#define POST_ACTOR_COMPILER 1
#line 1 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
/*
 * storageserver.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cinttypes>
#include <functional>
#include <iterator>
#include <limits>
#include <memory>
#include <type_traits>
#include <unordered_map>

#include "fdbclient/BlobCipher.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/Knobs.h"
#include "flow/ApiVersion.h"
#include "flow/Buggify.h"
#include "flow/Platform.h"
#include "flow/network.h"
#include "fmt/format.h"
#include "fdbclient/Audit.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/FDBTypes.h"
#include "fdbrpc/fdbrpc.h"
#include "fdbrpc/LoadBalance.h"
#include "fdbserver/OTELSpanContextMessage.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Hash3.h"
#include "flow/Histogram.h"
#include "flow/PriorityMultiLock.actor.h"
#include "flow/IRandom.h"
#include "flow/IndexedSet.h"
#include "flow/SystemMonitor.h"
#include "flow/Trace.h"
#include "fdbclient/Tracing.h"
#include "flow/Util.h"
#include "fdbclient/Atomic.h"
#include "fdbclient/AuditUtils.actor.h"
#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/KeyBackedTypes.actor.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/Notified.h"
#include "fdbclient/StatusClient.h"
#include "fdbclient/StorageServerShard.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/TransactionLineage.h"
#include "fdbclient/Tuple.h"
#include "fdbclient/VersionedMap.h"
#include "fdbrpc/sim_validation.h"
#include "fdbrpc/Smoother.h"
#include "fdbrpc/Stats.h"
#include "fdbserver/AccumulativeChecksumUtil.h"
#include "fdbserver/BulkDumpUtil.actor.h"
#include "fdbserver/BulkLoadUtil.actor.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/FDBExecHelper.actor.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/LatencyBandConfig.h"
#include "fdbserver/LogProtocolMessage.h"
#include "fdbserver/LogSystem.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/MutationTracking.h"
#include "fdbserver/OTELSpanContextMessage.h"
#include "fdbserver/Ratekeeper.h"
#include "fdbserver/ReadLatencySamples.h"
#include "fdbserver/RecoveryState.h"
#include "fdbserver/RocksDBCheckpointUtils.actor.h"
#include "fdbserver/ServerCheckpoint.actor.h"
#include "fdbserver/ServerDBInfo.h"
#include "fdbserver/SpanContextMessage.h"
#include "fdbserver/StorageMetrics.actor.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/TransactionTagCounter.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "fdbserver/StorageCorruptionBug.h"
#include "fdbserver/StorageServerUtils.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Hash3.h"
#include "flow/Histogram.h"
#include "flow/IRandom.h"
#include "flow/IndexedSet.h"
#include "flow/SystemMonitor.h"
#include "flow/TDMetric.actor.h"
#include "flow/Trace.h"
#include "flow/Util.h"
#include "flow/genericactors.actor.h"
#include "fdbserver/FDBRocksDBVersion.h"

#include "flow/actorcompiler.h" // This must be the last #include.

#ifndef __INTEL_COMPILER
#pragma region Data Structures
#endif

#define SHORT_CIRCUT_ACTUAL_STORAGE 0

namespace {
enum ChangeServerKeysContext { CSK_UPDATE, CSK_RESTORE, CSK_ASSIGN_EMPTY, CSK_FALL_BACK };

std::string changeServerKeysContextName(const ChangeServerKeysContext& context) {
	switch (context) {
	case CSK_UPDATE:
		return "Update";
	case CSK_RESTORE:
		return "Restore";
	case CSK_ASSIGN_EMPTY:
		return "AssignEmpty";
	case CSK_FALL_BACK:
		return "FallBackToFetchKeys";
	default:
		ASSERT(false);
	}
	return "UnknownContext";
}

bool canReplyWith(Error e) {
	switch (e.code()) {
	case error_code_transaction_too_old:
	case error_code_future_version:
	case error_code_wrong_shard_server:
	case error_code_process_behind:
	case error_code_watch_cancelled:
	case error_code_server_overloaded:
	// getMappedRange related exceptions that are not retriable:
	case error_code_mapper_bad_index:
	case error_code_mapper_no_such_key:
	case error_code_mapper_bad_range_decriptor:
	case error_code_quick_get_key_values_has_more:
	case error_code_quick_get_value_miss:
	case error_code_quick_get_key_values_miss:
	case error_code_get_mapped_key_values_has_more:
	case error_code_key_not_tuple:
	case error_code_value_not_tuple:
	case error_code_mapper_not_tuple:
		// case error_code_all_alternatives_failed:
		return true;
	default:
		return false;
	}
}

template <class Req>
Optional<ReadType> trackedReadType(Req const& req)
    requires requires(Req req) {
	    { req.options } -> std::same_as<Optional<ReadOptions>&>;
    }
{
	if (!SERVER_KNOBS->TRACK_READ_LATENCIES_PER_TYPE) {
		return {};
	}
	return req.options.map([](auto const& options) { return options.type; });
}

} // namespace

#define PERSIST_PREFIX "\xff\xff"

FDB_BOOLEAN_PARAM(UnlimitedCommitBytes);
FDB_BOOLEAN_PARAM(MoveInFailed);
FDB_BOOLEAN_PARAM(MoveInUpdatesSpilled);

// Immutable
static const KeyValueRef persistFormat(PERSIST_PREFIX "Format"_sr, "FoundationDB/StorageServer/1/4"_sr);
static const KeyValueRef persistShardAwareFormat(PERSIST_PREFIX "Format"_sr, "FoundationDB/StorageServer/1/5"_sr);
static const KeyRangeRef persistFormatReadableRange("FoundationDB/StorageServer/1/2"_sr,
                                                    "FoundationDB/StorageServer/1/6"_sr);
static const KeyRef persistID = PERSIST_PREFIX "ID"_sr;
static const KeyRef persistTssPairID = PERSIST_PREFIX "tssPairID"_sr;
static const KeyRef persistSSPairID = PERSIST_PREFIX "ssWithTSSPairID"_sr;
static const KeyRef persistTssQuarantine = PERSIST_PREFIX "tssQ"_sr;

// (Potentially) change with the durable version or when fetchKeys completes
static const KeyRef persistVersion = PERSIST_PREFIX "Version"_sr;
static const KeyRangeRef persistShardAssignedKeys =
    KeyRangeRef(PERSIST_PREFIX "ShardAssigned/"_sr, PERSIST_PREFIX "ShardAssigned0"_sr);
static const KeyRangeRef persistShardAvailableKeys =
    KeyRangeRef(PERSIST_PREFIX "ShardAvailable/"_sr, PERSIST_PREFIX "ShardAvailable0"_sr);
static const KeyRangeRef persistByteSampleKeys = KeyRangeRef(PERSIST_PREFIX "BS/"_sr, PERSIST_PREFIX "BS0"_sr);
static const KeyRangeRef persistByteSampleSampleKeys =
    KeyRangeRef(PERSIST_PREFIX "BS/"_sr PERSIST_PREFIX "BS/"_sr, PERSIST_PREFIX "BS/"_sr PERSIST_PREFIX "BS0"_sr);
static const KeyRef persistLogProtocol = PERSIST_PREFIX "LogProtocol"_sr;
static const KeyRef persistPrimaryLocality = PERSIST_PREFIX "PrimaryLocality"_sr;
// data keys are unmangled (but never start with PERSIST_PREFIX because they are always in allKeys)

static const KeyRangeRef persistStorageServerShardKeys =
    KeyRangeRef(PERSIST_PREFIX "StorageServerShard/"_sr, PERSIST_PREFIX "StorageServerShard0"_sr);

// Checkpoint related prefixes.
static const KeyRangeRef persistCheckpointKeys =
    KeyRangeRef(PERSIST_PREFIX "Checkpoint/"_sr, PERSIST_PREFIX "Checkpoint0"_sr);
static const KeyRangeRef persistPendingCheckpointKeys =
    KeyRangeRef(PERSIST_PREFIX "PendingCheckpoint/"_sr, PERSIST_PREFIX "PendingCheckpoint0"_sr);
static const std::string serverCheckpointFolder = "serverCheckpoints";
static const std::string checkpointBytesSampleTempFolder = "/metadata_temp";
static const std::string fetchedCheckpointFolder = "fetchedCheckpoints";
static const std::string serverBulkDumpFolder = "bulkDumpFiles";
static const std::string serverBulkLoadFolder = "bulkLoadFiles";
static const std::string invalidTeamId = "InvalidTeam";

static const KeyRangeRef persistBulkLoadTaskKeys =
    KeyRangeRef(PERSIST_PREFIX "BulkLoadTask/"_sr, PERSIST_PREFIX "BulkLoadTask0"_sr);

// Accumulative checksum related prefix
static const KeyRangeRef persistAccumulativeChecksumKeys =
    KeyRangeRef(PERSIST_PREFIX "AccumulativeChecksum/"_sr, PERSIST_PREFIX "AccumulativeChecksum0"_sr);

inline Key encodePersistAccumulativeChecksumKey(uint16_t acsIndex) {
	BinaryWriter wr(Unversioned());
	wr.serializeBytes(persistAccumulativeChecksumKeys.begin);
	wr << bigEndian16(acsIndex);
	return wr.toValue();
}

inline uint16_t decodePersistAccumulativeChecksumKey(const Key& key) {
	uint16_t acsIndex;
	BinaryReader rd(key.removePrefix(persistAccumulativeChecksumKeys.begin), Unversioned());
	rd >> acsIndex;
	return bigEndian16(acsIndex);
}

// MoveInUpdates caches new updates of a move-in shard, before that shard is ready to accept writes.
struct MoveInUpdates {
	MoveInUpdates() : spilled(MoveInUpdatesSpilled::False) {}
	MoveInUpdates(UID id,
	              Version version,
	              struct StorageServer* data,
	              IKeyValueStore* store,
	              MoveInUpdatesSpilled spilled);

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation,
	                 bool allowSpill);

	bool hasNext() const;

	std::vector<Standalone<VerUpdateRef>> next(const int byteLimit);
	const std::deque<Standalone<VerUpdateRef>>& getUpdatesQueue() const { return this->updates; }

	UID id;
	Version lastRepliedVersion;
	std::deque<Standalone<VerUpdateRef>> updates;
	std::vector<Standalone<VerUpdateRef>> spillBuffer;
	struct StorageServer* data;
	IKeyValueStore* store;
	KeyRange range;
	bool fail;
	MoveInUpdatesSpilled spilled;
	size_t size;
	Future<Void> loadFuture;
	Severity logSev;

private:
																#line 283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
[[nodiscard]] static Future<Void> loadUpdates( MoveInUpdates* const& self, Version const& begin, Version const& end );
template <class> friend class MoveInUpdates_LoadUpdatesActorState;

#line 282 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

	Key getPersistKey(const Version version, const int idx) const;
};

															#line 292 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
// This generated class is to be used only via loadUpdates()
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class MoveInUpdates_LoadUpdatesActor>
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class MoveInUpdates_LoadUpdatesActorState {
															#line 298 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates_LoadUpdatesActorState(MoveInUpdates* const& self,Version const& begin,Version const& end) 
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   begin(begin),
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   end(end)
															#line 309 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("loadUpdates", reinterpret_cast<unsigned long>(this));

	}
	~MoveInUpdates_LoadUpdatesActorState() 
	{
		fdb_probe_actor_destroy("loadUpdates", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 287 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spilled);
															#line 288 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (begin >= end)
															#line 326 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 289 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				self->spilled = MoveInUpdatesSpilled::False;
															#line 290 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 332 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~MoveInUpdates_LoadUpdatesActorState();
				static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 293 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const Key beginKey = persistUpdatesKey(self->id, begin), endKey = persistUpdatesKey(self->id, end);
															#line 294 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadBegin", self->id) .detail("BeginVersion", begin) .detail("EndVersion", end) .detail("BeginKey", beginKey) .detail("EndKey", endKey);
															#line 299 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(beginKey < endKey);
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = self->store->readRange(KeyRangeRef(beginKey, endKey), SERVER_KNOBS->FETCH_SHARD_UPDATES_BYTE_LIMIT, SERVER_KNOBS->FETCH_SHARD_UPDATES_BYTE_LIMIT);
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state = 1;
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*>(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)));
															#line 353 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& res,int loopDepth) 
	{
															#line 303 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Standalone<VerUpdateRef>> restored;
															#line 304 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 305 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const Version version = decodePersistUpdateVersion(res[i].key.removePrefix(self->range.begin));
															#line 306 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Standalone<VerUpdateRef> vur = BinaryReader::fromStringRef<Standalone<VerUpdateRef>>(res[i].value, IncludeVersion());
															#line 308 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version == vur.version);
															#line 309 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadedMutations", self->id) .detail("Version", version) .detail("Mutations", vur.mutations.size());
															#line 312 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			restored.push_back(std::move(vur));
															#line 388 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 315 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.more)
															#line 392 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 316 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = restored.size() - 1;i >= 0;--i) {
															#line 317 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (self->updates.empty() || restored[i].version < self->updates.front().version)
															#line 398 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 318 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					self->updates.push_front(std::move(restored[i]));
															#line 402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
			}
															#line 321 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			self->spilled = MoveInUpdatesSpilled::False;
															#line 407 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 323 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spillBuffer.empty());
															#line 324 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::swap(self->spillBuffer, restored);
															#line 415 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 327 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		self->loadFuture = Future<Void>();
															#line 328 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(self->logSev, "MoveInUpdatesLoadEnd", self->id) .detail("MinVersion", restored.empty() ? invalidVersion : restored.front().version) .detail("MaxVersion", restored.empty() ? invalidVersion : restored.back().version) .detail("VersionCount", restored.size()) .detail("LastBatch", !res.more);
															#line 334 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 423 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult && res,int loopDepth) 
	{
															#line 303 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Standalone<VerUpdateRef>> restored;
															#line 304 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 305 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const Version version = decodePersistUpdateVersion(res[i].key.removePrefix(self->range.begin));
															#line 306 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Standalone<VerUpdateRef> vur = BinaryReader::fromStringRef<Standalone<VerUpdateRef>>(res[i].value, IncludeVersion());
															#line 308 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version == vur.version);
															#line 309 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadedMutations", self->id) .detail("Version", version) .detail("Mutations", vur.mutations.size());
															#line 312 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			restored.push_back(std::move(vur));
															#line 447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 315 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.more)
															#line 451 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 316 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = restored.size() - 1;i >= 0;--i) {
															#line 317 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (self->updates.empty() || restored[i].version < self->updates.front().version)
															#line 457 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 318 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					self->updates.push_front(std::move(restored[i]));
															#line 461 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
			}
															#line 321 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			self->spilled = MoveInUpdatesSpilled::False;
															#line 466 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 323 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spillBuffer.empty());
															#line 324 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::swap(self->spillBuffer, restored);
															#line 474 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 327 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		self->loadFuture = Future<Void>();
															#line 328 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(self->logSev, "MoveInUpdatesLoadEnd", self->id) .detail("MinVersion", restored.empty() ? invalidVersion : restored.front().version) .detail("MaxVersion", restored.empty() ? invalidVersion : restored.back().version) .detail("VersionCount", restored.size()) .detail("LastBatch", !res.more);
															#line 334 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont1(res, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state > 0) static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state = 0;
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates* self;
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version begin;
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version end;
															#line 571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via loadUpdates()
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class MoveInUpdates_LoadUpdatesActor final : public Actor<Void>, public ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >, public FastAllocated<MoveInUpdates_LoadUpdatesActor>, public MoveInUpdates_LoadUpdatesActorState<MoveInUpdates_LoadUpdatesActor> {
															#line 576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<MoveInUpdates_LoadUpdatesActor>::operator new;
	using FastAllocated<MoveInUpdates_LoadUpdatesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8826889450964028416UL, 11767904229484728320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >;
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates_LoadUpdatesActor(MoveInUpdates* const& self,Version const& begin,Version const& end) 
															#line 593 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   MoveInUpdates_LoadUpdatesActorState<MoveInUpdates_LoadUpdatesActor>(self, begin, end),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5561495993262577152UL, 4925366652637659904UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadUpdates");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> MoveInUpdates::loadUpdates( MoveInUpdates* const& self, Version const& begin, Version const& end ) {
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new MoveInUpdates_LoadUpdatesActor(self, begin, end));
															#line 625 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 336 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

bool MoveInUpdates::hasNext() const {
	return this->spilled || (!this->updates.empty() && this->updates.back().version > this->lastRepliedVersion);
}

// MoveInShard corresponds to a move-in physical shard, a class representation of MoveInShardMetaData.
struct MoveInShard {
	std::shared_ptr<MoveInShardMetaData> meta;
	struct StorageServer* server;
	std::shared_ptr<MoveInUpdates> updates;
	bool isRestored;
	Version transferredVersion;
	ConductBulkLoad conductBulkLoad = ConductBulkLoad::False;

	Future<Void> fetchClient; // holds FetchShard() actor
	Promise<Void> fetchComplete;
	Promise<Void> readWrite;

	Severity logSev = static_cast<Severity>(SERVER_KNOBS->PHYSICAL_SHARD_MOVE_LOG_SEVERITY);

	MoveInShard() = default;
	MoveInShard(StorageServer* server,
	            const UID& id,
	            const UID& dataMoveId,
	            const Version version,
	            const ConductBulkLoad conductBulkLoad,
	            MoveInPhase phase);
	MoveInShard(StorageServer* server,
	            const UID& id,
	            const UID& dataMoveId,
	            const Version version,
	            const ConductBulkLoad conductBulkLoad);
	MoveInShard(StorageServer* server, MoveInShardMetaData meta);
	~MoveInShard();

	UID id() const { return this->meta->id; }
	UID dataMoveId() const { return this->meta->dataMoveId; }
	void setPhase(const MoveInPhase& phase) { this->meta->setPhase(phase); }
	MoveInPhase getPhase() const { return this->meta->getPhase(); }
	const std::vector<KeyRange>& ranges() const { return this->meta->ranges; }
	const std::vector<CheckpointMetaData>& checkpoints() const { return this->meta->checkpoints; }
	std::string destShardIdString() const { return this->meta->destShardIdString(); }
	void addRange(const KeyRangeRef range);
	void removeRange(const KeyRangeRef range);
	void cancel(const MoveInFailed failed = MoveInFailed::False);
	bool isDataTransferred() const { return meta->getPhase() >= MoveInPhase::ApplyingUpdates; }
	bool isDataAndCFTransferred() const { throw not_implemented(); }
	bool failed() const { return this->getPhase() == MoveInPhase::Cancel || this->getPhase() == MoveInPhase::Error; }
	void setHighWatermark(const Version version) { this->meta->highWatermark = version; }
	Version getHighWatermark() const { return this->meta->highWatermark; }

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);

	KeyRangeRef getAffectedRange(const MutationRef& mutation) const;

	std::string toString() const { return meta != nullptr ? meta->toString() : "Empty"; }
};

struct AddingShard : NonCopyable {
	KeyRange keys;
	Future<Void> fetchClient; // holds FetchKeys() actor
	Promise<Void> fetchComplete;
	Promise<Void> readWrite;
	DataMovementReason reason;
	SSBulkLoadMetadata ssBulkLoadMetadata;
	std::string teamId;

	// During the Fetching phase, it saves newer mutations whose version is greater or equal to fetchClient's
	// fetchVersion, while the shard is still busy catching up with fetchClient. It applies these updates after fetching
	// completes.
	std::deque<Standalone<VerUpdateRef>> updates;

	struct StorageServer* server;
	Version transferredVersion;
	Version fetchVersion;

	// To learn more details of the phase transitions, see function fetchKeys(). The phases below are sorted in
	// chronological order and do not go back.
	enum Phase {
		WaitPrevious,
		// During Fetching phase, it fetches data before fetchVersion and write it to storage, then let updater know it
		// is ready to update the deferred updates` (see the comment of member variable `updates` above).
		Fetching,
		// During the FetchingCF phase, the shard data is transferred but the remaining change feed data is still being
		// transferred. This is equivalent to the waiting phase for non-changefeed data.
		// TODO(gglass): remove FetchingCF.  Probably requires some refactoring of permanent logic,
		// not just flat out removal of CF-specific logic, so come back to this.
		FetchingCF,
		// During Waiting phase, it sends updater the deferred updates, and wait until they are durable.
		Waiting
		// The shard's state is changed from adding to readWrite then.
	};

	Phase phase;

	AddingShard(StorageServer* server,
	            KeyRangeRef const& keys,
	            DataMovementReason reason,
	            const SSBulkLoadMetadata& ssBulkLoadMetadata);

	// When fetchKeys "partially completes" (splits an adding shard in two), this is used to construct the left half
	AddingShard(AddingShard* prev, KeyRange const& keys)
	  : keys(keys), fetchClient(prev->fetchClient), server(prev->server), transferredVersion(prev->transferredVersion),
	    fetchVersion(prev->fetchVersion), phase(prev->phase), reason(prev->reason),
	    ssBulkLoadMetadata(prev->ssBulkLoadMetadata) {}
	~AddingShard() {
		if (!fetchComplete.isSet())
			fetchComplete.send(Void());
		if (!readWrite.isSet())
			readWrite.send(Void());
	}

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);

	bool isDataTransferred() const { return phase >= FetchingCF; }
	bool isDataAndCFTransferred() const { return phase >= Waiting; }

	SSBulkLoadMetadata getSSBulkLoadMetadata() const { return ssBulkLoadMetadata; }
};

class ShardInfo : public ReferenceCounted<ShardInfo>, NonCopyable {
private:
	ShardInfo(KeyRange keys, std::unique_ptr<AddingShard>&& adding, StorageServer* readWrite)
	  : adding(std::move(adding)), readWrite(readWrite), keys(keys), shardId(0LL), desiredShardId(0LL), version(0) {}
	ShardInfo(KeyRange keys, std::shared_ptr<MoveInShard> moveInShard)
	  : adding(nullptr), readWrite(nullptr), moveInShard(moveInShard), keys(keys),
	    shardId(moveInShard->meta->destShardId()), desiredShardId(moveInShard->meta->destShardId()),
	    version(moveInShard->meta->createVersion) {}

	// A shard has 4 mutual exclusive states: adding, moveInShard, readWrite and notAssigned.
	std::unique_ptr<AddingShard> adding;
	struct StorageServer* readWrite;
	std::shared_ptr<MoveInShard> moveInShard; // The shard is being moved in via physical-shard-move.
	KeyRange keys;
	uint64_t changeCounter;
	uint64_t shardId;
	uint64_t desiredShardId;
	std::string teamId = invalidTeamId;
	Version version;

public:
	static ShardInfo* newNotAssigned(KeyRange keys) { return new ShardInfo(keys, nullptr, nullptr); }
	static ShardInfo* newReadWrite(KeyRange keys, StorageServer* data) { return new ShardInfo(keys, nullptr, data); }
	static ShardInfo* newAdding(StorageServer* data,
	                            KeyRange keys,
	                            DataMovementReason reason,
	                            const SSBulkLoadMetadata& ssBulkLoadMetadata) {
		return new ShardInfo(keys, std::make_unique<AddingShard>(data, keys, reason, ssBulkLoadMetadata), nullptr);
	}
	static ShardInfo* addingSplitLeft(KeyRange keys, AddingShard* oldShard) {
		return new ShardInfo(keys, std::make_unique<AddingShard>(oldShard, keys), nullptr);
	}

	static ShardInfo* newShard(StorageServer* data, const StorageServerShard& shard);

	static bool canMerge(const ShardInfo* l, const ShardInfo* r) {
		if (l == nullptr || r == nullptr || l->keys.end != r->keys.begin || l->version == invalidVersion ||
		    r->version == invalidVersion) {
			return false;
		}
		if (l->getShardId() != r->getShardId() || l->getDesiredShardId() != r->getDesiredShardId()) {
			return false;
		}
		return (l->isReadable() && r->isReadable()) || (!l->assigned() && !r->assigned());
	}

	StorageServerShard toStorageServerShard() const {
		StorageServerShard::ShardState st = StorageServerShard::NotAssigned;
		Optional<UID> moveInShardId;
		if (this->isReadable()) {
			st = StorageServerShard::ReadWrite;
		} else if (!this->assigned()) {
			st = StorageServerShard::NotAssigned;
		} else if (this->getAddingShard()) {
			st = this->getAddingShard()->phase == AddingShard::Waiting ? StorageServerShard::ReadWritePending
			                                                           : StorageServerShard::Adding;
		} else {
			ASSERT(this->getMoveInShard());
			const MoveInPhase phase = this->getMoveInShard()->getPhase();
			if (phase < MoveInPhase::ReadWritePending) {
				st = StorageServerShard::MovingIn;
			} else if (phase == MoveInPhase::ReadWritePending) {
				st = StorageServerShard::ReadWritePending;
			} else if (phase == MoveInPhase::Complete) {
				st = StorageServerShard::ReadWrite;
			} else {
				st = StorageServerShard::MovingIn;
			}
			// Clear moveInShardId if the data move is complete.
			if (phase != MoveInPhase::ReadWritePending && phase != MoveInPhase::Complete) {
				moveInShardId = this->getMoveInShard()->id();
			}
		}
		return StorageServerShard(this->keys, this->version, this->shardId, this->desiredShardId, st, moveInShardId);
	}

	// Copies necessary information from `shard`.
	void populateShard(const StorageServerShard& shard) {
		this->version = shard.version;
		this->shardId = shard.id;
		this->desiredShardId = shard.desiredId;
	}

	// Returns true if the current shard is merged with `other`.
	bool mergeWith(const ShardInfo* other) {
		if (!canMerge(this, other)) {
			return false;
		}
		this->keys = KeyRangeRef(this->keys.begin, other->range().end);
		this->version = std::max(this->version, other->getVersion());
		return true;
	}

	void validate() const {
		// TODO: Complete this.
	}

	bool isReadable() const { return readWrite != nullptr; }
	bool notAssigned() const { return !readWrite && !adding && !moveInShard; }
	bool assigned() const { return readWrite || adding || moveInShard; }

	KeyRange range() const { return keys; }
	uint64_t getShardId() const { return shardId; }
	uint64_t getDesiredShardId() const { return desiredShardId; }
	uint64_t getChangeCounter() const { return changeCounter; }
	AddingShard* getAddingShard() const { return adding.get(); }
	std::shared_ptr<MoveInShard> getMoveInShard() const { return moveInShard; }
	Version getVersion() const { return version; }
	std::string getTeamId() const { return teamId; }

	void setChangeCounter(uint64_t shardChangeCounter) { changeCounter = shardChangeCounter; }
	void setShardId(uint64_t id) { shardId = id; }
	void setDesiredShardId(uint64_t id) { desiredShardId = id; }

	bool isInVersionedData() const {
		return readWrite || (adding && adding->isDataTransferred()) ||
		       (moveInShard && moveInShard->isDataTransferred());
	}
	bool isCFInVersionedData() const { return readWrite || (adding && adding->isDataAndCFTransferred()); }
	bool isReadWritePending() const {
		return isCFInVersionedData() || (moveInShard && (moveInShard->getPhase() == MoveInPhase::ReadWritePending ||
		                                                 moveInShard->getPhase() == MoveInPhase::Complete));
	}
	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);
	bool isFetched() const {
		return readWrite || (adding && adding->fetchComplete.isSet()) ||
		       (moveInShard && moveInShard->fetchComplete.isSet());
	}

	bool isFetching() const {
		if (adding) {
			return !adding->fetchComplete.isSet();
		}
		if (moveInShard) {
			return !moveInShard->fetchComplete.isSet();
		}
		return false;
	}

	std::string debugDescribeState() const {
		if (notAssigned()) {
			return "NotAssigned";
		} else if (adding && !adding->isDataAndCFTransferred()) {
			return "AddingFetchingCF";
		} else if (adding && !adding->isDataTransferred()) {
			return "AddingFetching";
		} else if (adding) {
			return "AddingTransferred";
		} else if (moveInShard) {
			return moveInShard->meta->toString();
		} else {
			return "ReadWrite";
		}
	}
};

struct StorageServerDisk {
	explicit StorageServerDisk(struct StorageServer* data, IKeyValueStore* storage) : data(data), storage(storage) {}

	IKeyValueStore* getKeyValueStore() const { return this->storage; }

	void makeNewStorageServerDurable(const bool shardAware);
	bool makeVersionMutationsDurable(Version& prevStorageVersion,
	                                 Version newStorageVersion,
	                                 int64_t& bytesLeft,
	                                 UnlimitedCommitBytes unlimitedCommitBytes,
	                                 int64_t& clearRangesLeft,
	                                 const UID& ssId,
	                                 bool verbose = false);
	void makeVersionDurable(Version version);
	void makeAccumulativeChecksumDurable(const AccumulativeChecksumState& acsState);
	void clearAccumulativeChecksumState(const AccumulativeChecksumState& acsState);
	void makeTssQuarantineDurable();
	Future<bool> restoreDurableState();

	void changeLogProtocol(Version version, ProtocolVersion protocol);

	void writeMutation(MutationRef mutation);
	void writeKeyValue(KeyValueRef kv);
	void clearRange(KeyRangeRef keys);

	Future<Void> addRange(KeyRangeRef range, std::string id) {
		return storage->addRange(range, id, !SERVER_KNOBS->SHARDED_ROCKSDB_DELAY_COMPACTION_FOR_DATA_MOVE);
	}

	std::vector<std::string> removeRange(KeyRangeRef range) { return storage->removeRange(range); }

	void markRangeAsActive(KeyRangeRef range) { storage->markRangeAsActive(range); }

	Future<Void> replaceRange(KeyRange range, Standalone<VectorRef<KeyValueRef>> data) {
		return storage->replaceRange(range, data);
	}

	void persistRangeMapping(KeyRangeRef range, bool isAdd) { storage->persistRangeMapping(range, isAdd); }

	CoalescedKeyRangeMap<std::string> getExistingRanges() { return storage->getExistingRanges(); }

	Future<Void> getError() { return storage->getError(); }
	Future<Void> init() { return storage->init(); }
	Future<Void> canCommit() { return storage->canCommit(); }
	Future<Void> commit() { return storage->commit(); }

	void logRecentRocksDBBackgroundWorkStats(UID ssId, std::string logReason) {
		return storage->logRecentRocksDBBackgroundWorkStats(ssId, logReason);
	}

	// SOMEDAY: Put readNextKeyInclusive in IKeyValueStore
	// Read the key that is equal or greater then 'key' from the storage engine.
	// For example, readNextKeyInclusive("a") should return:
	//  - "a", if key "a" exist
	//  - "b", if key "a" doesn't exist, and "b" is the next existing key in total order
	//  - allKeys.end, if keyrange [a, allKeys.end) is empty
	Future<Key> readNextKeyInclusive(KeyRef key, Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvScans);
		return readFirstKey(storage, KeyRangeRef(key, allKeys.end), options);
	}
	Future<Optional<Value>> readValue(KeyRef key, Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvGets);
		return storage->readValue(key, options);
	}
	Future<Optional<Value>> readValuePrefix(KeyRef key,
	                                        int maxLength,
	                                        Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvGets);
		return storage->readValuePrefix(key, maxLength, options);
	}
	Future<RangeResult> readRange(KeyRangeRef keys,
	                              int rowLimit = 1 << 30,
	                              int byteLimit = 1 << 30,
	                              Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvScans);
		return storage->readRange(keys, rowLimit, byteLimit, options);
	}

	Future<CheckpointMetaData> checkpoint(const CheckpointRequest& request) { return storage->checkpoint(request); }

	Future<Void> restore(const std::vector<CheckpointMetaData>& checkpoints) { return storage->restore(checkpoints); }

	Future<Void> restore(const std::string& shardId,
	                     const std::vector<KeyRange>& ranges,
	                     const std::vector<CheckpointMetaData>& checkpoints) {
		return storage->restore(shardId, ranges, checkpoints);
	}

	Future<Void> deleteCheckpoint(const CheckpointMetaData& checkpoint) {
		return storage->deleteCheckpoint(checkpoint);
	}

	KeyValueStoreType getKeyValueStoreType() const { return storage->getType(); }
	StorageBytes getStorageBytes() const { return storage->getStorageBytes(); }
	std::tuple<size_t, size_t, size_t> getSize() const { return storage->getSize(); }

	Future<EncryptionAtRestMode> encryptionMode() { return storage->encryptionMode(); }

	// The following are pointers to the Counters in StorageServer::counters of the same names.
	Counter* kvCommitLogicalBytes;
	Counter* kvClearRanges;
	Counter* kvClearSingleKey;
	Counter* kvGets;
	Counter* kvScans;
	Counter* kvCommits;

private:
	struct StorageServer* data;
	IKeyValueStore* storage;
	void writeMutations(const VectorRef<MutationRef>& mutations, Version debugVersion, const char* debugContext);
	void writeMutationsBuggy(const VectorRef<MutationRef>& mutations, Version debugVersion, const char* debugContext);

																#line 1026 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
// This generated class is to be used only via readFirstKey()
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class ReadFirstKeyActor>
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadFirstKeyActorState {
															#line 1032 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadFirstKeyActorState(IKeyValueStore* const& storage,KeyRangeRef const& range,Optional<ReadOptions> const& options) 
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : storage(storage),
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options)
															#line 1043 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readFirstKey", reinterpret_cast<unsigned long>(this));

	}
	~ReadFirstKeyActorState() 
	{
		fdb_probe_actor_destroy("readFirstKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 734 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = storage->readRange(range, 1, 1 << 30, options);
															#line 734 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<ReadFirstKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1060 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadFirstKeyActor*>(this)->actor_wait_state = 1;
															#line 734 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadFirstKeyActor, 0, RangeResult >*>(static_cast<ReadFirstKeyActor*>(this)));
															#line 1065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadFirstKeyActorState();
		static_cast<ReadFirstKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& r,int loopDepth) 
	{
															#line 735 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.size())
															#line 1088 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 736 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(r[0].key); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1092 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(r[0].key);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 738 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(range.end); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1102 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(range.end);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1cont1(RangeResult && r,int loopDepth) 
	{
															#line 735 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.size())
															#line 1115 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 736 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(r[0].key); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1119 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(r[0].key);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 738 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(range.end); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1129 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(range.end);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1when1(RangeResult const& r,int loopDepth) 
	{
		loopDepth = a_body1cont1(r, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && r,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadFirstKeyActor*>(this)->actor_wait_state > 0) static_cast<ReadFirstKeyActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFirstKeyActor*>(this)->ActorCallback< ReadFirstKeyActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	IKeyValueStore* storage;
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRangeRef range;
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 1219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readFirstKey()
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadFirstKeyActor final : public Actor<Key>, public ActorCallback< ReadFirstKeyActor, 0, RangeResult >, public FastAllocated<ReadFirstKeyActor>, public ReadFirstKeyActorState<ReadFirstKeyActor> {
															#line 1224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadFirstKeyActor>::operator new;
	using FastAllocated<ReadFirstKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15096462444186707968UL, 7765072823124777216UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Key>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadFirstKeyActor, 0, RangeResult >;
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadFirstKeyActor(IKeyValueStore* const& storage,KeyRangeRef const& range,Optional<ReadOptions> const& options) 
															#line 1241 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Key>(),
		   ReadFirstKeyActorState<ReadFirstKeyActor>(storage, range, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4418144802222492160UL, 5296783460543853568UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readFirstKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadFirstKeyActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] static Future<Key> readFirstKey( IKeyValueStore* const& storage, KeyRangeRef const& range, Optional<ReadOptions> const& options ) {
															#line 733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Key>(new ReadFirstKeyActor(storage, range, options));
															#line 1273 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 740 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
};

struct UpdateEagerReadInfo {
	std::vector<KeyRef> keyBegin;
	std::vector<Key> keyEnd; // these are for ClearRange

	std::vector<std::pair<KeyRef, int>> keys;
	std::vector<Optional<Value>> value;

	Arena arena;
	bool enableClearRangeEagerReads;

	UpdateEagerReadInfo(bool enableClearRangeEagerReads) : enableClearRangeEagerReads(enableClearRangeEagerReads) {}

	void addMutations(VectorRef<MutationRef> const& mutations) {
		for (auto& m : mutations)
			addMutation(m);
	}

	void addMutation(MutationRef const& m) {
		// SOMEDAY: Theoretically we can avoid a read if there is an earlier overlapping ClearRange
		if (m.type == MutationRef::ClearRange && !m.param2.startsWith(systemKeys.end) && enableClearRangeEagerReads)
			keyBegin.push_back(m.param2);
		else if (m.type == MutationRef::CompareAndClear) {
			if (enableClearRangeEagerReads)
				keyBegin.push_back(keyAfter(m.param1, arena));
			if (keys.size() > 0 && keys.back().first == m.param1) {
				// Don't issue a second read, if the last read was equal to the current key.
				// CompareAndClear is likely to be used after another atomic operation on same key.
				keys.back().second = std::max(keys.back().second, m.param2.size() + 1);
			} else {
				keys.emplace_back(m.param1, m.param2.size() + 1);
			}
		} else if ((m.type == MutationRef::AppendIfFits) || (m.type == MutationRef::ByteMin) ||
		           (m.type == MutationRef::ByteMax))
			keys.emplace_back(m.param1, CLIENT_KNOBS->VALUE_SIZE_LIMIT);
		else if (isAtomicOp((MutationRef::Type)m.type))
			keys.emplace_back(m.param1, m.param2.size());
	}

	void finishKeyBegin() {
		if (enableClearRangeEagerReads) {
			std::sort(keyBegin.begin(), keyBegin.end());
			keyBegin.resize(std::unique(keyBegin.begin(), keyBegin.end()) - keyBegin.begin());
		}
		std::sort(keys.begin(), keys.end(), [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			return (lhs.first < rhs.first) || (lhs.first == rhs.first && lhs.second > rhs.second);
		});
		keys.resize(std::unique(keys.begin(),
		                        keys.end(),
		                        [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			                        return lhs.first == rhs.first;
		                        }) -
		            keys.begin());
		// value gets populated in doEagerReads
	}

	Optional<Value>& getValue(KeyRef key) {
		int i = std::lower_bound(keys.begin(),
		                         keys.end(),
		                         std::pair<KeyRef, int>(key, 0),
		                         [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			                         return lhs.first < rhs.first;
		                         }) -
		        keys.begin();
		ASSERT(i < keys.size() && keys[i].first == key);
		return value[i];
	}

	KeyRef getKeyEnd(KeyRef key) {
		int i = std::lower_bound(keyBegin.begin(), keyBegin.end(), key) - keyBegin.begin();
		ASSERT(i < keyBegin.size() && keyBegin[i] == key);
		return keyEnd[i];
	}
};

const int VERSION_OVERHEAD =
    64 + sizeof(Version) + sizeof(Standalone<VerUpdateRef>) + // mutationLog, 64b overhead for map
    2 * (64 + sizeof(Version) +
         sizeof(Reference<VersionedMap<KeyRef, ValueOrClearToRef>::PTreeT>)); // versioned map [ x2 for
                                                                              // createNewVersion(version+1) ], 64b
                                                                              // overhead for map

static int mvccStorageBytes(MutationRef const& m) {
	return mvccStorageBytes(m.param1.size() + m.param2.size());
}

struct FetchInjectionInfo {
	Arena arena;
	std::vector<VerUpdateRef> changes;
};

class ServerWatchMetadata : public ReferenceCounted<ServerWatchMetadata> {
public:
	Key key;
	Optional<Value> value;
	Version version;
	Future<Version> watch_impl;
	Promise<Version> versionPromise;
	Optional<TagSet> tags;
	Optional<UID> debugID;

	ServerWatchMetadata(Key key, Optional<Value> value, Version version, Optional<TagSet> tags, Optional<UID> debugID)
	  : key(key), value(value), version(version), tags(tags), debugID(debugID) {}
};

struct BusiestWriteTagContext {
	const std::string busiestWriteTagTrackingKey;
	UID ratekeeperID;
	Reference<EventCacheHolder> busiestWriteTagEventHolder;
	double lastUpdateTime;

	BusiestWriteTagContext(const UID& thisServerID)
	  : busiestWriteTagTrackingKey(thisServerID.toString() + "/BusiestWriteTag"), ratekeeperID(UID()),
	    busiestWriteTagEventHolder(makeReference<EventCacheHolder>(busiestWriteTagTrackingKey)), lastUpdateTime(-1) {}
};

struct SSBulkLoadMetrics {
public:
	SSBulkLoadMetrics() : ongoingTasks(0) {}
	void addTask() { ongoingTasks++; }
	void removeTask() { ongoingTasks--; }
	int getOngoingTasks() { return ongoingTasks; }

private:
	int ongoingTasks = 0;
};

struct StorageServer : public IStorageMetricsService {
	typedef VersionedMap<KeyRef, ValueOrClearToRef> VersionedData;

private:
	// versionedData contains sets and clears.

	// * Nonoverlapping: No clear overlaps a set or another clear, or adjoins another clear.
	// ~ Clears are maximal: If versionedData.at(v) contains a clear [b,e) then
	//      there is a key data[e]@v, or e==allKeys.end, or a shard boundary or former boundary at e

	// * Reads are possible: When k is in a readable shard, for any v in [storageVersion, version.get()],
	//      storage[k] + versionedData.at(v)[k] = database[k] @ v    (storage[k] might be @ any version in
	//      [durableVersion, storageVersion])

	// * Transferred shards are partially readable: When k is in an adding, transferred shard, for any v in
	// [transferredVersion, version.get()],
	//      storage[k] + versionedData.at(v)[k] = database[k] @ v

	// * versionedData contains versions [storageVersion(), version.get()].  It might also contain version
	// (version.get()+1), in which changeDurableVersion may be deleting ghosts, and/or it might
	//      contain later versions if applyUpdate is on the stack.

	// * Old shards are erased: versionedData.atLatest() has entries (sets or intersecting clears) only for keys in
	// readable or adding,transferred shards.
	//   Earlier versions may have extra entries for shards that *were* readable or adding,transferred when those
	//   versions were the latest, but they eventually are forgotten.

	// * Old mutations are erased: All items in versionedData.atLatest() have insertVersion() > durableVersion(), but
	// views
	//   at older versions may contain older items which are also in storage (this is OK because of idempotency)

	VersionedData versionedData;
	std::map<Version, Standalone<VerUpdateRef>> mutationLog; // versions (durableVersion, version]

	using WatchMapKey = Key;
	using WatchMapKeyHasher = boost::hash<WatchMapKey>;
	using WatchMapValue = Reference<ServerWatchMetadata>;
	using WatchMap_t = std::unordered_map<WatchMapKey, WatchMapValue, WatchMapKeyHasher>;
	WatchMap_t watchMap; // keep track of server watches

public:
	struct PendingNewShard {
		PendingNewShard(uint64_t shardId, KeyRangeRef range) : shardId(format("%016llx", shardId)), range(range) {}

		std::string toString() const {
			return fmt::format("PendingNewShard: [ShardID]: {} [Range]: {}",
			                   this->shardId,
			                   Traceable<KeyRangeRef>::toString(this->range));
		}

		std::string shardId;
		KeyRange range;
	};

	std::map<Version, std::vector<CheckpointMetaData>> pendingCheckpoints; // Pending checkpoint requests
	std::unordered_map<UID, CheckpointMetaData> checkpoints; // Existing and deleting checkpoints
	std::unordered_map<UID, ICheckpointReader*> liveCheckpointReaders; // Active checkpoint readers
	std::map<Version, std::vector<PendingNewShard>>
	    pendingAddRanges; // Pending requests to add ranges to physical shards
	std::map<Version, std::vector<KeyRange>>
	    pendingRemoveRanges; // Pending requests to remove ranges from physical shards
	std::deque<std::pair<Standalone<StringRef>, Standalone<StringRef>>> constructedData;

	bool shardAware; // True if the storage server is aware of the physical shards.

	LocalityData locality; // Storage server's locality information.

	// Histograms
	struct FetchKeysHistograms {
		const Reference<Histogram> latency;
		const Reference<Histogram> bytes;
		const Reference<Histogram> bandwidth;
		const Reference<Histogram> bytesPerCommit;

		FetchKeysHistograms()
		  : latency(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                    FETCH_KEYS_LATENCY_HISTOGRAM,
		                                    Histogram::Unit::milliseconds)),
		    bytes(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                  FETCH_KEYS_BYTES_HISTOGRAM,
		                                  Histogram::Unit::bytes)),
		    bandwidth(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                      FETCH_KEYS_BYTES_PER_SECOND_HISTOGRAM,
		                                      Histogram::Unit::bytes_per_second)),
		    bytesPerCommit(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                           FETCH_KEYS_BYTES_PER_COMMIT_HISTOGRAM,
		                                           Histogram::Unit::bytes)) {}
	} fetchKeysHistograms;

	Reference<Histogram> tlogCursorReadsLatencyHistogram;
	Reference<Histogram> ssVersionLockLatencyHistogram;
	Reference<Histogram> eagerReadsLatencyHistogram;
	Reference<Histogram> fetchKeysPTreeUpdatesLatencyHistogram;
	Reference<Histogram> tLogMsgsPTreeUpdatesLatencyHistogram;
	Reference<Histogram> storageUpdatesDurableLatencyHistogram;
	Reference<Histogram> storageCommitLatencyHistogram;
	Reference<Histogram> ssDurableVersionUpdateLatencyHistogram;
	// Histograms of requests sent to KVS.
	Reference<Histogram> readRangeBytesReturnedHistogram;
	Reference<Histogram> readRangeBytesLimitHistogram;
	Reference<Histogram> readRangeKVPairsReturnedHistogram;

	// watch map operations
	Reference<ServerWatchMetadata> getWatchMetadata(KeyRef key) const;
	KeyRef setWatchMetadata(Reference<ServerWatchMetadata> metadata);
	void deleteWatchMetadata(KeyRef key);
	void clearWatchMetadata();

	std::vector<StorageServerShard> getStorageServerShards(KeyRangeRef range);
	std::shared_ptr<MoveInShard> getMoveInShard(const UID& dataMoveId,
	                                            const Version version,
	                                            const ConductBulkLoad conductBulkLoad);

	class CurrentRunningFetchKeys {
		std::unordered_map<UID, double> startTimeMap;
		std::unordered_map<UID, KeyRange> keyRangeMap;

		static const StringRef emptyString;
		static const KeyRangeRef emptyKeyRange;

	public:
		void recordStart(const UID id, const KeyRange& keyRange) {
			startTimeMap[id] = now();
			keyRangeMap[id] = keyRange;
		}

		void recordFinish(const UID id) {
			startTimeMap.erase(id);
			keyRangeMap.erase(id);
		}

		std::pair<double, KeyRange> longestTime() const {
			if (numRunning() == 0) {
				return { -1, emptyKeyRange };
			}

			const double currentTime = now();
			double longest = 0;
			UID UIDofLongest;
			for (const auto& kv : startTimeMap) {
				const double currentRunningTime = currentTime - kv.second;
				if (longest <= currentRunningTime) {
					longest = currentRunningTime;
					UIDofLongest = kv.first;
				}
			}
			if (BUGGIFY) {
				UIDofLongest = deterministicRandom()->randomUniqueID();
			}
			auto it = keyRangeMap.find(UIDofLongest);
			if (it != keyRangeMap.end()) {
				return { longest, it->second };
			}
			return { -1, emptyKeyRange };
		}

		int numRunning() const { return startTimeMap.size(); }
	} currentRunningFetchKeys;

	Tag tag;
	std::vector<std::pair<Version, Tag>> history;
	std::vector<std::pair<Version, Tag>> allHistory;
	Version poppedAllAfter;
	std::map<Version, Arena>
	    freeable; // for each version, an Arena that must be held until that version is < oldestVersion
	Arena lastArena;
	double cpuUsage;
	double diskUsage;

	std::map<Version, Standalone<VerUpdateRef>> const& getMutationLog() const { return mutationLog; }
	std::map<Version, Standalone<VerUpdateRef>>& getMutableMutationLog() { return mutationLog; }
	VersionedData const& data() const { return versionedData; }
	VersionedData& mutableData() { return versionedData; }

	mutable double old_rate = 1.0;
	double currentRate() const {
		auto versionLag = version.get() - durableVersion.get();
		double res;
		if (versionLag >= SERVER_KNOBS->STORAGE_DURABILITY_LAG_HARD_MAX) {
			res = 0.0;
		} else if (versionLag > SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX) {
			res =
			    1.0 -
			    (double(versionLag - SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX) /
			     double(SERVER_KNOBS->STORAGE_DURABILITY_LAG_HARD_MAX - SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX));
		} else {
			res = 1.0;
		}
		if (res != old_rate) {
			TraceEvent(SevDebug, "LocalRatekeeperChange", thisServerID)
			    .detail("Old", old_rate)
			    .detail("New", res)
			    .detail("NonDurableVersions", versionLag);
			old_rate = res;
		}
		return res;
	}

	void addMutationToMutationLogOrStorage(
	    Version ver,
	    MutationRef m); // Appends m to mutationLog@ver, or to storage if ver==invalidVersion

	// Update the byteSample, and write the updates to the mutation log@ver, or to storage if ver==invalidVersion
	void byteSampleApplyMutation(MutationRef const& m, Version ver);
	void byteSampleApplySet(KeyValueRef kv, Version ver);
	void byteSampleApplyClear(KeyRangeRef range, Version ver);

	void popVersion(Version v, bool popAllTags = false) {
		if (logSystem && !isTss()) {
			if (v > poppedAllAfter) {
				popAllTags = true;
				poppedAllAfter = std::numeric_limits<Version>::max();
			}

			std::vector<std::pair<Version, Tag>>* hist = &history;
			std::vector<std::pair<Version, Tag>> allHistoryCopy;
			if (popAllTags) {
				allHistoryCopy = allHistory;
				hist = &allHistoryCopy;
			}

			while (hist->size() && v > hist->back().first) {
				logSystem->pop(v, hist->back().second);
				hist->pop_back();
			}
			if (hist->size()) {
				logSystem->pop(v, hist->back().second);
			} else {
				logSystem->pop(v, tag);
			}
		}
	}

	Standalone<VerUpdateRef>& addVersionToMutationLog(Version v) {
		// return existing version...
		auto m = mutationLog.find(v);
		if (m != mutationLog.end())
			return m->second;

		// ...or create a new one
		auto& u = mutationLog[v];
		u.version = v;
		if (lastArena.getSize() >= 65536)
			lastArena = Arena(4096);
		u.arena() = lastArena;
		counters.bytesInput += VERSION_OVERHEAD;
		return u;
	}

	MutationRef addMutationToMutationLog(Standalone<VerUpdateRef>& mLV, MutationRef const& m) {
		byteSampleApplyMutation(m, mLV.version);
		counters.bytesInput += mvccStorageBytes(m);
		return mLV.push_back_deep(mLV.arena(), m);
	}

	void setTssPair(UID pairId) {
		tssPairID = Optional<UID>(pairId);

		// Set up tss fault injection here, only if we are in simulated mode and with fault injection.
		// With fault injection enabled, the tss will start acting normal for a bit, then after the specified delay
		// start behaving incorrectly.
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation &&
		    g_simulator->tssMode >= ISimulator::TSSMode::EnabledAddDelay) {
			tssFaultInjectTime = now() + deterministicRandom()->randomInt(60, 300);
			TraceEvent(SevWarnAlways, "TSSInjectFaultEnabled", thisServerID)
			    .detail("Mode", g_simulator->tssMode)
			    .detail("At", tssFaultInjectTime.get());
		}
	}

	// If a TSS is "in quarantine", it means it has incorrect data. It is effectively in a "zombie" state where it
	// rejects all read requests and ignores all non-private mutations and data movements, but otherwise is still part
	// of the cluster. The purpose of this state is to "freeze" the TSS state after a mismatch so a human operator can
	// investigate, but preventing a new storage process from replacing the TSS on the worker. It will still get removed
	// from the cluster if it falls behind on the mutation stream, or if its tss pair gets removed and its tag is no
	// longer valid.
	bool isTSSInQuarantine() const { return tssPairID.present() && tssInQuarantine; }

	void startTssQuarantine() {
		if (!tssInQuarantine) {
			// persist quarantine so it's still quarantined if rebooted
			storage.makeTssQuarantineDurable();
		}
		tssInQuarantine = true;
	}

	StorageServerDisk storage;

	KeyRangeMap<Reference<ShardInfo>> shards;
	KeyRangeMap<SSBulkLoadMetadata> ssBulkLoadMetadataMap; // store the latest bulkload task on ranges
	uint64_t shardChangeCounter; // max( shards->changecounter )

	KeyRangeMap<bool> cachedRangeMap; // indicates if a key-range is being cached

	// newestAvailableVersion[k]
	//   == invalidVersion -> k is unavailable at all versions
	//   <= storageVersion -> k is unavailable at all versions (but might be read anyway from storage if we are in the
	//   process of committing makeShardDurable)
	//   == v              -> k is readable (from storage+versionedData) @ [storageVersion,v], and not being updated
	//   when version increases
	//   == latestVersion  -> k is readable (from storage+versionedData) @ [storageVersion,version.get()], and thus
	//   stays available when version increases
	CoalescedKeyRangeMap<Version> newestAvailableVersion;

	CoalescedKeyRangeMap<Version> newestDirtyVersion; // Similar to newestAvailableVersion, but includes (only) keys
	                                                  // that were only partly available (due to cancelled fetchKeys)

	// The following are in rough order from newest to oldest
	Version lastTLogVersion, lastVersionWithData, restoredVersion, prevVersion;
	NotifiedVersion version;
	NotifiedVersion desiredOldestVersion; // We can increase oldestVersion (and then durableVersion) to this version
	                                      // when the disk permits
	NotifiedVersion oldestVersion; // See also storageVersion()
	NotifiedVersion durableVersion; // At least this version will be readable from storage after a power failure
	// In the event of the disk corruption, sqlite and redwood will either not recover, recover to durableVersion
	// but be unable to read some data, or they could lose the last commit. If we lose the last commit, the storage
	// might not be able to peek from the tlog (depending on when it sent the last pop). So this version just keeps
	// track of the version we committed to the storage engine before we did commit durableVersion.
	Version storageMinRecoverVersion = 0;
	Version rebootAfterDurableVersion;
	int8_t primaryLocality;
	NotifiedVersion knownCommittedVersion;

	Deque<std::pair<Version, Version>> recoveryVersionSkips;
	int64_t versionLag; // An estimate for how many versions it takes for the data to move from the logs to this storage
	                    // server

	Optional<UID> sourceTLogID; // the tLog from which the latest batch of versions were fetched

	ProtocolVersion logProtocol;

	Reference<ILogSystem> logSystem;
	Reference<ILogSystem::IPeekCursor> logCursor;

	// The version the cluster starts on. This value is not persisted and may
	// not be valid after a recovery.
	Version initialClusterVersion = 1;
	UID thisServerID;
	Optional<UID> tssPairID; // if this server is a tss, this is the id of its (ss) pair
	Optional<UID> ssPairID; // if this server is an ss, this is the id of its (tss) pair
	Optional<double> tssFaultInjectTime;
	bool tssInQuarantine;

	Key sk;
	Reference<AsyncVar<ServerDBInfo> const> db;
	Database cx;
	ActorCollection actors;

	CoalescedKeyRangeMap<bool, int64_t, KeyBytesMetric<int64_t>> byteSampleClears;
	AsyncVar<bool> byteSampleClearsTooLarge;
	Future<Void> byteSampleRecovery;
	Future<Void> durableInProgress;

	AsyncMap<Key, bool> watches;
	int64_t watchBytes;
	int64_t numWatches;
	AsyncVar<bool> noRecentUpdates;
	double lastUpdate;

	std::string folder;
	std::string checkpointFolder;
	std::string fetchedCheckpointFolder;
	std::string bulkDumpFolder;
	std::string bulkLoadFolder;

	// defined only during splitMutations()/addMutation()
	UpdateEagerReadInfo* updateEagerReads;

	FlowLock durableVersionLock;
	FlowLock fetchKeysParallelismLock;
	int64_t fetchKeysBytesBudget;
	AsyncVar<bool> fetchKeysBudgetUsed;
	int64_t fetchKeysTotalCommitBytes;
	std::vector<Promise<FetchInjectionInfo*>> readyFetchKeys;

	ThroughputLimiter fetchKeysLimiter;

	FlowLock serveFetchCheckpointParallelismLock;

	std::unordered_map<UID, std::shared_ptr<MoveInShard>> moveInShards;

	Reference<PriorityMultiLock> ssLock;
	std::vector<int> readPriorityRanks;

	Future<PriorityMultiLock::Lock> getReadLock(const Optional<ReadOptions>& options) {
		int readType = (int)(options.present() ? options.get().type : ReadType::NORMAL);
		readType = std::clamp<int>(readType, 0, readPriorityRanks.size() - 1);
		return ssLock->lock(readPriorityRanks[readType]);
	}

	FlowLock serveAuditStorageParallelismLock;

	FlowLock serveBulkDumpParallelismLock;

	int64_t instanceID;

	Promise<Void> otherError;
	Promise<Void> coreStarted;
	bool shuttingDown;

	Promise<Void> registerInterfaceAcceptingRequests;
	Future<Void> interfaceRegistered;

	bool behind;
	bool versionBehind;

	bool debug_inApplyUpdate;
	double debug_lastValidateTime;

	int64_t lastBytesInputEBrake;
	Version lastDurableVersionEBrake;

	int maxQueryQueue;
	int getAndResetMaxQueryQueueSize() {
		int val = maxQueryQueue;
		maxQueryQueue = 0;
		return val;
	}

	TransactionTagCounter transactionTagCounter;
	BusiestWriteTagContext busiestWriteTagContext;

	Optional<LatencyBandConfig> latencyBandConfig;

	Optional<EncryptionAtRestMode> encryptionMode;
	Reference<GetEncryptCipherKeysMonitor> getEncryptCipherKeysMonitor;

	struct Counters : CommonStorageCounters {

		Counter allQueries, systemKeyQueries, getKeyQueries, getValueQueries, getRangeQueries, getRangeSystemKeyQueries,
		    getRangeStreamQueries, lowPriorityQueries, rowsQueried, watchQueries, emptyQueries;

		// counters related to getMappedRange queries
		Counter getMappedRangeBytesQueried, finishedGetMappedRangeSecondaryQueries, getMappedRangeQueries,
		    finishedGetMappedRangeQueries;

		// Bytes pulled from TLogs, it counts the size of the key value pairs, e.g., key-value pair ("a", "b") is
		// counted as 2 Bytes.
		Counter logicalBytesInput;
		// Bytes pulled from TLogs for moving-in shards, it counts the mutations sent to the moving-in shard during
		// Fetching and Waiting phases.
		Counter logicalBytesMoveInOverhead;
		// Bytes committed to the underlying storage engine by SS, it counts the size of key value pairs.
		Counter kvCommitLogicalBytes;
		// Count of all clearRange operations to the storage engine.
		Counter kvClearRanges;
		// Count of all clearRange operations on a singlekeyRange(key delete) to the storage engine.
		Counter kvClearSingleKey;
		// ClearRange operations issued by FDB, instead of from users, e.g., ClearRange operations to remove a shard
		// from a storage server, as in removeDataRange().
		Counter kvSystemClearRanges;
		// Bytes of the mutations that have been removed from memory because they durable. The counting is same as
		// bytesInput, instead of the actual bytes taken in the storages, so that (bytesInput - bytesDurable) can
		// reflect the current memory footprint of MVCC.
		Counter bytesDurable;
		// Count of all fetchKey clearRange operations to the storage engine.
		Counter kvClearRangesInFetchKeys;

		// Bytes fetched by fetchChangeFeed for data movements.
		Counter feedBytesFetched;

		Counter sampledBytesCleared;
		Counter atomicMutations, changeFeedMutations, changeFeedMutationsDurable;
		Counter updateBatches, updateVersions;
		Counter loops;
		Counter fetchWaitingMS, fetchWaitingCount, fetchExecutingMS, fetchExecutingCount;
		Counter readsRejected;
		Counter wrongShardServer;
		Counter fetchedVersions;
		Counter fetchesFromLogs;
		// The following counters measure how many of lookups in the getMappedRangeQueries are effective. "Miss"
		// means fallback if fallback is enabled, otherwise means failure (so that another layer could implement
		// fallback).
		Counter quickGetValueHit, quickGetValueMiss, quickGetKeyValuesHit, quickGetKeyValuesMiss;

		// The number of logical bytes returned from storage engine, in response to readRange operations.
		Counter kvScanBytes;
		// The number of logical bytes returned from storage engine, in response to readValue operations.
		Counter kvGetBytes;
		// The number of keys read from storage engine by eagerReads.
		Counter eagerReadsKeys;
		// The count of readValue operation to the storage engine.
		Counter kvGets;
		// The count of readValue operation to the storage engine.
		Counter kvScans;
		// The count of commit operation to the storage engine.
		Counter kvCommits;
		// The count of change feed reads that hit disk
		Counter changeFeedDiskReads;
		// The count of ChangeServerKeys actions.
		Counter changeServerKeysAssigned;
		Counter changeServerKeysUnassigned;

		// The count of 'set' inserted to pTree. The actual ptree.insert() number could be higher, because of the range
		// clear split, see metric pTreeClearSplits.
		Counter pTreeSets;
		// The count of clear range inserted to pTree
		Counter pTreeClears;
		// If set is within a range of clear, the clear is split. It's tracking the number of splits, the split could be
		// expensive.
		Counter pTreeClearSplits;

		ReadLatencySamples readLatencySamples;
		std::unique_ptr<LatencySample> updateLatencySample;
		std::unique_ptr<LatencySample> updateEncryptionLatencySample;
		LatencyBands readLatencyBands;
		std::unique_ptr<LatencySample> ingestDurationLatencySample;

		explicit Counters(StorageServer* self)
		  : CommonStorageCounters("StorageServer", self->thisServerID.toString(), &self->metrics),
		    allQueries("QueryQueue", cc), systemKeyQueries("SystemKeyQueries", cc), getKeyQueries("GetKeyQueries", cc),
		    getValueQueries("GetValueQueries", cc), getRangeQueries("GetRangeQueries", cc),
		    getRangeSystemKeyQueries("GetRangeSystemKeyQueries", cc),
		    getMappedRangeQueries("GetMappedRangeQueries", cc), getRangeStreamQueries("GetRangeStreamQueries", cc),
		    lowPriorityQueries("LowPriorityQueries", cc), rowsQueried("RowsQueried", cc),
		    watchQueries("WatchQueries", cc), emptyQueries("EmptyQueries", cc),
		    logicalBytesInput("LogicalBytesInput", cc), logicalBytesMoveInOverhead("LogicalBytesMoveInOverhead", cc),
		    kvCommitLogicalBytes("KVCommitLogicalBytes", cc), kvClearRanges("KVClearRanges", cc),
		    kvClearSingleKey("KVClearSingleKey", cc), kvSystemClearRanges("KVSystemClearRanges", cc),
		    bytesDurable("BytesDurable", cc), feedBytesFetched("FeedBytesFetched", cc),
		    sampledBytesCleared("SampledBytesCleared", cc), atomicMutations("AtomicMutations", cc),
		    changeFeedMutations("ChangeFeedMutations", cc),
		    changeFeedMutationsDurable("ChangeFeedMutationsDurable", cc), updateBatches("UpdateBatches", cc),
		    updateVersions("UpdateVersions", cc), loops("Loops", cc), fetchWaitingMS("FetchWaitingMS", cc),
		    fetchWaitingCount("FetchWaitingCount", cc), fetchExecutingMS("FetchExecutingMS", cc),
		    fetchExecutingCount("FetchExecutingCount", cc), readsRejected("ReadsRejected", cc),
		    wrongShardServer("WrongShardServer", cc), fetchedVersions("FetchedVersions", cc),
		    fetchesFromLogs("FetchesFromLogs", cc), quickGetValueHit("QuickGetValueHit", cc),
		    quickGetValueMiss("QuickGetValueMiss", cc), quickGetKeyValuesHit("QuickGetKeyValuesHit", cc),
		    quickGetKeyValuesMiss("QuickGetKeyValuesMiss", cc), kvScanBytes("KVScanBytes", cc),
		    kvGetBytes("KVGetBytes", cc), eagerReadsKeys("EagerReadsKeys", cc), kvGets("KVGets", cc),
		    kvScans("KVScans", cc), kvCommits("KVCommits", cc), changeFeedDiskReads("ChangeFeedDiskReads", cc),
		    getMappedRangeBytesQueried("GetMappedRangeBytesQueried", cc),
		    finishedGetMappedRangeQueries("FinishedGetMappedRangeQueries", cc),
		    finishedGetMappedRangeSecondaryQueries("FinishedGetMappedRangeSecondaryQueries", cc),
		    pTreeSets("PTreeSets", cc), pTreeClears("PTreeClears", cc), pTreeClearSplits("PTreeClearSplits", cc),
		    changeServerKeysAssigned("ChangeServerKeysAssigned", cc),
		    changeServerKeysUnassigned("ChangeServerKeysUnassigned", cc),
		    kvClearRangesInFetchKeys("KvClearRangesInFetchKeys", cc), readLatencySamples(self->thisServerID),
		    updateLatencySample(std::make_unique<LatencySample>("UpdateLatencyMetrics",
		                                                        self->thisServerID,
		                                                        SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                        SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    updateEncryptionLatencySample(
		        std::make_unique<LatencySample>("UpdateEncryptionLatencyMetrics",
		                                        self->thisServerID,
		                                        SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                        SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readLatencyBands("ReadLatencyBands", self->thisServerID, SERVER_KNOBS->STORAGE_LOGGING_DELAY),
		    ingestDurationLatencySample(std::make_unique<LatencySample>("IngestDurationMetrics",
		                                                                self->thisServerID,
		                                                                SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                                SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)) {
			specialCounter(cc, "LastTLogVersion", [self]() { return self->lastTLogVersion; });
			specialCounter(cc, "Version", [self]() { return self->version.get(); });
			specialCounter(cc, "StorageVersion", [self]() { return self->storageVersion(); });
			specialCounter(cc, "DurableVersion", [self]() { return self->durableVersion.get(); });
			specialCounter(cc, "DesiredOldestVersion", [self]() { return self->desiredOldestVersion.get(); });
			specialCounter(cc, "VersionLag", [self]() { return self->versionLag; });
			specialCounter(cc, "LocalRate", [self] { return int64_t(self->currentRate() * 100); });

			specialCounter(
			    cc, "FetchKeysFetchActive", [self]() { return self->fetchKeysParallelismLock.activePermits(); });
			specialCounter(cc, "FetchKeysWaiting", [self]() { return self->fetchKeysParallelismLock.waiters(); });
			specialCounter(cc, "ServeFetchCheckpointActive", [self]() {
				return self->serveFetchCheckpointParallelismLock.activePermits();
			});
			specialCounter(cc, "ServeFetchCheckpointWaiting", [self]() {
				return self->serveFetchCheckpointParallelismLock.waiters();
			});
			specialCounter(cc, "ServeValidateStorageActive", [self]() {
				return self->serveAuditStorageParallelismLock.activePermits();
			});
			specialCounter(cc, "ServeValidateStorageWaiting", [self]() {
				return self->serveAuditStorageParallelismLock.waiters();
			});
			specialCounter(
			    cc, "ServerBulkDumpActive", [self]() { return self->serveBulkDumpParallelismLock.activePermits(); });
			specialCounter(
			    cc, "ServerBulkDumpWaiting", [self]() { return self->serveBulkDumpParallelismLock.waiters(); });
			specialCounter(cc, "QueryQueueMax", [self]() { return self->getAndResetMaxQueryQueueSize(); });
			specialCounter(cc, "ActiveWatches", [self]() { return self->numWatches; });
			specialCounter(cc, "WatchBytes", [self]() { return self->watchBytes; });
			specialCounter(cc, "KvstoreSizeTotal", [self]() { return std::get<0>(self->storage.getSize()); });
			specialCounter(cc, "KvstoreNodeTotal", [self]() { return std::get<1>(self->storage.getSize()); });
			specialCounter(cc, "KvstoreInlineKey", [self]() { return std::get<2>(self->storage.getSize()); });
		}
	} counters;

	// Bytes read from storage engine when a storage server starts.
	int64_t bytesRestored = 0;

	Reference<EventCacheHolder> storageServerSourceTLogIDEventHolder;

	std::shared_ptr<AccumulativeChecksumValidator> acsValidator = nullptr;

	std::shared_ptr<SSBulkLoadMetrics> bulkLoadMetrics = nullptr;

	StorageServer(IKeyValueStore* storage,
	              Reference<AsyncVar<ServerDBInfo> const> const& db,
	              StorageServerInterface const& ssi,
	              Reference<GetEncryptCipherKeysMonitor> encryptionMonitor)
	  : shardAware(false), locality(ssi.locality),
	    tlogCursorReadsLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                            TLOG_CURSOR_READS_LATENCY_HISTOGRAM,
	                                                            Histogram::Unit::milliseconds)),
	    ssVersionLockLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                          SS_VERSION_LOCK_LATENCY_HISTOGRAM,
	                                                          Histogram::Unit::milliseconds)),
	    eagerReadsLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                       EAGER_READS_LATENCY_HISTOGRAM,
	                                                       Histogram::Unit::milliseconds)),
	    fetchKeysPTreeUpdatesLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                  FETCH_KEYS_PTREE_UPDATES_LATENCY_HISTOGRAM,
	                                                                  Histogram::Unit::milliseconds)),
	    tLogMsgsPTreeUpdatesLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                 TLOG_MSGS_PTREE_UPDATES_LATENCY_HISTOGRAM,
	                                                                 Histogram::Unit::milliseconds)),
	    storageUpdatesDurableLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                  STORAGE_UPDATES_DURABLE_LATENCY_HISTOGRAM,
	                                                                  Histogram::Unit::milliseconds)),
	    storageCommitLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                          STORAGE_COMMIT_LATENCY_HISTOGRAM,
	                                                          Histogram::Unit::milliseconds)),
	    ssDurableVersionUpdateLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                   SS_DURABLE_VERSION_UPDATE_LATENCY_HISTOGRAM,
	                                                                   Histogram::Unit::milliseconds)),
	    readRangeBytesReturnedHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                            SS_READ_RANGE_BYTES_RETURNED_HISTOGRAM,
	                                                            Histogram::Unit::bytes)),
	    readRangeBytesLimitHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                         SS_READ_RANGE_BYTES_LIMIT_HISTOGRAM,
	                                                         Histogram::Unit::bytes)),
	    readRangeKVPairsReturnedHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                              SS_READ_RANGE_KV_PAIRS_RETURNED_HISTOGRAM,
	                                                              Histogram::Unit::countLinear)),
	    tag(invalidTag), poppedAllAfter(std::numeric_limits<Version>::max()), cpuUsage(0.0), diskUsage(0.0),
	    storage(this, storage), shardChangeCounter(0), lastTLogVersion(0), lastVersionWithData(0), restoredVersion(0),
	    prevVersion(0), rebootAfterDurableVersion(std::numeric_limits<Version>::max()),
	    primaryLocality(tagLocalityInvalid), knownCommittedVersion(0), versionLag(0), logProtocol(0),
	    thisServerID(ssi.id()), tssInQuarantine(false), db(db), actors(false),
	    trackShardAssignmentMinVersion(invalidVersion), byteSampleClears(false, "\xff\xff\xff"_sr),
	    durableInProgress(Void()), watchBytes(0), numWatches(0), noRecentUpdates(false), lastUpdate(now()),
	    updateEagerReads(nullptr), fetchKeysParallelismLock(SERVER_KNOBS->FETCH_KEYS_PARALLELISM),
	    fetchKeysBytesBudget(SERVER_KNOBS->STORAGE_FETCH_BYTES), fetchKeysBudgetUsed(false),
	    fetchKeysTotalCommitBytes(0), fetchKeysLimiter(SERVER_KNOBS->STORAGE_FETCH_KEYS_RATE_LIMIT),
	    serveFetchCheckpointParallelismLock(SERVER_KNOBS->SERVE_FETCH_CHECKPOINT_PARALLELISM),
	    ssLock(makeReference<PriorityMultiLock>(SERVER_KNOBS->STORAGE_SERVER_READ_CONCURRENCY,
	                                            SERVER_KNOBS->STORAGESERVER_READ_PRIORITIES)),
	    serveAuditStorageParallelismLock(SERVER_KNOBS->SERVE_AUDIT_STORAGE_PARALLELISM),
	    serveBulkDumpParallelismLock(SERVER_KNOBS->SS_SERVE_BULKDUMP_PARALLELISM),
	    instanceID(deterministicRandom()->randomUniqueID().first()), shuttingDown(false), behind(false),
	    versionBehind(false), debug_inApplyUpdate(false), debug_lastValidateTime(0), lastBytesInputEBrake(0),
	    lastDurableVersionEBrake(0), maxQueryQueue(0),
	    transactionTagCounter(ssi.id(),
	                          /*maxTagsTracked=*/SERVER_KNOBS->SS_THROTTLE_TAGS_TRACKED,
	                          /*minRateTracked=*/SERVER_KNOBS->MIN_TAG_READ_PAGES_RATE *
	                              CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE),
	    busiestWriteTagContext(ssi.id()), getEncryptCipherKeysMonitor(encryptionMonitor), counters(this),
	    storageServerSourceTLogIDEventHolder(
	        makeReference<EventCacheHolder>(ssi.id().toString() + "/StorageServerSourceTLogID")),
	    acsValidator(CLIENT_KNOBS->ENABLE_MUTATION_CHECKSUM && CLIENT_KNOBS->ENABLE_ACCUMULATIVE_CHECKSUM &&
	                         !SERVER_KNOBS->ENABLE_VERSION_VECTOR && !SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST
	                     ? std::make_shared<AccumulativeChecksumValidator>()
	                     : nullptr),
	    bulkLoadMetrics(std::make_shared<SSBulkLoadMetrics>()) {
		readPriorityRanks = parseStringToVector<int>(SERVER_KNOBS->STORAGESERVER_READTYPE_PRIORITY_MAP, ',');
		ASSERT(readPriorityRanks.size() > (int)ReadType::MAX);
		version.initMetric("StorageServer.Version"_sr, counters.cc.getId());
		oldestVersion.initMetric("StorageServer.OldestVersion"_sr, counters.cc.getId());
		durableVersion.initMetric("StorageServer.DurableVersion"_sr, counters.cc.getId());
		desiredOldestVersion.initMetric("StorageServer.DesiredOldestVersion"_sr, counters.cc.getId());

		newestAvailableVersion.insert(allKeys, invalidVersion);
		newestDirtyVersion.insert(allKeys, invalidVersion);
		if (storage->shardAware()) {
			addShard(ShardInfo::newShard(this, StorageServerShard::notAssigned(allKeys)));
		} else {
			addShard(ShardInfo::newNotAssigned(allKeys));
		}

		cx = openDBOnServer(db, TaskPriority::DefaultEndpoint, LockAware::True);

		this->storage.kvCommitLogicalBytes = &counters.kvCommitLogicalBytes;
		this->storage.kvClearRanges = &counters.kvClearRanges;
		this->storage.kvClearSingleKey = &counters.kvClearSingleKey;
		this->storage.kvGets = &counters.kvGets;
		this->storage.kvScans = &counters.kvScans;
		this->storage.kvCommits = &counters.kvCommits;
	}

	//~StorageServer() { fclose(log); }

	// Puts the given shard into shards.  The caller is responsible for adding shards
	//   for all ranges in shards.getAffectedRangesAfterInsertion(newShard->keys)), because these
	//   shards are invalidated by the call.
	void addShard(ShardInfo* newShard) {
		ASSERT(!newShard->range().empty());
		newShard->setChangeCounter(++shardChangeCounter);
		// TraceEvent("AddShard", this->thisServerID).detail("KeyBegin", newShard->keys.begin).detail("KeyEnd", newShard->keys.end).detail("State",newShard->isReadable() ? "Readable" : newShard->notAssigned() ? "NotAssigned" : "Adding").detail("Version", this->version.get());
		/*auto affected = shards.getAffectedRangesAfterInsertion( newShard->keys, Reference<ShardInfo>() );
		for(auto i = affected.begin(); i != affected.end(); ++i)
		    shards.insert( *i, Reference<ShardInfo>() );*/

		if (shardAware && newShard->notAssigned()) {
			auto sh = shards.intersectingRanges(newShard->range());
			for (auto it = sh.begin(); it != sh.end(); ++it) {
				if (it->value().isValid() && !it->value()->notAssigned()) {
					TraceEvent(SevVerbose, "StorageServerAddShardClear")
					    .detail("NewShardRange", newShard->range())
					    .detail("Range", it->value()->range())
					    .detail("ShardID", format("%016llx", it->value()->getShardId()))
					    .detail("NewShardID", format("%016llx", newShard->getDesiredShardId()))
					    .detail("NewShardActualID", format("%016llx", newShard->getShardId()));
				}
			}
		}

		Reference<ShardInfo> rShard(newShard);
		shards.insert(newShard->range(), rShard);
	}
	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation,
	                 KeyRangeRef const& shard,
	                 UpdateEagerReadInfo* eagerReads);
	void setInitialVersion(Version ver) {
		version = ver;
		desiredOldestVersion = ver;
		oldestVersion = ver;
		durableVersion = ver;
		storageMinRecoverVersion = ver;
		lastVersionWithData = ver;
		restoredVersion = ver;

		mutableData().createNewVersion(ver);
		mutableData().forgetVersionsBefore(ver);
	}

	bool isTss() const { return tssPairID.present(); }

	bool isSSWithTSSPair() const { return ssPairID.present(); }

	void setSSWithTssPair(UID idOfTSS) { ssPairID = Optional<UID>(idOfTSS); }

	void clearSSWithTssPair() { ssPairID = Optional<UID>(); }

	// This is the maximum version that might be read from storage (the minimum version is durableVersion)
	Version storageVersion() const { return oldestVersion.get(); }

	bool isReadable(KeyRangeRef const& keys) const override {
		auto sh = shards.intersectingRanges(keys);
		for (auto i = sh.begin(); i != sh.end(); ++i)
			if (!i->value()->isReadable())
				return false;
		return true;
	}

	void checkChangeCounter(uint64_t oldShardChangeCounter, KeyRef const& key) {
		if (oldShardChangeCounter != shardChangeCounter && shards[key]->getChangeCounter() > oldShardChangeCounter) {
			CODE_PROBE(true, "shard change during getValueQ");
			throw wrong_shard_server();
		}
	}

	void checkChangeCounter(uint64_t oldShardChangeCounter, KeyRangeRef const& keys) {
		if (oldShardChangeCounter != shardChangeCounter) {
			auto sh = shards.intersectingRanges(keys);
			for (auto i = sh.begin(); i != sh.end(); ++i)
				if (i->value()->getChangeCounter() > oldShardChangeCounter) {
					CODE_PROBE(true, "shard change during range operation");
					throw wrong_shard_server();
				}
		}
	}

	Counter::Value queueSize() const { return counters.bytesInput.getValue() - counters.bytesDurable.getValue(); }

	// penalty used by loadBalance() to balance requests among SSes. We prefer SS with less write queue size.
	double getPenalty() const override {
		return std::max(std::max(1.0,
		                         (queueSize() - (SERVER_KNOBS->TARGET_BYTES_PER_STORAGE_SERVER -
		                                         2.0 * SERVER_KNOBS->SPRING_BYTES_STORAGE_SERVER)) /
		                             SERVER_KNOBS->SPRING_BYTES_STORAGE_SERVER),
		                (currentRate() < 1e-6 ? 1e6 : 1.0 / currentRate()));
	}

	// Normally the storage server prefers to serve read requests over making mutations
	// durable to disk. However, when the storage server falls to far behind on
	// making mutations durable, this function will change the priority to prefer writes.
	Future<Void> getQueryDelay() {
		if ((version.get() - durableVersion.get() > SERVER_KNOBS->LOW_PRIORITY_DURABILITY_LAG) ||
		    (queueSize() > SERVER_KNOBS->LOW_PRIORITY_STORAGE_QUEUE_BYTES)) {
			++counters.lowPriorityQueries;
			return delay(0, TaskPriority::LowPriorityRead);
		}
		return delay(0, TaskPriority::DefaultEndpoint);
	}

	template <class Reply>
	using isLoadBalancedReply = std::is_base_of<LoadBalancedReply, Reply>;

	template <class Reply>
	typename std::enable_if<isLoadBalancedReply<Reply>::value, void>::type
	sendErrorWithPenalty(const ReplyPromise<Reply>& promise, const Error& err, double penalty) {
		if (err.code() == error_code_wrong_shard_server) {
			++counters.wrongShardServer;
		}
		Reply reply;
		reply.error = err;
		reply.penalty = penalty;
		promise.send(reply);
	}

	template <class Reply>
	typename std::enable_if<!isLoadBalancedReply<Reply>::value, void>::type
	sendErrorWithPenalty(const ReplyPromise<Reply>& promise, const Error& err, double) {
		if (err.code() == error_code_wrong_shard_server) {
			++counters.wrongShardServer;
		}
		promise.sendError(err);
	}

	template <class Request>
	bool shouldRead(const Request& request) {
		auto rate = currentRate();
		if (isTSSInQuarantine() || (rate < SERVER_KNOBS->STORAGE_DURABILITY_LAG_REJECT_THRESHOLD &&
		                            deterministicRandom()->random01() >
		                                std::max(SERVER_KNOBS->STORAGE_DURABILITY_LAG_MIN_RATE,
		                                         rate / SERVER_KNOBS->STORAGE_DURABILITY_LAG_REJECT_THRESHOLD))) {
			sendErrorWithPenalty(request.reply, server_overloaded(), getPenalty());
			++counters.readsRejected;
			return false;
		}
		return true;
	}

	template <class Request, class HandleFunction>
	Future<Void> readGuard(const Request& request, const HandleFunction& fun) {
		bool read = shouldRead(request);
		if (!read) {
			return Void();
		}
		return fun(this, request);
	}

	void getSplitPoints(SplitRangeRequest const& req) override {
		try {
			metrics.getSplitPoints(req, {});
		} catch (Error& e) {
			req.reply.sendError(e);
		}
	}

	void maybeInjectTargetedRestart(Version v) {
		// inject an SS restart at most once per test
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation &&
		    now() > g_simulator->injectTargetedSSRestartTime &&
		    rebootAfterDurableVersion == std::numeric_limits<Version>::max()) {
			CODE_PROBE(true, "Injecting SS targeted restart");
			TraceEvent("SimSSInjectTargetedRestart", thisServerID).detail("Version", v);
			rebootAfterDurableVersion = v;
			g_simulator->injectTargetedSSRestartTime = std::numeric_limits<double>::max();
		}
	}

	bool maybeInjectDelay() {
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation && now() > g_simulator->injectSSDelayTime) {
			CODE_PROBE(true, "Injecting SS targeted delay");
			TraceEvent("SimSSInjectDelay", thisServerID).log();
			g_simulator->injectSSDelayTime = std::numeric_limits<double>::max();
			return true;
		}
		return false;
	}

	Future<Void> waitMetricsForReal(const WaitMetricsRequest& req) override;

	void addActor(Future<Void> future) override { actors.add(future); }

	void getStorageMetrics(const GetStorageMetricsRequest& req) override {
		StorageBytes sb = storage.getStorageBytes();
		metrics.getStorageMetrics(req,
		                          sb,
		                          counters.bytesInput.getRate(),
		                          versionLag,
		                          lastUpdate,
		                          counters.bytesDurable.getValue(),
		                          counters.bytesInput.getValue(),
		                          bulkLoadMetrics->getOngoingTasks());
	}

	void getSplitMetrics(const SplitMetricsRequest& req) override { this->metrics.splitMetrics(req); }

	void getHotRangeMetrics(const ReadHotSubRangeRequest& req) override { this->metrics.getReadHotRanges(req); }

	int64_t getHotShardsMetrics(const KeyRange& range) override { return this->metrics.getHotShards(range); }

	// Used for recording shard assignment history for auditStorage
	std::vector<std::pair<Version, KeyRange>> shardAssignmentHistory;
	Version trackShardAssignmentMinVersion; // == invalidVersion means tracking stopped

	std::string printShardAssignmentHistory() {
		std::string toPrint = "";
		for (const auto& [version, range] : shardAssignmentHistory) {
			toPrint = toPrint + std::to_string(version) + " ";
		}
		return toPrint;
	}

	void startTrackShardAssignment(Version startVersion) {
		ASSERT(startVersion != invalidVersion);
		ASSERT(trackShardAssignmentMinVersion == invalidVersion);
		trackShardAssignmentMinVersion = startVersion;
		return;
	}

	void stopTrackShardAssignment() { trackShardAssignmentMinVersion = invalidVersion; }

	std::vector<std::pair<Version, KeyRangeRef>> getShardAssignmentHistory(Version early, Version later) {
		std::vector<std::pair<Version, KeyRangeRef>> res;
		for (const auto& shardAssignment : shardAssignmentHistory) {
			if (shardAssignment.first >= early && shardAssignment.first <= later) {
				TraceEvent(SevVerbose, "ShardAssignmentHistoryGetOne", thisServerID)
				    .detail("Keys", shardAssignment.second)
				    .detail("Version", shardAssignment.first);
				res.push_back(shardAssignment);
			} else {
				TraceEvent(SevVerbose, "ShardAssignmentHistoryGetSkip", thisServerID)
				    .detail("Keys", shardAssignment.second)
				    .detail("Version", shardAssignment.first)
				    .detail("EarlyVersion", early)
				    .detail("LaterVersion", later);
			}
		}
		TraceEvent(SevVerbose, "ShardAssignmentHistoryGetDone", thisServerID)
		    .detail("EarlyVersion", early)
		    .detail("LaterVersion", later)
		    .detail("HistoryTotalSize", shardAssignmentHistory.size())
		    .detail("HistoryTotal", printShardAssignmentHistory());
		return res;
	}
};

const StringRef StorageServer::CurrentRunningFetchKeys::emptyString = ""_sr;
const KeyRangeRef StorageServer::CurrentRunningFetchKeys::emptyKeyRange =
    KeyRangeRef(StorageServer::CurrentRunningFetchKeys::emptyString,
                StorageServer::CurrentRunningFetchKeys::emptyString);

// If and only if key:=value is in (storage+versionedData),    // NOT ACTUALLY: and key < allKeys.end,
//   and H(key) < |key+value|/bytesPerSample,
//     let sampledSize = max(|key+value|,bytesPerSample)
//     persistByteSampleKeys.begin()+key := sampledSize is in storage
//     (key,sampledSize) is in byteSample

// So P(key is sampled) * sampledSize == |key+value|

void StorageServer::byteSampleApplyMutation(MutationRef const& m, Version ver) {
	if (m.type == MutationRef::ClearRange)
		byteSampleApplyClear(KeyRangeRef(m.param1, m.param2), ver);
	else if (m.type == MutationRef::SetValue)
		byteSampleApplySet(KeyValueRef(m.param1, m.param2), ver);
	else
		ASSERT(false); // Mutation of unknown type modifying byte sample
}

// watchMap Operations
Reference<ServerWatchMetadata> StorageServer::getWatchMetadata(KeyRef key) const {
	const WatchMapKey mapKey(key);
	const auto it = watchMap.find(mapKey);
	if (it == watchMap.end())
		return Reference<ServerWatchMetadata>();
	return it->second;
}

KeyRef StorageServer::setWatchMetadata(Reference<ServerWatchMetadata> metadata) {
	KeyRef keyRef = metadata->key.contents();
	watchMap[keyRef] = metadata;
	return keyRef;
}

void StorageServer::deleteWatchMetadata(KeyRef key) {
	const WatchMapKey mapKey(key);
	watchMap.erase(mapKey);
}

void StorageServer::clearWatchMetadata() {
	watchMap.clear();
}

#ifndef __INTEL_COMPILER
#pragma endregion
#endif

/////////////////////////////////// Validation ///////////////////////////////////////
#ifndef __INTEL_COMPILER
#pragma region Validation
#endif
bool validateRange(StorageServer::VersionedData::ViewAtVersion const& view,
                   KeyRangeRef range,
                   Version version,
                   UID id,
                   Version minInsertVersion) {
	// * Nonoverlapping: No clear overlaps a set or another clear, or adjoins another clear.
	// * Old mutations are erased: All items in versionedData.atLatest() have insertVersion() > durableVersion()

	//TraceEvent("ValidateRange", id).detail("KeyBegin", range.begin).detail("KeyEnd", range.end).detail("Version", version);
	KeyRef k;
	bool ok = true;
	bool kIsClear = false;
	auto i = view.lower_bound(range.begin);
	if (i != view.begin())
		--i;
	for (; i != view.end() && i.key() < range.end; ++i) {
		ASSERT(i.insertVersion() > minInsertVersion);
		if (kIsClear && i->isClearTo() ? i.key() <= k : i.key() < k) {
			TraceEvent(SevError, "InvalidRange", id)
			    .detail("Key1", k)
			    .detail("Key2", i.key())
			    .detail("Version", version);
			ok = false;
		}
		// ASSERT( i.key() >= k );
		kIsClear = i->isClearTo();
		k = kIsClear ? i->getEndKey() : i.key();
	}
	return ok;
}

void validate(StorageServer* data, bool force = false) {
	try {
		if (!data->shuttingDown && (force || (EXPENSIVE_VALIDATION))) {
			data->newestAvailableVersion.validateCoalesced();
			data->newestDirtyVersion.validateCoalesced();

			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				TraceEvent(SevVerbose, "ValidateShard", data->thisServerID)
				    .detail("Range", s->range())
				    .detail("ShardID", format("%016llx", s->value()->getShardId()))
				    .detail("DesiredShardID", format("%016llx", s->value()->getDesiredShardId()))
				    .detail("ShardRange", s->value()->range())
				    .detail("ShardState", s->value()->debugDescribeState())
				    .log();
				ASSERT(s->value()->range() == s->range());
				ASSERT(!s->value()->range().empty());
				if (data->shardAware) {
					s->value()->validate();
				}
			}

			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				if (s->value()->isReadable()) {
					auto ar = data->newestAvailableVersion.intersectingRanges(s->range());
					for (auto a = ar.begin(); a != ar.end(); ++a) {
						TraceEvent(SevVerbose, "ValidateShardReadable", data->thisServerID)
						    .detail("Range", s->range())
						    .detail("ShardRange", s->value()->range())
						    .detail("ShardState", s->value()->debugDescribeState())
						    .detail("AvailableRange", a->range())
						    .detail("AvailableVersion", a->value())
						    .log();
						ASSERT(a->value() == latestVersion);
					}
				}
			}

			// * versionedData contains versions [storageVersion(), version.get()].  It might also contain version
			// (version.get()+1), in which changeDurableVersion may be deleting ghosts, and/or it might
			//      contain later versions if applyUpdate is on the stack.
			ASSERT(data->data().getOldestVersion() == data->storageVersion());
			ASSERT(data->data().getLatestVersion() == data->version.get() ||
			       data->data().getLatestVersion() == data->version.get() + 1 ||
			       (data->debug_inApplyUpdate && data->data().getLatestVersion() > data->version.get()));

			auto latest = data->data().atLatest();

			// * Old shards are erased: versionedData.atLatest() has entries (sets or clear *begins*) only for keys in
			// readable or adding,transferred shards.
			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				ShardInfo* shard = s->value().getPtr();
				if (!shard->isInVersionedData()) {
					auto beginNext = latest.lower_bound(s->begin());
					auto endNext = latest.lower_bound(s->end());
					if (beginNext != endNext) {
						TraceEvent(SevError, "VF", data->thisServerID)
						    .detail("LastValidTime", data->debug_lastValidateTime)
						    .detail("KeyBegin", s->begin())
						    .detail("KeyEnd", s->end())
						    .detail("DbgState", shard->debugDescribeState())
						    .detail("FirstKey", beginNext.key())
						    .detail("LastKey", endNext != latest.end() ? endNext.key() : "End"_sr)
						    .detail("FirstInsertV", beginNext.insertVersion())
						    .detail("LastInsertV", endNext != latest.end() ? endNext.insertVersion() : invalidVersion);
					}
					ASSERT(beginNext == endNext);
				}

				if (shard->assigned() && data->shardAware) {
					TraceEvent(SevVerbose, "ValidateAssignedShard", data->thisServerID)
					    .detail("Range", shard->range())
					    .detailf("ShardID", "%016llx", shard->getShardId())
					    .detailf("DesiredShardID", "%016llx", shard->getDesiredShardId())
					    .detail("State", shard->debugDescribeState());
					ASSERT(shard->getShardId() != 0UL && shard->getDesiredShardId() != 0UL);
				}
			}

			// FIXME: do some change feed validation?

			latest.validate();
			validateRange(latest, allKeys, data->version.get(), data->thisServerID, data->durableVersion.get());

			data->debug_lastValidateTime = now();
		}
	} catch (...) {
		TraceEvent(SevError, "ValidationFailure", data->thisServerID)
		    .detail("LastValidTime", data->debug_lastValidateTime);
		throw;
	}
}
#ifndef __INTEL_COMPILER
#pragma endregion
#endif

void updateProcessStats(StorageServer* self) {
	if (g_network->isSimulated()) {
		// diskUsage and cpuUsage are not relevant in the simulator,
		// and relying on the actual values could break seed determinism
		self->cpuUsage = 100.0;
		self->diskUsage = 100.0;
		return;
	}

	SystemStatistics sysStats = getSystemStatistics();
	if (sysStats.initialized) {
		self->cpuUsage = 100 * sysStats.processCPUSeconds / sysStats.elapsed;
		self->diskUsage = 100 * std::max(0.0, (sysStats.elapsed - sysStats.processDiskIdleSeconds) / sysStats.elapsed);
	}
}

///////////////////////////////////// Queries /////////////////////////////////
#ifndef __INTEL_COMPILER
#pragma region Queries
#endif

															#line 2551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via waitForVersionActor()
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForVersionActorActor>
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionActorActorState {
															#line 2558 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionActorActorState(StorageServer* const& data,Version const& version,SpanContext const& spanContext) 
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanContext(spanContext),
															#line 2015 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:WaitForVersion"_loc, spanContext)
															#line 2571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("waitForVersionActor", reinterpret_cast<unsigned long>(this));

	}
	~WaitForVersionActorActorState() 
	{
		fdb_probe_actor_destroy("waitForVersionActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2017 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2016 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForVersionActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2588 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2025 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 2592 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForVersionActorActor*>(this)->actor_wait_state = 1;
															#line 2017 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionActorActor, 0, Void >*>(static_cast<WaitForVersionActorActor*>(this)));
															#line 2025 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionActorActor, 1, Void >*>(static_cast<WaitForVersionActorActor*>(this)));
															#line 2599 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2020 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->oldestVersion.get())
															#line 2622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2021 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 2626 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2023 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionActorActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorActorState(); static_cast<WaitForVersionActorActor*>(this)->destroy(); return 0; }
															#line 2630 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WaitForVersionActorActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2020 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->oldestVersion.get())
															#line 2642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2021 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 2646 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2023 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionActorActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorActorState(); static_cast<WaitForVersionActorActor*>(this)->destroy(); return 0; }
															#line 2650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WaitForVersionActorActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2026 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 2662 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2027 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 2666 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2031 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 2670 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2026 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 2678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2027 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 2682 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2031 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 2686 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForVersionActorActor*>(this)->actor_wait_state > 0) static_cast<WaitForVersionActorActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForVersionActorActor*>(this)->ActorCallback< WaitForVersionActorActor, 0, Void >::remove();
		static_cast<WaitForVersionActorActor*>(this)->ActorCallback< WaitForVersionActorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForVersionActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForVersionActorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext spanContext;
															#line 2015 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 2819 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via waitForVersionActor()
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionActorActor final : public Actor<Version>, public ActorCallback< WaitForVersionActorActor, 0, Void >, public ActorCallback< WaitForVersionActorActor, 1, Void >, public FastAllocated<WaitForVersionActorActor>, public WaitForVersionActorActorState<WaitForVersionActorActor> {
															#line 2824 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<WaitForVersionActorActor>::operator new;
	using FastAllocated<WaitForVersionActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14100856757135141120UL, 9430833606160878848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForVersionActorActor, 0, Void >;
friend struct ActorCallback< WaitForVersionActorActor, 1, Void >;
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionActorActor(StorageServer* const& data,Version const& version,SpanContext const& spanContext) 
															#line 2842 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Version>(),
		   WaitForVersionActorActorState<WaitForVersionActorActor>(data, version, spanContext),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17177306746367876352UL, 6989725333754748672UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForVersionActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForVersionActorActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForVersionActor( StorageServer* const& data, Version const& version, SpanContext const& spanContext ) {
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForVersionActorActor(data, version, spanContext));
															#line 2875 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2035 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// If the latest commit version that mutated the shard(s) being served by the specified storage
// server is below the client specified read version then do a read at the latest commit version
// of the storage server.
Version getRealReadVersion(VersionVector& ssLatestCommitVersions, Tag& tag, Version specifiedReadVersion) {
	Version realReadVersion =
	    ssLatestCommitVersions.hasVersion(tag) ? ssLatestCommitVersions.getVersion(tag) : specifiedReadVersion;
	ASSERT(realReadVersion <= specifiedReadVersion);
	return realReadVersion;
}

// Find the latest commit version of the given tag.
Version getLatestCommitVersion(VersionVector& ssLatestCommitVersions, Tag& tag) {
	Version commitVersion =
	    ssLatestCommitVersions.hasVersion(tag) ? ssLatestCommitVersions.getVersion(tag) : invalidVersion;
	return commitVersion;
}

Future<Version> waitForVersion(StorageServer* data, Version version, SpanContext spanContext) {
	if (version == latestVersion) {
		version = std::max(Version(1), data->version.get());
	}

	if (version < data->oldestVersion.get() || version <= 0) {
		// TraceEvent(SevDebug, "WFVThrow", data->thisServerID).detail("Version", version).detail("OldestVersion", data->oldestVersion.get());
		return transaction_too_old();
	} else if (version <= data->version.get()) {
		return version;
	}

	if ((data->behind || data->versionBehind) && version > data->version.get()) {
		return process_behind();
	}

	if (deterministicRandom()->random01() < 0.001) {
		TraceEvent("WaitForVersion1000x").log();
	}
	return waitForVersionActor(data, version, spanContext);
}

Future<Version> waitForVersion(StorageServer* data,
                               Version commitVersion,
                               Version readVersion,
                               SpanContext spanContext) {
	ASSERT(commitVersion == invalidVersion || commitVersion < readVersion);

	if (commitVersion == invalidVersion) {
		return waitForVersion(data, readVersion, spanContext);
	}

	if (readVersion == latestVersion) {
		readVersion = std::max(Version(1), data->version.get());
	}

	if (readVersion < data->oldestVersion.get() || readVersion <= 0) {
		return transaction_too_old();
	} else {
		// It is correct to read any version between [commitVersion, readVersion],
		// because version vector guarantees no mutations between them.
		if (commitVersion < data->oldestVersion.get()) {
			if (data->version.get() < readVersion) {
				// Majority of the case, try using higher version to avoid
				// transaction_too_old error when oldestVersion advances.
				// BTW, any version in the range [oldestVersion, data->version.get()] is valid in this case.
				return data->version.get();
			} else {
				ASSERT(readVersion >= data->oldestVersion.get());
				return readVersion;
			}
		} else if (commitVersion <= data->version.get()) {
			return commitVersion;
		}
	}

	if ((data->behind || data->versionBehind) && commitVersion > data->version.get()) {
		return process_behind();
	}

	if (deterministicRandom()->random01() < 0.001) {
		TraceEvent("WaitForVersion1000x");
	}
	return waitForVersionActor(data, std::max(commitVersion, data->oldestVersion.get()), spanContext);
}

															#line 2963 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via waitForVersionNoTooOld()
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForVersionNoTooOldActor>
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionNoTooOldActorState {
															#line 2970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionNoTooOldActorState(StorageServer* const& data,Version const& version) 
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version)
															#line 2979 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this));

	}
	~WaitForVersionNoTooOldActorState() 
	{
		fdb_probe_actor_destroy("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2121 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version == latestVersion)
															#line 2994 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2122 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				version = std::max(Version(1), data->version.get());
															#line 2998 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2123 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version <= data->version.get())
															#line 3002 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2124 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3006 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
				this->~WaitForVersionNoTooOldActorState();
				static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2126 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2125 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3016 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2129 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 3020 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state = 1;
															#line 2126 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*>(static_cast<WaitForVersionNoTooOldActor*>(this)));
															#line 2129 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*>(static_cast<WaitForVersionNoTooOldActor*>(this)));
															#line 3027 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2127 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3050 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2127 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3062 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2130 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3074 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2131 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3078 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2135 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3082 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2130 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3090 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2131 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3094 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2135 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3098 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state > 0) static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForVersionNoTooOldActor*>(this)->ActorCallback< WaitForVersionNoTooOldActor, 0, Void >::remove();
		static_cast<WaitForVersionNoTooOldActor*>(this)->ActorCallback< WaitForVersionNoTooOldActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3227 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via waitForVersionNoTooOld()
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionNoTooOldActor final : public Actor<Version>, public ActorCallback< WaitForVersionNoTooOldActor, 0, Void >, public ActorCallback< WaitForVersionNoTooOldActor, 1, Void >, public FastAllocated<WaitForVersionNoTooOldActor>, public WaitForVersionNoTooOldActorState<WaitForVersionNoTooOldActor> {
															#line 3232 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<WaitForVersionNoTooOldActor>::operator new;
	using FastAllocated<WaitForVersionNoTooOldActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1844679308755470080UL, 1302204802163801344UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForVersionNoTooOldActor, 0, Void >;
friend struct ActorCallback< WaitForVersionNoTooOldActor, 1, Void >;
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionNoTooOldActor(StorageServer* const& data,Version const& version) 
															#line 3250 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Version>(),
		   WaitForVersionNoTooOldActorState<WaitForVersionNoTooOldActor>(data, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14315589494631704320UL, 1805358750176616448UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForVersionNoTooOld");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForVersionNoTooOld( StorageServer* const& data, Version const& version ) {
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForVersionNoTooOldActor(data, version));
															#line 3283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2139 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 3288 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via waitForMinVersion()
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForMinVersionActor>
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForMinVersionActorState {
															#line 3295 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForMinVersionActorState(StorageServer* const& data,Version const& version) 
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version)
															#line 3304 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("waitForMinVersion", reinterpret_cast<unsigned long>(this));

	}
	~WaitForMinVersionActorState() 
	{
		fdb_probe_actor_destroy("waitForMinVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2142 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version == latestVersion)
															#line 3319 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2143 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				version = std::max(Version(1), data->version.get());
															#line 3323 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2144 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version < data->oldestVersion.get() || version <= 0)
															#line 3327 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2145 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(data->oldestVersion.get()); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3331 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(data->oldestVersion.get());
				this->~WaitForMinVersionActorState();
				static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2146 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (version <= data->version.get())
															#line 3341 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2147 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3345 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
					this->~WaitForMinVersionActorState();
					static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 2150 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2149 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForMinVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3356 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2153 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 3360 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForMinVersionActor*>(this)->actor_wait_state = 1;
															#line 2150 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForMinVersionActor, 0, Void >*>(static_cast<WaitForMinVersionActor*>(this)));
															#line 2153 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForMinVersionActor, 1, Void >*>(static_cast<WaitForMinVersionActor*>(this)));
															#line 3367 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2151 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3390 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2151 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2154 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3414 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2155 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2159 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3422 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2154 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3430 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2155 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3434 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2159 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3438 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForMinVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitForMinVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForMinVersionActor*>(this)->ActorCallback< WaitForMinVersionActor, 0, Void >::remove();
		static_cast<WaitForMinVersionActor*>(this)->ActorCallback< WaitForMinVersionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForMinVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForMinVersionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3567 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via waitForMinVersion()
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForMinVersionActor final : public Actor<Version>, public ActorCallback< WaitForMinVersionActor, 0, Void >, public ActorCallback< WaitForMinVersionActor, 1, Void >, public FastAllocated<WaitForMinVersionActor>, public WaitForMinVersionActorState<WaitForMinVersionActor> {
															#line 3572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<WaitForMinVersionActor>::operator new;
	using FastAllocated<WaitForMinVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3939902797827745792UL, 14339974065825363712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForMinVersionActor, 0, Void >;
friend struct ActorCallback< WaitForMinVersionActor, 1, Void >;
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForMinVersionActor(StorageServer* const& data,Version const& version) 
															#line 3590 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Version>(),
		   WaitForMinVersionActorState<WaitForMinVersionActor>(data, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3358323868325720064UL, 6852897307082981376UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForMinVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForMinVersionActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForMinVersion( StorageServer* const& data, Version const& version ) {
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForMinVersionActor(data, version));
															#line 3623 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2163 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

std::vector<StorageServerShard> StorageServer::getStorageServerShards(KeyRangeRef range) {
	std::vector<StorageServerShard> res;
	for (auto t : this->shards.intersectingRanges(range)) {
		res.push_back(t.value()->toStorageServerShard());
	}
	return res;
}

static Error dataMoveConflictError(const bool isTss) {
	if (isTss && g_network->isSimulated()) {
		// TSS data move conflicts can happen in both sim and prod, but in sim,
		// the sev40s cause failures of Joshua tests. We have been using please_reboot
		// as means to avoid sev40s, but semantically this is undesired because rebooting
		// will not fix/heal the TSS.
		// TODO: think of a proper TSS move conflict error that does not trigger
		// reboot but also avoids sev40. And throw that error regardless of sim or prod.
		return please_reboot();
	}
	return data_move_conflict();
}

std::shared_ptr<MoveInShard> StorageServer::getMoveInShard(const UID& dataMoveId,
                                                           const Version version,
                                                           const ConductBulkLoad conductBulkLoad) {
	for (auto& [id, moveInShard] : this->moveInShards) {
		if (moveInShard->dataMoveId() == dataMoveId && moveInShard->meta->createVersion == version) {
			return moveInShard;
		}
	}

	const UID id = deterministicRandom()->randomUniqueID();
	std::shared_ptr<MoveInShard> shard = std::make_shared<MoveInShard>(this, id, dataMoveId, version, conductBulkLoad);
	auto [it, inserted] = this->moveInShards.emplace(id, shard);
	ASSERT(inserted);
	TraceEvent(SevDebug, "SSNewMoveInShard", this->thisServerID)
	    .detail("MoveInShard", shard->toString())
	    .detail("ConductBulkLoad", conductBulkLoad);
	return shard;
}

															#line 3668 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getValueQ()
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetValueQActor>
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetValueQActorState {
															#line 3675 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueQActorState(StorageServer* const& data,GetValueRequest const& req) 
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 2205 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultSize(0)
															#line 3686 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getValueQ", reinterpret_cast<unsigned long>(this));

	}
	~GetValueQActorState() 
	{
		fdb_probe_actor_destroy("getValueQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2206 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Span span("SS:getValue"_loc, req.spanContext);
															#line 3701 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			try {
															#line 2211 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.getValueQueries;
															#line 2212 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.allQueries;
															#line 2213 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.key.startsWith(systemKeys.begin))
															#line 3709 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2214 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					++data->counters.systemKeyQueries;
															#line 3713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 2216 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				data->maxQueryQueue = std::max<int>( data->maxQueryQueue, data->counters.allQueries.getValue() - data->counters.finishedQueries.getValue());
															#line 2221 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = data->getQueryDelay();
															#line 2221 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3721 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetValueQActor*>(this)->actor_wait_state = 1;
															#line 2221 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 0, Void >*>(static_cast<GetValueQActor*>(this)));
															#line 3726 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetValueQActorState();
		static_cast<GetValueQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2327 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->transactionTagCounter.addRequest(req.tags, req.key.size() + resultSize);
															#line 2329 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.finishedQueries;
															#line 2331 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - req.requestTime();
															#line 2332 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample(duration, ReadLatencySamples::READ, trackedReadType(req));
															#line 2333 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample(duration, ReadLatencySamples::READ_VALUE, trackedReadType(req));
															#line 2334 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->latencyBandConfig.present())
															#line 3765 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2335 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxReadBytes = data->latencyBandConfig.get().readConfig.maxReadBytes.orDefault(std::numeric_limits<int>::max());
															#line 2337 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencyBands.addMeasurement(duration, 1, Filtered(resultSize > maxReadBytes));
															#line 3771 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2340 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetValueQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetValueQActorState(); static_cast<GetValueQActor*>(this)->destroy(); return 0; }
															#line 3775 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<GetValueQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetValueQActorState();
		static_cast<GetValueQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2320 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 3788 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2321 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3792 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2322 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 3796 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3813 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 2;
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetValueQActor*>(this)));
															#line 3818 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3829 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 2;
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetValueQActor*>(this)));
															#line 3834 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2225 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		queueWaitEnd = g_network->timer();
															#line 2226 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample( queueWaitEnd - req.requestTime(), ReadLatencySamples::READ_QUEUE_WAIT, trackedReadType(req));
															#line 2229 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 3922 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2230 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.DoRead");
															#line 3926 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2234 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		v = Optional<Value>();
															#line 2235 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		Version commitVersion = getLatestCommitVersion(req.ssLatestCommitVersions, data->tag);
															#line 2236 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Version> __when_expr_2 = waitForVersion(data, commitVersion, req.version, req.spanContext);
															#line 2236 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3936 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 3;
															#line 2236 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 2, Version >*>(static_cast<GetValueQActor*>(this)));
															#line 3941 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(PriorityMultiLock::Lock const& __readLock,int loopDepth) 
	{
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readLock = __readLock;
															#line 3950 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(PriorityMultiLock::Lock && __readLock,int loopDepth) 
	{
		readLock = std::move(__readLock);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 2237 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample( g_network->timer() - queueWaitEnd, ReadLatencySamples::READ_VERSION_WAIT, trackedReadType(req));
															#line 2240 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 4031 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2241 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.AfterVersion");
															#line 4035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2245 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		changeCounter = data->shardChangeCounter;
															#line 2247 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!data->shards[req.key]->isReadable())
															#line 4041 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2249 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(wrong_shard_server(), loopDepth);
															#line 4045 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2252 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		path = 0;
															#line 2253 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		auto i = data->data().at(version).lastLessOrEqual(req.key);
															#line 2254 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (i && i->isValue() && i.key() == req.key)
															#line 4053 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2255 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			v = (Value)i->getValue();
															#line 2256 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			path = 1;
															#line 4059 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont7(loopDepth);
		}
		else
		{
															#line 2257 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!i || !i->isClearTo() || i->getEndKey() <= req.key)
															#line 4066 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2258 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				path = 2;
															#line 2259 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Optional<Value>> __when_expr_3 = data->storage.readValue(req.key, req.options);
															#line 2259 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4074 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
				static_cast<GetValueQActor*>(this)->actor_wait_state = 4;
															#line 2259 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 3, Optional<Value> >*>(static_cast<GetValueQActor*>(this)));
															#line 4079 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont11(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont4when1(Version const& __version,int loopDepth) 
	{
															#line 2236 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		version = __version;
															#line 4094 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 2, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 2, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 2, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 2, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 2270 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardGetValue", version, MutationRef(MutationRef::DebugKey, req.key, v.present() ? v.get() : "<null>"_sr), data->thisServerID);
															#line 2274 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardGetPath", version, MutationRef(MutationRef::DebugKey, req.key, path == 0 ? "0"_sr : path == 1 ? "1"_sr : "2"_sr), data->thisServerID);
															#line 2290 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (v.present())
															#line 4177 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2291 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.rowsQueried;
															#line 2292 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			resultSize = v.get().size();
															#line 2293 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.bytesQueried += resultSize;
															#line 4185 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2295 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 4191 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2298 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 4195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2300 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = v.present() ? std::max((int64_t)(req.key.size() + v.get().size()), SERVER_KNOBS->EMPTY_READ_PENALTY) : SERVER_KNOBS->EMPTY_READ_PENALTY;
															#line 2303 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(req.key, bytesReadPerKSecond);
															#line 4201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2306 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 4205 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2307 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.AfterRead");
															#line 4209 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2312 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		auto cached = data->cachedRangeMap[req.key];
															#line 2316 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		GetValueReply reply(v, cached);
															#line 2317 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.penalty = data->getPenalty();
															#line 2318 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 4219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont12(Optional<Value> const& vv,int loopDepth) 
	{
															#line 2260 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvGetBytes += vv.expectedSize();
															#line 2262 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->storageVersion())
															#line 4236 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2263 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "transaction_too_old after readValue");
															#line 2264 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(transaction_too_old(), loopDepth);
															#line 4242 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2266 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter(changeCounter, req.key);
															#line 2267 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		v = vv;
															#line 4248 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont12(Optional<Value> && vv,int loopDepth) 
	{
															#line 2260 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvGetBytes += vv.expectedSize();
															#line 2262 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->storageVersion())
															#line 4259 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2263 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "transaction_too_old after readValue");
															#line 2264 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(transaction_too_old(), loopDepth);
															#line 4265 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2266 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter(changeCounter, req.key);
															#line 2267 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		v = vv;
															#line 4271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Optional<Value> const& vv,int loopDepth) 
	{
		loopDepth = a_body1cont12(vv, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Optional<Value> && vv,int loopDepth) 
	{
		loopDepth = a_body1cont12(std::move(vv), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 3, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont18(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueRequest req;
															#line 2205 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t resultSize;
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	PriorityMultiLock::Lock readLock;
															#line 2225 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double queueWaitEnd;
															#line 2234 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<Value> v;
															#line 2236 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2245 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	uint64_t changeCounter;
															#line 2252 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int path;
															#line 4382 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getValueQ()
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetValueQActor final : public Actor<Void>, public ActorCallback< GetValueQActor, 0, Void >, public ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >, public ActorCallback< GetValueQActor, 2, Version >, public ActorCallback< GetValueQActor, 3, Optional<Value> >, public FastAllocated<GetValueQActor>, public GetValueQActorState<GetValueQActor> {
															#line 4387 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetValueQActor>::operator new;
	using FastAllocated<GetValueQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5751722378969451520UL, 11451848033936046080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetValueQActor, 0, Void >;
friend struct ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >;
friend struct ActorCallback< GetValueQActor, 2, Version >;
friend struct ActorCallback< GetValueQActor, 3, Optional<Value> >;
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueQActor(StorageServer* const& data,GetValueRequest const& req) 
															#line 4407 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetValueQActorState<GetValueQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17920039446934625280UL, 7083594275336435200UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getValueQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetValueQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetValueQActor, 2, Version >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetValueQActor, 3, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getValueQ( StorageServer* const& data, GetValueRequest const& req ) {
															#line 2204 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetValueQActor(data, req));
															#line 4443 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// Pessimistic estimate the number of overhead bytes used by each
// watch. Watch key references are stored in an AsyncMap<Key,bool>, and actors
// must be kept alive until the watch is finished.
extern size_t WATCH_OVERHEAD_WATCHQ, WATCH_OVERHEAD_WATCHIMPL;

															#line 4453 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via watchWaitForValueChange()
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WatchWaitForValueChangeActor>
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchWaitForValueChangeActorState {
															#line 4460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchWaitForValueChangeActorState(StorageServer* const& data,SpanContext const& parent,KeyRef const& key) 
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   parent(parent),
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   key(key),
															#line 2349 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanLocation("SS:watchWaitForValueChange"_loc),
															#line 2350 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   span(spanLocation, parent),
															#line 2351 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   metadata(data->getWatchMetadata(key))
															#line 4477 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("watchWaitForValueChange", reinterpret_cast<unsigned long>(this));

	}
	~WatchWaitForValueChangeActorState() 
	{
		fdb_probe_actor_destroy("watchWaitForValueChange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2352 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 4492 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2353 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.Before");
															#line 4496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2357 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			originalMetadataVersion = metadata->version;
															#line 2358 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(waitForVersionNoTooOld(data, metadata->version));
															#line 2358 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4504 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 1;
															#line 2358 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 0, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4509 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchWaitForValueChangeActorState();
		static_cast<WatchWaitForValueChangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2359 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2360 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterVersion");
															#line 4536 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2364 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		minVersion = data->data().latestVersion;
															#line 2365 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2366 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		options = ReadOptions();
															#line 2367 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 4546 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2359 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4555 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2360 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterVersion");
															#line 4559 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2364 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		minVersion = data->data().latestVersion;
															#line 2365 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2366 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		options = ReadOptions();
															#line 2367 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 4569 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2369 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			metadata = data->getWatchMetadata(key);
															#line 2370 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			latest = data->version.get();
															#line 2371 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			options.debugID = metadata->debugID;
															#line 2373 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(latest >= minVersion && latest < data->data().latestVersion, "Starting watch loop with latestVersion > data->version", probe::decoration::rare);
															#line 2376 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			GetValueRequest getReq(span.context, metadata->key, latest, metadata->tags, options, VersionVector());
															#line 2377 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			getValue = getValueQ( data, getReq);
															#line 2379 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetValueReply> __when_expr_1 = getReq.reply.getFuture();
															#line 2379 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4675 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 2;
															#line 2379 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4680 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 2457 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2459 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = data->version.whenAtLeast(data->data().latestVersion);
															#line 2459 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4699 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 4;
															#line 2459 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 3, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4704 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2450 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() != error_code_transaction_too_old)
															#line 4714 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2451 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 4718 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2454 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Reading a watched key failed with transaction_too_old");
															#line 4722 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(GetValueReply const& reply,int loopDepth) 
	{
															#line 2380 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		span = Span(spanLocation, parent);
															#line 2382 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.error.present())
															#line 4739 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2383 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(reply.error.get().code() != error_code_future_version);
															#line 2384 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(reply.error.get(), loopDepth);
															#line 4745 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2386 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (BUGGIFY)
															#line 4749 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2387 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(transaction_too_old(), loopDepth);
															#line 4753 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2390 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardWatchValue", latest, MutationRef(MutationRef::DebugKey, metadata->key, reply.value.present() ? StringRef(reply.value.get()) : "<null>"_sr), data->thisServerID);
															#line 2397 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4759 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2398 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterRead");
															#line 4763 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2406 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		Version waitVersion = minVersion;
															#line 2407 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.value != metadata->value)
															#line 4769 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2408 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest >= metadata->version)
															#line 4773 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2409 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WatchWaitForValueChangeActor*>(this)->SAV<Version>::futures) { (void)(latest); this->~WatchWaitForValueChangeActorState(); static_cast<WatchWaitForValueChangeActor*>(this)->destroy(); return 0; }
															#line 4777 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<WatchWaitForValueChangeActor*>(this)->SAV< Version >::value()) Version(std::move(latest)); // state_var_RVO
				this->~WatchWaitForValueChangeActorState();
				static_cast<WatchWaitForValueChangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2410 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (metadata->version > originalMetadataVersion)
															#line 4787 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2413 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "racing watches for same value at different versions", probe::decoration::rare);
															#line 2414 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (metadata->version > waitVersion)
															#line 4793 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 2415 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						waitVersion = metadata->version;
															#line 4797 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					}
				}
			}
		}
															#line 2420 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->watchBytes > SERVER_KNOBS->MAX_STORAGE_SERVER_WATCH_BYTES)
															#line 4804 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2421 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Too many watches, reverting to polling");
															#line 2422 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(watch_cancelled(), loopDepth);
															#line 4810 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2425 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		watchBytes = (metadata->key.expectedSize() + metadata->value.expectedSize() + key.expectedSize() + sizeof(Reference<ServerWatchMetadata>) + sizeof(ServerWatchMetadata) + WATCH_OVERHEAD_WATCHIMPL);
															#line 2429 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes += watchBytes;
															#line 4816 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 2431 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest < waitVersion)
															#line 4820 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2433 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || data->version.whenAtLeast(waitVersion);
															#line 4824 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2435 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (BUGGIFY)
															#line 4828 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2437 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || delay(deterministicRandom()->random01());
															#line 4832 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2440 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 4836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2441 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.WaitChange");
															#line 4840 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2443 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = watchFuture;
															#line 2443 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 4846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 3;
															#line 2443 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 2, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4851 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(GetValueReply && reply,int loopDepth) 
	{
															#line 2380 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		span = Span(spanLocation, parent);
															#line 2382 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.error.present())
															#line 4868 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2383 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(reply.error.get().code() != error_code_future_version);
															#line 2384 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(reply.error.get(), loopDepth);
															#line 4874 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2386 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (BUGGIFY)
															#line 4878 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2387 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(transaction_too_old(), loopDepth);
															#line 4882 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2390 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardWatchValue", latest, MutationRef(MutationRef::DebugKey, metadata->key, reply.value.present() ? StringRef(reply.value.get()) : "<null>"_sr), data->thisServerID);
															#line 2397 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4888 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2398 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterRead");
															#line 4892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2406 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		Version waitVersion = minVersion;
															#line 2407 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.value != metadata->value)
															#line 4898 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2408 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest >= metadata->version)
															#line 4902 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2409 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WatchWaitForValueChangeActor*>(this)->SAV<Version>::futures) { (void)(latest); this->~WatchWaitForValueChangeActorState(); static_cast<WatchWaitForValueChangeActor*>(this)->destroy(); return 0; }
															#line 4906 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<WatchWaitForValueChangeActor*>(this)->SAV< Version >::value()) Version(std::move(latest)); // state_var_RVO
				this->~WatchWaitForValueChangeActorState();
				static_cast<WatchWaitForValueChangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2410 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (metadata->version > originalMetadataVersion)
															#line 4916 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2413 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "racing watches for same value at different versions", probe::decoration::rare);
															#line 2414 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (metadata->version > waitVersion)
															#line 4922 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 2415 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						waitVersion = metadata->version;
															#line 4926 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					}
				}
			}
		}
															#line 2420 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->watchBytes > SERVER_KNOBS->MAX_STORAGE_SERVER_WATCH_BYTES)
															#line 4933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2421 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Too many watches, reverting to polling");
															#line 2422 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(watch_cancelled(), loopDepth);
															#line 4939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2425 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		watchBytes = (metadata->key.expectedSize() + metadata->value.expectedSize() + key.expectedSize() + sizeof(Reference<ServerWatchMetadata>) + sizeof(ServerWatchMetadata) + WATCH_OVERHEAD_WATCHIMPL);
															#line 2429 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes += watchBytes;
															#line 4945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 2431 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest < waitVersion)
															#line 4949 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2433 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || data->version.whenAtLeast(waitVersion);
															#line 4953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2435 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (BUGGIFY)
															#line 4957 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2437 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || delay(deterministicRandom()->random01());
															#line 4961 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2440 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 4965 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2441 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.WaitChange");
															#line 4969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2443 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = watchFuture;
															#line 2443 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 4975 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 3;
															#line 2443 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 2, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4980 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GetValueReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(reply, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GetValueReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,GetValueReply const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,GetValueReply && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2446 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes -= watchBytes;
															#line 2447 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(e, loopDepth);
															#line 5079 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont13(Void const& _,int loopDepth) 
	{
															#line 2444 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= watchBytes;
															#line 5093 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont13cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont13(Void && _,int loopDepth) 
	{
															#line 2444 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= watchBytes;
															#line 5102 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont13cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont13cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont17(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont18(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont18(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont18(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont18(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parent;
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef key;
															#line 2349 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Location spanLocation;
															#line 2350 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 2351 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<ServerWatchMetadata> metadata;
															#line 2357 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version originalMetadataVersion;
															#line 2364 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version minVersion;
															#line 2365 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Void> watchFuture;
															#line 2366 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadOptions options;
															#line 2370 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version latest;
															#line 2377 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Void> getValue;
															#line 2425 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t watchBytes;
															#line 5321 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via watchWaitForValueChange()
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchWaitForValueChangeActor final : public Actor<Version>, public ActorCallback< WatchWaitForValueChangeActor, 0, Void >, public ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >, public ActorCallback< WatchWaitForValueChangeActor, 2, Void >, public ActorCallback< WatchWaitForValueChangeActor, 3, Void >, public FastAllocated<WatchWaitForValueChangeActor>, public WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor> {
															#line 5326 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<WatchWaitForValueChangeActor>::operator new;
	using FastAllocated<WatchWaitForValueChangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6649371576236907520UL, 8745629963970053888UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WatchWaitForValueChangeActor, 0, Void >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 2, Void >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 3, Void >;
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchWaitForValueChangeActor(StorageServer* const& data,SpanContext const& parent,KeyRef const& key) 
															#line 5346 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Version>(),
		   WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor>(data, parent, key),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(510625173218191360UL, 2956297564950870528UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("watchWaitForValueChange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> watchWaitForValueChange( StorageServer* const& data, SpanContext const& parent, KeyRef const& key ) {
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WatchWaitForValueChangeActor(data, parent, key));
															#line 5382 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2462 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

void checkCancelWatchImpl(StorageServer* data, WatchValueRequest req) {
	Reference<ServerWatchMetadata> metadata = data->getWatchMetadata(req.key.contents());
	if (metadata.isValid() && metadata->versionPromise.getFutureReferenceCount() == 1) {
		// last watch timed out so cancel watch_impl and delete key from the map
		data->deleteWatchMetadata(req.key.contents());
		metadata->watch_impl.cancel();
	}
}

															#line 5396 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via watchValueSendReply()
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WatchValueSendReplyActor>
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchValueSendReplyActorState {
															#line 5403 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueSendReplyActorState(StorageServer* const& data,WatchValueRequest const& req,Future<Version> const& resp,SpanContext const& spanContext) 
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   resp(resp),
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanContext(spanContext),
															#line 2476 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:watchValue"_loc, spanContext),
															#line 2477 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   startTime(now())
															#line 5420 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("watchValueSendReply", reinterpret_cast<unsigned long>(this));

	}
	~WatchValueSendReplyActorState() 
	{
		fdb_probe_actor_destroy("watchValueSendReply", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2478 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.watchQueries;
															#line 2479 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->numWatches;
															#line 2480 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes += WATCH_OVERHEAD_WATCHQ;
															#line 2482 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 5441 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2483 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		double timeoutDelay = -1;
															#line 2484 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->noRecentUpdates.get())
															#line 5473 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2485 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			timeoutDelay = std::max(CLIENT_KNOBS->FAST_WATCH_TIMEOUT - (now() - startTime), 0.0);
															#line 5477 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2486 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!BUGGIFY)
															#line 5483 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2487 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				timeoutDelay = std::max(CLIENT_KNOBS->WATCH_TIMEOUT - (now() - startTime), 0.0);
															#line 5487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
		}
		try {
															#line 2492 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Version> __when_expr_0 = resp;
															#line 2491 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5495 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 2500 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = timeoutDelay < 0 ? Never() : delay(timeoutDelay);
															#line 5499 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 2508 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = data->noRecentUpdates.onChange();
															#line 5503 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1when3(__when_expr_2.get(), loopDepth); };
			static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state = 1;
															#line 2492 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 0, Version >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 2500 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 1, Void >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 2508 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 2, Void >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 5512 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2511 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2512 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			checkCancelWatchImpl(data, req);
															#line 2513 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			--data->numWatches;
															#line 2515 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 5540 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2516 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 5544 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2517 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 2518 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5550 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WatchValueSendReplyActorState();
			static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version const& ver,int loopDepth) 
	{
															#line 2494 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(WatchValueReply{ ver });
															#line 2495 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2496 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2497 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2498 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5582 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Version && ver,int loopDepth) 
	{
															#line 2494 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(WatchValueReply{ ver });
															#line 2495 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2496 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2497 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2498 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 2502 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 2503 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2504 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2505 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2506 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 2502 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 2503 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2504 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2505 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2506 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state > 0) static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state = 0;
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 0, Version >::remove();
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 1, Void >::remove();
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueRequest req;
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Version> resp;
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext spanContext;
															#line 2476 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 2477 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double startTime;
															#line 5866 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via watchValueSendReply()
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchValueSendReplyActor final : public Actor<Void>, public ActorCallback< WatchValueSendReplyActor, 0, Version >, public ActorCallback< WatchValueSendReplyActor, 1, Void >, public ActorCallback< WatchValueSendReplyActor, 2, Void >, public FastAllocated<WatchValueSendReplyActor>, public WatchValueSendReplyActorState<WatchValueSendReplyActor> {
															#line 5871 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<WatchValueSendReplyActor>::operator new;
	using FastAllocated<WatchValueSendReplyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12178598524076795904UL, 15268454913616272896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WatchValueSendReplyActor, 0, Version >;
friend struct ActorCallback< WatchValueSendReplyActor, 1, Void >;
friend struct ActorCallback< WatchValueSendReplyActor, 2, Void >;
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueSendReplyActor(StorageServer* const& data,WatchValueRequest const& req,Future<Version> const& resp,SpanContext const& spanContext) 
															#line 5890 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   WatchValueSendReplyActorState<WatchValueSendReplyActor>(data, req, resp, spanContext),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3738460070669966080UL, 521894726446841856UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("watchValueSendReply");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WatchValueSendReplyActor, 0, Version >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> watchValueSendReply( StorageServer* const& data, WatchValueRequest const& req, Future<Version> const& resp, SpanContext const& spanContext ) {
															#line 2472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new WatchValueSendReplyActor(data, req, resp, spanContext));
															#line 5923 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2522 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// Finds a checkpoint.
															#line 5929 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getCheckpointQ()
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetCheckpointQActor>
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetCheckpointQActorState {
															#line 5936 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointQActorState(StorageServer* const& self,GetCheckpointRequest const& req) 
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 5945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~GetCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("getCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2526 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->durableVersion.whenAtLeast(req.version + 1);
															#line 2526 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5962 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2526 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetCheckpointQActor, 0, Void >*>(static_cast<GetCheckpointQActor*>(this)));
															#line 5967 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetCheckpointQActorState();
		static_cast<GetCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2528 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevDebug, "ServeGetCheckpointVersionSatisfied", self->thisServerID) .detail("Version", req.version) .detail("Ranges", describe(req.ranges)) .detail("Format", static_cast<int>(req.format));
															#line 2532 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(req.ranges.size() == 1);
															#line 2533 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& range : req.ranges ) {
															#line 2534 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!self->isReadable(range))
															#line 5996 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2535 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(wrong_shard_server());
															#line 2536 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6002 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~GetCheckpointQActorState();
				static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
		try {
															#line 2541 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::unordered_map<UID, CheckpointMetaData>::iterator it = self->checkpoints.begin();
															#line 2542 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;it != self->checkpoints.end();++it) {
															#line 2543 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				const CheckpointMetaData& md = it->second;
															#line 2544 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (md.version == req.version && md.format == req.format && req.actionId == md.actionId && md.hasRanges(req.ranges) && md.getState() == CheckpointMetaData::Complete)
															#line 6018 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2546 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.send(md);
															#line 2547 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevDebug, "ServeGetCheckpointEnd", self->thisServerID).detail("Checkpoint", md.toString());
															#line 6024 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					break;
				}
			}
															#line 2552 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 6030 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2553 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 6034 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			loopDepth = a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2528 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevDebug, "ServeGetCheckpointVersionSatisfied", self->thisServerID) .detail("Version", req.version) .detail("Ranges", describe(req.ranges)) .detail("Format", static_cast<int>(req.format));
															#line 2532 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(req.ranges.size() == 1);
															#line 2533 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& range : req.ranges ) {
															#line 2534 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!self->isReadable(range))
															#line 6056 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2535 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(wrong_shard_server());
															#line 2536 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6062 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~GetCheckpointQActorState();
				static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
		try {
															#line 2541 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::unordered_map<UID, CheckpointMetaData>::iterator it = self->checkpoints.begin();
															#line 2542 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;it != self->checkpoints.end();++it) {
															#line 2543 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				const CheckpointMetaData& md = it->second;
															#line 2544 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (md.version == req.version && md.format == req.format && req.actionId == md.actionId && md.hasRanges(req.ranges) && md.getState() == CheckpointMetaData::Complete)
															#line 6078 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2546 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.send(md);
															#line 2547 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevDebug, "ServeGetCheckpointEnd", self->thisServerID).detail("Checkpoint", md.toString());
															#line 6084 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					break;
				}
			}
															#line 2552 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 6090 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2553 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 6094 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			loopDepth = a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<GetCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<GetCheckpointQActor*>(this)->ActorCallback< GetCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2561 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6185 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetCheckpointQActorState();
		static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2556 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 6198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2557 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6202 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2559 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(e);
															#line 6206 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointRequest req;
															#line 6234 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getCheckpointQ()
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetCheckpointQActor final : public Actor<Void>, public ActorCallback< GetCheckpointQActor, 0, Void >, public FastAllocated<GetCheckpointQActor>, public GetCheckpointQActorState<GetCheckpointQActor> {
															#line 6239 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetCheckpointQActor>::operator new;
	using FastAllocated<GetCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4444703378335054080UL, 16338566662265545472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetCheckpointQActor, 0, Void >;
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointQActor(StorageServer* const& self,GetCheckpointRequest const& req) 
															#line 6256 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetCheckpointQActorState<GetCheckpointQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7171777638048396032UL, 5824655659692947712UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getCheckpointQ( StorageServer* const& self, GetCheckpointRequest const& req ) {
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetCheckpointQActor(self, req));
															#line 6289 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2563 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// Delete the checkpoint from disk, as well as all related persisted meta data.
															#line 6295 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via deleteCheckpointQ()
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class DeleteCheckpointQActor>
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class DeleteCheckpointQActorState {
															#line 6302 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	DeleteCheckpointQActorState(StorageServer* const& self,Version const& version,CheckpointMetaData const& checkpoint) 
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   checkpoint(checkpoint)
															#line 6313 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("deleteCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~DeleteCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("deleteCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2566 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0, TaskPriority::Low);
															#line 2566 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6330 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2566 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 0, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6335 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeleteCheckpointQActorState();
		static_cast<DeleteCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2568 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->durableVersion.whenAtLeast(version);
															#line 2568 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6360 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2568 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 1, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6365 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2568 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->durableVersion.whenAtLeast(version);
															#line 2568 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6376 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2568 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 1, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6381 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2570 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointBegin", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2572 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		self->checkpoints.erase(checkpoint.checkpointID);
															#line 6467 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 2575 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = deleteCheckpoint(checkpoint);
															#line 2575 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6473 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2575 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 2, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6478 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2570 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointBegin", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2572 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		self->checkpoints.erase(checkpoint.checkpointID);
															#line 6495 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 2575 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = deleteCheckpoint(checkpoint);
															#line 2575 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2575 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 2, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6506 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2581 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		persistCheckpointKey = Key(persistCheckpointKeys.begin.toString() + checkpoint.checkpointID.toString());
															#line 2582 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		pendingCheckpointKey = Key(persistPendingCheckpointKeys.begin.toString() + checkpoint.checkpointID.toString());
															#line 2583 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		auto& mLV = self->addVersionToMutationLog(self->data().getLatestVersion());
															#line 2584 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		self->addMutationToMutationLog( mLV, MutationRef(MutationRef::ClearRange, pendingCheckpointKey, keyAfter(pendingCheckpointKey)));
															#line 2586 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		self->addMutationToMutationLog( mLV, MutationRef(MutationRef::ClearRange, persistCheckpointKey, keyAfter(persistCheckpointKey)));
															#line 2588 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointEnd", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2590 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<DeleteCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteCheckpointQActorState(); static_cast<DeleteCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6608 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<DeleteCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteCheckpointQActorState();
		static_cast<DeleteCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2578 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 6621 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	CheckpointMetaData checkpoint;
															#line 2581 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key persistCheckpointKey;
															#line 2582 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key pendingCheckpointKey;
															#line 6741 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via deleteCheckpointQ()
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class DeleteCheckpointQActor final : public Actor<Void>, public ActorCallback< DeleteCheckpointQActor, 0, Void >, public ActorCallback< DeleteCheckpointQActor, 1, Void >, public ActorCallback< DeleteCheckpointQActor, 2, Void >, public FastAllocated<DeleteCheckpointQActor>, public DeleteCheckpointQActorState<DeleteCheckpointQActor> {
															#line 6746 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<DeleteCheckpointQActor>::operator new;
	using FastAllocated<DeleteCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10697918008964403200UL, 3103484508091234048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeleteCheckpointQActor, 0, Void >;
friend struct ActorCallback< DeleteCheckpointQActor, 1, Void >;
friend struct ActorCallback< DeleteCheckpointQActor, 2, Void >;
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	DeleteCheckpointQActor(StorageServer* const& self,Version const& version,CheckpointMetaData const& checkpoint) 
															#line 6765 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DeleteCheckpointQActorState<DeleteCheckpointQActor>(self, version, checkpoint),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7967604653483254528UL, 11896452891661470208UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deleteCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> deleteCheckpointQ( StorageServer* const& self, Version const& version, CheckpointMetaData const& checkpoint ) {
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new DeleteCheckpointQActor(self, version, checkpoint));
															#line 6800 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2592 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// Serves FetchCheckpointRequests.
															#line 6806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via fetchCheckpointQ()
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FetchCheckpointQActor>
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointQActorState {
															#line 6813 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointQActorState(StorageServer* const& self,FetchCheckpointRequest const& req) 
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 6822 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("fetchCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~FetchCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("fetchCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2595 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Token", req.token);
															#line 2599 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = nullptr;
															#line 2600 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			totalSize = 0;
															#line 2602 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2605 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const auto it = self->checkpoints.find(req.checkpointID);
															#line 2606 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 6847 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2607 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 2608 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2609 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6855 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~FetchCheckpointQActorState();
				static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 2613 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				reader = newCheckpointReader(it->second, CheckpointAsKeyValues::False, deterministicRandom()->randomUniqueID());
															#line 2614 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = reader->init(req.token);
															#line 2614 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6868 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2614 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 0, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 6873 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2648 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_4 = reader->close();
															#line 2648 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 5;
															#line 2648 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 4, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 6909 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2625 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_end_of_stream || e.code() == error_code_checkpoint_not_found)
															#line 6919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(e);
															#line 2627 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent("ServeFetchCheckpointEnd", self->thisServerID) .error(e) .detail("CheckpointID", req.checkpointID) .detail("TotalSize", totalSize) .detail("Token", req.token);
															#line 6925 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = a_body1Catch2cont1(loopDepth);
			}
			else
			{
															#line 2632 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (e.code() != error_code_operation_obsolete)
															#line 6932 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2633 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevWarnAlways, "ServerFetchCheckpointFailure") .errorUnsuppressed(e) .detail("CheckpointID", req.checkpointID) .detail("Token", req.token);
															#line 2637 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (canReplyWith(e))
															#line 6938 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 2638 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						req.reply.sendError(e);
															#line 6942 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					}
															#line 2640 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					err = e;
															#line 2641 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (reader != nullptr)
															#line 6948 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 2642 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						StrictFuture<Void> __when_expr_3 = reader->close();
															#line 2642 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6954 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_3.get(), loopDepth); };
						static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 4;
															#line 2642 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 3, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 6959 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1Catch2cont4(loopDepth);
					}
				}
				else
				{
					loopDepth = a_body1Catch2cont3(loopDepth);
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2616 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 6985 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2616 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 6994 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 2617 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Standalone<StringRef>> __when_expr_1 = reader->nextChunk(CLIENT_KNOBS->REPLY_BYTE_LIMIT);
															#line 2617 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7087 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2617 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7092 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(int loopDepth) 
	{
															#line 2618 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = req.reply.onReady();
															#line 2618 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2618 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 2, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Standalone<StringRef> const& __data,int loopDepth) 
	{
															#line 2617 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data = __data;
															#line 7117 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Standalone<StringRef> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2619 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointReply reply(req.token);
															#line 2620 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.data = data;
															#line 2621 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 2622 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		totalSize += data.size();
															#line 7202 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2619 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointReply reply(req.token);
															#line 2620 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.data = data;
															#line 2621 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 2622 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		totalSize += data.size();
															#line 7217 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1Catch2cont1(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont3(int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont4(int loopDepth) 
	{
															#line 2644 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 7313 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1Catch2cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont4(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont4(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 2649 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 7408 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 2649 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 7420 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointRequest req;
															#line 2599 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	ICheckpointReader* reader;
															#line 2600 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t totalSize;
															#line 2617 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Standalone<StringRef> data;
															#line 2640 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Error err;
															#line 7515 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via fetchCheckpointQ()
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointQActor final : public Actor<Void>, public ActorCallback< FetchCheckpointQActor, 0, Void >, public ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >, public ActorCallback< FetchCheckpointQActor, 2, Void >, public ActorCallback< FetchCheckpointQActor, 3, Void >, public ActorCallback< FetchCheckpointQActor, 4, Void >, public FastAllocated<FetchCheckpointQActor>, public FetchCheckpointQActorState<FetchCheckpointQActor> {
															#line 7520 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<FetchCheckpointQActor>::operator new;
	using FastAllocated<FetchCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9497736148113723648UL, 8271642978390382080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchCheckpointQActor, 0, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >;
friend struct ActorCallback< FetchCheckpointQActor, 2, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 3, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 4, Void >;
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointQActor(StorageServer* const& self,FetchCheckpointRequest const& req) 
															#line 7541 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FetchCheckpointQActorState<FetchCheckpointQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13937557724080409344UL, 5112112820403248640UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> fetchCheckpointQ( StorageServer* const& self, FetchCheckpointRequest const& req ) {
															#line 2594 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new FetchCheckpointQActor(self, req));
															#line 7578 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2651 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// Serves FetchCheckpointKeyValuesRequest, reads local checkpoint and sends it to the client over wire.
															#line 7584 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via fetchCheckpointKeyValuesQ()
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FetchCheckpointKeyValuesQActor>
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointKeyValuesQActorState {
															#line 7591 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesQActorState(StorageServer* const& self,FetchCheckpointKeyValuesRequest const& req) 
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 7600 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	~FetchCheckpointKeyValuesQActorState() 
	{
		fdb_probe_actor_destroy("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2654 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->serveFetchCheckpointParallelismLock.take(TaskPriority::DefaultYield);
															#line 2654 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7617 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 1;
															#line 2654 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchCheckpointKeyValuesQActorState();
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2655 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(self->serveFetchCheckpointParallelismLock);
															#line 2657 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent("ServeFetchCheckpointKeyValuesBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2661 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		const auto it = self->checkpoints.find(req.checkpointID);
															#line 2665 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (it == self->checkpoints.end())
															#line 7653 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2666 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(checkpoint_not_found());
															#line 2667 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2668 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 7661 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FetchCheckpointKeyValuesQActorState();
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2671 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		reader = nullptr;
															#line 2672 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		auto crIt = self->liveCheckpointReaders.find(req.checkpointID);
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (crIt != self->liveCheckpointReaders.end())
															#line 7673 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2674 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = crIt->second;
															#line 7677 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2676 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = newCheckpointReader(it->second, CheckpointAsKeyValues::True, deterministicRandom()->randomUniqueID());
															#line 2677 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders[req.checkpointID] = reader;
															#line 7685 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2680 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = std::unique_ptr<ICheckpointIterator>();
															#line 7689 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 2682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = reader->init(BinaryWriter::toValue(req.range, IncludeVersion()));
															#line 2682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7695 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 2682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7700 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2655 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(self->serveFetchCheckpointParallelismLock);
															#line 2657 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent("ServeFetchCheckpointKeyValuesBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2661 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		const auto it = self->checkpoints.find(req.checkpointID);
															#line 2665 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (it == self->checkpoints.end())
															#line 7723 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2666 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(checkpoint_not_found());
															#line 2667 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2668 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 7731 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FetchCheckpointKeyValuesQActorState();
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2671 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		reader = nullptr;
															#line 2672 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		auto crIt = self->liveCheckpointReaders.find(req.checkpointID);
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (crIt != self->liveCheckpointReaders.end())
															#line 7743 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2674 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = crIt->second;
															#line 7747 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2676 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = newCheckpointReader(it->second, CheckpointAsKeyValues::True, deterministicRandom()->randomUniqueID());
															#line 2677 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders[req.checkpointID] = reader;
															#line 7755 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2680 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = std::unique_ptr<ICheckpointIterator>();
															#line 7759 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 2682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = reader->init(BinaryWriter::toValue(req.range, IncludeVersion()));
															#line 2682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7765 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 2682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7770 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2726 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		iter.reset();
															#line 2727 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reader->inUse())
															#line 7862 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2728 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders.erase(req.checkpointID);
															#line 2729 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = reader->close();
															#line 2729 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7870 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont2when1(__when_expr_4.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 5;
															#line 2729 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7875 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2709 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_end_of_stream || e.code() == error_code_checkpoint_not_found)
															#line 7890 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2710 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(e);
															#line 2711 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevInfo, "ServeFetchCheckpointKeyValuesEnd", self->thisServerID) .error(e) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 7896 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2716 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarnAlways, "ServerFetchCheckpointKeyValuesFailure") .errorUnsuppressed(e) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2720 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (canReplyWith(e))
															#line 7904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2721 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(e);
															#line 7908 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 2683 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = reader->getIterator(req.range);
															#line 2685 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 7927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 2683 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = reader->getIterator(req.range);
															#line 2685 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 7938 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1(int loopDepth) 
	{
															#line 2686 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_2 = iter->nextBatch(CLIENT_KNOBS->REPLY_BYTE_LIMIT, CLIENT_KNOBS->REPLY_BYTE_LIMIT);
															#line 2686 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8031 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 3;
															#line 2686 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8036 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1(int loopDepth) 
	{
															#line 2688 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.empty())
															#line 8045 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2689 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "FetchCheckpontKeyValuesReadRange", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("FirstReturnedKey", res.front().key) .detail("LastReturnedKey", res.back().key) .detail("Size", res.size());
															#line 8049 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "FetchCheckpontKeyValuesEmptyRange", self->thisServerID) .detail("CheckpointID", req.checkpointID);
															#line 8055 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2699 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = req.reply.onReady();
															#line 2699 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8061 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 4;
															#line 2699 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8066 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 2686 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		res = __res;
															#line 8075 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1cont6loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2700 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointKeyValuesStreamReply reply;
															#line 2701 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.arena.dependsOn(res.arena());
															#line 2702 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 2703 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.data.push_back(reply.arena, res[i]);
															#line 8160 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2706 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 8164 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2700 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointKeyValuesStreamReply reply;
															#line 2701 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.arena.dependsOn(res.arena());
															#line 2702 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 2703 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.data.push_back(reply.arena, res[i]);
															#line 8179 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 2706 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 8183 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 2731 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 8267 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointKeyValuesQActorState();
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesRequest req;
															#line 2655 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowLock::Releaser holder;
															#line 2671 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	ICheckpointReader* reader;
															#line 2680 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::unique_ptr<ICheckpointIterator> iter;
															#line 2686 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	RangeResult res;
															#line 8374 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via fetchCheckpointKeyValuesQ()
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointKeyValuesQActor final : public Actor<Void>, public ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >, public ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >, public FastAllocated<FetchCheckpointKeyValuesQActor>, public FetchCheckpointKeyValuesQActorState<FetchCheckpointKeyValuesQActor> {
															#line 8379 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<FetchCheckpointKeyValuesQActor>::operator new;
	using FastAllocated<FetchCheckpointKeyValuesQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11051104109437214720UL, 7175408126066668288UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >;
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesQActor(StorageServer* const& self,FetchCheckpointKeyValuesRequest const& req) 
															#line 8400 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FetchCheckpointKeyValuesQActorState<FetchCheckpointKeyValuesQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12412750992388461056UL, 6903825642391152640UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchCheckpointKeyValuesQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> fetchCheckpointKeyValuesQ( StorageServer* const& self, FetchCheckpointKeyValuesRequest const& req ) {
															#line 2653 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new FetchCheckpointKeyValuesQActor(self, req));
															#line 8437 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

MutationsAndVersionRef filterMutations(Arena& arena,
                                       EncryptedMutationsAndVersionRef const& m,
                                       KeyRange const& range,
                                       bool encrypted,
                                       int commonPrefixLength) {
	if (m.mutations.size() == 1 && m.mutations.back().param1 == lastEpochEndPrivateKey) {
		return MutationsAndVersionRef(m.mutations, m.version, m.knownCommittedVersion);
	}

	Optional<VectorRef<MutationRef>> modifiedMutations;
	for (int i = 0; i < m.mutations.size(); i++) {
		if (m.mutations[i].type == MutationRef::SetValue) {
			bool inRange = range.begin.compareSuffix(m.mutations[i].param1, commonPrefixLength) <= 0 &&
			               m.mutations[i].param1.compareSuffix(range.end, commonPrefixLength) < 0;
			if (modifiedMutations.present() && inRange) {
				modifiedMutations.get().push_back(
				    arena, encrypted && m.encrypted.present() ? m.encrypted.get()[i] : m.mutations[i]);
			}
			if (!modifiedMutations.present() && !inRange) {
				if (encrypted && m.encrypted.present()) {
					modifiedMutations = m.encrypted.get().slice(0, i);
				} else {
					modifiedMutations = m.mutations.slice(0, i);
				}
				arena.dependsOn(range.arena());
			}
		} else {
			ASSERT(m.mutations[i].type == MutationRef::ClearRange);
			// param1 < range.begin || param2 > range.end
			if (!modifiedMutations.present() &&
			    (m.mutations[i].param1.compareSuffix(range.begin, commonPrefixLength) < 0 ||
			     m.mutations[i].param2.compareSuffix(range.end, commonPrefixLength) > 0)) {
				if (encrypted && m.encrypted.present()) {
					modifiedMutations = m.encrypted.get().slice(0, i);
				} else {
					modifiedMutations = m.mutations.slice(0, i);
				}
				arena.dependsOn(range.arena());
			}
			if (modifiedMutations.present()) {
				// param1 < range.end && range.begin < param2
				if (m.mutations[i].param1.compareSuffix(range.end, commonPrefixLength) < 0 &&
				    range.begin.compareSuffix(m.mutations[i].param2, commonPrefixLength) < 0) {
					StringRef clearBegin = m.mutations[i].param1;
					StringRef clearEnd = m.mutations[i].param2;
					bool modified = false;
					if (clearBegin.compareSuffix(range.begin, commonPrefixLength) < 0) {
						clearBegin = range.begin;
						modified = true;
					}
					if (range.end.compareSuffix(clearEnd, commonPrefixLength) < 0) {
						clearEnd = range.end;
						modified = true;
					}
					if (modified) {
						MutationRef clearMutation = MutationRef(MutationRef::ClearRange, clearBegin, clearEnd);
						if (encrypted && m.encrypted.present() && m.encrypted.get()[i].isEncrypted()) {
							clearMutation = clearMutation.encrypt(m.cipherKeys[i], arena, BlobCipherMetrics::TLOG);
						}
						modifiedMutations.get().push_back(arena, clearMutation);
					} else {
						modifiedMutations.get().push_back(
						    arena, encrypted && m.encrypted.present() ? m.encrypted.get()[i] : m.mutations[i]);
					}
				}
			}
		}
	}
	if (modifiedMutations.present()) {
		return MutationsAndVersionRef(modifiedMutations.get(), m.version, m.knownCommittedVersion);
	}
	if (!encrypted || !m.encrypted.present()) {
		return MutationsAndVersionRef(m.mutations, m.version, m.knownCommittedVersion);
	}
	return MutationsAndVersionRef(m.encrypted.get(), m.version, m.knownCommittedVersion);
}

#ifdef NO_INTELLISENSE
size_t WATCH_OVERHEAD_WATCHQ =
    sizeof(WatchValueSendReplyActorState<WatchValueSendReplyActor>) + sizeof(WatchValueSendReplyActor);
size_t WATCH_OVERHEAD_WATCHIMPL =
    sizeof(WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor>) + sizeof(WatchWaitForValueChangeActor);
#else
size_t WATCH_OVERHEAD_WATCHQ = 0; // only used in IDE so value is irrelevant
size_t WATCH_OVERHEAD_WATCHIMPL = 0;
#endif

															#line 8529 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getShardState_impl()
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetShardState_implActor>
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardState_implActorState {
															#line 8536 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardState_implActorState(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 8545 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getShardState_impl", reinterpret_cast<unsigned long>(this));

	}
	~GetShardState_implActorState() 
	{
		fdb_probe_actor_destroy("getShardState_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2822 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(req.mode != GetShardStateRequest::NO_WAIT);
															#line 2824 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 8562 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetShardState_implActorState();
		static_cast<GetShardState_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2825 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Future<Void>> onChange;
															#line 2827 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto t : data->shards.intersectingRanges(req.keys) ) {
															#line 2828 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!t.value()->assigned())
															#line 8596 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2829 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				onChange.push_back(delay(SERVER_KNOBS->SHARD_READY_DELAY));
															#line 8600 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				break;
			}
															#line 2833 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.mode == GetShardStateRequest::READABLE && !t.value()->isReadable())
															#line 8605 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2834 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (t.value()->getAddingShard())
															#line 8609 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2835 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getAddingShard()->readWrite.getFuture());
															#line 8613 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 2837 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(t.value()->getMoveInShard());
															#line 2838 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getMoveInShard()->readWrite.getFuture());
															#line 8621 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
			}
															#line 2842 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.mode == GetShardStateRequest::FETCHING && !t.value()->isFetched())
															#line 8626 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2843 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (t.value()->getAddingShard())
															#line 8630 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 2844 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getAddingShard()->fetchComplete.getFuture());
															#line 8634 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 2846 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(t.value()->getMoveInShard());
															#line 2847 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getMoveInShard()->fetchComplete.getFuture());
															#line 8642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
			}
		}
															#line 2852 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!onChange.size())
															#line 8648 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2853 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			GetShardStateReply rep(data->version.get(), data->durableVersion.get());
															#line 2854 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.includePhysicalShard)
															#line 8654 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 2855 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				rep.shards = data->getStorageServerShards(req.keys);
															#line 8658 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 2857 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(rep);
															#line 2858 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<GetShardState_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetShardState_implActorState(); static_cast<GetShardState_implActor*>(this)->destroy(); return 0; }
															#line 8664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<GetShardState_implActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~GetShardState_implActorState();
			static_cast<GetShardState_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2861 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = waitForAll(onChange);
															#line 2861 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8674 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 1;
															#line 2861 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 0, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 8679 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 2862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 2862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8690 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 2;
															#line 2862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 1, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 8695 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 2862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 2862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8706 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 2;
															#line 2862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 1, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 8711 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state > 0) static_cast<GetShardState_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardState_implActor*>(this)->ActorCallback< GetShardState_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetShardState_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state > 0) static_cast<GetShardState_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardState_implActor*>(this)->ActorCallback< GetShardState_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetShardState_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateRequest req;
															#line 8882 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getShardState_impl()
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardState_implActor final : public Actor<Void>, public ActorCallback< GetShardState_implActor, 0, Void >, public ActorCallback< GetShardState_implActor, 1, Void >, public FastAllocated<GetShardState_implActor>, public GetShardState_implActorState<GetShardState_implActor> {
															#line 8887 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetShardState_implActor>::operator new;
	using FastAllocated<GetShardState_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16990453242155796736UL, 5905290578090532352UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetShardState_implActor, 0, Void >;
friend struct ActorCallback< GetShardState_implActor, 1, Void >;
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardState_implActor(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 8905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetShardState_implActorState<GetShardState_implActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3570919601400010496UL, 14753609829273848064UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getShardState_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetShardState_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetShardState_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getShardState_impl( StorageServer* const& data, GetShardStateRequest const& req ) {
															#line 2821 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetShardState_implActor(data, req));
															#line 8939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2865 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 8944 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getShardStateQ()
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetShardStateQActor>
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardStateQActorState {
															#line 8951 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateQActorState(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 8960 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getShardStateQ", reinterpret_cast<unsigned long>(this));

	}
	~GetShardStateQActorState() 
	{
		fdb_probe_actor_destroy("getShardStateQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2868 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = getShardState_impl(data, req);
															#line 2867 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetShardStateQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8977 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2869 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(g_network->isSimulated() ? 10 : 60);
															#line 8981 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<GetShardStateQActor*>(this)->actor_wait_state = 1;
															#line 2868 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetShardStateQActor, 0, Void >*>(static_cast<GetShardStateQActor*>(this)));
															#line 2869 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardStateQActor, 1, Void >*>(static_cast<GetShardStateQActor*>(this)));
															#line 8988 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetShardStateQActorState();
		static_cast<GetShardStateQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2873 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetShardStateQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetShardStateQActorState(); static_cast<GetShardStateQActor*>(this)->destroy(); return 0; }
															#line 9011 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<GetShardStateQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetShardStateQActorState();
		static_cast<GetShardStateQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2870 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 9035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2870 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 9044 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetShardStateQActor*>(this)->actor_wait_state > 0) static_cast<GetShardStateQActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardStateQActor*>(this)->ActorCallback< GetShardStateQActor, 0, Void >::remove();
		static_cast<GetShardStateQActor*>(this)->ActorCallback< GetShardStateQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetShardStateQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetShardStateQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateRequest req;
															#line 9174 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getShardStateQ()
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardStateQActor final : public Actor<Void>, public ActorCallback< GetShardStateQActor, 0, Void >, public ActorCallback< GetShardStateQActor, 1, Void >, public FastAllocated<GetShardStateQActor>, public GetShardStateQActorState<GetShardStateQActor> {
															#line 9179 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetShardStateQActor>::operator new;
	using FastAllocated<GetShardStateQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10430829645989972480UL, 14648274779329103616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetShardStateQActor, 0, Void >;
friend struct ActorCallback< GetShardStateQActor, 1, Void >;
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateQActor(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 9197 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetShardStateQActorState<GetShardStateQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8443565892279057664UL, 11087380315085298688UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getShardStateQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetShardStateQActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getShardStateQ( StorageServer* const& data, GetShardStateRequest const& req ) {
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetShardStateQActor(data, req));
															#line 9230 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 2875 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

KeyRef addPrefix(KeyRef const& key, Optional<KeyRef> prefix, Arena& arena) {
	if (prefix.present()) {
		return key.withPrefix(prefix.get(), arena);
	} else {
		return key;
	}
}

KeyValueRef removePrefix(KeyValueRef const& src, Optional<KeyRef> prefix) {
	if (prefix.present()) {
		return KeyValueRef(src.key.removePrefix(prefix.get()), src.value);
	} else {
		return src;
	}
}

// Combines data from base (at an older version) with sets from newer versions in [start, end) and appends the first (up
// to) |limit| rows to output If limit<0, base and output are in descending order, and start->key()>end->key(), but
// start is still inclusive and end is exclusive
void merge(Arena& arena,
           VectorRef<KeyValueRef, VecSerStrategy::String>& output,
           VectorRef<KeyValueRef> const& vm_output,
           RangeResult const& base,
           int& vCount,
           int limit,
           bool stopAtEndOfBase,
           int& pos,
           int limitBytes) {
	ASSERT(limit != 0);
	// Add a dependency of the new arena on the result from the KVS so that we don't have to copy any of the KVS
	// results.
	arena.dependsOn(base.arena());

	bool forward = limit > 0;
	if (!forward)
		limit = -limit;
	int adjustedLimit = limit + output.size();
	int accumulatedBytes = 0;
	KeyValueRef const* baseStart = base.begin();
	KeyValueRef const* baseEnd = base.end();
	while (baseStart != baseEnd && vCount > 0 && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
		if (forward ? baseStart->key < vm_output[pos].key : baseStart->key > vm_output[pos].key) {
			output.push_back(arena, *baseStart++);
		} else {
			output.push_back_deep(arena, vm_output[pos]);
			if (baseStart->key == vm_output[pos].key)
				++baseStart;
			++pos;
			vCount--;
		}
		accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
	}
	while (baseStart != baseEnd && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
		output.push_back(arena, *baseStart++);
		accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
	}
	if (!stopAtEndOfBase) {
		while (vCount > 0 && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
			output.push_back_deep(arena, vm_output[pos]);
			accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
			++pos;
			vCount--;
		}
	}
}

static inline void copyOptionalValue(Arena* a,
                                     GetValueReqAndResultRef& getValue,
                                     const Optional<Value>& optionalValue) {
	getValue.result = optionalValue.castTo<ValueRef>();
	if (optionalValue.present()) {
		a->dependsOn(optionalValue.get().arena());
	}
}
															#line 9309 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via quickGetValue()
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class QuickGetValueActor>
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetValueActorState {
															#line 9316 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetValueActorState(StorageServer* const& data,StringRef const& key,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   key(key),
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   a(a),
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOriginalReq(pOriginalReq),
															#line 2956 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValue(),
															#line 2957 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValueStart(g_network->timer())
															#line 9335 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("quickGetValue", reinterpret_cast<unsigned long>(this));

	}
	~QuickGetValueActorState() 
	{
		fdb_probe_actor_destroy("quickGetValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			getValue.key = key;
															#line 2960 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (data->shards[key]->isReadable())
															#line 9352 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
				try {
															#line 2963 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					GetValueRequest req( pOriginalReq->spanContext, key, version, pOriginalReq->tags, pOriginalReq->options, VersionVector());
															#line 2968 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					data->actors.add(getValueQ(data, req));
															#line 2969 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					StrictFuture<GetValueReply> __when_expr_0 = req.reply.getFuture();
															#line 2969 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (static_cast<QuickGetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<QuickGetValueActor*>(this)->actor_wait_state = 1;
															#line 2969 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< QuickGetValueActor, 0, GetValueReply >*>(static_cast<QuickGetValueActor*>(this)));
															#line 9368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1Catch2(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2985 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.quickGetValueMiss;
															#line 2986 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->QUICK_GET_VALUE_FALLBACK)
															#line 9404 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2987 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr = Transaction(data->cx);
															#line 2988 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setVersion(version);
															#line 2990 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.trState->taskID = TaskPriority::DefaultPromiseEndpoint;
															#line 2991 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Future<Optional<Value>> valueFuture = tr.get(key, Snapshot::True);
															#line 2993 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = valueFuture;
															#line 2993 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<QuickGetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<QuickGetValueActor*>(this)->actor_wait_state = 2;
															#line 2993 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< QuickGetValueActor, 1, Optional<Value> >*>(static_cast<QuickGetValueActor*>(this)));
															#line 9423 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 3000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(quick_get_value_miss(), loopDepth);
															#line 9430 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(GetValueReply const& reply,int loopDepth) 
	{
															#line 2970 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 9458 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2971 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetValueHit;
															#line 2972 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			copyOptionalValue(a, getValue, reply.value);
															#line 2973 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValueStart;
															#line 2974 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencySamples.sample( duration, ReadLatencySamples::MAPPED_RANGE_LOCAL, trackedReadType(*pOriginalReq));
															#line 2976 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 9470 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
			this->~QuickGetValueActorState();
			static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(GetValueReply && reply,int loopDepth) 
	{
															#line 2970 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 9484 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 2971 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetValueHit;
															#line 2972 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			copyOptionalValue(a, getValue, reply.value);
															#line 2973 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValueStart;
															#line 2974 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencySamples.sample( duration, ReadLatencySamples::MAPPED_RANGE_LOCAL, trackedReadType(*pOriginalReq));
															#line 2976 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 9496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
			this->~QuickGetValueActorState();
			static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetValueReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(reply, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetValueReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<QuickGetValueActor*>(this)->actor_wait_state > 0) static_cast<QuickGetValueActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetValueActor*>(this)->ActorCallback< QuickGetValueActor, 0, GetValueReply >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,GetValueReply const& value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,GetValueReply && value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> const& valueOption,int loopDepth) 
	{
															#line 2994 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		copyOptionalValue(a, getValue, valueOption);
															#line 2995 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - getValueStart;
															#line 2996 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample( duration, ReadLatencySamples::MAPPED_RANGE_REMOTE, trackedReadType(*pOriginalReq));
															#line 2998 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 9604 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> && valueOption,int loopDepth) 
	{
															#line 2994 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		copyOptionalValue(a, getValue, valueOption);
															#line 2995 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - getValueStart;
															#line 2996 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample( duration, ReadLatencySamples::MAPPED_RANGE_REMOTE, trackedReadType(*pOriginalReq));
															#line 2998 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 9622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Value> const& valueOption,int loopDepth) 
	{
		loopDepth = a_body1cont8(valueOption, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Value> && valueOption,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(valueOption), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<QuickGetValueActor*>(this)->actor_wait_state > 0) static_cast<QuickGetValueActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetValueActor*>(this)->ActorCallback< QuickGetValueActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StringRef key;
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Arena* a;
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetMappedKeyValuesRequest* pOriginalReq;
															#line 2956 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueReqAndResultRef getValue;
															#line 2957 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double getValueStart;
															#line 2987 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 9721 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via quickGetValue()
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetValueActor final : public Actor<GetValueReqAndResultRef>, public ActorCallback< QuickGetValueActor, 0, GetValueReply >, public ActorCallback< QuickGetValueActor, 1, Optional<Value> >, public FastAllocated<QuickGetValueActor>, public QuickGetValueActorState<QuickGetValueActor> {
															#line 9726 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<QuickGetValueActor>::operator new;
	using FastAllocated<QuickGetValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1525656798204458240UL, 4401066019922875648UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetValueReqAndResultRef>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< QuickGetValueActor, 0, GetValueReply >;
friend struct ActorCallback< QuickGetValueActor, 1, Optional<Value> >;
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetValueActor(StorageServer* const& data,StringRef const& key,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 9744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<GetValueReqAndResultRef>(),
		   QuickGetValueActorState<QuickGetValueActor>(data, key, version, a, pOriginalReq),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13229342185383356416UL, 6029856400378886912UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("quickGetValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< QuickGetValueActor, 0, GetValueReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< QuickGetValueActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetValueReqAndResultRef> quickGetValue( StorageServer* const& data, StringRef const& key, Version const& version, Arena* const& a, GetMappedKeyValuesRequest* const& pOriginalReq ) {
															#line 2950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetValueReqAndResultRef>(new QuickGetValueActor(data, key, version, a, pOriginalReq));
															#line 9778 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 3003 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// If limit>=0, it returns the first rows in the range (sorted ascending), otherwise the last rows (sorted descending).
// readRange has O(|result|) + O(log |data|) cost
															#line 9785 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via readRange()
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class ReadRangeActor>
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadRangeActorState {
															#line 9792 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadRangeActorState(StorageServer* const& data,Version const& version,KeyRange const& range,int const& limit,int* const& pLimitBytes,SpanContext const& parentSpan,Optional<ReadOptions> const& options) 
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   limit(limit),
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   pLimitBytes(pLimitBytes),
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   parentSpan(parentSpan),
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options),
															#line 3013 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   result(),
															#line 3014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   view(data->data().at(version)),
															#line 3015 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   vCurrent(view.end()),
															#line 3016 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   readBegin(),
															#line 3017 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   readEnd(),
															#line 3018 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   readBeginTemp(),
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   vCount(0),
															#line 3020 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:readRange"_loc, parentSpan),
															#line 3021 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultLogicalSize(0),
															#line 3022 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   logicalSize(0),
															#line 3025 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultCache(),
															#line 3028 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   pos(0)
															#line 9835 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readRange", reinterpret_cast<unsigned long>(this));

	}
	~ReadRangeActorState() 
	{
		fdb_probe_actor_destroy("readRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3031 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			auto containingRange = data->cachedRangeMap.rangeContaining(range.begin);
															#line 3032 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (containingRange.value() && containingRange->range().end >= range.end)
															#line 9852 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3035 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				result.cached = true;
															#line 9856 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 3037 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				result.cached = false;
															#line 9862 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3040 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (limit >= 0)
															#line 9866 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3043 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				vCurrent = view.lastLessOrEqual(range.begin);
															#line 3044 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent && vCurrent->isClearTo() && vCurrent->getEndKey() > range.begin)
															#line 9872 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 3045 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					readBegin = vCurrent->getEndKey();
															#line 9876 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 3047 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					readBegin = range.begin;
															#line 9882 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 3050 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent)
															#line 9886 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 3051 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (vCurrent.key() != readBegin)
															#line 9890 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 3052 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						++vCurrent;
															#line 9894 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					}
				}
				else
				{
															#line 3058 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					vCurrent = view.begin();
															#line 9901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 3060 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (EXPENSIVE_VALIDATION)
															#line 9905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 3061 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(vCurrent == view.lower_bound(readBegin));
															#line 9909 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 3064 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				;
															#line 9913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 3152 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				vCurrent = view.lastLess(range.end);
															#line 3155 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent && vCurrent->isClearTo() && vCurrent->getEndKey() >= range.end)
															#line 9922 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 3156 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					readEnd = vCurrent.key();
															#line 3157 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					--vCurrent;
															#line 9928 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 3159 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					readEnd = range.end;
															#line 9934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 3162 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				;
															#line 9938 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadRangeActorState();
		static_cast<ReadRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3240 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesReturnedHistogram->sample(resultLogicalSize);
															#line 3241 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeKVPairsReturnedHistogram->sample(result.data.size());
															#line 3244 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(result.data.size() == 0 || *pLimitBytes + result.data.end()[-1].expectedSize() + sizeof(KeyValueRef) > 0);
															#line 3245 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		result.more = limit == 0 || *pLimitBytes <= 0;
															#line 3246 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		result.version = version;
															#line 3247 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<ReadRangeActor*>(this)->SAV<GetKeyValuesReply>::futures) { (void)(result); this->~ReadRangeActorState(); static_cast<ReadRangeActor*>(this)->destroy(); return 0; }
															#line 9972 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<ReadRangeActor*>(this)->SAV< GetKeyValuesReply >::value()) GetKeyValuesReply(std::move(result)); // state_var_RVO
		this->~ReadRangeActorState();
		static_cast<ReadRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3064 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!(limit > 0 && *pLimitBytes > 0 && readBegin < range.end))
															#line 9997 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3065 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(!vCurrent || vCurrent.key() >= readBegin);
															#line 3066 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(data->storageVersion() <= version);
															#line 3069 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pos == resultCache.size())
															#line 10007 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3070 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent)
															#line 10011 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3071 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				auto b = vCurrent;
															#line 3072 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				--b;
															#line 3073 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(!b || b.key() < readBegin);
															#line 10019 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3077 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int vSize = 0;
															#line 3078 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;vCurrent && vCurrent.key() < range.end && !vCurrent->isClearTo() && vCount < limit && vSize < *pLimitBytes;) {
															#line 3081 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				resultCache.emplace_back(result.arena, vCurrent.key(), vCurrent->getValue());
															#line 3082 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				vSize += sizeof(KeyValueRef) + resultCache.cback().expectedSize();
															#line 3083 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCount;
															#line 3084 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 10033 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
		}
															#line 3089 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readEnd = vCurrent ? std::min(vCurrent.key(), range.end) : range.end;
															#line 3090 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_0 = data->storage.readRange(KeyRangeRef(readBegin, readEnd), limit, *pLimitBytes, options);
															#line 3090 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10042 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ReadRangeActor*>(this)->actor_wait_state = 1;
															#line 3090 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadRangeActor, 0, RangeResult >*>(static_cast<ReadRangeActor*>(this)));
															#line 10047 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(RangeResult const& atStorageVersion,int loopDepth) 
	{
															#line 3092 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 3093 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 3094 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 3095 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 3097 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= limit);
															#line 3098 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 10079 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3099 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 3104 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 10085 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3109 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 3110 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes);
															#line 3119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		limit -= result.data.size() - prevSize;
															#line 3121 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 3122 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 10097 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3125 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit <= 0 || *pLimitBytes <= 0)
															#line 10101 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3133 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 10107 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3134 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key));
															#line 3137 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			readBegin = readBeginTemp = keyAfter(atStorageVersion.end()[-1].key);
															#line 10113 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 3141 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 10119 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3142 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent->getEndKey() > readBegin);
															#line 3144 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				readBegin = vCurrent->getEndKey();
															#line 3145 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 10127 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 3147 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readEnd == range.end);
															#line 10133 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(RangeResult && atStorageVersion,int loopDepth) 
	{
															#line 3092 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 3093 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 3094 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 3095 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 3097 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= limit);
															#line 3098 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 10155 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3099 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 3104 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 10161 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3109 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 3110 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes);
															#line 3119 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		limit -= result.data.size() - prevSize;
															#line 3121 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 3122 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 10173 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3125 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit <= 0 || *pLimitBytes <= 0)
															#line 10177 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3133 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 10183 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3134 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key));
															#line 3137 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			readBegin = readBeginTemp = keyAfter(atStorageVersion.end()[-1].key);
															#line 10189 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 3141 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 10195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3142 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent->getEndKey() > readBegin);
															#line 3144 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				readBegin = vCurrent->getEndKey();
															#line 3145 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 10203 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 3147 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readEnd == range.end);
															#line 10209 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(atStorageVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(atStorageVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state > 0) static_cast<ReadRangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReadRangeActor*>(this)->ActorCallback< ReadRangeActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadRangeActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont11(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
															#line 3162 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!(limit < 0 && *pLimitBytes > 0 && readEnd > range.begin))
															#line 10309 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3163 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(!vCurrent || vCurrent.key() < readEnd);
															#line 3164 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(data->storageVersion() <= version);
															#line 3167 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pos == resultCache.size())
															#line 10319 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3168 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent)
															#line 10323 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3169 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				auto b = vCurrent;
															#line 3170 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++b;
															#line 3171 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(!b || b.key() >= readEnd);
															#line 10331 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3174 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			vCount = 0;
															#line 3175 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int vSize = 0;
															#line 3176 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;vCurrent && vCurrent.key() >= range.begin && !vCurrent->isClearTo() && vCount < -limit && vSize < *pLimitBytes;) {
															#line 3179 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				resultCache.emplace_back(result.arena, vCurrent.key(), vCurrent->getValue());
															#line 3180 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				vSize += sizeof(KeyValueRef) + resultCache.cback().expectedSize();
															#line 3181 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCount;
															#line 3182 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 10347 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
		}
															#line 3186 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readBegin = vCurrent ? std::max(vCurrent->isClearTo() ? vCurrent->getEndKey() : vCurrent.key(), range.begin) : range.begin;
															#line 3188 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = data->storage.readRange(KeyRangeRef(readBegin, readEnd), limit, *pLimitBytes, options);
															#line 3188 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10356 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadRangeActor*>(this)->actor_wait_state = 2;
															#line 3188 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadRangeActor, 1, RangeResult >*>(static_cast<ReadRangeActor*>(this)));
															#line 10361 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break2(int loopDepth) 
	{
		try {
			return a_body1cont11(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(RangeResult const& atStorageVersion,int loopDepth) 
	{
															#line 3190 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 3191 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 3192 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 3193 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 3195 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= -limit);
															#line 3196 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 10393 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3197 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 3202 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 10399 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3205 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 3206 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes);
															#line 3215 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		limit += result.data.size() - prevSize;
															#line 3217 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 3218 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 10411 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3221 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit >= 0 || *pLimitBytes <= 0)
															#line 10415 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3225 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 10421 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3226 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key));
															#line 3229 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			readEnd = atStorageVersion.end()[-1].key;
															#line 10427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 3230 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 10433 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3231 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent.key() < readEnd);
															#line 3232 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				readEnd = vCurrent.key();
															#line 3233 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 10441 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 3235 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readBegin == range.begin);
															#line 10447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				return a_body1break2(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2cont1(RangeResult && atStorageVersion,int loopDepth) 
	{
															#line 3190 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 3191 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 3192 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 3193 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 3195 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= -limit);
															#line 3196 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 10469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3197 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 3202 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 10475 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3205 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 3206 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes);
															#line 3215 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		limit += result.data.size() - prevSize;
															#line 3217 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 3218 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 10487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3221 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit >= 0 || *pLimitBytes <= 0)
															#line 10491 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3225 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 10497 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3226 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key));
															#line 3229 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			readEnd = atStorageVersion.end()[-1].key;
															#line 10503 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 3230 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 10509 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3231 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent.key() < readEnd);
															#line 3232 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				readEnd = vCurrent.key();
															#line 3233 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 10517 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 3235 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readBegin == range.begin);
															#line 10523 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				return a_body1break2(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2when1(RangeResult const& atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(atStorageVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(RangeResult && atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(std::move(atStorageVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state > 0) static_cast<ReadRangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReadRangeActor*>(this)->ActorCallback< ReadRangeActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadRangeActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange range;
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int limit;
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int* pLimitBytes;
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parentSpan;
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 3013 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesReply result;
															#line 3014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer::VersionedData::ViewAtVersion view;
															#line 3015 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer::VersionedData::iterator vCurrent;
															#line 3016 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef readBegin;
															#line 3017 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef readEnd;
															#line 3018 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key readBeginTemp;
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int vCount;
															#line 3020 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 3021 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int resultLogicalSize;
															#line 3022 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int logicalSize;
															#line 3025 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	VectorRef<KeyValueRef> resultCache;
															#line 3028 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int pos;
															#line 10644 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readRange()
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadRangeActor final : public Actor<GetKeyValuesReply>, public ActorCallback< ReadRangeActor, 0, RangeResult >, public ActorCallback< ReadRangeActor, 1, RangeResult >, public FastAllocated<ReadRangeActor>, public ReadRangeActorState<ReadRangeActor> {
															#line 10649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadRangeActor>::operator new;
	using FastAllocated<ReadRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7882077698660442624UL, 11676195279859821824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetKeyValuesReply>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadRangeActor, 0, RangeResult >;
friend struct ActorCallback< ReadRangeActor, 1, RangeResult >;
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadRangeActor(StorageServer* const& data,Version const& version,KeyRange const& range,int const& limit,int* const& pLimitBytes,SpanContext const& parentSpan,Optional<ReadOptions> const& options) 
															#line 10667 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<GetKeyValuesReply>(),
		   ReadRangeActorState<ReadRangeActor>(data, version, range, limit, pLimitBytes, parentSpan, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3635396024098275328UL, 3211122134102601216UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadRangeActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadRangeActor, 1, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetKeyValuesReply> readRange( StorageServer* const& data, Version const& version, KeyRange const& range, int const& limit, int* const& pLimitBytes, SpanContext const& parentSpan, Optional<ReadOptions> const& options ) {
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetKeyValuesReply>(new ReadRangeActor(data, version, range, limit, pLimitBytes, parentSpan, options));
															#line 10701 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 3249 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 10706 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via findKey()
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FindKeyActor>
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class FindKeyActorState {
															#line 10713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FindKeyActorState(StorageServer* const& data,KeySelectorRef const& sel,Version const& version,KeyRange const& range,int* const& pOffset,SpanContext const& parentSpan,Optional<ReadOptions> const& options) 
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   sel(sel),
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOffset(pOffset),
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   parentSpan(parentSpan),
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options)
															#line 10732 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("findKey", reinterpret_cast<unsigned long>(this));

	}
	~FindKeyActorState() 
	{
		fdb_probe_actor_destroy("findKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3268 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version != latestVersion);
															#line 3269 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(selectorInRange(sel, range) && version >= data->oldestVersion.get());
															#line 3272 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			forward = sel.offset > 0;
															#line 3273 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			sign = forward ? +1 : -1;
															#line 3274 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			skipEqualKey = sel.orEqual == forward;
															#line 3275 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			distance = forward ? sel.offset : 1 - sel.offset;
															#line 3276 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			span = Span("SS.findKey"_loc, parentSpan);
															#line 3280 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			maxBytes = int();
															#line 3281 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (sel.offset <= 1 && sel.offset >= 0)
															#line 10763 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3282 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				maxBytes = std::numeric_limits<int>::max();
															#line 10767 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 3284 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				maxBytes = (g_network->isSimulated() && g_simulator->tssMode == ISimulator::TSSMode::Disabled && BUGGIFY) ? SERVER_KNOBS->BUGGIFY_LIMIT_BYTES : SERVER_KNOBS->STORAGE_LIMIT_BYTES;
															#line 10773 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3288 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_0 = readRange(data, version, forward ? KeyRangeRef(sel.getKey(), range.end) : KeyRangeRef(range.begin, keyAfter(sel.getKey())), (distance + skipEqualKey) * sign, &maxBytes, span.context, options);
															#line 3288 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FindKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10779 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FindKeyActor*>(this)->actor_wait_state = 1;
															#line 3288 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*>(static_cast<FindKeyActor*>(this)));
															#line 10784 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FindKeyActorState();
		static_cast<FindKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3296 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 3300 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (more && !forward && rep.data.size() == 1)
															#line 10809 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3301 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Reverse key selector returned only one result in range read");
															#line 3302 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			maxBytes = std::numeric_limits<int>::max();
															#line 3303 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_1 = readRange( data, version, KeyRangeRef(range.begin, keyAfter(sel.getKey())), -2, &maxBytes, span.context, options);
															#line 3303 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FindKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10819 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FindKeyActor*>(this)->actor_wait_state = 2;
															#line 3303 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*>(static_cast<FindKeyActor*>(this)));
															#line 10824 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply const& __rep,int loopDepth) 
	{
															#line 3288 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = __rep;
															#line 10838 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply && __rep,int loopDepth) 
	{
		rep = std::move(__rep);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FindKeyActor*>(this)->actor_wait_state > 0) static_cast<FindKeyActor*>(this)->actor_wait_state = 0;
		static_cast<FindKeyActor*>(this)->ActorCallback< FindKeyActor, 0, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 3310 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		int index = distance - 1;
															#line 3311 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (skipEqualKey && rep.data.size() && rep.data[0].key == sel.getKey())
															#line 10919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3312 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++index;
															#line 10923 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3314 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (index < rep.data.size())
															#line 10927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3315 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			*pOffset = 0;
															#line 3317 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 10933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3318 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				int64_t bytesReadPerKSecond = std::max((int64_t)rep.data[index].key.size(), SERVER_KNOBS->EMPTY_READ_PENALTY);
															#line 3320 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				data->metrics.notifyBytesReadPerKSecond(sel.getKey(), bytesReadPerKSecond);
															#line 10939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3323 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(rep.data[index].key); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 10943 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(rep.data[index].key);
			this->~FindKeyActorState();
			static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 3325 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 10953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3326 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				int64_t bytesReadPerKSecond = SERVER_KNOBS->EMPTY_READ_PENALTY;
															#line 3327 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				data->metrics.notifyBytesReadPerKSecond(sel.getKey(), bytesReadPerKSecond);
															#line 10959 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3331 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			*pOffset = index - rep.data.size() + 1;
															#line 3332 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!forward)
															#line 10965 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3333 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				*pOffset = -*pOffset;
															#line 10969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3335 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (more)
															#line 10973 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3336 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				CODE_PROBE(true, "Key selector read range had more results");
															#line 3338 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(rep.data.size());
															#line 3339 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key returnKey = forward ? keyAfter(rep.data.back().key) : rep.data.back().key;
															#line 3343 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(returnKey != sel.getKey());
															#line 3344 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(returnKey); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 10985 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(returnKey);
				this->~FindKeyActorState();
				static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 3346 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(forward ? range.end : range.begin); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 10995 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(forward ? range.end : range.begin);
				this->~FindKeyActorState();
				static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}

		return loopDepth;
	}
	int a_body1cont5(GetKeyValuesReply const& rep2,int loopDepth) 
	{
															#line 3305 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = rep2;
															#line 3306 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 3307 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(rep.data.size() == 2 || !more);
															#line 11013 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(GetKeyValuesReply && rep2,int loopDepth) 
	{
															#line 3305 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = rep2;
															#line 3306 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 3307 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(rep.data.size() == 2 || !more);
															#line 11026 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(GetKeyValuesReply const& rep2,int loopDepth) 
	{
		loopDepth = a_body1cont5(rep2, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(GetKeyValuesReply && rep2,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(rep2), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FindKeyActor*>(this)->actor_wait_state > 0) static_cast<FindKeyActor*>(this)->actor_wait_state = 0;
		static_cast<FindKeyActor*>(this)->ActorCallback< FindKeyActor, 1, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeySelectorRef sel;
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange range;
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int* pOffset;
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parentSpan;
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 3272 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	bool forward;
															#line 3273 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int sign;
															#line 3274 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	bool skipEqualKey;
															#line 3275 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int distance;
															#line 3276 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 3280 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int maxBytes;
															#line 3288 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesReply rep;
															#line 3296 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	bool more;
															#line 11136 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via findKey()
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class FindKeyActor final : public Actor<Key>, public ActorCallback< FindKeyActor, 0, GetKeyValuesReply >, public ActorCallback< FindKeyActor, 1, GetKeyValuesReply >, public FastAllocated<FindKeyActor>, public FindKeyActorState<FindKeyActor> {
															#line 11141 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<FindKeyActor>::operator new;
	using FastAllocated<FindKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11322063836829577472UL, 17405128720505687040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Key>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FindKeyActor, 0, GetKeyValuesReply >;
friend struct ActorCallback< FindKeyActor, 1, GetKeyValuesReply >;
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FindKeyActor(StorageServer* const& data,KeySelectorRef const& sel,Version const& version,KeyRange const& range,int* const& pOffset,SpanContext const& parentSpan,Optional<ReadOptions> const& options) 
															#line 11159 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Key>(),
		   FindKeyActorState<FindKeyActor>(data, sel, version, range, pOffset, parentSpan, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11946933499123123200UL, 11635880071562820096UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("findKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Key> findKey( StorageServer* const& data, KeySelectorRef const& sel, Version const& version, KeyRange const& range, int* const& pOffset, SpanContext const& parentSpan, Optional<ReadOptions> const& options ) {
															#line 3250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Key>(new FindKeyActor(data, sel, version, range, pOffset, parentSpan, options));
															#line 11193 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 3350 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

KeyRange getShardKeyRange(StorageServer* data, const KeySelectorRef& sel)
// Returns largest range such that the shard state isReadable and selectorInRange(sel, range) or wrong_shard_server if
// no such range exists
{
	auto i = sel.isBackward() ? data->shards.rangeContainingKeyBefore(sel.getKey())
	                          : data->shards.rangeContaining(sel.getKey());
	auto fullRange = data->shards.ranges();
	if (!i->value()->isReadable())
		throw wrong_shard_server();
	ASSERT(selectorInRange(sel, i->range()));
	Key begin, end;
	if (sel.isBackward()) {
		end = i->range().end;
		while (i != fullRange.begin() && i.value()->isReadable()) {
			begin = i->range().begin;
			--i;
		}
		if (i.value()->isReadable()) {
			begin = i->range().begin;
		}
	} else {
		begin = i->range().begin;
		while (i != fullRange.end() && i.value()->isReadable()) {
			end = i->range().end;
			++i;
		}
	}
	return KeyRangeRef(begin, end);
}

void maybeInjectConsistencyScanCorruption(UID thisServerID, GetKeyValuesRequest const& req, GetKeyValuesReply& reply) {
	if (g_simulator->consistencyScanState != ISimulator::SimConsistencyScanState::Enabled_InjectCorruption ||
	    !req.options.present() || !req.options.get().consistencyCheckStartVersion.present() ||
	    !g_simulator->consistencyScanCorruptRequestKey.present()) {
		return;
	}

	UID destination = req.reply.getEndpoint().token;

	ASSERT(g_simulator->consistencyScanInjectedCorruptionType.present() ==
	       g_simulator->consistencyScanInjectedCorruptionDestination.present());
	// if we already injected a corruption, reinject it if this request was a retransmit of the same one we corrupted
	// could also check that this storage sent the corruption but the reply endpoints should be globally unique so this
	// covers it
	if (g_simulator->consistencyScanInjectedCorruptionDestination.present() &&
	    (g_simulator->consistencyScanInjectedCorruptionDestination.get() != destination)) {
		return;
	}

	CODE_PROBE(true, "consistency check injecting corruption");
	CODE_PROBE(g_simulator->consistencyScanInjectedCorruptionDestination.present() &&
	               g_simulator->consistencyScanInjectedCorruptionDestination.get() == destination,
	           "consistency check re-injecting corruption after retransmit",
	           probe::decoration::rare);

	g_simulator->consistencyScanInjectedCorruptionDestination = destination;
	// FIXME: reinject same type of corruption once we enable other types

	// FIXME: code probe for each type?

	if (true /*deterministicRandom()->random01() < 0.3*/) {
		// flip more flag
		reply.more = !reply.more;
		g_simulator->consistencyScanInjectedCorruptionType = ISimulator::SimConsistencyScanCorruptionType::FlipMoreFlag;
	} else {
		// FIXME: weird memory issues when messing with actual response data, enable and figure out later
		ASSERT(false);
		// make deep copy of request, since some of the underlying memory can reference storage engine data directly
		GetKeyValuesReply copy = reply;
		reply = GetKeyValuesReply();
		reply.more = copy.more;
		reply.cached = copy.cached;
		reply.version = copy.version;
		reply.data.append_deep(reply.arena, copy.data.begin(), copy.data.size());

		if (reply.data.empty()) {
			// add row to empty response
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::AddToEmpty;
			reply.data.push_back_deep(
			    reply.arena,
			    KeyValueRef(g_simulator->consistencyScanCorruptRequestKey.get(), "consistencyCheckCorruptValue"_sr));
		} else if (deterministicRandom()->coinflip() || reply.data.back().value.empty()) {
			// change value in non-empty response
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::RemoveLastRow;
			reply.data.pop_back();
		} else {
			// chop off last byte of first value
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::ChangeFirstValue;

			reply.data[0].value = reply.data[0].value.substr(0, reply.data[0].value.size() - 1);
		}
	}

	TraceEvent(SevWarnAlways, "InjectedConsistencyScanCorruption", thisServerID)
	    .detail("CorruptionType", g_simulator->consistencyScanInjectedCorruptionType.get())
	    .detail("Version", req.version)
	    .detail("Count", reply.data.size());
}

															#line 11300 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getKeyValuesQ()
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetKeyValuesQActor>
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetKeyValuesQActorState {
															#line 11307 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesQActorState(StorageServer* const& data,GetKeyValuesRequest const& req) 
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 3457 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:getKeyValues"_loc, req.spanContext),
															#line 3458 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultSize(0)
															#line 11320 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	~GetKeyValuesQActorState() 
	{
		fdb_probe_actor_destroy("getKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3460 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			getCurrentLineage()->modify(&TransactionLineage::txID) = req.spanContext.traceID;
															#line 3462 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.getRangeQueries;
															#line 3463 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.allQueries;
															#line 3464 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.begin.getKey().startsWith(systemKeys.begin))
															#line 11341 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3465 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.systemKeyQueries;
															#line 3466 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.getRangeSystemKeyQueries;
															#line 11347 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3468 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->maxQueryQueue = std::max<int>( data->maxQueryQueue, data->counters.allQueries.getValue() - data->counters.finishedQueries.getValue());
															#line 3473 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->getQueryDelay();
															#line 3473 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11355 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 1;
															#line 3473 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 0, Void >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11360 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetKeyValuesQActorState();
		static_cast<GetKeyValuesQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3474 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 3474 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11385 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 3474 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11390 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3474 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 3474 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11401 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 3474 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11406 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 3477 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		queueWaitEnd = g_network->timer();
															#line 3478 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample( queueWaitEnd - req.requestTime(), ReadLatencySamples::READ_QUEUE_WAIT, trackedReadType(req));
															#line 11492 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 3482 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 11496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3483 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.Before");
															#line 11500 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3486 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Version commitVersion = getLatestCommitVersion(req.ssLatestCommitVersions, data->tag);
															#line 3487 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Version> __when_expr_2 = waitForVersion(data, commitVersion, req.version, span.context);
															#line 3487 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 11508 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 3;
															#line 3487 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 2, Version >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11513 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(PriorityMultiLock::Lock const& __readLock,int loopDepth) 
	{
															#line 3474 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readLock = __readLock;
															#line 11528 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(PriorityMultiLock::Lock && __readLock,int loopDepth) 
	{
		readLock = std::move(__readLock);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 3643 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->transactionTagCounter.addRequest(req.tags, resultSize);
															#line 3644 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.finishedQueries;
															#line 3646 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - req.requestTime();
															#line 3647 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample(duration, ReadLatencySamples::READ, trackedReadType(req));
															#line 3648 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample(duration, ReadLatencySamples::READ_RANGE, trackedReadType(req));
															#line 3649 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->latencyBandConfig.present())
															#line 11617 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3650 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxReadBytes = data->latencyBandConfig.get().readConfig.maxReadBytes.orDefault(std::numeric_limits<int>::max());
															#line 3652 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxSelectorOffset = data->latencyBandConfig.get().readConfig.maxKeySelectorOffset.orDefault(std::numeric_limits<int>::max());
															#line 3654 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencyBands.addMeasurement(duration, 1, Filtered(resultSize > maxReadBytes || abs(req.begin.offset) > maxSelectorOffset || abs(req.end.offset) > maxSelectorOffset));
															#line 11625 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3661 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetKeyValuesQActorState(); static_cast<GetKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 11629 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<GetKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetKeyValuesQActorState();
		static_cast<GetKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3638 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 11642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3639 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 11646 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3640 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 11650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
															#line 3488 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		DisabledTraceEvent("VVV", data->thisServerID) .detail("Version", version) .detail("ReqVersion", req.version) .detail("Oldest", data->oldestVersion.get()) .detail("VV", req.ssLatestCommitVersions.toString()) .detail("DebugID", req.options.present() && req.options.get().debugID.present() ? req.options.get().debugID.get() : UID());
															#line 3496 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample( g_network->timer() - queueWaitEnd, ReadLatencySamples::READ_VERSION_WAIT, trackedReadType(req));
															#line 3499 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		changeCounter = data->shardChangeCounter;
															#line 3501 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		shard = getShardKeyRange(data, req.begin);
															#line 3503 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 11673 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3504 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterVersion");
															#line 11677 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3511 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!selectorInRange(req.end, shard) && !(req.end.isFirstGreaterOrEqual() && req.end.getKey() == shard.end))
															#line 11681 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3519 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont3Catch1(wrong_shard_server(), loopDepth);
															#line 11685 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3522 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		offset1 = 0;
															#line 3523 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		offset2 = int();
															#line 3524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fBegin = req.begin.isFirstGreaterOrEqual() ? Future<Key>(req.begin.getKey()) : findKey(data, req.begin, version, shard, &offset1, span.context, req.options);
															#line 3527 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fEnd = req.end.isFirstGreaterOrEqual() ? Future<Key>(req.end.getKey()) : findKey(data, req.end, version, shard, &offset2, span.context, req.options);
															#line 3530 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Key> __when_expr_3 = fBegin;
															#line 3530 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 11699 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont3Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 4;
															#line 3530 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 3, Key >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11704 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(Version const& __version,int loopDepth) 
	{
															#line 3487 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		version = __version;
															#line 11713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 2, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 2, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 2, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 2, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 3531 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Key> __when_expr_4 = fEnd;
															#line 3531 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 11794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont3Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 5;
															#line 3531 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 4, Key >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11799 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5when1(Key const& __begin,int loopDepth) 
	{
															#line 3530 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		begin = __begin;
															#line 11808 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Key && __begin,int loopDepth) 
	{
		begin = std::move(__begin);
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 3, Key >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 3, Key >*,Key const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 3, Key >*,Key && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 3, Key >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont10(int loopDepth) 
	{
															#line 3533 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 11887 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3534 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterKeys");
															#line 11891 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3542 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if ((offset1 && offset1 != 1) || (offset2 && offset2 != 1))
															#line 11895 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3543 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "wrong_shard_server due to offset");
															#line 3559 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont3Catch1(wrong_shard_server(), loopDepth);
															#line 11901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3562 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (begin >= end)
															#line 11905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3563 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 11909 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3564 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.Send");
															#line 11913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3568 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			GetKeyValuesReply none;
															#line 3569 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			none.version = version;
															#line 3570 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			none.more = false;
															#line 3571 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			none.penalty = data->getPenalty();
															#line 3573 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->checkChangeCounter(changeCounter, KeyRangeRef(std::min<KeyRef>(req.begin.getKey(), req.end.getKey()), std::max<KeyRef>(req.begin.getKey(), req.end.getKey())));
															#line 3577 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (g_network->isSimulated())
															#line 11927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3578 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				maybeInjectConsistencyScanCorruption(data->thisServerID, req, none);
															#line 11931 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3580 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(none);
															#line 11935 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont10cont1(loopDepth);
		}
		else
		{
															#line 3582 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			remainingLimitBytes = req.limitBytes;
															#line 3584 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			kvReadRange = g_network->timer();
															#line 3585 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_5 = readRange( data, version, KeyRangeRef(begin, end), req.limit, &remainingLimitBytes, span.context, req.options);
															#line 3585 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 11948 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont3Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont10when1(__when_expr_5.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 6;
															#line 3585 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont7when1(Key const& __end,int loopDepth) 
	{
															#line 3531 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		end = __end;
															#line 11963 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Key && __end,int loopDepth) 
	{
		end = std::move(__end);
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 4, Key >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 4, Key >*,Key const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 4, Key >*,Key && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 4, Key >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont10cont1(int loopDepth) 
	{
		loopDepth = a_body1cont10cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont7(GetKeyValuesReply const& _r,int loopDepth) 
	{
															#line 3587 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - kvReadRange;
															#line 3588 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample(duration, ReadLatencySamples::KV_READ_RANGE, trackedReadType(req));
															#line 3589 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesReply r = _r;
															#line 3591 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 12054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3592 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterReadRange");
															#line 12058 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3596 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter( changeCounter, KeyRangeRef(std::min<KeyRef>(begin, std::min<KeyRef>(req.begin.getKey(), req.end.getKey())), std::max<KeyRef>(end, std::max<KeyRef>(req.begin.getKey(), req.end.getKey()))));
															#line 3600 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (EXPENSIVE_VALIDATION)
															#line 12064 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3601 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = 0;i < r.data.size();i++) {
															#line 3602 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(r.data[i].key >= begin && r.data[i].key < end);
															#line 12070 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3604 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(r.data.size() <= std::abs(req.limit));
															#line 12074 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3608 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		int64_t totalByteSize = 0;
															#line 3609 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < r.data.size();i++) {
															#line 3610 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			totalByteSize += r.data[i].expectedSize();
															#line 12082 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3613 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.taskID.present() && req.taskID.get() == TaskPriority::FetchKeys)
															#line 12086 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3614 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchServed += r.data.size();
															#line 3615 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchBytesServed += (totalByteSize + (8 - (int)sizeof(KeyValueRef)) * r.data.size());
															#line 12092 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3618 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (totalByteSize > 0 && SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 12096 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3619 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = std::max(totalByteSize, SERVER_KNOBS->EMPTY_READ_PENALTY) / 2;
															#line 3620 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(r.data[0].key, bytesReadPerKSecond);
															#line 3621 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(r.data[r.data.size() - 1].key, bytesReadPerKSecond);
															#line 12104 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3624 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		r.penalty = data->getPenalty();
															#line 3625 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (g_network->isSimulated())
															#line 12110 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3626 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			maybeInjectConsistencyScanCorruption(data->thisServerID, req, r);
															#line 12114 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3628 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(r);
															#line 3630 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		resultSize = req.limitBytes - remainingLimitBytes;
															#line 3631 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.bytesQueried += resultSize;
															#line 3632 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.rowsQueried += r.data.size();
															#line 3633 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.data.size() == 0)
															#line 12126 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3634 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 12130 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont10cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont7(GetKeyValuesReply && _r,int loopDepth) 
	{
															#line 3587 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - kvReadRange;
															#line 3588 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample(duration, ReadLatencySamples::KV_READ_RANGE, trackedReadType(req));
															#line 3589 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesReply r = _r;
															#line 3591 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 12146 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3592 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterReadRange");
															#line 12150 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3596 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter( changeCounter, KeyRangeRef(std::min<KeyRef>(begin, std::min<KeyRef>(req.begin.getKey(), req.end.getKey())), std::max<KeyRef>(end, std::max<KeyRef>(req.begin.getKey(), req.end.getKey()))));
															#line 3600 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (EXPENSIVE_VALIDATION)
															#line 12156 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3601 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = 0;i < r.data.size();i++) {
															#line 3602 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(r.data[i].key >= begin && r.data[i].key < end);
															#line 12162 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3604 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(r.data.size() <= std::abs(req.limit));
															#line 12166 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3608 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		int64_t totalByteSize = 0;
															#line 3609 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < r.data.size();i++) {
															#line 3610 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			totalByteSize += r.data[i].expectedSize();
															#line 12174 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3613 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.taskID.present() && req.taskID.get() == TaskPriority::FetchKeys)
															#line 12178 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3614 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchServed += r.data.size();
															#line 3615 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchBytesServed += (totalByteSize + (8 - (int)sizeof(KeyValueRef)) * r.data.size());
															#line 12184 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3618 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (totalByteSize > 0 && SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 12188 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3619 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = std::max(totalByteSize, SERVER_KNOBS->EMPTY_READ_PENALTY) / 2;
															#line 3620 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(r.data[0].key, bytesReadPerKSecond);
															#line 3621 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(r.data[r.data.size() - 1].key, bytesReadPerKSecond);
															#line 12196 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3624 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		r.penalty = data->getPenalty();
															#line 3625 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (g_network->isSimulated())
															#line 12202 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3626 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			maybeInjectConsistencyScanCorruption(data->thisServerID, req, r);
															#line 12206 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3628 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(r);
															#line 3630 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		resultSize = req.limitBytes - remainingLimitBytes;
															#line 3631 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.bytesQueried += resultSize;
															#line 3632 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.rowsQueried += r.data.size();
															#line 3633 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.data.size() == 0)
															#line 12218 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3634 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 12222 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont10cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(GetKeyValuesReply const& _r,int loopDepth) 
	{
		loopDepth = a_body1cont10cont7(_r, loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(GetKeyValuesReply && _r,int loopDepth) 
	{
		loopDepth = a_body1cont10cont7(std::move(_r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont10cont17(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesRequest req;
															#line 3457 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 3458 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t resultSize;
															#line 3474 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	PriorityMultiLock::Lock readLock;
															#line 3477 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double queueWaitEnd;
															#line 3487 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3499 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	uint64_t changeCounter;
															#line 3501 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange shard;
															#line 3522 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int offset1;
															#line 3523 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int offset2;
															#line 3524 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Key> fBegin;
															#line 3527 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Key> fEnd;
															#line 3530 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key begin;
															#line 3531 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key end;
															#line 3582 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int remainingLimitBytes;
															#line 3584 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double kvReadRange;
															#line 12350 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getKeyValuesQ()
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetKeyValuesQActor final : public Actor<Void>, public ActorCallback< GetKeyValuesQActor, 0, Void >, public ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >, public ActorCallback< GetKeyValuesQActor, 2, Version >, public ActorCallback< GetKeyValuesQActor, 3, Key >, public ActorCallback< GetKeyValuesQActor, 4, Key >, public ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >, public FastAllocated<GetKeyValuesQActor>, public GetKeyValuesQActorState<GetKeyValuesQActor> {
															#line 12355 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetKeyValuesQActor>::operator new;
	using FastAllocated<GetKeyValuesQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(597209675464469248UL, 12153846849091392512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetKeyValuesQActor, 0, Void >;
friend struct ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >;
friend struct ActorCallback< GetKeyValuesQActor, 2, Version >;
friend struct ActorCallback< GetKeyValuesQActor, 3, Key >;
friend struct ActorCallback< GetKeyValuesQActor, 4, Key >;
friend struct ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >;
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesQActor(StorageServer* const& data,GetKeyValuesRequest const& req) 
															#line 12377 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetKeyValuesQActorState<GetKeyValuesQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15085295619130978816UL, 4562439640475096832UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getKeyValuesQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 2, Version >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 3, Key >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 4, Key >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getKeyValuesQ( StorageServer* const& data, GetKeyValuesRequest const& req ) {
															#line 3453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetKeyValuesQActor(data, req));
															#line 12415 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 3663 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 12420 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via quickGetKeyValues()
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class QuickGetKeyValuesActor>
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetKeyValuesActorState {
															#line 12427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetKeyValuesActorState(StorageServer* const& data,StringRef const& prefix,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   prefix(prefix),
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   a(a),
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOriginalReq(pOriginalReq),
															#line 3671 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   getRange(),
															#line 3672 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValuesStart(g_network->timer())
															#line 12446 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("quickGetKeyValues", reinterpret_cast<unsigned long>(this));

	}
	~QuickGetKeyValuesActorState() 
	{
		fdb_probe_actor_destroy("quickGetKeyValues", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3673 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.begin = firstGreaterOrEqual(KeyRef(*a, prefix));
															#line 3674 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.end = firstGreaterOrEqual(strinc(prefix, *a));
															#line 3675 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12465 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3676 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.Before");
															#line 12469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			try {
															#line 3681 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				GetKeyValuesRequest req;
															#line 3682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.spanContext = pOriginalReq->spanContext;
															#line 3683 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.options = pOriginalReq->options;
															#line 3684 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.arena = *a;
															#line 3685 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.begin = getRange.begin;
															#line 3686 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.end = getRange.end;
															#line 3687 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.version = version;
															#line 3690 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.limit = SERVER_KNOBS->QUICK_GET_KEY_VALUES_LIMIT;
															#line 3691 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.limitBytes = SERVER_KNOBS->QUICK_GET_KEY_VALUES_LIMIT_BYTES;
															#line 3692 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.options = pOriginalReq->options;
															#line 3694 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.tags = pOriginalReq->tags;
															#line 3695 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.ssLatestCommitVersions = VersionVector();
															#line 3700 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				data->actors.add(getKeyValuesQ(data, req));
															#line 3701 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<GetKeyValuesReply> __when_expr_0 = req.reply.getFuture();
															#line 3701 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12502 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 1;
															#line 3701 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*>(static_cast<QuickGetKeyValuesActor*>(this)));
															#line 12507 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3721 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.quickGetKeyValuesMiss;
															#line 3722 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->QUICK_GET_KEY_VALUES_FALLBACK)
															#line 12538 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3723 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr = Transaction(data->cx);
															#line 3724 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setVersion(version);
															#line 3725 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12546 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3726 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				tr.debugTransaction(pOriginalReq->options.get().debugID.get());
															#line 12550 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3729 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.trState->taskID = TaskPriority::DefaultPromiseEndpoint;
															#line 3730 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Future<RangeResult> rangeResultFuture = tr.getRange(prefixRange(prefix), GetRangeLimits::ROW_LIMIT_UNLIMITED, Snapshot::True);
															#line 3733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_1 = rangeResultFuture;
															#line 3733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12560 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 2;
															#line 3733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*>(static_cast<QuickGetKeyValuesActor*>(this)));
															#line 12565 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 3745 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(quick_get_key_values_miss(), loopDepth);
															#line 12572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(GetKeyValuesReply const& reply,int loopDepth) 
	{
															#line 3702 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 12594 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3703 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetKeyValuesHit;
															#line 3705 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			a->dependsOn(reply.arena);
															#line 3706 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.result = RangeResultRef(reply.data, reply.more);
															#line 3707 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValuesStart;
															#line 3708 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencySamples.sample( duration, ReadLatencySamples::MAPPED_RANGE_LOCAL, trackedReadType(*pOriginalReq));
															#line 3710 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12608 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3711 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterLocalFetch");
															#line 12612 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3714 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 12616 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
			this->~QuickGetKeyValuesActorState();
			static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(GetKeyValuesReply && reply,int loopDepth) 
	{
															#line 3702 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 12630 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3703 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetKeyValuesHit;
															#line 3705 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			a->dependsOn(reply.arena);
															#line 3706 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.result = RangeResultRef(reply.data, reply.more);
															#line 3707 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValuesStart;
															#line 3708 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencySamples.sample( duration, ReadLatencySamples::MAPPED_RANGE_LOCAL, trackedReadType(*pOriginalReq));
															#line 3710 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12644 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3711 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterLocalFetch");
															#line 12648 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3714 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 12652 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
			this->~QuickGetKeyValuesActorState();
			static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(reply, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state > 0) static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetKeyValuesActor*>(this)->ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(RangeResult const& rangeResult,int loopDepth) 
	{
															#line 3734 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		a->dependsOn(rangeResult.arena());
															#line 3735 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		getRange.result = rangeResult;
															#line 3736 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - getValuesStart;
															#line 3737 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample( duration, ReadLatencySamples::MAPPED_RANGE_REMOTE, trackedReadType(*pOriginalReq));
															#line 3739 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12762 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3740 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterRemoteFetch");
															#line 12766 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3743 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 12770 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(RangeResult && rangeResult,int loopDepth) 
	{
															#line 3734 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		a->dependsOn(rangeResult.arena());
															#line 3735 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		getRange.result = rangeResult;
															#line 3736 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - getValuesStart;
															#line 3737 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySamples.sample( duration, ReadLatencySamples::MAPPED_RANGE_REMOTE, trackedReadType(*pOriginalReq));
															#line 3739 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12790 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3740 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterRemoteFetch");
															#line 12794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 3743 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 12798 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult const& rangeResult,int loopDepth) 
	{
		loopDepth = a_body1cont9(rangeResult, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult && rangeResult,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(rangeResult), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state > 0) static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetKeyValuesActor*>(this)->ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StringRef prefix;
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Arena* a;
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetMappedKeyValuesRequest* pOriginalReq;
															#line 3671 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetRangeReqAndResultRef getRange;
															#line 3672 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double getValuesStart;
															#line 3723 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 12897 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via quickGetKeyValues()
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetKeyValuesActor final : public Actor<GetRangeReqAndResultRef>, public ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >, public ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >, public FastAllocated<QuickGetKeyValuesActor>, public QuickGetKeyValuesActorState<QuickGetKeyValuesActor> {
															#line 12902 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<QuickGetKeyValuesActor>::operator new;
	using FastAllocated<QuickGetKeyValuesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15947870667644671488UL, 3344309823605917184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetRangeReqAndResultRef>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >;
friend struct ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >;
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetKeyValuesActor(StorageServer* const& data,StringRef const& prefix,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 12920 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<GetRangeReqAndResultRef>(),
		   QuickGetKeyValuesActorState<QuickGetKeyValuesActor>(data, prefix, version, a, pOriginalReq),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10745188868789411328UL, 14092707623558697728UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("quickGetKeyValues");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetRangeReqAndResultRef> quickGetKeyValues( StorageServer* const& data, StringRef const& prefix, Version const& version, Arena* const& a, GetMappedKeyValuesRequest* const& pOriginalReq ) {
															#line 3664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetRangeReqAndResultRef>(new QuickGetKeyValuesActor(data, prefix, version, a, pOriginalReq));
															#line 12954 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 3748 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

void unpackKeyTuple(Tuple** referenceTuple, Optional<Tuple>& keyTuple, KeyValueRef* keyValue) {
	if (!keyTuple.present()) {
		// May throw exception if the key is not parsable as a tuple.
		try {
			keyTuple = Tuple::unpack(keyValue->key);
		} catch (Error& e) {
			TraceEvent("KeyNotTuple").error(e).detail("Key", keyValue->key.printable());
			throw key_not_tuple();
		}
	}
	*referenceTuple = &keyTuple.get();
}

void unpackValueTuple(Tuple** referenceTuple, Optional<Tuple>& valueTuple, KeyValueRef* keyValue) {
	if (!valueTuple.present()) {
		// May throw exception if the value is not parsable as a tuple.
		try {
			valueTuple = Tuple::unpack(keyValue->value);
		} catch (Error& e) {
			TraceEvent("ValueNotTuple").error(e).detail("Value", keyValue->value.printable());
			throw value_not_tuple();
		}
	}
	*referenceTuple = &valueTuple.get();
}

bool unescapeLiterals(std::string& s, std::string before, std::string after) {
	bool escaped = false;
	size_t p = 0;
	while (true) {
		size_t found = s.find(before, p);
		if (found == std::string::npos) {
			break;
		}
		s.replace(found, before.length(), after);
		p = found + after.length();
		escaped = true;
	}
	return escaped;
}

bool singleKeyOrValue(const std::string& s, size_t sz) {
	// format would be {K[??]} or {V[??]}
	return sz > 5 && s[0] == '{' && (s[1] == 'K' || s[1] == 'V') && s[2] == '[' && s[sz - 2] == ']' && s[sz - 1] == '}';
}

bool rangeQuery(const std::string& s) {
	return s == "{...}";
}

// create a vector of Optional<Tuple>
// in case of a singleKeyOrValue, insert an empty Tuple to vector as placeholder
// in case of a rangeQuery, insert Optional.empty as placeholder
// in other cases, insert the correct Tuple to be used.
void preprocessMappedKey(Tuple& mappedKeyFormatTuple, std::vector<Optional<Tuple>>& vt, bool& isRangeQuery) {
	vt.reserve(mappedKeyFormatTuple.size());

	for (int i = 0; i < mappedKeyFormatTuple.size(); i++) {
		Tuple::ElementType type = mappedKeyFormatTuple.getType(i);
		if (type == Tuple::BYTES || type == Tuple::UTF8) {
			std::string s = mappedKeyFormatTuple.getString(i).toString();
			auto sz = s.size();
			bool escaped = unescapeLiterals(s, "{{", "{");
			escaped = unescapeLiterals(s, "}}", "}") || escaped;
			if (escaped) {
				vt.emplace_back(Tuple::makeTuple(s));
			} else if (singleKeyOrValue(s, sz)) {
				// when it is SingleKeyOrValue, insert an empty Tuple to vector as placeholder
				vt.emplace_back(Tuple());
			} else if (rangeQuery(s)) {
				if (i != mappedKeyFormatTuple.size() - 1) {
					// It must be the last element of the mapper tuple
					throw mapper_bad_range_decriptor();
				}
				// when it is rangeQuery, insert Optional.empty as placeholder
				vt.emplace_back(Optional<Tuple>());
				isRangeQuery = true;
			} else {
				Tuple t;
				t.appendRaw(mappedKeyFormatTuple.subTupleRawString(i));
				vt.emplace_back(t);
			}
		} else {
			Tuple t;
			t.appendRaw(mappedKeyFormatTuple.subTupleRawString(i));
			vt.emplace_back(t);
		}
	}
}

Key constructMappedKey(KeyValueRef* keyValue, std::vector<Optional<Tuple>>& vec, Tuple& mappedKeyFormatTuple) {
	// Lazily parse key and/or value to tuple because they may not need to be a tuple if not used.
	Optional<Tuple> keyTuple;
	Optional<Tuple> valueTuple;
	Tuple mappedKeyTuple;

	mappedKeyTuple.reserve(vec.size());

	for (int i = 0; i < vec.size(); i++) {
		if (!vec[i].present()) {
			// rangeQuery
			continue;
		}
		if (vec[i].get().size()) {
			mappedKeyTuple.append(vec[i].get());
		} else {
			// singleKeyOrValue is true
			std::string s = mappedKeyFormatTuple.getString(i).toString();
			auto sz = s.size();
			int idx;
			Tuple* referenceTuple;
			try {
				idx = std::stoi(s.substr(3, sz - 5));
			} catch (std::exception& e) {
				throw mapper_bad_index();
			}
			if (s[1] == 'K') {
				unpackKeyTuple(&referenceTuple, keyTuple, keyValue);
			} else if (s[1] == 'V') {
				unpackValueTuple(&referenceTuple, valueTuple, keyValue);
			} else {
				ASSERT(false);
				throw internal_error();
			}
			if (idx < 0 || idx >= referenceTuple->size()) {
				throw mapper_bad_index();
			}
			mappedKeyTuple.appendRaw(referenceTuple->subTupleRawString(idx));
		}
	}

	return mappedKeyTuple.pack();
}

struct AuditGetShardInfoRes {
	Version readAtVersion;
	UID serverId;
	std::vector<KeyRange> ownRanges;
	AuditGetShardInfoRes() = default;
	AuditGetShardInfoRes(Version readAtVersion, UID serverId, std::vector<KeyRange> ownRanges)
	  : readAtVersion(readAtVersion), serverId(serverId), ownRanges(ownRanges) {}
};

// Given an input server, get ranges with in the input range
// from the perspective of SS->shardInfo
// Input: (1) SS ID; (2) within range
// Return AuditGetShardInfoRes including: (1) version of the read; (2) ranges of the SS
AuditGetShardInfoRes getThisServerShardInfo(StorageServer* data, KeyRange range) {
	std::vector<KeyRange> ownRange;
	for (auto& t : data->shards.intersectingRanges(range)) {
		KeyRange alignedRange = t.value()->range() & range;
		if (alignedRange.empty()) {
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways,
			           "SSAuditStorageReadShardInfoEmptyAlignedRange",
			           data->thisServerID)
			    .detail("Range", range);
			throw audit_storage_cancelled();
		}
		TraceEvent(SevVerbose, "SSAuditStorageGetThisServerShardInfo", data->thisServerID)
		    .detail("AlignedRange", alignedRange)
		    .detail("Range", t.value()->range())
		    .detail("AtVersion", data->version.get())
		    .detail("AuditServer", data->thisServerID)
		    .detail("ReadWrite", t.value()->isReadable() ? "True" : "False")
		    .detail("Adding", t.value()->isFetching() ? "True" : "False");
		if (t.value()->assigned()) {
			ownRange.push_back(alignedRange);
		}
	}
	return AuditGetShardInfoRes(data->version.get(), data->thisServerID, ownRange);
}

// Check consistency between StorageServer->shardInfo and ServerKeys system key space
															#line 13132 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via auditStorageServerShardQ()
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class AuditStorageServerShardQActor>
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class AuditStorageServerShardQActorState {
															#line 13139 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageServerShardQActorState(StorageServer* const& data,AuditStorageRequest const& req) 
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 13148 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this));

	}
	~AuditStorageServerShardQActorState() 
	{
		fdb_probe_actor_destroy("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3923 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(req.getType() == AuditType::ValidateStorageServerShard);
															#line 3924 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->serveAuditStorageParallelismLock.take(TaskPriority::DefaultYield);
															#line 3924 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13167 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 1;
															#line 3924 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 0, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13172 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AuditStorageServerShardQActorState();
		static_cast<AuditStorageServerShardQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3929 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->trackShardAssignmentMinVersion != invalidVersion)
															#line 13195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3931 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(audit_storage_cancelled());
															#line 3932 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "ExistStorageServerShardAuditExit") .detail("NewAuditId", req.id) .detail("NewAuditType", req.getType());
															#line 3936 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<AuditStorageServerShardQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageServerShardQActorState(); static_cast<AuditStorageServerShardQActor*>(this)->destroy(); return 0; }
															#line 13203 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<AuditStorageServerShardQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~AuditStorageServerShardQActorState();
			static_cast<AuditStorageServerShardQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 3938 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveAuditStorageParallelismLock);
															#line 3939 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageSsShardBegin", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range);
															#line 3942 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		res = AuditStorageState(req.id, data->thisServerID, req.getType());
															#line 3943 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		errors = std::vector<std::string>();
															#line 3944 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs = std::vector<Future<Void>>();
															#line 3945 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 3946 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyRes = AuditGetServerKeysRes();
															#line 3947 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = Version();
															#line 3948 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = KeyRange();
															#line 3949 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerRes = AuditGetKeyServersRes();
															#line 3950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = Version();
															#line 3951 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = KeyRange();
															#line 3952 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesLocalViewRes = AuditGetShardInfoRes();
															#line 3953 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localShardInfoReadAtVersion = Version();
															#line 3956 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByKeyServerMap = std::unordered_map<UID, std::vector<KeyRange>>();
															#line 3957 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByServerKey = std::vector<KeyRange>();
															#line 3958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByKeyServer = std::vector<KeyRange>();
															#line 3959 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesLocalView = std::vector<KeyRange>();
															#line 3960 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		failureReason = std::string();
															#line 3966 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = KeyRange();
															#line 3967 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToReadBegin = req.range.begin;
															#line 3968 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToRead = KeyRangeRef();
															#line 3969 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount = 0;
															#line 3970 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedLocalShardsNum = 0;
															#line 3971 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedServerKeysNum = 0;
															#line 3972 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiter = Reference<IRateControl>(new SpeedLimit(SERVER_KNOBS->AUDIT_STORAGE_RATE_PER_SERVER_MAX, 1));
															#line 3974 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = 0;
															#line 3975 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		startTime = now();
															#line 3976 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = 0;
															#line 3977 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = 0;
															#line 3978 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = 0;
															#line 13271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 3981 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 13275 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3929 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->trackShardAssignmentMinVersion != invalidVersion)
															#line 13290 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 3931 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(audit_storage_cancelled());
															#line 3932 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "ExistStorageServerShardAuditExit") .detail("NewAuditId", req.id) .detail("NewAuditType", req.getType());
															#line 3936 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<AuditStorageServerShardQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageServerShardQActorState(); static_cast<AuditStorageServerShardQActor*>(this)->destroy(); return 0; }
															#line 13298 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			new (&static_cast<AuditStorageServerShardQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~AuditStorageServerShardQActorState();
			static_cast<AuditStorageServerShardQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 3938 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveAuditStorageParallelismLock);
															#line 3939 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageSsShardBegin", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range);
															#line 3942 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		res = AuditStorageState(req.id, data->thisServerID, req.getType());
															#line 3943 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		errors = std::vector<std::string>();
															#line 3944 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs = std::vector<Future<Void>>();
															#line 3945 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 3946 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyRes = AuditGetServerKeysRes();
															#line 3947 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = Version();
															#line 3948 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = KeyRange();
															#line 3949 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerRes = AuditGetKeyServersRes();
															#line 3950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = Version();
															#line 3951 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = KeyRange();
															#line 3952 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesLocalViewRes = AuditGetShardInfoRes();
															#line 3953 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localShardInfoReadAtVersion = Version();
															#line 3956 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByKeyServerMap = std::unordered_map<UID, std::vector<KeyRange>>();
															#line 3957 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByServerKey = std::vector<KeyRange>();
															#line 3958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByKeyServer = std::vector<KeyRange>();
															#line 3959 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesLocalView = std::vector<KeyRange>();
															#line 3960 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		failureReason = std::string();
															#line 3966 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = KeyRange();
															#line 3967 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToReadBegin = req.range.begin;
															#line 3968 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToRead = KeyRangeRef();
															#line 3969 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount = 0;
															#line 3970 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedLocalShardsNum = 0;
															#line 3971 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedServerKeysNum = 0;
															#line 3972 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiter = Reference<IRateControl>(new SpeedLimit(SERVER_KNOBS->AUDIT_STORAGE_RATE_PER_SERVER_MAX, 1));
															#line 3974 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = 0;
															#line 3975 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		startTime = now();
															#line 3976 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = 0;
															#line 3977 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = 0;
															#line 3978 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = 0;
															#line 13366 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 3981 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 13370 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 4336 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->stopTrackShardAssignment();
															#line 4337 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevVerbose, "SSShardAssignmentHistoryRecordStopWhenExit", data->thisServerID).detail("AuditID", req.id);
															#line 4339 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<AuditStorageServerShardQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageServerShardQActorState(); static_cast<AuditStorageServerShardQActor*>(this)->destroy(); return 0; }
															#line 13464 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<AuditStorageServerShardQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AuditStorageServerShardQActorState();
		static_cast<AuditStorageServerShardQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4310 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 13477 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4311 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<AuditStorageServerShardQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageServerShardQActorState(); static_cast<AuditStorageServerShardQActor*>(this)->destroy(); return 0; }
															#line 13481 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<AuditStorageServerShardQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~AuditStorageServerShardQActorState();
				static_cast<AuditStorageServerShardQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 4313 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardFailed", data->thisServerID) .errorUnsuppressed(e) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("Reason", failureReason) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 4322 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->stopTrackShardAssignment();
															#line 4323 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevVerbose, "SSShardAssignmentHistoryRecordStopWhenError", data->thisServerID) .detail("AuditID", req.id);
															#line 4326 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_audit_storage_cancelled)
															#line 13495 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4327 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(audit_storage_cancelled());
															#line 13499 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 4328 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (e.code() == error_code_audit_storage_task_outdated)
															#line 13505 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4329 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(audit_storage_task_outdated());
															#line 13509 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 4331 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(audit_storage_failed());
															#line 13515 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 3983 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (data->version.get() == 0)
															#line 13546 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 3984 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				failureReason = "SS version is 0";
															#line 3985 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 13552 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 3989 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			errors.clear();
															#line 3991 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesLocalView.clear();
															#line 3992 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesSeenByServerKey.clear();
															#line 3993 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesSeenByKeyServer.clear();
															#line 3994 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesSeenByKeyServerMap.clear();
															#line 3995 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToRead = KeyRangeRef(rangeToReadBegin, req.range.end);
															#line 3997 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(!rangeToRead.empty());
															#line 4001 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesLocalViewRes = getThisServerShardInfo(data, rangeToRead);
															#line 4002 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localShardInfoReadAtVersion = ownRangesLocalViewRes.readAtVersion;
															#line 4003 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (localShardInfoReadAtVersion != data->version.get())
															#line 13574 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4004 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardGRVMismatchError", data->thisServerID);
															#line 4007 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 13580 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4011 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->startTrackShardAssignment(localShardInfoReadAtVersion);
															#line 4012 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevVerbose, "SSShardAssignmentHistoryRecordStart", data->thisServerID) .detail("AuditID", req.id);
															#line 4016 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 4017 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 4018 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.clear();
															#line 4019 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.push_back( store(serverKeyRes, getThisServerKeysFromServerKeys(data->thisServerID, &tr, rangeToRead)));
															#line 4021 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.push_back(store(keyServerRes, getShardMapFromKeyServers(data->thisServerID, &tr, rangeToRead)));
															#line 4022 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = waitForAll(fs);
															#line 4022 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13600 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 2;
															#line 4022 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 1, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13605 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 4304 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = now();
															#line 4305 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_9 = rateLimiter->getAllowance(remoteReadBytes);
															#line 4305 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13637 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_9.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 10;
															#line 4305 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 9, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4293 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 13652 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4298 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1Catch1(e, std::max(0, loopDepth - 1));
															#line 13656 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4300 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->stopTrackShardAssignment();
															#line 4301 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_8 = tr.onError(e);
															#line 4301 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_8.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 9;
															#line 4301 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 8, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 4024 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = serverKeyRes.completeRange;
															#line 4025 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = serverKeyRes.readAtVersion;
															#line 4027 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = keyServerRes.completeRange;
															#line 4028 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = keyServerRes.readAtVersion;
															#line 4030 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = keyServerRes.readBytes + serverKeyRes.readBytes;
															#line 4032 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 13694 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 4024 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = serverKeyRes.completeRange;
															#line 4025 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = serverKeyRes.readAtVersion;
															#line 4027 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = keyServerRes.completeRange;
															#line 4028 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = keyServerRes.readAtVersion;
															#line 4030 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = keyServerRes.readBytes + serverKeyRes.readBytes;
															#line 4032 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 13713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
															#line 4058 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (serverKeyReadAtVersion < localShardInfoReadAtVersion)
															#line 13797 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4059 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardComparedVersionError", data->thisServerID);
															#line 4062 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 13803 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4064 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (keyServerReadAtVersion != serverKeyReadAtVersion)
															#line 13807 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4065 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardKSVersionMismatchError", data->thisServerID);
															#line 4068 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 13813 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		try {
															#line 4071 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_5 = timeoutError(data->version.whenAtLeast(serverKeyReadAtVersion), 30);
															#line 4071 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont5Catch1(actor_cancelled(), loopDepth);
															#line 13820 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1cont5Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 6;
															#line 4071 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 5, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13825 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont5Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont5Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 4032 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!(serverKeyReadAtVersion < localShardInfoReadAtVersion))
															#line 13847 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
			return a_body1cont1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4033 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (retryCount >= SERVER_KNOBS->AUDIT_RETRY_COUNT_MAX)
															#line 13853 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4034 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			failureReason = "Read serverKeys retry count exceeds the max";
															#line 4035 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_failed(), std::max(0, loopDepth - 1));
															#line 13859 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4037 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = rateLimiter->getAllowance(remoteReadBytes);
															#line 4037 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13865 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 3;
															#line 4037 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 2, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13870 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 4038 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount++;
															#line 4039 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0.5);
															#line 4039 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13896 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 4;
															#line 4039 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 3, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 4038 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount++;
															#line 4039 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0.5);
															#line 4039 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13914 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 4;
															#line 4039 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 3, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 4040 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.reset();
															#line 4041 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 4042 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 4043 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.clear();
															#line 4044 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back( store(serverKeyRes, getThisServerKeysFromServerKeys(data->thisServerID, &tr, rangeToRead)));
															#line 4046 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back(store(keyServerRes, getShardMapFromKeyServers(data->thisServerID, &tr, rangeToRead)));
															#line 4047 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(fs);
															#line 4047 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14017 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 5;
															#line 4047 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 4, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14022 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 4040 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.reset();
															#line 4041 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 4042 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 4043 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.clear();
															#line 4044 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back( store(serverKeyRes, getThisServerKeysFromServerKeys(data->thisServerID, &tr, rangeToRead)));
															#line 4046 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back(store(keyServerRes, getShardMapFromKeyServers(data->thisServerID, &tr, rangeToRead)));
															#line 4047 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(fs);
															#line 4047 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14045 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 5;
															#line 4047 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 4, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14050 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont2loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 4049 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = serverKeyRes.completeRange;
															#line 4050 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = serverKeyRes.readAtVersion;
															#line 4052 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = keyServerRes.completeRange;
															#line 4053 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = keyServerRes.readAtVersion;
															#line 4055 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = keyServerRes.readBytes + serverKeyRes.readBytes;
															#line 14142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 4049 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = serverKeyRes.completeRange;
															#line 4050 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = serverKeyRes.readAtVersion;
															#line 4052 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = keyServerRes.completeRange;
															#line 4053 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = keyServerRes.readAtVersion;
															#line 4055 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = keyServerRes.readBytes + serverKeyRes.readBytes;
															#line 14159 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont2loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont2loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont6(int loopDepth) 
	{
															#line 4083 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->stopTrackShardAssignment();
															#line 4084 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevVerbose, "ShardAssignmentHistoryRecordStop", data->thisServerID) .detail("AuditID", req.id);
															#line 4088 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<std::pair<Version, KeyRangeRef>> shardAssignments = data->getShardAssignmentHistory(localShardInfoReadAtVersion, serverKeyReadAtVersion);
															#line 4090 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageSsShardGetHistory", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ServerKeyAtVersion", serverKeyReadAtVersion) .detail("LocalShardInfoAtVersion", localShardInfoReadAtVersion) .detail("ShardAssignmentsCount", shardAssignments.size());
															#line 4098 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!shardAssignments.empty())
															#line 14251 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4099 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			failureReason = "Shard assignment history is not empty";
															#line 4100 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 14257 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4103 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		KeyRange claimRange = rangeToRead;
															#line 4104 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = claimRange & serverKeyCompleteRange;
															#line 4105 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (claimRange.empty())
															#line 14265 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4106 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardOverlapRangeEmpty", data->thisServerID);
															#line 4109 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4111 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = claimRange & keyServerCompleteRange;
															#line 4112 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (claimRange.empty())
															#line 14277 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4113 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardOverlapRangeEmpty", data->thisServerID);
															#line 4116 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4120 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto& range : ownRangesLocalViewRes.ownRanges ) {
															#line 4121 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange overlappingRange = range & claimRange;
															#line 4122 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (overlappingRange.empty())
															#line 14291 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
				continue;
			}
															#line 4125 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesLocalView.push_back(overlappingRange);
															#line 14297 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4128 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto& range : serverKeyRes.ownRanges ) {
															#line 4129 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange overlappingRange = range & claimRange;
															#line 4130 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (overlappingRange.empty())
															#line 14305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
				continue;
			}
															#line 4133 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesSeenByServerKey.push_back(overlappingRange);
															#line 14311 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4136 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (keyServerRes.rangeOwnershipMap.contains(data->thisServerID))
															#line 14315 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4137 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::vector mergedRanges = coalesceRangeList(keyServerRes.rangeOwnershipMap[data->thisServerID]);
															#line 4138 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for( auto& range : mergedRanges ) {
															#line 4139 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyRange overlappingRange = range & claimRange;
															#line 4140 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (overlappingRange.empty())
															#line 14325 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
					continue;
				}
															#line 4143 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ownRangesSeenByKeyServer.push_back(overlappingRange);
															#line 14331 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
		}
															#line 4146 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageSsShardReadDone", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ClaimRange", claimRange) .detail("ServerKeyAtVersion", serverKeyReadAtVersion) .detail("ShardInfoAtVersion", data->version.get());
															#line 4153 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedLocalShardsNum = cumulatedValidatedLocalShardsNum + ownRangesLocalView.size();
															#line 4154 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedServerKeysNum = cumulatedValidatedServerKeysNum + ownRangesSeenByServerKey.size();
															#line 4155 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageStatisticShardInfo", data->thisServerID) .suppressFor(30.0) .detail("AuditType", req.getType()) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("CurrentValidatedLocalShardsNum", ownRangesLocalView.size()) .detail("CurrentValidatedServerKeysNum", ownRangesSeenByServerKey.size()) .detail("CurrentValidatedInclusiveRange", claimRange) .detail("CumulatedValidatedLocalShardsNum", cumulatedValidatedLocalShardsNum) .detail("CumulatedValidatedServerKeysNum", cumulatedValidatedServerKeysNum) .detail("CumulatedValidatedInclusiveRange", KeyRangeRef(req.range.begin, claimRange.end));
															#line 4169 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (ownRangesSeenByKeyServer.empty())
															#line 14344 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4170 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!ownRangesSeenByServerKey.empty())
															#line 14348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4171 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				std::string error = format("ServerKeys shows %zu ranges that not appear on keyServers for Server(%s): ", ownRangesSeenByServerKey.size(), data->thisServerID.toString().c_str(), describe(ownRangesSeenByServerKey).c_str());
															#line 4176 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevError, "SSAuditStorageSsShardError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ClaimRange", claimRange) .detail("ErrorMessage", error) .detail("MismatchedRangeByLocalView", describe(ownRangesSeenByServerKey)) .detail("AuditServer", data->thisServerID);
															#line 14354 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 4187 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Optional<std::pair<KeyRange, KeyRange>> anyMismatch = rangesSame(ownRangesSeenByServerKey, ownRangesSeenByKeyServer);
															#line 4189 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (anyMismatch.present())
															#line 14363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4190 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyRange mismatchedRangeByServerKey = anyMismatch.get().first;
															#line 4191 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyRange mismatchedRangeByKeyServer = anyMismatch.get().second;
															#line 4192 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				std::string error = format("KeyServers and serverKeys mismatch on Server(%s): ServerKey: %s; KeyServer: %s", data->thisServerID.toString().c_str(), mismatchedRangeByServerKey.toString().c_str(), mismatchedRangeByKeyServer.toString().c_str());
															#line 4197 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevError, "SSAuditStorageSsShardError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ClaimRange", claimRange) .detail("ErrorMessage", error) .detail("MismatchedRangeByKeyServer", mismatchedRangeByKeyServer) .detail("MismatchedRangeByServerKey", mismatchedRangeByServerKey) .detail("AuditServer", data->thisServerID);
															#line 4207 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				errors.push_back(error);
															#line 14375 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
		}
															#line 4212 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		Optional<std::pair<KeyRange, KeyRange>> anyMismatch = rangesSame(ownRangesSeenByServerKey, ownRangesLocalView);
															#line 4214 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (anyMismatch.present())
															#line 14382 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4215 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange mismatchedRangeByServerKey = anyMismatch.get().first;
															#line 4216 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange mismatchedRangeByLocalView = anyMismatch.get().second;
															#line 4217 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::string error = format("Storage server shard info mismatch on Server(%s): ServerKey: %s; ServerShardInfo: %s", data->thisServerID.toString().c_str(), mismatchedRangeByServerKey.toString().c_str(), mismatchedRangeByLocalView.toString().c_str());
															#line 4222 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevError, "SSAuditStorageSsShardError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ClaimRange", claimRange) .detail("ErrorMessage", error) .detail("MismatchedRangeByLocalView", mismatchedRangeByLocalView) .detail("MismatchedRangeByServerKey", mismatchedRangeByServerKey) .detail("AuditServer", data->thisServerID);
															#line 4232 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			errors.push_back(error);
															#line 14394 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4236 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!errors.empty())
															#line 14398 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4237 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevVerbose, "SSAuditStorageSsShardErrorEnd", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID);
															#line 4241 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.range = claimRange;
															#line 4242 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.setPhase(AuditPhase::Error);
															#line 4243 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!req.ddId.isValid())
															#line 14408 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4244 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardDDIdInvalid", data->thisServerID);
															#line 4247 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14414 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4249 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.ddId = req.ddId;
															#line 4250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_6 = persistAuditStateByServer(data->cx, res);
															#line 4250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14422 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1loopBody1cont6when1(__when_expr_6.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 7;
															#line 4250 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 6, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 4255 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.range = Standalone(KeyRangeRef(req.range.begin, claimRange.end));
															#line 4256 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.setPhase(AuditPhase::Complete);
															#line 4257 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!req.ddId.isValid())
															#line 14438 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4258 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardDDIdInvalid", data->thisServerID);
															#line 4261 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14444 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4263 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.ddId = req.ddId;
															#line 4264 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_7 = persistAuditStateByServer(data->cx, res);
															#line 4264 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14452 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont1loopBody1cont6when2(__when_expr_7.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 8;
															#line 4264 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 7, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14457 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4073 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "SSAuditStorageSsShardWaitSSVersionTooLong", data->thisServerID) .detail("ServerKeyReadAtVersion", serverKeyReadAtVersion) .detail("SSVersion", data->version.get());
															#line 4076 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			failureReason = "SS version takes long time to catch up with serverKeyReadAtVersion";
															#line 4077 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 14472 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont27(Void const& _,int loopDepth) 
	{
															#line 4251 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(audit_storage_error());
															#line 14592 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont27(Void && _,int loopDepth) 
	{
															#line 4251 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(audit_storage_error());
															#line 14601 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont27(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont27(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1cont29(Void const& _,int loopDepth) 
	{
															#line 4265 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (res.range.end < req.range.end)
															#line 14685 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4266 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardPartialDone", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("CompleteRange", res.range) .detail("ClaimRange", claimRange) .detail("RangeToReadEnd", req.range.end) .detail("LastRateLimiterWaitTime", lastRateLimiterWaitTime) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime);
															#line 4276 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToReadBegin = res.range.end;
															#line 14691 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 4278 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(res);
															#line 4279 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardComplete", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("ClaimRange", claimRange) .detail("CompleteRange", res.range) .detail("NumValidatedLocalShards", cumulatedValidatedLocalShardsNum) .detail("NumValidatedServerKeys", cumulatedValidatedServerKeysNum) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 14699 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont29(Void && _,int loopDepth) 
	{
															#line 4265 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (res.range.end < req.range.end)
															#line 14710 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4266 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardPartialDone", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("CompleteRange", res.range) .detail("ClaimRange", claimRange) .detail("RangeToReadEnd", req.range.end) .detail("LastRateLimiterWaitTime", lastRateLimiterWaitTime) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime);
															#line 4276 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToReadBegin = res.range.end;
															#line 14716 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 4278 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(res);
															#line 4279 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardComplete", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("ClaimRange", claimRange) .detail("CompleteRange", res.range) .detail("NumValidatedLocalShards", cumulatedValidatedLocalShardsNum) .detail("NumValidatedServerKeys", cumulatedValidatedServerKeysNum) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 14724 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont1loopBody1cont11cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont1loopBody1cont31(Void const& _,int loopDepth) 
	{
															#line 4306 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = now() - rateLimiterBeforeWaitTime;
															#line 4307 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = rateLimiterTotalWaitTime + lastRateLimiterWaitTime;
															#line 14912 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont31(Void && _,int loopDepth) 
	{
															#line 4306 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = now() - rateLimiterBeforeWaitTime;
															#line 4307 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = rateLimiterTotalWaitTime + lastRateLimiterWaitTime;
															#line 14923 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont31(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont31(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageRequest req;
															#line 3938 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowLock::Releaser holder;
															#line 3942 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageState res;
															#line 3943 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<std::string> errors;
															#line 3944 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<Future<Void>> fs;
															#line 3945 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 3946 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditGetServerKeysRes serverKeyRes;
															#line 3947 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version serverKeyReadAtVersion;
															#line 3948 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange serverKeyCompleteRange;
															#line 3949 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditGetKeyServersRes keyServerRes;
															#line 3950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version keyServerReadAtVersion;
															#line 3951 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange keyServerCompleteRange;
															#line 3952 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditGetShardInfoRes ownRangesLocalViewRes;
															#line 3953 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version localShardInfoReadAtVersion;
															#line 3956 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::unordered_map<UID, std::vector<KeyRange>> ownRangesSeenByKeyServerMap;
															#line 3957 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<KeyRange> ownRangesSeenByServerKey;
															#line 3958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<KeyRange> ownRangesSeenByKeyServer;
															#line 3959 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<KeyRange> ownRangesLocalView;
															#line 3960 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::string failureReason;
															#line 3966 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange claimRange;
															#line 3967 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key rangeToReadBegin;
															#line 3968 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRangeRef rangeToRead;
															#line 3969 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int retryCount;
															#line 3970 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t cumulatedValidatedLocalShardsNum;
															#line 3971 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t cumulatedValidatedServerKeysNum;
															#line 3972 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<IRateControl> rateLimiter;
															#line 3974 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t remoteReadBytes;
															#line 3975 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double startTime;
															#line 3976 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double lastRateLimiterWaitTime;
															#line 3977 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double rateLimiterBeforeWaitTime;
															#line 3978 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double rateLimiterTotalWaitTime;
															#line 15080 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via auditStorageServerShardQ()
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class AuditStorageServerShardQActor final : public Actor<Void>, public ActorCallback< AuditStorageServerShardQActor, 0, Void >, public ActorCallback< AuditStorageServerShardQActor, 1, Void >, public ActorCallback< AuditStorageServerShardQActor, 2, Void >, public ActorCallback< AuditStorageServerShardQActor, 3, Void >, public ActorCallback< AuditStorageServerShardQActor, 4, Void >, public ActorCallback< AuditStorageServerShardQActor, 5, Void >, public ActorCallback< AuditStorageServerShardQActor, 6, Void >, public ActorCallback< AuditStorageServerShardQActor, 7, Void >, public ActorCallback< AuditStorageServerShardQActor, 8, Void >, public ActorCallback< AuditStorageServerShardQActor, 9, Void >, public FastAllocated<AuditStorageServerShardQActor>, public AuditStorageServerShardQActorState<AuditStorageServerShardQActor> {
															#line 15085 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<AuditStorageServerShardQActor>::operator new;
	using FastAllocated<AuditStorageServerShardQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1273068346326933504UL, 3504070585136320768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AuditStorageServerShardQActor, 0, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 1, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 2, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 3, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 4, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 5, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 6, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 7, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 8, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 9, Void >;
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageServerShardQActor(StorageServer* const& data,AuditStorageRequest const& req) 
															#line 15111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   AuditStorageServerShardQActorState<AuditStorageServerShardQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10868456401308189184UL, 5217512031670601728UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("auditStorageServerShardQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 9, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> auditStorageServerShardQ( StorageServer* const& data, AuditStorageRequest const& req ) {
															#line 3922 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new AuditStorageServerShardQActor(data, req));
															#line 15153 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 4341 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 15158 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via auditStorageShardReplicaQ()
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class AuditStorageShardReplicaQActor>
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class AuditStorageShardReplicaQActorState {
															#line 15165 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageShardReplicaQActorState(StorageServer* const& data,AuditStorageRequest const& req) 
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 15174 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this));

	}
	~AuditStorageShardReplicaQActorState() 
	{
		fdb_probe_actor_destroy("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4343 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(req.getType() == AuditType::ValidateHA || req.getType() == AuditType::ValidateReplica);
															#line 4344 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->serveAuditStorageParallelismLock.take(TaskPriority::DefaultYield);
															#line 4344 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15193 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 1;
															#line 4344 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AuditStorageShardReplicaQActorState();
		static_cast<AuditStorageShardReplicaQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 4345 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveAuditStorageParallelismLock);
															#line 4347 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageShardReplicaBegin", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("TargetServers", describe(req.targetServers));
															#line 4353 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		res = AuditStorageState(req.id, req.getType());
															#line 4354 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverListValues = std::vector<Optional<Value>>();
															#line 4355 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs = std::vector<Future<ErrorOr<GetKeyValuesReply>>>();
															#line 4356 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		errors = std::vector<std::string>();
															#line 4357 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		version = Version();
															#line 4358 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 4359 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToRead = req.range;
															#line 4360 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToReadBegin = req.range.begin;
															#line 4361 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = KeyRange();
															#line 4362 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		limit = 1e4;
															#line 4363 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		limitBytes = CLIENT_KNOBS->REPLY_BYTE_LIMIT;
															#line 4364 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readBytes = 0;
															#line 4365 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		numValidatedKeys = 0;
															#line 4366 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		validatedBytes = 0;
															#line 4367 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		complete = false;
															#line 4368 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		checkTimes = 0;
															#line 4369 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		startTime = now();
															#line 4370 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = 0;
															#line 4371 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = 0;
															#line 4372 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = 0;
															#line 4373 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiter = Reference<IRateControl>(new SpeedLimit(SERVER_KNOBS->AUDIT_STORAGE_RATE_PER_SERVER_MAX, 1));
															#line 15265 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 4376 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 15269 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 4345 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveAuditStorageParallelismLock);
															#line 4347 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageShardReplicaBegin", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("TargetServers", describe(req.targetServers));
															#line 4353 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		res = AuditStorageState(req.id, req.getType());
															#line 4354 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		serverListValues = std::vector<Optional<Value>>();
															#line 4355 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs = std::vector<Future<ErrorOr<GetKeyValuesReply>>>();
															#line 4356 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		errors = std::vector<std::string>();
															#line 4357 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		version = Version();
															#line 4358 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 4359 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToRead = req.range;
															#line 4360 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToReadBegin = req.range.begin;
															#line 4361 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = KeyRange();
															#line 4362 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		limit = 1e4;
															#line 4363 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		limitBytes = CLIENT_KNOBS->REPLY_BYTE_LIMIT;
															#line 4364 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readBytes = 0;
															#line 4365 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		numValidatedKeys = 0;
															#line 4366 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		validatedBytes = 0;
															#line 4367 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		complete = false;
															#line 4368 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		checkTimes = 0;
															#line 4369 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		startTime = now();
															#line 4370 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = 0;
															#line 4371 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = 0;
															#line 4372 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = 0;
															#line 4373 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiter = Reference<IRateControl>(new SpeedLimit(SERVER_KNOBS->AUDIT_STORAGE_RATE_PER_SERVER_MAX, 1));
															#line 15328 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 4376 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 15332 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 4751 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<AuditStorageShardReplicaQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageShardReplicaQActorState(); static_cast<AuditStorageShardReplicaQActor*>(this)->destroy(); return 0; }
															#line 15422 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<AuditStorageShardReplicaQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AuditStorageShardReplicaQActorState();
		static_cast<AuditStorageShardReplicaQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4732 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 15435 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4733 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<AuditStorageShardReplicaQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageShardReplicaQActorState(); static_cast<AuditStorageShardReplicaQActor*>(this)->destroy(); return 0; }
															#line 15439 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				new (&static_cast<AuditStorageShardReplicaQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~AuditStorageShardReplicaQActorState();
				static_cast<AuditStorageShardReplicaQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 4735 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageShardReplicaFailed", data->thisServerID) .errorUnsuppressed(e) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 4742 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_audit_storage_cancelled)
															#line 15449 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4743 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(audit_storage_cancelled());
															#line 15453 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 4744 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (e.code() == error_code_audit_storage_task_outdated)
															#line 15459 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4745 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(audit_storage_task_outdated());
															#line 15463 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 4747 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(audit_storage_failed());
															#line 15469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 4378 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			readBytes = 0;
															#line 4379 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToRead = KeyRangeRef(rangeToReadBegin, req.range.end);
															#line 4380 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(!rangeToRead.empty());
															#line 4382 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "SSAuditStorageShardReplicaNewRoundBegin", data->thisServerID) .suppressFor(10.0) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReadRangeBegin", rangeToReadBegin) .detail("ReadRangeEnd", req.range.end);
															#line 4389 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			serverListValues.clear();
															#line 4390 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			errors.clear();
															#line 4391 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.clear();
															#line 4392 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.reset();
															#line 4393 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 4394 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 4395 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 4398 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::vector<Future<Optional<Value>>> serverListEntries;
															#line 4399 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for( const UID& id : req.targetServers ) {
															#line 4400 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (id != data->thisServerID)
															#line 15526 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4401 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					serverListEntries.push_back(tr.get(serverListKeyFor(id)));
															#line 15530 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
			}
															#line 4404 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = store(serverListValues, getAll(serverListEntries));
															#line 4404 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15537 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 2;
															#line 4404 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15542 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 4724 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = now();
															#line 4725 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_7 = rateLimiter->getAllowance(readBytes);
															#line 4725 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15574 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_7.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 8;
															#line 4725 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15579 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4721 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_6 = tr.onError(e);
															#line 4721 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15591 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_6.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 7;
															#line 4721 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15596 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 4407 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(version, tr.getReadVersion());
															#line 4407 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15613 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 3;
															#line 4407 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15618 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 4407 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(version, tr.getReadVersion());
															#line 4407 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15629 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 3;
															#line 4407 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15634 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 4410 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& v : serverListValues ) {
															#line 4411 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!v.present())
															#line 15720 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4412 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaRemoteServerNotFound", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type);
															#line 4416 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 15726 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4418 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StorageServerInterface remoteServer = decodeServerListValue(v.get());
															#line 4419 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			GetKeyValuesRequest req;
															#line 4420 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.begin = firstGreaterOrEqual(rangeToRead.begin);
															#line 4421 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.end = firstGreaterOrEqual(rangeToRead.end);
															#line 4422 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.limit = limit;
															#line 4423 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.limitBytes = limitBytes;
															#line 4424 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.version = version;
															#line 4425 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.tags = TagSet();
															#line 4426 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.push_back(remoteServer.getKeyValues.getReplyUnlessFailedFor(req, 2, 0));
															#line 15746 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4430 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesRequest localReq;
															#line 4431 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.begin = firstGreaterOrEqual(rangeToRead.begin);
															#line 4432 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.end = firstGreaterOrEqual(rangeToRead.end);
															#line 4433 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.limit = limit;
															#line 4434 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.limitBytes = limitBytes;
															#line 4435 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.version = version;
															#line 4436 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.tags = TagSet();
															#line 4437 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->actors.add(getKeyValuesQ(data, localReq));
															#line 4438 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back(errorOr(localReq.reply.getFuture()));
															#line 4439 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<std::vector<ErrorOr<GetKeyValuesReply>>> __when_expr_3 = getAll(fs);
															#line 4439 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15770 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 4;
															#line 4439 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15775 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 4410 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& v : serverListValues ) {
															#line 4411 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!v.present())
															#line 15786 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4412 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaRemoteServerNotFound", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type);
															#line 4416 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 15792 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4418 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StorageServerInterface remoteServer = decodeServerListValue(v.get());
															#line 4419 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			GetKeyValuesRequest req;
															#line 4420 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.begin = firstGreaterOrEqual(rangeToRead.begin);
															#line 4421 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.end = firstGreaterOrEqual(rangeToRead.end);
															#line 4422 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.limit = limit;
															#line 4423 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.limitBytes = limitBytes;
															#line 4424 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.version = version;
															#line 4425 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.tags = TagSet();
															#line 4426 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.push_back(remoteServer.getKeyValues.getReplyUnlessFailedFor(req, 2, 0));
															#line 15812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4430 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesRequest localReq;
															#line 4431 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.begin = firstGreaterOrEqual(rangeToRead.begin);
															#line 4432 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.end = firstGreaterOrEqual(rangeToRead.end);
															#line 4433 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.limit = limit;
															#line 4434 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.limitBytes = limitBytes;
															#line 4435 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.version = version;
															#line 4436 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.tags = TagSet();
															#line 4437 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		data->actors.add(getKeyValuesQ(data, localReq));
															#line 4438 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back(errorOr(localReq.reply.getFuture()));
															#line 4439 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<std::vector<ErrorOr<GetKeyValuesReply>>> __when_expr_3 = getAll(fs);
															#line 4439 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 4;
															#line 4439 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15841 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont6(std::vector<ErrorOr<GetKeyValuesReply>> const& reps,int loopDepth) 
	{
															#line 4443 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < reps.size();++i) {
															#line 4444 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reps[i].isError())
															#line 15927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4445 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaGetKeyValuesError", data->thisServerID) .errorUnsuppressed(reps[i].getError()) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReplyIndex", i) .detail("RangeRead", rangeToRead) .detail("TargetServers", describe(req.targetServers));
															#line 4453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(reps[i].getError(), loopDepth);
															#line 15933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4455 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reps[i].get().error.present())
															#line 15937 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4456 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaGetKeyValuesError", data->thisServerID) .errorUnsuppressed(reps[i].get().error.get()) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReplyIndex", i) .detail("RangeRead", rangeToRead) .detail("TargetServers", describe(req.targetServers));
															#line 4464 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(reps[i].get().error.get(), loopDepth);
															#line 15943 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4466 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			readBytes = readBytes + reps[i].get().data.expectedSize();
															#line 4467 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			validatedBytes = validatedBytes + reps[i].get().data.expectedSize();
															#line 4472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!reps[i].get().more)
															#line 15951 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4473 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				complete = true;
															#line 15955 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
		}
															#line 4478 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = rangeToRead;
															#line 4479 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		const GetKeyValuesReply& local = reps.back().get();
															#line 4480 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (serverListValues.size() != reps.size() - 1)
															#line 15964 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4481 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaRepsLengthWrong", data->thisServerID) .detail("ServerListValuesSize", serverListValues.size()) .detail("RepsSize", reps.size());
															#line 4486 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 15970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4488 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reps.size() == 1)
															#line 15974 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4490 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "SSAuditStorageShardReplicaNothingToCompare", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("TargetServers", describe(req.targetServers));
															#line 4495 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			complete = true;
															#line 15980 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4499 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int repIdx = 0;repIdx < reps.size() - 1;repIdx++) {
															#line 4500 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const GetKeyValuesReply& remote = reps[repIdx].get();
															#line 4502 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!serverListValues[repIdx].present())
															#line 15988 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4503 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaRepIdxNotPresent", data->thisServerID) .detail("RepIdx", repIdx);
															#line 4507 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 15994 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4509 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const StorageServerInterface& remoteServer = decodeServerListValue(serverListValues[repIdx].get());
															#line 4510 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Key lastKey = rangeToRead.begin;
															#line 4511 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const int end = std::min(local.data.size(), remote.data.size());
															#line 4512 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			bool missingKey = local.data.size() != remote.data.size();
															#line 4514 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::string error;
															#line 4515 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int i = 0;
															#line 4516 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;i < end;++i) {
															#line 4517 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyValueRef remoteKV = remote.data[i];
															#line 4518 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyValueRef localKV = local.data[i];
															#line 4519 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.range.contains(remoteKV.key) || !req.range.contains(localKV.key))
															#line 16016 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4520 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevWarn, "SSAuditStorageShardReplicaKeyOutOfRange", data->thisServerID) .detail("AuditRange", req.range) .detail("RemoteServer", remoteServer.toString()) .detail("LocalKey", localKV.key) .detail("RemoteKey", remoteKV.key);
															#line 4525 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(wrong_shard_server(), loopDepth);
															#line 16022 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 4528 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (remoteKV.key != localKV.key)
															#line 16026 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4529 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					error = format("Key Mismatch: local server (%016llx): %s, remote server(%016llx) %s", data->thisServerID.first(), Traceable<StringRef>::toString(localKV.key).c_str(), remoteServer.uniqueID.first(), Traceable<StringRef>::toString(remoteKV.key).c_str());
															#line 4534 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4542 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					errors.push_back(error);
															#line 16034 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					break;
				}
				else
				{
															#line 4544 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (remoteKV.value != localKV.value)
															#line 16041 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 4545 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						error = format( "Value Mismatch for Key %s: local server (%016llx): %s, remote server(%016llx) %s", Traceable<StringRef>::toString(localKV.key).c_str(), data->thisServerID.first(), Traceable<StringRef>::toString(localKV.value).c_str(), remoteServer.uniqueID.first(), Traceable<StringRef>::toString(remoteKV.value).c_str());
															#line 4552 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4560 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						errors.push_back(error);
															#line 16049 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						break;
					}
					else
					{
															#line 4563 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevVerbose, "SSAuditStorageShardReplicaValidatedKey", data->thisServerID) .detail("Key", localKV.key);
															#line 16056 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					}
				}
															#line 4566 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++numValidatedKeys;
															#line 4567 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				lastKey = localKV.key;
															#line 16063 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4569 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange completeRange = Standalone(KeyRangeRef(rangeToRead.begin, keyAfter(lastKey)));
															#line 4570 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (completeRange.empty() || claimRange.begin != completeRange.begin)
															#line 16069 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4571 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaCompleteRangeUnexpected", data->thisServerID) .detail("ClaimRange", claimRange) .detail("CompleteRange", completeRange);
															#line 4576 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16075 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4578 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			claimRange = claimRange & completeRange;
															#line 4579 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!error.empty())
															#line 16081 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
				continue;
			}
															#line 4582 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!local.more && !remote.more && local.data.size() == remote.data.size())
															#line 16087 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
				continue;
			}
			else
			{
															#line 4584 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (i >= local.data.size() && !local.more && i < remote.data.size())
															#line 16095 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4585 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (!missingKey)
															#line 16099 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 4586 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaMissingKeyUnexpected", data->thisServerID);
															#line 16103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					}
															#line 4590 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					std::string error = format("Missing key(s) form local server (%lld), next key: %s, remote server(%016llx) ", data->thisServerID.first(), Traceable<StringRef>::toString(remote.data[i].key).c_str(), remoteServer.uniqueID.first());
															#line 4595 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4603 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					errors.push_back(error);
															#line 16111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					continue;
				}
				else
				{
															#line 4605 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (i >= remote.data.size() && !remote.more && i < local.data.size())
															#line 16118 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 4606 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						if (!missingKey)
															#line 16122 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						{
															#line 4607 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
							TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaMissingKeyUnexpected", data->thisServerID);
															#line 16126 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						}
															#line 4611 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						std::string error = format("Missing key(s) form remote server (%lld), next local server(%016llx) key: %s", remoteServer.uniqueID.first(), data->thisServerID.first(), Traceable<StringRef>::toString(local.data[i].key).c_str());
															#line 4616 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4624 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						errors.push_back(error);
															#line 16134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						continue;
					}
				}
			}
		}
															#line 4629 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (claimRange.end >= req.range.end)
															#line 16142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4630 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			complete = true;
															#line 16146 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4633 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageStatisticValidateReplica", data->thisServerID) .suppressFor(30.0) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("AuditServer", data->thisServerID) .detail("ReplicaServers", req.targetServers) .detail("CheckTimes", checkTimes) .detail("NumValidatedKeys", numValidatedKeys) .detail("CurrentValidatedInclusiveRange", claimRange) .detail("CumulatedValidatedInclusiveRange", KeyRangeRef(req.range.begin, claimRange.end));
															#line 4646 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!errors.empty())
															#line 16152 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4647 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorCount", errors.size()) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4655 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.range = claimRange;
															#line 4656 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.setPhase(AuditPhase::Error);
															#line 4657 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!req.ddId.isValid())
															#line 16162 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4658 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaDDIdInvalid", data->thisServerID);
															#line 4661 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16168 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4663 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.ddId = req.ddId;
															#line 4664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = persistAuditStateByRange(data->cx, res);
															#line 4664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 5;
															#line 4664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16181 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 4668 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (complete || checkTimes % 100 == 0)
															#line 16188 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4669 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (complete)
															#line 16192 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4670 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					res.range = req.range;
															#line 16196 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 4672 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					res.range = Standalone(KeyRangeRef(req.range.begin, claimRange.end));
															#line 16202 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 4674 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				res.setPhase(AuditPhase::Complete);
															#line 4675 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.ddId.isValid())
															#line 16208 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4676 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaDDIdInvalid", data->thisServerID);
															#line 4679 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16214 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 4681 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				res.ddId = req.ddId;
															#line 4682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevInfo, "SSAuditStorageShardReplicaProgressPersist", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("Progress", res.toString());
															#line 4688 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_5 = persistAuditStateByRange(data->cx, res);
															#line 4688 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont6when2(__when_expr_5.get(), loopDepth); };
				static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 6;
															#line 4688 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1loopBody1cont36(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6(std::vector<ErrorOr<GetKeyValuesReply>> && reps,int loopDepth) 
	{
															#line 4443 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < reps.size();++i) {
															#line 4444 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reps[i].isError())
															#line 16246 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4445 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaGetKeyValuesError", data->thisServerID) .errorUnsuppressed(reps[i].getError()) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReplyIndex", i) .detail("RangeRead", rangeToRead) .detail("TargetServers", describe(req.targetServers));
															#line 4453 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(reps[i].getError(), loopDepth);
															#line 16252 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4455 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reps[i].get().error.present())
															#line 16256 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4456 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaGetKeyValuesError", data->thisServerID) .errorUnsuppressed(reps[i].get().error.get()) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReplyIndex", i) .detail("RangeRead", rangeToRead) .detail("TargetServers", describe(req.targetServers));
															#line 4464 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(reps[i].get().error.get(), loopDepth);
															#line 16262 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4466 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			readBytes = readBytes + reps[i].get().data.expectedSize();
															#line 4467 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			validatedBytes = validatedBytes + reps[i].get().data.expectedSize();
															#line 4472 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!reps[i].get().more)
															#line 16270 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4473 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				complete = true;
															#line 16274 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
		}
															#line 4478 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = rangeToRead;
															#line 4479 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		const GetKeyValuesReply& local = reps.back().get();
															#line 4480 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (serverListValues.size() != reps.size() - 1)
															#line 16283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4481 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaRepsLengthWrong", data->thisServerID) .detail("ServerListValuesSize", serverListValues.size()) .detail("RepsSize", reps.size());
															#line 4486 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16289 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4488 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reps.size() == 1)
															#line 16293 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4490 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "SSAuditStorageShardReplicaNothingToCompare", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("TargetServers", describe(req.targetServers));
															#line 4495 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			complete = true;
															#line 16299 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4499 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int repIdx = 0;repIdx < reps.size() - 1;repIdx++) {
															#line 4500 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const GetKeyValuesReply& remote = reps[repIdx].get();
															#line 4502 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!serverListValues[repIdx].present())
															#line 16307 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4503 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaRepIdxNotPresent", data->thisServerID) .detail("RepIdx", repIdx);
															#line 4507 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16313 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4509 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const StorageServerInterface& remoteServer = decodeServerListValue(serverListValues[repIdx].get());
															#line 4510 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Key lastKey = rangeToRead.begin;
															#line 4511 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			const int end = std::min(local.data.size(), remote.data.size());
															#line 4512 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			bool missingKey = local.data.size() != remote.data.size();
															#line 4514 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			std::string error;
															#line 4515 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			int i = 0;
															#line 4516 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;i < end;++i) {
															#line 4517 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyValueRef remoteKV = remote.data[i];
															#line 4518 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyValueRef localKV = local.data[i];
															#line 4519 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.range.contains(remoteKV.key) || !req.range.contains(localKV.key))
															#line 16335 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4520 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevWarn, "SSAuditStorageShardReplicaKeyOutOfRange", data->thisServerID) .detail("AuditRange", req.range) .detail("RemoteServer", remoteServer.toString()) .detail("LocalKey", localKV.key) .detail("RemoteKey", remoteKV.key);
															#line 4525 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(wrong_shard_server(), loopDepth);
															#line 16341 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 4528 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (remoteKV.key != localKV.key)
															#line 16345 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4529 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					error = format("Key Mismatch: local server (%016llx): %s, remote server(%016llx) %s", data->thisServerID.first(), Traceable<StringRef>::toString(localKV.key).c_str(), remoteServer.uniqueID.first(), Traceable<StringRef>::toString(remoteKV.key).c_str());
															#line 4534 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4542 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					errors.push_back(error);
															#line 16353 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					break;
				}
				else
				{
															#line 4544 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (remoteKV.value != localKV.value)
															#line 16360 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 4545 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						error = format( "Value Mismatch for Key %s: local server (%016llx): %s, remote server(%016llx) %s", Traceable<StringRef>::toString(localKV.key).c_str(), data->thisServerID.first(), Traceable<StringRef>::toString(localKV.value).c_str(), remoteServer.uniqueID.first(), Traceable<StringRef>::toString(remoteKV.value).c_str());
															#line 4552 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4560 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						errors.push_back(error);
															#line 16368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						break;
					}
					else
					{
															#line 4563 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevVerbose, "SSAuditStorageShardReplicaValidatedKey", data->thisServerID) .detail("Key", localKV.key);
															#line 16375 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					}
				}
															#line 4566 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				++numValidatedKeys;
															#line 4567 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				lastKey = localKV.key;
															#line 16382 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4569 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange completeRange = Standalone(KeyRangeRef(rangeToRead.begin, keyAfter(lastKey)));
															#line 4570 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (completeRange.empty() || claimRange.begin != completeRange.begin)
															#line 16388 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4571 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaCompleteRangeUnexpected", data->thisServerID) .detail("ClaimRange", claimRange) .detail("CompleteRange", completeRange);
															#line 4576 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16394 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4578 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			claimRange = claimRange & completeRange;
															#line 4579 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!error.empty())
															#line 16400 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
				continue;
			}
															#line 4582 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!local.more && !remote.more && local.data.size() == remote.data.size())
															#line 16406 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
				continue;
			}
			else
			{
															#line 4584 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (i >= local.data.size() && !local.more && i < remote.data.size())
															#line 16414 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4585 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (!missingKey)
															#line 16418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 4586 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaMissingKeyUnexpected", data->thisServerID);
															#line 16422 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					}
															#line 4590 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					std::string error = format("Missing key(s) form local server (%lld), next key: %s, remote server(%016llx) ", data->thisServerID.first(), Traceable<StringRef>::toString(remote.data[i].key).c_str(), remoteServer.uniqueID.first());
															#line 4595 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4603 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					errors.push_back(error);
															#line 16430 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					continue;
				}
				else
				{
															#line 4605 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					if (i >= remote.data.size() && !remote.more && i < local.data.size())
															#line 16437 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					{
															#line 4606 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						if (!missingKey)
															#line 16441 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						{
															#line 4607 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
							TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaMissingKeyUnexpected", data->thisServerID);
															#line 16445 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						}
															#line 4611 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						std::string error = format("Missing key(s) form remote server (%lld), next local server(%016llx) key: %s", remoteServer.uniqueID.first(), data->thisServerID.first(), Traceable<StringRef>::toString(local.data[i].key).c_str());
															#line 4616 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4624 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
						errors.push_back(error);
															#line 16453 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
						continue;
					}
				}
			}
		}
															#line 4629 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (claimRange.end >= req.range.end)
															#line 16461 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4630 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			complete = true;
															#line 16465 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4633 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageStatisticValidateReplica", data->thisServerID) .suppressFor(30.0) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("AuditServer", data->thisServerID) .detail("ReplicaServers", req.targetServers) .detail("CheckTimes", checkTimes) .detail("NumValidatedKeys", numValidatedKeys) .detail("CurrentValidatedInclusiveRange", claimRange) .detail("CumulatedValidatedInclusiveRange", KeyRangeRef(req.range.begin, claimRange.end));
															#line 4646 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!errors.empty())
															#line 16471 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4647 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorCount", errors.size()) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4655 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.range = claimRange;
															#line 4656 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.setPhase(AuditPhase::Error);
															#line 4657 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!req.ddId.isValid())
															#line 16481 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4658 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaDDIdInvalid", data->thisServerID);
															#line 4661 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4663 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			res.ddId = req.ddId;
															#line 4664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = persistAuditStateByRange(data->cx, res);
															#line 4664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16495 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 5;
															#line 4664 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16500 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 4668 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (complete || checkTimes % 100 == 0)
															#line 16507 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4669 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (complete)
															#line 16511 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4670 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					res.range = req.range;
															#line 16515 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 4672 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					res.range = Standalone(KeyRangeRef(req.range.begin, claimRange.end));
															#line 16521 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 4674 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				res.setPhase(AuditPhase::Complete);
															#line 4675 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.ddId.isValid())
															#line 16527 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				{
															#line 4676 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaDDIdInvalid", data->thisServerID);
															#line 4679 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16533 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				}
															#line 4681 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				res.ddId = req.ddId;
															#line 4682 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevInfo, "SSAuditStorageShardReplicaProgressPersist", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("Progress", res.toString());
															#line 4688 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_5 = persistAuditStateByRange(data->cx, res);
															#line 4688 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16543 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont6when2(__when_expr_5.get(), loopDepth); };
				static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 6;
															#line 4688 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16548 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1loopBody1cont36(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(std::vector<ErrorOr<GetKeyValuesReply>> const& reps,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(reps, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(std::vector<ErrorOr<GetKeyValuesReply>> && reps,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(std::move(reps), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*,std::vector<ErrorOr<GetKeyValuesReply>> const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*,std::vector<ErrorOr<GetKeyValuesReply>> && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont41(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont34(Void const& _,int loopDepth) 
	{
															#line 4665 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(audit_storage_error());
															#line 16644 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont34(Void && _,int loopDepth) 
	{
															#line 4665 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(audit_storage_error());
															#line 16653 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont34(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont34(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont36(int loopDepth) 
	{
															#line 4691 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (complete)
															#line 16737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4692 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(res);
															#line 4693 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageShardReplicaComplete", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("ReplicaServers", req.targetServers) .detail("ClaimRange", claimRange) .detail("CompleteRange", res.range) .detail("CheckTimes", checkTimes) .detail("NumValidatedKeys", numValidatedKeys) .detail("ValidatedBytes", validatedBytes) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 16743 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		else
		{
															#line 4707 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageShardReplicaPartialDone", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("ReplicaServers", req.targetServers) .detail("ClaimRange", claimRange) .detail("CompleteRange", res.range) .detail("LastRateLimiterWaitTime", lastRateLimiterWaitTime) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime);
															#line 4717 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToReadBegin = claimRange.end;
															#line 16752 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont37(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont36(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont37(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont36(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont37(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont37(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont41(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1cont42(Void const& _,int loopDepth) 
	{
															#line 4726 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = now() - rateLimiterBeforeWaitTime;
															#line 4727 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = rateLimiterTotalWaitTime + lastRateLimiterWaitTime;
															#line 4728 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		++checkTimes;
															#line 16953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont42(Void && _,int loopDepth) 
	{
															#line 4726 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = now() - rateLimiterBeforeWaitTime;
															#line 4727 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = rateLimiterTotalWaitTime + lastRateLimiterWaitTime;
															#line 4728 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		++checkTimes;
															#line 16966 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont42(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont42(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageRequest req;
															#line 4345 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowLock::Releaser holder;
															#line 4353 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageState res;
															#line 4354 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<Optional<Value>> serverListValues;
															#line 4355 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<Future<ErrorOr<GetKeyValuesReply>>> fs;
															#line 4356 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<std::string> errors;
															#line 4357 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 4358 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 4359 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange rangeToRead;
															#line 4360 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key rangeToReadBegin;
															#line 4361 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange claimRange;
															#line 4362 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int limit;
															#line 4363 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int limitBytes;
															#line 4364 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t readBytes;
															#line 4365 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t numValidatedKeys;
															#line 4366 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t validatedBytes;
															#line 4367 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	bool complete;
															#line 4368 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t checkTimes;
															#line 4369 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double startTime;
															#line 4370 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double lastRateLimiterWaitTime;
															#line 4371 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double rateLimiterBeforeWaitTime;
															#line 4372 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	double rateLimiterTotalWaitTime;
															#line 4373 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<IRateControl> rateLimiter;
															#line 17107 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via auditStorageShardReplicaQ()
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class AuditStorageShardReplicaQActor final : public Actor<Void>, public ActorCallback< AuditStorageShardReplicaQActor, 0, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 1, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 2, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >, public ActorCallback< AuditStorageShardReplicaQActor, 4, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 5, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 6, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 7, Void >, public FastAllocated<AuditStorageShardReplicaQActor>, public AuditStorageShardReplicaQActorState<AuditStorageShardReplicaQActor> {
															#line 17112 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<AuditStorageShardReplicaQActor>::operator new;
	using FastAllocated<AuditStorageShardReplicaQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(962436770010858240UL, 10583971810946573568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AuditStorageShardReplicaQActor, 0, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 1, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 2, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 4, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 5, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 6, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 7, Void >;
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageShardReplicaQActor(StorageServer* const& data,AuditStorageRequest const& req) 
															#line 17136 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   AuditStorageShardReplicaQActorState<AuditStorageShardReplicaQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16484260831731161088UL, 9346140896814425600UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("auditStorageShardReplicaQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> auditStorageShardReplicaQ( StorageServer* const& data, AuditStorageRequest const& req ) {
															#line 4342 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new AuditStorageShardReplicaQActor(data, req));
															#line 17176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 4753 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 17181 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getRangeDataToDump()
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetRangeDataToDumpActor>
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetRangeDataToDumpActorState {
															#line 17188 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetRangeDataToDumpActorState(StorageServer* const& data,KeyRange const& range,Version const& version,std::shared_ptr<RangeDumpRawData> const& output) 
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   output(output),
															#line 4758 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   kvsToDump()
															#line 17203 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getRangeDataToDump", reinterpret_cast<unsigned long>(this));

	}
	~GetRangeDataToDumpActorState() 
	{
		fdb_probe_actor_destroy("getRangeDataToDump", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4759 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			output->kvs.clear();
															#line 4760 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			output->sampled.clear();
															#line 4761 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			sample = std::map<Key, Value>();
															#line 4762 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			output->kvsBytes = 0;
															#line 4763 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			beginKey = range.begin;
															#line 4764 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			output->lastKey = range.begin;
															#line 4765 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			immediateError = true;
															#line 4767 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 17232 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetRangeDataToDumpActorState();
		static_cast<GetRangeDataToDumpActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 4825 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (immediateError)
															#line 17255 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4826 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(retry(), loopDepth);
															#line 17259 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4828 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetRangeDataToDumpActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetRangeDataToDumpActorState(); static_cast<GetRangeDataToDumpActor*>(this)->destroy(); return 0; }
															#line 17263 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<GetRangeDataToDumpActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetRangeDataToDumpActorState();
		static_cast<GetRangeDataToDumpActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 4769 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = ErrorOr<GetKeyValuesReply>();
															#line 17282 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		try {
															#line 4771 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localReq = GetKeyValuesRequest();
															#line 4772 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localReq.begin = firstGreaterOrEqual(beginKey);
															#line 4773 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localReq.end = firstGreaterOrEqual(range.end);
															#line 4774 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localReq.version = version;
															#line 4775 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localReq.limit = SERVER_KNOBS->MOVE_SHARD_KRM_ROW_LIMIT;
															#line 4776 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localReq.limitBytes = SERVER_KNOBS->MOVE_SHARD_KRM_BYTE_LIMIT;
															#line 4777 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localReq.tags = TagSet();
															#line 4778 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			data->actors.add(getKeyValuesQ(data, localReq));
															#line 4779 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(rep, errorOr(localReq.reply.getFuture()));
															#line 4779 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetRangeDataToDumpActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17304 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetRangeDataToDumpActor*>(this)->actor_wait_state = 1;
															#line 4779 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetRangeDataToDumpActor, 0, Void >*>(static_cast<GetRangeDataToDumpActor*>(this)));
															#line 17309 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 4794 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		immediateError = false;
															#line 4797 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& kv : rep.get().data ) {
															#line 4798 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			output->lastKey = kv.key;
															#line 4799 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			auto res = output->kvs.insert({ kv.key, kv.value });
															#line 4800 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(res.second);
															#line 4801 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ByteSampleInfo sampleInfo = isKeyValueInSample(KeyValueRef(kv.key, kv.value));
															#line 4802 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (sampleInfo.inSample)
															#line 17349 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4803 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				auto resSample = output->sampled.insert({ kv.key, BinaryWriter::toValue(sampleInfo.sampledSize, Unversioned()) });
															#line 4805 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(resSample.second);
															#line 17355 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4807 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			output->kvsBytes = output->kvsBytes + kv.expectedSize();
															#line 4808 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (output->kvsBytes >= SERVER_KNOBS->SS_BULKDUMP_BATCH_BYTES)
															#line 17361 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
				break;
			}
		}
															#line 4814 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (output->kvsBytes >= SERVER_KNOBS->SS_BULKDUMP_BATCH_BYTES)
															#line 17368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		else
		{
															#line 4816 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!rep.get().more)
															#line 17376 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4817 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				output->lastKey = range.end;
															#line 17380 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
															#line 4822 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		beginKey = keyAfter(output->lastKey);
															#line 17386 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4787 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 17396 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4788 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 17400 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 4780 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (rep.isError())
															#line 17416 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4781 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1loopBody1Catch1(rep.getError(), loopDepth);
															#line 17420 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4783 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (rep.get().error.present())
															#line 17424 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4784 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1loopBody1Catch1(rep.get().error.get(), loopDepth);
															#line 17428 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 4780 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (rep.isError())
															#line 17438 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4781 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1loopBody1Catch1(rep.getError(), loopDepth);
															#line 17442 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4783 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (rep.get().error.present())
															#line 17446 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4784 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1loopBody1Catch1(rep.get().error.get(), loopDepth);
															#line 17450 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetRangeDataToDumpActor*>(this)->actor_wait_state > 0) static_cast<GetRangeDataToDumpActor*>(this)->actor_wait_state = 0;
		static_cast<GetRangeDataToDumpActor*>(this)->ActorCallback< GetRangeDataToDumpActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetRangeDataToDumpActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRangeDataToDump", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeDataToDumpActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeDataToDump", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRangeDataToDumpActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRangeDataToDump", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeDataToDumpActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeDataToDump", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetRangeDataToDumpActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRangeDataToDump", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeDataToDumpActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeDataToDump", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange range;
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::shared_ptr<RangeDumpRawData> output;
															#line 4758 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::map<Key, Value> kvsToDump;
															#line 4761 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::map<Key, Value> sample;
															#line 4763 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key beginKey;
															#line 4765 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	bool immediateError;
															#line 4769 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	ErrorOr<GetKeyValuesReply> rep;
															#line 4771 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesRequest localReq;
															#line 17564 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getRangeDataToDump()
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class GetRangeDataToDumpActor final : public Actor<Void>, public ActorCallback< GetRangeDataToDumpActor, 0, Void >, public FastAllocated<GetRangeDataToDumpActor>, public GetRangeDataToDumpActorState<GetRangeDataToDumpActor> {
															#line 17569 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetRangeDataToDumpActor>::operator new;
	using FastAllocated<GetRangeDataToDumpActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3742905293383661824UL, 3136652802575597056UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetRangeDataToDumpActor, 0, Void >;
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	GetRangeDataToDumpActor(StorageServer* const& data,KeyRange const& range,Version const& version,std::shared_ptr<RangeDumpRawData> const& output) 
															#line 17586 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetRangeDataToDumpActorState<GetRangeDataToDumpActor>(data, range, version, output),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getRangeDataToDump", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3977532152012596480UL, 1927066618512968192UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeDataToDumpActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getRangeDataToDump");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getRangeDataToDump", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetRangeDataToDumpActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getRangeDataToDump( StorageServer* const& data, KeyRange const& range, Version const& version, std::shared_ptr<RangeDumpRawData> const& output ) {
															#line 4754 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetRangeDataToDumpActor(data, range, version, output));
															#line 17619 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 4830 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// The SS actor handling bulk dump task sent from DD.
// The SS partitions the task range into batches and make progress on each batch one by one.
// Each batch is a subrange of the task range sent from DD.
// When SS completes one batch, SS persists the metadata indicating this batch range completed.
// If the SS fails on dumping a batch data, the SS will send an error to DD and the leftover files
// is cleaned up when this actor returns.
// In the case of SS crashes, the leftover files will be cleared at the init step when the SS restores.
// If the SS uploads any file with succeed but the blob store is actually stored, this inconsistency will
// be captured by DD and DD will retry to dump the problematic range with a new task.
// DD will retry later if it receives any error from SS.
// Upload the data for the range with the following path organization:
//  <rootRemote>/<JobId>/<TaskId>/<batchNum>/<dumpVersion>-manifest.sst
//	<rootRemote>/<JobId>/<TaskId>/<batchNum>/<dumpVersion>-data.sst
//	<rootRemote>/<JobId>/<TaskId>/<batchNum>/<dumpVersion>-sample.sst
// where rootRemote = req.bulkDumpState.remoteRoot, jobId = req.bulkDumpState.jobId, taskId = req.bulkDumpState.taskId,
// batchNum and dumpVersion are dynamically generated.
// Each task must have one manifest file.
// If the task's range is empty, data file and sample file do not exist
// If the task's data size is too small, the sample file may omitted
															#line 17643 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via bulkDumpQ()
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class BulkDumpQActor>
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class BulkDumpQActorState {
															#line 17650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkDumpQActorState(StorageServer* const& data,BulkDumpRequest const& req) 
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 17659 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("bulkDumpQ", reinterpret_cast<unsigned long>(this));

	}
	~BulkDumpQActorState() 
	{
		fdb_probe_actor_destroy("bulkDumpQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4851 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->serveBulkDumpParallelismLock.take(TaskPriority::DefaultYield);
															#line 4851 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17676 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkDumpQActor*>(this)->actor_wait_state = 1;
															#line 4851 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 0, Void >*>(static_cast<BulkDumpQActor*>(this)));
															#line 17681 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkDumpQActorState();
		static_cast<BulkDumpQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 4852 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveBulkDumpParallelismLock);
															#line 4853 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeBegin = req.bulkDumpState.getRange().begin;
															#line 4854 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeEnd = req.bulkDumpState.getRange().end;
															#line 4855 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		transportMethod = req.bulkDumpState.getTransportMethod();
															#line 4856 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		dumpType = req.bulkDumpState.getType();
															#line 4857 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readBytes = 0;
															#line 4858 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount = 0;
															#line 4859 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		batchNum = 0;
															#line 4860 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		versionToDump = Version();
															#line 4861 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeDumpRawData = std::shared_ptr<RangeDumpRawData>();
															#line 4862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		jobId = req.bulkDumpState.getJobId();
															#line 4863 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rootFolderLocal = data->bulkDumpFolder;
															#line 4864 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rootFolderRemote = req.bulkDumpState.getJobRoot();
															#line 4866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(req.bulkDumpState.getTaskId().present());
															#line 4867 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		taskFolder = getBulkDumpJobTaskFolder(jobId, req.bulkDumpState.getTaskId().get());
															#line 4868 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		destinationFileSets = BulkLoadFileSet();
															#line 4869 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 4871 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 17738 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 4852 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveBulkDumpParallelismLock);
															#line 4853 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeBegin = req.bulkDumpState.getRange().begin;
															#line 4854 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeEnd = req.bulkDumpState.getRange().end;
															#line 4855 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		transportMethod = req.bulkDumpState.getTransportMethod();
															#line 4856 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		dumpType = req.bulkDumpState.getType();
															#line 4857 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readBytes = 0;
															#line 4858 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount = 0;
															#line 4859 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		batchNum = 0;
															#line 4860 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		versionToDump = Version();
															#line 4861 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeDumpRawData = std::shared_ptr<RangeDumpRawData>();
															#line 4862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		jobId = req.bulkDumpState.getJobId();
															#line 4863 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rootFolderLocal = data->bulkDumpFolder;
															#line 4864 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rootFolderRemote = req.bulkDumpState.getJobRoot();
															#line 4866 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(req.bulkDumpState.getTaskId().present());
															#line 4867 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		taskFolder = getBulkDumpJobTaskFolder(jobId, req.bulkDumpState.getTaskId().get());
															#line 4868 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		destinationFileSets = BulkLoadFileSet();
															#line 4869 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 4871 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 17781 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state > 0) static_cast<BulkDumpQActor*>(this)->actor_wait_state = 0;
		static_cast<BulkDumpQActor*>(this)->ActorCallback< BulkDumpQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkDumpQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 4995 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		clearFileFolder(abspath(joinPath(rootFolderLocal, taskFolder)), data->thisServerID, true);
															#line 4997 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<BulkDumpQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkDumpQActorState(); static_cast<BulkDumpQActor*>(this)->destroy(); return 0; }
															#line 17867 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<BulkDumpQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkDumpQActorState();
		static_cast<BulkDumpQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 4874 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeDumpRawData = std::make_shared<RangeDumpRawData>();
															#line 4877 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			clearFileFolder(abspath(joinPath(rootFolderLocal, taskFolder)));
															#line 4880 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToDump = Standalone(KeyRangeRef(rangeBegin, rangeEnd));
															#line 4884 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			relativeFolder = joinPath(taskFolder, std::to_string(batchNum));
															#line 4887 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.reset();
															#line 4888 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = store(versionToDump, tr.getReadVersion());
															#line 4888 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17899 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkDumpQActor*>(this)->actor_wait_state = 2;
															#line 4888 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 1, Void >*>(static_cast<BulkDumpQActor*>(this)));
															#line 17904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 4991 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(1.0);
															#line 4991 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<BulkDumpQActor*>(this)->actor_wait_state = 7;
															#line 4991 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 6, Void >*>(static_cast<BulkDumpQActor*>(this)));
															#line 17939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4970 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 17949 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4971 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 17953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
															#line 4973 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "SSBulkDumpError", data->thisServerID) .errorUnsuppressed(e) .detail("TaskID", req.bulkDumpState.getTaskId()) .detail("TaskRange", req.bulkDumpState.getRange()) .detail("JobID", req.bulkDumpState.getJobId()) .detail("RetryCount", retryCount) .detail("BatchNum", batchNum);
															#line 4980 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_bulkdump_task_outdated)
															#line 17959 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4981 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(bulkdump_task_outdated());
															#line 17963 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 4984 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_wrong_shard_server || e.code() == error_code_platform_error || e.code() == error_code_io_error || retryCount >= 50)
															#line 17968 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 4986 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(bulkdump_task_failed());
															#line 17972 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 4989 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			retryCount++;
															#line 17977 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 4892 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = getRangeDataToDump(data, rangeToDump, versionToDump, rangeDumpRawData);
															#line 4892 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17994 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkDumpQActor*>(this)->actor_wait_state = 3;
															#line 4892 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 2, Void >*>(static_cast<BulkDumpQActor*>(this)));
															#line 17999 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 4892 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = getRangeDataToDump(data, rangeToDump, versionToDump, rangeDumpRawData);
															#line 4892 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18010 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkDumpQActor*>(this)->actor_wait_state = 3;
															#line 4892 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 2, Void >*>(static_cast<BulkDumpQActor*>(this)));
															#line 18015 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state > 0) static_cast<BulkDumpQActor*>(this)->actor_wait_state = 0;
		static_cast<BulkDumpQActor*>(this)->ActorCallback< BulkDumpQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkDumpQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 4898 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		resFileSets = getLocalRemoteFileSetSetting(versionToDump, relativeFolder, rootFolderLocal, rootFolderRemote);
															#line 4905 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localFileSetSetting = resFileSets.first;
															#line 4906 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteFileSetSetting = resFileSets.second;
															#line 4909 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		BulkLoadByteSampleSetting byteSampleSetting(0, "hashlittle2", SERVER_KNOBS->BYTE_SAMPLING_FACTOR, SERVER_KNOBS->BYTE_SAMPLING_OVERHEAD, SERVER_KNOBS->MIN_BYTE_SAMPLING_PROBABILITY);
															#line 4916 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		dataRange = rangeToDump & KeyRangeRef(rangeBegin, keyAfter(rangeDumpRawData->lastKey));
															#line 4917 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<BulkLoadManifest> __when_expr_3 = dumpDataFileToLocalDirectory(data->thisServerID, rangeDumpRawData, localFileSetSetting, remoteFileSetSetting, byteSampleSetting, versionToDump, dataRange, dumpType, transportMethod);
															#line 4917 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<BulkDumpQActor*>(this)->actor_wait_state = 4;
															#line 4917 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >*>(static_cast<BulkDumpQActor*>(this)));
															#line 18116 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 4898 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		resFileSets = getLocalRemoteFileSetSetting(versionToDump, relativeFolder, rootFolderLocal, rootFolderRemote);
															#line 4905 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localFileSetSetting = resFileSets.first;
															#line 4906 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteFileSetSetting = resFileSets.second;
															#line 4909 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		BulkLoadByteSampleSetting byteSampleSetting(0, "hashlittle2", SERVER_KNOBS->BYTE_SAMPLING_FACTOR, SERVER_KNOBS->BYTE_SAMPLING_OVERHEAD, SERVER_KNOBS->MIN_BYTE_SAMPLING_PROBABILITY);
															#line 4916 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		dataRange = rangeToDump & KeyRangeRef(rangeBegin, keyAfter(rangeDumpRawData->lastKey));
															#line 4917 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<BulkLoadManifest> __when_expr_3 = dumpDataFileToLocalDirectory(data->thisServerID, rangeDumpRawData, localFileSetSetting, remoteFileSetSetting, byteSampleSetting, versionToDump, dataRange, dumpType, transportMethod);
															#line 4917 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18137 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<BulkDumpQActor*>(this)->actor_wait_state = 4;
															#line 4917 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >*>(static_cast<BulkDumpQActor*>(this)));
															#line 18142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state > 0) static_cast<BulkDumpQActor*>(this)->actor_wait_state = 0;
		static_cast<BulkDumpQActor*>(this)->ActorCallback< BulkDumpQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkDumpQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont4(int loopDepth) 
	{
															#line 4927 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		readBytes = readBytes + rangeDumpRawData->kvsBytes;
															#line 4928 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "SSBulkDumpDataFileGenerated", data->thisServerID) .detail("TaskID", req.bulkDumpState.getTaskId()) .detail("TaskRange", req.bulkDumpState.getRange()) .detail("JobID", req.bulkDumpState.getJobId()) .detail("ChecksumServers", describe(req.checksumServers)) .detail("RangeToDump", rangeToDump) .detail("DataRange", dataRange) .detail("RootFolderLocal", rootFolderLocal) .detail("RelativeFolder", relativeFolder) .detail("DataKeyCount", rangeDumpRawData->kvs.size()) .detail("DataBytes", rangeDumpRawData->kvsBytes) .detail("BatchNum", batchNum) .detail("RemoteFileSet", manifest.getFileSet().toString());
															#line 4943 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		localFileSet = localFileSetSetting;
															#line 4944 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!manifest.hasDataFile())
															#line 18232 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4945 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localFileSet.removeDataFile();
															#line 18236 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4947 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!manifest.hasByteSampleFile())
															#line 18240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4948 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			localFileSet.removeByteSampleFile();
															#line 18244 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		}
															#line 4950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_4 = uploadBulkDumpFileSet( req.bulkDumpState.getTransportMethod(), localFileSet, manifest.getFileSet(), data->thisServerID);
															#line 4950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18250 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<BulkDumpQActor*>(this)->actor_wait_state = 5;
															#line 4950 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 4, Void >*>(static_cast<BulkDumpQActor*>(this)));
															#line 18255 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(BulkLoadManifest const& __manifest,int loopDepth) 
	{
															#line 4917 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		manifest = __manifest;
															#line 18264 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(BulkLoadManifest && __manifest,int loopDepth) 
	{
		manifest = std::move(__manifest);
		loopDepth = a_body1cont1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state > 0) static_cast<BulkDumpQActor*>(this)->actor_wait_state = 0;
		static_cast<BulkDumpQActor*>(this)->ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >::remove();

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >*,BulkLoadManifest const& value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >*,BulkLoadManifest && value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >*,Error err) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 4955 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!dataRange.empty())
															#line 18343 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4957 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(dataRange == manifest.getRange());
															#line 4958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_5 = persistCompleteBulkDumpRange(data->cx, req.bulkDumpState.generateBulkDumpMetadataToPersist(manifest));
															#line 4958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18351 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
			static_cast<BulkDumpQActor*>(this)->actor_wait_state = 6;
															#line 4958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 5, Void >*>(static_cast<BulkDumpQActor*>(this)));
															#line 18356 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 4955 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!dataRange.empty())
															#line 18370 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4957 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(dataRange == manifest.getRange());
															#line 4958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_5 = persistCompleteBulkDumpRange(data->cx, req.bulkDumpState.generateBulkDumpMetadataToPersist(manifest));
															#line 4958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<BulkDumpQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18378 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
			static_cast<BulkDumpQActor*>(this)->actor_wait_state = 6;
															#line 4958 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< BulkDumpQActor, 5, Void >*>(static_cast<BulkDumpQActor*>(this)));
															#line 18383 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state > 0) static_cast<BulkDumpQActor*>(this)->actor_wait_state = 0;
		static_cast<BulkDumpQActor*>(this)->ActorCallback< BulkDumpQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< BulkDumpQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont8(int loopDepth) 
	{
															#line 4963 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeBegin = keyAfter(rangeDumpRawData->lastKey);
															#line 4964 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (rangeBegin >= rangeEnd || batchNum >= SERVER_KNOBS->SS_BULKDUMP_BATCH_COUNT_MAX_PER_REQUEST)
															#line 18474 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		{
															#line 4965 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(req.bulkDumpState);
															#line 18478 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4968 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		batchNum++;
															#line 18483 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopBody1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state > 0) static_cast<BulkDumpQActor*>(this)->actor_wait_state = 0;
		static_cast<BulkDumpQActor*>(this)->ActorCallback< BulkDumpQActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< BulkDumpQActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont13(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont13(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<BulkDumpQActor*>(this)->actor_wait_state > 0) static_cast<BulkDumpQActor*>(this)->actor_wait_state = 0;
		static_cast<BulkDumpQActor*>(this)->ActorCallback< BulkDumpQActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< BulkDumpQActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< BulkDumpQActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkDumpRequest req;
															#line 4852 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowLock::Releaser holder;
															#line 4853 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key rangeBegin;
															#line 4854 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Key rangeEnd;
															#line 4855 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkLoadTransportMethod transportMethod;
															#line 4856 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkLoadType dumpType;
															#line 4857 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t readBytes;
															#line 4858 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	int retryCount;
															#line 4859 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	uint64_t batchNum;
															#line 4860 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Version versionToDump;
															#line 4861 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::shared_ptr<RangeDumpRawData> rangeDumpRawData;
															#line 4862 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	UID jobId;
															#line 4863 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::string rootFolderLocal;
															#line 4864 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::string rootFolderRemote;
															#line 4867 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::string taskFolder;
															#line 4868 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkLoadFileSet destinationFileSets;
															#line 4869 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 4880 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange rangeToDump;
															#line 4884 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::string relativeFolder;
															#line 4898 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	std::pair<BulkLoadFileSet, BulkLoadFileSet> resFileSets;
															#line 4905 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkLoadFileSet localFileSetSetting;
															#line 4906 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkLoadFileSet remoteFileSetSetting;
															#line 4916 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange dataRange;
															#line 4917 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkLoadManifest manifest;
															#line 4943 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkLoadFileSet localFileSet;
															#line 18727 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via bulkDumpQ()
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class BulkDumpQActor final : public Actor<Void>, public ActorCallback< BulkDumpQActor, 0, Void >, public ActorCallback< BulkDumpQActor, 1, Void >, public ActorCallback< BulkDumpQActor, 2, Void >, public ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >, public ActorCallback< BulkDumpQActor, 4, Void >, public ActorCallback< BulkDumpQActor, 5, Void >, public ActorCallback< BulkDumpQActor, 6, Void >, public FastAllocated<BulkDumpQActor>, public BulkDumpQActorState<BulkDumpQActor> {
															#line 18732 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<BulkDumpQActor>::operator new;
	using FastAllocated<BulkDumpQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1456577046743170560UL, 2119913023481647872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkDumpQActor, 0, Void >;
friend struct ActorCallback< BulkDumpQActor, 1, Void >;
friend struct ActorCallback< BulkDumpQActor, 2, Void >;
friend struct ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >;
friend struct ActorCallback< BulkDumpQActor, 4, Void >;
friend struct ActorCallback< BulkDumpQActor, 5, Void >;
friend struct ActorCallback< BulkDumpQActor, 6, Void >;
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	BulkDumpQActor(StorageServer* const& data,BulkDumpRequest const& req) 
															#line 18755 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   BulkDumpQActorState<BulkDumpQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkDumpQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1152782711603816960UL, 11318211526931145216UL);
		ActorExecutionContextHelper __helper(static_cast<BulkDumpQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkDumpQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkDumpQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkDumpQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkDumpQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkDumpQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkDumpQActor, 3, BulkLoadManifest >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< BulkDumpQActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< BulkDumpQActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< BulkDumpQActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> bulkDumpQ( StorageServer* const& data, BulkDumpRequest const& req ) {
															#line 4850 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new BulkDumpQActor(data, req));
															#line 18794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}

#line 4999 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 18799 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase5000()
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FlowTestCase5000Actor>
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class FlowTestCase5000ActorState {
															#line 18806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowTestCase5000ActorState(UnitTestParameters const& params) 
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : params(params)
															#line 18813 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase5000", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase5000ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase5000", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 5001 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Key key = Tuple::makeTuple("key-0"_sr, "key-1"_sr, "key-2"_sr).getDataAsStandalone();
															#line 5002 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Value value = Tuple::makeTuple("value-0"_sr, "value-1"_sr, "value-2"_sr).getDataAsStandalone();
															#line 5003 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			kvr = KeyValueRef(key, value);
															#line 18832 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 5005 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Tuple mappedKeyFormatTuple = Tuple::makeTuple("normal"_sr, "{{escaped}}"_sr, "{K[2]}"_sr, "{V[0]}"_sr, "{...}"_sr);
															#line 5008 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				std::vector<Optional<Tuple>> vt;
															#line 5009 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				bool isRangeQuery = false;
															#line 5010 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				preprocessMappedKey(mappedKeyFormatTuple, vt, isRangeQuery);
															#line 5012 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key mappedKey = constructMappedKey(&kvr, vt, mappedKeyFormatTuple);
															#line 5014 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key expectedMappedKey = Tuple::makeTuple("normal"_sr, "{escaped}"_sr, "key-2"_sr, "value-0"_sr).getDataAsStandalone();
															#line 5017 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(mappedKey.compare(expectedMappedKey) == 0);
															#line 5018 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(isRangeQuery == true);
															#line 18850 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			{
															#line 5022 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Tuple mappedKeyFormatTuple = Tuple::makeTuple("{{{{}}"_sr, "}}"_sr);
															#line 5024 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				std::vector<Optional<Tuple>> vt;
															#line 5025 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				bool isRangeQuery = false;
															#line 5026 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				preprocessMappedKey(mappedKeyFormatTuple, vt, isRangeQuery);
															#line 5027 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key mappedKey = constructMappedKey(&kvr, vt, mappedKeyFormatTuple);
															#line 5029 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key expectedMappedKey = Tuple::makeTuple("{{}"_sr, "}"_sr).getDataAsStandalone();
															#line 5031 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(mappedKey.compare(expectedMappedKey) == 0);
															#line 5032 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(isRangeQuery == false);
															#line 18869 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			{
															#line 5035 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Tuple mappedKeyFormatTuple = Tuple::makeTuple("{{{{}}"_sr, "}}"_sr);
															#line 5037 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				std::vector<Optional<Tuple>> vt;
															#line 5038 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				bool isRangeQuery = false;
															#line 5039 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				preprocessMappedKey(mappedKeyFormatTuple, vt, isRangeQuery);
															#line 5040 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key mappedKey = constructMappedKey(&kvr, vt, mappedKeyFormatTuple);
															#line 5042 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key expectedMappedKey = Tuple::makeTuple("{{}"_sr, "}"_sr).getDataAsStandalone();
															#line 5044 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(mappedKey.compare(expectedMappedKey) == 0);
															#line 5045 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(isRangeQuery == false);
															#line 18888 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			}
			{
															#line 5048 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Tuple mappedKeyFormatTuple = Tuple::makeTuple("{K[100]}"_sr);
															#line 5049 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				throwException = false;
															#line 18895 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				try {
															#line 5051 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					std::vector<Optional<Tuple>> vt;
															#line 5052 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					bool isRangeQuery = false;
															#line 5053 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					preprocessMappedKey(mappedKeyFormatTuple, vt, isRangeQuery);
															#line 5055 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
					Key mappedKey = constructMappedKey(&kvr, vt, mappedKeyFormatTuple);
															#line 18905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
					loopDepth = a_body1cont7(loopDepth);
				}
				catch (Error& error) {
					loopDepth = a_body1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1Catch2(unknown_error(), loopDepth);
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase5000ActorState();
		static_cast<FlowTestCase5000Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
		{
															#line 5063 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Tuple mappedKeyFormatTuple = Tuple::makeTuple("{...}"_sr, "last-element"_sr);
															#line 5064 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			throwException2 = false;
															#line 18938 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			try {
															#line 5066 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				std::vector<Optional<Tuple>> vt;
															#line 5067 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				bool isRangeQuery = false;
															#line 5068 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				preprocessMappedKey(mappedKeyFormatTuple, vt, isRangeQuery);
															#line 5070 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key mappedKey = constructMappedKey(&kvr, vt, mappedKeyFormatTuple);
															#line 18948 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = a_body1cont12(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1cont1Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
															#line 5060 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(throwException);
															#line 18964 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 5057 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(e.code() == error_code_mapper_bad_index);
															#line 5058 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			throwException = true;
															#line 18976 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(int loopDepth) 
	{
		{
															#line 5078 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			Tuple mappedKeyFormatTuple = Tuple::makeTuple("{K[not-a-number]}"_sr);
															#line 5079 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			throwException3 = false;
															#line 19007 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			try {
															#line 5081 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				std::vector<Optional<Tuple>> vt;
															#line 5082 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				bool isRangeQuery = false;
															#line 5083 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				preprocessMappedKey(mappedKeyFormatTuple, vt, isRangeQuery);
															#line 5085 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				Key mappedKey = constructMappedKey(&kvr, vt, mappedKeyFormatTuple);
															#line 19017 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = a_body1cont16(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1cont9Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont9Catch1(unknown_error(), loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
															#line 5075 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(throwException2);
															#line 19033 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 5072 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(e.code() == error_code_mapper_bad_range_decriptor);
															#line 5073 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			throwException2 = true;
															#line 19045 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont13(int loopDepth) 
	{
															#line 5092 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FlowTestCase5000Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase5000ActorState(); static_cast<FlowTestCase5000Actor*>(this)->destroy(); return 0; }
															#line 19073 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase5000Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase5000ActorState();
		static_cast<FlowTestCase5000Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont14(int loopDepth) 
	{
															#line 5090 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(throwException3);
															#line 19085 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont9Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 5087 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(e.code() == error_code_mapper_bad_index);
															#line 5088 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			throwException3 = true;
															#line 19097 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			loopDepth = a_body1cont14(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont16(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont14(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	UnitTestParameters params;
															#line 5003 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyValueRef kvr;
															#line 5049 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	bool throwException;
															#line 5064 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	bool throwException2;
															#line 5079 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	bool throwException3;
															#line 19131 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase5000()
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class FlowTestCase5000Actor final : public Actor<Void>, public FastAllocated<FlowTestCase5000Actor>, public FlowTestCase5000ActorState<FlowTestCase5000Actor> {
															#line 19136 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase5000Actor>::operator new;
	using FastAllocated<FlowTestCase5000Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1280760525901070336UL, 11163078187380147200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowTestCase5000Actor(UnitTestParameters const& params) 
															#line 19152 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase5000ActorState<FlowTestCase5000Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase5000", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12285033626718163712UL, 15943551014899478784UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase5000Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase5000");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase5000", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
static Future<Void> flowTestCase5000( UnitTestParameters const& params ) {
															#line 5000 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new FlowTestCase5000Actor(params));
															#line 19184 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase5000, "/fdbserver/storageserver/constructMappedKey")

#line 5094 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"

// Issues a secondary query (either range and point read) and fills results into "kvm".
															#line 19191 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via mapSubquery()
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
template <class MapSubqueryActor>
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
class MapSubqueryActorState {
															#line 19198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
public:
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
	MapSubqueryActorState(StorageServer* const& data,Version const& version,GetMappedKeyValuesRequest* const& pOriginalReq,Arena* const& pArena,bool const& isRangeQuery,KeyValueRef* const& it,MappedKeyValueRef* const& kvm,Key const& mappedKey) 
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOriginalReq(pOriginalReq),
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   pArena(pArena),
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   isRangeQuery(isRangeQuery),
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   it(it),
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   kvm(kvm),
															#line 5096 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		   mappedKey(mappedKey)
															#line 19219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("mapSubquery", reinterpret_cast<unsigned long>(this));

	}
	~MapSubqueryActorState() 
	{
		fdb_probe_actor_destroy("mapSubquery", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 5104 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
			if (isRangeQuery)
															#line 19234 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
			{
															#line 5106 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<GetRangeReqAndResultRef> __when_expr_0 = quickGetKeyValues(data, mappedKey, version, pArena, pOriginalReq);
															#line 5106 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<MapSubqueryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 19240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<MapSubqueryActor*>(this)->actor_wait_state = 1;
															#line 5106 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MapSubqueryActor, 0, GetRangeReqAndResultRef >*>(static_cast<MapSubqueryActor*>(this)));
															#line 19245 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 5111 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<GetValueReqAndResultRef> __when_expr_1 = quickGetValue(data, mappedKey, version, pArena, pOriginalReq);
															#line 5111 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<MapSubqueryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 19254 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
				static_cast<MapSubqueryActor*>(this)->actor_wait_state = 2;
															#line 5111 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MapSubqueryActor, 1, GetValueReqAndResultRef >*>(static_cast<MapSubqueryActor*>(this)));
															#line 19259 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MapSubqueryActorState();
		static_cast<MapSubqueryActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 5114 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<MapSubqueryActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MapSubqueryActorState(); static_cast<MapSubqueryActor*>(this)->destroy(); return 0; }
															#line 19283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		new (&static_cast<MapSubqueryActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MapSubqueryActorState();
		static_cast<MapSubqueryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(GetRangeReqAndResultRef const& getRange,int loopDepth) 
	{
															#line 5107 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		kvm->key = it->key;
															#line 5108 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		kvm->value = it->value;
															#line 5109 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		kvm->reqAndResult = getRange;
															#line 19299 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(GetRangeReqAndResultRef && getRange,int loopDepth) 
	{
															#line 5107 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		kvm->key = it->key;
															#line 5108 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		kvm->value = it->value;
															#line 5109 "/Users/ec2-user/foundationdb/fdbserver/storageserver.actor.cpp"
		kvm->reqAndResult = getRange;
															#line 19312 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/storageserver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetRangeReqAndResultRef const& getRange,int loopDepth) 
	{
		loopDepth = a_body1cont2(getRange, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetRangeReqAndResultRef && getRange,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(getRange), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MapSubqueryActor*>(this)->actor_wait_state > 0) static_cast<MapSubqueryActor*>(this)->actor_wait_state = 0;
		static_cast<MapSubqueryActor*>(this)->ActorCallback< MapSubqueryActor, 0, GetRangeReqAndResultRef >::remove();

	}
	void a_callback_fire(ActorCallback< MapSubqueryActor, 0, GetRangeReqAndResultRef >*,GetRangeReqAndResultRef const& value) 
	{
		fdb_probe_actor_enter("mapSubquery", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MapSubqueryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_p