#define POST_ACTOR_COMPILER 1
#line 1 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
/*
 * DataDistributionQueue.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <limits>
#include <numeric>
#include <utility>
#include <vector>

#include "fdbserver/DataDistributionTeam.h"
#include "flow/ActorCollection.h"
#include "flow/Buggify.h"
#include "flow/FastRef.h"
#include "flow/Trace.h"
#include "fdbrpc/sim_validation.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbrpc/simulator.h"
#include "fdbserver/DDTxnProcessor.h"
#include "flow/DebugTrace.h"
#include "fdbserver/DDRelocationQueue.h"
#include "flow/actorcompiler.h" // This must be the last #include.

#define WORK_FULL_UTILIZATION 10000 // This is not a knob; it is a fixed point scaling factor!

typedef Reference<IDataDistributionTeam> ITeamRef;
typedef std::pair<ITeamRef, ITeamRef> SrcDestTeamPair;

inline bool isDataMovementForDiskBalancing(DataMovementReason reason) {
	return reason == DataMovementReason::REBALANCE_UNDERUTILIZED_TEAM ||
	       reason == DataMovementReason::REBALANCE_OVERUTILIZED_TEAM;
}

inline bool isDataMovementForReadBalancing(DataMovementReason reason) {
	return reason == DataMovementReason::REBALANCE_READ_OVERUTIL_TEAM ||
	       reason == DataMovementReason::REBALANCE_READ_UNDERUTIL_TEAM;
}

inline bool isDataMovementForMountainChopper(DataMovementReason reason) {
	return reason == DataMovementReason::REBALANCE_OVERUTILIZED_TEAM ||
	       reason == DataMovementReason::REBALANCE_READ_OVERUTIL_TEAM;
}

// FIXME: Always use DataMovementReason to invoke these functions.
inline bool isValleyFillerPriority(int priority) {
	return priority == SERVER_KNOBS->PRIORITY_REBALANCE_UNDERUTILIZED_TEAM ||
	       priority == SERVER_KNOBS->PRIORITY_REBALANCE_READ_UNDERUTIL_TEAM;
}

inline bool isDataMovementForValleyFiller(DataMovementReason reason) {
	return reason == DataMovementReason::REBALANCE_UNDERUTILIZED_TEAM ||
	       reason == DataMovementReason::REBALANCE_READ_UNDERUTIL_TEAM;
}

typedef std::map<DataMovementReason, int> DmReasonPriorityMapping;
typedef std::map<int, DataMovementReason> PriorityDmReasonMapping;
std::pair<const DmReasonPriorityMapping*, const PriorityDmReasonMapping*> buildPriorityMappings() {
	static DmReasonPriorityMapping reasonPriority{
		{ DataMovementReason::INVALID, -1 },
		{ DataMovementReason::RECOVER_MOVE, SERVER_KNOBS->PRIORITY_RECOVER_MOVE },
		{ DataMovementReason::REBALANCE_UNDERUTILIZED_TEAM, SERVER_KNOBS->PRIORITY_REBALANCE_UNDERUTILIZED_TEAM },
		{ DataMovementReason::REBALANCE_OVERUTILIZED_TEAM, SERVER_KNOBS->PRIORITY_REBALANCE_OVERUTILIZED_TEAM },
		{ DataMovementReason::REBALANCE_READ_OVERUTIL_TEAM, SERVER_KNOBS->PRIORITY_REBALANCE_READ_OVERUTIL_TEAM },
		{ DataMovementReason::REBALANCE_READ_UNDERUTIL_TEAM, SERVER_KNOBS->PRIORITY_REBALANCE_READ_UNDERUTIL_TEAM },
		{ DataMovementReason::PERPETUAL_STORAGE_WIGGLE, SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE },
		{ DataMovementReason::TEAM_HEALTHY, SERVER_KNOBS->PRIORITY_TEAM_HEALTHY },
		{ DataMovementReason::TEAM_CONTAINS_UNDESIRED_SERVER, SERVER_KNOBS->PRIORITY_TEAM_CONTAINS_UNDESIRED_SERVER },
		{ DataMovementReason::TEAM_REDUNDANT, SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT },
		{ DataMovementReason::MERGE_SHARD, SERVER_KNOBS->PRIORITY_MERGE_SHARD },
		{ DataMovementReason::POPULATE_REGION, SERVER_KNOBS->PRIORITY_POPULATE_REGION },
		{ DataMovementReason::TEAM_UNHEALTHY, SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY },
		{ DataMovementReason::TEAM_2_LEFT, SERVER_KNOBS->PRIORITY_TEAM_2_LEFT },
		{ DataMovementReason::TEAM_1_LEFT, SERVER_KNOBS->PRIORITY_TEAM_1_LEFT },
		{ DataMovementReason::TEAM_FAILED, SERVER_KNOBS->PRIORITY_TEAM_FAILED },
		{ DataMovementReason::TEAM_0_LEFT, SERVER_KNOBS->PRIORITY_TEAM_0_LEFT },
		{ DataMovementReason::SPLIT_SHARD, SERVER_KNOBS->PRIORITY_SPLIT_SHARD },
		{ DataMovementReason::ENFORCE_MOVE_OUT_OF_PHYSICAL_SHARD,
		  SERVER_KNOBS->PRIORITY_ENFORCE_MOVE_OUT_OF_PHYSICAL_SHARD },
		{ DataMovementReason::REBALANCE_STORAGE_QUEUE, SERVER_KNOBS->PRIORITY_REBALANCE_STORAGE_QUEUE },
		{ DataMovementReason::ASSIGN_EMPTY_RANGE, -2 }, // dummy reason, no corresponding actual data move
		{ DataMovementReason::SEED_SHARD_SERVER, -3 }, // dummy reason, no corresponding actual data move
		{ DataMovementReason::NUMBER_OF_REASONS, -4 }, // dummy reason, no corresponding actual data move
	};

	static PriorityDmReasonMapping priorityReason;
	if (priorityReason.empty()) { // only build once
		for (const auto& [r, p] : reasonPriority) {
			priorityReason[p] = r;
		}
		// Don't allow 2 priorities value being the same.
		if (priorityReason.size() != reasonPriority.size()) {
			TraceEvent(SevError, "DuplicateDataMovementPriority").log();
			ASSERT(false);
		}
	}

	return std::make_pair(&reasonPriority, &priorityReason);
}

// Return negative priority for invalid or dummy reasons
int dataMovementPriority(DataMovementReason reason) {
	auto [reasonPriority, _] = buildPriorityMappings();
	return reasonPriority->at(reason);
}

DataMovementReason priorityToDataMovementReason(int priority) {
	auto [_, priorityReason] = buildPriorityMappings();
	return priorityReason->at(priority);
}

RelocateData::RelocateData()
  : priority(-1), boundaryPriority(-1), healthPriority(-1), reason(RelocateReason::OTHER), startTime(-1),
    dataMoveId(anonymousShardId), workFactor(0), wantsNewServers(false), cancellable(false),
    interval("QueuedRelocation") {};

RelocateData::RelocateData(RelocateShard const& rs)
  : parent_range(rs.getParentRange()), keys(rs.keys), priority(rs.priority),
    boundaryPriority(isBoundaryPriority(rs.priority) ? rs.priority : -1),
    healthPriority(isHealthPriority(rs.priority) ? rs.priority : -1), reason(rs.reason), dmReason(rs.moveReason),
    startTime(now()), randomId(rs.traceId.isValid() ? rs.traceId : deterministicRandom()->randomUniqueID()),
    dataMoveId(rs.dataMoveId), workFactor(0),
    wantsNewServers(isDataMovementForMountainChopper(rs.moveReason) || isDataMovementForValleyFiller(rs.moveReason) ||
                    rs.moveReason == DataMovementReason::SPLIT_SHARD ||
                    rs.moveReason == DataMovementReason::TEAM_REDUNDANT ||
                    rs.moveReason == DataMovementReason::REBALANCE_STORAGE_QUEUE),
    cancellable(true), interval("QueuedRelocation", randomId), dataMove(rs.dataMove) {
	if (dataMove != nullptr) {
		this->src.insert(this->src.end(), dataMove->meta.src.begin(), dataMove->meta.src.end());
	}
}

bool RelocateData::isRestore() const {
	return this->dataMove != nullptr;
}

// Note: C++ standard library uses the Compare operator, uniqueness is determined by !comp(a, b) && !comp(b, a).
// So operator == and != is not used by std::set<RelocateData, std::greater<RelocateData>>
bool RelocateData::operator>(const RelocateData& rhs) const {
	if (priority != rhs.priority) {
		return priority > rhs.priority;
	} else if (startTime != rhs.startTime) {
		return startTime < rhs.startTime;
	} else if (randomId != rhs.randomId) {
		return randomId > rhs.randomId;
	} else if (keys.begin != rhs.keys.begin) {
		return keys.begin < rhs.keys.begin;
	} else {
		return keys.end < rhs.keys.end;
	}
}

bool RelocateData::operator==(const RelocateData& rhs) const {
	return priority == rhs.priority && boundaryPriority == rhs.boundaryPriority &&
	       healthPriority == rhs.healthPriority && reason == rhs.reason && keys == rhs.keys &&
	       startTime == rhs.startTime && workFactor == rhs.workFactor && src == rhs.src &&
	       completeSources == rhs.completeSources && wantsNewServers == rhs.wantsNewServers && randomId == rhs.randomId;
}
bool RelocateData::operator!=(const RelocateData& rhs) const {
	return !(*this == rhs);
}
Optional<KeyRange> RelocateData::getParentRange() const {
	return parent_range;
}

class ParallelTCInfo final : public ReferenceCounted<ParallelTCInfo>, public IDataDistributionTeam {
	std::vector<Reference<IDataDistributionTeam>> teams;
	std::vector<UID> tempServerIDs;

	template <typename NUM>
	NUM sum(std::function<NUM(IDataDistributionTeam const&)> func) const {
		NUM result = 0;
		for (const auto& team : teams) {
			result += func(*team);
		}
		return result;
	}

	template <class T>
	std::vector<T> collect(std::function<std::vector<T>(IDataDistributionTeam const&)> func) const {
		std::vector<T> result;

		for (const auto& team : teams) {
			std::vector<T> newItems = func(*team);
			result.insert(result.end(), newItems.begin(), newItems.end());
		}
		return result;
	}

	bool any(std::function<bool(IDataDistributionTeam const&)> func) const {
		for (const auto& team : teams) {
			if (func(*team)) {
				return true;
			}
		}
		return false;
	}

public:
	ParallelTCInfo() = default;
	explicit ParallelTCInfo(ParallelTCInfo const& info) : teams(info.teams), tempServerIDs(info.tempServerIDs) {};

	void addTeam(Reference<IDataDistributionTeam> team) { teams.push_back(team); }

	void clear() { teams.clear(); }

	bool all(std::function<bool(IDataDistributionTeam const&)> func) const {
		return !any([func](IDataDistributionTeam const& team) { return !func(team); });
	}

	std::vector<StorageServerInterface> getLastKnownServerInterfaces() const override {
		return collect<StorageServerInterface>(
		    [](IDataDistributionTeam const& team) { return team.getLastKnownServerInterfaces(); });
	}

	int size() const override {
		int totalSize = 0;
		for (auto it = teams.begin(); it != teams.end(); it++) {
			totalSize += (*it)->size();
		}
		return totalSize;
	}

	std::vector<UID> const& getServerIDs() const override {
		static std::vector<UID> tempServerIDs;
		tempServerIDs.clear();
		for (const auto& team : teams) {
			std::vector<UID> const& childIDs = team->getServerIDs();
			tempServerIDs.insert(tempServerIDs.end(), childIDs.begin(), childIDs.end());
		}
		return tempServerIDs;
	}

	void addDataInFlightToTeam(int64_t delta) override {
		for (auto& team : teams) {
			team->addDataInFlightToTeam(delta);
		}
	}

	void addReadInFlightToTeam(int64_t delta) override {
		for (auto& team : teams) {
			team->addReadInFlightToTeam(delta);
		}
	}

	int64_t getDataInFlightToTeam() const override {
		return sum<int64_t>([](IDataDistributionTeam const& team) { return team.getDataInFlightToTeam(); });
	}

	int64_t getLoadBytes(bool includeInFlight = true, double inflightPenalty = 1.0) const override {
		return sum<int64_t>([includeInFlight, inflightPenalty](IDataDistributionTeam const& team) {
			return team.getLoadBytes(includeInFlight, inflightPenalty);
		});
	}

	int64_t getReadInFlightToTeam() const override {
		return sum<int64_t>([](IDataDistributionTeam const& team) { return team.getReadInFlightToTeam(); });
	}

	double getReadLoad(bool includeInFlight = true, double inflightPenalty = 1.0) const override {
		return sum<double>([includeInFlight, inflightPenalty](IDataDistributionTeam const& team) {
			return team.getReadLoad(includeInFlight, inflightPenalty);
		});
	}

	double getAverageCPU() const override {
		return sum<double>([](IDataDistributionTeam const& team) { return team.getAverageCPU(); }) / teams.size();
	}

	bool hasLowerCpu(double cpuThreshold) const override {
		return all([cpuThreshold](IDataDistributionTeam const& team) { return team.hasLowerCpu(cpuThreshold); });
	}

	Optional<int64_t> getLongestStorageQueueSize() const override {
		int64_t maxQueueSize = 0;
		for (const auto& team : teams) {
			Optional<int64_t> queueSize = team->getLongestStorageQueueSize();
			if (!queueSize.present()) {
				return Optional<int64_t>();
			}
			maxQueueSize = std::max(maxQueueSize, queueSize.get());
		}
		return maxQueueSize;
	}

	Optional<int> getMaxOngoingBulkLoadTaskCount() const override {
		int maxOngoingBulkLoadTaskCount = 0;
		for (const auto& team : teams) {
			Optional<int> ongoingBulkLoadTaskCount = team->getMaxOngoingBulkLoadTaskCount();
			if (!ongoingBulkLoadTaskCount.present()) {
				// If a SS tracker cannot get the metrics from the SS, it is possible that this SS has some healthy
				// issue. So, return an empty result to avoid choosing this server.
				return Optional<int>();
			}
			maxOngoingBulkLoadTaskCount = std::max(maxOngoingBulkLoadTaskCount, ongoingBulkLoadTaskCount.get());
		}
		return maxOngoingBulkLoadTaskCount;
	}

	int64_t getMinAvailableSpace(bool includeInFlight = true) const override {
		int64_t result = std::numeric_limits<int64_t>::max();
		for (const auto& team : teams) {
			result = std::min(result, team->getMinAvailableSpace(includeInFlight));
		}
		return result;
	}

	double getMinAvailableSpaceRatio(bool includeInFlight = true) const override {
		double result = std::numeric_limits<double>::max();
		for (const auto& team : teams) {
			result = std::min(result, team->getMinAvailableSpaceRatio(includeInFlight));
		}
		return result;
	}

	bool hasHealthyAvailableSpace(double minRatio) const override {
		return all([minRatio](IDataDistributionTeam const& team) { return team.hasHealthyAvailableSpace(minRatio); });
	}

	Future<Void> updateStorageMetrics() override {
		std::vector<Future<Void>> futures;

		for (auto& team : teams) {
			futures.push_back(team->updateStorageMetrics());
		}
		return waitForAll(futures);
	}

	bool isOptimal() const override {
		return all([](IDataDistributionTeam const& team) { return team.isOptimal(); });
	}

	bool isWrongConfiguration() const override {
		return any([](IDataDistributionTeam const& team) { return team.isWrongConfiguration(); });
	}
	void setWrongConfiguration(bool wrongConfiguration) override {
		for (auto it = teams.begin(); it != teams.end(); it++) {
			(*it)->setWrongConfiguration(wrongConfiguration);
		}
	}

	bool isHealthy() const override {
		return all([](IDataDistributionTeam const& team) { return team.isHealthy(); });
	}

	void setHealthy(bool h) override {
		for (auto it = teams.begin(); it != teams.end(); it++) {
			(*it)->setHealthy(h);
		}
	}

	int getPriority() const override {
		int priority = 0;
		for (auto it = teams.begin(); it != teams.end(); it++) {
			priority = std::max(priority, (*it)->getPriority());
		}
		return priority;
	}

	void setPriority(int p) override {
		for (auto it = teams.begin(); it != teams.end(); it++) {
			(*it)->setPriority(p);
		}
	}
	void addref() const override { ReferenceCounted<ParallelTCInfo>::addref(); }
	void delref() const override { ReferenceCounted<ParallelTCInfo>::delref(); }

	void addServers(const std::vector<UID>& servers) override {
		ASSERT(!teams.empty());
		teams[0]->addServers(servers);
	}

	std::string getTeamID() const override {
		std::string id;
		for (int i = 0; i < teams.size(); i++) {
			auto const& team = teams[i];
			id += (i == teams.size() - 1) ? team->getTeamID() : format("%s, ", team->getTeamID().c_str());
		}
		return id;
	}
};

bool Busyness::canLaunch(int prio, int work) const {
	ASSERT(prio > 0 && prio < 1000);
	return ledger[prio / 100] <= WORK_FULL_UTILIZATION - work; // allow for rounding errors in double division
}

void Busyness::addWork(int prio, int work) {
	ASSERT(prio > 0 && prio < 1000);
	for (int i = 0; i <= (prio / 100); i++)
		ledger[i] += work;
}

void Busyness::removeWork(int prio, int work) {
	addWork(prio, -work);
}

std::string Busyness::toString() {
	std::string result;
	for (int i = 1; i < ledger.size();) {
		int j = i + 1;
		while (j < ledger.size() && ledger[i] == ledger[j])
			j++;
		if (i != 1)
			result += ", ";
		result += i + 1 == j ? format("%03d", i * 100) : format("%03d/%03d", i * 100, (j - 1) * 100);
		result += format("=%1.02f (%d/%d)", (float)ledger[i] / WORK_FULL_UTILIZATION, ledger[i], WORK_FULL_UTILIZATION);
		i = j;
	}
	return result;
}

double adjustRelocationParallelismForSrc(double srcParallelism) {
	double res = srcParallelism;
	if (SERVER_KNOBS->ENABLE_CONSERVATIVE_RELOCATION_WHEN_REPLICA_CONSISTENCY_CHECK &&
	    SERVER_KNOBS->ENABLE_REPLICA_CONSISTENCY_CHECK_ON_DATA_MOVEMENT &&
	    srcParallelism >= 1.0 + SERVER_KNOBS->DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS) {
		// DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS is the number of extra replicas that the destination
		// servers will read from the source team.
		res = res / (1.0 + SERVER_KNOBS->DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS);
	}
	ASSERT(res > 0);
	return res;
}

// find the "workFactor" for this, were it launched now
int getSrcWorkFactor(RelocateData const& relocation, int singleRegionTeamSize) {
	// RELOCATION_PARALLELISM_PER_SOURCE_SERVER is the number of concurrent replications that can be launched on a
	// single storage server at a time, given the team size is 1 --- only this storage server is available to serve
	// fetchKey read requests from the dest team.
	// The real parallelism is adjusted by the number of source servers of a source team that can serve
	// fetchKey requests.
	// When ENABLE_REPLICA_CONSISTENCY_CHECK_ON_DATA_MOVEMENT is enabled, the fetchKeys on
	// destination servers will read from DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS + 1 replicas from the source team
	// (suppose the team size is large enough). As a result it is possible that the source team can be overloaded by the
	// fetchKey read requests. This is especially true when the shard split data movements are launched. So, we
	// introduce ENABLE_CONSERVATIVE_RELOCATION_WHEN_REPLICA_CONSISTENCY_CHECK knob to adjust the relocation parallelism
	// accordingly. The adjustment is to reduce the relocation parallelism by a factor of
	// (1 + DATAMOVE_CONSISTENCY_CHECK_REQUIRED_REPLICAS).
	if (relocation.bulkLoadTask.present())
		return 0;
	else if (relocation.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_1_LEFT ||
	         relocation.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_0_LEFT)
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(SERVER_KNOBS->RELOCATION_PARALLELISM_PER_SOURCE_SERVER);
	else if (relocation.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_2_LEFT)
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(2 * SERVER_KNOBS->RELOCATION_PARALLELISM_PER_SOURCE_SERVER);
	else if (relocation.healthPriority == SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE)
		// we want to set PRIORITY_PERPETUAL_STORAGE_WIGGLE to a reasonably large value
		// to make this parallelism take effect
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(SERVER_KNOBS->WIGGLING_RELOCATION_PARALLELISM_PER_SOURCE_SERVER);
	else if (relocation.priority == SERVER_KNOBS->PRIORITY_MERGE_SHARD)
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(SERVER_KNOBS->MERGE_RELOCATION_PARALLELISM_PER_TEAM);
	else { // for now we assume that any message at a lower priority can best be assumed to have a full team left for
		   // work
		return WORK_FULL_UTILIZATION /
		       adjustRelocationParallelismForSrc(singleRegionTeamSize *
		                                         SERVER_KNOBS->RELOCATION_PARALLELISM_PER_SOURCE_SERVER);
	}
}

int getDestWorkFactor() {
	// Work of moving a shard is even across destination servers
	return WORK_FULL_UTILIZATION / SERVER_KNOBS->RELOCATION_PARALLELISM_PER_DEST_SERVER;
}

// Data movement's resource control: Do not overload servers used for the RelocateData
// return true if servers are not too busy to launch the relocation
// This ensure source servers will not be overloaded.
bool canLaunchSrc(RelocateData& relocation,
                  int teamSize,
                  int singleRegionTeamSize,
                  std::map<UID, Busyness>& busymap,
                  std::vector<RelocateData> cancellableRelocations) {
	// assert this has not already been launched
	ASSERT(relocation.workFactor == 0);
	ASSERT(relocation.src.size() != 0);
	ASSERT(teamSize >= singleRegionTeamSize);

	if (relocation.bulkLoadTask.present()) {
		// workFactor for bulk load task on source is always 0, therefore, we can safely launch
		// the data move with a bulk load task
		return true;
	}

	// find the "workFactor" for this, were it launched now
	int workFactor = getSrcWorkFactor(relocation, singleRegionTeamSize);
	int neededServers = std::min<int>(relocation.src.size(), teamSize - singleRegionTeamSize + 1);
	if (SERVER_KNOBS->USE_OLD_NEEDED_SERVERS) {
		neededServers = std::max(1, (int)relocation.src.size() - teamSize + 1);
	}
	// see if each of the SS can launch this task
	for (int i = 0; i < relocation.src.size(); i++) {
		// For each source server for this relocation, copy and modify its busyness to reflect work that WOULD be
		// cancelled
		auto busyCopy = busymap[relocation.src[i]];
		for (int j = 0; j < cancellableRelocations.size(); j++) {
			auto& servers = cancellableRelocations[j].src;
			if (std::count(servers.begin(), servers.end(), relocation.src[i]))
				busyCopy.removeWork(cancellableRelocations[j].priority, cancellableRelocations[j].workFactor);
		}
		// Use this modified busyness to check if this relocation could be launched
		if (busyCopy.canLaunch(relocation.priority, workFactor)) {
			--neededServers;
			if (neededServers == 0)
				return true;
		}
	}

	return false;
}

// candidateTeams is a vector containing one team per datacenter, the team(s) DD is planning on moving the shard to.
bool canLaunchDest(const std::vector<std::pair<Reference<IDataDistributionTeam>, bool>>& candidateTeams,
                   int priority,
                   std::map<UID, Busyness>& busymapDest) {
	// fail switch if this is causing issues
	if (SERVER_KNOBS->RELOCATION_PARALLELISM_PER_DEST_SERVER <= 0) {
		return true;
	}
	int workFactor = getDestWorkFactor();
	for (auto& [team, _] : candidateTeams) {
		for (UID id : team->getServerIDs()) {
			if (!busymapDest[id].canLaunch(priority, workFactor)) {
				return false;
			}
		}
	}
	return true;
}

// update busyness for each server
void launch(RelocateData& relocation, std::map<UID, Busyness>& busymap, int singleRegionTeamSize) {
	// if we are here this means that we can launch and should adjust all the work the servers can do
	relocation.workFactor = getSrcWorkFactor(relocation, singleRegionTeamSize);
	for (int i = 0; i < relocation.src.size(); i++)
		busymap[relocation.src[i]].addWork(relocation.priority, relocation.workFactor);
}

void launchDest(RelocateData& relocation,
                const std::vector<std::pair<Reference<IDataDistributionTeam>, bool>>& candidateTeams,
                std::map<UID, Busyness>& destBusymap) {
	ASSERT(relocation.completeDests.empty());
	int destWorkFactor = getDestWorkFactor();
	for (auto& [team, _] : candidateTeams) {
		for (UID id : team->getServerIDs()) {
			relocation.completeDests.push_back(id);
			destBusymap[id].addWork(relocation.priority, destWorkFactor);
		}
	}
}
void completeDest(RelocateData const& relocation, std::map<UID, Busyness>& destBusymap) {
	int destWorkFactor = getDestWorkFactor();
	for (UID id : relocation.completeDests) {
		destBusymap[id].removeWork(relocation.priority, destWorkFactor);
	}
}

void complete(RelocateData const& relocation, std::map<UID, Busyness>& busymap, std::map<UID, Busyness>& destBusymap) {
	ASSERT(relocation.bulkLoadTask.present() || relocation.workFactor > 0);
	for (int i = 0; i < relocation.src.size(); i++)
		busymap[relocation.src[i]].removeWork(relocation.priority, relocation.workFactor);

	completeDest(relocation, destBusymap);
}

// Cancels in-flight data moves intersecting with range.
															#line 591 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
[[nodiscard]] Future<Void> cancelDataMove( class DDQueue* const& self, KeyRange const& range, const DDEnabledState* const& ddEnabledState );

#line 590 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

															#line 596 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
[[nodiscard]] Future<Void> dataDistributionRelocator( class DDQueue* const& self, RelocateData const& rd, Future<Void> const& prevCleanup, const DDEnabledState* const& ddEnabledState );

#line 595 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

															#line 601 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getSourceServersForRange()
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class GetSourceServersForRangeActor>
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSourceServersForRangeActorState {
															#line 608 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSourceServersForRangeActorState(DDQueue* const& self,RelocateData const& input,PromiseStream<RelocateData> const& output,Reference<FlowLock> const& fetchLock) 
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   input(input),
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   output(output),
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   fetchLock(fetchLock)
															#line 621 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getSourceServersForRange", reinterpret_cast<unsigned long>(this));

	}
	~GetSourceServersForRangeActorState() 
	{
		fdb_probe_actor_destroy("getSourceServersForRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 602 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (input.priority == SERVER_KNOBS->PRIORITY_MERGE_SHARD)
															#line 636 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 603 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_0 = delay(0.5, TaskPriority::DataDistributionVeryLow);
															#line 603 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 1;
															#line 603 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 0, Void >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 647 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 605 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_1 = delay(0.0001, TaskPriority::DataDistributionLaunch);
															#line 605 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 656 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
				static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 2;
															#line 605 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 1, Void >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 661 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetSourceServersForRangeActorState();
		static_cast<GetSourceServersForRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 608 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_2 = fetchLock->take(TaskPriority::DataDistributionLaunch);
															#line 608 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 687 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 3;
															#line 608 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 2, Void >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 692 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state > 0) static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 0;
		static_cast<GetSourceServersForRangeActor*>(this)->ActorCallback< GetSourceServersForRangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetSourceServersForRangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state > 0) static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 0;
		static_cast<GetSourceServersForRangeActor*>(this)->ActorCallback< GetSourceServersForRangeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetSourceServersForRangeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 609 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		releaser = FlowLock::Releaser(*fetchLock);
															#line 611 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<IDDTxnProcessor::SourceServers> __when_expr_3 = self->txnProcessor->getSourceServersForRange(input.keys);
															#line 611 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 879 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 4;
															#line 611 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 884 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 609 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		releaser = FlowLock::Releaser(*fetchLock);
															#line 611 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<IDDTxnProcessor::SourceServers> __when_expr_3 = self->txnProcessor->getSourceServersForRange(input.keys);
															#line 611 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 897 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 4;
															#line 611 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*>(static_cast<GetSourceServersForRangeActor*>(this)));
															#line 902 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state > 0) static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 0;
		static_cast<GetSourceServersForRangeActor*>(this)->ActorCallback< GetSourceServersForRangeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetSourceServersForRangeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(IDDTxnProcessor::SourceServers const& res,int loopDepth) 
	{
															#line 612 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		input.src = std::move(res.srcServers);
															#line 613 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		input.completeSources = std::move(res.completeSources);
															#line 614 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		output.send(input);
															#line 615 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSourceServersForRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetSourceServersForRangeActorState(); static_cast<GetSourceServersForRangeActor*>(this)->destroy(); return 0; }
															#line 992 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<GetSourceServersForRangeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetSourceServersForRangeActorState();
		static_cast<GetSourceServersForRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(IDDTxnProcessor::SourceServers && res,int loopDepth) 
	{
															#line 612 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		input.src = std::move(res.srcServers);
															#line 613 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		input.completeSources = std::move(res.completeSources);
															#line 614 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		output.send(input);
															#line 615 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSourceServersForRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetSourceServersForRangeActorState(); static_cast<GetSourceServersForRangeActor*>(this)->destroy(); return 0; }
															#line 1010 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<GetSourceServersForRangeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetSourceServersForRangeActorState();
		static_cast<GetSourceServersForRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(IDDTxnProcessor::SourceServers const& res,int loopDepth) 
	{
		loopDepth = a_body1cont5(res, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(IDDTxnProcessor::SourceServers && res,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state > 0) static_cast<GetSourceServersForRangeActor*>(this)->actor_wait_state = 0;
		static_cast<GetSourceServersForRangeActor*>(this)->ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >::remove();

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*,IDDTxnProcessor::SourceServers const& value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*,IDDTxnProcessor::SourceServers && value) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*,Error err) 
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RelocateData input;
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PromiseStream<RelocateData> output;
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<FlowLock> fetchLock;
															#line 609 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	FlowLock::Releaser releaser;
															#line 1103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getSourceServersForRange()
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSourceServersForRangeActor final : public Actor<Void>, public ActorCallback< GetSourceServersForRangeActor, 0, Void >, public ActorCallback< GetSourceServersForRangeActor, 1, Void >, public ActorCallback< GetSourceServersForRangeActor, 2, Void >, public ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >, public FastAllocated<GetSourceServersForRangeActor>, public GetSourceServersForRangeActorState<GetSourceServersForRangeActor> {
															#line 1108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetSourceServersForRangeActor>::operator new;
	using FastAllocated<GetSourceServersForRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5327141336778016000UL, 14429565539857505536UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetSourceServersForRangeActor, 0, Void >;
friend struct ActorCallback< GetSourceServersForRangeActor, 1, Void >;
friend struct ActorCallback< GetSourceServersForRangeActor, 2, Void >;
friend struct ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >;
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSourceServersForRangeActor(DDQueue* const& self,RelocateData const& input,PromiseStream<RelocateData> const& output,Reference<FlowLock> const& fetchLock) 
															#line 1128 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetSourceServersForRangeActorState<GetSourceServersForRangeActor>(self, input, output, fetchLock),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getSourceServersForRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7022398268476599552UL, 5262681254722997504UL);
		ActorExecutionContextHelper __helper(static_cast<GetSourceServersForRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getSourceServersForRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getSourceServersForRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetSourceServersForRangeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetSourceServersForRangeActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetSourceServersForRangeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetSourceServersForRangeActor, 3, IDDTxnProcessor::SourceServers >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<Void> getSourceServersForRange( DDQueue* const& self, RelocateData const& input, PromiseStream<RelocateData> const& output, Reference<FlowLock> const& fetchLock ) {
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new GetSourceServersForRangeActor(self, input, output, fetchLock));
															#line 1164 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 617 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

DDQueue::DDQueue(DDQueueInitParams const& params)
  : IDDRelocationQueue(), distributorId(params.id), lock(params.lock), txnProcessor(params.db),
    teamCollections(params.teamCollections), shardsAffectedByTeamFailure(params.shardsAffectedByTeamFailure),
    physicalShardCollection(params.physicalShardCollection), bulkLoadTaskCollection(params.bulkLoadTaskCollection),
    getAverageShardBytes(params.getAverageShardBytes),
    startMoveKeysParallelismLock(SERVER_KNOBS->DD_MOVE_KEYS_PARALLELISM),
    finishMoveKeysParallelismLock(SERVER_KNOBS->DD_MOVE_KEYS_PARALLELISM),
    cleanUpDataMoveParallelismLock(SERVER_KNOBS->DD_MOVE_KEYS_PARALLELISM),
    fetchSourceLock(new FlowLock(SERVER_KNOBS->DD_FETCH_SOURCE_PARALLELISM)), activeRelocations(0),
    queuedRelocations(0), bytesWritten(0), teamSize(params.teamSize), singleRegionTeamSize(params.singleRegionTeamSize),
    output(params.relocationProducer), input(params.relocationConsumer), getShardMetrics(params.getShardMetrics),
    getTopKMetrics(params.getTopKMetrics), lastInterval(0), suppressIntervals(0),
    rawProcessingUnhealthy(new AsyncVar<bool>(false)), rawProcessingWiggle(new AsyncVar<bool>(false)),
    unhealthyRelocations(0), movedKeyServersEventHolder(makeReference<EventCacheHolder>("MovedKeyServers")),
    moveReusePhysicalShard(0), moveCreateNewPhysicalShard(0),
    retryFindDstReasonCount(static_cast<int>(RetryFindDstReason::NumberOfTypes), 0),
    moveBytesRate(SERVER_KNOBS->DD_TRACE_MOVE_BYTES_AVERAGE_INTERVAL) {}

void DDQueue::startRelocation(int priority, int healthPriority) {
	// Although PRIORITY_TEAM_REDUNDANT has lower priority than split and merge shard movement,
	// we must count it into unhealthyRelocations; because team removers relies on unhealthyRelocations to
	// ensure a team remover will not start before the previous one finishes removing a team and move away data
	// NOTE: split and merge shard have higher priority. If they have to wait for unhealthyRelocations = 0,
	// deadlock may happen: split/merge shard waits for unhealthyRelocations, while blocks team_redundant.
	if (healthPriority == SERVER_KNOBS->PRIORITY_POPULATE_REGION ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_2_LEFT || healthPriority == SERVER_KNOBS->PRIORITY_TEAM_1_LEFT ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_0_LEFT ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT) {
		unhealthyRelocations++;
		rawProcessingUnhealthy->set(true);
	}
	if (healthPriority == SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE) {
		rawProcessingWiggle->set(true);
	}
	priority_relocations[priority]++;
}

void DDQueue::finishRelocation(int priority, int healthPriority) {
	if (healthPriority == SERVER_KNOBS->PRIORITY_POPULATE_REGION ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_2_LEFT || healthPriority == SERVER_KNOBS->PRIORITY_TEAM_1_LEFT ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_0_LEFT ||
	    healthPriority == SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT) {
		unhealthyRelocations--;
		ASSERT(unhealthyRelocations >= 0);
		if (unhealthyRelocations == 0) {
			rawProcessingUnhealthy->set(false);
		}
	}
	priority_relocations[priority]--;
	if (priority_relocations[SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE] == 0) {
		rawProcessingWiggle->set(false);
	}
}

void DDQueue::validate() {
	if (EXPENSIVE_VALIDATION) {
		for (auto it = fetchingSourcesQueue.begin(); it != fetchingSourcesQueue.end(); ++it) {
			// relocates in the fetching queue do not have src servers yet.
			if (it->src.size())
				TraceEvent(SevError, "DDQueueValidateError1")
				    .detail("Problem", "relocates in the fetching queue do not have src servers yet");

			// relocates in the fetching queue do not have a work factor yet.
			if (it->workFactor != 0.0)
				TraceEvent(SevError, "DDQueueValidateError2")
				    .detail("Problem", "relocates in the fetching queue do not have a work factor yet");

			// relocates in the fetching queue are in the queueMap.
			auto range = queueMap.rangeContaining(it->keys.begin);
			if (range.value() != *it || range.range() != it->keys)
				TraceEvent(SevError, "DDQueueValidateError3")
				    .detail("Problem", "relocates in the fetching queue are in the queueMap");
		}

		/*
		for( auto it = queue.begin(); it != queue.end(); ++it ) {
		    for( auto rdit = it->second.begin(); rdit != it->second.end(); ++rdit ) {
		        // relocates in the queue are in the queueMap exactly.
		        auto range = queueMap.rangeContaining( rdit->keys.begin );
		        if( range.value() != *rdit || range.range() != rdit->keys )
		            TraceEvent(SevError, "DDQueueValidateError4").detail("Problem", "relocates in the queue are in the queueMap exactly")
		            .detail("RangeBegin", range.range().begin)
		            .detail("RangeEnd", range.range().end)
		            .detail("RelocateBegin2", range.value().keys.begin)
		            .detail("RelocateEnd2", range.value().keys.end)
		            .detail("RelocateStart", range.value().startTime)
		            .detail("MapStart", rdit->startTime)
		            .detail("RelocateWork", range.value().workFactor)
		            .detail("MapWork", rdit->workFactor)
		            .detail("RelocateSrc", range.value().src.size())
		            .detail("MapSrc", rdit->src.size())
		            .detail("RelocatePrio", range.value().priority)
		            .detail("MapPrio", rdit->priority);

		        // relocates in the queue have src servers
		        if( !rdit->src.size() )
		            TraceEvent(SevError, "DDQueueValidateError5").detail("Problem", "relocates in the queue have src servers");

		        // relocates in the queue do not have a work factor yet.
		        if( rdit->workFactor != 0.0 )
		            TraceEvent(SevError, "DDQueueValidateError6").detail("Problem", "relocates in the queue do not have a work factor yet");

		        bool contains = false;
		        for( int i = 0; i < rdit->src.size(); i++ ) {
		            if( rdit->src[i] == it->first ) {
		                contains = true;
		                break;
		            }
		        }
		        if( !contains )
		            TraceEvent(SevError, "DDQueueValidateError7").detail("Problem", "queued relocate data does not include ss under which its filed");
		    }
		}*/

		auto inFlightRanges = inFlight.ranges();
		for (auto it = inFlightRanges.begin(); it != inFlightRanges.end(); ++it) {
			for (int i = 0; i < it->value().src.size(); i++) {
				// each server in the inFlight map is in the busymap
				if (!busymap.contains(it->value().src[i]))
					TraceEvent(SevError, "DDQueueValidateError8")
					    .detail("Problem", "each server in the inFlight map is in the busymap");

				// relocate data that is inFlight is not also in the queue
				if (queue[it->value().src[i]].contains(it->value()))
					TraceEvent(SevError, "DDQueueValidateError9")
					    .detail("Problem", "relocate data that is inFlight is not also in the queue");
			}

			for (int i = 0; i < it->value().completeDests.size(); i++) {
				// each server in the inFlight map is in the dest busymap
				if (!destBusymap.contains(it->value().completeDests[i]))
					TraceEvent(SevError, "DDQueueValidateError10")
					    .detail("Problem", "each server in the inFlight map is in the destBusymap");
			}

			// in flight relocates have source servers
			if (it->value().startTime != -1 && !it->value().src.size())
				TraceEvent(SevError, "DDQueueValidateError11")
				    .detail("Problem", "in flight relocates have source servers");

			if (inFlightActors.liveActorAt(it->range().begin)) {
				// the key range in the inFlight map matches the key range in the RelocateData message
				if (it->value().keys != it->range())
					TraceEvent(SevError, "DDQueueValidateError12")
					    .detail("Problem",
					            "the key range in the inFlight map matches the key range in the RelocateData message");
			} else if (it->value().cancellable) {
				TraceEvent(SevError, "DDQueueValidateError13")
				    .detail("Problem", "key range is cancellable but not in flight!")
				    .detail("Range", it->range());
			}
		}

		for (auto it = busymap.begin(); it != busymap.end(); ++it) {
			for (int i = 0; i < it->second.ledger.size() - 1; i++) {
				if (it->second.ledger[i] < it->second.ledger[i + 1])
					TraceEvent(SevError, "DDQueueValidateError14")
					    .detail("Problem", "ascending ledger problem")
					    .detail("LedgerLevel", i)
					    .detail("LedgerValueA", it->second.ledger[i])
					    .detail("LedgerValueB", it->second.ledger[i + 1]);
				if (it->second.ledger[i] < 0.0)
					TraceEvent(SevError, "DDQueueValidateError15")
					    .detail("Problem", "negative ascending problem")
					    .detail("LedgerLevel", i)
					    .detail("LedgerValue", it->second.ledger[i]);
			}
		}

		for (auto it = destBusymap.begin(); it != destBusymap.end(); ++it) {
			for (int i = 0; i < it->second.ledger.size() - 1; i++) {
				if (it->second.ledger[i] < it->second.ledger[i + 1])
					TraceEvent(SevError, "DDQueueValidateError16")
					    .detail("Problem", "ascending ledger problem")
					    .detail("LedgerLevel", i)
					    .detail("LedgerValueA", it->second.ledger[i])
					    .detail("LedgerValueB", it->second.ledger[i + 1]);
				if (it->second.ledger[i] < 0.0)
					TraceEvent(SevError, "DDQueueValidateError17")
					    .detail("Problem", "negative ascending problem")
					    .detail("LedgerLevel", i)
					    .detail("LedgerValue", it->second.ledger[i]);
			}
		}

		std::set<RelocateData, std::greater<RelocateData>> queuedRelocationsMatch;
		for (auto it = queue.begin(); it != queue.end(); ++it)
			queuedRelocationsMatch.insert(it->second.begin(), it->second.end());
		ASSERT(queuedRelocations == queuedRelocationsMatch.size() + fetchingSourcesQueue.size());

		int testActive = 0;
		for (auto it = priority_relocations.begin(); it != priority_relocations.end(); ++it)
			testActive += it->second;
		ASSERT(activeRelocations + queuedRelocations == testActive);
	}
}

void DDQueue::queueRelocation(RelocateShard rs, std::set<UID>& serversToLaunchFrom) {
	//TraceEvent("QueueRelocationBegin").detail("Begin", rd.keys.begin).detail("End", rd.keys.end);

	// remove all items from both queues that are fully contained in the new relocation (i.e. will be overwritten)
	RelocateData rd(rs);
	bool hasHealthPriority = RelocateData::isHealthPriority(rd.priority);
	bool hasBoundaryPriority = RelocateData::isBoundaryPriority(rd.priority);

	auto ranges = queueMap.intersectingRanges(rd.keys);
	for (auto r = ranges.begin(); r != ranges.end(); ++r) {
		RelocateData& rrs = r->value();

		auto fetchingSourcesItr = fetchingSourcesQueue.find(rrs);
		bool foundActiveFetching = fetchingSourcesItr != fetchingSourcesQueue.end();
		std::set<RelocateData, std::greater<RelocateData>>* firstQueue;
		std::set<RelocateData, std::greater<RelocateData>>::iterator firstRelocationItr;
		bool foundActiveRelocation = false;

		if (!foundActiveFetching && rrs.src.size()) {
			firstQueue = &queue[rrs.src[0]];
			firstRelocationItr = firstQueue->find(rrs);
			foundActiveRelocation = firstRelocationItr != firstQueue->end();
		}

		// If there is a queued job that wants data relocation which we are about to cancel/modify,
		//  make sure that we keep the relocation intent for the job that we queue up
		if (foundActiveFetching || foundActiveRelocation) {
			rd.wantsNewServers |= rrs.wantsNewServers;
			rd.startTime = std::min(rd.startTime, rrs.startTime);
			if (!hasHealthPriority) {
				rd.healthPriority = std::max(rd.healthPriority, rrs.healthPriority);
			}
			if (!hasBoundaryPriority) {
				rd.boundaryPriority = std::max(rd.boundaryPriority, rrs.boundaryPriority);
			}
			rd.priority = std::max(rd.priority, std::max(rd.boundaryPriority, rd.healthPriority));
		}

		if (rd.keys.contains(rrs.keys)) {
			if (foundActiveFetching)
				fetchingSourcesQueue.erase(fetchingSourcesItr);
			else if (foundActiveRelocation) {
				firstQueue->erase(firstRelocationItr);
				for (int i = 1; i < rrs.src.size(); i++)
					queue[rrs.src[i]].erase(rrs);
			}
		}

		if (foundActiveFetching || foundActiveRelocation) {
			serversToLaunchFrom.insert(rrs.src.begin(), rrs.src.end());
			/*TraceEvent(rrs.interval.end(), mi.id()).detail("Result","Cancelled")
			    .detail("WasFetching", foundActiveFetching).detail("Contained", rd.keys.contains( rrs.keys ));*/
			queuedRelocations--;
			TraceEvent(SevVerbose, "QueuedRelocationsChanged")
			    .detail("DataMoveID", rrs.dataMoveId)
			    .detail("RandomID", rrs.randomId)
			    .detail("Total", queuedRelocations);
			finishRelocation(rrs.priority, rrs.healthPriority);
		}
	}

	// determine the final state of the relocations map
	auto affectedQueuedItems = queueMap.getAffectedRangesAfterInsertion(rd.keys, rd);

	// put the new request into the global map of requests (modifies the ranges already present)
	queueMap.insert(rd.keys, rd);

	// cancel all the getSourceServers actors that intersect the new range that we will be getting
	getSourceActors.cancel(KeyRangeRef(affectedQueuedItems.front().begin, affectedQueuedItems.back().end));

	// update fetchingSourcesQueue and the per-server queue based on truncated ranges after insertion, (re-)launch
	// getSourceServers
	auto queueMapItr = queueMap.rangeContaining(affectedQueuedItems[0].begin);

	// Put off erasing elements from fetchingSourcesQueue
	std::set<RelocateData, std::greater<RelocateData>> delayDelete;
	for (int r = 0; r < affectedQueuedItems.size(); ++r, ++queueMapItr) {
		// ASSERT(queueMapItr->value() == queueMap.rangeContaining(affectedQueuedItems[r].begin)->value());
		RelocateData& rrs = queueMapItr->value();

		if (rrs.src.size() == 0 && (rrs.keys == rd.keys || fetchingSourcesQueue.contains(rrs))) {
			if (rrs.keys != rd.keys) {
				delayDelete.insert(rrs);
			}

			rrs.keys = affectedQueuedItems[r];
			rrs.interval = TraceInterval("QueuedRelocation", rrs.randomId); // inherit the old randomId

			DebugRelocationTraceEvent(rrs.interval.begin(), distributorId)
			    .detail("KeyBegin", rrs.keys.begin)
			    .detail("KeyEnd", rrs.keys.end)
			    .detail("Priority", rrs.priority)
			    .detail("WantsNewServers", rrs.wantsNewServers);

			queuedRelocations++;
			TraceEvent(SevVerbose, "QueuedRelocationsChanged")
			    .detail("DataMoveID", rrs.dataMoveId)
			    .detail("RandomID", rrs.randomId)
			    .detail("Total", queuedRelocations);
			startRelocation(rrs.priority, rrs.healthPriority);

			fetchingSourcesQueue.insert(rrs);
			getSourceActors.insert(rrs.keys,
			                       getSourceServersForRange(this, rrs, fetchSourceServersComplete, fetchSourceLock));
		} else {
			RelocateData newData(rrs);
			newData.keys = affectedQueuedItems[r];
			ASSERT(rrs.src.size() || rrs.startTime == -1);

			bool foundActiveRelocation = false;
			for (int i = 0; i < rrs.src.size(); i++) {
				auto& serverQueue = queue[rrs.src[i]];

				if (serverQueue.erase(rrs) > 0) {
					if (!foundActiveRelocation) {
						newData.interval = TraceInterval("QueuedRelocation", rrs.randomId); // inherit the old randomId

						DebugRelocationTraceEvent(newData.interval.begin(), distributorId)
						    .detail("KeyBegin", newData.keys.begin)
						    .detail("KeyEnd", newData.keys.end)
						    .detail("Priority", newData.priority)
						    .detail("WantsNewServers", newData.wantsNewServers);

						queuedRelocations++;
						TraceEvent(SevVerbose, "QueuedRelocationsChanged")
						    .detail("DataMoveID", newData.dataMoveId)
						    .detail("RandomID", newData.randomId)
						    .detail("Total", queuedRelocations);
						startRelocation(newData.priority, newData.healthPriority);
						foundActiveRelocation = true;
					}

					serverQueue.insert(newData);
				} else
					break;
			}

			// We update the keys of a relocation even if it is "dead" since it helps validate()
			rrs.keys = affectedQueuedItems[r];
			rrs.interval = newData.interval;
		}
	}

	for (auto it : delayDelete) {
		fetchingSourcesQueue.erase(it);
	}
	DebugRelocationTraceEvent("ReceivedRelocateShard", distributorId)
	    .detail("KeyBegin", rd.keys.begin)
	    .detail("KeyEnd", rd.keys.end)
	    .detail("Priority", rd.priority)
	    .detail("AffectedRanges", affectedQueuedItems.size());
}

void DDQueue::completeSourceFetch(const RelocateData& results) {
	ASSERT(fetchingSourcesQueue.contains(results));

	// logRelocation( results, "GotSourceServers" );

	fetchingSourcesQueue.erase(results);
	queueMap.insert(results.keys, results);
	for (int i = 0; i < results.src.size(); i++) {
		queue[results.src[i]].insert(results);
	}
	updateLastAsSource(results.src);
	serverCounter.increaseForTeam(results.src, results.reason, ServerCounter::CountType::QueuedSource);
}

void DDQueue::logRelocation(const RelocateData& rd, const char* title) {
	std::string busyString;
	for (int i = 0; i < rd.src.size() && i < teamSize * 2; i++)
		busyString += describe(rd.src[i]) + " - (" + busymap[rd.src[i]].toString() + "); ";

	TraceEvent(title, distributorId)
	    .detail("KeyBegin", rd.keys.begin)
	    .detail("KeyEnd", rd.keys.end)
	    .detail("Priority", rd.priority)
	    .detail("WorkFactor", rd.workFactor)
	    .detail("SourceServerCount", rd.src.size())
	    .detail("SourceServers", describe(rd.src, teamSize * 2))
	    .detail("SourceBusyness", busyString);
}

void DDQueue::launchQueuedWork(KeyRange keys, const DDEnabledState* ddEnabledState) {
	// combine all queued work in the key range and check to see if there is anything to launch
	std::set<RelocateData, std::greater<RelocateData>> combined;
	auto f = queueMap.intersectingRanges(keys);
	for (auto it = f.begin(); it != f.end(); ++it) {
		if (it->value().src.size() && queue[it->value().src[0]].contains(it->value()))
			combined.insert(it->value());
	}
	launchQueuedWork(combined, ddEnabledState);
}

void DDQueue::launchQueuedWork(const std::set<UID>& serversToLaunchFrom, const DDEnabledState* ddEnabledState) {
	// combine all work from the source servers to see if there is anything new to launch
	std::set<RelocateData, std::greater<RelocateData>> combined;
	for (auto id : serversToLaunchFrom) {
		auto& queuedWork = queue[id];
		auto it = queuedWork.begin();
		for (int j = 0; j < teamSize && it != queuedWork.end(); j++) {
			combined.insert(*it);
			++it;
		}
	}
	launchQueuedWork(combined, ddEnabledState);
}

void DDQueue::launchQueuedWork(RelocateData launchData, const DDEnabledState* ddEnabledState) {
	// check a single RelocateData to see if it can be launched
	std::set<RelocateData, std::greater<RelocateData>> combined;
	combined.insert(launchData);
	launchQueuedWork(combined, ddEnabledState);
}

DataMoveType newDataMoveType(bool doBulkLoading) {
	DataMoveType type = DataMoveType::LOGICAL;
	if (deterministicRandom()->random01() < SERVER_KNOBS->DD_PHYSICAL_SHARD_MOVE_PROBABILITY) {
		type = DataMoveType::PHYSICAL;
	}
	if (type != DataMoveType::PHYSICAL && SERVER_KNOBS->ENABLE_PHYSICAL_SHARD_MOVE_EXPERIMENT) {
		type = DataMoveType::PHYSICAL_EXP;
	}
	if (doBulkLoading) {
		if (type == DataMoveType::LOGICAL) {
			type = DataMoveType::LOGICAL_BULKLOAD;
		} else if (type == DataMoveType::PHYSICAL || type == DataMoveType::PHYSICAL_EXP) {
			type = DataMoveType::PHYSICAL_BULKLOAD;
		} else {
			UNREACHABLE();
		}
	}
	return type;
}

bool runPendingBulkLoadTaskWithRelocateData(DDQueue* self, RelocateData& rd) {
	bool doBulkLoading = false;
	Optional<DDBulkLoadEngineTask> task = self->bulkLoadTaskCollection->getTaskByRange(rd.keys);
	if (task.present() && task.get().coreState.onAnyPhase({ BulkLoadPhase::Triggered, BulkLoadPhase::Running })) {
		rd.bulkLoadTask = task.get();
		doBulkLoading = true;
	}
	if (doBulkLoading) {
		try {
			self->bulkLoadTaskCollection->startTask(rd.bulkLoadTask.get().coreState);
		} catch (Error& e) {
			ASSERT_WE_THINK(e.code() == error_code_bulkload_task_outdated);
			if (e.code() == error_code_bulkload_task_outdated) {
				TraceEvent(SevError, "DDBulkLoadTaskOutdatedWhenStartRelocator", self->distributorId)
				    .detail("NewDataMoveID", rd.dataMoveId)
				    .detail("NewDataMovePriority", rd.priority)
				    .detail("NewDataMoveRange", rd.keys)
				    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
				    .detail("TaskJobID", rd.bulkLoadTask.get().coreState.getJobId())
				    .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange())
				    .detail("TaskDataMoveID",
				            rd.bulkLoadTask.get().coreState.getDataMoveId().present()
				                ? rd.bulkLoadTask.get().coreState.getDataMoveId().get().toString()
				                : "");
				throw movekeys_conflict();
			} else {
				throw e;
			}
		}
	}
	return doBulkLoading;
}

// For each relocateData rd in the queue, check if there exist inflight relocate data whose keyrange is overlapped
// with rd. If there exist, cancel them by cancelling their actors and reducing the src servers' busyness of those
// canceled inflight relocateData. Launch the relocation for the rd.
void DDQueue::launchQueuedWork(std::set<RelocateData, std::greater<RelocateData>> combined,
                               const DDEnabledState* ddEnabledState) {
	[[maybe_unused]] int startedHere = 0;
	double startTime = now();
	// kick off relocators from items in the queue as need be
	auto it = combined.begin();
	for (; it != combined.end(); it++) {
		RelocateData rd(*it);

		// If having a bulk load task overlapping the rd range,
		// attach bulk load task to the input rd if rd is not a data move
		// for unhealthy. Make the bulk load task visible on the global task map
		bool doBulkLoading = runPendingBulkLoadTaskWithRelocateData(this, rd);
		if (doBulkLoading) {
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskLaunchingDataMove", this->distributorId)
			    .detail("NewDataMoveId", rd.dataMoveId)
			    .detail("NewDataMovePriority", rd.priority)
			    .detail("NewDataMoveRange", rd.keys)
			    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
			    .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId());
		}

		// Check if there is an inflight shard that is overlapped with the queued relocateShard (rd)
		bool overlappingInFlight = false;
		auto intersectingInFlight = inFlight.intersectingRanges(rd.keys);
		for (auto it = intersectingInFlight.begin(); it != intersectingInFlight.end(); ++it) {
			if (fetchKeysComplete.contains(it->value()) && inFlightActors.liveActorAt(it->range().begin) &&
			    !rd.keys.contains(it->range()) && it->value().priority >= rd.priority &&
			    rd.healthPriority < SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY) {

				DebugRelocationTraceEvent("OverlappingInFlight", distributorId)
				    .detail("KeyBegin", it->value().keys.begin)
				    .detail("KeyEnd", it->value().keys.end)
				    .detail("Priority", it->value().priority);

				overlappingInFlight = true;
				break;
			}
		}

		if (overlappingInFlight) {
			ASSERT(!rd.isRestore());
			// logRelocation( rd, "SkippingOverlappingInFlight" );
			continue;
		}

		// Because the busyness of a server is decreased when a superseding relocation is issued, we
		//  need to consider what the busyness of a server WOULD be if
		auto containedRanges = inFlight.containedRanges(rd.keys);
		std::vector<RelocateData> cancellableRelocations;
		for (auto it = containedRanges.begin(); it != containedRanges.end(); ++it) {
			if (it.value().cancellable) {
				cancellableRelocations.push_back(it->value());
			}
		}

		// Data movement avoids overloading source servers in moving data.
		// SOMEDAY: the list of source servers may be outdated since they were fetched when the work was put in the
		// queue
		// FIXME: we need spare capacity even when we're just going to be cancelling work via TEAM_HEALTHY
		if (!rd.isRestore() && !canLaunchSrc(rd, teamSize, singleRegionTeamSize, busymap, cancellableRelocations)) {
			// logRelocation( rd, "SkippingQueuedRelocation" );
			if (rd.bulkLoadTask.present()) {
				TraceEvent(SevError, "DDBulkLoadTaskDelayedByBusySrc", this->distributorId)
				    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
				    .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId())
				    .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
			}
			continue;
		}

		// From now on, the source servers for the RelocateData rd have enough resource to move the data away,
		// because they do not have too much inflight data movement.

		// logRelocation( rd, "LaunchingRelocation" );
		DebugRelocationTraceEvent(rd.interval.end(), distributorId).detail("Result", "Success");

		if (!rd.isRestore()) {
			queuedRelocations--;
			TraceEvent(SevVerbose, "QueuedRelocationsChanged")
			    .detail("DataMoveID", rd.dataMoveId)
			    .detail("RandomID", rd.randomId)
			    .detail("Total", queuedRelocations);
			finishRelocation(rd.priority, rd.healthPriority);

			// now we are launching: remove this entry from the queue of all the src servers
			for (size_t i = 0; i < rd.src.size(); i++) {
				const auto result = queue[rd.src[i]].erase(rd);
				ASSERT(result);
			}
		}

		// If there is a job in flight that wants data relocation which we are about to cancel/modify,
		//     make sure that we keep the relocation intent for the job that we launch
		auto f = inFlight.intersectingRanges(rd.keys);
		for (auto it = f.begin(); it != f.end(); ++it) {
			if (inFlightActors.liveActorAt(it->range().begin)) {
				rd.wantsNewServers |= it->value().wantsNewServers;
			}
		}
		startedHere++;

		// update both inFlightActors and inFlight key range maps, cancelling deleted RelocateShards
		std::vector<KeyRange> ranges;
		inFlightActors.getRangesAffectedByInsertion(rd.keys, ranges);
		inFlightActors.cancel(KeyRangeRef(ranges.front().begin, ranges.back().end));
		// The cancelDataMove feature assumes inFlightActors are immediately cancelled.
		// If this is not true, multiple inflightActors can have overlapped range,
		// which leads to conflicts of moving keys

		Future<Void> fCleanup =
		    SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA ? cancelDataMove(this, rd.keys, ddEnabledState) : Void();

		inFlight.insert(rd.keys, rd);
		for (int r = 0; r < ranges.size(); r++) {
			RelocateData& rrs = inFlight.rangeContaining(ranges[r].begin)->value();
			rrs.keys = ranges[r];
			if (rrs.bulkLoadTask.present() && rrs.bulkLoadTask.get().coreState.getRange() != rrs.keys) {
				// The new bulk load data move partially overwrites an old bulk load data move.
				// In this case, the old bulk load task is cancelled.
				// For the range that is not covered by the new data move, drop the bulk load task and
				// run it as a normal data move.
				ASSERT(rrs.bulkLoadTask.get().coreState.getRange().contains(rrs.keys));
				rrs.bulkLoadTask.reset();
			}
			if (rd.keys == ranges[r] && rd.isRestore()) {
				ASSERT(rd.dataMove != nullptr);
				ASSERT(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA);
				rrs.dataMoveId = rd.dataMove->meta.id;
			} else {
				ASSERT_WE_THINK(!rd.isRestore()); // Restored data move should not overlap.
				// TODO(psm): The shard id is determined by DD.
				rrs.dataMove.reset();
				if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA) {
					if (SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD) {
						rrs.dataMoveId = UID();
					} else if (rrs.bulkLoadTask.present()) {
						// We have to decide this after prevCleanup completes.
						// For details, see the comment in dataDistributionRelocator.
						rrs.dataMoveId = UID();
					} else {
						DataMoveType dataMoveType = newDataMoveType(rrs.bulkLoadTask.present());
						rrs.dataMoveId = newDataMoveId(
						    deterministicRandom()->randomUInt64(), AssignEmptyRange::False, dataMoveType, rrs.dmReason);
						TraceEvent(SevInfo, "NewDataMoveWithRandomDestID", this->distributorId)
						    .detail("DataMoveID", rrs.dataMoveId.toString())
						    .detail("TrackID", rrs.randomId)
						    .detail("Range", rrs.keys)
						    .detail("Reason", rrs.reason.toString())
						    .detail("DataMoveType", dataMoveType)
						    .detail("DataMoveReason", static_cast<int>(rrs.dmReason));
					}
				} else {
					rrs.dataMoveId = anonymousShardId;
					TraceEvent(SevInfo, "NewDataMoveWithAnonymousDestID", this->distributorId)
					    .detail("DataMoveID", rrs.dataMoveId.toString())
					    .detail("TrackID", rrs.randomId)
					    .detail("Range", rrs.keys)
					    .detail("Reason", rrs.reason.toString())
					    .detail("DataMoveReason", static_cast<int>(rrs.dmReason));
				}
			}

			launch(rrs, busymap, singleRegionTeamSize);
			activeRelocations++;
			TraceEvent(SevVerbose, "InFlightRelocationChange")
			    .detail("Launch", rrs.dataMoveId)
			    .detail("Total", activeRelocations);
			startRelocation(rrs.priority, rrs.healthPriority);
			// Start the actor that relocates data in the rrs.keys
			inFlightActors.insert(rrs.keys, dataDistributionRelocator(this, rrs, fCleanup, ddEnabledState));
		}

		// logRelocation( rd, "LaunchedRelocation" );
	}
	if (now() - startTime > .001 && deterministicRandom()->random01() < 0.001)
		TraceEvent(SevWarnAlways, "LaunchingQueueSlowx1000").detail("Elapsed", now() - startTime);

	/*if( startedHere > 0 ) {
	    TraceEvent("StartedDDRelocators", distributorId)
	        .detail("QueueSize", queuedRelocations)
	        .detail("StartedHere", startedHere)
	        .detail("ActiveRelocations", activeRelocations);
	} */

	validate();
}

int DDQueue::getHighestPriorityRelocation() const {
	int highestPriority{ 0 };
	for (const auto& [priority, count] : priority_relocations) {
		if (count > 0) {
			highestPriority = std::max(highestPriority, priority);
		}
	}
	return highestPriority;
}

// return true if the servers are throttled as source for read rebalance
bool DDQueue::timeThrottle(const std::vector<UID>& ids) const {
	return std::any_of(ids.begin(), ids.end(), [this](const UID& id) {
		if (this->lastAsSource.contains(id)) {
			return (now() - this->lastAsSource.at(id)) * SERVER_KNOBS->READ_REBALANCE_SRC_PARALLELISM <
			       SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL;
		}
		return false;
	});
}

void DDQueue::updateLastAsSource(const std::vector<UID>& ids, double t) {
	for (auto& id : ids)
		lastAsSource[id] = t;
}

// Schedules cancellation of a data move.
void DDQueue::enqueueCancelledDataMove(UID dataMoveId, KeyRange range, const DDEnabledState* ddEnabledState) {
	ASSERT(!txnProcessor->isMocked()); // the mock implementation currently doesn't support data move
	std::vector<Future<Void>> cleanup;
	auto f = this->dataMoves.intersectingRanges(range);
	for (auto it = f.begin(); it != f.end(); ++it) {
		if (it->value().isValid()) {
			TraceEvent(SevError, "DDEnqueueCancelledDataMoveConflict", this->distributorId)
			    .detail("DataMoveID", dataMoveId)
			    .detail("CancelledRange", range)
			    .detail("ConflictingDataMoveID", it->value().id)
			    .detail("ConflictingRange", KeyRangeRef(it->range().begin, it->range().end));
			return;
		}
	}

	DDQueue::DDDataMove dataMove(dataMoveId);
	dataMove.cancel = cleanUpDataMove(txnProcessor->context(),
	                                  dataMoveId,
	                                  this->lock,
	                                  &this->cleanUpDataMoveParallelismLock,
	                                  range,
	                                  ddEnabledState,
	                                  this->addBackgroundCleanUpDataMoveActor);
	this->dataMoves.insert(range, dataMove);
	TraceEvent(SevInfo, "DDEnqueuedCancelledDataMove", this->distributorId)
	    .detail("DataMoveID", dataMoveId)
	    .detail("Range", range);
}

Future<Void> DDQueue::periodicalRefreshCounter() {
	auto f = [this]() {
		serverCounter.traceAll(distributorId);
		serverCounter.clear();
	};
	return recurring(f, SERVER_KNOBS->DD_QUEUE_COUNTER_REFRESH_INTERVAL);
}

int DDQueue::getUnhealthyRelocationCount() const {
	return unhealthyRelocations;
}

// Cancel existing relocation if exists, and serialize all concurrent relocations
															#line 1895 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via cancelDataMove()
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class CancelDataMoveActor>
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class CancelDataMoveActorState {
															#line 1902 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	CancelDataMoveActorState(class DDQueue* const& self,KeyRange const& range,const DDEnabledState* const& ddEnabledState) 
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   range(range),
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   ddEnabledState(ddEnabledState),
															#line 1345 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   cleanup(),
															#line 1346 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   lastObservedDataMoves()
															#line 1917 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("cancelDataMove", reinterpret_cast<unsigned long>(this));

	}
	~CancelDataMoveActorState() 
	{
		fdb_probe_actor_destroy("cancelDataMove", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1349 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				lastObservedDataMoves.clear();
															#line 1350 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto f = self->dataMoves.intersectingRanges(range);
															#line 1351 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for(auto it = f.begin();it != f.end();++it) {
															#line 1352 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (!it->value().isValid())
															#line 1939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					{
						continue;
					}
															#line 1355 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					KeyRange keys = KeyRangeRef(it->range().begin, it->range().end);
															#line 1356 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevInfo, "DDQueueCancelDataMove", self->distributorId) .detail("DataMoveID", it->value().id) .detail("DataMoveRange", keys) .detail("Range", range);
															#line 1360 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (!it->value().cancel.isValid())
															#line 1949 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					{
															#line 1361 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						it->value().cancel = cleanUpDataMove(self->txnProcessor->context(), it->value().id, self->lock, &self->cleanUpDataMoveParallelismLock, keys, ddEnabledState, self->addBackgroundCleanUpDataMoveActor);
															#line 1953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
															#line 1369 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					lastObservedDataMoves.push_back(std::make_pair(keys, it->value().id));
															#line 1370 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					cleanup.push_back(it->value().cancel);
															#line 1959 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
															#line 1373 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_0 = waitForAll(cleanup);
															#line 1373 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<CancelDataMoveActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1965 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<CancelDataMoveActor*>(this)->actor_wait_state = 1;
															#line 1373 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CancelDataMoveActor, 0, Void >*>(static_cast<CancelDataMoveActor*>(this)));
															#line 1970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CancelDataMoveActorState();
		static_cast<CancelDataMoveActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1404 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<CancelDataMoveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CancelDataMoveActorState(); static_cast<CancelDataMoveActor*>(this)->destroy(); return 0; }
															#line 1999 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<CancelDataMoveActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CancelDataMoveActorState();
		static_cast<CancelDataMoveActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1401 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 2012 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1380 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::vector<KeyRange> toResetRanges;
															#line 1381 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto observedDataMove : lastObservedDataMoves ) {
															#line 1382 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto f = self->dataMoves.intersectingRanges(observedDataMove.first);
															#line 1383 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(auto it = f.begin();it != f.end();++it) {
															#line 1384 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (it->value().id != observedDataMove.second)
															#line 2034 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1385 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevInfo, "DataMoveWrittenByConcurrentDataMove", self->distributorId) .detail("Range", range) .detail("OldRange", observedDataMove.first) .detail("LastObservedDataMoveID", observedDataMove.second) .detail("CurrentDataMoveID", it->value().id);
															#line 2038 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1391 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(!it->value().isValid() || (it->value().cancel.isValid() && it->value().cancel.isReady()));
															#line 1392 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					toResetRanges.push_back(Standalone(it->range()));
															#line 2046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
		}
															#line 1396 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto& toResetRange : toResetRanges ) {
															#line 1397 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataMoves.insert(toResetRange, DDQueue::DDDataMove());
															#line 2054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1380 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::vector<KeyRange> toResetRanges;
															#line 1381 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto observedDataMove : lastObservedDataMoves ) {
															#line 1382 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto f = self->dataMoves.intersectingRanges(observedDataMove.first);
															#line 1383 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(auto it = f.begin();it != f.end();++it) {
															#line 1384 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (it->value().id != observedDataMove.second)
															#line 2072 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1385 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevInfo, "DataMoveWrittenByConcurrentDataMove", self->distributorId) .detail("Range", range) .detail("OldRange", observedDataMove.first) .detail("LastObservedDataMoveID", observedDataMove.second) .detail("CurrentDataMoveID", it->value().id);
															#line 2076 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1391 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(!it->value().isValid() || (it->value().cancel.isValid() && it->value().cancel.isReady()));
															#line 1392 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					toResetRanges.push_back(Standalone(it->range()));
															#line 2084 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
		}
															#line 1396 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto& toResetRange : toResetRanges ) {
															#line 1397 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataMoves.insert(toResetRange, DDQueue::DDDataMove());
															#line 2092 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CancelDataMoveActor*>(this)->actor_wait_state > 0) static_cast<CancelDataMoveActor*>(this)->actor_wait_state = 0;
		static_cast<CancelDataMoveActor*>(this)->ActorCallback< CancelDataMoveActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CancelDataMoveActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CancelDataMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CancelDataMoveActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CancelDataMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CancelDataMoveActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CancelDataMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cancelDataMove", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	class DDQueue* self;
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	KeyRange range;
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	const DDEnabledState* ddEnabledState;
															#line 1345 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<Future<Void>> cleanup;
															#line 1346 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<std::pair<KeyRange, UID>> lastObservedDataMoves;
															#line 2196 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via cancelDataMove()
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class CancelDataMoveActor final : public Actor<Void>, public ActorCallback< CancelDataMoveActor, 0, Void >, public FastAllocated<CancelDataMoveActor>, public CancelDataMoveActorState<CancelDataMoveActor> {
															#line 2201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<CancelDataMoveActor>::operator new;
	using FastAllocated<CancelDataMoveActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6983351125395174400UL, 13389078191021468672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CancelDataMoveActor, 0, Void >;
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	CancelDataMoveActor(class DDQueue* const& self,KeyRange const& range,const DDEnabledState* const& ddEnabledState) 
															#line 2218 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CancelDataMoveActorState<CancelDataMoveActor>(self, range, ddEnabledState),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("cancelDataMove", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7738710457594625024UL, 3228702489229399296UL);
		ActorExecutionContextHelper __helper(static_cast<CancelDataMoveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("cancelDataMove");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("cancelDataMove", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CancelDataMoveActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<Void> cancelDataMove( class DDQueue* const& self, KeyRange const& range, const DDEnabledState* const& ddEnabledState ) {
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new CancelDataMoveActor(self, range, ddEnabledState));
															#line 2251 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 1406 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

static std::string destServersString(std::vector<std::pair<Reference<IDataDistributionTeam>, bool>> const& bestTeams) {
	std::stringstream ss;

	for (auto& tc : bestTeams) {
		for (const auto& id : tc.first->getServerIDs()) {
			ss << id.toString() << " ";
		}
	}

	return std::move(ss).str();
}

void traceRelocateDecision(TraceEvent& ev, const UID& pairId, const RelocateDecision& decision) {
	ev.detail("PairId", pairId)
	    .detail("Priority", decision.rd.priority)
	    .detail("KeyBegin", decision.rd.keys.begin)
	    .detail("KeyEnd", decision.rd.keys.end)
	    .detail("Reason", decision.rd.reason.toString())
	    .detail("SourceServers", describe(decision.rd.src))
	    .detail("DestinationTeam", describe(decision.destIds))
	    .detail("ExtraIds", describe(decision.extraIds));
	if (SERVER_KNOBS->DD_ENABLE_VERBOSE_TRACING) {
		// StorageMetrics is the rd shard's metrics, e.g., bytes and write bandwidth
		ev.detail("StorageMetrics", decision.metrics.toString());
	}

	if (decision.rd.reason == RelocateReason::WRITE_SPLIT) {
		// tell if the splitter acted as expected for write bandwidth splitting
		// SOMEDAY: trace the source team write bytes if necessary
		ev.detail("ShardWriteBytes", decision.metrics.bytesWrittenPerKSecond)
		    .detail("ParentShardWriteBytes", decision.parentMetrics.get().bytesWrittenPerKSecond);
	} else if (decision.rd.reason == RelocateReason::SIZE_SPLIT) {
		ev.detail("ShardSize", decision.metrics.bytes).detail("ParentShardSize", decision.parentMetrics.get().bytes);
	}
}

static int nonOverlappedServerCount(const std::vector<UID>& srcIds, const std::vector<UID>& destIds) {
	std::unordered_set<UID> srcSet{ srcIds.begin(), srcIds.end() };
	int count = 0;
	for (int i = 0; i < destIds.size(); i++) {
		if (!srcSet.contains(destIds[i])) {
			count++;
		}
	}
	return count;
}

void validateBulkLoadRelocateData(const RelocateData& rd, const std::vector<UID>& destIds, UID logId) {
	if (rd.keys != rd.bulkLoadTask.get().coreState.getRange()) {
		TraceEvent(SevError, "DDBulkLoadTaskLaunchFailed", logId)
		    .detail("Reason", "Wrong data move range")
		    .detail("DataMovePriority", rd.priority)
		    .detail("DataMoveId", rd.dataMoveId)
		    .detail("RelocatorRange", rd.keys)
		    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
		    .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId())
		    .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
		throw movekeys_conflict();
		// Very important invariant. If this error appears, check the logic
	}
	for (const auto& destId : destIds) {
		if (std::find(rd.src.begin(), rd.src.end(), destId) != rd.src.end()) {
			// In this case, getTeam has to select src as dest when remote team collection is not ready
			// This is not expected
			TraceEvent(SevError, "DDBulkLoadEngineTaskLaunchFailed", logId)
			    .detail("Reason", "Conflict src and destd due to remote recovery")
			    .detail("DataMovePriority", rd.priority)
			    .detail("DataMoveId", rd.dataMoveId)
			    .detail("RelocatorRange", rd.keys)
			    .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId())
			    .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId())
			    .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
			throw movekeys_conflict();
		}
	}
	return;
}

// With probability, set wantTrueBestIfMoveout true for teamUnhealthy data moves and teamRedundant data moves only.
// This flag takes effect in getTeam. When the flag is set true, DD always getBestTeam for teamRedundant data moves and
// do getBestTeam for a teamRedundant data move if the data move decides to move data out of a SS.
bool getWantTrueBestIfMoveout(int priority) {
	if (priority == SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY) {
		return deterministicRandom()->random01() <
		       SERVER_KNOBS->PROBABILITY_TEAM_UNHEALTHY_DATAMOVE_CHOOSE_TRUE_BEST_DEST;
	} else if (priority == SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT) {
		return deterministicRandom()->random01() <
		       SERVER_KNOBS->PROBABILITY_TEAM_REDUNDANT_DATAMOVE_CHOOSE_TRUE_BEST_DEST;
	} else {
		return false;
	}
}

// This actor relocates the specified keys to a good place.
// The inFlightActor key range map stores the actor for each RelocateData
															#line 2351 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via dataDistributionRelocator()
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class DataDistributionRelocatorActor>
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class DataDistributionRelocatorActorState {
															#line 2358 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataDistributionRelocatorActorState(DDQueue* const& self,RelocateData const& rd,Future<Void> const& prevCleanup,const DDEnabledState* const& ddEnabledState) 
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   rd(rd),
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   prevCleanup(prevCleanup),
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   ddEnabledState(ddEnabledState),
															#line 1506 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   errorOut(self->error),
															#line 1507 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   relocateShardInterval("RelocateShard", rd.randomId),
															#line 1508 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   dataTransferComplete(self->dataTransferComplete),
															#line 1509 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   relocationComplete(self->relocationComplete),
															#line 1510 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   signalledTransferComplete(false),
															#line 1511 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   distributorId(self->distributorId),
															#line 1512 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   healthyDestinations(),
															#line 1514 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   anyHealthy(false),
															#line 1515 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   allHealthy(true),
															#line 1516 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   anyWithSource(false),
															#line 1517 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   anyDestOverloaded(false),
															#line 1518 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destOverloadedCount(0),
															#line 1519 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   stuckCount(0),
															#line 1520 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   bestTeams(),
															#line 1521 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   startTime(now()),
															#line 1522 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destIds(),
															#line 1523 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   wantTrueBest(isValleyFillerPriority(rd.priority)),
															#line 1524 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   wantTrueBestIfMoveout(getWantTrueBestIfMoveout(rd.priority)),
															#line 1525 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   debugID(deterministicRandom()->randomUInt64()),
															#line 1526 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   enableShardMove(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD),
															#line 1530 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   doBulkLoading(rd.bulkLoadTask.present())
															#line 2413 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("dataDistributionRelocator", reinterpret_cast<unsigned long>(this));

	}
	~DataDistributionRelocatorActorState() 
	{
		fdb_probe_actor_destroy("dataDistributionRelocator", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1533 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (now() - self->lastInterval < 1.0)
															#line 2429 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1534 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					relocateShardInterval.severity = SevDebug;
															#line 1535 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->suppressIntervals++;
															#line 2435 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
															#line 1538 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(relocateShardInterval.begin(), distributorId) .detail("KeyBegin", rd.keys.begin) .detail("KeyEnd", rd.keys.end) .detail("Priority", rd.priority) .detail("WantTrueBestIfMoveout", wantTrueBestIfMoveout) .detail("SuppressedEventCount", self->suppressIntervals);
															#line 1545 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (relocateShardInterval.severity != SevDebug)
															#line 2441 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1546 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->lastInterval = now();
															#line 1547 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->suppressIntervals = 0;
															#line 2447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
															#line 1550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2451 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1551 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					auto inFlightRange = self->inFlight.rangeContaining(rd.keys.begin);
															#line 1552 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(inFlightRange.range() == rd.keys);
															#line 1553 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(inFlightRange.value().randomId == rd.randomId);
															#line 1554 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ASSERT(inFlightRange.value().dataMoveId == rd.dataMoveId);
															#line 1555 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					inFlightRange.value().cancellable = false;
															#line 1557 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					StrictFuture<Void> __when_expr_0 = prevCleanup;
															#line 1557 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2467 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 1;
															#line 1557 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 0, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2472 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DataDistributionRelocatorActorState();
		static_cast<DataDistributionRelocatorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2406 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			err = e;
															#line 2407 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(relocateShardInterval.end(), distributorId) .errorUnsuppressed(err) .detail("Duration", now() - startTime);
															#line 2410 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (now() - startTime > 600)
															#line 2511 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2411 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarnAlways, "RelocateShardTooLong") .errorUnsuppressed(err) .detail("Duration", now() - startTime) .detail("Dest", describe(destIds)) .detail("Src", describe(rd.src));
															#line 2515 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2417 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 2519 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2418 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				dataTransferComplete.send(rd);
															#line 2523 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2420 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			relocationComplete.send(rd);
															#line 2422 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading && e.code() != error_code_actor_cancelled && e.code() != error_code_movekeys_conflict)
															#line 2529 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2423 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskRelocatorFailed", self->distributorId) .errorUnsuppressed(e) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
															#line 2533 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2430 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (err.code() == error_code_data_move_dest_team_not_found)
															#line 2537 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2431 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_14 = cancelDataMove(self, rd.keys, ddEnabledState);
															#line 2431 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2543 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_14.isReady()) { if (__when_expr_14.isError()) return a_body1Catch1(__when_expr_14.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_14.get(), loopDepth); };
				static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 13;
															#line 2431 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_14.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 14, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2548 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 2435 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (err.code() != error_code_actor_cancelled && err.code() != error_code_data_move_cancelled)
															#line 2555 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 2436 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (errorOut.canBeSet())
															#line 2559 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					{
															#line 2437 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						errorOut.sendError(err);
															#line 2563 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
				}
				loopDepth = a_body1Catch2cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1646 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		parentMetrics = Optional<StorageMetrics>();
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		metrics = StorageMetrics();
															#line 1649 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		Future<StorageMetrics> metricsF = brokenPromiseToNever(self->getShardMetrics.getReply(GetMetricsRequest(rd.keys)));
															#line 1651 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (rd.getParentRange().present())
															#line 2587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1652 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			Future<StorageMetrics> parentMetricsF = brokenPromiseToNever(self->getShardMetrics.getReply(GetMetricsRequest(rd.getParentRange().get())));
															#line 1654 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_3 = store(metrics, metricsF) && store(parentMetrics, parentMetricsF);
															#line 1654 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2595 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont2when1(__when_expr_3.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 4;
															#line 1654 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 3, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2600 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 1656 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_4 = store(metrics, metricsF);
															#line 1656 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2609 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), loopDepth); else return a_body1cont2when2(__when_expr_4.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 5;
															#line 1656 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 4, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2614 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1559 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 2624 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1565 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			tr = Transaction(self->txnProcessor->context());
															#line 1566 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 2630 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = a_body1cont5loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1559 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 2644 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1565 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			tr = Transaction(self->txnProcessor->context());
															#line 1566 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 2650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = a_body1cont5loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 1622 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto f = self->dataMoves.intersectingRanges(rd.keys);
															#line 1623 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(auto it = f.begin();it != f.end();++it) {
															#line 1624 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			KeyRangeRef kr(it->range().begin, it->range().end);
															#line 1625 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			const UID mId = it->value().id;
															#line 1626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (mId.isValid() && mId != rd.dataMoveId)
															#line 2747 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1627 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent("DDRelocatorConflictingDataMove", distributorId) .detail("CurrentDataMoveID", rd.dataMoveId) .detail("DataMoveID", mId) .detail("Range", kr);
															#line 2751 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
															#line 1633 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (rd.isRestore() || !SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 2756 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1634 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 2760 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1635 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(rd.dataMoveId.isValid());
															#line 2764 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 1637 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataMoves.insert(rd.keys, DDQueue::DDDataMove(rd.dataMoveId));
															#line 2768 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
															#line 1609 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		DataMoveType dataMoveType = newDataMoveType(doBulkLoading);
															#line 1610 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		rd.dataMoveId = newDataMoveId( deterministicRandom()->randomUInt64(), AssignEmptyRange::False, dataMoveType, rd.dmReason);
															#line 1612 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskNewDataMoveID", self->distributorId) .detail("DataMoveID", rd.dataMoveId.toString()) .detail("TrackID", rd.randomId) .detail("Range", rd.keys) .detail("Priority", rd.priority) .detail("DataMoveType", dataMoveType) .detail("DoBulkLoading", doBulkLoading) .detail("DataMoveReason", static_cast<int>(rd.dmReason));
															#line 2782 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont5loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1(int loopDepth) 
	{
		try {
															#line 1568 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<BulkLoadTaskState> __when_expr_1 = getBulkLoadTask(&tr, rd.bulkLoadTask.get().coreState.getRange(), rd.bulkLoadTask.get().coreState.getTaskId(), { BulkLoadPhase::Triggered, BulkLoadPhase::Running });
															#line 1568 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1cont5loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2801 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont5loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont5loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 2;
															#line 1568 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont5loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont5loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5break1(int loopDepth) 
	{
		try {
			return a_body1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1584 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (e.code() == error_code_bulkload_task_outdated)
															#line 2841 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1588 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (rd.bulkLoadTask.get().completeAck.canBeSet())
															#line 2845 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1589 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					rd.bulkLoadTask.get().completeAck.sendError(bulkload_task_outdated());
															#line 2849 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
															#line 1591 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				doBulkLoading = false;
															#line 1598 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskFallbackToNormalDataMove", self->distributorId) .detail("TrackID", rd.randomId) .detail("DataMovePriority", rd.priority) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange());
															#line 2855 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1606 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1606 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2862 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 3;
															#line 1606 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 2, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2867 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2(BulkLoadTaskState const& currentBulkLoadTaskState,int loopDepth) 
	{
															#line 1573 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDataMoveLaunched", self->distributorId) .detail("TrackID", rd.randomId) .detail("DataMovePriority", rd.priority) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("ExistingJobID", currentBulkLoadTaskState.getJobId()) .detail("ExistingTaskID", currentBulkLoadTaskState.getTaskId()) .detail("ExistingTaskRange", currentBulkLoadTaskState.getRange());
															#line 2882 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2(BulkLoadTaskState && currentBulkLoadTaskState,int loopDepth) 
	{
															#line 1573 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDataMoveLaunched", self->distributorId) .detail("TrackID", rd.randomId) .detail("DataMovePriority", rd.priority) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("ExistingJobID", currentBulkLoadTaskState.getJobId()) .detail("ExistingTaskID", currentBulkLoadTaskState.getTaskId()) .detail("ExistingTaskRange", currentBulkLoadTaskState.getRange());
															#line 2891 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(BulkLoadTaskState const& currentBulkLoadTaskState,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont2(currentBulkLoadTaskState, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(BulkLoadTaskState && currentBulkLoadTaskState,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont2(std::move(currentBulkLoadTaskState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*,BulkLoadTaskState const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*,BulkLoadTaskState && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont14(int loopDepth) 
	{
															#line 1659 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		excludedDstPhysicalShards = std::unordered_set<uint64_t>();
															#line 1661 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(rd.src.size());
															#line 1662 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		;
															#line 3066 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont14loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont15(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont15(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont16(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont14loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont14loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1(int loopDepth) 
	{
															#line 1663 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destOverloadedCount = 0;
															#line 1664 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		stuckCount = 0;
															#line 1665 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		physicalShardIDCandidate = UID().first();
															#line 1666 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		forceToUseNewPhysicalShard = false;
															#line 1667 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		;
															#line 3264 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont14loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1(int loopDepth) 
	{
															#line 2003 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (enableShardMove)
															#line 3273 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2005 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!rd.isRestore())
															#line 3277 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2008 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(physicalShardIDCandidate != UID().first());
															#line 2009 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (self->physicalShardCollection->physicalShardExists(physicalShardIDCandidate))
															#line 3283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 2010 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->moveReusePhysicalShard++;
															#line 3287 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
				else
				{
															#line 2012 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->moveCreateNewPhysicalShard++;
															#line 3293 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
															#line 2014 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				rd.dataMoveId = newDataMoveId( physicalShardIDCandidate, AssignEmptyRange::False, newDataMoveType(doBulkLoading), rd.dmReason);
															#line 2016 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevInfo, "NewDataMoveWithPhysicalShard") .detail("DataMoveID", rd.dataMoveId.toString()) .detail("Reason", rd.reason.toString()) .detail("DataMoveReason", static_cast<int>(rd.dmReason));
															#line 2020 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto inFlightRange = self->inFlight.rangeContaining(rd.keys.begin);
															#line 2021 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				inFlightRange.value().dataMoveId = rd.dataMoveId;
															#line 2022 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto f = self->dataMoves.intersectingRanges(rd.keys);
															#line 2023 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for(auto it = f.begin();it != f.end();++it) {
															#line 2024 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					KeyRangeRef kr(it->range().begin, it->range().end);
															#line 2025 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					const UID mId = it->value().id;
															#line 2026 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (mId.isValid() && mId != rd.dataMoveId)
															#line 3313 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					{
															#line 2027 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						TraceEvent("DDRelocatorConflictingDataMoveAfterGetTeam", distributorId) .detail("CurrentDataMoveID", rd.dataMoveId) .detail("DataMoveID", mId) .detail("Range", kr);
															#line 3317 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
				}
															#line 2033 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->dataMoves.insert(rd.keys, DDQueue::DDDataMove(rd.dataMoveId));
															#line 3322 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2035 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rd.dataMoveId.first() != UID().first());
															#line 2036 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto dataMoveRange = self->dataMoves.rangeContaining(rd.keys.begin);
															#line 2037 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(dataMoveRange.value().id == rd.dataMoveId);
															#line 3330 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2041 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto inFlightRange = self->inFlight.rangeContaining(rd.keys.begin);
															#line 2042 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(inFlightRange.range() == rd.keys);
															#line 2043 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(inFlightRange.value().randomId == rd.randomId);
															#line 2044 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		inFlightRange.value().cancellable = false;
															#line 2046 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destIds.clear();
															#line 2047 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		healthyIds = std::vector<UID>();
															#line 2048 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		extraIds = std::vector<UID>();
															#line 2049 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destinationTeams = std::vector<ShardsAffectedByTeamFailure::Team>();
															#line 2051 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < bestTeams.size();i++) {
															#line 2052 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto& serverIds = bestTeams[i].first->getServerIDs();
															#line 2053 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			destinationTeams.push_back(ShardsAffectedByTeamFailure::Team(serverIds, i == 0));
															#line 2056 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (allHealthy && anyWithSource && !bestTeams[i].second && !doBulkLoading)
															#line 3356 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2065 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				int idx = deterministicRandom()->randomInt(0, serverIds.size());
															#line 2066 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				destIds.push_back(serverIds[idx]);
															#line 2067 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				healthyIds.push_back(serverIds[idx]);
															#line 2068 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for(int j = 0;j < serverIds.size();j++) {
															#line 2069 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (j != idx)
															#line 3368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					{
															#line 2070 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						extraIds.push_back(serverIds[j]);
															#line 3372 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
				}
															#line 2073 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				healthyDestinations.addTeam(bestTeams[i].first);
															#line 3377 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2075 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				destIds.insert(destIds.end(), serverIds.begin(), serverIds.end());
															#line 2076 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (bestTeams[i].first->isHealthy())
															#line 3385 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 2077 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					healthyIds.insert(healthyIds.end(), serverIds.begin(), serverIds.end());
															#line 2078 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					healthyDestinations.addTeam(bestTeams[i].first);
															#line 3391 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
		}
															#line 2084 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3397 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2085 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			validateBulkLoadRelocateData(rd, destIds, self->distributorId);
															#line 3401 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2087 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevInfo, "DDRelocatorGotDestTeam", self->distributorId) .detail("KeyBegin", rd.keys.begin) .detail("KeyEnd", rd.keys.end) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("SrcIds", describe(rd.src)) .detail("DestId", describe(destIds)) .detail("BulkLoadTaskID", doBulkLoading ? rd.bulkLoadTask.get().coreState.getTaskId().toString() : "");
															#line 2097 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		totalIds = 0;
															#line 2098 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for( auto& destTeam : destinationTeams ) {
															#line 2099 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			totalIds += destTeam.servers.size();
															#line 3411 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2101 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (totalIds < self->teamSize)
															#line 3415 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2102 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(SevWarn, "IncorrectDestTeamSize") .suppressFor(1.0) .detail("ExpectedTeamSize", self->teamSize) .detail("DestTeamSize", totalIds);
															#line 3419 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2108 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!rd.isRestore())
															#line 3423 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2109 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->shardsAffectedByTeamFailure->moveShard(rd.keys, destinationTeams);
															#line 3427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2113 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		healthyDestinations.addDataInFlightToTeam(+metrics.bytes);
															#line 2114 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		healthyDestinations.addReadInFlightToTeam(+metrics.readLoadKSecond());
															#line 2118 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchDest(rd, bestTeams, self->destBusymap);
															#line 2119 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3437 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2120 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for( const auto& [team, _] : bestTeams ) {
															#line 2121 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for( const UID& ssid : team->getServerIDs() ) {
															#line 2122 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->bulkLoadTaskCollection->busyMap.addTask(ssid);
															#line 3445 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
		}
															#line 2127 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent ev(relocateShardInterval.severity, "RelocateShardHasDestination", distributorId);
															#line 2128 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		RelocateDecision decision{ rd, destIds, extraIds, metrics, parentMetrics };
															#line 2129 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceRelocateDecision(ev, relocateShardInterval.pairID, decision);
															#line 2131 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->serverCounter.increaseForTeam(rd.src, rd.reason, DDQueue::ServerCounter::LaunchedSource);
															#line 2132 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->serverCounter.increaseForTeam(destIds, rd.reason, DDQueue::ServerCounter::LaunchedDest);
															#line 2133 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->serverCounter.increaseForTeam(extraIds, rd.reason, DDQueue::ServerCounter::LaunchedDest);
															#line 2135 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		error = success();
															#line 2136 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		dataMovementComplete = Promise<Void>();
															#line 2138 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::unique_ptr<MoveKeysParams> params;
															#line 2139 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 3469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			params = std::make_unique<MoveKeysParams>(rd.dataMoveId, std::vector<KeyRange>{ rd.keys }, destIds, healthyIds, self->lock, dataMovementComplete, &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, doBulkLoading ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 3473 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2155 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			params = std::make_unique<MoveKeysParams>(rd.dataMoveId, rd.keys, destIds, healthyIds, self->lock, dataMovementComplete, &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, doBulkLoading ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 3479 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2170 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		doMoveKeys = self->txnProcessor->moveKeys(*params);
															#line 2171 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		pollHealth = signalledTransferComplete ? Never() : delay(SERVER_KNOBS->HEALTH_POLL_TIME, TaskPriority::DataDistributionLaunch);
															#line 3485 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		try {
															#line 2175 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 3489 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = a_body1cont14loopBody1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont14loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont14loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont14loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1(int loopDepth) 
	{
															#line 1668 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		tciIndex = 0;
															#line 1669 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		foundTeams = true;
															#line 1670 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bestTeamReady = false;
															#line 1671 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyHealthy = false;
															#line 1672 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		allHealthy = true;
															#line 1673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyWithSource = false;
															#line 1674 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyDestOverloaded = false;
															#line 1675 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bestTeams.clear();
															#line 1677 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		;
															#line 3527 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont14loopBody1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont14loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1910 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyDestOverloaded = !canLaunchDest(bestTeams, rd.priority, self->destBusymap);
															#line 1911 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1912 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyDestOverloaded = false;
															#line 3555 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 1915 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (foundTeams && anyHealthy && !anyDestOverloaded)
															#line 3559 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1916 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rd.completeDests.empty());
															#line 3563 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1920 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (foundTeams)
															#line 3568 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1921 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!anyHealthy)
															#line 3572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1922 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::NoAnyHealthy]++;
															#line 3576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1923 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (anyDestOverloaded)
															#line 3582 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1924 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::DstOverloaded]++;
															#line 3586 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
		}
															#line 1928 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (anyDestOverloaded)
															#line 3592 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1929 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			CODE_PROBE(true, "Destination overloaded throttled move");
															#line 1930 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			destOverloadedCount++;
															#line 1931 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(destOverloadedCount > 50 ? SevInfo : SevDebug, "DestSSBusy", distributorId) .suppressFor(1.0) .detail("TraceID", rd.randomId) .detail("WantTrueBestIfMoveout", wantTrueBestIfMoveout) .detail("IsRestore", rd.isRestore()) .detail("Priority", rd.priority) .detail("StuckCount", stuckCount) .detail("DestOverloadedCount", destOverloadedCount) .detail("TeamCollectionId", tciIndex) .detail("AnyDestOverloaded", anyDestOverloaded) .detail("NumOfTeamCollections", self->teamCollections.size()) .detail("Servers", destServersString(bestTeams));
															#line 1943 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (enableShardMove)
															#line 3602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1944 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (rd.isRestore() && destOverloadedCount > 50)
															#line 3606 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1945 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					return a_body1Catch2(data_move_dest_team_not_found(), std::max(0, loopDepth - 2));
															#line 3610 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
															#line 1948 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_7 = delay(SERVER_KNOBS->DEST_OVERLOADED_DELAY, TaskPriority::DataDistributionLaunch);
															#line 1948 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 3617 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch2(__when_expr_7.getError(), std::max(0, loopDepth - 2)); else return a_body1cont14loopBody1loopBody1cont1when1(__when_expr_7.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 8;
															#line 1948 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 7, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 3622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 1950 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			CODE_PROBE(true, "did not find a healthy destination team on the first attempt");
															#line 1951 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			stuckCount++;
															#line 1952 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(stuckCount > 50 ? SevWarnAlways : SevWarn, "BestTeamStuck", distributorId) .suppressFor(1.0) .detail("TraceID", rd.randomId) .detail("WantTrueBestIfMoveout", wantTrueBestIfMoveout) .detail("IsRestore", rd.isRestore()) .detail("Priority", rd.priority) .detail("StuckCount", stuckCount) .detail("DestOverloadedCount", destOverloadedCount) .detail("TeamCollectionId", tciIndex) .detail("AnyDestOverloaded", anyDestOverloaded) .detail("NumOfTeamCollections", self->teamCollections.size()) .detail("ConductingBulkLoad", doBulkLoading);
															#line 1964 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading && stuckCount == 50)
															#line 3635 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1965 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(rd.bulkLoadTask.present());
															#line 1966 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskStuck", self->distributorId) .detail("TraceID", rd.randomId) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("DataMoveID", rd.dataMoveId) .detail("Reason", rd.reason.toString()) .detail("Priority", rd.priority) .detail("DataMoveReason", static_cast<int>(rd.dmReason));
															#line 1975 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (rd.bulkLoadTask.get().completeAck.canBeSet())
															#line 3643 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1977 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					rd.bulkLoadTask.get().completeAck.send(BulkLoadAck( true, rd.priority));
															#line 1978 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					return a_body1Catch2(data_move_dest_team_not_found(), std::max(0, loopDepth - 2));
															#line 3649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
															#line 1984 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rd.isRestore() && stuckCount > 50)
															#line 3654 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1985 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				return a_body1Catch2(data_move_dest_team_not_found(), std::max(0, loopDepth - 2));
															#line 3658 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 1987 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_8 = delay(SERVER_KNOBS->BEST_TEAM_STUCK_DELAY, TaskPriority::DataDistributionLaunch);
															#line 1987 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 3664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch2(__when_expr_8.getError(), std::max(0, loopDepth - 2)); else return a_body1cont14loopBody1loopBody1cont1when2(__when_expr_8.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 9;
															#line 1987 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 8, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 3669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont14loopBody1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1(int loopDepth) 
	{
															#line 1677 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!(tciIndex < self->teamCollections.size()))
															#line 3686 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1678 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && rd.isRestore())
															#line 3692 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1679 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto req = GetTeamRequest(tciIndex == 0 ? rd.dataMove->primaryDest : rd.dataMove->remoteDest);
															#line 1680 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.keys = rd.keys;
															#line 1681 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			Future<std::pair<Optional<Reference<IDataDistributionTeam>>, bool>> fbestTeam = brokenPromiseToNever(self->teamCollections[tciIndex].getTeam.getReply(req));
															#line 1683 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeamReady = fbestTeam.isReady();
															#line 1684 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<std::pair<Optional<Reference<IDataDistributionTeam>>, bool>> __when_expr_5 = fbestTeam;
															#line 1684 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 3));
															#line 3706 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), std::max(0, loopDepth - 3)); else return a_body1cont14loopBody1loopBody1loopBody1when1(__when_expr_5.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 6;
															#line 1684 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 3711 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
															#line 1721 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			double inflightPenalty = SERVER_KNOBS->INFLIGHT_PENALTY_HEALTHY;
															#line 1722 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rd.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY || rd.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_2_LEFT)
															#line 3720 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1724 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				inflightPenalty = SERVER_KNOBS->INFLIGHT_PENALTY_UNHEALTHY;
															#line 3724 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 1725 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rd.healthPriority == SERVER_KNOBS->PRIORITY_POPULATE_REGION || rd.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_1_LEFT || rd.healthPriority == SERVER_KNOBS->PRIORITY_TEAM_0_LEFT)
															#line 3728 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1728 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				inflightPenalty = SERVER_KNOBS->INFLIGHT_PENALTY_ONE_LEFT;
															#line 3732 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 1730 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TeamSelect destTeamSelect;
															#line 1731 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!rd.wantsNewServers)
															#line 3738 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1732 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				destTeamSelect = TeamSelect::WANT_COMPLETE_SRCS;
															#line 3742 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1733 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (wantTrueBest)
															#line 3748 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1734 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					destTeamSelect = TeamSelect::WANT_TRUE_BEST;
															#line 3752 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1736 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					destTeamSelect = TeamSelect::ANY;
															#line 3758 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
															#line 1738 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			PreferLowerReadUtil preferLowerReadTeam = SERVER_KNOBS->DD_PREFER_LOW_READ_UTIL_TEAM || rd.reason == RelocateReason::REBALANCE_READ ? PreferLowerReadUtil::True : PreferLowerReadUtil::False;
															#line 1742 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto req = GetTeamRequest(destTeamSelect, PreferLowerDiskUtil::True, TeamMustHaveShards::False, preferLowerReadTeam, PreferWithinShardLimit::True, ForReadBalance(rd.reason == RelocateReason::REBALANCE_READ), inflightPenalty, rd.keys);
															#line 1751 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.src = rd.src;
															#line 1752 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.completeSources = rd.completeSources;
															#line 1753 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.storageQueueAware = SERVER_KNOBS->ENABLE_STORAGE_QUEUE_AWARE_TEAM_SELECTION;
															#line 1754 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.findTeamForBulkLoad = doBulkLoading;
															#line 1755 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req.wantTrueBestIfMoveout = wantTrueBestIfMoveout;
															#line 1757 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (enableShardMove && tciIndex == 1)
															#line 3777 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1758 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(physicalShardIDCandidate != UID().first() && physicalShardIDCandidate != anonymousShardId.first());
															#line 1760 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				std::pair<Optional<ShardsAffectedByTeamFailure::Team>, bool> remoteTeamWithPhysicalShard = self->physicalShardCollection->tryGetAvailableRemoteTeamWith( physicalShardIDCandidate, metrics, debugID);
															#line 1763 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!remoteTeamWithPhysicalShard.second)
															#line 3785 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1766 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]++;
															#line 1767 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 3791 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
															#line 1770 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (remoteTeamWithPhysicalShard.first.present())
															#line 3796 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1773 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					req = GetTeamRequest(remoteTeamWithPhysicalShard.first.get().servers);
															#line 1774 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					req.keys = rd.keys;
															#line 3802 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
															#line 1781 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			Future<std::pair<Optional<Reference<IDataDistributionTeam>>, bool>> fbestTeam = brokenPromiseToNever(self->teamCollections[tciIndex].getTeam.getReply(req));
															#line 1783 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeamReady = fbestTeam.isReady();
															#line 1784 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<std::pair<Optional<Reference<IDataDistributionTeam>>, bool>> __when_expr_6 = fbestTeam;
															#line 1784 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 3));
															#line 3813 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch2(__when_expr_6.getError(), std::max(0, loopDepth - 3)); else return a_body1cont14loopBody1loopBody1loopBody1when2(__when_expr_6.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 7;
															#line 1784 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 3818 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont14loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1905 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		tciIndex++;
															#line 3841 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont14loopBody1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1cont3(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& bestTeam,int loopDepth) 
	{
															#line 1685 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (tciIndex > 0 && !bestTeamReady)
															#line 3850 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1689 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("BestTeamNotReady") .detail("TraceID", rd.randomId) .detail("TeamCollectionIndex", tciIndex) .detail("RestoreDataMoveForDest", describe(tciIndex == 0 ? rd.dataMove->primaryDest : rd.dataMove->remoteDest));
															#line 1694 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]++;
															#line 1695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 3858 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1698 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.present() || !bestTeam.first.get()->isHealthy())
															#line 3863 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1699 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[tciIndex == 0 ? DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam : DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]++;
															#line 1702 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 3869 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyHealthy = true;
															#line 1706 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bestTeams.emplace_back(bestTeam.first.get(), bestTeam.second);
															#line 1707 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3878 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1708 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSelectDestTeam", self->distributorId) .detail("Context", "Restore") .detail("SrcIds", describe(rd.src)) .detail("DestIds", bestTeam.first.get()->getServerIDs()) .detail("DestTeam", bestTeam.first.get()->getTeamID()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("Primary", tciIndex == 0);
															#line 3882 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1cont3(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && bestTeam,int loopDepth) 
	{
															#line 1685 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (tciIndex > 0 && !bestTeamReady)
															#line 3892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1689 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("BestTeamNotReady") .detail("TraceID", rd.randomId) .detail("TeamCollectionIndex", tciIndex) .detail("RestoreDataMoveForDest", describe(tciIndex == 0 ? rd.dataMove->primaryDest : rd.dataMove->remoteDest));
															#line 1694 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]++;
															#line 1695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 3900 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1698 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.present() || !bestTeam.first.get()->isHealthy())
															#line 3905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1699 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[tciIndex == 0 ? DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam : DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]++;
															#line 1702 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 3911 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		anyHealthy = true;
															#line 1706 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bestTeams.emplace_back(bestTeam.first.get(), bestTeam.second);
															#line 1707 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 3920 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1708 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSelectDestTeam", self->distributorId) .detail("Context", "Restore") .detail("SrcIds", describe(rd.src)) .detail("DestIds", bestTeam.first.get()->getServerIDs()) .detail("DestTeam", bestTeam.first.get()->getTeamID()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskRange", rd.bulkLoadTask.get().coreState.getRange()) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("Primary", tciIndex == 0);
															#line 3924 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1when1(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& bestTeam,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont3(bestTeam, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1when1(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && bestTeam,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont3(std::move(bestTeam), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont14loopBody1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont14loopBody1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont14loopBody1loopBody1loopBody1cont8(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& bestTeam,int loopDepth) 
	{
															#line 1785 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 4009 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1786 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorBestTeamReceived", self->distributorId) .detail("DataMoveID", rd.dataMoveId) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("BestTeamReady", bestTeamReady);
															#line 4013 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 1794 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (tciIndex > 0 && !bestTeamReady)
															#line 4017 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1798 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("BestTeamNotReady");
															#line 1799 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]++;
															#line 1800 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 4025 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1805 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.present())
															#line 4030 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[tciIndex == 0 ? DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam : DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]++;
															#line 1809 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 4036 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1812 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.get()->isHealthy())
															#line 4041 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1813 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			allHealthy = false;
															#line 4045 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 1815 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyHealthy = true;
															#line 4051 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 1818 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (bestTeam.second)
															#line 4055 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1819 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyWithSource = true;
															#line 4059 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 1822 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (enableShardMove)
															#line 4063 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1823 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (tciIndex == 1 && !forceToUseNewPhysicalShard)
															#line 4067 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1830 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				bool minAvailableSpaceRatio = bestTeam.first.get()->getMinAvailableSpaceRatio(true);
															#line 1831 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (minAvailableSpaceRatio < SERVER_KNOBS->TARGET_AVAILABLE_SPACE_RATIO)
															#line 4073 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1832 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsFull]++;
															#line 1833 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 4079 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
															#line 1843 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!bestTeam.first.get()->isHealthy())
															#line 4084 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1844 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::RemoteTeamIsNotHealthy]++;
															#line 1846 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 4090 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
			}
															#line 1851 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeams.emplace_back(bestTeam.first.get(), true);
															#line 1858 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (tciIndex == 0)
															#line 4098 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1859 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(foundTeams);
															#line 1860 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ShardsAffectedByTeamFailure::Team primaryTeam = ShardsAffectedByTeamFailure::Team(bestTeams[0].first->getServerIDs(), true);
															#line 1863 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (forceToUseNewPhysicalShard)
															#line 4106 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1864 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					physicalShardIDCandidate = self->physicalShardCollection->generateNewPhysicalShardID(debugID);
															#line 4110 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1867 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					Optional<uint64_t> candidate = self->physicalShardCollection->trySelectAvailablePhysicalShardFor( primaryTeam, metrics, excludedDstPhysicalShards, debugID);
															#line 1870 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (candidate.present())
															#line 4118 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					{
															#line 1871 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						physicalShardIDCandidate = candidate.get();
															#line 4122 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
					else
					{
															#line 1873 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]++;
															#line 1875 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						if (wantTrueBest)
															#line 4130 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
						{
															#line 1879 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
							forceToUseNewPhysicalShard = true;
															#line 4134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
						}
															#line 1881 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						foundTeams = false;
															#line 4138 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
						return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
					}
				}
															#line 1885 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(physicalShardIDCandidate != UID().first() && physicalShardIDCandidate != anonymousShardId.first());
															#line 4144 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 1889 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeams.emplace_back(bestTeam.first.get(), bestTeam.second);
															#line 1890 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 4153 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1891 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent( bulkLoadVerboseEventSev(), "DDBulkLoadTaskSelectDestTeam", self->distributorId) .detail("Context", "New") .detail("SrcIds", describe(rd.src)) .detail("DestIds", bestTeam.first.get()->getServerIDs()) .detail("DestTeam", bestTeam.first.get()->getTeamID()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("Primary", tciIndex == 0);
															#line 4157 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1cont8(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && bestTeam,int loopDepth) 
	{
															#line 1785 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (doBulkLoading)
															#line 4168 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1786 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorBestTeamReceived", self->distributorId) .detail("DataMoveID", rd.dataMoveId) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("BestTeamReady", bestTeamReady);
															#line 4172 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 1794 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (tciIndex > 0 && !bestTeamReady)
															#line 4176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1798 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("BestTeamNotReady");
															#line 1799 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]++;
															#line 1800 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 4184 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1805 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.present())
															#line 4189 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->retryFindDstReasonCount[tciIndex == 0 ? DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam : DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]++;
															#line 1809 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			foundTeams = false;
															#line 4195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1812 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!bestTeam.first.get()->isHealthy())
															#line 4200 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1813 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			allHealthy = false;
															#line 4204 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 1815 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyHealthy = true;
															#line 4210 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 1818 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (bestTeam.second)
															#line 4214 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1819 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			anyWithSource = true;
															#line 4218 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 1822 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (enableShardMove)
															#line 4222 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1823 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (tciIndex == 1 && !forceToUseNewPhysicalShard)
															#line 4226 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1830 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				bool minAvailableSpaceRatio = bestTeam.first.get()->getMinAvailableSpaceRatio(true);
															#line 1831 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (minAvailableSpaceRatio < SERVER_KNOBS->TARGET_AVAILABLE_SPACE_RATIO)
															#line 4232 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1832 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsFull]++;
															#line 1833 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 4238 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
															#line 1843 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!bestTeam.first.get()->isHealthy())
															#line 4243 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1844 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::RemoteTeamIsNotHealthy]++;
															#line 1846 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					foundTeams = false;
															#line 4249 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
				}
			}
															#line 1851 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeams.emplace_back(bestTeam.first.get(), true);
															#line 1858 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (tciIndex == 0)
															#line 4257 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1859 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(foundTeams);
															#line 1860 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ShardsAffectedByTeamFailure::Team primaryTeam = ShardsAffectedByTeamFailure::Team(bestTeams[0].first->getServerIDs(), true);
															#line 1863 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (forceToUseNewPhysicalShard)
															#line 4265 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 1864 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					physicalShardIDCandidate = self->physicalShardCollection->generateNewPhysicalShardID(debugID);
															#line 4269 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1867 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					Optional<uint64_t> candidate = self->physicalShardCollection->trySelectAvailablePhysicalShardFor( primaryTeam, metrics, excludedDstPhysicalShards, debugID);
															#line 1870 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (candidate.present())
															#line 4277 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					{
															#line 1871 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						physicalShardIDCandidate = candidate.get();
															#line 4281 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
					else
					{
															#line 1873 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]++;
															#line 1875 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						if (wantTrueBest)
															#line 4289 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
						{
															#line 1879 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
							forceToUseNewPhysicalShard = true;
															#line 4293 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
						}
															#line 1881 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						foundTeams = false;
															#line 4297 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
						return a_body1cont14loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
					}
				}
															#line 1885 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ASSERT(physicalShardIDCandidate != UID().first() && physicalShardIDCandidate != anonymousShardId.first());
															#line 4303 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 1889 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			bestTeams.emplace_back(bestTeam.first.get(), bestTeam.second);
															#line 1890 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 4312 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1891 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent( bulkLoadVerboseEventSev(), "DDBulkLoadTaskSelectDestTeam", self->distributorId) .detail("Context", "New") .detail("SrcIds", describe(rd.src)) .detail("DestIds", bestTeam.first.get()->getServerIDs()) .detail("DestTeam", bestTeam.first.get()->getTeamID()) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId()) .detail("Priority", rd.priority) .detail("DataMoveId", rd.dataMoveId) .detail("Primary", tciIndex == 0);
															#line 4316 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1when2(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& bestTeam,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont8(bestTeam, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1loopBody1when2(std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && bestTeam,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1loopBody1cont8(std::move(bestTeam), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,std::pair<Optional<Reference<IDataDistributionTeam>>, bool> const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont14loopBody1loopBody1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,std::pair<Optional<Reference<IDataDistributionTeam>>, bool> && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont14loopBody1loopBody1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont14loopBody1loopBody1cont2(int loopDepth) 
	{
															#line 1992 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (enableShardMove)
															#line 4402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 1993 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (destOverloadedCount + stuckCount > 20)
															#line 4406 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 1994 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RetryLimitReached]++;
															#line 1995 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				forceToUseNewPhysicalShard = true;
															#line 4412 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 1997 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			excludedDstPhysicalShards.insert(physicalShardIDCandidate);
															#line 4416 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		if (loopDepth == 0) return a_body1cont14loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont14loopBody1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont14loopBody1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont14loopBody1loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1loopBody1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont14loopBody1loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont14loopBody1loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont14loopBody1cont2(int loopDepth) 
	{
															#line 2259 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (error.code() != error_code_move_to_removed_server)
															#line 4600 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2260 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!error.code())
															#line 4604 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
				try {
															#line 2262 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					StrictFuture<Void> __when_expr_12 = healthyDestinations .updateStorageMetrics();
															#line 2262 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1cont14loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 4611 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1cont14loopBody1cont2Catch1(__when_expr_12.getError(), loopDepth); else return a_body1cont14loopBody1cont2when1(__when_expr_12.get(), loopDepth); };
					static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 11;
															#line 2262 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 12, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 4616 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1cont14loopBody1cont2Catch1(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1cont14loopBody1cont2Catch1(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont14loopBody1cont26(loopDepth);
			}
		}
		else
		{
															#line 2375 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			CODE_PROBE(true, "move to removed server", probe::decoration::rare);
															#line 2376 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			healthyDestinations.addDataInFlightToTeam(-metrics.bytes);
															#line 2377 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto readLoad = metrics.readLoadKSecond();
															#line 2378 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto& destinationRef = healthyDestinations;
															#line 2379 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->noErrorActors.add( trigger([destinationRef, readLoad]() mutable { destinationRef.addReadInFlightToTeam(-readLoad); }, delay(SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL)));
															#line 2383 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 4644 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2386 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				completeDest(rd, self->destBusymap);
															#line 4648 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2388 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			rd.completeDests.clear();
															#line 2390 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 4654 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2391 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorError") .errorUnsuppressed(error) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId());
															#line 2395 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for( const auto& [team, _] : bestTeams ) {
															#line 2396 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					for( const UID& ssid : team->getServerIDs() ) {
															#line 2397 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->bulkLoadTaskCollection->busyMap.removeTask(ssid);
															#line 4664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
				}
			}
															#line 2402 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Void> __when_expr_13 = delay(SERVER_KNOBS->RETRY_RELOCATESHARD_DELAY, TaskPriority::DataDistributionLaunch);
															#line 2402 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4672 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1Catch2(__when_expr_13.getError(), std::max(0, loopDepth - 1)); else return a_body1cont14loopBody1cont2when2(__when_expr_13.get(), loopDepth); };
			static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 12;
															#line 2402 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_13.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 13, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 4677 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2254 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			error = e;
															#line 4688 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = a_body1cont14loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont24(int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont24cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont14loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 2177 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_9 = doMoveKeys;
															#line 2176 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state < 0) return a_body1cont14loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4718 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont14loopBody1cont1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1cont14loopBody1cont1loopBody1when1(__when_expr_9.get(), loopDepth); };
															#line 2233 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_10 = pollHealth;
															#line 4722 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont14loopBody1cont1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1cont14loopBody1cont1loopBody1when2(__when_expr_10.get(), loopDepth); };
															#line 2244 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_11 = signalledTransferComplete ? Never() : dataMovementComplete.getFuture();
															#line 4726 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1cont14loopBody1cont1Catch1(__when_expr_11.getError(), std::max(0, loopDepth - 1)); else return a_body1cont14loopBody1cont1loopBody1when3(__when_expr_11.get(), loopDepth); };
		static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 10;
															#line 2177 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 9, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2233 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 10, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 2244 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< DataDistributionRelocatorActor, 11, Void >*>(static_cast<DataDistributionRelocatorActor*>(this)));
															#line 4735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont14loopBody1cont24(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont14loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont14loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont14loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 2178 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (extraIds.size())
															#line 4763 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2179 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			destIds.insert(destIds.end(), extraIds.begin(), extraIds.end());
															#line 2180 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			healthyIds.insert(healthyIds.end(), extraIds.begin(), extraIds.end());
															#line 2181 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			extraIds.clear();
															#line 2182 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(totalIds == destIds.size());
															#line 2183 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			std::unique_ptr<MoveKeysParams> params;
															#line 2184 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 4777 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2185 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				params = std::make_unique<MoveKeysParams>(rd.dataMoveId, std::vector<KeyRange>{ rd.keys }, destIds, healthyIds, self->lock, Promise<Void>(), &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, rd.bulkLoadTask.present() ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 4781 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2201 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				params = std::make_unique<MoveKeysParams>(rd.dataMoveId, rd.keys, destIds, healthyIds, self->lock, Promise<Void>(), &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, rd.bulkLoadTask.present() ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 4787 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2217 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			doMoveKeys = self->txnProcessor->moveKeys(*params);
															#line 4791 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2219 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->fetchKeysComplete.insert(rd);
															#line 2220 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 4799 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2221 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto ranges = self->dataMoves.getAffectedRangesAfterInsertion(rd.keys);
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (ranges.size() == 1 && static_cast<KeyRange>(ranges[0]) == rd.keys && ranges[0].value.id == rd.dataMoveId && !ranges[0].value.cancel.isValid())
															#line 4805 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 2224 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->dataMoves.insert(rd.keys, DDQueue::DDDataMove());
															#line 2225 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevVerbose, "DequeueDataMoveOnSuccess", self->distributorId) .detail("DataMoveID", rd.dataMoveId) .detail("DataMoveRange", rd.keys);
															#line 4811 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
			return a_body1cont14loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 2178 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (extraIds.size())
															#line 4824 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2179 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			destIds.insert(destIds.end(), extraIds.begin(), extraIds.end());
															#line 2180 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			healthyIds.insert(healthyIds.end(), extraIds.begin(), extraIds.end());
															#line 2181 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			extraIds.clear();
															#line 2182 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(totalIds == destIds.size());
															#line 2183 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			std::unique_ptr<MoveKeysParams> params;
															#line 2184 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 4838 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2185 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				params = std::make_unique<MoveKeysParams>(rd.dataMoveId, std::vector<KeyRange>{ rd.keys }, destIds, healthyIds, self->lock, Promise<Void>(), &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, rd.bulkLoadTask.present() ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 4842 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2201 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				params = std::make_unique<MoveKeysParams>(rd.dataMoveId, rd.keys, destIds, healthyIds, self->lock, Promise<Void>(), &self->startMoveKeysParallelismLock, &self->finishMoveKeysParallelismLock, self->teamCollections.size() > 1, relocateShardInterval.pairID, ddEnabledState, CancelConflictingDataMoves::False, rd.bulkLoadTask.present() ? rd.bulkLoadTask.get().coreState : Optional<BulkLoadTaskState>());
															#line 4848 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2217 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			doMoveKeys = self->txnProcessor->moveKeys(*params);
															#line 4852 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2219 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->fetchKeysComplete.insert(rd);
															#line 2220 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 4860 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2221 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto ranges = self->dataMoves.getAffectedRangesAfterInsertion(rd.keys);
															#line 2222 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (ranges.size() == 1 && static_cast<KeyRange>(ranges[0]) == rd.keys && ranges[0].value.id == rd.dataMoveId && !ranges[0].value.cancel.isValid())
															#line 4866 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 2224 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->dataMoves.insert(rd.keys, DDQueue::DDDataMove());
															#line 2225 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent(SevVerbose, "DequeueDataMoveOnSuccess", self->distributorId) .detail("DataMoveID", rd.dataMoveId) .detail("DataMoveRange", rd.keys);
															#line 4872 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
			return a_body1cont14loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 2234 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!healthyDestinations.isHealthy())
															#line 4885 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2235 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 4889 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2236 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				signalledTransferComplete = true;
															#line 2237 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->dataTransferComplete.send(rd);
															#line 4895 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
															#line 2240 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		pollHealth = signalledTransferComplete ? Never() : delay(SERVER_KNOBS->HEALTH_POLL_TIME, TaskPriority::DataDistributionLaunch);
															#line 4900 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 2234 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!healthyDestinations.isHealthy())
															#line 4909 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2235 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 4913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2236 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				signalledTransferComplete = true;
															#line 2237 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->dataTransferComplete.send(rd);
															#line 4919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
															#line 2240 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		pollHealth = signalledTransferComplete ? Never() : delay(SERVER_KNOBS->HEALTH_POLL_TIME, TaskPriority::DataDistributionLaunch);
															#line 4924 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when3(Void const& _,int loopDepth) 
	{
															#line 2245 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->fetchKeysComplete.insert(rd);
															#line 2246 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!signalledTransferComplete)
															#line 4935 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2247 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			signalledTransferComplete = true;
															#line 2248 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataTransferComplete.send(rd);
															#line 4941 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont1loopBody1when3(Void && _,int loopDepth) 
	{
															#line 2245 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->fetchKeysComplete.insert(rd);
															#line 2246 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!signalledTransferComplete)
															#line 4953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2247 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			signalledTransferComplete = true;
															#line 2248 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->dataTransferComplete.send(rd);
															#line 4959 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont14loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 9, Void >::remove();
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 10, Void >::remove();
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont14loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont14loopBody1cont24cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont14loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont25(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont14loopHead1(0);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26(int loopDepth) 
	{
															#line 2270 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		healthyDestinations.addDataInFlightToTeam(-metrics.bytes);
															#line 2271 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto readLoad = metrics.readLoadKSecond();
															#line 2276 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& destinationRef = healthyDestinations;
															#line 2277 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->noErrorActors.add( trigger([destinationRef, readLoad]() mutable { destinationRef.addReadInFlightToTeam(-readLoad); }, delay(SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL)));
															#line 2282 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!error.code())
															#line 5175 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2283 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(relocateShardInterval.end(), distributorId) .detail("Duration", now() - startTime) .detail("Result", "Success");
															#line 2286 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("DataMoveStats", distributorId) .detail("Duration", now() - startTime) .detail("Bytes", metrics.bytes) .detail("Rate", static_cast<double>(metrics.bytes) / (now() - startTime)) .detail("Reason", rd.reason.toString()) .detail("DataMoveReason", static_cast<int>(rd.dmReason)) .detail("DataMoveID", rd.dataMoveId) .detail("DataMoveType", getDataMoveTypeFromDataMoveId(rd.dataMoveId));
															#line 2294 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (now() - startTime > 600)
															#line 5183 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2295 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevWarnAlways, "RelocateShardTooLong") .detail("Duration", now() - startTime) .detail("Dest", describe(destIds)) .detail("Src", describe(rd.src));
															#line 5187 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2300 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rd.keys.begin == keyServersPrefix)
															#line 5191 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2301 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent("MovedKeyServerKeys") .detail("Dest", describe(destIds)) .trackLatest(self->movedKeyServersEventHolder->trackingKey);
															#line 5195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2306 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!signalledTransferComplete)
															#line 5199 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2307 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				signalledTransferComplete = true;
															#line 2308 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				dataTransferComplete.send(rd);
															#line 5205 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2314 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			const int nonOverlappingCount = nonOverlappedServerCount(rd.completeSources, destIds);
															#line 2315 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->bytesWritten += metrics.bytes;
															#line 2316 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveBytesRate.addSample(metrics.bytes * nonOverlappingCount);
															#line 2317 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->shardsAffectedByTeamFailure->finishMove(rd.keys);
															#line 2318 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			relocationComplete.send(rd);
															#line 2320 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (enableShardMove)
															#line 5219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2322 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				std::vector<ShardsAffectedByTeamFailure::Team> selectedTeams;
															#line 2323 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for(int i = 0;i < bestTeams.size();i++) {
															#line 2324 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					auto serverIds = bestTeams[i].first->getServerIDs();
															#line 2325 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					selectedTeams.push_back(ShardsAffectedByTeamFailure::Team(serverIds, i == 0));
															#line 5229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
															#line 2329 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->physicalShardCollection->updatePhysicalShardCollection( rd.keys, rd.isRestore(), selectedTeams, rd.dataMoveId.first(), metrics, debugID);
															#line 5233 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2333 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 5237 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2334 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for( const auto& [team, _] : bestTeams ) {
															#line 2335 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					for( const UID& ssid : team->getServerIDs() ) {
															#line 2336 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->bulkLoadTaskCollection->busyMap.removeTask(ssid);
															#line 5245 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
				}
				try {
															#line 2340 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					self->bulkLoadTaskCollection->terminateTask(rd.bulkLoadTask.get().coreState);
															#line 2341 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					TraceEvent( bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorComplete", self->distributorId) .detail("Dests", describe(destIds)) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId());
															#line 5253 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					loopDepth = a_body1cont14loopBody1cont26cont12(loopDepth);
				}
				catch (Error& error) {
					loopDepth = a_body1cont14loopBody1cont26Catch1(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1cont14loopBody1cont26Catch1(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont14loopBody1cont26cont2(loopDepth);
			}
		}
		else
		{
															#line 2361 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (doBulkLoading)
															#line 5271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2362 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorError") .errorUnsuppressed(error) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId());
															#line 2366 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				for( const auto& [team, _] : bestTeams ) {
															#line 2367 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					for( const UID& ssid : team->getServerIDs() ) {
															#line 2368 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
						self->bulkLoadTaskCollection->busyMap.removeTask(ssid);
															#line 5281 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
					}
				}
			}
															#line 2372 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			return a_body1Catch2(error, std::max(0, loopDepth - 1));
															#line 5287 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont27(int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2266 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			error = e;
															#line 5303 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = a_body1cont14loopBody1cont27(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont28(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont28cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont28(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont28cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont28(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont28(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont14loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont14loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont14loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont14loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont14loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont14loopBody1cont28cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont14loopBody1cont27(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26cont2(int loopDepth) 
	{
															#line 2359 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<DataDistributionRelocatorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DataDistributionRelocatorActorState(); static_cast<DataDistributionRelocatorActor*>(this)->destroy(); return 0; }
															#line 5418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<DataDistributionRelocatorActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DataDistributionRelocatorActorState();
		static_cast<DataDistributionRelocatorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26cont8(int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont26cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26Catch1(const Error& bulkLoadError,int loopDepth=0) 
	{
		try {
															#line 2347 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT_WE_THINK(bulkLoadError.code() == error_code_bulkload_task_outdated);
															#line 2348 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (bulkLoadError.code() != error_code_bulkload_task_outdated)
															#line 5439 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2349 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				return a_body1Catch2(bulkLoadError, std::max(0, loopDepth - 1));
															#line 5443 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2351 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskRelocatorCompleteButOutdated", self->distributorId) .detail("Dests", describe(destIds)) .detail("JobID", rd.bulkLoadTask.get().coreState.getJobId()) .detail("TaskID", rd.bulkLoadTask.get().coreState.getTaskId());
															#line 5447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = a_body1cont14loopBody1cont26cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont26cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont14loopBody1cont26cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont14loopBody1cont29(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont25(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont29(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont25(loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont14loopBody1cont2when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14loopBody1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 13, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 13, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont14loopBody1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 13, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont14loopBody1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 13, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 13);

	}
	int a_body1Catch2cont1(int loopDepth) 
	{
															#line 2440 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 5562 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1Catch2cont5(Void const& _,int loopDepth) 
	{
															#line 2432 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevWarnAlways, "RelocateShardCancelDataMoveTeamNotFound") .detail("Src", describe(rd.src)) .detail("DataMoveMetaData", rd.dataMove != nullptr ? rd.dataMove->meta.toString() : "Empty");
															#line 5570 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont5(Void && _,int loopDepth) 
	{
															#line 2432 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevWarnAlways, "RelocateShardCancelDataMoveTeamNotFound") .detail("Src", describe(rd.src)) .detail("DataMoveMetaData", rd.dataMove != nullptr ? rd.dataMove->meta.toString() : "Empty");
															#line 5579 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state > 0) static_cast<DataDistributionRelocatorActor*>(this)->actor_wait_state = 0;
		static_cast<DataDistributionRelocatorActor*>(this)->ActorCallback< DataDistributionRelocatorActor, 14, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 14, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_fire(ActorCallback< DataDistributionRelocatorActor, 14, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_error(ActorCallback< DataDistributionRelocatorActor, 14, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), 14);

	}
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RelocateData rd;
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> prevCleanup;
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	const DDEnabledState* ddEnabledState;
															#line 1506 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Promise<Void> errorOut;
															#line 1507 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceInterval relocateShardInterval;
															#line 1508 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PromiseStream<RelocateData> dataTransferComplete;
															#line 1509 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PromiseStream<RelocateData> relocationComplete;
															#line 1510 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool signalledTransferComplete;
															#line 1511 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	UID distributorId;
															#line 1512 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	ParallelTCInfo healthyDestinations;
															#line 1514 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool anyHealthy;
															#line 1515 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool allHealthy;
															#line 1516 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool anyWithSource;
															#line 1517 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool anyDestOverloaded;
															#line 1518 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int destOverloadedCount;
															#line 1519 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int stuckCount;
															#line 1520 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<std::pair<Reference<IDataDistributionTeam>, bool>> bestTeams;
															#line 1521 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	double startTime;
															#line 1522 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<UID> destIds;
															#line 1523 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	WantTrueBest wantTrueBest;
															#line 1524 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	WantTrueBestIfMoveout wantTrueBestIfMoveout;
															#line 1525 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	uint64_t debugID;
															#line 1526 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool enableShardMove;
															#line 1530 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool doBulkLoading;
															#line 1565 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Transaction tr;
															#line 1646 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Optional<StorageMetrics> parentMetrics;
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	StorageMetrics metrics;
															#line 1659 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::unordered_set<uint64_t> excludedDstPhysicalShards;
															#line 1665 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	uint64_t physicalShardIDCandidate;
															#line 1666 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool forceToUseNewPhysicalShard;
															#line 1668 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int tciIndex;
															#line 1669 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool foundTeams;
															#line 1670 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool bestTeamReady;
															#line 2047 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<UID> healthyIds;
															#line 2048 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<UID> extraIds;
															#line 2049 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<ShardsAffectedByTeamFailure::Team> destinationTeams;
															#line 2097 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int totalIds;
															#line 2135 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Error error;
															#line 2136 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Promise<Void> dataMovementComplete;
															#line 2170 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> doMoveKeys;
															#line 2171 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> pollHealth;
															#line 2406 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Error err;
															#line 5745 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via dataDistributionRelocator()
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class DataDistributionRelocatorActor final : public Actor<Void>, public ActorCallback< DataDistributionRelocatorActor, 0, Void >, public ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >, public ActorCallback< DataDistributionRelocatorActor, 2, Void >, public ActorCallback< DataDistributionRelocatorActor, 3, Void >, public ActorCallback< DataDistributionRelocatorActor, 4, Void >, public ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >, public ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >, public ActorCallback< DataDistributionRelocatorActor, 7, Void >, public ActorCallback< DataDistributionRelocatorActor, 8, Void >, public ActorCallback< DataDistributionRelocatorActor, 9, Void >, public ActorCallback< DataDistributionRelocatorActor, 10, Void >, public ActorCallback< DataDistributionRelocatorActor, 11, Void >, public ActorCallback< DataDistributionRelocatorActor, 12, Void >, public ActorCallback< DataDistributionRelocatorActor, 13, Void >, public ActorCallback< DataDistributionRelocatorActor, 14, Void >, public FastAllocated<DataDistributionRelocatorActor>, public DataDistributionRelocatorActorState<DataDistributionRelocatorActor> {
															#line 5750 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<DataDistributionRelocatorActor>::operator new;
	using FastAllocated<DataDistributionRelocatorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16221220449786696960UL, 13667875461431749120UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DataDistributionRelocatorActor, 0, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >;
friend struct ActorCallback< DataDistributionRelocatorActor, 2, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 3, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 4, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >;
friend struct ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >;
friend struct ActorCallback< DataDistributionRelocatorActor, 7, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 8, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 9, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 10, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 11, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 12, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 13, Void >;
friend struct ActorCallback< DataDistributionRelocatorActor, 14, Void >;
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataDistributionRelocatorActor(DDQueue* const& self,RelocateData const& rd,Future<Void> const& prevCleanup,const DDEnabledState* const& ddEnabledState) 
															#line 5781 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DataDistributionRelocatorActorState<DataDistributionRelocatorActor>(self, rd, prevCleanup, ddEnabledState),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2631274672844690944UL, 4050065294653027072UL);
		ActorExecutionContextHelper __helper(static_cast<DataDistributionRelocatorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("dataDistributionRelocator");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("dataDistributionRelocator", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 1, BulkLoadTaskState >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 5, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 6, std::pair<Optional<Reference<IDataDistributionTeam>>, bool> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 12, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 13, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< DataDistributionRelocatorActor, 14, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<Void> dataDistributionRelocator( DDQueue* const& self, RelocateData const& rd, Future<Void> const& prevCleanup, const DDEnabledState* const& ddEnabledState ) {
															#line 1502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new DataDistributionRelocatorActor(self, rd, prevCleanup, ddEnabledState));
															#line 5826 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 2443 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

inline double getWorstCpu(const HealthMetrics& metrics, const std::vector<UID>& ids) {
	double cpu = 0;
	for (auto& id : ids) {
		if (metrics.storageStats.contains(id)) {
			cpu = std::max(cpu, metrics.storageStats.at(id).cpuUsage);
		} else {
			// assume the server is too busy to report its stats
			cpu = std::max(cpu, 100.0);
			break;
		}
	}
	return cpu;
}

// Move the shard with the top K highest read density of sourceTeam's to destTeam if sourceTeam has much more read
// load than destTeam
															#line 5847 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via rebalanceReadLoad()
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class RebalanceReadLoadActor>
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RebalanceReadLoadActorState {
															#line 5854 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RebalanceReadLoadActorState(DDQueue* const& self,DataMovementReason const& moveReason,Reference<IDataDistributionTeam> const& sourceTeam,Reference<IDataDistributionTeam> const& destTeam,bool const& primary,TraceEvent* const& traceEvent) 
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   moveReason(moveReason),
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   sourceTeam(sourceTeam),
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destTeam(destTeam),
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   primary(primary),
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   traceEvent(traceEvent)
															#line 5871 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("rebalanceReadLoad", reinterpret_cast<unsigned long>(this));

	}
	~RebalanceReadLoadActorState() 
	{
		fdb_probe_actor_destroy("rebalanceReadLoad", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2466 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (g_network->isSimulated() && g_simulator->speedUpSimulation)
															#line 5886 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2467 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("CancelingDueToSimulationSpeedup", true);
															#line 2468 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 5892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2471 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2473 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("ShardsInSource", shards.size());
															#line 2477 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (shards.size() <= 1)
															#line 5904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2478 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("SkipReason", "NoShardOnSource");
															#line 2479 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 5910 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2486 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (self->timeThrottle(sourceTeam->getServerIDs()))
															#line 5918 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2487 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("SkipReason", "SourceTeamThrottle");
															#line 2488 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 5924 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2491 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			auto srcLoad = sourceTeam->getReadLoad(false), destLoad = destTeam->getReadLoad();
															#line 2492 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SrcReadBandwidth", srcLoad).detail("DestReadBandwidth", destLoad);
															#line 2495 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if ((1.0 - SERVER_KNOBS->READ_REBALANCE_DIFF_FRAC) * srcLoad <= destLoad)
															#line 5936 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2496 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("SkipReason", "TeamTooSimilar");
															#line 2497 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 5942 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2500 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			int topK = std::max(1, std::min(int(0.1 * shards.size()), SERVER_KNOBS->READ_REBALANCE_SHARD_TOPK));
															#line 2501 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			healthMetrics = self->txnProcessor->getHealthMetrics(true);
															#line 2502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			req = GetTopKMetricsRequest(shards, topK, (srcLoad - destLoad) * SERVER_KNOBS->READ_REBALANCE_MAX_SHARD_FRAC, std::min(srcLoad / shards.size(), SERVER_KNOBS->READ_REBALANCE_MIN_READ_BYTES_KS));
															#line 2506 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<GetTopKMetricsReply> __when_expr_0 = brokenPromiseToNever(self->getTopKMetrics.getReply(req));
															#line 2506 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5958 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state = 1;
															#line 2506 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*>(static_cast<RebalanceReadLoadActor*>(this)));
															#line 5963 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RebalanceReadLoadActorState();
		static_cast<RebalanceReadLoadActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2507 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_1 = ready(healthMetrics);
															#line 2507 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5988 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state = 2;
															#line 2507 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RebalanceReadLoadActor, 1, Void >*>(static_cast<RebalanceReadLoadActor*>(this)));
															#line 5993 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(GetTopKMetricsReply const& __reply,int loopDepth) 
	{
															#line 2506 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		reply = __reply;
															#line 6002 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetTopKMetricsReply && __reply,int loopDepth) 
	{
		reply = std::move(__reply);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state > 0) static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state = 0;
		static_cast<RebalanceReadLoadActor*>(this)->ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >::remove();

	}
	void a_callback_fire(ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*,GetTopKMetricsReply const& value) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*,GetTopKMetricsReply && value) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*,Error err) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 2508 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto cpu = getWorstCpu(healthMetrics.get(), sourceTeam->getServerIDs());
															#line 2509 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (cpu < SERVER_KNOBS->READ_REBALANCE_CPU_THRESHOLD)
															#line 6083 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2510 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "LowReadLoad").detail("WorstSrcCpu", cpu);
															#line 2511 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6089 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceReadLoadActorState();
			static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2514 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& metricsList = reply.shardMetrics;
															#line 2516 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		deterministicRandom()->randomShuffle(metricsList);
															#line 2517 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("MinReadLoad", reply.minReadLoad).detail("MaxReadLoad", reply.maxReadLoad);
															#line 2519 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (metricsList.empty())
															#line 6103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2520 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "NoEligibleShards");
															#line 2521 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6109 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceReadLoadActorState();
			static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& [shard, metrics] = metricsList[0];
															#line 2525 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("ShardReadBandwidth", metrics.bytesReadPerKSecond) .detail("ShardReadOps", metrics.opsReadPerKSecond);
															#line 2529 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2531 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < shards.size();i++) {
															#line 2532 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (shard == shards[i])
															#line 6125 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2533 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				UID traceId = deterministicRandom()->randomUniqueID();
															#line 2534 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->output.send(RelocateShard(shard, moveReason, RelocateReason::REBALANCE_READ, traceId));
															#line 2535 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("TraceId", traceId);
															#line 2537 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto serverIds = sourceTeam->getServerIDs();
															#line 2538 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->updateLastAsSource(serverIds);
															#line 2540 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->serverCounter.increaseForTeam( serverIds, RelocateReason::REBALANCE_READ, DDQueue::ServerCounter::ProposedSource);
															#line 2542 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(true); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6141 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(true);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2545 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SkipReason", "ShardNotPresent");
															#line 2546 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6152 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
		this->~RebalanceReadLoadActorState();
		static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 2508 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto cpu = getWorstCpu(healthMetrics.get(), sourceTeam->getServerIDs());
															#line 2509 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (cpu < SERVER_KNOBS->READ_REBALANCE_CPU_THRESHOLD)
															#line 6166 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2510 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "LowReadLoad").detail("WorstSrcCpu", cpu);
															#line 2511 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6172 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceReadLoadActorState();
			static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2514 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& metricsList = reply.shardMetrics;
															#line 2516 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		deterministicRandom()->randomShuffle(metricsList);
															#line 2517 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("MinReadLoad", reply.minReadLoad).detail("MaxReadLoad", reply.maxReadLoad);
															#line 2519 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (metricsList.empty())
															#line 6186 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2520 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "NoEligibleShards");
															#line 2521 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6192 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceReadLoadActorState();
			static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2524 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto& [shard, metrics] = metricsList[0];
															#line 2525 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("ShardReadBandwidth", metrics.bytesReadPerKSecond) .detail("ShardReadOps", metrics.opsReadPerKSecond);
															#line 2529 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2531 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < shards.size();i++) {
															#line 2532 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (shard == shards[i])
															#line 6208 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2533 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				UID traceId = deterministicRandom()->randomUniqueID();
															#line 2534 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->output.send(RelocateShard(shard, moveReason, RelocateReason::REBALANCE_READ, traceId));
															#line 2535 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("TraceId", traceId);
															#line 2537 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				auto serverIds = sourceTeam->getServerIDs();
															#line 2538 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->updateLastAsSource(serverIds);
															#line 2540 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->serverCounter.increaseForTeam( serverIds, RelocateReason::REBALANCE_READ, DDQueue::ServerCounter::ProposedSource);
															#line 2542 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(true); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(true);
				this->~RebalanceReadLoadActorState();
				static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2545 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SkipReason", "ShardNotPresent");
															#line 2546 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<RebalanceReadLoadActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceReadLoadActorState(); static_cast<RebalanceReadLoadActor*>(this)->destroy(); return 0; }
															#line 6235 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<RebalanceReadLoadActor*>(this)->SAV< bool >::value()) bool(false);
		this->~RebalanceReadLoadActorState();
		static_cast<RebalanceReadLoadActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state > 0) static_cast<RebalanceReadLoadActor*>(this)->actor_wait_state = 0;
		static_cast<RebalanceReadLoadActor*>(this)->ActorCallback< RebalanceReadLoadActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RebalanceReadLoadActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RebalanceReadLoadActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RebalanceReadLoadActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataMovementReason moveReason;
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam> sourceTeam;
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam> destTeam;
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool primary;
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceEvent* traceEvent;
															#line 2471 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<KeyRange> shards;
															#line 2501 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<HealthMetrics> healthMetrics;
															#line 2502 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetTopKMetricsRequest req;
															#line 2506 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetTopKMetricsReply reply;
															#line 6338 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via rebalanceReadLoad()
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RebalanceReadLoadActor final : public Actor<bool>, public ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >, public ActorCallback< RebalanceReadLoadActor, 1, Void >, public FastAllocated<RebalanceReadLoadActor>, public RebalanceReadLoadActorState<RebalanceReadLoadActor> {
															#line 6343 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<RebalanceReadLoadActor>::operator new;
	using FastAllocated<RebalanceReadLoadActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15598829513541780736UL, 4825804428113594112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >;
friend struct ActorCallback< RebalanceReadLoadActor, 1, Void >;
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RebalanceReadLoadActor(DDQueue* const& self,DataMovementReason const& moveReason,Reference<IDataDistributionTeam> const& sourceTeam,Reference<IDataDistributionTeam> const& destTeam,bool const& primary,TraceEvent* const& traceEvent) 
															#line 6361 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<bool>(),
		   RebalanceReadLoadActorState<RebalanceReadLoadActor>(self, moveReason, sourceTeam, destTeam, primary, traceEvent),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(257547752678163200UL, 2881630330149849344UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceReadLoadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("rebalanceReadLoad");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("rebalanceReadLoad", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RebalanceReadLoadActor, 0, GetTopKMetricsReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RebalanceReadLoadActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<bool> rebalanceReadLoad( DDQueue* const& self, DataMovementReason const& moveReason, Reference<IDataDistributionTeam> const& sourceTeam, Reference<IDataDistributionTeam> const& destTeam, bool const& primary, TraceEvent* const& traceEvent ) {
															#line 2460 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<bool>(new RebalanceReadLoadActor(self, moveReason, sourceTeam, destTeam, primary, traceEvent));
															#line 6395 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 2548 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

// Move a random shard from sourceTeam if sourceTeam has much more data than provided destTeam
															#line 6401 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via rebalanceTeams()
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class RebalanceTeamsActor>
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RebalanceTeamsActorState {
															#line 6408 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RebalanceTeamsActorState(DDQueue* const& self,DataMovementReason const& moveReason,Reference<IDataDistributionTeam const> const& sourceTeam,Reference<IDataDistributionTeam const> const& destTeam,bool const& primary,TraceEvent* const& traceEvent) 
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   moveReason(moveReason),
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   sourceTeam(sourceTeam),
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destTeam(destTeam),
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   primary(primary),
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   traceEvent(traceEvent)
															#line 6425 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("rebalanceTeams", reinterpret_cast<unsigned long>(this));

	}
	~RebalanceTeamsActorState() 
	{
		fdb_probe_actor_destroy("rebalanceTeams", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2556 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (g_network->isSimulated() && g_simulator->speedUpSimulation)
															#line 6440 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2557 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("CancelingDueToSimulationSpeedup", true);
															#line 2558 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6446 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(false);
				this->~RebalanceTeamsActorState();
				static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2561 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			Promise<int64_t> req;
															#line 2562 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->getAverageShardBytes.send(req);
															#line 2564 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<int64_t> __when_expr_0 = req.getFuture();
															#line 2564 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<RebalanceTeamsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RebalanceTeamsActor*>(this)->actor_wait_state = 1;
															#line 2564 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RebalanceTeamsActor, 0, int64_t >*>(static_cast<RebalanceTeamsActor*>(this)));
															#line 6465 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RebalanceTeamsActorState();
		static_cast<RebalanceTeamsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2568 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("AverageShardBytes", averageShardBytes).detail("ShardsInSource", shards.size());
															#line 2570 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!shards.size())
															#line 6492 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2571 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", "NoShardOnSource");
															#line 2572 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6498 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceTeamsActorState();
			static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2575 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		moveShard = KeyRange();
															#line 2576 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		metrics = StorageMetrics();
															#line 2577 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		retries = 0;
															#line 2578 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		;
															#line 6512 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(int64_t const& __averageShardBytes,int loopDepth) 
	{
															#line 2564 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		averageShardBytes = __averageShardBytes;
															#line 6521 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(int64_t && __averageShardBytes,int loopDepth) 
	{
		averageShardBytes = std::move(__averageShardBytes);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RebalanceTeamsActor*>(this)->actor_wait_state > 0) static_cast<RebalanceTeamsActor*>(this)->actor_wait_state = 0;
		static_cast<RebalanceTeamsActor*>(this)->ActorCallback< RebalanceTeamsActor, 0, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< RebalanceTeamsActor, 0, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RebalanceTeamsActor, 0, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RebalanceTeamsActor, 0, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2592 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		int64_t sourceBytes = sourceTeam->getLoadBytes(false);
															#line 2593 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		int64_t destBytes = destTeam->getLoadBytes();
															#line 2595 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bool sourceAndDestTooSimilar = sourceBytes - destBytes <= 3 * std::max<int64_t>(SERVER_KNOBS->MIN_SHARD_BYTES, metrics.bytes);
															#line 2597 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SourceBytes", sourceBytes) .detail("DestBytes", destBytes) .detail("ShardBytes", metrics.bytes) .detail("SourceAndDestTooSimilar", sourceAndDestTooSimilar);
															#line 2602 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (sourceAndDestTooSimilar || metrics.bytes == 0)
															#line 6608 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2603 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			traceEvent->detail("SkipReason", sourceAndDestTooSimilar ? "TeamTooSimilar" : "ShardZeroSize");
															#line 2604 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6614 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(false);
			this->~RebalanceTeamsActorState();
			static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2608 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		shards = self->shardsAffectedByTeamFailure->getShardsFor( ShardsAffectedByTeamFailure::Team(sourceTeam->getServerIDs(), primary));
															#line 2610 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < shards.size();i++) {
															#line 2611 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (moveShard == shards[i])
															#line 6626 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2612 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				UID traceId = deterministicRandom()->randomUniqueID();
															#line 2613 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->output.send(RelocateShard(moveShard, moveReason, RelocateReason::REBALANCE_DISK, traceId));
															#line 2614 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				traceEvent->detail("TraceId", traceId);
															#line 2616 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->serverCounter.increaseForTeam( sourceTeam->getServerIDs(), RelocateReason::REBALANCE_DISK, DDQueue::ServerCounter::ProposedSource);
															#line 2618 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(true); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6638 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(true);
				this->~RebalanceTeamsActorState();
				static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2622 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SkipReason", "ShardNotPresent");
															#line 2623 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<RebalanceTeamsActor*>(this)->SAV<bool>::futures) { (void)(false); this->~RebalanceTeamsActorState(); static_cast<RebalanceTeamsActor*>(this)->destroy(); return 0; }
															#line 6649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<RebalanceTeamsActor*>(this)->SAV< bool >::value()) bool(false);
		this->~RebalanceTeamsActorState();
		static_cast<RebalanceTeamsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 2578 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!(retries < SERVER_KNOBS->REBALANCE_MAX_RETRIES))
															#line 6668 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2579 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		testShard = deterministicRandom()->randomChoice(shards);
															#line 2580 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<StorageMetrics> __when_expr_1 = brokenPromiseToNever(self->getShardMetrics.getReply(GetMetricsRequest(testShard)));
															#line 2580 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<RebalanceTeamsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RebalanceTeamsActor*>(this)->actor_wait_state = 2;
															#line 2580 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*>(static_cast<RebalanceTeamsActor*>(this)));
															#line 6683 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(StorageMetrics const& testMetrics,int loopDepth) 
	{
															#line 2582 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (testMetrics.bytes > metrics.bytes)
															#line 6705 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2583 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			moveShard = testShard;
															#line 2584 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			metrics = testMetrics;
															#line 2585 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (metrics.bytes > averageShardBytes)
															#line 6713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
				return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
															#line 2589 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		retries++;
															#line 6720 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(StorageMetrics && testMetrics,int loopDepth) 
	{
															#line 2582 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (testMetrics.bytes > metrics.bytes)
															#line 6729 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2583 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			moveShard = testShard;
															#line 2584 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			metrics = testMetrics;
															#line 2585 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (metrics.bytes > averageShardBytes)
															#line 6737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
				return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
															#line 2589 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		retries++;
															#line 6744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(StorageMetrics const& testMetrics,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(testMetrics, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(StorageMetrics && testMetrics,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(testMetrics), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RebalanceTeamsActor*>(this)->actor_wait_state > 0) static_cast<RebalanceTeamsActor*>(this)->actor_wait_state = 0;
		static_cast<RebalanceTeamsActor*>(this)->ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >::remove();

	}
	void a_callback_fire(ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*,StorageMetrics const& value) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*,StorageMetrics && value) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*,Error err) 
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataMovementReason moveReason;
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam const> sourceTeam;
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam const> destTeam;
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool primary;
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceEvent* traceEvent;
															#line 2564 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int64_t averageShardBytes;
															#line 2565 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<KeyRange> shards;
															#line 2575 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	KeyRange moveShard;
															#line 2576 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	StorageMetrics metrics;
															#line 2577 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int retries;
															#line 2579 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	KeyRange testShard;
															#line 6848 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via rebalanceTeams()
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RebalanceTeamsActor final : public Actor<bool>, public ActorCallback< RebalanceTeamsActor, 0, int64_t >, public ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >, public FastAllocated<RebalanceTeamsActor>, public RebalanceTeamsActorState<RebalanceTeamsActor> {
															#line 6853 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<RebalanceTeamsActor>::operator new;
	using FastAllocated<RebalanceTeamsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4463030793906839808UL, 8991344503366864896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RebalanceTeamsActor, 0, int64_t >;
friend struct ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >;
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RebalanceTeamsActor(DDQueue* const& self,DataMovementReason const& moveReason,Reference<IDataDistributionTeam const> const& sourceTeam,Reference<IDataDistributionTeam const> const& destTeam,bool const& primary,TraceEvent* const& traceEvent) 
															#line 6871 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<bool>(),
		   RebalanceTeamsActorState<RebalanceTeamsActor>(self, moveReason, sourceTeam, destTeam, primary, traceEvent),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("rebalanceTeams", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(846430350490556672UL, 3554085506117065472UL);
		ActorExecutionContextHelper __helper(static_cast<RebalanceTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("rebalanceTeams");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("rebalanceTeams", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RebalanceTeamsActor, 0, int64_t >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RebalanceTeamsActor, 1, StorageMetrics >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] static Future<bool> rebalanceTeams( DDQueue* const& self, DataMovementReason const& moveReason, Reference<IDataDistributionTeam const> const& sourceTeam, Reference<IDataDistributionTeam const> const& destTeam, bool const& primary, TraceEvent* const& traceEvent ) {
															#line 2550 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<bool>(new RebalanceTeamsActor(self, moveReason, sourceTeam, destTeam, primary, traceEvent));
															#line 6905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 2625 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

															#line 6910 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getSrcDestTeams()
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class GetSrcDestTeamsActor>
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSrcDestTeamsActorState {
															#line 6917 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSrcDestTeamsActorState(DDQueue* const& self,int const& teamCollectionIndex,GetTeamRequest const& srcReq,GetTeamRequest const& destReq,int const& priority,TraceEvent* const& traceEvent) 
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   teamCollectionIndex(teamCollectionIndex),
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   srcReq(srcReq),
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   destReq(destReq),
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   priority(priority),
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   traceEvent(traceEvent)
															#line 6934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getSrcDestTeams", reinterpret_cast<unsigned long>(this));

	}
	~GetSrcDestTeamsActorState() 
	{
		fdb_probe_actor_destroy("getSrcDestTeams", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2633 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<std::pair<Optional<ITeamRef>, bool>> __when_expr_0 = brokenPromiseToNever(self->teamCollections[teamCollectionIndex].getTeam.getReply(destReq));
															#line 2633 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6951 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state = 1;
															#line 2633 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*>(static_cast<GetSrcDestTeamsActor*>(this)));
															#line 6956 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetSrcDestTeamsActorState();
		static_cast<GetSrcDestTeamsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2635 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("DestTeam", printable(randomTeam.first.mapRef(&IDataDistributionTeam::getDesc)));
															#line 2637 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (randomTeam.first.present())
															#line 6981 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2638 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<std::pair<Optional<ITeamRef>, bool>> __when_expr_1 = brokenPromiseToNever(self->teamCollections[teamCollectionIndex].getTeam.getReply(srcReq));
															#line 2638 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6987 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state = 2;
															#line 2638 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*>(static_cast<GetSrcDestTeamsActor*>(this)));
															#line 6992 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(std::pair<Optional<ITeamRef>, bool> const& __randomTeam,int loopDepth) 
	{
															#line 2633 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		randomTeam = __randomTeam;
															#line 7006 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<Optional<ITeamRef>, bool> && __randomTeam,int loopDepth) 
	{
		randomTeam = std::move(__randomTeam);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state > 0) static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state = 0;
		static_cast<GetSrcDestTeamsActor*>(this)->ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*,std::pair<Optional<ITeamRef>, bool> const& value) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*,std::pair<Optional<ITeamRef>, bool> && value) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2647 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSrcDestTeamsActor*>(this)->SAV<SrcDestTeamPair>::futures) { (void)({}); this->~GetSrcDestTeamsActorState(); static_cast<GetSrcDestTeamsActor*>(this)->destroy(); return 0; }
															#line 7085 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<GetSrcDestTeamsActor*>(this)->SAV< SrcDestTeamPair >::value()) SrcDestTeamPair({});
		this->~GetSrcDestTeamsActorState();
		static_cast<GetSrcDestTeamsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2641 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent->detail("SourceTeam", printable(loadedTeam.first.mapRef(&IDataDistributionTeam::getDesc)));
															#line 2643 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (loadedTeam.first.present())
															#line 7099 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2644 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<GetSrcDestTeamsActor*>(this)->SAV<SrcDestTeamPair>::futures) { (void)(std::make_pair(loadedTeam.first.get(), randomTeam.first.get())); this->~GetSrcDestTeamsActorState(); static_cast<GetSrcDestTeamsActor*>(this)->destroy(); return 0; }
															#line 7103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<GetSrcDestTeamsActor*>(this)->SAV< SrcDestTeamPair >::value()) SrcDestTeamPair(std::make_pair(loadedTeam.first.get(), randomTeam.first.get()));
			this->~GetSrcDestTeamsActorState();
			static_cast<GetSrcDestTeamsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::pair<Optional<ITeamRef>, bool> const& __loadedTeam,int loopDepth) 
	{
															#line 2638 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		loadedTeam = __loadedTeam;
															#line 7117 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::pair<Optional<ITeamRef>, bool> && __loadedTeam,int loopDepth) 
	{
		loadedTeam = std::move(__loadedTeam);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state > 0) static_cast<GetSrcDestTeamsActor*>(this)->actor_wait_state = 0;
		static_cast<GetSrcDestTeamsActor*>(this)->ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >::remove();

	}
	void a_callback_fire(ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*,std::pair<Optional<ITeamRef>, bool> const& value) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*,std::pair<Optional<ITeamRef>, bool> && value) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*,Error err) 
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int teamCollectionIndex;
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetTeamRequest srcReq;
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetTeamRequest destReq;
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int priority;
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceEvent* traceEvent;
															#line 2633 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::pair<Optional<ITeamRef>, bool> randomTeam;
															#line 2638 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::pair<Optional<ITeamRef>, bool> loadedTeam;
															#line 7208 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getSrcDestTeams()
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSrcDestTeamsActor final : public Actor<SrcDestTeamPair>, public ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >, public ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >, public FastAllocated<GetSrcDestTeamsActor>, public GetSrcDestTeamsActorState<GetSrcDestTeamsActor> {
															#line 7213 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetSrcDestTeamsActor>::operator new;
	using FastAllocated<GetSrcDestTeamsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6852716116236748288UL, 1655208759141107456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<SrcDestTeamPair>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >;
friend struct ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >;
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSrcDestTeamsActor(DDQueue* const& self,int const& teamCollectionIndex,GetTeamRequest const& srcReq,GetTeamRequest const& destReq,int const& priority,TraceEvent* const& traceEvent) 
															#line 7231 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<SrcDestTeamPair>(),
		   GetSrcDestTeamsActorState<GetSrcDestTeamsActor>(self, teamCollectionIndex, srcReq, destReq, priority, traceEvent),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getSrcDestTeams", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17622014136999606784UL, 6190347938397110016UL);
		ActorExecutionContextHelper __helper(static_cast<GetSrcDestTeamsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getSrcDestTeams");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getSrcDestTeams", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetSrcDestTeamsActor, 0, std::pair<Optional<ITeamRef>, bool> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetSrcDestTeamsActor, 1, std::pair<Optional<ITeamRef>, bool> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<SrcDestTeamPair> getSrcDestTeams( DDQueue* const& self, int const& teamCollectionIndex, GetTeamRequest const& srcReq, GetTeamRequest const& destReq, int const& priority, TraceEvent* const& traceEvent ) {
															#line 2626 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<SrcDestTeamPair>(new GetSrcDestTeamsActor(self, teamCollectionIndex, srcReq, destReq, priority, traceEvent));
															#line 7265 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 2649 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

Future<SrcDestTeamPair> DDQueue::getSrcDestTeams(const int& teamCollectionIndex,
                                                 const GetTeamRequest& srcReq,
                                                 const GetTeamRequest& destReq,
                                                 const int& priority,
                                                 TraceEvent* traceEvent) {
	return ::getSrcDestTeams(this, teamCollectionIndex, srcReq, destReq, priority, traceEvent);
}
Future<bool> DDQueue::rebalanceReadLoad(DataMovementReason moveReason,
                                        Reference<IDataDistributionTeam> sourceTeam,
                                        Reference<IDataDistributionTeam> destTeam,
                                        bool primary,
                                        TraceEvent* traceEvent) {
	return ::rebalanceReadLoad(this, moveReason, sourceTeam, destTeam, primary, traceEvent);
}

Future<bool> DDQueue::rebalanceTeams(DataMovementReason moveReason,
                                     Reference<const IDataDistributionTeam> sourceTeam,
                                     Reference<const IDataDistributionTeam> destTeam,
                                     bool primary,
                                     TraceEvent* traceEvent) {
	return ::rebalanceTeams(this, moveReason, sourceTeam, destTeam, primary, traceEvent);
}

															#line 7293 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getSkipRebalanceValue()
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class GetSkipRebalanceValueActor>
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSkipRebalanceValueActorState {
															#line 7300 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSkipRebalanceValueActorState(Reference<IDDTxnProcessor> const& txnProcessor,bool const& readRebalance) 
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : txnProcessor(txnProcessor),
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   readRebalance(readRebalance)
															#line 7309 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this));

	}
	~GetSkipRebalanceValueActorState() 
	{
		fdb_probe_actor_destroy("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2674 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = txnProcessor->readRebalanceDDIgnoreKey();
															#line 2674 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (static_cast<GetSkipRebalanceValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7326 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetSkipRebalanceValueActor*>(this)->actor_wait_state = 1;
															#line 2674 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*>(static_cast<GetSkipRebalanceValueActor*>(this)));
															#line 7331 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetSkipRebalanceValueActorState();
		static_cast<GetSkipRebalanceValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> const& val,int loopDepth) 
	{
															#line 2676 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!val.present())
															#line 7354 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2677 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<GetSkipRebalanceValueActor*>(this)->SAV<bool>::futures) { (void)(false); this->~GetSkipRebalanceValueActorState(); static_cast<GetSkipRebalanceValueActor*>(this)->destroy(); return 0; }
															#line 7358 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<GetSkipRebalanceValueActor*>(this)->SAV< bool >::value()) bool(false);
			this->~GetSkipRebalanceValueActorState();
			static_cast<GetSkipRebalanceValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2679 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bool skipCurrentLoop = false;
															#line 2681 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (val.get().size() > 0 && val.get() != "on"_sr)
															#line 7368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2682 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			int ddIgnore = BinaryReader::fromStringRef<uint8_t>(val.get(), Unversioned());
															#line 2683 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (readRebalance)
															#line 7374 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2684 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				skipCurrentLoop = (ddIgnore & DDIgnore::REBALANCE_READ) > 0;
															#line 7378 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2686 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				skipCurrentLoop = (ddIgnore & DDIgnore::REBALANCE_DISK) > 0;
															#line 7384 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 2689 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			skipCurrentLoop = true;
															#line 7391 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2692 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSkipRebalanceValueActor*>(this)->SAV<bool>::futures) { (void)(skipCurrentLoop); this->~GetSkipRebalanceValueActorState(); static_cast<GetSkipRebalanceValueActor*>(this)->destroy(); return 0; }
															#line 7395 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<GetSkipRebalanceValueActor*>(this)->SAV< bool >::value()) bool(skipCurrentLoop);
		this->~GetSkipRebalanceValueActorState();
		static_cast<GetSkipRebalanceValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> && val,int loopDepth) 
	{
															#line 2676 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!val.present())
															#line 7407 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2677 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!static_cast<GetSkipRebalanceValueActor*>(this)->SAV<bool>::futures) { (void)(false); this->~GetSkipRebalanceValueActorState(); static_cast<GetSkipRebalanceValueActor*>(this)->destroy(); return 0; }
															#line 7411 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			new (&static_cast<GetSkipRebalanceValueActor*>(this)->SAV< bool >::value()) bool(false);
			this->~GetSkipRebalanceValueActorState();
			static_cast<GetSkipRebalanceValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2679 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		bool skipCurrentLoop = false;
															#line 2681 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (val.get().size() > 0 && val.get() != "on"_sr)
															#line 7421 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2682 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			int ddIgnore = BinaryReader::fromStringRef<uint8_t>(val.get(), Unversioned());
															#line 2683 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (readRebalance)
															#line 7427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2684 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				skipCurrentLoop = (ddIgnore & DDIgnore::REBALANCE_READ) > 0;
															#line 7431 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2686 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				skipCurrentLoop = (ddIgnore & DDIgnore::REBALANCE_DISK) > 0;
															#line 7437 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 2689 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			skipCurrentLoop = true;
															#line 7444 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2692 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<GetSkipRebalanceValueActor*>(this)->SAV<bool>::futures) { (void)(skipCurrentLoop); this->~GetSkipRebalanceValueActorState(); static_cast<GetSkipRebalanceValueActor*>(this)->destroy(); return 0; }
															#line 7448 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<GetSkipRebalanceValueActor*>(this)->SAV< bool >::value()) bool(skipCurrentLoop);
		this->~GetSkipRebalanceValueActorState();
		static_cast<GetSkipRebalanceValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont1(val, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetSkipRebalanceValueActor*>(this)->actor_wait_state > 0) static_cast<GetSkipRebalanceValueActor*>(this)->actor_wait_state = 0;
		static_cast<GetSkipRebalanceValueActor*>(this)->ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSkipRebalanceValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetSkipRebalanceValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetSkipRebalanceValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDDTxnProcessor> txnProcessor;
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool readRebalance;
															#line 7535 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getSkipRebalanceValue()
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class GetSkipRebalanceValueActor final : public Actor<bool>, public ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >, public FastAllocated<GetSkipRebalanceValueActor>, public GetSkipRebalanceValueActorState<GetSkipRebalanceValueActor> {
															#line 7540 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetSkipRebalanceValueActor>::operator new;
	using FastAllocated<GetSkipRebalanceValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13248959606247227904UL, 16959163992145479168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >;
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	GetSkipRebalanceValueActor(Reference<IDDTxnProcessor> const& txnProcessor,bool const& readRebalance) 
															#line 7557 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<bool>(),
		   GetSkipRebalanceValueActorState<GetSkipRebalanceValueActor>(txnProcessor, readRebalance),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9255883138487682304UL, 8872371140409258752UL);
		ActorExecutionContextHelper __helper(static_cast<GetSkipRebalanceValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getSkipRebalanceValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getSkipRebalanceValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetSkipRebalanceValueActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<bool> getSkipRebalanceValue( Reference<IDDTxnProcessor> const& txnProcessor, bool const& readRebalance ) {
															#line 2673 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<bool>(new GetSkipRebalanceValueActor(txnProcessor, readRebalance));
															#line 7590 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 2694 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

															#line 7595 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via BgDDLoadRebalance()
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class BgDDLoadRebalanceActor>
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class BgDDLoadRebalanceActorState {
															#line 7602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	BgDDLoadRebalanceActorState(DDQueue* const& self,int const& teamCollectionIndex,DataMovementReason const& reason) 
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   teamCollectionIndex(teamCollectionIndex),
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   reason(reason),
															#line 2696 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   resetCount(0),
															#line 2697 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   lastRead(0),
															#line 2698 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   skipCurrentLoop(false),
															#line 2699 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   readRebalance(isDataMovementForReadBalancing(reason)),
															#line 2700 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   moveType(isDataMovementForMountainChopper(reason) ? "BgDDMountainChopper" : "BgDDValleyFiller"),
															#line 2702 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   ddPriority(dataMovementPriority(reason)),
															#line 2703 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   mcMove(isDataMovementForMountainChopper(reason)),
															#line 2704 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   preferLowerReadTeam(readRebalance || SERVER_KNOBS->DD_PREFER_LOW_READ_UTIL_TEAM ? PreferLowerReadUtil::True : PreferLowerReadUtil::False),
															#line 2707 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   rebalancePollingInterval(0)
															#line 7631 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this));

	}
	~BgDDLoadRebalanceActorState() 
	{
		fdb_probe_actor_destroy("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2709 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 7646 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BgDDLoadRebalanceActorState();
		static_cast<BgDDLoadRebalanceActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2710 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		moved = false;
															#line 2711 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		sourceTeam = Reference<IDataDistributionTeam>();
															#line 2712 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destTeam = Reference<IDataDistributionTeam>();
															#line 2715 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(rebalancePollingInterval, TaskPriority::DataDistributionLaunch);
															#line 2715 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7684 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 1;
															#line 2715 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 0, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 7689 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 2717 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if ((now() - lastRead) > SERVER_KNOBS->BG_REBALANCE_SWITCH_CHECK_INTERVAL)
															#line 7699 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2718 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_1 = store(skipCurrentLoop, getSkipRebalanceValue(self->txnProcessor, readRebalance));
															#line 2718 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7705 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 2;
															#line 2718 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 1, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 7710 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont3(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 2717 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if ((now() - lastRead) > SERVER_KNOBS->BG_REBALANCE_SWITCH_CHECK_INTERVAL)
															#line 7731 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2718 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_1 = store(skipCurrentLoop, getSkipRebalanceValue(self->txnProcessor, readRebalance));
															#line 2718 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 2;
															#line 2718 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 1, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 7742 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont3(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2799 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("RebalanceMoveError", self->distributorId).detail("MoveType", moveType).errorUnsuppressed(e);
															#line 2800 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 7846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 2722 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (skipCurrentLoop)
															#line 7860 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2723 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			rebalancePollingInterval = std::max(rebalancePollingInterval, SERVER_KNOBS->BG_REBALANCE_SWITCH_CHECK_INTERVAL);
															#line 2725 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("DDRebalancePaused", self->distributorId) .suppressFor(5.0) .detail("MoveType", moveType) .detail("Reason", "Disabled");
															#line 7866 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1loopHead1(loopDepth); // continue
		}
															#line 2732 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (self->priority_relocations[ddPriority] >= SERVER_KNOBS->DD_REBALANCE_PARALLELISM)
															#line 7871 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2733 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			rebalancePollingInterval = std::min(rebalancePollingInterval * 2, SERVER_KNOBS->BG_REBALANCE_MAX_POLLING_INTERVAL);
															#line 2735 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("DDRebalancePaused", self->distributorId) .suppressFor(5.0) .detail("MoveType", moveType) .detail("Reason", "DataMoveLimitReached") .detail("QueuedRelocations", self->priority_relocations[ddPriority]) .detail("PollingInterval", rebalancePollingInterval);
															#line 7877 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			return a_body1loopHead1(loopDepth); // continue
		}
															#line 2744 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		rebalancePollingInterval = std::max(rebalancePollingInterval / 2, SERVER_KNOBS->BG_REBALANCE_POLLING_INTERVAL);
															#line 2747 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent = TraceEvent(mcMove ? "MountainChopperSample" : "ValleyFillerSample", self->distributorId);
															#line 2748 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent.suppressFor(5.0);
															#line 2749 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		GetTeamRequest srcReq = GetTeamRequest(mcMove ? TeamSelect::WANT_TRUE_BEST : TeamSelect::ANY, PreferLowerDiskUtil::False, TeamMustHaveShards::True, PreferLowerReadUtil::False, PreferWithinShardLimit::False, ForReadBalance(readRebalance));
															#line 2756 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		GetTeamRequest destReq = GetTeamRequest(!mcMove ? TeamSelect::WANT_TRUE_BEST : TeamSelect::ANY, PreferLowerDiskUtil::True, TeamMustHaveShards::False, preferLowerReadTeam, PreferWithinShardLimit::False, ForReadBalance(readRebalance));
															#line 2762 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		getTeamFuture = self->getSrcDestTeams(teamCollectionIndex, srcReq, destReq, ddPriority, &traceEvent);
															#line 2764 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_2 = ready(getTeamFuture);
															#line 2764 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7896 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 3;
															#line 2764 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 2, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 7901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 2719 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		lastRead = now();
															#line 7910 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 2719 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		lastRead = now();
															#line 7919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2765 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		sourceTeam = getTeamFuture.get().first;
															#line 2766 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destTeam = getTeamFuture.get().second;
															#line 2769 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (sourceTeam.isValid() && destTeam.isValid())
															#line 8007 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2770 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (readRebalance)
															#line 8011 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2771 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_3 = store(moved,self->rebalanceReadLoad( reason, sourceTeam, destTeam, teamCollectionIndex == 0, &traceEvent));
															#line 2771 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8017 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_3.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 4;
															#line 2771 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 3, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 8022 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 2773 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_4 = store(moved,self->rebalanceTeams( reason, sourceTeam, destTeam, teamCollectionIndex == 0, &traceEvent));
															#line 2773 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8031 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_4.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 5;
															#line 2773 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 4, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 8036 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2765 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		sourceTeam = getTeamFuture.get().first;
															#line 2766 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		destTeam = getTeamFuture.get().second;
															#line 2769 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (sourceTeam.isValid() && destTeam.isValid())
															#line 8055 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2770 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (readRebalance)
															#line 8059 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2771 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_3 = store(moved,self->rebalanceReadLoad( reason, sourceTeam, destTeam, teamCollectionIndex == 0, &traceEvent));
															#line 2771 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_3.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 4;
															#line 2771 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 3, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 8070 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 2773 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				StrictFuture<Void> __when_expr_4 = store(moved,self->rebalanceTeams( reason, sourceTeam, destTeam, teamCollectionIndex == 0, &traceEvent));
															#line 2773 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 8079 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_4.get(), loopDepth); };
				static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 5;
															#line 2773 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BgDDLoadRebalanceActor, 4, Void >*>(static_cast<BgDDLoadRebalanceActor*>(this)));
															#line 8084 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 2777 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		traceEvent.detail("Moved", moved).log();
															#line 2779 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (moved)
															#line 8176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2780 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			resetCount = 0;
															#line 2781 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(mcMove ? "MountainChopperMoved" : "ValleyFillerMoved", self->distributorId) .suppressFor(5.0) .detail("QueuedRelocations", self->priority_relocations[ddPriority]) .detail("PollingInterval", rebalancePollingInterval);
															#line 8182 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2786 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			++resetCount;
															#line 2787 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (resetCount > 30)
															#line 8190 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2788 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				rebalancePollingInterval = SERVER_KNOBS->BG_REBALANCE_MAX_POLLING_INTERVAL;
															#line 8194 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 2790 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(mcMove ? "MountainChopperSkipped" : "ValleyFillerSkipped", self->distributorId) .suppressFor(5.0) .detail("QueuedRelocations", self->priority_relocations[ddPriority]) .detail("ResetCount", resetCount) .detail("PollingInterval", rebalancePollingInterval);
															#line 8198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state > 0) static_cast<BgDDLoadRebalanceActor*>(this)->actor_wait_state = 0;
		static_cast<BgDDLoadRebalanceActor*>(this)->ActorCallback< BgDDLoadRebalanceActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< BgDDLoadRebalanceActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< BgDDLoadRebalanceActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont17(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue* self;
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int teamCollectionIndex;
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DataMovementReason reason;
															#line 2696 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int resetCount;
															#line 2697 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	double lastRead;
															#line 2698 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool skipCurrentLoop;
															#line 2699 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	const bool readRebalance;
															#line 2700 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	const std::string moveType;
															#line 2702 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	int ddPriority;
															#line 2703 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool mcMove;
															#line 2704 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PreferLowerReadUtil preferLowerReadTeam;
															#line 2707 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	double rebalancePollingInterval;
															#line 2710 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	bool moved;
															#line 2711 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam> sourceTeam;
															#line 2712 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<IDataDistributionTeam> destTeam;
															#line 2747 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	TraceEvent traceEvent;
															#line 2762 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<SrcDestTeamPair> getTeamFuture;
															#line 8431 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via BgDDLoadRebalance()
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class BgDDLoadRebalanceActor final : public Actor<Void>, public ActorCallback< BgDDLoadRebalanceActor, 0, Void >, public ActorCallback< BgDDLoadRebalanceActor, 1, Void >, public ActorCallback< BgDDLoadRebalanceActor, 2, Void >, public ActorCallback< BgDDLoadRebalanceActor, 3, Void >, public ActorCallback< BgDDLoadRebalanceActor, 4, Void >, public FastAllocated<BgDDLoadRebalanceActor>, public BgDDLoadRebalanceActorState<BgDDLoadRebalanceActor> {
															#line 8436 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<BgDDLoadRebalanceActor>::operator new;
	using FastAllocated<BgDDLoadRebalanceActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14628289023124753408UL, 1256087141258132992UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BgDDLoadRebalanceActor, 0, Void >;
friend struct ActorCallback< BgDDLoadRebalanceActor, 1, Void >;
friend struct ActorCallback< BgDDLoadRebalanceActor, 2, Void >;
friend struct ActorCallback< BgDDLoadRebalanceActor, 3, Void >;
friend struct ActorCallback< BgDDLoadRebalanceActor, 4, Void >;
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	BgDDLoadRebalanceActor(DDQueue* const& self,int const& teamCollectionIndex,DataMovementReason const& reason) 
															#line 8457 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   BgDDLoadRebalanceActorState<BgDDLoadRebalanceActor>(self, teamCollectionIndex, reason),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16598836187423810560UL, 4740218205161676032UL);
		ActorExecutionContextHelper __helper(static_cast<BgDDLoadRebalanceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("BgDDLoadRebalance");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("BgDDLoadRebalance", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< BgDDLoadRebalanceActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] Future<Void> BgDDLoadRebalance( DDQueue* const& self, int const& teamCollectionIndex, DataMovementReason const& reason ) {
															#line 2695 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new BgDDLoadRebalanceActor(self, teamCollectionIndex, reason));
															#line 8494 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 2804 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"

struct DDQueueImpl {
																#line 8500 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
// This generated class is to be used only via run()
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class RunActor>
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RunActorState {
															#line 8506 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RunActorState(Reference<DDQueue> const& self,Reference<AsyncVar<bool>> const& processingUnhealthy,Reference<AsyncVar<bool>> const& processingWiggle,FutureStream<Promise<int>> const& getUnhealthyRelocationCount) 
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : self(self),
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   processingUnhealthy(processingUnhealthy),
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   processingWiggle(processingWiggle),
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   getUnhealthyRelocationCount(getUnhealthyRelocationCount),
															#line 2811 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   serversToLaunchFrom(),
															#line 2812 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   keysToLaunchFrom(),
															#line 2813 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   launchData(),
															#line 2814 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   recordMetrics(delay(SERVER_KNOBS->DD_QUEUE_LOGGING_INTERVAL)),
															#line 2816 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   ddQueueFutures(),
															#line 2818 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   rangesComplete(),
															#line 2819 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   launchQueuedWorkTimeout(Never()),
															#line 2820 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   onCleanUpDataMoveActorError(actorCollection(self->addBackgroundCleanUpDataMoveActor.getFuture()))
															#line 8535 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("run", reinterpret_cast<unsigned long>(this));

	}
	~RunActorState() 
	{
		fdb_probe_actor_destroy("run", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2823 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(int i = 0;i < self->teamCollections.size();i++) {
															#line 2824 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ddQueueFutures.push_back( BgDDLoadRebalance(self.getPtr(), i, DataMovementReason::REBALANCE_OVERUTILIZED_TEAM));
															#line 2826 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				ddQueueFutures.push_back( BgDDLoadRebalance(self.getPtr(), i, DataMovementReason::REBALANCE_UNDERUTILIZED_TEAM));
															#line 2828 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 8556 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 2829 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ddQueueFutures.push_back( BgDDLoadRebalance(self.getPtr(), i, DataMovementReason::REBALANCE_READ_OVERUTIL_TEAM));
															#line 2831 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					ddQueueFutures.push_back( BgDDLoadRebalance(self.getPtr(), i, DataMovementReason::REBALANCE_READ_UNDERUTIL_TEAM));
															#line 8562 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
															#line 2835 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ddQueueFutures.push_back(delayedAsyncVar(self->rawProcessingUnhealthy, processingUnhealthy, 0));
															#line 2836 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ddQueueFutures.push_back(delayedAsyncVar(self->rawProcessingWiggle, processingWiggle, 0));
															#line 2837 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ddQueueFutures.push_back(self->periodicalRefreshCounter());
															#line 8571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			try {
															#line 2840 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				;
															#line 8575 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunActorState();
		static_cast<RunActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3001 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (e.code() != error_code_broken_promise && e.code() != error_code_movekeys_conflict && e.code() != error_code_data_move_cancelled && e.code() != error_code_data_move_dest_team_not_found)
															#line 8605 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 3005 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				TraceEvent(SevError, "DataDistributionQueueError", self->distributorId).error(e);
															#line 8609 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
															#line 3006 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 8613 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2841 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->validate();
															#line 2845 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (launchData.startTime != -1)
															#line 8636 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2847 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->launchQueuedWork(launchData, self->ddEnabledState);
															#line 2848 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			launchData = RelocateData();
															#line 8642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2849 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (!keysToLaunchFrom.empty())
															#line 8648 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2850 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->launchQueuedWork(keysToLaunchFrom, self->ddEnabledState);
															#line 2851 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				keysToLaunchFrom = KeyRangeRef();
															#line 8654 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
															#line 2854 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(launchData.startTime == -1 && keysToLaunchFrom.empty());
															#line 2857 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_0 = self->input;
															#line 8661 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_0), FutureStream<RelocateShard>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<RelocateShard>>::value, "invalid type");
															#line 2856 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8665 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.pop(), loopDepth); };
															#line 2871 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_1 = launchQueuedWorkTimeout;
															#line 8669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 2876 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_2 = self->fetchSourceServersComplete.getFuture();
															#line 8673 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_2), FutureStream<RelocateData>>::value || std::is_same<decltype(__when_expr_2), ThreadFutureStream<RelocateData>>::value, "invalid type");
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when3(__when_expr_2.pop(), loopDepth); };
															#line 2881 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_3 = self->dataTransferComplete.getFuture();
															#line 8678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_3), FutureStream<RelocateData>>::value || std::is_same<decltype(__when_expr_3), ThreadFutureStream<RelocateData>>::value, "invalid type");
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when4(__when_expr_3.pop(), loopDepth); };
															#line 2887 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_4 = self->relocationComplete.getFuture();
															#line 8683 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_4), FutureStream<RelocateData>>::value || std::is_same<decltype(__when_expr_4), ThreadFutureStream<RelocateData>>::value, "invalid type");
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when5(__when_expr_4.pop(), loopDepth); };
															#line 2905 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_5 = rangesComplete.getFuture();
															#line 8688 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_5), FutureStream<KeyRange>>::value || std::is_same<decltype(__when_expr_5), ThreadFutureStream<KeyRange>>::value, "invalid type");
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when6(__when_expr_5.pop(), loopDepth); };
															#line 2908 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_6 = recordMetrics;
															#line 8693 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch2(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when7(__when_expr_6.get(), loopDepth); };
															#line 2992 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_7 = self->error.getFuture();
															#line 8697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch2(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when8(__when_expr_7.get(), loopDepth); };
															#line 2993 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_8 = waitForAll(ddQueueFutures);
															#line 8701 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch2(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when9(__when_expr_8.get(), loopDepth); };
															#line 2994 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto __when_expr_9 = getUnhealthyRelocationCount;
															#line 8705 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_9), FutureStream<Promise<int>>>::value || std::is_same<decltype(__when_expr_9), ThreadFutureStream<Promise<int>>>::value, "invalid type");
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch2(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when10(__when_expr_9.pop(), loopDepth); };
															#line 2997 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_10 = onCleanUpDataMoveActorError;
															#line 8710 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch2(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when11(__when_expr_10.get(), loopDepth); };
		static_cast<RunActor*>(this)->actor_wait_state = 1;
															#line 2857 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 0, RelocateShard >*>(static_cast<RunActor*>(this)));
															#line 2871 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunActor, 1, Void >*>(static_cast<RunActor*>(this)));
															#line 2876 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 2, RelocateData >*>(static_cast<RunActor*>(this)));
															#line 2881 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 3, RelocateData >*>(static_cast<RunActor*>(this)));
															#line 2887 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 4, RelocateData >*>(static_cast<RunActor*>(this)));
															#line 2905 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 5, KeyRange >*>(static_cast<RunActor*>(this)));
															#line 2908 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< RunActor, 6, Void >*>(static_cast<RunActor*>(this)));
															#line 2992 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< RunActor, 7, Void >*>(static_cast<RunActor*>(this)));
															#line 2993 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< RunActor, 8, Void >*>(static_cast<RunActor*>(this)));
															#line 2994 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorSingleCallback< RunActor, 9, Promise<int> >*>(static_cast<RunActor*>(this)));
															#line 2997 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< RunActor, 10, Void >*>(static_cast<RunActor*>(this)));
															#line 8735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(RelocateShard const& rs,int loopDepth) 
	{
															#line 2858 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (rs.isRestore())
															#line 8750 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2859 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rs.dataMove != nullptr);
															#line 2860 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rs.dataMoveId.isValid());
															#line 2861 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->launchQueuedWork(RelocateData(rs), self->ddEnabledState);
															#line 8758 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2862 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rs.cancelled)
															#line 8764 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2863 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->enqueueCancelledDataMove(rs.dataMoveId, rs.keys, self->ddEnabledState);
															#line 8768 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2865 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				bool wasEmpty = serversToLaunchFrom.empty();
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->queueRelocation(rs, serversToLaunchFrom);
															#line 2867 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (wasEmpty && !serversToLaunchFrom.empty())
															#line 8778 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 2868 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					launchQueuedWorkTimeout = delay(0, TaskPriority::DataDistributionLaunch);
															#line 8782 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RelocateShard && rs,int loopDepth) 
	{
															#line 2858 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (rs.isRestore())
															#line 8794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2859 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rs.dataMove != nullptr);
															#line 2860 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			ASSERT(rs.dataMoveId.isValid());
															#line 2861 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->launchQueuedWork(RelocateData(rs), self->ddEnabledState);
															#line 8802 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2862 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			if (rs.cancelled)
															#line 8808 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			{
															#line 2863 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->enqueueCancelledDataMove(rs.dataMoveId, rs.keys, self->ddEnabledState);
															#line 8812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2865 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				bool wasEmpty = serversToLaunchFrom.empty();
															#line 2866 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->queueRelocation(rs, serversToLaunchFrom);
															#line 2867 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				if (wasEmpty && !serversToLaunchFrom.empty())
															#line 8822 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				{
															#line 2868 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
					launchQueuedWorkTimeout = delay(0, TaskPriority::DataDistributionLaunch);
															#line 8826 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
				}
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 2872 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->launchQueuedWork(serversToLaunchFrom, self->ddEnabledState);
															#line 2873 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		serversToLaunchFrom = std::set<UID>();
															#line 2874 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchQueuedWorkTimeout = Never();
															#line 8842 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 2872 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->launchQueuedWork(serversToLaunchFrom, self->ddEnabledState);
															#line 2873 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		serversToLaunchFrom = std::set<UID>();
															#line 2874 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchQueuedWorkTimeout = Never();
															#line 8855 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(RelocateData const& results,int loopDepth) 
	{
															#line 2878 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->completeSourceFetch(results);
															#line 2879 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchData = results;
															#line 8866 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(RelocateData && results,int loopDepth) 
	{
															#line 2878 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->completeSourceFetch(results);
															#line 2879 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		launchData = results;
															#line 8877 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(RelocateData const& done,int loopDepth) 
	{
															#line 2882 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		complete(done, self->busymap, self->destBusymap);
															#line 2883 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (serversToLaunchFrom.empty() && !done.src.empty())
															#line 8888 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2884 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			launchQueuedWorkTimeout = delay(0, TaskPriority::DataDistributionLaunch);
															#line 8892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2885 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		serversToLaunchFrom.insert(done.src.begin(), done.src.end());
															#line 8896 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(RelocateData && done,int loopDepth) 
	{
															#line 2882 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		complete(done, self->busymap, self->destBusymap);
															#line 2883 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (serversToLaunchFrom.empty() && !done.src.empty())
															#line 8907 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2884 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			launchQueuedWorkTimeout = delay(0, TaskPriority::DataDistributionLaunch);
															#line 8911 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 2885 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		serversToLaunchFrom.insert(done.src.begin(), done.src.end());
															#line 8915 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when5(RelocateData const& done,int loopDepth) 
	{
															#line 2888 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->activeRelocations--;
															#line 2889 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevVerbose, "InFlightRelocationChange") .detail("Complete", done.dataMoveId) .detail("IsRestore", done.isRestore()) .detail("Total", self->activeRelocations);
															#line 2893 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->finishRelocation(done.priority, done.healthPriority);
															#line 2894 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->fetchKeysComplete.erase(done);
															#line 2896 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->noErrorActors.add( tag(delay(0, TaskPriority::DataDistributionLaunch), done.keys, rangesComplete));
															#line 2898 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (g_network->isSimulated() && debug_isCheckRelocationDuration() && now() - done.startTime > 60)
															#line 8934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2900 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(SevWarnAlways, "RelocationDurationTooLong") .detail("Duration", now() - done.startTime);
															#line 2902 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			debug_setCheckRelocationDuration(false);
															#line 8940 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when5(RelocateData && done,int loopDepth) 
	{
															#line 2888 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->activeRelocations--;
															#line 2889 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent(SevVerbose, "InFlightRelocationChange") .detail("Complete", done.dataMoveId) .detail("IsRestore", done.isRestore()) .detail("Total", self->activeRelocations);
															#line 2893 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->finishRelocation(done.priority, done.healthPriority);
															#line 2894 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->fetchKeysComplete.erase(done);
															#line 2896 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->noErrorActors.add( tag(delay(0, TaskPriority::DataDistributionLaunch), done.keys, rangesComplete));
															#line 2898 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (g_network->isSimulated() && debug_isCheckRelocationDuration() && now() - done.startTime > 60)
															#line 8960 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2900 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent(SevWarnAlways, "RelocationDurationTooLong") .detail("Duration", now() - done.startTime);
															#line 2902 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			debug_setCheckRelocationDuration(false);
															#line 8966 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when6(KeyRange const& done,int loopDepth) 
	{
															#line 2906 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		keysToLaunchFrom = done;
															#line 8976 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when6(KeyRange && done,int loopDepth) 
	{
															#line 2906 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		keysToLaunchFrom = done;
															#line 8985 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when7(Void const& _,int loopDepth) 
	{
															#line 2909 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		Promise<int64_t> req;
															#line 2910 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->getAverageShardBytes.send(req);
															#line 2912 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		recordMetrics = delay(SERVER_KNOBS->DD_QUEUE_LOGGING_INTERVAL, TaskPriority::FlushTrace);
															#line 2914 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto const highestPriorityRelocation = self->getHighestPriorityRelocation();
															#line 2916 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent("MovingData", self->distributorId) .detail("InFlight", self->activeRelocations) .detail("InQueue", self->queuedRelocations) .detail("AverageShardSize", req.getFuture().isReady() ? req.getFuture().get() : -1) .detail("UnhealthyRelocations", self->unhealthyRelocations) .detail("HighestPriority", highestPriorityRelocation) .detail("BytesWritten", self->moveBytesRate.getTotal()) .detail("BytesWrittenAverageRate", self->moveBytesRate.getAverage()) .detail("PriorityRecoverMove", self->priority_relocations[SERVER_KNOBS->PRIORITY_RECOVER_MOVE]) .detail("PriorityRebalanceUnderutilizedTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_UNDERUTILIZED_TEAM]) .detail("PriorityRebalanceOverutilizedTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_OVERUTILIZED_TEAM]) .detail("PriorityRebalanceReadUnderutilTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_READ_UNDERUTIL_TEAM]) .detail("PriorityRebalanceReadOverutilTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_READ_OVERUTIL_TEAM]) .detail("PriorityStorageWiggle", self->priority_relocations[SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE]) .detail("PriorityTeamHealthy", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_HEALTHY]) .detail("PriorityTeamContainsUndesiredServer", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_CONTAINS_UNDESIRED_SERVER]) .detail("PriorityTeamRedundant", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT]) .detail("PriorityMergeShard", self->priority_relocations[SERVER_KNOBS->PRIORITY_MERGE_SHARD]) .detail("PriorityPopulateRegion", self->priority_relocations[SERVER_KNOBS->PRIORITY_POPULATE_REGION]) .detail("PriorityTeamUnhealthy", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY]) .detail("PriorityTeam2Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_2_LEFT]) .detail("PriorityTeam1Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_1_LEFT]) .detail("PriorityTeam0Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_0_LEFT]) .detail("PrioritySplitShard", self->priority_relocations[SERVER_KNOBS->PRIORITY_SPLIT_SHARD]) .trackLatest("MovingData");
															#line 2957 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 9004 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2958 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("PhysicalShardMoveStats") .detail("MoveCreateNewPhysicalShard", self->moveCreateNewPhysicalShard) .detail("MoveReusePhysicalShard", self->moveReusePhysicalShard) .detail( "RemoteBestTeamNotReady", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]) .detail( "PrimaryNoHealthyTeam", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam]) .detail("RemoteNoHealthyTeam", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]) .detail("RemoteTeamIsFull", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsFull]) .detail( "RemoteTeamIsNotHealthy", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsNotHealthy]) .detail("UnknownForceNew", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::UnknownForceNew]) .detail("NoAnyHealthy", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::NoAnyHealthy]) .detail("DstOverloaded", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::DstOverloaded]) .detail("NoAvailablePhysicalShard", self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]) .detail("RetryLimitReached", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RetryLimitReached]);
															#line 2985 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveCreateNewPhysicalShard = 0;
															#line 2986 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveReusePhysicalShard = 0;
															#line 2987 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(int i = 0;i < self->retryFindDstReasonCount.size();++i) {
															#line 2988 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->retryFindDstReasonCount[i] = 0;
															#line 9016 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when7(Void && _,int loopDepth) 
	{
															#line 2909 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		Promise<int64_t> req;
															#line 2910 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self->getAverageShardBytes.send(req);
															#line 2912 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		recordMetrics = delay(SERVER_KNOBS->DD_QUEUE_LOGGING_INTERVAL, TaskPriority::FlushTrace);
															#line 2914 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto const highestPriorityRelocation = self->getHighestPriorityRelocation();
															#line 2916 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		TraceEvent("MovingData", self->distributorId) .detail("InFlight", self->activeRelocations) .detail("InQueue", self->queuedRelocations) .detail("AverageShardSize", req.getFuture().isReady() ? req.getFuture().get() : -1) .detail("UnhealthyRelocations", self->unhealthyRelocations) .detail("HighestPriority", highestPriorityRelocation) .detail("BytesWritten", self->moveBytesRate.getTotal()) .detail("BytesWrittenAverageRate", self->moveBytesRate.getAverage()) .detail("PriorityRecoverMove", self->priority_relocations[SERVER_KNOBS->PRIORITY_RECOVER_MOVE]) .detail("PriorityRebalanceUnderutilizedTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_UNDERUTILIZED_TEAM]) .detail("PriorityRebalanceOverutilizedTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_OVERUTILIZED_TEAM]) .detail("PriorityRebalanceReadUnderutilTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_READ_UNDERUTIL_TEAM]) .detail("PriorityRebalanceReadOverutilTeam", self->priority_relocations[SERVER_KNOBS->PRIORITY_REBALANCE_READ_OVERUTIL_TEAM]) .detail("PriorityStorageWiggle", self->priority_relocations[SERVER_KNOBS->PRIORITY_PERPETUAL_STORAGE_WIGGLE]) .detail("PriorityTeamHealthy", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_HEALTHY]) .detail("PriorityTeamContainsUndesiredServer", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_CONTAINS_UNDESIRED_SERVER]) .detail("PriorityTeamRedundant", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_REDUNDANT]) .detail("PriorityMergeShard", self->priority_relocations[SERVER_KNOBS->PRIORITY_MERGE_SHARD]) .detail("PriorityPopulateRegion", self->priority_relocations[SERVER_KNOBS->PRIORITY_POPULATE_REGION]) .detail("PriorityTeamUnhealthy", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_UNHEALTHY]) .detail("PriorityTeam2Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_2_LEFT]) .detail("PriorityTeam1Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_1_LEFT]) .detail("PriorityTeam0Left", self->priority_relocations[SERVER_KNOBS->PRIORITY_TEAM_0_LEFT]) .detail("PrioritySplitShard", self->priority_relocations[SERVER_KNOBS->PRIORITY_SPLIT_SHARD]) .trackLatest("MovingData");
															#line 2957 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 9037 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		{
															#line 2958 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			TraceEvent("PhysicalShardMoveStats") .detail("MoveCreateNewPhysicalShard", self->moveCreateNewPhysicalShard) .detail("MoveReusePhysicalShard", self->moveReusePhysicalShard) .detail( "RemoteBestTeamNotReady", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteBestTeamNotReady]) .detail( "PrimaryNoHealthyTeam", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::PrimaryNoHealthyTeam]) .detail("RemoteNoHealthyTeam", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteNoHealthyTeam]) .detail("RemoteTeamIsFull", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsFull]) .detail( "RemoteTeamIsNotHealthy", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RemoteTeamIsNotHealthy]) .detail("UnknownForceNew", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::UnknownForceNew]) .detail("NoAnyHealthy", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::NoAnyHealthy]) .detail("DstOverloaded", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::DstOverloaded]) .detail("NoAvailablePhysicalShard", self->retryFindDstReasonCount [DDQueue::RetryFindDstReason::NoAvailablePhysicalShard]) .detail("RetryLimitReached", self->retryFindDstReasonCount[DDQueue::RetryFindDstReason::RetryLimitReached]);
															#line 2985 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveCreateNewPhysicalShard = 0;
															#line 2986 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			self->moveReusePhysicalShard = 0;
															#line 2987 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			for(int i = 0;i < self->retryFindDstReasonCount.size();++i) {
															#line 2988 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
				self->retryFindDstReasonCount[i] = 0;
															#line 9049 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when10(Promise<int> const& r,int loopDepth) 
	{
															#line 2995 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		r.send(self->getUnhealthyRelocationCount());
															#line 9084 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when10(Promise<int> && r,int loopDepth) 
	{
															#line 2995 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		r.send(self->getUnhealthyRelocationCount());
															#line 9093 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 0, RelocateShard >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 1, Void >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 2, RelocateData >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 3, RelocateData >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 4, RelocateData >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 5, KeyRange >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 6, Void >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 7, Void >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 8, Void >::remove();
		static_cast<RunActor*>(this)->ActorSingleCallback< RunActor, 9, Promise<int> >::remove();
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 0, RelocateShard >*,RelocateShard const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 0, RelocateShard >*,RelocateShard && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 0, RelocateShard >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RunActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RunActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 2, RelocateData >*,RelocateData const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 2, RelocateData >*,RelocateData && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 2, RelocateData >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 3, RelocateData >*,RelocateData const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 3, RelocateData >*,RelocateData && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 3, RelocateData >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 4, RelocateData >*,RelocateData const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 4, RelocateData >*,RelocateData && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 4, RelocateData >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 5, KeyRange >*,KeyRange const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 5, KeyRange >*,KeyRange && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 5, KeyRange >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< RunActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RunActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< RunActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RunActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when8(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< RunActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< RunActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< RunActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when9(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< RunActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when9(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< RunActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 9, Promise<int> >*,Promise<int> const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when10(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorSingleCallback< RunActor, 9, Promise<int> >*,Promise<int> && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when10(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorSingleCallback< RunActor, 9, Promise<int> >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< RunActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when11(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< RunActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when11(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< RunActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 10);

	}
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<DDQueue> self;
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<AsyncVar<bool>> processingUnhealthy;
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Reference<AsyncVar<bool>> processingWiggle;
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	FutureStream<Promise<int>> getUnhealthyRelocationCount;
															#line 2811 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::set<UID> serversToLaunchFrom;
															#line 2812 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	KeyRange keysToLaunchFrom;
															#line 2813 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RelocateData launchData;
															#line 2814 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> recordMetrics;
															#line 2816 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	std::vector<Future<Void>> ddQueueFutures;
															#line 2818 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	PromiseStream<KeyRange> rangesComplete;
															#line 2819 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> launchQueuedWorkTimeout;
															#line 2820 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> onCleanUpDataMoveActorError;
															#line 9777 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via run()
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class RunActor final : public Actor<Void>, public ActorSingleCallback< RunActor, 0, RelocateShard >, public ActorCallback< RunActor, 1, Void >, public ActorSingleCallback< RunActor, 2, RelocateData >, public ActorSingleCallback< RunActor, 3, RelocateData >, public ActorSingleCallback< RunActor, 4, RelocateData >, public ActorSingleCallback< RunActor, 5, KeyRange >, public ActorCallback< RunActor, 6, Void >, public ActorCallback< RunActor, 7, Void >, public ActorCallback< RunActor, 8, Void >, public ActorSingleCallback< RunActor, 9, Promise<int> >, public ActorCallback< RunActor, 10, Void >, public FastAllocated<RunActor>, public RunActorState<RunActor> {
															#line 9782 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<RunActor>::operator new;
	using FastAllocated<RunActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2283477971368940544UL, 1375116389852092160UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< RunActor, 0, RelocateShard >;
friend struct ActorCallback< RunActor, 1, Void >;
friend struct ActorSingleCallback< RunActor, 2, RelocateData >;
friend struct ActorSingleCallback< RunActor, 3, RelocateData >;
friend struct ActorSingleCallback< RunActor, 4, RelocateData >;
friend struct ActorSingleCallback< RunActor, 5, KeyRange >;
friend struct ActorCallback< RunActor, 6, Void >;
friend struct ActorCallback< RunActor, 7, Void >;
friend struct ActorCallback< RunActor, 8, Void >;
friend struct ActorSingleCallback< RunActor, 9, Promise<int> >;
friend struct ActorCallback< RunActor, 10, Void >;
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	RunActor(Reference<DDQueue> const& self,Reference<AsyncVar<bool>> const& processingUnhealthy,Reference<AsyncVar<bool>> const& processingWiggle,FutureStream<Promise<int>> const& getUnhealthyRelocationCount) 
															#line 9809 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RunActorState<RunActor>(self, processingUnhealthy, processingWiggle, getUnhealthyRelocationCount),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(633236942609902848UL, 16810589115111353856UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("run");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< RunActor, 0, RelocateShard >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
[[nodiscard]] static Future<Void> run( Reference<DDQueue> const& self, Reference<AsyncVar<bool>> const& processingUnhealthy, Reference<AsyncVar<bool>> const& processingWiggle, FutureStream<Promise<int>> const& getUnhealthyRelocationCount ) {
															#line 2806 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new RunActor(self, processingUnhealthy, processingWiggle, getUnhealthyRelocationCount));
															#line 9841 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}

#line 3009 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
};

Future<Void> DDQueue::run(Reference<DDQueue> self,
                          Reference<AsyncVar<bool>> processingUnhealthy,
                          Reference<AsyncVar<bool>> processingWiggle,
                          FutureStream<Promise<int>> getUnhealthyRelocationCount,
                          const DDEnabledState* ddEnabledState) {
	self->ddEnabledState = ddEnabledState;
	return DDQueueImpl::run(self, processingUnhealthy, processingWiggle, getUnhealthyRelocationCount);
}
															#line 9855 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase3019()
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
template <class FlowTestCase3019Actor>
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class FlowTestCase3019ActorState {
															#line 9862 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	FlowTestCase3019ActorState(UnitTestParameters const& params) 
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		 : params(params),
															#line 3020 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   duration(2.5 * SERVER_KNOBS->DD_QUEUE_COUNTER_REFRESH_INTERVAL),
															#line 3021 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   self(),
															#line 3022 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   counterFuture(self.periodicalRefreshCounter()),
															#line 3023 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		   finishFuture(delay(duration))
															#line 9877 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase3019", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase3019ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase3019", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3024 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			std::cout << "Start trace counter unit test for " << duration << "s ...\n";
															#line 3025 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			;
															#line 9894 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase3019ActorState();
		static_cast<FlowTestCase3019Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3041 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::cout << "Finished.";
															#line 3042 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (!static_cast<FlowTestCase3019Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase3019ActorState(); static_cast<FlowTestCase3019Actor*>(this)->destroy(); return 0; }
															#line 9919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase3019Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase3019ActorState();
		static_cast<FlowTestCase3019Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3026 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_0 = counterFuture;
															#line 3025 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		if (static_cast<FlowTestCase3019Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9940 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 3027 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_1 = finishFuture;
															#line 9944 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 3030 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delayJittered(2.0);
															#line 9948 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when3(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase3019Actor*>(this)->actor_wait_state = 1;
															#line 3026 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3019Actor, 0, Void >*>(static_cast<FlowTestCase3019Actor*>(this)));
															#line 3027 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3019Actor, 1, Void >*>(static_cast<FlowTestCase3019Actor*>(this)));
															#line 3030 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3019Actor, 2, Void >*>(static_cast<FlowTestCase3019Actor*>(this)));
															#line 9957 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when3(Void const& _,int loopDepth) 
	{
															#line 3031 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::vector<UID> team(3);
															#line 3032 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < team.size();++i) {
															#line 3033 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			team[i] = UID(deterministicRandom()->randomInt(1, 400), 0);
															#line 10013 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 3035 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto reason = RelocateReason(deterministicRandom()->randomInt(0, RelocateReason::typeCount()));
															#line 3036 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto countType = DDQueue::ServerCounter::randomCountType();
															#line 3037 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self.serverCounter.increaseForTeam(team, reason, countType);
															#line 3038 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(self.serverCounter.get(team[0], reason, countType));
															#line 10023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void && _,int loopDepth) 
	{
															#line 3031 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		std::vector<UID> team(3);
															#line 3032 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		for(int i = 0;i < team.size();++i) {
															#line 3033 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
			team[i] = UID(deterministicRandom()->randomInt(1, 400), 0);
															#line 10036 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		}
															#line 3035 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto reason = RelocateReason(deterministicRandom()->randomInt(0, RelocateReason::typeCount()));
															#line 3036 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		auto countType = DDQueue::ServerCounter::randomCountType();
															#line 3037 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		self.serverCounter.increaseForTeam(team, reason, countType);
															#line 3038 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
		ASSERT(self.serverCounter.get(team[0], reason, countType));
															#line 10046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase3019Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3019Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3019Actor*>(this)->ActorCallback< FlowTestCase3019Actor, 0, Void >::remove();
		static_cast<FlowTestCase3019Actor*>(this)->ActorCallback< FlowTestCase3019Actor, 1, Void >::remove();
		static_cast<FlowTestCase3019Actor*>(this)->ActorCallback< FlowTestCase3019Actor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3019Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3019Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase3019Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3019Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3019Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase3019Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3019Actor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3019Actor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase3019Actor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	UnitTestParameters params;
															#line 3020 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	double duration;
															#line 3021 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	DDQueue self;
															#line 3022 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> counterFuture;
															#line 3023 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	Future<Void> finishFuture;
															#line 10240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase3019()
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
class FlowTestCase3019Actor final : public Actor<Void>, public ActorCallback< FlowTestCase3019Actor, 0, Void >, public ActorCallback< FlowTestCase3019Actor, 1, Void >, public ActorCallback< FlowTestCase3019Actor, 2, Void >, public FastAllocated<FlowTestCase3019Actor>, public FlowTestCase3019ActorState<FlowTestCase3019Actor> {
															#line 10245 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase3019Actor>::operator new;
	using FastAllocated<FlowTestCase3019Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4028369263809280768UL, 6260827598966004736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase3019Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase3019Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase3019Actor, 2, Void >;
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	FlowTestCase3019Actor(UnitTestParameters const& params) 
															#line 10264 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase3019ActorState<FlowTestCase3019Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase3019", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7361445140096338176UL, 9241696329374182656UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3019Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase3019");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase3019", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase3019Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
static Future<Void> flowTestCase3019( UnitTestParameters const& params ) {
															#line 3019 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
	return Future<Void>(new FlowTestCase3019Actor(params));
															#line 10297 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DDRelocationQueue.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase3019, "/DataDistribution/DDQueue/ServerCounterTrace")

#line 3044 "/Users/ec2-user/foundationdb/fdbserver/DDRelocationQueue.actor.cpp"
