#define POST_ACTOR_COMPILER 1
#line 1 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
/*
 * Resolver.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <algorithm>
#include <memory>

#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/Notified.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbrpc/Stats.h"
#include "fdbserver/ApplyMetadataMutation.h"
#include "fdbserver/ConflictSet.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/LogSystem.h"
#include "fdbserver/LogSystemDiskQueueAdapter.h"
#include "fdbserver/MasterInterface.h"
#include "fdbserver/ResolverInterface.h"
#include "fdbserver/RestoreUtil.h"
#include "fdbserver/ServerDBInfo.h"
#include "fdbserver/StorageMetrics.actor.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "flow/ActorCollection.h"
#include "flow/Error.h"
#include "flow/Histogram.h"

#include "flow/actorcompiler.h" // This must be the last #include.

namespace {
struct ProxyRequestsInfo {
	std::map<Version, ResolveTransactionBatchReply> outstandingBatches;
	Version lastVersion;

	ProxyRequestsInfo() : lastVersion(-1) {}
};
} // namespace

namespace {

class RecentStateTransactionsInfo {
public:
	RecentStateTransactionsInfo() = default;

	// Erases state transactions up to the given version (inclusive) and returns
	// the number of bytes for the erased mutations.
	int64_t eraseUpTo(Version oldestVersion) {
		recentStateTransactions.erase(recentStateTransactions.begin(),
		                              recentStateTransactions.upper_bound(oldestVersion));

		int64_t stateBytes = 0;
		while (recentStateTransactionSizes.size() && recentStateTransactionSizes.front().first <= oldestVersion) {
			stateBytes += recentStateTransactionSizes.front().second;
			recentStateTransactionSizes.pop_front();
		}
		return stateBytes;
	}

	// Adds state transactions between two versions to the reply message.
	// "initialShardChanged" indicates if commitVersion has shard changes.
	// Returns if shardChanged or a state transaction has ever happened for these versions.
	[[nodiscard]] bool applyStateTxnsToBatchReply(ResolveTransactionBatchReply* reply,
	                                              Version firstUnseenVersion,
	                                              Version commitVersion,
	                                              bool initialShardChanged) {
		bool shardChanged = initialShardChanged;
		auto stateTransactionItr = recentStateTransactions.lower_bound(firstUnseenVersion);
		auto endItr = recentStateTransactions.lower_bound(commitVersion);
		// Resolver only sends back prior state txns back, because the proxy
		// sends this request has them and will apply them via applyMetadataToCommittedTransactions();
		// and other proxies will get this version's state txns as a prior version.
		for (; stateTransactionItr != endItr; ++stateTransactionItr) {
			shardChanged = shardChanged || stateTransactionItr->value.first;
			reply->stateMutations.push_back(reply->arena, stateTransactionItr->value.second);
			reply->arena.dependsOn(stateTransactionItr->value.second.arena());
		}
		return shardChanged;
	}

	bool empty() const { return recentStateTransactionSizes.empty(); }
	// Returns the number of versions with non-empty state transactions.
	uint32_t size() const { return recentStateTransactionSizes.size(); }

	// Returns the first/smallest version of the state transactions.
	// This can only be called when empty() returns false or size() > 0.
	Version firstVersion() const { return recentStateTransactionSizes.front().first; }

	// Records non-zero stateBytes for a version.
	void addVersionBytes(Version commitVersion, int64_t stateBytes) {
		if (stateBytes > 0)
			recentStateTransactionSizes.emplace_back(commitVersion, stateBytes);
	}

	// Returns the reference to the pair of (shardChanged, stateMutations) for the given version
	std::pair<bool, Standalone<VectorRef<StateTransactionRef>>>& getStateTransactionsRef(Version commitVersion) {
		return recentStateTransactions[commitVersion];
	}

private:
	// Commit version to a pair of (shardChanged, stateMutations).
	Map<Version, std::pair<bool, Standalone<VectorRef<StateTransactionRef>>>> recentStateTransactions;

	// Only keep versions with non-zero size state transactions.
	Deque<std::pair<Version, int64_t>> recentStateTransactionSizes;
};

struct Resolver : ReferenceCounted<Resolver> {
	const UID dbgid;
	const int commitProxyCount, resolverCount;
	NotifiedVersion version;
	AsyncVar<Version> neededVersion;

	RecentStateTransactionsInfo recentStateTransactionsInfo;
	AsyncVar<int64_t> totalStateBytes;
	AsyncTrigger checkNeededVersion;
	std::map<NetworkAddress, ProxyRequestsInfo> proxyInfoMap;
	ConflictSet* conflictSet;
	TransientStorageMetricSample iopsSample;

	// Use LogSystem as backend for txnStateStore. However, the real commit
	// happens at commit proxies and we never "write" to the LogSystem at
	// Resolvers.
	LogSystemDiskQueueAdapter* logAdapter = nullptr;
	Reference<ILogSystem> logSystem;
	IKeyValueStore* txnStateStore = nullptr;
	int localTLogCount = -1;

	std::map<UID, Reference<StorageInfo>> storageCache;
	KeyRangeMap<ServerCacheInfo> keyInfo; // keyrange -> all storage servers in all DCs for the keyrange
	std::unordered_map<UID, StorageServerInterface> tssMapping;
	bool forceRecovery = false;

	Version debugMinRecentStateVersion = 0;

	// The previous commit versions per tlog
	std::vector<Version> tpcvVector;

	CounterCollection cc;
	Counter resolveBatchIn;
	Counter resolveBatchStart;
	Counter resolvedTransactions;
	Counter resolvedBytes;
	Counter resolvedReadConflictRanges;
	Counter resolvedWriteConflictRanges;
	Counter transactionsAccepted;
	Counter transactionsTooOld;
	Counter transactionsConflicted;
	Counter resolvedStateTransactions;
	Counter resolvedStateMutations;
	Counter resolvedStateBytes;
	Counter resolveBatchOut;
	Counter metricsRequests;
	Counter splitRequests;
	int numLogs;

	// End-to-end server latency of resolver requests.
	Reference<Histogram> resolverLatencyDist;

	// Queue wait times, per request.
	Reference<Histogram> queueWaitLatencyDist;

	// Actual work, per req request.
	Reference<Histogram> computeTimeDist;

	// Distribution of waiters in queue.
	// 0 or 1 will be most common, but higher values are interesting.
	Reference<Histogram> queueDepthDist;

	Future<Void> logger;

	EncryptionAtRestMode encryptMode;

	Version lastShardMove;

	Resolver(UID dbgid, int commitProxyCount, int resolverCount, EncryptionAtRestMode encryptMode)
	  : dbgid(dbgid), commitProxyCount(commitProxyCount), resolverCount(resolverCount), encryptMode(encryptMode),
	    version(-1), conflictSet(newConflictSet()), iopsSample(SERVER_KNOBS->KEY_BYTES_PER_SAMPLE),
	    cc("Resolver", dbgid.toString()), resolveBatchIn("ResolveBatchIn", cc),
	    resolveBatchStart("ResolveBatchStart", cc), resolvedTransactions("ResolvedTransactions", cc),
	    resolvedBytes("ResolvedBytes", cc), resolvedReadConflictRanges("ResolvedReadConflictRanges", cc),
	    resolvedWriteConflictRanges("ResolvedWriteConflictRanges", cc),
	    transactionsAccepted("TransactionsAccepted", cc), transactionsTooOld("TransactionsTooOld", cc),
	    transactionsConflicted("TransactionsConflicted", cc),
	    resolvedStateTransactions("ResolvedStateTransactions", cc),
	    resolvedStateMutations("ResolvedStateMutations", cc), resolvedStateBytes("ResolvedStateBytes", cc),
	    resolveBatchOut("ResolveBatchOut", cc), metricsRequests("MetricsRequests", cc),
	    splitRequests("SplitRequests", cc),
	    resolverLatencyDist(Histogram::getHistogram("Resolver"_sr, "Latency"_sr, Histogram::Unit::milliseconds)),
	    queueWaitLatencyDist(Histogram::getHistogram("Resolver"_sr, "QueueWait"_sr, Histogram::Unit::milliseconds)),
	    computeTimeDist(Histogram::getHistogram("Resolver"_sr, "ComputeTime"_sr, Histogram::Unit::milliseconds)),
	    // Distribution of queue depths, with knowledge that Histogram has 32 buckets, and each bucket will have size 1.
	    queueDepthDist(Histogram::getHistogram("Resolver"_sr, "QueueDepth"_sr, Histogram::Unit::countLinear, 0, 31)),
	    lastShardMove(invalidVersion) {
		specialCounter(cc, "Version", [this]() { return this->version.get(); });
		specialCounter(cc, "NeededVersion", [this]() { return this->neededVersion.get(); });
		specialCounter(cc, "TotalStateBytes", [this]() { return this->totalStateBytes.get(); });

		logger = cc.traceCounters("ResolverMetrics", dbgid, SERVER_KNOBS->WORKER_LOGGING_INTERVAL, "ResolverMetrics");
	}
	~Resolver() { destroyConflictSet(conflictSet); }
};
} // namespace

															#line 224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
namespace {
// This generated class is to be used only via versionReady()
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
template <class VersionReadyActor>
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class VersionReadyActorState {
															#line 231 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	VersionReadyActorState(Resolver* const& self,ProxyRequestsInfo* const& proxyInfo,Version const& prevVersion) 
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		 : self(self),
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   proxyInfo(proxyInfo),
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   prevVersion(prevVersion)
															#line 242 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
	{
		fdb_probe_actor_create("versionReady", reinterpret_cast<unsigned long>(this));

	}
	~VersionReadyActorState() 
	{
		fdb_probe_actor_destroy("versionReady", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 223 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			;
															#line 257 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~VersionReadyActorState();
		static_cast<VersionReadyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 224 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (self->recentStateTransactionsInfo.size() && proxyInfo->lastVersion <= self->recentStateTransactionsInfo.firstVersion())
															#line 287 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 226 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->neededVersion.set(std::max(self->neededVersion.get(), prevVersion));
															#line 291 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 230 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		int waiters = self->version.numWaiting();
															#line 231 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (self->version.get() < prevVersion)
															#line 297 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 232 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			waiters++;
															#line 301 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 234 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->queueDepthDist->sampleRecordCounter(waiters);
															#line 237 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->version.whenAtLeast(prevVersion);
															#line 236 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<VersionReadyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 309 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 242 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->checkNeededVersion.onTrigger();
															#line 313 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<VersionReadyActor*>(this)->actor_wait_state = 1;
															#line 237 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< VersionReadyActor, 0, Void >*>(static_cast<VersionReadyActor*>(this)));
															#line 242 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< VersionReadyActor, 1, Void >*>(static_cast<VersionReadyActor*>(this)));
															#line 320 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 239 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->queueDepthDist->sampleRecordCounter(self->version.numWaiting());
															#line 240 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<VersionReadyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~VersionReadyActorState(); static_cast<VersionReadyActor*>(this)->destroy(); return 0; }
															#line 337 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<VersionReadyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~VersionReadyActorState();
		static_cast<VersionReadyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 239 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->queueDepthDist->sampleRecordCounter(self->version.numWaiting());
															#line 240 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<VersionReadyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~VersionReadyActorState(); static_cast<VersionReadyActor*>(this)->destroy(); return 0; }
															#line 351 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<VersionReadyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~VersionReadyActorState();
		static_cast<VersionReadyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<VersionReadyActor*>(this)->actor_wait_state > 0) static_cast<VersionReadyActor*>(this)->actor_wait_state = 0;
		static_cast<VersionReadyActor*>(this)->ActorCallback< VersionReadyActor, 0, Void >::remove();
		static_cast<VersionReadyActor*>(this)->ActorCallback< VersionReadyActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< VersionReadyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< VersionReadyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< VersionReadyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< VersionReadyActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< VersionReadyActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< VersionReadyActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Resolver* self;
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ProxyRequestsInfo* proxyInfo;
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Version prevVersion;
															#line 498 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
};
// This generated class is to be used only via versionReady()
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class VersionReadyActor final : public Actor<Void>, public ActorCallback< VersionReadyActor, 0, Void >, public ActorCallback< VersionReadyActor, 1, Void >, public FastAllocated<VersionReadyActor>, public VersionReadyActorState<VersionReadyActor> {
															#line 503 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
	using FastAllocated<VersionReadyActor>::operator new;
	using FastAllocated<VersionReadyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17825276450841492736UL, 8394792501737435648UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< VersionReadyActor, 0, Void >;
friend struct ActorCallback< VersionReadyActor, 1, Void >;
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	VersionReadyActor(Resolver* const& self,ProxyRequestsInfo* const& proxyInfo,Version const& prevVersion) 
															#line 521 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		 : Actor<Void>(),
		   VersionReadyActorState<VersionReadyActor>(self, proxyInfo, prevVersion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14159113740431511296UL, 12582743155816252928UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("versionReady");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< VersionReadyActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> versionReady( Resolver* const& self, ProxyRequestsInfo* const& proxyInfo, Version const& prevVersion ) {
															#line 222 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new VersionReadyActor(self, proxyInfo, prevVersion));
															#line 554 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
}

#line 246 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"

															#line 559 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
namespace {
// This generated class is to be used only via resolveBatch()
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ResolveBatchActor>
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolveBatchActorState {
															#line 566 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolveBatchActorState(Reference<Resolver> const& self,ResolveTransactionBatchRequest const& req,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		 : self(self),
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   req(req),
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db),
															#line 250 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   debugID(),
															#line 251 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   span("R:resolveBatch"_loc, req.spanContext),
															#line 254 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   proxyAddress(req.prevVersion >= 0 ? req.reply.getEndpoint().getPrimaryAddress() : NetworkAddress()),
															#line 256 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   proxyInfo(self->proxyInfoMap[proxyAddress]),
															#line 258 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   cipherKeys()
															#line 587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
	{
		fdb_probe_actor_create("resolveBatch", reinterpret_cast<unsigned long>(this));

	}
	~ResolveBatchActorState() 
	{
		fdb_probe_actor_destroy("resolveBatch", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 259 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (self->encryptMode.isEncryptionEnabled())
															#line 602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 260 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				static const std::unordered_set<EncryptCipherDomainId> metadataDomainIds = { SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID, ENCRYPT_HEADER_DOMAIN_ID };
															#line 262 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				StrictFuture<std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>> __when_expr_0 = GetEncryptCipherKeys<ServerDBInfo>::getLatestEncryptCipherKeys( db, metadataDomainIds, BlobCipherMetrics::TLOG);
															#line 262 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 610 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ResolveBatchActor*>(this)->actor_wait_state = 1;
															#line 262 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*>(static_cast<ResolveBatchActor*>(this)));
															#line 615 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResolveBatchActorState();
		static_cast<ResolveBatchActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 268 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->resolveBatchIn;
															#line 270 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (req.debugID.present())
															#line 645 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 271 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			debugID = nondeterministicRandom()->randomUniqueID();
															#line 272 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			g_traceBatch.addAttach("CommitAttachID", req.debugID.get().first(), debugID.get().first());
															#line 273 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			g_traceBatch.addEvent("CommitDebug", debugID.get().first(), "Resolver.resolveBatch.Before");
															#line 653 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 282 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		;
															#line 657 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& cks,int loopDepth) 
	{
															#line 265 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		cipherKeys = cks;
															#line 666 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && cks,int loopDepth) 
	{
															#line 265 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		cipherKeys = cks;
															#line 675 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& cks,int loopDepth) 
	{
		loopDepth = a_body1cont2(cks, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && cks,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(cks), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state > 0) static_cast<ResolveBatchActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveBatchActor*>(this)->ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,Error err) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 294 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (debugID.present())
															#line 759 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 295 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			g_traceBatch.addEvent("CommitDebug", debugID.get().first(), "Resolver.resolveBatch.AfterQueueSizeCheck");
															#line 763 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 298 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = versionReady(self.getPtr(), &proxyInfo, req.prevVersion);
															#line 298 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 769 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ResolveBatchActor*>(this)->actor_wait_state = 3;
															#line 298 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 2, Void >*>(static_cast<ResolveBatchActor*>(this)));
															#line 774 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 282 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!(self->totalStateBytes.get() > SERVER_KNOBS->RESOLVER_STATE_MEMORY_LIMIT && self->recentStateTransactionsInfo.size() && proxyInfo.lastVersion > self->recentStateTransactionsInfo.firstVersion() && req.version > self->neededVersion.get()))
															#line 790 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 291 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->totalStateBytes.onChange() || self->neededVersion.onChange();
															#line 291 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 798 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ResolveBatchActor*>(this)->actor_wait_state = 2;
															#line 291 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 1, Void >*>(static_cast<ResolveBatchActor*>(this)));
															#line 803 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state > 0) static_cast<ResolveBatchActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveBatchActor*>(this)->ActorCallback< ResolveBatchActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResolveBatchActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (check_yield(TaskPriority::DefaultEndpoint))
															#line 912 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 301 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0, TaskPriority::Low) || delay(SERVER_KNOBS->COMMIT_SLEEP_TIME);
															#line 301 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 918 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont6when1(__when_expr_3.get(), loopDepth); };
			static_cast<ResolveBatchActor*>(this)->actor_wait_state = 4;
															#line 301 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 3, Void >*>(static_cast<ResolveBatchActor*>(this)));
															#line 923 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (check_yield(TaskPriority::DefaultEndpoint))
															#line 937 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 301 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0, TaskPriority::Low) || delay(SERVER_KNOBS->COMMIT_SLEEP_TIME);
															#line 301 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 943 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont6when1(__when_expr_3.get(), loopDepth); };
			static_cast<ResolveBatchActor*>(this)->actor_wait_state = 4;
															#line 301 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 3, Void >*>(static_cast<ResolveBatchActor*>(this)));
															#line 948 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state > 0) static_cast<ResolveBatchActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveBatchActor*>(this)->ActorCallback< ResolveBatchActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ResolveBatchActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 306 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		double queueWaitEndTime = g_network->timer();
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->queueWaitLatencyDist->sampleSeconds(queueWaitEndTime - req.requestTime());
															#line 309 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (self->version.get() == req.prevVersion)
															#line 1041 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 313 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			const double beginComputeTime = g_network->timer();
															#line 315 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			++self->resolveBatchStart;
															#line 316 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedTransactions += req.transactions.size();
															#line 317 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedBytes += req.transactions.expectedSize();
															#line 319 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (proxyInfo.lastVersion > 0)
															#line 1053 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 320 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				proxyInfo.outstandingBatches.erase(proxyInfo.outstandingBatches.begin(), proxyInfo.outstandingBatches.upper_bound(req.lastReceivedVersion));
															#line 1057 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 324 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			Version firstUnseenVersion = proxyInfo.lastVersion + 1;
															#line 325 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			proxyInfo.lastVersion = req.version;
															#line 327 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (req.debugID.present())
															#line 1065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 328 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				g_traceBatch.addEvent("CommitDebug", debugID.get().first(), "Resolver.resolveBatch.AfterOrderer");
															#line 1069 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 330 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ResolveTransactionBatchReply& reply = proxyInfo.outstandingBatches[req.version];
															#line 331 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			reply.writtenTags = req.writtenTags;
															#line 333 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			std::vector<int> commitList;
															#line 334 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			std::vector<int> tooOldList;
															#line 337 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			double expire = now() + SERVER_KNOBS->SAMPLE_EXPIRATION_TIME;
															#line 338 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ConflictBatch conflictBatch(self->conflictSet, &reply.conflictingKeyRangeMap, &reply.arena);
															#line 339 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			const Version newOldestVersion = req.version - SERVER_KNOBS->MAX_WRITE_TRANSACTION_LIFE_VERSIONS;
															#line 340 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			for(int t = 0;t < req.transactions.size();t++) {
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				conflictBatch.addTransaction(req.transactions[t], newOldestVersion);
															#line 342 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				self->resolvedReadConflictRanges += req.transactions[t].read_conflict_ranges.size();
															#line 343 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				self->resolvedWriteConflictRanges += req.transactions[t].write_conflict_ranges.size();
															#line 345 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				if (self->resolverCount > 1)
															#line 1095 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				{
															#line 346 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					for( auto it : req.transactions[t].write_conflict_ranges ) {
															#line 347 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
						self->iopsSample.addAndExpire( it.begin, SERVER_KNOBS->SAMPLE_OFFSET_PER_KEY + it.begin.size(), expire);
															#line 1101 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
					}
															#line 349 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					for( auto it : req.transactions[t].read_conflict_ranges ) {
															#line 350 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
						self->iopsSample.addAndExpire( it.begin, SERVER_KNOBS->SAMPLE_OFFSET_PER_KEY + it.begin.size(), expire);
															#line 1107 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
					}
				}
			}
															#line 354 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			conflictBatch.detectConflicts(req.version, newOldestVersion, commitList, &tooOldList);
															#line 356 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			reply.debugID = req.debugID;
															#line 357 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			reply.committed.resize(reply.arena, req.transactions.size());
															#line 358 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			for(int c = 0;c < commitList.size();c++) {
															#line 359 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				reply.committed[commitList[c]] = ConflictBatch::TransactionCommitted;
															#line 1121 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 361 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			for(int c = 0;c < tooOldList.size();c++) {
															#line 362 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				ASSERT(reply.committed[tooOldList[c]] == ConflictBatch::TransactionConflict);
															#line 363 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				reply.committed[tooOldList[c]] = ConflictBatch::TransactionTooOld;
															#line 1129 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 366 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->transactionsAccepted += commitList.size();
															#line 367 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->transactionsTooOld += tooOldList.size();
															#line 368 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->transactionsConflicted += req.transactions.size() - commitList.size() - tooOldList.size();
															#line 370 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(req.prevVersion >= 0 || req.txnStateTransactions.size() == 0);
															#line 373 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			auto& stateTransactionsPair = self->recentStateTransactionsInfo.getStateTransactionsRef(req.version);
															#line 374 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			auto& stateTransactions = stateTransactionsPair.second;
															#line 375 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			int64_t stateMutations = 0;
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			int64_t stateBytes = 0;
															#line 377 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			std::unique_ptr<LogPushData> toCommit(nullptr);
															#line 378 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			std::unique_ptr<ResolverData> resolverData(nullptr);
															#line 379 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			bool isLocked = false;
															#line 380 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 1155 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 381 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				auto lockedKey = self->txnStateStore->readValue(databaseLockedKey).get();
															#line 382 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				isLocked = lockedKey.present() && lockedKey.get().size();
															#line 383 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				toCommit.reset(new LogPushData(self->logSystem, self->localTLogCount));
															#line 384 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				resolverData.reset(new ResolverData(self->dbgid, self->logSystem, self->txnStateStore, &self->keyInfo, toCommit.get(), self->forceRecovery, req.version + 1, &self->storageCache, &self->tssMapping));
															#line 1165 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 394 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			for( int t : req.txnStateTransactions ) {
															#line 395 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				stateMutations += req.transactions[t].mutations.size();
															#line 396 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				stateBytes += req.transactions[t].mutations.expectedSize();
															#line 397 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				stateTransactions.push_back_deep( stateTransactions.arena(), StateTransactionRef(reply.committed[t] == ConflictBatch::TransactionCommitted, req.transactions[t].mutations, req.transactions[t].tenantIds));
															#line 408 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				if (reply.committed[t] == ConflictBatch::TransactionCommitted && !self->forceRecovery && SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS && (!isLocked || req.transactions[t].lock_aware))
															#line 1177 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				{
															#line 410 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					SpanContext spanContext = req.transactions[t].spanContext.present() ? req.transactions[t].spanContext.get() : SpanContext();
															#line 413 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					applyMetadataMutations(spanContext, *resolverData, req.transactions[t].mutations, self->encryptMode.isEncryptionEnabled() ? &cipherKeys : nullptr, self->encryptMode);
															#line 1183 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				}
															#line 419 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				CODE_PROBE(self->forceRecovery, "Resolver detects forced recovery");
															#line 1187 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 422 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedStateTransactions += req.txnStateTransactions.size();
															#line 423 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedStateMutations += stateMutations;
															#line 424 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedStateBytes += stateBytes;
															#line 426 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->recentStateTransactionsInfo.addVersionBytes(req.version, stateBytes);
															#line 428 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(req.version >= firstUnseenVersion);
															#line 429 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(firstUnseenVersion >= self->debugMinRecentStateVersion);
															#line 431 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			CODE_PROBE(firstUnseenVersion == req.version, "Resolver first unseen version is current version");
															#line 435 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			stateTransactionsPair.first = toCommit && toCommit->isShardChanged();
															#line 436 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			bool shardChanged = self->recentStateTransactionsInfo.applyStateTxnsToBatchReply( &reply, firstUnseenVersion, req.version, toCommit && toCommit->isShardChanged());
															#line 440 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 1209 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 441 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				auto privateMutations = toCommit->getAllMessages();
															#line 442 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				for( const auto& mutations : privateMutations ) {
															#line 443 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					reply.privateMutations.push_back(reply.arena, mutations);
															#line 444 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					reply.arena.dependsOn(mutations.arena());
															#line 1219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				}
															#line 447 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				toCommit->saveTags(reply.writtenTags);
															#line 448 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				reply.privateMutationCount = toCommit->getMutationCount();
															#line 1225 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 453 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(!proxyInfo.outstandingBatches.empty());
															#line 454 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(self->proxyInfoMap.size() <= self->commitProxyCount + 1);
															#line 457 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			Version oldestProxyVersion = req.version;
															#line 458 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			for(auto itr = self->proxyInfoMap.begin();itr != self->proxyInfoMap.end();++itr) {
															#line 460 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				if (itr->first.isValid())
															#line 1237 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				{
															#line 461 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					oldestProxyVersion = std::min(itr->second.lastVersion, oldestProxyVersion);
															#line 1241 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				}
				else
				{
															#line 464 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					ASSERT(self->debugMinRecentStateVersion == 0 || self->debugMinRecentStateVersion > itr->second.lastVersion);
															#line 1247 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				}
			}
															#line 469 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			CODE_PROBE(oldestProxyVersion == req.version, "The proxy that sent this request has the oldest current version");
															#line 471 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			CODE_PROBE(oldestProxyVersion != req.version, "The proxy that sent this request does not have the oldest current version");
															#line 474 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			bool anyPopped = false;
															#line 475 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (firstUnseenVersion <= oldestProxyVersion && self->proxyInfoMap.size() == self->commitProxyCount + 1)
															#line 1258 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 476 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				CODE_PROBE(true, "Deleting old state transactions");
															#line 477 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				int64_t erasedBytes = self->recentStateTransactionsInfo.eraseUpTo(oldestProxyVersion);
															#line 478 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				self->debugMinRecentStateVersion = oldestProxyVersion + 1;
															#line 479 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				anyPopped = erasedBytes > 0;
															#line 480 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				stateBytes -= erasedBytes;
															#line 1270 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 483 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST)
															#line 1274 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 484 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				if (!self->numLogs)
															#line 1278 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				{
															#line 485 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					reply.tpcvMap.clear();
															#line 1282 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				}
				else
				{
															#line 487 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					std::set<uint16_t> writtenTLogs;
															#line 488 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					if (req.lastShardMove < self->lastShardMove || shardChanged || req.txnStateTransactions.size() || !req.writtenTags.size())
															#line 1290 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
					{
															#line 490 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
						for(int i = 0;i < self->numLogs;i++) {
															#line 491 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
							writtenTLogs.insert(i);
															#line 1296 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
						}
															#line 493 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
						if (shardChanged)
															#line 1300 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
						{
															#line 494 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
							self->lastShardMove = req.version;
															#line 1304 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
						}
					}
					else
					{
															#line 497 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
						toCommit->getLocations(reply.writtenTags, writtenTLogs);
															#line 1311 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
					}
															#line 499 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					if (self->tpcvVector[0] == invalidVersion)
															#line 1315 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
					{
															#line 500 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
						std::fill(self->tpcvVector.begin(), self->tpcvVector.end(), req.prevVersion);
															#line 1319 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
					}
															#line 502 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
					for( uint16_t tLog : writtenTLogs ) {
															#line 503 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
						reply.tpcvMap[tLog] = self->tpcvVector[tLog];
															#line 504 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
						self->tpcvVector[tLog] = req.version;
															#line 1327 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
					}
				}
															#line 507 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				reply.lastShardMove = self->lastShardMove;
															#line 1332 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 509 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->version.set(req.version);
															#line 510 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			bool breachedLimit = self->totalStateBytes.get() <= SERVER_KNOBS->RESOLVER_STATE_MEMORY_LIMIT && self->totalStateBytes.get() + stateBytes > SERVER_KNOBS->RESOLVER_STATE_MEMORY_LIMIT;
															#line 512 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->totalStateBytes.setUnconditional(self->totalStateBytes.get() + stateBytes);
															#line 513 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (anyPopped || breachedLimit)
															#line 1342 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 514 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				self->checkNeededVersion.trigger();
															#line 1346 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 518 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			const double endComputeTime = g_network->timer();
															#line 519 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->computeTimeDist->sampleSeconds(endComputeTime - beginComputeTime);
															#line 521 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (req.debugID.present())
															#line 1354 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 522 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				g_traceBatch.addEvent("CommitDebug", debugID.get().first(), "Resolver.resolveBatch.After");
															#line 1358 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
		}
		else
		{
															#line 524 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			CODE_PROBE(true, "Duplicate resolve batch request");
															#line 1365 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 528 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		auto proxyInfoItr = self->proxyInfoMap.find(proxyAddress);
															#line 530 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (proxyInfoItr != self->proxyInfoMap.end())
															#line 1371 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 531 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			auto batchItr = proxyInfoItr->second.outstandingBatches.find(req.version);
															#line 532 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (batchItr != proxyInfoItr->second.outstandingBatches.end())
															#line 1377 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 533 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				req.reply.send(batchItr->second);
															#line 1381 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
			else
			{
															#line 535 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				CODE_PROBE(true, "No outstanding batches for version on proxy", probe::decoration::rare);
															#line 536 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				req.reply.send(Never());
															#line 1389 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
		}
		else
		{
															#line 539 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT_WE_THINK(false);
															#line 542 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			req.reply.send(Never());
															#line 1398 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 547 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		const double endTime = g_network->timer();
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->resolverLatencyDist->sampleSeconds(endTime - req.requestTime());
															#line 550 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->resolveBatchOut;
															#line 552 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ResolveBatchActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResolveBatchActorState(); static_cast<ResolveBatchActor*>(this)->destroy(); return 0; }
															#line 1408 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<ResolveBatchActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResolveBatchActorState();
		static_cast<ResolveBatchActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
															#line 302 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		g_network->setCurrentTask(TaskPriority::DefaultEndpoint);
															#line 1420 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
															#line 302 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		g_network->setCurrentTask(TaskPriority::DefaultEndpoint);
															#line 1429 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state > 0) static_cast<ResolveBatchActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveBatchActor*>(this)->ActorCallback< ResolveBatchActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ResolveBatchActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<Resolver> self;
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolveTransactionBatchRequest req;
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 250 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Optional<UID> debugID;
															#line 251 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Span span;
															#line 254 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	NetworkAddress proxyAddress;
															#line 256 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ProxyRequestsInfo& proxyInfo;
															#line 258 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> cipherKeys;
															#line 1525 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
};
// This generated class is to be used only via resolveBatch()
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolveBatchActor final : public Actor<Void>, public ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >, public ActorCallback< ResolveBatchActor, 1, Void >, public ActorCallback< ResolveBatchActor, 2, Void >, public ActorCallback< ResolveBatchActor, 3, Void >, public FastAllocated<ResolveBatchActor>, public ResolveBatchActorState<ResolveBatchActor> {
															#line 1530 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
	using FastAllocated<ResolveBatchActor>::operator new;
	using FastAllocated<ResolveBatchActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10801859968065771008UL, 12615149806211552000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >;
friend struct ActorCallback< ResolveBatchActor, 1, Void >;
friend struct ActorCallback< ResolveBatchActor, 2, Void >;
friend struct ActorCallback< ResolveBatchActor, 3, Void >;
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolveBatchActor(Reference<Resolver> const& self,ResolveTransactionBatchRequest const& req,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 1550 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		 : Actor<Void>(),
		   ResolveBatchActorState<ResolveBatchActor>(self, req, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4614177339645472000UL, 11787679010325726976UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resolveBatch");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResolveBatchActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ResolveBatchActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ResolveBatchActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> resolveBatch( Reference<Resolver> const& self, ResolveTransactionBatchRequest const& req, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 247 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ResolveBatchActor(self, req, db));
															#line 1586 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
}

#line 554 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"

namespace {

// TODO: refactor with the one in CommitProxyServer.actor.cpp
struct TransactionStateResolveContext {
	// Maximum sequence for txnStateRequest, this is defined when the request last flag is set.
	Sequence maxSequence = std::numeric_limits<Sequence>::max();

	// Flags marks received transaction state requests, we only process the transaction request when *all* requests are
	// received.
	std::unordered_set<Sequence> receivedSequences;

	Reference<Resolver> pResolverData;

	// Pointer to transaction state store, shortcut for commitData.txnStateStore
	IKeyValueStore* pTxnStateStore = nullptr;

	// Actor streams
	PromiseStream<Future<Void>>* pActors = nullptr;

	// Flag reports if the transaction state request is complete. This request should only happen during recover, i.e.
	// once per Resolver.
	bool processed = false;

	TransactionStateResolveContext() = default;

	TransactionStateResolveContext(Reference<Resolver> pResolverData_, PromiseStream<Future<Void>>* pActors_)
	  : pResolverData(pResolverData_), pTxnStateStore(pResolverData_->txnStateStore), pActors(pActors_) {
		ASSERT(pTxnStateStore != nullptr || !SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS);
	}
};

															#line 1622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
// This generated class is to be used only via processCompleteTransactionStateRequest()
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ProcessCompleteTransactionStateRequestActor>
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ProcessCompleteTransactionStateRequestActorState {
															#line 1628 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ProcessCompleteTransactionStateRequestActorState(Reference<Resolver> const& self,TransactionStateResolveContext* const& pContext,Reference<AsyncVar<ServerDBInfo> const> const& db,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>* const& cipherKeys) 
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		 : self(self),
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   pContext(pContext),
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db),
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   cipherKeys(cipherKeys),
															#line 591 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   txnKeys(allKeys),
															#line 592 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   tag_uid()
															#line 1645 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
	{
		fdb_probe_actor_create("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this));

	}
	~ProcessCompleteTransactionStateRequestActorState() 
	{
		fdb_probe_actor_destroy("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			RangeResult UIDtoTagMap = pContext->pTxnStateStore->readRange(serverTagKeys).get();
															#line 595 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			for( const KeyValueRef& kv : UIDtoTagMap ) {
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				tag_uid[decodeServerTagValue(kv.value)] = decodeServerTagKey(kv.key);
															#line 1664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			}
															#line 599 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			;
															#line 1668 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ProcessCompleteTransactionStateRequestActorState();
		static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		auto lockedKey = pContext->pTxnStateStore->readValue(databaseLockedKey).get();
															#line 664 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pTxnStateStore->enableSnapshot();
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessCompleteTransactionStateRequestActorState(); static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->destroy(); return 0; }
															#line 1695 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessCompleteTransactionStateRequestActorState();
		static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = yield();
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1716 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->actor_wait_state = 1;
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*>(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)));
															#line 1721 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 602 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		RangeResult data = pContext->pTxnStateStore ->readRange(txnKeys, SERVER_KNOBS->BUGGIFIED_ROW_LIMIT, SERVER_KNOBS->APPLY_MUTATION_BYTES) .get();
															#line 606 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!data.size())
															#line 1745 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 609 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		((KeyRangeRef&)txnKeys) = KeyRangeRef(keyAfter(data.back().key, txnKeys.arena()), txnKeys.end);
															#line 611 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		MutationsVec mutations;
															#line 612 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		std::vector<std::pair<MapPair<Key, ServerCacheInfo>, int>> keyInfoData;
															#line 613 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		std::vector<UID> src, dest;
															#line 614 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		ServerCacheInfo info;
															#line 616 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		auto updateTagInfo = [pContext = pContext](const std::vector<UID>& uids, std::vector<Tag>& tags, std::vector<Reference<StorageInfo>>& storageInfoItems) { for (const auto& id : uids) { auto storageInfo = getStorageInfo(id, &pContext->pResolverData->storageCache, pContext->pTxnStateStore); ASSERT(storageInfo->tag != invalidTag); tags.push_back(storageInfo->tag); storageInfoItems.push_back(storageInfo); } };
															#line 626 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		for( auto& kv : data ) {
															#line 627 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (!kv.key.startsWith(keyServersPrefix))
															#line 1765 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 628 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				mutations.emplace_back(mutations.arena(), MutationRef::SetValue, kv.key, kv.value);
															#line 1769 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				continue;
			}
															#line 632 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			KeyRef k = kv.key.removePrefix(keyServersPrefix);
															#line 633 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (k == allKeys.end)
															#line 1776 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
				continue;
			}
															#line 636 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			decodeKeyServersValue(tag_uid, kv.value, src, dest);
															#line 638 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			info.tags.clear();
															#line 640 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			info.src_info.clear();
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			updateTagInfo(src, info.tags, info.src_info);
															#line 643 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			info.dest_info.clear();
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			updateTagInfo(dest, info.tags, info.dest_info);
															#line 646 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			uniquify(info.tags);
															#line 647 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			keyInfoData.emplace_back(MapPair<Key, ServerCacheInfo>(k, info), 1);
															#line 1796 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 652 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pResolverData->keyInfo.rawInsert(keyInfoData);
															#line 654 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		bool confChanges;
															#line 655 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		ResolverData resolverData( pContext->pResolverData->dbgid, pContext->pTxnStateStore, &pContext->pResolverData->keyInfo, confChanges);
															#line 657 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		applyMetadataMutations(SpanContext(), resolverData, mutations, cipherKeys, self->encryptMode);
															#line 1806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 602 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		RangeResult data = pContext->pTxnStateStore ->readRange(txnKeys, SERVER_KNOBS->BUGGIFIED_ROW_LIMIT, SERVER_KNOBS->APPLY_MUTATION_BYTES) .get();
															#line 606 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!data.size())
															#line 1817 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 609 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		((KeyRangeRef&)txnKeys) = KeyRangeRef(keyAfter(data.back().key, txnKeys.arena()), txnKeys.end);
															#line 611 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		MutationsVec mutations;
															#line 612 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		std::vector<std::pair<MapPair<Key, ServerCacheInfo>, int>> keyInfoData;
															#line 613 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		std::vector<UID> src, dest;
															#line 614 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		ServerCacheInfo info;
															#line 616 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		auto updateTagInfo = [pContext = pContext](const std::vector<UID>& uids, std::vector<Tag>& tags, std::vector<Reference<StorageInfo>>& storageInfoItems) { for (const auto& id : uids) { auto storageInfo = getStorageInfo(id, &pContext->pResolverData->storageCache, pContext->pTxnStateStore); ASSERT(storageInfo->tag != invalidTag); tags.push_back(storageInfo->tag); storageInfoItems.push_back(storageInfo); } };
															#line 626 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		for( auto& kv : data ) {
															#line 627 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (!kv.key.startsWith(keyServersPrefix))
															#line 1837 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 628 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				mutations.emplace_back(mutations.arena(), MutationRef::SetValue, kv.key, kv.value);
															#line 1841 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				continue;
			}
															#line 632 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			KeyRef k = kv.key.removePrefix(keyServersPrefix);
															#line 633 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (k == allKeys.end)
															#line 1848 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
				continue;
			}
															#line 636 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			decodeKeyServersValue(tag_uid, kv.value, src, dest);
															#line 638 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			info.tags.clear();
															#line 640 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			info.src_info.clear();
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			updateTagInfo(src, info.tags, info.src_info);
															#line 643 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			info.dest_info.clear();
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			updateTagInfo(dest, info.tags, info.dest_info);
															#line 646 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			uniquify(info.tags);
															#line 647 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			keyInfoData.emplace_back(MapPair<Key, ServerCacheInfo>(k, info), 1);
															#line 1868 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 652 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pResolverData->keyInfo.rawInsert(keyInfoData);
															#line 654 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		bool confChanges;
															#line 655 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		ResolverData resolverData( pContext->pResolverData->dbgid, pContext->pTxnStateStore, &pContext->pResolverData->keyInfo, confChanges);
															#line 657 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		applyMetadataMutations(SpanContext(), resolverData, mutations, cipherKeys, self->encryptMode);
															#line 1878 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->actor_wait_state > 0) static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<Resolver> self;
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	TransactionStateResolveContext* pContext;
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>* cipherKeys;
															#line 591 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	KeyRange txnKeys;
															#line 592 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	std::map<Tag, UID> tag_uid;
															#line 1970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
};
// This generated class is to be used only via processCompleteTransactionStateRequest()
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ProcessCompleteTransactionStateRequestActor final : public Actor<Void>, public ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >, public FastAllocated<ProcessCompleteTransactionStateRequestActor>, public ProcessCompleteTransactionStateRequestActorState<ProcessCompleteTransactionStateRequestActor> {
															#line 1975 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
	using FastAllocated<ProcessCompleteTransactionStateRequestActor>::operator new;
	using FastAllocated<ProcessCompleteTransactionStateRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(788390845804629248UL, 1393619339681845504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >;
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ProcessCompleteTransactionStateRequestActor(Reference<Resolver> const& self,TransactionStateResolveContext* const& pContext,Reference<AsyncVar<ServerDBInfo> const> const& db,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>* const& cipherKeys) 
															#line 1992 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		 : Actor<Void>(),
		   ProcessCompleteTransactionStateRequestActorState<ProcessCompleteTransactionStateRequestActor>(self, pContext, db, cipherKeys),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8499504063907237888UL, 15656031568419423488UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("processCompleteTransactionStateRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> processCompleteTransactionStateRequest( Reference<Resolver> const& self, TransactionStateResolveContext* const& pContext, Reference<AsyncVar<ServerDBInfo> const> const& db, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>* const& cipherKeys ) {
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ProcessCompleteTransactionStateRequestActor(self, pContext, db, cipherKeys));
															#line 2024 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
}

#line 668 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"

															#line 2029 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
// This generated class is to be used only via processTransactionStateRequestPart()
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ProcessTransactionStateRequestPartActor>
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ProcessTransactionStateRequestPartActorState {
															#line 2035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ProcessTransactionStateRequestPartActorState(Reference<Resolver> const& self,TransactionStateResolveContext* const& pContext,TxnStateRequest const& request,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		 : self(self),
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   pContext(pContext),
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   request(request),
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db)
															#line 2048 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
	{
		fdb_probe_actor_create("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this));

	}
	~ProcessTransactionStateRequestPartActorState() 
	{
		fdb_probe_actor_destroy("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 673 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(pContext->pResolverData.getPtr() != nullptr);
															#line 674 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(pContext->pActors != nullptr);
															#line 676 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (pContext->receivedSequences.contains(request.sequence))
															#line 2067 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 678 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				pContext->pActors->send(broadcastTxnRequest(request, SERVER_KNOBS->TXN_STATE_SEND_AMOUNT, true));
															#line 679 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = yield();
															#line 679 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2075 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 1;
															#line 679 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*>(static_cast<ProcessTransactionStateRequestPartActor*>(this)));
															#line 2080 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 683 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (request.last)
															#line 2108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 685 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			pContext->maxSequence = request.sequence + 1;
															#line 2112 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 687 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->receivedSequences.insert(request.sequence);
															#line 691 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		for( auto& kv : request.data ) {
															#line 692 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			pContext->pTxnStateStore->set(kv, &request.arena);
															#line 2120 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 694 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pTxnStateStore->commit(true);
															#line 696 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (pContext->receivedSequences.size() == pContext->maxSequence)
															#line 2126 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 698 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(!pContext->processed);
															#line 699 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			cipherKeys = std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>();
															#line 700 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (self->encryptMode.isEncryptionEnabled())
															#line 2134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 701 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				static const std::unordered_set<EncryptCipherDomainId> metadataDomainIds = { SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID, ENCRYPT_HEADER_DOMAIN_ID };
															#line 704 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				StrictFuture<std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>> __when_expr_1 = GetEncryptCipherKeys<ServerDBInfo>::getLatestEncryptCipherKeys( db, metadataDomainIds, BlobCipherMetrics::TLOG);
															#line 704 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 2;
															#line 704 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*>(static_cast<ProcessTransactionStateRequestPartActor*>(this)));
															#line 2147 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont7(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 680 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessTransactionStateRequestPartActorState(); static_cast<ProcessTransactionStateRequestPartActor*>(this)->destroy(); return 0; }
															#line 2166 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 680 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessTransactionStateRequestPartActorState(); static_cast<ProcessTransactionStateRequestPartActor*>(this)->destroy(); return 0; }
															#line 2178 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state > 0) static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 714 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pActors->send(broadcastTxnRequest(request, SERVER_KNOBS->TXN_STATE_SEND_AMOUNT, true));
															#line 715 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = yield();
															#line 715 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2269 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 4;
															#line 715 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*>(static_cast<ProcessTransactionStateRequestPartActor*>(this)));
															#line 2274 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
															#line 709 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = processCompleteTransactionStateRequest( self, pContext, db, self->encryptMode.isEncryptionEnabled() ? &cipherKeys : nullptr);
															#line 709 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2285 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont7when1(__when_expr_2.get(), loopDepth); };
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 3;
															#line 709 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*>(static_cast<ProcessTransactionStateRequestPartActor*>(this)));
															#line 2290 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& cks,int loopDepth) 
	{
															#line 707 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		cipherKeys = cks;
															#line 2299 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && cks,int loopDepth) 
	{
															#line 707 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		cipherKeys = cks;
															#line 2308 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& cks,int loopDepth) 
	{
		loopDepth = a_body1cont8(cks, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && cks,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(cks), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state > 0) static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,Error err) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
															#line 711 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->processed = true;
															#line 2392 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
															#line 711 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->processed = true;
															#line 2401 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state > 0) static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont11(Void const& _,int loopDepth) 
	{
															#line 716 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessTransactionStateRequestPartActorState(); static_cast<ProcessTransactionStateRequestPartActor*>(this)->destroy(); return 0; }
															#line 2485 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11(Void && _,int loopDepth) 
	{
															#line 716 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessTransactionStateRequestPartActorState(); static_cast<ProcessTransactionStateRequestPartActor*>(this)->destroy(); return 0; }
															#line 2497 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state > 0) static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<Resolver> self;
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	TransactionStateResolveContext* pContext;
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	TxnStateRequest request;
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 699 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> cipherKeys;
															#line 2590 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
};
// This generated class is to be used only via processTransactionStateRequestPart()
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ProcessTransactionStateRequestPartActor final : public Actor<Void>, public ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >, public ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >, public ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >, public ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >, public FastAllocated<ProcessTransactionStateRequestPartActor>, public ProcessTransactionStateRequestPartActorState<ProcessTransactionStateRequestPartActor> {
															#line 2595 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
	using FastAllocated<ProcessTransactionStateRequestPartActor>::operator new;
	using FastAllocated<ProcessTransactionStateRequestPartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9691265197512057088UL, 9234985123382214400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >;
friend struct ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >;
friend struct ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >;
friend struct ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >;
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ProcessTransactionStateRequestPartActor(Reference<Resolver> const& self,TransactionStateResolveContext* const& pContext,TxnStateRequest const& request,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 2615 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		 : Actor<Void>(),
		   ProcessTransactionStateRequestPartActorState<ProcessTransactionStateRequestPartActor>(self, pContext, request, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6249502656935116032UL, 7110093004433443328UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("processTransactionStateRequestPart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> processTransactionStateRequestPart( Reference<Resolver> const& self, TransactionStateResolveContext* const& pContext, TxnStateRequest const& request, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ProcessTransactionStateRequestPartActor(self, pContext, request, db));
															#line 2650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
}

#line 718 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"

} // anonymous namespace

															#line 2657 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
namespace {
// This generated class is to be used only via resolverCore()
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ResolverCoreActor>
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolverCoreActorState {
															#line 2664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverCoreActorState(ResolverInterface const& resolver,InitializeResolverRequest const& initReq,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		 : resolver(resolver),
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   initReq(initReq),
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db),
															#line 724 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   self(new Resolver(resolver.id(), initReq.commitProxyCount, initReq.resolverCount, initReq.encryptMode)),
															#line 726 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   actors(false),
															#line 727 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   doPollMetrics(self->resolverCount > 1 ? Void() : Future<Void>(Never())),
															#line 728 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   addActor()
															#line 2683 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
	{
		fdb_probe_actor_create("resolverCore", reinterpret_cast<unsigned long>(this));

	}
	~ResolverCoreActorState() 
	{
		fdb_probe_actor_destroy("resolverCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 729 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			actors.add(waitFailureServer(resolver.waitFailure.getFuture()));
															#line 730 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			actors.add(traceRole(Role::RESOLVER, resolver.id()));
															#line 732 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			TraceEvent("ResolverInit", resolver.id()) .detail("RecoveryCount", initReq.recoveryCount) .detail("EncryptMode", initReq.encryptMode.toString());
															#line 737 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			;
															#line 2704 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResolverCoreActorState();
		static_cast<ResolverCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 744 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->logSystem = ILogSystem::fromServerDBInfo(resolver.id(), db->get(), false, addActor);
															#line 745 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->localTLogCount = db->get().logSystemConfig.numLogs();
															#line 746 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		onError = transformError(actorCollection(addActor.getFuture()), broken_promise(), resolver_failed());
															#line 748 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		transactionStateResolveContext = TransactionStateResolveContext();
															#line 749 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 2735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 750 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->logAdapter = new LogSystemDiskQueueAdapter(self->logSystem, Reference<AsyncVar<PeekTxsInfo>>(), 1, false);
															#line 751 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->txnStateStore = keyValueStoreLogSystem( self->logAdapter, db, resolver.id(), 2e9, true, true, true, self->encryptMode.isEncryptionEnabled());
															#line 755 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(self->txnStateStore->readValue(StringRef()));
															#line 755 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (static_cast<ResolverCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2745 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ResolverCoreActor*>(this)->actor_wait_state = 2;
															#line 755 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 1, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 2750 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 737 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!(!(initReq.masterLifetime.isEqual(db->get().masterLifetime) && db->get().recoveryState >= RecoveryState::RECOVERY_TRANSACTION)))
															#line 2771 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 740 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 740 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2779 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ResolverCoreActor*>(this)->actor_wait_state = 1;
															#line 740 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 0, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 2784 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state > 0) static_cast<ResolverCoreActor*>(this)->actor_wait_state = 0;
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 767 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		;
															#line 2893 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 758 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		transactionStateResolveContext = TransactionStateResolveContext(self, &addActor);
															#line 760 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST)
															#line 2904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 761 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->numLogs = db->get().logSystemConfig.numLogs();
															#line 762 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->tpcvVector.resize(1 + self->numLogs, 0);
															#line 763 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			std::fill(self->tpcvVector.begin(), self->tpcvVector.end(), invalidVersion);
															#line 2912 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 758 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		transactionStateResolveContext = TransactionStateResolveContext(self, &addActor);
															#line 760 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST)
															#line 2924 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 761 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->numLogs = db->get().logSystemConfig.numLogs();
															#line 762 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			self->tpcvVector.resize(1 + self->numLogs, 0);
															#line 763 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			std::fill(self->tpcvVector.begin(), self->tpcvVector.end(), invalidVersion);
															#line 2932 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state > 0) static_cast<ResolverCoreActor*>(this)->actor_wait_state = 0;
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 768 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		FutureStream<ResolveTransactionBatchRequest> __when_expr_2 = resolver.resolve.getFuture();
															#line 767 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3026 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_2.pop(), loopDepth); };
															#line 771 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		FutureStream<ResolutionMetricsRequest> __when_expr_3 = resolver.metrics.getFuture();
															#line 3030 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when2(__when_expr_3.pop(), loopDepth); };
															#line 776 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		FutureStream<ResolutionSplitRequest> __when_expr_4 = resolver.split.getFuture();
															#line 3034 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when3(__when_expr_4.pop(), loopDepth); };
															#line 784 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_5 = actors.getResult();
															#line 3038 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when4(__when_expr_5.get(), loopDepth); };
															#line 785 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_6 = onError;
															#line 3042 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when5(__when_expr_6.get(), loopDepth); };
															#line 786 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_7 = doPollMetrics;
															#line 3046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when6(__when_expr_7.get(), loopDepth); };
															#line 790 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		FutureStream<TxnStateRequest> __when_expr_8 = resolver.txnState.getFuture();
															#line 3050 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when7(__when_expr_8.pop(), loopDepth); };
		static_cast<ResolverCoreActor*>(this)->actor_wait_state = 3;
															#line 768 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*>(static_cast<ResolverCoreActor*>(this)));
															#line 771 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >*>(static_cast<ResolverCoreActor*>(this)));
															#line 776 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >*>(static_cast<ResolverCoreActor*>(this)));
															#line 784 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 5, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 785 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 6, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 786 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 7, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 790 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >*>(static_cast<ResolverCoreActor*>(this)));
															#line 3067 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(ResolveTransactionBatchRequest const& batch,int loopDepth) 
	{
															#line 769 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		actors.add(resolveBatch(self, batch, db));
															#line 3082 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(ResolveTransactionBatchRequest && batch,int loopDepth) 
	{
															#line 769 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		actors.add(resolveBatch(self, batch, db));
															#line 3091 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when2(ResolutionMetricsRequest const& req,int loopDepth) 
	{
															#line 772 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->metricsRequests;
															#line 773 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		req.reply.send(self->iopsSample.getEstimate(SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS ? normalKeys : allKeys));
															#line 3102 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when2(ResolutionMetricsRequest && req,int loopDepth) 
	{
															#line 772 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->metricsRequests;
															#line 773 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		req.reply.send(self->iopsSample.getEstimate(SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS ? normalKeys : allKeys));
															#line 3113 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when3(ResolutionSplitRequest const& req,int loopDepth) 
	{
															#line 777 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->splitRequests;
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		ResolutionSplitReply rep;
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		rep.key = self->iopsSample.splitEstimate(req.range, req.offset, req.front);
															#line 780 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		rep.used = self->iopsSample.getEstimate(req.front ? KeyRangeRef(req.range.begin, rep.key) : KeyRangeRef(rep.key, req.range.end));
															#line 782 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		req.reply.send(rep);
															#line 3130 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when3(ResolutionSplitRequest && req,int loopDepth) 
	{
															#line 777 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->splitRequests;
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		ResolutionSplitReply rep;
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		rep.key = self->iopsSample.splitEstimate(req.range, req.offset, req.front);
															#line 780 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		rep.used = self->iopsSample.getEstimate(req.front ? KeyRangeRef(req.range.begin, rep.key) : KeyRangeRef(rep.key, req.range.end));
															#line 782 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		req.reply.send(rep);
															#line 3147 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when6(Void const& _,int loopDepth) 
	{
															#line 787 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->iopsSample.poll();
															#line 788 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		doPollMetrics = delay(SERVER_KNOBS->SAMPLE_POLL_TIME);
															#line 3182 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when6(Void && _,int loopDepth) 
	{
															#line 787 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		self->iopsSample.poll();
															#line 788 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		doPollMetrics = delay(SERVER_KNOBS->SAMPLE_POLL_TIME);
															#line 3193 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when7(TxnStateRequest const& request,int loopDepth) 
	{
															#line 791 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 3202 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 792 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			addActor.send(processTransactionStateRequestPart(self, &transactionStateResolveContext, request, db));
															#line 3206 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
		else
		{
															#line 794 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(false);
															#line 3212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when7(TxnStateRequest && request,int loopDepth) 
	{
															#line 791 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 3222 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 792 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			addActor.send(processTransactionStateRequestPart(self, &transactionStateResolveContext, request, db));
															#line 3226 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
		else
		{
															#line 794 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(false);
															#line 3232 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state > 0) static_cast<ResolverCoreActor*>(this)->actor_wait_state = 0;
		static_cast<ResolverCoreActor*>(this)->ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 5, Void >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 6, Void >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 7, Void >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >::remove();

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*,ResolveTransactionBatchRequest const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*,ResolveTransactionBatchRequest && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >*,ResolutionMetricsRequest const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >*,ResolutionMetricsRequest && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >*,ResolutionSplitRequest const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >*,ResolutionSplitRequest && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >*,TxnStateRequest const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >*,TxnStateRequest && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 8);

	}
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverInterface resolver;
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	InitializeResolverRequest initReq;
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 724 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<Resolver> self;
															#line 726 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ActorCollection actors;
															#line 727 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Future<Void> doPollMetrics;
															#line 728 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 746 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Future<Void> onError;
															#line 748 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	TransactionStateResolveContext transactionStateResolveContext;
															#line 3667 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
};
// This generated class is to be used only via resolverCore()
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolverCoreActor final : public Actor<Void>, public ActorCallback< ResolverCoreActor, 0, Void >, public ActorCallback< ResolverCoreActor, 1, Void >, public ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >, public ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >, public ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >, public ActorCallback< ResolverCoreActor, 5, Void >, public ActorCallback< ResolverCoreActor, 6, Void >, public ActorCallback< ResolverCoreActor, 7, Void >, public ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >, public FastAllocated<ResolverCoreActor>, public ResolverCoreActorState<ResolverCoreActor> {
															#line 3672 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
	using FastAllocated<ResolverCoreActor>::operator new;
	using FastAllocated<ResolverCoreActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17510283862942207232UL, 16992208973932771328UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResolverCoreActor, 0, Void >;
friend struct ActorCallback< ResolverCoreActor, 1, Void >;
friend struct ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >;
friend struct ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >;
friend struct ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >;
friend struct ActorCallback< ResolverCoreActor, 5, Void >;
friend struct ActorCallback< ResolverCoreActor, 6, Void >;
friend struct ActorCallback< ResolverCoreActor, 7, Void >;
friend struct ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >;
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverCoreActor(ResolverInterface const& resolver,InitializeResolverRequest const& initReq,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 3697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		 : Actor<Void>(),
		   ResolverCoreActorState<ResolverCoreActor>(resolver, initReq, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(574549493760680960UL, 16065327264668821760UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resolverCore");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResolverCoreActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResolverCoreActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> resolverCore( ResolverInterface const& resolver, InitializeResolverRequest const& initReq, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ResolverCoreActor(resolver, initReq, db));
															#line 3732 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
}

#line 799 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"

															#line 3737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
namespace {
// This generated class is to be used only via checkRemoved()
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
template <class CheckRemovedActor>
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class CheckRemovedActorState {
															#line 3744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	CheckRemovedActorState(Reference<AsyncVar<ServerDBInfo> const> const& db,uint64_t const& recoveryCount,ResolverInterface const& myInterface) 
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		 : db(db),
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   recoveryCount(recoveryCount),
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   myInterface(myInterface)
															#line 3755 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
	{
		fdb_probe_actor_create("checkRemoved", reinterpret_cast<unsigned long>(this));

	}
	~CheckRemovedActorState() 
	{
		fdb_probe_actor_destroy("checkRemoved", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 803 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			;
															#line 3770 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckRemovedActorState();
		static_cast<CheckRemovedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 804 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (db->get().recoveryCount >= recoveryCount && std::find(db->get().resolvers.begin(), db->get().resolvers.end(), myInterface) == db->get().resolvers.end())
															#line 3800 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		{
															#line 806 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			return a_body1Catch1(worker_removed(), std::max(0, loopDepth - 1));
															#line 3804 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
															#line 807 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 807 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<CheckRemovedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3810 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<CheckRemovedActor*>(this)->actor_wait_state = 1;
															#line 807 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckRemovedActor, 0, Void >*>(static_cast<CheckRemovedActor*>(this)));
															#line 3815 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckRemovedActor*>(this)->actor_wait_state > 0) static_cast<CheckRemovedActor*>(this)->actor_wait_state = 0;
		static_cast<CheckRemovedActor*>(this)->ActorCallback< CheckRemovedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckRemovedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckRemovedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckRemovedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	uint64_t recoveryCount;
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverInterface myInterface;
															#line 3913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
};
// This generated class is to be used only via checkRemoved()
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class CheckRemovedActor final : public Actor<Void>, public ActorCallback< CheckRemovedActor, 0, Void >, public FastAllocated<CheckRemovedActor>, public CheckRemovedActorState<CheckRemovedActor> {
															#line 3918 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
	using FastAllocated<CheckRemovedActor>::operator new;
	using FastAllocated<CheckRemovedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1309474672110072576UL, 8840002601574094336UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckRemovedActor, 0, Void >;
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	CheckRemovedActor(Reference<AsyncVar<ServerDBInfo> const> const& db,uint64_t const& recoveryCount,ResolverInterface const& myInterface) 
															#line 3935 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		 : Actor<Void>(),
		   CheckRemovedActorState<CheckRemovedActor>(db, recoveryCount, myInterface),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15518436168711072512UL, 6662910151609573120UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkRemoved");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckRemovedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> checkRemoved( Reference<AsyncVar<ServerDBInfo> const> const& db, uint64_t const& recoveryCount, ResolverInterface const& myInterface ) {
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new CheckRemovedActor(db, recoveryCount, myInterface));
															#line 3968 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
}

#line 810 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"

															#line 3973 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
namespace {
// This generated class is to be used only via resolver()
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ResolverActor>
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolverActorState {
															#line 3980 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverActorState(ResolverInterface const& resolver,InitializeResolverRequest const& initReq,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		 : resolver(resolver),
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   initReq(initReq),
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db)
															#line 3991 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
	{
		fdb_probe_actor_create("resolver", reinterpret_cast<unsigned long>(this));

	}
	~ResolverActorState() 
	{
		fdb_probe_actor_destroy("resolver", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 815 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				core = resolverCore(resolver, initReq, db);
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				;
															#line 4009 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResolverActorState();
		static_cast<ResolverActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 823 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			if (e.code() == error_code_actor_cancelled || e.code() == error_code_worker_removed)
															#line 4039 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
			{
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				TraceEvent("ResolverTerminated", resolver.id()).errorUnsuppressed(e);
															#line 825 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
				if (!static_cast<ResolverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResolverActorState(); static_cast<ResolverActor*>(this)->destroy(); return 0; }
															#line 4045 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
				new (&static_cast<ResolverActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~ResolverActorState();
				static_cast<ResolverActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 827 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 4053 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 817 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = core;
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolverActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4076 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 820 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = checkRemoved(db, initReq.recoveryCount, resolver);
															#line 4080 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<ResolverActor*>(this)->actor_wait_state = 1;
															#line 817 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResolverActor, 0, Void >*>(static_cast<ResolverActor*>(this)));
															#line 820 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResolverActor, 1, Void >*>(static_cast<ResolverActor*>(this)));
															#line 4087 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 818 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ResolverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResolverActorState(); static_cast<ResolverActor*>(this)->destroy(); return 0; }
															#line 4102 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<ResolverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResolverActorState();
		static_cast<ResolverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 818 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ResolverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResolverActorState(); static_cast<ResolverActor*>(this)->destroy(); return 0; }
															#line 4114 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		new (&static_cast<ResolverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResolverActorState();
		static_cast<ResolverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResolverActor*>(this)->actor_wait_state > 0) static_cast<ResolverActor*>(this)->actor_wait_state = 0;
		static_cast<ResolverActor*>(this)->ActorCallback< ResolverActor, 0, Void >::remove();
		static_cast<ResolverActor*>(this)->ActorCallback< ResolverActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolverActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolverActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResolverActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolverActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResolverActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResolverActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverInterface resolver;
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	InitializeResolverRequest initReq;
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 815 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	Future<Void> core;
															#line 4263 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
};
// This generated class is to be used only via resolver()
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolverActor final : public Actor<Void>, public ActorCallback< ResolverActor, 0, Void >, public ActorCallback< ResolverActor, 1, Void >, public FastAllocated<ResolverActor>, public ResolverActorState<ResolverActor> {
															#line 4268 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
public:
	using FastAllocated<ResolverActor>::operator new;
	using FastAllocated<ResolverActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9906724072783484928UL, 2817238968894130432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResolverActor, 0, Void >;
friend struct ActorCallback< ResolverActor, 1, Void >;
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverActor(ResolverInterface const& resolver,InitializeResolverRequest const& initReq,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 4286 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
		 : Actor<Void>(),
		   ResolverActorState<ResolverActor>(resolver, initReq, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18233311343312639744UL, 9120145405007874816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resolver");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResolverActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> resolver( ResolverInterface const& resolver, InitializeResolverRequest const& initReq, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ResolverActor(resolver, initReq, db));
															#line 4319 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/Resolver.actor.g.cpp"
}

#line 830 "/Users/ec2-user/foundationdb/fdbserver/Resolver.actor.cpp"
