#define POST_ACTOR_COMPILER 1
#line 1 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
/*
 * DataDistribution.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/Audit.h"
#include "fdbclient/AuditUtils.actor.h"
#include "fdbclient/BulkDumping.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/RunRYWTransaction.actor.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/BulkDumpUtil.actor.h"
#include "fdbserver/BulkLoadUtil.actor.h"
#include "fdbserver/DDSharedContext.h"
#include "fdbserver/DDTeamCollection.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/FDBExecHelper.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/QuietDatabase.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/TenantCache.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbserver/MockDataDistributor.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Platform.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/serialize.h"
#include "flow/actorcompiler.h" // This must be the last #include.

static const std::string ddServerBulkDumpFolder = "ddBulkDumpFiles";
static const std::string ddServerBulkLoadFolder = "ddBulkLoadFiles";

DataMoveType getDataMoveTypeFromDataMoveId(const UID& dataMoveId) {
	bool assigned, emptyRange;
	DataMoveType dataMoveType;
	DataMovementReason dataMoveReason;
	decodeDataMoveId(dataMoveId, assigned, emptyRange, dataMoveType, dataMoveReason);
	return dataMoveType;
}

void RelocateShard::setParentRange(KeyRange const& parent) {
	ASSERT(reason == RelocateReason::WRITE_SPLIT || reason == RelocateReason::SIZE_SPLIT);
	parent_range = parent;
}

Optional<KeyRange> RelocateShard::getParentRange() const {
	return parent_range;
}

ShardSizeBounds ShardSizeBounds::shardSizeBoundsBeforeTrack() {
	return ShardSizeBounds{ .max = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                           .iosPerKSecond = StorageMetrics::infinity,
		                                           .bytesReadPerKSecond = StorageMetrics::infinity,
		                                           .opsReadPerKSecond = StorageMetrics::infinity },
		                    .min = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = 0,
		                                           .iosPerKSecond = 0,
		                                           .bytesReadPerKSecond = 0,
		                                           .opsReadPerKSecond = 0 },
		                    .permittedError = StorageMetrics{ .bytes = -1,
		                                                      .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                                      .iosPerKSecond = StorageMetrics::infinity,
		                                                      .bytesReadPerKSecond = StorageMetrics::infinity,
		                                                      .opsReadPerKSecond = StorageMetrics::infinity } };
}

namespace {

std::set<int> const& normalDDQueueErrors() {
	static std::set<int> s{ error_code_movekeys_conflict,
		                    error_code_broken_promise,
		                    error_code_data_move_cancelled,
		                    error_code_data_move_dest_team_not_found };
	return s;
}

} // anonymous namespace

enum class DDAuditContext : uint8_t {
	INVALID = 0,
	RESUME = 1,
	LAUNCH = 2,
	RETRY = 3,
};

struct DDAudit {
	DDAudit(AuditStorageState coreState)
	  : coreState(coreState), actors(true), foundError(false), auditStorageAnyChildFailed(false), retryCount(0),
	    cancelled(false), overallCompleteDoAuditCount(0), overallIssuedDoAuditCount(0), overallSkippedDoAuditCount(0),
	    remainingBudgetForAuditTasks(SERVER_KNOBS->CONCURRENT_AUDIT_TASK_COUNT_MAX), context(DDAuditContext::INVALID) {}

	AuditStorageState coreState;
	ActorCollection actors;
	Future<Void> auditActor;
	bool foundError;
	int retryCount;
	bool auditStorageAnyChildFailed;
	bool cancelled; // use to cancel any actor beyond auditActor
	int64_t overallIssuedDoAuditCount;
	int64_t overallCompleteDoAuditCount;
	int64_t overallSkippedDoAuditCount;
	AsyncVar<int> remainingBudgetForAuditTasks;
	DDAuditContext context;
	std::unordered_set<UID> serversFinishedSSShardAudit; // dedicated to ssshard

	inline void setAuditRunActor(Future<Void> actor) { auditActor = actor; }
	inline Future<Void> getAuditRunActor() { return auditActor; }

	inline void setDDAuditContext(DDAuditContext context_) { this->context = context_; }
	inline DDAuditContext getDDAuditContext() const { return context; }

	// auditActor and actors are guaranteed to deliver a cancel signal
	void cancel() {
		auditActor.cancel();
		actors.clear(true);
		cancelled = true;
	}

	bool isCancelled() const { return cancelled; }
};

void DataMove::validateShard(const DDShardInfo& shard, KeyRangeRef range, int priority) {
	if (!valid) {
		if (shard.hasDest && shard.destId != anonymousShardId) {
			TraceEvent(SevError, "DataMoveValidationError")
			    .detail("Range", range)
			    .detail("Reason", "DataMoveMissing")
			    .detail("DestID", shard.destId)
			    .detail("ShardPrimaryDest", describe(shard.primaryDest))
			    .detail("ShardRemoteDest", describe(shard.remoteDest));
		}
		return;
	}

	ASSERT(!this->meta.ranges.empty() && this->meta.ranges.front().contains(range));

	if (!shard.hasDest) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "ShardMissingDest")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest));
		cancelled = true;
		return;
	}

	if (shard.destId != this->meta.id) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveIDMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardMoveID", shard.destId);
		cancelled = true;
		return;
	}

	if (!std::equal(
	        this->primaryDest.begin(), this->primaryDest.end(), shard.primaryDest.begin(), shard.primaryDest.end()) ||
	    !std::equal(
	        this->remoteDest.begin(), this->remoteDest.end(), shard.remoteDest.begin(), shard.remoteDest.end())) {
		TraceEvent(g_network->isSimulated() ? SevWarn : SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveDestMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest))
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		cancelled = true;
	}
}

Future<Void> StorageWiggler::onCheck() const {
	return delay(MIN_ON_CHECK_DELAY_SEC);
}

// add server to wiggling queue
void StorageWiggler::addServer(const UID& serverId, const StorageMetadataType& metadata) {
	// std::cout << "size: " << pq_handles.size() << " add " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	ASSERT(!pq_handles.contains(serverId));
	pq_handles[serverId] = wiggle_pq.emplace(metadata, serverId);
}

void StorageWiggler::removeServer(const UID& serverId) {
	// std::cout << "size: " << pq_handles.size() << " remove " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	if (contains(serverId)) { // server haven't been popped
		auto handle = pq_handles.at(serverId);
		pq_handles.erase(serverId);
		wiggle_pq.erase(handle);
	}
}

void StorageWiggler::updateMetadata(const UID& serverId, const StorageMetadataType& metadata) {
	//	std::cout << "size: " << pq_handles.size() << " update " << serverId.toString()
	//	          << " DC: " << teamCollection->isPrimary() << std::endl;
	auto handle = pq_handles.at(serverId);
	if ((*handle).first == metadata) {
		return;
	}
	wiggle_pq.update(handle, std::make_pair(metadata, serverId));
}

bool StorageWiggler::necessary(const UID& serverId, const StorageMetadataType& metadata) const {
	return metadata.wrongConfiguredForWiggle ||
	       (now() - metadata.createdTime > SERVER_KNOBS->DD_STORAGE_WIGGLE_MIN_SS_AGE_SEC);
}

Optional<UID> StorageWiggler::getNextServerId(bool necessaryOnly) {
	if (!wiggle_pq.empty()) {
		auto [metadata, id] = wiggle_pq.top();
		if (necessaryOnly && !necessary(id, metadata)) {
			return {};
		}
		wiggle_pq.pop();
		pq_handles.erase(id);
		return Optional<UID>(id);
	}
	return Optional<UID>();
}

Future<Void> StorageWiggler::resetStats() {
	metrics.reset();
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.resetStorageWiggleMetrics(tr, PrimaryRegion(teamCollection->isPrimary()), metrics);
	    });
}

Future<Void> StorageWiggler::restoreStats() {
	auto readFuture = wiggleData.storageWiggleMetrics(PrimaryRegion(teamCollection->isPrimary()))
	                      .getD(teamCollection->dbContext().getReference(), Snapshot::False, metrics);
	return store(metrics, readFuture);
}

Future<Void> StorageWiggler::startWiggle() {
	metrics.last_wiggle_start = StorageMetadataType::currentTime();
	if (shouldStartNewRound()) {
		metrics.last_round_start = metrics.last_wiggle_start;
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

Future<Void> StorageWiggler::finishWiggle() {
	metrics.last_wiggle_finish = StorageMetadataType::currentTime();
	metrics.finished_wiggle += 1;
	auto duration = metrics.last_wiggle_finish - metrics.last_wiggle_start;
	metrics.smoothed_wiggle_duration.setTotal((double)duration);

	if (shouldFinishRound()) {
		metrics.last_round_finish = metrics.last_wiggle_finish;
		metrics.finished_round += 1;
		duration = metrics.last_round_finish - metrics.last_round_start;
		metrics.smoothed_round_duration.setTotal((double)duration);
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

															#line 298 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via remoteRecovered()
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoteRecoveredActor>
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActorState {
															#line 305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActorState(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : db(db)
															#line 312 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	~RemoteRecoveredActorState() 
	{
		fdb_probe_actor_destroy("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 297 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDTrackerStarting").log();
															#line 298 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 329 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 302 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoteRecoveredActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoteRecoveredActorState(); static_cast<RemoteRecoveredActor*>(this)->destroy(); return 0; }
															#line 352 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoteRecoveredActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 298 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(db->get().recoveryState < RecoveryState::ALL_LOGS_RECRUITED))
															#line 371 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 299 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDTrackerStarting").detail("RecoveryState", (int)db->get().recoveryState);
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 381 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 1;
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoteRecoveredActor, 0, Void >*>(static_cast<RemoteRecoveredActor*>(this)));
															#line 386 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state > 0) static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 0;
		static_cast<RemoteRecoveredActor*>(this)->ActorCallback< RemoteRecoveredActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoteRecoveredActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 493 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via remoteRecovered()
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActor final : public Actor<Void>, public ActorCallback< RemoteRecoveredActor, 0, Void >, public FastAllocated<RemoteRecoveredActor>, public RemoteRecoveredActorState<RemoteRecoveredActor> {
															#line 498 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoteRecoveredActor>::operator new;
	using FastAllocated<RemoteRecoveredActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4754438293371068416UL, 12537620733448050176UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoteRecoveredActor, 0, Void >;
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActor(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 515 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoteRecoveredActorState<RemoteRecoveredActor>(db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17462346244829468672UL, 10124612030083111680UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("remoteRecovered");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoteRecoveredActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> remoteRecovered( Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoteRecoveredActor(db));
															#line 548 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 304 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Ensures that the serverKeys key space is properly coalesced
// This method is only used for testing and is not implemented in a manner that is safe for large databases
															#line 555 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via debugCheckCoalescing()
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DebugCheckCoalescingActor>
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActorState {
															#line 562 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActorState(Database const& cx) 
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : cx(cx),
															#line 308 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	~DebugCheckCoalescingActorState() 
	{
		fdb_probe_actor_destroy("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 309 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 586 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 311 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(serverListKeys, CLIENT_KNOBS->TOO_MANY);
															#line 311 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 619 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 1;
															#line 311 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 624 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 331 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 331 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 648 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 3;
															#line 331 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 2, Void >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 653 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 312 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 314 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = int();
															#line 315 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 672 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __serverList,int loopDepth) 
	{
															#line 311 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		serverList = __serverList;
															#line 681 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __serverList,int loopDepth) 
	{
		serverList = std::move(__serverList);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 328 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DoneCheckingCoalescing").log();
															#line 329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DebugCheckCoalescingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DebugCheckCoalescingActorState(); static_cast<DebugCheckCoalescingActor*>(this)->destroy(); return 0; }
															#line 762 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DebugCheckCoalescingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 315 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < serverList.size()))
															#line 781 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 316 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		id = decodeServerListValue(serverList[i].value).id();
															#line 317 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, serverKeysPrefixFor(id), allKeys);
															#line 317 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 791 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 2;
															#line 317 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 796 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult const& ranges,int loopDepth) 
	{
															#line 318 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 320 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 321 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 822 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 322 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 826 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 315 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 831 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult && ranges,int loopDepth) 
	{
															#line 318 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 320 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 321 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 844 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 322 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 848 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 315 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 853 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult const& ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(ranges, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult && ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(std::move(ranges), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 308 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 311 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult serverList;
															#line 314 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 316 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID id;
															#line 1030 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via debugCheckCoalescing()
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActor final : public Actor<Void>, public ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 2, Void >, public FastAllocated<DebugCheckCoalescingActor>, public DebugCheckCoalescingActorState<DebugCheckCoalescingActor> {
															#line 1035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DebugCheckCoalescingActor>::operator new;
	using FastAllocated<DebugCheckCoalescingActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3501084614479814400UL, 7828141602350221824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 2, Void >;
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActor(Database const& cx) 
															#line 1054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DebugCheckCoalescingActorState<DebugCheckCoalescingActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13012747114799893760UL, 10690014687488268032UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("debugCheckCoalescing");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> debugCheckCoalescing( Database const& cx ) {
															#line 307 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DebugCheckCoalescingActor(cx));
															#line 1089 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 335 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DataDistributor;
void runAuditStorage(
    Reference<DataDistributor> self,
    AuditStorageState auditStates,
    int retryCount,
    DDAuditContext context,
    Optional<std::unordered_set<UID>> serversFinishedSSShardAudit = Optional<std::unordered_set<UID>>());
															#line 1101 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorageCore( Reference<DataDistributor> const& self, UID const& auditID, AuditType const& auditType, int const& currentRetryCount );

#line 347 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1105 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<UID> launchAudit( Reference<DataDistributor> const& self, KeyRange const& auditRange, AuditType const& auditType, KeyValueStoreType const& auditStorageEngineType );

#line 351 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1109 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorage( Reference<DataDistributor> const& self, TriggerAuditRequest const& req );

#line 352 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
void loadAndDispatchAudit(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit);
															#line 1114 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorageServerShard( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 354 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1118 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditStorageShardOnServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi );

#line 357 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1122 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorage( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 358 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1126 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 361 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1130 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& auditRange );

#line 364 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 367 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1138 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditOnStorageServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi, AuditStorageRequest const& req );

#line 371 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> skipAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& rangeToSkip );

#line 374 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

void runBulkLoadTaskAsync(Reference<DataDistributor> self, KeyRange range, UID taskId, bool restart);
															#line 1148 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self );

#line 377 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DDBulkLoadJobManager {
	BulkLoadJobState jobState;
	BulkLoadTransportMethod jobTransportMethod;
	std::shared_ptr<BulkLoadManifestFileMap> manifestEntryMap;
	std::string manifestLocalTempFolder;
	bool allTaskSubmitted = false;

	DDBulkLoadJobManager() = default;
	DDBulkLoadJobManager(const BulkLoadJobState& jobState, const std::string& manifestLocalTempFolder)
	  : jobState(jobState), manifestLocalTempFolder(manifestLocalTempFolder), allTaskSubmitted(false) {
		manifestEntryMap = std::make_shared<BulkLoadManifestFileMap>();
	}

	bool isValid() const { return jobState.isValid(); }
};

struct DDBulkDumpJobManager {
	BulkDumpState jobState;
	std::map<Key, BulkLoadManifest> taskManifestMap;

	DDBulkDumpJobManager() = default;
	DDBulkDumpJobManager(const BulkDumpState& jobState) : jobState(jobState) {}

	bool isValid() const { return jobState.isValid(); }
};

struct DataDistributor : NonCopyable, ReferenceCounted<DataDistributor> {
public:
	Reference<AsyncVar<ServerDBInfo> const> dbInfo;
	Reference<DDSharedContext> context;
	UID ddId;
	PromiseStream<Future<Void>> addActor;

	// State initialized when bootstrap
	Reference<IDDTxnProcessor> txnProcessor;
	MoveKeysLock& lock; // reference to context->lock
	DatabaseConfiguration& configuration; // reference to context->configuration
	std::vector<Optional<Key>> primaryDcId;
	std::vector<Optional<Key>> remoteDcIds;
	Reference<InitialDataDistribution> initData;

	Reference<EventCacheHolder> initialDDEventHolder;
	Reference<EventCacheHolder> movingDataEventHolder;
	Reference<EventCacheHolder> totalDataInFlightEventHolder;
	Reference<EventCacheHolder> totalDataInFlightRemoteEventHolder;

	// Optional components that can be set after ::init(). They're optional when test, but required for DD being
	// fully-functional.
	DDTeamCollection* teamCollection;
	Reference<ShardsAffectedByTeamFailure> shardsAffectedByTeamFailure;
	// consumer is a yield stream from producer. The RelocateShard is pushed into relocationProducer and popped from
	// relocationConsumer (by DDQueue)
	PromiseStream<RelocateShard> relocationProducer, relocationConsumer;
	PromiseStream<BulkLoadShardRequest> triggerShardBulkLoading;
	Reference<PhysicalShardCollection> physicalShardCollection;
	Reference<BulkLoadTaskCollection> bulkLoadTaskCollection;

	Promise<Void> initialized;

	std::unordered_map<AuditType, std::unordered_map<UID, std::shared_ptr<DDAudit>>> audits;
	FlowLock auditStorageHaLaunchingLock;
	FlowLock auditStorageReplicaLaunchingLock;
	FlowLock auditStorageLocationMetadataLaunchingLock;
	FlowLock auditStorageSsShardLaunchingLock;
	Promise<Void> auditStorageInitialized;
	bool auditStorageInitStarted;

	Optional<Reference<TenantCache>> ddTenantCache;

	// monitor DD configuration change
	Promise<Version> configChangeWatching;
	Future<Void> onConfigChange;

	ActorCollection bulkLoadActors;
	bool bulkLoadEnabled = false;
	ParallelismLimitor bulkLoadParallelismLimitor;
	ParallelismLimitor bulkLoadEngineParallelismLimitor;
	std::string bulkLoadFolder;

	Optional<DDBulkLoadJobManager> bulkLoadJobManager;

	bool bulkDumpEnabled = false;
	ParallelismLimitor bulkDumpParallelismLimitor;
	std::string folder;
	std::string bulkDumpFolder;

	DDBulkDumpJobManager bulkDumpJobManager;

	DataDistributor(Reference<AsyncVar<ServerDBInfo> const> const& db,
	                UID id,
	                Reference<DDSharedContext> context,
	                std::string folder)
	  : dbInfo(db), context(context), ddId(id), txnProcessor(nullptr), lock(context->lock),
	    configuration(context->configuration), initialDDEventHolder(makeReference<EventCacheHolder>("InitialDD")),
	    movingDataEventHolder(makeReference<EventCacheHolder>("MovingData")),
	    totalDataInFlightEventHolder(makeReference<EventCacheHolder>("TotalDataInFlight")),
	    totalDataInFlightRemoteEventHolder(makeReference<EventCacheHolder>("TotalDataInFlightRemote")),
	    teamCollection(nullptr), bulkLoadTaskCollection(nullptr), auditStorageHaLaunchingLock(1),
	    auditStorageReplicaLaunchingLock(1), auditStorageLocationMetadataLaunchingLock(1),
	    auditStorageSsShardLaunchingLock(1), auditStorageInitStarted(false), bulkLoadActors(false),
	    bulkLoadEnabled(false), bulkLoadParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM),
	    bulkLoadEngineParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM), bulkDumpEnabled(false),
	    bulkDumpParallelismLimitor(SERVER_KNOBS->DD_BULKDUMP_PARALLELISM), folder(folder) {
		if (!folder.empty()) {
			bulkDumpFolder = abspath(joinPath(folder, ddServerBulkDumpFolder));
			// TODO(BulkDump): clear this folder in the presence of crash
			bulkLoadFolder = abspath(joinPath(folder, ddServerBulkLoadFolder));
			// TODO(BulkLoad): clear this folder in the presence of crash
		}
	}

	// bootstrap steps

	Future<Void> takeMoveKeysLock() { return store(lock, txnProcessor->takeMoveKeysLock(ddId)); }

	Future<Void> loadDatabaseConfiguration() { return store(configuration, txnProcessor->getDatabaseConfiguration()); }

	Future<Void> updateReplicaKeys() {
		return txnProcessor->updateReplicaKeys(primaryDcId, remoteDcIds, configuration);
	}

	Future<Void> loadInitialDataDistribution() {
		return store(initData,
		             txnProcessor->getInitialDataDistribution(
		                 ddId,
		                 lock,
		                 configuration.usableRegions > 1 ? remoteDcIds : std::vector<Optional<Key>>(),
		                 context->ddEnabledState.get(),
		                 SkipDDModeCheck::False));
	}

	void initDcInfo() {
		primaryDcId.clear();
		remoteDcIds.clear();
		const std::vector<RegionInfo>& regions = configuration.regions;
		if (configuration.regions.size() > 0) {
			primaryDcId.push_back(regions[0].dcId);
		}
		if (configuration.regions.size() > 1) {
			remoteDcIds.push_back(regions[1].dcId);
		}
	}

	Future<Void> waitDataDistributorEnabled() const {
		return txnProcessor->waitForDataDistributionEnabled(context->ddEnabledState.get());
	}

	// Resume in-memory audit instances and issue background audit metadata cleanup
	void resumeAuditStorage(Reference<DataDistributor> self, std::vector<AuditStorageState> auditStates) {
		for (const auto& auditState : auditStates) {
			if (auditState.getPhase() != AuditPhase::Running) {
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "WrongAuditStateToResume")
				    .detail("AuditState", auditState.toString());
				return;
			}
			if (self->audits.contains(auditState.getType()) &&
			    self->audits[auditState.getType()].contains(auditState.id)) {
				// Ignore any RUNNING auditState with an alive audit
				// instance in DD audits map
				continue;
			}
			runAuditStorage(self, auditState, 0, DDAuditContext::RESUME);
			TraceEvent(SevInfo, "AuditStorageResumed", self->ddId)
			    .detail("AuditID", auditState.id)
			    .detail("AuditType", auditState.getType())
			    .detail("AuditState", auditState.toString());
		}
		return;
	}

																#line 1323 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via initAuditStorage()
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitAuditStorageActor>
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActorState {
															#line 1329 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActorState(Reference<DataDistributor> const& self) 
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 1336 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	~InitAuditStorageActorState() 
	{
		fdb_probe_actor_destroy("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 549 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->auditStorageInitStarted = true;
															#line 550 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			MoveKeyLockInfo lockInfo;
															#line 551 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.myOwner = self->lock.myOwner;
															#line 552 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevOwner = self->lock.prevOwner;
															#line 553 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevWrite = self->lock.prevWrite;
															#line 554 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<std::vector<AuditStorageState>> __when_expr_0 = initAuditMetadata(self->txnProcessor->context(), lockInfo, self->context->isDDEnabled(), self->ddId, SERVER_KNOBS->PERSIST_FINISH_AUDIT_COUNT);
															#line 554 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 1;
															#line 554 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*>(static_cast<InitAuditStorageActor*>(this)));
															#line 1368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
															#line 560 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 561 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 562 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1395 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
															#line 560 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 561 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 562 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1411 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(auditStatesToResume, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(auditStatesToResume), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state > 0) static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 0;
		static_cast<InitAuditStorageActor*>(this)->ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >::remove();

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> const& value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> && value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,Error err) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via initAuditStorage()
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActor final : public Actor<Void>, public ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >, public FastAllocated<InitAuditStorageActor>, public InitAuditStorageActorState<InitAuditStorageActor> {
															#line 1501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitAuditStorageActor>::operator new;
	using FastAllocated<InitAuditStorageActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13861049474155709952UL, 7847454862576393984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >;
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActor(Reference<DataDistributor> const& self) 
															#line 1518 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitAuditStorageActorState<InitAuditStorageActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8855224104647952640UL, 5194562846350463232UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("initAuditStorage");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> initAuditStorage( Reference<DataDistributor> const& self ) {
															#line 548 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitAuditStorageActor(self));
															#line 1550 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 564 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 1555 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class WaitUntilDataDistributorExitSecurityModeActor>
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActorState {
															#line 1561 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActorState(Reference<DataDistributor> const& self) 
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 566 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(self->txnProcessor->context())
															#line 1570 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	~WaitUntilDataDistributorExitSecurityModeActorState() 
	{
		fdb_probe_actor_destroy("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 567 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 1585 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitUntilDataDistributorExitSecurityModeActorState();
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 568 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->DD_ENABLED_CHECK_DELAY, TaskPriority::DataDistribution);
															#line 568 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1617 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 1;
															#line 568 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 569 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 570 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 571 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1635 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 573 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 573 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1641 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 573 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1646 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 569 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 570 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 571 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1665 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 573 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 573 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1671 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 573 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1676 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1775 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 4;
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1780 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> const& mode,int loopDepth) 
	{
															#line 574 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1795 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 575 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1799 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 577 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 578 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 580 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1813 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 581 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1817 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 583 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 583 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1827 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 583 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1832 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> && mode,int loopDepth) 
	{
															#line 574 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1841 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 575 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1845 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 577 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 578 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 580 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1859 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 581 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1863 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 583 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 583 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1873 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 583 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1878 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> const& mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(mode, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> && mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(mode), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 584 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1962 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 584 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1971 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 566 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 2155 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActor final : public Actor<Void>, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >, public FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>, public WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor> {
															#line 2160 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator new;
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8389439829996272384UL, 9703104253709778432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >;
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActor(Reference<DataDistributor> const& self) 
															#line 2180 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4834519635010961920UL, 15563594834600371456UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitUntilDataDistributorExitSecurityMode");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> waitUntilDataDistributorExitSecurityMode( Reference<DataDistributor> const& self ) {
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new WaitUntilDataDistributorExitSecurityModeActor(self));
															#line 2215 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 590 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Initialize the required internal states of DataDistributor from system metadata. It's necessary before
	// DataDistributor start working. Doesn't include initialization of optional components, like TenantCache, DDQueue,
	// Tracker, TeamCollection. The components should call its own ::init methods.
																#line 2223 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via init()
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitActor>
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActorState {
															#line 2229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActorState(Reference<DataDistributor> const& self) 
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 2236 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("init", reinterpret_cast<unsigned long>(this));

	}
	~InitActorState() 
	{
		fdb_probe_actor_destroy("init", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 595 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 2251 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitActorState();
		static_cast<InitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 707 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitActorState(); static_cast<InitActor*>(this)->destroy(); return 0; }
															#line 2274 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitActorState();
		static_cast<InitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->waitDataDistributorEnabled();
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2295 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 1;
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitActor, 0, Void >*>(static_cast<InitActor*>(this)));
															#line 2300 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 597 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 599 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2328 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2333 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 597 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 599 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2353 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 601 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 605 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2439 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 607 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2443 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 609 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 616 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2452 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 618 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 618 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2458 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 618 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2463 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 601 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 605 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2479 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 607 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2483 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 609 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 616 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2492 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 618 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 618 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2498 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 618 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2503 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< InitActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 626 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributorExitSecurityMode").log();
															#line 628 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->loadDatabaseConfiguration();
															#line 628 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2596 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 4;
															#line 628 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< InitActor, 3, Void >*>(static_cast<InitActor*>(this)));
															#line 2601 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< InitActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 629 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 630 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 634 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2701 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 636 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 638 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2707 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2718 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 629 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 630 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 634 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2731 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 636 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 638 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2743 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2748 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< InitActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
															#line 642 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2841 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
															#line 642 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2854 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2859 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< InitActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 646 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 2943 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2947 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 654 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 662 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 2957 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 664 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 2961 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 668 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 672 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 694 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 700 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 2972 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 646 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 2981 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2985 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 654 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2991 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 662 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 2995 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 664 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 2999 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 668 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 672 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 694 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 700 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 3010 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< InitActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 3092 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via init()
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActor final : public Actor<Void>, public ActorCallback< InitActor, 0, Void >, public ActorCallback< InitActor, 1, Void >, public ActorCallback< InitActor, 2, Void >, public ActorCallback< InitActor, 3, Void >, public ActorCallback< InitActor, 4, Void >, public ActorCallback< InitActor, 5, Void >, public FastAllocated<InitActor>, public InitActorState<InitActor> {
															#line 3097 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitActor>::operator new;
	using FastAllocated<InitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14946979269166259200UL, 4609695802952166400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitActor, 0, Void >;
friend struct ActorCallback< InitActor, 1, Void >;
friend struct ActorCallback< InitActor, 2, Void >;
friend struct ActorCallback< InitActor, 3, Void >;
friend struct ActorCallback< InitActor, 4, Void >;
friend struct ActorCallback< InitActor, 5, Void >;
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActor(Reference<DataDistributor> const& self) 
															#line 3119 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitActorState<InitActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1770802396446596864UL, 4776746127134105600UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("init");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< InitActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< InitActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< InitActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< InitActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< InitActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> init( Reference<DataDistributor> const& self ) {
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitActor(self));
															#line 3156 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 709 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3161 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoveDataMoveTombstoneBackgroundActor>
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActorState {
															#line 3167 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActorState(Reference<DataDistributor> const& self) 
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 711 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentID()
															#line 3176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	~RemoveDataMoveTombstoneBackgroundActorState() 
	{
		fdb_probe_actor_destroy("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 713 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				cx = openDBOnServer(self->dbInfo, TaskPriority::DefaultEndpoint, LockAware::True);
															#line 714 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr = Transaction(cx);
															#line 715 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				;
															#line 3196 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 740 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoveDataMoveTombstoneBackgroundActorState(); static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->destroy(); return 0; }
															#line 3225 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 731 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 3238 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 732 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3242 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 734 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "RemoveDataMoveTombstoneError", self->ddId) .errorUnsuppressed(e) .detail("CurrentDataMoveID", currentID);
															#line 3246 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 717 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 718 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 719 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			for( UID& dataMoveID : self->initData->toCleanDataMoveTombstone ) {
															#line 720 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				currentID = dataMoveID;
															#line 721 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr.clear(dataMoveKeyFor(currentID));
															#line 722 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "RemoveDataMoveTombstone", self->ddId).detail("DataMoveID", currentID);
															#line 3285 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 724 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr.commit();
															#line 724 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3291 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 1;
															#line 724 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3296 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 727 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 727 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3333 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 2;
															#line 727 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3338 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 711 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID currentID;
															#line 713 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 714 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 3544 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActor final : public Actor<Void>, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >, public FastAllocated<RemoveDataMoveTombstoneBackgroundActor>, public RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor> {
															#line 3549 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator new;
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7907695875525825792UL, 10374567205397853440UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >;
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >;
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActor(Reference<DataDistributor> const& self) 
															#line 3567 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929663207261195520UL, 2170269584154920960UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("removeDataMoveTombstoneBackground");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> removeDataMoveTombstoneBackground( Reference<DataDistributor> const& self ) {
															#line 710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoveDataMoveTombstoneBackgroundActor(self));
															#line 3600 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 742 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3605 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromShards()
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromShardsActor>
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActorState {
															#line 3611 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActorState(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   traceShard(traceShard)
															#line 3620 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromShardsActorState() 
	{
		fdb_probe_actor_destroy("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 745 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 3635 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 746 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for(int i = 0;i < self->initData->shards.size() - 1;i++) {
															#line 747 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					const DDShardInfo& iShard = self->initData->shards[i];
															#line 748 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					KeyRangeRef keys = KeyRangeRef(iShard.key, self->initData->shards[i + 1].key);
															#line 749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 750 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					teams.emplace_back(iShard.primarySrc, true);
															#line 751 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (self->configuration.usableRegions > 1)
															#line 3649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 752 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.emplace_back(iShard.remoteSrc, false);
															#line 3653 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					}
															#line 754 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->physicalShardCollection->initPhysicalShardCollection(keys, teams, iShard.srcId.first(), 0);
															#line 3657 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 758 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundaries = std::vector<Key>();
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadIsEnabled(self->initData->bulkLoadMode))
															#line 3664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 761 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevInfo, "DDInitCustomRangeConfigDisabledByBulkLoadMode", self->ddId);
															#line 3668 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 763 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for( auto it : self->initData->userRangeConfig->ranges() ) {
															#line 764 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					auto range = it->range();
															#line 765 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					customBoundaries.push_back(range.begin);
															#line 766 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDInitCustomRangeConfig", self->ddId) .detail("Range", KeyRangeRef(range.begin, range.end)) .detail("Config", it->value());
															#line 3680 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 772 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			shard = 0;
															#line 773 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary = 0;
															#line 774 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			overreplicatedCount = 0;
															#line 775 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 3691 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 849 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromShardsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromShardsActorState(); static_cast<ResumeFromShardsActor*>(this)->destroy(); return 0; }
															#line 3714 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromShardsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 775 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(shard < self->initData->shards.size() - 1))
															#line 3733 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DDShardInfo& iShard = self->initData->shards[shard];
															#line 777 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<KeyRangeRef> ranges;
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key beginKey = iShard.key;
															#line 780 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key endKey = self->initData->shards[shard + 1].key;
															#line 781 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] <= beginKey;) {
															#line 782 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3749 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 784 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] < endKey;) {
															#line 785 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ranges.push_back(KeyRangeRef(beginKey, customBoundaries[customBoundary]));
															#line 786 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = customBoundaries[customBoundary];
															#line 787 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3759 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 789 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ranges.push_back(KeyRangeRef(beginKey, endKey));
															#line 791 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 792 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.primarySrc, true));
															#line 793 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.usableRegions > 1)
															#line 3769 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 794 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.remoteSrc, false));
															#line 3773 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 797 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int r = 0;r < ranges.size();r++) {
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto& keys = ranges[r];
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->defineShard(keys);
															#line 801 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->initData->userRangeConfig->rangeContaining(keys.begin);
															#line 802 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			int customReplicas = std::max(self->configuration.storageTeamSize, it->value().replicationFactor.orDefault(0));
															#line 804 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT_WE_THINK(KeyRangeRef(it->range().begin, it->range().end).contains(keys));
															#line 806 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bool unhealthy = iShard.primarySrc.size() != customReplicas;
															#line 807 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && self->configuration.usableRegions > 1)
															#line 3791 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 808 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				unhealthy = iShard.remoteSrc.size() != customReplicas;
															#line 3795 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 810 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && iShard.primarySrc.size() > self->configuration.storageTeamSize)
															#line 3799 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (++overreplicatedCount > SERVER_KNOBS->DD_MAX_SHARDS_ON_LARGE_TEAMS)
															#line 3803 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 812 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					unhealthy = true;
															#line 3807 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (traceShard)
															#line 3812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 817 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDInitShard", self->ddId) .detail("Keys", keys) .detail("PrimarySrc", describe(iShard.primarySrc)) .detail("RemoteSrc", describe(iShard.remoteSrc)) .detail("PrimaryDest", describe(iShard.primaryDest)) .detail("RemoteDest", describe(iShard.remoteDest)) .detail("SrcID", iShard.srcId) .detail("DestID", iShard.destId) .detail("CustomReplicas", customReplicas) .detail("StorageTeamSize", self->configuration.storageTeamSize) .detail("Unhealthy", unhealthy) .detail("Overreplicated", overreplicatedCount);
															#line 3816 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 831 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->moveShard(keys, teams);
															#line 832 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if ((ddLargeTeamEnabled() && (unhealthy || r > 0)) || (iShard.hasDest && iShard.destId == anonymousShardId))
															#line 3822 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 837 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				DataMovementReason reason = DataMovementReason::RECOVER_MOVE;
															#line 838 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (unhealthy)
															#line 3828 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					reason = DataMovementReason::TEAM_UNHEALTHY;
															#line 3832 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 840 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (r > 0)
															#line 3838 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						reason = DataMovementReason::SPLIT_SHARD;
															#line 3842 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					}
				}
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(RelocateShard(keys, reason, RelocateReason::OTHER));
															#line 3847 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 847 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = yield(TaskPriority::DataDistribution);
															#line 847 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3854 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 1;
															#line 847 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromShardsActor, 0, Void >*>(static_cast<ResumeFromShardsActor*>(this)));
															#line 3859 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 775 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3881 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 775 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3890 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromShardsActor*>(this)->ActorCallback< ResumeFromShardsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromShardsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool traceShard;
															#line 758 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Key> customBoundaries;
															#line 772 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int shard;
															#line 773 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int customBoundary;
															#line 774 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int overreplicatedCount;
															#line 3982 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromShards()
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActor final : public Actor<Void>, public ActorCallback< ResumeFromShardsActor, 0, Void >, public FastAllocated<ResumeFromShardsActor>, public ResumeFromShardsActorState<ResumeFromShardsActor> {
															#line 3987 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromShardsActor>::operator new;
	using FastAllocated<ResumeFromShardsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14588957832858729984UL, 8729021681823816192UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromShardsActor, 0, Void >;
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActor(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 4004 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromShardsActorState<ResumeFromShardsActor>(self, traceShard),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17921411212936075776UL, 15009917662088599808UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromShards");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromShardsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromShards( Reference<DataDistributor> const& self, bool const& traceShard ) {
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromShardsActor(self, traceShard));
															#line 4036 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 851 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// TODO: unit test needed
																#line 4042 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromDataMoves()
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromDataMovesActor>
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActorState {
															#line 4048 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart),
															#line 854 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   it(self->initData->dataMoveMap.ranges().begin())
															#line 4059 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromDataMovesActorState() 
	{
		fdb_probe_actor_destroy("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 856 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 856 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4076 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 1;
															#line 856 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 0, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4081 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 858 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4104 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 858 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4113 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 920 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 4197 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 921 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->addActor.send(self->removeDataMoveTombstoneBackground(self));
															#line 4201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 923 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromDataMovesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromDataMovesActorState(); static_cast<ResumeFromDataMovesActor*>(this)->destroy(); return 0; }
															#line 4205 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromDataMovesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 858 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(it != self->initData->dataMoveMap.ranges().end()))
															#line 4224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 859 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DataMoveMetaData& meta = it.value()->meta;
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		DataMoveType dataMoveType = getDataMoveTypeFromDataMoveId(meta.id);
															#line 861 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.ranges.empty())
															#line 4234 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 862 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "EmptyDataMoveRange", self->ddId).detail("DataMoveMetaData", meta.toString());
															#line 4238 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1cont1continue1(loopDepth); // continue
		}
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.bulkLoadTaskState.present())
															#line 4243 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 866 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.dataMoveId = meta.id;
															#line 868 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.cancelled = true;
															#line 869 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->relocationProducer.send(rs);
															#line 872 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMove", self->ddId) .detail("Reason", "DDInit") .detail("DataMove", meta.toString());
															#line 4255 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		else
		{
															#line 875 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (dataMoveType == DataMoveType::LOGICAL_BULKLOAD || dataMoveType == DataMoveType::PHYSICAL_BULKLOAD)
															#line 4262 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 878 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 879 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.dataMoveId = meta.id;
															#line 880 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.cancelled = true;
															#line 881 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(rs);
															#line 882 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMoveForWrongType", self->ddId) .detail("Reason", "WrongTypeWhenDDInit") .detail("DataMoveType", dataMoveType) .detail("DataMove", meta.toString());
															#line 4274 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1cont1loopBody1cont5(loopDepth);
			}
			else
			{
															#line 886 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (it.value()->isCancelled() || (it.value()->valid && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA))
															#line 4281 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 889 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.dataMoveId = meta.id;
															#line 890 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.cancelled = true;
															#line 891 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->relocationProducer.send(rs);
															#line 892 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent("DDInitScheduledCancelDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 4293 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = a_body1cont1loopBody1cont7(loopDepth);
				}
				else
				{
															#line 893 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (it.value()->valid)
															#line 4300 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 894 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						TraceEvent(SevDebug, "DDInitFoundDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 895 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						ASSERT(meta.ranges.front() == it.range());
															#line 897 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 898 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMoveId = meta.id;
															#line 899 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMove = it.value();
															#line 900 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 901 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->primaryDest, true));
															#line 902 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (!rs.dataMove->remoteDest.empty())
															#line 4318 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 903 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
							teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->remoteDest, false));
															#line 4322 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
						}
															#line 908 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->restartShardTracker.send(rs.keys);
															#line 909 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->defineShard(rs.keys);
															#line 913 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->moveShard(rs.keys, teams);
															#line 914 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->relocationProducer.send(rs);
															#line 915 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						StrictFuture<Void> __when_expr_1 = yield(TaskPriority::DataDistribution);
															#line 915 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4336 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
						if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
						static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 2;
															#line 915 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 1, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4341 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1cont1loopBody1cont9(loopDepth);
					}
				}
			}
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1continue1(int loopDepth) 
	{
															#line 858 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4371 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 858 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4380 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 854 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRangeMap<std::shared_ptr<DataMove>>::iterator it;
															#line 4496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromDataMoves()
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActor final : public Actor<Void>, public ActorCallback< ResumeFromDataMovesActor, 0, Void >, public ActorCallback< ResumeFromDataMovesActor, 1, Void >, public FastAllocated<ResumeFromDataMovesActor>, public ResumeFromDataMovesActorState<ResumeFromDataMovesActor> {
															#line 4501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromDataMovesActor>::operator new;
	using FastAllocated<ResumeFromDataMovesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14811559024679276800UL, 6625740153063138816UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromDataMovesActor, 0, Void >;
friend struct ActorCallback< ResumeFromDataMovesActor, 1, Void >;
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 4519 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromDataMovesActorState<ResumeFromDataMovesActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929780632836526848UL, 14642014520803479552UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromDataMoves");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromDataMoves( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromDataMovesActor(self, readyToStart));
															#line 4552 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 925 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Resume inflight relocations from the previous DD
	// TODO: The initialDataDistribution is unused once resumeRelocations,
	// DataDistributionTracker::trackInitialShards, and DDTeamCollection::init are done. In the future, we can release
	// the object to save memory usage if it turns out to be a problem.
	Future<Void> resumeRelocations() {
		ASSERT(shardsAffectedByTeamFailure); // has to be allocated
		Future<Void> shardsReady = resumeFromShards(Reference<DataDistributor>::addRef(this), g_network->isSimulated());
		return resumeFromDataMoves(Reference<DataDistributor>::addRef(this), shardsReady);
	}

	Future<Void> pollMoveKeysLock() const {
		return txnProcessor->pollMoveKeysLock(lock, context->ddEnabledState.get());
	}

	Future<bool> isDataDistributionEnabled() const {
		return txnProcessor->isDataDistributionEnabled(context->ddEnabledState.get());
	}

	Future<Void> removeKeysFromFailedServer(const UID& serverID, const std::vector<UID>& teamForDroppedRange) const {
		return txnProcessor->removeKeysFromFailedServer(
		    serverID, teamForDroppedRange, lock, context->ddEnabledState.get());
	}

	Future<Void> removeStorageServer(const UID& serverID, const Optional<UID>& tssPairID = Optional<UID>()) const {
		return txnProcessor->removeStorageServer(serverID, tssPairID, lock, context->ddEnabledState.get());
	}

	Future<Void> initDDConfigWatch();

	Future<Void> initTenantCache();
};

Future<Void> DataDistributor::initDDConfigWatch() {
	if (txnProcessor->isMocked()) {
		onConfigChange = Never();
		return Void();
	}
	onConfigChange = map(DDConfiguration().trigger.onChange(
	                         SystemDBWriteLockedNow(txnProcessor->context().getReference()), {}, configChangeWatching),
	                     [](Version v) {
		                     CODE_PROBE(true, "DataDistribution change detected");
		                     TraceEvent("DataDistributionConfigChanged").detail("ChangeVersion", v);
		                     throw dd_config_changed();
		                     return Void();
	                     });

	return success(configChangeWatching.getFuture());
}

Future<Void> DataDistributor::initTenantCache() {
	// SOMEDAY: support tenant cache in MockDD
	ASSERT(!txnProcessor->isMocked());
	ddTenantCache = makeReference<TenantCache>(txnProcessor->context(), ddId);
	return ddTenantCache.get()->build();
}

inline void addAuditToAuditMap(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit) {
	AuditType auditType = audit->coreState.getType();
	UID auditID = audit->coreState.id;
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "addAuditToAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(!self->audits[auditType].contains(auditID));
	self->audits[auditType][auditID] = audit;
	return;
}

inline std::shared_ptr<DDAudit> getAuditFromAuditMap(Reference<DataDistributor> self,
                                                     AuditType auditType,
                                                     UID auditID) {
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "getAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	return self->audits[auditType][auditID];
}

inline void removeAuditFromAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	std::shared_ptr<DDAudit> audit = self->audits[auditType][auditID];
	audit->cancel();
	self->audits[auditType].erase(auditID);
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "removeAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	return;
}

inline bool auditExistInAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	return self->audits.contains(auditType) && self->audits[auditType].contains(auditID);
}

inline bool existAuditInAuditMapForType(Reference<DataDistributor> self, AuditType auditType) {
	return self->audits.contains(auditType) && !self->audits[auditType].empty();
}

inline std::unordered_map<UID, std::shared_ptr<DDAudit>> getAuditsForType(Reference<DataDistributor> self,
                                                                          AuditType auditType) {
	ASSERT(self->audits.contains(auditType));
	return self->audits[auditType];
}

// Periodically check and log the physicalShard status; clean up empty physicalShard;
															#line 4663 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MonitorPhysicalShardStatusActor>
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActorState {
															#line 4670 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActorState(Reference<PhysicalShardCollection> const& self) 
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 4677 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	~MonitorPhysicalShardStatusActorState() 
	{
		fdb_probe_actor_destroy("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1033 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA);
															#line 1034 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD);
															#line 1035 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 4696 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorPhysicalShardStatusActorState();
		static_cast<MonitorPhysicalShardStatusActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1036 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->cleanUpPhysicalShardCollection();
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->logPhysicalShardCollection();
															#line 1038 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->PHYSICAL_SHARD_METRICS_DELAY);
															#line 1038 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4732 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 1;
															#line 1038 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*>(static_cast<MonitorPhysicalShardStatusActor*>(this)));
															#line 4737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state > 0) static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorPhysicalShardStatusActor*>(this)->ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<PhysicalShardCollection> self;
															#line 4831 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActor final : public Actor<Void>, public ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >, public FastAllocated<MonitorPhysicalShardStatusActor>, public MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor> {
															#line 4836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator new;
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4749816246071341312UL, 13034380876347677952UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >;
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActor(Reference<PhysicalShardCollection> const& self) 
															#line 4853 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8017156993063803648UL, 8326163650824365056UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorPhysicalShardStatus");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> monitorPhysicalShardStatus( Reference<PhysicalShardCollection> const& self ) {
															#line 1032 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MonitorPhysicalShardStatusActor(self));
															#line 4886 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1041 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// This actor must be a singleton
															#line 4892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via prepareDataMigration()
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class PrepareDataMigrationActor>
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PrepareDataMigrationActorState {
															#line 4899 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareDataMigrationActorState(PrepareBlobRestoreRequest const& req,Reference<DDSharedContext> const& context,Database const& cx) 
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : req(req),
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   context(context),
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(cx)
															#line 4910 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("prepareDataMigration", reinterpret_cast<unsigned long>(this));

	}
	~PrepareDataMigrationActorState() 
	{
		fdb_probe_actor_destroy("prepareDataMigration", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1048 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<std::pair<Version, Tag>> __when_expr_0 = addStorageServer(cx, req.ssi);
															#line 1048 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4928 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 1;
															#line 1048 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 4933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PrepareDataMigrationActorState();
		static_cast<PrepareDataMigrationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1064 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(context->ddEnabledState->trySetEnabled(req.requesterID));
															#line 1065 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<PrepareDataMigrationActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PrepareDataMigrationActorState(); static_cast<PrepareDataMigrationActor*>(this)->destroy(); return 0; }
															#line 4964 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<PrepareDataMigrationActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PrepareDataMigrationActorState();
		static_cast<PrepareDataMigrationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1059 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 4977 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1060 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4981 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1061 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(e);
															#line 4985 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(std::pair<Version, Tag> const& verAndTag,int loopDepth) 
	{
															#line 1049 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevDebug, "BlobRestorePrepare", context->id()) .detail("State", "BMAdded") .detail("ReqId", req.requesterID) .detail("Version", verAndTag.first) .detail("Tag", verAndTag.second);
															#line 1055 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = prepareBlobRestore( cx, context->lock, context->ddEnabledState.get(), context->id(), req.keys, req.ssi.id(), req.requesterID);
															#line 1055 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5004 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 2;
															#line 1055 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 1, Void >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 5009 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(std::pair<Version, Tag> && verAndTag,int loopDepth) 
	{
															#line 1049 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevDebug, "BlobRestorePrepare", context->id()) .detail("State", "BMAdded") .detail("ReqId", req.requesterID) .detail("Version", verAndTag.first) .detail("Tag", verAndTag.second);
															#line 1055 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = prepareBlobRestore( cx, context->lock, context->ddEnabledState.get(), context->id(), req.keys, req.ssi.id(), req.requesterID);
															#line 1055 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5022 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 2;
															#line 1055 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PrepareDataMigrationActor, 1, Void >*>(static_cast<PrepareDataMigrationActor*>(this)));
															#line 5027 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(std::pair<Version, Tag> const& verAndTag,int loopDepth) 
	{
		loopDepth = a_body1cont2(verAndTag, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<Version, Tag> && verAndTag,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(verAndTag), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state > 0) static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 0;
		static_cast<PrepareDataMigrationActor*>(this)->ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >::remove();

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,std::pair<Version, Tag> const& value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,std::pair<Version, Tag> && value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*,Error err) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1057 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		req.reply.send(PrepareBlobRestoreReply(PrepareBlobRestoreReply::SUCCESS));
															#line 5111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1057 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		req.reply.send(PrepareBlobRestoreReply(PrepareBlobRestoreReply::SUCCESS));
															#line 5120 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state > 0) static_cast<PrepareDataMigrationActor*>(this)->actor_wait_state = 0;
		static_cast<PrepareDataMigrationActor*>(this)->ActorCallback< PrepareDataMigrationActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PrepareDataMigrationActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareBlobRestoreRequest req;
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DDSharedContext> context;
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 5219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via prepareDataMigration()
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PrepareDataMigrationActor final : public Actor<Void>, public ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >, public ActorCallback< PrepareDataMigrationActor, 1, Void >, public FastAllocated<PrepareDataMigrationActor>, public PrepareDataMigrationActorState<PrepareDataMigrationActor> {
															#line 5224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<PrepareDataMigrationActor>::operator new;
	using FastAllocated<PrepareDataMigrationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5034784314895188992UL, 2745360904154581760UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >;
friend struct ActorCallback< PrepareDataMigrationActor, 1, Void >;
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PrepareDataMigrationActor(PrepareBlobRestoreRequest const& req,Reference<DDSharedContext> const& context,Database const& cx) 
															#line 5242 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   PrepareDataMigrationActorState<PrepareDataMigrationActor>(req, context, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("prepareDataMigration", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12945933996581036800UL, 2706979800331646720UL);
		ActorExecutionContextHelper __helper(static_cast<PrepareDataMigrationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("prepareDataMigration");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("prepareDataMigration", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PrepareDataMigrationActor, 0, std::pair<Version, Tag> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PrepareDataMigrationActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> prepareDataMigration( PrepareBlobRestoreRequest const& req, Reference<DDSharedContext> const& context, Database const& cx ) {
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new PrepareDataMigrationActor(req, context, cx));
															#line 5276 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1067 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 5281 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via serveBlobMigratorRequests()
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ServeBlobMigratorRequestsActor>
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ServeBlobMigratorRequestsActorState {
															#line 5288 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ServeBlobMigratorRequestsActorState(Reference<DataDistributor> const& self,Reference<DataDistributionTracker> const& tracker,Reference<DDQueue> const& queue) 
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tracker(tracker),
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   queue(queue)
															#line 5299 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this));

	}
	~ServeBlobMigratorRequestsActorState() 
	{
		fdb_probe_actor_destroy("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->initialized.getFuture();
															#line 1071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5316 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 1;
															#line 1071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*>(static_cast<ServeBlobMigratorRequestsActor*>(this)));
															#line 5321 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ServeBlobMigratorRequestsActorState();
		static_cast<ServeBlobMigratorRequestsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1072 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 5344 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1072 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 5353 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state > 0) static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 0;
		static_cast<ServeBlobMigratorRequestsActor*>(this)->ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1073 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		FutureStream<PrepareBlobRestoreRequest> __when_expr_1 = self->context->interface.prepareBlobRestoreReq.getFuture();
															#line 1073 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5446 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.pop(), loopDepth); };
		static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 2;
															#line 1073 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*>(static_cast<ServeBlobMigratorRequestsActor*>(this)));
															#line 5451 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(PrepareBlobRestoreRequest const& req,int loopDepth) 
	{
															#line 1074 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (BlobMigratorInterface::isBlobMigrator(req.ssi.id()))
															#line 5460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1075 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->sameId(req.requesterID) && self->context->ddEnabledState->isBlobRestorePreparing())
															#line 5464 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1078 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				CODE_PROBE(true, "Receive repeated PrepareBlobRestoreRequest");
															#line 5468 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
															#line 1081 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->trySetBlobRestorePreparing(req.requesterID))
															#line 5473 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1083 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(prepareDataMigration(req, self->context, self->txnProcessor->context()));
															#line 1085 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(dd_config_changed(), std::max(0, loopDepth - 1));
															#line 5479 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 1087 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				auto reason = self->context->ddEnabledState->isBlobRestorePreparing() ? PrepareBlobRestoreReply::CONFLICT_BLOB_RESTORE : PrepareBlobRestoreReply::CONFLICT_SNAPSHOT;
															#line 1090 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				req.reply.send(PrepareBlobRestoreReply(reason));
															#line 5487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
		}
		else
		{
															#line 1094 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(operation_failed());
															#line 5495 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(PrepareBlobRestoreRequest && req,int loopDepth) 
	{
															#line 1074 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (BlobMigratorInterface::isBlobMigrator(req.ssi.id()))
															#line 5505 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1075 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->sameId(req.requesterID) && self->context->ddEnabledState->isBlobRestorePreparing())
															#line 5509 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1078 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				CODE_PROBE(true, "Receive repeated PrepareBlobRestoreRequest");
															#line 5513 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
															#line 1081 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (self->context->ddEnabledState->trySetBlobRestorePreparing(req.requesterID))
															#line 5518 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1083 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(prepareDataMigration(req, self->context, self->txnProcessor->context()));
															#line 1085 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(dd_config_changed(), std::max(0, loopDepth - 1));
															#line 5524 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 1087 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				auto reason = self->context->ddEnabledState->isBlobRestorePreparing() ? PrepareBlobRestoreReply::CONFLICT_BLOB_RESTORE : PrepareBlobRestoreReply::CONFLICT_SNAPSHOT;
															#line 1090 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				req.reply.send(PrepareBlobRestoreReply(reason));
															#line 5532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				return a_body1cont1loopHead1(loopDepth); // continue
			}
		}
		else
		{
															#line 1094 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			req.reply.sendError(operation_failed());
															#line 5540 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(PrepareBlobRestoreRequest const& req,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(req, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(PrepareBlobRestoreRequest && req,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(req), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state > 0) static_cast<ServeBlobMigratorRequestsActor*>(this)->actor_wait_state = 0;
		static_cast<ServeBlobMigratorRequestsActor*>(this)->ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >::remove();

	}
	void a_callback_fire(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,PrepareBlobRestoreRequest const& value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,PrepareBlobRestoreRequest && value) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*,Error err) 
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributionTracker> tracker;
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DDQueue> queue;
															#line 5627 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via serveBlobMigratorRequests()
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ServeBlobMigratorRequestsActor final : public Actor<Void>, public ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >, public ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >, public FastAllocated<ServeBlobMigratorRequestsActor>, public ServeBlobMigratorRequestsActorState<ServeBlobMigratorRequestsActor> {
															#line 5632 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ServeBlobMigratorRequestsActor>::operator new;
	using FastAllocated<ServeBlobMigratorRequestsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2962024048285282816UL, 13338862073583597824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >;
friend struct ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >;
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ServeBlobMigratorRequestsActor(Reference<DataDistributor> const& self,Reference<DataDistributionTracker> const& tracker,Reference<DDQueue> const& queue) 
															#line 5650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ServeBlobMigratorRequestsActorState<ServeBlobMigratorRequestsActor>(self, tracker, queue),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13290839411857998080UL, 673675850101827840UL);
		ActorExecutionContextHelper __helper(static_cast<ServeBlobMigratorRequestsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("serveBlobMigratorRequests");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("serveBlobMigratorRequests", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ServeBlobMigratorRequestsActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< ServeBlobMigratorRequestsActor, 1, PrepareBlobRestoreRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> serveBlobMigratorRequests( Reference<DataDistributor> const& self, Reference<DataDistributionTracker> const& tracker, Reference<DDQueue> const& queue ) {
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ServeBlobMigratorRequestsActor(self, tracker, queue));
															#line 5684 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1098 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Trigger a task on range based on the current bulk load task metadata
															#line 5690 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class TriggerBulkLoadTaskActor>
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActorState {
															#line 5697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId)
															#line 5708 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~TriggerBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1103 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 5723 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1104 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Database cx = self->txnProcessor->context();
															#line 1105 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr = Transaction(cx);
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState = BulkLoadTaskState();
															#line 5757 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 1108 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1109 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1110 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1110 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5767 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1110 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5772 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1136 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 5794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1137 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistTriggerStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 5798 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1142 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1142 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5804 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1142 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5809 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1111 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1112 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1112 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5828 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1112 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5833 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1111 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1112 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1112 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1112 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5851 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1118 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1119 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1120 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1121 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1122 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1122 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1122 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5950 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1118 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1119 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1120 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1121 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1122 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1122 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1122 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5974 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1126 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1126 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6060 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1126 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 6065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1126 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1126 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6076 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1126 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 6081 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1127 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1132 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1133 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6171 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1127 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1132 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1133 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6189 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1105 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState newBulkLoadTaskState;
															#line 6369 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActor final : public Actor<std::pair<BulkLoadTaskState, Version>>, public ActorCallback< TriggerBulkLoadTaskActor, 0, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 1, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 2, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 3, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 4, Void >, public FastAllocated<TriggerBulkLoadTaskActor>, public TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor> {
															#line 6374 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<TriggerBulkLoadTaskActor>::operator new;
	using FastAllocated<TriggerBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16341390813417238528UL, 15920065914083934464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::pair<BulkLoadTaskState, Version>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TriggerBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 4, Void >;
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 6395 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<std::pair<BulkLoadTaskState, Version>>(),
		   TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9998265325154424320UL, 14265371869313678848UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("triggerBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<std::pair<BulkLoadTaskState, Version>> triggerBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<std::pair<BulkLoadTaskState, Version>>(new TriggerBulkLoadTaskActor(self, taskRange, taskId));
															#line 6432 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1146 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// TODO(BulkLoad): add reason to persist
															#line 6438 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via failBulkLoadTask()
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FailBulkLoadTaskActor>
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActorState {
															#line 6445 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cancelledDataMovePriority(cancelledDataMovePriority),
															#line 1152 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1153 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1154 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState()
															#line 6464 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~FailBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1155 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 6479 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1185 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FailBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FailBulkLoadTaskActorState(); static_cast<FailBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6502 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FailBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1157 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1158 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6528 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 0, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6533 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1176 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 6568 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1177 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistErrorStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1182 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1182 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6578 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1182 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 4, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6583 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1160 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1160 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6600 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1160 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6605 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1160 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1160 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6616 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1160 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6621 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1162 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1163 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1164 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1165 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1166 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1166 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6715 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1166 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6720 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1162 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1163 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1164 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1165 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1166 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1166 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6739 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1166 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1168 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1168 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6830 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1168 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6835 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1168 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1168 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1168 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6851 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1169 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1170 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6937 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1169 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1170 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6948 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int cancelledDataMovePriority;
															#line 1152 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1153 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1154 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 7129 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via failBulkLoadTask()
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< FailBulkLoadTaskActor, 0, Void >, public ActorCallback< FailBulkLoadTaskActor, 1, Void >, public ActorCallback< FailBulkLoadTaskActor, 2, Void >, public ActorCallback< FailBulkLoadTaskActor, 3, Void >, public ActorCallback< FailBulkLoadTaskActor, 4, Void >, public FastAllocated<FailBulkLoadTaskActor>, public FailBulkLoadTaskActorState<FailBulkLoadTaskActor> {
															#line 7134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FailBulkLoadTaskActor>::operator new;
	using FastAllocated<FailBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14740079401493580800UL, 9862220687738861056UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FailBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 4, Void >;
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 7155 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FailBulkLoadTaskActorState<FailBulkLoadTaskActor>(self, taskRange, taskId, cancelledDataMovePriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5276853651913060352UL, 273265509463260672UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("failBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> failBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId, int const& cancelledDataMovePriority ) {
															#line 1148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FailBulkLoadTaskActor(self, taskRange, taskId, cancelledDataMovePriority));
															#line 7192 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1187 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// A bulk load task is guaranteed to be either complete or overwritten by another task
// When a bulk load task is trigged, the range traffic is turned off atomically
// If the task completes, the task re-enables the traffic atomically
															#line 7200 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via doBulkLoadTask()
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DoBulkLoadTaskActor>
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActorState {
															#line 7207 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1192 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   completeAck(),
															#line 1193 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   triggeredBulkLoadTask(),
															#line 1194 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   commitVersion(invalidVersion),
															#line 1195 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 7226 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~DoBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1196 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadEngineParallelismLimitor.canStart());
															#line 1197 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadEngineParallelismLimitor.incrementTaskCounter();
															#line 7243 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1200 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<std::pair<BulkLoadTaskState, Version>> __when_expr_0 = triggerBulkLoadTask(self, range, taskId);
															#line 1200 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7249 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1200 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7254 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1278 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadEngineParallelismLimitor.decrementTaskCounter();
															#line 1279 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DoBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBulkLoadTaskActorState(); static_cast<DoBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 7285 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DoBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1264 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 7298 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1265 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 7302 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1267 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(e) .detail("Phase", "Error") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1273 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 7308 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1274 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 7312 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1201 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1202 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1203 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1210 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1216 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1221 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1222 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1222 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7342 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1222 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7347 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1201 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1202 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1203 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1210 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1216 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1221 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1222 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1222 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7370 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1222 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7375 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(triggeredBulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(triggeredBulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1223 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ack.unretrievableError)
															#line 7459 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1224 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "See unretrievable error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7463 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1232 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_2 = failBulkLoadTask(self, range, taskId, ack.dataMovePriority);
															#line 1232 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 7469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1232 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 2, Void >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 7474 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont3Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 1257 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Task complete") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck const& __ack,int loopDepth) 
	{
															#line 1222 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ack = __ack;
															#line 7497 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck && __ack,int loopDepth) 
	{
		ack = std::move(__ack);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& failTaskError,int loopDepth=0) 
	{
		try {
															#line 1240 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_actor_cancelled)
															#line 7589 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1241 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 7593 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1243 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(failTaskError) .detail("Phase", "Failed to mark task error. Error should be bulkload_task_outdated") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1250 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_movekeys_conflict)
															#line 7599 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1251 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 7603 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1253 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(failTaskError.code() == error_code_bulkload_task_outdated);
															#line 7607 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 1233 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 1233 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 7631 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1192 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<BulkLoadAck> completeAck;
															#line 1193 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState triggeredBulkLoadTask;
															#line 1194 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Version commitVersion;
															#line 1195 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1222 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadAck ack;
															#line 7753 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via doBulkLoadTask()
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >, public ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >, public ActorCallback< DoBulkLoadTaskActor, 2, Void >, public FastAllocated<DoBulkLoadTaskActor>, public DoBulkLoadTaskActorState<DoBulkLoadTaskActor> {
															#line 7758 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DoBulkLoadTaskActor>::operator new;
	using FastAllocated<DoBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1154971080244963328UL, 12978934465382019072UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >;
friend struct ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >;
friend struct ActorCallback< DoBulkLoadTaskActor, 2, Void >;
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 7777 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DoBulkLoadTaskActorState<DoBulkLoadTaskActor>(self, range, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9618644142710713856UL, 6256719351634756608UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> doBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& taskId ) {
															#line 1191 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DoBulkLoadTaskActor(self, range, taskId));
															#line 7812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1281 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 7817 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class EraseBulkLoadTaskActor>
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActorState {
															#line 7824 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1283 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1284 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1285 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask()
															#line 7841 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~EraseBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1286 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 7856 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1326 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<EraseBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EraseBulkLoadTaskActorState(); static_cast<EraseBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 7879 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<EraseBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1288 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1289 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1290 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(bulkLoadTask, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Acknowledged }));
															#line 1290 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1290 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 0, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7910 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1313 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 7945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1314 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskEraseStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7949 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1319 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_bulkload_task_outdated)
															#line 7953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1323 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1323 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7961 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1323 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 3, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7966 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1291 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1291 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7983 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1291 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7988 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1291 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1291 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7999 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1291 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8004 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1293 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1293 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8090 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1293 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8095 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1293 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1293 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8106 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1293 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 8111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1294 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1295 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1299 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1300 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1301 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 8203 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1305 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1306 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 8209 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1294 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1295 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1299 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1300 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1301 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 8227 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1305 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1306 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 8233 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1283 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1284 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1285 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 8413 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< EraseBulkLoadTaskActor, 0, Void >, public ActorCallback< EraseBulkLoadTaskActor, 1, Void >, public ActorCallback< EraseBulkLoadTaskActor, 2, Void >, public ActorCallback< EraseBulkLoadTaskActor, 3, Void >, public FastAllocated<EraseBulkLoadTaskActor>, public EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor> {
															#line 8418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<EraseBulkLoadTaskActor>::operator new;
	using FastAllocated<EraseBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5152067689425866240UL, 4838213614500430848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< EraseBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 3, Void >;
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 8438 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(445300323391651072UL, 14768580555042305792UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("eraseBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> eraseBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1282 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new EraseBulkLoadTaskActor(self, taskRange, taskId));
															#line 8474 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1328 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 8479 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadTasksActor>
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActorState {
															#line 8486 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActorState(Reference<DataDistributor> const& self) 
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1330 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginKey(normalKeys.begin),
															#line 1331 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   endKey(normalKeys.end),
															#line 1332 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeToRead(),
															#line 1333 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1334 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1335 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   i(0),
															#line 1336 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1337 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   result(),
															#line 1338 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadActors(),
															#line 1339 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range()
															#line 8513 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadTasksActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1340 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 8528 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1401 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = waitForAll(bulkLoadActors);
															#line 1401 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8553 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 7;
															#line 1401 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8558 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1340 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 8574 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1342 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1343 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			result.clear();
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store( result, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead, SERVER_KNOBS->DD_BULKLOAD_TASK_METADATA_READ_SIZE));
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 1;
															#line 1344 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8592 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1395 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 8627 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1396 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 8631 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1398 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = tr.onError(e);
															#line 1398 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8637 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_5.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 6;
															#line 1398 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1347 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1348 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 8659 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1347 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1348 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 8670 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1393 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = result.back().key;
															#line 8754 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1348 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < result.size() - 1))
															#line 8770 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1349 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (result[i].value.empty())
															#line 8776 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3continue1(loopDepth); // continue
		}
															#line 1352 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		range = Standalone(KeyRangeRef(result[i].key, result[i + 1].key));
															#line 1353 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[i].value);
															#line 1354 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 8786 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1355 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 1355 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8792 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 2;
															#line 1355 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8797 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3continue1(int loopDepth) 
	{
															#line 1348 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 8824 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1358 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (range != bulkLoadTaskState.getRange())
															#line 8833 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1360 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(0.1);
															#line 1360 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8839 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 3;
															#line 1360 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8844 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont6(int loopDepth) 
	{
															#line 1363 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.phase == BulkLoadPhase::Submitted || bulkLoadTaskState.phase == BulkLoadPhase::Triggered || bulkLoadTaskState.phase == BulkLoadPhase::Running)
															#line 8945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1367 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 8949 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont3loopBody1cont6loopHead1(loopDepth);
		}
		else
		{
															#line 1379 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadTaskState.phase == BulkLoadPhase::Acknowledged)
															#line 8956 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1380 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskClearMetadata", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 1384 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_4 = eraseBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId());
															#line 1384 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8964 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
				static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 5;
															#line 1384 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1385 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (bulkLoadTaskState.phase == BulkLoadPhase::Error)
															#line 8976 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1386 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevWarnAlways, "DDBulkLoadTaskUnretriableError", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 8980 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 1390 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(bulkLoadTaskState.phase == BulkLoadPhase::Complete);
															#line 8986 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				}
				loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont3loopBody1cont9(int loopDepth) 
	{
															#line 1348 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 9085 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont10(int loopDepth) 
	{
															#line 1373 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSchedule", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("Phase", bulkLoadTaskState.phase);
															#line 1377 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadActors.push_back( doBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId()));
															#line 9096 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1(int loopDepth) 
	{
															#line 1368 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadEngineParallelismLimitor.canStart())
															#line 9112 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3loopBody1cont6break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1371 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->bulkLoadEngineParallelismLimitor.waitUntilCounterChanged();
															#line 1371 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9120 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont6loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 4;
															#line 1371 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 9125 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont3loopBody1cont11(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1402 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 9427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1402 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 9439 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1330 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1331 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1332 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1333 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1334 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1335 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1336 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1337 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult result;
															#line 1338 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> bulkLoadActors;
															#line 1339 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 9544 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >, public FastAllocated<ScheduleBulkLoadTasksActor>, public ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor> {
															#line 9549 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator new;
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9249233369020088832UL, 7331943351695459584UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >;
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActor(Reference<DataDistributor> const& self) 
															#line 9572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14856297450692458752UL, 7460955458194361088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadTasks");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self ) {
															#line 1329 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadTasksActor(self));
															#line 9611 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1404 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 9616 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadTaskCoreActor>
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActorState {
															#line 9623 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart)
															#line 9632 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadTaskCoreActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1406 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 1406 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 1;
															#line 1406 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 0, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9654 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadTaskCoreActorState();
		static_cast<BulkLoadTaskCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1407 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1408 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1408 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9681 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1408 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9686 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1407 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1408 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1408 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9699 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1408 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9704 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1409 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1410 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9790 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1409 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1410 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9801 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
		try {
															#line 1412 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = scheduleBulkLoadTasks(self);
															#line 1412 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9895 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 3;
															#line 1412 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 2, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9900 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1422 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(SERVER_KNOBS->DD_BULKLOAD_SCHEDULE_MIN_INTERVAL_SEC);
															#line 1422 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9917 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 4;
															#line 1422 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 3, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9922 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1414 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 9932 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1415 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9936 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1417 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadTaskCoreError", self->ddId).errorUnsuppressed(e);
															#line 1418 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 9942 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1419 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9946 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 1407 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 10151 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActor final : public Actor<Void>, public ActorCallback< BulkLoadTaskCoreActor, 0, Void >, public ActorCallback< BulkLoadTaskCoreActor, 1, Void >, public ActorCallback< BulkLoadTaskCoreActor, 2, Void >, public ActorCallback< BulkLoadTaskCoreActor, 3, Void >, public FastAllocated<BulkLoadTaskCoreActor>, public BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor> {
															#line 10156 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadTaskCoreActor>::operator new;
	using FastAllocated<BulkLoadTaskCoreActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6903399204662472192UL, 4577084401014476800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadTaskCoreActor, 0, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 1, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 2, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 3, Void >;
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 10176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12210363843676168704UL, 8182787864321598208UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadTaskCore");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadTaskCore( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 1405 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadTaskCoreActor(self, readyToStart));
															#line 10212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1425 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 10217 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via checkBulkLoadJob()
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class CheckBulkLoadJobActor>
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadJobActorState {
															#line 10224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadJobActorState(Transaction* const& tr,UID const& jobId) 
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : tr(tr),
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1427 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeResult()
															#line 10235 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~CheckBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("checkBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1429 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(rangeResult, krmGetRanges(tr, bulkLoadJobPrefix, normalKeys));
															#line 1429 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10252 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1429 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadJobActor, 0, Void >*>(static_cast<CheckBulkLoadJobActor*>(this)));
															#line 10257 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1430 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1431 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1432 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10284 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1433 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10288 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1435 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1436 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10294 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1437 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10298 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1439 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10302 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1440 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10306 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1442 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckBulkLoadJobActorState(); static_cast<CheckBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10310 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1430 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1431 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1432 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10326 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1433 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10330 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1435 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1436 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10336 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1437 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10340 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1439 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10344 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1440 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1442 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckBulkLoadJobActorState(); static_cast<CheckBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10352 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckBulkLoadJobActorState();
		static_cast<CheckBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadJobActor*>(this)->ActorCallback< CheckBulkLoadJobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadJobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction* tr;
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1427 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult rangeResult;
															#line 10441 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via checkBulkLoadJob()
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadJobActor final : public Actor<Void>, public ActorCallback< CheckBulkLoadJobActor, 0, Void >, public FastAllocated<CheckBulkLoadJobActor>, public CheckBulkLoadJobActorState<CheckBulkLoadJobActor> {
															#line 10446 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<CheckBulkLoadJobActor>::operator new;
	using FastAllocated<CheckBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16686170609753829376UL, 16397454695523041024UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBulkLoadJobActor, 0, Void >;
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadJobActor(Transaction* const& tr,UID const& jobId) 
															#line 10463 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   CheckBulkLoadJobActorState<CheckBulkLoadJobActor>(tr, jobId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6430116811539734272UL, 4354220408088791808UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBulkLoadJobActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> checkBulkLoadJob( Transaction* const& tr, UID const& jobId ) {
															#line 1426 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new CheckBulkLoadJobActor(tr, jobId));
															#line 10496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1444 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 10501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via getBulkLoadJob()
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class GetBulkLoadJobActor>
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActorState {
															#line 10508 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActorState(Transaction* const& tr,UID const& jobId) 
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : tr(tr),
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1446 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeResult()
															#line 10519 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~GetBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1448 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(rangeResult, krmGetRanges(tr, bulkLoadJobPrefix, normalKeys));
															#line 1448 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10536 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1448 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetBulkLoadJobActor, 0, Void >*>(static_cast<GetBulkLoadJobActor*>(this)));
															#line 10541 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1449 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1450 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1451 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10568 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1452 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1454 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1455 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10578 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1456 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10582 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1458 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10586 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1459 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10590 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1461 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10594 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1449 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult.size() >= 2 && rangeResult.size() <= 3);
															#line 1450 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(rangeResult[0].key == normalKeys.begin);
															#line 1451 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 10610 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1452 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10614 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1454 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1455 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 10620 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1456 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10624 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1458 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId)
															#line 10628 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1459 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(bulkload_task_outdated(), loopDepth);
															#line 10632 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1461 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 10636 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<GetBulkLoadJobActor*>(this)->ActorCallback< GetBulkLoadJobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction* tr;
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1446 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult rangeResult;
															#line 10725 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via getBulkLoadJob()
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActor final : public Actor<BulkLoadJobState>, public ActorCallback< GetBulkLoadJobActor, 0, Void >, public FastAllocated<GetBulkLoadJobActor>, public GetBulkLoadJobActorState<GetBulkLoadJobActor> {
															#line 10730 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<GetBulkLoadJobActor>::operator new;
	using FastAllocated<GetBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13402810676553510912UL, 6369290153926454784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadJobState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetBulkLoadJobActor, 0, Void >;
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActor(Transaction* const& tr,UID const& jobId) 
															#line 10747 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadJobState>(),
		   GetBulkLoadJobActorState<GetBulkLoadJobActor>(tr, jobId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17307579102949951488UL, 17075247084345564672UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetBulkLoadJobActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadJobState> getBulkLoadJob( Transaction* const& tr, UID const& jobId ) {
															#line 1445 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadJobState>(new GetBulkLoadJobActor(tr, jobId));
															#line 10780 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1463 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Find task metadata for a bulk load job with jobId and input range
															#line 10786 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobFindTaskActor>
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActorState {
															#line 10793 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,UID const& logId) 
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   logId(logId),
															#line 1469 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1470 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1471 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 10812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobFindTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1472 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 10827 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1501 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(bulkLoadTaskState); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 10850 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(std::move(bulkLoadTaskState)); // state_var_RVO
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1474 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1475 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBulkLoadJob(&tr, jobId);
															#line 1475 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10874 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 1;
															#line 1475 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10879 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1498 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1498 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10916 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 3;
															#line 1498 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10921 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1476 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1477 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1477 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10940 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1477 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1476 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1477 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1477 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10958 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1477 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 10963 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(RangeResult const& result,int loopDepth) 
	{
															#line 1479 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1480 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1481 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 11051 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1482 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 11055 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1484 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1485 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1486 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 11067 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1487 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1494 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 11073 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3(RangeResult && result,int loopDepth) 
	{
															#line 1479 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1480 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1481 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 11087 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1482 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 11091 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1484 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1485 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1486 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 11103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1487 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1494 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 11109 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(result, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID logId;
															#line 1469 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1470 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1471 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 11291 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActor final : public Actor<Optional<BulkLoadTaskState>>, public ActorCallback< BulkLoadJobFindTaskActor, 0, Void >, public ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >, public ActorCallback< BulkLoadJobFindTaskActor, 2, Void >, public FastAllocated<BulkLoadJobFindTaskActor>, public BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor> {
															#line 11296 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobFindTaskActor>::operator new;
	using FastAllocated<BulkLoadJobFindTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9338484050132740352UL, 4551812253436582144UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<BulkLoadTaskState>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobFindTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 2, Void >;
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,UID const& logId) 
															#line 11315 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Optional<BulkLoadTaskState>>(),
		   BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor>(self, range, jobId, logId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3995601487305067776UL, 1783749224615820032UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobFindTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Optional<BulkLoadTaskState>> bulkLoadJobFindTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& jobId, UID const& logId ) {
															#line 1465 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Optional<BulkLoadTaskState>>(new BulkLoadJobFindTaskActor(self, range, jobId, logId));
															#line 11350 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1503 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Submit a bulkload task for the given jobId
															#line 11356 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobSubmitTaskActor>
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActorState {
															#line 11363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests) 
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(manifests),
															#line 1508 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1509 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1511 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(jobId, manifests)
															#line 11380 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobSubmitTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1512 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 11395 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1532 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV<BulkLoadTaskState>::futures) { (void)(bulkLoadTask); this->~BulkLoadJobSubmitTaskActorState(); static_cast<BulkLoadJobSubmitTaskActor*>(this)->destroy(); return 0; }
															#line 11418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV< BulkLoadTaskState >::value()) BulkLoadTaskState(std::move(bulkLoadTask)); // state_var_RVO
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1515 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1515 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11440 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 1;
															#line 1515 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11445 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1529 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1529 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 4;
															#line 1529 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1516 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1516 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11504 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1516 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11509 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1516 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1516 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11520 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1516 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11525 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1518 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1518 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11611 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1518 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11616 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1518 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1518 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11627 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1518 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 11632 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1519 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1520 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 11718 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1519 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1520 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 11729 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1508 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1509 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1511 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 11908 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActor final : public Actor<BulkLoadTaskState>, public ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >, public FastAllocated<BulkLoadJobSubmitTaskActor>, public BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor> {
															#line 11913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator new;
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13497705060275542784UL, 15971096285420478976UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadTaskState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >;
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests) 
															#line 11933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadTaskState>(),
		   BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor>(self, jobId, manifests),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9698732467664213248UL, 5402256740808584704UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobSubmitTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadTaskState> bulkLoadJobSubmitTask( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadManifestSet const& manifests ) {
															#line 1505 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadTaskState>(new BulkLoadJobSubmitTaskActor(self, jobId, manifests));
															#line 11969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1534 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// If the task reaches unretrievable error, return true; otherwise, return false.
															#line 11975 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobWaitUntilTaskCompleteOrErrorActor>
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActorState {
															#line 11982 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(bulkLoadTask)
															#line 11993 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1539 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(bulkLoadTask.isValid());
															#line 1540 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1541 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1542 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentTask = BulkLoadTaskState();
															#line 1543 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 12016 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1545 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentTask, getBulkLoadTask(&tr, bulkLoadTask.getRange(), bulkLoadTask.getTaskId(), { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running, BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }));
															#line 1545 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12049 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1545 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1572 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_JOB_MONITOR_PERIOD_SEC);
															#line 1572 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12071 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 1572 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12076 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1570 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1570 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12088 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1570 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 12093 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1555 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 12108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1556 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 12112 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1558 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 12116 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1559 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretrievableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1564 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12122 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1566 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 12130 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1567 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1555 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 12148 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1556 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 12152 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1558 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 12156 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1559 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretrievableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1564 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12162 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1566 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 12170 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1567 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 12174 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1540 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1541 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1542 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState currentTask;
															#line 12458 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActor final : public Actor<Void>, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>, public BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor> {
															#line 12463 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5123904649918049280UL, 11209088874171390208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >;
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 12482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>(self, jobId, bulkLoadTask),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16093638368250768128UL, 12615896230081500416UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobWaitUntilTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobWaitUntilTaskCompleteOrError( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadTaskState const& bulkLoadTask ) {
															#line 1536 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobWaitUntilTaskCompleteOrErrorActor(self, jobId, bulkLoadTask));
															#line 12517 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1575 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Given a list of manifestEntry, create a bulkload task.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 12525 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobNewTaskActor>
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActorState {
															#line 12532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRoot(jobRoot),
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestLocalTempFolder(manifestLocalTempFolder),
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntries(manifestEntries),
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1586 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1587 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1588 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(),
															#line 1589 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 12559 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobNewTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1590 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!manifestEntries.empty());
															#line 12574 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1593 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(manifests, getBulkLoadManifestMetadataFromEntry( manifestEntries, manifestLocalTempFolder, jobTransportMethod, jobRoot, self->ddId));
															#line 1593 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12580 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 1;
															#line 1593 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12585 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1641 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12614 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1626 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 12627 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1627 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 12631 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1629 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1635 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 12637 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1636 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 12641 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1599 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask( self, KeyRangeRef(manifests.getMinBeginKey(), manifests.getMaxEndKey()), jobId, self->ddId);
															#line 1599 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12659 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1599 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1599 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask( self, KeyRangeRef(manifests.getMinBeginKey(), manifests.getMaxEndKey()), jobId, self->ddId);
															#line 1599 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12675 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1599 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12680 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1601 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 12764 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1603 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12768 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1610 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1611 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests));
															#line 1611 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12780 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1611 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12785 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1601 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 12794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1603 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 12798 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1610 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1611 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests));
															#line 1611 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12810 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1611 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 12815 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1613 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1620 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1622 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1623 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12907 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1613 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1620 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1622 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1623 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12925 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string jobRoot;
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string manifestLocalTempFolder;
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1586 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1587 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1588 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1589 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 13041 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobNewTaskActor, 0, Void >, public ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobNewTaskActor, 2, Void >, public FastAllocated<BulkLoadJobNewTaskActor>, public BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor> {
															#line 13046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobNewTaskActor>::operator new;
	using FastAllocated<BulkLoadJobNewTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13824581548151361024UL, 9444798390567234560UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobNewTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 2, Void >;
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActor(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 13065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor>(self, jobId, jobRoot, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8655289167937893888UL, 2026549218517418240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobNewTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobNewTask( Reference<DataDistributor> const& self, UID const& jobId, std::string const& jobRoot, BulkLoadTransportMethod const& jobTransportMethod, std::string const& manifestLocalTempFolder, std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries, Promise<Void> const& errorOut ) {
															#line 1579 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobNewTaskActor(self, jobId, jobRoot, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut));
															#line 13100 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1643 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Given a bulkload task range, find the task and wait until the task is complete or error.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 13108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobMonitorTaskActor>
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActorState {
															#line 13115 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1651 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1652 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1653 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 13134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobMonitorTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1654 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadParallelismLimitor.canStart());
															#line 1655 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.incrementTaskCounter();
															#line 13151 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1658 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Optional<BulkLoadTaskState>> __when_expr_0 = bulkLoadJobFindTask(self, taskRange, jobId, self->ddId);
															#line 1658 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13157 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 1;
															#line 1658 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13162 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1702 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13191 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1687 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 13204 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1688 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 13208 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1690 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskMonitorError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1695 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1696 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 13216 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1697 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 13220 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1659 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 13236 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1661 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1662 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13242 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1664 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1665 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1672 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 13254 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1674 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1675 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 13260 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1679 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1679 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13266 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1679 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1659 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 13280 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1661 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1662 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 13286 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1664 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1665 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1672 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 13298 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1674 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1675 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 13304 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1679 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1679 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 13310 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1679 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 13315 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1680 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1685 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 13401 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1680 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1685 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 13412 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1651 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1652 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1653 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 13519 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >, public FastAllocated<BulkLoadJobMonitorTaskActor>, public BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor> {
															#line 13524 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator new;
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15769414295445147648UL, 3189923607606486016UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >;
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActor(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 13542 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor>(self, jobId, taskRange, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3763012194873441280UL, 18076434267272298240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobMonitorTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobMonitorTask( Reference<DataDistributor> const& self, UID const& jobId, KeyRange const& taskRange, Promise<Void> const& errorOut ) {
															#line 1647 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobMonitorTaskActor(self, jobId, taskRange, errorOut));
															#line 13576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1704 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 13581 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class PersistBulkLoadJobTaskCountActor>
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActorState {
															#line 13588 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActorState(Reference<DataDistributor> const& self) 
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1706 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1707 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 13599 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	~PersistBulkLoadJobTaskCountActorState() 
	{
		fdb_probe_actor_destroy("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1708 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1709 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = jobState.getJobId();
															#line 1711 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = jobState.getJobRange();
															#line 1712 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(jobState.getTaskCount().present());
															#line 1713 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			taskCount = jobState.getTaskCount().get();
															#line 1714 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentJobState = BulkLoadJobState();
															#line 1715 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 13628 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1744 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13651 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1717 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1717 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13673 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 1;
															#line 1717 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1741 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1741 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13715 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 4;
															#line 1741 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13720 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1718 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 13735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1719 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 13739 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1720 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1726 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 13745 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1728 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13749 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1730 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1731 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1731 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13761 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1731 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13766 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1718 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 13775 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1719 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 13779 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1720 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1726 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 13785 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1728 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 13789 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1730 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1731 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1731 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13801 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1731 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1732 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1732 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1732 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13897 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1732 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1732 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13908 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1732 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 13913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1733 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1734 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 13999 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1733 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1734 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 14010 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1706 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1707 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1709 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1710 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1711 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1713 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	uint64_t taskCount;
															#line 1714 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 14193 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActor final : public Actor<Void>, public ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >, public FastAllocated<PersistBulkLoadJobTaskCountActor>, public PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor> {
															#line 14198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator new;
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12895010126530304000UL, 14275659048149996032UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >;
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActor(Reference<DataDistributor> const& self) 
															#line 14218 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10651365252717723648UL, 4800485421425282816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistBulkLoadJobTaskCount");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> persistBulkLoadJobTaskCount( Reference<DataDistributor> const& self ) {
															#line 1705 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new PersistBulkLoadJobTaskCountActor(self));
															#line 14254 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1746 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Remove the bulkload job metadata from the range map metadata.
// Add the job metadata to the history metadata with the error message.
															#line 14261 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MoveErrorBulkLoadJobToHistoryActor>
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActorState {
															#line 14268 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActorState(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorMessage(errorMessage),
															#line 1750 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1751 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1752 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentJobState()
															#line 14283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	~MoveErrorBulkLoadJobToHistoryActorState() 
	{
		fdb_probe_actor_destroy("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1753 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1754 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = self->bulkLoadJobManager.get().jobState.getJobId();
															#line 1755 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 14302 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1770 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveErrorBulkLoadJobToHistoryActorState(); static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->destroy(); return 0; }
															#line 14325 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1757 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1757 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14347 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 1;
															#line 1757 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14352 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1767 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = tr.onError(e);
															#line 1767 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14389 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_5.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 6;
															#line 1767 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14394 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1758 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1758 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14411 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1758 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14416 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1758 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId));
															#line 1758 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1758 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14432 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1759 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1759 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14518 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1759 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14523 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1759 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1759 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14534 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1759 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14539 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1761 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1762 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1763 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1763 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14629 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1763 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14634 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1761 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1762 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1763 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1763 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1763 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14654 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1764 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = tr.commit();
															#line 1764 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14740 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1764 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14745 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1764 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = tr.commit();
															#line 1764 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14756 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1764 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 14761 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string errorMessage;
															#line 1750 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1751 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1752 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 1754 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 15027 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActor final : public Actor<Void>, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >, public FastAllocated<MoveErrorBulkLoadJobToHistoryActor>, public MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor> {
															#line 15032 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator new;
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13493791532589790464UL, 5113038557044698368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >;
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActor(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 15054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor>(self, errorMessage),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10076677940119608320UL, 17212560479686694656UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("moveErrorBulkLoadJobToHistory");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> moveErrorBulkLoadJobToHistory( Reference<DataDistributor> const& self, std::string const& errorMessage ) {
															#line 1749 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MoveErrorBulkLoadJobToHistoryActor(self, errorMessage));
															#line 15092 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1772 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Download the job manifest file from the remoteJobManifestFilePath to the localJobManifestFilePath.
// Build the bulkload manifest range map based on the localJobManifestFilePath file content.
// For any failure, we mark the job metadata as error and move the metadata to the history.
															#line 15100 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FetchBulkLoadTaskManifestEntryMapActor>
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActorState {
															#line 15107 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActorState(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   localJobManifestFilePath(localJobManifestFilePath),
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   remoteJobManifestFilePath(remoteJobManifestFilePath)
															#line 15120 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	~FetchBulkLoadTaskManifestEntryMapActorState() 
	{
		fdb_probe_actor_destroy("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1780 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid() && self->bulkLoadJobManager.get().manifestEntryMap->empty());
															#line 1782 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginTime = now();
															#line 1783 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = self->bulkLoadJobManager.get().jobState.getJobRange();
															#line 15139 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1785 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!fileExists(abspath(localJobManifestFilePath)))
															#line 15143 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1786 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDBulkLoadJobManagerDownloadJobManifest", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath);
															#line 1790 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					StrictFuture<Void> __when_expr_0 = downloadBulkLoadJobManifestFile( jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath, self->ddId);
															#line 1790 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 15151 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 1;
															#line 1790 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15156 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1838 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchBulkLoadTaskManifestEntryMapActorState(); static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->destroy(); return 0; }
															#line 15190 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1814 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 15203 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1815 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 15207 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1817 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			err = e;
															#line 1818 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1824 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string errorMessage = "Failed to build job-manifest map with error code " + std::to_string(err.code()) + ". The remote file path is " + remoteJobManifestFilePath + ". The local file path is " + localJobManifestFilePath + ". The transport method is " + convertBulkLoadTransportMethodToString(jobTransportMethod) + ".";
															#line 1829 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = moveErrorBulkLoadJobToHistory(self, errorMessage);
															#line 1829 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_2.get(), loopDepth); };
			static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 3;
															#line 1829 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1801 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = getBulkLoadJobFileManifestEntryFromJobManifestFile( localJobManifestFilePath, jobRange, self->ddId, self->bulkLoadJobManager.get().manifestEntryMap);
															#line 1801 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 15241 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 2;
															#line 1801 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 15246 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1792 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 15255 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1792 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 15264 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1806 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1807 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 15350 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1806 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1807 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 15361 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch2cont1(Void const& _,int loopDepth) 
	{
															#line 1830 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1836 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 15460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2cont1(Void && _,int loopDepth) 
	{
															#line 1830 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretrievableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1836 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 15470 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string localJobManifestFilePath;
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string remoteJobManifestFilePath;
															#line 1782 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1783 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1817 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Error err;
															#line 15563 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActor final : public Actor<Void>, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >, public FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>, public FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor> {
															#line 15568 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator new;
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9767248698875538176UL, 16441752953021687040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >;
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActor(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 15587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor>(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7119443845485172480UL, 13876208275314761984UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchBulkLoadTaskManifestEntryMap");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> fetchBulkLoadTaskManifestEntryMap( Reference<DataDistributor> const& self, BulkLoadTransportMethod const& jobTransportMethod, std::string const& localJobManifestFilePath, std::string const& remoteJobManifestFilePath ) {
															#line 1776 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FetchBulkLoadTaskManifestEntryMapActor(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath));
															#line 15622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1840 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 15627 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadJobActor>
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActorState {
															#line 15634 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActorState(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1842 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntry()
															#line 15645 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1843 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1844 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1845 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1846 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors = std::vector<Future<Void>>();
															#line 1847 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1848 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1852 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 15672 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1937 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(actors);
															#line 1937 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 1937 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15702 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1854 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, jobState.getJobRange().end));
															#line 1854 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15721 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1854 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15726 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1934 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1934 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15763 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 1934 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15768 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1856 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1857 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 15785 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 1854 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		res = __res;
															#line 15794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1928 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (beginKey == jobState.getJobRange().end)
															#line 15873 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1930 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager.get().allTaskSubmitted = true;
															#line 15877 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1857 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < res.size() - 1))
															#line 15895 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1861 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!res[i].value.empty())
															#line 15901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1862 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			task = decodeBulkLoadTaskState(res[i].value);
															#line 1863 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (task.isValid())
															#line 15907 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1864 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.getJobId() != self->bulkLoadJobManager.get().jobState.getJobId())
															#line 15911 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1865 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					return a_body1loopBody1Catch1(bulkload_task_outdated(), std::max(0, loopDepth - 1));
															#line 15915 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				}
															#line 1868 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(self->bulkLoadJobManager.get().manifestEntryMap->find(task.getRange().begin) != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1870 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.onAnyPhase( { BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }))
															#line 15921 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1872 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(task.getRange().end == res[i + 1].key);
															#line 1873 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					beginKey = task.getRange().end;
															#line 15927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					return a_body1loopBody1cont2continue1(loopDepth); // continue
				}
				else
				{
															#line 1876 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (task.onAnyPhase({ BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }))
															#line 15934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 1881 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (self->bulkLoadJobManager.get().allTaskSubmitted)
															#line 15938 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 1883 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
							;
															#line 15942 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
							loopDepth = a_body1loopBody1cont2loopBody1loopHead1(loopDepth);
						}
						else
						{
							loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);
						}
					}
					else
					{
															#line 1897 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
						UNREACHABLE();
															#line 15954 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = a_body1loopBody1cont2loopBody1cont7(loopDepth);
					}
				}
			}
			else
			{
				loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2continue1(int loopDepth) 
	{
															#line 1857 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 15988 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1905 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i].key);
															#line 1906 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 15999 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont8(int loopDepth) 
	{
															#line 1893 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(task.getRange().end == res[i + 1].key);
															#line 1894 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = task.getRange().end;
															#line 16028 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont2continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont9(int loopDepth) 
	{
															#line 1890 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back( bulkLoadJobMonitorTask(self, task.getJobId(), task.getRange(), errorOut));
															#line 16037 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1(int loopDepth) 
	{
															#line 1884 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadParallelismLimitor.canStart())
															#line 16053 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1887 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->bulkLoadParallelismLimitor.waitUntilCounterChanged();
															#line 1887 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 16061 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 1887 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 16066 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2loopBody1cont13(int loopDepth) 
	{
															#line 1926 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i + 1].key);
															#line 1857 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 16177 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 1906 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < res[i + 1].key))
															#line 16193 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1907 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1908 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;manifestEntries.size() < SERVER_KNOBS->MANIFEST_COUNT_MAX_PER_BULKLOAD_TASK && beginKey < res[i + 1].key;) {
															#line 1910 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->bulkLoadJobManager.get().manifestEntryMap->find(beginKey);
															#line 1911 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(it != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1912 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntry = it->second;
															#line 1913 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntries.push_back(manifestEntry);
															#line 1914 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = manifestEntry.getEndKey();
															#line 16211 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1916 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!manifestEntries.empty());
															#line 1917 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back(bulkLoadJobNewTask(self, jobState.getJobId(), jobState.getJobRoot(), jobState.getTransportMethod(), self->bulkLoadJobManager.get().manifestLocalTempFolder, manifestEntries, errorOut));
															#line 1924 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_TASK_SUBMISSION_INTERVAL_SEC);
															#line 1924 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 16221 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 1924 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 16226 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont13(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1938 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 16435 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1938 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 16447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1842 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFileManifestEntry manifestEntry;
															#line 1844 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1845 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1846 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> actors;
															#line 1847 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1848 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1854 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult res;
															#line 1856 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1862 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState task;
															#line 16552 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >, public ActorCallback< ScheduleBulkLoadJobActor, 1, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 2, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 3, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 4, Void >, public FastAllocated<ScheduleBulkLoadJobActor>, public ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor> {
															#line 16557 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadJobActor>::operator new;
	using FastAllocated<ScheduleBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11483070649125406720UL, 15869896022407095040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 4, Void >;
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActor(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 16578 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor>(self, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4684338048225246976UL, 14035353769110992896UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadJob( Reference<DataDistributor> const& self, Promise<Void> const& errorOut ) {
															#line 1841 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadJobActor(self, errorOut));
															#line 16615 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1940 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 16620 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class CheckBulkLoadTaskCompleteOrErrorActor>
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActorState {
															#line 16627 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self) 
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1942 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1943 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 16638 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~CheckBulkLoadTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1944 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1945 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1946 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1947 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 1948 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 1949 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = KeyRange();
															#line 1950 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 1951 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 16667 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1996 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(true); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16690 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(true);
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1951 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 16709 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1953 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 1954 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1955 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBulkLoadJob(&tr, jobState.getJobId());
															#line 1955 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16722 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1955 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16727 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1993 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1993 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16764 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 1993 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16769 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1956 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1956 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16786 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1956 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16791 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1956 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1956 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16802 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1956 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16807 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1957 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1958 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1959 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1960 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16897 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1962 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1966 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16909 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1967 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1972 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16915 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1974 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1975 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16923 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 1982 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16928 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1983 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1988 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1991 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 16943 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1957 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1958 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1959 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1960 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16958 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1962 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16962 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1966 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1967 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1972 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16976 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1974 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16980 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1975 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16984 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 1982 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16989 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1983 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1988 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16995 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1991 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 17004 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1942 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1943 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1945 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1946 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1947 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1948 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState existTask;
															#line 1949 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1950 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult bulkLoadTaskResult;
															#line 17202 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActor final : public Actor<bool>, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>, public CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor> {
															#line 17207 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10738023392804713216UL, 5934652075669778688UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >;
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActor(Reference<DataDistributor> const& self) 
															#line 17226 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<bool>(),
		   CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13835388531052272896UL, 2426800946393579264UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBulkLoadTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<bool> checkBulkLoadTaskCompleteOrError( Reference<DataDistributor> const& self ) {
															#line 1941 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<bool>(new CheckBulkLoadTaskCompleteOrErrorActor(self));
															#line 17261 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1998 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 17266 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via finalizeBulkLoadJob()
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FinalizeBulkLoadJobActor>
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FinalizeBulkLoadJobActorState {
															#line 17273 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FinalizeBulkLoadJobActorState(Reference<DataDistributor> const& self) 
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2000 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 2001 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 17284 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~FinalizeBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2002 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 2003 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 2004 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 2005 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 2006 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = Optional<Key>();
															#line 2007 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 2008 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 2009 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobCompleteRange = KeyRange();
															#line 2010 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = false;
															#line 2011 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			i = 0;
															#line 2012 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			allFinish = false;
															#line 2013 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 17321 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2092 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FinalizeBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FinalizeBulkLoadJobActorState(); static_cast<FinalizeBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 17344 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FinalizeBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2013 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 17363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 2015 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.reset();
															#line 2016 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 2017 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobState.getJobId());
															#line 2017 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17376 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 2017 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17381 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2089 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_8 = tr.onError(e);
															#line 2089 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_8.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 9;
															#line 2089 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17423 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
															#line 2018 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2019 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2019 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17442 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2019 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState && currentJobState,int loopDepth) 
	{
															#line 2018 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2019 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2019 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2019 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17465 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(currentJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(currentJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 2020 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2021 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 17551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 2020 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2021 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 17562 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 2059 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (hasError)
															#line 17646 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2060 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setErrorPhase("BulkLoadTask Error.");
															#line 17650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 2062 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setCompletePhase();
															#line 17656 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 2064 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(lastKey.present());
															#line 2065 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobCompleteRange = KeyRangeRef(jobState.getJobRange().begin, lastKey.get());
															#line 2066 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 2066 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17666 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 2066 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17671 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1(int loopDepth) 
	{
															#line 2021 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < bulkLoadTaskResult.size() - 1))
															#line 17687 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2022 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 2023 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 2024 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!existTask.isValid())
															#line 17697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2025 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = bulkLoadTaskResult[i + 1].key;
															#line 17701 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1loopBody1cont4continue1(loopDepth); // continue
		}
															#line 2028 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getRange() == KeyRangeRef(bulkLoadTaskResult[i].key, bulkLoadTaskResult[i + 1].key));
															#line 2033 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getJobId() == jobState.getJobId());
															#line 2034 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (existTask.phase == BulkLoadPhase::Error)
															#line 17710 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2035 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerStopClearMetadata", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2042 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = true;
															#line 17716 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);
		}
		else
		{
															#line 2044 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete && existTask.phase != BulkLoadPhase::Acknowledged)
															#line 17723 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2045 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerWrongTaskPhase", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2051 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 17729 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2054 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = setBulkLoadFinalizeTransaction(&tr, existTask.getRange(), existTask.getTaskId());
															#line 2054 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 2054 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17740 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4continue1(int loopDepth) 
	{
															#line 2021 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 17763 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont1(int loopDepth) 
	{
															#line 2056 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		lastKey = bulkLoadTaskResult[i + 1].key;
															#line 17772 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2067 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17870 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2070 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17878 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17883 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2074 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_6 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2074 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_6.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2074 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17897 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2067 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2068 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17909 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2070 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17917 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2071 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17922 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2074 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_6 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2074 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17931 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_6.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2074 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17936 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 2076 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_7 = tr.commit();
															#line 2076 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_7.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 8;
															#line 2076 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18028 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 2072 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2072 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18039 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2072 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18044 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 2072 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2072 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18055 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2072 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 18060 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
															#line 2077 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 2078 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerFinalizeRange", self->ddId) .detail("JobCompleteRange", jobCompleteRange) .detail("CommitVersion", commitVersion) .detail("AllFinish", allFinish) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2087 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = lastKey.get();
															#line 18322 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont12cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
															#line 2077 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 2078 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerFinalizeRange", self->ddId) .detail("JobCompleteRange", jobCompleteRange) .detail("CommitVersion", commitVersion) .detail("AllFinish", allFinish) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2087 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = lastKey.get();
															#line 18335 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont12cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1loopBody1cont12cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 2000 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 2001 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 2003 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 2004 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 2005 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 2006 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Optional<Key> lastKey;
															#line 2007 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState existTask;
															#line 2008 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult bulkLoadTaskResult;
															#line 2009 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobCompleteRange;
															#line 2010 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool hasError;
															#line 2011 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 2012 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool allFinish;
															#line 18541 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via finalizeBulkLoadJob()
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FinalizeBulkLoadJobActor final : public Actor<Void>, public ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >, public ActorCallback< FinalizeBulkLoadJobActor, 1, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 2, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 3, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 4, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 5, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 6, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 7, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 8, Void >, public FastAllocated<FinalizeBulkLoadJobActor>, public FinalizeBulkLoadJobActorState<FinalizeBulkLoadJobActor> {
															#line 18546 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FinalizeBulkLoadJobActor>::operator new;
	using FastAllocated<FinalizeBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10058113230511092224UL, 8748724029763760384UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 1, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 2, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 3, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 4, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 5, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 6, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 7, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 8, Void >;
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FinalizeBulkLoadJobActor(Reference<DataDistributor> const& self) 
															#line 18571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FinalizeBulkLoadJobActorState<FinalizeBulkLoadJobActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13125497442541706752UL, 11431860539656438528UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("finalizeBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> finalizeBulkLoadJob( Reference<DataDistributor> const& self ) {
															#line 1999 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FinalizeBulkLoadJobActor(self));
															#line 18612 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 2094 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 18617 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobManager()
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobManagerActor>
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobManagerActorState {
															#line 18624 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobManagerActorState(Reference<DataDistributor> const& self) 
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2097 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context())
															#line 18633 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobManager", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobManagerActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobManager", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2098 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<BulkLoadJobState>> __when_expr_0 = getRunningBulkLoadJob(cx);
															#line 2098 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 1;
															#line 2098 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18655 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobManagerActorState();
		static_cast<BulkLoadJobManagerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2099 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!job.present())
															#line 18678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2100 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobManagerNoJobExist", self->ddId);
															#line 2101 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager.reset();
															#line 2102 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadTaskCollection->removeBulkLoadJobRange();
															#line 2103 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobManagerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobManagerActorState(); static_cast<BulkLoadJobManagerActor*>(this)->destroy(); return 0; }
															#line 18688 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobManagerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobManagerActorState();
			static_cast<BulkLoadJobManagerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2105 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobId = job.get().getJobId();
															#line 2106 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobRange = job.get().getJobRange();
															#line 2107 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobRoot = job.get().getJobRoot();
															#line 2108 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobTransportMethod = job.get().getTransportMethod();
															#line 2109 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->setBulkLoadJobRange(jobRange);
															#line 2112 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->bulkLoadJobManager.present() || self->bulkLoadJobManager.get().jobState.getJobId() != jobId)
															#line 18706 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2113 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadJobManagerBuild", self->ddId) .detail("OldJobID", self->bulkLoadJobManager.present() ? self->bulkLoadJobManager.get().jobState.getJobId().toString() : "No old job") .detail("NewJobId", jobId) .detail("NewJobRange", jobRange) .detail("NewJobRoot", jobRoot) .detail("NewJobTransportMethod", jobTransportMethod);
															#line 2122 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string localFolder = getBulkLoadJobRoot(self->bulkLoadFolder, jobId);
															#line 2123 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string manifestLocalTempFolder = abspath(joinPath(localFolder, "manifest-temp"));
															#line 2124 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			resetFileFolder(manifestLocalTempFolder);
															#line 2125 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string remoteFolder = getBulkLoadJobRoot(jobRoot, jobId);
															#line 2126 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string jobManifestFileName = getBulkLoadJobManifestFileName();
															#line 2127 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string localJobManifestFilePath = joinPath(localFolder, jobManifestFileName);
															#line 2128 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string remoteJobManifestFilePath = appendToPath(remoteFolder, jobManifestFileName);
															#line 2129 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager = DDBulkLoadJobManager(job.get(), manifestLocalTempFolder);
															#line 2130 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = fetchBulkLoadTaskManifestEntryMap( self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath);
															#line 2130 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18730 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 2;
															#line 2130 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 1, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2134 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadJobManagerExist", self->ddId) .detail("JobID", self->bulkLoadJobManager.get().jobState.getJobId()) .detail("JobRange", self->bulkLoadJobManager.get().jobState.getJobRange());
															#line 18742 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadJobState> const& __job,int loopDepth) 
	{
															#line 2098 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		job = __job;
															#line 18752 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadJobState> && __job,int loopDepth) 
	{
		job = std::move(__job);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Optional<BulkLoadJobState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Optional<BulkLoadJobState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2140 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 2146 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		errorOut = Promise<Void>();
															#line 2147 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 18835 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2132 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = persistBulkLoadJobTaskCount(self);
															#line 2132 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 3;
															#line 2132 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 2, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18851 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2132 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = persistBulkLoadJobTaskCount(self);
															#line 2132 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18862 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 3;
															#line 2132 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 2, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18867 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 2169 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobManagerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobManagerActorState(); static_cast<BulkLoadJobManagerActor*>(this)->destroy(); return 0; }
															#line 19038 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobManagerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobManagerActorState();
		static_cast<BulkLoadJobManagerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 2148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<bool> __when_expr_3 = checkBulkLoadTaskCompleteOrError(self);
															#line 2148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19059 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 4;
															#line 2148 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 3, bool >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 19064 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(bool const& complete,int loopDepth) 
	{
															#line 2149 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (complete)
															#line 19086 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2150 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadJobManagerAllTaskComplete", self->ddId) .detail("JobID", self->bulkLoadJobManager.get().jobState.getJobId()) .detail("JobRange", self->bulkLoadJobManager.get().jobState.getJobRange());
															#line 2153 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = finalizeBulkLoadJob(self);
															#line 2153 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19094 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_4.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 5;
															#line 2153 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 4, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 19099 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2156 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::vector<Future<Void>> actors;
															#line 2157 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors.push_back(errorOut.getFuture());
															#line 2158 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors.push_back(scheduleBulkLoadJob(self, errorOut));
															#line 2159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = waitForAny(actors);
															#line 2159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19114 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when2(__when_expr_5.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 6;
															#line 2159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 5, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 19119 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(bool && complete,int loopDepth) 
	{
															#line 2149 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (complete)
															#line 19129 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2150 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadJobManagerAllTaskComplete", self->ddId) .detail("JobID", self->bulkLoadJobManager.get().jobState.getJobId()) .detail("JobRange", self->bulkLoadJobManager.get().jobState.getJobRange());
															#line 2153 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = finalizeBulkLoadJob(self);
															#line 2153 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19137 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_4.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 5;
															#line 2153 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 4, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 19142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2156 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::vector<Future<Void>> actors;
															#line 2157 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors.push_back(errorOut.getFuture());
															#line 2158 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors.push_back(scheduleBulkLoadJob(self, errorOut));
															#line 2159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = waitForAny(actors);
															#line 2159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19157 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when2(__when_expr_5.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 6;
															#line 2159 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 5, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 19162 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(bool const& complete,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(complete, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(bool && complete,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(std::move(complete), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 3, bool >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 3, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 3, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 3, bool >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2loopBody1cont2(int loopDepth) 
	{
															#line 2167 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(SERVER_KNOBS->DD_BULKLOAD_SCHEDULE_MIN_INTERVAL_SEC);
															#line 2167 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19249 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont2when1(__when_expr_6.get(), loopDepth); };
		static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 7;
															#line 2167 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 6, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 19254 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3(Void const& _,int loopDepth) 
	{
		return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3(Void && _,int loopDepth) 
	{
		return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont2loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 2163 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerTaskDispatched", self->ddId) .detail("JobID", self->bulkLoadJobManager.get().jobState.getJobId()) .detail("JobRange", self->bulkLoadJobManager.get().jobState.getJobRange());
															#line 19350 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 2163 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerTaskDispatched", self->ddId) .detail("JobID", self->bulkLoadJobManager.get().jobState.getJobId()) .detail("JobRange", self->bulkLoadJobManager.get().jobState.getJobRange());
															#line 19359 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont2loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont2loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont2loopBody1cont7(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont7(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont2loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont2loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 2097 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 2098 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Optional<BulkLoadJobState> job;
															#line 2105 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 2106 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 2107 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string jobRoot;
															#line 2108 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 2146 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 19542 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobManager()
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobManagerActor final : public Actor<Void>, public ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >, public ActorCallback< BulkLoadJobManagerActor, 1, Void >, public ActorCallback< BulkLoadJobManagerActor, 2, Void >, public ActorCallback< BulkLoadJobManagerActor, 3, bool >, public ActorCallback< BulkLoadJobManagerActor, 4, Void >, public ActorCallback< BulkLoadJobManagerActor, 5, Void >, public ActorCallback< BulkLoadJobManagerActor, 6, Void >, public FastAllocated<BulkLoadJobManagerActor>, public BulkLoadJobManagerActorState<BulkLoadJobManagerActor> {
															#line 19547 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobManagerActor>::operator new;
	using FastAllocated<BulkLoadJobManagerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7149438110462579456UL, 11920537388442615552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >;
friend struct ActorCallback< BulkLoadJobManagerActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobManagerActor, 2, Void >;
friend struct ActorCallback< BulkLoadJobManagerActor, 3, bool >;
friend struct ActorCallback< BulkLoadJobManagerActor, 4, Void >;
friend struct ActorCallback< BulkLoadJobManagerActor, 5, Void >;
friend struct ActorCallback< BulkLoadJobManagerActor, 6, Void >;
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobManagerActor(Reference<DataDistributor> const& self) 
															#line 19570 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobManagerActorState<BulkLoadJobManagerActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18271078866073528064UL, 6796315670317911296UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobManager");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobManagerActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobManagerActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadJobManagerActor, 3, bool >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< BulkLoadJobManagerActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< BulkLoadJobManagerActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< BulkLoadJobManagerActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobManager( Reference<DataDistributor> const& self ) {
															#line 2095 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobManagerActor(self));
															#line 19609 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
}

#line 2171 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 19614 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobCore()
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobCoreActor>
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobCoreActorState {
															#line 19621 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobCoreActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart)
															#line 19630 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobCore", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobCoreActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2173 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 2173 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 19647 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state = 1;
															#line 2173 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobCoreActor, 0, Void >*>(static_cast<BulkLoadJobCoreActor*>(this)));
															#line 19652 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobCoreActorState();
		static_cast<BulkLoadJobCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2174 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 19675 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2174 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 19684 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobCoreActor*>(this)->ActorCallback< BulkLoadJobCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2176 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = bulkLoadJobManager(self);
															#line 2176 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 19778 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state = 2;
															#line 2176 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobCoreActor, 1, Void >*>(static_cast<BulkLoadJobCoreActor*>(this)));
															#line 19783 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 2186 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_SCHEDULE_MIN_INTERVAL_SEC);
															#line 2186 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19800 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state = 3;
															#line 2186 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobCoreActor, 2, Void >*>(static_cast<BulkLoadJobCoreActor*>(this)));
															#line 19805 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2178 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 19815 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2179 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 19819 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2181 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobManagerError", self->ddId).errorUnsuppressed(e);
															#line 2182 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 19825 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2183 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 19829 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobCoreActor*>(this)->ActorCallback< BulkLoadJobCoreActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobCoreActor*>(this)->ActorCallback< BulkLoadJobCoreActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobCoreActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobCoreActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobCoreActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobCore", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 20032 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobCore()
															#line 2172 "/Users/ec2-user/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobCoreActor final : public Actor<Void>, pu