#define POST_ACTOR_COMPILER 1
#line 1 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
/*
 * GlobalTagThrottler.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/FDBTypes.h"
#include "fdbclient/TagThrottle.actor.h"
#include "fdbrpc/Smoother.h"
#include "fdbserver/ServerThroughputTracker.h"
#include "fdbserver/TagThrottler.h"

#include <limits>

#include "flow/actorcompiler.h" // must be last include

// In the function names below, several terms are used repeatedly. The context-specific are defined here:
//
// Cost: Every read or write operation has an associated cost, determined by the number of bytes accessed.
//       Global tag throttling quotas are specified in terms of the amount of this cost that can be consumed
//       per second. In the global tag throttler, cost refers to the per second rate of cost consumption.
//
// TPS: Transactions per second. Quotas are not specified in terms of TPS, but the limits given to clients must
//      be specified in terms of TPS because throttling is performed at the front end of transactions (before costs are
//      known).
//
// Total: Refers to the total quota specified by clients through the global tag throttling API. The sum of the
//        costs of all operations (cluster-wide) with a particular tag cannot exceed the tag's specified total quota,
//        even if the cluster has no saturated processes.
//
// Desired TPS: Assuming that a tag is able to achieve its total quota, this is the TPS it would be able to perform.
//
// Reserved: Refers to the reserved quota specified by clients through the global tag throttling API. As long as the
//           sum of the costs of all operations (cluster-wide) with a particular tag are not above the tag's
//           specified reserved quota, the tag should not experience any throttling from the global tag throttler.
//
// Current [Cost|TPS]: Measuring the current throughput on the cluster, independent of any specified quotas.
//
// ThrottlingRatio: Based on the health of each storage server, a throttling ratio is provided,
//                  informing the global tag throttler what ratio of the current throughput can be maintained.
//
// Limiting [Cost|TPS]: Based on the health of storage servers, a limiting throughput may be enforced.
//
// Target [Cost|TPS]: Based on reserved, limiting, and desired throughputs, this is the target throughput
//                    that the global tag throttler aims to achieve (across all clients).
//
// PerClient TPS: Because the target throughput must be shared across multiple clients, and all clients must
//           be given the same limits, a per-client limit is calculated based on the current and target throughputs.

class GlobalTagThrottlerImpl {
	template <class K, class V>
	static Optional<V> tryGet(std::unordered_map<K, V> const& m, K const& k) {
		auto it = m.find(k);
		if (it == m.end()) {
			return {};
		} else {
			return it->second;
		}
	}

	static Optional<double> getMin(Optional<double> a, Optional<double> b) {
		if (a.present() && b.present()) {
			return std::min(a.get(), b.get());
		} else if (a.present()) {
			return a;
		} else {
			return b;
		}
	}

	static Optional<double> getMax(Optional<double> a, Optional<double> b) {
		if (a.present() && b.present()) {
			return std::max(a.get(), b.get());
		} else if (a.present()) {
			return a;
		} else {
			return b;
		}
	}

	enum class LimitType { RESERVED, TOTAL };
	enum class OpType { READ, WRITE };

	class ThroughputCounters {
		Smoother readCost;
		Smoother writeCost;

	public:
		ThroughputCounters()
		  : readCost(SERVER_KNOBS->GLOBAL_TAG_THROTTLING_COST_FOLDING_TIME),
		    writeCost(SERVER_KNOBS->GLOBAL_TAG_THROTTLING_COST_FOLDING_TIME) {}

		void updateCost(double newCost, OpType opType) {
			if (opType == OpType::READ) {
				readCost.setTotal(newCost);
			} else {
				writeCost.setTotal(newCost);
			}
		}

		double getCost() const { return readCost.smoothTotal() + writeCost.smoothTotal(); }
	};

	// Track various statistics per tag, aggregated across all storage servers
	class PerTagStatistics {
		Optional<ThrottleApi::TagQuotaValue> quota;
		HoltLinearSmoother transactionCounter;
		Smoother perClientRate;
		Smoother targetRate;
		double transactionsLastAdded;
		double lastLogged;

	public:
		explicit PerTagStatistics()
		  : transactionCounter(SERVER_KNOBS->GLOBAL_TAG_THROTTLING_TRANSACTION_COUNT_FOLDING_TIME,
		                       SERVER_KNOBS->GLOBAL_TAG_THROTTLING_TRANSACTION_RATE_FOLDING_TIME),
		    perClientRate(SERVER_KNOBS->GLOBAL_TAG_THROTTLING_TARGET_RATE_FOLDING_TIME),
		    targetRate(SERVER_KNOBS->GLOBAL_TAG_THROTTLING_TARGET_RATE_FOLDING_TIME), transactionsLastAdded(now()),
		    lastLogged(0) {}

		Optional<ThrottleApi::TagQuotaValue> getQuota() const { return quota; }

		void setQuota(ThrottleApi::TagQuotaValue quota) { this->quota = quota; }

		void clearQuota() { quota = {}; }

		void addTransactions(int count) {
			transactionsLastAdded = now();
			transactionCounter.addDelta(count);
		}

		double getTransactionRate() const { return transactionCounter.smoothRate(); }

		ClientTagThrottleLimits updateAndGetPerClientLimit(double targetTps) {
			auto newPerClientRate = std::max(
			    SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MIN_RATE,
			    std::min(targetTps, (targetTps / transactionCounter.smoothRate()) * perClientRate.smoothTotal()));
			perClientRate.setTotal(newPerClientRate);
			return ClientTagThrottleLimits(
			    std::max(perClientRate.smoothTotal(), SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MIN_RATE),
			    ClientTagThrottleLimits::NO_EXPIRATION);
		}

		double updateAndGetTargetLimit(double targetTps) {
			targetRate.setTotal(targetTps);
			return targetRate.smoothTotal();
		}

		bool recentTransactionsAdded() const {
			return now() - transactionsLastAdded < SERVER_KNOBS->GLOBAL_TAG_THROTTLING_TAG_EXPIRE_AFTER;
		}

		bool canLog() const { return now() - lastLogged > SERVER_KNOBS->GLOBAL_TAG_THROTTLING_TRACE_INTERVAL; }

		void updateLastLogged() { lastLogged = now(); }
	};

	struct StorageServerInfo {
		Optional<Standalone<StringRef>> zoneId;
		Optional<double> throttlingRatio;
	};

	Database db;
	UID id;
	int maxFallingBehind{ 0 };
	double limitingThreshold{ 0.0 };
	uint64_t throttledTagChangeId{ 0 };
	uint32_t lastBusyTagCount{ 0 };

	std::unordered_map<UID, StorageServerInfo> ssInfos;
	std::unordered_map<TransactionTag, PerTagStatistics> tagStatistics;
	ServerThroughputTracker throughputTracker;

	// For transactions with the provided tag, returns the average cost of all transactions
	// across the cluster. The minimum cost is one page. If the transaction rate is too low,
	// return an empty Optional, because no accurate estimation can be made.
	Optional<double> getAverageTransactionCost(TransactionTag tag, TraceEvent& te) const {
		auto const cost = throughputTracker.getThroughput(tag);
		auto const stats = tryGet(tagStatistics, tag);
		if (!stats.present()) {
			return CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
		}
		auto const transactionRate = stats.get().getTransactionRate();
		te.detail("TransactionRate", transactionRate);
		te.detail("Cost", cost);
		if (transactionRate < SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MIN_RATE) {
			return {};
		} else {
			return std::max(static_cast<double>(CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE), cost / transactionRate);
		}
	}

	Optional<double> getQuota(TransactionTag tag, LimitType limitType) const {
		auto const stats = tryGet(tagStatistics, tag);
		if (!stats.present()) {
			return {};
		}
		auto const quota = stats.get().getQuota();
		if (!quota.present()) {
			return {};
		}
		if (limitType == LimitType::TOTAL) {
			return quota.get().totalQuota;
		} else {
			return quota.get().reservedQuota;
		}
	}

	Optional<double> getLimitingThrottlingRatio(TransactionTag tag) const {
		std::unordered_map<Optional<Standalone<StringRef>>, double> zoneThrottlingRatios;
		for (auto const& [id, ssInfo] : ssInfos) {
			// Ignore storage servers where throttlingId is not significantly contributing
			auto const throughput = throughputTracker.getThroughput(id, tag);
			if (!throughput.present() || throughput.get() < limitingThreshold) {
				continue;
			}

			if (ssInfo.throttlingRatio.present()) {
				auto const [it, inserted] =
				    zoneThrottlingRatios.try_emplace(ssInfo.zoneId, ssInfo.throttlingRatio.get());
				if (!inserted) {
					auto& zoneThrottlingRatio = it->second;
					zoneThrottlingRatio = std::min<double>(zoneThrottlingRatio, ssInfo.throttlingRatio.get());
				}
			}
		}
		if (zoneThrottlingRatios.size() <= maxFallingBehind) {
			return {};
		} else {
			std::priority_queue<double, std::vector<double>> maxHeap;
			ASSERT_GE(maxFallingBehind, 0);
			for (auto const& [_, throttlingRatio] : zoneThrottlingRatios) {
				if (maxHeap.size() < maxFallingBehind + 1) {
					maxHeap.push(throttlingRatio);
				} else if (maxHeap.top() > throttlingRatio) {
					maxHeap.pop();
					maxHeap.push(throttlingRatio);
				}
			}
			return maxHeap.top();
		}
	}

	// Return the limiting transaction rate, aggregated across all storage servers.
	// The limits from the worst maxFallingBehind zones are
	// ignored, because we do not non-workload related issues (e.g. slow disks)
	// to affect tag throttling. If more than maxFallingBehind zones are at
	// or near saturation, this indicates that throttling should take place.
	Optional<double> getLimitingTps(TransactionTag tag) const {
		auto const stats = tryGet(tagStatistics, tag);
		if (!stats.present()) {
			return {};
		}
		auto const limitingThrottlingRatio = getLimitingThrottlingRatio(tag);
		if (!limitingThrottlingRatio.present()) {
			return {};
		}
		return stats.get().getTransactionRate() * limitingThrottlingRatio.get();
	}

	void removeUnseenQuotas(std::unordered_set<TransactionTag> const& tagsWithQuota) {
		for (auto& [tag, stats] : tagStatistics) {
			if (!tagsWithQuota.count(tag)) {
				stats.clearQuota();
			}
		}
	}

																#line 285 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
// This generated class is to be used only via monitorThrottlingChanges()
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class MonitorThrottlingChangesActor>
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class MonitorThrottlingChangesActorState {
															#line 291 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	MonitorThrottlingChangesActorState(GlobalTagThrottlerImpl* const& self) 
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : self(self),
															#line 284 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagsWithQuota()
															#line 300 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this));

	}
	~MonitorThrottlingChangesActorState() 
	{
		fdb_probe_actor_destroy("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 286 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			;
															#line 315 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorThrottlingChangesActorState();
		static_cast<MonitorThrottlingChangesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 287 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		tr = ReadYourWritesTransaction(self->db);
															#line 288 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		;
															#line 347 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
		try {
															#line 290 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 291 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 293 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagsWithQuota.clear();
															#line 294 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(tagQuotaKeys, CLIENT_KNOBS->TOO_MANY);
															#line 294 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 378 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state = 1;
															#line 294 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorThrottlingChangesActor, 0, RangeResult >*>(static_cast<MonitorThrottlingChangesActor*>(this)));
															#line 383 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 308 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TraceEvent("GlobalTagThrottler_MonitoringChangesError", self->id).error(e);
															#line 309 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 309 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 422 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state = 3;
															#line 309 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MonitorThrottlingChangesActor, 2, Void >*>(static_cast<MonitorThrottlingChangesActor*>(this)));
															#line 427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(int loopDepth) 
	{
															#line 295 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		TraceEvent("GlobalTagThrottler_ReadCurrentQuotas", self->id).detail("Size", currentQuotas.size());
															#line 296 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		for( auto const kv : currentQuotas ) {
															#line 297 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			auto const tag = kv.key.removePrefix(tagQuotaPrefix);
															#line 298 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			auto const quota = ThrottleApi::TagQuotaValue::fromValue(kv.value);
															#line 299 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			self->tagStatistics[tag].setQuota(quota);
															#line 300 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagsWithQuota.insert(tag);
															#line 452 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		}
															#line 302 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		self->removeUnseenQuotas(tagsWithQuota);
															#line 303 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		self->removeExpiredTags();
															#line 304 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		++self->throttledTagChangeId;
															#line 305 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(5.0);
															#line 305 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 464 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state = 2;
															#line 305 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorThrottlingChangesActor, 1, Void >*>(static_cast<MonitorThrottlingChangesActor*>(this)));
															#line 469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(RangeResult const& __currentQuotas,int loopDepth) 
	{
															#line 294 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		currentQuotas = __currentQuotas;
															#line 478 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(RangeResult && __currentQuotas,int loopDepth) 
	{
		currentQuotas = std::move(__currentQuotas);
		loopDepth = a_body1loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorThrottlingChangesActor*>(this)->ActorCallback< MonitorThrottlingChangesActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorThrottlingChangesActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorThrottlingChangesActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorThrottlingChangesActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorThrottlingChangesActor*>(this)->ActorCallback< MonitorThrottlingChangesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorThrottlingChangesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MonitorThrottlingChangesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MonitorThrottlingChangesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorThrottlingChangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorThrottlingChangesActor*>(this)->ActorCallback< MonitorThrottlingChangesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorThrottlingChangesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MonitorThrottlingChangesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MonitorThrottlingChangesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottlerImpl* self;
															#line 284 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	std::unordered_set<TransactionTag> tagsWithQuota;
															#line 287 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ReadYourWritesTransaction tr;
															#line 294 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	RangeResult currentQuotas;
															#line 735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via monitorThrottlingChanges()
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class MonitorThrottlingChangesActor final : public Actor<Void>, public ActorCallback< MonitorThrottlingChangesActor, 0, RangeResult >, public ActorCallback< MonitorThrottlingChangesActor, 1, Void >, public ActorCallback< MonitorThrottlingChangesActor, 2, Void >, public FastAllocated<MonitorThrottlingChangesActor>, public MonitorThrottlingChangesActorState<MonitorThrottlingChangesActor> {
															#line 740 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<MonitorThrottlingChangesActor>::operator new;
	using FastAllocated<MonitorThrottlingChangesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14186849513445227776UL, 7381023855897504512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorThrottlingChangesActor, 0, RangeResult >;
friend struct ActorCallback< MonitorThrottlingChangesActor, 1, Void >;
friend struct ActorCallback< MonitorThrottlingChangesActor, 2, Void >;
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	MonitorThrottlingChangesActor(GlobalTagThrottlerImpl* const& self) 
															#line 759 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorThrottlingChangesActorState<MonitorThrottlingChangesActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10553190831330820864UL, 13570587432161133824UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorThrottlingChangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorThrottlingChanges");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorThrottlingChanges", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorThrottlingChangesActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MonitorThrottlingChangesActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MonitorThrottlingChangesActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
[[nodiscard]] static Future<Void> monitorThrottlingChanges( GlobalTagThrottlerImpl* const& self ) {
															#line 283 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new MonitorThrottlingChangesActor(self));
															#line 793 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}

#line 314 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

	Optional<double> getTargetTps(TransactionTag tag, bool& isBusy, TraceEvent& te) {
		auto const limitingTps = getLimitingTps(tag);
		auto const averageTransactionCost = getAverageTransactionCost(tag, te);
		auto const totalQuota = getQuota(tag, LimitType::TOTAL);
		auto const reservedQuota = getQuota(tag, LimitType::RESERVED);
		if (!averageTransactionCost.present() || !totalQuota.present() || !reservedQuota.present()) {
			return {};
		}
		auto const desiredTps = totalQuota.get() / averageTransactionCost.get();
		auto const reservedTps = reservedQuota.get() / averageTransactionCost.get();
		auto const targetTps = getMax(reservedTps, getMin(desiredTps, limitingTps));

		isBusy = limitingTps.present() && limitingTps.get() < desiredTps;

		te.detail("Tag", tag)
		    .detail("TargetTps", targetTps)
		    .detail("AverageTransactionCost", averageTransactionCost)
		    .detail("LimitingTps", limitingTps)
		    .detail("ReservedTps", reservedTps)
		    .detail("DesiredTps", desiredTps)
		    .detail("NumStorageServers", throughputTracker.storageServersTracked())
		    .detail("TotalQuota", totalQuota)
		    .detail("ReservedQuota", reservedQuota);

		return targetTps;
	}

public:
	GlobalTagThrottlerImpl(Database db, UID id, int maxFallingBehind, double limitingThreshold)
	  : db(db), id(id), maxFallingBehind(maxFallingBehind), limitingThreshold(limitingThreshold) {}
	Future<Void> monitorThrottlingChanges() { return monitorThrottlingChanges(this); }
	void addRequests(TransactionTag tag, int count) {
		auto it = tagStatistics.find(tag);
		if (it == tagStatistics.end()) {
			if (tagStatistics.size() == SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MAX_TAGS_TRACKED) {
				CODE_PROBE(true,
				           "Global tag throttler ignoring transactions because maximum number of trackable tags has "
				           "been reached");
				TraceEvent("GlobalTagThrottler_IgnoringRequests")
				    .suppressFor(60.0)
				    .detail("Tag", tag)
				    .detail("Count", count);
			} else {
				tagStatistics[tag].addTransactions(static_cast<double>(count));
			}
		} else {
			it->second.addTransactions(static_cast<double>(count));
		}
	}
	uint64_t getThrottledTagChangeId() const { return throttledTagChangeId; }

	TransactionTagMap<double> getProxyRates(int numProxies) {
		TransactionTagMap<double> result;
		lastBusyTagCount = 0;

		for (auto& [tag, stats] : tagStatistics) {
			// Currently there is no differentiation between batch priority and default priority transactions
			TraceEvent te("GlobalTagThrottler_GotRate", id);
			bool const traceEnabled = stats.canLog();
			if (!traceEnabled) {
				te.disable();
			}
			bool isBusy{ false };
			auto const targetTps = getTargetTps(tag, isBusy, te);
			if (isBusy) {
				++lastBusyTagCount;
			}
			if (targetTps.present()) {
				auto const smoothedTargetTps = stats.updateAndGetTargetLimit(targetTps.get());
				te.detail("SmoothedTargetTps", smoothedTargetTps).detail("NumProxies", numProxies);
				result[tag] = std::max(1.0, smoothedTargetTps / numProxies);
				if (traceEnabled) {
					stats.updateLastLogged();
				}
			} else {
				te.disable();
			}
		}

		return result;
	}

	PrioritizedTransactionTagMap<ClientTagThrottleLimits> getClientRates() {
		PrioritizedTransactionTagMap<ClientTagThrottleLimits> result;
		lastBusyTagCount = 0;

		for (auto& [tag, stats] : tagStatistics) {
			// Currently there is no differentiation between batch priority and default priority transactions
			bool isBusy{ false };
			TraceEvent te("GlobalTagThrottler_GotClientRate", id);
			bool const traceEnabled = stats.canLog();
			if (!traceEnabled) {
				te.disable();
			}
			auto const targetTps = getTargetTps(tag, isBusy, te);

			if (isBusy) {
				++lastBusyTagCount;
			}

			if (targetTps.present()) {
				auto const clientRate = stats.updateAndGetPerClientLimit(targetTps.get());
				result[TransactionPriority::BATCH][tag] = result[TransactionPriority::DEFAULT][tag] = clientRate;
				te.detail("ClientTps", clientRate.tpsRate);
				if (traceEnabled) {
					stats.updateLastLogged();
				}
			} else {
				te.disable();
			}
		}
		return result;
	}

	int64_t autoThrottleCount() const {
		int64_t result{ 0 };
		for (const auto& [tag, stats] : tagStatistics) {
			if (stats.getQuota().present()) {
				++result;
			}
		}
		return result;
	}
	uint32_t busyReadTagCount() const { return lastBusyTagCount; }
	uint32_t busyWriteTagCount() const { return lastBusyTagCount; }
	int64_t manualThrottleCount() const { return 0; }

	void updateThrottling(Map<UID, StorageQueueInfo> const& sqInfos) {
		for (auto it = sqInfos.begin(); it != sqInfos.end(); ++it) {
			auto& ss = it->value;
			auto& ssInfo = ssInfos[ss.id];
			ssInfo.throttlingRatio =
			    ss.getTagThrottlingRatio(SERVER_KNOBS->AUTO_TAG_THROTTLE_STORAGE_QUEUE_BYTES,
			                             SERVER_KNOBS->AUTO_TAG_THROTTLE_SPRING_BYTES_STORAGE_SERVER);
			ssInfo.zoneId = ss.locality.zoneId();
		}
		throughputTracker.update(sqInfos);
	}

	void setQuota(TransactionTagRef tag, ThrottleApi::TagQuotaValue const& tagQuotaValue) {
		tagStatistics[tag].setQuota(tagQuotaValue);
	}

	void removeQuota(TransactionTagRef tag) { tagStatistics[tag].clearQuota(); }

	void removeExpiredTags() {
		for (auto it = tagStatistics.begin(); it != tagStatistics.end();) {
			const auto& [tag, stats] = *it;
			if (!stats.recentTransactionsAdded()) {
				throughputTracker.removeTag(tag);
				it = tagStatistics.erase(it);
			} else {
				++it;
			}
		}
	}

	uint32_t tagsTracked() const { return tagStatistics.size(); }
};

GlobalTagThrottler::GlobalTagThrottler(Database db, UID id, int maxFallingBehind, double limitingThreshold)
  : impl(PImpl<GlobalTagThrottlerImpl>::create(db, id, maxFallingBehind, limitingThreshold)) {}

GlobalTagThrottler::~GlobalTagThrottler() = default;

Future<Void> GlobalTagThrottler::monitorThrottlingChanges() {
	return impl->monitorThrottlingChanges();
}
void GlobalTagThrottler::addRequests(TransactionTag tag, int count) {
	return impl->addRequests(tag, count);
}
uint64_t GlobalTagThrottler::getThrottledTagChangeId() const {
	return impl->getThrottledTagChangeId();
}
PrioritizedTransactionTagMap<ClientTagThrottleLimits> GlobalTagThrottler::getClientRates() {
	return impl->getClientRates();
}
TransactionTagMap<double> GlobalTagThrottler::getProxyRates(int numProxies) {
	return impl->getProxyRates(numProxies);
}
int64_t GlobalTagThrottler::autoThrottleCount() const {
	return impl->autoThrottleCount();
}
uint32_t GlobalTagThrottler::busyReadTagCount() const {
	return impl->busyReadTagCount();
}
uint32_t GlobalTagThrottler::busyWriteTagCount() const {
	return impl->busyWriteTagCount();
}
int64_t GlobalTagThrottler::manualThrottleCount() const {
	return impl->manualThrottleCount();
}
bool GlobalTagThrottler::isAutoThrottlingEnabled() const {
	return true;
}
void GlobalTagThrottler::updateThrottling(Map<UID, StorageQueueInfo> const& sqInfos) {
	return impl->updateThrottling(sqInfos);
}

void GlobalTagThrottler::setQuota(TransactionTagRef tag, ThrottleApi::TagQuotaValue const& tagQuotaValue) {
	return impl->setQuota(tag, tagQuotaValue);
}

void GlobalTagThrottler::removeQuota(TransactionTagRef tag) {
	return impl->removeQuota(tag);
}

uint32_t GlobalTagThrottler::tagsTracked() const {
	return impl->tagsTracked();
}

void GlobalTagThrottler::removeExpiredTags() {
	return impl->removeExpiredTags();
}

namespace {

enum class LimitType { RESERVED, TOTAL };
enum class OpType { READ, WRITE };

Optional<double> getTPSLimit(GlobalTagThrottler& globalTagThrottler, TransactionTag tag) {
	auto clientRates = globalTagThrottler.getClientRates();
	auto it1 = clientRates.find(TransactionPriority::DEFAULT);
	if (it1 != clientRates.end()) {
		auto it2 = it1->second.find(tag);
		if (it2 != it1->second.end()) {
			return it2->second.tpsRate;
		}
	}
	return {};
}

class MockStorageServer {
	class Cost {
		Smoother smoother;

	public:
		Cost() : smoother(SERVER_KNOBS->GLOBAL_TAG_THROTTLING_COST_FOLDING_TIME) {}
		Cost& operator+=(double delta) {
			smoother.addDelta(delta);
			return *this;
		}
		double smoothRate() const { return smoother.smoothRate(); }
	};

	UID id;
	// pages/second that this storage server can handle
	double capacity;
	std::map<TransactionTag, Cost> readCosts, writeCosts;
	Cost totalReadCost, totalWriteCost;

public:
	explicit MockStorageServer(UID id, double capacity) : id(id), capacity(capacity) { ASSERT_GT(capacity, 0); }
	void addReadCost(TransactionTag tag, double cost) {
		readCosts[tag] += cost;
		totalReadCost += cost;
	}
	void addWriteCost(TransactionTag tag, double cost) {
		writeCosts[tag] += cost;
		totalWriteCost += cost;
	}

	void setCapacity(double value) { capacity = value; }

	StorageQueueInfo getStorageQueueInfo() const {
		StorageQueueInfo result(id, LocalityData({}, Value(id.toString()), {}, {}));
		for (const auto& [tag, readCost] : readCosts) {
			double fractionalBusyness{ 0.0 }; // unused for global tag throttling
			result.busiestReadTags.emplace_back(tag, readCost.smoothRate(), fractionalBusyness);
		}
		for (const auto& [tag, writeCost] : writeCosts) {
			double fractionalBusyness{ 0.0 }; // unused for global tag throttling
			result.busiestWriteTags.emplace_back(tag, writeCost.smoothRate(), fractionalBusyness);
		}
		result.lastReply.bytesInput = ((totalReadCost.smoothRate() + totalWriteCost.smoothRate()) /
		                               (capacity * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE)) *
		                              SERVER_KNOBS->AUTO_TAG_THROTTLE_STORAGE_QUEUE_BYTES;
		result.valid = true;
		return result;
	}
};

class StorageServerCollection {
	std::vector<MockStorageServer> storageServers;

public:
	StorageServerCollection(size_t size, double capacity) {
		ASSERT_GT(size, 0);
		storageServers.reserve(size);
		for (int i = 0; i < size; ++i) {
			storageServers.emplace_back(UID(i, i), capacity);
		}
	}

	void addCost(TransactionTag tag,
	             double pagesPerSecond,
	             std::vector<int> const& storageServerIndices,
	             OpType opType) {
		if (storageServerIndices.empty()) {
			auto costPerSS = CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE * (pagesPerSecond / storageServers.size());
			if (opType == OpType::WRITE) {
				costPerSS *= CLIENT_KNOBS->GLOBAL_TAG_THROTTLING_RW_FUNGIBILITY_RATIO;
			}
			for (auto& storageServer : storageServers) {
				if (opType == OpType::READ) {
					storageServer.addReadCost(tag, costPerSS);
				} else {
					storageServer.addWriteCost(tag, costPerSS);
				}
			}
		} else {
			auto const costPerSS =
			    CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE * (pagesPerSecond / storageServerIndices.size());
			for (auto i : storageServerIndices) {
				if (opType == OpType::READ) {
					storageServers[i].addReadCost(tag, costPerSS);
				} else {
					storageServers[i].addWriteCost(tag, costPerSS);
				}
			}
		}
	}

	void setCapacity(int index, double value) { storageServers[index].setCapacity(value); }

	Map<UID, StorageQueueInfo> getStorageQueueInfos() const {
		Map<UID, StorageQueueInfo> result;
		for (const auto& storageServer : storageServers) {
			auto const sqInfo = storageServer.getStorageQueueInfo();
			result.insert(mapPair(sqInfo.id, sqInfo));
		}
		return result;
	}
};

															#line 1133 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
// This generated class is to be used only via runClient()
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class RunClientActor>
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class RunClientActorState {
															#line 1139 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	RunClientActorState(GlobalTagThrottler* const& globalTagThrottler,StorageServerCollection* const& storageServers,TransactionTag const& tag,double const& tpsRate,double const& costPerTransaction,OpType const& opType,std::vector<int> const& storageServerIndices = std::vector<int>()) 
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : globalTagThrottler(globalTagThrottler),
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(storageServers),
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tag(tag),
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tpsRate(tpsRate),
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   costPerTransaction(costPerTransaction),
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   opType(opType),
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServerIndices(storageServerIndices)
															#line 1158 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("runClient", reinterpret_cast<unsigned long>(this));

	}
	~RunClientActorState() 
	{
		fdb_probe_actor_destroy("runClient", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 657 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			;
															#line 1173 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunClientActorState();
		static_cast<RunClientActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 658 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		auto tpsLimit = getTPSLimit(*globalTagThrottler, tag);
															#line 659 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		enforcedRate = tpsLimit.present() ? std::min<double>(tpsRate, tpsLimit.get()) : tpsRate;
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(1 / enforcedRate);
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<RunClientActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1209 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RunClientActor*>(this)->actor_wait_state = 1;
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunClientActor, 0, Void >*>(static_cast<RunClientActor*>(this)));
															#line 1214 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 661 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		storageServers->addCost(tag, costPerTransaction, storageServerIndices, opType);
															#line 662 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler->addRequests(tag, 1);
															#line 1225 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 661 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		storageServers->addCost(tag, costPerTransaction, storageServerIndices, opType);
															#line 662 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler->addRequests(tag, 1);
															#line 1236 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunClientActor*>(this)->actor_wait_state > 0) static_cast<RunClientActor*>(this)->actor_wait_state = 0;
		static_cast<RunClientActor*>(this)->ActorCallback< RunClientActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunClientActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunClientActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunClientActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runClient", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler* globalTagThrottler;
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection* storageServers;
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	TransactionTag tag;
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	double tpsRate;
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	double costPerTransaction;
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	OpType opType;
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	std::vector<int> storageServerIndices;
															#line 659 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	double enforcedRate;
															#line 1332 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via runClient()
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class RunClientActor final : public Actor<Void>, public ActorCallback< RunClientActor, 0, Void >, public FastAllocated<RunClientActor>, public RunClientActorState<RunClientActor> {
															#line 1337 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<RunClientActor>::operator new;
	using FastAllocated<RunClientActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1184647659419586560UL, 13901978466263382784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunClientActor, 0, Void >;
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	RunClientActor(GlobalTagThrottler* const& globalTagThrottler,StorageServerCollection* const& storageServers,TransactionTag const& tag,double const& tpsRate,double const& costPerTransaction,OpType const& opType,std::vector<int> const& storageServerIndices = std::vector<int>()) 
															#line 1354 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   RunClientActorState<RunClientActor>(globalTagThrottler, storageServers, tag, tpsRate, costPerTransaction, opType, storageServerIndices),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runClient", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17320794064144289280UL, 14301634185507036928UL);
		ActorExecutionContextHelper __helper(static_cast<RunClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runClient");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runClient", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunClientActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
[[nodiscard]] Future<Void> runClient( GlobalTagThrottler* const& globalTagThrottler, StorageServerCollection* const& storageServers, TransactionTag const& tag, double const& tpsRate, double const& costPerTransaction, OpType const& opType, std::vector<int> const& storageServerIndices = std::vector<int>() ) {
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new RunClientActor(globalTagThrottler, storageServers, tag, tpsRate, costPerTransaction, opType, storageServerIndices));
															#line 1386 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}

#line 665 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

															#line 1391 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
// This generated class is to be used only via monitorActor()
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class Check, class MonitorActorActor>
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class MonitorActorActorState {
															#line 1397 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	MonitorActorActorState(GlobalTagThrottler* const& globalTagThrottler,Check const& check) 
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : globalTagThrottler(globalTagThrottler),
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   check(check),
															#line 668 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   successes(0)
															#line 1408 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorActor", reinterpret_cast<unsigned long>(this));

	}
	~MonitorActorActorState() 
	{
		fdb_probe_actor_destroy("monitorActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 669 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			;
															#line 1423 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorActorActorState();
		static_cast<MonitorActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 670 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 670 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<MonitorActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1455 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MonitorActorActor*>(this)->actor_wait_state = 1;
															#line 670 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorActorActor, 0, Void >*>(static_cast<MonitorActorActor*>(this)));
															#line 1460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 671 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (check(*globalTagThrottler))
															#line 1469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		{
															#line 674 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (++successes == 10)
															#line 1473 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			{
															#line 675 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
				if (!static_cast<MonitorActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MonitorActorActorState(); static_cast<MonitorActorActor*>(this)->destroy(); return 0; }
															#line 1477 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
				new (&static_cast<MonitorActorActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~MonitorActorActorState();
				static_cast<MonitorActorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
		else
		{
															#line 678 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			successes = 0;
															#line 1488 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 671 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (check(*globalTagThrottler))
															#line 1498 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		{
															#line 674 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (++successes == 10)
															#line 1502 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			{
															#line 675 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
				if (!static_cast<MonitorActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MonitorActorActorState(); static_cast<MonitorActorActor*>(this)->destroy(); return 0; }
															#line 1506 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
				new (&static_cast<MonitorActorActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~MonitorActorActorState();
				static_cast<MonitorActorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
		else
		{
															#line 678 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			successes = 0;
															#line 1517 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorActorActor*>(this)->actor_wait_state > 0) static_cast<MonitorActorActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorActorActor*>(this)->ActorCallback< MonitorActorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorActor", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler* globalTagThrottler;
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Check check;
															#line 668 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	int successes;
															#line 1604 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via monitorActor()
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class Check>
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class MonitorActorActor final : public Actor<Void>, public ActorCallback< MonitorActorActor<Check>, 0, Void >, public FastAllocated<MonitorActorActor<Check>>, public MonitorActorActorState<Check, MonitorActorActor<Check>> {
															#line 1611 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<MonitorActorActor<Check>>::operator new;
	using FastAllocated<MonitorActorActor<Check>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10240689717627382528UL, 4367424148545790720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorActorActor<Check>, 0, Void >;
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	MonitorActorActor(GlobalTagThrottler* const& globalTagThrottler,Check const& check) 
															#line 1628 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorActorActorState<Check, MonitorActorActor<Check>>(globalTagThrottler, check),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(673194977085094400UL, 11501339878130125056UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorActorActor<Check>, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class Check>
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
[[nodiscard]] Future<Void> monitorActor( GlobalTagThrottler* const& globalTagThrottler, Check const& check ) {
															#line 666 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new MonitorActorActor<Check>(globalTagThrottler, check));
															#line 1662 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}

#line 682 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

bool isNear(double a, double b) {
	return abs(a - b) < 0.1 * a;
}

bool isNear(Optional<double> a, Optional<double> b) {
	if (a.present()) {
		return b.present() && isNear(a.get(), b.get());
	} else {
		return !b.present();
	}
}

bool targetRateIsNear(GlobalTagThrottler& globalTagThrottler, TransactionTag tag, Optional<double> expected) {
	Optional<double> rate;
	auto const targetRates = globalTagThrottler.getProxyRates(1);
	auto it = targetRates.find(tag);
	if (it != targetRates.end()) {
		rate = it->second;
	}
	TraceEvent("GlobalTagThrottling_RateMonitor")
	    .detail("Tag", tag)
	    .detail("CurrentTPSRate", rate)
	    .detail("ExpectedTPSRate", expected);
	return isNear(rate, expected);
}

bool totalTargetRateIsNear(GlobalTagThrottler& globalTagThrottler, double expected) {
	auto const targetRates = globalTagThrottler.getProxyRates(1);
	double targetRateSum = 0.0;
	for (auto const& [_, targetRate] : targetRates) {
		targetRateSum += targetRate;
	}
	TraceEvent("GlobalTagThrottling_TotalRateMonitor")
	    .detail("NumTags", targetRates.size())
	    .detail("CurrentTotalTPSRate", targetRateSum)
	    .detail("ExpectedTotalTPSRate", expected);
	return isNear(targetRateSum, expected);
}

bool clientRateIsNear(GlobalTagThrottler& globalTagThrottler, TransactionTag tag, Optional<double> expected) {
	Optional<double> rate;
	auto clientRates = globalTagThrottler.getClientRates();
	auto it1 = clientRates.find(TransactionPriority::DEFAULT);
	if (it1 != clientRates.end()) {
		auto it2 = it1->second.find(tag);
		if (it2 != it1->second.end()) {
			rate = it2->second.tpsRate;
		}
	}
	TraceEvent("GlobalTagThrottling_ClientRateMonitor")
	    .detail("Tag", tag)
	    .detail("CurrentTPSRate", rate)
	    .detail("ExpectedTPSRate", expected);
	return isNear(rate, expected);
}

															#line 1723 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
// This generated class is to be used only via updateGlobalTagThrottler()
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class UpdateGlobalTagThrottlerActor>
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class UpdateGlobalTagThrottlerActorState {
															#line 1729 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UpdateGlobalTagThrottlerActorState(GlobalTagThrottler* const& globalTagThrottler,StorageServerCollection const* const& storageServers) 
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : globalTagThrottler(globalTagThrottler),
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(storageServers)
															#line 1738 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this));

	}
	~UpdateGlobalTagThrottlerActorState() 
	{
		fdb_probe_actor_destroy("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 741 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			;
															#line 1753 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateGlobalTagThrottlerActorState();
		static_cast<UpdateGlobalTagThrottlerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 742 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 742 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<UpdateGlobalTagThrottlerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1785 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<UpdateGlobalTagThrottlerActor*>(this)->actor_wait_state = 1;
															#line 742 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateGlobalTagThrottlerActor, 0, Void >*>(static_cast<UpdateGlobalTagThrottlerActor*>(this)));
															#line 1790 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		auto const storageQueueInfos = storageServers->getStorageQueueInfos();
															#line 744 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler->updateThrottling(storageServers->getStorageQueueInfos());
															#line 1801 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 743 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		auto const storageQueueInfos = storageServers->getStorageQueueInfos();
															#line 744 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler->updateThrottling(storageServers->getStorageQueueInfos());
															#line 1812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateGlobalTagThrottlerActor*>(this)->actor_wait_state > 0) static_cast<UpdateGlobalTagThrottlerActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateGlobalTagThrottlerActor*>(this)->ActorCallback< UpdateGlobalTagThrottlerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateGlobalTagThrottlerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGlobalTagThrottlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateGlobalTagThrottlerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGlobalTagThrottlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateGlobalTagThrottlerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGlobalTagThrottlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler* globalTagThrottler;
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection const* storageServers;
															#line 1896 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via updateGlobalTagThrottler()
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class UpdateGlobalTagThrottlerActor final : public Actor<Void>, public ActorCallback< UpdateGlobalTagThrottlerActor, 0, Void >, public FastAllocated<UpdateGlobalTagThrottlerActor>, public UpdateGlobalTagThrottlerActorState<UpdateGlobalTagThrottlerActor> {
															#line 1901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<UpdateGlobalTagThrottlerActor>::operator new;
	using FastAllocated<UpdateGlobalTagThrottlerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7528416839084440576UL, 15846966554806137600UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateGlobalTagThrottlerActor, 0, Void >;
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UpdateGlobalTagThrottlerActor(GlobalTagThrottler* const& globalTagThrottler,StorageServerCollection const* const& storageServers) 
															#line 1918 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateGlobalTagThrottlerActorState<UpdateGlobalTagThrottlerActor>(globalTagThrottler, storageServers),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12903385154211538688UL, 13112762669264635904UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGlobalTagThrottlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateGlobalTagThrottler");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateGlobalTagThrottler", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateGlobalTagThrottlerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
[[nodiscard]] Future<Void> updateGlobalTagThrottler( GlobalTagThrottler* const& globalTagThrottler, StorageServerCollection const* const& storageServers ) {
															#line 739 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new UpdateGlobalTagThrottlerActor(globalTagThrottler, storageServers));
															#line 1950 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}

#line 747 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

GlobalTagThrottler getTestGlobalTagThrottler(int maxFallingBehind = 0) {
	return GlobalTagThrottler(
	    Database{}, UID{}, maxFallingBehind, /*limitingThreshold=*/CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
}

} // namespace

// 10 storage servers can handle 100 pages/second each.
// Total quota set to 100 pages/second.
// Client attempts 5 6-byte read transactions per second.
// Limit should adjust to allow 100/6 transactions per second.
															#line 1966 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase759()
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase759Actor>
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase759ActorState {
															#line 1973 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase759ActorState(UnitTestParameters const& params) 
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 760 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 761 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100)
															#line 1984 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase759", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase759ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase759", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 762 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 763 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 764 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 765 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 766 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 5.0, 6.0, OpType::READ);
															#line 767 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag](auto& gtt) { return targetRateIsNear(gtt, testTag, 100.0 / 6.0); });
															#line 769 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 770 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 770 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase759Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2015 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase759Actor*>(this)->actor_wait_state = 1;
															#line 770 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase759Actor, 0, Void >*>(static_cast<FlowTestCase759Actor*>(this)));
															#line 2020 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase759ActorState();
		static_cast<FlowTestCase759Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 771 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase759Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase759ActorState(); static_cast<FlowTestCase759Actor*>(this)->destroy(); return 0; }
															#line 2043 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase759Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase759ActorState();
		static_cast<FlowTestCase759Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 771 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase759Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase759ActorState(); static_cast<FlowTestCase759Actor*>(this)->destroy(); return 0; }
															#line 2055 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase759Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase759ActorState();
		static_cast<FlowTestCase759Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase759Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase759Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase759Actor*>(this)->ActorCallback< FlowTestCase759Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase759Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase759", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase759Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase759", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase759Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase759", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase759Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase759", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase759Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase759", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase759Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase759", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 760 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 761 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 766 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 767 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 769 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 2150 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase759()
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase759Actor final : public Actor<Void>, public ActorCallback< FlowTestCase759Actor, 0, Void >, public FastAllocated<FlowTestCase759Actor>, public FlowTestCase759ActorState<FlowTestCase759Actor> {
															#line 2155 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase759Actor>::operator new;
	using FastAllocated<FlowTestCase759Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2426131005993694976UL, 880784806206811136UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase759Actor, 0, Void >;
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase759Actor(UnitTestParameters const& params) 
															#line 2172 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase759ActorState<FlowTestCase759Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase759", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1781295376941480704UL, 8841052743729362432UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase759Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase759");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase759", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase759Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase759( UnitTestParameters const& params ) {
															#line 759 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase759Actor(params));
															#line 2205 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase759, "/GlobalTagThrottler/Simple")

#line 773 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 100 pages/second each.
// Total quota set to 100 pages/second.
// Client attempts 5 6-byte write transactions per second.
// Limit should adjust to allow 100/(6*<fungibility_ratio>) transactions per second.
															#line 2215 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase778()
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase778Actor>
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase778ActorState {
															#line 2222 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase778ActorState(UnitTestParameters const& params) 
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 780 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100)
															#line 2233 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase778", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase778ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase778", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 781 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 782 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 783 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 784 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 785 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 5.0, 6.0, OpType::WRITE);
															#line 786 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag](auto& gtt) { return targetRateIsNear(gtt, testTag, 100.0 / (6.0 * CLIENT_KNOBS->GLOBAL_TAG_THROTTLING_RW_FUNGIBILITY_RATIO)); });
															#line 790 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 791 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 791 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase778Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2264 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase778Actor*>(this)->actor_wait_state = 1;
															#line 791 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase778Actor, 0, Void >*>(static_cast<FlowTestCase778Actor*>(this)));
															#line 2269 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase778ActorState();
		static_cast<FlowTestCase778Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 792 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase778Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase778ActorState(); static_cast<FlowTestCase778Actor*>(this)->destroy(); return 0; }
															#line 2292 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase778Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase778ActorState();
		static_cast<FlowTestCase778Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 792 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase778Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase778ActorState(); static_cast<FlowTestCase778Actor*>(this)->destroy(); return 0; }
															#line 2304 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase778Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase778ActorState();
		static_cast<FlowTestCase778Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase778Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase778Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase778Actor*>(this)->ActorCallback< FlowTestCase778Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase778Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase778", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase778Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase778", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase778Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase778", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase778Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase778", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase778Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase778", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase778Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase778", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 780 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 785 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 786 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 790 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 2399 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase778()
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase778Actor final : public Actor<Void>, public ActorCallback< FlowTestCase778Actor, 0, Void >, public FastAllocated<FlowTestCase778Actor>, public FlowTestCase778ActorState<FlowTestCase778Actor> {
															#line 2404 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase778Actor>::operator new;
	using FastAllocated<FlowTestCase778Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6879995330017923072UL, 2307317141494754560UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase778Actor, 0, Void >;
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase778Actor(UnitTestParameters const& params) 
															#line 2421 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase778ActorState<FlowTestCase778Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase778", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7403420887684603904UL, 11117729944543677184UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase778Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase778");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase778", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase778Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase778( UnitTestParameters const& params ) {
															#line 778 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase778Actor(params));
															#line 2454 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase778, "/GlobalTagThrottler/WriteThrottling")

#line 794 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 100 pages/second each.
// Total quota set to 100 pages/second for each tag.
// 2 clients each attempt 5 6-byte read transactions per second.
// Both limits should adjust to allow 100/6 transactions per second.
															#line 2464 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase799()
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase799Actor>
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase799ActorState {
															#line 2471 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase799ActorState(UnitTestParameters const& params) 
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 801 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100)
															#line 2482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase799", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase799ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase799", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 802 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 803 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag1 = "sampleTag1"_sr;
															#line 804 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag2 = "sampleTag2"_sr;
															#line 805 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 806 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag1, tagQuotaValue);
															#line 807 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag2, tagQuotaValue);
															#line 808 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures = std::vector<Future<Void>>();
															#line 809 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitorFutures = std::vector<Future<Void>>();
															#line 810 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(runClient(&globalTagThrottler, &storageServers, testTag1, 5.0, 6.0, OpType::READ));
															#line 811 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(runClient(&globalTagThrottler, &storageServers, testTag2, 5.0, 6.0, OpType::READ));
															#line 812 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(updateGlobalTagThrottler(&globalTagThrottler, &storageServers));
															#line 813 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag1, testTag2](auto& gtt) { return targetRateIsNear(gtt, testTag1, 100.0 / 6.0) && targetRateIsNear(gtt, testTag2, 100.0 / 6.0); });
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(waitForAny(futures) || monitor, 600.0);
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase799Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2523 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase799Actor*>(this)->actor_wait_state = 1;
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase799Actor, 0, Void >*>(static_cast<FlowTestCase799Actor*>(this)));
															#line 2528 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase799ActorState();
		static_cast<FlowTestCase799Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 817 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase799Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase799ActorState(); static_cast<FlowTestCase799Actor*>(this)->destroy(); return 0; }
															#line 2551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase799Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase799ActorState();
		static_cast<FlowTestCase799Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 817 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase799Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase799ActorState(); static_cast<FlowTestCase799Actor*>(this)->destroy(); return 0; }
															#line 2563 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase799Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase799ActorState();
		static_cast<FlowTestCase799Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase799Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase799Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase799Actor*>(this)->ActorCallback< FlowTestCase799Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase799Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase799", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase799Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase799", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase799Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase799", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase799Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase799", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase799Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase799", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase799Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase799", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 801 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 808 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	std::vector<Future<Void>> futures;
															#line 809 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	std::vector<Future<Void>> monitorFutures;
															#line 813 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 2658 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase799()
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase799Actor final : public Actor<Void>, public ActorCallback< FlowTestCase799Actor, 0, Void >, public FastAllocated<FlowTestCase799Actor>, public FlowTestCase799ActorState<FlowTestCase799Actor> {
															#line 2663 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase799Actor>::operator new;
	using FastAllocated<FlowTestCase799Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11536073867488851712UL, 4703385197996388864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase799Actor, 0, Void >;
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase799Actor(UnitTestParameters const& params) 
															#line 2680 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase799ActorState<FlowTestCase799Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase799", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6884708122001271552UL, 1784450062992275968UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase799Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase799");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase799", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase799Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase799( UnitTestParameters const& params ) {
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase799Actor(params));
															#line 2713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase799, "/GlobalTagThrottler/MultiTagThrottling")

#line 819 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 100 pages/second each.
// Total quota set to 100 pages/second.
// Client attempts 20 10-byte read transactions per second.
// Limit should adjust to allow 100/10 transactions per second.
															#line 2723 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase824()
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase824Actor>
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase824ActorState {
															#line 2730 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase824ActorState(UnitTestParameters const& params) 
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 825 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 826 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100)
															#line 2741 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase824", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase824ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase824", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 827 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 828 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 829 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 830 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 831 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 20.0, 10.0, OpType::READ);
															#line 832 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag](auto& gtt) { return targetRateIsNear(gtt, testTag, 10.0); });
															#line 834 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 835 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 835 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase824Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2772 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase824Actor*>(this)->actor_wait_state = 1;
															#line 835 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase824Actor, 0, Void >*>(static_cast<FlowTestCase824Actor*>(this)));
															#line 2777 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase824ActorState();
		static_cast<FlowTestCase824Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 836 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase824Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase824ActorState(); static_cast<FlowTestCase824Actor*>(this)->destroy(); return 0; }
															#line 2800 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase824Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase824ActorState();
		static_cast<FlowTestCase824Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 836 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase824Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase824ActorState(); static_cast<FlowTestCase824Actor*>(this)->destroy(); return 0; }
															#line 2812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase824Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase824ActorState();
		static_cast<FlowTestCase824Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase824Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase824Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase824Actor*>(this)->ActorCallback< FlowTestCase824Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase824Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase824", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase824Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase824", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase824Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase824", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase824Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase824", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase824Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase824", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase824Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase824", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 825 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 826 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 831 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 832 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 834 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 2907 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase824()
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase824Actor final : public Actor<Void>, public ActorCallback< FlowTestCase824Actor, 0, Void >, public FastAllocated<FlowTestCase824Actor>, public FlowTestCase824ActorState<FlowTestCase824Actor> {
															#line 2912 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase824Actor>::operator new;
	using FastAllocated<FlowTestCase824Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(495468991815889152UL, 3557051788484495616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase824Actor, 0, Void >;
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase824Actor(UnitTestParameters const& params) 
															#line 2929 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase824ActorState<FlowTestCase824Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase824", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10781417912220723456UL, 1708651621433870848UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase824Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase824");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase824", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase824Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase824( UnitTestParameters const& params ) {
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase824Actor(params));
															#line 2962 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase824, "/GlobalTagThrottler/AttemptWorkloadAboveQuota")

#line 838 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 100 pages/second each.
// Total quota set to 100 pages/second.
// 2 clients each attempt 5 6-byte transactions per second.
// Limit should adjust to allow 100/6 transactions per second.
															#line 2972 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase843()
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase843Actor>
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase843ActorState {
															#line 2979 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase843ActorState(UnitTestParameters const& params) 
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 844 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 845 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100)
															#line 2990 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase843", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase843ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase843", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 846 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 847 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 848 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 849 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 850 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 5.0, 6.0, OpType::READ);
															#line 851 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client2 = runClient(&globalTagThrottler, &storageServers, testTag, 5.0, 6.0, OpType::READ);
															#line 852 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag](auto& gtt) { return targetRateIsNear(gtt, testTag, 100.0 / 6.0) && clientRateIsNear(gtt, testTag, 100.0 / 6.0); });
															#line 855 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 856 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || client2 || updater, 600.0);
															#line 856 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase843Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase843Actor*>(this)->actor_wait_state = 1;
															#line 856 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase843Actor, 0, Void >*>(static_cast<FlowTestCase843Actor*>(this)));
															#line 3028 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase843ActorState();
		static_cast<FlowTestCase843Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 857 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase843Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase843ActorState(); static_cast<FlowTestCase843Actor*>(this)->destroy(); return 0; }
															#line 3051 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase843Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase843ActorState();
		static_cast<FlowTestCase843Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 857 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase843Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase843ActorState(); static_cast<FlowTestCase843Actor*>(this)->destroy(); return 0; }
															#line 3063 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase843Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase843ActorState();
		static_cast<FlowTestCase843Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase843Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase843Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase843Actor*>(this)->ActorCallback< FlowTestCase843Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase843Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase843", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase843Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase843", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase843Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase843", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase843Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase843", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase843Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase843", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase843Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase843", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 844 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 845 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 850 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 851 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client2;
															#line 852 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 855 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 3160 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase843()
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase843Actor final : public Actor<Void>, public ActorCallback< FlowTestCase843Actor, 0, Void >, public FastAllocated<FlowTestCase843Actor>, public FlowTestCase843ActorState<FlowTestCase843Actor> {
															#line 3165 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase843Actor>::operator new;
	using FastAllocated<FlowTestCase843Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14135564429543582976UL, 11698509236397068800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase843Actor, 0, Void >;
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase843Actor(UnitTestParameters const& params) 
															#line 3182 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase843ActorState<FlowTestCase843Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase843", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8828930445531627264UL, 638537177824591360UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase843Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase843");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase843", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase843Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase843( UnitTestParameters const& params ) {
															#line 843 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase843Actor(params));
															#line 3215 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase843, "/GlobalTagThrottler/MultiClientThrottling")

#line 859 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 100 pages/second each.
// Total quota set to 100 pages/second.
// 2 clients each attempt 20 10-page transactions per second.
// Target rate should adjust to allow 100/10 transactions per second.
// Each client is throttled to only perform (100/10)/2 transactions per second.
															#line 3226 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase865()
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase865Actor>
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase865ActorState {
															#line 3233 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase865ActorState(UnitTestParameters const& params) 
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 866 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100)
															#line 3244 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase865", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase865ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase865", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 868 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 869 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 870 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 871 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 872 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 20.0, 10.0, OpType::READ);
															#line 873 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client2 = runClient(&globalTagThrottler, &storageServers, testTag, 20.0, 10.0, OpType::READ);
															#line 874 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag](auto& gtt) { return targetRateIsNear(gtt, testTag, 10.0) && clientRateIsNear(gtt, testTag, 5.0); });
															#line 877 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 878 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 878 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase865Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3277 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase865Actor*>(this)->actor_wait_state = 1;
															#line 878 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase865Actor, 0, Void >*>(static_cast<FlowTestCase865Actor*>(this)));
															#line 3282 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase865ActorState();
		static_cast<FlowTestCase865Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 879 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase865Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase865ActorState(); static_cast<FlowTestCase865Actor*>(this)->destroy(); return 0; }
															#line 3305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase865Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase865ActorState();
		static_cast<FlowTestCase865Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 879 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase865Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase865ActorState(); static_cast<FlowTestCase865Actor*>(this)->destroy(); return 0; }
															#line 3317 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase865Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase865ActorState();
		static_cast<FlowTestCase865Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase865Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase865Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase865Actor*>(this)->ActorCallback< FlowTestCase865Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase865Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase865", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase865Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase865", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase865Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase865", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase865Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase865", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase865Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase865", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase865Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase865", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 866 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 872 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 873 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client2;
															#line 874 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 877 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 3414 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase865()
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase865Actor final : public Actor<Void>, public ActorCallback< FlowTestCase865Actor, 0, Void >, public FastAllocated<FlowTestCase865Actor>, public FlowTestCase865ActorState<FlowTestCase865Actor> {
															#line 3419 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase865Actor>::operator new;
	using FastAllocated<FlowTestCase865Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7602065165334157312UL, 5756658825408898304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase865Actor, 0, Void >;
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase865Actor(UnitTestParameters const& params) 
															#line 3436 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase865ActorState<FlowTestCase865Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase865", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14596960196615233024UL, 17598253760259343360UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase865Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase865");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase865", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase865Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase865( UnitTestParameters const& params ) {
															#line 865 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase865Actor(params));
															#line 3469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase865, "/GlobalTagThrottler/MultiClientThrottling2")

#line 881 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 100 pages/second each.
// Total quota set to 100 pages/second.
// One client attempts 5 5-page read transactions per second.
// Another client attempts 25 5-page read transactions per second.
// Target rate should adjust to allow 100/5 transactions per second.
// This 20 transactions/second limit is split with a distribution of (5, 15) between the 2 clients.
															#line 3481 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase888()
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase888Actor>
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase888ActorState {
															#line 3488 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase888ActorState(UnitTestParameters const& params) 
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 889 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 890 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100)
															#line 3499 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase888", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase888ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase888", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 891 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 892 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 893 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 894 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 895 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 5.0, 5.0, OpType::READ);
															#line 896 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client2 = runClient(&globalTagThrottler, &storageServers, testTag, 25.0, 5.0, OpType::READ);
															#line 897 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag](auto& gtt) { return targetRateIsNear(gtt, testTag, 20.0) && clientRateIsNear(gtt, testTag, 15.0); });
															#line 900 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 901 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 901 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase888Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase888Actor*>(this)->actor_wait_state = 1;
															#line 901 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase888Actor, 0, Void >*>(static_cast<FlowTestCase888Actor*>(this)));
															#line 3537 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase888ActorState();
		static_cast<FlowTestCase888Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 902 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase888Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase888ActorState(); static_cast<FlowTestCase888Actor*>(this)->destroy(); return 0; }
															#line 3560 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase888Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase888ActorState();
		static_cast<FlowTestCase888Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 902 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase888Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase888ActorState(); static_cast<FlowTestCase888Actor*>(this)->destroy(); return 0; }
															#line 3572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase888Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase888ActorState();
		static_cast<FlowTestCase888Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase888Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase888Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase888Actor*>(this)->ActorCallback< FlowTestCase888Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase888Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase888", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase888Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase888", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase888Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase888", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase888Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase888", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase888Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase888", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase888Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase888", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 889 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 890 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 895 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 896 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client2;
															#line 897 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 900 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 3669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase888()
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase888Actor final : public Actor<Void>, public ActorCallback< FlowTestCase888Actor, 0, Void >, public FastAllocated<FlowTestCase888Actor>, public FlowTestCase888ActorState<FlowTestCase888Actor> {
															#line 3674 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase888Actor>::operator new;
	using FastAllocated<FlowTestCase888Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13364455438897252096UL, 11139672562053559296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase888Actor, 0, Void >;
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase888Actor(UnitTestParameters const& params) 
															#line 3691 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase888ActorState<FlowTestCase888Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase888", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15206641667834682368UL, 2432834138882270464UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase888Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase888");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase888", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase888Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase888( UnitTestParameters const& params ) {
															#line 888 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase888Actor(params));
															#line 3724 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase888, "/GlobalTagThrottler/SkewedMultiClientThrottling")

#line 904 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 100 pages/second each.
// Total quota is initially set to 100 pages/second.
// Client attempts 5 6-page transactions per second.
// Test that the tag throttler can reach equilibrium, then adjust to a new equilibrium once the quota is changed
// Target rate should adjust to allow 100/6 transactions per second.
// Total quota is modified to 50 pages/second.
// Target rate should adjust to allow 50/6 transactions per second.
															#line 3737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase912()
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase912Actor>
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase912ActorState {
															#line 3744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase912ActorState(UnitTestParameters const& params) 
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 913 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 914 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100),
															#line 915 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue(),
															#line 916 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   testTag("sampleTag1"_sr)
															#line 3759 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase912", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase912ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase912", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 917 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 918 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 919 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 5.0, 6.0, OpType::READ);
															#line 920 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor( &globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, 100.0 / 6.0); });
															#line 922 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 923 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 923 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase912Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3786 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase912Actor*>(this)->actor_wait_state = 1;
															#line 923 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase912Actor, 0, Void >*>(static_cast<FlowTestCase912Actor*>(this)));
															#line 3791 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase912ActorState();
		static_cast<FlowTestCase912Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 924 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		tagQuotaValue.totalQuota = 50 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 925 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 926 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		monitor = monitorActor(&globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, 50.0 / 6.0); });
															#line 928 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeoutError(monitor || client || updater, 600.0);
															#line 928 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase912Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3822 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase912Actor*>(this)->actor_wait_state = 2;
															#line 928 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase912Actor, 1, Void >*>(static_cast<FlowTestCase912Actor*>(this)));
															#line 3827 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 924 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		tagQuotaValue.totalQuota = 50 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 925 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 926 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		monitor = monitorActor(&globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, 50.0 / 6.0); });
															#line 928 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeoutError(monitor || client || updater, 600.0);
															#line 928 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase912Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3844 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase912Actor*>(this)->actor_wait_state = 2;
															#line 928 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase912Actor, 1, Void >*>(static_cast<FlowTestCase912Actor*>(this)));
															#line 3849 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase912Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase912Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase912Actor*>(this)->ActorCallback< FlowTestCase912Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase912Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase912", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase912Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase912", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase912Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase912", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase912Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase912", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase912Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase912", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase912Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase912", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 929 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase912Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase912ActorState(); static_cast<FlowTestCase912Actor*>(this)->destroy(); return 0; }
															#line 3933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase912Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase912ActorState();
		static_cast<FlowTestCase912Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 929 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase912Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase912ActorState(); static_cast<FlowTestCase912Actor*>(this)->destroy(); return 0; }
															#line 3945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase912Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase912ActorState();
		static_cast<FlowTestCase912Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase912Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase912Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase912Actor*>(this)->ActorCallback< FlowTestCase912Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase912Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase912", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase912Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase912", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase912Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase912", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase912Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase912", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase912Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase912", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase912Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase912", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 913 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 914 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 915 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 916 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	TransactionTag testTag;
															#line 919 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 920 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 922 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 4044 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase912()
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase912Actor final : public Actor<Void>, public ActorCallback< FlowTestCase912Actor, 0, Void >, public ActorCallback< FlowTestCase912Actor, 1, Void >, public FastAllocated<FlowTestCase912Actor>, public FlowTestCase912ActorState<FlowTestCase912Actor> {
															#line 4049 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase912Actor>::operator new;
	using FastAllocated<FlowTestCase912Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9418300087775710464UL, 15210485572399219968UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase912Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase912Actor, 1, Void >;
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase912Actor(UnitTestParameters const& params) 
															#line 4067 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase912ActorState<FlowTestCase912Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase912", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(827380864981028864UL, 18031906037941957632UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase912Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase912");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase912", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase912Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase912Actor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase912( UnitTestParameters const& params ) {
															#line 912 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase912Actor(params));
															#line 4101 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase912, "/GlobalTagThrottler/UpdateQuota")

#line 931 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 100 pages/second each.
// Total quota is initially set to 100 pages/second.
// Client attempts 5 6-page read transactions per second.
// Target limit adjusts to allow 100/6 transactions per second.
// Then Quota is removed.
// Target limit is removed as a result.
															#line 4113 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase938()
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase938Actor>
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase938ActorState {
															#line 4120 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase938ActorState(UnitTestParameters const& params) 
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 939 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 940 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100),
															#line 941 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue(),
															#line 942 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   testTag("sampleTag1"_sr)
															#line 4135 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase938", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase938ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase938", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 943 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 944 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 945 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 5.0, 6.0, OpType::READ);
															#line 946 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor( &globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, 100.0 / 6.0); });
															#line 948 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 949 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 949 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase938Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4162 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase938Actor*>(this)->actor_wait_state = 1;
															#line 949 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase938Actor, 0, Void >*>(static_cast<FlowTestCase938Actor*>(this)));
															#line 4167 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase938ActorState();
		static_cast<FlowTestCase938Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 950 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler.removeQuota(testTag);
															#line 951 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		monitor = monitorActor(&globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, {}); });
															#line 952 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeoutError(monitor || client || updater, 600.0);
															#line 952 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase938Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4196 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase938Actor*>(this)->actor_wait_state = 2;
															#line 952 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase938Actor, 1, Void >*>(static_cast<FlowTestCase938Actor*>(this)));
															#line 4201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 950 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler.removeQuota(testTag);
															#line 951 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		monitor = monitorActor(&globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, {}); });
															#line 952 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeoutError(monitor || client || updater, 600.0);
															#line 952 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase938Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4216 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase938Actor*>(this)->actor_wait_state = 2;
															#line 952 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase938Actor, 1, Void >*>(static_cast<FlowTestCase938Actor*>(this)));
															#line 4221 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase938Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase938Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase938Actor*>(this)->ActorCallback< FlowTestCase938Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase938Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase938", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase938Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase938", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase938Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase938", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase938Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase938", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase938Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase938", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase938Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase938", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 953 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase938Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase938ActorState(); static_cast<FlowTestCase938Actor*>(this)->destroy(); return 0; }
															#line 4305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase938Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase938ActorState();
		static_cast<FlowTestCase938Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 953 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase938Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase938ActorState(); static_cast<FlowTestCase938Actor*>(this)->destroy(); return 0; }
															#line 4317 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase938Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase938ActorState();
		static_cast<FlowTestCase938Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase938Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase938Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase938Actor*>(this)->ActorCallback< FlowTestCase938Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase938Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase938", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase938Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase938", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase938Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase938", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase938Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase938", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase938Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase938", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase938Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase938", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 939 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 940 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 941 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 942 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	TransactionTag testTag;
															#line 945 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 946 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 948 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 4416 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase938()
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase938Actor final : public Actor<Void>, public ActorCallback< FlowTestCase938Actor, 0, Void >, public ActorCallback< FlowTestCase938Actor, 1, Void >, public FastAllocated<FlowTestCase938Actor>, public FlowTestCase938ActorState<FlowTestCase938Actor> {
															#line 4421 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase938Actor>::operator new;
	using FastAllocated<FlowTestCase938Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7813331566611371264UL, 14585764788814420992UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase938Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase938Actor, 1, Void >;
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase938Actor(UnitTestParameters const& params) 
															#line 4439 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase938ActorState<FlowTestCase938Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase938", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13068121403876472832UL, 834915236736823040UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase938Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase938");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase938", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase938Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase938Actor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase938( UnitTestParameters const& params ) {
															#line 938 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase938Actor(params));
															#line 4473 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase938, "/GlobalTagThrottler/RemoveQuota")

#line 955 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 5 pages/second each.
// Total quota is set to 100 pages/second.
// Client attempts 10 6-page transactions per second
// Target is adjusted to 50/6 transactions per second, to match the total capacity all storage servers.
															#line 4483 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase960()
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase960Actor>
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase960ActorState {
															#line 4490 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase960ActorState(UnitTestParameters const& params) 
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 961 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 962 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 5),
															#line 963 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue()
															#line 4503 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase960", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase960ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase960", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 964 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 965 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 966 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 967 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 10.0, 6.0, OpType::READ);
															#line 968 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag](auto& gtt) { return targetRateIsNear(gtt, testTag, 50 / 6.0) && gtt.busyReadTagCount() == 1; });
															#line 971 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 972 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 972 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase960Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase960Actor*>(this)->actor_wait_state = 1;
															#line 972 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase960Actor, 0, Void >*>(static_cast<FlowTestCase960Actor*>(this)));
															#line 4537 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase960ActorState();
		static_cast<FlowTestCase960Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 973 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase960Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase960ActorState(); static_cast<FlowTestCase960Actor*>(this)->destroy(); return 0; }
															#line 4560 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase960Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase960ActorState();
		static_cast<FlowTestCase960Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 973 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase960Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase960ActorState(); static_cast<FlowTestCase960Actor*>(this)->destroy(); return 0; }
															#line 4572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase960Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase960ActorState();
		static_cast<FlowTestCase960Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase960Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase960Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase960Actor*>(this)->ActorCallback< FlowTestCase960Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase960Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase960", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase960Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase960", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase960Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase960", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase960Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase960", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase960Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase960", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase960Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase960", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 961 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 962 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 963 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 967 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 968 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 971 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 4669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase960()
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase960Actor final : public Actor<Void>, public ActorCallback< FlowTestCase960Actor, 0, Void >, public FastAllocated<FlowTestCase960Actor>, public FlowTestCase960ActorState<FlowTestCase960Actor> {
															#line 4674 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase960Actor>::operator new;
	using FastAllocated<FlowTestCase960Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8724898885602724352UL, 13408895079173339904UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase960Actor, 0, Void >;
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase960Actor(UnitTestParameters const& params) 
															#line 4691 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase960ActorState<FlowTestCase960Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase960", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16647012115615295232UL, 3354942169579841792UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase960Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase960");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase960", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase960Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase960( UnitTestParameters const& params ) {
															#line 960 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase960Actor(params));
															#line 4724 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase960, "/GlobalTagThrottler/ActiveThrottling")

#line 975 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can handle 5 pages/second each.
// Total quota is set to 50 pages/second for one tag, 100 pages/second for another.
// For each tag, a client attempts to execute 10 6-page read transactions per second.
// Target rates are adjusted to utilize the full 50 pages/second capacity of the
// storage servers.
															#line 4735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase981()
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase981Actor>
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase981ActorState {
															#line 4742 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase981ActorState(UnitTestParameters const& params) 
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 982 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 983 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 5),
															#line 984 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue1(),
															#line 985 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue2()
															#line 4757 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase981", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase981ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase981", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 986 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag1 = "sampleTag1"_sr;
															#line 987 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag2 = "sampleTag2"_sr;
															#line 988 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue1.totalQuota = 50 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 989 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue2.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 990 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag1, tagQuotaValue1);
															#line 991 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag2, tagQuotaValue2);
															#line 992 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			std::vector<Future<Void>> futures;
															#line 993 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(runClient(&globalTagThrottler, &storageServers, testTag1, 10.0, 6.0, OpType::READ));
															#line 994 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(runClient(&globalTagThrottler, &storageServers, testTag2, 10.0, 6.0, OpType::READ));
															#line 995 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag1, testTag2](auto& gtt) { return totalTargetRateIsNear(gtt, 50 / 6.0) && gtt.busyReadTagCount() == 2; });
															#line 998 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(updateGlobalTagThrottler(&globalTagThrottler, &storageServers));
															#line 999 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(waitForAny(futures) || monitor, 600.0);
															#line 999 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase981Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4796 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase981Actor*>(this)->actor_wait_state = 1;
															#line 999 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase981Actor, 0, Void >*>(static_cast<FlowTestCase981Actor*>(this)));
															#line 4801 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase981ActorState();
		static_cast<FlowTestCase981Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1000 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase981Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase981ActorState(); static_cast<FlowTestCase981Actor*>(this)->destroy(); return 0; }
															#line 4824 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase981Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase981ActorState();
		static_cast<FlowTestCase981Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1000 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase981Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase981ActorState(); static_cast<FlowTestCase981Actor*>(this)->destroy(); return 0; }
															#line 4836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase981Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase981ActorState();
		static_cast<FlowTestCase981Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase981Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase981Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase981Actor*>(this)->ActorCallback< FlowTestCase981Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase981Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase981", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase981Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase981", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase981Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase981", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase981Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase981", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase981Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase981", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase981Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase981", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 982 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 983 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 984 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue1;
															#line 985 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue2;
															#line 995 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 4931 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase981()
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase981Actor final : public Actor<Void>, public ActorCallback< FlowTestCase981Actor, 0, Void >, public FastAllocated<FlowTestCase981Actor>, public FlowTestCase981ActorState<FlowTestCase981Actor> {
															#line 4936 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase981Actor>::operator new;
	using FastAllocated<FlowTestCase981Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2759025293359133184UL, 8246693857427756032UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase981Actor, 0, Void >;
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase981Actor(UnitTestParameters const& params) 
															#line 4953 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase981ActorState<FlowTestCase981Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase981", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2814831999360399872UL, 7282302431154646528UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase981Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase981");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase981", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase981Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase981( UnitTestParameters const& params ) {
															#line 981 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase981Actor(params));
															#line 4986 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase981, "/GlobalTagThrottler/MultiTagActiveThrottling1")

#line 1002 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 3 storage servers can handle 50 pages/second each.
// Total quota is set to 100 pages/second for each tag.
// Each client attempts 10 6-page read transactions per second.
// This workload is sent to 2 storage servers per client (with an overlap of one storage server).
// The total target rate summed across both tags is adjusted to match the throughput that the
// busiest server can handle (50/3 transactions per second).
															#line 4998 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1009()
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase1009Actor>
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1009ActorState {
															#line 5005 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1009ActorState(UnitTestParameters const& params) 
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 1010 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 1011 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(3, 50),
															#line 1012 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue1(),
															#line 1013 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue2()
															#line 5020 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1009", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1009ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1009", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1014 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag1 = "sampleTag1"_sr;
															#line 1015 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag2 = "sampleTag2"_sr;
															#line 1016 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue1.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 1017 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue2.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 1018 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag1, tagQuotaValue1);
															#line 1019 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag2, tagQuotaValue2);
															#line 1020 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			std::vector<Future<Void>> futures;
															#line 1021 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(runClient(&globalTagThrottler, &storageServers, testTag1, 10.0, 6.0, OpType::READ, { 0, 1 }));
															#line 1022 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(runClient(&globalTagThrottler, &storageServers, testTag2, 10.0, 6.0, OpType::READ, { 1, 2 }));
															#line 1023 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag1, testTag2](auto& gtt) { return totalTargetRateIsNear(gtt, 50 / 3.0) && gtt.busyReadTagCount() == 2; });
															#line 1026 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(updateGlobalTagThrottler(&globalTagThrottler, &storageServers));
															#line 1027 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(waitForAny(futures) || monitor, 600.0);
															#line 1027 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase1009Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5059 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase1009Actor*>(this)->actor_wait_state = 1;
															#line 1027 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1009Actor, 0, Void >*>(static_cast<FlowTestCase1009Actor*>(this)));
															#line 5064 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1009ActorState();
		static_cast<FlowTestCase1009Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1028 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1009Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1009ActorState(); static_cast<FlowTestCase1009Actor*>(this)->destroy(); return 0; }
															#line 5087 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1009Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1009ActorState();
		static_cast<FlowTestCase1009Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1028 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1009Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1009ActorState(); static_cast<FlowTestCase1009Actor*>(this)->destroy(); return 0; }
															#line 5099 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1009Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1009ActorState();
		static_cast<FlowTestCase1009Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase1009Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1009Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1009Actor*>(this)->ActorCallback< FlowTestCase1009Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1009Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1009", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1009Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1009", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1009Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1009", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1009Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1009", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase1009Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1009", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1009Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1009", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 1010 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 1011 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 1012 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue1;
															#line 1013 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue2;
															#line 1023 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 5194 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1009()
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1009Actor final : public Actor<Void>, public ActorCallback< FlowTestCase1009Actor, 0, Void >, public FastAllocated<FlowTestCase1009Actor>, public FlowTestCase1009ActorState<FlowTestCase1009Actor> {
															#line 5199 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1009Actor>::operator new;
	using FastAllocated<FlowTestCase1009Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14368328115702563840UL, 14223201553024557312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase1009Actor, 0, Void >;
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1009Actor(UnitTestParameters const& params) 
															#line 5216 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1009ActorState<FlowTestCase1009Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1009", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12142580165321210880UL, 16248409784321250048UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1009Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1009");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1009", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase1009Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase1009( UnitTestParameters const& params ) {
															#line 1009 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase1009Actor(params));
															#line 5249 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1009, "/GlobalTagThrottler/MultiTagActiveThrottling2")

#line 1030 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 3 storage servers can handle 50 pages/second each.
// Total quota is set to 100 pages/second for each tag.
// One client attempts 10 6-page read transactions per second, all directed towards a single storage server.
// Another client, using a different tag, attempts 10 6-page read transactions split across the other two storage
// servers. Target rates adjust to 50/6 and 100/6 transactions per second for the two clients, based on the capacities
// of the storage servers being accessed.
															#line 5261 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1037()
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase1037Actor>
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1037ActorState {
															#line 5268 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1037ActorState(UnitTestParameters const& params) 
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 1038 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 1039 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(3, 50),
															#line 1040 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue1(),
															#line 1041 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue2()
															#line 5283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1037", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1037ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1037", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1042 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag1 = "sampleTag1"_sr;
															#line 1043 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag2 = "sampleTag2"_sr;
															#line 1044 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue1.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 1045 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue2.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 1046 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag1, tagQuotaValue1);
															#line 1047 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag2, tagQuotaValue2);
															#line 1048 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			std::vector<Future<Void>> futures;
															#line 1049 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(runClient(&globalTagThrottler, &storageServers, testTag1, 10.0, 6.0, OpType::READ, { 0 }));
															#line 1050 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(runClient(&globalTagThrottler, &storageServers, testTag2, 10.0, 6.0, OpType::READ, { 1, 2 }));
															#line 1051 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag1, testTag2](auto& gtt) { return targetRateIsNear(gtt, testTag1, 50 / 6.0) && targetRateIsNear(gtt, testTag2, 100 / 6.0) && gtt.busyReadTagCount() == 1; });
															#line 1055 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			futures.push_back(updateGlobalTagThrottler(&globalTagThrottler, &storageServers));
															#line 1056 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(waitForAny(futures) || monitor, 600.0);
															#line 1056 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase1037Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5322 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase1037Actor*>(this)->actor_wait_state = 1;
															#line 1056 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1037Actor, 0, Void >*>(static_cast<FlowTestCase1037Actor*>(this)));
															#line 5327 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1037ActorState();
		static_cast<FlowTestCase1037Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1057 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1037Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1037ActorState(); static_cast<FlowTestCase1037Actor*>(this)->destroy(); return 0; }
															#line 5350 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1037Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1037ActorState();
		static_cast<FlowTestCase1037Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1057 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1037Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1037ActorState(); static_cast<FlowTestCase1037Actor*>(this)->destroy(); return 0; }
															#line 5362 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1037Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1037ActorState();
		static_cast<FlowTestCase1037Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase1037Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1037Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1037Actor*>(this)->ActorCallback< FlowTestCase1037Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1037Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1037", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1037Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1037", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1037Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1037", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1037Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1037", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase1037Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1037", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1037Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1037", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 1038 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 1039 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 1040 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue1;
															#line 1041 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue2;
															#line 1051 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 5457 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1037()
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1037Actor final : public Actor<Void>, public ActorCallback< FlowTestCase1037Actor, 0, Void >, public FastAllocated<FlowTestCase1037Actor>, public FlowTestCase1037ActorState<FlowTestCase1037Actor> {
															#line 5462 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1037Actor>::operator new;
	using FastAllocated<FlowTestCase1037Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2420321944365058816UL, 3336801088887533056UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase1037Actor, 0, Void >;
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1037Actor(UnitTestParameters const& params) 
															#line 5479 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1037ActorState<FlowTestCase1037Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1037", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17455681774589509376UL, 7475541557322951424UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1037Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1037");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1037", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase1037Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase1037( UnitTestParameters const& params ) {
															#line 1037 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase1037Actor(params));
															#line 5512 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1037, "/GlobalTagThrottler/MultiTagActiveThrottling3")

#line 1059 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 10 storage servers can serve 5 pages/second each.
// Total quota is set to 100 pages/second.
// Reserved quota is set to 70 pages/second.
// A client attempts to execute 10 6-page read transactions per second.
// Despite the storage server only having capacity to serve 50/6 transactions per second,
//   the reserved quota will ensure the target rate adjusts to 70/6 transactions per second.
															#line 5524 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1066()
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase1066Actor>
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1066ActorState {
															#line 5531 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1066ActorState(UnitTestParameters const& params) 
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 1067 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 5),
															#line 1069 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   tagQuotaValue()
															#line 5544 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1066", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1066ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1066", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1070 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 1071 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 1072 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.reservedQuota = 70 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 1073 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 1074 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 10.0, 6.0, OpType::READ);
															#line 1075 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor(&globalTagThrottler, [testTag](auto& gtt) { return targetRateIsNear(gtt, testTag, 70 / 6.0); });
															#line 1077 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 1078 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 1078 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase1066Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5575 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase1066Actor*>(this)->actor_wait_state = 1;
															#line 1078 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1066Actor, 0, Void >*>(static_cast<FlowTestCase1066Actor*>(this)));
															#line 5580 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1066ActorState();
		static_cast<FlowTestCase1066Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1079 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1066Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1066ActorState(); static_cast<FlowTestCase1066Actor*>(this)->destroy(); return 0; }
															#line 5603 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1066Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1066ActorState();
		static_cast<FlowTestCase1066Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1079 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1066Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1066ActorState(); static_cast<FlowTestCase1066Actor*>(this)->destroy(); return 0; }
															#line 5615 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1066Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1066ActorState();
		static_cast<FlowTestCase1066Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase1066Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1066Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1066Actor*>(this)->ActorCallback< FlowTestCase1066Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1066Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1066", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1066", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1066Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1066", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1066", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase1066Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1066", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1066", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 1067 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 1068 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 1069 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 1074 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 1075 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 1077 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 5712 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1066()
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1066Actor final : public Actor<Void>, public ActorCallback< FlowTestCase1066Actor, 0, Void >, public FastAllocated<FlowTestCase1066Actor>, public FlowTestCase1066ActorState<FlowTestCase1066Actor> {
															#line 5717 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1066Actor>::operator new;
	using FastAllocated<FlowTestCase1066Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13602508254939718912UL, 16500221515602818816UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase1066Actor, 0, Void >;
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1066Actor(UnitTestParameters const& params) 
															#line 5734 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1066ActorState<FlowTestCase1066Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1066", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8356652071045351680UL, 2071736920068913152UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1066");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1066", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase1066Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase1066( UnitTestParameters const& params ) {
															#line 1066 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase1066Actor(params));
															#line 5767 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1066, "/GlobalTagThrottler/ReservedQuota")

#line 1081 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// Test that tags are expired iff a sufficient amount of time has passed since the
// last transaction with that tag
															#line 5775 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1084()
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase1084Actor>
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1084ActorState {
															#line 5782 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1084ActorState(UnitTestParameters const& params) 
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 1085 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 1086 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 5)
															#line 5793 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1084", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1084ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1084", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1087 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			TransactionTag testTag = "sampleTag1"_sr;
															#line 1089 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = timeout(runClient(&globalTagThrottler, &storageServers, testTag, 10.0, 6.0, OpType::READ), 60.0, Void());
															#line 1091 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = timeout(updateGlobalTagThrottler(&globalTagThrottler, &storageServers), 60.0, Void());
															#line 1092 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = client && updater;
															#line 1092 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5816 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state = 1;
															#line 1092 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1084Actor, 0, Void >*>(static_cast<FlowTestCase1084Actor*>(this)));
															#line 5821 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1084ActorState();
		static_cast<FlowTestCase1084Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1093 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		client.cancel();
															#line 1094 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		updater.cancel();
															#line 1095 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), 1);
															#line 1096 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler.removeExpiredTags();
															#line 1097 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), 1);
															#line 1098 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->GLOBAL_TAG_THROTTLING_TAG_EXPIRE_AFTER + 1.0);
															#line 1098 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5856 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state = 2;
															#line 1098 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1084Actor, 1, Void >*>(static_cast<FlowTestCase1084Actor*>(this)));
															#line 5861 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1093 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		client.cancel();
															#line 1094 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		updater.cancel();
															#line 1095 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), 1);
															#line 1096 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler.removeExpiredTags();
															#line 1097 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), 1);
															#line 1098 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->GLOBAL_TAG_THROTTLING_TAG_EXPIRE_AFTER + 1.0);
															#line 1098 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5882 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state = 2;
															#line 1098 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1084Actor, 1, Void >*>(static_cast<FlowTestCase1084Actor*>(this)));
															#line 5887 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1084Actor*>(this)->ActorCallback< FlowTestCase1084Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1084Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1084", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1084Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1084", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1084Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1084", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1084Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1084", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase1084Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1084", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1084Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1084", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1099 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), 1);
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler.removeExpiredTags();
															#line 1101 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), 0);
															#line 1102 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1084Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1084ActorState(); static_cast<FlowTestCase1084Actor*>(this)->destroy(); return 0; }
															#line 5977 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1084Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1084ActorState();
		static_cast<FlowTestCase1084Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1099 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), 1);
															#line 1100 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		globalTagThrottler.removeExpiredTags();
															#line 1101 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), 0);
															#line 1102 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1084Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1084ActorState(); static_cast<FlowTestCase1084Actor*>(this)->destroy(); return 0; }
															#line 5995 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1084Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1084ActorState();
		static_cast<FlowTestCase1084Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1084Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1084Actor*>(this)->ActorCallback< FlowTestCase1084Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1084Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1084", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1084Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1084", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1084Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1084", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1084Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1084", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase1084Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1084", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1084Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1084", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 1085 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 1086 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 1089 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 1091 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 6088 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1084()
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1084Actor final : public Actor<Void>, public ActorCallback< FlowTestCase1084Actor, 0, Void >, public ActorCallback< FlowTestCase1084Actor, 1, Void >, public FastAllocated<FlowTestCase1084Actor>, public FlowTestCase1084ActorState<FlowTestCase1084Actor> {
															#line 6093 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1084Actor>::operator new;
	using FastAllocated<FlowTestCase1084Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4740304398461038336UL, 296595888701986560UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase1084Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase1084Actor, 1, Void >;
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1084Actor(UnitTestParameters const& params) 
															#line 6111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1084ActorState<FlowTestCase1084Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1084", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7990276998591725568UL, 17780638665742168320UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1084Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1084");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1084", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase1084Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase1084Actor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase1084( UnitTestParameters const& params ) {
															#line 1084 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase1084Actor(params));
															#line 6145 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1084, "/GlobalTagThrottler/ExpireTags")

#line 1104 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// Test that the number of tags tracked does not grow beyond SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MAX_TAGS_TRACKED
															#line 6152 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1106()
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase1106Actor>
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1106ActorState {
															#line 6159 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1106ActorState(UnitTestParameters const& params) 
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 1107 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler()),
															#line 1108 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 5)
															#line 6170 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1106", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1106ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1106", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1109 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			std::vector<Future<Void>> futures;
															#line 1110 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			for(int i = 0;i < 2 * SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MAX_TAGS_TRACKED;++i) {
															#line 1111 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
				Arena arena;
															#line 1112 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
				TransactionTag tag = makeString(8, arena);
															#line 1113 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
				deterministicRandom()->randomBytes(mutateString(tag), tag.size());
															#line 1114 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
				futures.push_back(runClient(&globalTagThrottler, &storageServers, tag, 1.0, 6.0, OpType::READ));
															#line 6195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			}
															#line 1116 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeout(waitForAll(futures), 60.0, Void());
															#line 1116 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase1106Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase1106Actor*>(this)->actor_wait_state = 1;
															#line 1116 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1106Actor, 0, Void >*>(static_cast<FlowTestCase1106Actor*>(this)));
															#line 6206 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1106ActorState();
		static_cast<FlowTestCase1106Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1117 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MAX_TAGS_TRACKED);
															#line 1118 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1106Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1106ActorState(); static_cast<FlowTestCase1106Actor*>(this)->destroy(); return 0; }
															#line 6231 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1106Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1106ActorState();
		static_cast<FlowTestCase1106Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1117 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		ASSERT_EQ(globalTagThrottler.tagsTracked(), SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MAX_TAGS_TRACKED);
															#line 1118 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1106Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1106ActorState(); static_cast<FlowTestCase1106Actor*>(this)->destroy(); return 0; }
															#line 6245 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1106Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1106ActorState();
		static_cast<FlowTestCase1106Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase1106Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1106Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1106Actor*>(this)->ActorCallback< FlowTestCase1106Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1106Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1106", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1106Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1106", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1106Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1106", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1106Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1106", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase1106Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1106", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1106Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1106", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 1107 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 1108 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 6334 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1106()
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1106Actor final : public Actor<Void>, public ActorCallback< FlowTestCase1106Actor, 0, Void >, public FastAllocated<FlowTestCase1106Actor>, public FlowTestCase1106ActorState<FlowTestCase1106Actor> {
															#line 6339 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1106Actor>::operator new;
	using FastAllocated<FlowTestCase1106Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11429785871106904064UL, 10268887156912777984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase1106Actor, 0, Void >;
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1106Actor(UnitTestParameters const& params) 
															#line 6356 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1106ActorState<FlowTestCase1106Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1106", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12615296786427559168UL, 6921431730183740672UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1106Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1106");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1106", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase1106Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase1106( UnitTestParameters const& params ) {
															#line 1106 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase1106Actor(params));
															#line 6389 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1106, "/GlobalTagThrottler/TagLimit")

#line 1120 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"

// 9 storage servers can handle 100 pages/second each.
// 1 unhealthy storage server can only handle 1 page/second.
// Total quota is set to 100 pages/second.
// Client attempts 5 6-page transactions per second.
// Target rate adjusts to 100/6 transactions per second, ignoring the worst storage server.
// Then, a second storage server becomes unhealthy and can only handle 1 page/second.
// Target rate adjusts down to 10/6 transactions per second, because only one bad zone can be ignored.
															#line 6402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1128()
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
template <class FlowTestCase1128Actor>
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1128ActorState {
															#line 6409 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1128ActorState(UnitTestParameters const& params) 
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		 : params(params),
															#line 1129 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   globalTagThrottler(getTestGlobalTagThrottler( 1)),
															#line 1130 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   storageServers(10, 100),
															#line 1131 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		   testTag("sampleTag1"_sr)
															#line 6422 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1128", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1128ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1128", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1132 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			storageServers.setCapacity(0, 1);
															#line 1133 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			ThrottleApi::TagQuotaValue tagQuotaValue;
															#line 1134 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			tagQuotaValue.totalQuota = 100 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE;
															#line 1135 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			globalTagThrottler.setQuota(testTag, tagQuotaValue);
															#line 1136 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			client = runClient(&globalTagThrottler, &storageServers, testTag, 5.0, 6.0, OpType::READ);
															#line 1137 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			monitor = monitorActor( &globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, 100.0 / 6.0); });
															#line 1139 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			updater = updateGlobalTagThrottler(&globalTagThrottler, &storageServers);
															#line 1140 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeoutError(monitor || client || updater, 600.0);
															#line 1140 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6453 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state = 1;
															#line 1140 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1128Actor, 0, Void >*>(static_cast<FlowTestCase1128Actor*>(this)));
															#line 6458 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1128ActorState();
		static_cast<FlowTestCase1128Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1141 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		storageServers.setCapacity(1, 1);
															#line 1142 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		monitor = monitorActor(&globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, 10.0 / 6.0); });
															#line 1144 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeoutError(monitor || client || updater, 600.0);
															#line 1144 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state = 2;
															#line 1144 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1128Actor, 1, Void >*>(static_cast<FlowTestCase1128Actor*>(this)));
															#line 6492 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1141 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		storageServers.setCapacity(1, 1);
															#line 1142 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		monitor = monitorActor(&globalTagThrottler, [](auto& gtt) { return targetRateIsNear(gtt, "sampleTag1"_sr, 10.0 / 6.0); });
															#line 1144 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeoutError(monitor || client || updater, 600.0);
															#line 1144 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6507 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state = 2;
															#line 1144 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1128Actor, 1, Void >*>(static_cast<FlowTestCase1128Actor*>(this)));
															#line 6512 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1128Actor*>(this)->ActorCallback< FlowTestCase1128Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1128Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1128", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1128Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1128", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1128Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1128", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1128Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1128", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase1128Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1128", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1128Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1128", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1145 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1128Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1128ActorState(); static_cast<FlowTestCase1128Actor*>(this)->destroy(); return 0; }
															#line 6596 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1128Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1128ActorState();
		static_cast<FlowTestCase1128Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1145 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase1128Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1128ActorState(); static_cast<FlowTestCase1128Actor*>(this)->destroy(); return 0; }
															#line 6608 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase1128Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1128ActorState();
		static_cast<FlowTestCase1128Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1128Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1128Actor*>(this)->ActorCallback< FlowTestCase1128Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1128Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1128", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1128Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1128", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1128Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1128", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1128Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1128", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase1128Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1128", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1128Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1128", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 1129 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	GlobalTagThrottler globalTagThrottler;
															#line 1130 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	StorageServerCollection storageServers;
															#line 1131 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	TransactionTag testTag;
															#line 1136 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> client;
															#line 1137 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> monitor;
															#line 1139 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	Future<Void> updater;
															#line 6705 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1128()
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
class FlowTestCase1128Actor final : public Actor<Void>, public ActorCallback< FlowTestCase1128Actor, 0, Void >, public ActorCallback< FlowTestCase1128Actor, 1, Void >, public FastAllocated<FlowTestCase1128Actor>, public FlowTestCase1128ActorState<FlowTestCase1128Actor> {
															#line 6710 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1128Actor>::operator new;
	using FastAllocated<FlowTestCase1128Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8001181096280197376UL, 9830215811241459712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase1128Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase1128Actor, 1, Void >;
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	FlowTestCase1128Actor(UnitTestParameters const& params) 
															#line 6728 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1128ActorState<FlowTestCase1128Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1128", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9879293587874395136UL, 13007380007972398592UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1128Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1128");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1128", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase1128Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase1128Actor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
static Future<Void> flowTestCase1128( UnitTestParameters const& params ) {
															#line 1128 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase1128Actor(params));
															#line 6762 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/GlobalTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1128, "/GlobalTagThrottler/IgnoreWorstZone")

#line 1147 "/Users/ec2-user/foundationdb/fdbserver/GlobalTagThrottler.actor.cpp"
