#define POST_ACTOR_COMPILER 1
#line 1 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
/*
 * Mako.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/NativeAPI.actor.h"
#include "fdbserver/TesterInterface.actor.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbserver/workloads/BulkSetup.actor.h"
#include "fdbclient/ReadYourWrites.h"
#include "fdbclient/zipf.h"
#include "crc32/crc32c.h"
#include "flow/actorcompiler.h"

enum {
	OP_GETREADVERSION,
	OP_GET,
	OP_GETRANGE,
	OP_SGET,
	OP_SGETRANGE,
	OP_UPDATE,
	OP_INSERT,
	OP_INSERTRANGE,
	OP_CLEAR,
	OP_SETCLEAR,
	OP_CLEARRANGE,
	OP_SETCLEARRANGE,
	OP_COMMIT,
	MAX_OP
};
enum { OP_COUNT, OP_RANGE };
struct MakoWorkload : TestWorkload {
	static constexpr auto NAME = "Mako";

	uint64_t rowCount, seqNumLen, sampleSize, actorCountPerClient, keyBytes, maxValueBytes, minValueBytes, csSize,
	    csCount, csPartitionSize, csStepSizeInPartition;
	double testDuration, loadTime, warmingDelay, maxInsertRate, transactionsPerSecond, allowedLatency,
	    periodicLoggingInterval, zipfConstant;
	bool enableLogging, commitGet, populateData, runBenchmark, preserveData, zipf, checksumVerification,
	    doChecksumVerificationOnly, latencyForLocalOperation;
	PerfIntCounter xacts, retries, conflicts, commits, totalOps;
	std::vector<PerfIntCounter> opCounters;
	std::vector<uint64_t> insertionCountsToMeasure;
	std::vector<std::pair<uint64_t, double>> ratesAtKeyCounts;
	std::string operationsSpec;
	// store operations to execute
	int operations[MAX_OP][2];
	// used for periodically tracing
	std::vector<PerfMetric> periodicMetrics;
	// store latency of each operation with sampling
	std::vector<DDSketch<double>> opLatencies;
	// key used to store checkSum for given key range
	std::vector<Key> csKeys;
	// key prefix of for all generated keys
	std::string keyPrefix;
	int KEYPREFIXLEN;
	const std::array<std::string, MAX_OP> opNames = { "GRV",       "GET",      "GETRANGE",   "SGET",
		                                              "SGETRANGE", "UPDATE",   "INSERT",     "INSERTRANGE",
		                                              "CLEAR",     "SETCLEAR", "CLEARRANGE", "SETCLEARRANGE",
		                                              "COMMIT" };
	MakoWorkload(WorkloadContext const& wcx)
	  : TestWorkload(wcx), loadTime(0.0), xacts("Transactions"), retries("Retries"), conflicts("Conflicts"),
	    commits("Commits"), totalOps("Operations") {
		// init parameters from test file
		// Number of rows populated
		rowCount = getOption(options, "rows"_sr, (uint64_t)10000);
		// Test duration in seconds
		testDuration = getOption(options, "testDuration"_sr, 30.0);
		warmingDelay = getOption(options, "warmingDelay"_sr, 0.0);
		maxInsertRate = getOption(options, "maxInsertRate"_sr, 1e12);
		// Flag to control whether to populate data into database
		populateData = getOption(options, "populateData"_sr, true);
		// Flag to control whether to run benchmark
		runBenchmark = getOption(options, "runBenchmark"_sr, true);
		// Flag to control whether to clean data in the database
		preserveData = getOption(options, "preserveData"_sr, true);
		// If true, force commit for read-only transactions
		commitGet = getOption(options, "commitGet"_sr, false);
		// If true, log latency for set, clear and clearrange
		latencyForLocalOperation = getOption(options, "latencyForLocalOperation"_sr, false);
		// Target total transaction-per-second (TPS) of all clients
		transactionsPerSecond = getOption(options, "transactionsPerSecond"_sr, 100000.0) / clientCount;
		actorCountPerClient = getOption(options, "actorCountPerClient"_sr, 16);
		// Sampling rate (1 sample / <sampleSize> ops) for latency stats
		sampleSize = getOption(options, "sampleSize"_sr, rowCount / 100);
		// If true, record latency metrics per periodicLoggingInterval; For details, see tracePeriodically()
		enableLogging = getOption(options, "enableLogging"_sr, false);
		periodicLoggingInterval = getOption(options, "periodicLoggingInterval"_sr, 5.0);
		// All the generated keys will start with the specified prefix
		keyPrefix = getOption(options, "keyPrefix"_sr, "mako"_sr).toString();
		KEYPREFIXLEN = keyPrefix.size();
		// If true, the workload will picking up keys which are zipfian distributed
		zipf = getOption(options, "zipf"_sr, false);
		zipfConstant = getOption(options, "zipfConstant"_sr, 0.99);
		// Specified length of keys and length range of values
		keyBytes = std::max(getOption(options, "keyBytes"_sr, 16), 16);
		maxValueBytes = getOption(options, "valueBytes"_sr, 16);
		minValueBytes = getOption(options, "minValueBytes"_sr, maxValueBytes);
		ASSERT(minValueBytes <= maxValueBytes);
		// The inserted key is formatted as: fixed prefix('mako') + sequential number + padding('x')
		// assume we want to insert 10000 rows with keyBytes set to 16,
		// then the key goes from 'mako00000xxxxxxx' to 'mako09999xxxxxxx'
		seqNumLen = digits(rowCount);
		// check keyBytes, maxValueBytes is valid
		ASSERT(seqNumLen + KEYPREFIXLEN <= keyBytes);
		// user input: a sequence of operations to be executed; e.g. "g10i5" means to do GET 10 times and Insert 5 times
		// One operation type is defined as "<Type><Count>" or "<Type><Count>:<Range>".
		// When Count is omitted, it's equivalent to setting it to 1.  (e.g. "g" is equivalent to "g1")
		// Multiple operation types can be concatenated.  (e.g. "g9u1" = 9 GETs and 1 update)
		// For RANGE operations, "Range" needs to be specified in addition to "Count".
		// Below are all allowed inputs:
		// g – GET
		// gr – GET RANGE
		// sg – Snapshot GET
		// sgr – Snapshot GET RANGE
		// u – Update (= GET followed by SET)
		// i – Insert (= SET with a new key)
		// ir – Insert Range (Sequential)
		// c – CLEAR
		// sc – SET & CLEAR
		// cr – CLEAR RANGE
		// scr – SET & CLEAR RANGE
		// grv – GetReadVersion()
		// Every transaction is committed unless it contains only GET / GET RANGE operations.
		operationsSpec = getOption(options, "operations"_sr, "g100"_sr).contents().toString();
		//  parse the sequence and extract operations to be executed
		parseOperationsSpec();
		for (int i = 0; i < MAX_OP; ++i) {
			// initialize per-operation latency record
			opLatencies.push_back(DDSketch<double>());
			// initialize per-operation counter
			opCounters.push_back(PerfIntCounter(opNames[i]));
		}
		if (zipf) {
			zipfian_generator3(0, (int)rowCount - 1, zipfConstant);
		}
		// Added for checksum verification
		csSize = getOption(options, "csSize"_sr, rowCount / 100);
		ASSERT(csSize <= rowCount);
		csCount = getOption(options, "csCount"_sr, 0);
		checksumVerification = (csCount != 0);
		doChecksumVerificationOnly = getOption(options, "doChecksumVerificationOnly"_sr, false);
		if (doChecksumVerificationOnly)
			ASSERT(checksumVerification); // csCount should be non-zero when you do checksum verification only
		if (csCount) {
			csPartitionSize = rowCount / csSize;
			ASSERT(csCount <= csPartitionSize);
			csStepSizeInPartition = csPartitionSize / csCount;
			for (int i = 0; i < csCount; ++i) {
				csKeys.emplace_back(format((keyPrefix + "_crc32c_%u_%u").c_str(), i, rowCount));
			}
		}
	}

	Future<Void> setup(Database const& cx) override {
		if (doChecksumVerificationOnly)
			return Void();
		return _setup(cx, this);
	}

	Future<Void> start(Database const& cx) override {
		if (doChecksumVerificationOnly)
			return Void();
		return _start(cx, this);
	}

	Future<bool> check(Database const& cx) override {
		if (!checksumVerification) {
			return true;
		}
		// verify checksum consistency
		return dochecksumVerification(cx, this);
	}

	// disable the default timeout setting
	double getCheckTimeout() const override { return std::numeric_limits<double>::max(); }

	void getMetrics(std::vector<PerfMetric>& m) override {
		// metrics of population process
		if (populateData) {
			m.emplace_back("Mean load time (seconds)", loadTime, Averaged::True);
			// The importing rate of keys, controlled by parameter "insertionCountsToMeasure"
			auto ratesItr = ratesAtKeyCounts.begin();
			for (; ratesItr != ratesAtKeyCounts.end(); ratesItr++) {
				m.emplace_back(
				    format("%lld keys imported bytes/sec", ratesItr->first), ratesItr->second, Averaged::False);
			}
		}
		// benchmark
		if (runBenchmark) {
			m.emplace_back("Measured Duration", testDuration, Averaged::True);
			m.push_back(xacts.getMetric());
			m.emplace_back("Transactions/sec", xacts.getValue() / testDuration, Averaged::True);
			m.push_back(totalOps.getMetric());
			m.emplace_back("Operations/sec", totalOps.getValue() / testDuration, Averaged::True);
			m.push_back(conflicts.getMetric());
			m.emplace_back("Conflicts/sec", conflicts.getValue() / testDuration, Averaged::True);
			m.push_back(retries.getMetric());

			// count of each operation
			for (int i = 0; i < MAX_OP; ++i) {
				m.push_back(opCounters[i].getMetric());
			}

			// Meaningful Latency metrics
			const int opExecutedAtOnce[] = { OP_GETREADVERSION, OP_GET, OP_GETRANGE, OP_SGET, OP_SGETRANGE, OP_COMMIT };
			for (const int& op : opExecutedAtOnce) {
				m.emplace_back("Mean " + opNames[op] + " Latency (us)", 1e6 * opLatencies[op].mean(), Averaged::True);
				m.emplace_back(
				    "Max " + opNames[op] + " Latency (us, averaged)", 1e6 * opLatencies[op].max(), Averaged::True);
				m.emplace_back(
				    "Min " + opNames[op] + " Latency (us, averaged)", 1e6 * opLatencies[op].min(), Averaged::True);
			}
			// Latency for local operations if needed
			if (latencyForLocalOperation) {
				const int localOp[] = { OP_INSERT, OP_CLEAR, OP_CLEARRANGE };
				for (const int& op : localOp) {
					TraceEvent(SevDebug, "LocalLatency")
					    .detail("Name", opNames[op])
					    .detail("Size", opLatencies[op].getPopulationSize());
					m.emplace_back(
					    "Mean " + opNames[op] + " Latency (us)", 1e6 * opLatencies[op].mean(), Averaged::True);
					m.emplace_back(
					    "Max " + opNames[op] + " Latency (us, averaged)", 1e6 * opLatencies[op].max(), Averaged::True);
					m.emplace_back(
					    "Min " + opNames[op] + " Latency (us, averaged)", 1e6 * opLatencies[op].min(), Averaged::True);
				}
			}

			// insert logging metrics if exists
			m.insert(m.end(), periodicMetrics.begin(), periodicMetrics.end());
		}
	}
	static std::string randStr(int len) {
		std::string result(len, '.');
		for (int i = 0; i < len; ++i) {
			result[i] = deterministicRandom()->randomAlphaNumeric();
		}
		return result;
	}

	static void randStr(char* str, int len) {
		for (int i = 0; i < len; ++i) {
			str[i] = deterministicRandom()->randomAlphaNumeric();
		}
	}

	Value randomValue() {
		const int length = deterministicRandom()->randomInt(minValueBytes, maxValueBytes + 1);
		std::string valueString = randStr(length);
		return StringRef(reinterpret_cast<const uint8_t*>(valueString.c_str()), length);
	}

	Key keyForIndex(uint64_t ind) {
		Key result = makeString(keyBytes);
		char* data = reinterpret_cast<char*>(mutateString(result));
		format((keyPrefix + "%0*d").c_str(), seqNumLen, ind).copy(data, KEYPREFIXLEN + seqNumLen);
		for (int i = KEYPREFIXLEN + seqNumLen; i < keyBytes; ++i)
			data[i] = 'x';
		return result;
	}

	/* number of digits */
	static uint64_t digits(uint64_t num) {
		uint64_t digits = 0;
		while (num > 0) {
			num /= 10;
			digits++;
		}
		return digits;
	}

	static void updateCSFlags(MakoWorkload* self, std::vector<bool>& flags, uint64_t startIdx, uint64_t endIdx) {
		// We deal with cases where rowCount % csCount != 0 and csPartitionSize % csSize != 0;
		// In particular, all keys with index in range [csSize * csPartitionSize, rowCount) will not be used for
		// checksum By the same way, for any i in range [0, csSize): keys with index in range [ i*csPartitionSize,
		// i*csPartitionSize + csCount*csStepSizeInPartition) will not be used for checksum
		uint64_t boundary = self->csSize * self->csPartitionSize;
		if (startIdx >= boundary)
			return;
		else if (endIdx > boundary)
			endIdx = boundary;

		// If all checksums need to be updated, just return
		if (std::all_of(flags.begin(), flags.end(), [](bool flag) { return flag; }))
			return;

		if (startIdx + 1 == endIdx) {
			// single key case
			startIdx = startIdx % self->csPartitionSize;
			if ((startIdx < self->csCount * self->csStepSizeInPartition) &&
			    (startIdx % self->csStepSizeInPartition == 0)) {
				flags.at(startIdx / self->csStepSizeInPartition) = true;
			}
		} else {
			// key range case
			uint64_t count = self->csCount;
			uint64_t base = (startIdx / self->csPartitionSize) * self->csPartitionSize;
			startIdx -= base;
			endIdx -= base;
			uint64_t startStepIdx = std::min(startIdx / self->csStepSizeInPartition, self->csCount - 1);

			// if changed range size is more than one csPartitionSize, which means every checksum needs to be updated
			if ((endIdx - startIdx) < self->csPartitionSize) {
				uint64_t endStepIdx;
				if (endIdx > self->csPartitionSize) {
					endStepIdx =
					    self->csCount +
					    std::min((endIdx - 1 - self->csPartitionSize) / self->csStepSizeInPartition, self->csCount);
				} else {
					endStepIdx = std::min((endIdx - 1) / self->csStepSizeInPartition, self->csCount - 1);
				}
				// All the left boundary of csStep should be updated
				// Also, check the startIdx whether it is the left boundary of a csStep
				if (startIdx == self->csStepSizeInPartition * startStepIdx)
					flags[startStepIdx] = true;
				count = endStepIdx - startStepIdx;
			}
			for (int i = 1; i <= count; ++i) {
				flags[(startStepIdx + i) % self->csCount] = true;
			}
		}
	}
	Standalone<KeyValueRef> operator()(uint64_t n) { return KeyValueRef(keyForIndex(n), randomValue()); }

																#line 343 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via tracePeriodically()
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class TracePeriodicallyActor>
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class TracePeriodicallyActorState {
															#line 349 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	TracePeriodicallyActorState(MakoWorkload* const& self) 
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : self(self),
															#line 342 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   start(timer()),
															#line 343 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   elapsed(0.0),
															#line 344 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   last_ops(0),
															#line 345 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   last_xacts(0)
															#line 364 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("tracePeriodically", reinterpret_cast<unsigned long>(this));

	}
	~TracePeriodicallyActorState() 
	{
		fdb_probe_actor_destroy("tracePeriodically", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 347 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			;
															#line 379 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TracePeriodicallyActorState();
		static_cast<TracePeriodicallyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 348 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		elapsed += self->periodicLoggingInterval;
															#line 349 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delayUntil(start + elapsed);
															#line 349 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<TracePeriodicallyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 413 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<TracePeriodicallyActor*>(this)->actor_wait_state = 1;
															#line 349 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TracePeriodicallyActor, 0, Void >*>(static_cast<TracePeriodicallyActor*>(this)));
															#line 418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 350 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent((self->description() + "_CommitLatency").c_str()) .detail("Mean", self->opLatencies[OP_COMMIT].mean()) .detail("Median", self->opLatencies[OP_COMMIT].median()) .detail("Percentile5", self->opLatencies[OP_COMMIT].percentile(.05)) .detail("Percentile95", self->opLatencies[OP_COMMIT].percentile(.95)) .detail("Count", self->opCounters[OP_COMMIT].getValue()) .detail("Elapsed", elapsed);
															#line 357 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent((self->description() + "_GRVLatency").c_str()) .detail("Mean", self->opLatencies[OP_GETREADVERSION].mean()) .detail("Median", self->opLatencies[OP_GETREADVERSION].median()) .detail("Percentile5", self->opLatencies[OP_GETREADVERSION].percentile(.05)) .detail("Percentile95", self->opLatencies[OP_GETREADVERSION].percentile(.95)) .detail("Count", self->opCounters[OP_GETREADVERSION].getValue());
															#line 364 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		std::string ts = format("T=%04.0fs: ", elapsed);
															#line 365 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->periodicMetrics.emplace_back(ts + "Transactions/sec", (self->xacts.getValue() - last_xacts) / self->periodicLoggingInterval, Averaged::False);
															#line 368 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->periodicMetrics.emplace_back(ts + "Operations/sec", (self->totalOps.getValue() - last_ops) / self->periodicLoggingInterval, Averaged::False);
															#line 372 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		last_xacts = self->xacts.getValue();
															#line 373 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		last_ops = self->totalOps.getValue();
															#line 439 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 350 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent((self->description() + "_CommitLatency").c_str()) .detail("Mean", self->opLatencies[OP_COMMIT].mean()) .detail("Median", self->opLatencies[OP_COMMIT].median()) .detail("Percentile5", self->opLatencies[OP_COMMIT].percentile(.05)) .detail("Percentile95", self->opLatencies[OP_COMMIT].percentile(.95)) .detail("Count", self->opCounters[OP_COMMIT].getValue()) .detail("Elapsed", elapsed);
															#line 357 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent((self->description() + "_GRVLatency").c_str()) .detail("Mean", self->opLatencies[OP_GETREADVERSION].mean()) .detail("Median", self->opLatencies[OP_GETREADVERSION].median()) .detail("Percentile5", self->opLatencies[OP_GETREADVERSION].percentile(.05)) .detail("Percentile95", self->opLatencies[OP_GETREADVERSION].percentile(.95)) .detail("Count", self->opCounters[OP_GETREADVERSION].getValue());
															#line 364 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		std::string ts = format("T=%04.0fs: ", elapsed);
															#line 365 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->periodicMetrics.emplace_back(ts + "Transactions/sec", (self->xacts.getValue() - last_xacts) / self->periodicLoggingInterval, Averaged::False);
															#line 368 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->periodicMetrics.emplace_back(ts + "Operations/sec", (self->totalOps.getValue() - last_ops) / self->periodicLoggingInterval, Averaged::False);
															#line 372 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		last_xacts = self->xacts.getValue();
															#line 373 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		last_ops = self->totalOps.getValue();
															#line 460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TracePeriodicallyActor*>(this)->actor_wait_state > 0) static_cast<TracePeriodicallyActor*>(this)->actor_wait_state = 0;
		static_cast<TracePeriodicallyActor*>(this)->ActorCallback< TracePeriodicallyActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TracePeriodicallyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("tracePeriodically", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TracePeriodicallyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tracePeriodically", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TracePeriodicallyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("tracePeriodically", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TracePeriodicallyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tracePeriodically", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TracePeriodicallyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("tracePeriodically", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TracePeriodicallyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tracePeriodically", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 342 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	double start;
															#line 343 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	double elapsed;
															#line 344 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int64_t last_ops;
															#line 345 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int64_t last_xacts;
															#line 550 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via tracePeriodically()
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class TracePeriodicallyActor final : public Actor<Void>, public ActorCallback< TracePeriodicallyActor, 0, Void >, public FastAllocated<TracePeriodicallyActor>, public TracePeriodicallyActorState<TracePeriodicallyActor> {
															#line 555 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<TracePeriodicallyActor>::operator new;
	using FastAllocated<TracePeriodicallyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6802699535875629056UL, 15988797751672558336UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TracePeriodicallyActor, 0, Void >;
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	TracePeriodicallyActor(MakoWorkload* const& self) 
															#line 572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   TracePeriodicallyActorState<TracePeriodicallyActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tracePeriodically", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15798896764292167168UL, 11037908824035524864UL);
		ActorExecutionContextHelper __helper(static_cast<TracePeriodicallyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tracePeriodically");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tracePeriodically", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TracePeriodicallyActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] static Future<Void> tracePeriodically( MakoWorkload* const& self ) {
															#line 341 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new TracePeriodicallyActor(self));
															#line 604 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
																#line 608 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via _setup()
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class _setupActor>
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class _setupActorState {
															#line 614 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	_setupActorState(Database const& cx,MakoWorkload* const& self) 
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : cx(cx),
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self)
															#line 623 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("_setup", reinterpret_cast<unsigned long>(this));

	}
	~_setupActorState() 
	{
		fdb_probe_actor_destroy("_setup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 378 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (self->populateData)
															#line 638 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			{
															#line 379 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				loadTime = Promise<double>();
															#line 380 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				ratesAtKeyCounts = Promise<std::vector<std::pair<uint64_t, double>>>();
															#line 382 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				StrictFuture<Void> __when_expr_0 = bulkSetup(cx, self, self->rowCount, loadTime, self->insertionCountsToMeasure.empty(), self->warmingDelay, self->maxInsertRate, self->insertionCountsToMeasure, ratesAtKeyCounts);
															#line 382 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 648 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<_setupActor*>(this)->actor_wait_state = 1;
															#line 382 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 0, Void >*>(static_cast<_setupActor*>(this)));
															#line 653 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 398 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (self->checksumVerification && self->clientId == 0)
															#line 681 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 399 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_1 = generateChecksum(cx, self);
															#line 399 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 687 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<_setupActor*>(this)->actor_wait_state = 2;
															#line 399 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 1, Void >*>(static_cast<_setupActor*>(this)));
															#line 692 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 393 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->loadTime = loadTime.getFuture().get();
															#line 395 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->ratesAtKeyCounts = ratesAtKeyCounts.getFuture().get();
															#line 708 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 393 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->loadTime = loadTime.getFuture().get();
															#line 395 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->ratesAtKeyCounts = ratesAtKeyCounts.getFuture().get();
															#line 719 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _setupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 402 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 803 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _setupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Database cx;
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 379 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Promise<double> loadTime;
															#line 380 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Promise<std::vector<std::pair<uint64_t, double>>> ratesAtKeyCounts;
															#line 906 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via _setup()
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class _setupActor final : public Actor<Void>, public ActorCallback< _setupActor, 0, Void >, public ActorCallback< _setupActor, 1, Void >, public FastAllocated<_setupActor>, public _setupActorState<_setupActor> {
															#line 911 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<_setupActor>::operator new;
	using FastAllocated<_setupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12422264608937916416UL, 10338400788025701120UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _setupActor, 0, Void >;
friend struct ActorCallback< _setupActor, 1, Void >;
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	_setupActor(Database const& cx,MakoWorkload* const& self) 
															#line 929 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   _setupActorState<_setupActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3044502327521131520UL, 13387266639422932480UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_setup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _setupActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _setupActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] Future<Void> _setup( Database const& cx, MakoWorkload* const& self ) {
															#line 376 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new _setupActor(cx, self));
															#line 962 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 404 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

																#line 967 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via _start()
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class _startActor>
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class _startActorState {
															#line 973 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	_startActorState(Database const& cx,MakoWorkload* const& self) 
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : cx(cx),
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self)
															#line 982 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("_start", reinterpret_cast<unsigned long>(this));

	}
	~_startActorState() 
	{
		fdb_probe_actor_destroy("_start", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 407 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (self->runBenchmark)
															#line 997 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			{
															#line 408 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				StrictFuture<Void> __when_expr_0 = self->_runBenchmark(cx, self);
															#line 408 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1003 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<_startActor*>(this)->actor_wait_state = 1;
															#line 408 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _startActor, 0, Void >*>(static_cast<_startActor*>(this)));
															#line 1008 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_startActorState();
		static_cast<_startActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 410 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!self->preserveData && self->clientId == 0)
															#line 1036 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 411 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_1 = self->cleanup(cx, self);
															#line 411 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1042 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<_startActor*>(this)->actor_wait_state = 2;
															#line 411 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _startActor, 1, Void >*>(static_cast<_startActor*>(this)));
															#line 1047 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _startActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _startActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 413 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<_startActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_startActorState(); static_cast<_startActor*>(this)->destroy(); return 0; }
															#line 1148 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<_startActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_startActorState();
		static_cast<_startActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _startActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _startActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Database cx;
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 1247 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via _start()
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class _startActor final : public Actor<Void>, public ActorCallback< _startActor, 0, Void >, public ActorCallback< _startActor, 1, Void >, public FastAllocated<_startActor>, public _startActorState<_startActor> {
															#line 1252 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<_startActor>::operator new;
	using FastAllocated<_startActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9751680300496770048UL, 7855157880303864576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _startActor, 0, Void >;
friend struct ActorCallback< _startActor, 1, Void >;
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	_startActor(Database const& cx,MakoWorkload* const& self) 
															#line 1270 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   _startActorState<_startActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16516870674810565632UL, 12235738791954228224UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_start");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _startActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _startActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] Future<Void> _start( Database const& cx, MakoWorkload* const& self ) {
															#line 405 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new _startActor(cx, self));
															#line 1303 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 415 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

																#line 1308 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via _runBenchmark()
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class _runBenchmarkActor>
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class _runBenchmarkActorState {
															#line 1314 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	_runBenchmarkActorState(Database const& cx,MakoWorkload* const& self) 
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : cx(cx),
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self)
															#line 1323 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("_runBenchmark", reinterpret_cast<unsigned long>(this));

	}
	~_runBenchmarkActorState() 
	{
		fdb_probe_actor_destroy("_runBenchmark", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 417 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			std::vector<Future<Void>> clients;
															#line 418 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			clients.reserve(self->actorCountPerClient);
															#line 419 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			for(int c = 0;c < self->actorCountPerClient;++c) {
															#line 420 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				clients.push_back(self->makoClient(cx, self, self->actorCountPerClient / self->transactionsPerSecond, c));
															#line 1344 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			}
															#line 423 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (self->enableLogging)
															#line 1348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			{
															#line 424 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				clients.push_back(tracePeriodically(self));
															#line 1352 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			}
															#line 426 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeout(waitForAll(clients), self->testDuration, Void());
															#line 426 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<_runBenchmarkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1358 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_runBenchmarkActor*>(this)->actor_wait_state = 1;
															#line 426 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _runBenchmarkActor, 0, Void >*>(static_cast<_runBenchmarkActor*>(this)));
															#line 1363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_runBenchmarkActorState();
		static_cast<_runBenchmarkActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 427 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<_runBenchmarkActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_runBenchmarkActorState(); static_cast<_runBenchmarkActor*>(this)->destroy(); return 0; }
															#line 1386 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<_runBenchmarkActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_runBenchmarkActorState();
		static_cast<_runBenchmarkActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 427 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<_runBenchmarkActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_runBenchmarkActorState(); static_cast<_runBenchmarkActor*>(this)->destroy(); return 0; }
															#line 1398 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<_runBenchmarkActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_runBenchmarkActorState();
		static_cast<_runBenchmarkActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_runBenchmarkActor*>(this)->actor_wait_state > 0) static_cast<_runBenchmarkActor*>(this)->actor_wait_state = 0;
		static_cast<_runBenchmarkActor*>(this)->ActorCallback< _runBenchmarkActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _runBenchmarkActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_runBenchmark", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_runBenchmarkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_runBenchmark", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _runBenchmarkActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_runBenchmark", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_runBenchmarkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_runBenchmark", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _runBenchmarkActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_runBenchmark", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_runBenchmarkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_runBenchmark", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Database cx;
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 1485 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via _runBenchmark()
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class _runBenchmarkActor final : public Actor<Void>, public ActorCallback< _runBenchmarkActor, 0, Void >, public FastAllocated<_runBenchmarkActor>, public _runBenchmarkActorState<_runBenchmarkActor> {
															#line 1490 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<_runBenchmarkActor>::operator new;
	using FastAllocated<_runBenchmarkActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10223031878351485696UL, 7660867352996132096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _runBenchmarkActor, 0, Void >;
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	_runBenchmarkActor(Database const& cx,MakoWorkload* const& self) 
															#line 1507 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   _runBenchmarkActorState<_runBenchmarkActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_runBenchmark", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(971222033845392128UL, 4646333185837490432UL);
		ActorExecutionContextHelper __helper(static_cast<_runBenchmarkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_runBenchmark");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_runBenchmark", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _runBenchmarkActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] Future<Void> _runBenchmark( Database const& cx, MakoWorkload* const& self ) {
															#line 416 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new _runBenchmarkActor(cx, self));
															#line 1539 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 429 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

																#line 1544 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via makoClient()
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class MakoClientActor>
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class MakoClientActorState {
															#line 1550 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoClientActorState(Database const& cx,MakoWorkload* const& self,double const& delay,int const& actorIndex) 
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : cx(cx),
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self),
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   delay(delay),
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   actorIndex(actorIndex),
															#line 432 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   rkey2(),
															#line 433 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   rval(),
															#line 434 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   tr(cx),
															#line 435 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   doCommit(),
															#line 436 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   count(),
															#line 437 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   rangeLen(),
															#line 438 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   rkeyRangeRef(),
															#line 439 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   perOpCount(MAX_OP, 0),
															#line 441 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   csChangedFlags(self->csCount, false),
															#line 442 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   lastTime(timer()),
															#line 443 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   commitStart()
															#line 1585 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("makoClient", reinterpret_cast<unsigned long>(this));

	}
	~MakoClientActorState() 
	{
		fdb_probe_actor_destroy("makoClient", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 445 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent("ClientStarting") .detail("ActorIndex", actorIndex) .detail("ClientIndex", self->clientId) .detail("NumActors", self->actorCountPerClient);
															#line 450 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			;
															#line 1602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MakoClientActorState();
		static_cast<MakoClientActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 452 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_0 = poisson(&lastTime, delay);
															#line 452 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1634 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MakoClientActor*>(this)->actor_wait_state = 1;
															#line 452 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 0, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 1639 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 455 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			doCommit = self->commitGet;
															#line 456 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			i = 0;
															#line 1651 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 455 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			doCommit = self->commitGet;
															#line 456 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			i = 0;
															#line 1669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 636 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		std::fill(perOpCount.begin(), perOpCount.end(), 0);
															#line 637 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		tr.reset();
															#line 1761 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 626 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent("FailedToExecOperations").error(e);
															#line 627 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 1773 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			{
															#line 628 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 1777 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			}
			else
			{
															#line 629 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				if (e.code() == error_code_not_committed)
															#line 1783 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				{
															#line 630 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
					++self->conflicts;
															#line 1787 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				}
			}
															#line 632 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_13 = tr.onError(e);
															#line 632 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1Catch1(__when_expr_13.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_13.get(), loopDepth); };
			static_cast<MakoClientActor*>(this)->actor_wait_state = 14;
															#line 632 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_13.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 13, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 1799 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 612 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (doCommit)
															#line 1814 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 613 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_11 = self->updateCSBeforeCommit(&tr, self, &csChangedFlags);
															#line 613 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1820 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1loopBody1cont1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_11.get(), loopDepth); };
			static_cast<MakoClientActor*>(this)->actor_wait_state = 12;
															#line 613 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 11, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 1825 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 456 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!(i < MAX_OP))
															#line 1846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
			return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 457 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (i == OP_COMMIT)
															#line 1852 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
			return a_body1loopBody1cont1continue1(loopDepth); // continue
		}
															#line 459 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		count = 0;
															#line 1858 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1continue1(int loopDepth) 
	{
															#line 456 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++i;
															#line 1880 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(int loopDepth) 
	{
															#line 456 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++i;
															#line 1889 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1(int loopDepth) 
	{
															#line 459 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!(count < self->operations[i][0]))
															#line 1905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
			return a_body1loopBody1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 460 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		range = self->operations[i][1];
															#line 461 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		rangeLen = digits(range);
															#line 463 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		indBegin = self->getRandomKeyIndex(self->rowCount);
															#line 464 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		rkey = self->keyForIndex(indBegin);
															#line 465 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		rval = self->randomValue();
															#line 466 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		indEnd = std::min(indBegin + range, self->rowCount);
															#line 467 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		rkey2 = self->keyForIndex(indEnd);
															#line 469 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		rkeyRangeRef = KeyRangeRef(rkey, rkey2);
															#line 472 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (self->checksumVerification)
															#line 1927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 473 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (i == OP_INSERT | i == OP_UPDATE | i == OP_CLEAR)
															#line 1931 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			{
															#line 474 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				updateCSFlags(self, csChangedFlags, indBegin, indBegin + 1);
															#line 1935 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			}
			else
			{
															#line 475 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				if (i == OP_CLEARRANGE)
															#line 1941 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				{
															#line 476 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
					updateCSFlags(self, csChangedFlags, indBegin, indEnd);
															#line 1945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				}
			}
		}
															#line 480 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (i == OP_GETREADVERSION)
															#line 1951 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 481 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_1 = logLatency(tr.getReadVersion(), &self->opLatencies[i]);
															#line 481 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1957 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<MakoClientActor*>(this)->actor_wait_state = 2;
															#line 481 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 1, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 1962 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 482 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (i == OP_GET)
															#line 1969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			{
															#line 483 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				StrictFuture<Void> __when_expr_2 = logLatency(tr.get(rkey, Snapshot::False), &self->opLatencies[i]);
															#line 483 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1975 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1when2(__when_expr_2.get(), loopDepth); };
				static_cast<MakoClientActor*>(this)->actor_wait_state = 3;
															#line 483 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 2, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 1980 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 484 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				if (i == OP_GETRANGE)
															#line 1987 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				{
															#line 485 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
					StrictFuture<Void> __when_expr_3 = logLatency(tr.getRange(rkeyRangeRef, CLIENT_KNOBS->TOO_MANY, Snapshot::False), &self->opLatencies[i]);
															#line 485 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
					if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1993 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
					if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1when3(__when_expr_3.get(), loopDepth); };
					static_cast<MakoClientActor*>(this)->actor_wait_state = 4;
															#line 485 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
					__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 3, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 1998 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 487 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
					if (i == OP_SGET)
															#line 2005 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
					{
															#line 488 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
						StrictFuture<Void> __when_expr_4 = logLatency(tr.get(rkey, Snapshot::True), &self->opLatencies[i]);
															#line 488 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
						if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2011 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
						if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1when4(__when_expr_4.get(), loopDepth); };
						static_cast<MakoClientActor*>(this)->actor_wait_state = 5;
															#line 488 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
						__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 4, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 2016 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
															#line 489 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
						if (i == OP_SGETRANGE)
															#line 2023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
						{
															#line 491 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
							StrictFuture<Void> __when_expr_5 = logLatency(tr.getRange(rkeyRangeRef, CLIENT_KNOBS->TOO_MANY, Snapshot::True), &self->opLatencies[i]);
															#line 491 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
							if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2029 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
							if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1cont1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1when5(__when_expr_5.get(), loopDepth); };
							static_cast<MakoClientActor*>(this)->actor_wait_state = 6;
															#line 491 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
							__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 5, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 2034 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
							loopDepth = 0;
						}
						else
						{
															#line 493 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
							if (i == OP_UPDATE)
															#line 2041 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
							{
															#line 494 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
								StrictFuture<Void> __when_expr_6 = logLatency(tr.get(rkey, Snapshot::False), &self->opLatencies[OP_GET]);
															#line 494 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
								if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2047 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
								if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1cont1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1when6(__when_expr_6.get(), loopDepth); };
								static_cast<MakoClientActor*>(this)->actor_wait_state = 7;
															#line 494 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
								__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 6, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 2052 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 503 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
								if (i == OP_INSERT)
															#line 2059 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
								{
															#line 506 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
									randStr(reinterpret_cast<char*>(mutateString(rkey)) + self->KEYPREFIXLEN, self->keyBytes - self->KEYPREFIXLEN);
															#line 508 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
									if (self->latencyForLocalOperation)
															#line 2065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
									{
															#line 509 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										double opBegin = timer();
															#line 510 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										tr.set(rkey, rval);
															#line 511 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										self->opLatencies[OP_INSERT].addSample(timer() - opBegin);
															#line 2073 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
									}
									else
									{
															#line 513 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										tr.set(rkey, rval);
															#line 2079 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
									}
															#line 515 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
									doCommit = true;
															#line 2083 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
									loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont18(loopDepth);
								}
								else
								{
															#line 516 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
									if (i == OP_INSERTRANGE)
															#line 2090 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
									{
															#line 517 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										char* rkeyPtr = reinterpret_cast<char*>(mutateString(rkey));
															#line 518 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										randStr(rkeyPtr + self->KEYPREFIXLEN, self->keyBytes - self->KEYPREFIXLEN);
															#line 519 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										for(int range_i = 0;range_i < range;++range_i) {
															#line 520 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
											format("%0.*d", rangeLen, range_i).copy(rkeyPtr + self->keyBytes - rangeLen, rangeLen);
															#line 521 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
											if (self->latencyForLocalOperation)
															#line 2102 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
											{
															#line 522 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												double opBegin = timer();
															#line 523 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												tr.set(rkey, self->randomValue());
															#line 524 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												self->opLatencies[OP_INSERT].addSample(timer() - opBegin);
															#line 2110 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
											}
											else
											{
															#line 526 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												tr.set(rkey, self->randomValue());
															#line 2116 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
											}
										}
															#line 529 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										doCommit = true;
															#line 2121 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
										loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont22(loopDepth);
									}
									else
									{
															#line 530 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
										if (i == OP_CLEAR)
															#line 2128 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
										{
															#line 531 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
											if (self->latencyForLocalOperation)
															#line 2132 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
											{
															#line 532 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												double opBegin = timer();
															#line 533 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												tr.clear(rkey);
															#line 534 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												self->opLatencies[OP_CLEAR].addSample(timer() - opBegin);
															#line 2140 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
											}
											else
											{
															#line 536 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												tr.clear(rkey);
															#line 2146 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
											}
															#line 538 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
											doCommit = true;
															#line 2150 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
											loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont27(loopDepth);
										}
										else
										{
															#line 539 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
											if (i == OP_SETCLEAR)
															#line 2157 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
											{
															#line 540 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												randStr(reinterpret_cast<char*>(mutateString(rkey)) + self->KEYPREFIXLEN, self->keyBytes - self->KEYPREFIXLEN);
															#line 542 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												if (self->latencyForLocalOperation)
															#line 2163 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
												{
															#line 543 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
													double opBegin = timer();
															#line 544 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
													tr.set(rkey, rval);
															#line 545 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
													self->opLatencies[OP_INSERT].addSample(timer() - opBegin);
															#line 2171 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
												}
												else
												{
															#line 547 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
													tr.set(rkey, rval);
															#line 2177 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
												}
															#line 549 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												StrictFuture<Void> __when_expr_7 = self->updateCSBeforeCommit(&tr, self, &csChangedFlags);
															#line 549 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2183 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
												if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1cont1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1when7(__when_expr_7.get(), loopDepth); };
												static_cast<MakoClientActor*>(this)->actor_wait_state = 8;
															#line 549 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 7, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 2188 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
												loopDepth = 0;
											}
											else
											{
															#line 564 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
												if (i == OP_CLEARRANGE)
															#line 2195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
												{
															#line 565 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
													if (self->latencyForLocalOperation)
															#line 2199 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
													{
															#line 566 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														double opBegin = timer();
															#line 567 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														tr.clear(rkeyRangeRef);
															#line 568 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														self->opLatencies[OP_CLEARRANGE].addSample(timer() - opBegin);
															#line 2207 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
													}
													else
													{
															#line 570 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														tr.clear(rkeyRangeRef);
															#line 2213 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
													}
															#line 572 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
													doCommit = true;
															#line 2217 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
													loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont35(loopDepth);
												}
												else
												{
															#line 573 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
													if (i == OP_SETCLEARRANGE)
															#line 2224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
													{
															#line 574 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														char* rkeyPtr = reinterpret_cast<char*>(mutateString(rkey));
															#line 575 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														randStr(rkeyPtr + self->KEYPREFIXLEN, self->keyBytes - self->KEYPREFIXLEN);
															#line 576 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														scr_start_key = std::string();
															#line 577 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														scr_end_key = std::string();
															#line 578 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														scr_key_range_ref = KeyRangeRef();
															#line 579 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														for(int range_i = 0;range_i < range;++range_i) {
															#line 580 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															format("%0.*d", rangeLen, range_i).copy(rkeyPtr + self->keyBytes - rangeLen, rangeLen);
															#line 581 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															if (self->latencyForLocalOperation)
															#line 2242 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
															{
															#line 582 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
																double opBegin = timer();
															#line 583 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
																tr.set(rkey, self->randomValue());
															#line 584 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
																self->opLatencies[OP_INSERT].addSample(timer() - opBegin);
															#line 2250 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
															}
															else
															{
															#line 586 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
																tr.set(rkey, self->randomValue());
															#line 2256 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
															}
															#line 588 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															if (range_i == 0)
															#line 2260 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
															{
															#line 589 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
																scr_start_key = rkey.toString();
															#line 2264 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
															}
														}
															#line 591 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														scr_end_key = rkey.toString();
															#line 592 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														scr_key_range_ref = KeyRangeRef(KeyRef(scr_start_key), KeyRef(scr_end_key));
															#line 593 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														StrictFuture<Void> __when_expr_9 = self->updateCSBeforeCommit(&tr, self, &csChangedFlags);
															#line 593 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2275 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
														if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1loopBody1cont1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1when8(__when_expr_9.get(), loopDepth); };
														static_cast<MakoClientActor*>(this)->actor_wait_state = 10;
															#line 593 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
														__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 9, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 2280 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
														loopDepth = 0;
													}
													else
													{
														loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont39(loopDepth);
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 608 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++perOpCount[i];
															#line 459 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++count;
															#line 2320 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont8(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont10(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont12(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont13(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont13(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont14(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont15(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont15(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont15(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont15(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont16(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont17(Void const& _,int loopDepth) 
	{
															#line 495 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (self->latencyForLocalOperation)
															#line 2794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 496 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			double opBegin = timer();
															#line 497 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.set(rkey, rval);
															#line 498 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			self->opLatencies[OP_INSERT].addSample(timer() - opBegin);
															#line 2802 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		else
		{
															#line 500 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.set(rkey, rval);
															#line 2808 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 502 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		doCommit = true;
															#line 2812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont16(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont17(Void && _,int loopDepth) 
	{
															#line 495 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (self->latencyForLocalOperation)
															#line 2821 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 496 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			double opBegin = timer();
															#line 497 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.set(rkey, rval);
															#line 498 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			self->opLatencies[OP_INSERT].addSample(timer() - opBegin);
															#line 2829 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		else
		{
															#line 500 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.set(rkey, rval);
															#line 2835 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 502 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		doCommit = true;
															#line 2839 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont16(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont17(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont17(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when6(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont18(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont16(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont22(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont27(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont22(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont31(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont27(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont32(Void const& _,int loopDepth) 
	{
															#line 551 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		commitStart = timer();
															#line 552 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_8 = tr.commit();
															#line 552 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2951 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1loopBody1cont1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1cont32when1(__when_expr_8.get(), loopDepth); };
		static_cast<MakoClientActor*>(this)->actor_wait_state = 9;
															#line 552 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 8, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 2956 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont32(Void && _,int loopDepth) 
	{
															#line 551 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		commitStart = timer();
															#line 552 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_8 = tr.commit();
															#line 552 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1loopBody1cont1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1cont32when1(__when_expr_8.get(), loopDepth); };
		static_cast<MakoClientActor*>(this)->actor_wait_state = 9;
															#line 552 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 8, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 2974 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont32(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont32(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when7(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont32cont1(Void const& _,int loopDepth) 
	{
															#line 553 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->opLatencies[OP_COMMIT].addSample(timer() - commitStart);
															#line 554 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++perOpCount[OP_COMMIT];
															#line 555 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		tr.reset();
															#line 556 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (self->latencyForLocalOperation)
															#line 3064 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 557 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			double opBegin = timer();
															#line 558 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(rkey);
															#line 559 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			self->opLatencies[OP_CLEAR].addSample(timer() - opBegin);
															#line 3072 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		else
		{
															#line 561 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(rkey);
															#line 3078 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 563 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		doCommit = true;
															#line 3082 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont31(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont32cont1(Void && _,int loopDepth) 
	{
															#line 553 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->opLatencies[OP_COMMIT].addSample(timer() - commitStart);
															#line 554 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++perOpCount[OP_COMMIT];
															#line 555 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		tr.reset();
															#line 556 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (self->latencyForLocalOperation)
															#line 3097 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 557 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			double opBegin = timer();
															#line 558 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(rkey);
															#line 559 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			self->opLatencies[OP_CLEAR].addSample(timer() - opBegin);
															#line 3105 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		else
		{
															#line 561 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(rkey);
															#line 3111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 563 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		doCommit = true;
															#line 3115 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont31(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont32when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont32cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont32when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont32cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont1loopBody1loopBody1cont32when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont1loopBody1loopBody1cont32when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont35(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont31(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont39(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont35(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont40(Void const& _,int loopDepth) 
	{
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		commitStart = timer();
															#line 595 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_10 = tr.commit();
															#line 595 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 3215 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1loopBody1cont1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1cont40when1(__when_expr_10.get(), loopDepth); };
		static_cast<MakoClientActor*>(this)->actor_wait_state = 11;
															#line 595 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 10, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 3220 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont40(Void && _,int loopDepth) 
	{
															#line 594 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		commitStart = timer();
															#line 595 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_10 = tr.commit();
															#line 595 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 3233 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1loopBody1cont1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1loopBody1cont40when1(__when_expr_10.get(), loopDepth); };
		static_cast<MakoClientActor*>(this)->actor_wait_state = 11;
															#line 595 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 10, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 3238 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont40(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont40(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when8(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1loopBody1cont1loopBody1loopBody1cont40cont1(Void const& _,int loopDepth) 
	{
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->opLatencies[OP_COMMIT].addSample(timer() - commitStart);
															#line 597 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++perOpCount[OP_COMMIT];
															#line 598 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		tr.reset();
															#line 599 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (self->latencyForLocalOperation)
															#line 3328 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			double opBegin = timer();
															#line 601 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(scr_key_range_ref);
															#line 602 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			self->opLatencies[OP_CLEARRANGE].addSample(timer() - opBegin);
															#line 3336 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		else
		{
															#line 604 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(scr_key_range_ref);
															#line 3342 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 606 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		doCommit = true;
															#line 3346 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont39(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont40cont1(Void && _,int loopDepth) 
	{
															#line 596 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->opLatencies[OP_COMMIT].addSample(timer() - commitStart);
															#line 597 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++perOpCount[OP_COMMIT];
															#line 598 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		tr.reset();
															#line 599 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (self->latencyForLocalOperation)
															#line 3361 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 600 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			double opBegin = timer();
															#line 601 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(scr_key_range_ref);
															#line 602 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			self->opLatencies[OP_CLEARRANGE].addSample(timer() - opBegin);
															#line 3369 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		else
		{
															#line 604 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(scr_key_range_ref);
															#line 3375 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 606 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		doCommit = true;
															#line 3379 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont39(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont40when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont40cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont40when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont40cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1loopBody1cont1loopBody1loopBody1cont40when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1loopBody1cont1loopBody1loopBody1cont40when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 620 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++self->xacts;
															#line 621 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		for(int op = 0;op < MAX_OP;++op) {
															#line 622 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			self->opCounters[op] += perOpCount[op];
															#line 623 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			self->totalOps += perOpCount[op];
															#line 3469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 614 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		commitStart = timer();
															#line 615 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_12 = tr.commit();
															#line 615 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3483 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1loopBody1cont1Catch1(__when_expr_12.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_12.get(), loopDepth); };
		static_cast<MakoClientActor*>(this)->actor_wait_state = 13;
															#line 615 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 12, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 3488 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 614 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		commitStart = timer();
															#line 615 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_12 = tr.commit();
															#line 615 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<MakoClientActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1loopBody1cont1Catch1(__when_expr_12.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_12.get(), loopDepth); };
		static_cast<MakoClientActor*>(this)->actor_wait_state = 13;
															#line 615 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< MakoClientActor, 12, Void >*>(static_cast<MakoClientActor*>(this)));
															#line 3506 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 616 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->opLatencies[OP_COMMIT].addSample(timer() - commitStart);
															#line 617 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++perOpCount[OP_COMMIT];
															#line 3592 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 616 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		self->opLatencies[OP_COMMIT].addSample(timer() - commitStart);
															#line 617 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++perOpCount[OP_COMMIT];
															#line 3603 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1loopBody1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
															#line 633 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++self->retries;
															#line 3700 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
															#line 633 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++self->retries;
															#line 3709 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose14() 
	{
		if (static_cast<MakoClientActor*>(this)->actor_wait_state > 0) static_cast<MakoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MakoClientActor*>(this)->ActorCallback< MakoClientActor, 13, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 13, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorCallback< MakoClientActor, 13, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_error(ActorCallback< MakoClientActor, 13, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), 13);

	}
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Database cx;
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	double delay;
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int actorIndex;
															#line 432 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Key rkey, rkey2;
															#line 433 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Value rval;
															#line 434 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	ReadYourWritesTransaction tr;
															#line 435 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	bool doCommit;
															#line 436 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int i, count;
															#line 437 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	uint64_t range, indBegin, indEnd, rangeLen;
															#line 438 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	KeyRangeRef rkeyRangeRef;
															#line 439 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	std::vector<int> perOpCount;
															#line 441 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	std::vector<bool> csChangedFlags;
															#line 442 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	double lastTime;
															#line 443 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	double commitStart;
															#line 576 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	std::string scr_start_key;
															#line 577 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	std::string scr_end_key;
															#line 578 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	KeyRangeRef scr_key_range_ref;
															#line 3825 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via makoClient()
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class MakoClientActor final : public Actor<Void>, public ActorCallback< MakoClientActor, 0, Void >, public ActorCallback< MakoClientActor, 1, Void >, public ActorCallback< MakoClientActor, 2, Void >, public ActorCallback< MakoClientActor, 3, Void >, public ActorCallback< MakoClientActor, 4, Void >, public ActorCallback< MakoClientActor, 5, Void >, public ActorCallback< MakoClientActor, 6, Void >, public ActorCallback< MakoClientActor, 7, Void >, public ActorCallback< MakoClientActor, 8, Void >, public ActorCallback< MakoClientActor, 9, Void >, public ActorCallback< MakoClientActor, 10, Void >, public ActorCallback< MakoClientActor, 11, Void >, public ActorCallback< MakoClientActor, 12, Void >, public ActorCallback< MakoClientActor, 13, Void >, public FastAllocated<MakoClientActor>, public MakoClientActorState<MakoClientActor> {
															#line 3830 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<MakoClientActor>::operator new;
	using FastAllocated<MakoClientActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6550731396490403072UL, 10566241721354860800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MakoClientActor, 0, Void >;
friend struct ActorCallback< MakoClientActor, 1, Void >;
friend struct ActorCallback< MakoClientActor, 2, Void >;
friend struct ActorCallback< MakoClientActor, 3, Void >;
friend struct ActorCallback< MakoClientActor, 4, Void >;
friend struct ActorCallback< MakoClientActor, 5, Void >;
friend struct ActorCallback< MakoClientActor, 6, Void >;
friend struct ActorCallback< MakoClientActor, 7, Void >;
friend struct ActorCallback< MakoClientActor, 8, Void >;
friend struct ActorCallback< MakoClientActor, 9, Void >;
friend struct ActorCallback< MakoClientActor, 10, Void >;
friend struct ActorCallback< MakoClientActor, 11, Void >;
friend struct ActorCallback< MakoClientActor, 12, Void >;
friend struct ActorCallback< MakoClientActor, 13, Void >;
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoClientActor(Database const& cx,MakoWorkload* const& self,double const& delay,int const& actorIndex) 
															#line 3860 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   MakoClientActorState<MakoClientActor>(cx, self, delay, actorIndex),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("makoClient", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7120238470895654400UL, 8761946908235474432UL);
		ActorExecutionContextHelper __helper(static_cast<MakoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("makoClient");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("makoClient", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MakoClientActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MakoClientActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MakoClientActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MakoClientActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MakoClientActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MakoClientActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< MakoClientActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< MakoClientActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< MakoClientActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< MakoClientActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< MakoClientActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< MakoClientActor, 11, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< MakoClientActor, 12, Void >*)0, actor_cancelled()); break;
		case 14: this->a_callback_error((ActorCallback< MakoClientActor, 13, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] Future<Void> makoClient( Database const& cx, MakoWorkload* const& self, double const& delay, int const& actorIndex ) {
															#line 430 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new MakoClientActor(cx, self, delay, actorIndex));
															#line 3905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 640 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

																#line 3910 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via cleanup()
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class CleanupActor>
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class CleanupActorState {
															#line 3916 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	CleanupActorState(Database const& cx,MakoWorkload* const& self) 
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : cx(cx),
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self),
															#line 643 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   keyPrefix(self->keyPrefix),
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   tr(cx)
															#line 3929 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("cleanup", reinterpret_cast<unsigned long>(this));

	}
	~CleanupActorState() 
	{
		fdb_probe_actor_destroy("cleanup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 646 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			;
															#line 3944 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CleanupActorState();
		static_cast<CleanupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 658 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<CleanupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CleanupActorState(); static_cast<CleanupActor*>(this)->destroy(); return 0; }
															#line 3967 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<CleanupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CleanupActorState();
		static_cast<CleanupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 648 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.clear(prefixRange(keyPrefix));
															#line 649 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr.commit();
															#line 649 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<CleanupActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3991 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CleanupActor*>(this)->actor_wait_state = 1;
															#line 649 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CleanupActor, 0, Void >*>(static_cast<CleanupActor*>(this)));
															#line 3996 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 653 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent("FailedToCleanData").error(e);
															#line 654 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 654 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<CleanupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CleanupActor*>(this)->actor_wait_state = 2;
															#line 654 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CleanupActor, 1, Void >*>(static_cast<CleanupActor*>(this)));
															#line 4040 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent("CleanUpMakoRelatedData").detail("KeyPrefix", self->keyPrefix);
															#line 4055 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 650 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent("CleanUpMakoRelatedData").detail("KeyPrefix", self->keyPrefix);
															#line 4064 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CleanupActor*>(this)->actor_wait_state > 0) static_cast<CleanupActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupActor*>(this)->ActorCallback< CleanupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CleanupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CleanupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CleanupActor*>(this)->actor_wait_state > 0) static_cast<CleanupActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupActor*>(this)->ActorCallback< CleanupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CleanupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CleanupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanup", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Database cx;
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 643 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	std::string keyPrefix;
															#line 644 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	ReadYourWritesTransaction tr;
															#line 4239 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via cleanup()
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class CleanupActor final : public Actor<Void>, public ActorCallback< CleanupActor, 0, Void >, public ActorCallback< CleanupActor, 1, Void >, public FastAllocated<CleanupActor>, public CleanupActorState<CleanupActor> {
															#line 4244 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<CleanupActor>::operator new;
	using FastAllocated<CleanupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9040345950657581056UL, 11572414677467175936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CleanupActor, 0, Void >;
friend struct ActorCallback< CleanupActor, 1, Void >;
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	CleanupActor(Database const& cx,MakoWorkload* const& self) 
															#line 4262 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   CleanupActorState<CleanupActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("cleanup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17613239497936119296UL, 12602339259201360896UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("cleanup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("cleanup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CleanupActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CleanupActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] Future<Void> cleanup( Database const& cx, MakoWorkload* const& self ) {
															#line 641 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new CleanupActor(cx, self));
															#line 4295 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
																#line 4299 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via logLatency()
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class T, class LogLatencyActor>
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class LogLatencyActorState {
															#line 4305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	LogLatencyActorState(Future<T> const& f,DDSketch<double>* const& opLatencies) 
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : f(f),
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   opLatencies(opLatencies),
															#line 662 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   opBegin(timer())
															#line 4316 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("logLatency", reinterpret_cast<unsigned long>(this));

	}
	~LogLatencyActorState() 
	{
		fdb_probe_actor_destroy("logLatency", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 663 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(f);
															#line 663 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<LogLatencyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4333 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LogLatencyActor*>(this)->actor_wait_state = 1;
															#line 663 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LogLatencyActor, 0, Void >*>(static_cast<LogLatencyActor*>(this)));
															#line 4338 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LogLatencyActorState();
		static_cast<LogLatencyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 664 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		opLatencies->addSample(timer() - opBegin);
															#line 665 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<LogLatencyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LogLatencyActorState(); static_cast<LogLatencyActor*>(this)->destroy(); return 0; }
															#line 4363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<LogLatencyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LogLatencyActorState();
		static_cast<LogLatencyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 664 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		opLatencies->addSample(timer() - opBegin);
															#line 665 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<LogLatencyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LogLatencyActorState(); static_cast<LogLatencyActor*>(this)->destroy(); return 0; }
															#line 4377 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<LogLatencyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LogLatencyActorState();
		static_cast<LogLatencyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LogLatencyActor*>(this)->actor_wait_state > 0) static_cast<LogLatencyActor*>(this)->actor_wait_state = 0;
		static_cast<LogLatencyActor*>(this)->ActorCallback< LogLatencyActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LogLatencyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("logLatency", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LogLatencyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("logLatency", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LogLatencyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("logLatency", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LogLatencyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("logLatency", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LogLatencyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("logLatency", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LogLatencyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("logLatency", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Future<T> f;
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	DDSketch<double>* opLatencies;
															#line 662 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	double opBegin;
															#line 4466 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via logLatency()
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class T>
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class LogLatencyActor final : public Actor<Void>, public ActorCallback< LogLatencyActor<T>, 0, Void >, public FastAllocated<LogLatencyActor<T>>, public LogLatencyActorState<T, LogLatencyActor<T>> {
															#line 4473 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<LogLatencyActor<T>>::operator new;
	using FastAllocated<LogLatencyActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17739121634659300864UL, 281411185943536128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LogLatencyActor<T>, 0, Void >;
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	LogLatencyActor(Future<T> const& f,DDSketch<double>* const& opLatencies) 
															#line 4490 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   LogLatencyActorState<T, LogLatencyActor<T>>(f, opLatencies),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("logLatency", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10170994824737762048UL, 197458950419200512UL);
		ActorExecutionContextHelper __helper(static_cast<LogLatencyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("logLatency");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("logLatency", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LogLatencyActor<T>, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class T>
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] static Future<Void> logLatency( Future<T> const& f, DDSketch<double>* const& opLatencies ) {
															#line 660 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new LogLatencyActor<T>(f, opLatencies));
															#line 4524 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 667 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

	int64_t getRandomKeyIndex(uint64_t rowCount) {
		int64_t randomKeyIndex;
		if (zipf) {
			randomKeyIndex = zipfian_next();
		} else {
			randomKeyIndex = deterministicRandom()->randomInt64(0, rowCount);
		}
		return randomKeyIndex;
	}
	void parseOperationsSpec() {
		const char* ptr = operationsSpec.c_str();
		int op = 0;
		int rangeop = 0;
		int num;
		int error = 0;

		for (op = 0; op < MAX_OP; op++) {
			operations[op][OP_COUNT] = 0;
			operations[op][OP_RANGE] = 0;
		}

		op = 0;
		while (*ptr) {
			if (strncmp(ptr, "grv", 3) == 0) {
				op = OP_GETREADVERSION;
				ptr += 3;
			} else if (strncmp(ptr, "gr", 2) == 0) {
				op = OP_GETRANGE;
				rangeop = 1;
				ptr += 2;
			} else if (strncmp(ptr, "g", 1) == 0) {
				op = OP_GET;
				ptr++;
			} else if (strncmp(ptr, "sgr", 3) == 0) {
				op = OP_SGETRANGE;
				rangeop = 1;
				ptr += 3;
			} else if (strncmp(ptr, "sg", 2) == 0) {
				op = OP_SGET;
				ptr += 2;
			} else if (strncmp(ptr, "u", 1) == 0) {
				op = OP_UPDATE;
				ptr++;
			} else if (strncmp(ptr, "ir", 2) == 0) {
				op = OP_INSERTRANGE;
				rangeop = 1;
				ptr += 2;
			} else if (strncmp(ptr, "i", 1) == 0) {
				op = OP_INSERT;
				ptr++;
			} else if (strncmp(ptr, "cr", 2) == 0) {
				op = OP_CLEARRANGE;
				rangeop = 1;
				ptr += 2;
			} else if (strncmp(ptr, "c", 1) == 0) {
				op = OP_CLEAR;
				ptr++;
			} else if (strncmp(ptr, "scr", 3) == 0) {
				op = OP_SETCLEARRANGE;
				rangeop = 1;
				ptr += 3;
			} else if (strncmp(ptr, "sc", 2) == 0) {
				op = OP_SETCLEAR;
				ptr += 2;
			} else {
				error = 1;
				break;
			}

			/* count */
			num = 0;
			if ((*ptr < '0') || (*ptr > '9')) {
				num = 1; /* if omitted, set it to 1 */
			} else {
				while ((*ptr >= '0') && (*ptr <= '9')) {
					num = num * 10 + *ptr - '0';
					ptr++;
				}
			}
			/* set count */
			operations[op][OP_COUNT] = num;

			if (rangeop) {
				if (*ptr != ':') {
					error = 1;
					break;
				} else {
					ptr++; /* skip ':' */
					num = 0;
					if ((*ptr < '0') || (*ptr > '9')) {
						error = 1;
						break;
					}
					while ((*ptr >= '0') && (*ptr <= '9')) {
						num = num * 10 + *ptr - '0';
						ptr++;
					}
					/* set range */
					operations[op][OP_RANGE] = num;
				}
			}
			rangeop = 0;
		}

		if (error) {
			TraceEvent(SevError, "TestFailure")
			    .detail("Reason", "InvalidTransactionSpecification")
			    .detail("operations", operationsSpec);
		}
	}

																#line 4640 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via calcCheckSum()
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class CalcCheckSumActor>
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class CalcCheckSumActorState {
															#line 4646 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	CalcCheckSumActorState(ReadYourWritesTransaction* const& tr,MakoWorkload* const& self,int const& csIndex) 
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : tr(tr),
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self),
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   csIndex(csIndex),
															#line 780 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   result(0),
															#line 781 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   i(),
															#line 782 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   csKey()
															#line 4663 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("calcCheckSum", reinterpret_cast<unsigned long>(this));

	}
	~CalcCheckSumActorState() 
	{
		fdb_probe_actor_destroy("calcCheckSum", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 783 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			i = 0;
															#line 4678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CalcCheckSumActorState();
		static_cast<CalcCheckSumActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 795 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<CalcCheckSumActor*>(this)->SAV<uint32_t>::futures) { (void)(result); this->~CalcCheckSumActorState(); static_cast<CalcCheckSumActor*>(this)->destroy(); return 0; }
															#line 4701 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<CalcCheckSumActor*>(this)->SAV< uint32_t >::value()) uint32_t(std::move(result)); // state_var_RVO
		this->~CalcCheckSumActorState();
		static_cast<CalcCheckSumActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 783 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!(i < self->csSize))
															#line 4720 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 784 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		int idx = csIndex * self->csStepSizeInPartition + i * self->csPartitionSize;
															#line 785 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		csKey = self->keyForIndex(idx);
															#line 786 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_0 = tr->get(csKey);
															#line 786 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<CalcCheckSumActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4732 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<CalcCheckSumActor*>(this)->actor_wait_state = 1;
															#line 786 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CalcCheckSumActor, 0, Optional<Value> >*>(static_cast<CalcCheckSumActor*>(this)));
															#line 4737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Optional<Value> const& temp,int loopDepth) 
	{
															#line 787 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (temp.present())
															#line 4759 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 788 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			Value val = temp.get();
															#line 789 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			result = crc32c_append(result, val.begin(), val.size());
															#line 4765 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		else
		{
															#line 792 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			result = crc32c_append(result, csKey.begin(), csKey.size());
															#line 4771 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 783 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++i;
															#line 4775 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Optional<Value> && temp,int loopDepth) 
	{
															#line 787 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (temp.present())
															#line 4784 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 788 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			Value val = temp.get();
															#line 789 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			result = crc32c_append(result, val.begin(), val.size());
															#line 4790 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
		else
		{
															#line 792 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			result = crc32c_append(result, csKey.begin(), csKey.size());
															#line 4796 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 783 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++i;
															#line 4800 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& temp,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(temp, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && temp,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(temp), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CalcCheckSumActor*>(this)->actor_wait_state > 0) static_cast<CalcCheckSumActor*>(this)->actor_wait_state = 0;
		static_cast<CalcCheckSumActor*>(this)->ActorCallback< CalcCheckSumActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< CalcCheckSumActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("calcCheckSum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CalcCheckSumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("calcCheckSum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CalcCheckSumActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("calcCheckSum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CalcCheckSumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("calcCheckSum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CalcCheckSumActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("calcCheckSum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CalcCheckSumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("calcCheckSum", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	ReadYourWritesTransaction* tr;
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int csIndex;
															#line 780 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	uint32_t result;
															#line 781 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int i;
															#line 782 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Key csKey;
															#line 4892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via calcCheckSum()
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class CalcCheckSumActor final : public Actor<uint32_t>, public ActorCallback< CalcCheckSumActor, 0, Optional<Value> >, public FastAllocated<CalcCheckSumActor>, public CalcCheckSumActorState<CalcCheckSumActor> {
															#line 4897 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<CalcCheckSumActor>::operator new;
	using FastAllocated<CalcCheckSumActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18235122495843304960UL, 478162504238883840UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<uint32_t>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CalcCheckSumActor, 0, Optional<Value> >;
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	CalcCheckSumActor(ReadYourWritesTransaction* const& tr,MakoWorkload* const& self,int const& csIndex) 
															#line 4914 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<uint32_t>(),
		   CalcCheckSumActorState<CalcCheckSumActor>(tr, self, csIndex),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("calcCheckSum", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16272585567577648128UL, 7880584558769032192UL);
		ActorExecutionContextHelper __helper(static_cast<CalcCheckSumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("calcCheckSum");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("calcCheckSum", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CalcCheckSumActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] static Future<uint32_t> calcCheckSum( ReadYourWritesTransaction* const& tr, MakoWorkload* const& self, int const& csIndex ) {
															#line 779 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<uint32_t>(new CalcCheckSumActor(tr, self, csIndex));
															#line 4946 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 797 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

																#line 4951 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via dochecksumVerification()
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class DochecksumVerificationActor>
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class DochecksumVerificationActorState {
															#line 4957 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	DochecksumVerificationActorState(Database const& cx,MakoWorkload* const& self) 
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : cx(cx),
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self),
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   tr(cx),
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   csIdx(),
															#line 801 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   csValue()
															#line 4972 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("dochecksumVerification", reinterpret_cast<unsigned long>(this));

	}
	~DochecksumVerificationActorState() 
	{
		fdb_probe_actor_destroy("dochecksumVerification", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 803 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			;
															#line 4987 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DochecksumVerificationActorState();
		static_cast<DochecksumVerificationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 805 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 806 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			csIdx = 0;
															#line 5020 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 833 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent("FailedToCalculateChecksum").error(e).detail("ChecksumIndex", csIdx);
															#line 834 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 834 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<DochecksumVerificationActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<DochecksumVerificationActor*>(this)->actor_wait_state = 3;
															#line 834 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DochecksumVerificationActor, 2, Void >*>(static_cast<DochecksumVerificationActor*>(this)));
															#line 5051 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 831 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<DochecksumVerificationActor*>(this)->SAV<bool>::futures) { (void)(true); this->~DochecksumVerificationActorState(); static_cast<DochecksumVerificationActor*>(this)->destroy(); return 0; }
															#line 5066 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<DochecksumVerificationActor*>(this)->SAV< bool >::value()) bool(true);
		this->~DochecksumVerificationActorState();
		static_cast<DochecksumVerificationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 806 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!(csIdx < self->csCount))
															#line 5085 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 807 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_0 = tr.get(self->csKeys[csIdx]);
															#line 807 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<DochecksumVerificationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5093 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<DochecksumVerificationActor*>(this)->actor_wait_state = 1;
															#line 807 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DochecksumVerificationActor, 0, Optional<Value> >*>(static_cast<DochecksumVerificationActor*>(this)));
															#line 5098 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Optional<Value> const& temp,int loopDepth) 
	{
															#line 808 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!temp.present())
															#line 5120 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 809 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent(SevError, "TestFailure") .detail("Reason", "NoExistingChecksum") .detail("missedChecksumIndex", csIdx);
															#line 812 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (!static_cast<DochecksumVerificationActor*>(this)->SAV<bool>::futures) { (void)(false); this->~DochecksumVerificationActorState(); static_cast<DochecksumVerificationActor*>(this)->destroy(); return 0; }
															#line 5126 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			new (&static_cast<DochecksumVerificationActor*>(this)->SAV< bool >::value()) bool(false);
			this->~DochecksumVerificationActorState();
			static_cast<DochecksumVerificationActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 814 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			csValue = temp.get();
															#line 815 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			ASSERT(csValue.size() == sizeof(uint32_t));
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<uint32_t> __when_expr_1 = calcCheckSum(&tr, self, csIdx);
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<DochecksumVerificationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<DochecksumVerificationActor*>(this)->actor_wait_state = 2;
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DochecksumVerificationActor, 1, uint32_t >*>(static_cast<DochecksumVerificationActor*>(this)));
															#line 5147 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Optional<Value> && temp,int loopDepth) 
	{
															#line 808 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!temp.present())
															#line 5157 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 809 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent(SevError, "TestFailure") .detail("Reason", "NoExistingChecksum") .detail("missedChecksumIndex", csIdx);
															#line 812 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (!static_cast<DochecksumVerificationActor*>(this)->SAV<bool>::futures) { (void)(false); this->~DochecksumVerificationActorState(); static_cast<DochecksumVerificationActor*>(this)->destroy(); return 0; }
															#line 5163 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			new (&static_cast<DochecksumVerificationActor*>(this)->SAV< bool >::value()) bool(false);
			this->~DochecksumVerificationActorState();
			static_cast<DochecksumVerificationActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 814 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			csValue = temp.get();
															#line 815 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			ASSERT(csValue.size() == sizeof(uint32_t));
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<uint32_t> __when_expr_1 = calcCheckSum(&tr, self, csIdx);
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<DochecksumVerificationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5179 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<DochecksumVerificationActor*>(this)->actor_wait_state = 2;
															#line 816 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DochecksumVerificationActor, 1, uint32_t >*>(static_cast<DochecksumVerificationActor*>(this)));
															#line 5184 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Optional<Value> const& temp,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(temp, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Optional<Value> && temp,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(std::move(temp), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DochecksumVerificationActor*>(this)->actor_wait_state > 0) static_cast<DochecksumVerificationActor*>(this)->actor_wait_state = 0;
		static_cast<DochecksumVerificationActor*>(this)->ActorCallback< DochecksumVerificationActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< DochecksumVerificationActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DochecksumVerificationActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DochecksumVerificationActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
															#line 806 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++csIdx;
															#line 5269 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont5(uint32_t const& calculatedCS,int loopDepth) 
	{
															#line 817 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		uint32_t existingCS = *(reinterpret_cast<const uint32_t*>(csValue.begin()));
															#line 818 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (existingCS != calculatedCS)
															#line 5280 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 819 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent(SevError, "TestFailure") .detail("Reason", "ChecksumVerificationFailure") .detail("ChecksumIndex", csIdx) .detail("ExistingChecksum", existingCS) .detail("CurrentChecksum", calculatedCS);
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (!static_cast<DochecksumVerificationActor*>(this)->SAV<bool>::futures) { (void)(false); this->~DochecksumVerificationActorState(); static_cast<DochecksumVerificationActor*>(this)->destroy(); return 0; }
															#line 5286 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			new (&static_cast<DochecksumVerificationActor*>(this)->SAV< bool >::value()) bool(false);
			this->~DochecksumVerificationActorState();
			static_cast<DochecksumVerificationActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 826 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent("ChecksumVerificationPass") .detail("ChecksumIndex", csIdx) .detail("ChecksumValue", existingCS);
															#line 5294 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont5(uint32_t && calculatedCS,int loopDepth) 
	{
															#line 817 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		uint32_t existingCS = *(reinterpret_cast<const uint32_t*>(csValue.begin()));
															#line 818 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (existingCS != calculatedCS)
															#line 5305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 819 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent(SevError, "TestFailure") .detail("Reason", "ChecksumVerificationFailure") .detail("ChecksumIndex", csIdx) .detail("ExistingChecksum", existingCS) .detail("CurrentChecksum", calculatedCS);
															#line 824 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (!static_cast<DochecksumVerificationActor*>(this)->SAV<bool>::futures) { (void)(false); this->~DochecksumVerificationActorState(); static_cast<DochecksumVerificationActor*>(this)->destroy(); return 0; }
															#line 5311 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			new (&static_cast<DochecksumVerificationActor*>(this)->SAV< bool >::value()) bool(false);
			this->~DochecksumVerificationActorState();
			static_cast<DochecksumVerificationActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 826 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent("ChecksumVerificationPass") .detail("ChecksumIndex", csIdx) .detail("ChecksumValue", existingCS);
															#line 5319 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(uint32_t const& calculatedCS,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont5(calculatedCS, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(uint32_t && calculatedCS,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont5(std::move(calculatedCS), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DochecksumVerificationActor*>(this)->actor_wait_state > 0) static_cast<DochecksumVerificationActor*>(this)->actor_wait_state = 0;
		static_cast<DochecksumVerificationActor*>(this)->ActorCallback< DochecksumVerificationActor, 1, uint32_t >::remove();

	}
	void a_callback_fire(ActorCallback< DochecksumVerificationActor, 1, uint32_t >*,uint32_t const& value) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DochecksumVerificationActor, 1, uint32_t >*,uint32_t && value) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DochecksumVerificationActor, 1, uint32_t >*,Error err) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DochecksumVerificationActor*>(this)->actor_wait_state > 0) static_cast<DochecksumVerificationActor*>(this)->actor_wait_state = 0;
		static_cast<DochecksumVerificationActor*>(this)->ActorCallback< DochecksumVerificationActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DochecksumVerificationActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DochecksumVerificationActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DochecksumVerificationActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Database cx;
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 799 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	ReadYourWritesTransaction tr;
															#line 800 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int csIdx;
															#line 801 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Value csValue;
															#line 5496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via dochecksumVerification()
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class DochecksumVerificationActor final : public Actor<bool>, public ActorCallback< DochecksumVerificationActor, 0, Optional<Value> >, public ActorCallback< DochecksumVerificationActor, 1, uint32_t >, public ActorCallback< DochecksumVerificationActor, 2, Void >, public FastAllocated<DochecksumVerificationActor>, public DochecksumVerificationActorState<DochecksumVerificationActor> {
															#line 5501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<DochecksumVerificationActor>::operator new;
	using FastAllocated<DochecksumVerificationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16123102473159578368UL, 7307648569781626368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DochecksumVerificationActor, 0, Optional<Value> >;
friend struct ActorCallback< DochecksumVerificationActor, 1, uint32_t >;
friend struct ActorCallback< DochecksumVerificationActor, 2, Void >;
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	DochecksumVerificationActor(Database const& cx,MakoWorkload* const& self) 
															#line 5520 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<bool>(),
		   DochecksumVerificationActorState<DochecksumVerificationActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("dochecksumVerification", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2876640143980600064UL, 6828659106957041920UL);
		ActorExecutionContextHelper __helper(static_cast<DochecksumVerificationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("dochecksumVerification");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("dochecksumVerification", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DochecksumVerificationActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DochecksumVerificationActor, 1, uint32_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DochecksumVerificationActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] static Future<bool> dochecksumVerification( Database const& cx, MakoWorkload* const& self ) {
															#line 798 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<bool>(new DochecksumVerificationActor(cx, self));
															#line 5554 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 838 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

																#line 5559 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via generateChecksum()
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class GenerateChecksumActor>
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class GenerateChecksumActorState {
															#line 5565 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	GenerateChecksumActorState(Database const& cx,MakoWorkload* const& self) 
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : cx(cx),
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self),
															#line 840 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   tr(cx),
															#line 841 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   csIdx()
															#line 5578 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("generateChecksum", reinterpret_cast<unsigned long>(this));

	}
	~GenerateChecksumActorState() 
	{
		fdb_probe_actor_destroy("generateChecksum", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 842 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			;
															#line 5593 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GenerateChecksumActorState();
		static_cast<GenerateChecksumActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 857 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<GenerateChecksumActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GenerateChecksumActorState(); static_cast<GenerateChecksumActor*>(this)->destroy(); return 0; }
															#line 5616 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<GenerateChecksumActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GenerateChecksumActorState();
		static_cast<GenerateChecksumActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 844 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			csIdx = 0;
															#line 5636 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 853 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent("FailedToGenerateChecksumForPopulatedData").error(e);
															#line 854 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 854 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5675 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 4;
															#line 854 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GenerateChecksumActor, 3, Void >*>(static_cast<GenerateChecksumActor*>(this)));
															#line 5680 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 850 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 850 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 3;
															#line 850 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GenerateChecksumActor, 2, Void >*>(static_cast<GenerateChecksumActor*>(this)));
															#line 5702 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 844 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!(csIdx < self->csCount))
															#line 5718 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 845 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_0 = tr.get(self->csKeys[csIdx]);
															#line 845 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5726 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 1;
															#line 845 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GenerateChecksumActor, 0, Optional<Value> >*>(static_cast<GenerateChecksumActor*>(this)));
															#line 5731 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Optional<Value> const& temp,int loopDepth) 
	{
															#line 846 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (temp.present())
															#line 5753 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 847 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent("DuplicatePopulationOnSamePrefix").detail("KeyPrefix", self->keyPrefix);
															#line 5757 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 848 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->updateCheckSum(&tr, self, csIdx);
															#line 848 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5763 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 2;
															#line 848 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GenerateChecksumActor, 1, Void >*>(static_cast<GenerateChecksumActor*>(this)));
															#line 5768 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Optional<Value> && temp,int loopDepth) 
	{
															#line 846 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (temp.present())
															#line 5777 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 847 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			TraceEvent("DuplicatePopulationOnSamePrefix").detail("KeyPrefix", self->keyPrefix);
															#line 5781 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		}
															#line 848 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->updateCheckSum(&tr, self, csIdx);
															#line 848 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5787 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 2;
															#line 848 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GenerateChecksumActor, 1, Void >*>(static_cast<GenerateChecksumActor*>(this)));
															#line 5792 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Optional<Value> const& temp,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(temp, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Optional<Value> && temp,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(std::move(temp), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state > 0) static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 0;
		static_cast<GenerateChecksumActor*>(this)->ActorCallback< GenerateChecksumActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GenerateChecksumActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GenerateChecksumActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GenerateChecksumActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 844 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++csIdx;
															#line 5876 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 844 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++csIdx;
															#line 5885 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state > 0) static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 0;
		static_cast<GenerateChecksumActor*>(this)->ActorCallback< GenerateChecksumActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GenerateChecksumActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GenerateChecksumActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GenerateChecksumActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state > 0) static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 0;
		static_cast<GenerateChecksumActor*>(this)->ActorCallback< GenerateChecksumActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GenerateChecksumActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GenerateChecksumActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GenerateChecksumActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GenerateChecksumActor*>(this)->actor_wait_state > 0) static_cast<GenerateChecksumActor*>(this)->actor_wait_state = 0;
		static_cast<GenerateChecksumActor*>(this)->ActorCallback< GenerateChecksumActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GenerateChecksumActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GenerateChecksumActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GenerateChecksumActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	Database cx;
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 840 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	ReadYourWritesTransaction tr;
															#line 841 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int csIdx;
															#line 6147 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via generateChecksum()
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class GenerateChecksumActor final : public Actor<Void>, public ActorCallback< GenerateChecksumActor, 0, Optional<Value> >, public ActorCallback< GenerateChecksumActor, 1, Void >, public ActorCallback< GenerateChecksumActor, 2, Void >, public ActorCallback< GenerateChecksumActor, 3, Void >, public FastAllocated<GenerateChecksumActor>, public GenerateChecksumActorState<GenerateChecksumActor> {
															#line 6152 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<GenerateChecksumActor>::operator new;
	using FastAllocated<GenerateChecksumActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(222452211364673792UL, 14335253691097342208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GenerateChecksumActor, 0, Optional<Value> >;
friend struct ActorCallback< GenerateChecksumActor, 1, Void >;
friend struct ActorCallback< GenerateChecksumActor, 2, Void >;
friend struct ActorCallback< GenerateChecksumActor, 3, Void >;
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	GenerateChecksumActor(Database const& cx,MakoWorkload* const& self) 
															#line 6172 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   GenerateChecksumActorState<GenerateChecksumActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("generateChecksum", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14491780255820148992UL, 2957429005388427264UL);
		ActorExecutionContextHelper __helper(static_cast<GenerateChecksumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("generateChecksum");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("generateChecksum", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GenerateChecksumActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GenerateChecksumActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GenerateChecksumActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GenerateChecksumActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] static Future<Void> generateChecksum( Database const& cx, MakoWorkload* const& self ) {
															#line 839 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new GenerateChecksumActor(cx, self));
															#line 6207 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 859 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

																#line 6212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via updateCheckSum()
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class UpdateCheckSumActor>
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class UpdateCheckSumActorState {
															#line 6218 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	UpdateCheckSumActorState(ReadYourWritesTransaction* const& tr,MakoWorkload* const& self,int const& csIdx) 
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : tr(tr),
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self),
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   csIdx(csIdx)
															#line 6229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("updateCheckSum", reinterpret_cast<unsigned long>(this));

	}
	~UpdateCheckSumActorState() 
	{
		fdb_probe_actor_destroy("updateCheckSum", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 861 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<uint32_t> __when_expr_0 = calcCheckSum(tr, self, csIdx);
															#line 861 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<UpdateCheckSumActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6246 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateCheckSumActor*>(this)->actor_wait_state = 1;
															#line 861 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateCheckSumActor, 0, uint32_t >*>(static_cast<UpdateCheckSumActor*>(this)));
															#line 6251 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateCheckSumActorState();
		static_cast<UpdateCheckSumActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 862 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		TraceEvent("UpdateCheckSum").detail("ChecksumIndex", csIdx).detail("Checksum", csVal);
															#line 863 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		tr->set(self->csKeys[csIdx], ValueRef(reinterpret_cast<const uint8_t*>(&csVal), sizeof(uint32_t)));
															#line 864 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<UpdateCheckSumActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateCheckSumActorState(); static_cast<UpdateCheckSumActor*>(this)->destroy(); return 0; }
															#line 6278 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<UpdateCheckSumActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateCheckSumActorState();
		static_cast<UpdateCheckSumActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(uint32_t const& __csVal,int loopDepth) 
	{
															#line 861 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		csVal = __csVal;
															#line 6290 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(uint32_t && __csVal,int loopDepth) 
	{
		csVal = std::move(__csVal);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateCheckSumActor*>(this)->actor_wait_state > 0) static_cast<UpdateCheckSumActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateCheckSumActor*>(this)->ActorCallback< UpdateCheckSumActor, 0, uint32_t >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateCheckSumActor, 0, uint32_t >*,uint32_t const& value) 
	{
		fdb_probe_actor_enter("updateCheckSum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCheckSumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCheckSum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateCheckSumActor, 0, uint32_t >*,uint32_t && value) 
	{
		fdb_probe_actor_enter("updateCheckSum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCheckSumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCheckSum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateCheckSumActor, 0, uint32_t >*,Error err) 
	{
		fdb_probe_actor_enter("updateCheckSum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCheckSumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCheckSum", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	ReadYourWritesTransaction* tr;
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int csIdx;
															#line 861 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	uint32_t csVal;
															#line 6373 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via updateCheckSum()
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class UpdateCheckSumActor final : public Actor<Void>, public ActorCallback< UpdateCheckSumActor, 0, uint32_t >, public FastAllocated<UpdateCheckSumActor>, public UpdateCheckSumActorState<UpdateCheckSumActor> {
															#line 6378 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<UpdateCheckSumActor>::operator new;
	using FastAllocated<UpdateCheckSumActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12767907045955126016UL, 9070239055112624640UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateCheckSumActor, 0, uint32_t >;
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	UpdateCheckSumActor(ReadYourWritesTransaction* const& tr,MakoWorkload* const& self,int const& csIdx) 
															#line 6395 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateCheckSumActorState<UpdateCheckSumActor>(tr, self, csIdx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateCheckSum", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6768543439570458368UL, 3309653214232725760UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCheckSumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateCheckSum");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateCheckSum", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateCheckSumActor, 0, uint32_t >*)0, actor_cancelled()); break;
		}

	}
};
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] static Future<Void> updateCheckSum( ReadYourWritesTransaction* const& tr, MakoWorkload* const& self, int const& csIdx ) {
															#line 860 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new UpdateCheckSumActor(tr, self, csIdx));
															#line 6427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 866 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"

																#line 6432 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
// This generated class is to be used only via updateCSBeforeCommit()
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
template <class UpdateCSBeforeCommitActor>
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class UpdateCSBeforeCommitActorState {
															#line 6438 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	UpdateCSBeforeCommitActorState(ReadYourWritesTransaction* const& tr,MakoWorkload* const& self,std::vector<bool>* const& flags) 
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		 : tr(tr),
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   self(self),
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		   flags(flags)
															#line 6449 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
	{
		fdb_probe_actor_create("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this));

	}
	~UpdateCSBeforeCommitActorState() 
	{
		fdb_probe_actor_destroy("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 870 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (!self->checksumVerification)
															#line 6464 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			{
															#line 871 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
				if (!static_cast<UpdateCSBeforeCommitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateCSBeforeCommitActorState(); static_cast<UpdateCSBeforeCommitActor*>(this)->destroy(); return 0; }
															#line 6468 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
				new (&static_cast<UpdateCSBeforeCommitActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~UpdateCSBeforeCommitActorState();
				static_cast<UpdateCSBeforeCommitActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 873 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			csIdx = int();
															#line 874 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			csIdx = 0;
															#line 6478 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateCSBeforeCommitActorState();
		static_cast<UpdateCSBeforeCommitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 880 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!static_cast<UpdateCSBeforeCommitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateCSBeforeCommitActorState(); static_cast<UpdateCSBeforeCommitActor*>(this)->destroy(); return 0; }
															#line 6501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		new (&static_cast<UpdateCSBeforeCommitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateCSBeforeCommitActorState();
		static_cast<UpdateCSBeforeCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 874 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if (!(csIdx < self->csCount))
															#line 6520 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 875 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		if ((*flags)[csIdx])
															#line 6526 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		{
															#line 876 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			StrictFuture<Void> __when_expr_0 = updateCheckSum(tr, self, csIdx);
															#line 876 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			if (static_cast<UpdateCSBeforeCommitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateCSBeforeCommitActor*>(this)->actor_wait_state = 1;
															#line 876 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateCSBeforeCommitActor, 0, Void >*>(static_cast<UpdateCSBeforeCommitActor*>(this)));
															#line 6537 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 874 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		++csIdx;
															#line 6564 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 877 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		(*flags)[csIdx] = false;
															#line 6573 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 877 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
		(*flags)[csIdx] = false;
															#line 6582 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateCSBeforeCommitActor*>(this)->actor_wait_state > 0) static_cast<UpdateCSBeforeCommitActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateCSBeforeCommitActor*>(this)->ActorCallback< UpdateCSBeforeCommitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateCSBeforeCommitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCSBeforeCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateCSBeforeCommitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCSBeforeCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateCSBeforeCommitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCSBeforeCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	ReadYourWritesTransaction* tr;
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	MakoWorkload* self;
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	std::vector<bool>* flags;
															#line 873 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	int csIdx;
															#line 6670 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
};
// This generated class is to be used only via updateCSBeforeCommit()
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
class UpdateCSBeforeCommitActor final : public Actor<Void>, public ActorCallback< UpdateCSBeforeCommitActor, 0, Void >, public FastAllocated<UpdateCSBeforeCommitActor>, public UpdateCSBeforeCommitActorState<UpdateCSBeforeCommitActor> {
															#line 6675 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
public:
	using FastAllocated<UpdateCSBeforeCommitActor>::operator new;
	using FastAllocated<UpdateCSBeforeCommitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9894666902790475776UL, 8366720442109312000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateCSBeforeCommitActor, 0, Void >;
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	UpdateCSBeforeCommitActor(ReadYourWritesTransaction* const& tr,MakoWorkload* const& self,std::vector<bool>* const& flags) 
															#line 6692 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateCSBeforeCommitActorState<UpdateCSBeforeCommitActor>(tr, self, flags),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3748874192729633792UL, 567939706041592320UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCSBeforeCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateCSBeforeCommit");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateCSBeforeCommit", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateCSBeforeCommitActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
[[nodiscard]] static Future<Void> updateCSBeforeCommit( ReadYourWritesTransaction* const& tr, MakoWorkload* const& self, std::vector<bool>* const& flags ) {
															#line 867 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
	return Future<Void>(new UpdateCSBeforeCommitActor(tr, self, flags));
															#line 6724 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbserver/workloads/Mako.actor.g.cpp"
}

#line 882 "/Users/ec2-user/foundationdb/fdbserver/workloads/Mako.actor.cpp"
};

WorkloadFactory<MakoWorkload> MakoloadFactory;
