#define POST_ACTOR_COMPILER 1
#line 1 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
/*
 * S3BlobStore.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/S3BlobStore.h"

#include "fdbclient/ClientKnobs.h"
#include "fdbclient/Knobs.h"
#include "flow/FastRef.h"
#include "flow/IConnection.h"
#include "flow/Trace.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "md5/md5.h"
#include "libb64/encode.h"
#include "fdbclient/sha1/SHA1.h"
#include <climits>
#include <iostream>
#include <time.h>
#include <iomanip>
#include <openssl/sha.h>
#include <openssl/evp.h>
#include <openssl/hmac.h>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/hex.hpp>
#include "flow/IAsyncFile.h"
#include "flow/Hostname.h"
#include "flow/UnitTest.h"
#include "rapidxml/rapidxml.hpp"
#ifdef WITH_AWS_BACKUP
#include "fdbclient/FDBAWSCredentialsProvider.h"
#endif

#include "flow/actorcompiler.h" // has to be last include

using namespace rapidxml;

json_spirit::mObject S3BlobStoreEndpoint::Stats::getJSON() {
	json_spirit::mObject o;

	o["requests_failed"] = requests_failed;
	o["requests_successful"] = requests_successful;
	o["bytes_sent"] = bytes_sent;

	return o;
}

S3BlobStoreEndpoint::Stats S3BlobStoreEndpoint::Stats::operator-(const Stats& rhs) {
	Stats r;
	r.requests_failed = requests_failed - rhs.requests_failed;
	r.requests_successful = requests_successful - rhs.requests_successful;
	r.bytes_sent = bytes_sent - rhs.bytes_sent;
	return r;
}

S3BlobStoreEndpoint::Stats S3BlobStoreEndpoint::s_stats;
std::unique_ptr<S3BlobStoreEndpoint::BlobStats> S3BlobStoreEndpoint::blobStats;
Future<Void> S3BlobStoreEndpoint::statsLogger = Never();

std::unordered_map<BlobStoreConnectionPoolKey, Reference<S3BlobStoreEndpoint::ConnectionPoolData>>
    S3BlobStoreEndpoint::globalConnectionPool;

S3BlobStoreEndpoint::BlobKnobs::BlobKnobs() {
	secure_connection = 1;
	connect_tries = CLIENT_KNOBS->BLOBSTORE_CONNECT_TRIES;
	connect_timeout = CLIENT_KNOBS->BLOBSTORE_CONNECT_TIMEOUT;
	max_connection_life = CLIENT_KNOBS->BLOBSTORE_MAX_CONNECTION_LIFE;
	request_tries = CLIENT_KNOBS->BLOBSTORE_REQUEST_TRIES;
	request_timeout_min = CLIENT_KNOBS->BLOBSTORE_REQUEST_TIMEOUT_MIN;
	requests_per_second = CLIENT_KNOBS->BLOBSTORE_REQUESTS_PER_SECOND;
	concurrent_requests = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_REQUESTS;
	list_requests_per_second = CLIENT_KNOBS->BLOBSTORE_LIST_REQUESTS_PER_SECOND;
	write_requests_per_second = CLIENT_KNOBS->BLOBSTORE_WRITE_REQUESTS_PER_SECOND;
	read_requests_per_second = CLIENT_KNOBS->BLOBSTORE_READ_REQUESTS_PER_SECOND;
	delete_requests_per_second = CLIENT_KNOBS->BLOBSTORE_DELETE_REQUESTS_PER_SECOND;
	multipart_max_part_size = CLIENT_KNOBS->BLOBSTORE_MULTIPART_MAX_PART_SIZE;
	multipart_min_part_size = CLIENT_KNOBS->BLOBSTORE_MULTIPART_MIN_PART_SIZE;
	concurrent_uploads = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_UPLOADS;
	concurrent_lists = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_LISTS;
	concurrent_reads_per_file = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_READS_PER_FILE;
	concurrent_writes_per_file = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_WRITES_PER_FILE;
	enable_read_cache = CLIENT_KNOBS->BLOBSTORE_ENABLE_READ_CACHE;
	read_block_size = CLIENT_KNOBS->BLOBSTORE_READ_BLOCK_SIZE;
	read_ahead_blocks = CLIENT_KNOBS->BLOBSTORE_READ_AHEAD_BLOCKS;
	read_cache_blocks_per_file = CLIENT_KNOBS->BLOBSTORE_READ_CACHE_BLOCKS_PER_FILE;
	max_send_bytes_per_second = CLIENT_KNOBS->BLOBSTORE_MAX_SEND_BYTES_PER_SECOND;
	max_recv_bytes_per_second = CLIENT_KNOBS->BLOBSTORE_MAX_RECV_BYTES_PER_SECOND;
	max_delay_retryable_error = CLIENT_KNOBS->BLOBSTORE_MAX_DELAY_RETRYABLE_ERROR;
	max_delay_connection_failed = CLIENT_KNOBS->BLOBSTORE_MAX_DELAY_CONNECTION_FAILED;
	sdk_auth = false;
	enable_object_integrity_check = CLIENT_KNOBS->BLOBSTORE_ENABLE_OBJECT_INTEGRITY_CHECK;
	global_connection_pool = CLIENT_KNOBS->BLOBSTORE_GLOBAL_CONNECTION_POOL;
}

bool S3BlobStoreEndpoint::BlobKnobs::set(StringRef name, int value) {
#define TRY_PARAM(n, sn)                                                                                               \
	if (name == #n || name == #sn) {                                                                                   \
		n = value;                                                                                                     \
		return true;                                                                                                   \
	}
	TRY_PARAM(secure_connection, sc)
	TRY_PARAM(connect_tries, ct);
	TRY_PARAM(connect_timeout, cto);
	TRY_PARAM(max_connection_life, mcl);
	TRY_PARAM(request_tries, rt);
	TRY_PARAM(request_timeout_min, rtom);
	// TODO: For backward compatibility because request_timeout was renamed to request_timeout_min
	if (name == "request_timeout"_sr || name == "rto"_sr) {
		request_timeout_min = value;
		return true;
	}
	TRY_PARAM(requests_per_second, rps);
	TRY_PARAM(list_requests_per_second, lrps);
	TRY_PARAM(write_requests_per_second, wrps);
	TRY_PARAM(read_requests_per_second, rrps);
	TRY_PARAM(delete_requests_per_second, drps);
	TRY_PARAM(concurrent_requests, cr);
	TRY_PARAM(multipart_max_part_size, maxps);
	TRY_PARAM(multipart_min_part_size, minps);
	TRY_PARAM(multipart_retry_delay_ms, mrd);
	TRY_PARAM(concurrent_uploads, cu);
	TRY_PARAM(concurrent_lists, cl);
	TRY_PARAM(concurrent_reads_per_file, crpf);
	TRY_PARAM(concurrent_writes_per_file, cwpf);
	TRY_PARAM(enable_read_cache, erc);
	TRY_PARAM(read_block_size, rbs);
	TRY_PARAM(read_ahead_blocks, rab);
	TRY_PARAM(read_cache_blocks_per_file, rcb);
	TRY_PARAM(max_send_bytes_per_second, sbps);
	TRY_PARAM(max_recv_bytes_per_second, rbps);
	TRY_PARAM(max_delay_retryable_error, dre);
	TRY_PARAM(max_delay_connection_failed, dcf);
	TRY_PARAM(sdk_auth, sa);
	TRY_PARAM(enable_object_integrity_check, eoic);
	TRY_PARAM(global_connection_pool, gcp);
#undef TRY_PARAM
	return false;
}

// Returns an S3 Blob URL parameter string that specifies all of the non-default options for the endpoint using option
// short names.
std::string S3BlobStoreEndpoint::BlobKnobs::getURLParameters() const {
	static BlobKnobs defaults;
	std::string r;
#define _CHECK_PARAM(n, sn)                                                                                            \
	if (n != defaults.n) {                                                                                             \
		r += format("%s%s=%d", r.empty() ? "" : "&", #sn, n);                                                          \
	}
	_CHECK_PARAM(secure_connection, sc);
	_CHECK_PARAM(connect_tries, ct);
	_CHECK_PARAM(connect_timeout, cto);
	_CHECK_PARAM(max_connection_life, mcl);
	_CHECK_PARAM(request_tries, rt);
	_CHECK_PARAM(request_timeout_min, rto);
	_CHECK_PARAM(requests_per_second, rps);
	_CHECK_PARAM(list_requests_per_second, lrps);
	_CHECK_PARAM(write_requests_per_second, wrps);
	_CHECK_PARAM(read_requests_per_second, rrps);
	_CHECK_PARAM(delete_requests_per_second, drps);
	_CHECK_PARAM(concurrent_requests, cr);
	_CHECK_PARAM(multipart_max_part_size, maxps);
	_CHECK_PARAM(multipart_min_part_size, minps);
	_CHECK_PARAM(concurrent_uploads, cu);
	_CHECK_PARAM(concurrent_lists, cl);
	_CHECK_PARAM(concurrent_reads_per_file, crpf);
	_CHECK_PARAM(concurrent_writes_per_file, cwpf);
	_CHECK_PARAM(enable_read_cache, erc);
	_CHECK_PARAM(read_block_size, rbs);
	_CHECK_PARAM(read_ahead_blocks, rab);
	_CHECK_PARAM(read_cache_blocks_per_file, rcb);
	_CHECK_PARAM(max_send_bytes_per_second, sbps);
	_CHECK_PARAM(max_recv_bytes_per_second, rbps);
	_CHECK_PARAM(sdk_auth, sa);
	_CHECK_PARAM(enable_object_integrity_check, eoic);
	_CHECK_PARAM(global_connection_pool, gcp);
	_CHECK_PARAM(max_delay_retryable_error, dre);
	_CHECK_PARAM(max_delay_connection_failed, dcf);
#undef _CHECK_PARAM
	return r;
}

std::string guessRegionFromDomain(std::string domain) {
	static const std::vector<const char*> knownServices = { "s3.", "cos.", "oss-", "obs." };
	boost::algorithm::to_lower(domain);

	for (int i = 0; i < knownServices.size(); ++i) {
		const char* service = knownServices[i];

		std::size_t p = domain.find(service);
		if (p == std::string::npos || (p >= 1 && domain[p - 1] != '.')) {
			// eg. 127.0.0.1, example.com, s3-service.example.com, mys3.example.com
			continue;
		}

		StringRef h = StringRef(domain).substr(p);

		if (!h.startsWith("oss-"_sr)) {
			h.eat(service); // ignore s3 service
		}

		return h.eat(".").toString();
	}

	return "";
}

Reference<S3BlobStoreEndpoint> S3BlobStoreEndpoint::fromString(const std::string& url,
                                                               const Optional<std::string>& proxy,
                                                               std::string* resourceFromURL,
                                                               std::string* error,
                                                               ParametersT* ignored_parameters) {
	if (resourceFromURL)
		resourceFromURL->clear();

	try {
		// Replace HTML-encoded ampersands with raw ampersands
		std::string decoded_url = url;
		size_t pos = 0;
		while ((pos = decoded_url.find("&amp;", pos)) != std::string::npos) {
			decoded_url.replace(pos, 5, "&");
			pos += 1;
		}

		// Also handle double-encoded ampersands
		pos = 0;
		while ((pos = decoded_url.find("&amp;amp;", pos)) != std::string::npos) {
			decoded_url.replace(pos, 9, "&");
			pos += 1;
		}

		StringRef t(decoded_url);
		StringRef prefix = t.eat("://");
		if (prefix != "blobstore"_sr)
			throw format("Invalid blobstore URL prefix '%s'", prefix.toString().c_str());

		Optional<std::string> proxyHost, proxyPort;
		if (proxy.present()) {
			StringRef proxyRef(proxy.get());
			if (proxy.get().find("://") != std::string::npos) {
				StringRef proxyPrefix = proxyRef.eat("://");
				if (proxyPrefix != "http"_sr) {
					throw format("Invalid proxy URL prefix '%s'. Either don't use a prefix, or use http://",
					             proxyPrefix.toString().c_str());
				}
			}
			std::string proxyBody = proxyRef.eat().toString();
			if (!Hostname::isHostname(proxyBody) && !NetworkAddress::parseOptional(proxyBody).present()) {
				throw format("'%s' is not a valid value for proxy. Format should be either IP:port or host:port.",
				             proxyBody.c_str());
			}
			StringRef p(proxyBody);
			proxyHost = p.eat(":").toString();
			proxyPort = p.eat().toString();
		}

		Optional<StringRef> cred;
		if (url.find("@") != std::string::npos) {
			cred = t.eat("@");
		}
		uint8_t foundSeparator = 0;
		StringRef hostPort = t.eatAny("/?", &foundSeparator);
		StringRef resource;
		if (foundSeparator == '/') {
			resource = t.eat("?");
		}

		// hostPort is at least a host or IP address, optionally followed by :portNumber or :serviceName
		StringRef h(hostPort);
		StringRef host = h.eat(":");
		if (host.size() == 0)
			throw std::string("host cannot be empty");

		StringRef service = h.eat();

		std::string region = guessRegionFromDomain(host.toString());

		BlobKnobs knobs;
		HTTP::Headers extraHeaders;
		while (1) {
			StringRef name = t.eat("=");
			if (name.size() == 0)
				break;
			StringRef value = t.eat("&");

			// Special case for header
			if (name == "header"_sr) {
				StringRef originalValue = value;
				StringRef headerFieldName = value.eat(":");
				StringRef headerFieldValue = value;
				if (headerFieldName.size() == 0 || headerFieldValue.size() == 0) {
					throw format("'%s' is not a valid value for '%s' parameter.  Format is <FieldName>:<FieldValue> "
					             "where strings are not empty.",
					             originalValue.toString().c_str(),
					             name.toString().c_str());
				}
				std::string& fieldValue = extraHeaders[headerFieldName.toString()];
				// RFC 2616 section 4.2 says header field names can repeat but only if it is valid to concatenate their
				// values with comma separation
				if (!fieldValue.empty()) {
					fieldValue.append(",");
				}
				fieldValue.append(headerFieldValue.toString());
				continue;
			}

			// overwrite s3 region from parameter
			if (name == "region"_sr) {
				region = value.toString();
				continue;
			}

			// See if the parameter is a knob
			// First try setting a dummy value (all knobs are currently numeric) just to see if this parameter is known
			// to S3BlobStoreEndpoint. If it is, then we will set it to a good value or throw below, so the dummy set
			// has no bad side effects.
			bool known = knobs.set(name, 0);

			// If the parameter is not known to S3BlobStoreEndpoint then throw unless there is an ignored_parameters set
			// to add it to
			if (!known) {
				if (ignored_parameters == nullptr) {
					throw format("%s is not a valid parameter name", name.toString().c_str());
				}
				(*ignored_parameters)[name.toString()] = value.toString();
				continue;
			}

			// The parameter is known to S3BlobStoreEndpoint so it must be numeric and valid.
			char* valueEnd = nullptr;
			std::string s = value.toString();
			long int ivalue = strtol(s.c_str(), &valueEnd, 10);
			if (*valueEnd || (ivalue == 0 && s != "0") ||
			    (((ivalue == LONG_MAX) || (ivalue == LONG_MIN)) && errno == ERANGE))
				throw format("%s is not a valid value for %s", s.c_str(), name.toString().c_str());

			// It should not be possible for this set to fail now since the dummy set above had to have worked.
			ASSERT(knobs.set(name, ivalue));
		}

		if (resourceFromURL != nullptr)
			*resourceFromURL = resource.toString();

		Optional<S3BlobStoreEndpoint::Credentials> creds;
		if (cred.present()) {
			StringRef c(cred.get());
			StringRef key = c.eat(":");
			StringRef secret = c.eat(":");
			StringRef securityToken = c.eat();
			creds = S3BlobStoreEndpoint::Credentials{ key.toString(), secret.toString(), securityToken.toString() };
		}

		if (region.empty() && CLIENT_KNOBS->HTTP_REQUEST_AWS_V4_HEADER) {
			throw std::string(
			    "Failed to get region from host or parameter in url, region is required for aws v4 signature");
		}

		return makeReference<S3BlobStoreEndpoint>(
		    host.toString(), service.toString(), region, proxyHost, proxyPort, creds, knobs, extraHeaders);

	} catch (std::string& err) {
		if (error != nullptr)
			*error = err;
		TraceEvent(SevWarnAlways, "S3BlobStoreEndpointBadURL")
		    .suppressFor(60)
		    .detail("Description", err)
		    .detail("Format", getURLFormat())
		    .detail("URL", url);
		throw backup_invalid_url();
	}
}

std::string S3BlobStoreEndpoint::getResourceURL(std::string resource, std::string params) const {
	std::string hostPort = host;
	if (!service.empty()) {
		hostPort.append(":");
		hostPort.append(service);
	}

	// If secret isn't being looked up from credentials files then it was passed explicitly in the URL so show it here.
	std::string credsString;
	if (credentials.present()) {
		if (!lookupKey) {
			credsString = credentials.get().key;
		}
		if (!lookupSecret) {
			credsString += ":" + credentials.get().secret;
			if (!credentials.get().securityToken.empty()) {
				credsString += ":" + credentials.get().securityToken;
			}
		}
		credsString += "@";
	}

	std::string r = format("blobstore://%s%s/%s", credsString.c_str(), hostPort.c_str(), resource.c_str());

	// Get params that are deviations from knob defaults
	std::string knobParams = knobs.getURLParameters();
	if (!knobParams.empty()) {
		if (!params.empty()) {
			params.append("&");
		}
		params.append(knobParams);
	}

	for (const auto& [k, v] : extraHeaders) {
		if (!params.empty()) {
			params.append("&");
		}
		params.append("header=");
		params.append(k);
		params.append(":");
		params.append(v);
	}

	if (!params.empty()) {
		r.append("?").append(params);
	}

	return r;
}

std::string S3BlobStoreEndpoint::constructResourcePath(const std::string& bucket, const std::string& object) const {
	std::string resource;

	if (host.find(bucket + ".") != 0) {
		resource += std::string("/") + bucket; // not virtual hosting mode
	}

	if (!object.empty()) {
		// Don't add a slash if the object starts with one
		if (!object.starts_with("/")) {
			resource += "/";
		}
		resource += object;
	}

	return resource;
}

// Forward declaration for doRequest_impl to fix compilation order issue
															#line 462 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
[[nodiscard]] Future<Reference<HTTP::IncomingResponse>> doRequest_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& verb, std::string const& resource, HTTP::Headers const& headers, UnsentPacketQueue* const& pContent, int const& contentLen, std::set<unsigned int> const& successCodes );

#line 467 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

															#line 467 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via bucketExists_impl()
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class BucketExists_implActor>
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class BucketExists_implActorState {
															#line 474 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	BucketExists_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket) 
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket)
															#line 483 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("bucketExists_impl", reinterpret_cast<unsigned long>(this));

	}
	~BucketExists_implActorState() 
	{
		fdb_probe_actor_destroy("bucketExists_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 469 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateRead->getAllowance(1);
															#line 469 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<BucketExists_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 500 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BucketExists_implActor*>(this)->actor_wait_state = 1;
															#line 469 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BucketExists_implActor, 0, Void >*>(static_cast<BucketExists_implActor*>(this)));
															#line 505 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BucketExists_implActorState();
		static_cast<BucketExists_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 471 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, "");
															#line 472 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 530 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 475 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(b, "HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 475 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<BucketExists_implActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 536 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BucketExists_implActor*>(this)->actor_wait_state = 2;
															#line 475 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<BucketExists_implActor*>(this)));
															#line 541 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 471 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, "");
															#line 472 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 558 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 475 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(b, "HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 475 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<BucketExists_implActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 564 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BucketExists_implActor*>(this)->actor_wait_state = 2;
															#line 475 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<BucketExists_implActor*>(this)));
															#line 569 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BucketExists_implActor*>(this)->actor_wait_state > 0) static_cast<BucketExists_implActor*>(this)->actor_wait_state = 0;
		static_cast<BucketExists_implActor*>(this)->ActorCallback< BucketExists_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BucketExists_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BucketExists_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BucketExists_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 479 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevError, "S3ClientBucketExistsError") .detail("Bucket", bucket) .detail("Host", b->host) .errorUnsuppressed(e);
															#line 483 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 662 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 477 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<BucketExists_implActor*>(this)->SAV<bool>::futures) { (void)(r->code == 200); this->~BucketExists_implActorState(); static_cast<BucketExists_implActor*>(this)->destroy(); return 0; }
															#line 676 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<BucketExists_implActor*>(this)->SAV< bool >::value()) bool(r->code == 200);
		this->~BucketExists_implActorState();
		static_cast<BucketExists_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 477 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<BucketExists_implActor*>(this)->SAV<bool>::futures) { (void)(r->code == 200); this->~BucketExists_implActorState(); static_cast<BucketExists_implActor*>(this)->destroy(); return 0; }
															#line 688 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<BucketExists_implActor*>(this)->SAV< bool >::value()) bool(r->code == 200);
		this->~BucketExists_implActorState();
		static_cast<BucketExists_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont3(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BucketExists_implActor*>(this)->actor_wait_state > 0) static_cast<BucketExists_implActor*>(this)->actor_wait_state = 0;
		static_cast<BucketExists_implActor*>(this)->ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 471 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 472 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 779 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via bucketExists_impl()
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class BucketExists_implActor final : public Actor<bool>, public ActorCallback< BucketExists_implActor, 0, Void >, public ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<BucketExists_implActor>, public BucketExists_implActorState<BucketExists_implActor> {
															#line 784 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<BucketExists_implActor>::operator new;
	using FastAllocated<BucketExists_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4370542329954182656UL, 12221098736638565120UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BucketExists_implActor, 0, Void >;
friend struct ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	BucketExists_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket) 
															#line 802 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<bool>(),
		   BucketExists_implActorState<BucketExists_implActor>(b, bucket),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14807564591002974976UL, 18309392055811966464UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bucketExists_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BucketExists_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<bool> bucketExists_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket ) {
															#line 468 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<bool>(new BucketExists_implActor(b, bucket));
															#line 836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 486 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<bool> S3BlobStoreEndpoint::bucketExists(std::string const& bucket) {
	return bucketExists_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket);
}

															#line 845 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via objectExists_impl()
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ObjectExists_implActor>
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ObjectExists_implActorState {
															#line 852 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ObjectExists_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 863 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("objectExists_impl", reinterpret_cast<unsigned long>(this));

	}
	~ObjectExists_implActorState() 
	{
		fdb_probe_actor_destroy("objectExists_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 492 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateRead->getAllowance(1);
															#line 492 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 880 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 1;
															#line 492 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ObjectExists_implActor, 0, Void >*>(static_cast<ObjectExists_implActor*>(this)));
															#line 885 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ObjectExists_implActorState();
		static_cast<ObjectExists_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 494 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 495 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 497 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(b, "HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 497 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 914 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 2;
															#line 497 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ObjectExists_implActor*>(this)));
															#line 919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 494 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 495 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 497 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(b, "HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 497 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 2;
															#line 497 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ObjectExists_implActor*>(this)));
															#line 939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state > 0) static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 0;
		static_cast<ObjectExists_implActor*>(this)->ActorCallback< ObjectExists_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ObjectExists_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ObjectExists_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ObjectExists_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 498 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ObjectExists_implActor*>(this)->SAV<bool>::futures) { (void)(r->code == 200); this->~ObjectExists_implActorState(); static_cast<ObjectExists_implActor*>(this)->destroy(); return 0; }
															#line 1023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ObjectExists_implActor*>(this)->SAV< bool >::value()) bool(r->code == 200);
		this->~ObjectExists_implActorState();
		static_cast<ObjectExists_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 498 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ObjectExists_implActor*>(this)->SAV<bool>::futures) { (void)(r->code == 200); this->~ObjectExists_implActorState(); static_cast<ObjectExists_implActor*>(this)->destroy(); return 0; }
															#line 1035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ObjectExists_implActor*>(this)->SAV< bool >::value()) bool(r->code == 200);
		this->~ObjectExists_implActorState();
		static_cast<ObjectExists_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state > 0) static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 0;
		static_cast<ObjectExists_implActor*>(this)->ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 494 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 495 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1128 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via objectExists_impl()
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ObjectExists_implActor final : public Actor<bool>, public ActorCallback< ObjectExists_implActor, 0, Void >, public ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ObjectExists_implActor>, public ObjectExists_implActorState<ObjectExists_implActor> {
															#line 1133 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ObjectExists_implActor>::operator new;
	using FastAllocated<ObjectExists_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16283442770161119744UL, 16726954803447360768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ObjectExists_implActor, 0, Void >;
friend struct ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ObjectExists_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 1151 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<bool>(),
		   ObjectExists_implActorState<ObjectExists_implActor>(b, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1440333905109885440UL, 8919820050798510080UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("objectExists_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ObjectExists_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<bool> objectExists_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket, std::string const& object ) {
															#line 491 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<bool>(new ObjectExists_implActor(b, bucket, object));
															#line 1185 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 500 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<bool> S3BlobStoreEndpoint::objectExists(std::string const& bucket, std::string const& object) {
	return objectExists_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 1194 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via deleteObject_impl()
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class DeleteObject_implActor>
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DeleteObject_implActorState {
															#line 1201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DeleteObject_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 1212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("deleteObject_impl", reinterpret_cast<unsigned long>(this));

	}
	~DeleteObject_implActorState() 
	{
		fdb_probe_actor_destroy("deleteObject_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 506 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateDelete->getAllowance(1);
															#line 506 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 1;
															#line 506 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeleteObject_implActor, 0, Void >*>(static_cast<DeleteObject_implActor*>(this)));
															#line 1234 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeleteObject_implActorState();
		static_cast<DeleteObject_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 508 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 509 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 512 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(b, "DELETE", resource, headers, nullptr, 0, { 200, 204, 404 });
															#line 512 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1263 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 2;
															#line 512 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<DeleteObject_implActor*>(this)));
															#line 1268 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 508 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 509 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 512 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(b, "DELETE", resource, headers, nullptr, 0, { 200, 204, 404 });
															#line 512 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1283 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 2;
															#line 512 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<DeleteObject_implActor*>(this)));
															#line 1288 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteObject_implActor*>(this)->ActorCallback< DeleteObject_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteObject_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeleteObject_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeleteObject_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 516 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 1372 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 517 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarnAlways, "S3BlobStoreEndpointDeleteObjectMissing") .detail("Host", b->host) .detail("Bucket", bucket) .detail("Object", object);
															#line 1376 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 523 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<DeleteObject_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteObject_implActorState(); static_cast<DeleteObject_implActor*>(this)->destroy(); return 0; }
															#line 1380 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<DeleteObject_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteObject_implActorState();
		static_cast<DeleteObject_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 516 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 1392 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 517 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarnAlways, "S3BlobStoreEndpointDeleteObjectMissing") .detail("Host", b->host) .detail("Bucket", bucket) .detail("Object", object);
															#line 1396 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 523 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<DeleteObject_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteObject_implActorState(); static_cast<DeleteObject_implActor*>(this)->destroy(); return 0; }
															#line 1400 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<DeleteObject_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteObject_implActorState();
		static_cast<DeleteObject_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteObject_implActor*>(this)->ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 508 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 509 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1493 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via deleteObject_impl()
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DeleteObject_implActor final : public Actor<Void>, public ActorCallback< DeleteObject_implActor, 0, Void >, public ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<DeleteObject_implActor>, public DeleteObject_implActorState<DeleteObject_implActor> {
															#line 1498 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<DeleteObject_implActor>::operator new;
	using FastAllocated<DeleteObject_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3586375500435316736UL, 2705342117864334336UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeleteObject_implActor, 0, Void >;
friend struct ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DeleteObject_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 1516 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   DeleteObject_implActorState<DeleteObject_implActor>(b, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14670167000419629568UL, 2229091376396350720UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deleteObject_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeleteObject_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> deleteObject_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket, std::string const& object ) {
															#line 505 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new DeleteObject_implActor(b, bucket, object));
															#line 1550 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 525 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::deleteObject(std::string const& bucket, std::string const& object) {
	return deleteObject_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 1559 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via deleteRecursively_impl()
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class DeleteRecursively_implActor>
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DeleteRecursively_implActorState {
															#line 1566 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DeleteRecursively_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& prefix,int* const& pNumDeleted,int64_t* const& pBytesDeleted) 
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   prefix(prefix),
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pNumDeleted(pNumDeleted),
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pBytesDeleted(pBytesDeleted),
															#line 535 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resultStream(),
															#line 537 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   done(b->listObjectsStream(bucket, resultStream, prefix, '/', std::numeric_limits<int>::max()))
															#line 1585 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("deleteRecursively_impl", reinterpret_cast<unsigned long>(this));

	}
	~DeleteRecursively_implActorState() 
	{
		fdb_probe_actor_destroy("deleteRecursively_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 540 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			done = map(done, [=](Void) mutable { resultStream.sendError(end_of_stream()); return Void(); });
															#line 545 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			deleteFutures = std::list<Future<Void>>();
															#line 1602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			try {
															#line 547 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				;
															#line 1606 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeleteRecursively_implActorState();
		static_cast<DeleteRecursively_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 580 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 1635 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 576 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_end_of_stream)
															#line 1645 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 577 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 1649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 550 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_0 = done;
															#line 548 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1674 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 554 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		auto __when_expr_1 = resultStream.getFuture();
															#line 1678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<S3BlobStoreEndpoint::ListResult>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<S3BlobStoreEndpoint::ListResult>>::value, "invalid type");
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.pop(), loopDepth); };
		static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 1;
															#line 550 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeleteRecursively_implActor, 0, Void >*>(static_cast<DeleteRecursively_implActor*>(this)));
															#line 554 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >*>(static_cast<DeleteRecursively_implActor*>(this)));
															#line 1686 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 570 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 1695 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 551 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		done = Never();
															#line 1704 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 551 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		done = Never();
															#line 1713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(S3BlobStoreEndpoint::ListResult const& list,int loopDepth) 
	{
															#line 555 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& object : list.objects ) {
															#line 556 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			deleteFutures.push_back(map(b->deleteObject(bucket, object.name), [=](Void) -> Void { if (pNumDeleted != nullptr) { ++*pNumDeleted; } if (pBytesDeleted != nullptr) { *pBytesDeleted += object.size; } return Void(); }));
															#line 1724 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(S3BlobStoreEndpoint::ListResult && list,int loopDepth) 
	{
															#line 555 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& object : list.objects ) {
															#line 556 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			deleteFutures.push_back(map(b->deleteObject(bucket, object.name), [=](Void) -> Void { if (pNumDeleted != nullptr) { ++*pNumDeleted; } if (pBytesDeleted != nullptr) { *pBytesDeleted += object.size; } return Void(); }));
															#line 1736 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteRecursively_implActor*>(this)->ActorCallback< DeleteRecursively_implActor, 0, Void >::remove();
		static_cast<DeleteRecursively_implActor*>(this)->ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeleteRecursively_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >*,S3BlobStoreEndpoint::ListResult const& value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >*,S3BlobStoreEndpoint::ListResult && value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >*,Error err) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 570 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!(deleteFutures.size() > CLIENT_KNOBS->BLOBSTORE_CONCURRENT_REQUESTS))
															#line 1880 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
			return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 571 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deleteFutures.front();
															#line 571 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1888 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 2;
															#line 571 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeleteRecursively_implActor, 2, Void >*>(static_cast<DeleteRecursively_implActor*>(this)));
															#line 1893 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 572 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		deleteFutures.pop_front();
															#line 1915 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 572 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		deleteFutures.pop_front();
															#line 1924 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteRecursively_implActor*>(this)->ActorCallback< DeleteRecursively_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeleteRecursively_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 585 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<DeleteRecursively_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteRecursively_implActorState(); static_cast<DeleteRecursively_implActor*>(this)->destroy(); return 0; }
															#line 2008 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<DeleteRecursively_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteRecursively_implActorState();
		static_cast<DeleteRecursively_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 580 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!(deleteFutures.size() > 0))
															#line 2027 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 581 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_3 = deleteFutures.front();
															#line 581 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 3;
															#line 581 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeleteRecursively_implActor, 3, Void >*>(static_cast<DeleteRecursively_implActor*>(this)));
															#line 2040 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 582 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		deleteFutures.pop_front();
															#line 2062 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 582 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		deleteFutures.pop_front();
															#line 2071 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteRecursively_implActor*>(this)->ActorCallback< DeleteRecursively_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DeleteRecursively_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string prefix;
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int* pNumDeleted;
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int64_t* pBytesDeleted;
															#line 535 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PromiseStream<S3BlobStoreEndpoint::ListResult> resultStream;
															#line 537 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Future<Void> done;
															#line 545 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::list<Future<Void>> deleteFutures;
															#line 2167 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via deleteRecursively_impl()
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DeleteRecursively_implActor final : public Actor<Void>, public ActorCallback< DeleteRecursively_implActor, 0, Void >, public ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >, public ActorCallback< DeleteRecursively_implActor, 2, Void >, public ActorCallback< DeleteRecursively_implActor, 3, Void >, public FastAllocated<DeleteRecursively_implActor>, public DeleteRecursively_implActorState<DeleteRecursively_implActor> {
															#line 2172 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<DeleteRecursively_implActor>::operator new;
	using FastAllocated<DeleteRecursively_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8242382639751734016UL, 12050531015013355520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeleteRecursively_implActor, 0, Void >;
friend struct ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >;
friend struct ActorCallback< DeleteRecursively_implActor, 2, Void >;
friend struct ActorCallback< DeleteRecursively_implActor, 3, Void >;
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DeleteRecursively_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& prefix,int* const& pNumDeleted,int64_t* const& pBytesDeleted) 
															#line 2192 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   DeleteRecursively_implActorState<DeleteRecursively_implActor>(b, bucket, prefix, pNumDeleted, pBytesDeleted),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14744632183853925888UL, 8951783522599013632UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deleteRecursively_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeleteRecursively_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeleteRecursively_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeleteRecursively_implActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> deleteRecursively_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket, std::string const& prefix, int* const& pNumDeleted, int64_t* const& pBytesDeleted ) {
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new DeleteRecursively_implActor(b, bucket, prefix, pNumDeleted, pBytesDeleted));
															#line 2227 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 587 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::deleteRecursively(std::string const& bucket,
                                                    std::string prefix,
                                                    int* pNumDeleted,
                                                    int64_t* pBytesDeleted) {
	return deleteRecursively_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), bucket, prefix, pNumDeleted, pBytesDeleted);
}

															#line 2240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via createBucket_impl()
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class CreateBucket_implActor>
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class CreateBucket_implActorState {
															#line 2247 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	CreateBucket_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket) 
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 597 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   packets()
															#line 2258 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("createBucket_impl", reinterpret_cast<unsigned long>(this));

	}
	~CreateBucket_implActorState() 
	{
		fdb_probe_actor_destroy("createBucket_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 598 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateWrite->getAllowance(1);
															#line 598 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2275 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 1;
															#line 598 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 0, Void >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2280 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CreateBucket_implActorState();
		static_cast<CreateBucket_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 600 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<bool> __when_expr_1 = b->bucketExists(bucket);
															#line 600 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 2;
															#line 600 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 1, bool >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2310 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 600 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<bool> __when_expr_1 = b->bucketExists(bucket);
															#line 600 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2321 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 2;
															#line 600 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 1, bool >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2326 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state > 0) static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 0;
		static_cast<CreateBucket_implActor*>(this)->ActorCallback< CreateBucket_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CreateBucket_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(bool const& exists,int loopDepth) 
	{
															#line 601 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!exists)
															#line 2410 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 602 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			resource = b->constructResourcePath(bucket, "");
															#line 603 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers = HTTP::Headers();
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string region = b->getRegion();
															#line 606 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (region.empty())
															#line 2420 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 607 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = doRequest_impl(b, "PUT", resource, headers, nullptr, 0, { 200, 409 });
															#line 607 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2426 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 3;
															#line 607 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2431 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 610 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				Standalone<StringRef> body( format("<CreateBucketConfiguration xmlns=\"http://s3.amazonaws.com/doc/2006-03-01/\">" "  <LocationConstraint>%s</LocationConstraint>" "</CreateBucketConfiguration>", region.c_str()));
															#line 615 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				PacketWriter pw(packets.getWriteBuffer(), nullptr, Unversioned());
															#line 616 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				pw.serializeBytes(body);
															#line 618 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_3 = doRequest_impl(b, "PUT", resource, headers, &packets, body.size(), { 200, 409 });
															#line 618 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2446 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when2(__when_expr_3.get(), loopDepth); };
				static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 4;
															#line 618 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2451 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(bool && exists,int loopDepth) 
	{
															#line 601 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!exists)
															#line 2466 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 602 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			resource = b->constructResourcePath(bucket, "");
															#line 603 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers = HTTP::Headers();
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string region = b->getRegion();
															#line 606 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (region.empty())
															#line 2476 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 607 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = doRequest_impl(b, "PUT", resource, headers, nullptr, 0, { 200, 409 });
															#line 607 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 3;
															#line 607 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 610 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				Standalone<StringRef> body( format("<CreateBucketConfiguration xmlns=\"http://s3.amazonaws.com/doc/2006-03-01/\">" "  <LocationConstraint>%s</LocationConstraint>" "</CreateBucketConfiguration>", region.c_str()));
															#line 615 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				PacketWriter pw(packets.getWriteBuffer(), nullptr, Unversioned());
															#line 616 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				pw.serializeBytes(body);
															#line 618 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_3 = doRequest_impl(b, "PUT", resource, headers, &packets, body.size(), { 200, 409 });
															#line 618 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2502 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when2(__when_expr_3.get(), loopDepth); };
				static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 4;
															#line 618 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2507 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(bool const& exists,int loopDepth) 
	{
		loopDepth = a_body1cont2(exists, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(bool && exists,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(exists), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state > 0) static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 0;
		static_cast<CreateBucket_implActor*>(this)->ActorCallback< CreateBucket_implActor, 1, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 1, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 1, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CreateBucket_implActor, 1, bool >*,Error err) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 622 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<CreateBucket_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CreateBucket_implActorState(); static_cast<CreateBucket_implActor*>(this)->destroy(); return 0; }
															#line 2597 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<CreateBucket_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CreateBucket_implActorState();
		static_cast<CreateBucket_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont5(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state > 0) static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 0;
		static_cast<CreateBucket_implActor*>(this)->ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont6(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state > 0) static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 0;
		static_cast<CreateBucket_implActor*>(this)->ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 597 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue packets;
															#line 602 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 603 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 2795 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via createBucket_impl()
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class CreateBucket_implActor final : public Actor<Void>, public ActorCallback< CreateBucket_implActor, 0, Void >, public ActorCallback< CreateBucket_implActor, 1, bool >, public ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >, public ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >, public FastAllocated<CreateBucket_implActor>, public CreateBucket_implActorState<CreateBucket_implActor> {
															#line 2800 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<CreateBucket_implActor>::operator new;
	using FastAllocated<CreateBucket_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3986791070375695360UL, 9899054833098301184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CreateBucket_implActor, 0, Void >;
friend struct ActorCallback< CreateBucket_implActor, 1, bool >;
friend struct ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >;
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	CreateBucket_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket) 
															#line 2820 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   CreateBucket_implActorState<CreateBucket_implActor>(b, bucket),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13190372849516335360UL, 14355671705917093888UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("createBucket_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CreateBucket_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CreateBucket_implActor, 1, bool >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> createBucket_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket ) {
															#line 596 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new CreateBucket_implActor(b, bucket));
															#line 2856 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 624 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::createBucket(std::string const& bucket) {
	return createBucket_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket);
}

															#line 2865 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via objectSize_impl()
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ObjectSize_implActor>
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ObjectSize_implActorState {
															#line 2872 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ObjectSize_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 2883 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("objectSize_impl", reinterpret_cast<unsigned long>(this));

	}
	~ObjectSize_implActorState() 
	{
		fdb_probe_actor_destroy("objectSize_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 630 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateRead->getAllowance(1);
															#line 630 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2900 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 1;
															#line 630 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ObjectSize_implActor, 0, Void >*>(static_cast<ObjectSize_implActor*>(this)));
															#line 2905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ObjectSize_implActorState();
		static_cast<ObjectSize_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 632 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 633 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 635 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(b, "HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 635 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 2;
															#line 635 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ObjectSize_implActor*>(this)));
															#line 2939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 632 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 633 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 635 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(b, "HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 635 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2954 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 2;
															#line 635 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ObjectSize_implActor*>(this)));
															#line 2959 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state > 0) static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 0;
		static_cast<ObjectSize_implActor*>(this)->ActorCallback< ObjectSize_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ObjectSize_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ObjectSize_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ObjectSize_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 636 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 3043 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 637 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(file_not_found(), loopDepth);
															#line 3047 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 638 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ObjectSize_implActor*>(this)->SAV<int64_t>::futures) { (void)(r->data.contentLen); this->~ObjectSize_implActorState(); static_cast<ObjectSize_implActor*>(this)->destroy(); return 0; }
															#line 3051 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ObjectSize_implActor*>(this)->SAV< int64_t >::value()) int64_t(r->data.contentLen);
		this->~ObjectSize_implActorState();
		static_cast<ObjectSize_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 636 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 3063 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 637 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(file_not_found(), loopDepth);
															#line 3067 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 638 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ObjectSize_implActor*>(this)->SAV<int64_t>::futures) { (void)(r->data.contentLen); this->~ObjectSize_implActorState(); static_cast<ObjectSize_implActor*>(this)->destroy(); return 0; }
															#line 3071 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ObjectSize_implActor*>(this)->SAV< int64_t >::value()) int64_t(r->data.contentLen);
		this->~ObjectSize_implActorState();
		static_cast<ObjectSize_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state > 0) static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 0;
		static_cast<ObjectSize_implActor*>(this)->ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 632 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 633 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 3164 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via objectSize_impl()
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ObjectSize_implActor final : public Actor<int64_t>, public ActorCallback< ObjectSize_implActor, 0, Void >, public ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ObjectSize_implActor>, public ObjectSize_implActorState<ObjectSize_implActor> {
															#line 3169 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ObjectSize_implActor>::operator new;
	using FastAllocated<ObjectSize_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3865974153018148608UL, 10134878806287088384UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int64_t>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ObjectSize_implActor, 0, Void >;
friend struct ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ObjectSize_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 3187 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<int64_t>(),
		   ObjectSize_implActorState<ObjectSize_implActor>(b, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4058515382792528128UL, 9575206348893316608UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("objectSize_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ObjectSize_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<int64_t> objectSize_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket, std::string const& object ) {
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<int64_t>(new ObjectSize_implActor(b, bucket, object));
															#line 3221 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 640 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<int64_t> S3BlobStoreEndpoint::objectSize(std::string const& bucket, std::string const& object) {
	return objectSize_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

// Try to read a file, parse it as JSON, and return the resulting document.
// It will NOT throw if any errors are encountered, it will just return an empty
// JSON object and will log trace events for the errors encountered.
															#line 3233 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via tryReadJSONFile()
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class TryReadJSONFileActor>
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class TryReadJSONFileActorState {
															#line 3240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	TryReadJSONFileActorState(std::string const& path) 
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : path(path),
															#line 649 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   content(),
															#line 652 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   errorEventType("BlobCredentialFileError")
															#line 3251 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("tryReadJSONFile", reinterpret_cast<unsigned long>(this));

	}
	~TryReadJSONFileActorState() 
	{
		fdb_probe_actor_destroy("tryReadJSONFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 655 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_NO_AIO | IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED, 0);
															#line 655 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3269 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 1;
															#line 655 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*>(static_cast<TryReadJSONFileActor*>(this)));
															#line 3274 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TryReadJSONFileActorState();
		static_cast<TryReadJSONFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 677 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<TryReadJSONFileActor*>(this)->SAV<Optional<json_spirit::mObject>>::futures) { (void)(Optional<json_spirit::mObject>()); this->~TryReadJSONFileActorState(); static_cast<TryReadJSONFileActor*>(this)->destroy(); return 0; }
															#line 3303 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<TryReadJSONFileActor*>(this)->SAV< Optional<json_spirit::mObject> >::value()) Optional<json_spirit::mObject>(Optional<json_spirit::mObject>());
		this->~TryReadJSONFileActorState();
		static_cast<TryReadJSONFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 673 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 3316 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 674 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, errorEventType).errorUnsuppressed(e).suppressFor(60).detail("File", path);
															#line 3320 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 657 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = f->size();
															#line 657 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3338 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 2;
															#line 657 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TryReadJSONFileActor, 1, int64_t >*>(static_cast<TryReadJSONFileActor*>(this)));
															#line 3343 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __f,int loopDepth) 
	{
															#line 655 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		f = __f;
															#line 3352 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __f,int loopDepth) 
	{
		f = std::move(__f);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state > 0) static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 0;
		static_cast<TryReadJSONFileActor*>(this)->ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 658 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		buf = makeString(size);
															#line 659 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<int> __when_expr_2 = f->read(mutateString(buf), size, 0);
															#line 659 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3435 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 3;
															#line 659 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TryReadJSONFileActor, 2, int >*>(static_cast<TryReadJSONFileActor*>(this)));
															#line 3440 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(int64_t const& __size,int loopDepth) 
	{
															#line 657 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		size = __size;
															#line 3449 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(int64_t && __size,int loopDepth) 
	{
		size = std::move(__size);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state > 0) static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 0;
		static_cast<TryReadJSONFileActor*>(this)->ActorCallback< TryReadJSONFileActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TryReadJSONFileActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int const& r,int loopDepth) 
	{
															#line 660 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		ASSERT(r == size);
															#line 661 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		content = buf.toString();
															#line 664 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		errorEventType = "BlobCredentialFileParseFailed";
															#line 665 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		json_spirit::mValue json;
															#line 666 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		json_spirit::read_string(content, json);
															#line 667 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (json.type() == json_spirit::obj_type)
															#line 3538 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 668 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<TryReadJSONFileActor*>(this)->SAV<Optional<json_spirit::mObject>>::futures) { (void)(json.get_obj()); this->~TryReadJSONFileActorState(); static_cast<TryReadJSONFileActor*>(this)->destroy(); return 0; }
															#line 3542 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<TryReadJSONFileActor*>(this)->SAV< Optional<json_spirit::mObject> >::value()) Optional<json_spirit::mObject>(json.get_obj());
			this->~TryReadJSONFileActorState();
			static_cast<TryReadJSONFileActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 670 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "BlobCredentialFileNotJSONObject").suppressFor(60).detail("File", path);
															#line 3552 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(int && r,int loopDepth) 
	{
															#line 660 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		ASSERT(r == size);
															#line 661 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		content = buf.toString();
															#line 664 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		errorEventType = "BlobCredentialFileParseFailed";
															#line 665 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		json_spirit::mValue json;
															#line 666 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		json_spirit::read_string(content, json);
															#line 667 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (json.type() == json_spirit::obj_type)
															#line 3572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 668 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<TryReadJSONFileActor*>(this)->SAV<Optional<json_spirit::mObject>>::futures) { (void)(json.get_obj()); this->~TryReadJSONFileActorState(); static_cast<TryReadJSONFileActor*>(this)->destroy(); return 0; }
															#line 3576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<TryReadJSONFileActor*>(this)->SAV< Optional<json_spirit::mObject> >::value()) Optional<json_spirit::mObject>(json.get_obj());
			this->~TryReadJSONFileActorState();
			static_cast<TryReadJSONFileActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 670 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "BlobCredentialFileNotJSONObject").suppressFor(60).detail("File", path);
															#line 3586 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(int const& r,int loopDepth) 
	{
		loopDepth = a_body1cont4(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(int && r,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state > 0) static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 0;
		static_cast<TryReadJSONFileActor*>(this)->ActorCallback< TryReadJSONFileActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TryReadJSONFileActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string path;
															#line 649 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string content;
															#line 652 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	const char* errorEventType;
															#line 655 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<IAsyncFile> f;
															#line 657 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int64_t size;
															#line 658 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Standalone<StringRef> buf;
															#line 3692 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via tryReadJSONFile()
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class TryReadJSONFileActor final : public Actor<Optional<json_spirit::mObject>>, public ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >, public ActorCallback< TryReadJSONFileActor, 1, int64_t >, public ActorCallback< TryReadJSONFileActor, 2, int >, public FastAllocated<TryReadJSONFileActor>, public TryReadJSONFileActorState<TryReadJSONFileActor> {
															#line 3697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<TryReadJSONFileActor>::operator new;
	using FastAllocated<TryReadJSONFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6541871712287012864UL, 9166111087234443776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<json_spirit::mObject>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< TryReadJSONFileActor, 1, int64_t >;
friend struct ActorCallback< TryReadJSONFileActor, 2, int >;
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	TryReadJSONFileActor(std::string const& path) 
															#line 3716 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Optional<json_spirit::mObject>>(),
		   TryReadJSONFileActorState<TryReadJSONFileActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16806631701943569920UL, 8005239482977197312UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tryReadJSONFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TryReadJSONFileActor, 1, int64_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TryReadJSONFileActor, 2, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Optional<json_spirit::mObject>> tryReadJSONFile( std::string const& path ) {
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Optional<json_spirit::mObject>>(new TryReadJSONFileActor(path));
															#line 3751 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 679 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

// If the credentials expire, the connection will eventually fail and be discarded from the pool, and then a new
// connection will be constructed, which will call this again to get updated credentials
static S3BlobStoreEndpoint::Credentials getSecretSdk() {
#ifdef WITH_AWS_BACKUP
	double elapsed = -timer_monotonic();
	Aws::Auth::AWSCredentials awsCreds = FDBAWSCredentialsProvider::getAwsCredentials();
	elapsed += timer_monotonic();

	if (awsCreds.IsEmpty()) {
		TraceEvent(SevWarn, "S3BlobStoreAWSCredsEmpty");
		throw backup_auth_missing();
	}

	S3BlobStoreEndpoint::Credentials fdbCreds;
	fdbCreds.key = awsCreds.GetAWSAccessKeyId();
	fdbCreds.secret = awsCreds.GetAWSSecretKey();
	fdbCreds.securityToken = awsCreds.GetSessionToken();

	TraceEvent("S3BlobStoreGotSdkCredentials").suppressFor(60).detail("Duration", elapsed);

	return fdbCreds;
#else
	TraceEvent(SevError, "S3BlobStoreNoSDK");
	throw backup_auth_missing();
#endif
}

															#line 3783 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via updateSecret_impl()
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class UpdateSecret_implActor>
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class UpdateSecret_implActorState {
															#line 3790 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UpdateSecret_implActorState(Reference<S3BlobStoreEndpoint> const& b) 
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b)
															#line 3797 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("updateSecret_impl", reinterpret_cast<unsigned long>(this));

	}
	~UpdateSecret_implActorState() 
	{
		fdb_probe_actor_destroy("updateSecret_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 708 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (b->knobs.sdk_auth)
															#line 3812 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 709 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				b->credentials = getSecretSdk();
															#line 710 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 3818 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~UpdateSecret_implActorState();
				static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 712 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::vector<std::string>* pFiles = (std::vector<std::string>*)g_network->global(INetwork::enBlobCredentialFiles);
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (pFiles == nullptr)
															#line 3828 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 714 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 3832 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~UpdateSecret_implActorState();
				static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 716 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!b->credentials.present())
															#line 3840 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 717 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 3844 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~UpdateSecret_implActorState();
				static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 720 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			reads = std::vector<Future<Optional<json_spirit::mObject>>>();
															#line 721 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for( auto& f : *pFiles ) {
															#line 722 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				reads.push_back(tryReadJSONFile(f));
															#line 3856 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 724 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitForAll(reads);
															#line 724 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<UpdateSecret_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3862 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateSecret_implActor*>(this)->actor_wait_state = 1;
															#line 724 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateSecret_implActor, 0, Void >*>(static_cast<UpdateSecret_implActor*>(this)));
															#line 3867 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateSecret_implActorState();
		static_cast<UpdateSecret_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 726 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string accessKey = b->lookupKey ? "" : b->credentials.get().key;
															#line 727 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string credentialsFileKey = accessKey + "@" + b->host;
															#line 729 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		int invalid = 0;
															#line 731 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& f : reads ) {
															#line 733 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!f.get().present())
															#line 3898 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 734 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, "S3BlobStoreAuthMissingNotPresent") .detail("Endpoint", b->host) .detail("Service", b->service);
															#line 737 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				++invalid;
															#line 3904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				continue;
			}
															#line 741 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			JSONDoc doc(f.get().get());
															#line 742 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (doc.has("accounts") && doc.last().type() == json_spirit::obj_type)
															#line 3911 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 743 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				JSONDoc accounts(doc.last().get_obj());
															#line 744 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (accounts.has(credentialsFileKey, false) && accounts.last().type() == json_spirit::obj_type)
															#line 3917 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 745 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					JSONDoc account(accounts.last());
															#line 746 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					S3BlobStoreEndpoint::Credentials creds = b->credentials.get();
															#line 747 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (b->lookupKey)
															#line 3925 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 748 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string apiKey;
															#line 749 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (account.tryGet("api_key", apiKey))
															#line 3931 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 750 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							creds.key = apiKey;
															#line 3935 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
							continue;
						}
					}
															#line 754 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (b->lookupSecret)
															#line 3944 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 755 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string secret;
															#line 756 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (account.tryGet("secret", secret))
															#line 3950 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 757 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							creds.secret = secret;
															#line 3954 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
							continue;
						}
					}
															#line 761 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					std::string token;
															#line 762 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (account.tryGet("token", token))
															#line 3965 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 763 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						creds.securityToken = token;
															#line 3969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 764 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					b->credentials = creds;
															#line 765 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 3975 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~UpdateSecret_implActorState();
					static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
				else
				{
															#line 767 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevWarn, "S3BlobStoreAuthFoundAccountFAILED") .detail("CredentialFileKey", credentialsFileKey);
															#line 3985 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
		}
															#line 774 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (invalid > 0)
															#line 3991 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 775 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreInvalidCredentials") .detail("Endpoint", b->host) .detail("Service", b->service) .detail("Invalid", invalid);
															#line 779 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(backup_auth_unreadable(), loopDepth);
															#line 3997 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 783 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent(SevWarn, "S3BlobStoreAuthMissing") .detail("Endpoint", b->host) .detail("Service", b->service) .detail("Reason", "No valid credentials found");
															#line 787 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		return a_body1Catch1(backup_auth_missing(), loopDepth);
															#line 4003 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 726 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string accessKey = b->lookupKey ? "" : b->credentials.get().key;
															#line 727 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string credentialsFileKey = accessKey + "@" + b->host;
															#line 729 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		int invalid = 0;
															#line 731 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& f : reads ) {
															#line 733 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!f.get().present())
															#line 4019 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 734 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, "S3BlobStoreAuthMissingNotPresent") .detail("Endpoint", b->host) .detail("Service", b->service);
															#line 737 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				++invalid;
															#line 4025 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				continue;
			}
															#line 741 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			JSONDoc doc(f.get().get());
															#line 742 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (doc.has("accounts") && doc.last().type() == json_spirit::obj_type)
															#line 4032 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 743 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				JSONDoc accounts(doc.last().get_obj());
															#line 744 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (accounts.has(credentialsFileKey, false) && accounts.last().type() == json_spirit::obj_type)
															#line 4038 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 745 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					JSONDoc account(accounts.last());
															#line 746 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					S3BlobStoreEndpoint::Credentials creds = b->credentials.get();
															#line 747 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (b->lookupKey)
															#line 4046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 748 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string apiKey;
															#line 749 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (account.tryGet("api_key", apiKey))
															#line 4052 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 750 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							creds.key = apiKey;
															#line 4056 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
							continue;
						}
					}
															#line 754 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (b->lookupSecret)
															#line 4065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 755 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string secret;
															#line 756 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (account.tryGet("secret", secret))
															#line 4071 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 757 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							creds.secret = secret;
															#line 4075 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
							continue;
						}
					}
															#line 761 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					std::string token;
															#line 762 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (account.tryGet("token", token))
															#line 4086 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 763 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						creds.securityToken = token;
															#line 4090 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 764 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					b->credentials = creds;
															#line 765 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 4096 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~UpdateSecret_implActorState();
					static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
				else
				{
															#line 767 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevWarn, "S3BlobStoreAuthFoundAccountFAILED") .detail("CredentialFileKey", credentialsFileKey);
															#line 4106 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
		}
															#line 774 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (invalid > 0)
															#line 4112 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 775 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreInvalidCredentials") .detail("Endpoint", b->host) .detail("Service", b->service) .detail("Invalid", invalid);
															#line 779 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(backup_auth_unreadable(), loopDepth);
															#line 4118 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 783 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent(SevWarn, "S3BlobStoreAuthMissing") .detail("Endpoint", b->host) .detail("Service", b->service) .detail("Reason", "No valid credentials found");
															#line 787 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		return a_body1Catch1(backup_auth_missing(), loopDepth);
															#line 4124 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateSecret_implActor*>(this)->actor_wait_state > 0) static_cast<UpdateSecret_implActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateSecret_implActor*>(this)->ActorCallback< UpdateSecret_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateSecret_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateSecret_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateSecret_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateSecret_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateSecret_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateSecret_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 720 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::vector<Future<Optional<json_spirit::mObject>>> reads;
															#line 4207 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via updateSecret_impl()
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class UpdateSecret_implActor final : public Actor<Void>, public ActorCallback< UpdateSecret_implActor, 0, Void >, public FastAllocated<UpdateSecret_implActor>, public UpdateSecret_implActorState<UpdateSecret_implActor> {
															#line 4212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<UpdateSecret_implActor>::operator new;
	using FastAllocated<UpdateSecret_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15665212411835073536UL, 4900518122576739840UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateSecret_implActor, 0, Void >;
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UpdateSecret_implActor(Reference<S3BlobStoreEndpoint> const& b) 
															#line 4229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateSecret_implActorState<UpdateSecret_implActor>(b),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateSecret_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7400895728881971712UL, 1782739488599957760UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateSecret_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateSecret_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateSecret_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateSecret_implActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> updateSecret_impl( Reference<S3BlobStoreEndpoint> const& b ) {
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new UpdateSecret_implActor(b));
															#line 4262 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 789 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::updateSecret() {
	return updateSecret_impl(Reference<S3BlobStoreEndpoint>::addRef(this));
}

															#line 4271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via connect_impl()
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class Connect_implActor>
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class Connect_implActorState {
															#line 4278 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Connect_implActorState(Reference<S3BlobStoreEndpoint> const& b,bool* const& reusingConn) 
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   reusingConn(reusingConn)
															#line 4287 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("connect_impl", reinterpret_cast<unsigned long>(this));

	}
	~Connect_implActorState() 
	{
		fdb_probe_actor_destroy("connect_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 797 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			*reusingConn = false;
															#line 798 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;!b->connectionPool->pool.empty();) {
															#line 799 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				S3BlobStoreEndpoint::ReusableConnection rconn = b->connectionPool->pool.front();
															#line 800 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				b->connectionPool->pool.pop();
															#line 803 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (rconn.expirationTime > now())
															#line 4310 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 804 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					*reusingConn = true;
															#line 805 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					++b->blobStats->reusedConnections;
															#line 806 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent("S3BlobStoreEndpointReusingConnected") .suppressFor(60) .detail("RemoteEndpoint", rconn.conn->getPeerAddress()) .detail("ExpiresIn", rconn.expirationTime - now()) .detail("Proxy", b->proxyHost.orDefault(""));
															#line 811 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<Connect_implActor*>(this)->SAV<S3BlobStoreEndpoint::ReusableConnection>::futures) { (void)(rconn); this->~Connect_implActorState(); static_cast<Connect_implActor*>(this)->destroy(); return 0; }
															#line 4320 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<Connect_implActor*>(this)->SAV< S3BlobStoreEndpoint::ReusableConnection >::value()) S3BlobStoreEndpoint::ReusableConnection(rconn);
					this->~Connect_implActorState();
					static_cast<Connect_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 813 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				++b->blobStats->expiredConnections;
															#line 4328 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 815 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++b->blobStats->newConnections;
															#line 816 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string host = b->host, service = b->service;
															#line 817 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevDebug, "S3BlobStoreEndpointBuildingNewConnection") .detail("UseProxy", b->useProxy) .detail("TLS", b->knobs.secure_connection == 1) .detail("Host", host) .detail("Service", service) .log();
															#line 823 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (service.empty())
															#line 4338 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 824 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (b->useProxy)
															#line 4342 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 825 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					fprintf(stderr, "ERROR: Port can't be empty when using HTTP proxy.\n");
															#line 826 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1Catch1(connection_failed(), loopDepth);
															#line 4348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 828 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				service = b->knobs.secure_connection ? "https" : "http";
															#line 4352 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 830 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			bool isTLS = b->knobs.isTLS();
															#line 831 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			conn = Reference<IConnection>();
															#line 832 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (b->useProxy)
															#line 4360 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 833 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (isTLS)
															#line 4364 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 834 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					StrictFuture<Reference<IConnection>> __when_expr_0 = HTTP::proxyConnect(host, service, b->proxyHost.get(), b->proxyPort.get());
															#line 834 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4370 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<Connect_implActor*>(this)->actor_wait_state = 1;
															#line 834 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 0, Reference<IConnection> >*>(static_cast<Connect_implActor*>(this)));
															#line 4375 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 838 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					host = b->proxyHost.get();
															#line 839 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					service = b->proxyPort.get();
															#line 840 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					StrictFuture<Reference<IConnection>> __when_expr_1 = INetworkConnections::net()->connect(host, service, false);
															#line 840 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4388 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
					static_cast<Connect_implActor*>(this)->actor_wait_state = 2;
															#line 840 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 1, Reference<IConnection> >*>(static_cast<Connect_implActor*>(this)));
															#line 4393 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					loopDepth = 0;
				}
			}
			else
			{
															#line 844 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Void> __when_expr_2 = store(conn, INetworkConnections::net()->connect(host, service, isTLS));
															#line 844 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4403 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1when3(__when_expr_2.get(), loopDepth); };
				static_cast<Connect_implActor*>(this)->actor_wait_state = 3;
															#line 844 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 2, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 4408 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Connect_implActorState();
		static_cast<Connect_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 846 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_3 = conn->connectHandshake();
															#line 846 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4434 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<Connect_implActor*>(this)->actor_wait_state = 4;
															#line 846 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 3, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 4439 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Reference<IConnection> const& _conn,int loopDepth) 
	{
															#line 836 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		conn = _conn;
															#line 4454 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Reference<IConnection> && _conn,int loopDepth) 
	{
															#line 836 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		conn = _conn;
															#line 4463 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IConnection> const& _conn,int loopDepth) 
	{
		loopDepth = a_body1cont7(_conn, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IConnection> && _conn,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_conn), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 0, Reference<IConnection> >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont9(Reference<IConnection> const& _conn,int loopDepth) 
	{
															#line 841 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		conn = _conn;
															#line 4547 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Reference<IConnection> && _conn,int loopDepth) 
	{
															#line 841 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		conn = _conn;
															#line 4556 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Reference<IConnection> const& _conn,int loopDepth) 
	{
		loopDepth = a_body1cont9(_conn, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Reference<IConnection> && _conn,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_conn), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 1, Reference<IConnection> >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 1, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 1, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 1, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont12(Void const& _,int loopDepth) 
	{
															#line 848 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent("S3BlobStoreEndpointNewConnectionSuccess") .suppressFor(60) .detail("RemoteEndpoint", conn->getPeerAddress()) .detail("ExpiresIn", b->knobs.max_connection_life) .detail("Proxy", b->proxyHost.orDefault(""));
															#line 854 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (b->lookupKey || b->lookupSecret || b->knobs.sdk_auth)
															#line 4729 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 855 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = b->updateSecret();
															#line 855 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont12when1(__when_expr_4.get(), loopDepth); };
			static_cast<Connect_implActor*>(this)->actor_wait_state = 5;
															#line 855 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 4, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 4740 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont12cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont12(Void && _,int loopDepth) 
	{
															#line 848 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent("S3BlobStoreEndpointNewConnectionSuccess") .suppressFor(60) .detail("RemoteEndpoint", conn->getPeerAddress()) .detail("ExpiresIn", b->knobs.max_connection_life) .detail("Proxy", b->proxyHost.orDefault(""));
															#line 854 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (b->lookupKey || b->lookupSecret || b->knobs.sdk_auth)
															#line 4756 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 855 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = b->updateSecret();
															#line 855 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4762 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont12when1(__when_expr_4.get(), loopDepth); };
			static_cast<Connect_implActor*>(this)->actor_wait_state = 5;
															#line 855 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 4, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 4767 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont12cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont12cont1(int loopDepth) 
	{
															#line 857 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<Connect_implActor*>(this)->SAV<S3BlobStoreEndpoint::ReusableConnection>::futures) { (void)(S3BlobStoreEndpoint::ReusableConnection({ conn, now() + b->knobs.max_connection_life })); this->~Connect_implActorState(); static_cast<Connect_implActor*>(this)->destroy(); return 0; }
															#line 4856 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<Connect_implActor*>(this)->SAV< S3BlobStoreEndpoint::ReusableConnection >::value()) S3BlobStoreEndpoint::ReusableConnection(S3BlobStoreEndpoint::ReusableConnection({ conn, now() + b->knobs.max_connection_life }));
		this->~Connect_implActorState();
		static_cast<Connect_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont12cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont12cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont12when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont12when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont12when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont12when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool* reusingConn;
															#line 831 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<IConnection> conn;
															#line 4957 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via connect_impl()
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class Connect_implActor final : public Actor<S3BlobStoreEndpoint::ReusableConnection>, public ActorCallback< Connect_implActor, 0, Reference<IConnection> >, public ActorCallback< Connect_implActor, 1, Reference<IConnection> >, public ActorCallback< Connect_implActor, 2, Void >, public ActorCallback< Connect_implActor, 3, Void >, public ActorCallback< Connect_implActor, 4, Void >, public FastAllocated<Connect_implActor>, public Connect_implActorState<Connect_implActor> {
															#line 4962 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<Connect_implActor>::operator new;
	using FastAllocated<Connect_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2873266999606263552UL, 12662092946374958848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<S3BlobStoreEndpoint::ReusableConnection>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Connect_implActor, 0, Reference<IConnection> >;
friend struct ActorCallback< Connect_implActor, 1, Reference<IConnection> >;
friend struct ActorCallback< Connect_implActor, 2, Void >;
friend struct ActorCallback< Connect_implActor, 3, Void >;
friend struct ActorCallback< Connect_implActor, 4, Void >;
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Connect_implActor(Reference<S3BlobStoreEndpoint> const& b,bool* const& reusingConn) 
															#line 4983 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<S3BlobStoreEndpoint::ReusableConnection>(),
		   Connect_implActorState<Connect_implActor>(b, reusingConn),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7092978256320716288UL, 13901349198287939840UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connect_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Connect_implActor, 0, Reference<IConnection> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Connect_implActor, 1, Reference<IConnection> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< Connect_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< Connect_implActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< Connect_implActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<S3BlobStoreEndpoint::ReusableConnection> connect_impl( Reference<S3BlobStoreEndpoint> const& b, bool* const& reusingConn ) {
															#line 794 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<S3BlobStoreEndpoint::ReusableConnection>(new Connect_implActor(b, reusingConn));
															#line 5020 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 859 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<S3BlobStoreEndpoint::ReusableConnection> S3BlobStoreEndpoint::connect(bool* reusing) {
	return connect_impl(Reference<S3BlobStoreEndpoint>::addRef(this), reusing);
}

void S3BlobStoreEndpoint::returnConnection(ReusableConnection& rconn) {
	// If it expires in the future then add it to the pool in the front
	if (rconn.expirationTime > now()) {
		connectionPool->pool.push(rconn);
	} else {
		++blobStats->expiredConnections;
	}
	rconn.conn = Reference<IConnection>();
}

std::string awsCanonicalURI(const std::string& resource, std::vector<std::string>& queryParameters, bool isV4) {
	StringRef resourceRef(resource);
	resourceRef.eat("/");
	std::string canonicalURI("/" + resourceRef.toString());
	size_t q = canonicalURI.find_last_of('?');
	if (q != canonicalURI.npos)
		canonicalURI.resize(q);
	if (isV4) {
		canonicalURI = HTTP::awsV4URIEncode(canonicalURI, false);
	} else {
		canonicalURI = HTTP::urlEncode(canonicalURI);
	}

	// Create the canonical query string
	std::string queryString;
	q = resource.find_last_of('?');
	if (q != queryString.npos)
		queryString = resource.substr(q + 1);

	StringRef qStr(queryString);
	StringRef queryParameter;
	while ((queryParameter = qStr.eat("&")) != StringRef()) {
		StringRef param = queryParameter.eat("=");
		StringRef value = queryParameter.eat();

		if (isV4) {
			queryParameters.push_back(HTTP::awsV4URIEncode(param.toString(), true) + "=" +
			                          HTTP::awsV4URIEncode(value.toString(), true));
		} else {
			queryParameters.push_back(HTTP::urlEncode(param.toString()) + "=" + HTTP::urlEncode(value.toString()));
		}
	}

	return canonicalURI;
}

// ref: https://docs.aws.amazon.com/AmazonS3/latest/API/ErrorResponses.html
std::string parseErrorCodeFromS3(std::string xmlResponse) {
	// Copy XML string to a modifiable buffer
	try {
		std::vector<char> xmlBuffer(xmlResponse.begin(), xmlResponse.end());
		xmlBuffer.push_back('\0'); // Ensure null-terminated string
		// Parse the XML
		xml_document<> doc;
		doc.parse<0>(&xmlBuffer[0]);
		// Find the root node
		xml_node<>* root = doc.first_node("Error");
		if (!root) {
			TraceEvent(SevWarn, "ParseS3XMLResponseNoError").detail("Response", xmlResponse).log();
			return "";
		}
		// Find the <Code> node
		xml_node<>* codeNode = root->first_node("Code");
		if (!codeNode) {
			TraceEvent(SevWarn, "ParseS3XMLResponseNoErrorCode").detail("Response", xmlResponse).log();
			return "";
		}
		return std::string(codeNode->value());
	} catch (Error e) {
		TraceEvent("BackupParseS3ErrorCodeFailure").errorUnsuppressed(e);
		throw backup_parse_s3_response_failure();
	} catch (...) {
		throw backup_parse_s3_response_failure();
	}
}

bool isS3TokenError(const std::string& s3Error) {
	return s3Error == "InvalidToken" || s3Error == "ExpiredToken";
}

void setHeaders(Reference<S3BlobStoreEndpoint> bstore, Reference<HTTP::OutgoingRequest> req) {
	// Finish/update the request headers (which includes Date header)
	// This must be done AFTER the connection is ready because if credentials are coming from disk they are
	// refreshed when a new connection is established and setAuthHeaders() would need the updated secret.
	if (bstore->credentials.present() && !bstore->credentials.get().securityToken.empty())
		req->data.headers["x-amz-security-token"] = bstore->credentials.get().securityToken;
	if (CLIENT_KNOBS->HTTP_REQUEST_AWS_V4_HEADER) {
		bstore->setV4AuthHeaders(req->verb, req->resource, req->data.headers);
	} else {
		bstore->setAuthHeaders(req->verb, req->resource, req->data.headers);
	}
}

std::string getCanonicalURI(Reference<S3BlobStoreEndpoint> bstore, Reference<HTTP::OutgoingRequest> req) {
	std::vector<std::string> queryParameters;
	std::string canonicalURI =
	    awsCanonicalURI(req->resource, queryParameters, CLIENT_KNOBS->HTTP_REQUEST_AWS_V4_HEADER);
	if (!queryParameters.empty()) {
		canonicalURI += "?";
		// Use raw ampersands when joining query parameters
		for (size_t i = 0; i < queryParameters.size(); ++i) {
			if (i > 0) {
				canonicalURI += "&";
			}
			canonicalURI += queryParameters[i];
		}
	}

	if (bstore->useProxy && bstore->knobs.secure_connection == 0) {
		// Has to be in absolute-form.
		canonicalURI = "http://" + bstore->host + ":" + bstore->service + canonicalURI;
	}
	return canonicalURI;
}

void populateDryrunRequest(Reference<HTTP::OutgoingRequest> dryrunRequest,
                           Reference<S3BlobStoreEndpoint> bstore,
                           std::string bucket) {
	// dryrun with a check bucket exist request, to avoid sending duplicate data
	HTTP::Headers headers;
	dryrunRequest->verb = "GET";
	dryrunRequest->data.contentLen = 0;
	dryrunRequest->data.headers = headers;
	dryrunRequest->data.headers["Host"] = bstore->host;
	dryrunRequest->data.headers["Accept"] = "application/xml";

	dryrunRequest->resource = bstore->constructResourcePath(bucket, "");
}

bool isWriteRequest(std::string verb) {
	return verb == "POST" || verb == "PUT";
}

std::string parseBucketFromURI(std::string uri) {
	if (uri.size() <= 1 || uri[0] != '/') {
		// there is no bucket in the uri
		return "";
	}
	uri = uri.substr(1);
	size_t secondSlash = uri.find('/');
	if (secondSlash == std::string::npos) {
		return uri;
	}
	return uri.substr(0, secondSlash);
}

// Do a request, get a Response.
// Request content is provided as UnsentPacketQueue *pContent which will be depleted as bytes are sent but the queue
// itself must live for the life of this actor and be destroyed by the caller
															#line 5178 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via doRequest_impl()
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class DoRequest_implActor>
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DoRequest_implActorState {
															#line 5185 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DoRequest_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& verb,std::string const& resource,HTTP::Headers const& headers,UnsentPacketQueue* const& pContent,int const& contentLen,std::set<unsigned int> const& successCodes) 
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   verb(verb),
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resource(resource),
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   headers(headers),
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pContent(pContent),
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentLen(contentLen),
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   successCodes(successCodes),
															#line 1021 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentCopy(),
															#line 1022 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   dryrunContentCopy(),
															#line 1023 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   req(makeReference<HTTP::OutgoingRequest>()),
															#line 1024 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   dryrunRequest(makeReference<HTTP::OutgoingRequest>())
															#line 5212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("doRequest_impl", reinterpret_cast<unsigned long>(this));

	}
	~DoRequest_implActorState() 
	{
		fdb_probe_actor_destroy("doRequest_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->verb = verb;
															#line 1026 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.content = &contentCopy;
															#line 1027 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.contentLen = contentLen;
															#line 1029 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.headers = headers;
															#line 1030 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.headers["Host"] = bstore->host;
															#line 1031 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.headers["Accept"] = "application/xml";
															#line 1034 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (resource.empty())
															#line 5239 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1035 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				resource = "/";
															#line 5243 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1039 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for( const auto& [k, v] : bstore->extraHeaders ) {
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string& fieldValue = req->data.headers[k];
															#line 1041 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!fieldValue.empty())
															#line 5251 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1042 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					fieldValue.append(",");
															#line 5255 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1044 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				fieldValue.append(v);
															#line 5259 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1049 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			int bandwidthThisRequest = 1 + bstore->knobs.max_send_bytes_per_second / bstore->knobs.concurrent_uploads;
															#line 1050 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			int contentUploadSeconds = contentLen / bandwidthThisRequest;
															#line 1051 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			requestTimeout = std::max(bstore->knobs.request_timeout_min, 3 * contentUploadSeconds);
															#line 1053 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->concurrentRequests.take();
															#line 1053 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 1;
															#line 1053 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 0, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5276 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoRequest_implActorState();
		static_cast<DoRequest_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1054 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		globalReleaser = FlowLock::Releaser(bstore->concurrentRequests, 1);
															#line 1056 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxTries = std::min(bstore->knobs.request_tries, bstore->knobs.connect_tries);
															#line 1057 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		thisTry = 1;
															#line 1058 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		badRequestCode = 400;
															#line 1059 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		s3TokenError = false;
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		nextRetryDelay = 2.0;
															#line 1062 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 5311 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1054 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		globalReleaser = FlowLock::Releaser(bstore->concurrentRequests, 1);
															#line 1056 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxTries = std::min(bstore->knobs.request_tries, bstore->knobs.connect_tries);
															#line 1057 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		thisTry = 1;
															#line 1058 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		badRequestCode = 400;
															#line 1059 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		s3TokenError = false;
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		nextRetryDelay = 2.0;
															#line 1062 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 5332 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1063 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		err = Optional<Error>();
															#line 1064 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		remoteAddress = Optional<NetworkAddress>();
															#line 1065 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connectionEstablished = false;
															#line 1066 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = Reference<HTTP::IncomingResponse>();
															#line 1067 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		dryrunR = Reference<HTTP::IncomingResponse>();
															#line 1068 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		canonicalURI = resource;
															#line 1070 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		req->resource = resource;
															#line 1071 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connID = UID();
															#line 1072 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		reqStartTimer = double();
															#line 1073 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connectStartTimer = g_network->timer();
															#line 1074 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		reusingConn = false;
															#line 1075 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fastRetry = false;
															#line 1076 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		simulateS3TokenError = false;
															#line 5447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1080 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			Future<S3BlobStoreEndpoint::ReusableConnection> frconn = bstore->connect(&reusingConn);
															#line 1084 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.content->discardAll();
															#line 1085 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (pContent != nullptr)
															#line 5455 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1086 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				PacketBuffer* pFirst = pContent->getUnsent();
															#line 1087 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				PacketBuffer* pLast = nullptr;
															#line 1088 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(PacketBuffer* p = pFirst;p != nullptr;p = p->nextPacketBuffer()) {
															#line 1089 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					p->addref();
															#line 1091 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					p->bytes_sent = 0;
															#line 1092 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					pLast = p;
															#line 5469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1094 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				req->data.content->prependWriteBuffer(pFirst, pLast);
															#line 5473 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1098 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<S3BlobStoreEndpoint::ReusableConnection> __when_expr_1 = timeoutError(frconn, bstore->knobs.connect_timeout);
															#line 1098 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5479 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 2;
															#line 1098 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5484 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 1215 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		double end = g_network->timer();
															#line 1216 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		double connectDuration = reqStartTimer - connectStartTimer;
															#line 1217 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		double reqDuration = end - reqStartTimer;
															#line 1218 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		bstore->blobStats->requestLatency.addMeasurement(reqDuration);
															#line 1222 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!err.present() && successCodes.count(r->code) != 0)
															#line 5507 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1223 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevDebug, "S3BlobStoreDoRequestSuccessful") .detail("Verb", verb) .detail("Error", err.present()) .detail("ErrorString", err.present() ? err.get().name() : "") .detail("Resource", resource) .detail("ResponseCode", r->code) .detail("ResponseContentSize", r->data.content.size()) .log();
															#line 1231 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			bstore->s_stats.requests_successful++;
															#line 1232 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++bstore->blobStats->requestsSuccessful;
															#line 1233 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<DoRequest_implActor*>(this)->SAV<Reference<HTTP::IncomingResponse>>::futures) { (void)(r); this->~DoRequest_implActorState(); static_cast<DoRequest_implActor*>(this)->destroy(); return 0; }
															#line 5517 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<DoRequest_implActor*>(this)->SAV< Reference<HTTP::IncomingResponse> >::value()) Reference<HTTP::IncomingResponse>(std::move(r)); // state_var_RVO
			this->~DoRequest_implActorState();
			static_cast<DoRequest_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1237 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		bstore->s_stats.requests_failed++;
															#line 1238 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		++bstore->blobStats->requestsFailed;
															#line 1241 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		bool retryable = err.present() || r->code == 500 || r->code == 502 || r->code == 503 || r->code == 429;
															#line 1243 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		retryable = retryable && (thisTry < maxTries);
															#line 1245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!retryable || !err.present())
															#line 5533 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1246 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			fastRetry = false;
															#line 5537 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1249 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent event(SevWarn, retryable ? (fastRetry ? "S3BlobStoreEndpointRequestFailedFastRetryable" : "S3BlobStoreEndpointRequestFailedRetryable") : "S3BlobStoreEndpointRequestFailed");
															#line 1254 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		bool connectionFailed = false;
															#line 1256 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (err.present())
															#line 5545 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1257 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.errorUnsuppressed(err.get());
															#line 1258 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (err.get().code() == error_code_connection_failed)
															#line 5551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1259 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				connectionFailed = true;
															#line 5555 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 1262 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		event.suppressFor(60);
															#line 1264 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!err.present())
															#line 5562 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1265 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("ResponseCode", r->code);
															#line 1266 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string s3Error = parseErrorCodeFromS3(r->data.content);
															#line 1267 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("S3ErrorCode", s3Error);
															#line 1268 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (r->code == badRequestCode)
															#line 5572 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1269 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (isS3TokenError(s3Error) || simulateS3TokenError)
															#line 5576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1270 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					s3TokenError = true;
															#line 5580 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1272 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarnAlways, "S3BlobStoreBadRequest") .detail("HttpCode", r->code) .detail("HttpResponseContent", r->data.content) .detail("S3Error", s3Error) .log();
															#line 5584 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 1280 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		event.detail("ConnectionEstablished", connectionEstablished);
															#line 1281 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		event.detail("ReusingConn", reusingConn);
															#line 1282 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (connectionEstablished)
															#line 5593 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1283 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("ConnID", connID);
															#line 1284 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("ConnectDuration", connectDuration);
															#line 1285 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("ReqDuration", reqDuration);
															#line 5601 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1288 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (remoteAddress.present())
															#line 5605 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1289 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("RemoteEndpoint", remoteAddress.get());
															#line 5609 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1291 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("RemoteHost", bstore->host);
															#line 5615 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1293 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		event.detail("Verb", verb) .detail("Resource", resource) .detail("ThisTry", thisTry) .detail("URI", canonicalURI) .detail("Proxy", bstore->proxyHost.orDefault(""));
															#line 1301 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!fastRetry && (!r || r->code != 429))
															#line 5621 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1302 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++thisTry;
															#line 5625 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1304 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (fastRetry)
															#line 5629 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1305 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++bstore->blobStats->fastRetries;
															#line 1306 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_7 = delay(0);
															#line 1306 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5637 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_7.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 8;
															#line 1306 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 7, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1307 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (retryable || s3TokenError)
															#line 5649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1309 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				double delay = nextRetryDelay;
															#line 1312 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				double limit = connectionFailed ? bstore->knobs.max_delay_connection_failed : bstore->knobs.max_delay_retryable_error;
															#line 1315 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				nextRetryDelay = std::min(nextRetryDelay * 2, limit);
															#line 1317 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (r)
															#line 5659 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1318 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					auto iRetryAfter = r->data.headers.find("Retry-After");
															#line 1319 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (iRetryAfter != r->data.headers.end())
															#line 5665 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1320 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						event.detail("RetryAfterHeader", iRetryAfter->second);
															#line 1321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						char* pEnd;
															#line 1322 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						double retryAfter = strtod(iRetryAfter->second.c_str(), &pEnd);
															#line 1323 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (*pEnd)
															#line 5675 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1325 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							retryAfter = 300;
															#line 5679 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
															#line 1327 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						delay = std::max(delay, retryAfter);
															#line 5683 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
				}
															#line 1332 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				event.detail("RetryDelay", delay);
															#line 1333 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Void> __when_expr_8 = ::delay(delay);
															#line 1333 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5692 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when2(__when_expr_8.get(), loopDepth); };
				static_cast<DoRequest_implActor*>(this)->actor_wait_state = 9;
															#line 1333 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 8, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1338 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (r && r->code == 406)
															#line 5704 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1339 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1Catch1(http_not_accepted(), std::max(0, loopDepth - 1));
															#line 5708 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1341 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (r && r->code == 401)
															#line 5712 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1342 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1Catch1(http_auth_failed(), std::max(0, loopDepth - 1));
															#line 5716 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1345 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (err.present())
															#line 5720 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1346 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					int code = err.get().code();
															#line 1353 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (code == error_code_timed_out && !connectionEstablished)
															#line 5726 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1354 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						TraceEvent(SevWarn, "S3BlobStoreEndpointConnectTimeout") .suppressFor(60) .detail("Timeout", requestTimeout);
															#line 1357 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						return a_body1Catch1(connection_failed(), std::max(0, loopDepth - 1));
															#line 5732 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 1360 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (code == error_code_timed_out || code == error_code_connection_failed || code == error_code_lookup_failed)
															#line 5736 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1362 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						return a_body1Catch1(err.get(), std::max(0, loopDepth - 1));
															#line 5740 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
				}
															#line 1365 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(http_request_failed(), std::max(0, loopDepth - 1));
															#line 5745 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1205 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreDoRequestError") .errorUnsuppressed(e) .detail("Verb", verb) .detail("Resource", resource);
															#line 1209 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 5758 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1210 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 5762 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1212 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			err = e;
															#line 5766 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(int loopDepth) 
	{
															#line 1100 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connectionEstablished = true;
															#line 1101 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connID = rconn.conn->getDebugID();
															#line 1102 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		reqStartTimer = g_network->timer();
															#line 1103 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent(SevDebug, "S3BlobStoreEndpointConnected") .suppressFor(60) .detail("RemoteEndpoint", rconn.conn->getPeerAddress()) .detail("Reusing", reusingConn) .detail("ConnID", connID) .detail("Verb", req->verb) .detail("Resource", resource) .detail("Proxy", bstore->proxyHost.orDefault(""));
															#line 5787 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1113 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (s3TokenError && isWriteRequest(req->verb) && CLIENT_KNOBS->BACKUP_ALLOW_DRYRUN)
															#line 5791 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1117 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				bucket = parseBucketFromURI(resource);
															#line 1118 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (bucket.empty())
															#line 5797 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1119 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevError, "EmptyBucketRequest") .detail("S3TokenError", s3TokenError) .detail("Verb", req->verb) .detail("Resource", resource) .log();
															#line 1124 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1cont1loopBody1cont2Catch1(bucket_not_in_url(), loopDepth);
															#line 5803 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1126 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				dryrunRequest->data.content = &dryrunContentCopy;
															#line 1127 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				dryrunRequest->data.content->discardAll();
															#line 1128 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				populateDryrunRequest(dryrunRequest, bstore, bucket);
															#line 1129 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				setHeaders(bstore, dryrunRequest);
															#line 1130 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				dryrunRequest->resource = getCanonicalURI(bstore, dryrunRequest);
															#line 1131 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent("RetryS3RequestDueToTokenIssue") .detail("S3TokenError", s3TokenError) .detail("OriginalResource", resource) .detail("DryrunResource", dryrunRequest->resource) .detail("Bucket", bucket) .detail("V4", CLIENT_KNOBS->HTTP_REQUEST_AWS_V4_HEADER) .log();
															#line 1138 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Void> __when_expr_2 = bstore->requestRate->getAllowance(1);
															#line 1138 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5821 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<DoRequest_implActor*>(this)->actor_wait_state = 3;
															#line 1138 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 2, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5826 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1loopBody1cont6(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(S3BlobStoreEndpoint::ReusableConnection const& __rconn,int loopDepth) 
	{
															#line 1098 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rconn = __rconn;
															#line 5846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(S3BlobStoreEndpoint::ReusableConnection && __rconn,int loopDepth) 
	{
		rconn = std::move(__rconn);
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*,S3BlobStoreEndpoint::ReusableConnection const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*,S3BlobStoreEndpoint::ReusableConnection && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
															#line 1174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		setHeaders(bstore, req);
															#line 1175 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		req->resource = getCanonicalURI(bstore, req);
															#line 1177 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		remoteAddress = rconn.conn->getPeerAddress();
															#line 1178 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_5 = bstore->requestRate->getAllowance(1);
															#line 1178 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 6;
															#line 1178 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 5, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5938 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1172 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevError, "ErrorDuringRetryS3TokenIssue").errorUnsuppressed(e);
															#line 5948 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 1139 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		Future<Reference<HTTP::IncomingResponse>> dryrunResponse = HTTP::doRequest( rconn.conn, dryrunRequest, bstore->sendRate, &bstore->s_stats.bytes_sent, bstore->recvRate);
															#line 1141 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_3 = timeoutError(dryrunResponse, requestTimeout);
															#line 1141 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5973 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont7when1(__when_expr_3.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 4;
															#line 1141 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5978 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 1139 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		Future<Reference<HTTP::IncomingResponse>> dryrunResponse = HTTP::doRequest( rconn.conn, dryrunRequest, bstore->sendRate, &bstore->s_stats.bytes_sent, bstore->recvRate);
															#line 1141 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_3 = timeoutError(dryrunResponse, requestTimeout);
															#line 1141 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5991 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont7when1(__when_expr_3.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 4;
															#line 1141 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5996 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont9(Reference<HTTP::IncomingResponse> const& _dryrunR,int loopDepth) 
	{
															#line 1142 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		dryrunR = _dryrunR;
															#line 1143 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string s3Error = parseErrorCodeFromS3(dryrunR->data.content);
															#line 1144 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (dryrunR->code == badRequestCode && isS3TokenError(s3Error))
															#line 6084 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1148 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(bstore->knobs.max_delay_retryable_error);
															#line 1148 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6090 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont9when1(__when_expr_4.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 5;
															#line 1148 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 4, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 6095 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1149 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (dryrunR->code == 200 || dryrunR->code == 404)
															#line 6102 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1152 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent("S3TokenIssueResolved") .detail("HttpCode", dryrunR->code) .detail("HttpResponseContent", dryrunR->data.content) .detail("S3Error", s3Error) .detail("URI", dryrunRequest->resource) .log();
															#line 1158 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3TokenError = false;
															#line 6108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			else
			{
															#line 1160 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevError, "S3UnexpectedError") .detail("HttpCode", dryrunR->code) .detail("HttpResponseContent", dryrunR->data.content) .detail("S3Error", s3Error) .detail("URI", dryrunRequest->resource) .log();
															#line 1166 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1cont1loopBody1cont2Catch1(http_bad_response(), loopDepth);
															#line 6116 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1loopBody1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Reference<HTTP::IncomingResponse> && _dryrunR,int loopDepth) 
	{
															#line 1142 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		dryrunR = _dryrunR;
															#line 1143 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string s3Error = parseErrorCodeFromS3(dryrunR->data.content);
															#line 1144 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (dryrunR->code == badRequestCode && isS3TokenError(s3Error))
															#line 6131 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1148 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(bstore->knobs.max_delay_retryable_error);
															#line 1148 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6137 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont9when1(__when_expr_4.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 5;
															#line 1148 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 4, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 6142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1149 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (dryrunR->code == 200 || dryrunR->code == 404)
															#line 6149 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1152 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent("S3TokenIssueResolved") .detail("HttpCode", dryrunR->code) .detail("HttpResponseContent", dryrunR->data.content) .detail("S3Error", s3Error) .detail("URI", dryrunRequest->resource) .log();
															#line 1158 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3TokenError = false;
															#line 6155 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			else
			{
															#line 1160 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevError, "S3UnexpectedError") .detail("HttpCode", dryrunR->code) .detail("HttpResponseContent", dryrunR->data.content) .detail("S3Error", s3Error) .detail("URI", dryrunRequest->resource) .log();
															#line 1166 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1cont1loopBody1cont2Catch1(http_bad_response(), loopDepth);
															#line 6163 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1loopBody1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7when1(Reference<HTTP::IncomingResponse> const& _dryrunR,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(_dryrunR, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7when1(Reference<HTTP::IncomingResponse> && _dryrunR,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(std::move(_dryrunR), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont10(int loopDepth) 
	{
		return a_body1cont1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont15(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16(Void const& _,int loopDepth) 
	{
															#line 1180 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		reqF = HTTP::doRequest(rconn.conn, req, bstore->sendRate, &bstore->s_stats.bytes_sent, bstore->recvRate);
															#line 1184 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (reqF.isReady() && reusingConn)
															#line 6357 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1185 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			fastRetry = true;
															#line 6361 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1188 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_6 = timeoutError(reqF, requestTimeout);
															#line 1188 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6367 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1loopBody1cont16when1(__when_expr_6.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 7;
															#line 1188 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*>(static_cast<DoRequest_implActor*>(this)));
															#line 6372 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16(Void && _,int loopDepth) 
	{
															#line 1180 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		reqF = HTTP::doRequest(rconn.conn, req, bstore->sendRate, &bstore->s_stats.bytes_sent, bstore->recvRate);
															#line 1184 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (reqF.isReady() && reusingConn)
															#line 6383 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1185 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			fastRetry = true;
															#line 6387 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1188 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_6 = timeoutError(reqF, requestTimeout);
															#line 1188 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6393 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1loopBody1cont16when1(__when_expr_6.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 7;
															#line 1188 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*>(static_cast<DoRequest_implActor*>(this)));
															#line 6398 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont16cont1(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
															#line 1189 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (g_network->isSimulated() && BUGGIFY && deterministicRandom()->random01() < 0.1)
															#line 6482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1191 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			_r->code = badRequestCode;
															#line 1192 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			simulateS3TokenError = true;
															#line 6488 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1194 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = _r;
															#line 1198 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->data.headers["Connection"] != "close")
															#line 6494 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1199 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			bstore->returnConnection(rconn);
															#line 6498 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1201 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++bstore->blobStats->expiredConnections;
															#line 6504 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1203 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rconn.conn.clear();
															#line 6508 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont16cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16cont1(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
															#line 1189 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (g_network->isSimulated() && BUGGIFY && deterministicRandom()->random01() < 0.1)
															#line 6517 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1191 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			_r->code = badRequestCode;
															#line 1192 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			simulateS3TokenError = true;
															#line 6523 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1194 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = _r;
															#line 1198 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->data.headers["Connection"] != "close")
															#line 6529 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1199 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			bstore->returnConnection(rconn);
															#line 6533 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1201 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++bstore->blobStats->expiredConnections;
															#line 6539 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1203 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rconn.conn.clear();
															#line 6543 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont16cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16when1(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont16cont1(_r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16when1(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont16cont1(std::move(_r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont16when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont16when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1cont16cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont17(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont29(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont29(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont1loopBody1cont30(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont31(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont31(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont31(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont31(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);

	}
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string verb;
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue* pContent;
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int contentLen;
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::set<unsigned int> successCodes;
															#line 1021 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue contentCopy;
															#line 1022 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue dryrunContentCopy;
															#line 1023 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::OutgoingRequest> req;
															#line 1024 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::OutgoingRequest> dryrunRequest;
															#line 1051 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int requestTimeout;
															#line 1054 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser globalReleaser;
															#line 1056 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int maxTries;
															#line 1057 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int thisTry;
															#line 1058 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int badRequestCode;
															#line 1059 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool s3TokenError;
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double nextRetryDelay;
															#line 1063 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<Error> err;
															#line 1064 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<NetworkAddress> remoteAddress;
															#line 1065 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool connectionEstablished;
															#line 1066 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 1067 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::IncomingResponse> dryrunR;
															#line 1068 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string canonicalURI;
															#line 1071 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UID connID;
															#line 1072 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double reqStartTimer;
															#line 1073 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double connectStartTimer;
															#line 1074 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool reusingConn;
															#line 1075 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool fastRetry;
															#line 1076 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool simulateS3TokenError;
															#line 1098 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	S3BlobStoreEndpoint::ReusableConnection rconn;
															#line 1117 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1180 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Future<Reference<HTTP::IncomingResponse>> reqF;
															#line 6888 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via doRequest_impl()
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DoRequest_implActor final : public Actor<Reference<HTTP::IncomingResponse>>, public ActorCallback< DoRequest_implActor, 0, Void >, public ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >, public ActorCallback< DoRequest_implActor, 2, Void >, public ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >, public ActorCallback< DoRequest_implActor, 4, Void >, public ActorCallback< DoRequest_implActor, 5, Void >, public ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >, public ActorCallback< DoRequest_implActor, 7, Void >, public ActorCallback< DoRequest_implActor, 8, Void >, public FastAllocated<DoRequest_implActor>, public DoRequest_implActorState<DoRequest_implActor> {
															#line 6893 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<DoRequest_implActor>::operator new;
	using FastAllocated<DoRequest_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1696277915996261888UL, 15146383275710662656UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<HTTP::IncomingResponse>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoRequest_implActor, 0, Void >;
friend struct ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >;
friend struct ActorCallback< DoRequest_implActor, 2, Void >;
friend struct ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< DoRequest_implActor, 4, Void >;
friend struct ActorCallback< DoRequest_implActor, 5, Void >;
friend struct ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< DoRequest_implActor, 7, Void >;
friend struct ActorCallback< DoRequest_implActor, 8, Void >;
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DoRequest_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& verb,std::string const& resource,HTTP::Headers const& headers,UnsentPacketQueue* const& pContent,int const& contentLen,std::set<unsigned int> const& successCodes) 
															#line 6918 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Reference<HTTP::IncomingResponse>>(),
		   DoRequest_implActorState<DoRequest_implActor>(bstore, verb, resource, headers, pContent, contentLen, successCodes),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8762720438240590848UL, 16675552070853705472UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doRequest_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoRequest_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoRequest_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< DoRequest_implActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< DoRequest_implActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< DoRequest_implActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< DoRequest_implActor, 8, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Reference<HTTP::IncomingResponse>> doRequest_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& verb, std::string const& resource, HTTP::Headers const& headers, UnsentPacketQueue* const& pContent, int const& contentLen, std::set<unsigned int> const& successCodes ) {
															#line 1013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Reference<HTTP::IncomingResponse>>(new DoRequest_implActor(bstore, verb, resource, headers, pContent, contentLen, successCodes));
															#line 6959 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1369 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Reference<HTTP::IncomingResponse>> S3BlobStoreEndpoint::doRequest(std::string const& verb,
                                                                         std::string const& resource,
                                                                         const HTTP::Headers& headers,
                                                                         UnsentPacketQueue* pContent,
                                                                         int contentLen,
                                                                         std::set<unsigned int> successCodes) {
	return doRequest_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), verb, resource, headers, pContent, contentLen, successCodes);
}

															#line 6974 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via listObjectsStream_impl()
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ListObjectsStream_implActor>
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListObjectsStream_implActorState {
															#line 6981 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListObjectsStream_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,PromiseStream<S3BlobStoreEndpoint::ListResult> const& results,Optional<std::string> const& prefix,Optional<char> const& delimiter,int const& maxDepth,std::function<bool(std::string const&)> const& recurseFilter) 
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   results(results),
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   prefix(prefix),
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   delimiter(delimiter),
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   maxDepth(maxDepth),
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   recurseFilter(recurseFilter),
															#line 1387 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resource(bstore->constructResourcePath(bucket, ""))
															#line 7002 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("listObjectsStream_impl", reinterpret_cast<unsigned long>(this));

	}
	~ListObjectsStream_implActorState() 
	{
		fdb_probe_actor_destroy("listObjectsStream_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1391 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (resource.empty())
															#line 7017 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1392 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				resource = "/";
															#line 7021 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1394 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			resource.append("?list-type=2&max-keys=").append(std::to_string(CLIENT_KNOBS->BLOBSTORE_LIST_MAX_KEYS_PER_PAGE));
															#line 1396 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (prefix.present())
															#line 7027 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1397 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				resource.append("&prefix=").append(prefix.get());
															#line 7031 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1398 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (delimiter.present())
															#line 7035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1399 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				resource.append("&delimiter=").append(std::string(1, delimiter.get()));
															#line 7039 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1401 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			continuationToken = std::string();
															#line 1402 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			more = true;
															#line 1403 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			subLists = std::vector<Future<Void>>();
															#line 1405 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			;
															#line 7049 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ListObjectsStream_implActorState();
		static_cast<ListObjectsStream_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1510 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_2 = waitForAll(subLists);
															#line 1510 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7074 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 3;
															#line 1510 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ListObjectsStream_implActor, 2, Void >*>(static_cast<ListObjectsStream_implActor*>(this)));
															#line 7079 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1405 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!(more))
															#line 7095 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1406 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_0 = bstore->concurrentLists.take();
															#line 1406 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 1;
															#line 1406 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ListObjectsStream_implActor, 0, Void >*>(static_cast<ListObjectsStream_implActor*>(this)));
															#line 7108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1407 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser = FlowLock::Releaser(bstore->concurrentLists, 1);
															#line 1409 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1410 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fullResource = resource;
															#line 1411 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!continuationToken.empty())
															#line 7136 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1412 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			fullResource.append("&continuation-token=").append(continuationToken);
															#line 7140 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1415 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", fullResource, headers, nullptr, 0, { 200, 404 });
															#line 1415 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7146 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 2;
															#line 1415 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ListObjectsStream_implActor*>(this)));
															#line 7151 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1407 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser = FlowLock::Releaser(bstore->concurrentLists, 1);
															#line 1409 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1410 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fullResource = resource;
															#line 1411 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!continuationToken.empty())
															#line 7166 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1412 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			fullResource.append("&continuation-token=").append(continuationToken);
															#line 7170 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1415 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", fullResource, headers, nullptr, 0, { 200, 404 });
															#line 1415 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 2;
															#line 1415 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ListObjectsStream_implActor*>(this)));
															#line 7181 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state > 0) static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListObjectsStream_implActor*>(this)->ActorCallback< ListObjectsStream_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ListObjectsStream_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 1417 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser.release();
															#line 7265 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1420 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::ListResult listResult;
															#line 1423 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (r->code == 404)
															#line 7271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevError, "S3BlobStoreResourceNotFound") .detail("Bucket", bucket) .detail("Prefix", prefix.present() ? prefix.get() : "") .detail("Resource", fullResource);
															#line 1428 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(resource_not_found(), loopDepth);
															#line 7277 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1431 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 1434 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 1435 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 1438 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 1439 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result == nullptr || strcmp(result->name(), "ListBucketResult") != 0)
															#line 7289 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1440 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7293 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1443 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* n = result->first_node();
															#line 1444 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			more = false;
															#line 1445 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			continuationToken.clear();
															#line 1447 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;n != nullptr;) {
															#line 1448 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				const char* name = n->name();
															#line 1449 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (strcmp(name, "IsTruncated") == 0)
															#line 7307 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1450 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					const char* val = n->value();
															#line 1451 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(val, "true") == 0)
															#line 7313 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						more = true;
															#line 7317 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
					else
					{
															#line 1453 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (strcmp(val, "false") == 0)
															#line 7323 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1454 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							more = false;
															#line 7327 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
															#line 1456 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7333 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
				}
				else
				{
															#line 1458 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(name, "NextContinuationToken") == 0)
															#line 7341 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1459 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (n->value() != nullptr)
															#line 7345 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1460 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							continuationToken = n->value();
															#line 7349 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
					else
					{
															#line 1462 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (strcmp(name, "Contents") == 0)
															#line 7356 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1463 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							S3BlobStoreEndpoint::ObjectInfo object;
															#line 1465 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							xml_node<>* key = n->first_node("Key");
															#line 1466 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (key == nullptr)
															#line 7364 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 1467 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							}
															#line 1469 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							object.name = key->value();
															#line 1471 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							xml_node<>* size = n->first_node("Size");
															#line 1472 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (size == nullptr)
															#line 7376 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 1473 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7380 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							}
															#line 1475 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							object.size = strtoull(size->value(), nullptr, 10);
															#line 1477 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							listResult.objects.push_back(object);
															#line 7386 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
															#line 1478 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (strcmp(name, "CommonPrefixes") == 0)
															#line 7392 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 1479 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								xml_node<>* prefixNode = n->first_node("Prefix");
															#line 1480 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								for(;prefixNode != nullptr;) {
															#line 1481 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									const char* prefix = prefixNode->value();
															#line 1483 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									if (maxDepth > 0)
															#line 7402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
									{
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
										if (!recurseFilter || recurseFilter(prefix))
															#line 7406 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
										{
															#line 1486 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
											subLists.push_back(bstore->listObjectsStream( bucket, results, prefix, Optional<char>(), maxDepth - 1, recurseFilter));
															#line 7410 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
										}
									}
									else
									{
															#line 1490 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
										listResult.commonPrefixes.push_back(prefix);
															#line 7417 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
									}
															#line 1492 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									prefixNode = prefixNode->next_sibling("Prefix");
															#line 7421 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
								}
							}
						}
					}
				}
															#line 1495 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				n = n->next_sibling();
															#line 7429 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1498 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			results.send(listResult);
															#line 7433 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1loopBody1cont28(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 1417 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser.release();
															#line 7448 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1420 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::ListResult listResult;
															#line 1423 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (r->code == 404)
															#line 7454 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevError, "S3BlobStoreResourceNotFound") .detail("Bucket", bucket) .detail("Prefix", prefix.present() ? prefix.get() : "") .detail("Resource", fullResource);
															#line 1428 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(resource_not_found(), loopDepth);
															#line 7460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1431 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 1434 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 1435 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 1438 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 1439 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result == nullptr || strcmp(result->name(), "ListBucketResult") != 0)
															#line 7472 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1440 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7476 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1443 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* n = result->first_node();
															#line 1444 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			more = false;
															#line 1445 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			continuationToken.clear();
															#line 1447 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;n != nullptr;) {
															#line 1448 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				const char* name = n->name();
															#line 1449 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (strcmp(name, "IsTruncated") == 0)
															#line 7490 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1450 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					const char* val = n->value();
															#line 1451 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(val, "true") == 0)
															#line 7496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						more = true;
															#line 7500 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
					else
					{
															#line 1453 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (strcmp(val, "false") == 0)
															#line 7506 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1454 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							more = false;
															#line 7510 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
															#line 1456 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7516 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
				}
				else
				{
															#line 1458 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(name, "NextContinuationToken") == 0)
															#line 7524 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1459 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (n->value() != nullptr)
															#line 7528 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1460 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							continuationToken = n->value();
															#line 7532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
					else
					{
															#line 1462 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (strcmp(name, "Contents") == 0)
															#line 7539 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1463 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							S3BlobStoreEndpoint::ObjectInfo object;
															#line 1465 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							xml_node<>* key = n->first_node("Key");
															#line 1466 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (key == nullptr)
															#line 7547 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 1467 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							}
															#line 1469 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							object.name = key->value();
															#line 1471 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							xml_node<>* size = n->first_node("Size");
															#line 1472 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (size == nullptr)
															#line 7559 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 1473 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7563 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							}
															#line 1475 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							object.size = strtoull(size->value(), nullptr, 10);
															#line 1477 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							listResult.objects.push_back(object);
															#line 7569 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
															#line 1478 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (strcmp(name, "CommonPrefixes") == 0)
															#line 7575 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 1479 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								xml_node<>* prefixNode = n->first_node("Prefix");
															#line 1480 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								for(;prefixNode != nullptr;) {
															#line 1481 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									const char* prefix = prefixNode->value();
															#line 1483 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									if (maxDepth > 0)
															#line 7585 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
									{
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
										if (!recurseFilter || recurseFilter(prefix))
															#line 7589 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
										{
															#line 1486 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
											subLists.push_back(bstore->listObjectsStream( bucket, results, prefix, Optional<char>(), maxDepth - 1, recurseFilter));
															#line 7593 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
										}
									}
									else
									{
															#line 1490 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
										listResult.commonPrefixes.push_back(prefix);
															#line 7600 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
									}
															#line 1492 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									prefixNode = prefixNode->next_sibling("Prefix");
															#line 7604 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
								}
							}
						}
					}
				}
															#line 1495 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				n = n->next_sibling();
															#line 7612 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1498 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			results.send(listResult);
															#line 7616 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1loopBody1cont28(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(r, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state > 0) static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListObjectsStream_implActor*>(this)->ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1500 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 7713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1501 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, "S3BlobStoreEndpointListResultParseError") .errorUnsuppressed(e) .suppressFor(60) .detail("Resource", fullResource);
															#line 7717 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1506 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 7721 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont28(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1511 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ListObjectsStream_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ListObjectsStream_implActorState(); static_cast<ListObjectsStream_implActor*>(this)->destroy(); return 0; }
															#line 7748 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ListObjectsStream_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ListObjectsStream_implActorState();
		static_cast<ListObjectsStream_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1511 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ListObjectsStream_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ListObjectsStream_implActorState(); static_cast<ListObjectsStream_implActor*>(this)->destroy(); return 0; }
															#line 7760 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ListObjectsStream_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ListObjectsStream_implActorState();
		static_cast<ListObjectsStream_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state > 0) static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListObjectsStream_implActor*>(this)->ActorCallback< ListObjectsStream_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ListObjectsStream_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PromiseStream<S3BlobStoreEndpoint::ListResult> results;
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<std::string> prefix;
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<char> delimiter;
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int maxDepth;
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::function<bool(std::string const&)> recurseFilter;
															#line 1387 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1401 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string continuationToken;
															#line 1402 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool more;
															#line 1403 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::vector<Future<Void>> subLists;
															#line 1407 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser listReleaser;
															#line 1409 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1410 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string fullResource;
															#line 7871 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via listObjectsStream_impl()
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListObjectsStream_implActor final : public Actor<Void>, public ActorCallback< ListObjectsStream_implActor, 0, Void >, public ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >, public ActorCallback< ListObjectsStream_implActor, 2, Void >, public FastAllocated<ListObjectsStream_implActor>, public ListObjectsStream_implActorState<ListObjectsStream_implActor> {
															#line 7876 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ListObjectsStream_implActor>::operator new;
	using FastAllocated<ListObjectsStream_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1764520531503481600UL, 344841072473491712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ListObjectsStream_implActor, 0, Void >;
friend struct ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< ListObjectsStream_implActor, 2, Void >;
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListObjectsStream_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,PromiseStream<S3BlobStoreEndpoint::ListResult> const& results,Optional<std::string> const& prefix,Optional<char> const& delimiter,int const& maxDepth,std::function<bool(std::string const&)> const& recurseFilter) 
															#line 7895 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   ListObjectsStream_implActorState<ListObjectsStream_implActor>(bstore, bucket, results, prefix, delimiter, maxDepth, recurseFilter),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2139282905966115328UL, 13274857023515672576UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("listObjectsStream_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ListObjectsStream_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ListObjectsStream_implActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> listObjectsStream_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, PromiseStream<S3BlobStoreEndpoint::ListResult> const& results, Optional<std::string> const& prefix, Optional<char> const& delimiter, int const& maxDepth, std::function<bool(std::string const&)> const& recurseFilter ) {
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new ListObjectsStream_implActor(bstore, bucket, results, prefix, delimiter, maxDepth, recurseFilter));
															#line 7930 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1513 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::listObjectsStream(std::string const& bucket,
                                                    PromiseStream<ListResult> results,
                                                    Optional<std::string> prefix,
                                                    Optional<char> delimiter,
                                                    int maxDepth,
                                                    std::function<bool(std::string const&)> recurseFilter) {
	return listObjectsStream_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), bucket, results, prefix, delimiter, maxDepth, recurseFilter);
}

															#line 7945 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via listObjects_impl()
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ListObjects_implActor>
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListObjects_implActorState {
															#line 7952 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListObjects_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,Optional<std::string> const& prefix,Optional<char> const& delimiter,int const& maxDepth,std::function<bool(std::string const&)> const& recurseFilter) 
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   prefix(prefix),
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   delimiter(delimiter),
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   maxDepth(maxDepth),
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   recurseFilter(recurseFilter),
															#line 1530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   results(),
															#line 1531 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resultStream(),
															#line 1532 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   done(bstore->listObjectsStream(bucket, resultStream, prefix, delimiter, maxDepth, recurseFilter))
															#line 7975 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("listObjects_impl", reinterpret_cast<unsigned long>(this));

	}
	~ListObjects_implActorState() 
	{
		fdb_probe_actor_destroy("listObjects_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1536 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			done = map(done, [=](Void) mutable { resultStream.sendError(end_of_stream()); return Void(); });
															#line 7990 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			try {
															#line 1542 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				;
															#line 7994 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ListObjects_implActorState();
		static_cast<ListObjects_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1561 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ListObjects_implActor*>(this)->SAV<S3BlobStoreEndpoint::ListResult>::futures) { (void)(results); this->~ListObjects_implActorState(); static_cast<ListObjects_implActor*>(this)->destroy(); return 0; }
															#line 8023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ListObjects_implActor*>(this)->SAV< S3BlobStoreEndpoint::ListResult >::value()) S3BlobStoreEndpoint::ListResult(std::move(results)); // state_var_RVO
		this->~ListObjects_implActorState();
		static_cast<ListObjects_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1557 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_end_of_stream)
															#line 8036 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1558 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 8040 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1545 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_0 = done;
															#line 1543 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjects_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 1549 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		auto __when_expr_1 = resultStream.getFuture();
															#line 8069 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<S3BlobStoreEndpoint::ListResult>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<S3BlobStoreEndpoint::ListResult>>::value, "invalid type");
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.pop(), loopDepth); };
		static_cast<ListObjects_implActor*>(this)->actor_wait_state = 1;
															#line 1545 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ListObjects_implActor, 0, Void >*>(static_cast<ListObjects_implActor*>(this)));
															#line 1549 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >*>(static_cast<ListObjects_implActor*>(this)));
															#line 8077 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 1546 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		done = Never();
															#line 8092 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 1546 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		done = Never();
															#line 8101 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(S3BlobStoreEndpoint::ListResult const& info,int loopDepth) 
	{
															#line 1550 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		results.commonPrefixes.insert( results.commonPrefixes.end(), info.commonPrefixes.begin(), info.commonPrefixes.end());
															#line 1552 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		results.objects.insert(results.objects.end(), info.objects.begin(), info.objects.end());
															#line 8112 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(S3BlobStoreEndpoint::ListResult && info,int loopDepth) 
	{
															#line 1550 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		results.commonPrefixes.insert( results.commonPrefixes.end(), info.commonPrefixes.begin(), info.commonPrefixes.end());
															#line 1552 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		results.objects.insert(results.objects.end(), info.objects.begin(), info.objects.end());
															#line 8123 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ListObjects_implActor*>(this)->actor_wait_state > 0) static_cast<ListObjects_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListObjects_implActor*>(this)->ActorCallback< ListObjects_implActor, 0, Void >::remove();
		static_cast<ListObjects_implActor*>(this)->ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >::remove();

	}
	void a_callback_fire(ActorCallback< ListObjects_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ListObjects_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ListObjects_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >*,S3BlobStoreEndpoint::ListResult const& value) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >*,S3BlobStoreEndpoint::ListResult && value) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >*,Error err) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<std::string> prefix;
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<char> delimiter;
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int maxDepth;
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::function<bool(std::string const&)> recurseFilter;
															#line 1530 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	S3BlobStoreEndpoint::ListResult results;
															#line 1531 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PromiseStream<S3BlobStoreEndpoint::ListResult> resultStream;
															#line 1532 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Future<Void> done;
															#line 8267 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via listObjects_impl()
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListObjects_implActor final : public Actor<S3BlobStoreEndpoint::ListResult>, public ActorCallback< ListObjects_implActor, 0, Void >, public ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >, public FastAllocated<ListObjects_implActor>, public ListObjects_implActorState<ListObjects_implActor> {
															#line 8272 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ListObjects_implActor>::operator new;
	using FastAllocated<ListObjects_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1474432368146315264UL, 15095902967842709760UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<S3BlobStoreEndpoint::ListResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ListObjects_implActor, 0, Void >;
friend struct ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >;
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListObjects_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,Optional<std::string> const& prefix,Optional<char> const& delimiter,int const& maxDepth,std::function<bool(std::string const&)> const& recurseFilter) 
															#line 8290 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<S3BlobStoreEndpoint::ListResult>(),
		   ListObjects_implActorState<ListObjects_implActor>(bstore, bucket, prefix, delimiter, maxDepth, recurseFilter),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1153006404795665152UL, 10728906690639680256UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("listObjects_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ListObjects_implActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<S3BlobStoreEndpoint::ListResult> listObjects_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, Optional<std::string> const& prefix, Optional<char> const& delimiter, int const& maxDepth, std::function<bool(std::string const&)> const& recurseFilter ) {
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<S3BlobStoreEndpoint::ListResult>(new ListObjects_implActor(bstore, bucket, prefix, delimiter, maxDepth, recurseFilter));
															#line 8323 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1563 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<S3BlobStoreEndpoint::ListResult> S3BlobStoreEndpoint::listObjects(
    std::string const& bucket,
    Optional<std::string> prefix,
    Optional<char> delimiter,
    int maxDepth,
    std::function<bool(std::string const&)> recurseFilter) {
	return listObjects_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), bucket, prefix, delimiter, maxDepth, recurseFilter);
}

															#line 8338 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via listBuckets_impl()
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ListBuckets_implActor>
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListBuckets_implActorState {
															#line 8345 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListBuckets_implActorState(Reference<S3BlobStoreEndpoint> const& bstore) 
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1575 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resource("/?marker="),
															#line 1576 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   lastName(),
															#line 1577 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   more(true),
															#line 1578 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   buckets()
															#line 8360 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("listBuckets_impl", reinterpret_cast<unsigned long>(this));

	}
	~ListBuckets_implActorState() 
	{
		fdb_probe_actor_destroy("listBuckets_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1580 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			;
															#line 8375 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ListBuckets_implActorState();
		static_cast<ListBuckets_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1638 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ListBuckets_implActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(buckets); this->~ListBuckets_implActorState(); static_cast<ListBuckets_implActor*>(this)->destroy(); return 0; }
															#line 8398 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ListBuckets_implActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(std::move(buckets)); // state_var_RVO
		this->~ListBuckets_implActorState();
		static_cast<ListBuckets_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1580 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!(more))
															#line 8417 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1581 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_0 = bstore->concurrentLists.take();
															#line 1581 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8425 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 1;
															#line 1581 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ListBuckets_implActor, 0, Void >*>(static_cast<ListBuckets_implActor*>(this)));
															#line 8430 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1582 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser = FlowLock::Releaser(bstore->concurrentLists, 1);
															#line 1584 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1585 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fullResource = resource + lastName;
															#line 1586 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", fullResource, headers, nullptr, 0, { 200 });
															#line 1586 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 2;
															#line 1586 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ListBuckets_implActor*>(this)));
															#line 8465 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1582 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser = FlowLock::Releaser(bstore->concurrentLists, 1);
															#line 1584 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1585 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fullResource = resource + lastName;
															#line 1586 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", fullResource, headers, nullptr, 0, { 200 });
															#line 1586 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 2;
															#line 1586 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ListBuckets_implActor*>(this)));
															#line 8487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state > 0) static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListBuckets_implActor*>(this)->ActorCallback< ListBuckets_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ListBuckets_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ListBuckets_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ListBuckets_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 1588 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser.release();
															#line 8571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1591 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 1594 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 1595 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 1598 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 1599 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result == nullptr || strcmp(result->name(), "ListAllMyBucketsResult") != 0)
															#line 8583 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1600 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 8587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1603 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			more = false;
															#line 1604 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* truncated = result->first_node("IsTruncated");
															#line 1605 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (truncated != nullptr && strcmp(truncated->value(), "true") == 0)
															#line 8595 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1606 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				more = true;
															#line 8599 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1609 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* bucketsNode = result->first_node("Buckets");
															#line 1610 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (bucketsNode != nullptr)
															#line 8605 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1611 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				xml_node<>* bucketNode = bucketsNode->first_node("Bucket");
															#line 1612 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(;bucketNode != nullptr;) {
															#line 1613 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					xml_node<>* nameNode = bucketNode->first_node("Name");
															#line 1614 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (nameNode == nullptr)
															#line 8615 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1615 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 8619 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 1617 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					const char* name = nameNode->value();
															#line 1618 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					buckets.push_back(name);
															#line 1620 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					bucketNode = bucketNode->next_sibling("Bucket");
															#line 8627 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
															#line 1624 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (more)
															#line 8632 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1625 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				lastName = buckets.back();
															#line 8636 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1loopBody1cont12(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 1588 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser.release();
															#line 8652 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1591 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 1594 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 1595 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 1598 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 1599 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result == nullptr || strcmp(result->name(), "ListAllMyBucketsResult") != 0)
															#line 8664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1600 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 8668 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1603 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			more = false;
															#line 1604 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* truncated = result->first_node("IsTruncated");
															#line 1605 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (truncated != nullptr && strcmp(truncated->value(), "true") == 0)
															#line 8676 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1606 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				more = true;
															#line 8680 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1609 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* bucketsNode = result->first_node("Buckets");
															#line 1610 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (bucketsNode != nullptr)
															#line 8686 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1611 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				xml_node<>* bucketNode = bucketsNode->first_node("Bucket");
															#line 1612 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(;bucketNode != nullptr;) {
															#line 1613 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					xml_node<>* nameNode = bucketNode->first_node("Name");
															#line 1614 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (nameNode == nullptr)
															#line 8696 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1615 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 8700 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 1617 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					const char* name = nameNode->value();
															#line 1618 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					buckets.push_back(name);
															#line 1620 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					bucketNode = bucketNode->next_sibling("Bucket");
															#line 8708 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
															#line 1624 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (more)
															#line 8713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1625 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				lastName = buckets.back();
															#line 8717 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1loopBody1cont12(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(r, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state > 0) static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListBuckets_implActor*>(this)->ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1629 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 8815 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1630 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, "S3BlobStoreEndpointListBucketResultParseError") .errorUnsuppressed(e) .suppressFor(60) .detail("Resource", fullResource);
															#line 8819 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1634 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 8823 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1575 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1576 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string lastName;
															#line 1577 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool more;
															#line 1578 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::vector<std::string> buckets;
															#line 1582 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser listReleaser;
															#line 1584 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1585 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string fullResource;
															#line 8862 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via listBuckets_impl()
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListBuckets_implActor final : public Actor<std::vector<std::string>>, public ActorCallback< ListBuckets_implActor, 0, Void >, public ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ListBuckets_implActor>, public ListBuckets_implActorState<ListBuckets_implActor> {
															#line 8867 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ListBuckets_implActor>::operator new;
	using FastAllocated<ListBuckets_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3977925312582570752UL, 1143124160313779968UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ListBuckets_implActor, 0, Void >;
friend struct ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListBuckets_implActor(Reference<S3BlobStoreEndpoint> const& bstore) 
															#line 8885 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   ListBuckets_implActorState<ListBuckets_implActor>(bstore),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5964580658728215808UL, 13914731692276139264UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("listBuckets_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ListBuckets_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> listBuckets_impl( Reference<S3BlobStoreEndpoint> const& bstore ) {
															#line 1574 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::vector<std::string>>(new ListBuckets_implActor(bstore));
															#line 8919 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1640 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::vector<std::string>> S3BlobStoreEndpoint::listBuckets() {
	return listBuckets_impl(Reference<S3BlobStoreEndpoint>::addRef(this));
}

std::string S3BlobStoreEndpoint::hmac_sha1(Credentials const& creds, std::string const& msg) {
	std::string key = creds.secret;

	// Hash key to shorten it if it is longer than SHA1 block size
	if (key.size() > 64) {
		key = SHA1::from_string(key);
	}

	// Pad key up to SHA1 block size if needed
	key.append(64 - key.size(), '\0');

	std::string kipad = key;
	for (int i = 0; i < 64; ++i)
		kipad[i] ^= '\x36';

	std::string kopad = key;
	for (int i = 0; i < 64; ++i)
		kopad[i] ^= '\x5c';

	kipad.append(msg);
	std::string hkipad = SHA1::from_string(kipad);
	kopad.append(hkipad);
	return SHA1::from_string(kopad);
}

static void sha256(const unsigned char* data, const size_t len, unsigned char* hash) {
	SHA256_CTX sha256;
	SHA256_Init(&sha256);
	SHA256_Update(&sha256, data, len);
	SHA256_Final(hash, &sha256);
}

std::string sha256_hex(std::string str) {
	unsigned char hash[SHA256_DIGEST_LENGTH];
	sha256((const unsigned char*)str.c_str(), str.size(), hash);
	std::stringstream ss;
	for (int i = 0; i < SHA256_DIGEST_LENGTH; i++) {
		ss << std::hex << std::setw(2) << std::setfill('0') << (int)hash[i];
	}
	return ss.str();
}

// Return base64'd SHA256 hash of input string.
std::string sha256_base64(std::string str) {
	unsigned char hash[SHA256_DIGEST_LENGTH];
	sha256((const unsigned char*)str.c_str(), str.size(), hash);
	std::string hashAsStr = std::string((char*)hash, SHA256_DIGEST_LENGTH);
	std::string sig = base64::encoder::from_string(hashAsStr);
	// base64 encoded blocks end in \n so remove last character.
	sig.resize(sig.size() - 1);
	return sig;
}

std::string hmac_sha256_hex(std::string key, std::string msg) {
	unsigned char hash[32];

	HMAC_CTX* hmac = HMAC_CTX_new();
	HMAC_Init_ex(hmac, &key[0], key.length(), EVP_sha256(), NULL);
	HMAC_Update(hmac, (unsigned char*)&msg[0], msg.length());
	unsigned int len = 32;
	HMAC_Final(hmac, hash, &len);
	HMAC_CTX_free(hmac);

	std::stringstream ss;
	ss << std::hex << std::setfill('0');
	for (int i = 0; i < len; i++) {
		ss << std::hex << std::setw(2) << (unsigned int)hash[i];
	}
	return (ss.str());
}

std::string hmac_sha256(std::string key, std::string msg) {
	unsigned char hash[32];

	HMAC_CTX* hmac = HMAC_CTX_new();
	HMAC_Init_ex(hmac, &key[0], key.length(), EVP_sha256(), NULL);
	HMAC_Update(hmac, (unsigned char*)&msg[0], msg.length());
	unsigned int len = 32;
	HMAC_Final(hmac, hash, &len);
	HMAC_CTX_free(hmac);

	std::stringstream ss;
	ss << std::setfill('0');
	for (int i = 0; i < len; i++) {
		ss << hash[i];
	}
	return (ss.str());
}

// Date and Time parameters are used for unit testing
void S3BlobStoreEndpoint::setV4AuthHeaders(std::string const& verb,
                                           std::string const& resource,
                                           HTTP::Headers& headers,
                                           std::string date,
                                           std::string datestamp) {
	if (!credentials.present()) {
		return;
	}
	Credentials creds = credentials.get();
	// std::cout << "========== Starting===========" << std::endl;
	std::string accessKey = creds.key;
	std::string secretKey = creds.secret;
	// Create a date for headers and the credential string
	std::string amzDate;
	std::string dateStamp;
	if (date.empty() || datestamp.empty()) {
		time_t ts;
		time(&ts);
		char dateBuf[20];
		// ISO 8601 format YYYYMMDD'T'HHMMSS'Z'
		strftime(dateBuf, 20, "%Y%m%dT%H%M%SZ", gmtime(&ts));
		amzDate = dateBuf;
		strftime(dateBuf, 20, "%Y%m%d", gmtime(&ts));
		dateStamp = dateBuf;
	} else {
		amzDate = date;
		dateStamp = datestamp;
	}

	// ************* TASK 1: CREATE A CANONICAL REQUEST *************
	// Create Create canonical URI--the part of the URI from domain to query string (use '/' if no path)
	std::vector<std::string> queryParameters;
	std::string canonicalURI = awsCanonicalURI(resource, queryParameters, true);

	std::string canonicalQueryString;
	if (!queryParameters.empty()) {
		std::sort(queryParameters.begin(), queryParameters.end());
		canonicalQueryString = boost::algorithm::join(queryParameters, "&");
	}

	using namespace boost::algorithm;
	// Create the canonical headers and signed headers
	ASSERT(!headers["Host"].empty());
	// Be careful. There is x-amz-content-sha256 for auth and then
	// x-amz-checksum-sha256 for object integrity check.
	headers["x-amz-content-sha256"] = "UNSIGNED-PAYLOAD";
	headers["x-amz-date"] = amzDate;
	std::vector<std::pair<std::string, std::string>> headersList;
	headersList.push_back({ "host", trim_copy(headers["Host"]) + "\n" });
	if (headers.find("Content-Type") != headers.end())
		headersList.push_back({ "content-type", trim_copy(headers["Content-Type"]) + "\n" });
	if (headers.find("Content-MD5") != headers.end())
		headersList.push_back({ "content-md5", trim_copy(headers["Content-MD5"]) + "\n" });
	for (auto h : headers) {
		if (StringRef(h.first).startsWith("x-amz"_sr))
			headersList.push_back({ to_lower_copy(h.first), trim_copy(h.second) + "\n" });
	}
	std::sort(headersList.begin(), headersList.end());
	std::string canonicalHeaders;
	std::string signedHeaders;
	for (auto& i : headersList) {
		canonicalHeaders += i.first + ":" + i.second;
		signedHeaders += i.first + ";";
	}
	signedHeaders.pop_back();
	std::string canonicalRequest = verb + "\n" + canonicalURI + "\n" + canonicalQueryString + "\n" + canonicalHeaders +
	                               "\n" + signedHeaders + "\n" + headers["x-amz-content-sha256"];

	// ************* TASK 2: CREATE THE STRING TO SIGN*************
	std::string algorithm = "AWS4-HMAC-SHA256";
	std::string credentialScope = dateStamp + "/" + region + "/s3/" + "aws4_request";
	std::string stringToSign =
	    algorithm + "\n" + amzDate + "\n" + credentialScope + "\n" + sha256_hex(canonicalRequest);

	// ************* TASK 3: CALCULATE THE SIGNATURE *************
	// Create the signing key using the function defined above.
	std::string signingKey =
	    hmac_sha256(hmac_sha256(hmac_sha256(hmac_sha256("AWS4" + secretKey, dateStamp), region), "s3"), "aws4_request");
	// Sign the string_to_sign using the signing_key
	std::string signature = hmac_sha256_hex(signingKey, stringToSign);
	// ************* TASK 4: ADD SIGNING INFORMATION TO THE Header *************
	std::string authorizationHeader = algorithm + " " + "Credential=" + accessKey + "/" + credentialScope + ", " +
	                                  "SignedHeaders=" + signedHeaders + ", " + "Signature=" + signature;
	headers["Authorization"] = authorizationHeader;
}

void S3BlobStoreEndpoint::setAuthHeaders(std::string const& verb, std::string const& resource, HTTP::Headers& headers) {
	if (!credentials.present()) {
		return;
	}
	Credentials creds = credentials.get();

	std::string& date = headers["Date"];

	char dateBuf[64];
	time_t ts;
	time(&ts);
	strftime(dateBuf, 64, "%a, %d %b %Y %H:%M:%S GMT", gmtime(&ts));
	date = dateBuf;

	std::string msg;
	msg.append(verb);
	msg.append("\n");
	auto contentMD5 = headers.find("Content-MD5");
	if (contentMD5 != headers.end())
		msg.append(contentMD5->second);
	msg.append("\n");
	auto contentType = headers.find("Content-Type");
	if (contentType != headers.end())
		msg.append(contentType->second);
	msg.append("\n");
	msg.append(date);
	msg.append("\n");
	for (auto h : headers) {
		StringRef name = h.first;
		if (name.startsWith("x-amz"_sr) || name.startsWith("x-icloud"_sr)) {
			msg.append(h.first);
			msg.append(":");
			msg.append(h.second);
			msg.append("\n");
		}
	}

	msg.append(resource);
	if (verb == "GET") {
		size_t q = resource.find_last_of('?');
		if (q != resource.npos)
			msg.resize(msg.size() - (resource.size() - q));
	}

	std::string sig = base64::encoder::from_string(hmac_sha1(creds, msg));
	// base64 encoded blocks end in \n so remove it.
	sig.resize(sig.size() - 1);
	std::string auth = "AWS ";
	auth.append(creds.key);
	auth.append(":");
	auth.append(sig);
	headers["Authorization"] = auth;
}

															#line 9158 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via readEntireFile_impl()
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ReadEntireFile_implActor>
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ReadEntireFile_implActorState {
															#line 9165 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ReadEntireFile_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 9176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("readEntireFile_impl", reinterpret_cast<unsigned long>(this));

	}
	~ReadEntireFile_implActorState() 
	{
		fdb_probe_actor_destroy("readEntireFile_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1878 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateRead->getAllowance(1);
															#line 1878 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9193 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 1;
															#line 1878 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadEntireFile_implActor, 0, Void >*>(static_cast<ReadEntireFile_implActor*>(this)));
															#line 9198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadEntireFile_implActorState();
		static_cast<ReadEntireFile_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1880 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 1881 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1884 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["x-amz-checksum-mode"] = "ENABLED";
															#line 1885 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", resource, headers, nullptr, 0, { 200, 404 });
															#line 1885 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 2;
															#line 1885 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ReadEntireFile_implActor*>(this)));
															#line 9234 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1880 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 1881 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1884 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["x-amz-checksum-mode"] = "ENABLED";
															#line 1885 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", resource, headers, nullptr, 0, { 200, 404 });
															#line 1885 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9251 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 2;
															#line 1885 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ReadEntireFile_implActor*>(this)));
															#line 9256 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state > 0) static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 0;
		static_cast<ReadEntireFile_implActor*>(this)->ActorCallback< ReadEntireFile_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadEntireFile_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadEntireFile_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadEntireFile_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 1887 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 9340 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1888 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(file_not_found(), loopDepth);
															#line 9344 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1889 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 9348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1894 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string checksumSHA256 = r->data.headers["x-amz-checksum-sha256"];
															#line 1895 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (checksumSHA256.empty())
															#line 9354 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1897 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(http_bad_response(), loopDepth);
															#line 9358 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1900 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string contentSHA256 = sha256_base64(r->data.content);
															#line 1901 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (checksumSHA256 != contentSHA256)
															#line 9364 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1902 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(checksum_failed(), loopDepth);
															#line 9368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 1905 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ReadEntireFile_implActor*>(this)->SAV<std::string>::futures) { (void)(r->data.content); this->~ReadEntireFile_implActorState(); static_cast<ReadEntireFile_implActor*>(this)->destroy(); return 0; }
															#line 9373 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ReadEntireFile_implActor*>(this)->SAV< std::string >::value()) std::string(r->data.content);
		this->~ReadEntireFile_implActorState();
		static_cast<ReadEntireFile_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 1887 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 9385 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1888 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(file_not_found(), loopDepth);
															#line 9389 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1889 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 9393 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1894 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string checksumSHA256 = r->data.headers["x-amz-checksum-sha256"];
															#line 1895 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (checksumSHA256.empty())
															#line 9399 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1897 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(http_bad_response(), loopDepth);
															#line 9403 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1900 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string contentSHA256 = sha256_base64(r->data.content);
															#line 1901 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (checksumSHA256 != contentSHA256)
															#line 9409 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1902 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(checksum_failed(), loopDepth);
															#line 9413 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 1905 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ReadEntireFile_implActor*>(this)->SAV<std::string>::futures) { (void)(r->data.content); this->~ReadEntireFile_implActorState(); static_cast<ReadEntireFile_implActor*>(this)->destroy(); return 0; }
															#line 9418 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ReadEntireFile_implActor*>(this)->SAV< std::string >::value()) std::string(r->data.content);
		this->~ReadEntireFile_implActorState();
		static_cast<ReadEntireFile_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state > 0) static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 0;
		static_cast<ReadEntireFile_implActor*>(this)->ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 1880 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1881 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 9511 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via readEntireFile_impl()
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ReadEntireFile_implActor final : public Actor<std::string>, public ActorCallback< ReadEntireFile_implActor, 0, Void >, public ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ReadEntireFile_implActor>, public ReadEntireFile_implActorState<ReadEntireFile_implActor> {
															#line 9516 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ReadEntireFile_implActor>::operator new;
	using FastAllocated<ReadEntireFile_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17465419384577442048UL, 11119268078152743168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadEntireFile_implActor, 0, Void >;
friend struct ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ReadEntireFile_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 9534 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::string>(),
		   ReadEntireFile_implActorState<ReadEntireFile_implActor>(bstore, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9869764261496093184UL, 2881618862979785984UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readEntireFile_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadEntireFile_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<std::string> readEntireFile_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object ) {
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::string>(new ReadEntireFile_implActor(bstore, bucket, object));
															#line 9568 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1907 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::string> S3BlobStoreEndpoint::readEntireFile(std::string const& bucket, std::string const& object) {
	return readEntireFile_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 9577 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via writeEntireFileFromBuffer_impl()
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class WriteEntireFileFromBuffer_implActor>
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class WriteEntireFileFromBuffer_implActorState {
															#line 9584 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	WriteEntireFileFromBuffer_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,UnsentPacketQueue* const& pContent,int const& contentLen,std::string const& contentHash) 
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pContent(pContent),
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentLen(contentLen),
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentHash(contentHash)
															#line 9601 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this));

	}
	~WriteEntireFileFromBuffer_implActorState() 
	{
		fdb_probe_actor_destroy("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1918 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (contentLen > bstore->knobs.multipart_max_part_size)
															#line 9616 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1919 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(file_too_large(), loopDepth);
															#line 9620 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1921 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 1921 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9626 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 1;
															#line 1921 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9631 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteEntireFileFromBuffer_implActorState();
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1922 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bstore->concurrentUploads.take();
															#line 1922 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9656 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 2;
															#line 1922 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9661 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1922 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bstore->concurrentUploads.take();
															#line 1922 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9672 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 2;
															#line 1922 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9677 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1923 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		uploadReleaser = FlowLock::Releaser(bstore->concurrentUploads, 1);
															#line 1925 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 1926 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1931 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 9767 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1932 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-sha256"] = contentHash;
															#line 1933 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-algorithm"] = "SHA256";
															#line 9773 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1935 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-MD5"] = contentHash;
															#line 9779 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1937 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE.empty())
															#line 9783 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1938 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-server-side-encryption"] = CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE;
															#line 9787 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1939 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = doRequest_impl(bstore, "PUT", resource, headers, pContent, contentLen, { 200 });
															#line 1939 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9793 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 3;
															#line 1939 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9798 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1923 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		uploadReleaser = FlowLock::Releaser(bstore->concurrentUploads, 1);
															#line 1925 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 1926 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1931 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 9813 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1932 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-sha256"] = contentHash;
															#line 1933 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-algorithm"] = "SHA256";
															#line 9819 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1935 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-MD5"] = contentHash;
															#line 9825 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1937 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE.empty())
															#line 9829 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1938 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-server-side-encryption"] = CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE;
															#line 9833 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1939 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = doRequest_impl(bstore, "PUT", resource, headers, pContent, contentLen, { 200 });
															#line 1939 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9839 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 3;
															#line 1939 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9844 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 1941 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<WriteEntireFileFromBuffer_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteEntireFileFromBuffer_implActorState(); static_cast<WriteEntireFileFromBuffer_implActor*>(this)->destroy(); return 0; }
															#line 9928 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<WriteEntireFileFromBuffer_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteEntireFileFromBuffer_implActorState();
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 1941 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<WriteEntireFileFromBuffer_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteEntireFileFromBuffer_implActorState(); static_cast<WriteEntireFileFromBuffer_implActor*>(this)->destroy(); return 0; }
															#line 9940 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<WriteEntireFileFromBuffer_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteEntireFileFromBuffer_implActorState();
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont4(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue* pContent;
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int contentLen;
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string contentHash;
															#line 1923 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser uploadReleaser;
															#line 1925 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1926 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 10041 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via writeEntireFileFromBuffer_impl()
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class WriteEntireFileFromBuffer_implActor final : public Actor<Void>, public ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >, public ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >, public ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >, public FastAllocated<WriteEntireFileFromBuffer_implActor>, public WriteEntireFileFromBuffer_implActorState<WriteEntireFileFromBuffer_implActor> {
															#line 10046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<WriteEntireFileFromBuffer_implActor>::operator new;
	using FastAllocated<WriteEntireFileFromBuffer_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14379524943934968320UL, 14094530018279057920UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >;
friend struct ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >;
friend struct ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >;
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	WriteEntireFileFromBuffer_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,UnsentPacketQueue* const& pContent,int const& contentLen,std::string const& contentHash) 
															#line 10065 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   WriteEntireFileFromBuffer_implActorState<WriteEntireFileFromBuffer_implActor>(bstore, bucket, object, pContent, contentLen, contentHash),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15594163113904011264UL, 9204598572552334336UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeEntireFileFromBuffer_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> writeEntireFileFromBuffer_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, UnsentPacketQueue* const& pContent, int const& contentLen, std::string const& contentHash ) {
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new WriteEntireFileFromBuffer_implActor(bstore, bucket, object, pContent, contentLen, contentHash));
															#line 10100 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1943 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

															#line 10105 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via writeEntireFile_impl()
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class WriteEntireFile_implActor>
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class WriteEntireFile_implActorState {
															#line 10112 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	WriteEntireFile_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& content) 
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   content(content),
															#line 1948 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   packets()
															#line 10127 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("writeEntireFile_impl", reinterpret_cast<unsigned long>(this));

	}
	~WriteEntireFile_implActorState() 
	{
		fdb_probe_actor_destroy("writeEntireFile_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1949 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (content.size() > bstore->knobs.multipart_max_part_size)
															#line 10142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1950 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(file_too_large(), loopDepth);
															#line 10146 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1952 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			PacketWriter pw(packets.getWriteBuffer(content.size()), nullptr, Unversioned());
															#line 1953 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			pw.serializeBytes(content);
															#line 1959 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = yield();
															#line 1959 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10156 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 1;
															#line 1959 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFile_implActor, 0, Void >*>(static_cast<WriteEntireFile_implActor*>(this)));
															#line 10161 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteEntireFile_implActorState();
		static_cast<WriteEntireFile_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1970 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string contentHash;
															#line 1971 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (CLIENT_KNOBS->BLOBSTORE_ENABLE_OBJECT_INTEGRITY_CHECK)
															#line 10186 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1973 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash = sha256_base64(content);
															#line 10190 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1975 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			MD5_CTX sum;
															#line 1976 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Init(&sum);
															#line 1977 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Update(&sum, content.data(), content.size());
															#line 1978 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string sumBytes;
															#line 1979 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			sumBytes.resize(16);
															#line 1980 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Final((unsigned char*)sumBytes.data(), &sum);
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash = base64::encoder::from_string(sumBytes);
															#line 1982 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash.resize(contentHash.size() - 1);
															#line 10210 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1985 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = writeEntireFileFromBuffer_impl(bstore, bucket, object, &packets, content.size(), contentHash);
															#line 1985 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10216 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 2;
															#line 1985 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFile_implActor, 1, Void >*>(static_cast<WriteEntireFile_implActor*>(this)));
															#line 10221 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1970 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string contentHash;
															#line 1971 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (CLIENT_KNOBS->BLOBSTORE_ENABLE_OBJECT_INTEGRITY_CHECK)
															#line 10232 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1973 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash = sha256_base64(content);
															#line 10236 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1975 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			MD5_CTX sum;
															#line 1976 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Init(&sum);
															#line 1977 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Update(&sum, content.data(), content.size());
															#line 1978 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string sumBytes;
															#line 1979 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			sumBytes.resize(16);
															#line 1980 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Final((unsigned char*)sumBytes.data(), &sum);
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash = base64::encoder::from_string(sumBytes);
															#line 1982 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash.resize(contentHash.size() - 1);
															#line 10256 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1985 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = writeEntireFileFromBuffer_impl(bstore, bucket, object, &packets, content.size(), contentHash);
															#line 1985 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10262 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 2;
															#line 1985 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFile_implActor, 1, Void >*>(static_cast<WriteEntireFile_implActor*>(this)));
															#line 10267 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFile_implActor*>(this)->ActorCallback< WriteEntireFile_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFile_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteEntireFile_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteEntireFile_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1986 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<WriteEntireFile_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteEntireFile_implActorState(); static_cast<WriteEntireFile_implActor*>(this)->destroy(); return 0; }
															#line 10351 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<WriteEntireFile_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteEntireFile_implActorState();
		static_cast<WriteEntireFile_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1986 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<WriteEntireFile_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteEntireFile_implActorState(); static_cast<WriteEntireFile_implActor*>(this)->destroy(); return 0; }
															#line 10363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<WriteEntireFile_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteEntireFile_implActorState();
		static_cast<WriteEntireFile_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFile_implActor*>(this)->ActorCallback< WriteEntireFile_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFile_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteEntireFile_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteEntireFile_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string content;
															#line 1948 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue packets;
															#line 10456 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via writeEntireFile_impl()
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class WriteEntireFile_implActor final : public Actor<Void>, public ActorCallback< WriteEntireFile_implActor, 0, Void >, public ActorCallback< WriteEntireFile_implActor, 1, Void >, public FastAllocated<WriteEntireFile_implActor>, public WriteEntireFile_implActorState<WriteEntireFile_implActor> {
															#line 10461 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<WriteEntireFile_implActor>::operator new;
	using FastAllocated<WriteEntireFile_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6665598190599031040UL, 17239259188628011264UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteEntireFile_implActor, 0, Void >;
friend struct ActorCallback< WriteEntireFile_implActor, 1, Void >;
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	WriteEntireFile_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& content) 
															#line 10479 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   WriteEntireFile_implActorState<WriteEntireFile_implActor>(bstore, bucket, object, content),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18357809352626313472UL, 5788567832381847040UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeEntireFile_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteEntireFile_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteEntireFile_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> writeEntireFile_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::string const& content ) {
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new WriteEntireFile_implActor(bstore, bucket, object, content));
															#line 10513 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1988 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::writeEntireFile(std::string const& bucket,
                                                  std::string const& object,
                                                  std::string const& content) {
	return writeEntireFile_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, content);
}

Future<Void> S3BlobStoreEndpoint::writeEntireFileFromBuffer(std::string const& bucket,
                                                            std::string const& object,
                                                            UnsentPacketQueue* pContent,
                                                            int contentLen,
                                                            std::string const& contentHash) {
	return writeEntireFileFromBuffer_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, pContent, contentLen, contentHash);
}

															#line 10533 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via readObject_impl()
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ReadObject_implActor>
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ReadObject_implActorState {
															#line 10540 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ReadObject_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,void* const& data,int const& length,int64_t const& offset) 
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   data(data),
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   length(length),
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   offset(offset)
															#line 10557 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("readObject_impl", reinterpret_cast<unsigned long>(this));

	}
	~ReadObject_implActorState() 
	{
		fdb_probe_actor_destroy("readObject_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 2011 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (length <= 0)
															#line 10573 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 2012 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevWarn, "S3BlobStoreReadObjectEmptyRead").detail("Length", length);
															#line 2013 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<ReadObject_implActor*>(this)->SAV<int>::futures) { (void)(0); this->~ReadObject_implActorState(); static_cast<ReadObject_implActor*>(this)->destroy(); return 0; }
															#line 10579 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<ReadObject_implActor*>(this)->SAV< int >::value()) int(0);
					this->~ReadObject_implActorState();
					static_cast<ReadObject_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 2017 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Void> __when_expr_0 = bstore->requestRateRead->getAllowance(1);
															#line 2017 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<ReadObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 10589 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ReadObject_implActor*>(this)->actor_wait_state = 1;
															#line 2017 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadObject_implActor, 0, Void >*>(static_cast<ReadObject_implActor*>(this)));
															#line 10594 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadObject_implActorState();
		static_cast<ReadObject_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2051 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreEndpoint_ReadError") .error(e) .detail("Bucket", bucket) .detail("Object", object) .detail("Length", length) .detail("Offset", offset);
															#line 2057 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 10626 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2019 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2020 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2021 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["Range"] = format("bytes=%lld-%lld", offset, offset + length - 1);
															#line 2024 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = Reference<HTTP::IncomingResponse>();
															#line 2025 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", resource, headers, nullptr, 0, { 200, 206, 404 });
															#line 2025 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ReadObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 10650 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadObject_implActor*>(this)->actor_wait_state = 2;
															#line 2025 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ReadObject_implActor*>(this)));
															#line 10655 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2019 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2020 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2021 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["Range"] = format("bytes=%lld-%lld", offset, offset + length - 1);
															#line 2024 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = Reference<HTTP::IncomingResponse>();
															#line 2025 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", resource, headers, nullptr, 0, { 200, 206, 404 });
															#line 2025 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ReadObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 10674 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadObject_implActor*>(this)->actor_wait_state = 2;
															#line 2025 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ReadObject_implActor*>(this)));
															#line 10679 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadObject_implActor*>(this)->actor_wait_state > 0) static_cast<ReadObject_implActor*>(this)->actor_wait_state = 0;
		static_cast<ReadObject_implActor*>(this)->ActorCallback< ReadObject_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadObject_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadObject_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadObject_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
															#line 2027 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = _r;
															#line 2029 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 10765 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(file_not_found(), loopDepth);
															#line 10769 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2034 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->data.contentLen != r->data.content.size())
															#line 10773 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2035 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreReadObjectContentLengthMismatch") .detail("Expected", r->data.contentLen) .detail("Actual", r->data.content.size());
															#line 2038 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(io_error(), loopDepth);
															#line 10779 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		try {
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			memcpy(data, r->data.content.data(), std::min<int64_t>(r->data.contentLen, length));
															#line 2045 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<ReadObject_implActor*>(this)->SAV<int>::futures) { (void)(r->data.contentLen); this->~ReadObject_implActorState(); static_cast<ReadObject_implActor*>(this)->destroy(); return 0; }
															#line 10786 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<ReadObject_implActor*>(this)->SAV< int >::value()) int(r->data.contentLen);
			this->~ReadObject_implActorState();
			static_cast<ReadObject_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont4Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont4Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
															#line 2027 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = _r;
															#line 2029 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 10806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(file_not_found(), loopDepth);
															#line 10810 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2034 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->data.contentLen != r->data.content.size())
															#line 10814 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2035 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreReadObjectContentLengthMismatch") .detail("Expected", r->data.contentLen) .detail("Actual", r->data.content.size());
															#line 2038 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(io_error(), loopDepth);
															#line 10820 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		try {
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			memcpy(data, r->data.content.data(), std::min<int64_t>(r->data.contentLen, length));
															#line 2045 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<ReadObject_implActor*>(this)->SAV<int>::futures) { (void)(r->data.contentLen); this->~ReadObject_implActorState(); static_cast<ReadObject_implActor*>(this)->destroy(); return 0; }
															#line 10827 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<ReadObject_implActor*>(this)->SAV< int >::value()) int(r->data.contentLen);
			this->~ReadObject_implActorState();
			static_cast<ReadObject_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont4Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont4Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
		loopDepth = a_body1cont4(_r, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadObject_implActor*>(this)->actor_wait_state > 0) static_cast<ReadObject_implActor*>(this)->actor_wait_state = 0;
		static_cast<ReadObject_implActor*>(this)->ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2047 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreReadObjectMemcpyError").detail("Error", e.what());
															#line 2048 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(io_error(), loopDepth);
															#line 10923 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	void* data;
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int length;
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int64_t offset;
															#line 2019 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2020 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 2024 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 10951 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via readObject_impl()
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ReadObject_implActor final : public Actor<int>, public ActorCallback< ReadObject_implActor, 0, Void >, public ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ReadObject_implActor>, public ReadObject_implActorState<ReadObject_implActor> {
															#line 10956 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ReadObject_implActor>::operator new;
	using FastAllocated<ReadObject_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14078678576366191104UL, 4770617376688913152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadObject_implActor, 0, Void >;
friend struct ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ReadObject_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,void* const& data,int const& length,int64_t const& offset) 
															#line 10974 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<int>(),
		   ReadObject_implActorState<ReadObject_implActor>(bstore, bucket, object, data, length, offset),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7919612023543018496UL, 14030950089782221056UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readObject_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadObject_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<int> readObject_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, void* const& data, int const& length, int64_t const& offset ) {
															#line 2004 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<int>(new ReadObject_implActor(bstore, bucket, object, data, length, offset));
															#line 11008 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2060 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<int> S3BlobStoreEndpoint::readObject(std::string const& bucket,
                                            std::string const& object,
                                            void* data,
                                            int length,
                                            int64_t offset) {
	return readObject_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, data, length, offset);
}

															#line 11021 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via beginMultiPartUpload_impl()
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class BeginMultiPartUpload_implActor>
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class BeginMultiPartUpload_implActorState {
															#line 11028 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	BeginMultiPartUpload_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 11039 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	~BeginMultiPartUpload_implActorState() 
	{
		fdb_probe_actor_destroy("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2072 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2072 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11056 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 1;
															#line 2072 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*>(static_cast<BeginMultiPartUpload_implActor*>(this)));
															#line 11061 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BeginMultiPartUpload_implActorState();
		static_cast<BeginMultiPartUpload_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2074 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2075 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?uploads";
															#line 2076 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2077 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE.empty())
															#line 11090 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2078 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-server-side-encryption"] = CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE;
															#line 11094 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2079 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 11098 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2083 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-algorithm"] = "SHA256";
															#line 11102 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2085 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "POST", resource, headers, nullptr, 0, { 200 });
															#line 2085 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2085 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<BeginMultiPartUpload_implActor*>(this)));
															#line 11113 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2074 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2075 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?uploads";
															#line 2076 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2077 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE.empty())
															#line 11128 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2078 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-server-side-encryption"] = CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE;
															#line 11132 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2079 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 11136 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2083 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-algorithm"] = "SHA256";
															#line 11140 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2085 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "POST", resource, headers, nullptr, 0, { 200 });
															#line 2085 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11146 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2085 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<BeginMultiPartUpload_implActor*>(this)));
															#line 11151 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<BeginMultiPartUpload_implActor*>(this)->ActorCallback< BeginMultiPartUpload_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		try {
															#line 2088 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 2090 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 2092 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 2095 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 2096 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result != nullptr && strcmp(result->name(), "InitiateMultipartUploadResult") == 0)
															#line 11244 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2097 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				xml_node<>* id = result->first_node("UploadId");
															#line 2098 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (id != nullptr)
															#line 11250 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<BeginMultiPartUpload_implActor*>(this)->SAV<std::string>::futures) { (void)(id->value()); this->~BeginMultiPartUpload_implActorState(); static_cast<BeginMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 11254 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<BeginMultiPartUpload_implActor*>(this)->SAV< std::string >::value()) std::string(id->value());
					this->~BeginMultiPartUpload_implActorState();
					static_cast<BeginMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
			loopDepth = a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		try {
															#line 2088 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 2090 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 2092 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 2095 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 2096 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result != nullptr && strcmp(result->name(), "InitiateMultipartUploadResult") == 0)
															#line 11284 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2097 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				xml_node<>* id = result->first_node("UploadId");
															#line 2098 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (id != nullptr)
															#line 11290 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<BeginMultiPartUpload_implActor*>(this)->SAV<std::string>::futures) { (void)(id->value()); this->~BeginMultiPartUpload_implActorState(); static_cast<BeginMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 11294 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<BeginMultiPartUpload_implActor*>(this)->SAV< std::string >::value()) std::string(id->value());
					this->~BeginMultiPartUpload_implActorState();
					static_cast<BeginMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
			loopDepth = a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<BeginMultiPartUpload_implActor*>(this)->ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 2104 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		return a_body1Catch1(http_bad_response(), loopDepth);
															#line 11390 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont2Catch1(const Error& __current_error,int loopDepth=0) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2074 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2076 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 11430 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via beginMultiPartUpload_impl()
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class BeginMultiPartUpload_implActor final : public Actor<std::string>, public ActorCallback< BeginMultiPartUpload_implActor, 0, Void >, public ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<BeginMultiPartUpload_implActor>, public BeginMultiPartUpload_implActorState<BeginMultiPartUpload_implActor> {
															#line 11435 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<BeginMultiPartUpload_implActor>::operator new;
	using FastAllocated<BeginMultiPartUpload_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5354238901276868608UL, 3164497992322843136UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BeginMultiPartUpload_implActor, 0, Void >;
friend struct ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	BeginMultiPartUpload_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 11453 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::string>(),
		   BeginMultiPartUpload_implActorState<BeginMultiPartUpload_implActor>(bstore, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4949678732428246272UL, 2657853908492627456UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("beginMultiPartUpload_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] static Future<std::string> beginMultiPartUpload_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object ) {
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::string>(new BeginMultiPartUpload_implActor(bstore, bucket, object));
															#line 11487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2106 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::string> S3BlobStoreEndpoint::beginMultiPartUpload(std::string const& bucket, std::string const& object) {
	return beginMultiPartUpload_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 11496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via uploadPart_impl()
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class UploadPart_implActor>
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class UploadPart_implActorState {
															#line 11503 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UploadPart_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID,unsigned int const& partNumber,UnsentPacketQueue* const& pContent,int const& contentLen,std::string const& contentMD5) 
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   uploadID(uploadID),
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   partNumber(partNumber),
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pContent(pContent),
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentLen(contentLen),
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentMD5(contentMD5)
															#line 11524 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("uploadPart_impl", reinterpret_cast<unsigned long>(this));

	}
	~UploadPart_implActorState() 
	{
		fdb_probe_actor_destroy("uploadPart_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2119 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2119 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11541 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UploadPart_implActor*>(this)->actor_wait_state = 1;
															#line 2119 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 0, Void >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11546 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UploadPart_implActorState();
		static_cast<UploadPart_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2120 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bstore->concurrentUploads.take();
															#line 2120 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<UploadPart_implActor*>(this)->actor_wait_state = 2;
															#line 2120 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 1, Void >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2120 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bstore->concurrentUploads.take();
															#line 2120 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<UploadPart_implActor*>(this)->actor_wait_state = 2;
															#line 2120 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 1, Void >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11592 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state > 0) static_cast<UploadPart_implActor*>(this)->actor_wait_state = 0;
		static_cast<UploadPart_implActor*>(this)->ActorCallback< UploadPart_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UploadPart_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2121 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		uploadReleaser = FlowLock::Releaser(bstore->concurrentUploads, 1);
															#line 2123 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2124 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?partNumber=%d&uploadId=%s", partNumber, uploadID.c_str());
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2128 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 11684 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2129 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-sha256"] = contentMD5;
															#line 2130 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-algorithm"] = "SHA256";
															#line 11690 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 2132 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-MD5"] = contentMD5;
															#line 11696 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = doRequest_impl(bstore, "PUT", resource, headers, pContent, contentLen, { 200 });
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11702 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<UploadPart_implActor*>(this)->actor_wait_state = 3;
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11707 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2121 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		uploadReleaser = FlowLock::Releaser(bstore->concurrentUploads, 1);
															#line 2123 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2124 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?partNumber=%d&uploadId=%s", partNumber, uploadID.c_str());
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2128 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 11724 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2129 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-sha256"] = contentMD5;
															#line 2130 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-algorithm"] = "SHA256";
															#line 11730 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 2132 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-MD5"] = contentMD5;
															#line 11736 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = doRequest_impl(bstore, "PUT", resource, headers, pContent, contentLen, { 200 });
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11742 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<UploadPart_implActor*>(this)->actor_wait_state = 3;
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11747 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state > 0) static_cast<UploadPart_implActor*>(this)->actor_wait_state = 0;
		static_cast<UploadPart_implActor*>(this)->ActorCallback< UploadPart_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< UploadPart_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2142 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!bstore->knobs.enable_object_integrity_check)
															#line 11831 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!HTTP::verifyMD5(&r->data, false, contentMD5))
															#line 11835 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2144 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(checksum_failed(), loopDepth);
															#line 11839 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 2150 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string etag = r->data.headers["ETag"];
															#line 2151 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (etag.empty())
															#line 11846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2152 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 11850 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2154 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<UploadPart_implActor*>(this)->SAV<std::string>::futures) { (void)(etag); this->~UploadPart_implActorState(); static_cast<UploadPart_implActor*>(this)->destroy(); return 0; }
															#line 11854 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<UploadPart_implActor*>(this)->SAV< std::string >::value()) std::string(etag);
		this->~UploadPart_implActorState();
		static_cast<UploadPart_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> const& __r,int loopDepth) 
	{
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = __r;
															#line 11866 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state > 0) static_cast<UploadPart_implActor*>(this)->actor_wait_state = 0;
		static_cast<UploadPart_implActor*>(this)->ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string uploadID;
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	unsigned int partNumber;
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue* pContent;
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int contentLen;
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string contentMD5;
															#line 2121 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser uploadReleaser;
															#line 2123 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 11965 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via uploadPart_impl()
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class UploadPart_implActor final : public Actor<std::string>, public ActorCallback< UploadPart_implActor, 0, Void >, public ActorCallback< UploadPart_implActor, 1, Void >, public ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >, public FastAllocated<UploadPart_implActor>, public UploadPart_implActorState<UploadPart_implActor> {
															#line 11970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<UploadPart_implActor>::operator new;
	using FastAllocated<UploadPart_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15984101439949168384UL, 8909375212530276352UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UploadPart_implActor, 0, Void >;
friend struct ActorCallback< UploadPart_implActor, 1, Void >;
friend struct ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >;
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UploadPart_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID,unsigned int const& partNumber,UnsentPacketQueue* const& pContent,int const& contentLen,std::string const& contentMD5) 
															#line 11989 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::string>(),
		   UploadPart_implActorState<UploadPart_implActor>(bstore, bucket, object, uploadID, partNumber, pContent, contentLen, contentMD5),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1338121866253869824UL, 10943970907985309952UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("uploadPart_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UploadPart_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< UploadPart_implActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<std::string> uploadPart_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::string const& uploadID, unsigned int const& partNumber, UnsentPacketQueue* const& pContent, int const& contentLen, std::string const& contentMD5 ) {
															#line 2111 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::string>(new UploadPart_implActor(bstore, bucket, object, uploadID, partNumber, pContent, contentLen, contentMD5));
															#line 12024 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2156 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::string> S3BlobStoreEndpoint::uploadPart(std::string const& bucket,
                                                    std::string const& object,
                                                    std::string const& uploadID,
                                                    unsigned int partNumber,
                                                    UnsentPacketQueue* pContent,
                                                    int contentLen,
                                                    std::string const& contentMD5) {
	return uploadPart_impl(Reference<S3BlobStoreEndpoint>::addRef(this),
	                       bucket,
	                       object,
	                       uploadID,
	                       partNumber,
	                       pContent,
	                       contentLen,
	                       contentMD5);
}

															#line 12046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via finishMultiPartUpload_impl()
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class FinishMultiPartUpload_implActor>
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FinishMultiPartUpload_implActorState {
															#line 12053 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FinishMultiPartUpload_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID,S3BlobStoreEndpoint::MultiPartSetT const& parts) 
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   uploadID(uploadID),
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   parts(parts),
															#line 2179 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   part_list()
															#line 12070 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	~FinishMultiPartUpload_implActorState() 
	{
		fdb_probe_actor_destroy("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2180 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2180 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12087 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 1;
															#line 2180 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*>(static_cast<FinishMultiPartUpload_implActor*>(this)));
															#line 12092 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FinishMultiPartUpload_implActorState();
		static_cast<FinishMultiPartUpload_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2182 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string manifest = "<CompleteMultipartUpload>";
															#line 2183 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& p : parts ) {
															#line 2184 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			manifest += format("<Part><PartNumber>%d</PartNumber><ETag>%s</ETag>", p.first, p.second.etag.c_str());
															#line 2186 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (bstore->knobs.enable_object_integrity_check && !p.second.checksum.empty())
															#line 12121 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2187 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				manifest += format("<ChecksumSHA256>%s</ChecksumSHA256>", p.second.checksum.c_str());
															#line 12125 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2189 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			manifest += "</Part>\n";
															#line 12129 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2191 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		manifest += "</CompleteMultipartUpload>";
															#line 2193 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2194 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?uploadId=%s", uploadID.c_str());
															#line 2195 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2196 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		PacketWriter pw(part_list.getWriteBuffer(manifest.size()), nullptr, Unversioned());
															#line 2197 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		pw.serializeBytes(manifest);
															#line 2198 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "POST", resource, headers, &part_list, manifest.size(), { 200 });
															#line 2198 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12147 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2198 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<FinishMultiPartUpload_implActor*>(this)));
															#line 12152 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2182 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string manifest = "<CompleteMultipartUpload>";
															#line 2183 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& p : parts ) {
															#line 2184 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			manifest += format("<Part><PartNumber>%d</PartNumber><ETag>%s</ETag>", p.first, p.second.etag.c_str());
															#line 2186 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (bstore->knobs.enable_object_integrity_check && !p.second.checksum.empty())
															#line 12167 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2187 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				manifest += format("<ChecksumSHA256>%s</ChecksumSHA256>", p.second.checksum.c_str());
															#line 12171 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2189 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			manifest += "</Part>\n";
															#line 12175 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2191 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		manifest += "</CompleteMultipartUpload>";
															#line 2193 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2194 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?uploadId=%s", uploadID.c_str());
															#line 2195 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2196 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		PacketWriter pw(part_list.getWriteBuffer(manifest.size()), nullptr, Unversioned());
															#line 2197 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		pw.serializeBytes(manifest);
															#line 2198 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "POST", resource, headers, &part_list, manifest.size(), { 200 });
															#line 2198 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12193 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2198 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<FinishMultiPartUpload_implActor*>(this)));
															#line 12198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<FinishMultiPartUpload_implActor*>(this)->ActorCallback< FinishMultiPartUpload_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 2209 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<FinishMultiPartUpload_implActor*>(this)->SAV<Optional<std::string>>::futures) { (void)(Optional<std::string>()); this->~FinishMultiPartUpload_implActorState(); static_cast<FinishMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 12282 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<FinishMultiPartUpload_implActor*>(this)->SAV< Optional<std::string> >::value()) Optional<std::string>(Optional<std::string>());
		this->~FinishMultiPartUpload_implActorState();
		static_cast<FinishMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 2209 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<FinishMultiPartUpload_implActor*>(this)->SAV<Optional<std::string>>::futures) { (void)(Optional<std::string>()); this->~FinishMultiPartUpload_implActorState(); static_cast<FinishMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 12294 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<FinishMultiPartUpload_implActor*>(this)->SAV< Optional<std::string> >::value()) Optional<std::string>(Optional<std::string>());
		this->~FinishMultiPartUpload_implActorState();
		static_cast<FinishMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<FinishMultiPartUpload_implActor*>(this)->ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string uploadID;
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	S3BlobStoreEndpoint::MultiPartSetT parts;
															#line 2179 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue part_list;
															#line 2193 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2195 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 12393 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via finishMultiPartUpload_impl()
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FinishMultiPartUpload_implActor final : public Actor<Optional<std::string>>, public ActorCallback< FinishMultiPartUpload_implActor, 0, Void >, public ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<FinishMultiPartUpload_implActor>, public FinishMultiPartUpload_implActorState<FinishMultiPartUpload_implActor> {
															#line 12398 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<FinishMultiPartUpload_implActor>::operator new;
	using FastAllocated<FinishMultiPartUpload_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8136049497288206336UL, 5800962181273271808UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FinishMultiPartUpload_implActor, 0, Void >;
friend struct ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FinishMultiPartUpload_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID,S3BlobStoreEndpoint::MultiPartSetT const& parts) 
															#line 12416 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Optional<std::string>>(),
		   FinishMultiPartUpload_implActorState<FinishMultiPartUpload_implActor>(bstore, bucket, object, uploadID, parts),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(499877118889630720UL, 2657922083509416704UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("finishMultiPartUpload_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Optional<std::string>> finishMultiPartUpload_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::string const& uploadID, S3BlobStoreEndpoint::MultiPartSetT const& parts ) {
															#line 2174 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Optional<std::string>>(new FinishMultiPartUpload_implActor(bstore, bucket, object, uploadID, parts));
															#line 12450 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2211 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Optional<std::string>> S3BlobStoreEndpoint::finishMultiPartUpload(std::string const& bucket,
                                                                         std::string const& object,
                                                                         std::string const& uploadID,
                                                                         MultiPartSetT const& parts) {
	return finishMultiPartUpload_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, uploadID, parts);
}

															#line 12462 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via abortMultiPartUpload_impl()
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class AbortMultiPartUpload_implActor>
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class AbortMultiPartUpload_implActorState {
															#line 12469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	AbortMultiPartUpload_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID) 
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   uploadID(uploadID)
															#line 12482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	~AbortMultiPartUpload_implActorState() 
	{
		fdb_probe_actor_destroy("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2223 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2223 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12499 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 1;
															#line 2223 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*>(static_cast<AbortMultiPartUpload_implActor*>(this)));
															#line 12504 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AbortMultiPartUpload_implActorState();
		static_cast<AbortMultiPartUpload_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2225 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string resource = bstore->constructResourcePath(bucket, object);
															#line 2226 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?uploadId=%s", uploadID.c_str());
															#line 2228 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		HTTP::Headers headers;
															#line 2229 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "DELETE", resource, headers, nullptr, 0, { 200, 204 });
															#line 2229 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12535 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2229 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<AbortMultiPartUpload_implActor*>(this)));
															#line 12540 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2225 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string resource = bstore->constructResourcePath(bucket, object);
															#line 2226 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?uploadId=%s", uploadID.c_str());
															#line 2228 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		HTTP::Headers headers;
															#line 2229 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "DELETE", resource, headers, nullptr, 0, { 200, 204 });
															#line 2229 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12557 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2229 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<AbortMultiPartUpload_implActor*>(this)));
															#line 12562 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<AbortMultiPartUpload_implActor*>(this)->ActorCallback< AbortMultiPartUpload_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 2231 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<AbortMultiPartUpload_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AbortMultiPartUpload_implActorState(); static_cast<AbortMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 12646 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<AbortMultiPartUpload_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AbortMultiPartUpload_implActorState();
		static_cast<AbortMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 2231 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<AbortMultiPartUpload_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AbortMultiPartUpload_implActorState(); static_cast<AbortMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 12658 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<AbortMultiPartUpload_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AbortMultiPartUpload_implActorState();
		static_cast<AbortMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<AbortMultiPartUpload_implActor*>(this)->ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string uploadID;
															#line 12749 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via abortMultiPartUpload_impl()
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class AbortMultiPartUpload_implActor final : public Actor<Void>, public ActorCallback< AbortMultiPartUpload_implActor, 0, Void >, public ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<AbortMultiPartUpload_implActor>, public AbortMultiPartUpload_implActorState<AbortMultiPartUpload_implActor> {
															#line 12754 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<AbortMultiPartUpload_implActor>::operator new;
	using FastAllocated<AbortMultiPartUpload_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1990111951736385024UL, 8209986278351305472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AbortMultiPartUpload_implActor, 0, Void >;
friend struct ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	AbortMultiPartUpload_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID) 
															#line 12772 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   AbortMultiPartUpload_implActorState<AbortMultiPartUpload_implActor>(bstore, bucket, object, uploadID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16979502986786407680UL, 15532907730366778880UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("abortMultiPartUpload_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> abortMultiPartUpload_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::string const& uploadID ) {
															#line 2219 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new AbortMultiPartUpload_implActor(bstore, bucket, object, uploadID));
															#line 12806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2233 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::abortMultiPartUpload(std::string const& bucket,
                                                       std::string const& object,
                                                       std::string const& uploadID) {
	return abortMultiPartUpload_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, uploadID);
}

// Forward declarations
															#line 12818 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
[[nodiscard]] Future<std::map<std::string, std::string>> getObjectTags_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object );

#line 2244 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

															#line 12823 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via putObjectTags_impl()
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class PutObjectTags_implActor>
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class PutObjectTags_implActorState {
															#line 12830 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PutObjectTags_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& tags) 
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   tags(tags),
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   packets()
															#line 12845 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("putObjectTags_impl", reinterpret_cast<unsigned long>(this));

	}
	~PutObjectTags_implActorState() 
	{
		fdb_probe_actor_destroy("putObjectTags_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2250 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2250 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12862 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 1;
															#line 2250 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 0, Void >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 12867 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PutObjectTags_implActorState();
		static_cast<PutObjectTags_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2251 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2252 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?tagging";
															#line 2253 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxRetries = 5;
															#line 2254 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		retryCount = 0;
															#line 2255 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = 1.0;
															#line 2256 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxBackoff = 8.0;
															#line 2258 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 12902 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2251 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2252 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?tagging";
															#line 2253 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxRetries = 5;
															#line 2254 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		retryCount = 0;
															#line 2255 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = 1.0;
															#line 2256 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxBackoff = 8.0;
															#line 2258 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 12923 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2260 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string manifest = "<Tagging xmlns=\"http://s3.amazonaws.com/doc/2006-03-01/\"><TagSet>";
															#line 2261 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(auto itr = tags.begin();itr != tags.end();++itr) {
															#line 2262 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				manifest += "<Tag><Key>" + itr->first + "</Key><Value>" + itr->second + "</Value></Tag>";
															#line 13019 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2264 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			manifest += "</TagSet></Tagging>";
															#line 2266 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			PacketWriter pw(packets.getWriteBuffer(manifest.size()), nullptr, Unversioned());
															#line 2267 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			pw.serializeBytes(manifest);
															#line 2269 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			HTTP::Headers headers;
															#line 2270 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-Type"] = "application/xml";
															#line 2271 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-Length"] = format("%d", manifest.size());
															#line 2273 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "PUT", resource, headers, &packets, manifest.size(), { 200 });
															#line 2273 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13037 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 2;
															#line 2273 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13042 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2294 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 13064 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2295 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 13068 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2298 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStorePutTagsError") .error(e) .detail("Bucket", bucket) .detail("Object", object) .detail("RetryCount", retryCount);
															#line 2304 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (++retryCount >= maxRetries)
															#line 13074 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2305 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 13078 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2309 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(backoff * (0.9 + 0.2 * deterministicRandom()->random01()));
															#line 2309 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13084 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 5;
															#line 2309 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 4, Void >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13089 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 2277 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<std::map<std::string, std::string>> __when_expr_2 = getObjectTags_impl(bstore, bucket, object);
															#line 2277 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13106 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 3;
															#line 2277 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 2277 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<std::map<std::string, std::string>> __when_expr_2 = getObjectTags_impl(bstore, bucket, object);
															#line 2277 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13122 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 3;
															#line 2277 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13127 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont4(std::map<std::string, std::string> const& verifyTags,int loopDepth) 
	{
															#line 2278 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (verifyTags == tags)
															#line 13211 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2279 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<PutObjectTags_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PutObjectTags_implActorState(); static_cast<PutObjectTags_implActor*>(this)->destroy(); return 0; }
															#line 13215 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<PutObjectTags_implActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PutObjectTags_implActorState();
			static_cast<PutObjectTags_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2282 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (++retryCount >= maxRetries)
															#line 13223 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2283 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStorePutTagsMaxRetriesExceeded") .detail("Bucket", bucket) .detail("Object", object);
															#line 2286 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1cont1loopBody1Catch1(operation_failed(), loopDepth);
															#line 13229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2290 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(backoff * (0.9 + 0.2 * deterministicRandom()->random01()));
															#line 2290 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13235 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 4;
															#line 2290 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 3, Void >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(std::map<std::string, std::string> && verifyTags,int loopDepth) 
	{
															#line 2278 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (verifyTags == tags)
															#line 13249 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2279 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<PutObjectTags_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PutObjectTags_implActorState(); static_cast<PutObjectTags_implActor*>(this)->destroy(); return 0; }
															#line 13253 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<PutObjectTags_implActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PutObjectTags_implActorState();
			static_cast<PutObjectTags_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2282 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (++retryCount >= maxRetries)
															#line 13261 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2283 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStorePutTagsMaxRetriesExceeded") .detail("Bucket", bucket) .detail("Object", object);
															#line 2286 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1cont1loopBody1Catch1(operation_failed(), loopDepth);
															#line 13267 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2290 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(backoff * (0.9 + 0.2 * deterministicRandom()->random01()));
															#line 2290 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13273 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 4;
															#line 2290 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 3, Void >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13278 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(std::map<std::string, std::string> const& verifyTags,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(verifyTags, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(std::map<std::string, std::string> && verifyTags,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(std::move(verifyTags), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*,std::map<std::string, std::string> const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*,std::map<std::string, std::string> && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 2291 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = std::min(backoff * 2, maxBackoff);
															#line 13362 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 2291 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = std::min(backoff * 2, maxBackoff);
															#line 13371 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
															#line 2310 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = std::min(backoff * 2, maxBackoff);
															#line 13468 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
															#line 2310 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = std::min(backoff * 2, maxBackoff);
															#line 13477 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::map<std::string, std::string> tags;
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue packets;
															#line 2251 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2253 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int maxRetries;
															#line 2254 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int retryCount;
															#line 2255 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double backoff;
															#line 2256 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double maxBackoff;
															#line 13577 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via putObjectTags_impl()
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class PutObjectTags_implActor final : public Actor<Void>, public ActorCallback< PutObjectTags_implActor, 0, Void >, public ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >, public ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >, public ActorCallback< PutObjectTags_implActor, 3, Void >, public ActorCallback< PutObjectTags_implActor, 4, Void >, public FastAllocated<PutObjectTags_implActor>, public PutObjectTags_implActorState<PutObjectTags_implActor> {
															#line 13582 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<PutObjectTags_implActor>::operator new;
	using FastAllocated<PutObjectTags_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9082566429130456832UL, 17193981632970009600UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PutObjectTags_implActor, 0, Void >;
friend struct ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >;
friend struct ActorCallback< PutObjectTags_implActor, 3, Void >;
friend struct ActorCallback< PutObjectTags_implActor, 4, Void >;
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PutObjectTags_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& tags) 
															#line 13603 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   PutObjectTags_implActorState<PutObjectTags_implActor>(bstore, bucket, object, tags),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12648391953656857344UL, 8578338389149528320UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("putObjectTags_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> putObjectTags_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& tags ) {
															#line 2245 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new PutObjectTags_implActor(bstore, bucket, object, tags));
															#line 13640 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2314 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::putObjectTags(std::string const& bucket,
                                                std::string const& object,
                                                std::map<std::string, std::string> const& tags) {
	return putObjectTags_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, tags);
}

															#line 13651 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via getObjectTags_impl()
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class GetObjectTags_implActor>
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class GetObjectTags_implActorState {
															#line 13658 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	GetObjectTags_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 13669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("getObjectTags_impl", reinterpret_cast<unsigned long>(this));

	}
	~GetObjectTags_implActorState() 
	{
		fdb_probe_actor_destroy("getObjectTags_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2324 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateRead->getAllowance(1);
															#line 2324 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13686 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 1;
															#line 2324 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetObjectTags_implActor, 0, Void >*>(static_cast<GetObjectTags_implActor*>(this)));
															#line 13691 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetObjectTags_implActorState();
		static_cast<GetObjectTags_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2326 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2327 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?tagging";
															#line 2328 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2330 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", resource, headers, nullptr, 0, { 200 });
															#line 2330 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13722 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 2;
															#line 2330 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<GetObjectTags_implActor*>(this)));
															#line 13727 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2326 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2327 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?tagging";
															#line 2328 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2330 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = doRequest_impl(bstore, "GET", resource, headers, nullptr, 0, { 200 });
															#line 2330 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 2;
															#line 2330 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<GetObjectTags_implActor*>(this)));
															#line 13749 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetObjectTags_implActor*>(this)->ActorCallback< GetObjectTags_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetObjectTags_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetObjectTags_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetObjectTags_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 2332 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rapidxml::xml_document<> doc;
															#line 2333 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		doc.parse<rapidxml::parse_default>((char*)r->data.content.c_str());
															#line 2335 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::map<std::string, std::string> tags;
															#line 2338 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rapidxml::xml_node<>* tagging = doc.first_node();
															#line 2339 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for(;tagging && strcmp(tagging->name(), "Tagging") != 0;) {
															#line 2340 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			tagging = tagging->next_sibling();
															#line 13843 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2343 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (tagging)
															#line 13847 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2345 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			rapidxml::xml_node<>* tagSet = tagging->first_node();
															#line 2346 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;tagSet && strcmp(tagSet->name(), "TagSet") != 0;) {
															#line 2347 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				tagSet = tagSet->next_sibling();
															#line 13855 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2350 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (tagSet)
															#line 13859 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2352 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(rapidxml::xml_node<>* tag = tagSet->first_node();tag;tag = tag->next_sibling()) {
															#line 2353 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(tag->name(), "Tag") == 0)
															#line 13865 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 2354 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string key, value;
															#line 2356 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						for(rapidxml::xml_node<>* node = tag->first_node();node;node = node->next_sibling()) {
															#line 2357 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (strcmp(node->name(), "Key") == 0)
															#line 13873 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 2358 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								key = node->value();
															#line 13877 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							}
							else
							{
															#line 2359 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								if (strcmp(node->name(), "Value") == 0)
															#line 13883 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
								{
															#line 2360 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									value = node->value();
															#line 13887 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
								}
							}
						}
															#line 2363 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (!key.empty())
															#line 13893 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 2364 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							tags[key] = value;
															#line 13897 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
				}
			}
		}
															#line 2371 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<GetObjectTags_implActor*>(this)->SAV<std::map<std::string, std::string>>::futures) { (void)(tags); this->~GetObjectTags_implActorState(); static_cast<GetObjectTags_implActor*>(this)->destroy(); return 0; }
															#line 13905 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<GetObjectTags_implActor*>(this)->SAV< std::map<std::string, std::string> >::value()) std::map<std::string, std::string>(tags);
		this->~GetObjectTags_implActorState();
		static_cast<GetObjectTags_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 2332 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rapidxml::xml_document<> doc;
															#line 2333 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		doc.parse<rapidxml::parse_default>((char*)r->data.content.c_str());
															#line 2335 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::map<std::string, std::string> tags;
															#line 2338 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rapidxml::xml_node<>* tagging = doc.first_node();
															#line 2339 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for(;tagging && strcmp(tagging->name(), "Tagging") != 0;) {
															#line 2340 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			tagging = tagging->next_sibling();
															#line 13927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2343 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (tagging)
															#line 13931 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2345 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			rapidxml::xml_node<>* tagSet = tagging->first_node();
															#line 2346 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;tagSet && strcmp(tagSet->name(), "TagSet") != 0;) {
															#line 2347 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				tagSet = tagSet->next_sibling();
															#line 13939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2350 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (tagSet)
															#line 13943 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2352 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(rapidxml::xml_node<>* tag = tagSet->first_node();tag;tag = tag->next_sibling()) {
															#line 2353 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(tag->name(), "Tag") == 0)
															#line 13949 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 2354 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string key, value;
															#line 2356 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						for(rapidxml::xml_node<>* node = tag->first_node();node;node = node->next_sibling()) {
															#line 2357 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (strcmp(node->name(), "Key") == 0)
															#line 13957 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 2358 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								key = node->value();
															#line 13961 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
							}
							else
							{
															#line 2359 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								if (strcmp(node->name(), "Value") == 0)
															#line 13967 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
								{
															#line 2360 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									value = node->value();
															#line 13971 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
								}
							}
						}
															#line 2363 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (!key.empty())
															#line 13977 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 2364 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							tags[key] = value;
															#line 13981 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
				}
			}
		}
															#line 2371 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<GetObjectTags_implActor*>(this)->SAV<std::map<std::string, std::string>>::futures) { (void)(tags); this->~GetObjectTags_implActorState(); static_cast<GetObjectTags_implActor*>(this)->destroy(); return 0; }
															#line 13989 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<GetObjectTags_implActor*>(this)->SAV< std::map<std::string, std::string> >::value()) std::map<std::string, std::string>(tags);
		this->~GetObjectTags_implActorState();
		static_cast<GetObjectTags_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetObjectTags_implActor*>(this)->ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2326 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2328 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 14082 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via getObjectTags_impl()
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class GetObjectTags_implActor final : public Actor<std::map<std::string, std::string>>, public ActorCallback< GetObjectTags_implActor, 0, Void >, public ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<GetObjectTags_implActor>, public GetObjectTags_implActorState<GetObjectTags_implActor> {
															#line 14087 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<GetObjectTags_implActor>::operator new;
	using FastAllocated<GetObjectTags_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8115013185247424768UL, 3457416788358617344UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::map<std::string, std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetObjectTags_implActor, 0, Void >;
friend struct ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	GetObjectTags_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 14105 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::map<std::string, std::string>>(),
		   GetObjectTags_implActorState<GetObjectTags_implActor>(bstore, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9536500618548032768UL, 11160301285233402112UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getObjectTags_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetObjectTags_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<std::map<std::string, std::string>> getObjectTags_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object ) {
															#line 2321 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::map<std::string, std::string>>(new GetObjectTags_implActor(bstore, bucket, object));
															#line 14139 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2373 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::map<std::string, std::string>> S3BlobStoreEndpoint::getObjectTags(std::string const& bucket,
                                                                              std::string const& object) {
	return getObjectTags_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 14149 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2379()
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class FlowTestCase2379Actor>
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2379ActorState {
															#line 14156 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2379ActorState(UnitTestParameters const& params) 
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : params(params)
															#line 14163 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2379", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2379ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2379", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2380 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::Credentials creds{ "AKIAIOSFODNN7EXAMPLE", "wJalrXUtnFEMI/K7MDENG/bPxRfiCYEXAMPLEKEY", "" };
															#line 14178 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2383 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				S3BlobStoreEndpoint s3("s3.amazonaws.com", "443", "amazonaws", "proxy", "port", creds);
															#line 2384 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string verb("GET");
															#line 2385 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string resource("/test.txt");
															#line 2386 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				HTTP::Headers headers;
															#line 2387 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				headers["Host"] = "s3.amazonaws.com";
															#line 2388 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3.setV4AuthHeaders(verb, resource, headers, "20130524T000000Z", "20130524");
															#line 2389 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Authorization"] == "AWS4-HMAC-SHA256 Credential=AKIAIOSFODNN7EXAMPLE/20130524/amazonaws/s3/aws4_request, " "SignedHeaders=host;x-amz-content-sha256;x-amz-date, " "Signature=c6037f4b174f2019d02d7085a611cef8adfe1efe583e220954dc85d59cd31ba3");
															#line 2393 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["x-amz-date"] == "20130524T000000Z");
															#line 14196 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			{
															#line 2398 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				S3BlobStoreEndpoint s3("s3.amazonaws.com", "443", "amazonaws", "proxy", "port", creds);
															#line 2399 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string verb("GET");
															#line 2400 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string resource("/test/examplebucket?Action=DescribeRegions&Version=2013-10-15");
															#line 2401 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				HTTP::Headers headers;
															#line 2402 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				headers["Host"] = "s3.amazonaws.com";
															#line 2403 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3.setV4AuthHeaders(verb, resource, headers, "20130524T000000Z", "20130524");
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Authorization"] == "AWS4-HMAC-SHA256 Credential=AKIAIOSFODNN7EXAMPLE/20130524/amazonaws/s3/aws4_request, " "SignedHeaders=host;x-amz-content-sha256;x-amz-date, " "Signature=426f04e71e191fbc30096c306fe1b11ce8f026a7be374541862bbee320cce71c");
															#line 2408 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["x-amz-date"] == "20130524T000000Z");
															#line 14215 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
			{
															#line 2413 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				S3BlobStoreEndpoint s3("s3.us-west-2.amazonaws.com", "443", "us-west-2", "proxy", "port", creds);
															#line 2414 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string verb("POST");
															#line 2415 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string resource("/simple.json");
															#line 2416 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				HTTP::Headers headers;
															#line 2417 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				headers["Host"] = "s3.us-west-2.amazonaws.com";
															#line 2418 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				headers["Content-Type"] = "Application/x-amz-json-1.0";
															#line 2419 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3.setV4AuthHeaders(verb, resource, headers, "20130524T000000Z", "20130524");
															#line 2420 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Authorization"] == "AWS4-HMAC-SHA256 Credential=AKIAIOSFODNN7EXAMPLE/20130524/us-west-2/s3/aws4_request, " "SignedHeaders=content-type;host;x-amz-content-sha256;x-amz-date, " "Signature=cf095e36bed9cd3139c2e8b3e20c296a79d8540987711bf3a0d816b19ae00314");
															#line 2424 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["x-amz-date"] == "20130524T000000Z");
															#line 2425 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Host"] == "s3.us-west-2.amazonaws.com");
															#line 2426 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Content-Type"] == "Application/x-amz-json-1.0");
															#line 14240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<FlowTestCase2379Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2379ActorState(); static_cast<FlowTestCase2379Actor*>(this)->destroy(); return 0; }
															#line 14244 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<FlowTestCase2379Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2379ActorState();
			static_cast<FlowTestCase2379Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2379ActorState();
		static_cast<FlowTestCase2379Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnitTestParameters params;
															#line 14268 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2379()
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2379Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2379Actor>, public FlowTestCase2379ActorState<FlowTestCase2379Actor> {
															#line 14273 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2379Actor>::operator new;
	using FastAllocated<FlowTestCase2379Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12486752309311830016UL, 10874361554747227904UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2379Actor(UnitTestParameters const& params) 
															#line 14289 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2379ActorState<FlowTestCase2379Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2379", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12792862716113816064UL, 13815771760951638016UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2379Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2379");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2379", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
static Future<Void> flowTestCase2379( UnitTestParameters const& params ) {
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new FlowTestCase2379Actor(params));
															#line 14321 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2379, "/backup/s3/v4headers")

#line 2431 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

															#line 14327 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2432()
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class FlowTestCase2432Actor>
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2432ActorState {
															#line 14334 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2432ActorState(UnitTestParameters const& params) 
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : params(params)
															#line 14341 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2432", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2432ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2432", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2433 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string url = "blobstore://s3.us-west-2.amazonaws.com/resource_name?bucket=bucket_name&sa=1";
															#line 2435 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string resource;
															#line 2436 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string error;
															#line 2437 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::ParametersT parameters;
															#line 2438 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			Reference<S3BlobStoreEndpoint> s3 = S3BlobStoreEndpoint::fromString(url, {}, &resource, &error, &parameters);
															#line 2439 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT(s3->getRegion() == "us-west-2");
															#line 2441 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			url = "blobstore://s3.us-west-2.amazonaws.com/resource_name?bucket=bucket_name&sc=922337203685477580700";
															#line 14368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			try {
															#line 2443 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3 = S3BlobStoreEndpoint::fromString(url, {}, &resource, &error, &parameters);
															#line 2444 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(false);
															#line 14374 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2432ActorState();
		static_cast<FlowTestCase2432Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2449 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<FlowTestCase2432Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2432ActorState(); static_cast<FlowTestCase2432Actor*>(this)->destroy(); return 0; }
															#line 14403 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<FlowTestCase2432Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2432ActorState();
		static_cast<FlowTestCase2432Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2447 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT_EQ(e.code(), error_code_backup_invalid_url);
															#line 14416 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnitTestParameters params;
															#line 14442 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2432()
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2432Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2432Actor>, public FlowTestCase2432ActorState<FlowTestCase2432Actor> {
															#line 14447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2432Actor>::operator new;
	using FastAllocated<FlowTestCase2432Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17845483761453268992UL, 8451184455077786880UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2432Actor(UnitTestParameters const& params) 
															#line 14463 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2432ActorState<FlowTestCase2432Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2432", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8108605102170561024UL, 18279702463451747072UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2432Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2432");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2432", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
static Future<Void> flowTestCase2432( UnitTestParameters const& params ) {
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new FlowTestCase2432Actor(params));
															#line 14495 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2432, "/backup/s3/guess_region")

#line 2451 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"

															#line 14501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2452()
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class FlowTestCase2452Actor>
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2452ActorState {
															#line 14508 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2452ActorState(UnitTestParameters const& params) 
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : params(params)
															#line 14515 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2452", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2452ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2452", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2463 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string url = "blobstore://@test-bucket.s3.us-west-2.amazonaws.com/bulkload/path?bucket=test-bucket";
															#line 2464 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string resource;
															#line 2465 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string error;
															#line 2466 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::ParametersT parameters;
															#line 2467 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			Reference<S3BlobStoreEndpoint> s3 = S3BlobStoreEndpoint::fromString(url, {}, &resource, &error, &parameters);
															#line 2469 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT(s3.isValid());
															#line 2470 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT(resource == "bulkload/path");
															#line 2473 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string listResource = s3->constructResourcePath("test-bucket", "");
															#line 2474 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT(listResource == "");
															#line 2483 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			url = "blobstore://s3.us-west-2.amazonaws.com/resource_path?bucket=test-bucket";
															#line 2484 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			s3 = S3BlobStoreEndpoint::fromString(url, {}, &resource, &error, &parameters);
															#line 2485 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT(s3.isValid());
															#line 2486 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT(resource == "resource_path");
															#line 2489 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			listResource = s3->constructResourcePath("test-bucket", "");
															#line 2490 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT(listResource == "/test-bucket");
															#line 2492 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<FlowTestCase2452Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2452ActorState(); static_cast<FlowTestCase2452Actor*>(this)->destroy(); return 0; }
															#line 14560 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<FlowTestCase2452Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2452ActorState();
			static_cast<FlowTestCase2452Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2452ActorState();
		static_cast<FlowTestCase2452Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnitTestParameters params;
															#line 14584 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2452()
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2452Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2452Actor>, public FlowTestCase2452ActorState<FlowTestCase2452Actor> {
															#line 14589 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2452Actor>::operator new;
	using FastAllocated<FlowTestCase2452Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15286861569589403392UL, 10224410252062280704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2452Actor(UnitTestParameters const& params) 
															#line 14605 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2452ActorState<FlowTestCase2452Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2452", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2593523183904721664UL, 6362905900191486976UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2452Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2452");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2452", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
static Future<Void> flowTestCase2452( UnitTestParameters const& params ) {
															#line 2452 "/Users/ec2-user/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new FlowTestCase2452Actor(params));
															#line 14637 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/S3BlobStore.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2452, "/backup/s3/virtual_hosting_list_resource_path")

