#define POST_ACTOR_COMPILER 1
#line 1 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
/*
 * ManagementAPI.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cinttypes>
#include <cstddef>
#include <string>
#include <vector>

#include "fdbclient/BulkDumping.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/GenericManagementAPI.actor.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/RangeLock.h"
#include "flow/Error.h"
#include "fmt/format.h"
#include "fdbclient/Knobs.h"
#include "flow/Arena.h"
#include "fdbclient/ClusterConnectionMemoryRecord.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/ReadYourWrites.h"
#include "fdbclient/ManagementAPI.actor.h"

#include "fdbclient/SystemData.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/CoordinationInterface.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbrpc/simulator.h"
#include "fdbclient/StatusClient.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"
#include "fdbrpc/ReplicationPolicy.h"
#include "fdbrpc/Replication.h"
#include "fdbclient/Schemas.h"
#include "fdbrpc/SimulatorProcessInfo.h"

#include "flow/actorcompiler.h" // This must be the last #include.

bool isInteger(const std::string& s) {
	if (s.empty())
		return false;
	char* p;
	strtol(s.c_str(), &p, 10);
	return (*p == 0);
}

// Defines the mapping between configuration names (as exposed by fdbcli, buildConfiguration()) and actual configuration
// parameters
std::map<std::string, std::string> configForToken(std::string const& mode) {
	std::map<std::string, std::string> out;
	std::string p = configKeysPrefix.toString();

	if (mode == "new") {
		out[p + "initialized"] = "1";
		return out;
	}

	if (mode == "tss") {
		// Set temporary marker in config map to mark that this is a tss configuration and not a normal storage/log
		// configuration. A bit of a hack but reuses the parsing code nicely.
		out[p + "istss"] = "1";
		return out;
	}

	if (mode == "locked") {
		// Setting this key is interpreted as an instruction to use the normal version-stamp-based mechanism for locking
		// the database.
		out[databaseLockedKey.toString()] = deterministicRandom()->randomUniqueID().toString();
		return out;
	}

	size_t pos;

	// key:=value is unvalidated and unchecked
	pos = mode.find(":=");
	if (pos != std::string::npos) {
		out[p + mode.substr(0, pos)] = mode.substr(pos + 2);
		return out;
	}

	// key=value is constrained to a limited set of options and basic validation is performed
	pos = mode.find("=");
	if (pos != std::string::npos) {
		std::string key = mode.substr(0, pos);
		std::string value = mode.substr(pos + 1);

		if (key == "proxies" && isInteger(value)) {
			printf("Warning: Proxy role is being split into GRV Proxy and Commit Proxy, now prefer configuring "
			       "'grv_proxies' and 'commit_proxies' separately. Generally we should follow that 'commit_proxies'"
			       " is three times of 'grv_proxies' count and 'grv_proxies' should be not more than 4.\n");
			int proxiesCount = atoi(value.c_str());
			if (proxiesCount == -1) {
				proxiesCount = CLIENT_KNOBS->DEFAULT_AUTO_GRV_PROXIES + CLIENT_KNOBS->DEFAULT_AUTO_COMMIT_PROXIES;
				ASSERT_WE_THINK(proxiesCount >= 2);
			}

			if (proxiesCount < 2) {
				printf("Error: At least 2 proxies (1 GRV proxy and 1 Commit proxy) are required.\n");
				return out;
			}

			int grvProxyCount = std::max(1,
			                             std::min(CLIENT_KNOBS->DEFAULT_MAX_GRV_PROXIES,
			                                      proxiesCount / (CLIENT_KNOBS->DEFAULT_COMMIT_GRV_PROXIES_RATIO + 1)));
			int commitProxyCount = proxiesCount - grvProxyCount;
			ASSERT_WE_THINK(grvProxyCount >= 1 && commitProxyCount >= 1);

			out[p + "grv_proxies"] = std::to_string(grvProxyCount);
			out[p + "commit_proxies"] = std::to_string(commitProxyCount);
			printf("%d proxies are automatically converted into %d GRV proxies and %d Commit proxies.\n",
			       proxiesCount,
			       grvProxyCount,
			       commitProxyCount);

			TraceEvent("DatabaseConfigurationProxiesSpecified")
			    .detail("SpecifiedProxies", atoi(value.c_str()))
			    .detail("EffectiveSpecifiedProxies", proxiesCount)
			    .detail("ConvertedGrvProxies", grvProxyCount)
			    .detail("ConvertedCommitProxies", commitProxyCount);
		}

		if ((key == "logs" || key == "commit_proxies" || key == "grv_proxies" || key == "resolvers" ||
		     key == "remote_logs" || key == "log_routers" || key == "usable_regions" ||
		     key == "repopulate_anti_quorum" || key == "count") &&
		    isInteger(value)) {
			out[p + key] = value;
		}

		if (key == "regions") {
			json_spirit::mValue mv;
			json_spirit::read_string(value, mv);

			StatusObject regionObj;
			regionObj["regions"] = mv;
			out[p + key] =
			    BinaryWriter::toValue(regionObj, IncludeVersion(ProtocolVersion::withRegionConfiguration())).toString();
		}

		if (key == "perpetual_storage_wiggle" && isInteger(value)) {
			int ppWiggle = std::stoi(value);
			if (ppWiggle >= 2 || ppWiggle < 0) {
				printf("Error: Only 0 and 1 are valid values of perpetual_storage_wiggle at present.\n");
				return out;
			}
			out[p + key] = value;
		}
		if (key == "perpetual_storage_wiggle_locality") {
			if (!isValidPerpetualStorageWiggleLocality(value)) {
				printf("Error: perpetual_storage_wiggle_locality should be in <locality_key>:<locality_value> "
				       "format or enter 0 to disable the locality match for wiggling.\n");
				return out;
			}
			out[p + key] = value;
		}
		if (key == "storage_migration_type") {
			StorageMigrationType type;
			if (value == "disabled") {
				type = StorageMigrationType::DISABLED;
			} else if (value == "aggressive") {
				type = StorageMigrationType::AGGRESSIVE;
			} else if (value == "gradual") {
				type = StorageMigrationType::GRADUAL;
			} else {
				printf("Error: Only disabled|aggressive|gradual are valid for storage_migration_type.\n");
				return out;
			}
			out[p + key] = format("%d", type);
		}

		if (key == "tenant_mode") {
			TenantMode tenantMode;
			if (value == "disabled") {
				tenantMode = TenantMode::DISABLED;
			} else if (value == "optional_experimental") {
				tenantMode = TenantMode::OPTIONAL_TENANT;
			} else if (value == "required_experimental") {
				tenantMode = TenantMode::REQUIRED;
			} else {
				printf("Error: Only disabled|optional_experimental|required_experimental are valid for tenant_mode.\n");
				return out;
			}
			out[p + key] = format("%d", tenantMode);
		}

		if (key == "encryption_at_rest_mode") {
			EncryptionAtRestMode mode;
			if (value == "disabled") {
				mode = EncryptionAtRestMode::DISABLED;
			} else if (value == "domain_aware") {
				mode = EncryptionAtRestMode::DOMAIN_AWARE;
			} else if (value == "cluster_aware") {
				mode = EncryptionAtRestMode::CLUSTER_AWARE;
			} else {
				printf("Error: Only disabled|domain_aware|cluster_aware are valid for encryption_at_rest_mode.\n");
				return out;
			}
			out[p + key] = format("%d", mode);
		}

		if (key == "exclude") {
			int p = 0;
			while (p < value.size()) {
				int end = value.find_first_of(',', p);
				if (end == value.npos) {
					end = value.size();
				}
				auto addrRef = StringRef(value).substr(p, end - p);
				AddressExclusion addr = AddressExclusion::parse(addrRef);
				if (addr.isValid()) {
					out[encodeExcludedServersKey(addr)] = "";
				} else {
					printf("Error: invalid address format: %s\n", addrRef.toString().c_str());
				}
				p = end + 1;
			}
		}

		if (key == "storage_engine" || key == "log_engine" || key == "perpetual_storage_wiggle_engine") {
			StringRef s = value;

			// Parse as engine_name[:p=v]... to handle future storage engine params
			Value engine = s.eat(":");
			std::map<Key, Value> params;
			while (!s.empty()) {
				params[s.eat("=")] = s.eat(":");
			}

			try {
				out[p + key] = format("%d", KeyValueStoreType::fromString(engine.toString()).storeType());
			} catch (Error& e) {
				printf("Error: Invalid value for %s (%s): %s\n", key.c_str(), value.c_str(), e.what());
			}
			return out;
		}

		return out;
	}

	Optional<KeyValueStoreType> logType;
	Optional<KeyValueStoreType> storeType;

	// These are legacy shorthand commands to set a specific log engine and storage engine
	// based only on the storage engine name.  Most of them assume SQLite should be the
	// log engine.
	if (mode == "ssd-1") {
		logType = KeyValueStoreType::SSD_BTREE_V1;
		storeType = KeyValueStoreType::SSD_BTREE_V1;
	} else if (mode == "ssd" || mode == "ssd-2") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::SSD_BTREE_V2;
	} else if (mode == "ssd-redwood-1") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::SSD_REDWOOD_V1;
	} else if (mode == "ssd-rocksdb-v1") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::SSD_ROCKSDB_V1;
	} else if (mode == "ssd-sharded-rocksdb") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::SSD_SHARDED_ROCKSDB;
	} else if (mode == "memory" || mode == "memory-2") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::MEMORY;
	} else if (mode == "memory-1") {
		logType = KeyValueStoreType::MEMORY;
		storeType = KeyValueStoreType::MEMORY;
	} else if (mode == "memory-radixtree" || mode == "memory-radixtree-beta") {
		logType = KeyValueStoreType::SSD_BTREE_V2;
		storeType = KeyValueStoreType::MEMORY_RADIXTREE;
	}
	// Add any new store types to fdbserver/workloads/ConfigureDatabase, too

	if (storeType.present()) {
		out[p + "log_engine"] = format("%d", logType.get().storeType());
		out[p + "storage_engine"] = format("%d", storeType.get().storeType());
		return out;
	}

	std::string redundancy, log_replicas;
	Reference<IReplicationPolicy> storagePolicy;
	Reference<IReplicationPolicy> tLogPolicy;

	bool redundancySpecified = true;
	if (mode == "single") {
		redundancy = "1";
		log_replicas = "1";
		storagePolicy = tLogPolicy = Reference<IReplicationPolicy>(new PolicyOne());

	} else if (mode == "double" || mode == "fast_recovery_double") {
		redundancy = "2";
		log_replicas = "2";
		storagePolicy = tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
	} else if (mode == "triple" || mode == "fast_recovery_triple") {
		redundancy = "3";
		log_replicas = "3";
		storagePolicy = tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(3, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
	} else if (mode == "three_datacenter" || mode == "multi_dc") {
		redundancy = "6";
		log_replicas = "4";
		storagePolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(3,
		                     "dcid",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
		tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "dcid",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else if (mode == "three_datacenter_fallback") {
		redundancy = "4";
		log_replicas = "4";
		storagePolicy = tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "dcid",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else if (mode == "three_data_hall") {
		redundancy = "3";
		log_replicas = "4";
		storagePolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(3, "data_hall", Reference<IReplicationPolicy>(new PolicyOne())));
		tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "data_hall",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else if (mode == "three_data_hall_fallback") {
		redundancy = "2";
		log_replicas = "4";
		storagePolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2, "data_hall", Reference<IReplicationPolicy>(new PolicyOne())));
		tLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "data_hall",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else
		redundancySpecified = false;
	if (redundancySpecified) {
		out[p + "storage_replicas"] = redundancy;
		out[p + "log_replicas"] = log_replicas;
		out[p + "log_anti_quorum"] = "0";

		BinaryWriter policyWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, storagePolicy);
		out[p + "storage_replication_policy"] = policyWriter.toValue().toString();

		policyWriter = BinaryWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, tLogPolicy);
		out[p + "log_replication_policy"] = policyWriter.toValue().toString();
		return out;
	}

	std::string remote_redundancy, remote_log_replicas;
	Reference<IReplicationPolicy> remoteTLogPolicy;
	bool remoteRedundancySpecified = true;
	if (mode == "remote_default") {
		remote_redundancy = "0";
		remote_log_replicas = "0";
		remoteTLogPolicy = Reference<IReplicationPolicy>();
	} else if (mode == "remote_single") {
		remote_redundancy = "1";
		remote_log_replicas = "1";
		remoteTLogPolicy = Reference<IReplicationPolicy>(new PolicyOne());
	} else if (mode == "remote_double") {
		remote_redundancy = "2";
		remote_log_replicas = "2";
		remoteTLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
	} else if (mode == "remote_triple") {
		remote_redundancy = "3";
		remote_log_replicas = "3";
		remoteTLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(3, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
	} else if (mode == "remote_three_data_hall") { // FIXME: not tested in simulation
		remote_redundancy = "3";
		remote_log_replicas = "4";
		remoteTLogPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(2,
		                     "data_hall",
		                     Reference<IReplicationPolicy>(
		                         new PolicyAcross(2, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())))));
	} else
		remoteRedundancySpecified = false;
	if (remoteRedundancySpecified) {
		out[p + "remote_log_replicas"] = remote_log_replicas;

		BinaryWriter policyWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, remoteTLogPolicy);
		out[p + "remote_log_policy"] = policyWriter.toValue().toString();
		return out;
	}

	return out;
}

ConfigurationResult buildConfiguration(std::vector<StringRef> const& modeTokens,
                                       std::map<std::string, std::string>& outConf) {
	for (auto it : modeTokens) {
		std::string mode = it.toString();
		auto m = configForToken(mode);
		if (!m.size()) {
			TraceEvent(SevWarnAlways, "UnknownOption").detail("Option", mode);
			return ConfigurationResult::UNKNOWN_OPTION;
		}

		for (auto t = m.begin(); t != m.end(); ++t) {
			if (outConf.count(t->first)) {
				TraceEvent(SevWarnAlways, "ConflictingOption")
				    .detail("Option", t->first)
				    .detail("Value", t->second)
				    .detail("ExistingValue", outConf[t->first]);
				return ConfigurationResult::CONFLICTING_OPTIONS;
			}
			outConf[t->first] = t->second;
		}
	}
	auto p = configKeysPrefix.toString();
	if (!outConf.count(p + "storage_replication_policy") && outConf.count(p + "storage_replicas")) {
		int storageCount = stoi(outConf[p + "storage_replicas"]);
		Reference<IReplicationPolicy> storagePolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(storageCount, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
		BinaryWriter policyWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, storagePolicy);
		outConf[p + "storage_replication_policy"] = policyWriter.toValue().toString();
	}

	if (!outConf.count(p + "log_replication_policy") && outConf.count(p + "log_replicas")) {
		int logCount = stoi(outConf[p + "log_replicas"]);
		Reference<IReplicationPolicy> logPolicy = Reference<IReplicationPolicy>(
		    new PolicyAcross(logCount, "zoneid", Reference<IReplicationPolicy>(new PolicyOne())));
		BinaryWriter policyWriter(IncludeVersion(ProtocolVersion::withReplicationPolicy()));
		serializeReplicationPolicy(policyWriter, logPolicy);
		outConf[p + "log_replication_policy"] = policyWriter.toValue().toString();
	}
	if (outConf.count(p + "istss")) {
		// redo config parameters to be tss config instead of normal config

		// save param values from parsing as a normal config
		bool isNew = outConf.count(p + "initialized");
		Optional<std::string> count;
		Optional<std::string> storageEngine;
		if (outConf.count(p + "count")) {
			count = Optional<std::string>(outConf[p + "count"]);
		}
		if (outConf.count(p + "storage_engine")) {
			storageEngine = Optional<std::string>(outConf[p + "storage_engine"]);
		}

		// A new tss setup must have count + storage engine. An adjustment must have at least one.
		if ((isNew && (!count.present() || !storageEngine.present())) ||
		    (!isNew && !count.present() && !storageEngine.present())) {
			return ConfigurationResult::INCOMPLETE_CONFIGURATION;
		}

		// clear map and only reset tss parameters
		outConf.clear();
		if (count.present()) {
			outConf[p + "tss_count"] = count.get();
		}
		if (storageEngine.present()) {
			outConf[p + "tss_storage_engine"] = storageEngine.get();
		}
	}
	return ConfigurationResult::SUCCESS;
}

ConfigurationResult buildConfiguration(std::string const& configMode, std::map<std::string, std::string>& outConf) {
	std::vector<StringRef> modes;

	int p = 0;
	while (p < configMode.size()) {
		int end = configMode.find_first_of(' ', p);
		if (end == configMode.npos)
			end = configMode.size();
		modes.push_back(StringRef(configMode).substr(p, end - p));
		p = end + 1;
	}

	return buildConfiguration(modes, outConf);
}

bool isCompleteConfiguration(std::map<std::string, std::string> const& options) {
	std::string p = configKeysPrefix.toString();

	return options.count(p + "log_replicas") == 1 && options.count(p + "log_anti_quorum") == 1 &&
	       options.count(p + "storage_replicas") == 1 && options.count(p + "log_engine") == 1 &&
	       options.count(p + "storage_engine") == 1;
}

															#line 512 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via disableBackupWorker()
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class DisableBackupWorkerActor>
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class DisableBackupWorkerActorState {
															#line 519 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	DisableBackupWorkerActorState(Database const& cx) 
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx)
															#line 526 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("disableBackupWorker", reinterpret_cast<unsigned long>(this));

	}
	~DisableBackupWorkerActorState() 
	{
		fdb_probe_actor_destroy("disableBackupWorker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 511 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<DatabaseConfiguration> __when_expr_0 = getDatabaseConfiguration(cx);
															#line 511 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 543 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 1;
															#line 511 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*>(static_cast<DisableBackupWorkerActor*>(this)));
															#line 548 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DisableBackupWorkerActorState();
		static_cast<DisableBackupWorkerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration const& configuration,int loopDepth) 
	{
															#line 512 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!configuration.backupWorkerEnabled)
															#line 571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 513 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerAlreadyDisabled");
															#line 514 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<DisableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DisableBackupWorkerActorState(); static_cast<DisableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 577 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<DisableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DisableBackupWorkerActorState();
			static_cast<DisableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 516 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<ConfigurationResult> __when_expr_1 = ManagementAPI::changeConfig(cx.getReference(), "backup_worker_enabled:=0", true);
															#line 516 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 2;
															#line 516 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*>(static_cast<DisableBackupWorkerActor*>(this)));
															#line 592 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration && configuration,int loopDepth) 
	{
															#line 512 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!configuration.backupWorkerEnabled)
															#line 601 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 513 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerAlreadyDisabled");
															#line 514 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<DisableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DisableBackupWorkerActorState(); static_cast<DisableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 607 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<DisableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DisableBackupWorkerActorState();
			static_cast<DisableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 516 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<ConfigurationResult> __when_expr_1 = ManagementAPI::changeConfig(cx.getReference(), "backup_worker_enabled:=0", true);
															#line 516 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 617 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 2;
															#line 516 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*>(static_cast<DisableBackupWorkerActor*>(this)));
															#line 622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration const& configuration,int loopDepth) 
	{
		loopDepth = a_body1cont1(configuration, loopDepth);

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration && configuration,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(configuration), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state > 0) static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<DisableBackupWorkerActor*>(this)->ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(ConfigurationResult const& res,int loopDepth) 
	{
															#line 517 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (res != ConfigurationResult::SUCCESS)
															#line 706 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 518 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerDisableFailed").detail("Result", res);
															#line 519 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(operation_failed(), loopDepth);
															#line 712 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 521 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<DisableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DisableBackupWorkerActorState(); static_cast<DisableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 716 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<DisableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DisableBackupWorkerActorState();
		static_cast<DisableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(ConfigurationResult && res,int loopDepth) 
	{
															#line 517 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (res != ConfigurationResult::SUCCESS)
															#line 728 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 518 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerDisableFailed").detail("Result", res);
															#line 519 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(operation_failed(), loopDepth);
															#line 734 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 521 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<DisableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DisableBackupWorkerActorState(); static_cast<DisableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 738 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<DisableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DisableBackupWorkerActorState();
		static_cast<DisableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(ConfigurationResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont2(res, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(ConfigurationResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state > 0) static_cast<DisableBackupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<DisableBackupWorkerActor*>(this)->ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >::remove();

	}
	void a_callback_fire(ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*,ConfigurationResult const& value) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*,ConfigurationResult && value) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*,Error err) 
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 823 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via disableBackupWorker()
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class DisableBackupWorkerActor final : public Actor<Void>, public ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >, public ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >, public FastAllocated<DisableBackupWorkerActor>, public DisableBackupWorkerActorState<DisableBackupWorkerActor> {
															#line 828 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<DisableBackupWorkerActor>::operator new;
	using FastAllocated<DisableBackupWorkerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(19507444347246080UL, 12558165981376123136UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >;
friend struct ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >;
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	DisableBackupWorkerActor(Database const& cx) 
															#line 846 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   DisableBackupWorkerActorState<DisableBackupWorkerActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("disableBackupWorker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(814777889414831360UL, 5570830813057092096UL);
		ActorExecutionContextHelper __helper(static_cast<DisableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("disableBackupWorker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("disableBackupWorker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DisableBackupWorkerActor, 0, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DisableBackupWorkerActor, 1, ConfigurationResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> disableBackupWorker( Database const& cx ) {
															#line 510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new DisableBackupWorkerActor(cx));
															#line 880 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 523 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 885 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via enableBackupWorker()
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class EnableBackupWorkerActor>
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class EnableBackupWorkerActorState {
															#line 892 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	EnableBackupWorkerActorState(Database const& cx) 
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx)
															#line 899 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("enableBackupWorker", reinterpret_cast<unsigned long>(this));

	}
	~EnableBackupWorkerActorState() 
	{
		fdb_probe_actor_destroy("enableBackupWorker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 525 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<DatabaseConfiguration> __when_expr_0 = getDatabaseConfiguration(cx);
															#line 525 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 916 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 1;
															#line 525 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*>(static_cast<EnableBackupWorkerActor*>(this)));
															#line 921 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EnableBackupWorkerActorState();
		static_cast<EnableBackupWorkerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration const& configuration,int loopDepth) 
	{
															#line 526 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (configuration.backupWorkerEnabled)
															#line 944 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 527 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerAlreadyEnabled");
															#line 528 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<EnableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableBackupWorkerActorState(); static_cast<EnableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 950 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<EnableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~EnableBackupWorkerActorState();
			static_cast<EnableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<ConfigurationResult> __when_expr_1 = ManagementAPI::changeConfig(cx.getReference(), "backup_worker_enabled:=1", true);
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 960 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 2;
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*>(static_cast<EnableBackupWorkerActor*>(this)));
															#line 965 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration && configuration,int loopDepth) 
	{
															#line 526 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (configuration.backupWorkerEnabled)
															#line 974 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 527 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerAlreadyEnabled");
															#line 528 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<EnableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableBackupWorkerActorState(); static_cast<EnableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 980 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<EnableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~EnableBackupWorkerActorState();
			static_cast<EnableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<ConfigurationResult> __when_expr_1 = ManagementAPI::changeConfig(cx.getReference(), "backup_worker_enabled:=1", true);
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 990 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 2;
															#line 530 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*>(static_cast<EnableBackupWorkerActor*>(this)));
															#line 995 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration const& configuration,int loopDepth) 
	{
		loopDepth = a_body1cont1(configuration, loopDepth);

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration && configuration,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(configuration), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state > 0) static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<EnableBackupWorkerActor*>(this)->ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(ConfigurationResult const& res,int loopDepth) 
	{
															#line 531 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (res != ConfigurationResult::SUCCESS)
															#line 1079 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 532 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerEnableFailed").detail("Result", res);
															#line 533 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(operation_failed(), loopDepth);
															#line 1085 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 535 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<EnableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableBackupWorkerActorState(); static_cast<EnableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 1089 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<EnableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EnableBackupWorkerActorState();
		static_cast<EnableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(ConfigurationResult && res,int loopDepth) 
	{
															#line 531 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (res != ConfigurationResult::SUCCESS)
															#line 1101 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 532 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("BackupWorkerEnableFailed").detail("Result", res);
															#line 533 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(operation_failed(), loopDepth);
															#line 1107 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 535 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<EnableBackupWorkerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableBackupWorkerActorState(); static_cast<EnableBackupWorkerActor*>(this)->destroy(); return 0; }
															#line 1111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<EnableBackupWorkerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EnableBackupWorkerActorState();
		static_cast<EnableBackupWorkerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(ConfigurationResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont2(res, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(ConfigurationResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state > 0) static_cast<EnableBackupWorkerActor*>(this)->actor_wait_state = 0;
		static_cast<EnableBackupWorkerActor*>(this)->ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >::remove();

	}
	void a_callback_fire(ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*,ConfigurationResult const& value) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*,ConfigurationResult && value) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*,Error err) 
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1196 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via enableBackupWorker()
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class EnableBackupWorkerActor final : public Actor<Void>, public ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >, public ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >, public FastAllocated<EnableBackupWorkerActor>, public EnableBackupWorkerActorState<EnableBackupWorkerActor> {
															#line 1201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<EnableBackupWorkerActor>::operator new;
	using FastAllocated<EnableBackupWorkerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3202402914084803072UL, 9420687037872217856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >;
friend struct ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >;
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	EnableBackupWorkerActor(Database const& cx) 
															#line 1219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   EnableBackupWorkerActorState<EnableBackupWorkerActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("enableBackupWorker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12529125006175537920UL, 10565899026445989632UL);
		ActorExecutionContextHelper __helper(static_cast<EnableBackupWorkerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("enableBackupWorker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("enableBackupWorker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< EnableBackupWorkerActor, 0, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< EnableBackupWorkerActor, 1, ConfigurationResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> enableBackupWorker( Database const& cx ) {
															#line 524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new EnableBackupWorkerActor(cx));
															#line 1253 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 537 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

/*
    - Validates encryption and tenant mode configurations
    - During cluster creation (configure new) we allow the following:
        - If encryption mode is disabled/cluster_aware then any tenant mode is allowed
        - If the encryption mode is domain_aware then the only allowed tenant mode is required
    - During cluster configuration changes the following is allowed:
        - Encryption mode cannot be changed (can only be set during creation)
        - If the encryption mode is disabled/cluster_aware then any tenant mode changes are allowed
        - If the encryption mode is domain_aware then tenant mode changes are not allowed (as the only supported mode is
          required)
*/
bool isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration> oldConfiguration,
                                       std::map<std::string, std::string> newConfig,
                                       bool creating) {
	EncryptionAtRestMode encryptMode;
	TenantMode tenantMode;
	if (creating) {
		if (newConfig.count(encryptionAtRestModeConfKey.toString()) != 0) {
			encryptMode = EncryptionAtRestMode::fromValueRef(
			    ValueRef(newConfig.find(encryptionAtRestModeConfKey.toString())->second));
			// check if the tenant mode is being set during configure new (otherwise assume tenants are disabled)
			if (newConfig.count(tenantModeConfKey.toString()) != 0) {
				tenantMode = TenantMode::fromValue(ValueRef(newConfig.find(tenantModeConfKey.toString())->second));
			}
		}
	} else {
		ASSERT(oldConfiguration.present());
		encryptMode = oldConfiguration.get().encryptionAtRestMode;
		if (newConfig.count(tenantModeConfKey.toString()) != 0) {
			tenantMode = TenantMode::fromValue(ValueRef(newConfig.find(tenantModeConfKey.toString())->second));
		} else {
			// Tenant mode and encryption mode didn't change
			return true;
		}
	}
	TraceEvent(SevDebug, "EncryptAndTenantModes")
	    .detail("EncryptMode", encryptMode.toString())
	    .detail("TenantMode", tenantMode.toString());

	if (encryptMode.mode == EncryptionAtRestMode::DOMAIN_AWARE && tenantMode != TenantMode::REQUIRED) {
		// For domain aware encryption only the required tenant mode is currently supported
		TraceEvent(SevWarnAlways, "InvalidEncryptAndTenantConfiguration")
		    .detail("EncryptMode", encryptMode.toString())
		    .detail("TenantMode", tenantMode.toString());
		return false;
	}

	return true;
}

bool isTenantModeModeConfigValid(DatabaseConfiguration oldConfiguration, DatabaseConfiguration newConfiguration) {
	TenantMode oldTenantMode = oldConfiguration.tenantMode;
	TenantMode newTenantMode = newConfiguration.tenantMode;
	TraceEvent(SevDebug, "TenantModes")
	    .detail("OldTenantMode", oldTenantMode.toString())
	    .detail("NewTenantMode", newTenantMode.toString());
	if (oldTenantMode != TenantMode::REQUIRED && newTenantMode == TenantMode::REQUIRED) {
		// TODO: Changing from optional/disabled to required tenant mode should be allowed if there is no non-tenant
		// data present
		TraceEvent(SevWarnAlways, "InvalidTenantConfiguration")
		    .detail("OldTenantMode", oldTenantMode.toString())
		    .detail("NewTenantMode", newTenantMode.toString());
		return false;
	}
	return true;
}

															#line 1325 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase605()
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class FlowTestCase605Actor>
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class FlowTestCase605ActorState {
															#line 1332 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	FlowTestCase605ActorState(UnitTestParameters const& params) 
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : params(params)
															#line 1339 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase605", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase605ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase605", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 606 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			DatabaseConfiguration oldConfig;
															#line 607 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			DatabaseConfiguration newConfig;
															#line 608 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::vector<TenantMode> tenantModes = { TenantMode::DISABLED, TenantMode::OPTIONAL_TENANT, TenantMode::REQUIRED };
															#line 610 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = TenantMode::REQUIRED;
															#line 611 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.tenantMode = deterministicRandom()->randomChoice(tenantModes);
															#line 612 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isTenantModeModeConfigValid(oldConfig, newConfig));
															#line 614 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT;
															#line 615 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.tenantMode = deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT;
															#line 616 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isTenantModeModeConfigValid(oldConfig, newConfig));
															#line 618 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT;
															#line 619 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.tenantMode = TenantMode::REQUIRED;
															#line 620 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(!isTenantModeModeConfigValid(oldConfig, newConfig));
															#line 622 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<FlowTestCase605Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase605ActorState(); static_cast<FlowTestCase605Actor*>(this)->destroy(); return 0; }
															#line 1378 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<FlowTestCase605Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase605ActorState();
			static_cast<FlowTestCase605Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase605ActorState();
		static_cast<FlowTestCase605Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	UnitTestParameters params;
															#line 1402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase605()
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class FlowTestCase605Actor final : public Actor<Void>, public FastAllocated<FlowTestCase605Actor>, public FlowTestCase605ActorState<FlowTestCase605Actor> {
															#line 1407 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase605Actor>::operator new;
	using FastAllocated<FlowTestCase605Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(552434248270049280UL, 15079095694060191232UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	FlowTestCase605Actor(UnitTestParameters const& params) 
															#line 1423 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase605ActorState<FlowTestCase605Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase605", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2598018080392104704UL, 15836946076326249728UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase605Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase605");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase605", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
static Future<Void> flowTestCase605( UnitTestParameters const& params ) {
															#line 605 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new FlowTestCase605Actor(params));
															#line 1455 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase605, "/ManagementAPI/ChangeConfig/TenantMode")

#line 624 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// unit test for changing encryption/tenant mode config options
															#line 1462 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase626()
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class FlowTestCase626Actor>
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class FlowTestCase626ActorState {
															#line 1469 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	FlowTestCase626ActorState(UnitTestParameters const& params) 
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : params(params)
															#line 1476 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase626", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase626ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase626", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 627 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::map<std::string, std::string> newConfig;
															#line 628 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string encryptModeKey = encryptionAtRestModeConfKey.toString();
															#line 629 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string tenantModeKey = tenantModeConfKey.toString();
															#line 630 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::vector<TenantMode> tenantModes = { TenantMode::DISABLED, TenantMode::OPTIONAL_TENANT, TenantMode::REQUIRED };
															#line 631 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::vector<EncryptionAtRestMode> encryptionModes = { EncryptionAtRestMode::DISABLED, EncryptionAtRestMode::CLUSTER_AWARE, EncryptionAtRestMode::DOMAIN_AWARE };
															#line 637 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::DISABLED);
															#line 638 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 639 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::CLUSTER_AWARE);
															#line 643 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 644 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 647 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::DOMAIN_AWARE);
															#line 648 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT);
															#line 650 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(!isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 651 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(TenantMode::REQUIRED);
															#line 652 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 655 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.erase(encryptModeKey);
															#line 656 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 657 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 660 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.erase(tenantModeKey);
															#line 661 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::DOMAIN_AWARE);
															#line 662 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(!isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 663 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[encryptModeKey] = std::to_string(EncryptionAtRestMode::CLUSTER_AWARE);
															#line 664 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(Optional<DatabaseConfiguration>(), newConfig, true));
															#line 667 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			DatabaseConfiguration oldConfig;
															#line 670 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.encryptionAtRestMode = EncryptionAtRestMode::DISABLED;
															#line 671 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->randomChoice(tenantModes);
															#line 672 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 673 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 676 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.encryptionAtRestMode = EncryptionAtRestMode::DOMAIN_AWARE;
															#line 677 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = TenantMode::REQUIRED;
															#line 678 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->coinflip() ? TenantMode::DISABLED : TenantMode::OPTIONAL_TENANT);
															#line 680 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(!isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 681 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(TenantMode::REQUIRED);
															#line 682 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 685 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.encryptionAtRestMode = EncryptionAtRestMode::CLUSTER_AWARE;
															#line 687 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->randomChoice(tenantModes);
															#line 688 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig[tenantModeKey] = std::to_string(deterministicRandom()->randomChoice(tenantModes));
															#line 689 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 692 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newConfig.erase(tenantModeKey);
															#line 693 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.tenantMode = deterministicRandom()->randomChoice(tenantModes);
															#line 694 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldConfig.encryptionAtRestMode = deterministicRandom()->randomChoice(encryptionModes);
															#line 695 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(isEncryptionAtRestModeConfigValid(oldConfig, newConfig, false));
															#line 697 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<FlowTestCase626Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase626ActorState(); static_cast<FlowTestCase626Actor*>(this)->destroy(); return 0; }
															#line 1577 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<FlowTestCase626Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase626ActorState();
			static_cast<FlowTestCase626Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase626ActorState();
		static_cast<FlowTestCase626Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	UnitTestParameters params;
															#line 1601 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase626()
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class FlowTestCase626Actor final : public Actor<Void>, public FastAllocated<FlowTestCase626Actor>, public FlowTestCase626ActorState<FlowTestCase626Actor> {
															#line 1606 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase626Actor>::operator new;
	using FastAllocated<FlowTestCase626Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6531807777174181632UL, 4294557709600756736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	FlowTestCase626Actor(UnitTestParameters const& params) 
															#line 1622 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase626ActorState<FlowTestCase626Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase626", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14617865719834517504UL, 14913502343025108224UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase626Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase626");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase626", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
static Future<Void> flowTestCase626( UnitTestParameters const& params ) {
															#line 626 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new FlowTestCase626Actor(params));
															#line 1654 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase626, "/ManagementAPI/ChangeConfig/TenantAndEncryptMode")

#line 699 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 1660 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getDatabaseConfiguration()
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetDatabaseConfigurationActor>
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDatabaseConfigurationActorState {
															#line 1667 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDatabaseConfigurationActorState(Transaction* const& tr,bool const& useSystemPriority) 
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   useSystemPriority(useSystemPriority)
															#line 1676 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this));

	}
	~GetDatabaseConfigurationActorState() 
	{
		fdb_probe_actor_destroy("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 701 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (useSystemPriority)
															#line 1691 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 702 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1695 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 704 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 705 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(configKeys, CLIENT_KNOBS->TOO_MANY);
															#line 706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDatabaseConfigurationActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1705 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetDatabaseConfigurationActor*>(this)->actor_wait_state = 1;
															#line 706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*>(static_cast<GetDatabaseConfigurationActor*>(this)));
															#line 1710 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetDatabaseConfigurationActorState();
		static_cast<GetDatabaseConfigurationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& res,int loopDepth) 
	{
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(res.size() < CLIENT_KNOBS->TOO_MANY);
															#line 708 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		DatabaseConfiguration config;
															#line 709 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		config.fromKeyValues((VectorRef<KeyValueRef>)res);
															#line 710 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDatabaseConfigurationActor*>(this)->SAV<DatabaseConfiguration>::futures) { (void)(config); this->~GetDatabaseConfigurationActorState(); static_cast<GetDatabaseConfigurationActor*>(this)->destroy(); return 0; }
															#line 1739 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDatabaseConfigurationActor*>(this)->SAV< DatabaseConfiguration >::value()) DatabaseConfiguration(config);
		this->~GetDatabaseConfigurationActorState();
		static_cast<GetDatabaseConfigurationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult && res,int loopDepth) 
	{
															#line 707 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(res.size() < CLIENT_KNOBS->TOO_MANY);
															#line 708 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		DatabaseConfiguration config;
															#line 709 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		config.fromKeyValues((VectorRef<KeyValueRef>)res);
															#line 710 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDatabaseConfigurationActor*>(this)->SAV<DatabaseConfiguration>::futures) { (void)(config); this->~GetDatabaseConfigurationActorState(); static_cast<GetDatabaseConfigurationActor*>(this)->destroy(); return 0; }
															#line 1757 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDatabaseConfigurationActor*>(this)->SAV< DatabaseConfiguration >::value()) DatabaseConfiguration(config);
		this->~GetDatabaseConfigurationActorState();
		static_cast<GetDatabaseConfigurationActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont1(res, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetDatabaseConfigurationActor*>(this)->actor_wait_state > 0) static_cast<GetDatabaseConfigurationActor*>(this)->actor_wait_state = 0;
		static_cast<GetDatabaseConfigurationActor*>(this)->ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool useSystemPriority;
															#line 1844 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getDatabaseConfiguration()
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDatabaseConfigurationActor final : public Actor<DatabaseConfiguration>, public ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >, public FastAllocated<GetDatabaseConfigurationActor>, public GetDatabaseConfigurationActorState<GetDatabaseConfigurationActor> {
															#line 1849 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetDatabaseConfigurationActor>::operator new;
	using FastAllocated<GetDatabaseConfigurationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5548653345701468160UL, 1097306476174464000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<DatabaseConfiguration>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >;
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDatabaseConfigurationActor(Transaction* const& tr,bool const& useSystemPriority) 
															#line 1866 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<DatabaseConfiguration>(),
		   GetDatabaseConfigurationActorState<GetDatabaseConfigurationActor>(tr, useSystemPriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17873082326995364352UL, 2518798019749376UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getDatabaseConfiguration");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetDatabaseConfigurationActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<DatabaseConfiguration> getDatabaseConfiguration( Transaction* const& tr, bool const& useSystemPriority ) {
															#line 700 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<DatabaseConfiguration>(new GetDatabaseConfigurationActor(tr, useSystemPriority));
															#line 1899 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 712 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 1904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getDatabaseConfiguration()
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetDatabaseConfigurationActor1>
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDatabaseConfigurationActor1State {
															#line 1911 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDatabaseConfigurationActor1State(Database const& cx,bool const& useSystemPriority) 
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   useSystemPriority(useSystemPriority),
															#line 714 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 1922 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this));

	}
	~GetDatabaseConfigurationActor1State() 
	{
		fdb_probe_actor_destroy("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 715 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 1937 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetDatabaseConfigurationActor1State();
		static_cast<GetDatabaseConfigurationActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<DatabaseConfiguration> __when_expr_0 = getDatabaseConfiguration(&tr, useSystemPriority);
															#line 717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state = 1;
															#line 717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*>(static_cast<GetDatabaseConfigurationActor1*>(this)));
															#line 1975 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 720 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 720 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1999 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state = 2;
															#line 720 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*>(static_cast<GetDatabaseConfigurationActor1*>(this)));
															#line 2004 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(DatabaseConfiguration const& config,int loopDepth) 
	{
															#line 718 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDatabaseConfigurationActor1*>(this)->SAV<DatabaseConfiguration>::futures) { (void)(config); this->~GetDatabaseConfigurationActor1State(); static_cast<GetDatabaseConfigurationActor1*>(this)->destroy(); return 0; }
															#line 2019 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDatabaseConfigurationActor1*>(this)->SAV< DatabaseConfiguration >::value()) DatabaseConfiguration(config);
		this->~GetDatabaseConfigurationActor1State();
		static_cast<GetDatabaseConfigurationActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(DatabaseConfiguration && config,int loopDepth) 
	{
															#line 718 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDatabaseConfigurationActor1*>(this)->SAV<DatabaseConfiguration>::futures) { (void)(config); this->~GetDatabaseConfigurationActor1State(); static_cast<GetDatabaseConfigurationActor1*>(this)->destroy(); return 0; }
															#line 2031 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDatabaseConfigurationActor1*>(this)->SAV< DatabaseConfiguration >::value()) DatabaseConfiguration(config);
		this->~GetDatabaseConfigurationActor1State();
		static_cast<GetDatabaseConfigurationActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(DatabaseConfiguration const& config,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(config, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(DatabaseConfiguration && config,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(config), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state > 0) static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state = 0;
		static_cast<GetDatabaseConfigurationActor1*>(this)->ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state > 0) static_cast<GetDatabaseConfigurationActor1*>(this)->actor_wait_state = 0;
		static_cast<GetDatabaseConfigurationActor1*>(this)->ActorCallback< GetDatabaseConfigurationActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool useSystemPriority;
															#line 714 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 2207 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getDatabaseConfiguration()
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDatabaseConfigurationActor1 final : public Actor<DatabaseConfiguration>, public ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >, public ActorCallback< GetDatabaseConfigurationActor1, 1, Void >, public FastAllocated<GetDatabaseConfigurationActor1>, public GetDatabaseConfigurationActor1State<GetDatabaseConfigurationActor1> {
															#line 2212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetDatabaseConfigurationActor1>::operator new;
	using FastAllocated<GetDatabaseConfigurationActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5548653345701468160UL, 1097306476174464000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<DatabaseConfiguration>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >;
friend struct ActorCallback< GetDatabaseConfigurationActor1, 1, Void >;
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDatabaseConfigurationActor1(Database const& cx,bool const& useSystemPriority) 
															#line 2230 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<DatabaseConfiguration>(),
		   GetDatabaseConfigurationActor1State<GetDatabaseConfigurationActor1>(cx, useSystemPriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1609963244257551360UL, 6418944901107183872UL);
		ActorExecutionContextHelper __helper(static_cast<GetDatabaseConfigurationActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getDatabaseConfiguration");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getDatabaseConfiguration", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetDatabaseConfigurationActor1, 0, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetDatabaseConfigurationActor1, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<DatabaseConfiguration> getDatabaseConfiguration( Database const& cx, bool const& useSystemPriority ) {
															#line 713 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<DatabaseConfiguration>(new GetDatabaseConfigurationActor1(cx, useSystemPriority));
															#line 2264 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 724 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

ConfigureAutoResult parseConfig(StatusObject const& status) {
	ConfigureAutoResult result;
	StatusObjectReader statusObj(status);

	StatusObjectReader statusObjCluster;
	if (!statusObj.get("cluster", statusObjCluster))
		return ConfigureAutoResult();

	StatusObjectReader statusObjConfig;
	if (!statusObjCluster.get("configuration", statusObjConfig))
		return ConfigureAutoResult();

	if (!statusObjConfig.get("redundancy.factor", result.old_replication))
		return ConfigureAutoResult();

	result.auto_replication = result.old_replication;

	[[maybe_unused]] int storage_replication;
	int log_replication;
	if (result.old_replication == "single") {
		result.auto_replication = "double";
		storage_replication = 2;
		log_replication = 2;
	} else if (result.old_replication == "double" || result.old_replication == "fast_recovery_double") {
		storage_replication = 2;
		log_replication = 2;
	} else if (result.old_replication == "triple" || result.old_replication == "fast_recovery_triple") {
		storage_replication = 3;
		log_replication = 3;
	} else if (result.old_replication == "three_datacenter") {
		storage_replication = 6;
		log_replication = 4;
	} else if (result.old_replication == "three_datacenter_fallback") {
		storage_replication = 4;
		log_replication = 4;
	} else if (result.old_replication == "three_data_hall") {
		storage_replication = 3;
		log_replication = 4;
	} else if (result.old_replication == "three_data_hall_fallback") {
		storage_replication = 2;
		log_replication = 4;
	} else
		return ConfigureAutoResult();

	StatusObjectReader machinesMap;
	if (!statusObjCluster.get("machines", machinesMap))
		return ConfigureAutoResult();

	std::map<std::string, std::string> machineid_dcid;
	std::set<std::string> datacenters;
	int machineCount = 0;
	for (auto mach : machinesMap.obj()) {
		StatusObjectReader machine(mach.second);
		std::string dcId;
		if (machine.get("datacenter_id", dcId)) {
			machineid_dcid[mach.first] = dcId;
			datacenters.insert(dcId);
		}
		machineCount++;
	}

	result.machines = machineCount;

	if (datacenters.size() > 1)
		return ConfigureAutoResult();

	StatusObjectReader processesMap;
	if (!statusObjCluster.get("processes", processesMap))
		return ConfigureAutoResult();

	std::set<std::string> oldMachinesWithTransaction;
	int oldTransactionProcesses = 0;
	std::map<std::string, std::vector<std::pair<NetworkAddress, ProcessClass>>> machine_processes;
	int processCount = 0;
	for (auto proc : processesMap.obj()) {
		StatusObjectReader process(proc.second);
		if (!process.has("excluded") || !process.last().get_bool()) {
			std::string addrStr;
			if (!process.get("address", addrStr))
				return ConfigureAutoResult();
			std::string class_source;
			if (!process.get("class_source", class_source))
				return ConfigureAutoResult();
			std::string class_type;
			if (!process.get("class_type", class_type))
				return ConfigureAutoResult();
			std::string machineId;
			if (!process.get("machine_id", machineId))
				return ConfigureAutoResult();

			NetworkAddress addr = NetworkAddress::parse(addrStr);
			ProcessClass processClass(class_type, class_source);

			if (processClass.classType() == ProcessClass::TransactionClass ||
			    processClass.classType() == ProcessClass::LogClass) {
				oldMachinesWithTransaction.insert(machineId);
			}

			if (processClass.classType() == ProcessClass::TransactionClass ||
			    processClass.classType() == ProcessClass::CommitProxyClass ||
			    processClass.classType() == ProcessClass::GrvProxyClass ||
			    processClass.classType() == ProcessClass::ResolutionClass ||
			    processClass.classType() == ProcessClass::StatelessClass ||
			    processClass.classType() == ProcessClass::LogClass) {
				oldTransactionProcesses++;
			}

			if (processClass.classSource() == ProcessClass::AutoSource) {
				processClass = ProcessClass(ProcessClass::UnsetClass, ProcessClass::CommandLineSource);
				result.address_class[addr] = processClass;
			}

			if (processClass.classType() != ProcessClass::TesterClass) {
				machine_processes[machineId].emplace_back(addr, processClass);
				processCount++;
			}
		}
	}

	result.processes = processCount;
	result.old_processes_with_transaction = oldTransactionProcesses;
	result.old_machines_with_transaction = oldMachinesWithTransaction.size();

	std::map<std::pair<int, std::string>, std::vector<std::pair<NetworkAddress, ProcessClass>>> count_processes;
	for (auto& it : machine_processes) {
		count_processes[std::make_pair(it.second.size(), it.first)] = it.second;
	}

	std::set<std::string> machinesWithTransaction;
	std::set<std::string> machinesWithStorage;
	int totalTransactionProcesses = 0;
	int existingProxyCount = 0;
	int existingGrvProxyCount = 0;
	int existingResolverCount = 0;
	int existingStatelessCount = 0;
	for (auto& it : machine_processes) {
		for (auto& proc : it.second) {
			if (proc.second == ProcessClass::TransactionClass || proc.second == ProcessClass::LogClass) {
				totalTransactionProcesses++;
				machinesWithTransaction.insert(it.first);
			}
			if (proc.second == ProcessClass::StatelessClass) {
				existingStatelessCount++;
			}
			if (proc.second == ProcessClass::CommitProxyClass) {
				existingProxyCount++;
			}
			if (proc.second == ProcessClass::GrvProxyClass) {
				existingGrvProxyCount++;
			}
			if (proc.second == ProcessClass::ResolutionClass) {
				existingResolverCount++;
			}
			if (proc.second == ProcessClass::StorageClass) {
				machinesWithStorage.insert(it.first);
			}
			if (proc.second == ProcessClass::UnsetClass && proc.second.classSource() == ProcessClass::DBSource) {
				machinesWithStorage.insert(it.first);
			}
		}
	}

	if (processCount < 10)
		return ConfigureAutoResult();

	result.desired_resolvers = 1;
	int resolverCount;
	if (!statusObjConfig.get("resolvers", result.old_resolvers)) {
		result.old_resolvers = CLIENT_KNOBS->DEFAULT_AUTO_RESOLVERS;
		statusObjConfig.get("auto_resolvers", result.old_resolvers);
		result.auto_resolvers = result.desired_resolvers;
		resolverCount = result.auto_resolvers;
	} else {
		result.auto_resolvers = result.old_resolvers;
		resolverCount = result.old_resolvers;
	}

	result.desired_commit_proxies = std::max(std::min(12, processCount / 15), 1);
	int proxyCount;
	if (!statusObjConfig.get("commit_proxies", result.old_commit_proxies)) {
		result.old_commit_proxies = CLIENT_KNOBS->DEFAULT_AUTO_COMMIT_PROXIES;
		statusObjConfig.get("auto_commit_proxies", result.old_commit_proxies);
		result.auto_commit_proxies = result.desired_commit_proxies;
		proxyCount = result.auto_commit_proxies;
	} else {
		result.auto_commit_proxies = result.old_commit_proxies;
		proxyCount = result.old_commit_proxies;
	}

	result.desired_grv_proxies = std::max(std::min(4, processCount / 20), 1);
	int grvProxyCount;
	if (!statusObjConfig.get("grv_proxies", result.old_grv_proxies)) {
		result.old_grv_proxies = CLIENT_KNOBS->DEFAULT_AUTO_GRV_PROXIES;
		statusObjConfig.get("auto_grv_proxies", result.old_grv_proxies);
		result.auto_grv_proxies = result.desired_grv_proxies;
		grvProxyCount = result.auto_grv_proxies;
	} else {
		result.auto_grv_proxies = result.old_grv_proxies;
		grvProxyCount = result.old_grv_proxies;
	}

	result.desired_logs = std::min(12, processCount / 20);
	result.desired_logs = std::max(result.desired_logs, log_replication + 1);
	result.desired_logs = std::min<int>(result.desired_logs, machine_processes.size());
	int logCount;
	if (!statusObjConfig.get("logs", result.old_logs)) {
		result.old_logs = CLIENT_KNOBS->DEFAULT_AUTO_LOGS;
		statusObjConfig.get("auto_logs", result.old_logs);
		result.auto_logs = result.desired_logs;
		logCount = result.auto_logs;
	} else {
		result.auto_logs = result.old_logs;
		logCount = result.old_logs;
	}

	logCount = std::max(logCount, log_replication);

	totalTransactionProcesses += std::min(existingProxyCount, proxyCount);
	totalTransactionProcesses += std::min(existingGrvProxyCount, grvProxyCount);
	totalTransactionProcesses += std::min(existingResolverCount, resolverCount);
	totalTransactionProcesses += existingStatelessCount;

	// if one process on a machine is transaction class, make them all transaction class
	for (auto& it : count_processes) {
		if (machinesWithTransaction.count(it.first.second) && !machinesWithStorage.count(it.first.second)) {
			for (auto& proc : it.second) {
				if (proc.second == ProcessClass::UnsetClass &&
				    proc.second.classSource() == ProcessClass::CommandLineSource) {
					result.address_class[proc.first] =
					    ProcessClass(ProcessClass::TransactionClass, ProcessClass::AutoSource);
					totalTransactionProcesses++;
				}
			}
		}
	}

	int desiredTotalTransactionProcesses = logCount + resolverCount + proxyCount + grvProxyCount;

	// add machines with all transaction class until we have enough processes and enough machines
	for (auto& it : count_processes) {
		if (machinesWithTransaction.size() >= logCount && totalTransactionProcesses >= desiredTotalTransactionProcesses)
			break;

		if (!machinesWithTransaction.count(it.first.second) && !machinesWithStorage.count(it.first.second)) {
			for (auto& proc : it.second) {
				if (proc.second == ProcessClass::UnsetClass &&
				    proc.second.classSource() == ProcessClass::CommandLineSource) {
					ASSERT(proc.second != ProcessClass::TransactionClass);
					result.address_class[proc.first] =
					    ProcessClass(ProcessClass::TransactionClass, ProcessClass::AutoSource);
					totalTransactionProcesses++;
					machinesWithTransaction.insert(it.first.second);
				}
			}
		}
	}

	if (machinesWithTransaction.size() < logCount || totalTransactionProcesses < desiredTotalTransactionProcesses)
		return ConfigureAutoResult();

	result.auto_processes_with_transaction = totalTransactionProcesses;
	result.auto_machines_with_transaction = machinesWithTransaction.size();

	if (3 * totalTransactionProcesses > processCount)
		return ConfigureAutoResult();

	return result;
}

															#line 2538 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getWorkers()
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetWorkersActor>
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetWorkersActorState {
															#line 2545 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetWorkersActorState(Transaction* const& tr) 
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 995 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   processClasses(tr->getRange(processClassKeys, CLIENT_KNOBS->TOO_MANY)),
															#line 996 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   processData(tr->getRange(workerListKeys, CLIENT_KNOBS->TOO_MANY))
															#line 2556 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getWorkers", reinterpret_cast<unsigned long>(this));

	}
	~GetWorkersActorState() 
	{
		fdb_probe_actor_destroy("getWorkers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 998 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(processClasses) && success(processData);
															#line 998 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetWorkersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2573 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetWorkersActor*>(this)->actor_wait_state = 1;
															#line 998 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetWorkersActor, 0, Void >*>(static_cast<GetWorkersActor*>(this)));
															#line 2578 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetWorkersActorState();
		static_cast<GetWorkersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 999 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!processClasses.get().more && processClasses.get().size() < CLIENT_KNOBS->TOO_MANY);
															#line 1000 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!processData.get().more && processData.get().size() < CLIENT_KNOBS->TOO_MANY);
															#line 1002 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::map<Optional<Standalone<StringRef>>, ProcessClass> id_class;
															#line 1003 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < processClasses.get().size();i++) {
															#line 1004 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			id_class[decodeProcessClassKey(processClasses.get()[i].key)] = decodeProcessClassValue(processClasses.get()[i].value);
															#line 2609 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1008 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<ProcessData> results;
															#line 1010 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < processData.get().size();i++) {
															#line 1011 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ProcessData data = decodeWorkerListValue(processData.get()[i].value);
															#line 1012 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ProcessClass processClass = id_class[data.locality.processId()];
															#line 1014 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (processClass.classSource() == ProcessClass::DBSource || data.processClass.classType() == ProcessClass::UnsetClass)
															#line 2621 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				data.processClass = processClass;
															#line 2625 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1018 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (data.processClass.classType() != ProcessClass::TesterClass)
															#line 2629 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1019 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				results.push_back(data);
															#line 2633 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1022 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetWorkersActor*>(this)->SAV<std::vector<ProcessData>>::futures) { (void)(results); this->~GetWorkersActorState(); static_cast<GetWorkersActor*>(this)->destroy(); return 0; }
															#line 2638 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetWorkersActor*>(this)->SAV< std::vector<ProcessData> >::value()) std::vector<ProcessData>(results);
		this->~GetWorkersActorState();
		static_cast<GetWorkersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 999 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!processClasses.get().more && processClasses.get().size() < CLIENT_KNOBS->TOO_MANY);
															#line 1000 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!processData.get().more && processData.get().size() < CLIENT_KNOBS->TOO_MANY);
															#line 1002 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::map<Optional<Standalone<StringRef>>, ProcessClass> id_class;
															#line 1003 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < processClasses.get().size();i++) {
															#line 1004 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			id_class[decodeProcessClassKey(processClasses.get()[i].key)] = decodeProcessClassValue(processClasses.get()[i].value);
															#line 2658 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1008 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<ProcessData> results;
															#line 1010 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < processData.get().size();i++) {
															#line 1011 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ProcessData data = decodeWorkerListValue(processData.get()[i].value);
															#line 1012 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ProcessClass processClass = id_class[data.locality.processId()];
															#line 1014 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (processClass.classSource() == ProcessClass::DBSource || data.processClass.classType() == ProcessClass::UnsetClass)
															#line 2670 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				data.processClass = processClass;
															#line 2674 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1018 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (data.processClass.classType() != ProcessClass::TesterClass)
															#line 2678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1019 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				results.push_back(data);
															#line 2682 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1022 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetWorkersActor*>(this)->SAV<std::vector<ProcessData>>::futures) { (void)(results); this->~GetWorkersActorState(); static_cast<GetWorkersActor*>(this)->destroy(); return 0; }
															#line 2687 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetWorkersActor*>(this)->SAV< std::vector<ProcessData> >::value()) std::vector<ProcessData>(results);
		this->~GetWorkersActorState();
		static_cast<GetWorkersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetWorkersActor*>(this)->actor_wait_state > 0) static_cast<GetWorkersActor*>(this)->actor_wait_state = 0;
		static_cast<GetWorkersActor*>(this)->ActorCallback< GetWorkersActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetWorkersActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetWorkersActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetWorkersActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 995 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<RangeResult> processClasses;
															#line 996 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<RangeResult> processData;
															#line 2776 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getWorkers()
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetWorkersActor final : public Actor<std::vector<ProcessData>>, public ActorCallback< GetWorkersActor, 0, Void >, public FastAllocated<GetWorkersActor>, public GetWorkersActorState<GetWorkersActor> {
															#line 2781 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetWorkersActor>::operator new;
	using FastAllocated<GetWorkersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7023438796570172928UL, 13591632497469995776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<ProcessData>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetWorkersActor, 0, Void >;
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetWorkersActor(Transaction* const& tr) 
															#line 2798 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<ProcessData>>(),
		   GetWorkersActorState<GetWorkersActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14779239272732970240UL, 16642038601798370048UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getWorkers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetWorkersActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<ProcessData>> getWorkers( Transaction* const& tr ) {
															#line 994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<ProcessData>>(new GetWorkersActor(tr));
															#line 2831 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1024 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 2836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getWorkers()
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetWorkersActor1>
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetWorkersActor1State {
															#line 2843 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetWorkersActor1State(Database const& cx) 
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1026 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 2852 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getWorkers", reinterpret_cast<unsigned long>(this));

	}
	~GetWorkersActor1State() 
	{
		fdb_probe_actor_destroy("getWorkers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1027 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 2867 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetWorkersActor1State();
		static_cast<GetWorkersActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1029 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1031 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1032 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<ProcessData>> __when_expr_0 = getWorkers(&tr);
															#line 1032 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetWorkersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2906 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetWorkersActor1*>(this)->actor_wait_state = 1;
															#line 1032 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*>(static_cast<GetWorkersActor1*>(this)));
															#line 2911 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1035 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1035 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetWorkersActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2935 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetWorkersActor1*>(this)->actor_wait_state = 2;
															#line 1035 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetWorkersActor1, 1, Void >*>(static_cast<GetWorkersActor1*>(this)));
															#line 2940 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<ProcessData> const& workers,int loopDepth) 
	{
															#line 1033 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetWorkersActor1*>(this)->SAV<std::vector<ProcessData>>::futures) { (void)(workers); this->~GetWorkersActor1State(); static_cast<GetWorkersActor1*>(this)->destroy(); return 0; }
															#line 2955 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetWorkersActor1*>(this)->SAV< std::vector<ProcessData> >::value()) std::vector<ProcessData>(workers);
		this->~GetWorkersActor1State();
		static_cast<GetWorkersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<ProcessData> && workers,int loopDepth) 
	{
															#line 1033 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetWorkersActor1*>(this)->SAV<std::vector<ProcessData>>::futures) { (void)(workers); this->~GetWorkersActor1State(); static_cast<GetWorkersActor1*>(this)->destroy(); return 0; }
															#line 2967 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetWorkersActor1*>(this)->SAV< std::vector<ProcessData> >::value()) std::vector<ProcessData>(workers);
		this->~GetWorkersActor1State();
		static_cast<GetWorkersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<ProcessData> const& workers,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(workers, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<ProcessData> && workers,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(workers), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetWorkersActor1*>(this)->actor_wait_state > 0) static_cast<GetWorkersActor1*>(this)->actor_wait_state = 0;
		static_cast<GetWorkersActor1*>(this)->ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >::remove();

	}
	void a_callback_fire(ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*,std::vector<ProcessData> const& value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*,std::vector<ProcessData> && value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*,Error err) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetWorkersActor1*>(this)->actor_wait_state > 0) static_cast<GetWorkersActor1*>(this)->actor_wait_state = 0;
		static_cast<GetWorkersActor1*>(this)->ActorCallback< GetWorkersActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetWorkersActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetWorkersActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetWorkersActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1026 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 3141 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getWorkers()
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetWorkersActor1 final : public Actor<std::vector<ProcessData>>, public ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >, public ActorCallback< GetWorkersActor1, 1, Void >, public FastAllocated<GetWorkersActor1>, public GetWorkersActor1State<GetWorkersActor1> {
															#line 3146 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetWorkersActor1>::operator new;
	using FastAllocated<GetWorkersActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7023438796570172928UL, 13591632497469995776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<ProcessData>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >;
friend struct ActorCallback< GetWorkersActor1, 1, Void >;
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetWorkersActor1(Database const& cx) 
															#line 3164 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<ProcessData>>(),
		   GetWorkersActor1State<GetWorkersActor1>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getWorkers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12240367636695030272UL, 7610739061935181824UL);
		ActorExecutionContextHelper __helper(static_cast<GetWorkersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getWorkers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getWorkers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetWorkersActor1, 0, std::vector<ProcessData> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetWorkersActor1, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<ProcessData>> getWorkers( Database const& cx ) {
															#line 1025 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<ProcessData>>(new GetWorkersActor1(cx));
															#line 3198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1039 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 3203 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getConnectionString()
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetConnectionStringActor>
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetConnectionStringActorState {
															#line 3210 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetConnectionStringActorState(Database const& cx) 
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1041 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 3219 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getConnectionString", reinterpret_cast<unsigned long>(this));

	}
	~GetConnectionStringActorState() 
	{
		fdb_probe_actor_destroy("getConnectionString", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1042 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 3234 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetConnectionStringActorState();
		static_cast<GetConnectionStringActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 1045 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1046 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(coordinatorsKey);
															#line 1046 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetConnectionStringActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetConnectionStringActor*>(this)->actor_wait_state = 1;
															#line 1046 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*>(static_cast<GetConnectionStringActor*>(this)));
															#line 3276 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1051 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1051 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetConnectionStringActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3300 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetConnectionStringActor*>(this)->actor_wait_state = 2;
															#line 1051 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetConnectionStringActor, 1, Void >*>(static_cast<GetConnectionStringActor*>(this)));
															#line 3305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& currentKey,int loopDepth) 
	{
															#line 1047 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!currentKey.present())
															#line 3320 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1048 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetConnectionStringActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(Optional<ClusterConnectionString>()); this->~GetConnectionStringActorState(); static_cast<GetConnectionStringActor*>(this)->destroy(); return 0; }
															#line 3324 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetConnectionStringActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(Optional<ClusterConnectionString>());
			this->~GetConnectionStringActorState();
			static_cast<GetConnectionStringActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1049 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetConnectionStringActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(ClusterConnectionString(currentKey.get().toString())); this->~GetConnectionStringActorState(); static_cast<GetConnectionStringActor*>(this)->destroy(); return 0; }
															#line 3332 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetConnectionStringActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(ClusterConnectionString(currentKey.get().toString()));
		this->~GetConnectionStringActorState();
		static_cast<GetConnectionStringActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && currentKey,int loopDepth) 
	{
															#line 1047 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!currentKey.present())
															#line 3344 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1048 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetConnectionStringActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(Optional<ClusterConnectionString>()); this->~GetConnectionStringActorState(); static_cast<GetConnectionStringActor*>(this)->destroy(); return 0; }
															#line 3348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetConnectionStringActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(Optional<ClusterConnectionString>());
			this->~GetConnectionStringActorState();
			static_cast<GetConnectionStringActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1049 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetConnectionStringActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(ClusterConnectionString(currentKey.get().toString())); this->~GetConnectionStringActorState(); static_cast<GetConnectionStringActor*>(this)->destroy(); return 0; }
															#line 3356 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetConnectionStringActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(ClusterConnectionString(currentKey.get().toString()));
		this->~GetConnectionStringActorState();
		static_cast<GetConnectionStringActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& currentKey,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(currentKey, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && currentKey,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(currentKey), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetConnectionStringActor*>(this)->actor_wait_state > 0) static_cast<GetConnectionStringActor*>(this)->actor_wait_state = 0;
		static_cast<GetConnectionStringActor*>(this)->ActorCallback< GetConnectionStringActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetConnectionStringActor*>(this)->actor_wait_state > 0) static_cast<GetConnectionStringActor*>(this)->actor_wait_state = 0;
		static_cast<GetConnectionStringActor*>(this)->ActorCallback< GetConnectionStringActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetConnectionStringActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetConnectionStringActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetConnectionStringActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1041 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 3530 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getConnectionString()
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetConnectionStringActor final : public Actor<Optional<ClusterConnectionString>>, public ActorCallback< GetConnectionStringActor, 0, Optional<Value> >, public ActorCallback< GetConnectionStringActor, 1, Void >, public FastAllocated<GetConnectionStringActor>, public GetConnectionStringActorState<GetConnectionStringActor> {
															#line 3535 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetConnectionStringActor>::operator new;
	using FastAllocated<GetConnectionStringActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14601089463279200000UL, 4604285476255192832UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<ClusterConnectionString>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetConnectionStringActor, 0, Optional<Value> >;
friend struct ActorCallback< GetConnectionStringActor, 1, Void >;
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetConnectionStringActor(Database const& cx) 
															#line 3553 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Optional<ClusterConnectionString>>(),
		   GetConnectionStringActorState<GetConnectionStringActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getConnectionString", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15280877619308916480UL, 17895028133604852992UL);
		ActorExecutionContextHelper __helper(static_cast<GetConnectionStringActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getConnectionString");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getConnectionString", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetConnectionStringActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetConnectionStringActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Optional<ClusterConnectionString>> getConnectionString( Database const& cx ) {
															#line 1040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Optional<ClusterConnectionString>>(new GetConnectionStringActor(cx));
															#line 3587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1055 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

static std::vector<std::string> connectionStrings;

namespace {

															#line 3596 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via getClusterConnectionStringFromStorageServer()
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetClusterConnectionStringFromStorageServerActor>
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetClusterConnectionStringFromStorageServerActorState {
															#line 3602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetClusterConnectionStringFromStorageServerActorState(Transaction* const& tr) 
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 3609 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this));

	}
	~GetClusterConnectionStringFromStorageServerActorState() 
	{
		fdb_probe_actor_destroy("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1062 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1063 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1064 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1065 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1067 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			retryTimes = 0;
															#line 1068 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 3634 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetClusterConnectionStringFromStorageServerActorState();
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1069 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (retryTimes >= CLIENT_KNOBS->CHANGE_QUORUM_BAD_STATE_RETRY_TIMES)
															#line 3664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1070 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(Optional<ClusterConnectionString>()); this->~GetClusterConnectionStringFromStorageServerActorState(); static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->destroy(); return 0; }
															#line 3668 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(Optional<ClusterConnectionString>());
			this->~GetClusterConnectionStringFromStorageServerActorState();
			static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1073 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_0 = tr->get(coordinatorsKey);
															#line 1073 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3678 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 1;
															#line 1073 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*>(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)));
															#line 3683 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1074 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (g_network->isSimulated() && currentKey.present())
															#line 3692 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1082 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			int n = connectionStrings.size() > 0 ? connectionStrings.size() - 1 : 0;
															#line 1083 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(int i = 0;i < n;++i) {
															#line 1084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ASSERT(currentKey.get() != connectionStrings.at(i));
															#line 3700 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1088 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!currentKey.present())
															#line 3705 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1090 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			++retryTimes;
															#line 1091 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(CLIENT_KNOBS->CHANGE_QUORUM_BAD_STATE_RETRY_DELAY);
															#line 1091 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 2;
															#line 1091 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*>(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)));
															#line 3718 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& __currentKey,int loopDepth) 
	{
															#line 1073 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		currentKey = __currentKey;
															#line 3732 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && __currentKey,int loopDepth) 
	{
		currentKey = std::move(__currentKey);
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state > 0) static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 0;
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1095 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		clusterConnectionString = ClusterConnectionString(currentKey.get().toString());
															#line 1096 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (tr->getDatabase()->getConnectionRecord() && clusterConnectionString.clusterKeyName().toString() != tr->getDatabase()->getConnectionRecord()->getConnectionString().clusterKeyName())
															#line 3813 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1100 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			++retryTimes;
															#line 1101 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(CLIENT_KNOBS->CHANGE_QUORUM_BAD_STATE_RETRY_DELAY);
															#line 1101 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3821 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 3;
															#line 1101 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*>(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)));
															#line 3826 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state > 0) static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 0;
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
															#line 1105 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->SAV<Optional<ClusterConnectionString>>::futures) { (void)(clusterConnectionString); this->~GetClusterConnectionStringFromStorageServerActorState(); static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->destroy(); return 0; }
															#line 3927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->SAV< Optional<ClusterConnectionString> >::value()) Optional<ClusterConnectionString>(std::move(clusterConnectionString)); // state_var_RVO
		this->~GetClusterConnectionStringFromStorageServerActorState();
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state > 0) static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->actor_wait_state = 0;
		static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1067 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int retryTimes;
															#line 1073 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Optional<Value> currentKey;
															#line 1095 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString clusterConnectionString;
															#line 4030 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getClusterConnectionStringFromStorageServer()
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetClusterConnectionStringFromStorageServerActor final : public Actor<Optional<ClusterConnectionString>>, public ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >, public ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >, public ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >, public FastAllocated<GetClusterConnectionStringFromStorageServerActor>, public GetClusterConnectionStringFromStorageServerActorState<GetClusterConnectionStringFromStorageServerActor> {
															#line 4035 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetClusterConnectionStringFromStorageServerActor>::operator new;
	using FastAllocated<GetClusterConnectionStringFromStorageServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5387647723938747904UL, 1776308152287766528UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<ClusterConnectionString>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >;
friend struct ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >;
friend struct ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >;
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetClusterConnectionStringFromStorageServerActor(Transaction* const& tr) 
															#line 4054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Optional<ClusterConnectionString>>(),
		   GetClusterConnectionStringFromStorageServerActorState<GetClusterConnectionStringFromStorageServerActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(625065204163832320UL, 1122260541233559808UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterConnectionStringFromStorageServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getClusterConnectionStringFromStorageServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getClusterConnectionStringFromStorageServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetClusterConnectionStringFromStorageServerActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetClusterConnectionStringFromStorageServerActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetClusterConnectionStringFromStorageServerActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Optional<ClusterConnectionString>> getClusterConnectionStringFromStorageServer( Transaction* const& tr ) {
															#line 1060 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Optional<ClusterConnectionString>>(new GetClusterConnectionStringFromStorageServerActor(tr));
															#line 4088 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1108 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 4093 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via verifyConfigurationDatabaseAlive()
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class VerifyConfigurationDatabaseAliveActor>
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class VerifyConfigurationDatabaseAliveActorState {
															#line 4099 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	VerifyConfigurationDatabaseAliveActorState(Database const& cx) 
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1110 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   backoff(),
															#line 1111 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   configTr()
															#line 4110 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this));

	}
	~VerifyConfigurationDatabaseAliveActorState() 
	{
		fdb_probe_actor_destroy("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 4125 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~VerifyConfigurationDatabaseAliveActorState();
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1116 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			configTr = ISingleThreadTransaction::create(ISingleThreadTransaction::Type::PAXOS_CONFIG, cx);
															#line 1117 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			Tuple tuple;
															#line 1118 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tuple.appendNull();
															#line 1119 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tuple << "test"_sr;
															#line 1120 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = configTr->get(tuple.pack());
															#line 1120 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4166 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 1;
															#line 1120 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*>(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)));
															#line 4171 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1124 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ChangeQuorumCheckerNewCoordinatorsError").error(e);
															#line 1125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (e.code() == error_code_coordinators_changed)
															#line 4195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1126 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<Void> __when_expr_1 = backoff.onError();
															#line 1126 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
				static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 2;
															#line 1126 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*>(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)));
															#line 4206 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1129 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<Void> __when_expr_2 = configTr->onError(e);
															#line 1129 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4215 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when2(__when_expr_2.get(), loopDepth); };
				static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 3;
															#line 1129 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*>(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)));
															#line 4220 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& serializedValue,int loopDepth) 
	{
															#line 1121 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ChangeQuorumCheckerNewCoordinatorsOnline").log();
															#line 1122 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~VerifyConfigurationDatabaseAliveActorState(); static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->destroy(); return 0; }
															#line 4238 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~VerifyConfigurationDatabaseAliveActorState();
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && serializedValue,int loopDepth) 
	{
															#line 1121 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ChangeQuorumCheckerNewCoordinatorsOnline").log();
															#line 1122 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~VerifyConfigurationDatabaseAliveActorState(); static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->destroy(); return 0; }
															#line 4252 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~VerifyConfigurationDatabaseAliveActorState();
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& serializedValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(serializedValue, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && serializedValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(serializedValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state > 0) static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 0;
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont2(Void const& _,int loopDepth) 
	{
															#line 1127 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		configTr->reset();
															#line 4345 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont2(Void && _,int loopDepth) 
	{
															#line 1127 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		configTr->reset();
															#line 4354 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state > 0) static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 0;
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state > 0) static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->actor_wait_state = 0;
		static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1110 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Backoff backoff;
															#line 1111 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<ISingleThreadTransaction> configTr;
															#line 4527 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via verifyConfigurationDatabaseAlive()
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class VerifyConfigurationDatabaseAliveActor final : public Actor<Void>, public ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >, public ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >, public ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >, public FastAllocated<VerifyConfigurationDatabaseAliveActor>, public VerifyConfigurationDatabaseAliveActorState<VerifyConfigurationDatabaseAliveActor> {
															#line 4532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<VerifyConfigurationDatabaseAliveActor>::operator new;
	using FastAllocated<VerifyConfigurationDatabaseAliveActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12134509241220544768UL, 7012879785730547712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >;
friend struct ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >;
friend struct ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >;
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	VerifyConfigurationDatabaseAliveActor(Database const& cx) 
															#line 4551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   VerifyConfigurationDatabaseAliveActorState<VerifyConfigurationDatabaseAliveActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3250116751694520064UL, 7074429358866028800UL);
		ActorExecutionContextHelper __helper(static_cast<VerifyConfigurationDatabaseAliveActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("verifyConfigurationDatabaseAlive");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("verifyConfigurationDatabaseAlive", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< VerifyConfigurationDatabaseAliveActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< VerifyConfigurationDatabaseAliveActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< VerifyConfigurationDatabaseAliveActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> verifyConfigurationDatabaseAlive( Database const& cx ) {
															#line 1109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new VerifyConfigurationDatabaseAliveActor(cx));
															#line 4585 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1134 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 4590 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via resetPreviousCoordinatorsKey()
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ResetPreviousCoordinatorsKeyActor>
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ResetPreviousCoordinatorsKeyActorState {
															#line 4596 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ResetPreviousCoordinatorsKeyActorState(Database const& cx) 
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx)
															#line 4603 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this));

	}
	~ResetPreviousCoordinatorsKeyActorState() 
	{
		fdb_probe_actor_destroy("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1136 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 4618 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResetPreviousCoordinatorsKeyActorState();
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1142 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		clearTr = ISingleThreadTransaction::create(ISingleThreadTransaction::Type::RYW, cx);
															#line 4648 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		try {
															#line 1145 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			clearTr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1146 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			clearTr->clear(previousCoordinatorsKey);
															#line 1147 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = clearTr->commit();
															#line 1147 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4658 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state = 1;
															#line 1147 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*>(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)));
															#line 4663 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e2,int loopDepth=0) 
	{
		try {
															#line 1150 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = clearTr->onError(e2);
															#line 1150 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4687 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state = 2;
															#line 1150 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*>(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)));
															#line 4692 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1148 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResetPreviousCoordinatorsKeyActorState(); static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->destroy(); return 0; }
															#line 4707 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResetPreviousCoordinatorsKeyActorState();
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1148 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResetPreviousCoordinatorsKeyActorState(); static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->destroy(); return 0; }
															#line 4719 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResetPreviousCoordinatorsKeyActorState();
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state > 0) static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state = 0;
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state > 0) static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->actor_wait_state = 0;
		static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1142 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<ISingleThreadTransaction> clearTr;
															#line 4893 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via resetPreviousCoordinatorsKey()
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ResetPreviousCoordinatorsKeyActor final : public Actor<Void>, public ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >, public ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >, public FastAllocated<ResetPreviousCoordinatorsKeyActor>, public ResetPreviousCoordinatorsKeyActorState<ResetPreviousCoordinatorsKeyActor> {
															#line 4898 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ResetPreviousCoordinatorsKeyActor>::operator new;
	using FastAllocated<ResetPreviousCoordinatorsKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2120594236947170048UL, 2330044488284179200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >;
friend struct ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >;
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ResetPreviousCoordinatorsKeyActor(Database const& cx) 
															#line 4916 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ResetPreviousCoordinatorsKeyActorState<ResetPreviousCoordinatorsKeyActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2328696218968888064UL, 1221210460215735552UL);
		ActorExecutionContextHelper __helper(static_cast<ResetPreviousCoordinatorsKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resetPreviousCoordinatorsKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resetPreviousCoordinatorsKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResetPreviousCoordinatorsKeyActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResetPreviousCoordinatorsKeyActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> resetPreviousCoordinatorsKey( Database const& cx ) {
															#line 1135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ResetPreviousCoordinatorsKeyActor(cx));
															#line 4949 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1154 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

} // namespace

															#line 4956 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via changeQuorumChecker()
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ChangeQuorumCheckerActor>
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ChangeQuorumCheckerActorState {
															#line 4963 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ChangeQuorumCheckerActorState(Transaction* const& tr,ClusterConnectionString* const& conn,std::string const& newName,bool const& disableConfigDB) 
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   conn(conn),
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   newName(newName),
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   disableConfigDB(disableConfigDB)
															#line 4976 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("changeQuorumChecker", reinterpret_cast<unsigned long>(this));

	}
	~ChangeQuorumCheckerActorState() 
	{
		fdb_probe_actor_destroy("changeQuorumChecker", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1161 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ChangeQuorumCheckerStart").detail("NewConnectionString", conn->toString());
															#line 1162 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<ClusterConnectionString>> __when_expr_0 = getClusterConnectionStringFromStorageServer(tr);
															#line 1162 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4995 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 1;
															#line 1162 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5000 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1165 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!clusterConnectionStringOptional.present())
															#line 5023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1166 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::BAD_DATABASE_STATE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5027 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::BAD_DATABASE_STATE);
			this->~ChangeQuorumCheckerActorState();
			static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1170 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		old = clusterConnectionStringOptional.get();
															#line 1172 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (conn->hostnames.size() + conn->coords.size() == 0)
															#line 5037 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1173 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			conn->hostnames = old.hostnames;
															#line 1174 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			conn->coords = old.coords;
															#line 5043 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1176 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<NetworkAddress>> __when_expr_1 = conn->tryResolveHostnames();
															#line 1176 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5049 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 2;
															#line 1176 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<ClusterConnectionString> const& __clusterConnectionStringOptional,int loopDepth) 
	{
															#line 1162 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		clusterConnectionStringOptional = __clusterConnectionStringOptional;
															#line 5063 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<ClusterConnectionString> && __clusterConnectionStringOptional,int loopDepth) 
	{
		clusterConnectionStringOptional = std::move(__clusterConnectionStringOptional);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1177 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (desiredCoordinators.size() != conn->hostnames.size() + conn->coords.size())
															#line 5142 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1178 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ChangeQuorumCheckerEarlyTermination") .detail("Reason", "One or more hostnames are unresolvable") .backtrace();
															#line 1181 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5148 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::COORDINATOR_UNREACHABLE);
			this->~ChangeQuorumCheckerActorState();
			static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1184 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (newName.empty())
															#line 5156 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1185 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newName = old.clusterKeyName().toString();
															#line 5160 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1187 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(conn->hostnames.begin(), conn->hostnames.end());
															#line 1188 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(conn->coords.begin(), conn->coords.end());
															#line 1189 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(old.hostnames.begin(), old.hostnames.end());
															#line 1190 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(old.coords.begin(), old.coords.end());
															#line 1191 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (conn->hostnames == old.hostnames && conn->coords == old.coords && old.clusterKeyName() == newName)
															#line 5172 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1192 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			connectionStrings.clear();
															#line 1193 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (g_network->isSimulated() && g_simulator->configDBType == ConfigDBType::DISABLED)
															#line 5178 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1194 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				disableConfigDB = true;
															#line 5182 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1196 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!disableConfigDB)
															#line 5186 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1197 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<Void> __when_expr_2 = verifyConfigurationDatabaseAlive(tr->getDatabase());
															#line 1197 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5192 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 3;
															#line 1197 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 2, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5197 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont8(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(std::vector<NetworkAddress> const& __desiredCoordinators,int loopDepth) 
	{
															#line 1176 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCoordinators = __desiredCoordinators;
															#line 5216 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::vector<NetworkAddress> && __desiredCoordinators,int loopDepth) 
	{
		desiredCoordinators = std::move(__desiredCoordinators);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 1211 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		conn->parseKey(newName + ':' + deterministicRandom()->randomAlphaNumeric(32));
															#line 1212 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		connectionStrings.push_back(conn->toString());
															#line 1214 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (g_network->isSimulated())
															#line 5299 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1215 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			int i = 0;
															#line 1216 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			int protectedCount = 0;
															#line 1217 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			int minimumCoordinators = (desiredCoordinators.size() / 2) + 1;
															#line 1218 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(;protectedCount < minimumCoordinators && i < desiredCoordinators.size();) {
															#line 1219 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto process = g_simulator->getProcessByAddress(desiredCoordinators[i]);
															#line 1220 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto addresses = process->addresses;
															#line 1222 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!process->isReliable())
															#line 5315 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1223 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					i++;
															#line 5319 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					continue;
				}
															#line 1227 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				g_simulator->protectedAddresses.insert(process->addresses.address);
															#line 1228 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (addresses.secondaryAddress.present())
															#line 5326 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1229 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					g_simulator->protectedAddresses.insert(process->addresses.secondaryAddress.get());
															#line 5330 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
															#line 1231 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				TraceEvent("ProtectCoordinator").detail("Address", desiredCoordinators[i]).backtrace();
															#line 1232 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				protectedCount++;
															#line 1233 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				i++;
															#line 5338 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1236 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (protectedCount < minimumCoordinators)
															#line 5342 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1237 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				TraceEvent("NotEnoughReliableCoordinators") .detail("NumReliable", protectedCount) .detail("MinimumRequired", minimumCoordinators) .detail("ConnectionString", conn->toString());
															#line 1242 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5348 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::COORDINATOR_UNREACHABLE);
				this->~ChangeQuorumCheckerActorState();
				static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1246 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<Future<Optional<LeaderInfo>>> leaderServers;
															#line 1247 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ClientCoordinators coord(Reference<ClusterConnectionMemoryRecord>(new ClusterConnectionMemoryRecord(*conn)));
															#line 1249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		leaderServers.reserve(coord.clientLeaderServers.size());
															#line 1250 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < coord.clientLeaderServers.size();i++) {
															#line 1251 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (coord.clientLeaderServers[i].hostname.present())
															#line 5365 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1252 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				leaderServers.push_back(retryGetReplyFromHostname(GetLeaderRequest(coord.clusterKey, UID()), coord.clientLeaderServers[i].hostname.get(), WLTOKEN_CLIENTLEADERREG_GETLEADER, TaskPriority::CoordinationReply));
															#line 5369 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1257 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				leaderServers.push_back(retryBrokenPromise(coord.clientLeaderServers[i].getLeader, GetLeaderRequest(coord.clusterKey, UID()), TaskPriority::CoordinationReply));
															#line 5375 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1264 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_5 = waitForAll(leaderServers);
															#line 1263 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5382 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5when1(__when_expr_5.get(), loopDepth); };
															#line 1265 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(5.0);
															#line 5386 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont5when2(__when_expr_6.get(), loopDepth); };
		static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 6;
															#line 1264 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 5, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 1265 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 6, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5393 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
															#line 1199 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (BUGGIFY_WITH_PROB(0.1))
															#line 5402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1205 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(deterministicRandom()->random01() * 10);
															#line 1205 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5408 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont8when1(__when_expr_3.get(), loopDepth); };
			static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 4;
															#line 1205 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 3, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5413 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont11(int loopDepth) 
	{
															#line 1207 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_4 = resetPreviousCoordinatorsKey(tr->getDatabase());
															#line 1207 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5516 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont11when1(__when_expr_4.get(), loopDepth); };
		static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 5;
															#line 1207 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumCheckerActor, 4, Void >*>(static_cast<ChangeQuorumCheckerActor*>(this)));
															#line 5521 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont11cont1(Void const& _,int loopDepth) 
	{
															#line 1208 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::SAME_NETWORK_ADDRESSES); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5617 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::SAME_NETWORK_ADDRESSES);
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11cont1(Void && _,int loopDepth) 
	{
															#line 1208 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::SAME_NETWORK_ADDRESSES); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5629 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::SAME_NETWORK_ADDRESSES);
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont13(int loopDepth) 
	{
															#line 1269 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ChangeQuorumCheckerSetCoordinatorsKey") .detail("CurrentCoordinators", old.toString()) .detail("NewCoordinators", conn->toString());
															#line 1272 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr->set(coordinatorsKey, conn->toString());
															#line 1273 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(Optional<CoordinatorsResult>()); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5720 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(Optional<CoordinatorsResult>());
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when2(Void const& _,int loopDepth) 
	{
															#line 1266 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::COORDINATOR_UNREACHABLE);
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5when2(Void && _,int loopDepth) 
	{
															#line 1266 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumCheckerActor*>(this)->SAV<Optional<CoordinatorsResult>>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumCheckerActorState(); static_cast<ChangeQuorumCheckerActor*>(this)->destroy(); return 0; }
															#line 5756 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumCheckerActor*>(this)->SAV< Optional<CoordinatorsResult> >::value()) Optional<CoordinatorsResult>(CoordinatorsResult::COORDINATOR_UNREACHABLE);
		this->~ChangeQuorumCheckerActorState();
		static_cast<ChangeQuorumCheckerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumCheckerActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 5, Void >::remove();
		static_cast<ChangeQuorumCheckerActor*>(this)->ActorCallback< ChangeQuorumCheckerActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumCheckerActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ChangeQuorumCheckerActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString* conn;
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::string newName;
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool disableConfigDB;
															#line 1162 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Optional<ClusterConnectionString> clusterConnectionStringOptional;
															#line 1170 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString old;
															#line 1176 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> desiredCoordinators;
															#line 5899 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via changeQuorumChecker()
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ChangeQuorumCheckerActor final : public Actor<Optional<CoordinatorsResult>>, public ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >, public ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >, public ActorCallback< ChangeQuorumCheckerActor, 2, Void >, public ActorCallback< ChangeQuorumCheckerActor, 3, Void >, public ActorCallback< ChangeQuorumCheckerActor, 4, Void >, public ActorCallback< ChangeQuorumCheckerActor, 5, Void >, public ActorCallback< ChangeQuorumCheckerActor, 6, Void >, public FastAllocated<ChangeQuorumCheckerActor>, public ChangeQuorumCheckerActorState<ChangeQuorumCheckerActor> {
															#line 5904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ChangeQuorumCheckerActor>::operator new;
	using FastAllocated<ChangeQuorumCheckerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5328413485742110720UL, 12675828290282356736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<CoordinatorsResult>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 2, Void >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 3, Void >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 4, Void >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 5, Void >;
friend struct ActorCallback< ChangeQuorumCheckerActor, 6, Void >;
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ChangeQuorumCheckerActor(Transaction* const& tr,ClusterConnectionString* const& conn,std::string const& newName,bool const& disableConfigDB) 
															#line 5927 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Optional<CoordinatorsResult>>(),
		   ChangeQuorumCheckerActorState<ChangeQuorumCheckerActor>(tr, conn, newName, disableConfigDB),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("changeQuorumChecker", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2226565757528493312UL, 6695731912205238272UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumCheckerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("changeQuorumChecker");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("changeQuorumChecker", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 0, Optional<ClusterConnectionString> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 1, std::vector<NetworkAddress> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ChangeQuorumCheckerActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Optional<CoordinatorsResult>> changeQuorumChecker( Transaction* const& tr, ClusterConnectionString* const& conn, std::string const& newName, bool const& disableConfigDB ) {
															#line 1157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Optional<CoordinatorsResult>>(new ChangeQuorumCheckerActor(tr, conn, newName, disableConfigDB));
															#line 5965 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1275 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 5970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via changeQuorum()
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ChangeQuorumActor>
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ChangeQuorumActorState {
															#line 5977 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ChangeQuorumActorState(Database const& cx,Reference<IQuorumChange> const& change) 
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   change(change),
															#line 1277 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx),
															#line 1278 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   retries(0),
															#line 1279 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   desiredCoordinators(),
															#line 1280 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   notEnoughMachineResults(0)
															#line 5994 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("changeQuorum", reinterpret_cast<unsigned long>(this));

	}
	~ChangeQuorumActorState() 
	{
		fdb_probe_actor_destroy("changeQuorum", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1282 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 6009 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ChangeQuorumActorState();
		static_cast<ChangeQuorumActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1284 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<ClusterConnectionString>> __when_expr_0 = getClusterConnectionStringFromStorageServer(&tr);
															#line 1284 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6042 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 1;
															#line 1284 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6047 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1379 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("RetryQuorumChange").error(e).detail("Retries", retries);
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_8 = tr.onError(e);
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6073 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_8.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 8;
															#line 1380 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 8, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6078 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1287 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!clusterConnectionStringOptional.present())
															#line 6093 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1288 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::BAD_DATABASE_STATE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6097 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::BAD_DATABASE_STATE);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1292 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		oldClusterConnectionString = clusterConnectionStringOptional.get();
															#line 1293 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		oldClusterKeyName = oldClusterConnectionString.clusterKeyName();
															#line 1295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<NetworkAddress>> __when_expr_1 = oldClusterConnectionString.tryResolveHostnames();
															#line 1295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 2;
															#line 1295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6116 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<ClusterConnectionString> const& __clusterConnectionStringOptional,int loopDepth) 
	{
															#line 1284 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		clusterConnectionStringOptional = __clusterConnectionStringOptional;
															#line 6125 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<ClusterConnectionString> && __clusterConnectionStringOptional,int loopDepth) 
	{
		clusterConnectionStringOptional = std::move(__clusterConnectionStringOptional);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1296 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		result = CoordinatorsResult::SUCCESS;
															#line 1297 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!desiredCoordinators.size())
															#line 6206 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1298 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<NetworkAddress>> __when_expr_2 = change->getDesiredCoordinators(&tr, oldCoordinators, Reference<ClusterConnectionMemoryRecord>( new ClusterConnectionMemoryRecord(oldClusterConnectionString)), result);
															#line 1298 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 3;
															#line 1298 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6217 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(std::vector<NetworkAddress> const& __oldCoordinators,int loopDepth) 
	{
															#line 1295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		oldCoordinators = __oldCoordinators;
															#line 6231 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(std::vector<NetworkAddress> && __oldCoordinators,int loopDepth) 
	{
		oldCoordinators = std::move(__oldCoordinators);
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 1307 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (result == CoordinatorsResult::NOT_ENOUGH_MACHINES && notEnoughMachineResults < 1)
															#line 6310 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1310 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			notEnoughMachineResults++;
															#line 1311 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(1.0);
															#line 1311 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6318 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 4;
															#line 1311 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 3, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6323 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(std::vector<NetworkAddress> const& _desiredCoordinators,int loopDepth) 
	{
															#line 1304 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCoordinators = _desiredCoordinators;
															#line 6337 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(std::vector<NetworkAddress> && _desiredCoordinators,int loopDepth) 
	{
															#line 1304 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCoordinators = _desiredCoordinators;
															#line 6346 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(std::vector<NetworkAddress> const& _desiredCoordinators,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_desiredCoordinators, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(std::vector<NetworkAddress> && _desiredCoordinators,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_desiredCoordinators), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
															#line 1315 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (result != CoordinatorsResult::SUCCESS)
															#line 6430 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1316 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(result); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6434 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(std::move(result)); // state_var_RVO
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1317 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!desiredCoordinators.size())
															#line 6442 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1318 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::INVALID_NETWORK_ADDRESSES); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6446 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::INVALID_NETWORK_ADDRESSES);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1319 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::sort(desiredCoordinators.begin(), desiredCoordinators.end());
															#line 1321 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::string newName = change->getDesiredClusterKeyName();
															#line 1322 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (newName.empty())
															#line 6458 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1323 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			newName = oldClusterKeyName.toString();
															#line 6462 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1325 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (oldCoordinators == desiredCoordinators && oldClusterKeyName == newName)
															#line 6466 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1326 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(retries ? CoordinatorsResult::SUCCESS : CoordinatorsResult::SAME_NETWORK_ADDRESSES); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6470 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(retries ? CoordinatorsResult::SUCCESS : CoordinatorsResult::SAME_NETWORK_ADDRESSES);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1328 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		newClusterConnectionString = ClusterConnectionString(desiredCoordinators, StringRef(newName + ':' + deterministicRandom()->randomAlphaNumeric(32)));
															#line 1330 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		newClusterKeyName = newClusterConnectionString.clusterKeyName();
															#line 1332 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (g_network->isSimulated())
															#line 6482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1333 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(int i = 0;i < (desiredCoordinators.size() / 2) + 1;i++) {
															#line 1334 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto process = g_simulator->getProcessByAddress(desiredCoordinators[i]);
															#line 1335 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ASSERT(process->isReliable() || process->rebooting);
															#line 1337 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				g_simulator->protectedAddresses.insert(process->addresses.address);
															#line 1338 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (process->addresses.secondaryAddress.present())
															#line 6494 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1339 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					g_simulator->protectedAddresses.insert(process->addresses.secondaryAddress.get());
															#line 6498 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
															#line 1341 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				TraceEvent("ProtectCoordinator").detail("Address", desiredCoordinators[i]).backtrace();
															#line 6502 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1345 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("AttemptingQuorumChange") .detail("FromCS", oldClusterConnectionString.toString()) .detail("ToCS", newClusterConnectionString.toString());
															#line 1348 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		CODE_PROBE(oldClusterKeyName != newClusterKeyName, "Quorum change with new name");
															#line 1349 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		CODE_PROBE(oldClusterKeyName == newClusterKeyName, "Quorum change with unchanged name");
															#line 1351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		leaderServers = std::vector<Future<Optional<LeaderInfo>>>();
															#line 1352 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		coord = ClientCoordinators(Reference<ClusterConnectionMemoryRecord>( new ClusterConnectionMemoryRecord(newClusterConnectionString)));
															#line 1355 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!change->getDesiredClusterKeyName().empty())
															#line 6517 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1356 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<CheckDescriptorMutableReply> __when_expr_4 = coord.clientLeaderServers[0].checkDescriptorMutable.getReply(CheckDescriptorMutableRequest());
															#line 1356 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6523 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont8when1(__when_expr_4.get(), loopDepth); };
			static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 5;
															#line 1356 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6528 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 1312 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.reset();
															#line 6542 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 1312 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.reset();
															#line 6551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
															#line 1362 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		leaderServers.reserve(coord.clientLeaderServers.size());
															#line 1363 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < coord.clientLeaderServers.size();i++) {
															#line 1364 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			leaderServers.push_back(retryBrokenPromise(coord.clientLeaderServers[i].getLeader, GetLeaderRequest(coord.clusterKey, UID()), TaskPriority::CoordinationReply));
															#line 6639 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1368 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_5 = waitForAll(leaderServers);
															#line 1367 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6645 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont11when1(__when_expr_5.get(), loopDepth); };
															#line 1369 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(5.0);
															#line 6649 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont11when2(__when_expr_6.get(), loopDepth); };
		static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 6;
															#line 1368 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 5, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 1369 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 6, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6656 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont19(CheckDescriptorMutableReply const& mutabilityReply,int loopDepth) 
	{
															#line 1358 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!mutabilityReply.isMutable)
															#line 6665 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1359 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::BAD_DATABASE_STATE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::BAD_DATABASE_STATE);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont19(CheckDescriptorMutableReply && mutabilityReply,int loopDepth) 
	{
															#line 1358 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!mutabilityReply.isMutable)
															#line 6683 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1359 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::BAD_DATABASE_STATE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6687 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::BAD_DATABASE_STATE);
			this->~ChangeQuorumActorState();
			static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(CheckDescriptorMutableReply const& mutabilityReply,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont19(mutabilityReply, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(CheckDescriptorMutableReply && mutabilityReply,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont19(std::move(mutabilityReply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*,CheckDescriptorMutableReply const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*,CheckDescriptorMutableReply && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont11cont1(int loopDepth) 
	{
															#line 1374 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(coordinatorsKey, newClusterConnectionString.toString());
															#line 1376 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_7 = tr.commit();
															#line 1376 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6780 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont11cont1when1(__when_expr_7.get(), loopDepth); };
		static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 7;
															#line 1376 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ChangeQuorumActor, 7, Void >*>(static_cast<ChangeQuorumActor*>(this)));
															#line 6785 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11when2(Void const& _,int loopDepth) 
	{
															#line 1370 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::COORDINATOR_UNREACHABLE);
		this->~ChangeQuorumActorState();
		static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont11when2(Void && _,int loopDepth) 
	{
															#line 1370 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ChangeQuorumActor*>(this)->SAV<CoordinatorsResult>::futures) { (void)(CoordinatorsResult::COORDINATOR_UNREACHABLE); this->~ChangeQuorumActorState(); static_cast<ChangeQuorumActor*>(this)->destroy(); return 0; }
															#line 6818 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ChangeQuorumActor*>(this)->SAV< CoordinatorsResult >::value()) CoordinatorsResult(CoordinatorsResult::COORDINATOR_UNREACHABLE);
		this->~ChangeQuorumActorState();
		static_cast<ChangeQuorumActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 5, Void >::remove();
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont11when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont11when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont11cont3(Void const& _,int loopDepth) 
	{
															#line 1377 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(false);
															#line 6951 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11cont3(Void && _,int loopDepth) 
	{
															#line 1377 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(false);
															#line 6960 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1loopBody1cont11cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
															#line 1381 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		++retries;
															#line 7057 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
															#line 1381 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		++retries;
															#line 7066 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<ChangeQuorumActor*>(this)->actor_wait_state > 0) static_cast<ChangeQuorumActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeQuorumActor*>(this)->ActorCallback< ChangeQuorumActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< ChangeQuorumActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< ChangeQuorumActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), 8);

	}
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<IQuorumChange> change;
															#line 1277 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 1278 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int retries;
															#line 1279 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> desiredCoordinators;
															#line 1280 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int notEnoughMachineResults;
															#line 1284 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Optional<ClusterConnectionString> clusterConnectionStringOptional;
															#line 1292 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString oldClusterConnectionString;
															#line 1293 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Key oldClusterKeyName;
															#line 1295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> oldCoordinators;
															#line 1296 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CoordinatorsResult result;
															#line 1328 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClusterConnectionString newClusterConnectionString;
															#line 1330 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Key newClusterKeyName;
															#line 1351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<Future<Optional<LeaderInfo>>> leaderServers;
															#line 1352 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClientCoordinators coord;
															#line 7176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via changeQuorum()
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ChangeQuorumActor final : public Actor<CoordinatorsResult>, public ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >, public ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >, public ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >, public ActorCallback< ChangeQuorumActor, 3, Void >, public ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >, public ActorCallback< ChangeQuorumActor, 5, Void >, public ActorCallback< ChangeQuorumActor, 6, Void >, public ActorCallback< ChangeQuorumActor, 7, Void >, public ActorCallback< ChangeQuorumActor, 8, Void >, public FastAllocated<ChangeQuorumActor>, public ChangeQuorumActorState<ChangeQuorumActor> {
															#line 7181 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ChangeQuorumActor>::operator new;
	using FastAllocated<ChangeQuorumActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12562814177430369792UL, 861730360597012480UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<CoordinatorsResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >;
friend struct ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >;
friend struct ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >;
friend struct ActorCallback< ChangeQuorumActor, 3, Void >;
friend struct ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >;
friend struct ActorCallback< ChangeQuorumActor, 5, Void >;
friend struct ActorCallback< ChangeQuorumActor, 6, Void >;
friend struct ActorCallback< ChangeQuorumActor, 7, Void >;
friend struct ActorCallback< ChangeQuorumActor, 8, Void >;
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ChangeQuorumActor(Database const& cx,Reference<IQuorumChange> const& change) 
															#line 7206 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<CoordinatorsResult>(),
		   ChangeQuorumActorState<ChangeQuorumActor>(cx, change),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("changeQuorum", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11745850326369364480UL, 4959262309748762880UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeQuorumActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("changeQuorum");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("changeQuorum", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ChangeQuorumActor, 0, Optional<ClusterConnectionString> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ChangeQuorumActor, 1, std::vector<NetworkAddress> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ChangeQuorumActor, 2, std::vector<NetworkAddress> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ChangeQuorumActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ChangeQuorumActor, 4, CheckDescriptorMutableReply >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ChangeQuorumActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ChangeQuorumActor, 7, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< ChangeQuorumActor, 8, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<CoordinatorsResult> changeQuorum( Database const& cx, Reference<IQuorumChange> const& change ) {
															#line 1276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<CoordinatorsResult>(new ChangeQuorumActor(cx, change));
															#line 7246 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1385 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

struct NameQuorumChange final : IQuorumChange {
	std::string newName;
	Reference<IQuorumChange> otherChange;
	explicit NameQuorumChange(std::string const& newName, Reference<IQuorumChange> const& otherChange)
	  : newName(newName), otherChange(otherChange) {}
	Future<std::vector<NetworkAddress>> getDesiredCoordinators(Transaction* tr,
	                                                           std::vector<NetworkAddress> oldCoordinators,
	                                                           Reference<IClusterConnectionRecord> ccr,
	                                                           CoordinatorsResult& t) override {
		return otherChange->getDesiredCoordinators(tr, oldCoordinators, ccr, t);
	}
	std::string getDesiredClusterKeyName() const override { return newName; }
};
Reference<IQuorumChange> nameQuorumChange(std::string const& name, Reference<IQuorumChange> const& other) {
	return Reference<IQuorumChange>(new NameQuorumChange(name, other));
}

struct AutoQuorumChange final : IQuorumChange {
	int desired;
	explicit AutoQuorumChange(int desired) : desired(desired) {}

	Future<std::vector<NetworkAddress>> getDesiredCoordinators(Transaction* tr,
	                                                           std::vector<NetworkAddress> oldCoordinators,
	                                                           Reference<IClusterConnectionRecord> ccr,
	                                                           CoordinatorsResult& err) override {
		return getDesired(Reference<AutoQuorumChange>::addRef(this), tr, oldCoordinators, ccr, &err);
	}

																#line 7279 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via getRedundancy()
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetRedundancyActor>
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetRedundancyActorState {
															#line 7285 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetRedundancyActorState(AutoQuorumChange* const& self,Transaction* const& tr) 
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : self(self),
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(tr),
															#line 1415 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fStorageReplicas(tr->get("storage_replicas"_sr.withPrefix(configKeysPrefix))),
															#line 1416 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fLogReplicas(tr->get("log_replicas"_sr.withPrefix(configKeysPrefix)))
															#line 7298 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getRedundancy", reinterpret_cast<unsigned long>(this));

	}
	~GetRedundancyActorState() 
	{
		fdb_probe_actor_destroy("getRedundancy", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1417 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(fStorageReplicas) && success(fLogReplicas);
															#line 1417 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetRedundancyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7315 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetRedundancyActor*>(this)->actor_wait_state = 1;
															#line 1417 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetRedundancyActor, 0, Void >*>(static_cast<GetRedundancyActor*>(this)));
															#line 7320 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetRedundancyActorState();
		static_cast<GetRedundancyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1418 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		int redundancy = std::min(atoi(fStorageReplicas.get().get().toString().c_str()), atoi(fLogReplicas.get().get().toString().c_str()));
															#line 1421 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetRedundancyActor*>(this)->SAV<int>::futures) { (void)(redundancy); this->~GetRedundancyActorState(); static_cast<GetRedundancyActor*>(this)->destroy(); return 0; }
															#line 7345 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetRedundancyActor*>(this)->SAV< int >::value()) int(redundancy);
		this->~GetRedundancyActorState();
		static_cast<GetRedundancyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1418 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		int redundancy = std::min(atoi(fStorageReplicas.get().get().toString().c_str()), atoi(fLogReplicas.get().get().toString().c_str()));
															#line 1421 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetRedundancyActor*>(this)->SAV<int>::futures) { (void)(redundancy); this->~GetRedundancyActorState(); static_cast<GetRedundancyActor*>(this)->destroy(); return 0; }
															#line 7359 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetRedundancyActor*>(this)->SAV< int >::value()) int(redundancy);
		this->~GetRedundancyActorState();
		static_cast<GetRedundancyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetRedundancyActor*>(this)->actor_wait_state > 0) static_cast<GetRedundancyActor*>(this)->actor_wait_state = 0;
		static_cast<GetRedundancyActor*>(this)->ActorCallback< GetRedundancyActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetRedundancyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRedundancy", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRedundancyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRedundancy", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRedundancyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRedundancy", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRedundancyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRedundancy", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetRedundancyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRedundancy", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRedundancyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRedundancy", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	AutoQuorumChange* self;
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1415 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<Optional<Value>> fStorageReplicas;
															#line 1416 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<Optional<Value>> fLogReplicas;
															#line 7450 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getRedundancy()
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetRedundancyActor final : public Actor<int>, public ActorCallback< GetRedundancyActor, 0, Void >, public FastAllocated<GetRedundancyActor>, public GetRedundancyActorState<GetRedundancyActor> {
															#line 7455 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetRedundancyActor>::operator new;
	using FastAllocated<GetRedundancyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15752461682149526784UL, 7060542805840554496UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetRedundancyActor, 0, Void >;
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetRedundancyActor(AutoQuorumChange* const& self,Transaction* const& tr) 
															#line 7472 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<int>(),
		   GetRedundancyActorState<GetRedundancyActor>(self, tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getRedundancy", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17051584921099603200UL, 8631560012367626240UL);
		ActorExecutionContextHelper __helper(static_cast<GetRedundancyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getRedundancy");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getRedundancy", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetRedundancyActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] static Future<int> getRedundancy( AutoQuorumChange* const& self, Transaction* const& tr ) {
															#line 1414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<int>(new GetRedundancyActor(self, tr));
															#line 7504 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1423 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

																#line 7509 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via isAcceptable()
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class IsAcceptableActor>
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IsAcceptableActorState {
															#line 7515 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IsAcceptableActorState(AutoQuorumChange* const& self,Transaction* const& tr,std::vector<NetworkAddress> const& oldCoordinators,Reference<IClusterConnectionRecord> const& ccr,int const& desiredCount,std::set<AddressExclusion>* const& excluded) 
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : self(self),
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(tr),
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   oldCoordinators(oldCoordinators),
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   ccr(ccr),
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   desiredCount(desiredCount),
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   excluded(excluded)
															#line 7532 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("isAcceptable", reinterpret_cast<unsigned long>(this));

	}
	~IsAcceptableActorState() 
	{
		fdb_probe_actor_destroy("isAcceptable", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1430 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ClusterConnectionString cs = ccr->getConnectionString();
															#line 1431 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (oldCoordinators.size() != cs.hostnames.size() + cs.coords.size())
															#line 7549 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1432 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7553 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1436 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (oldCoordinators.size() < desiredCount)
															#line 7561 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1437 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7565 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1438 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (oldCoordinators.size() % 2 != 1)
															#line 7573 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1439 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7577 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1442 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( auto& c : oldCoordinators ) {
															#line 1443 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (addressExcluded(*excluded, c))
															#line 7587 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1444 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7591 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
					this->~IsAcceptableActorState();
					static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 1449 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::sort(oldCoordinators.begin(), oldCoordinators.end());
															#line 1450 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(int i = 1;i < oldCoordinators.size();i++) {
															#line 1451 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (oldCoordinators[i - 1].ip == oldCoordinators[i].ip)
															#line 7604 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1452 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7608 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
					this->~IsAcceptableActorState();
					static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 1455 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ClientCoordinators coord(ccr);
															#line 1456 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::vector<Future<Optional<LeaderInfo>>> leaderServers;
															#line 1457 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			leaderServers.reserve(coord.clientLeaderServers.size());
															#line 1458 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for(int i = 0;i < coord.clientLeaderServers.size();i++) {
															#line 1459 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (coord.clientLeaderServers[i].hostname.present())
															#line 7625 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1460 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					leaderServers.push_back(retryGetReplyFromHostname(GetLeaderRequest(coord.clusterKey, UID()), coord.clientLeaderServers[i].hostname.get(), WLTOKEN_CLIENTLEADERREG_GETLEADER, TaskPriority::CoordinationReply));
															#line 7629 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 1465 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					leaderServers.push_back(retryBrokenPromise(coord.clientLeaderServers[i].getLeader, GetLeaderRequest(coord.clusterKey, UID()), TaskPriority::CoordinationReply));
															#line 7635 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
			}
															#line 1470 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<std::vector<Optional<LeaderInfo>>>> __when_expr_0 = timeout(getAll(leaderServers), CLIENT_KNOBS->IS_ACCEPTABLE_DELAY);
															#line 1470 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IsAcceptableActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7642 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<IsAcceptableActor*>(this)->actor_wait_state = 1;
															#line 1470 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*>(static_cast<IsAcceptableActor*>(this)));
															#line 7647 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IsAcceptableActorState();
		static_cast<IsAcceptableActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<std::vector<Optional<LeaderInfo>>> const& results,int loopDepth) 
	{
															#line 1472 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!results.present())
															#line 7670 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1473 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7674 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
			this->~IsAcceptableActorState();
			static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1475 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& r : results.get() ) {
															#line 1476 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!r.present())
															#line 7684 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1477 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7688 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1481 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(true); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(true);
		this->~IsAcceptableActorState();
		static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<std::vector<Optional<LeaderInfo>>> && results,int loopDepth) 
	{
															#line 1472 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!results.present())
															#line 7709 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1473 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7713 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
			this->~IsAcceptableActorState();
			static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1475 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& r : results.get() ) {
															#line 1476 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!r.present())
															#line 7723 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1477 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7727 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(false);
				this->~IsAcceptableActorState();
				static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 1481 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IsAcceptableActor*>(this)->SAV<bool>::futures) { (void)(true); this->~IsAcceptableActorState(); static_cast<IsAcceptableActor*>(this)->destroy(); return 0; }
															#line 7736 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IsAcceptableActor*>(this)->SAV< bool >::value()) bool(true);
		this->~IsAcceptableActorState();
		static_cast<IsAcceptableActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<std::vector<Optional<LeaderInfo>>> const& results,int loopDepth) 
	{
		loopDepth = a_body1cont1(results, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<std::vector<Optional<LeaderInfo>>> && results,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(results), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IsAcceptableActor*>(this)->actor_wait_state > 0) static_cast<IsAcceptableActor*>(this)->actor_wait_state = 0;
		static_cast<IsAcceptableActor*>(this)->ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >::remove();

	}
	void a_callback_fire(ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*,Optional<std::vector<Optional<LeaderInfo>>> const& value) 
	{
		fdb_probe_actor_enter("isAcceptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IsAcceptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("isAcceptable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*,Optional<std::vector<Optional<LeaderInfo>>> && value) 
	{
		fdb_probe_actor_enter("isAcceptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IsAcceptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("isAcceptable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*,Error err) 
	{
		fdb_probe_actor_enter("isAcceptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IsAcceptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("isAcceptable", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	AutoQuorumChange* self;
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> oldCoordinators;
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<IClusterConnectionRecord> ccr;
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int desiredCount;
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<AddressExclusion>* excluded;
															#line 7831 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via isAcceptable()
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IsAcceptableActor final : public Actor<bool>, public ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >, public FastAllocated<IsAcceptableActor>, public IsAcceptableActorState<IsAcceptableActor> {
															#line 7836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<IsAcceptableActor>::operator new;
	using FastAllocated<IsAcceptableActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1941964052630162432UL, 4133873664639733760UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >;
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IsAcceptableActor(AutoQuorumChange* const& self,Transaction* const& tr,std::vector<NetworkAddress> const& oldCoordinators,Reference<IClusterConnectionRecord> const& ccr,int const& desiredCount,std::set<AddressExclusion>* const& excluded) 
															#line 7853 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<bool>(),
		   IsAcceptableActorState<IsAcceptableActor>(self, tr, oldCoordinators, ccr, desiredCount, excluded),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("isAcceptable", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9830863436459782912UL, 6089237774462111488UL);
		ActorExecutionContextHelper __helper(static_cast<IsAcceptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("isAcceptable");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("isAcceptable", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IsAcceptableActor, 0, Optional<std::vector<Optional<LeaderInfo>>> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] static Future<bool> isAcceptable( AutoQuorumChange* const& self, Transaction* const& tr, std::vector<NetworkAddress> const& oldCoordinators, Reference<IClusterConnectionRecord> const& ccr, int const& desiredCount, std::set<AddressExclusion>* const& excluded ) {
															#line 1424 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<bool>(new IsAcceptableActor(self, tr, oldCoordinators, ccr, desiredCount, excluded));
															#line 7885 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1483 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

																#line 7890 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
// This generated class is to be used only via getDesired()
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetDesiredActor>
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDesiredActorState {
															#line 7896 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDesiredActorState(Reference<AutoQuorumChange> const& self,Transaction* const& tr,std::vector<NetworkAddress> const& oldCoordinators,Reference<IClusterConnectionRecord> const& ccr,CoordinatorsResult* const& err) 
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : self(self),
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(tr),
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   oldCoordinators(oldCoordinators),
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   ccr(ccr),
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   err(err),
															#line 1489 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   desiredCount(self->desired)
															#line 7913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getDesired", reinterpret_cast<unsigned long>(this));

	}
	~GetDesiredActorState() 
	{
		fdb_probe_actor_destroy("getDesired", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1491 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (desiredCount == -1)
															#line 7928 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1492 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<int> __when_expr_0 = getRedundancy(self.getPtr(), tr);
															#line 1492 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7934 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetDesiredActor*>(this)->actor_wait_state = 1;
															#line 1492 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 0, int >*>(static_cast<GetDesiredActor*>(this)));
															#line 7939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetDesiredActorState();
		static_cast<GetDesiredActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1496 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<AddressExclusion>> __when_expr_1 = getAllExcludedServers(tr);
															#line 1496 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7969 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetDesiredActor*>(this)->actor_wait_state = 2;
															#line 1496 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*>(static_cast<GetDesiredActor*>(this)));
															#line 7974 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(int const& redundancy,int loopDepth) 
	{
															#line 1493 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCount = redundancy * 2 - 1;
															#line 7983 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(int && redundancy,int loopDepth) 
	{
															#line 1493 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		desiredCount = redundancy * 2 - 1;
															#line 7992 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(int const& redundancy,int loopDepth) 
	{
		loopDepth = a_body1cont2(redundancy, loopDepth);

		return loopDepth;
	}
	int a_body1when1(int && redundancy,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(redundancy), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state > 0) static_cast<GetDesiredActor*>(this)->actor_wait_state = 0;
		static_cast<GetDesiredActor*>(this)->ActorCallback< GetDesiredActor, 0, int >::remove();

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 0, int >*,int const& value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 0, int >*,int && value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetDesiredActor, 0, int >*,Error err) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(std::vector<AddressExclusion> const& excl,int loopDepth) 
	{
															#line 1497 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		excluded = std::set<AddressExclusion>(excl.begin(), excl.end());
															#line 1499 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<ProcessData>> __when_expr_2 = getWorkers(tr);
															#line 1499 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8080 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetDesiredActor*>(this)->actor_wait_state = 3;
															#line 1499 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*>(static_cast<GetDesiredActor*>(this)));
															#line 8085 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(std::vector<AddressExclusion> && excl,int loopDepth) 
	{
															#line 1497 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		excluded = std::set<AddressExclusion>(excl.begin(), excl.end());
															#line 1499 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<std::vector<ProcessData>> __when_expr_2 = getWorkers(tr);
															#line 1499 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8098 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetDesiredActor*>(this)->actor_wait_state = 3;
															#line 1499 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*>(static_cast<GetDesiredActor*>(this)));
															#line 8103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(std::vector<AddressExclusion> const& excl,int loopDepth) 
	{
		loopDepth = a_body1cont4(excl, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::vector<AddressExclusion> && excl,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(excl), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state > 0) static_cast<GetDesiredActor*>(this)->actor_wait_state = 0;
		static_cast<GetDesiredActor*>(this)->ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >::remove();

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> const& value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> && value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*,Error err) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(std::vector<ProcessData> const& _workers,int loopDepth) 
	{
															#line 1500 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		workers = _workers;
															#line 1502 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::map<NetworkAddress, LocalityData> addr_locality;
															#line 1503 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto w : workers ) {
															#line 1504 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			addr_locality[w.address] = w.locality;
															#line 8193 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1509 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool checkAcceptable = true;
															#line 1510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<Optional<Standalone<StringRef>>> checkDuplicates;
															#line 1511 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto addr : oldCoordinators ) {
															#line 1512 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto findResult = addr_locality.find(addr);
															#line 1513 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (findResult == addr_locality.end() || checkDuplicates.count(findResult->second.zoneId()))
															#line 8205 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1514 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				checkAcceptable = false;
															#line 8209 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				break;
			}
															#line 1517 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			checkDuplicates.insert(findResult->second.zoneId());
															#line 8214 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1520 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (checkAcceptable)
															#line 8218 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1521 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<bool> __when_expr_3 = isAcceptable(self.getPtr(), tr, oldCoordinators, ccr, desiredCount, &excluded);
															#line 1521 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8224 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
			static_cast<GetDesiredActor*>(this)->actor_wait_state = 4;
															#line 1521 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 3, bool >*>(static_cast<GetDesiredActor*>(this)));
															#line 8229 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(std::vector<ProcessData> && _workers,int loopDepth) 
	{
															#line 1500 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		workers = _workers;
															#line 1502 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::map<NetworkAddress, LocalityData> addr_locality;
															#line 1503 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto w : workers ) {
															#line 1504 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			addr_locality[w.address] = w.locality;
															#line 8249 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1509 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool checkAcceptable = true;
															#line 1510 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<Optional<Standalone<StringRef>>> checkDuplicates;
															#line 1511 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto addr : oldCoordinators ) {
															#line 1512 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto findResult = addr_locality.find(addr);
															#line 1513 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (findResult == addr_locality.end() || checkDuplicates.count(findResult->second.zoneId()))
															#line 8261 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1514 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				checkAcceptable = false;
															#line 8265 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				break;
			}
															#line 1517 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			checkDuplicates.insert(findResult->second.zoneId());
															#line 8270 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1520 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (checkAcceptable)
															#line 8274 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1521 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<bool> __when_expr_3 = isAcceptable(self.getPtr(), tr, oldCoordinators, ccr, desiredCount, &excluded);
															#line 1521 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetDesiredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8280 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
			static_cast<GetDesiredActor*>(this)->actor_wait_state = 4;
															#line 1521 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetDesiredActor, 3, bool >*>(static_cast<GetDesiredActor*>(this)));
															#line 8285 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4when1(std::vector<ProcessData> const& _workers,int loopDepth) 
	{
		loopDepth = a_body1cont5(_workers, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(std::vector<ProcessData> && _workers,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_workers), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state > 0) static_cast<GetDesiredActor*>(this)->actor_wait_state = 0;
		static_cast<GetDesiredActor*>(this)->ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >::remove();

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*,std::vector<ProcessData> const& value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*,std::vector<ProcessData> && value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*,Error err) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 1528 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<NetworkAddress> chosen;
															#line 1529 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		self->addDesiredWorkers(chosen, workers, desiredCount, excluded);
															#line 1531 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (chosen.size() < desiredCount)
															#line 8378 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1532 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (chosen.empty() || chosen.size() < oldCoordinators.size())
															#line 8382 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1533 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				TraceEvent("NotEnoughMachinesForCoordinators") .detail("EligibleWorkers", workers.size()) .detail("ChosenWorkers", chosen.size()) .detail("DesiredCoordinators", desiredCount) .detail("CurrentCoordinators", oldCoordinators.size());
															#line 1538 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				*err = CoordinatorsResult::NOT_ENOUGH_MACHINES;
															#line 1539 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<GetDesiredActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(std::vector<NetworkAddress>()); this->~GetDesiredActorState(); static_cast<GetDesiredActor*>(this)->destroy(); return 0; }
															#line 8390 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<GetDesiredActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(std::vector<NetworkAddress>());
				this->~GetDesiredActorState();
				static_cast<GetDesiredActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1541 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			chosen.resize((chosen.size() - 1) | 1);
															#line 8398 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1544 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetDesiredActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(chosen); this->~GetDesiredActorState(); static_cast<GetDesiredActor*>(this)->destroy(); return 0; }
															#line 8402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetDesiredActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(chosen);
		this->~GetDesiredActorState();
		static_cast<GetDesiredActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10(bool const& ok,int loopDepth) 
	{
															#line 1522 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 8414 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1523 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			*err = CoordinatorsResult::SAME_NETWORK_ADDRESSES;
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetDesiredActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(oldCoordinators); this->~GetDesiredActorState(); static_cast<GetDesiredActor*>(this)->destroy(); return 0; }
															#line 8420 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetDesiredActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(std::move(oldCoordinators)); // state_var_RVO
			this->~GetDesiredActorState();
			static_cast<GetDesiredActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(bool && ok,int loopDepth) 
	{
															#line 1522 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 8434 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1523 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			*err = CoordinatorsResult::SAME_NETWORK_ADDRESSES;
															#line 1524 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<GetDesiredActor*>(this)->SAV<std::vector<NetworkAddress>>::futures) { (void)(oldCoordinators); this->~GetDesiredActorState(); static_cast<GetDesiredActor*>(this)->destroy(); return 0; }
															#line 8440 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<GetDesiredActor*>(this)->SAV< std::vector<NetworkAddress> >::value()) std::vector<NetworkAddress>(std::move(oldCoordinators)); // state_var_RVO
			this->~GetDesiredActorState();
			static_cast<GetDesiredActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(bool const& ok,int loopDepth) 
	{
		loopDepth = a_body1cont10(ok, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(bool && ok,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(ok), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetDesiredActor*>(this)->actor_wait_state > 0) static_cast<GetDesiredActor*>(this)->actor_wait_state = 0;
		static_cast<GetDesiredActor*>(this)->ActorCallback< GetDesiredActor, 3, bool >::remove();

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 3, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetDesiredActor, 3, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetDesiredActor, 3, bool >*,Error err) 
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<AutoQuorumChange> self;
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<NetworkAddress> oldCoordinators;
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Reference<IClusterConnectionRecord> ccr;
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CoordinatorsResult* err;
															#line 1489 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int desiredCount;
															#line 1497 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<AddressExclusion> excluded;
															#line 1500 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<ProcessData> workers;
															#line 8541 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getDesired()
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetDesiredActor final : public Actor<std::vector<NetworkAddress>>, public ActorCallback< GetDesiredActor, 0, int >, public ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >, public ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >, public ActorCallback< GetDesiredActor, 3, bool >, public FastAllocated<GetDesiredActor>, public GetDesiredActorState<GetDesiredActor> {
															#line 8546 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetDesiredActor>::operator new;
	using FastAllocated<GetDesiredActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3830385086023952640UL, 2663178220359692800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<NetworkAddress>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetDesiredActor, 0, int >;
friend struct ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >;
friend struct ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >;
friend struct ActorCallback< GetDesiredActor, 3, bool >;
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetDesiredActor(Reference<AutoQuorumChange> const& self,Transaction* const& tr,std::vector<NetworkAddress> const& oldCoordinators,Reference<IClusterConnectionRecord> const& ccr,CoordinatorsResult* const& err) 
															#line 8566 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<NetworkAddress>>(),
		   GetDesiredActorState<GetDesiredActor>(self, tr, oldCoordinators, ccr, err),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getDesired", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3649354580216779264UL, 10244989468105284608UL);
		ActorExecutionContextHelper __helper(static_cast<GetDesiredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getDesired");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getDesired", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetDesiredActor, 0, int >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetDesiredActor, 1, std::vector<AddressExclusion> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetDesiredActor, 2, std::vector<ProcessData> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetDesiredActor, 3, bool >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] static Future<std::vector<NetworkAddress>> getDesired( Reference<AutoQuorumChange> const& self, Transaction* const& tr, std::vector<NetworkAddress> const& oldCoordinators, Reference<IClusterConnectionRecord> const& ccr, CoordinatorsResult* const& err ) {
															#line 1484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<NetworkAddress>>(new GetDesiredActor(self, tr, oldCoordinators, ccr, err));
															#line 8601 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1546 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

	// Select a desired set of workers such that
	// (1) the number of workers at each locality type (e.g., dcid) <= desiredCount; and
	// (2) prefer workers at a locality where less workers has been chosen than other localities: evenly distribute
	// workers.
	void addDesiredWorkers(std::vector<NetworkAddress>& chosen,
	                       const std::vector<ProcessData>& workers,
	                       int desiredCount,
	                       const std::set<AddressExclusion>& excluded) {
		std::vector<ProcessData> remainingWorkers(workers);
		deterministicRandom()->randomShuffle(remainingWorkers);

		std::partition(remainingWorkers.begin(), remainingWorkers.end(), [](const ProcessData& data) {
			return (data.processClass == ProcessClass::CoordinatorClass);
		});

		TraceEvent(SevDebug, "AutoSelectCoordinators").detail("CandidateWorkers", remainingWorkers.size());
		for (auto worker = remainingWorkers.begin(); worker != remainingWorkers.end(); worker++) {
			TraceEvent(SevDebug, "AutoSelectCoordinators")
			    .detail("Worker", worker->processClass.toString())
			    .detail("Address", worker->address.toString())
			    .detail("Locality", worker->locality.toString());
		}
		TraceEvent(SevDebug, "AutoSelectCoordinators").detail("ExcludedAddress", excluded.size());
		for (auto& excludedAddr : excluded) {
			TraceEvent(SevDebug, "AutoSelectCoordinators").detail("ExcludedAddress", excludedAddr.toString());
		}

		std::map<StringRef, int> maxCounts;
		std::map<StringRef, std::map<StringRef, int>> currentCounts;
		std::map<StringRef, int> hardLimits;

		std::vector<StringRef> fields({ "dcid"_sr, "data_hall"_sr, "zoneid"_sr, "machineid"_sr });

		for (auto field = fields.begin(); field != fields.end(); field++) {
			if (field->toString() == "zoneid") {
				hardLimits[*field] = 1;
			} else {
				hardLimits[*field] = desiredCount;
			}
		}

		while (chosen.size() < desiredCount) {
			bool found = false;
			for (auto worker = remainingWorkers.begin(); worker != remainingWorkers.end(); worker++) {
				if (addressExcluded(excluded, worker->address)) {
					continue;
				}
				// Exclude faulty node due to machine assassination
				if (g_network->isSimulated() && !g_simulator->getProcessByAddress(worker->address)->isReliable()) {
					TraceEvent("AutoSelectCoordinators").detail("SkipUnreliableWorker", worker->address.toString());
					continue;
				}
				bool valid = true;
				for (auto field = fields.begin(); field != fields.end(); field++) {
					if (maxCounts[*field] == 0) {
						maxCounts[*field] = 1;
					}
					auto value = worker->locality.get(*field).orDefault(""_sr);
					auto currentCount = currentCounts[*field][value];
					if (currentCount >= maxCounts[*field]) {
						valid = false;
						break;
					}
				}
				if (valid) {
					for (auto field = fields.begin(); field != fields.end(); field++) {
						auto value = worker->locality.get(*field).orDefault(""_sr);
						currentCounts[*field][value] += 1;
					}
					chosen.push_back(worker->address);
					remainingWorkers.erase(worker);
					found = true;
					break;
				}
			}
			if (!found) {
				bool canIncrement = false;
				for (auto field = fields.begin(); field != fields.end(); field++) {
					if (maxCounts[*field] < hardLimits[*field]) {
						maxCounts[*field] += 1;
						canIncrement = true;
						break;
					}
				}
				if (!canIncrement) {
					break;
				}
			}
		}
	}
};
Reference<IQuorumChange> autoQuorumChange(int desired) {
	return Reference<IQuorumChange>(new AutoQuorumChange(desired));
}

															#line 8701 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via excludeServers()
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ExcludeServersActor>
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeServersActorState {
															#line 8708 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeServersActorState(Transaction* const& tr,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   servers(servers),
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed)
															#line 8719 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("excludeServers", reinterpret_cast<unsigned long>(this));

	}
	~ExcludeServersActorState() 
	{
		fdb_probe_actor_destroy("excludeServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1643 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1644 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1645 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1646 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1647 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<AddressExclusion>> __when_expr_0 = failed ? getExcludedFailedServerList(tr) : getExcludedServerList(tr);
															#line 1647 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExcludeServersActor*>(this)->actor_wait_state = 1;
															#line 1647 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*>(static_cast<ExcludeServersActor*>(this)));
															#line 8749 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExcludeServersActorState();
		static_cast<ExcludeServersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AddressExclusion> const& excl,int loopDepth) 
	{
															#line 1648 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<AddressExclusion> exclusions(excl.begin(), excl.end());
															#line 1649 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool containNewExclusion = false;
															#line 1650 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& s : servers ) {
															#line 1651 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (exclusions.find(s) != exclusions.end())
															#line 8778 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
				continue;
			}
															#line 1654 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			containNewExclusion = true;
															#line 1655 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 8786 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1656 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeFailedServersKey(s), StringRef());
															#line 8790 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1658 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeExcludedServersKey(s), StringRef());
															#line 8796 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1662 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (containNewExclusion)
															#line 8801 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1663 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string excludeVersionKey = deterministicRandom()->randomUniqueID().toString();
															#line 1664 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto serversVersionKey = failed ? failedServersVersionKey : excludedServersVersionKey;
															#line 1665 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(serversVersionKey));
															#line 1666 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->set(serversVersionKey, excludeVersionKey);
															#line 8811 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1668 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ExcludeServersCommit") .detail("Servers", describe(servers)) .detail("ExcludeFailed", failed) .detail("ExclusionUpdated", containNewExclusion);
															#line 1672 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActorState(); static_cast<ExcludeServersActor*>(this)->destroy(); return 0; }
															#line 8817 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActorState();
		static_cast<ExcludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AddressExclusion> && excl,int loopDepth) 
	{
															#line 1648 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<AddressExclusion> exclusions(excl.begin(), excl.end());
															#line 1649 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool containNewExclusion = false;
															#line 1650 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& s : servers ) {
															#line 1651 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (exclusions.find(s) != exclusions.end())
															#line 8835 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
				continue;
			}
															#line 1654 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			containNewExclusion = true;
															#line 1655 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 8843 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1656 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeFailedServersKey(s), StringRef());
															#line 8847 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1658 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeExcludedServersKey(s), StringRef());
															#line 8853 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1662 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (containNewExclusion)
															#line 8858 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1663 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string excludeVersionKey = deterministicRandom()->randomUniqueID().toString();
															#line 1664 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto serversVersionKey = failed ? failedServersVersionKey : excludedServersVersionKey;
															#line 1665 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(serversVersionKey));
															#line 1666 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->set(serversVersionKey, excludeVersionKey);
															#line 8868 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1668 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ExcludeServersCommit") .detail("Servers", describe(servers)) .detail("ExcludeFailed", failed) .detail("ExclusionUpdated", containNewExclusion);
															#line 1672 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActorState(); static_cast<ExcludeServersActor*>(this)->destroy(); return 0; }
															#line 8874 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActorState();
		static_cast<ExcludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<AddressExclusion> const& excl,int loopDepth) 
	{
		loopDepth = a_body1cont1(excl, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<AddressExclusion> && excl,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(excl), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExcludeServersActor*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor*>(this)->ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> servers;
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 8963 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via excludeServers()
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeServersActor final : public Actor<Void>, public ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >, public FastAllocated<ExcludeServersActor>, public ExcludeServersActorState<ExcludeServersActor> {
															#line 8968 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ExcludeServersActor>::operator new;
	using FastAllocated<ExcludeServersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4381307120490042368UL, 8270453186728130048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >;
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeServersActor(Transaction* const& tr,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 8985 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ExcludeServersActorState<ExcludeServersActor>(tr, servers, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15846472300878847232UL, 13853541035855110912UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("excludeServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExcludeServersActor, 0, std::vector<AddressExclusion> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> excludeServers( Transaction* const& tr, std::vector<AddressExclusion> const& servers, bool const& failed ) {
															#line 1642 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ExcludeServersActor(tr, servers, failed));
															#line 9018 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1674 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 9023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via excludeServers()
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ExcludeServersActor1>
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeServersActor1State {
															#line 9030 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeServersActor1State(Database const& cx,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   servers(servers),
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed)
															#line 9041 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("excludeServers", reinterpret_cast<unsigned long>(this));

	}
	~ExcludeServersActor1State() 
	{
		fdb_probe_actor_destroy("excludeServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1676 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (cx->apiVersionAtLeast(700))
															#line 9056 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1677 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw = ReadYourWritesTransaction(cx);
															#line 1678 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 9062 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 1702 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr = Transaction(cx);
															#line 1703 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 9071 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1680 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1681 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1682 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.set( SpecialKeySpace::getManagementApiCommandOptionSpecialKey(failed ? "failed" : "excluded", "force"), ValueRef());
															#line 1685 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( auto& s : servers ) {
															#line 1686 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				Key addr = failed ? SpecialKeySpace::getManagementApiCommandPrefix("failed").withSuffix(s.toString()) : SpecialKeySpace::getManagementApiCommandPrefix("exclude").withSuffix(s.toString());
															#line 1689 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw.set(addr, ValueRef());
															#line 9113 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1691 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeServersSpecialKeySpaceCommit") .detail("Servers", describe(servers)) .detail("ExcludeFailed", failed);
															#line 1694 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = ryw.commit();
															#line 1694 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9121 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 1;
															#line 1694 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 0, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9126 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1697 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeServersError").errorUnsuppressed(e);
															#line 1698 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ryw.onError(e);
															#line 1698 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9152 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 2;
															#line 1698 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 1, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9157 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1695 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActor1State(); static_cast<ExcludeServersActor1*>(this)->destroy(); return 0; }
															#line 9172 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1695 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActor1State(); static_cast<ExcludeServersActor1*>(this)->destroy(); return 0; }
															#line 9184 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
		try {
															#line 1705 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = excludeServers(&tr, servers, failed);
															#line 1705 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 9368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody2when1(__when_expr_2.get(), loopDepth); };
			static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 3;
															#line 1705 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 2, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9373 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1709 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeServersError").errorUnsuppressed(e);
															#line 1710 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1710 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9399 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 5;
															#line 1710 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 4, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9404 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void const& _,int loopDepth) 
	{
															#line 1706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 9421 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody2cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 4;
															#line 1706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 3, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9426 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void && _,int loopDepth) 
	{
															#line 1706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 9437 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody2cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 4;
															#line 1706 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ExcludeServersActor1, 3, Void >*>(static_cast<ExcludeServersActor1*>(this)));
															#line 9442 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody2cont3(Void const& _,int loopDepth) 
	{
															#line 1707 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActor1State(); static_cast<ExcludeServersActor1*>(this)->destroy(); return 0; }
															#line 9526 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont3(Void && _,int loopDepth) 
	{
															#line 1707 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeServersActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeServersActor1State(); static_cast<ExcludeServersActor1*>(this)->destroy(); return 0; }
															#line 9538 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeServersActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeServersActor1State();
		static_cast<ExcludeServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ExcludeServersActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeServersActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeServersActor1*>(this)->ActorCallback< ExcludeServersActor1, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ExcludeServersActor1, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ExcludeServersActor1, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> servers;
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 1677 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction ryw;
															#line 1702 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 9718 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via excludeServers()
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeServersActor1 final : public Actor<Void>, public ActorCallback< ExcludeServersActor1, 0, Void >, public ActorCallback< ExcludeServersActor1, 1, Void >, public ActorCallback< ExcludeServersActor1, 2, Void >, public ActorCallback< ExcludeServersActor1, 3, Void >, public ActorCallback< ExcludeServersActor1, 4, Void >, public FastAllocated<ExcludeServersActor1>, public ExcludeServersActor1State<ExcludeServersActor1> {
															#line 9723 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ExcludeServersActor1>::operator new;
	using FastAllocated<ExcludeServersActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4381307120490042368UL, 8270453186728130048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExcludeServersActor1, 0, Void >;
friend struct ActorCallback< ExcludeServersActor1, 1, Void >;
friend struct ActorCallback< ExcludeServersActor1, 2, Void >;
friend struct ActorCallback< ExcludeServersActor1, 3, Void >;
friend struct ActorCallback< ExcludeServersActor1, 4, Void >;
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeServersActor1(Database const& cx,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 9744 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ExcludeServersActor1State<ExcludeServersActor1>(cx, servers, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("excludeServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1716282752885344256UL, 5229742794320878592UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("excludeServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("excludeServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExcludeServersActor1, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ExcludeServersActor1, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ExcludeServersActor1, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ExcludeServersActor1, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ExcludeServersActor1, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> excludeServers( Database const& cx, std::vector<AddressExclusion> const& servers, bool const& failed ) {
															#line 1675 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ExcludeServersActor1(cx, servers, failed));
															#line 9781 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1715 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// excludes localities by setting the keys in api version below 7.0
															#line 9787 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via excludeLocalities()
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ExcludeLocalitiesActor>
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeLocalitiesActorState {
															#line 9794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeLocalitiesActorState(Transaction* const& tr,std::unordered_set<std::string> const& localities,bool const& failed) 
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   localities(localities),
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed)
															#line 9805 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("excludeLocalities", reinterpret_cast<unsigned long>(this));

	}
	~ExcludeLocalitiesActorState() 
	{
		fdb_probe_actor_destroy("excludeLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1718 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1719 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1720 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1721 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1722 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<std::string>> __when_expr_0 = failed ? getExcludedFailedLocalityList(tr) : getExcludedLocalityList(tr);
															#line 1722 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9830 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor*>(this)->actor_wait_state = 1;
															#line 1722 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*>(static_cast<ExcludeLocalitiesActor*>(this)));
															#line 9835 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExcludeLocalitiesActorState();
		static_cast<ExcludeLocalitiesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<std::string> const& excl,int loopDepth) 
	{
															#line 1723 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<std::string> exclusion(excl.begin(), excl.end());
															#line 1724 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool containNewExclusion = false;
															#line 1725 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& l : localities ) {
															#line 1726 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (exclusion.find(l) != exclusion.end())
															#line 9864 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
				continue;
			}
															#line 1729 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			containNewExclusion = true;
															#line 1730 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 9872 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1731 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeFailedLocalityKey(l), StringRef());
															#line 9876 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1733 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeExcludedLocalityKey(l), StringRef());
															#line 9882 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1736 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (containNewExclusion)
															#line 9887 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1737 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string excludeVersionKey = deterministicRandom()->randomUniqueID().toString();
															#line 1738 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto localityVersionKey = failed ? failedLocalityVersionKey : excludedLocalityVersionKey;
															#line 1739 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(localityVersionKey));
															#line 1740 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->set(localityVersionKey, excludeVersionKey);
															#line 9897 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1742 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ExcludeLocalitiesCommit") .detail("Localities", describe(localities)) .detail("ExcludeFailed", failed) .detail("ExclusionUpdated", containNewExclusion);
															#line 1746 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActorState(); static_cast<ExcludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 9903 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActorState();
		static_cast<ExcludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<std::string> && excl,int loopDepth) 
	{
															#line 1723 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::set<std::string> exclusion(excl.begin(), excl.end());
															#line 1724 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool containNewExclusion = false;
															#line 1725 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& l : localities ) {
															#line 1726 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (exclusion.find(l) != exclusion.end())
															#line 9921 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
				continue;
			}
															#line 1729 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			containNewExclusion = true;
															#line 1730 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 9929 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1731 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeFailedLocalityKey(l), StringRef());
															#line 9933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1733 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr->set(encodeExcludedLocalityKey(l), StringRef());
															#line 9939 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 1736 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (containNewExclusion)
															#line 9944 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 1737 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::string excludeVersionKey = deterministicRandom()->randomUniqueID().toString();
															#line 1738 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto localityVersionKey = failed ? failedLocalityVersionKey : excludedLocalityVersionKey;
															#line 1739 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(localityVersionKey));
															#line 1740 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->set(localityVersionKey, excludeVersionKey);
															#line 9954 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 1742 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("ExcludeLocalitiesCommit") .detail("Localities", describe(localities)) .detail("ExcludeFailed", failed) .detail("ExclusionUpdated", containNewExclusion);
															#line 1746 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActorState(); static_cast<ExcludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 9960 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActorState();
		static_cast<ExcludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<std::string> const& excl,int loopDepth) 
	{
		loopDepth = a_body1cont1(excl, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<std::string> && excl,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(excl), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExcludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor*>(this)->ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*,std::vector<std::string> const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*,std::vector<std::string> && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::unordered_set<std::string> localities;
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 10049 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via excludeLocalities()
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeLocalitiesActor final : public Actor<Void>, public ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >, public FastAllocated<ExcludeLocalitiesActor>, public ExcludeLocalitiesActorState<ExcludeLocalitiesActor> {
															#line 10054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ExcludeLocalitiesActor>::operator new;
	using FastAllocated<ExcludeLocalitiesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12718188777737456128UL, 4179321876736922624UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >;
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeLocalitiesActor(Transaction* const& tr,std::unordered_set<std::string> const& localities,bool const& failed) 
															#line 10071 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ExcludeLocalitiesActorState<ExcludeLocalitiesActor>(tr, localities, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8343920748801204224UL, 15351083160916243200UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("excludeLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor, 0, std::vector<std::string> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> excludeLocalities( Transaction* const& tr, std::unordered_set<std::string> const& localities, bool const& failed ) {
															#line 1717 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ExcludeLocalitiesActor(tr, localities, failed));
															#line 10104 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1748 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// Exclude the servers matching the given set of localities from use as state servers.
// excludes localities by setting the keys.
															#line 10111 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via excludeLocalities()
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ExcludeLocalitiesActor1>
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeLocalitiesActor1State {
															#line 10118 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeLocalitiesActor1State(Database const& cx,std::unordered_set<std::string> const& localities,bool const& failed) 
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   localities(localities),
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed)
															#line 10129 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("excludeLocalities", reinterpret_cast<unsigned long>(this));

	}
	~ExcludeLocalitiesActor1State() 
	{
		fdb_probe_actor_destroy("excludeLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1752 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (cx->apiVersionAtLeast(700))
															#line 10144 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1753 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw = ReadYourWritesTransaction(cx);
															#line 1754 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 10150 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 1779 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr = Transaction(cx);
															#line 1780 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 10159 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1756 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1757 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1758 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.set(SpecialKeySpace::getManagementApiCommandOptionSpecialKey( failed ? "failed_locality" : "excluded_locality", "force"), ValueRef());
															#line 1761 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( const auto& l : localities ) {
															#line 1762 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				Key addr = failed ? SpecialKeySpace::getManagementApiCommandPrefix("failedlocality").withSuffix(l) : SpecialKeySpace::getManagementApiCommandPrefix("excludedlocality").withSuffix(l);
															#line 1765 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw.set(addr, ValueRef());
															#line 10201 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1767 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeLocalitiesSpecialKeySpaceCommit") .detail("Localities", describe(localities)) .detail("ExcludeFailed", failed);
															#line 1771 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = ryw.commit();
															#line 1771 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10209 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 1;
															#line 1771 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 0, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10214 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1774 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeLocalitiesError").errorUnsuppressed(e);
															#line 1775 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ryw.onError(e);
															#line 1775 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 2;
															#line 1775 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 1, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10245 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1772 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActor1State(); static_cast<ExcludeLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 10260 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1772 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActor1State(); static_cast<ExcludeLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 10272 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
		try {
															#line 1782 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = excludeLocalities(&tr, localities, failed);
															#line 1782 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 10456 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody2when1(__when_expr_2.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 3;
															#line 1782 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 2, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10461 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1786 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ExcludeLocalitiesError").errorUnsuppressed(e);
															#line 1787 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1787 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 5;
															#line 1787 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 4, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10492 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void const& _,int loopDepth) 
	{
															#line 1783 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1783 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 10509 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody2cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 4;
															#line 1783 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 3, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10514 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void && _,int loopDepth) 
	{
															#line 1783 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1783 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 10525 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody2cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 4;
															#line 1783 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ExcludeLocalitiesActor1, 3, Void >*>(static_cast<ExcludeLocalitiesActor1*>(this)));
															#line 10530 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody2cont3(Void const& _,int loopDepth) 
	{
															#line 1784 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActor1State(); static_cast<ExcludeLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 10614 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont3(Void && _,int loopDepth) 
	{
															#line 1784 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ExcludeLocalitiesActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ExcludeLocalitiesActor1State(); static_cast<ExcludeLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 10626 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ExcludeLocalitiesActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~ExcludeLocalitiesActor1State();
		static_cast<ExcludeLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<ExcludeLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<ExcludeLocalitiesActor1*>(this)->ActorCallback< ExcludeLocalitiesActor1, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ExcludeLocalitiesActor1, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ExcludeLocalitiesActor1, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::unordered_set<std::string> localities;
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 1753 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction ryw;
															#line 1779 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 10806 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via excludeLocalities()
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ExcludeLocalitiesActor1 final : public Actor<Void>, public ActorCallback< ExcludeLocalitiesActor1, 0, Void >, public ActorCallback< ExcludeLocalitiesActor1, 1, Void >, public ActorCallback< ExcludeLocalitiesActor1, 2, Void >, public ActorCallback< ExcludeLocalitiesActor1, 3, Void >, public ActorCallback< ExcludeLocalitiesActor1, 4, Void >, public FastAllocated<ExcludeLocalitiesActor1>, public ExcludeLocalitiesActor1State<ExcludeLocalitiesActor1> {
															#line 10811 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ExcludeLocalitiesActor1>::operator new;
	using FastAllocated<ExcludeLocalitiesActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12718188777737456128UL, 4179321876736922624UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ExcludeLocalitiesActor1, 0, Void >;
friend struct ActorCallback< ExcludeLocalitiesActor1, 1, Void >;
friend struct ActorCallback< ExcludeLocalitiesActor1, 2, Void >;
friend struct ActorCallback< ExcludeLocalitiesActor1, 3, Void >;
friend struct ActorCallback< ExcludeLocalitiesActor1, 4, Void >;
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ExcludeLocalitiesActor1(Database const& cx,std::unordered_set<std::string> const& localities,bool const& failed) 
															#line 10832 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ExcludeLocalitiesActor1State<ExcludeLocalitiesActor1>(cx, localities, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("excludeLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13644543871368453632UL, 7369712116135787008UL);
		ActorExecutionContextHelper __helper(static_cast<ExcludeLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("excludeLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("excludeLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ExcludeLocalitiesActor1, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> excludeLocalities( Database const& cx, std::unordered_set<std::string> const& localities, bool const& failed ) {
															#line 1751 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new ExcludeLocalitiesActor1(cx, localities, failed));
															#line 10869 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1792 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 10874 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via includeServers()
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class IncludeServersActor>
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IncludeServersActorState {
															#line 10881 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IncludeServersActorState(Database const& cx,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   servers(servers),
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed),
															#line 1794 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   versionKey(deterministicRandom()->randomUniqueID().toString())
															#line 10894 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("includeServers", reinterpret_cast<unsigned long>(this));

	}
	~IncludeServersActorState() 
	{
		fdb_probe_actor_destroy("includeServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1795 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (cx->apiVersionAtLeast(700))
															#line 10909 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1796 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw = ReadYourWritesTransaction(cx);
															#line 1797 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 10915 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 1835 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr = Transaction(cx);
															#line 1836 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 10924 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1799 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1800 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1801 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( auto& s : servers ) {
															#line 1802 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!s.isValid())
															#line 10962 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1803 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (failed)
															#line 10966 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1804 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						ryw.clear(SpecialKeySpace::getManagementApiCommandRange("failed"));
															#line 10970 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					}
					else
					{
															#line 1806 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						ryw.clear(SpecialKeySpace::getManagementApiCommandRange("exclude"));
															#line 10976 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					}
				}
				else
				{
															#line 1809 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					Key addr = failed ? SpecialKeySpace::getManagementApiCommandPrefix("failed").withSuffix(s.toString()) : SpecialKeySpace::getManagementApiCommandPrefix("exclude").withSuffix(s.toString());
															#line 1812 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					ryw.clear(addr);
															#line 1821 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (s.isWholeMachine())
															#line 10987 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1822 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						ryw.clear(KeyRangeRef(addr.withSuffix(":"_sr), addr.withSuffix(";"_sr)));
															#line 10991 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					}
				}
			}
															#line 1825 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeServersCommit").detail("Servers", describe(servers)).detail("Failed", failed);
															#line 1827 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = ryw.commit();
															#line 1827 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11001 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<IncludeServersActor*>(this)->actor_wait_state = 1;
															#line 1827 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IncludeServersActor, 0, Void >*>(static_cast<IncludeServersActor*>(this)));
															#line 11006 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1830 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeServersError").errorUnsuppressed(e);
															#line 1831 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ryw.onError(e);
															#line 1831 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11032 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<IncludeServersActor*>(this)->actor_wait_state = 2;
															#line 1831 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< IncludeServersActor, 1, Void >*>(static_cast<IncludeServersActor*>(this)));
															#line 11037 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1828 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeServersActorState(); static_cast<IncludeServersActor*>(this)->destroy(); return 0; }
															#line 11052 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1828 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeServersActorState(); static_cast<IncludeServersActor*>(this)->destroy(); return 0; }
															#line 11064 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IncludeServersActor*>(this)->actor_wait_state > 0) static_cast<IncludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeServersActor*>(this)->ActorCallback< IncludeServersActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IncludeServersActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<IncludeServersActor*>(this)->actor_wait_state > 0) static_cast<IncludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeServersActor*>(this)->ActorCallback< IncludeServersActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< IncludeServersActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
		try {
															#line 1838 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1839 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1840 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1841 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1845 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::CAUSAL_WRITE_RISKY);
															#line 1846 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 11256 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1847 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.addReadConflictRange(singleKeyRange(failedServersVersionKey));
															#line 1848 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.set(failedServersVersionKey, versionKey);
															#line 11262 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1850 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.addReadConflictRange(singleKeyRange(excludedServersVersionKey));
															#line 1851 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.set(excludedServersVersionKey, versionKey);
															#line 11270 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1854 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			for( auto& s : servers ) {
															#line 1855 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!s.isValid())
															#line 11276 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1856 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (failed)
															#line 11280 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1857 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(failedServersKeys);
															#line 11284 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					}
					else
					{
															#line 1859 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(excludedServersKeys);
															#line 11290 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					}
				}
				else
				{
															#line 1861 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (s.isWholeMachine())
															#line 11297 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1870 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						auto addr = failed ? encodeFailedServersKey(s) : encodeExcludedServersKey(s);
															#line 1871 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(singleKeyRange(addr));
															#line 1872 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(KeyRangeRef(addr + ':', addr + char(':' + 1)));
															#line 11305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					}
					else
					{
															#line 1874 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						if (failed)
															#line 11311 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
						{
															#line 1875 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
							tr.clear(encodeFailedServersKey(s));
															#line 11315 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
						}
						else
						{
															#line 1877 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
							tr.clear(encodeExcludedServersKey(s));
															#line 11321 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
						}
					}
				}
			}
															#line 1882 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeServersCommit").detail("Servers", describe(servers)).detail("Failed", failed);
															#line 1884 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1884 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 11332 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody2when1(__when_expr_2.get(), loopDepth); };
			static_cast<IncludeServersActor*>(this)->actor_wait_state = 3;
															#line 1884 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< IncludeServersActor, 2, Void >*>(static_cast<IncludeServersActor*>(this)));
															#line 11337 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1887 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeServersError").errorUnsuppressed(e);
															#line 1888 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1888 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11363 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<IncludeServersActor*>(this)->actor_wait_state = 4;
															#line 1888 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< IncludeServersActor, 3, Void >*>(static_cast<IncludeServersActor*>(this)));
															#line 11368 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void const& _,int loopDepth) 
	{
															#line 1885 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeServersActorState(); static_cast<IncludeServersActor*>(this)->destroy(); return 0; }
															#line 11383 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void && _,int loopDepth) 
	{
															#line 1885 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeServersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeServersActorState(); static_cast<IncludeServersActor*>(this)->destroy(); return 0; }
															#line 11395 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeServersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeServersActorState();
		static_cast<IncludeServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<IncludeServersActor*>(this)->actor_wait_state > 0) static_cast<IncludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeServersActor*>(this)->ActorCallback< IncludeServersActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< IncludeServersActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<IncludeServersActor*>(this)->actor_wait_state > 0) static_cast<IncludeServersActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeServersActor*>(this)->ActorCallback< IncludeServersActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< IncludeServersActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< IncludeServersActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> servers;
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 1794 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::string versionKey;
															#line 1796 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction ryw;
															#line 1835 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 11577 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via includeServers()
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IncludeServersActor final : public Actor<Void>, public ActorCallback< IncludeServersActor, 0, Void >, public ActorCallback< IncludeServersActor, 1, Void >, public ActorCallback< IncludeServersActor, 2, Void >, public ActorCallback< IncludeServersActor, 3, Void >, public FastAllocated<IncludeServersActor>, public IncludeServersActorState<IncludeServersActor> {
															#line 11582 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<IncludeServersActor>::operator new;
	using FastAllocated<IncludeServersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11047490305281981440UL, 16874277495497292800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IncludeServersActor, 0, Void >;
friend struct ActorCallback< IncludeServersActor, 1, Void >;
friend struct ActorCallback< IncludeServersActor, 2, Void >;
friend struct ActorCallback< IncludeServersActor, 3, Void >;
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IncludeServersActor(Database const& cx,std::vector<AddressExclusion> const& servers,bool const& failed) 
															#line 11602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   IncludeServersActorState<IncludeServersActor>(cx, servers, failed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("includeServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7035060310568136960UL, 2930223583431837696UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("includeServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("includeServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IncludeServersActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< IncludeServersActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< IncludeServersActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< IncludeServersActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> includeServers( Database const& cx, std::vector<AddressExclusion> const& servers, bool const& failed ) {
															#line 1793 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new IncludeServersActor(cx, servers, failed));
															#line 11638 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1893 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// Remove the given localities from the exclusion list.
// include localities by clearing the keys.
															#line 11645 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via includeLocalities()
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class IncludeLocalitiesActor>
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IncludeLocalitiesActorState {
															#line 11652 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IncludeLocalitiesActorState(Database const& cx,std::vector<std::string> const& localities,bool const& failed,bool const& includeAll) 
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   localities(localities),
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   failed(failed),
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   includeAll(includeAll),
															#line 1897 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   versionKey(deterministicRandom()->randomUniqueID().toString())
															#line 11667 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("includeLocalities", reinterpret_cast<unsigned long>(this));

	}
	~IncludeLocalitiesActorState() 
	{
		fdb_probe_actor_destroy("includeLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1898 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (cx->apiVersionAtLeast(700))
															#line 11682 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1899 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ryw = ReadYourWritesTransaction(cx);
															#line 1900 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 11688 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 1931 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr = Transaction(cx);
															#line 1932 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				;
															#line 11697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1902 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1903 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ryw.setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES);
															#line 1904 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (includeAll)
															#line 11733 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1905 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (failed)
															#line 11737 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1906 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					ryw.clear(SpecialKeySpace::getManagementApiCommandRange("failedlocality"));
															#line 11741 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 1908 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					ryw.clear(SpecialKeySpace::getManagementApiCommandRange("excludedlocality"));
															#line 11747 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
			}
			else
			{
															#line 1911 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				for( const auto& l : localities ) {
															#line 1912 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					Key locality = failed ? SpecialKeySpace::getManagementApiCommandPrefix("failedlocality").withSuffix(l) : SpecialKeySpace::getManagementApiCommandPrefix("excludedlocality").withSuffix(l);
															#line 1915 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					ryw.clear(locality);
															#line 11758 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
			}
															#line 1918 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeLocalitiesCommit") .detail("Localities", describe(localities)) .detail("Failed", failed) .detail("IncludeAll", includeAll);
															#line 1923 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = ryw.commit();
															#line 1923 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11767 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 1;
															#line 1923 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IncludeLocalitiesActor, 0, Void >*>(static_cast<IncludeLocalitiesActor*>(this)));
															#line 11772 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1926 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeLocalitiesError").errorUnsuppressed(e);
															#line 1927 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = ryw.onError(e);
															#line 1927 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11798 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 2;
															#line 1927 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< IncludeLocalitiesActor, 1, Void >*>(static_cast<IncludeLocalitiesActor*>(this)));
															#line 11803 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1924 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeLocalitiesActorState(); static_cast<IncludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 11818 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1924 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeLocalitiesActorState(); static_cast<IncludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 11830 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeLocalitiesActor*>(this)->ActorCallback< IncludeLocalitiesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IncludeLocalitiesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeLocalitiesActor*>(this)->ActorCallback< IncludeLocalitiesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< IncludeLocalitiesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
		try {
															#line 1934 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1935 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1936 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1937 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1941 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::CAUSAL_WRITE_RISKY);
															#line 1942 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (failed)
															#line 12022 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1943 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.addReadConflictRange(singleKeyRange(failedLocalityVersionKey));
															#line 1944 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.set(failedLocalityVersionKey, versionKey);
															#line 12028 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 1946 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.addReadConflictRange(singleKeyRange(excludedLocalityVersionKey));
															#line 1947 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				tr.set(excludedLocalityVersionKey, versionKey);
															#line 12036 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 1950 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (includeAll)
															#line 12040 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1951 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (failed)
															#line 12044 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1952 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.clear(failedLocalityKeys);
															#line 12048 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 1954 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.clear(excludedLocalityKeys);
															#line 12054 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
			}
			else
			{
															#line 1957 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				for( const auto& l : localities ) {
															#line 1958 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					if (failed)
															#line 12063 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					{
															#line 1959 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(encodeFailedLocalityKey(l));
															#line 12067 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					}
					else
					{
															#line 1961 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
						tr.clear(encodeExcludedLocalityKey(l));
															#line 12073 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
					}
				}
			}
															#line 1966 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeLocalitiesCommit") .detail("Localities", describe(localities)) .detail("Failed", failed) .detail("IncludeAll", includeAll);
															#line 1971 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1971 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1loopBody2Catch1(actor_cancelled(), loopDepth);
															#line 12083 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody2when1(__when_expr_2.get(), loopDepth); };
			static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 3;
															#line 1971 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< IncludeLocalitiesActor, 2, Void >*>(static_cast<IncludeLocalitiesActor*>(this)));
															#line 12088 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1974 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("IncludeLocalitiesError").errorUnsuppressed(e);
															#line 1975 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1975 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12114 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 4;
															#line 1975 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< IncludeLocalitiesActor, 3, Void >*>(static_cast<IncludeLocalitiesActor*>(this)));
															#line 12119 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void const& _,int loopDepth) 
	{
															#line 1972 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeLocalitiesActorState(); static_cast<IncludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 12134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2cont2(Void && _,int loopDepth) 
	{
															#line 1972 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<IncludeLocalitiesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~IncludeLocalitiesActorState(); static_cast<IncludeLocalitiesActor*>(this)->destroy(); return 0; }
															#line 12146 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<IncludeLocalitiesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~IncludeLocalitiesActorState();
		static_cast<IncludeLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeLocalitiesActor*>(this)->ActorCallback< IncludeLocalitiesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< IncludeLocalitiesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<IncludeLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<IncludeLocalitiesActor*>(this)->ActorCallback< IncludeLocalitiesActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< IncludeLocalitiesActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< IncludeLocalitiesActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<std::string> localities;
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool failed;
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool includeAll;
															#line 1897 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::string versionKey;
															#line 1899 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction ryw;
															#line 1931 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 12330 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via includeLocalities()
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class IncludeLocalitiesActor final : public Actor<Void>, public ActorCallback< IncludeLocalitiesActor, 0, Void >, public ActorCallback< IncludeLocalitiesActor, 1, Void >, public ActorCallback< IncludeLocalitiesActor, 2, Void >, public ActorCallback< IncludeLocalitiesActor, 3, Void >, public FastAllocated<IncludeLocalitiesActor>, public IncludeLocalitiesActorState<IncludeLocalitiesActor> {
															#line 12335 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<IncludeLocalitiesActor>::operator new;
	using FastAllocated<IncludeLocalitiesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12383088541254423296UL, 6929031979279031296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IncludeLocalitiesActor, 0, Void >;
friend struct ActorCallback< IncludeLocalitiesActor, 1, Void >;
friend struct ActorCallback< IncludeLocalitiesActor, 2, Void >;
friend struct ActorCallback< IncludeLocalitiesActor, 3, Void >;
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	IncludeLocalitiesActor(Database const& cx,std::vector<std::string> const& localities,bool const& failed,bool const& includeAll) 
															#line 12355 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   IncludeLocalitiesActorState<IncludeLocalitiesActor>(cx, localities, failed, includeAll),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("includeLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3203046928570983680UL, 8290273524939964416UL);
		ActorExecutionContextHelper __helper(static_cast<IncludeLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("includeLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("includeLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IncludeLocalitiesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< IncludeLocalitiesActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< IncludeLocalitiesActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< IncludeLocalitiesActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> includeLocalities( Database const& cx, std::vector<std::string> const& localities, bool const& failed, bool const& includeAll ) {
															#line 1896 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new IncludeLocalitiesActor(cx, localities, failed, includeAll));
															#line 12391 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 1980 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 12396 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via setClass()
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class SetClassActor>
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetClassActorState {
															#line 12403 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetClassActorState(Database const& cx,AddressExclusion const& server,ProcessClass const& processClass) 
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   server(server),
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   processClass(processClass),
															#line 1982 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 12416 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("setClass", reinterpret_cast<unsigned long>(this));

	}
	~SetClassActorState() 
	{
		fdb_probe_actor_destroy("setClass", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1984 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 12431 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SetClassActorState();
		static_cast<SetClassActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1986 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1987 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1988 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1989 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::USE_PROVISIONAL_PROXIES);
															#line 1991 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<ProcessData>> __when_expr_0 = getWorkers(&tr);
															#line 1991 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetClassActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12472 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SetClassActor*>(this)->actor_wait_state = 1;
															#line 1991 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*>(static_cast<SetClassActor*>(this)));
															#line 12477 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2011 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2011 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetClassActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12501 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<SetClassActor*>(this)->actor_wait_state = 3;
															#line 2011 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SetClassActor, 2, Void >*>(static_cast<SetClassActor*>(this)));
															#line 12506 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<ProcessData> const& workers,int loopDepth) 
	{
															#line 1993 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool foundChange = false;
															#line 1994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < workers.size();i++) {
															#line 1995 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (server.excludes(workers[i].address))
															#line 12525 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1996 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (processClass.classType() != ProcessClass::InvalidClass)
															#line 12529 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1997 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.set(processClassKeyFor(workers[i].locality.processId().get()), processClassValue(processClass));
															#line 12533 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 2000 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.clear(processClassKeyFor(workers[i].locality.processId().get()));
															#line 12539 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
															#line 2001 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				foundChange = true;
															#line 12543 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2005 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (foundChange)
															#line 12548 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2006 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.set(processClassChangeKey, deterministicRandom()->randomUniqueID().toString());
															#line 12552 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2008 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2008 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetClassActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12558 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<SetClassActor*>(this)->actor_wait_state = 2;
															#line 2008 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetClassActor, 1, Void >*>(static_cast<SetClassActor*>(this)));
															#line 12563 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<ProcessData> && workers,int loopDepth) 
	{
															#line 1993 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		bool foundChange = false;
															#line 1994 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < workers.size();i++) {
															#line 1995 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (server.excludes(workers[i].address))
															#line 12576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 1996 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (processClass.classType() != ProcessClass::InvalidClass)
															#line 12580 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				{
															#line 1997 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.set(processClassKeyFor(workers[i].locality.processId().get()), processClassValue(processClass));
															#line 12584 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
				else
				{
															#line 2000 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
					tr.clear(processClassKeyFor(workers[i].locality.processId().get()));
															#line 12590 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				}
															#line 2001 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				foundChange = true;
															#line 12594 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2005 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (foundChange)
															#line 12599 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2006 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.set(processClassChangeKey, deterministicRandom()->randomUniqueID().toString());
															#line 12603 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2008 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2008 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetClassActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12609 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<SetClassActor*>(this)->actor_wait_state = 2;
															#line 2008 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetClassActor, 1, Void >*>(static_cast<SetClassActor*>(this)));
															#line 12614 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<ProcessData> const& workers,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(workers, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<ProcessData> && workers,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(workers), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SetClassActor*>(this)->actor_wait_state > 0) static_cast<SetClassActor*>(this)->actor_wait_state = 0;
		static_cast<SetClassActor*>(this)->ActorCallback< SetClassActor, 0, std::vector<ProcessData> >::remove();

	}
	void a_callback_fire(ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*,std::vector<ProcessData> const& value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*,std::vector<ProcessData> && value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*,Error err) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 2009 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetClassActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SetClassActorState(); static_cast<SetClassActor*>(this)->destroy(); return 0; }
															#line 12698 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetClassActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SetClassActorState();
		static_cast<SetClassActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 2009 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetClassActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SetClassActorState(); static_cast<SetClassActor*>(this)->destroy(); return 0; }
															#line 12710 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetClassActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SetClassActorState();
		static_cast<SetClassActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SetClassActor*>(this)->actor_wait_state > 0) static_cast<SetClassActor*>(this)->actor_wait_state = 0;
		static_cast<SetClassActor*>(this)->ActorCallback< SetClassActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetClassActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SetClassActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SetClassActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SetClassActor*>(this)->actor_wait_state > 0) static_cast<SetClassActor*>(this)->actor_wait_state = 0;
		static_cast<SetClassActor*>(this)->ActorCallback< SetClassActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetClassActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SetClassActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SetClassActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	AddressExclusion server;
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ProcessClass processClass;
															#line 1982 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 12888 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via setClass()
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetClassActor final : public Actor<Void>, public ActorCallback< SetClassActor, 0, std::vector<ProcessData> >, public ActorCallback< SetClassActor, 1, Void >, public ActorCallback< SetClassActor, 2, Void >, public FastAllocated<SetClassActor>, public SetClassActorState<SetClassActor> {
															#line 12893 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<SetClassActor>::operator new;
	using FastAllocated<SetClassActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(825851376152366080UL, 492413447179878912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SetClassActor, 0, std::vector<ProcessData> >;
friend struct ActorCallback< SetClassActor, 1, Void >;
friend struct ActorCallback< SetClassActor, 2, Void >;
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetClassActor(Database const& cx,AddressExclusion const& server,ProcessClass const& processClass) 
															#line 12912 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   SetClassActorState<SetClassActor>(cx, server, processClass),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("setClass", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7738751807625442048UL, 7840563547066985984UL);
		ActorExecutionContextHelper __helper(static_cast<SetClassActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("setClass");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("setClass", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SetClassActor, 0, std::vector<ProcessData> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SetClassActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SetClassActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> setClass( Database const& cx, AddressExclusion const& server, ProcessClass const& processClass ) {
															#line 1981 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new SetClassActor(cx, server, processClass));
															#line 12947 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2015 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 12952 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getExcludedServerList()
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetExcludedServerListActor>
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedServerListActorState {
															#line 12959 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedServerListActorState(Transaction* const& tr) 
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 12966 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getExcludedServerList", reinterpret_cast<unsigned long>(this));

	}
	~GetExcludedServerListActorState() 
	{
		fdb_probe_actor_destroy("getExcludedServerList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2017 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(excludedServersKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2017 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetExcludedServerListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12983 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetExcludedServerListActor*>(this)->actor_wait_state = 1;
															#line 2017 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetExcludedServerListActor, 0, RangeResult >*>(static_cast<GetExcludedServerListActor*>(this)));
															#line 12988 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetExcludedServerListActorState();
		static_cast<GetExcludedServerListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2018 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!r.more && r.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2020 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<AddressExclusion> exclusions;
															#line 2021 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(auto i = r.begin();i != r.end();++i) {
															#line 2022 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto a = decodeExcludedServersKey(i->key);
															#line 2023 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (a.isValid())
															#line 13019 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2024 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				exclusions.push_back(a);
															#line 13023 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2026 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2027 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetExcludedServerListActor*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetExcludedServerListActorState(); static_cast<GetExcludedServerListActor*>(this)->destroy(); return 0; }
															#line 13030 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetExcludedServerListActor*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(exclusions);
		this->~GetExcludedServerListActorState();
		static_cast<GetExcludedServerListActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __r,int loopDepth) 
	{
															#line 2017 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		r = __r;
															#line 13042 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetExcludedServerListActor*>(this)->actor_wait_state > 0) static_cast<GetExcludedServerListActor*>(this)->actor_wait_state = 0;
		static_cast<GetExcludedServerListActor*>(this)->ActorCallback< GetExcludedServerListActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetExcludedServerListActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetExcludedServerListActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetExcludedServerListActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2017 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	RangeResult r;
															#line 13121 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getExcludedServerList()
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedServerListActor final : public Actor<std::vector<AddressExclusion>>, public ActorCallback< GetExcludedServerListActor, 0, RangeResult >, public FastAllocated<GetExcludedServerListActor>, public GetExcludedServerListActorState<GetExcludedServerListActor> {
															#line 13126 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetExcludedServerListActor>::operator new;
	using FastAllocated<GetExcludedServerListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4340903684745712384UL, 11351119170224390400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<AddressExclusion>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetExcludedServerListActor, 0, RangeResult >;
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedServerListActor(Transaction* const& tr) 
															#line 13143 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<AddressExclusion>>(),
		   GetExcludedServerListActorState<GetExcludedServerListActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getExcludedServerList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7528942178455984128UL, 12820870738981153792UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getExcludedServerList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getExcludedServerList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetExcludedServerListActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<AddressExclusion>> getExcludedServerList( Transaction* const& tr ) {
															#line 2016 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<AddressExclusion>>(new GetExcludedServerListActor(tr));
															#line 13176 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2029 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 13181 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getExcludedFailedServerList()
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetExcludedFailedServerListActor>
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedFailedServerListActorState {
															#line 13188 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedFailedServerListActorState(Transaction* const& tr) 
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 13195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this));

	}
	~GetExcludedFailedServerListActorState() 
	{
		fdb_probe_actor_destroy("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2031 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(failedServersKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2031 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetExcludedFailedServerListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetExcludedFailedServerListActor*>(this)->actor_wait_state = 1;
															#line 2031 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*>(static_cast<GetExcludedFailedServerListActor*>(this)));
															#line 13217 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetExcludedFailedServerListActorState();
		static_cast<GetExcludedFailedServerListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2032 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!r.more && r.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2034 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<AddressExclusion> exclusions;
															#line 2035 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(auto i = r.begin();i != r.end();++i) {
															#line 2036 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto a = decodeFailedServersKey(i->key);
															#line 2037 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (a.isValid())
															#line 13248 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2038 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				exclusions.push_back(a);
															#line 13252 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2040 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2041 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetExcludedFailedServerListActor*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetExcludedFailedServerListActorState(); static_cast<GetExcludedFailedServerListActor*>(this)->destroy(); return 0; }
															#line 13259 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetExcludedFailedServerListActor*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(exclusions);
		this->~GetExcludedFailedServerListActorState();
		static_cast<GetExcludedFailedServerListActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __r,int loopDepth) 
	{
															#line 2031 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		r = __r;
															#line 13271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetExcludedFailedServerListActor*>(this)->actor_wait_state > 0) static_cast<GetExcludedFailedServerListActor*>(this)->actor_wait_state = 0;
		static_cast<GetExcludedFailedServerListActor*>(this)->ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2031 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	RangeResult r;
															#line 13350 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getExcludedFailedServerList()
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedFailedServerListActor final : public Actor<std::vector<AddressExclusion>>, public ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >, public FastAllocated<GetExcludedFailedServerListActor>, public GetExcludedFailedServerListActorState<GetExcludedFailedServerListActor> {
															#line 13355 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetExcludedFailedServerListActor>::operator new;
	using FastAllocated<GetExcludedFailedServerListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12233782463996453120UL, 4691984257380872960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<AddressExclusion>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >;
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedFailedServerListActor(Transaction* const& tr) 
															#line 13372 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<AddressExclusion>>(),
		   GetExcludedFailedServerListActorState<GetExcludedFailedServerListActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7137961867629435392UL, 15389248370416580864UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedServerListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getExcludedFailedServerList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getExcludedFailedServerList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetExcludedFailedServerListActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<AddressExclusion>> getExcludedFailedServerList( Transaction* const& tr ) {
															#line 2030 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<AddressExclusion>>(new GetExcludedFailedServerListActor(tr));
															#line 13405 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2043 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 13410 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getAllExcludedServers()
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetAllExcludedServersActor>
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedServersActorState {
															#line 13417 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedServersActorState(Transaction* const& tr) 
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 2045 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   exclusions(),
															#line 2047 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fExcludedServers(getExcludedServerList(tr)),
															#line 2048 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fExcludedFailed(getExcludedFailedServerList(tr)),
															#line 2049 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fExcludedLocalities(getAllExcludedLocalities(tr)),
															#line 2050 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fWorkers(getWorkers(tr))
															#line 13434 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getAllExcludedServers", reinterpret_cast<unsigned long>(this));

	}
	~GetAllExcludedServersActorState() 
	{
		fdb_probe_actor_destroy("getAllExcludedServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2054 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(fExcludedServers) && success(fExcludedFailed) && success(fExcludedLocalities);
															#line 2054 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13451 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 1;
															#line 2054 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor, 0, Void >*>(static_cast<GetAllExcludedServersActor*>(this)));
															#line 13456 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetAllExcludedServersActorState();
		static_cast<GetAllExcludedServersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2056 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedServers = fExcludedServers.get();
															#line 2057 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedServers.begin(), excludedServers.end());
															#line 2058 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedFailed = fExcludedFailed.get();
															#line 2059 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedFailed.begin(), excludedFailed.end());
															#line 2063 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		excludedLocalities = fExcludedLocalities.get();
															#line 2066 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!excludedLocalities.empty())
															#line 13489 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2068 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(fWorkers);
															#line 2068 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13495 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 2;
															#line 2068 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor, 1, Void >*>(static_cast<GetAllExcludedServersActor*>(this)));
															#line 13500 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2056 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedServers = fExcludedServers.get();
															#line 2057 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedServers.begin(), excludedServers.end());
															#line 2058 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedFailed = fExcludedFailed.get();
															#line 2059 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedFailed.begin(), excludedFailed.end());
															#line 2063 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		excludedLocalities = fExcludedLocalities.get();
															#line 2066 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!excludedLocalities.empty())
															#line 13524 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2068 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(fWorkers);
															#line 2068 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13530 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 2;
															#line 2068 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor, 1, Void >*>(static_cast<GetAllExcludedServersActor*>(this)));
															#line 13535 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedServersActor*>(this)->ActorCallback< GetAllExcludedServersActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetAllExcludedServersActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2080 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2081 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedServersActor*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetAllExcludedServersActorState(); static_cast<GetAllExcludedServersActor*>(this)->destroy(); return 0; }
															#line 13626 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedServersActor*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(std::move(exclusions)); // state_var_RVO
		this->~GetAllExcludedServersActorState();
		static_cast<GetAllExcludedServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		workers = fWorkers.get();
															#line 2071 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& locality : excludedLocalities ) {
															#line 2072 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::set<AddressExclusion> localityAddresses = getAddressesByLocality(workers, locality);
															#line 2073 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!localityAddresses.empty())
															#line 13644 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2075 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				exclusions.insert(exclusions.end(), localityAddresses.begin(), localityAddresses.end());
															#line 13648 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		workers = fWorkers.get();
															#line 2071 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& locality : excludedLocalities ) {
															#line 2072 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			std::set<AddressExclusion> localityAddresses = getAddressesByLocality(workers, locality);
															#line 2073 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!localityAddresses.empty())
															#line 13665 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2075 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				exclusions.insert(exclusions.end(), localityAddresses.begin(), localityAddresses.end());
															#line 13669 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedServersActor*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedServersActor*>(this)->ActorCallback< GetAllExcludedServersActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetAllExcludedServersActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2045 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> exclusions;
															#line 2047 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<AddressExclusion>> fExcludedServers;
															#line 2048 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<AddressExclusion>> fExcludedFailed;
															#line 2049 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<std::string>> fExcludedLocalities;
															#line 2050 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<ProcessData>> fWorkers;
															#line 2063 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<std::string> excludedLocalities;
															#line 2069 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<ProcessData> workers;
															#line 13767 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getAllExcludedServers()
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedServersActor final : public Actor<std::vector<AddressExclusion>>, public ActorCallback< GetAllExcludedServersActor, 0, Void >, public ActorCallback< GetAllExcludedServersActor, 1, Void >, public FastAllocated<GetAllExcludedServersActor>, public GetAllExcludedServersActorState<GetAllExcludedServersActor> {
															#line 13772 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetAllExcludedServersActor>::operator new;
	using FastAllocated<GetAllExcludedServersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14640256366557981184UL, 13127514102173330688UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<AddressExclusion>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetAllExcludedServersActor, 0, Void >;
friend struct ActorCallback< GetAllExcludedServersActor, 1, Void >;
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedServersActor(Transaction* const& tr) 
															#line 13790 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<AddressExclusion>>(),
		   GetAllExcludedServersActorState<GetAllExcludedServersActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17118544508197446144UL, 12587635006963120896UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getAllExcludedServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetAllExcludedServersActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetAllExcludedServersActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<AddressExclusion>> getAllExcludedServers( Transaction* const& tr ) {
															#line 2044 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<AddressExclusion>>(new GetAllExcludedServersActor(tr));
															#line 13824 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2083 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 13829 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getAllExcludedServers()
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetAllExcludedServersActor1>
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedServersActor1State {
															#line 13836 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedServersActor1State(Database const& cx) 
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2085 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 13845 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getAllExcludedServers", reinterpret_cast<unsigned long>(this));

	}
	~GetAllExcludedServersActor1State() 
	{
		fdb_probe_actor_destroy("getAllExcludedServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2086 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 13860 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetAllExcludedServersActor1State();
		static_cast<GetAllExcludedServersActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2088 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2089 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2090 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2091 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<AddressExclusion>> __when_expr_0 = getAllExcludedServers(&tr);
															#line 2091 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13899 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state = 1;
															#line 2091 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*>(static_cast<GetAllExcludedServersActor1*>(this)));
															#line 13904 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2094 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2094 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13928 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state = 2;
															#line 2094 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedServersActor1, 1, Void >*>(static_cast<GetAllExcludedServersActor1*>(this)));
															#line 13933 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<AddressExclusion> const& exclusions,int loopDepth) 
	{
															#line 2092 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedServersActor1*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetAllExcludedServersActor1State(); static_cast<GetAllExcludedServersActor1*>(this)->destroy(); return 0; }
															#line 13948 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedServersActor1*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(exclusions);
		this->~GetAllExcludedServersActor1State();
		static_cast<GetAllExcludedServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<AddressExclusion> && exclusions,int loopDepth) 
	{
															#line 2092 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedServersActor1*>(this)->SAV<std::vector<AddressExclusion>>::futures) { (void)(exclusions); this->~GetAllExcludedServersActor1State(); static_cast<GetAllExcludedServersActor1*>(this)->destroy(); return 0; }
															#line 13960 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedServersActor1*>(this)->SAV< std::vector<AddressExclusion> >::value()) std::vector<AddressExclusion>(exclusions);
		this->~GetAllExcludedServersActor1State();
		static_cast<GetAllExcludedServersActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<AddressExclusion> const& exclusions,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(exclusions, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<AddressExclusion> && exclusions,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(exclusions), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedServersActor1*>(this)->ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*,std::vector<AddressExclusion> && value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedServersActor1*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedServersActor1*>(this)->ActorCallback< GetAllExcludedServersActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedServersActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetAllExcludedServersActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2085 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 14134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getAllExcludedServers()
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedServersActor1 final : public Actor<std::vector<AddressExclusion>>, public ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >, public ActorCallback< GetAllExcludedServersActor1, 1, Void >, public FastAllocated<GetAllExcludedServersActor1>, public GetAllExcludedServersActor1State<GetAllExcludedServersActor1> {
															#line 14139 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetAllExcludedServersActor1>::operator new;
	using FastAllocated<GetAllExcludedServersActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14640256366557981184UL, 13127514102173330688UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<AddressExclusion>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >;
friend struct ActorCallback< GetAllExcludedServersActor1, 1, Void >;
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedServersActor1(Database const& cx) 
															#line 14157 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<AddressExclusion>>(),
		   GetAllExcludedServersActor1State<GetAllExcludedServersActor1>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getAllExcludedServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4564542305281620992UL, 15906870840917760000UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedServersActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getAllExcludedServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getAllExcludedServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetAllExcludedServersActor1, 0, std::vector<AddressExclusion> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetAllExcludedServersActor1, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<AddressExclusion>> getAllExcludedServers( Database const& cx ) {
															#line 2084 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<AddressExclusion>>(new GetAllExcludedServersActor1(cx));
															#line 14191 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2098 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 14196 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getExcludedLocalityList()
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetExcludedLocalityListActor>
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedLocalityListActorState {
															#line 14203 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedLocalityListActorState(Transaction* const& tr) 
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 14210 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getExcludedLocalityList", reinterpret_cast<unsigned long>(this));

	}
	~GetExcludedLocalityListActorState() 
	{
		fdb_probe_actor_destroy("getExcludedLocalityList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2100 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(excludedLocalityKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2100 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetExcludedLocalityListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14227 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetExcludedLocalityListActor*>(this)->actor_wait_state = 1;
															#line 2100 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*>(static_cast<GetExcludedLocalityListActor*>(this)));
															#line 14232 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetExcludedLocalityListActorState();
		static_cast<GetExcludedLocalityListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2101 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!r.more && r.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2103 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<std::string> excludedLocalities;
															#line 2104 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& i : r ) {
															#line 2105 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto a = decodeExcludedLocalityKey(i.key);
															#line 2106 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			excludedLocalities.push_back(a);
															#line 14263 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2108 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(excludedLocalities);
															#line 2109 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetExcludedLocalityListActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(excludedLocalities); this->~GetExcludedLocalityListActorState(); static_cast<GetExcludedLocalityListActor*>(this)->destroy(); return 0; }
															#line 14269 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetExcludedLocalityListActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(excludedLocalities);
		this->~GetExcludedLocalityListActorState();
		static_cast<GetExcludedLocalityListActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __r,int loopDepth) 
	{
															#line 2100 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		r = __r;
															#line 14281 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetExcludedLocalityListActor*>(this)->actor_wait_state > 0) static_cast<GetExcludedLocalityListActor*>(this)->actor_wait_state = 0;
		static_cast<GetExcludedLocalityListActor*>(this)->ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2100 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	RangeResult r;
															#line 14360 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getExcludedLocalityList()
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedLocalityListActor final : public Actor<std::vector<std::string>>, public ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >, public FastAllocated<GetExcludedLocalityListActor>, public GetExcludedLocalityListActorState<GetExcludedLocalityListActor> {
															#line 14365 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetExcludedLocalityListActor>::operator new;
	using FastAllocated<GetExcludedLocalityListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10034134042722728192UL, 8822587497000743168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >;
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedLocalityListActor(Transaction* const& tr) 
															#line 14382 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   GetExcludedLocalityListActorState<GetExcludedLocalityListActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3296438703473487872UL, 3197736188210309376UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getExcludedLocalityList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getExcludedLocalityList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetExcludedLocalityListActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> getExcludedLocalityList( Transaction* const& tr ) {
															#line 2099 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<std::string>>(new GetExcludedLocalityListActor(tr));
															#line 14415 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2111 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 14420 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getExcludedFailedLocalityList()
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetExcludedFailedLocalityListActor>
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedFailedLocalityListActorState {
															#line 14427 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedFailedLocalityListActorState(Transaction* const& tr) 
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr)
															#line 14434 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this));

	}
	~GetExcludedFailedLocalityListActorState() 
	{
		fdb_probe_actor_destroy("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2113 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(failedLocalityKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2113 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetExcludedFailedLocalityListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14451 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetExcludedFailedLocalityListActor*>(this)->actor_wait_state = 1;
															#line 2113 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*>(static_cast<GetExcludedFailedLocalityListActor*>(this)));
															#line 14456 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetExcludedFailedLocalityListActorState();
		static_cast<GetExcludedFailedLocalityListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2114 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!r.more && r.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2116 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		std::vector<std::string> excludedLocalities;
															#line 2117 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( const auto& i : r ) {
															#line 2118 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto a = decodeFailedLocalityKey(i.key);
															#line 2119 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			excludedLocalities.push_back(a);
															#line 14487 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2121 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(excludedLocalities);
															#line 2122 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetExcludedFailedLocalityListActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(excludedLocalities); this->~GetExcludedFailedLocalityListActorState(); static_cast<GetExcludedFailedLocalityListActor*>(this)->destroy(); return 0; }
															#line 14493 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetExcludedFailedLocalityListActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(excludedLocalities);
		this->~GetExcludedFailedLocalityListActorState();
		static_cast<GetExcludedFailedLocalityListActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __r,int loopDepth) 
	{
															#line 2113 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		r = __r;
															#line 14505 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetExcludedFailedLocalityListActor*>(this)->actor_wait_state > 0) static_cast<GetExcludedFailedLocalityListActor*>(this)->actor_wait_state = 0;
		static_cast<GetExcludedFailedLocalityListActor*>(this)->ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2113 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	RangeResult r;
															#line 14584 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getExcludedFailedLocalityList()
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetExcludedFailedLocalityListActor final : public Actor<std::vector<std::string>>, public ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >, public FastAllocated<GetExcludedFailedLocalityListActor>, public GetExcludedFailedLocalityListActorState<GetExcludedFailedLocalityListActor> {
															#line 14589 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetExcludedFailedLocalityListActor>::operator new;
	using FastAllocated<GetExcludedFailedLocalityListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12572448584432027136UL, 12309285900773993984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >;
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetExcludedFailedLocalityListActor(Transaction* const& tr) 
															#line 14606 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   GetExcludedFailedLocalityListActorState<GetExcludedFailedLocalityListActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11099463865640149248UL, 96070748485650944UL);
		ActorExecutionContextHelper __helper(static_cast<GetExcludedFailedLocalityListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getExcludedFailedLocalityList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getExcludedFailedLocalityList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetExcludedFailedLocalityListActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> getExcludedFailedLocalityList( Transaction* const& tr ) {
															#line 2112 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<std::string>>(new GetExcludedFailedLocalityListActor(tr));
															#line 14639 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2124 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 14644 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getAllExcludedLocalities()
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetAllExcludedLocalitiesActor>
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedLocalitiesActorState {
															#line 14651 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedLocalitiesActorState(Transaction* const& tr) 
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 2126 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   exclusions(),
															#line 2127 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fExcludedLocalities(getExcludedLocalityList(tr)),
															#line 2128 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   fFailedLocalities(getExcludedFailedLocalityList(tr))
															#line 14664 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this));

	}
	~GetAllExcludedLocalitiesActorState() 
	{
		fdb_probe_actor_destroy("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2131 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(fExcludedLocalities) && success(fFailedLocalities);
															#line 2131 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedLocalitiesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14681 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetAllExcludedLocalitiesActor*>(this)->actor_wait_state = 1;
															#line 2131 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*>(static_cast<GetAllExcludedLocalitiesActor*>(this)));
															#line 14686 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetAllExcludedLocalitiesActorState();
		static_cast<GetAllExcludedLocalitiesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2133 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedLocalities = fExcludedLocalities.get();
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedLocalities.begin(), excludedLocalities.end());
															#line 2135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto failedLocalities = fFailedLocalities.get();
															#line 2136 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), failedLocalities.begin(), failedLocalities.end());
															#line 2138 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2139 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedLocalitiesActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(exclusions); this->~GetAllExcludedLocalitiesActorState(); static_cast<GetAllExcludedLocalitiesActor*>(this)->destroy(); return 0; }
															#line 14719 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedLocalitiesActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(std::move(exclusions)); // state_var_RVO
		this->~GetAllExcludedLocalitiesActorState();
		static_cast<GetAllExcludedLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2133 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto excludedLocalities = fExcludedLocalities.get();
															#line 2134 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), excludedLocalities.begin(), excludedLocalities.end());
															#line 2135 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto failedLocalities = fFailedLocalities.get();
															#line 2136 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		exclusions.insert(exclusions.end(), failedLocalities.begin(), failedLocalities.end());
															#line 2138 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		uniquify(exclusions);
															#line 2139 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedLocalitiesActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(exclusions); this->~GetAllExcludedLocalitiesActorState(); static_cast<GetAllExcludedLocalitiesActor*>(this)->destroy(); return 0; }
															#line 14741 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedLocalitiesActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(std::move(exclusions)); // state_var_RVO
		this->~GetAllExcludedLocalitiesActorState();
		static_cast<GetAllExcludedLocalitiesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetAllExcludedLocalitiesActor*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedLocalitiesActor*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedLocalitiesActor*>(this)->ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2126 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<std::string> exclusions;
															#line 2127 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<std::string>> fExcludedLocalities;
															#line 2128 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Future<std::vector<std::string>> fFailedLocalities;
															#line 14832 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getAllExcludedLocalities()
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedLocalitiesActor final : public Actor<std::vector<std::string>>, public ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >, public FastAllocated<GetAllExcludedLocalitiesActor>, public GetAllExcludedLocalitiesActorState<GetAllExcludedLocalitiesActor> {
															#line 14837 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetAllExcludedLocalitiesActor>::operator new;
	using FastAllocated<GetAllExcludedLocalitiesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10243597620363918592UL, 15678368360877609728UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >;
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedLocalitiesActor(Transaction* const& tr) 
															#line 14854 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   GetAllExcludedLocalitiesActorState<GetAllExcludedLocalitiesActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15824813939058471168UL, 12546097955091880448UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getAllExcludedLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetAllExcludedLocalitiesActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> getAllExcludedLocalities( Transaction* const& tr ) {
															#line 2125 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<std::string>>(new GetAllExcludedLocalitiesActor(tr));
															#line 14887 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2141 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// Get the list of excluded localities by reading the keys.
															#line 14893 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getAllExcludedLocalities()
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class GetAllExcludedLocalitiesActor1>
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedLocalitiesActor1State {
															#line 14900 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedLocalitiesActor1State(Database const& cx) 
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2144 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 14909 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this));

	}
	~GetAllExcludedLocalitiesActor1State() 
	{
		fdb_probe_actor_destroy("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2145 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 14924 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetAllExcludedLocalitiesActor1State();
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2147 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2148 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2149 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2150 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<std::vector<std::string>> __when_expr_0 = getAllExcludedLocalities(&tr);
															#line 2150 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14963 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state = 1;
															#line 2150 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*>(static_cast<GetAllExcludedLocalitiesActor1*>(this)));
															#line 14968 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2153 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2153 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14992 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state = 2;
															#line 2153 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*>(static_cast<GetAllExcludedLocalitiesActor1*>(this)));
															#line 14997 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<std::string> const& exclusions,int loopDepth) 
	{
															#line 2151 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedLocalitiesActor1*>(this)->SAV<std::vector<std::string>>::futures) { (void)(exclusions); this->~GetAllExcludedLocalitiesActor1State(); static_cast<GetAllExcludedLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 15012 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedLocalitiesActor1*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(exclusions);
		this->~GetAllExcludedLocalitiesActor1State();
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(std::vector<std::string> && exclusions,int loopDepth) 
	{
															#line 2151 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<GetAllExcludedLocalitiesActor1*>(this)->SAV<std::vector<std::string>>::futures) { (void)(exclusions); this->~GetAllExcludedLocalitiesActor1State(); static_cast<GetAllExcludedLocalitiesActor1*>(this)->destroy(); return 0; }
															#line 15024 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<GetAllExcludedLocalitiesActor1*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(exclusions);
		this->~GetAllExcludedLocalitiesActor1State();
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<std::string> const& exclusions,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(exclusions, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::vector<std::string> && exclusions,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(exclusions), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*,std::vector<std::string> const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*,std::vector<std::string> && value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state > 0) static_cast<GetAllExcludedLocalitiesActor1*>(this)->actor_wait_state = 0;
		static_cast<GetAllExcludedLocalitiesActor1*>(this)->ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2144 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 15198 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via getAllExcludedLocalities()
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class GetAllExcludedLocalitiesActor1 final : public Actor<std::vector<std::string>>, public ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >, public ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >, public FastAllocated<GetAllExcludedLocalitiesActor1>, public GetAllExcludedLocalitiesActor1State<GetAllExcludedLocalitiesActor1> {
															#line 15203 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<GetAllExcludedLocalitiesActor1>::operator new;
	using FastAllocated<GetAllExcludedLocalitiesActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10243597620363918592UL, 15678368360877609728UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >;
friend struct ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >;
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	GetAllExcludedLocalitiesActor1(Database const& cx) 
															#line 15221 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   GetAllExcludedLocalitiesActor1State<GetAllExcludedLocalitiesActor1>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3977877330550522880UL, 15060849589996873216UL);
		ActorExecutionContextHelper __helper(static_cast<GetAllExcludedLocalitiesActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getAllExcludedLocalities");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getAllExcludedLocalities", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetAllExcludedLocalitiesActor1, 0, std::vector<std::string> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetAllExcludedLocalitiesActor1, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> getAllExcludedLocalities( Database const& cx ) {
															#line 2143 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::vector<std::string>>(new GetAllExcludedLocalitiesActor1(cx));
															#line 15255 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2157 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

// Decodes the locality string to a pair of locality prefix and its value.
// The prefix could be dcid, processid, machineid, processid.
std::pair<std::string, std::string> decodeLocality(const std::string& locality) {
	StringRef localityRef((const uint8_t*)(locality.c_str()), locality.size());

	std::string localityKeyValue = localityRef.removePrefix(LocalityData::ExcludeLocalityPrefix).toString();
	int split = localityKeyValue.find(':');
	if (split != std::string::npos) {
		return std::make_pair(localityKeyValue.substr(0, split), localityKeyValue.substr(split + 1));
	}

	return std::make_pair("", "");
}

// Returns the list of IPAddresses of the servers that match the given locality.
// Example: locality="dcid:primary" returns all the ip addresses of the servers in the primary dc.
std::set<AddressExclusion> getServerAddressesByLocality(
    const std::map<std::string, StorageServerInterface> server_interfaces,
    const std::string& locality) {
	std::pair<std::string, std::string> locality_key_value = decodeLocality(locality);
	std::set<AddressExclusion> locality_addresses;

	for (auto& server : server_interfaces) {
		auto locality_value = server.second.locality.get(locality_key_value.first);
		if (!locality_value.present()) {
			continue;
		}

		if (locality_value.get() != locality_key_value.second) {
			continue;
		}

		auto primary_address = server.second.address();
		locality_addresses.insert(AddressExclusion(primary_address.ip, primary_address.port));
		if (server.second.secondaryAddress().present()) {
			auto secondary_address = server.second.secondaryAddress().get();
			locality_addresses.insert(AddressExclusion(secondary_address.ip, secondary_address.port));
		}
	}

	return locality_addresses;
}

// Returns the list of IPAddresses of the workers that match the given locality.
// Example: locality="locality_dcid:primary" returns all the ip addresses of the workers in the primary dc.
std::set<AddressExclusion> getAddressesByLocality(const std::vector<ProcessData>& workers,
                                                  const std::string& locality) {
	std::pair<std::string, std::string> locality_key_value = decodeLocality(locality);
	std::set<AddressExclusion> locality_addresses;

	for (int i = 0; i < workers.size(); i++) {
		auto locality_value = workers[i].locality.get(locality_key_value.first);
		if (!locality_value.present()) {
			continue;
		}

		if (locality_value.get() != locality_key_value.second) {
			continue;
		}

		locality_addresses.insert(AddressExclusion(workers[i].address.ip, workers[i].address.port));
	}

	return locality_addresses;
}

															#line 15326 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via printHealthyZone()
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class PrintHealthyZoneActor>
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class PrintHealthyZoneActorState {
															#line 15333 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	PrintHealthyZoneActorState(Database const& cx) 
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2225 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 15342 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("printHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	~PrintHealthyZoneActorState() 
	{
		fdb_probe_actor_destroy("printHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2226 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 15357 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PrintHealthyZoneActorState();
		static_cast<PrintHealthyZoneActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2228 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 2229 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2230 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(healthyZoneKey);
															#line 2230 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15394 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state = 1;
															#line 2230 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*>(static_cast<PrintHealthyZoneActor*>(this)));
															#line 15399 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2244 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2244 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15423 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2244 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PrintHealthyZoneActor, 1, Void >*>(static_cast<PrintHealthyZoneActor*>(this)));
															#line 15428 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& val,int loopDepth) 
	{
															#line 2231 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 15443 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2232 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			printf("Data distribution has been disabled for all storage server failures in this cluster and thus " "maintenance mode is not active.\n");
															#line 15447 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
		else
		{
															#line 2234 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!val.present() || decodeHealthyZoneValue(val.get()).second <= tr.getReadVersion().get())
															#line 15453 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2235 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("No ongoing maintenance.\n");
															#line 15457 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 2237 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto healthyZone = decodeHealthyZoneValue(val.get());
															#line 2238 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				fmt::print("Maintenance for zone {0} will continue for {1} seconds.\n", healthyZone.first.toString(), (healthyZone.second - tr.getReadVersion().get()) / CLIENT_KNOBS->CORE_VERSIONSPERSECOND);
															#line 15465 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2242 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<PrintHealthyZoneActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PrintHealthyZoneActorState(); static_cast<PrintHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 15470 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<PrintHealthyZoneActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PrintHealthyZoneActorState();
		static_cast<PrintHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && val,int loopDepth) 
	{
															#line 2231 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 15482 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2232 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			printf("Data distribution has been disabled for all storage server failures in this cluster and thus " "maintenance mode is not active.\n");
															#line 15486 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
		else
		{
															#line 2234 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!val.present() || decodeHealthyZoneValue(val.get()).second <= tr.getReadVersion().get())
															#line 15492 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2235 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("No ongoing maintenance.\n");
															#line 15496 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
			else
			{
															#line 2237 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				auto healthyZone = decodeHealthyZoneValue(val.get());
															#line 2238 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				fmt::print("Maintenance for zone {0} will continue for {1} seconds.\n", healthyZone.first.toString(), (healthyZone.second - tr.getReadVersion().get()) / CLIENT_KNOBS->CORE_VERSIONSPERSECOND);
															#line 15504 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2242 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<PrintHealthyZoneActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PrintHealthyZoneActorState(); static_cast<PrintHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 15509 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<PrintHealthyZoneActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PrintHealthyZoneActorState();
		static_cast<PrintHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(val, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<PrintHealthyZoneActor*>(this)->ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<PrintHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<PrintHealthyZoneActor*>(this)->ActorCallback< PrintHealthyZoneActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PrintHealthyZoneActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PrintHealthyZoneActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PrintHealthyZoneActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2225 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 15683 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via printHealthyZone()
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class PrintHealthyZoneActor final : public Actor<Void>, public ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >, public ActorCallback< PrintHealthyZoneActor, 1, Void >, public FastAllocated<PrintHealthyZoneActor>, public PrintHealthyZoneActorState<PrintHealthyZoneActor> {
															#line 15688 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<PrintHealthyZoneActor>::operator new;
	using FastAllocated<PrintHealthyZoneActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5347877199630744576UL, 2267995477140187392UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >;
friend struct ActorCallback< PrintHealthyZoneActor, 1, Void >;
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	PrintHealthyZoneActor(Database const& cx) 
															#line 15706 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   PrintHealthyZoneActorState<PrintHealthyZoneActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("printHealthyZone", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3507524340034560UL, 4848501524512482048UL);
		ActorExecutionContextHelper __helper(static_cast<PrintHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("printHealthyZone");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("printHealthyZone", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PrintHealthyZoneActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PrintHealthyZoneActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> printHealthyZone( Database const& cx ) {
															#line 2224 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new PrintHealthyZoneActor(cx));
															#line 15740 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2248 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 15745 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via clearHealthyZone()
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class ClearHealthyZoneActor>
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ClearHealthyZoneActorState {
															#line 15752 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClearHealthyZoneActorState(Database const& cx,bool const& printWarning,bool const& clearSSFailureZoneString) 
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   printWarning(printWarning),
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   clearSSFailureZoneString(clearSSFailureZoneString),
															#line 2250 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 15765 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("clearHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	~ClearHealthyZoneActorState() 
	{
		fdb_probe_actor_destroy("clearHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2251 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("ClearHealthyZone").detail("ClearSSFailureZoneString", clearSSFailureZoneString);
															#line 2252 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 15782 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ClearHealthyZoneActorState();
		static_cast<ClearHealthyZoneActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2254 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2255 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2256 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2257 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(healthyZoneKey);
															#line 2257 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15821 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 1;
															#line 2257 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*>(static_cast<ClearHealthyZoneActor*>(this)));
															#line 15826 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2271 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2271 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15850 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 3;
															#line 2271 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ClearHealthyZoneActor, 2, Void >*>(static_cast<ClearHealthyZoneActor*>(this)));
															#line 15855 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& val,int loopDepth) 
	{
															#line 2258 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!clearSSFailureZoneString && val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 15870 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2260 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (printWarning)
															#line 15874 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2261 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("ERROR: Maintenance mode cannot be used while data distribution is disabled for storage " "server failures. Use 'datadistribution on' to reenable data distribution.\n");
															#line 15878 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2264 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ClearHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~ClearHealthyZoneActorState(); static_cast<ClearHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 15882 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ClearHealthyZoneActor*>(this)->SAV< bool >::value()) bool(false);
			this->~ClearHealthyZoneActorState();
			static_cast<ClearHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2267 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.clear(healthyZoneKey);
															#line 2268 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2268 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15894 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2268 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ClearHealthyZoneActor, 1, Void >*>(static_cast<ClearHealthyZoneActor*>(this)));
															#line 15899 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && val,int loopDepth) 
	{
															#line 2258 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!clearSSFailureZoneString && val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 15908 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2260 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (printWarning)
															#line 15912 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2261 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("ERROR: Maintenance mode cannot be used while data distribution is disabled for storage " "server failures. Use 'datadistribution on' to reenable data distribution.\n");
															#line 15916 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2264 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<ClearHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~ClearHealthyZoneActorState(); static_cast<ClearHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 15920 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<ClearHealthyZoneActor*>(this)->SAV< bool >::value()) bool(false);
			this->~ClearHealthyZoneActorState();
			static_cast<ClearHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2267 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.clear(healthyZoneKey);
															#line 2268 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2268 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15932 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2268 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ClearHealthyZoneActor, 1, Void >*>(static_cast<ClearHealthyZoneActor*>(this)));
															#line 15937 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(val, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<ClearHealthyZoneActor*>(this)->ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 2269 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ClearHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~ClearHealthyZoneActorState(); static_cast<ClearHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16021 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ClearHealthyZoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~ClearHealthyZoneActorState();
		static_cast<ClearHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 2269 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<ClearHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~ClearHealthyZoneActorState(); static_cast<ClearHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16033 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<ClearHealthyZoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~ClearHealthyZoneActorState();
		static_cast<ClearHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<ClearHealthyZoneActor*>(this)->ActorCallback< ClearHealthyZoneActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ClearHealthyZoneActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<ClearHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<ClearHealthyZoneActor*>(this)->ActorCallback< ClearHealthyZoneActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ClearHealthyZoneActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ClearHealthyZoneActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool printWarning;
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool clearSSFailureZoneString;
															#line 2250 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 16211 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via clearHealthyZone()
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class ClearHealthyZoneActor final : public Actor<bool>, public ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >, public ActorCallback< ClearHealthyZoneActor, 1, Void >, public ActorCallback< ClearHealthyZoneActor, 2, Void >, public FastAllocated<ClearHealthyZoneActor>, public ClearHealthyZoneActorState<ClearHealthyZoneActor> {
															#line 16216 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<ClearHealthyZoneActor>::operator new;
	using FastAllocated<ClearHealthyZoneActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13575226811177831936UL, 2962264050135314688UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >;
friend struct ActorCallback< ClearHealthyZoneActor, 1, Void >;
friend struct ActorCallback< ClearHealthyZoneActor, 2, Void >;
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ClearHealthyZoneActor(Database const& cx,bool const& printWarning,bool const& clearSSFailureZoneString) 
															#line 16235 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<bool>(),
		   ClearHealthyZoneActorState<ClearHealthyZoneActor>(cx, printWarning, clearSSFailureZoneString),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("clearHealthyZone", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(868363240144782336UL, 16824621560516897536UL);
		ActorExecutionContextHelper __helper(static_cast<ClearHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("clearHealthyZone");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("clearHealthyZone", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ClearHealthyZoneActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ClearHealthyZoneActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ClearHealthyZoneActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<bool> clearHealthyZone( Database const& cx, bool const& printWarning, bool const& clearSSFailureZoneString ) {
															#line 2249 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<bool>(new ClearHealthyZoneActor(cx, printWarning, clearSSFailureZoneString));
															#line 16270 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2275 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 16275 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via setHealthyZone()
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class SetHealthyZoneActor>
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetHealthyZoneActorState {
															#line 16282 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetHealthyZoneActorState(Database const& cx,StringRef const& zoneId,double const& seconds,bool const& printWarning) 
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   zoneId(zoneId),
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   seconds(seconds),
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   printWarning(printWarning),
															#line 2277 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 16297 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("setHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	~SetHealthyZoneActorState() 
	{
		fdb_probe_actor_destroy("setHealthyZone", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2278 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("SetHealthyZone").detail("Zone", zoneId).detail("DurationSeconds", seconds);
															#line 2279 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 16314 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SetHealthyZoneActorState();
		static_cast<SetHealthyZoneActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2281 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2282 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2283 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2284 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(healthyZoneKey);
															#line 2284 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16353 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 1;
															#line 2284 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16358 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2298 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 2298 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16382 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 4;
															#line 2298 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 3, Void >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16387 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& val,int loopDepth) 
	{
															#line 2285 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 16402 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2286 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (printWarning)
															#line 16406 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2287 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("ERROR: Maintenance mode cannot be used while data distribution is disabled for storage " "server failures. Use 'datadistribution on' to reenable data distribution.\n");
															#line 16410 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2290 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<SetHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~SetHealthyZoneActorState(); static_cast<SetHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16414 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<SetHealthyZoneActor*>(this)->SAV< bool >::value()) bool(false);
			this->~SetHealthyZoneActorState();
			static_cast<SetHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2292 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Version> __when_expr_1 = tr.getReadVersion();
															#line 2292 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16424 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2292 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 1, Version >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16429 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && val,int loopDepth) 
	{
															#line 2285 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present() && decodeHealthyZoneValue(val.get()).first == ignoreSSFailuresZoneString)
															#line 16438 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2286 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (printWarning)
															#line 16442 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2287 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				printf("ERROR: Maintenance mode cannot be used while data distribution is disabled for storage " "server failures. Use 'datadistribution on' to reenable data distribution.\n");
															#line 16446 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2290 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<SetHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~SetHealthyZoneActorState(); static_cast<SetHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16450 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<SetHealthyZoneActor*>(this)->SAV< bool >::value()) bool(false);
			this->~SetHealthyZoneActorState();
			static_cast<SetHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2292 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Version> __when_expr_1 = tr.getReadVersion();
															#line 2292 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16460 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 2;
															#line 2292 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 1, Version >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16465 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(val, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<SetHealthyZoneActor*>(this)->ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Version const& readVersion,int loopDepth) 
	{
															#line 2293 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(healthyZoneKey, healthyZoneValue(zoneId, readVersion + (seconds * CLIENT_KNOBS->CORE_VERSIONSPERSECOND)));
															#line 2295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16553 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 3;
															#line 2295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 2, Void >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16558 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Version && readVersion,int loopDepth) 
	{
															#line 2293 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(healthyZoneKey, healthyZoneValue(zoneId, readVersion + (seconds * CLIENT_KNOBS->CORE_VERSIONSPERSECOND)));
															#line 2295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16571 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 3;
															#line 2295 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SetHealthyZoneActor, 2, Void >*>(static_cast<SetHealthyZoneActor*>(this)));
															#line 16576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Version const& readVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(readVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Version && readVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(readVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<SetHealthyZoneActor*>(this)->ActorCallback< SetHealthyZoneActor, 1, Version >::remove();

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 1, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 1, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SetHealthyZoneActor, 1, Version >*,Error err) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2296 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~SetHealthyZoneActorState(); static_cast<SetHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16660 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetHealthyZoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~SetHealthyZoneActorState();
		static_cast<SetHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2296 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetHealthyZoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~SetHealthyZoneActorState(); static_cast<SetHealthyZoneActor*>(this)->destroy(); return 0; }
															#line 16672 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetHealthyZoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~SetHealthyZoneActorState();
		static_cast<SetHealthyZoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<SetHealthyZoneActor*>(this)->ActorCallback< SetHealthyZoneActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SetHealthyZoneActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SetHealthyZoneActor*>(this)->actor_wait_state > 0) static_cast<SetHealthyZoneActor*>(this)->actor_wait_state = 0;
		static_cast<SetHealthyZoneActor*>(this)->ActorCallback< SetHealthyZoneActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SetHealthyZoneActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SetHealthyZoneActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	StringRef zoneId;
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	double seconds;
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool printWarning;
															#line 2277 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 16852 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via setHealthyZone()
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetHealthyZoneActor final : public Actor<bool>, public ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >, public ActorCallback< SetHealthyZoneActor, 1, Version >, public ActorCallback< SetHealthyZoneActor, 2, Void >, public ActorCallback< SetHealthyZoneActor, 3, Void >, public FastAllocated<SetHealthyZoneActor>, public SetHealthyZoneActorState<SetHealthyZoneActor> {
															#line 16857 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<SetHealthyZoneActor>::operator new;
	using FastAllocated<SetHealthyZoneActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10320839426559179264UL, 8767651095270539776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >;
friend struct ActorCallback< SetHealthyZoneActor, 1, Version >;
friend struct ActorCallback< SetHealthyZoneActor, 2, Void >;
friend struct ActorCallback< SetHealthyZoneActor, 3, Void >;
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetHealthyZoneActor(Database const& cx,StringRef const& zoneId,double const& seconds,bool const& printWarning) 
															#line 16877 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<bool>(),
		   SetHealthyZoneActorState<SetHealthyZoneActor>(cx, zoneId, seconds, printWarning),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("setHealthyZone", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14961976220816392704UL, 7810201303269410048UL);
		ActorExecutionContextHelper __helper(static_cast<SetHealthyZoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("setHealthyZone");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("setHealthyZone", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SetHealthyZoneActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SetHealthyZoneActor, 1, Version >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SetHealthyZoneActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< SetHealthyZoneActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<bool> setHealthyZone( Database const& cx, StringRef const& zoneId, double const& seconds, bool const& printWarning ) {
															#line 2276 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<bool>(new SetHealthyZoneActor(cx, zoneId, seconds, printWarning));
															#line 16913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2302 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 16918 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via setDDMode()
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class SetDDModeActor>
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetDDModeActorState {
															#line 16925 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetDDModeActorState(Database const& cx,int const& mode) 
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   mode(mode),
															#line 2304 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx),
															#line 2305 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   oldMode(-1),
															#line 2306 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   wr(Unversioned())
															#line 16940 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("setDDMode", reinterpret_cast<unsigned long>(this));

	}
	~SetDDModeActorState() 
	{
		fdb_probe_actor_destroy("setDDMode", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2307 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			wr << mode;
															#line 2309 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 16957 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SetDDModeActorState();
		static_cast<SetDDModeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2311 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2312 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2313 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2314 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(dataDistributionModeKey);
															#line 2314 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16996 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SetDDModeActor*>(this)->actor_wait_state = 1;
															#line 2314 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 0, Optional<Value> >*>(static_cast<SetDDModeActor*>(this)));
															#line 17001 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2345 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("SetDDModeRetrying").error(e);
															#line 2346 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 2346 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17027 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<SetDDModeActor*>(this)->actor_wait_state = 4;
															#line 2346 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 3, Void >*>(static_cast<SetDDModeActor*>(this)));
															#line 17032 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& old,int loopDepth) 
	{
															#line 2315 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (oldMode < 0)
															#line 17047 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2316 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldMode = 1;
															#line 2317 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (old.present())
															#line 17053 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2318 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				BinaryReader rd(old.get(), Unversioned());
															#line 2319 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				rd >> oldMode;
															#line 17059 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2322 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		BinaryWriter wrMyOwner(Unversioned());
															#line 2323 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		wrMyOwner << dataDistributionModeLock;
															#line 2324 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(moveKeysLockOwnerKey, wrMyOwner.toValue());
															#line 2325 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		BinaryWriter wrLastWrite(Unversioned());
															#line 2326 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		wrLastWrite << deterministicRandom()->randomUniqueID();
															#line 2327 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(moveKeysLockWriteKey, wrLastWrite.toValue());
															#line 2329 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(dataDistributionModeKey, wr.toValue());
															#line 2330 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (mode)
															#line 17078 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2334 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(healthyZoneKey);
															#line 2334 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17084 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<SetDDModeActor*>(this)->actor_wait_state = 2;
															#line 2334 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 1, Optional<Value> >*>(static_cast<SetDDModeActor*>(this)));
															#line 17089 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && old,int loopDepth) 
	{
															#line 2315 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (oldMode < 0)
															#line 17103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2316 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			oldMode = 1;
															#line 2317 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (old.present())
															#line 17109 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2318 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				BinaryReader rd(old.get(), Unversioned());
															#line 2319 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				rd >> oldMode;
															#line 17115 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2322 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		BinaryWriter wrMyOwner(Unversioned());
															#line 2323 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		wrMyOwner << dataDistributionModeLock;
															#line 2324 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(moveKeysLockOwnerKey, wrMyOwner.toValue());
															#line 2325 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		BinaryWriter wrLastWrite(Unversioned());
															#line 2326 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		wrLastWrite << deterministicRandom()->randomUniqueID();
															#line 2327 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(moveKeysLockWriteKey, wrLastWrite.toValue());
															#line 2329 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.set(dataDistributionModeKey, wr.toValue());
															#line 2330 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (mode)
															#line 17134 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2334 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(healthyZoneKey);
															#line 2334 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17140 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<SetDDModeActor*>(this)->actor_wait_state = 2;
															#line 2334 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 1, Optional<Value> >*>(static_cast<SetDDModeActor*>(this)));
															#line 17145 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& old,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(old, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && old,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(old), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state > 0) static_cast<SetDDModeActor*>(this)->actor_wait_state = 0;
		static_cast<SetDDModeActor*>(this)->ActorCallback< SetDDModeActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SetDDModeActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 2342 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2342 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17236 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<SetDDModeActor*>(this)->actor_wait_state = 3;
															#line 2342 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SetDDModeActor, 2, Void >*>(static_cast<SetDDModeActor*>(this)));
															#line 17241 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Optional<Value> const& currentHealthyZoneValue,int loopDepth) 
	{
															#line 2335 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (currentHealthyZoneValue.present() && decodeHealthyZoneValue(currentHealthyZoneValue.get()).first == ignoreSSFailuresZoneString)
															#line 17250 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2338 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.clear(healthyZoneKey);
															#line 17254 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2340 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.clear(rebalanceDDIgnoreKey);
															#line 17258 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Optional<Value> && currentHealthyZoneValue,int loopDepth) 
	{
															#line 2335 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (currentHealthyZoneValue.present() && decodeHealthyZoneValue(currentHealthyZoneValue.get()).first == ignoreSSFailuresZoneString)
															#line 17267 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2338 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.clear(healthyZoneKey);
															#line 17271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2340 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.clear(rebalanceDDIgnoreKey);
															#line 17275 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Optional<Value> const& currentHealthyZoneValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(currentHealthyZoneValue, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Optional<Value> && currentHealthyZoneValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(currentHealthyZoneValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state > 0) static_cast<SetDDModeActor*>(this)->actor_wait_state = 0;
		static_cast<SetDDModeActor*>(this)->ActorCallback< SetDDModeActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SetDDModeActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 2343 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetDDModeActor*>(this)->SAV<int>::futures) { (void)(oldMode); this->~SetDDModeActorState(); static_cast<SetDDModeActor*>(this)->destroy(); return 0; }
															#line 17359 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetDDModeActor*>(this)->SAV< int >::value()) int(std::move(oldMode)); // state_var_RVO
		this->~SetDDModeActorState();
		static_cast<SetDDModeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 2343 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<SetDDModeActor*>(this)->SAV<int>::futures) { (void)(oldMode); this->~SetDDModeActorState(); static_cast<SetDDModeActor*>(this)->destroy(); return 0; }
															#line 17371 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<SetDDModeActor*>(this)->SAV< int >::value()) int(std::move(oldMode)); // state_var_RVO
		this->~SetDDModeActorState();
		static_cast<SetDDModeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state > 0) static_cast<SetDDModeActor*>(this)->actor_wait_state = 0;
		static_cast<SetDDModeActor*>(this)->ActorCallback< SetDDModeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SetDDModeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SetDDModeActor*>(this)->actor_wait_state > 0) static_cast<SetDDModeActor*>(this)->actor_wait_state = 0;
		static_cast<SetDDModeActor*>(this)->ActorCallback< SetDDModeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SetDDModeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SetDDModeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int mode;
															#line 2304 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 2305 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	int oldMode;
															#line 2306 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	BinaryWriter wr;
															#line 17551 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via setDDMode()
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class SetDDModeActor final : public Actor<int>, public ActorCallback< SetDDModeActor, 0, Optional<Value> >, public ActorCallback< SetDDModeActor, 1, Optional<Value> >, public ActorCallback< SetDDModeActor, 2, Void >, public ActorCallback< SetDDModeActor, 3, Void >, public FastAllocated<SetDDModeActor>, public SetDDModeActorState<SetDDModeActor> {
															#line 17556 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<SetDDModeActor>::operator new;
	using FastAllocated<SetDDModeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15475990744171659264UL, 7870226871020589312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SetDDModeActor, 0, Optional<Value> >;
friend struct ActorCallback< SetDDModeActor, 1, Optional<Value> >;
friend struct ActorCallback< SetDDModeActor, 2, Void >;
friend struct ActorCallback< SetDDModeActor, 3, Void >;
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	SetDDModeActor(Database const& cx,int const& mode) 
															#line 17576 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<int>(),
		   SetDDModeActorState<SetDDModeActor>(cx, mode),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("setDDMode", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3750982474694156544UL, 11167414918040745728UL);
		ActorExecutionContextHelper __helper(static_cast<SetDDModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("setDDMode");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("setDDMode", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SetDDModeActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SetDDModeActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SetDDModeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< SetDDModeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<int> setDDMode( Database const& cx, int const& mode ) {
															#line 2303 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<int>(new SetDDModeActor(cx, mode));
															#line 17612 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2350 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 17617 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via checkForExcludingServersTxActor()
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class CheckForExcludingServersTxActorActor>
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class CheckForExcludingServersTxActorActorState {
															#line 17624 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CheckForExcludingServersTxActorActorState(ReadYourWritesTransaction* const& tr,std::set<AddressExclusion>* const& exclusions,std::set<NetworkAddress>* const& inProgressExclusion) 
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   exclusions(exclusions),
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   inProgressExclusion(inProgressExclusion)
															#line 17635 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this));

	}
	~CheckForExcludingServersTxActorActorState() 
	{
		fdb_probe_actor_destroy("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2355 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			ASSERT(inProgressExclusion->size() == 0);
															#line 2356 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!exclusions->size())
															#line 17652 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2357 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<CheckForExcludingServersTxActorActor*>(this)->SAV<bool>::futures) { (void)(true); this->~CheckForExcludingServersTxActorActorState(); static_cast<CheckForExcludingServersTxActorActor*>(this)->destroy(); return 0; }
															#line 17656 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<CheckForExcludingServersTxActorActor*>(this)->SAV< bool >::value()) bool(true);
				this->~CheckForExcludingServersTxActorActorState();
				static_cast<CheckForExcludingServersTxActorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2359 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2360 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2361 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2367 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(serverListKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2367 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17672 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 1;
															#line 2367 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*>(static_cast<CheckForExcludingServersTxActorActor*>(this)));
															#line 17677 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckForExcludingServersTxActorActorState();
		static_cast<CheckForExcludingServersTxActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& serverList,int loopDepth) 
	{
															#line 2368 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2370 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ok = true;
															#line 2371 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& s : serverList ) {
															#line 2372 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto addresses = decodeServerListValue(s.value).getKeyValues.getEndpoint().addresses;
															#line 2373 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (addressExcluded(*exclusions, addresses.address))
															#line 17708 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2374 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2375 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(addresses.address);
															#line 17714 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2377 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (addresses.secondaryAddress.present() && addressExcluded(*exclusions, addresses.secondaryAddress.get()))
															#line 17718 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2378 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(addresses.secondaryAddress.get());
															#line 17724 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2383 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 17729 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2384 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Standalone<StringRef>>> __when_expr_1 = tr->get(logsKey);
															#line 2384 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17735 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 2;
															#line 2384 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*>(static_cast<CheckForExcludingServersTxActorActor*>(this)));
															#line 17740 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(RangeResult && serverList,int loopDepth) 
	{
															#line 2368 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2370 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ok = true;
															#line 2371 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& s : serverList ) {
															#line 2372 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			auto addresses = decodeServerListValue(s.value).getKeyValues.getEndpoint().addresses;
															#line 2373 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (addressExcluded(*exclusions, addresses.address))
															#line 17762 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2374 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2375 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(addresses.address);
															#line 17768 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
															#line 2377 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (addresses.secondaryAddress.present() && addressExcluded(*exclusions, addresses.secondaryAddress.get()))
															#line 17772 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2378 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2379 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(addresses.secondaryAddress.get());
															#line 17778 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2383 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 17783 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2384 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Standalone<StringRef>>> __when_expr_1 = tr->get(logsKey);
															#line 2384 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17789 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 2;
															#line 2384 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*>(static_cast<CheckForExcludingServersTxActorActor*>(this)));
															#line 17794 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(RangeResult const& serverList,int loopDepth) 
	{
		loopDepth = a_body1cont1(serverList, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && serverList,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(serverList), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersTxActorActor*>(this)->ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2401 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<CheckForExcludingServersTxActorActor*>(this)->SAV<bool>::futures) { (void)(ok); this->~CheckForExcludingServersTxActorActorState(); static_cast<CheckForExcludingServersTxActorActor*>(this)->destroy(); return 0; }
															#line 17883 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<CheckForExcludingServersTxActorActor*>(this)->SAV< bool >::value()) bool(std::move(ok)); // state_var_RVO
		this->~CheckForExcludingServersTxActorActorState();
		static_cast<CheckForExcludingServersTxActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Optional<Standalone<StringRef>> const& value,int loopDepth) 
	{
															#line 2385 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(value.present());
															#line 2386 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto logs = decodeLogsValue(value.get());
															#line 2387 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto const& log : logs.first ) {
															#line 2388 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (log.second == NetworkAddress() || addressExcluded(*exclusions, log.second))
															#line 17901 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2389 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2390 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(log.second);
															#line 17907 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2393 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto const& log : logs.second ) {
															#line 2394 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (log.second == NetworkAddress() || addressExcluded(*exclusions, log.second))
															#line 17914 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2395 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2396 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(log.second);
															#line 17920 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Optional<Standalone<StringRef>> && value,int loopDepth) 
	{
															#line 2385 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(value.present());
															#line 2386 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		auto logs = decodeLogsValue(value.get());
															#line 2387 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto const& log : logs.first ) {
															#line 2388 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (log.second == NetworkAddress() || addressExcluded(*exclusions, log.second))
															#line 17937 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2389 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2390 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(log.second);
															#line 17943 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2393 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto const& log : logs.second ) {
															#line 2394 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (log.second == NetworkAddress() || addressExcluded(*exclusions, log.second))
															#line 17950 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2395 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				ok = false;
															#line 2396 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				inProgressExclusion->insert(log.second);
															#line 17956 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Standalone<StringRef>> const& value,int loopDepth) 
	{
		loopDepth = a_body1cont7(value, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Standalone<StringRef>> && value,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(value), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersTxActorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersTxActorActor*>(this)->ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*,Optional<Standalone<StringRef>> const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*,Optional<Standalone<StringRef>> && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction* tr;
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<AddressExclusion>* exclusions;
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<NetworkAddress>* inProgressExclusion;
															#line 2370 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool ok;
															#line 18046 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via checkForExcludingServersTxActor()
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class CheckForExcludingServersTxActorActor final : public Actor<bool>, public ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >, public ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >, public FastAllocated<CheckForExcludingServersTxActorActor>, public CheckForExcludingServersTxActorActorState<CheckForExcludingServersTxActorActor> {
															#line 18051 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<CheckForExcludingServersTxActorActor>::operator new;
	using FastAllocated<CheckForExcludingServersTxActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(88321450997157632UL, 15131550140251389696UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >;
friend struct ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >;
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CheckForExcludingServersTxActorActor(ReadYourWritesTransaction* const& tr,std::set<AddressExclusion>* const& exclusions,std::set<NetworkAddress>* const& inProgressExclusion) 
															#line 18069 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<bool>(),
		   CheckForExcludingServersTxActorActorState<CheckForExcludingServersTxActorActor>(tr, exclusions, inProgressExclusion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(835331421418836480UL, 17258140674795182080UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersTxActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkForExcludingServersTxActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkForExcludingServersTxActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckForExcludingServersTxActorActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckForExcludingServersTxActorActor, 1, Optional<Standalone<StringRef>> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<bool> checkForExcludingServersTxActor( ReadYourWritesTransaction* const& tr, std::set<AddressExclusion>* const& exclusions, std::set<NetworkAddress>* const& inProgressExclusion ) {
															#line 2351 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<bool>(new CheckForExcludingServersTxActorActor(tr, exclusions, inProgressExclusion));
															#line 18103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2403 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 18108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via checkForExcludingServers()
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class CheckForExcludingServersActor>
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class CheckForExcludingServersActorState {
															#line 18115 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CheckForExcludingServersActorState(Database const& cx,std::vector<AddressExclusion> const& excl,bool const& waitForAllExcluded) 
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   excl(excl),
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   waitForAllExcluded(waitForAllExcluded),
															#line 2407 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   exclusions(excl.begin(), excl.end()),
															#line 2408 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   inProgressExclusion()
															#line 18130 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("checkForExcludingServers", reinterpret_cast<unsigned long>(this));

	}
	~CheckForExcludingServersActorState() 
	{
		fdb_probe_actor_destroy("checkForExcludingServers", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2410 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 18145 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckForExcludingServersActorState();
		static_cast<CheckForExcludingServersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2426 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<CheckForExcludingServersActor*>(this)->SAV<std::set<NetworkAddress>>::futures) { (void)(inProgressExclusion); this->~CheckForExcludingServersActorState(); static_cast<CheckForExcludingServersActor*>(this)->destroy(); return 0; }
															#line 18168 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<CheckForExcludingServersActor*>(this)->SAV< std::set<NetworkAddress> >::value()) std::set<NetworkAddress>(std::move(inProgressExclusion)); // state_var_RVO
		this->~CheckForExcludingServersActorState();
		static_cast<CheckForExcludingServersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2411 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr = ReadYourWritesTransaction(cx);
															#line 2412 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		inProgressExclusion.clear();
															#line 18189 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		try {
															#line 2414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<bool> __when_expr_0 = checkForExcludingServersTxActor(&tr, &exclusions, &inProgressExclusion);
															#line 2414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18195 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 1;
															#line 2414 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersActor, 0, bool >*>(static_cast<CheckForExcludingServersActor*>(this)));
															#line 18200 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2422 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent("CheckForExcludingServersError").error(e);
															#line 2423 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2423 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18239 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 3;
															#line 2423 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersActor, 2, Void >*>(static_cast<CheckForExcludingServersActor*>(this)));
															#line 18244 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(bool const& ok,int loopDepth) 
	{
															#line 2415 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 18259 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2416 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<CheckForExcludingServersActor*>(this)->SAV<std::set<NetworkAddress>>::futures) { (void)(inProgressExclusion); this->~CheckForExcludingServersActorState(); static_cast<CheckForExcludingServersActor*>(this)->destroy(); return 0; }
															#line 18263 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<CheckForExcludingServersActor*>(this)->SAV< std::set<NetworkAddress> >::value()) std::set<NetworkAddress>(std::move(inProgressExclusion)); // state_var_RVO
			this->~CheckForExcludingServersActorState();
			static_cast<CheckForExcludingServersActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2417 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!waitForAllExcluded)
															#line 18271 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2420 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delayJittered(1.0);
															#line 2420 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18279 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 2;
															#line 2420 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersActor, 1, Void >*>(static_cast<CheckForExcludingServersActor*>(this)));
															#line 18284 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(bool && ok,int loopDepth) 
	{
															#line 2415 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (ok)
															#line 18293 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2416 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<CheckForExcludingServersActor*>(this)->SAV<std::set<NetworkAddress>>::futures) { (void)(inProgressExclusion); this->~CheckForExcludingServersActorState(); static_cast<CheckForExcludingServersActor*>(this)->destroy(); return 0; }
															#line 18297 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<CheckForExcludingServersActor*>(this)->SAV< std::set<NetworkAddress> >::value()) std::set<NetworkAddress>(std::move(inProgressExclusion)); // state_var_RVO
			this->~CheckForExcludingServersActorState();
			static_cast<CheckForExcludingServersActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2417 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!waitForAllExcluded)
															#line 18305 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2420 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delayJittered(1.0);
															#line 2420 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18313 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 2;
															#line 2420 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckForExcludingServersActor, 1, Void >*>(static_cast<CheckForExcludingServersActor*>(this)));
															#line 18318 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(bool const& ok,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(ok, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(bool && ok,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(ok), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersActor*>(this)->ActorCallback< CheckForExcludingServersActor, 0, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 0, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 0, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersActor, 0, bool >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersActor*>(this)->ActorCallback< CheckForExcludingServersActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state > 0) static_cast<CheckForExcludingServersActor*>(this)->actor_wait_state = 0;
		static_cast<CheckForExcludingServersActor*>(this)->ActorCallback< CheckForExcludingServersActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckForExcludingServersActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckForExcludingServersActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<AddressExclusion> excl;
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	bool waitForAllExcluded;
															#line 2407 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<AddressExclusion> exclusions;
															#line 2408 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::set<NetworkAddress> inProgressExclusion;
															#line 2411 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction tr;
															#line 18597 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via checkForExcludingServers()
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class CheckForExcludingServersActor final : public Actor<std::set<NetworkAddress>>, public ActorCallback< CheckForExcludingServersActor, 0, bool >, public ActorCallback< CheckForExcludingServersActor, 1, Void >, public ActorCallback< CheckForExcludingServersActor, 2, Void >, public FastAllocated<CheckForExcludingServersActor>, public CheckForExcludingServersActorState<CheckForExcludingServersActor> {
															#line 18602 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<CheckForExcludingServersActor>::operator new;
	using FastAllocated<CheckForExcludingServersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13296239110591824128UL, 3559763972327933184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::set<NetworkAddress>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckForExcludingServersActor, 0, bool >;
friend struct ActorCallback< CheckForExcludingServersActor, 1, Void >;
friend struct ActorCallback< CheckForExcludingServersActor, 2, Void >;
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	CheckForExcludingServersActor(Database const& cx,std::vector<AddressExclusion> const& excl,bool const& waitForAllExcluded) 
															#line 18621 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<std::set<NetworkAddress>>(),
		   CheckForExcludingServersActorState<CheckForExcludingServersActor>(cx, excl, waitForAllExcluded),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkForExcludingServers", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6376532781582041856UL, 15345192279576478720UL);
		ActorExecutionContextHelper __helper(static_cast<CheckForExcludingServersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkForExcludingServers");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkForExcludingServers", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckForExcludingServersActor, 0, bool >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckForExcludingServersActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckForExcludingServersActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<std::set<NetworkAddress>> checkForExcludingServers( Database const& cx, std::vector<AddressExclusion> const& excl, bool const& waitForAllExcluded ) {
															#line 2404 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<std::set<NetworkAddress>>(new CheckForExcludingServersActor(cx, excl, waitForAllExcluded));
															#line 18656 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2428 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 18661 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via mgmtSnapCreate()
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class MgmtSnapCreateActor>
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class MgmtSnapCreateActorState {
															#line 18668 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	MgmtSnapCreateActorState(Database const& cx,Standalone<StringRef> const& snapCmd,UID const& snapUID) 
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   snapCmd(snapCmd),
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   snapUID(snapUID)
															#line 18679 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("mgmtSnapCreate", reinterpret_cast<unsigned long>(this));

	}
	~MgmtSnapCreateActorState() 
	{
		fdb_probe_actor_destroy("mgmtSnapCreate", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 2431 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				StrictFuture<Void> __when_expr_0 = snapCreate(cx, snapCmd, snapUID);
															#line 2431 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (static_cast<MgmtSnapCreateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 18697 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<MgmtSnapCreateActor*>(this)->actor_wait_state = 1;
															#line 2431 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MgmtSnapCreateActor, 0, Void >*>(static_cast<MgmtSnapCreateActor*>(this)));
															#line 18702 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MgmtSnapCreateActorState();
		static_cast<MgmtSnapCreateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2435 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			TraceEvent(SevWarn, "SnapCreateFailed").error(e).detail("snapUID", snapUID);
															#line 2436 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 18734 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("SnapCreateSucceeded").detail("snapUID", snapUID);
															#line 2433 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<MgmtSnapCreateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MgmtSnapCreateActorState(); static_cast<MgmtSnapCreateActor*>(this)->destroy(); return 0; }
															#line 18750 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<MgmtSnapCreateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MgmtSnapCreateActorState();
		static_cast<MgmtSnapCreateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2432 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		TraceEvent("SnapCreateSucceeded").detail("snapUID", snapUID);
															#line 2433 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<MgmtSnapCreateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MgmtSnapCreateActorState(); static_cast<MgmtSnapCreateActor*>(this)->destroy(); return 0; }
															#line 18764 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<MgmtSnapCreateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MgmtSnapCreateActorState();
		static_cast<MgmtSnapCreateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MgmtSnapCreateActor*>(this)->actor_wait_state > 0) static_cast<MgmtSnapCreateActor*>(this)->actor_wait_state = 0;
		static_cast<MgmtSnapCreateActor*>(this)->ActorCallback< MgmtSnapCreateActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MgmtSnapCreateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("mgmtSnapCreate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MgmtSnapCreateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mgmtSnapCreate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MgmtSnapCreateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("mgmtSnapCreate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MgmtSnapCreateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mgmtSnapCreate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MgmtSnapCreateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("mgmtSnapCreate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MgmtSnapCreateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mgmtSnapCreate", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Standalone<StringRef> snapCmd;
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	UID snapUID;
															#line 18853 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via mgmtSnapCreate()
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class MgmtSnapCreateActor final : public Actor<Void>, public ActorCallback< MgmtSnapCreateActor, 0, Void >, public FastAllocated<MgmtSnapCreateActor>, public MgmtSnapCreateActorState<MgmtSnapCreateActor> {
															#line 18858 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<MgmtSnapCreateActor>::operator new;
	using FastAllocated<MgmtSnapCreateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18414145460991661568UL, 2348246289642665728UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MgmtSnapCreateActor, 0, Void >;
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	MgmtSnapCreateActor(Database const& cx,Standalone<StringRef> const& snapCmd,UID const& snapUID) 
															#line 18875 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   MgmtSnapCreateActorState<MgmtSnapCreateActor>(cx, snapCmd, snapUID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("mgmtSnapCreate", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8701768690566651136UL, 7813893908165313792UL);
		ActorExecutionContextHelper __helper(static_cast<MgmtSnapCreateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("mgmtSnapCreate");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("mgmtSnapCreate", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MgmtSnapCreateActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> mgmtSnapCreate( Database const& cx, Standalone<StringRef> const& snapCmd, UID const& snapUID ) {
															#line 2429 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new MgmtSnapCreateActor(cx, snapCmd, snapUID));
															#line 18908 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2439 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 18913 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForFullReplication()
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class WaitForFullReplicationActor>
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class WaitForFullReplicationActorState {
															#line 18920 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	WaitForFullReplicationActorState(Database const& cx) 
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx),
															#line 2441 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   tr(cx)
															#line 18929 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForFullReplication", reinterpret_cast<unsigned long>(this));

	}
	~WaitForFullReplicationActorState() 
	{
		fdb_probe_actor_destroy("waitForFullReplication", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2442 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 18944 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForFullReplicationActorState();
		static_cast<WaitForFullReplicationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2444 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 2445 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2446 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2448 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(configKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2448 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18983 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 1;
															#line 2448 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForFullReplicationActor, 0, RangeResult >*>(static_cast<WaitForFullReplicationActor*>(this)));
															#line 18988 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2475 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 2475 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19012 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 5;
															#line 2475 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WaitForFullReplicationActor, 4, Void >*>(static_cast<WaitForFullReplicationActor*>(this)));
															#line 19017 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(RangeResult const& confResults,int loopDepth) 
	{
															#line 2449 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!confResults.more && confResults.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2450 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		config = DatabaseConfiguration();
															#line 2451 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		config.fromKeyValues((VectorRef<KeyValueRef>)confResults);
															#line 2453 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		replicasFutures = std::vector<Future<Optional<Value>>>();
															#line 2454 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& region : config.regions ) {
															#line 2455 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			replicasFutures.push_back(tr.get(datacenterReplicasKeyFor(region.dcId)));
															#line 19042 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2457 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitForAll(replicasFutures);
															#line 2457 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 19048 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 2;
															#line 2457 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForFullReplicationActor, 1, Void >*>(static_cast<WaitForFullReplicationActor*>(this)));
															#line 19053 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(RangeResult && confResults,int loopDepth) 
	{
															#line 2449 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		ASSERT(!confResults.more && confResults.size() < CLIENT_KNOBS->TOO_MANY);
															#line 2450 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		config = DatabaseConfiguration();
															#line 2451 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		config.fromKeyValues((VectorRef<KeyValueRef>)confResults);
															#line 2453 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		replicasFutures = std::vector<Future<Optional<Value>>>();
															#line 2454 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for( auto& region : config.regions ) {
															#line 2455 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			replicasFutures.push_back(tr.get(datacenterReplicasKeyFor(region.dcId)));
															#line 19072 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		}
															#line 2457 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitForAll(replicasFutures);
															#line 2457 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 19078 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 2;
															#line 2457 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForFullReplicationActor, 1, Void >*>(static_cast<WaitForFullReplicationActor*>(this)));
															#line 19083 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& confResults,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(confResults, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && confResults,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(confResults), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state > 0) static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForFullReplicationActor*>(this)->ActorCallback< WaitForFullReplicationActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForFullReplicationActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 2459 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		watchFutures = std::vector<Future<Void>>();
															#line 2460 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < config.regions.size();i++) {
															#line 2461 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!replicasFutures[i].get().present() || decodeDatacenterReplicasValue(replicasFutures[i].get().get()) < config.storageTeamSize)
															#line 19171 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2463 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				watchFutures.push_back(tr.watch(datacenterReplicasKeyFor(config.regions[i].dcId)));
															#line 19175 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2467 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!watchFutures.size() || (config.usableRegions == 1 && watchFutures.size() < config.regions.size()))
															#line 19180 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2468 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<WaitForFullReplicationActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitForFullReplicationActorState(); static_cast<WaitForFullReplicationActor*>(this)->destroy(); return 0; }
															#line 19184 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<WaitForFullReplicationActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitForFullReplicationActorState();
			static_cast<WaitForFullReplicationActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2471 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2471 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 19194 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 3;
															#line 2471 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitForFullReplicationActor, 2, Void >*>(static_cast<WaitForFullReplicationActor*>(this)));
															#line 19199 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 2459 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		watchFutures = std::vector<Future<Void>>();
															#line 2460 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		for(int i = 0;i < config.regions.size();i++) {
															#line 2461 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!replicasFutures[i].get().present() || decodeDatacenterReplicasValue(replicasFutures[i].get().get()) < config.storageTeamSize)
															#line 19212 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2463 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				watchFutures.push_back(tr.watch(datacenterReplicasKeyFor(config.regions[i].dcId)));
															#line 19216 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2467 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!watchFutures.size() || (config.usableRegions == 1 && watchFutures.size() < config.regions.size()))
															#line 19221 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2468 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (!static_cast<WaitForFullReplicationActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitForFullReplicationActorState(); static_cast<WaitForFullReplicationActor*>(this)->destroy(); return 0; }
															#line 19225 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			new (&static_cast<WaitForFullReplicationActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitForFullReplicationActorState();
			static_cast<WaitForFullReplicationActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2471 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2471 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 19235 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 3;
															#line 2471 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitForFullReplicationActor, 2, Void >*>(static_cast<WaitForFullReplicationActor*>(this)));
															#line 19240 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state > 0) static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForFullReplicationActor*>(this)->ActorCallback< WaitForFullReplicationActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForFullReplicationActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 2472 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = waitForAny(watchFutures);
															#line 2472 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 19326 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 4;
															#line 2472 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitForFullReplicationActor, 3, Void >*>(static_cast<WaitForFullReplicationActor*>(this)));
															#line 19331 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 2472 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = waitForAny(watchFutures);
															#line 2472 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 19342 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 4;
															#line 2472 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitForFullReplicationActor, 3, Void >*>(static_cast<WaitForFullReplicationActor*>(this)));
															#line 19347 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state > 0) static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForFullReplicationActor*>(this)->ActorCallback< WaitForFullReplicationActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitForFullReplicationActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 2473 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.reset();
															#line 19431 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 2473 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr.reset();
															#line 19440 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state > 0) static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForFullReplicationActor*>(this)->ActorCallback< WaitForFullReplicationActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitForFullReplicationActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state > 0) static_cast<WaitForFullReplicationActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForFullReplicationActor*>(this)->ActorCallback< WaitForFullReplicationActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< WaitForFullReplicationActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< WaitForFullReplicationActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2441 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	ReadYourWritesTransaction tr;
															#line 2450 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	DatabaseConfiguration config;
															#line 2453 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<Future<Optional<Value>>> replicasFutures;
															#line 2459 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	std::vector<Future<Void>> watchFutures;
															#line 19630 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via waitForFullReplication()
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class WaitForFullReplicationActor final : public Actor<Void>, public ActorCallback< WaitForFullReplicationActor, 0, RangeResult >, public ActorCallback< WaitForFullReplicationActor, 1, Void >, public ActorCallback< WaitForFullReplicationActor, 2, Void >, public ActorCallback< WaitForFullReplicationActor, 3, Void >, public ActorCallback< WaitForFullReplicationActor, 4, Void >, public FastAllocated<WaitForFullReplicationActor>, public WaitForFullReplicationActorState<WaitForFullReplicationActor> {
															#line 19635 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<WaitForFullReplicationActor>::operator new;
	using FastAllocated<WaitForFullReplicationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12900296594137798656UL, 7645364426118868480UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForFullReplicationActor, 0, RangeResult >;
friend struct ActorCallback< WaitForFullReplicationActor, 1, Void >;
friend struct ActorCallback< WaitForFullReplicationActor, 2, Void >;
friend struct ActorCallback< WaitForFullReplicationActor, 3, Void >;
friend struct ActorCallback< WaitForFullReplicationActor, 4, Void >;
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	WaitForFullReplicationActor(Database const& cx) 
															#line 19656 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   WaitForFullReplicationActorState<WaitForFullReplicationActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForFullReplication", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7093723045182751488UL, 16279882402874844928UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForFullReplicationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForFullReplication");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForFullReplication", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForFullReplicationActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitForFullReplicationActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitForFullReplicationActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WaitForFullReplicationActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< WaitForFullReplicationActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> waitForFullReplication( Database const& cx ) {
															#line 2440 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new WaitForFullReplicationActor(cx));
															#line 19693 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2479 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 19698 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via timeKeeperSetDisable()
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class TimeKeeperSetDisableActor>
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class TimeKeeperSetDisableActorState {
															#line 19705 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	TimeKeeperSetDisableActorState(Database const& cx) 
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : cx(cx)
															#line 19712 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this));

	}
	~TimeKeeperSetDisableActorState() 
	{
		fdb_probe_actor_destroy("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2481 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			;
															#line 19727 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TimeKeeperSetDisableActorState();
		static_cast<TimeKeeperSetDisableActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2482 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr = Transaction(cx);
															#line 19757 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		try {
															#line 2484 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2485 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2486 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr.set(timeKeeperDisableKey, StringRef());
															#line 2487 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr.commit();
															#line 2487 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<TimeKeeperSetDisableActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 19769 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TimeKeeperSetDisableActor*>(this)->actor_wait_state = 1;
															#line 2487 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TimeKeeperSetDisableActor, 0, Void >*>(static_cast<TimeKeeperSetDisableActor*>(this)));
															#line 19774 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2490 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2490 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<TimeKeeperSetDisableActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 19798 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<TimeKeeperSetDisableActor*>(this)->actor_wait_state = 2;
															#line 2490 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TimeKeeperSetDisableActor, 1, Void >*>(static_cast<TimeKeeperSetDisableActor*>(this)));
															#line 19803 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2488 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<TimeKeeperSetDisableActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TimeKeeperSetDisableActorState(); static_cast<TimeKeeperSetDisableActor*>(this)->destroy(); return 0; }
															#line 19818 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<TimeKeeperSetDisableActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TimeKeeperSetDisableActorState();
		static_cast<TimeKeeperSetDisableActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2488 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<TimeKeeperSetDisableActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TimeKeeperSetDisableActorState(); static_cast<TimeKeeperSetDisableActor*>(this)->destroy(); return 0; }
															#line 19830 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<TimeKeeperSetDisableActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TimeKeeperSetDisableActorState();
		static_cast<TimeKeeperSetDisableActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TimeKeeperSetDisableActor*>(this)->actor_wait_state > 0) static_cast<TimeKeeperSetDisableActor*>(this)->actor_wait_state = 0;
		static_cast<TimeKeeperSetDisableActor*>(this)->ActorCallback< TimeKeeperSetDisableActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TimeKeeperSetDisableActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TimeKeeperSetDisableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TimeKeeperSetDisableActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TimeKeeperSetDisableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TimeKeeperSetDisableActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TimeKeeperSetDisableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TimeKeeperSetDisableActor*>(this)->actor_wait_state > 0) static_cast<TimeKeeperSetDisableActor*>(this)->actor_wait_state = 0;
		static_cast<TimeKeeperSetDisableActor*>(this)->ActorCallback< TimeKeeperSetDisableActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TimeKeeperSetDisableActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TimeKeeperSetDisableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TimeKeeperSetDisableActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TimeKeeperSetDisableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TimeKeeperSetDisableActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TimeKeeperSetDisableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Database cx;
															#line 2482 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction tr;
															#line 20004 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via timeKeeperSetDisable()
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class TimeKeeperSetDisableActor final : public Actor<Void>, public ActorCallback< TimeKeeperSetDisableActor, 0, Void >, public ActorCallback< TimeKeeperSetDisableActor, 1, Void >, public FastAllocated<TimeKeeperSetDisableActor>, public TimeKeeperSetDisableActorState<TimeKeeperSetDisableActor> {
															#line 20009 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<TimeKeeperSetDisableActor>::operator new;
	using FastAllocated<TimeKeeperSetDisableActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3309881924243289856UL, 5281314297534302464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TimeKeeperSetDisableActor, 0, Void >;
friend struct ActorCallback< TimeKeeperSetDisableActor, 1, Void >;
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	TimeKeeperSetDisableActor(Database const& cx) 
															#line 20027 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   TimeKeeperSetDisableActorState<TimeKeeperSetDisableActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11842399027119930112UL, 15387329726068933120UL);
		ActorExecutionContextHelper __helper(static_cast<TimeKeeperSetDisableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("timeKeeperSetDisable");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("timeKeeperSetDisable", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TimeKeeperSetDisableActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TimeKeeperSetDisableActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> timeKeeperSetDisable( Database const& cx ) {
															#line 2480 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new TimeKeeperSetDisableActor(cx));
															#line 20061 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2494 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 20066 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via lockDatabase()
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class LockDatabaseActor>
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class LockDatabaseActorState {
															#line 20073 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	LockDatabaseActorState(Transaction* const& tr,UID const& id) 
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		 : tr(tr),
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		   id(id)
															#line 20082 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("lockDatabase", reinterpret_cast<unsigned long>(this));

	}
	~LockDatabaseActorState() 
	{
		fdb_probe_actor_destroy("lockDatabase", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2496 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2497 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2498 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr->get(databaseLockedKey);
															#line 2498 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (static_cast<LockDatabaseActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 20103 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LockDatabaseActor*>(this)->actor_wait_state = 1;
															#line 2498 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LockDatabaseActor, 0, Optional<Value> >*>(static_cast<LockDatabaseActor*>(this)));
															#line 20108 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LockDatabaseActorState();
		static_cast<LockDatabaseActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> const& val,int loopDepth) 
	{
															#line 2500 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present())
															#line 20131 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2501 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (BinaryReader::fromStringRef<UID>(val.get().substr(10), Unversioned()) == id)
															#line 20135 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2502 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<LockDatabaseActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LockDatabaseActorState(); static_cast<LockDatabaseActor*>(this)->destroy(); return 0; }
															#line 20139 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<LockDatabaseActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LockDatabaseActorState();
				static_cast<LockDatabaseActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2505 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				return a_body1Catch1(database_locked(), loopDepth);
															#line 20149 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2509 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr->atomicOp(databaseLockedKey, BinaryWriter::toValue(id, Unversioned()).withPrefix("0123456789"_sr).withSuffix("\x00\x00\x00\x00"_sr), MutationRef::SetVersionstampedValue);
															#line 2512 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr->addWriteConflictRange(normalKeys);
															#line 2513 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<LockDatabaseActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LockDatabaseActorState(); static_cast<LockDatabaseActor*>(this)->destroy(); return 0; }
															#line 20158 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<LockDatabaseActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LockDatabaseActorState();
		static_cast<LockDatabaseActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> && val,int loopDepth) 
	{
															#line 2500 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (val.present())
															#line 20170 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		{
															#line 2501 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
			if (BinaryReader::fromStringRef<UID>(val.get().substr(10), Unversioned()) == id)
															#line 20174 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			{
															#line 2502 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				if (!static_cast<LockDatabaseActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LockDatabaseActorState(); static_cast<LockDatabaseActor*>(this)->destroy(); return 0; }
															#line 20178 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
				new (&static_cast<LockDatabaseActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LockDatabaseActorState();
				static_cast<LockDatabaseActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2505 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
				return a_body1Catch1(database_locked(), loopDepth);
															#line 20188 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
			}
		}
															#line 2509 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr->atomicOp(databaseLockedKey, BinaryWriter::toValue(id, Unversioned()).withPrefix("0123456789"_sr).withSuffix("\x00\x00\x00\x00"_sr), MutationRef::SetVersionstampedValue);
															#line 2512 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		tr->addWriteConflictRange(normalKeys);
															#line 2513 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
		if (!static_cast<LockDatabaseActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LockDatabaseActorState(); static_cast<LockDatabaseActor*>(this)->destroy(); return 0; }
															#line 20197 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		new (&static_cast<LockDatabaseActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LockDatabaseActorState();
		static_cast<LockDatabaseActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont1(val, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LockDatabaseActor*>(this)->actor_wait_state > 0) static_cast<LockDatabaseActor*>(this)->actor_wait_state = 0;
		static_cast<LockDatabaseActor*>(this)->ActorCallback< LockDatabaseActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< LockDatabaseActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("lockDatabase", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LockDatabaseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lockDatabase", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LockDatabaseActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("lockDatabase", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LockDatabaseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lockDatabase", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LockDatabaseActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("lockDatabase", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LockDatabaseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lockDatabase", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	Transaction* tr;
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	UID id;
															#line 20284 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
};
// This generated class is to be used only via lockDatabase()
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class LockDatabaseActor final : public Actor<Void>, public ActorCallback< LockDatabaseActor, 0, Optional<Value> >, public FastAllocated<LockDatabaseActor>, public LockDatabaseActorState<LockDatabaseActor> {
															#line 20289 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
	using FastAllocated<LockDatabaseActor>::operator new;
	using FastAllocated<LockDatabaseActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8351088034109426432UL, 15680878095689640704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LockDatabaseActor, 0, Optional<Value> >;
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	LockDatabaseActor(Transaction* const& tr,UID const& id) 
															#line 20306 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
		 : Actor<Void>(),
		   LockDatabaseActorState<LockDatabaseActor>(tr, id),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("lockDatabase", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15971211100070603520UL, 4275338283381576192UL);
		ActorExecutionContextHelper __helper(static_cast<LockDatabaseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("lockDatabase");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("lockDatabase", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LockDatabaseActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
[[nodiscard]] Future<Void> lockDatabase( Transaction* const& tr, UID const& id ) {
															#line 2495 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
	return Future<Void>(new LockDatabaseActor(tr, id));
															#line 20339 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
}

#line 2515 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"

															#line 20344 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via lockDatabase()
															#line 2516 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
template <class LockDatabaseActor1>
															#line 2516 "/Users/ec2-user/foundationdb/fdbclient/ManagementAPI.actor.cpp"
class LockDatabaseActor1State {
															#line 20351 "/Users/ec2-user/foundationdb_build_output_macos_arm64/fdbclient/ManagementAPI.actor.g.cpp"
public:
															#line 2516 "/Users/ec2-user/foundationdb/fdbclient/