﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/textract/model/EntityType.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>

using namespace Aws::Utils;


namespace Aws
{
  namespace Textract
  {
    namespace Model
    {
      namespace EntityTypeMapper
      {

        static const int KEY_HASH = HashingUtils::HashString("KEY");
        static const int VALUE_HASH = HashingUtils::HashString("VALUE");
        static const int COLUMN_HEADER_HASH = HashingUtils::HashString("COLUMN_HEADER");


        EntityType GetEntityTypeForName(const Aws::String& name)
        {
          int hashCode = HashingUtils::HashString(name.c_str());
          if (hashCode == KEY_HASH)
          {
            return EntityType::KEY;
          }
          else if (hashCode == VALUE_HASH)
          {
            return EntityType::VALUE;
          }
          else if (hashCode == COLUMN_HEADER_HASH)
          {
            return EntityType::COLUMN_HEADER;
          }
          EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
          if(overflowContainer)
          {
            overflowContainer->StoreOverflow(hashCode, name);
            return static_cast<EntityType>(hashCode);
          }

          return EntityType::NOT_SET;
        }

        Aws::String GetNameForEntityType(EntityType enumValue)
        {
          switch(enumValue)
          {
          case EntityType::KEY:
            return "KEY";
          case EntityType::VALUE:
            return "VALUE";
          case EntityType::COLUMN_HEADER:
            return "COLUMN_HEADER";
          default:
            EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
            if(overflowContainer)
            {
              return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
            }

            return {};
          }
        }

      } // namespace EntityTypeMapper
    } // namespace Model
  } // namespace Textract
} // namespace Aws
