﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/core/client/AWSError.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/redshift-data/RedshiftDataAPIServiceErrors.h>
#include <aws/redshift-data/model/BatchExecuteStatementException.h>
#include <aws/redshift-data/model/ExecuteStatementException.h>
#include <aws/redshift-data/model/ResourceNotFoundException.h>

using namespace Aws::Client;
using namespace Aws::Utils;
using namespace Aws::RedshiftDataAPIService;
using namespace Aws::RedshiftDataAPIService::Model;

namespace Aws
{
namespace RedshiftDataAPIService
{
template<> AWS_REDSHIFTDATAAPISERVICE_API BatchExecuteStatementException RedshiftDataAPIServiceError::GetModeledError()
{
  assert(this->GetErrorType() == RedshiftDataAPIServiceErrors::BATCH_EXECUTE_STATEMENT);
  return BatchExecuteStatementException(this->GetJsonPayload().View());
}

template<> AWS_REDSHIFTDATAAPISERVICE_API ExecuteStatementException RedshiftDataAPIServiceError::GetModeledError()
{
  assert(this->GetErrorType() == RedshiftDataAPIServiceErrors::EXECUTE_STATEMENT);
  return ExecuteStatementException(this->GetJsonPayload().View());
}

template<> AWS_REDSHIFTDATAAPISERVICE_API ResourceNotFoundException RedshiftDataAPIServiceError::GetModeledError()
{
  assert(this->GetErrorType() == RedshiftDataAPIServiceErrors::RESOURCE_NOT_FOUND);
  return ResourceNotFoundException(this->GetJsonPayload().View());
}

namespace RedshiftDataAPIServiceErrorMapper
{

static const int BATCH_EXECUTE_STATEMENT_HASH = HashingUtils::HashString("BatchExecuteStatementException");
static const int EXECUTE_STATEMENT_HASH = HashingUtils::HashString("ExecuteStatementException");
static const int INTERNAL_SERVER_HASH = HashingUtils::HashString("InternalServerException");
static const int DATABASE_CONNECTION_HASH = HashingUtils::HashString("DatabaseConnectionException");
static const int QUERY_TIMEOUT_HASH = HashingUtils::HashString("QueryTimeoutException");
static const int ACTIVE_SESSIONS_EXCEEDED_HASH = HashingUtils::HashString("ActiveSessionsExceededException");
static const int ACTIVE_STATEMENTS_EXCEEDED_HASH = HashingUtils::HashString("ActiveStatementsExceededException");


AWSError<CoreErrors> GetErrorForName(const char* errorName)
{
  int hashCode = HashingUtils::HashString(errorName);

  if (hashCode == BATCH_EXECUTE_STATEMENT_HASH)
  {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(RedshiftDataAPIServiceErrors::BATCH_EXECUTE_STATEMENT), RetryableType::NOT_RETRYABLE);
  }
  else if (hashCode == EXECUTE_STATEMENT_HASH)
  {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(RedshiftDataAPIServiceErrors::EXECUTE_STATEMENT), RetryableType::NOT_RETRYABLE);
  }
  else if (hashCode == INTERNAL_SERVER_HASH)
  {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(RedshiftDataAPIServiceErrors::INTERNAL_SERVER), RetryableType::NOT_RETRYABLE);
  }
  else if (hashCode == DATABASE_CONNECTION_HASH)
  {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(RedshiftDataAPIServiceErrors::DATABASE_CONNECTION), RetryableType::NOT_RETRYABLE);
  }
  else if (hashCode == QUERY_TIMEOUT_HASH)
  {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(RedshiftDataAPIServiceErrors::QUERY_TIMEOUT), RetryableType::NOT_RETRYABLE);
  }
  else if (hashCode == ACTIVE_SESSIONS_EXCEEDED_HASH)
  {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(RedshiftDataAPIServiceErrors::ACTIVE_SESSIONS_EXCEEDED), RetryableType::NOT_RETRYABLE);
  }
  else if (hashCode == ACTIVE_STATEMENTS_EXCEEDED_HASH)
  {
    return AWSError<CoreErrors>(static_cast<CoreErrors>(RedshiftDataAPIServiceErrors::ACTIVE_STATEMENTS_EXCEEDED), RetryableType::NOT_RETRYABLE);
  }
  return AWSError<CoreErrors>(CoreErrors::UNKNOWN, false);
}

} // namespace RedshiftDataAPIServiceErrorMapper
} // namespace RedshiftDataAPIService
} // namespace Aws
