// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0 OR ISC

#include <gtest/gtest.h>
#include <openssl/base.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/mem.h>
#include <openssl/obj.h>
#include "../test/test_util.h"

#include <vector>
#include "../fipsmodule/evp/internal.h"
#include "../internal.h"
#include "internal.h"

#ifdef ENABLE_DILITHIUM

#include "../test/file_test.h"
#include "../test/test_util.h"
#include "ml_dsa.h"

// mldsa65kPublicKey is an example ML-DSA-65 public key
static const uint8_t mldsa65kPublicKey[] = {
0x9B, 0x77, 0xAB, 0x96, 0x9D, 0x65, 0xA2, 0xC1, 0x55, 0x65, 0x2, 0x9B, 0xA5,
0xD4, 0xE5, 0x93, 0xA1, 0xAC, 0xE7, 0x3E, 0x8C, 0x61, 0xB7, 0xCB, 0xA1, 0x3E,
0x74, 0x8A, 0xC9, 0xC0, 0xA0, 0x63, 0x4A, 0xF6, 0xF4, 0x1C, 0x72, 0x37, 0xB0,
0x31, 0x9E, 0xB7, 0x51, 0x55, 0xCF, 0x5B, 0x4E, 0x3, 0x46, 0x7C, 0x26, 0xBE,
0x84, 0x73, 0xD8, 0x50, 0xDF, 0x72, 0x87, 0xC0, 0x18, 0xED, 0xE7, 0xE4, 0x12,
0x4F, 0xCA, 0x4E, 0x1A, 0xFA, 0x76, 0x82, 0xD4, 0xA6, 0x3E, 0xDA, 0xEC, 0x74,
0x53, 0xFF, 0xDD, 0x69, 0x5C, 0x9F, 0xFD, 0x69, 0xA3, 0xED, 0x4F, 0xEB, 0xFB,
0xEF, 0xD2, 0x98, 0x8B, 0x45, 0x6, 0xBA, 0xD5, 0xF8, 0x9E, 0xA, 0x2D, 0xA2,
0xC7, 0x96, 0x4B, 0x79, 0xE9, 0xA9, 0xA6, 0x73, 0x69, 0xF8, 0x8C, 0x1, 0x69,
0xF2, 0x66, 0x5, 0x37, 0x31, 0x65, 0xA9, 0x9, 0x3E, 0xE, 0x73, 0x95, 0x67,
0xC9, 0x33, 0xA6, 0x57, 0xDF, 0xDD, 0xC0, 0x55, 0x1A, 0x89, 0x6F, 0xC8, 0x30,
0x71, 0x68, 0x3C, 0x2A, 0x7E, 0x61, 0x86, 0xAC, 0x70, 0x6A, 0x27, 0x31, 0x9B,
0x9A, 0xEC, 0x8F, 0x37, 0x2B, 0x71, 0x91, 0x91, 0x6C, 0x8B, 0x35, 0xED, 0xF1,
0x97, 0x87, 0x58, 0xD1, 0x4F, 0xF2, 0x6, 0x23, 0xE6, 0x1C, 0x44, 0x63, 0x2,
0x9E, 0x9, 0x76, 0x6C, 0x72, 0xBD, 0xD, 0xB3, 0xE2, 0x1D, 0x92, 0xAA, 0x8D,
0x7B, 0x78, 0xD8, 0xB3, 0xA7, 0x5A, 0xAB, 0xBF, 0x22, 0xBB, 0x30, 0x5B, 0xFB,
0xB4, 0x3C, 0x52, 0xD2, 0xA2, 0xED, 0x3B, 0x99, 0x43, 0xCB, 0x29, 0x66, 0x2A,
0xBD, 0x52, 0x1B, 0x1C, 0xB4, 0xE5, 0xE3, 0x6E, 0xFF, 0xAD, 0xEF, 0x8B, 0xE1,
0xF9, 0xB5, 0x5E, 0xCB, 0xF2, 0x8E, 0xCD, 0x53, 0x39, 0xBE, 0xBE, 0x61, 0x72,
0x86, 0x31, 0x65, 0xA0, 0xFC, 0xC1, 0xFC, 0x31, 0x79, 0x93, 0xDF, 0x76, 0x13,
0x71, 0xE4, 0x61, 0xF, 0x6B, 0x32, 0x78, 0xD2, 0x24, 0xB7, 0x8C, 0xE8, 0x84,
0xE3, 0xB8, 0xF6, 0x4, 0xF3, 0x30, 0xE9, 0x5B, 0xA5, 0xD8, 0x94, 0xA7, 0xA3,
0xF0, 0xE8, 0xAC, 0x70, 0x32, 0x42, 0xB5, 0x8, 0xEE, 0x2A, 0x77, 0xFA, 0x4,
0x49, 0xE9, 0x7A, 0xB7, 0xA, 0x95, 0x5, 0x86, 0x33, 0xA5, 0xE4, 0x5A, 0xC6,
0xE1, 0xE7, 0x48, 0xBD, 0xBA, 0x80, 0xE7, 0x21, 0x61, 0x45, 0x24, 0x5E, 0xA9,
0x7F, 0x2D, 0x75, 0xF, 0xE9, 0xEE, 0x79, 0x88, 0x64, 0xF3, 0xE7, 0xC, 0xA0,
0xEB, 0x93, 0x2C, 0x6B, 0xD3, 0x51, 0x12, 0xE7, 0x62, 0x8D, 0x71, 0x10, 0x6D,
0x5B, 0x3A, 0x27, 0xF4, 0xEA, 0x80, 0xFC, 0xCD, 0x58, 0x81, 0x43, 0xEB, 0xA0,
0x4E, 0xF5, 0xA1, 0x68, 0x67, 0x74, 0x7C, 0x14, 0x12, 0xA6, 0x78, 0xC2, 0x8,
0x58, 0x3F, 0x20, 0x96, 0x52, 0xD2, 0x61, 0xDA, 0xED, 0x5F, 0x7F, 0xAD, 0x40,
0x93, 0x21, 0xEB, 0xC4, 0x37, 0x5C, 0xD1, 0x72, 0xE6, 0x6, 0x37, 0xD9, 0xF6,
0x9, 0xD4, 0xC9, 0x6D, 0xED, 0x7, 0xF6, 0xD2, 0x15, 0x94, 0xFD, 0xF6, 0xC3,
0x9, 0x60, 0x6D, 0x6A, 0x23, 0x50, 0x8C, 0xDD, 0x61, 0xDD, 0x66, 0x81, 0xB0,
0xAC, 0x7C, 0xE7, 0x7F, 0xED, 0x3C, 0x2F, 0x19, 0xB5, 0xF9, 0xB7, 0x2E, 0x35,
0xF7, 0xF4, 0x98, 0xE, 0x6A, 0x9E, 0x6D, 0xAC, 0xF1, 0xF, 0x90, 0x25, 0xED,
0xC5, 0x94, 0x9E, 0x10, 0x29, 0x97, 0x47, 0x5, 0x3D, 0x3, 0x6F, 0x69, 0xAE,
0x84, 0x8, 0x9B, 0x33, 0xC, 0x1F, 0x26, 0x65, 0xC7, 0x86, 0x25, 0x10, 0x11,
0x97, 0x33, 0x3D, 0x98, 0x43, 0xB5, 0x7F, 0x9C, 0x19, 0x62, 0xE5, 0x46, 0x6D,
0x3B, 0xA2, 0xDC, 0xD4, 0x17, 0x85, 0x9A, 0xE8, 0x2C, 0xF3, 0x1, 0x5F, 0x39,
0xD1, 0xBC, 0x7, 0x8E, 0xAC, 0xC9, 0x28, 0xC, 0x7B, 0xD8, 0x2, 0xFE, 0x46,
0x12, 0xA8, 0xBD, 0xE, 0x6B, 0x23, 0x65, 0x5B, 0xAA, 0xFC, 0x32, 0x20, 0xF7,
0xCC, 0xC7, 0x6, 0x80, 0x9, 0xA, 0x95, 0xD9, 0x69, 0xED, 0x3C, 0x6C, 0xEB,
0x62, 0x28, 0xE6, 0x4E, 0xF4, 0xFA, 0x9B, 0x5C, 0x36, 0x7, 0xE0, 0x25, 0x20,
0xB8, 0xF4, 0x1F, 0x2E, 0x78, 0x21, 0xEE, 0xFA, 0x9E, 0x80, 0x14, 0xAD, 0xAD,
0x83, 0x39, 0x2E, 0xD0, 0xE9, 0x56, 0xE3, 0x88, 0xC, 0xC4, 0xD7, 0xBE, 0xB1,
0xE4, 0xD0, 0x42, 0xE6, 0xED, 0xDC, 0x44, 0x65, 0x51, 0x1F, 0x95, 0x9A, 0xAA,
0xBF, 0x83, 0x7B, 0xD7, 0x14, 0x23, 0x18, 0x81, 0x91, 0xA, 0x7, 0x97, 0x10,
0x6F, 0x3C, 0x16, 0xF2, 0xF0, 0x3E, 0xE1, 0x45, 0x40, 0xB0, 0x39, 0x98, 0x33,
0x55, 0xFF, 0x7E, 0x75, 0x31, 0xE0, 0x10, 0x16, 0x81, 0x36, 0x56, 0x86, 0x34,
0x1C, 0x61, 0x10, 0x25, 0xAE, 0x98, 0x6E, 0xBE, 0xC9, 0x47, 0xCD, 0x14, 0x1C,
0x52, 0x8C, 0x27, 0xEE, 0x28, 0xDA, 0x18, 0x96, 0x4D, 0x16, 0x6D, 0x17, 0x2E,
0x5B, 0x7E, 0x88, 0x70, 0xC8, 0x3D, 0x31, 0x34, 0xE5, 0xEA, 0x8, 0x40, 0x25,
0x7B, 0x3, 0x75, 0x47, 0xAD, 0x19, 0x2, 0x7E, 0xCC, 0xB6, 0x43, 0xD1, 0xC9,
0xB2, 0x95, 0x7F, 0x9F, 0x93, 0xC4, 0xD7, 0x33, 0x5A, 0x7E, 0xA4, 0x51, 0x58,
0xC5, 0xA7, 0x23, 0x25, 0xF8, 0xF4, 0xDE, 0xEF, 0x84, 0x72, 0xE, 0x8D, 0xE7,
0x9E, 0x1E, 0x40, 0xB3, 0xA6, 0x58, 0x34, 0x4E, 0xB8, 0x56, 0x6B, 0xA1, 0x50,
0x2B, 0x1C, 0xF9, 0xA6, 0x88, 0x21, 0x34, 0x79, 0x99, 0x5F, 0x24, 0xD6, 0x96,
0x67, 0xB5, 0x7E, 0x9C, 0xD2, 0xFB, 0x11, 0x40, 0xA6, 0xE6, 0x20, 0xD2, 0x8C,
0x38, 0x62, 0x9B, 0xC1, 0xD7, 0x57, 0x42, 0xE0, 0xD7, 0x34, 0xF3, 0x90, 0xF9,
0x60, 0xDD, 0xEA, 0x24, 0x67, 0x6A, 0xC0, 0xC7, 0xEF, 0xA7, 0x1B, 0xDC, 0xAD,
0x3D, 0xD, 0x17, 0x90, 0x66, 0x70, 0xB2, 0x98, 0x24, 0x1B, 0x58, 0x79, 0xAC,
0x3E, 0x61, 0x9C, 0x67, 0xB4, 0xEE, 0x9, 0x6, 0x20, 0xCE, 0x39, 0x3, 0x57,
0xD4, 0xB5, 0x44, 0x3C, 0x35, 0x80, 0xDD, 0xEF, 0xC3, 0xC5, 0xC4, 0x93, 0x79,
0xF8, 0x84, 0x60, 0x31, 0x27, 0xB7, 0xF8, 0xEB, 0x63, 0xE8, 0x75, 0x74, 0x31,
0x29, 0xF4, 0xE7, 0x6, 0x51, 0x74, 0x72, 0x71, 0x9D, 0xA1, 0x3F, 0x3C, 0x73,
0xCF, 0x7, 0xA9, 0x98, 0x23, 0x1F, 0x62, 0x9C, 0x9E, 0x27, 0xFD, 0x1E, 0xC8,
0x1C, 0xB9, 0xBD, 0x16, 0xB5, 0x4C, 0x1A, 0xC2, 0x8D, 0xCF, 0x4D, 0xB8, 0xC2,
0x4D, 0x94, 0xE6, 0x12, 0x6D, 0x14, 0xFA, 0x2B, 0xF4, 0x4A, 0x2B, 0xD9, 0x7D,
0xEF, 0xF8, 0x81, 0x2C, 0xF7, 0x7B, 0x98, 0x44, 0x12, 0x58, 0xD5, 0x82, 0xAA,
0xED, 0x49, 0x40, 0x87, 0xBA, 0x11, 0x29, 0x7E, 0xFD, 0x4, 0x67, 0x20, 0x5D,
0x2B, 0x79, 0x42, 0x7, 0x3, 0x5C, 0x36, 0xD7, 0xBE, 0x72, 0xCA, 0x13, 0xCF,
0x93, 0x2D, 0xD8, 0xA9, 0xEE, 0x6, 0xB, 0xCF, 0x5A, 0x46, 0x88, 0x57, 0x9E,
0x18, 0x92, 0x3B, 0x5F, 0x2F, 0x86, 0xCD, 0x3D, 0x49, 0xF6, 0xA3, 0x5, 0xE6,
0xE4, 0x68, 0xA4, 0x79, 0xA6, 0xEE, 0x85, 0xF4, 0x2B, 0xF6, 0x6E, 0x1B, 0x7A,
0xBD, 0x77, 0xEA, 0x6A, 0xC9, 0x31, 0x34, 0x8E, 0x5F, 0xC2, 0xF3, 0x87, 0x3D,
0x8F, 0xD7, 0xB0, 0x16, 0x28, 0x3F, 0x2C, 0x87, 0xA0, 0xA3, 0x56, 0xE8, 0x21,
0x83, 0x53, 0xCB, 0xE9, 0x1D, 0x28, 0x57, 0x93, 0xDB, 0x5B, 0xE9, 0xF0, 0x7B,
0x7F, 0xF4, 0x6A, 0x51, 0x48, 0xFC, 0xAB, 0xF5, 0x3B, 0x44, 0xA7, 0x5E, 0x67,
0x3A, 0x6B, 0x43, 0x9C, 0xD1, 0x3, 0xDF, 0xF8, 0xD5, 0x7F, 0x7B, 0x9, 0x62,
0xBF, 0x28, 0xBD, 0xC6, 0x3E, 0xC3, 0x6C, 0x91, 0x1, 0x45, 0x3F, 0xE2, 0x1F,
0xEF, 0x2A, 0x8F, 0xB2, 0x1B, 0x72, 0x35, 0x4D, 0x18, 0x6F, 0x4D, 0x57, 0xBF,
0x6A, 0x69, 0x2, 0x69, 0x4A, 0xE5, 0x5F, 0x74, 0xF7, 0x69, 0x5B, 0x89, 0x8,
0xCE, 0xCE, 0x15, 0x56, 0x3F, 0x21, 0x1A, 0xB8, 0xEC, 0x4D, 0xB0, 0x7E, 0xF,
0x89, 0xB0, 0x5C, 0x6D, 0xDB, 0x53, 0x9E, 0xA9, 0x27, 0x28, 0x52, 0xE5, 0x9E,
0x1F, 0xEF, 0x84, 0x1A, 0x9A, 0xAE, 0x86, 0x8B, 0x25, 0x3B, 0xC6, 0x3B, 0x8E,
0x9C, 0x32, 0xD9, 0x89, 0x3B, 0xA2, 0xCB, 0x59, 0x35, 0xC3, 0x71, 0xEE, 0x22,
0xC, 0x61, 0xEA, 0x59, 0x33, 0x25, 0x39, 0xAF, 0xF0, 0x12, 0x81, 0x55, 0x4A,
0x9D, 0xC, 0x3E, 0x5E, 0x34, 0x9F, 0xA7, 0xD8, 0xC5, 0xB5, 0xA, 0xC3, 0xA2,
0x0, 0x3F, 0x59, 0x3D, 0x7, 0x5F, 0x2B, 0xC1, 0x6F, 0x6A, 0xE3, 0x94, 0x90,
0xAF, 0x81, 0x11, 0x82, 0x89, 0xF4, 0x9D, 0x8B, 0x5, 0xE2, 0x7C, 0x22, 0x2,
0xEC, 0x0, 0x38, 0x39, 0xED, 0x4, 0xB2, 0xC9, 0xD8, 0xA1, 0x1B, 0xED, 0xB9,
0xE1, 0x62, 0x82, 0xC4, 0xCC, 0xA0, 0x61, 0xEE, 0x7A, 0x17, 0xA0, 0x99, 0xAC,
0xAC, 0x85, 0xA7, 0x5F, 0xC9, 0xC3, 0xC5, 0x63, 0x8F, 0x5A, 0xE7, 0x41, 0xAC,
0xB7, 0x89, 0x13, 0x38, 0xD8, 0x58, 0xBF, 0x71, 0xA5, 0x4F, 0x9D, 0x4C, 0x72,
0x57, 0x88, 0x2E, 0xAB, 0xD4, 0x74, 0xDE, 0x46, 0x9F, 0xF4, 0xBA, 0xB1, 0x55,
0x6A, 0x18, 0xF4, 0x87, 0xB9, 0x24, 0xA7, 0xD9, 0xF4, 0x9A, 0x3C, 0xEF, 0xF4,
0xA2, 0x2D, 0xF, 0xC9, 0xE4, 0x45, 0xC2, 0xC9, 0x6F, 0x2D, 0xB6, 0xDA, 0xE6,
0x89, 0x38, 0x80, 0x2A, 0x89, 0xE2, 0xF5, 0x3D, 0x77, 0x5E, 0x61, 0x6E, 0x9C,
0xF9, 0x87, 0x89, 0xD4, 0x70, 0x23, 0x79, 0x93, 0xDA, 0xCE, 0x62, 0x89, 0xEB,
0x13, 0x77, 0xB0, 0x49, 0xB2, 0xF9, 0xFC, 0x84, 0xD3, 0x6, 0xD2, 0x8D, 0x5A,
0x94, 0x64, 0xC1, 0xA8, 0x9A, 0x60, 0x57, 0x8A, 0x8F, 0x62, 0x4A, 0x78, 0x12,
0x6B, 0x87, 0x6F, 0x6D, 0xC8, 0x32, 0xF3, 0xC6, 0x8D, 0xDB, 0x3A, 0x67, 0x95,
0xCD, 0xAF, 0x48, 0x28, 0x79, 0xC2, 0xB6, 0xDB, 0xD8, 0xFE, 0x82, 0x15, 0xE6,
0xE4, 0xEC, 0x79, 0xE2, 0xB4, 0x21, 0x5C, 0x30, 0x45, 0xD7, 0x3B, 0xA0, 0x1A,
0x3B, 0xAA, 0x3D, 0x6C, 0x1C, 0xC3, 0x1E, 0xDE, 0x4D, 0x75, 0x1D, 0x9A, 0x96,
0x51, 0xF9, 0x4F, 0x10, 0x28, 0x7E, 0x88, 0xEE, 0x3B, 0x93, 0x4A, 0xB, 0x9,
0x44, 0x9C, 0x20, 0x34, 0xF6, 0xEE, 0x6F, 0x26, 0xB9, 0x4C, 0x76, 0xCC, 0xE1,
0x6F, 0x9, 0x91, 0xAF, 0x48, 0x8C, 0xC4, 0x31, 0xA2, 0xF9, 0x44, 0x77, 0x19,
0xA7, 0x0, 0x33, 0x77, 0x31, 0xF2, 0xF5, 0xF7, 0x30, 0xDF, 0xAB, 0xFE, 0x7E,
0xE6, 0x83, 0xE1, 0xC9, 0x2A, 0xC8, 0xE0, 0xA6, 0xAC, 0x5A, 0x28, 0x7F, 0xC4,
0xB, 0xEB, 0x55, 0xD9, 0x5D, 0xBD, 0xB5, 0xD2, 0xF6, 0xB4, 0xA9, 0x76, 0x2B,
0x35, 0x10, 0x36, 0x3B, 0xCC, 0x61, 0x6C, 0x79, 0xCE, 0xC3, 0x9A, 0x2, 0x9A,
0x0, 0xBA, 0x43, 0x20, 0x3F, 0x26, 0x36, 0x66, 0x7, 0x11, 0x68, 0x51, 0x47,
0xBE, 0x78, 0xED, 0x4A, 0xFA, 0xBC, 0xDA, 0xCD, 0xFD, 0x2, 0xDB, 0xD1, 0x8B,
0xE0, 0xBD, 0x13, 0xFE, 0xED, 0x26, 0x77, 0xE4, 0x83, 0xAE, 0xB7, 0xAB, 0xFD,
0x2A, 0x5E, 0xA3, 0x28, 0xFD, 0x90, 0x40, 0x3D, 0x34, 0xF7, 0xF8, 0x35, 0x80,
0xF6, 0x6F, 0xA0, 0xE9, 0xCD, 0x9A, 0x54, 0x6F, 0x41, 0xA5, 0xC7, 0xED, 0xEA,
0xDC, 0x52, 0x23, 0xF1, 0x96, 0x19, 0x8E, 0x2B, 0x94, 0x3F, 0xD9, 0x27, 0x60,
0x1E, 0x27, 0xC1, 0x39, 0x68, 0x78, 0x7B, 0x47, 0x8F, 0xCC, 0xCD, 0xBE, 0xE4,
0xBD, 0xB, 0x73, 0x3, 0xFB, 0xFE, 0xC0, 0x50, 0x38, 0x70, 0xDF, 0x81, 0x5D,
0x22, 0x4C, 0x5B, 0xCB, 0x27, 0x5D, 0xD2, 0x94, 0x64, 0xA, 0x88, 0x67, 0x31,
0xE9, 0x8, 0xF0, 0x88, 0x20, 0xF2, 0x86, 0xCA, 0xBD, 0x18, 0x5F, 0x34, 0xD0,
0x96, 0xD, 0x4A, 0x62, 0x4D, 0xBE, 0xE8, 0xA6, 0x4, 0xA6, 0x69, 0xCE, 0xCD,
0xE9, 0x5A, 0x1D, 0xD2, 0xF8, 0xCF, 0x19, 0x6, 0x17, 0x5, 0x82, 0x6B, 0x60,
0x3E, 0x5E, 0x6B, 0x1D, 0x1E, 0x13, 0x51, 0x5D, 0xFE, 0x95, 0x38, 0x33, 0x62,
0x9B, 0xBF, 0xD5, 0x3E, 0x3B, 0x8B, 0xD2, 0x6F, 0x24, 0x6D, 0x24, 0xC9, 0xD,
0x2D, 0x52, 0xBF, 0xDA, 0xCE, 0x5E, 0xFE, 0x9D, 0xB8, 0x5D, 0x61, 0x57, 0xBC,
0x8C, 0x7A, 0x17, 0x75, 0x80, 0xEE, 0x52, 0x2F, 0xF5, 0x25, 0x48, 0x3A, 0x9E,
0x27, 0xF4, 0xEB, 0xE1, 0x1, 0xE4, 0xA7, 0x48, 0x93, 0xAA, 0x92, 0x68, 0xC0,
0x3B, 0x1A, 0x5A, 0xC5, 0x6D, 0xD0, 0x91, 0xB9, 0x8D, 0x44, 0xD4, 0xE1, 0x9C,
0x74, 0xEA, 0x14, 0xFA, 0xF6, 0x1E, 0x1, 0xC0, 0x89, 0x24, 0x90, 0x71, 0xAF,
0xF5, 0x2D, 0x6C, 0x35, 0x13, 0xA6, 0x73, 0x14, 0xAC, 0xE5, 0xAE, 0x88, 0x2F,
0x9D, 0x77, 0x3B, 0x8F, 0x61, 0xB1, 0x47, 0x66, 0x72, 0x14, 0x91, 0x40, 0xD7,
0x50, 0xDC, 0xEA, 0xFF, 0x49, 0x9E, 0x17, 0x75, 0x25, 0x49, 0x7C, 0x57, 0x41,
0xA7, 0x8C, 0x4D, 0x3B, 0x94, 0x9D, 0x65, 0x83, 0x62, 0x6F, 0x16, 0xBF, 0xC,
0x87, 0x3, 0x61, 0xB4, 0x3B, 0x60, 0x6D, 0x7, 0x56, 0xB8, 0x1F, 0x89, 0xAD,
0x0, 0x25, 0x10, 0x4A, 0x34, 0x4C, 0x9A, 0x26, 0xDA, 0x6, 0x25, 0x9C, 0x91,
0xA6, 0xA5, 0xAD, 0x4D, 0x6E, 0xE9, 0x2F, 0x18, 0xC4, 0x1D, 0x9, 0xE1, 0xAA,
0x66, 0x1, 0x31, 0x6D, 0x12, 0x30, 0xED, 0x97, 0x3F, 0x67, 0xCE, 0x4E, 0x26,
0xB, 0xF5, 0x5E, 0x81, 0xA7, 0x1F, 0x83, 0x68, 0x91, 0xC3, 0xD0, 0x4C, 0x2E,
0xD4, 0xDE, 0xEF, 0x34, 0xF9, 0x61, 0x83, 0x6F, 0xD6, 0x6E, 0x40, 0x87, 0x48,
0x7E, 0xCF, 0x56, 0x42, 0x21, 0xBA, 0x40, 0x64, 0x17, 0xFA, 0x97, 0xFF, 0x8D,
0xC8, 0x32, 0xFA, 0xB7, 0x45, 0xB0, 0xEC, 0xBD, 0xE, 0x51, 0x63, 0x90, 0x5,
0x68, 0x7A, 0x45, 0x86, 0x68, 0x2A, 0xE, 0x81, 0x5F, 0xDD, 0x12, 0xAD, 0x48,
0xF6, 0x87, 0x2E, 0x8D, 0xF6, 0x86, 0xC3, 0x6D, 0x69, 0xD5, 0x4E, 0x52, 0x8A,
0x8E, 0xE8, 0x1, 0x56, 0x11, 0xCC, 0x2E, 0x3F, 0xB5, 0x46, 0x1D, 0xF6, 0x6E,
0x4A, 0xEE, 0x1C, 0x60, 0x15, 0x85, 0xF6, 0x40, 0xFD, 0x56, 0xDC, 0x10, 0x1,
0xC3, 0xBD, 0xAE, 0x5A, 0x13, 0x1F, 0x15, 0x16, 0x10, 0x92, 0xC5, 0x2, 0xC2,
0x81, 0xB5, 0x6A, 0x4D, 0x37, 0x29, 0x40, 0x8B, 0xAA, 0x5F, 0xC9, 0x4C, 0x26,
0x7B, 0x2C, 0x21, 0x9E, 0xE2, 0xF2, 0x5A, 0x20, 0x88, 0x3F, 0x40, 0x30, 0xC5,
0x64, 0xE };

// mldsa65kPublicKeySPKI is the above example ML-DSA-65 public key encoded
static const uint8_t mldsa65kPublicKeySPKI[] = {
0x30, 0x82, 0x7, 0xB2, 0x30, 0xB, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65,
0x3, 0x4, 0x3, 0x12, 0x3, 0x82, 0x7, 0xA1, 0x0, 0x9B, 0x77, 0xAB, 0x96,
0x9D, 0x65, 0xA2, 0xC1, 0x55, 0x65, 0x2, 0x9B, 0xA5, 0xD4, 0xE5, 0x93, 0xA1,
0xAC, 0xE7, 0x3E, 0x8C, 0x61, 0xB7, 0xCB, 0xA1, 0x3E, 0x74, 0x8A, 0xC9, 0xC0,
0xA0, 0x63, 0x4A, 0xF6, 0xF4, 0x1C, 0x72, 0x37, 0xB0, 0x31, 0x9E, 0xB7, 0x51,
0x55, 0xCF, 0x5B, 0x4E, 0x3, 0x46, 0x7C, 0x26, 0xBE, 0x84, 0x73, 0xD8, 0x50,
0xDF, 0x72, 0x87, 0xC0, 0x18, 0xED, 0xE7, 0xE4, 0x12, 0x4F, 0xCA, 0x4E, 0x1A,
0xFA, 0x76, 0x82, 0xD4, 0xA6, 0x3E, 0xDA, 0xEC, 0x74, 0x53, 0xFF, 0xDD, 0x69,
0x5C, 0x9F, 0xFD, 0x69, 0xA3, 0xED, 0x4F, 0xEB, 0xFB, 0xEF, 0xD2, 0x98, 0x8B,
0x45, 0x6, 0xBA, 0xD5, 0xF8, 0x9E, 0xA, 0x2D, 0xA2, 0xC7, 0x96, 0x4B, 0x79,
0xE9, 0xA9, 0xA6, 0x73, 0x69, 0xF8, 0x8C, 0x1, 0x69, 0xF2, 0x66, 0x5, 0x37,
0x31, 0x65, 0xA9, 0x9, 0x3E, 0xE, 0x73, 0x95, 0x67, 0xC9, 0x33, 0xA6, 0x57,
0xDF, 0xDD, 0xC0, 0x55, 0x1A, 0x89, 0x6F, 0xC8, 0x30, 0x71, 0x68, 0x3C, 0x2A,
0x7E, 0x61, 0x86, 0xAC, 0x70, 0x6A, 0x27, 0x31, 0x9B, 0x9A, 0xEC, 0x8F, 0x37,
0x2B, 0x71, 0x91, 0x91, 0x6C, 0x8B, 0x35, 0xED, 0xF1, 0x97, 0x87, 0x58, 0xD1,
0x4F, 0xF2, 0x6, 0x23, 0xE6, 0x1C, 0x44, 0x63, 0x2, 0x9E, 0x9, 0x76, 0x6C,
0x72, 0xBD, 0xD, 0xB3, 0xE2, 0x1D, 0x92, 0xAA, 0x8D, 0x7B, 0x78, 0xD8, 0xB3,
0xA7, 0x5A, 0xAB, 0xBF, 0x22, 0xBB, 0x30, 0x5B, 0xFB, 0xB4, 0x3C, 0x52, 0xD2,
0xA2, 0xED, 0x3B, 0x99, 0x43, 0xCB, 0x29, 0x66, 0x2A, 0xBD, 0x52, 0x1B, 0x1C,
0xB4, 0xE5, 0xE3, 0x6E, 0xFF, 0xAD, 0xEF, 0x8B, 0xE1, 0xF9, 0xB5, 0x5E, 0xCB,
0xF2, 0x8E, 0xCD, 0x53, 0x39, 0xBE, 0xBE, 0x61, 0x72, 0x86, 0x31, 0x65, 0xA0,
0xFC, 0xC1, 0xFC, 0x31, 0x79, 0x93, 0xDF, 0x76, 0x13, 0x71, 0xE4, 0x61, 0xF,
0x6B, 0x32, 0x78, 0xD2, 0x24, 0xB7, 0x8C, 0xE8, 0x84, 0xE3, 0xB8, 0xF6, 0x4,
0xF3, 0x30, 0xE9, 0x5B, 0xA5, 0xD8, 0x94, 0xA7, 0xA3, 0xF0, 0xE8, 0xAC, 0x70,
0x32, 0x42, 0xB5, 0x8, 0xEE, 0x2A, 0x77, 0xFA, 0x4, 0x49, 0xE9, 0x7A, 0xB7,
0xA, 0x95, 0x5, 0x86, 0x33, 0xA5, 0xE4, 0x5A, 0xC6, 0xE1, 0xE7, 0x48, 0xBD,
0xBA, 0x80, 0xE7, 0x21, 0x61, 0x45, 0x24, 0x5E, 0xA9, 0x7F, 0x2D, 0x75, 0xF,
0xE9, 0xEE, 0x79, 0x88, 0x64, 0xF3, 0xE7, 0xC, 0xA0, 0xEB, 0x93, 0x2C, 0x6B,
0xD3, 0x51, 0x12, 0xE7, 0x62, 0x8D, 0x71, 0x10, 0x6D, 0x5B, 0x3A, 0x27, 0xF4,
0xEA, 0x80, 0xFC, 0xCD, 0x58, 0x81, 0x43, 0xEB, 0xA0, 0x4E, 0xF5, 0xA1, 0x68,
0x67, 0x74, 0x7C, 0x14, 0x12, 0xA6, 0x78, 0xC2, 0x8, 0x58, 0x3F, 0x20, 0x96,
0x52, 0xD2, 0x61, 0xDA, 0xED, 0x5F, 0x7F, 0xAD, 0x40, 0x93, 0x21, 0xEB, 0xC4,
0x37, 0x5C, 0xD1, 0x72, 0xE6, 0x6, 0x37, 0xD9, 0xF6, 0x9, 0xD4, 0xC9, 0x6D,
0xED, 0x7, 0xF6, 0xD2, 0x15, 0x94, 0xFD, 0xF6, 0xC3, 0x9, 0x60, 0x6D, 0x6A,
0x23, 0x50, 0x8C, 0xDD, 0x61, 0xDD, 0x66, 0x81, 0xB0, 0xAC, 0x7C, 0xE7, 0x7F,
0xED, 0x3C, 0x2F, 0x19, 0xB5, 0xF9, 0xB7, 0x2E, 0x35, 0xF7, 0xF4, 0x98, 0xE,
0x6A, 0x9E, 0x6D, 0xAC, 0xF1, 0xF, 0x90, 0x25, 0xED, 0xC5, 0x94, 0x9E, 0x10,
0x29, 0x97, 0x47, 0x5, 0x3D, 0x3, 0x6F, 0x69, 0xAE, 0x84, 0x8, 0x9B, 0x33,
0xC, 0x1F, 0x26, 0x65, 0xC7, 0x86, 0x25, 0x10, 0x11, 0x97, 0x33, 0x3D, 0x98,
0x43, 0xB5, 0x7F, 0x9C, 0x19, 0x62, 0xE5, 0x46, 0x6D, 0x3B, 0xA2, 0xDC, 0xD4,
0x17, 0x85, 0x9A, 0xE8, 0x2C, 0xF3, 0x1, 0x5F, 0x39, 0xD1, 0xBC, 0x7, 0x8E,
0xAC, 0xC9, 0x28, 0xC, 0x7B, 0xD8, 0x2, 0xFE, 0x46, 0x12, 0xA8, 0xBD, 0xE,
0x6B, 0x23, 0x65, 0x5B, 0xAA, 0xFC, 0x32, 0x20, 0xF7, 0xCC, 0xC7, 0x6, 0x80,
0x9, 0xA, 0x95, 0xD9, 0x69, 0xED, 0x3C, 0x6C, 0xEB, 0x62, 0x28, 0xE6, 0x4E,
0xF4, 0xFA, 0x9B, 0x5C, 0x36, 0x7, 0xE0, 0x25, 0x20, 0xB8, 0xF4, 0x1F, 0x2E,
0x78, 0x21, 0xEE, 0xFA, 0x9E, 0x80, 0x14, 0xAD, 0xAD, 0x83, 0x39, 0x2E, 0xD0,
0xE9, 0x56, 0xE3, 0x88, 0xC, 0xC4, 0xD7, 0xBE, 0xB1, 0xE4, 0xD0, 0x42, 0xE6,
0xED, 0xDC, 0x44, 0x65, 0x51, 0x1F, 0x95, 0x9A, 0xAA, 0xBF, 0x83, 0x7B, 0xD7,
0x14, 0x23, 0x18, 0x81, 0x91, 0xA, 0x7, 0x97, 0x10, 0x6F, 0x3C, 0x16, 0xF2,
0xF0, 0x3E, 0xE1, 0x45, 0x40, 0xB0, 0x39, 0x98, 0x33, 0x55, 0xFF, 0x7E, 0x75,
0x31, 0xE0, 0x10, 0x16, 0x81, 0x36, 0x56, 0x86, 0x34, 0x1C, 0x61, 0x10, 0x25,
0xAE, 0x98, 0x6E, 0xBE, 0xC9, 0x47, 0xCD, 0x14, 0x1C, 0x52, 0x8C, 0x27, 0xEE,
0x28, 0xDA, 0x18, 0x96, 0x4D, 0x16, 0x6D, 0x17, 0x2E, 0x5B, 0x7E, 0x88, 0x70,
0xC8, 0x3D, 0x31, 0x34, 0xE5, 0xEA, 0x8, 0x40, 0x25, 0x7B, 0x3, 0x75, 0x47,
0xAD, 0x19, 0x2, 0x7E, 0xCC, 0xB6, 0x43, 0xD1, 0xC9, 0xB2, 0x95, 0x7F, 0x9F,
0x93, 0xC4, 0xD7, 0x33, 0x5A, 0x7E, 0xA4, 0x51, 0x58, 0xC5, 0xA7, 0x23, 0x25,
0xF8, 0xF4, 0xDE, 0xEF, 0x84, 0x72, 0xE, 0x8D, 0xE7, 0x9E, 0x1E, 0x40, 0xB3,
0xA6, 0x58, 0x34, 0x4E, 0xB8, 0x56, 0x6B, 0xA1, 0x50, 0x2B, 0x1C, 0xF9, 0xA6,
0x88, 0x21, 0x34, 0x79, 0x99, 0x5F, 0x24, 0xD6, 0x96, 0x67, 0xB5, 0x7E, 0x9C,
0xD2, 0xFB, 0x11, 0x40, 0xA6, 0xE6, 0x20, 0xD2, 0x8C, 0x38, 0x62, 0x9B, 0xC1,
0xD7, 0x57, 0x42, 0xE0, 0xD7, 0x34, 0xF3, 0x90, 0xF9, 0x60, 0xDD, 0xEA, 0x24,
0x67, 0x6A, 0xC0, 0xC7, 0xEF, 0xA7, 0x1B, 0xDC, 0xAD, 0x3D, 0xD, 0x17, 0x90,
0x66, 0x70, 0xB2, 0x98, 0x24, 0x1B, 0x58, 0x79, 0xAC, 0x3E, 0x61, 0x9C, 0x67,
0xB4, 0xEE, 0x9, 0x6, 0x20, 0xCE, 0x39, 0x3, 0x57, 0xD4, 0xB5, 0x44, 0x3C,
0x35, 0x80, 0xDD, 0xEF, 0xC3, 0xC5, 0xC4, 0x93, 0x79, 0xF8, 0x84, 0x60, 0x31,
0x27, 0xB7, 0xF8, 0xEB, 0x63, 0xE8, 0x75, 0x74, 0x31, 0x29, 0xF4, 0xE7, 0x6,
0x51, 0x74, 0x72, 0x71, 0x9D, 0xA1, 0x3F, 0x3C, 0x73, 0xCF, 0x7, 0xA9, 0x98,
0x23, 0x1F, 0x62, 0x9C, 0x9E, 0x27, 0xFD, 0x1E, 0xC8, 0x1C, 0xB9, 0xBD, 0x16,
0xB5, 0x4C, 0x1A, 0xC2, 0x8D, 0xCF, 0x4D, 0xB8, 0xC2, 0x4D, 0x94, 0xE6, 0x12,
0x6D, 0x14, 0xFA, 0x2B, 0xF4, 0x4A, 0x2B, 0xD9, 0x7D, 0xEF, 0xF8, 0x81, 0x2C,
0xF7, 0x7B, 0x98, 0x44, 0x12, 0x58, 0xD5, 0x82, 0xAA, 0xED, 0x49, 0x40, 0x87,
0xBA, 0x11, 0x29, 0x7E, 0xFD, 0x4, 0x67, 0x20, 0x5D, 0x2B, 0x79, 0x42, 0x7,
0x3, 0x5C, 0x36, 0xD7, 0xBE, 0x72, 0xCA, 0x13, 0xCF, 0x93, 0x2D, 0xD8, 0xA9,
0xEE, 0x6, 0xB, 0xCF, 0x5A, 0x46, 0x88, 0x57, 0x9E, 0x18, 0x92, 0x3B, 0x5F,
0x2F, 0x86, 0xCD, 0x3D, 0x49, 0xF6, 0xA3, 0x5, 0xE6, 0xE4, 0x68, 0xA4, 0x79,
0xA6, 0xEE, 0x85, 0xF4, 0x2B, 0xF6, 0x6E, 0x1B, 0x7A, 0xBD, 0x77, 0xEA, 0x6A,
0xC9, 0x31, 0x34, 0x8E, 0x5F, 0xC2, 0xF3, 0x87, 0x3D, 0x8F, 0xD7, 0xB0, 0x16,
0x28, 0x3F, 0x2C, 0x87, 0xA0, 0xA3, 0x56, 0xE8, 0x21, 0x83, 0x53, 0xCB, 0xE9,
0x1D, 0x28, 0x57, 0x93, 0xDB, 0x5B, 0xE9, 0xF0, 0x7B, 0x7F, 0xF4, 0x6A, 0x51,
0x48, 0xFC, 0xAB, 0xF5, 0x3B, 0x44, 0xA7, 0x5E, 0x67, 0x3A, 0x6B, 0x43, 0x9C,
0xD1, 0x3, 0xDF, 0xF8, 0xD5, 0x7F, 0x7B, 0x9, 0x62, 0xBF, 0x28, 0xBD, 0xC6,
0x3E, 0xC3, 0x6C, 0x91, 0x1, 0x45, 0x3F, 0xE2, 0x1F, 0xEF, 0x2A, 0x8F, 0xB2,
0x1B, 0x72, 0x35, 0x4D, 0x18, 0x6F, 0x4D, 0x57, 0xBF, 0x6A, 0x69, 0x2, 0x69,
0x4A, 0xE5, 0x5F, 0x74, 0xF7, 0x69, 0x5B, 0x89, 0x8, 0xCE, 0xCE, 0x15, 0x56,
0x3F, 0x21, 0x1A, 0xB8, 0xEC, 0x4D, 0xB0, 0x7E, 0xF, 0x89, 0xB0, 0x5C, 0x6D,
0xDB, 0x53, 0x9E, 0xA9, 0x27, 0x28, 0x52, 0xE5, 0x9E, 0x1F, 0xEF, 0x84, 0x1A,
0x9A, 0xAE, 0x86, 0x8B, 0x25, 0x3B, 0xC6, 0x3B, 0x8E, 0x9C, 0x32, 0xD9, 0x89,
0x3B, 0xA2, 0xCB, 0x59, 0x35, 0xC3, 0x71, 0xEE, 0x22, 0xC, 0x61, 0xEA, 0x59,
0x33, 0x25, 0x39, 0xAF, 0xF0, 0x12, 0x81, 0x55, 0x4A, 0x9D, 0xC, 0x3E, 0x5E,
0x34, 0x9F, 0xA7, 0xD8, 0xC5, 0xB5, 0xA, 0xC3, 0xA2, 0x0, 0x3F, 0x59, 0x3D,
0x7, 0x5F, 0x2B, 0xC1, 0x6F, 0x6A, 0xE3, 0x94, 0x90, 0xAF, 0x81, 0x11, 0x82,
0x89, 0xF4, 0x9D, 0x8B, 0x5, 0xE2, 0x7C, 0x22, 0x2, 0xEC, 0x0, 0x38, 0x39,
0xED, 0x4, 0xB2, 0xC9, 0xD8, 0xA1, 0x1B, 0xED, 0xB9, 0xE1, 0x62, 0x82, 0xC4,
0xCC, 0xA0, 0x61, 0xEE, 0x7A, 0x17, 0xA0, 0x99, 0xAC, 0xAC, 0x85, 0xA7, 0x5F,
0xC9, 0xC3, 0xC5, 0x63, 0x8F, 0x5A, 0xE7, 0x41, 0xAC, 0xB7, 0x89, 0x13, 0x38,
0xD8, 0x58, 0xBF, 0x71, 0xA5, 0x4F, 0x9D, 0x4C, 0x72, 0x57, 0x88, 0x2E, 0xAB,
0xD4, 0x74, 0xDE, 0x46, 0x9F, 0xF4, 0xBA, 0xB1, 0x55, 0x6A, 0x18, 0xF4, 0x87,
0xB9, 0x24, 0xA7, 0xD9, 0xF4, 0x9A, 0x3C, 0xEF, 0xF4, 0xA2, 0x2D, 0xF, 0xC9,
0xE4, 0x45, 0xC2, 0xC9, 0x6F, 0x2D, 0xB6, 0xDA, 0xE6, 0x89, 0x38, 0x80, 0x2A,
0x89, 0xE2, 0xF5, 0x3D, 0x77, 0x5E, 0x61, 0x6E, 0x9C, 0xF9, 0x87, 0x89, 0xD4,
0x70, 0x23, 0x79, 0x93, 0xDA, 0xCE, 0x62, 0x89, 0xEB, 0x13, 0x77, 0xB0, 0x49,
0xB2, 0xF9, 0xFC, 0x84, 0xD3, 0x6, 0xD2, 0x8D, 0x5A, 0x94, 0x64, 0xC1, 0xA8,
0x9A, 0x60, 0x57, 0x8A, 0x8F, 0x62, 0x4A, 0x78, 0x12, 0x6B, 0x87, 0x6F, 0x6D,
0xC8, 0x32, 0xF3, 0xC6, 0x8D, 0xDB, 0x3A, 0x67, 0x95, 0xCD, 0xAF, 0x48, 0x28,
0x79, 0xC2, 0xB6, 0xDB, 0xD8, 0xFE, 0x82, 0x15, 0xE6, 0xE4, 0xEC, 0x79, 0xE2,
0xB4, 0x21, 0x5C, 0x30, 0x45, 0xD7, 0x3B, 0xA0, 0x1A, 0x3B, 0xAA, 0x3D, 0x6C,
0x1C, 0xC3, 0x1E, 0xDE, 0x4D, 0x75, 0x1D, 0x9A, 0x96, 0x51, 0xF9, 0x4F, 0x10,
0x28, 0x7E, 0x88, 0xEE, 0x3B, 0x93, 0x4A, 0xB, 0x9, 0x44, 0x9C, 0x20, 0x34,
0xF6, 0xEE, 0x6F, 0x26, 0xB9, 0x4C, 0x76, 0xCC, 0xE1, 0x6F, 0x9, 0x91, 0xAF,
0x48, 0x8C, 0xC4, 0x31, 0xA2, 0xF9, 0x44, 0x77, 0x19, 0xA7, 0x0, 0x33, 0x77,
0x31, 0xF2, 0xF5, 0xF7, 0x30, 0xDF, 0xAB, 0xFE, 0x7E, 0xE6, 0x83, 0xE1, 0xC9,
0x2A, 0xC8, 0xE0, 0xA6, 0xAC, 0x5A, 0x28, 0x7F, 0xC4, 0xB, 0xEB, 0x55, 0xD9,
0x5D, 0xBD, 0xB5, 0xD2, 0xF6, 0xB4, 0xA9, 0x76, 0x2B, 0x35, 0x10, 0x36, 0x3B,
0xCC, 0x61, 0x6C, 0x79, 0xCE, 0xC3, 0x9A, 0x2, 0x9A, 0x0, 0xBA, 0x43, 0x20,
0x3F, 0x26, 0x36, 0x66, 0x7, 0x11, 0x68, 0x51, 0x47, 0xBE, 0x78, 0xED, 0x4A,
0xFA, 0xBC, 0xDA, 0xCD, 0xFD, 0x2, 0xDB, 0xD1, 0x8B, 0xE0, 0xBD, 0x13, 0xFE,
0xED, 0x26, 0x77, 0xE4, 0x83, 0xAE, 0xB7, 0xAB, 0xFD, 0x2A, 0x5E, 0xA3, 0x28,
0xFD, 0x90, 0x40, 0x3D, 0x34, 0xF7, 0xF8, 0x35, 0x80, 0xF6, 0x6F, 0xA0, 0xE9,
0xCD, 0x9A, 0x54, 0x6F, 0x41, 0xA5, 0xC7, 0xED, 0xEA, 0xDC, 0x52, 0x23, 0xF1,
0x96, 0x19, 0x8E, 0x2B, 0x94, 0x3F, 0xD9, 0x27, 0x60, 0x1E, 0x27, 0xC1, 0x39,
0x68, 0x78, 0x7B, 0x47, 0x8F, 0xCC, 0xCD, 0xBE, 0xE4, 0xBD, 0xB, 0x73, 0x3,
0xFB, 0xFE, 0xC0, 0x50, 0x38, 0x70, 0xDF, 0x81, 0x5D, 0x22, 0x4C, 0x5B, 0xCB,
0x27, 0x5D, 0xD2, 0x94, 0x64, 0xA, 0x88, 0x67, 0x31, 0xE9, 0x8, 0xF0, 0x88,
0x20, 0xF2, 0x86, 0xCA, 0xBD, 0x18, 0x5F, 0x34, 0xD0, 0x96, 0xD, 0x4A, 0x62,
0x4D, 0xBE, 0xE8, 0xA6, 0x4, 0xA6, 0x69, 0xCE, 0xCD, 0xE9, 0x5A, 0x1D, 0xD2,
0xF8, 0xCF, 0x19, 0x6, 0x17, 0x5, 0x82, 0x6B, 0x60, 0x3E, 0x5E, 0x6B, 0x1D,
0x1E, 0x13, 0x51, 0x5D, 0xFE, 0x95, 0x38, 0x33, 0x62, 0x9B, 0xBF, 0xD5, 0x3E,
0x3B, 0x8B, 0xD2, 0x6F, 0x24, 0x6D, 0x24, 0xC9, 0xD, 0x2D, 0x52, 0xBF, 0xDA,
0xCE, 0x5E, 0xFE, 0x9D, 0xB8, 0x5D, 0x61, 0x57, 0xBC, 0x8C, 0x7A, 0x17, 0x75,
0x80, 0xEE, 0x52, 0x2F, 0xF5, 0x25, 0x48, 0x3A, 0x9E, 0x27, 0xF4, 0xEB, 0xE1,
0x1, 0xE4, 0xA7, 0x48, 0x93, 0xAA, 0x92, 0x68, 0xC0, 0x3B, 0x1A, 0x5A, 0xC5,
0x6D, 0xD0, 0x91, 0xB9, 0x8D, 0x44, 0xD4, 0xE1, 0x9C, 0x74, 0xEA, 0x14, 0xFA,
0xF6, 0x1E, 0x1, 0xC0, 0x89, 0x24, 0x90, 0x71, 0xAF, 0xF5, 0x2D, 0x6C, 0x35,
0x13, 0xA6, 0x73, 0x14, 0xAC, 0xE5, 0xAE, 0x88, 0x2F, 0x9D, 0x77, 0x3B, 0x8F,
0x61, 0xB1, 0x47, 0x66, 0x72, 0x14, 0x91, 0x40, 0xD7, 0x50, 0xDC, 0xEA, 0xFF,
0x49, 0x9E, 0x17, 0x75, 0x25, 0x49, 0x7C, 0x57, 0x41, 0xA7, 0x8C, 0x4D, 0x3B,
0x94, 0x9D, 0x65, 0x83, 0x62, 0x6F, 0x16, 0xBF, 0xC, 0x87, 0x3, 0x61, 0xB4,
0x3B, 0x60, 0x6D, 0x7, 0x56, 0xB8, 0x1F, 0x89, 0xAD, 0x0, 0x25, 0x10, 0x4A,
0x34, 0x4C, 0x9A, 0x26, 0xDA, 0x6, 0x25, 0x9C, 0x91, 0xA6, 0xA5, 0xAD, 0x4D,
0x6E, 0xE9, 0x2F, 0x18, 0xC4, 0x1D, 0x9, 0xE1, 0xAA, 0x66, 0x1, 0x31, 0x6D,
0x12, 0x30, 0xED, 0x97, 0x3F, 0x67, 0xCE, 0x4E, 0x26, 0xB, 0xF5, 0x5E, 0x81,
0xA7, 0x1F, 0x83, 0x68, 0x91, 0xC3, 0xD0, 0x4C, 0x2E, 0xD4, 0xDE, 0xEF, 0x34,
0xF9, 0x61, 0x83, 0x6F, 0xD6, 0x6E, 0x40, 0x87, 0x48, 0x7E, 0xCF, 0x56, 0x42,
0x21, 0xBA, 0x40, 0x64, 0x17, 0xFA, 0x97, 0xFF, 0x8D, 0xC8, 0x32, 0xFA, 0xB7,
0x45, 0xB0, 0xEC, 0xBD, 0xE, 0x51, 0x63, 0x90, 0x5, 0x68, 0x7A, 0x45, 0x86,
0x68, 0x2A, 0xE, 0x81, 0x5F, 0xDD, 0x12, 0xAD, 0x48, 0xF6, 0x87, 0x2E, 0x8D,
0xF6, 0x86, 0xC3, 0x6D, 0x69, 0xD5, 0x4E, 0x52, 0x8A, 0x8E, 0xE8, 0x1, 0x56,
0x11, 0xCC, 0x2E, 0x3F, 0xB5, 0x46, 0x1D, 0xF6, 0x6E, 0x4A, 0xEE, 0x1C, 0x60,
0x15, 0x85, 0xF6, 0x40, 0xFD, 0x56, 0xDC, 0x10, 0x1, 0xC3, 0xBD, 0xAE, 0x5A,
0x13, 0x1F, 0x15, 0x16, 0x10, 0x92, 0xC5, 0x2, 0xC2, 0x81, 0xB5, 0x6A, 0x4D,
0x37, 0x29, 0x40, 0x8B, 0xAA, 0x5F, 0xC9, 0x4C, 0x26, 0x7B, 0x2C, 0x21, 0x9E,
0xE2, 0xF2, 0x5A, 0x20, 0x88, 0x3F, 0x40, 0x30, 0xC5, 0x64, 0xE };

struct PQDSATestVector {
  const char name[20];
  const int nid;
  const size_t public_key_len;
  const size_t private_key_len;
  const size_t signature_len;
  const char *kat_filename;
  const uint8_t *kPublicKey;
  const uint8_t *kPublicKeySPKI;
  const size_t kPublicKeySPKI_len;
};

#define CMP_VEC_AND_PTR(vec, ptr, len)         \
          {                                    \
            std::vector<uint8_t> tmp(len);     \
            tmp.assign(ptr, ptr + len);        \
            EXPECT_EQ(Bytes(vec), Bytes(tmp)); \
          }

#define CMP_VEC_AND_PKEY_PUBLIC(vec, pkey, len) \
CMP_VEC_AND_PTR(vec, pkey->pkey.pqdsa_key->public_key, len)

#define CMP_VEC_AND_PKEY_SECRET(vec, pkey, len) \
CMP_VEC_AND_PTR(vec, pkey->pkey.pqdsa_key->private_key, len)

static const struct PQDSATestVector parameterSet[] = {
  {"MLDSA65", NID_MLDSA65, 1952, 4032, 3309,  "dilithium/kat/MLDSA_65_hedged_pure.txt", mldsa65kPublicKey, mldsa65kPublicKeySPKI, 1974},
};

class PQDSAParameterTest : public testing::TestWithParam<PQDSATestVector> {};

INSTANTIATE_TEST_SUITE_P(All, PQDSAParameterTest, testing::ValuesIn(parameterSet),
                         [](const testing::TestParamInfo<PQDSATestVector> &params)
                             -> std::string { return params.param.name; });

TEST_P(PQDSAParameterTest, KAT) {
  std::string kat_filepath = "crypto/";
  kat_filepath += GetParam().kat_filename;

  FileTestGTest(kat_filepath.c_str(), [&](FileTest *t) {
    std::string count, mlen, smlen;
    std::vector<uint8_t> xi, rng, seed, msg, pk, sk, sm, ctxstr;

    ASSERT_TRUE(t->GetAttribute(&count, "count"));
    ASSERT_TRUE(t->GetBytes(&xi, "xi"));
    ASSERT_TRUE(t->GetBytes(&rng, "rng"));
    ASSERT_TRUE(t->GetBytes(&seed, "seed"));
    ASSERT_TRUE(t->GetBytes(&pk, "pk"));
    ASSERT_TRUE(t->GetBytes(&sk, "sk"));
    ASSERT_TRUE(t->GetBytes(&msg, "msg"));
    ASSERT_TRUE(t->GetAttribute(&mlen, "mlen"));
    ASSERT_TRUE(t->GetBytes(&sm, "sm"));
    ASSERT_TRUE(t->GetAttribute(&smlen, "smlen"));
    ASSERT_TRUE(t->GetBytes(&ctxstr, "ctx"));

    size_t pk_len = GetParam().public_key_len;
    size_t sk_len = GetParam().private_key_len;
    size_t sig_len = GetParam().signature_len;

    std::vector<uint8_t> pub(pk_len);
    std::vector<uint8_t> priv(sk_len);
    std::vector<uint8_t> signature(sig_len);

    std::string name = GetParam().name;
    size_t mlen_int = std::stoi(mlen);

    // The KATs provide the signed message, which is the signature appended with
    // the message that was signed. We use the ML-DSA APIs for sign_signature
    // and not sign_message, which return the signature without the appended
    // message, so we truncate the signed messaged to |sig_len|.
    sm.resize(sig_len);

    // Generate key pair from seed xi and assert that public and private keys
    // are equal to expected values from KAT
    if (name == "MLDSA65") {
      ASSERT_TRUE(ml_dsa_65_keypair_internal(pub.data(), priv.data(), xi.data()));
    }
    EXPECT_EQ(Bytes(pub), Bytes(pk));
    EXPECT_EQ(Bytes(priv), Bytes(sk));

    // Prepare m_prime = (0 || ctxlen || ctx)
    // See both FIPS 204: Algorithm 2 line 10 and FIPS 205: Algorithm 22 line 8
    uint8_t m_prime[257];
    size_t m_prime_len = ctxstr.size() + 2;
    m_prime[0] = 0;
    m_prime[1] = ctxstr.size();
    ASSERT_TRUE(ctxstr.size() <= 255);
    OPENSSL_memcpy(m_prime + 2 , ctxstr.data(), ctxstr.size());

    // Generate signature by signing |msg|, assert that signature is equal
    // to expected value from KAT, then verify signature.
    if (name == "MLDSA65") {
      ASSERT_TRUE(ml_dsa_65_sign_internal(priv.data(),
                                          signature.data(), &sig_len,
                                          msg.data(), mlen_int,
                                          m_prime, m_prime_len,
                                          rng.data()));
      ASSERT_EQ(Bytes(signature), Bytes(sm));
      ASSERT_TRUE(ml_dsa_65_verify_internal(pub.data(),
                                            signature.data(), sig_len,
                                            msg.data(), mlen_int,
                                            m_prime, m_prime_len));
    }
  });
}

TEST_P(PQDSAParameterTest, KeyGen) {
  // Basic key generation tests for MLDSA
  // Generate a MLDSA key
  int nid = GetParam().nid;
  size_t pk_len = GetParam().public_key_len;
  size_t sk_len = GetParam().private_key_len;

  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));
  ASSERT_TRUE(ctx);
  ASSERT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(ctx.get(),nid));
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  EVP_PKEY *raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  bssl::UniquePtr<EVP_PKEY> pkey(raw);

  // Extract public key and check it is of the correct size
  uint8_t *buf = nullptr;
  size_t buf_size;
  EXPECT_TRUE(EVP_PKEY_get_raw_public_key(pkey.get(), buf, &buf_size));
  EXPECT_EQ(pk_len, buf_size);

  buf = (uint8_t *)OPENSSL_malloc(buf_size);
  ASSERT_NE(buf, nullptr);
  EXPECT_TRUE(EVP_PKEY_get_raw_public_key(pkey.get(), buf, &buf_size));

  buf_size = 0;
  EXPECT_FALSE(EVP_PKEY_get_raw_public_key(pkey.get(), buf, &buf_size));

  uint32_t err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_BUFFER_TOO_SMALL, ERR_GET_REASON(err));
  OPENSSL_free(buf);
  buf = nullptr;

  // Extract private key and check it is of the correct size
  EXPECT_TRUE(EVP_PKEY_get_raw_private_key(pkey.get(), buf, &buf_size));
  EXPECT_EQ((size_t)sk_len, buf_size);

  buf = (uint8_t *)OPENSSL_malloc(buf_size);
  ASSERT_NE(buf, nullptr);
  EXPECT_TRUE(EVP_PKEY_get_raw_private_key(pkey.get(), buf, &buf_size));

  buf_size = 0;
  EXPECT_FALSE(EVP_PKEY_get_raw_private_key(pkey.get(), buf, &buf_size));
  err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_BUFFER_TOO_SMALL, ERR_GET_REASON(err));
  OPENSSL_free(buf);
}

TEST_P(PQDSAParameterTest, KeyCmp) {
  // Generate two MLDSA keys are check that they are not equal.
  const int nid = GetParam().nid;

  // Generate first keypair
  bssl::UniquePtr<EVP_PKEY_CTX> ctx1(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));
  ASSERT_TRUE(ctx1);
  ASSERT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(ctx1.get(),nid));
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx1.get()));
  EVP_PKEY *raw1 = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx1.get(), &raw1));
  bssl::UniquePtr<EVP_PKEY> pkey1(raw1);

  // Generate second keypair
  bssl::UniquePtr<EVP_PKEY_CTX> ctx2(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));
  ASSERT_TRUE(ctx2);
  ASSERT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(ctx2.get(),nid));
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx2.get()));
  EVP_PKEY *raw2 = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx2.get(), &raw2));
  bssl::UniquePtr<EVP_PKEY> pkey2(raw2);

  // Compare keys
  EXPECT_EQ(0, EVP_PKEY_cmp(pkey1.get(), pkey2.get()));
}

TEST_P(PQDSAParameterTest, KeySize) {
  // Test the key size of MLDSA key is as expected
  int nid = GetParam().nid;
  int pk_len = GetParam().public_key_len;
  int sig_len = GetParam().signature_len;

  // generate an MLDSA keypair
  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));
  ASSERT_TRUE(ctx);
  ASSERT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(ctx.get(),nid));
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  EVP_PKEY *raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  bssl::UniquePtr<EVP_PKEY> pkey(raw);

  EXPECT_EQ(sig_len, EVP_PKEY_size(pkey.get()));
  EXPECT_EQ(8*(pk_len), EVP_PKEY_bits(pkey.get()));
}

TEST_P(PQDSAParameterTest, NewKeyFromBytes) {
  // Test the generation of a MLDSA key from bytes
  int nid = GetParam().nid;
  size_t pk_len = GetParam().public_key_len;
  size_t sk_len = GetParam().private_key_len;

  // Source key
  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));
  ASSERT_TRUE(ctx);
  ASSERT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(ctx.get(), nid));
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  EVP_PKEY *raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  bssl::UniquePtr<EVP_PKEY> pkey(raw);

  // New raw pkey to store raw public key
  bssl::UniquePtr<EVP_PKEY> new_pkey(EVP_PKEY_pqdsa_new_raw_public_key(nid, pkey->pkey.pqdsa_key->public_key, pk_len));

  // check that public key is present and secret key is not present
  ASSERT_NE(new_pkey, nullptr);
  EXPECT_NE(new_pkey->pkey.pqdsa_key->public_key, nullptr);
  EXPECT_EQ(new_pkey->pkey.pqdsa_key->private_key, nullptr);

  // check that EVP_PKEY_get_raw_private_key fails correctly
  uint8_t *buf = nullptr;
  size_t buf_size;
  EXPECT_FALSE(EVP_PKEY_get_raw_private_key(new_pkey.get(), buf, &buf_size));
  uint32_t err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_NOT_A_PRIVATE_KEY, ERR_GET_REASON(err));

  // EVP_PKEY_cmp returns 1 on success
  EXPECT_EQ(1, EVP_PKEY_cmp(pkey.get(), new_pkey.get()));

  // New raw pkey to store raw secret key
  bssl::UniquePtr<EVP_PKEY> private_pkey(EVP_PKEY_pqdsa_new_raw_private_key(nid, pkey->pkey.pqdsa_key->private_key, sk_len));

  // check that secret key is present and public key is not present
  ASSERT_NE(private_pkey, nullptr);
  EXPECT_EQ(private_pkey->pkey.pqdsa_key->public_key, nullptr);
  EXPECT_NE(private_pkey->pkey.pqdsa_key->private_key, nullptr);
  EXPECT_EQ(0, OPENSSL_memcmp(private_pkey->pkey.pqdsa_key->private_key, pkey->pkey.pqdsa_key->private_key, sk_len));
}

TEST_P(PQDSAParameterTest, RawFunctions) {
  // Test EVP_PKEY_get_raw_public_key for extracting public keys
  // Test EVP_PKEY_get_raw_private_key for extracting private keys
  // Test EVP_PKEY_pqdsa_new_raw_public_key for generating a new PKEY from raw pub
  // Test EVP_parse_public_key can parse the DER to a PKEY
  // Test EVP_PKEY_pqdsa_new_raw_private_key for generating a new PKEY from raw priv

  int nid = GetParam().nid;
  size_t pk_len = GetParam().public_key_len;
  size_t sk_len = GetParam().private_key_len;

  // Generate mldsa key
  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));
  ASSERT_TRUE(ctx);
  ASSERT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(ctx.get(),nid));
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  EVP_PKEY *raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  bssl::UniquePtr<EVP_PKEY> pkey(raw);

  // The public key must be extractable.
  std::vector<uint8_t> pub_buf(pk_len);
  size_t pub_len;
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(pkey.get(), nullptr, &pub_len));
  EXPECT_EQ(pub_len, pk_len);
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(pkey.get(), pub_buf.data(), &pub_len));

  bssl::UniquePtr<EVP_PKEY> pkey_pk_new(EVP_PKEY_pqdsa_new_raw_public_key(nid, pub_buf.data(), pk_len));
  ASSERT_TRUE(pkey_pk_new);

  // The public key must encode properly.
  bssl::ScopedCBB cbb;
  uint8_t *der;
  size_t der_len;
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_public_key(cbb.get(), pkey_pk_new.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  bssl::UniquePtr<uint8_t> free_der(der);

  // The public key must parse properly.
  CBS cbs;
  CBS_init(&cbs, der, der_len);
  bssl::UniquePtr<EVP_PKEY> pkey_from_der(EVP_parse_public_key(&cbs));
  ASSERT_TRUE(pkey_from_der.get());
  EXPECT_EQ(1, EVP_PKEY_cmp(pkey.get(), pkey_from_der.get()));

  // The secret key must be extractable.
  std::vector<uint8_t> priv_buf(sk_len);
  size_t priv_len;
  ASSERT_TRUE(EVP_PKEY_get_raw_private_key(pkey.get(), nullptr, &priv_len));
  EXPECT_EQ(priv_len, sk_len);
  ASSERT_TRUE(EVP_PKEY_get_raw_private_key(pkey.get(), priv_buf.data(), &priv_len));

  bssl::UniquePtr<EVP_PKEY> pkey_sk_new(EVP_PKEY_pqdsa_new_raw_private_key(nid, priv_buf.data(), sk_len));
  ASSERT_TRUE(pkey_sk_new);

  // The private key must encode properly.
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_private_key(cbb.get(), pkey_sk_new.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  free_der.reset(der);

  // private key parse
  CBS_init(&cbs, der, der_len);
  bssl::UniquePtr<EVP_PKEY> pkey_priv_from_der(EVP_parse_private_key(&cbs));
  ASSERT_TRUE(pkey_priv_from_der);

  EXPECT_EQ(Bytes(pkey_priv_from_der->pkey.pqdsa_key->private_key, priv_len),
            Bytes(priv_buf.data(), sk_len));
}

TEST_P(PQDSAParameterTest, SIGOperations) {
  // Test basic functionality for MLDSA
  int nid = GetParam().nid;
  size_t sig_len = GetParam().signature_len;

  // Generate a mldsa key
  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));
  ASSERT_TRUE(ctx);
  ASSERT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(ctx.get(),nid));
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  EVP_PKEY *raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  bssl::UniquePtr<EVP_PKEY> pkey(raw);

  // Sign a message
  bssl::ScopedEVP_MD_CTX md_ctx;
  std::vector<uint8_t> signature1(sig_len);

 // msg2 differs from msg1 by one byte
  std::vector<uint8_t> msg1 = {
      0x4a, 0x41, 0x4b, 0x45, 0x20, 0x4d, 0x41, 0x53, 0x53, 0x49,
      0x4d, 0x4f, 0x20, 0x41, 0x57, 0x53, 0x32, 0x30, 0x32, 0x32, 0x2e};
  std::vector<uint8_t> msg2 = {
      0x4a, 0x41, 0x4b, 0x45, 0x20, 0x4d, 0x41, 0x53, 0x53, 0x49,
      0x4d, 0x4f, 0x20, 0x41, 0x57, 0x53, 0x32, 0x30, 0x32, 0x31, 0x2e};

  ASSERT_TRUE(EVP_DigestSignInit(md_ctx.get(), nullptr, nullptr,
                                 nullptr, pkey.get()));
  ASSERT_TRUE(EVP_DigestSign(md_ctx.get(), signature1.data(), &sig_len,
                             msg1.data(), msg1.size()));

  // Verify the correct signed message
  ASSERT_TRUE(EVP_DigestVerify(md_ctx.get(), signature1.data(), sig_len,
                               msg1.data(), msg1.size()));

  // Verify the signed message fails upon a different message
  ASSERT_FALSE(EVP_DigestVerify(md_ctx.get(), signature1.data(), sig_len,
                                msg2.data(), msg2.size()));

  // Sign the different message

  std::vector<uint8_t> signature2(sig_len);
  md_ctx.Reset();
  ASSERT_TRUE(EVP_DigestSignInit(md_ctx.get(), nullptr, nullptr,
                                nullptr, pkey.get()));

  ASSERT_TRUE(EVP_DigestSign(md_ctx.get(), signature2.data(), &sig_len,
                             msg2.data(), msg2.size()));

  // Check that the two signatures are not equal
  EXPECT_NE(0, OPENSSL_memcmp(signature1.data(), signature2.data(), sig_len));

  // Verify the signed message fails upon a different signature
  ASSERT_FALSE(EVP_DigestVerify(md_ctx.get(), signature2.data(), sig_len,
                                msg1.data(), msg1.size()));
  md_ctx.Reset();
}

TEST_P(PQDSAParameterTest, MarshalParse) {
  // Test the example public key kPublicKey encodes correctly as kPublicKeySPKI
  // Test that the DER encoding can be parsed as a PKEY

  int nid = GetParam().nid;
  size_t pk_len = GetParam().public_key_len;
  const uint8_t * kPublicKey = GetParam().kPublicKey;
  const uint8_t * kPublicKeySPKI = GetParam().kPublicKeySPKI;
  size_t kPublicKeySPKI_len = GetParam().kPublicKeySPKI_len;

  bssl::UniquePtr<EVP_PKEY> pkey_pk_new(EVP_PKEY_pqdsa_new_raw_public_key(nid, kPublicKey, pk_len));
  ASSERT_TRUE(pkey_pk_new);

  // Encode the public key as DER
  bssl::ScopedCBB cbb;
  uint8_t *der;
  size_t der_len;
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_public_key(cbb.get(), pkey_pk_new.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  bssl::UniquePtr<uint8_t> free_der(der);

  // Test that the encoded public key encodes as expected
  EXPECT_EQ(Bytes(kPublicKeySPKI, kPublicKeySPKI_len), Bytes(der, der_len));

  // decode the DER structure, then parse as a PKEY.
  CBS cbs;
  CBS_init(&cbs, der, der_len);
  bssl::UniquePtr<EVP_PKEY> pkey_from_der(EVP_parse_public_key(&cbs));
  ASSERT_TRUE(pkey_from_der);
}

#else

TEST(PQDSATest, EvpDisabled) {
  ASSERT_EQ(nullptr, EVP_PKEY_CTX_new_id(EVP_PKEY_NONE, nullptr));
  bssl::UniquePtr<EVP_PKEY> pkey(EVP_PKEY_new());
  ASSERT_FALSE(EVP_PKEY_set_type(pkey.get(), EVP_PKEY_NONE));
}

#endif
