﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/states/model/StateMachineStatus.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>

using namespace Aws::Utils;


namespace Aws
{
  namespace SFN
  {
    namespace Model
    {
      namespace StateMachineStatusMapper
      {

        static const int ACTIVE_HASH = HashingUtils::HashString("ACTIVE");
        static const int DELETING_HASH = HashingUtils::HashString("DELETING");


        StateMachineStatus GetStateMachineStatusForName(const Aws::String& name)
        {
          int hashCode = HashingUtils::HashString(name.c_str());
          if (hashCode == ACTIVE_HASH)
          {
            return StateMachineStatus::ACTIVE;
          }
          else if (hashCode == DELETING_HASH)
          {
            return StateMachineStatus::DELETING;
          }
          EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
          if(overflowContainer)
          {
            overflowContainer->StoreOverflow(hashCode, name);
            return static_cast<StateMachineStatus>(hashCode);
          }

          return StateMachineStatus::NOT_SET;
        }

        Aws::String GetNameForStateMachineStatus(StateMachineStatus enumValue)
        {
          switch(enumValue)
          {
          case StateMachineStatus::NOT_SET:
            return {};
          case StateMachineStatus::ACTIVE:
            return "ACTIVE";
          case StateMachineStatus::DELETING:
            return "DELETING";
          default:
            EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
            if(overflowContainer)
            {
              return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
            }

            return {};
          }
        }

      } // namespace StateMachineStatusMapper
    } // namespace Model
  } // namespace SFN
} // namespace Aws
