#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
/*
 * RESTSimKmsVault.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/BlobMetadataUtils.h"
#include "fdbserver/RESTSimKmsVault.h"
#include "fdbclient/SimKmsVault.h"
#include "fdbrpc/HTTP.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/RESTKmsConnectorUtils.h"
#include "flow/Arena.h"
#include "flow/EncryptUtils.h"

#include <cstring>
#include <rapidjson/document.h>
#include <rapidjson/rapidjson.h>
#include <rapidjson/stringbuffer.h>
#include <rapidjson/writer.h>

#include "flow/FastRef.h"
#include "flow/IRandom.h"
#include "flow/Knobs.h"
#include "flow/Trace.h"
#include "flow/actorcompiler.h" // This must be the last #include.

using DomIdVec = std::vector<EncryptCipherDomainId>;
using BaseCipherDomIdVec = std::vector<std::pair<EncryptCipherBaseKeyId, Optional<EncryptCipherDomainId>>>;

using namespace RESTKmsConnectorUtils;

namespace {
const std::string missingVersionMsg = "Missing version";
const std::string missingVersionCode = "1234";
const std::string invalidVersionMsg = "Invalid version";
const std::string invalidVersionCode = "5678";
const std::string missingTokensMsg = "Missing validation tokens";
const std::string missingTokenCode = "0123";

const std::string bgUrl = "file://simfdb/fdbblob/";

struct VaultResponse {
	bool failed;
	std::string buff;

	VaultResponse() : failed(false), buff("") {}
};

} // namespace

int64_t getRefreshInterval(const int64_t now, const int64_t defaultTtl) {
	if (BUGGIFY) {
		return now;
	}
	return (now + defaultTtl);
}

int64_t getExpireInterval(const int64_t refTS, const int64_t defaultTtl) {

	if (BUGGIFY) {
		return -1;
	}
	return (refTS + defaultTtl);
}

void validateHeaders(const HTTP::Headers& toCompare) {
	if (toCompare != RESTKmsConnectorUtils::getHTTPHeaders()) {
		throw rest_malformed_response();
	}
}

void addErrorToDoc(rapidjson::Document& doc, const ErrorDetail& details) {
	rapidjson::Value errorDetail(rapidjson::kObjectType);
	if (!details.errorMsg.empty()) {
		// Add "errorMsg"
		rapidjson::Value key(ERROR_MSG_TAG, doc.GetAllocator());
		rapidjson::Value errMsg;
		errMsg.SetString(details.errorMsg.data(), details.errorMsg.size(), doc.GetAllocator());
		errorDetail.AddMember(key, errMsg, doc.GetAllocator());
	}
	if (!details.errorCode.empty()) {
		// Add "value" - token value
		rapidjson::Value key(ERROR_CODE_TAG, doc.GetAllocator());
		rapidjson::Value errCode;
		errCode.SetString(details.errorCode.data(), details.errorCode.size(), doc.GetAllocator());
		errorDetail.AddMember(key, errCode, doc.GetAllocator());
	}

	// Append "error"
	rapidjson::Value key(ERROR_TAG, doc.GetAllocator());
	doc.AddMember(key, errorDetail, doc.GetAllocator());
}

void prepareErrorResponse(VaultResponse* response,
                          const ErrorDetail& errorDetail,
                          Optional<int> version = Optional<int>()) {
	rapidjson::Document doc;
	doc.SetObject();

	if (version.present()) {
		addVersionToDoc(doc, version.get());
	}

	addErrorToDoc(doc, errorDetail);

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	ASSERT(!response->failed);
	response->failed = true;
	response->buff = std::string(sb.GetString(), sb.GetSize());
}

// Helper routine to extract 'version' from the input json document. If 'version' is missing or 'invalid', the routine
// is responsible to populate required error details to the 'response'
bool extractVersion(const rapidjson::Document& doc, VaultResponse* response, int* version) {
	// check version tag sanityrest_malformed_response
	if (!doc.HasMember(REQUEST_VERSION_TAG) || !doc[REQUEST_VERSION_TAG].IsInt()) {
		prepareErrorResponse(response, ErrorDetail(missingVersionCode, missingVersionMsg));
		CODE_PROBE(true, "RESTSimKmsVault missing version");
		return false;
	}

	*version = doc[REQUEST_VERSION_TAG].GetInt();
	if (*version < 0 || *version > SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION) {
		prepareErrorResponse(response, ErrorDetail(invalidVersionCode, invalidVersionMsg));
		CODE_PROBE(true, "RESTSimKmsVault invalid version");
		return false;
	}

	return true;
}

// Helper routine to validate 'validation-token(s)' from the input json document. If tokens are missing the routine is
// responsible to populate appropriate error to the 'response'
bool checkValidationTokens(const rapidjson::Document& doc, const int version, VaultResponse* response) {
	ASSERT(!response->failed);
	if (!doc.HasMember(VALIDATION_TOKENS_TAG) || !doc[VALIDATION_TOKENS_TAG].IsArray()) {
		prepareErrorResponse(response, ErrorDetail(missingTokenCode, missingTokensMsg), version);
		CODE_PROBE(true, "RESTSimKmsVault missing validation tokens");
		return false;
	}
	ASSERT(!response->failed);
	return true;
}

void addCipherDetailToRespDoc(rapidjson::Document& doc,
                              rapidjson::Value& cipherDetails,
                              const Reference<SimKmsVaultKeyCtx>& keyCtx,
                              const Optional<EncryptCipherDomainId> domId) {
	rapidjson::Value cipherDetail(rapidjson::kObjectType);

	// Add 'base_cipher_id'
	rapidjson::Value key(BASE_CIPHER_ID_TAG, doc.GetAllocator());
	rapidjson::Value baseKeyId;
	baseKeyId.SetUint64(keyCtx->id);
	cipherDetail.AddMember(key, baseKeyId, doc.GetAllocator());

	// Add 'encrypt_domain_id'
	if (domId.present()) {
		key.SetString(ENCRYPT_DOMAIN_ID_TAG, doc.GetAllocator());
		rapidjson::Value domainId;
		domainId.SetInt64(domId.get());
		cipherDetail.AddMember(key, domainId, doc.GetAllocator());
	}

	// Add 'cipher'
	key.SetString(BASE_CIPHER_TAG, doc.GetAllocator());
	rapidjson::Value cipher;
	ASSERT_EQ(keyCtx->key.size(), keyCtx->keyLen);
	cipher.SetString(reinterpret_cast<const char*>(keyCtx->key.begin()), keyCtx->keyLen, doc.GetAllocator());
	cipherDetail.AddMember(key, cipher, doc.GetAllocator());

	// Add 'refreshAt'
	key.SetString(REFRESH_AFTER_SEC, doc.GetAllocator());
	const int64_t refreshAt = getRefreshInterval(now(), FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value refreshInterval;
	refreshInterval.SetInt64(refreshAt);
	cipherDetail.AddMember(key, refreshInterval, doc.GetAllocator());

	// Add 'expireAt
	key.SetString(EXPIRE_AFTER_SEC, doc.GetAllocator());
	const int64_t expireAt = getExpireInterval(refreshAt, FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value expireInterval;
	expireInterval.SetInt64(expireAt);
	cipherDetail.AddMember(key, expireInterval, doc.GetAllocator());

	// push above object to the array
	cipherDetails.PushBack(cipherDetail, doc.GetAllocator());
}

void addBlobMetadaToResDoc(rapidjson::Document& doc, rapidjson::Value& blobDetails, const EncryptCipherDomainId domId) {
	Standalone<BlobMetadataDetailsRef> detailsRef = SimKmsVault::getBlobMetadata(domId, bgUrl);
	rapidjson::Value blobDetail(rapidjson::kObjectType);

	rapidjson::Value key(BLOB_METADATA_DOMAIN_ID_TAG, doc.GetAllocator());
	rapidjson::Value domainId;
	domainId.SetInt64(domId);
	blobDetail.AddMember(key, domainId, doc.GetAllocator());

	rapidjson::Value locations(rapidjson::kArrayType);
	for (const auto& loc : detailsRef.locations) {
		rapidjson::Value location(rapidjson::kObjectType);

		// set location-id
		key.SetString(BLOB_METADATA_LOCATION_ID_TAG, doc.GetAllocator());
		rapidjson::Value id;
		id.SetInt64(loc.locationId);
		location.AddMember(key, id, doc.GetAllocator());

		// set location-path
		key.SetString(BLOB_METADATA_LOCATION_PATH_TAG, doc.GetAllocator());
		rapidjson::Value path;
		path.SetString(reinterpret_cast<const char*>(loc.path.begin()), loc.path.size(), doc.GetAllocator());
		location.AddMember(key, path, doc.GetAllocator());

		locations.PushBack(location, doc.GetAllocator());
	}
	key.SetString(BLOB_METADATA_LOCATIONS_TAG, doc.GetAllocator());
	blobDetail.AddMember(key, locations, doc.GetAllocator());

	// Add required key lifecycle timestamps to match real KMS API behavior
	// These fields enable clients to implement proper key rotation schedules
	// (Checked by EncryptedRangeFlowWriter)
	rapidjson::Value refreshKey(REFRESH_AFTER_SEC, doc.GetAllocator());
	const int64_t refreshAt = getRefreshInterval(now(), FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value refreshInterval;
	refreshInterval.SetInt64(refreshAt);
	blobDetail.AddMember(refreshKey, refreshInterval, doc.GetAllocator());

	// Expiration time prevents indefinite key usage for security compliance
	rapidjson::Value expireKey(EXPIRE_AFTER_SEC, doc.GetAllocator());
	const int64_t expireAt = getExpireInterval(refreshAt, FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value expireInterval;
	expireInterval.SetInt64(expireAt);
	blobDetail.AddMember(expireKey, expireInterval, doc.GetAllocator());

	blobDetails.PushBack(blobDetail, doc.GetAllocator());
}

void addKmsUrlsToDoc(rapidjson::Document& doc) {
	rapidjson::Value kmsUrls(rapidjson::kArrayType);
	// FIXME: fetch latest KMS URLs && append to the doc
	rapidjson::Value memberKey(KMS_URLS_TAG, doc.GetAllocator());
	doc.AddMember(memberKey, kmsUrls, doc.GetAllocator());
}

// TODO: inject faults
VaultResponse handleFetchKeysByDomainIds(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;
	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'cipher_key_details' as json array
	rapidjson::Value cipherDetails(rapidjson::kArrayType);
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		// Check if ENCRYPT_DOMAIN_ID_TAG exists before accessing it
		if (!cipherDetail.HasMember(ENCRYPT_DOMAIN_ID_TAG) || !cipherDetail[ENCRYPT_DOMAIN_ID_TAG].IsInt64()) {
			continue; // Skip invalid entries
		}
		EncryptCipherDomainId domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domainId);
		ASSERT(keyCtx.isValid());
		addCipherDetailToRespDoc(result, cipherDetails, keyCtx, domainId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, cipherDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchByDomainIdsResponseStr").detail("Str", response->buff);
	return response;
}

VaultResponse handleFetchKeysByKeyIds(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;

	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'cipher_key_details' as json array
	rapidjson::Value cipherDetails(rapidjson::kArrayType);
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		Optional<EncryptCipherDomainId> domainId;
		if (cipherDetail.HasMember(ENCRYPT_DOMAIN_ID_TAG) && cipherDetail[ENCRYPT_DOMAIN_ID_TAG].IsInt64()) {
			domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		}
		EncryptCipherBaseKeyId baseCipherId = cipherDetail[BASE_CIPHER_ID_TAG].GetUint64();
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByBaseCipherId(baseCipherId);
		ASSERT(keyCtx.isValid());
		addCipherDetailToRespDoc(result, cipherDetails, keyCtx, domainId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, cipherDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchByKeyIdsResponseStr").detail("Str", response.buff);
	return response;
}

VaultResponse handleFetchBlobMetada(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;

	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'blob_metadata_details' as json array
	rapidjson::Value blobDetails(rapidjson::kArrayType);
	for (const auto& blobDetail : doc[BLOB_METADATA_DETAILS_TAG].GetArray()) {
		// Check if BLOB_METADATA_DOMAIN_ID_TAG exists before accessing it
		if (!blobDetail.HasMember(BLOB_METADATA_DOMAIN_ID_TAG) || !blobDetail[BLOB_METADATA_DOMAIN_ID_TAG].IsInt64()) {
			continue; // Skip invalid entries
		}
		EncryptCipherDomainId domainId = blobDetail[BLOB_METADATA_DOMAIN_ID_TAG].GetInt64();
		addBlobMetadaToResDoc(doc, blobDetails, domainId);
	}
	rapidjson::Value memberKey(BLOB_METADATA_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, blobDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchBlobMetadataResponeStr").detail("Str", response.buff);
	return response;
}

															#line 439 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via simKmsVaultRequestHandler()
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class SimKmsVaultRequestHandlerActor>
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class SimKmsVaultRequestHandlerActorState {
															#line 446 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	SimKmsVaultRequestHandlerActorState(Reference<HTTP::IncomingRequest> const& request,Reference<HTTP::OutgoingResponse> const& response) 
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : request(request),
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(response)
															#line 455 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this));

	}
	~SimKmsVaultRequestHandlerActorState() 
	{
		fdb_probe_actor_destroy("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 439 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0);
															#line 439 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 472 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state = 1;
															#line 439 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*>(static_cast<SimKmsVaultRequestHandlerActor*>(this)));
															#line 477 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 440 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT_EQ(request->verb, HTTP::HTTP_VERB_POST);
															#line 442 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		validateHeaders(request->data.headers);
															#line 444 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		vaultResponse = VaultResponse();
															#line 445 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_KEY_IDS_RESOURCE) == 0)
															#line 506 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		{
															#line 446 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			vaultResponse = handleFetchKeysByKeyIds(request->data.content);
															#line 510 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		}
		else
		{
															#line 447 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_DOMAIN_IDS_RESOURCE) == 0)
															#line 516 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			{
															#line 448 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				vaultResponse = handleFetchKeysByDomainIds(request->data.content);
															#line 520 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			}
			else
			{
															#line 449 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (request->resource.compare(REST_SIM_KMS_VAULT_GET_BLOB_METADATA_RESOURCE) == 0)
															#line 526 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				{
															#line 450 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					vaultResponse = handleFetchBlobMetada(request->data.content);
															#line 530 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				}
				else
				{
															#line 452 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					TraceEvent("UnexpectedResource").detail("Resource", request->resource);
															#line 453 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					return a_body1Catch1(http_bad_response(), loopDepth);
															#line 538 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				}
			}
		}
															#line 456 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->code = HTTP::HTTP_STATUS_CODE_OK;
															#line 457 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.headers = request->data.headers;
															#line 458 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		PacketWriter pw(response->data.content->getWriteBuffer(vaultResponse.buff.size()), nullptr, Unversioned());
															#line 459 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		pw.serializeBytes(vaultResponse.buff.data(), vaultResponse.buff.size());
															#line 460 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.contentLen = vaultResponse.buff.size();
															#line 462 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SimKmsVaultRequestHandlerActorState(); static_cast<SimKmsVaultRequestHandlerActor*>(this)->destroy(); return 0; }
															#line 554 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		new (&static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 440 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT_EQ(request->verb, HTTP::HTTP_VERB_POST);
															#line 442 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		validateHeaders(request->data.headers);
															#line 444 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		vaultResponse = VaultResponse();
															#line 445 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_KEY_IDS_RESOURCE) == 0)
															#line 572 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		{
															#line 446 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			vaultResponse = handleFetchKeysByKeyIds(request->data.content);
															#line 576 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		}
		else
		{
															#line 447 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_DOMAIN_IDS_RESOURCE) == 0)
															#line 582 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			{
															#line 448 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				vaultResponse = handleFetchKeysByDomainIds(request->data.content);
															#line 586 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			}
			else
			{
															#line 449 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (request->resource.compare(REST_SIM_KMS_VAULT_GET_BLOB_METADATA_RESOURCE) == 0)
															#line 592 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				{
															#line 450 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					vaultResponse = handleFetchBlobMetada(request->data.content);
															#line 596 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				}
				else
				{
															#line 452 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					TraceEvent("UnexpectedResource").detail("Resource", request->resource);
															#line 453 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					return a_body1Catch1(http_bad_response(), loopDepth);
															#line 604 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				}
			}
		}
															#line 456 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->code = HTTP::HTTP_STATUS_CODE_OK;
															#line 457 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.headers = request->data.headers;
															#line 458 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		PacketWriter pw(response->data.content->getWriteBuffer(vaultResponse.buff.size()), nullptr, Unversioned());
															#line 459 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		pw.serializeBytes(vaultResponse.buff.data(), vaultResponse.buff.size());
															#line 460 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.contentLen = vaultResponse.buff.size();
															#line 462 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SimKmsVaultRequestHandlerActorState(); static_cast<SimKmsVaultRequestHandlerActor*>(this)->destroy(); return 0; }
															#line 620 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		new (&static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state > 0) static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state = 0;
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 444 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	VaultResponse vaultResponse;
															#line 709 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via simKmsVaultRequestHandler()
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class SimKmsVaultRequestHandlerActor final : public Actor<Void>, public ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >, public FastAllocated<SimKmsVaultRequestHandlerActor>, public SimKmsVaultRequestHandlerActorState<SimKmsVaultRequestHandlerActor> {
															#line 714 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<SimKmsVaultRequestHandlerActor>::operator new;
	using FastAllocated<SimKmsVaultRequestHandlerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6292728182553107968UL, 10259380704818319872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >;
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	SimKmsVaultRequestHandlerActor(Reference<HTTP::IncomingRequest> const& request,Reference<HTTP::OutgoingResponse> const& response) 
															#line 731 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   SimKmsVaultRequestHandlerActorState<SimKmsVaultRequestHandlerActor>(request, response),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3523274803376955648UL, 4651189297213340672UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("simKmsVaultRequestHandler");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
[[nodiscard]] Future<Void> simKmsVaultRequestHandler( Reference<HTTP::IncomingRequest> const& request, Reference<HTTP::OutgoingResponse> const& response ) {
															#line 437 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new SimKmsVaultRequestHandlerActor(request, response));
															#line 764 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}

#line 464 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

Future<Void> RESTSimKmsVaultRequestHandler::handleRequest(Reference<HTTP::IncomingRequest> request,
                                                          Reference<HTTP::OutgoingResponse> response) {
	return simKmsVaultRequestHandler(request, response);
}

// Only used to link unit tests
void forceLinkRESTSimKmsVaultTest() {}

namespace {

enum class FaultType { NONE = 1, MISSING_VERSION = 2, INVALID_VERSION = 3, MISSING_VALIDATION_TOKEN = 4 };

void addFakeValidationTokens(rapidjson::Document& doc) {
	ValidationTokenMap tokenMap;
	tokenMap.emplace("foo", ValidationTokenCtx("bar", ValidationTokenSource::VALIDATION_TOKEN_SOURCE_FILE));

	addValidationTokensSectionToJsonDoc(doc, tokenMap);
}

void constructDomainIds(EncryptCipherDomainIdVec& domIds) {
	domIds.push_back(SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID);
	domIds.push_back(FDB_DEFAULT_ENCRYPT_DOMAIN_ID);
	domIds.push_back(ENCRYPT_HEADER_DOMAIN_ID);

	int idx = deterministicRandom()->randomInt(512, 786);
	int count = deterministicRandom()->randomInt(5, 100);
	while (count--) {
		domIds.push_back(idx++);
	}
}

std::string getFakeDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds,
                                           const char* rootTag,
                                           const char* elementTag,
                                           FaultType fault = FaultType::NONE) {
	rapidjson::Document doc;
	doc.SetObject();

	if (fault == FaultType::INVALID_VERSION) {
		addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION + 1);
	} else if (fault == FaultType::MISSING_VERSION) {
		// Skip adding the version
	} else {
		addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION);
	}

	if (fault != FaultType::MISSING_VALIDATION_TOKEN) {
		addFakeValidationTokens(doc);
	}

	constructDomainIds(domIds);
	addLatestDomainDetailsToDoc(doc, rootTag, elementTag, domIds);

	addRefreshKmsUrlsSectionToJsonDoc(doc, deterministicRandom()->coinflip());

	if (deterministicRandom()->coinflip()) {
		addDebugUidSectionToJsonDoc(doc, deterministicRandom()->randomUniqueID());
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	std::string resp(sb.GetString(), sb.GetSize());
	/*TraceEvent(SevDebug, "FakeDomainIdsRequest")
	    .detail("Str", resp)
	    .detail("RootTag", rootTag)
	    .detail("ElementTag", elementTag);*/
	return resp;
}

std::string getFakeEncryptDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	return getFakeDomainIdsRequestContent(domIds, CIPHER_KEY_DETAILS_TAG, ENCRYPT_DOMAIN_ID_TAG, fault);
}

std::string getFakeBlobDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	return getFakeDomainIdsRequestContent(domIds, BLOB_METADATA_DETAILS_TAG, BLOB_METADATA_DOMAIN_ID_TAG, fault);
}

std::string getFakeBaseCipherIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	rapidjson::Document doc;
	doc.SetObject();

	if (fault != FaultType::MISSING_VERSION) {
		if (fault == FaultType::INVALID_VERSION) {
			addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION + 1);
		} else {
			// Fix: Only add valid version for non-invalid-version faults
			addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION);
		}
	}

	if (fault != FaultType::MISSING_VALIDATION_TOKEN) {
		addFakeValidationTokens(doc);
	}

	constructDomainIds(domIds);
	rapidjson::Value keyIdDetails(rapidjson::kArrayType);
	for (auto domId : domIds) {
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domId);
		ASSERT(keyCtx.isValid());
		addBaseCipherIdDomIdToDoc(doc, keyIdDetails, keyCtx->id, domId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, doc.GetAllocator());
	doc.AddMember(memberKey, keyIdDetails, doc.GetAllocator());

	addRefreshKmsUrlsSectionToJsonDoc(doc, deterministicRandom()->coinflip());

	if (deterministicRandom()->coinflip()) {
		addDebugUidSectionToJsonDoc(doc, deterministicRandom()->randomUniqueID());
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	std::string resp(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FakeKeyIdsRequest").detail("Str", resp);
	return resp;
}

Optional<ErrorDetail> getErrorDetail(const std::string& buff) {
	rapidjson::Document doc;
	doc.Parse(buff.data());
	return RESTKmsConnectorUtils::getError(doc);
}

void validateEncryptLookup(const VaultResponse& response, const EncryptCipherDomainIdVec& domIds) {
	ASSERT(!response.failed);

	//TraceEvent(SevDebug, "VaultEncryptResponse").detail("Str", response.buff);

	rapidjson::Document doc;
	doc.Parse(response.buff.data());

	ASSERT(doc.HasMember(CIPHER_KEY_DETAILS_TAG) && doc[CIPHER_KEY_DETAILS_TAG].IsArray());

	std::unordered_set<EncryptCipherDomainId> domIdSet(domIds.begin(), domIds.end());
	int count = 0;
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		EncryptCipherDomainId domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		EncryptCipherBaseKeyId baseCipherId = cipherDetail[BASE_CIPHER_ID_TAG].GetUint64();
		const int cipherKeyLen = cipherDetail[BASE_CIPHER_TAG].GetStringLength();
		Standalone<StringRef> cipherKeyRef = makeString(cipherKeyLen);
		memcpy(mutateString(cipherKeyRef), cipherDetail[BASE_CIPHER_TAG].GetString(), cipherKeyLen);

		ASSERT(domIdSet.find(domainId) != domIdSet.end());

		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domainId);
		ASSERT_EQ(keyCtx->id, baseCipherId);
		ASSERT_EQ(keyCtx->key.compare(cipherKeyRef), 0);
		const int64_t refreshAfterSec = cipherDetail[REFRESH_AFTER_SEC].GetInt64();
		const int64_t expireAfterSec = cipherDetail[EXPIRE_AFTER_SEC].GetInt64();
		ASSERT(refreshAfterSec <= expireAfterSec || expireAfterSec == -1);
		count++;
	}
	ASSERT_EQ(count, domIds.size());
}

void validateBlobLookup(const VaultResponse& response, const EncryptCipherDomainIdVec& domIds) {
	ASSERT(!response.failed);

	//TraceEvent(SevDebug, "VaultBlobResponse").detail("Str", response.buff);

	rapidjson::Document doc;
	doc.Parse(response.buff.data());

	ASSERT(doc.HasMember(BLOB_METADATA_DETAILS_TAG) && doc[BLOB_METADATA_DETAILS_TAG].IsArray());

	std::unordered_set<EncryptCipherDomainId> domIdSet(domIds.begin(), domIds.end());
	int count = 0;
	for (const auto& blobDetail : doc[BLOB_METADATA_DETAILS_TAG].GetArray()) {
		EncryptCipherDomainId domainId = blobDetail[BLOB_METADATA_DOMAIN_ID_TAG].GetInt64();
		Standalone<BlobMetadataDetailsRef> details = SimKmsVault::getBlobMetadata(domainId, bgUrl);

		std::unordered_map<BlobMetadataLocationId, Standalone<StringRef>> locMap;
		for (const auto& loc : details.locations) {
			locMap[loc.locationId] = loc.path;
		}
		for (const auto& location : blobDetail[BLOB_METADATA_LOCATIONS_TAG].GetArray()) {
			BlobMetadataLocationId locationId = location[BLOB_METADATA_LOCATION_ID_TAG].GetInt64();
			Standalone<StringRef> path = makeString(location[BLOB_METADATA_LOCATION_PATH_TAG].GetStringLength());
			memcpy(mutateString(path),
			       location[BLOB_METADATA_LOCATION_PATH_TAG].GetString(),
			       location[BLOB_METADATA_LOCATION_PATH_TAG].GetStringLength());
			auto it = locMap.find(locationId);
			ASSERT(it != locMap.end());
			ASSERT_EQ(it->second.compare(path), 0);
		}
		const int64_t refreshAfterSec = blobDetail[REFRESH_AFTER_SEC].GetInt64();
		const int64_t expireAfterSec = blobDetail[EXPIRE_AFTER_SEC].GetInt64();
		ASSERT(refreshAfterSec <= expireAfterSec || expireAfterSec == -1);
		count++;
	}
	ASSERT_EQ(count, domIds.size());
}

} // namespace

															#line 970 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase666()
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase666Actor>
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase666ActorState {
															#line 977 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase666ActorState(UnitTestParameters const& params) 
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params),
															#line 667 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   request(makeReference<HTTP::IncomingRequest>()),
															#line 668 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(makeReference<HTTP::OutgoingResponse>())
															#line 988 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase666", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase666ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase666", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 670 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->verb = HTTP::HTTP_VERB_POST;
															#line 671 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->resource = "/whatever";
															#line 672 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers = RESTKmsConnectorUtils::getHTTPHeaders();
															#line 1007 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			try {
															#line 674 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				StrictFuture<Void> __when_expr_0 = simKmsVaultRequestHandler(request, response);
															#line 674 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (static_cast<FlowTestCase666Actor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1013 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FlowTestCase666Actor*>(this)->actor_wait_state = 1;
															#line 674 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase666Actor, 0, Void >*>(static_cast<FlowTestCase666Actor*>(this)));
															#line 1018 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase666ActorState();
		static_cast<FlowTestCase666Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 679 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<FlowTestCase666Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase666ActorState(); static_cast<FlowTestCase666Actor*>(this)->destroy(); return 0; }
															#line 1047 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase666Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase666ActorState();
		static_cast<FlowTestCase666Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 677 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT_EQ(e.code(), error_code_http_bad_response);
															#line 1060 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 675 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1075 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 675 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1084 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase666Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase666Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase666Actor*>(this)->ActorCallback< FlowTestCase666Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase666Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase666", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase666Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase666", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase666Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase666", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase666Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase666", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase666Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase666", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase666Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase666", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 667 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 668 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1183 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase666()
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase666Actor final : public Actor<Void>, public ActorCallback< FlowTestCase666Actor, 0, Void >, public FastAllocated<FlowTestCase666Actor>, public FlowTestCase666ActorState<FlowTestCase666Actor> {
															#line 1188 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase666Actor>::operator new;
	using FastAllocated<FlowTestCase666Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1312056414185601280UL, 3727952365015484672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase666Actor, 0, Void >;
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase666Actor(UnitTestParameters const& params) 
															#line 1205 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase666ActorState<FlowTestCase666Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase666", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14063394408406652928UL, 10810305699797864960UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase666Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase666");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase666", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase666Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase666( UnitTestParameters const& params ) {
															#line 666 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase666Actor(params));
															#line 1238 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase666, "/restSimKmsVault/invalidResource")

#line 681 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1244 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase682()
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase682Actor>
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase682ActorState {
															#line 1251 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase682ActorState(UnitTestParameters const& params) 
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params),
															#line 683 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   request(makeReference<HTTP::IncomingRequest>()),
															#line 684 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(makeReference<HTTP::OutgoingResponse>())
															#line 1262 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase682", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase682ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase682", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 686 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->verb = HTTP::HTTP_VERB_POST;
															#line 687 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->resource = "/whatever";
															#line 688 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers = RESTKmsConnectorUtils::getHTTPHeaders();
															#line 689 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers["Foo"] = "Bar";
															#line 1283 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			try {
															#line 691 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				StrictFuture<Void> __when_expr_0 = simKmsVaultRequestHandler(request, response);
															#line 691 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (static_cast<FlowTestCase682Actor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1289 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FlowTestCase682Actor*>(this)->actor_wait_state = 1;
															#line 691 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase682Actor, 0, Void >*>(static_cast<FlowTestCase682Actor*>(this)));
															#line 1294 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase682ActorState();
		static_cast<FlowTestCase682Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 696 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<FlowTestCase682Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase682ActorState(); static_cast<FlowTestCase682Actor*>(this)->destroy(); return 0; }
															#line 1323 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase682Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase682ActorState();
		static_cast<FlowTestCase682Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 694 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT_EQ(e.code(), error_code_rest_malformed_response);
															#line 1336 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 692 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1351 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 692 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1360 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase682Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase682Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase682Actor*>(this)->ActorCallback< FlowTestCase682Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase682Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase682", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase682Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase682", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase682Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase682", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase682Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase682", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase682Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase682", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase682Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase682", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 683 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 684 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1459 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase682()
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase682Actor final : public Actor<Void>, public ActorCallback< FlowTestCase682Actor, 0, Void >, public FastAllocated<FlowTestCase682Actor>, public FlowTestCase682ActorState<FlowTestCase682Actor> {
															#line 1464 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase682Actor>::operator new;
	using FastAllocated<FlowTestCase682Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12399419653735536128UL, 9232024263473615360UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase682Actor, 0, Void >;
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase682Actor(UnitTestParameters const& params) 
															#line 1481 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase682ActorState<FlowTestCase682Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase682", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6883151412054854912UL, 18284203850337170176UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase682Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase682");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase682", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase682Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase682( UnitTestParameters const& params ) {
															#line 682 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase682Actor(params));
															#line 1514 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase682, "/restSimKmsVault/invalidHeader")

#line 698 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1520 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase699()
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase699Actor>
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase699ActorState {
															#line 1527 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase699ActorState(UnitTestParameters const& params) 
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1534 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase699", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase699ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase699", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 700 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 701 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 702 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 703 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 704 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 705 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 706 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 708 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase699Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase699ActorState(); static_cast<FlowTestCase699Actor*>(this)->destroy(); return 0; }
															#line 1563 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase699Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase699ActorState();
			static_cast<FlowTestCase699Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase699ActorState();
		static_cast<FlowTestCase699Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1587 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase699()
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase699Actor final : public Actor<Void>, public FastAllocated<FlowTestCase699Actor>, public FlowTestCase699ActorState<FlowTestCase699Actor> {
															#line 1592 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase699Actor>::operator new;
	using FastAllocated<FlowTestCase699Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(67040157071526144UL, 13924554882101881856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase699Actor(UnitTestParameters const& params) 
															#line 1608 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase699ActorState<FlowTestCase699Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase699", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6006357672719014656UL, 14736324602556891648UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase699Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase699");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase699", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase699( UnitTestParameters const& params ) {
															#line 699 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase699Actor(params));
															#line 1640 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase699, "/restSimKmsVault/GetByDomainIds/missingVersion")

#line 710 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1646 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase711()
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase711Actor>
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase711ActorState {
															#line 1653 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase711ActorState(UnitTestParameters const& params) 
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1660 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase711", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase711ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase711", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 712 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 713 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 714 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 715 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 716 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 717 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 718 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 720 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase711Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase711ActorState(); static_cast<FlowTestCase711Actor*>(this)->destroy(); return 0; }
															#line 1689 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase711Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase711ActorState();
			static_cast<FlowTestCase711Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase711ActorState();
		static_cast<FlowTestCase711Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1713 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase711()
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase711Actor final : public Actor<Void>, public FastAllocated<FlowTestCase711Actor>, public FlowTestCase711ActorState<FlowTestCase711Actor> {
															#line 1718 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase711Actor>::operator new;
	using FastAllocated<FlowTestCase711Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11047949248402619648UL, 3726263610667685888UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase711Actor(UnitTestParameters const& params) 
															#line 1734 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase711ActorState<FlowTestCase711Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase711", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10441158914127045120UL, 15935437430891565568UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase711Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase711");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase711", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase711( UnitTestParameters const& params ) {
															#line 711 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase711Actor(params));
															#line 1766 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase711, "/restSimKmsVault/GetByDomainIds/invalidVersion")

#line 722 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1772 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase723()
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase723Actor>
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase723ActorState {
															#line 1779 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase723ActorState(UnitTestParameters const& params) 
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1786 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase723", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase723ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase723", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 724 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 725 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 727 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 728 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 729 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 730 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 731 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 733 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase723Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase723ActorState(); static_cast<FlowTestCase723Actor*>(this)->destroy(); return 0; }
															#line 1815 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase723Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase723ActorState();
			static_cast<FlowTestCase723Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase723ActorState();
		static_cast<FlowTestCase723Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1839 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase723()
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase723Actor final : public Actor<Void>, public FastAllocated<FlowTestCase723Actor>, public FlowTestCase723ActorState<FlowTestCase723Actor> {
															#line 1844 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase723Actor>::operator new;
	using FastAllocated<FlowTestCase723Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5776141259092383744UL, 17874634911971949056UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase723Actor(UnitTestParameters const& params) 
															#line 1860 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase723ActorState<FlowTestCase723Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase723", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7924626676723581440UL, 2193516560803552000UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase723Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase723");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase723", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase723( UnitTestParameters const& params ) {
															#line 723 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase723Actor(params));
															#line 1892 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase723, "/restSimKmsVault/GetByDomainIds/missingValidationTokens")

#line 735 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1898 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase736()
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase736Actor>
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase736ActorState {
															#line 1905 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase736ActorState(UnitTestParameters const& params) 
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1912 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase736", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase736ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase736", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 737 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 738 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds);
															#line 740 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 741 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateEncryptLookup(response, domIds);
															#line 742 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase736Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase736ActorState(); static_cast<FlowTestCase736Actor*>(this)->destroy(); return 0; }
															#line 1935 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase736Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase736ActorState();
			static_cast<FlowTestCase736Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase736ActorState();
		static_cast<FlowTestCase736Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1959 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase736()
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase736Actor final : public Actor<Void>, public FastAllocated<FlowTestCase736Actor>, public FlowTestCase736ActorState<FlowTestCase736Actor> {
															#line 1964 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase736Actor>::operator new;
	using FastAllocated<FlowTestCase736Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16949559530645973504UL, 1043458593680510464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase736Actor(UnitTestParameters const& params) 
															#line 1980 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase736ActorState<FlowTestCase736Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase736", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16511195595519303168UL, 17683067396571736320UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase736Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase736");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase736", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase736( UnitTestParameters const& params ) {
															#line 736 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase736Actor(params));
															#line 2012 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase736, "/restSimKmsVault/GetByDomainIds")

#line 744 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2018 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase745()
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase745Actor>
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase745ActorState {
															#line 2025 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase745ActorState(UnitTestParameters const& params) 
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2032 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase745", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase745ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase745", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 746 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 747 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 749 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 750 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 751 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 752 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 753 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 755 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase745Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase745ActorState(); static_cast<FlowTestCase745Actor*>(this)->destroy(); return 0; }
															#line 2061 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase745Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase745ActorState();
			static_cast<FlowTestCase745Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase745ActorState();
		static_cast<FlowTestCase745Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2085 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase745()
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase745Actor final : public Actor<Void>, public FastAllocated<FlowTestCase745Actor>, public FlowTestCase745ActorState<FlowTestCase745Actor> {
															#line 2090 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase745Actor>::operator new;
	using FastAllocated<FlowTestCase745Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7934820647223196416UL, 3872004460315096064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase745Actor(UnitTestParameters const& params) 
															#line 2106 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase745ActorState<FlowTestCase745Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase745", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5601893455724070144UL, 680185364163385344UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase745Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase745");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase745", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase745( UnitTestParameters const& params ) {
															#line 745 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase745Actor(params));
															#line 2138 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase745, "/restSimKmsVault/GetByKeyIds/missingVersion")

#line 757 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2144 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase758()
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase758Actor>
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase758ActorState {
															#line 2151 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase758ActorState(UnitTestParameters const& params) 
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2158 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase758", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase758ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase758", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 759 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 760 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 762 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 763 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 764 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 765 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 766 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 768 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase758Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase758ActorState(); static_cast<FlowTestCase758Actor*>(this)->destroy(); return 0; }
															#line 2187 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase758Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase758ActorState();
			static_cast<FlowTestCase758Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase758ActorState();
		static_cast<FlowTestCase758Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2211 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase758()
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase758Actor final : public Actor<Void>, public FastAllocated<FlowTestCase758Actor>, public FlowTestCase758ActorState<FlowTestCase758Actor> {
															#line 2216 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase758Actor>::operator new;
	using FastAllocated<FlowTestCase758Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6819608382829896448UL, 3280579991891119360UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase758Actor(UnitTestParameters const& params) 
															#line 2232 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase758ActorState<FlowTestCase758Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase758", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5558394056721664UL, 3471552018802204672UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase758Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase758");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase758", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase758( UnitTestParameters const& params ) {
															#line 758 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase758Actor(params));
															#line 2264 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase758, "/restSimKmsVault/GetByKeyIds/invalidVersion")

#line 770 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2270 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase771()
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase771Actor>
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase771ActorState {
															#line 2277 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase771ActorState(UnitTestParameters const& params) 
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2284 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase771", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase771ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase771", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 772 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 773 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 775 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 776 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 777 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 778 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 779 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 781 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase771Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase771ActorState(); static_cast<FlowTestCase771Actor*>(this)->destroy(); return 0; }
															#line 2313 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase771Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase771ActorState();
			static_cast<FlowTestCase771Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase771ActorState();
		static_cast<FlowTestCase771Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2337 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase771()
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase771Actor final : public Actor<Void>, public FastAllocated<FlowTestCase771Actor>, public FlowTestCase771ActorState<FlowTestCase771Actor> {
															#line 2342 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase771Actor>::operator new;
	using FastAllocated<FlowTestCase771Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11558016175035567104UL, 2830326529068114688UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase771Actor(UnitTestParameters const& params) 
															#line 2358 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase771ActorState<FlowTestCase771Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase771", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15424100140347215616UL, 5145059038419179008UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase771Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase771");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase771", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase771( UnitTestParameters const& params ) {
															#line 771 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase771Actor(params));
															#line 2390 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase771, "/restSimKmsVault/GetByKeyIds/missingValidationTokens")

#line 783 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2396 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase784()
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase784Actor>
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase784ActorState {
															#line 2403 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase784ActorState(UnitTestParameters const& params) 
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2410 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase784", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase784ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase784", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 785 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 786 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 788 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 789 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 790 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 791 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 792 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 794 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase784Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase784ActorState(); static_cast<FlowTestCase784Actor*>(this)->destroy(); return 0; }
															#line 2439 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase784Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase784ActorState();
			static_cast<FlowTestCase784Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase784ActorState();
		static_cast<FlowTestCase784Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2463 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase784()
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase784Actor final : public Actor<Void>, public FastAllocated<FlowTestCase784Actor>, public FlowTestCase784ActorState<FlowTestCase784Actor> {
															#line 2468 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase784Actor>::operator new;
	using FastAllocated<FlowTestCase784Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1023446981267279872UL, 14694356997357515008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase784Actor(UnitTestParameters const& params) 
															#line 2484 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase784ActorState<FlowTestCase784Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase784", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9763689986929070848UL, 7234430394623382272UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase784Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase784");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase784", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase784( UnitTestParameters const& params ) {
															#line 784 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase784Actor(params));
															#line 2516 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase784, "/restSimKmsVault/GetByKeyIds/missingValidationTokensBlob")

#line 796 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2522 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase797()
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase797Actor>
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase797ActorState {
															#line 2529 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase797ActorState(UnitTestParameters const& params) 
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2536 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase797", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase797ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase797", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 798 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 799 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds);
															#line 801 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 802 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateEncryptLookup(response, domIds);
															#line 803 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase797Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase797ActorState(); static_cast<FlowTestCase797Actor*>(this)->destroy(); return 0; }
															#line 2559 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase797Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase797ActorState();
			static_cast<FlowTestCase797Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase797ActorState();
		static_cast<FlowTestCase797Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2583 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase797()
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase797Actor final : public Actor<Void>, public FastAllocated<FlowTestCase797Actor>, public FlowTestCase797ActorState<FlowTestCase797Actor> {
															#line 2588 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase797Actor>::operator new;
	using FastAllocated<FlowTestCase797Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1349788550986007296UL, 11509296329183874304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase797Actor(UnitTestParameters const& params) 
															#line 2604 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase797ActorState<FlowTestCase797Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase797", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10326023024507611392UL, 2447539992465652480UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase797Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase797");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase797", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase797( UnitTestParameters const& params ) {
															#line 797 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase797Actor(params));
															#line 2636 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase797, "/restSimKmsVault/GetByKeyIds")

#line 805 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2642 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase806()
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase806Actor>
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase806ActorState {
															#line 2649 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase806ActorState(UnitTestParameters const& params) 
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2656 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase806", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase806ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase806", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 807 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 808 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 810 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 811 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 812 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 813 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 814 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 816 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase806Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase806ActorState(); static_cast<FlowTestCase806Actor*>(this)->destroy(); return 0; }
															#line 2685 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase806Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase806ActorState();
			static_cast<FlowTestCase806Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase806ActorState();
		static_cast<FlowTestCase806Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2709 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase806()
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase806Actor final : public Actor<Void>, public FastAllocated<FlowTestCase806Actor>, public FlowTestCase806ActorState<FlowTestCase806Actor> {
															#line 2714 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase806Actor>::operator new;
	using FastAllocated<FlowTestCase806Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5191202549875046656UL, 11775309533804142336UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase806Actor(UnitTestParameters const& params) 
															#line 2730 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase806ActorState<FlowTestCase806Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase806", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13119145203896912896UL, 7539303562061006336UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase806Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase806");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase806", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase806( UnitTestParameters const& params ) {
															#line 806 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase806Actor(params));
															#line 2762 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase806, "/restSimKmsVault/GetBlobMetadata/missingVersion")

#line 818 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2768 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase819()
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase819Actor>
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase819ActorState {
															#line 2775 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase819ActorState(UnitTestParameters const& params) 
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2782 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase819", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase819ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase819", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 820 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 821 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 823 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 824 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 825 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 826 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 827 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 829 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase819Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase819ActorState(); static_cast<FlowTestCase819Actor*>(this)->destroy(); return 0; }
															#line 2811 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase819Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase819ActorState();
			static_cast<FlowTestCase819Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase819ActorState();
		static_cast<FlowTestCase819Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2835 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase819()
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase819Actor final : public Actor<Void>, public FastAllocated<FlowTestCase819Actor>, public FlowTestCase819ActorState<FlowTestCase819Actor> {
															#line 2840 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase819Actor>::operator new;
	using FastAllocated<FlowTestCase819Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(765625920764516352UL, 3205079859826971136UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase819Actor(UnitTestParameters const& params) 
															#line 2856 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase819ActorState<FlowTestCase819Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase819", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14117879735483337984UL, 2931431795118776832UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase819Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase819");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase819", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase819( UnitTestParameters const& params ) {
															#line 819 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase819Actor(params));
															#line 2888 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase819, "/restSimKmsVault/GetBlobMetadata/invalidVersion")

#line 831 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2894 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase832()
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase832Actor>
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase832ActorState {
															#line 2901 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase832ActorState(UnitTestParameters const& params) 
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2908 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase832", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase832ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase832", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 833 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 834 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds);
															#line 836 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 837 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateBlobLookup(response, domIds);
															#line 838 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase832Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase832ActorState(); static_cast<FlowTestCase832Actor*>(this)->destroy(); return 0; }
															#line 2931 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase832Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase832ActorState();
			static_cast<FlowTestCase832Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase832ActorState();
		static_cast<FlowTestCase832Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2955 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase832()
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase832Actor final : public Actor<Void>, public FastAllocated<FlowTestCase832Actor>, public FlowTestCase832ActorState<FlowTestCase832Actor> {
															#line 2960 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase832Actor>::operator new;
	using FastAllocated<FlowTestCase832Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17333231771222107648UL, 4771075764863005952UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase832Actor(UnitTestParameters const& params) 
															#line 2976 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase832ActorState<FlowTestCase832Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase832", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(805177850810109952UL, 3376717215393643008UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase832Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase832");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase832", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase832( UnitTestParameters const& params ) {
															#line 832 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase832Actor(params));
															#line 3008 "/codebuild/output/src2047728441/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase832, "/restSimKmsVault/GetBlobMetadata/foo")

