#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
/*
 * DataDistribution.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/Audit.h"
#include "fdbclient/AuditUtils.actor.h"
#include "fdbclient/BulkDumping.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/RunRYWTransaction.actor.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/BulkDumpUtil.actor.h"
#include "fdbserver/BulkLoadUtil.actor.h"
#include "fdbserver/DDSharedContext.h"
#include "fdbserver/DDTeamCollection.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/FDBExecHelper.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/QuietDatabase.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/TenantCache.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbserver/MockDataDistributor.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Buggify.h"
#include "flow/Error.h"
#include "flow/Platform.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/serialize.h"
#include "flow/actorcompiler.h" // This must be the last #include.

static const std::string ddServerBulkDumpFolder = "ddBulkDumpFiles";
static const std::string ddServerBulkLoadFolder = "ddBulkLoadFiles";

DataMoveType getDataMoveTypeFromDataMoveId(const UID& dataMoveId) {
	bool assigned, emptyRange;
	DataMoveType dataMoveType;
	DataMovementReason dataMoveReason;
	decodeDataMoveId(dataMoveId, assigned, emptyRange, dataMoveType, dataMoveReason);
	return dataMoveType;
}

void RelocateShard::setParentRange(KeyRange const& parent) {
	ASSERT(reason == RelocateReason::WRITE_SPLIT || reason == RelocateReason::SIZE_SPLIT);
	parent_range = parent;
}

Optional<KeyRange> RelocateShard::getParentRange() const {
	return parent_range;
}

ShardSizeBounds ShardSizeBounds::shardSizeBoundsBeforeTrack() {
	return ShardSizeBounds{ .max = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                           .iosPerKSecond = StorageMetrics::infinity,
		                                           .bytesReadPerKSecond = StorageMetrics::infinity,
		                                           .opsReadPerKSecond = StorageMetrics::infinity },
		                    .min = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = 0,
		                                           .iosPerKSecond = 0,
		                                           .bytesReadPerKSecond = 0,
		                                           .opsReadPerKSecond = 0 },
		                    .permittedError = StorageMetrics{ .bytes = -1,
		                                                      .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                                      .iosPerKSecond = StorageMetrics::infinity,
		                                                      .bytesReadPerKSecond = StorageMetrics::infinity,
		                                                      .opsReadPerKSecond = StorageMetrics::infinity } };
}

namespace {

std::set<int> const& normalDDQueueErrors() {
	static std::set<int> s{ error_code_movekeys_conflict,
		                    error_code_broken_promise,
		                    error_code_data_move_cancelled,
		                    error_code_data_move_dest_team_not_found };
	return s;
}

} // anonymous namespace

enum class DDAuditContext : uint8_t {
	INVALID = 0,
	RESUME = 1,
	LAUNCH = 2,
	RETRY = 3,
};

struct DDAudit {
	DDAudit(AuditStorageState coreState)
	  : coreState(coreState), actors(true), foundError(false), auditStorageAnyChildFailed(false), retryCount(0),
	    cancelled(false), overallCompleteDoAuditCount(0), overallIssuedDoAuditCount(0), overallSkippedDoAuditCount(0),
	    remainingBudgetForAuditTasks(SERVER_KNOBS->CONCURRENT_AUDIT_TASK_COUNT_MAX), context(DDAuditContext::INVALID) {}

	AuditStorageState coreState;
	ActorCollection actors;
	Future<Void> auditActor;
	bool foundError;
	int retryCount;
	bool auditStorageAnyChildFailed;
	bool cancelled; // use to cancel any actor beyond auditActor
	int64_t overallIssuedDoAuditCount;
	int64_t overallCompleteDoAuditCount;
	int64_t overallSkippedDoAuditCount;
	AsyncVar<int> remainingBudgetForAuditTasks;
	DDAuditContext context;
	std::unordered_set<UID> serversFinishedSSShardAudit; // dedicated to ssshard

	inline void setAuditRunActor(Future<Void> actor) { auditActor = actor; }
	inline Future<Void> getAuditRunActor() { return auditActor; }

	inline void setDDAuditContext(DDAuditContext context_) { this->context = context_; }
	inline DDAuditContext getDDAuditContext() const { return context; }

	// auditActor and actors are guaranteed to deliver a cancel signal
	void cancel() {
		auditActor.cancel();
		actors.clear(true);
		cancelled = true;
	}

	bool isCancelled() const { return cancelled; }
};

void DataMove::validateShard(const DDShardInfo& shard, KeyRangeRef range, int priority) {
	if (!valid) {
		if (shard.hasDest && shard.destId != anonymousShardId) {
			TraceEvent(SevError, "DataMoveValidationError")
			    .detail("Range", range)
			    .detail("Reason", "DataMoveMissing")
			    .detail("DestID", shard.destId)
			    .detail("ShardPrimaryDest", describe(shard.primaryDest))
			    .detail("ShardRemoteDest", describe(shard.remoteDest));
		}
		return;
	}

	if (this->meta.ranges.empty()) {
		TraceEvent(SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveMetatdataRangeEmpty")
		    .detail("DestID", shard.destId)
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		ASSERT(false);
	}

	if (!this->meta.ranges.front().contains(range)) {
		TraceEvent(SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveMetatdataRangeMismatch")
		    .detail("DestID", shard.destId)
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		ASSERT(false);
	}

	if (!shard.hasDest) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "ShardMissingDest")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest));
		cancelled = true;
		return;
	}

	if (shard.destId != this->meta.id) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveIDMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardMoveID", shard.destId);
		cancelled = true;
		return;
	}

	if (!std::equal(
	        this->primaryDest.begin(), this->primaryDest.end(), shard.primaryDest.begin(), shard.primaryDest.end()) ||
	    !std::equal(
	        this->remoteDest.begin(), this->remoteDest.end(), shard.remoteDest.begin(), shard.remoteDest.end())) {
		TraceEvent(g_network->isSimulated() ? SevWarn : SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveDestMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest))
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		cancelled = true;
	}
}

Future<Void> StorageWiggler::onCheck() const {
	return delay(MIN_ON_CHECK_DELAY_SEC);
}

// add server to wiggling queue
void StorageWiggler::addServer(const UID& serverId, const StorageMetadataType& metadata) {
	// std::cout << "size: " << pq_handles.size() << " add " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	ASSERT(!pq_handles.contains(serverId));
	pq_handles[serverId] = wiggle_pq.emplace(metadata, serverId);
}

void StorageWiggler::removeServer(const UID& serverId) {
	// std::cout << "size: " << pq_handles.size() << " remove " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	if (contains(serverId)) { // server haven't been popped
		auto handle = pq_handles.at(serverId);
		pq_handles.erase(serverId);
		wiggle_pq.erase(handle);
	}
}

void StorageWiggler::updateMetadata(const UID& serverId, const StorageMetadataType& metadata) {
	//	std::cout << "size: " << pq_handles.size() << " update " << serverId.toString()
	//	          << " DC: " << teamCollection->isPrimary() << std::endl;
	auto handle = pq_handles.at(serverId);
	if ((*handle).first == metadata) {
		return;
	}
	wiggle_pq.update(handle, std::make_pair(metadata, serverId));
}

bool StorageWiggler::necessary(const UID& serverId, const StorageMetadataType& metadata) const {
	return metadata.wrongConfiguredForWiggle ||
	       (now() - metadata.createdTime > SERVER_KNOBS->DD_STORAGE_WIGGLE_MIN_SS_AGE_SEC);
}

Optional<UID> StorageWiggler::getNextServerId(bool necessaryOnly) {
	if (!wiggle_pq.empty()) {
		auto [metadata, id] = wiggle_pq.top();
		if (necessaryOnly && !necessary(id, metadata)) {
			return {};
		}
		wiggle_pq.pop();
		pq_handles.erase(id);
		return Optional<UID>(id);
	}
	return Optional<UID>();
}

Future<Void> StorageWiggler::resetStats() {
	metrics.reset();
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.resetStorageWiggleMetrics(tr, PrimaryRegion(teamCollection->isPrimary()), metrics);
	    });
}

Future<Void> StorageWiggler::restoreStats() {
	auto readFuture = wiggleData.storageWiggleMetrics(PrimaryRegion(teamCollection->isPrimary()))
	                      .getD(teamCollection->dbContext().getReference(), Snapshot::False, metrics);
	return store(metrics, readFuture);
}

Future<Void> StorageWiggler::startWiggle() {
	metrics.last_wiggle_start = StorageMetadataType::currentTime();
	if (shouldStartNewRound()) {
		metrics.last_round_start = metrics.last_wiggle_start;
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

Future<Void> StorageWiggler::finishWiggle() {
	metrics.last_wiggle_finish = StorageMetadataType::currentTime();
	metrics.finished_wiggle += 1;
	auto duration = metrics.last_wiggle_finish - metrics.last_wiggle_start;
	metrics.smoothed_wiggle_duration.setTotal((double)duration);

	if (shouldFinishRound()) {
		metrics.last_round_finish = metrics.last_wiggle_finish;
		metrics.finished_round += 1;
		duration = metrics.last_round_finish - metrics.last_round_start;
		metrics.smoothed_round_duration.setTotal((double)duration);
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

															#line 319 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via remoteRecovered()
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoteRecoveredActor>
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActorState {
															#line 326 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActorState(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : db(db)
															#line 333 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	~RemoteRecoveredActorState() 
	{
		fdb_probe_actor_destroy("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 318 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDTrackerStarting").log();
															#line 319 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 350 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 323 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoteRecoveredActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoteRecoveredActorState(); static_cast<RemoteRecoveredActor*>(this)->destroy(); return 0; }
															#line 373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoteRecoveredActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 319 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(db->get().recoveryState < RecoveryState::ALL_LOGS_RECRUITED))
															#line 392 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 320 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDTrackerStarting").detail("RecoveryState", (int)db->get().recoveryState);
															#line 321 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 321 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 402 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 1;
															#line 321 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoteRecoveredActor, 0, Void >*>(static_cast<RemoteRecoveredActor*>(this)));
															#line 407 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state > 0) static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 0;
		static_cast<RemoteRecoveredActor*>(this)->ActorCallback< RemoteRecoveredActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoteRecoveredActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 514 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via remoteRecovered()
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActor final : public Actor<Void>, public ActorCallback< RemoteRecoveredActor, 0, Void >, public FastAllocated<RemoteRecoveredActor>, public RemoteRecoveredActorState<RemoteRecoveredActor> {
															#line 519 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoteRecoveredActor>::operator new;
	using FastAllocated<RemoteRecoveredActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(501509099070746880UL, 2789736369696224000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoteRecoveredActor, 0, Void >;
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActor(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 536 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoteRecoveredActorState<RemoteRecoveredActor>(db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17462346244829468672UL, 10124612030083111680UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("remoteRecovered");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoteRecoveredActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> remoteRecovered( Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoteRecoveredActor(db));
															#line 569 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 325 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Ensures that the serverKeys key space is properly coalesced
// This method is only used for testing and is not implemented in a manner that is safe for large databases
															#line 576 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via debugCheckCoalescing()
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DebugCheckCoalescingActor>
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActorState {
															#line 583 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActorState(Database const& cx) 
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : cx(cx),
															#line 329 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	~DebugCheckCoalescingActorState() 
	{
		fdb_probe_actor_destroy("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 330 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 607 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 332 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(serverListKeys, CLIENT_KNOBS->TOO_MANY);
															#line 332 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 640 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 1;
															#line 332 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 352 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 352 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 669 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 3;
															#line 352 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 2, Void >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 674 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 333 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 335 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = int();
															#line 336 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 693 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __serverList,int loopDepth) 
	{
															#line 332 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		serverList = __serverList;
															#line 702 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __serverList,int loopDepth) 
	{
		serverList = std::move(__serverList);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 349 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DoneCheckingCoalescing").log();
															#line 350 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DebugCheckCoalescingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DebugCheckCoalescingActorState(); static_cast<DebugCheckCoalescingActor*>(this)->destroy(); return 0; }
															#line 783 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DebugCheckCoalescingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 336 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < serverList.size()))
															#line 802 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 337 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		id = decodeServerListValue(serverList[i].value).id();
															#line 338 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, serverKeysPrefixFor(id), allKeys);
															#line 338 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 812 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 2;
															#line 338 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 817 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult const& ranges,int loopDepth) 
	{
															#line 339 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 341 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 342 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 843 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 343 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 847 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 336 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 852 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult && ranges,int loopDepth) 
	{
															#line 339 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 341 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 342 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 865 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 343 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 869 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 336 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 874 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult const& ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(ranges, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult && ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(std::move(ranges), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 329 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 332 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult serverList;
															#line 335 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 337 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID id;
															#line 1051 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via debugCheckCoalescing()
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActor final : public Actor<Void>, public ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 2, Void >, public FastAllocated<DebugCheckCoalescingActor>, public DebugCheckCoalescingActorState<DebugCheckCoalescingActor> {
															#line 1056 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DebugCheckCoalescingActor>::operator new;
	using FastAllocated<DebugCheckCoalescingActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10081829265818783744UL, 12211397008499659776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 2, Void >;
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActor(Database const& cx) 
															#line 1075 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DebugCheckCoalescingActorState<DebugCheckCoalescingActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13012747114799893760UL, 10690014687488268032UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("debugCheckCoalescing");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> debugCheckCoalescing( Database const& cx ) {
															#line 328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DebugCheckCoalescingActor(cx));
															#line 1110 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 356 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DataDistributor;
void runAuditStorage(
    Reference<DataDistributor> self,
    AuditStorageState auditStates,
    int retryCount,
    DDAuditContext context,
    Optional<std::unordered_set<UID>> serversFinishedSSShardAudit = Optional<std::unordered_set<UID>>());
															#line 1122 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorageCore( Reference<DataDistributor> const& self, UID const& auditID, AuditType const& auditType, int const& currentRetryCount );

#line 368 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1126 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<UID> launchAudit( Reference<DataDistributor> const& self, KeyRange const& auditRange, AuditType const& auditType, KeyValueStoreType const& auditStorageEngineType );

#line 372 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1130 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorage( Reference<DataDistributor> const& self, TriggerAuditRequest const& req );

#line 373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
void loadAndDispatchAudit(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit);
															#line 1135 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorageServerShard( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 375 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1139 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditStorageShardOnServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi );

#line 378 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1143 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorage( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 379 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1147 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 382 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1151 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& auditRange );

#line 385 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1155 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 388 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditOnStorageServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi, AuditStorageRequest const& req );

#line 392 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> skipAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& rangeToSkip );

#line 395 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

void runBulkLoadTaskAsync(Reference<DataDistributor> self, KeyRange range, UID taskId, bool restart);
															#line 1169 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self );

#line 398 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DDBulkLoadJobManager {
	BulkLoadJobState jobState;
	BulkLoadTransportMethod jobTransportMethod;
	// manifestEntryMap is a map from the begin key of the manifest to the manifest entry.
	// The end key of the current manifest is the begin key of the next manifest.
	// When the task range is aligned with the manifest range, every key is the begin key of the corresponding manifest.
	// When the task range is not aligned with the manifest range, the first key is the task begin key which can be
	// larger than the corresponding manifest begin key.
	std::shared_ptr<BulkLoadManifestFileMap> manifestEntryMap;
	std::string manifestLocalTempFolder;
	bool allTaskSubmitted = false;

	DDBulkLoadJobManager() = default;
	DDBulkLoadJobManager(const BulkLoadJobState& jobState, const std::string& manifestLocalTempFolder)
	  : jobState(jobState), manifestLocalTempFolder(manifestLocalTempFolder), allTaskSubmitted(false) {
		manifestEntryMap = std::make_shared<BulkLoadManifestFileMap>();
	}

	bool isValid() const { return jobState.isValid(); }
};

struct DDBulkDumpJobManager {
	BulkDumpState jobState;
	std::map<Key, BulkLoadManifest> taskManifestMap;

	DDBulkDumpJobManager() = default;
	DDBulkDumpJobManager(const BulkDumpState& jobState) : jobState(jobState) {}

	bool isValid() const { return jobState.isValid(); }
};

struct DataDistributor : NonCopyable, ReferenceCounted<DataDistributor> {
public:
	Reference<AsyncVar<ServerDBInfo> const> dbInfo;
	Reference<DDSharedContext> context;
	UID ddId;
	PromiseStream<Future<Void>> addActor;

	// State initialized when bootstrap
	Reference<IDDTxnProcessor> txnProcessor;
	MoveKeysLock& lock; // reference to context->lock
	DatabaseConfiguration& configuration; // reference to context->configuration
	std::vector<Optional<Key>> primaryDcId;
	std::vector<Optional<Key>> remoteDcIds;
	Reference<InitialDataDistribution> initData;

	Reference<EventCacheHolder> initialDDEventHolder;
	Reference<EventCacheHolder> movingDataEventHolder;
	Reference<EventCacheHolder> totalDataInFlightEventHolder;
	Reference<EventCacheHolder> totalDataInFlightRemoteEventHolder;

	// Optional components that can be set after ::init(). They're optional when test, but required for DD being
	// fully-functional.
	DDTeamCollection* teamCollection;
	Reference<ShardsAffectedByTeamFailure> shardsAffectedByTeamFailure;
	// consumer is a yield stream from producer. The RelocateShard is pushed into relocationProducer and popped from
	// relocationConsumer (by DDQueue)
	PromiseStream<RelocateShard> relocationProducer, relocationConsumer;
	PromiseStream<BulkLoadShardRequest> triggerShardBulkLoading;
	Reference<PhysicalShardCollection> physicalShardCollection;
	Reference<BulkLoadTaskCollection> bulkLoadTaskCollection;

	Promise<Void> initialized;

	std::unordered_map<AuditType, std::unordered_map<UID, std::shared_ptr<DDAudit>>> audits;
	FlowLock auditStorageHaLaunchingLock;
	FlowLock auditStorageReplicaLaunchingLock;
	FlowLock auditStorageLocationMetadataLaunchingLock;
	FlowLock auditStorageSsShardLaunchingLock;
	FlowLock auditStorageRestoreLaunchingLock;
	Promise<Void> auditStorageInitialized;
	bool auditStorageInitStarted;

	Optional<Reference<TenantCache>> ddTenantCache;

	// monitor DD configuration change
	Promise<Version> configChangeWatching;
	Future<Void> onConfigChange;

	ActorCollection bulkLoadActors;
	bool bulkLoadEnabled = false;
	ParallelismLimitor bulkLoadParallelismLimitor;
	ParallelismLimitor bulkLoadEngineParallelismLimitor;
	std::string bulkLoadFolder;

	Optional<DDBulkLoadJobManager> bulkLoadJobManager;

	bool bulkDumpEnabled = false;
	ParallelismLimitor bulkDumpParallelismLimitor;
	std::string folder;
	std::string bulkDumpFolder;

	DDBulkDumpJobManager bulkDumpJobManager;

	DataDistributor(Reference<AsyncVar<ServerDBInfo> const> const& db,
	                UID id,
	                Reference<DDSharedContext> context,
	                std::string folder)
	  : dbInfo(db), context(context), ddId(id), txnProcessor(nullptr), lock(context->lock),
	    configuration(context->configuration), initialDDEventHolder(makeReference<EventCacheHolder>("InitialDD")),
	    movingDataEventHolder(makeReference<EventCacheHolder>("MovingData")),
	    totalDataInFlightEventHolder(makeReference<EventCacheHolder>("TotalDataInFlight")),
	    totalDataInFlightRemoteEventHolder(makeReference<EventCacheHolder>("TotalDataInFlightRemote")),
	    teamCollection(nullptr), bulkLoadTaskCollection(nullptr), auditStorageHaLaunchingLock(1),
	    auditStorageReplicaLaunchingLock(1), auditStorageLocationMetadataLaunchingLock(1),
	    auditStorageSsShardLaunchingLock(1), auditStorageInitStarted(false), bulkLoadActors(false),
	    bulkLoadEnabled(false), bulkLoadParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM),
	    bulkLoadEngineParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM), bulkDumpEnabled(false),
	    bulkDumpParallelismLimitor(SERVER_KNOBS->DD_BULKDUMP_PARALLELISM), folder(folder) {
		if (!folder.empty()) {
			bulkDumpFolder = abspath(joinPath(folder, ddServerBulkDumpFolder));
			// TODO(BulkDump): clear this folder in the presence of crash
			bulkLoadFolder = abspath(joinPath(folder, ddServerBulkLoadFolder));
			// TODO(BulkLoad): clear this folder in the presence of crash
		}
	}

	// bootstrap steps

	Future<Void> takeMoveKeysLock() { return store(lock, txnProcessor->takeMoveKeysLock(ddId)); }

	Future<Void> loadDatabaseConfiguration() { return store(configuration, txnProcessor->getDatabaseConfiguration()); }

	Future<Void> updateReplicaKeys() {
		return txnProcessor->updateReplicaKeys(primaryDcId, remoteDcIds, configuration);
	}

	Future<Void> loadInitialDataDistribution() {
		return store(initData,
		             txnProcessor->getInitialDataDistribution(
		                 ddId,
		                 lock,
		                 configuration.usableRegions > 1 ? remoteDcIds : std::vector<Optional<Key>>(),
		                 context->ddEnabledState.get(),
		                 SkipDDModeCheck::False));
	}

	void initDcInfo() {
		primaryDcId.clear();
		remoteDcIds.clear();
		const std::vector<RegionInfo>& regions = configuration.regions;
		if (configuration.regions.size() > 0) {
			primaryDcId.push_back(regions[0].dcId);
		}
		if (configuration.regions.size() > 1) {
			remoteDcIds.push_back(regions[1].dcId);
		}
	}

	Future<Void> waitDataDistributorEnabled() const {
		return txnProcessor->waitForDataDistributionEnabled(context->ddEnabledState.get());
	}

	// Resume in-memory audit instances and issue background audit metadata cleanup
	void resumeAuditStorage(Reference<DataDistributor> self, std::vector<AuditStorageState> auditStates) {
		for (const auto& auditState : auditStates) {
			if (auditState.getPhase() != AuditPhase::Running) {
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "WrongAuditStateToResume")
				    .detail("AuditState", auditState.toString());
				return;
			}
			if (self->audits.contains(auditState.getType()) &&
			    self->audits[auditState.getType()].contains(auditState.id)) {
				// Ignore any RUNNING auditState with an alive audit
				// instance in DD audits map
				continue;
			}
			runAuditStorage(self, auditState, 0, DDAuditContext::RESUME);
			TraceEvent(SevInfo, "AuditStorageResumed", self->ddId)
			    .detail("AuditID", auditState.id)
			    .detail("AuditType", auditState.getType())
			    .detail("AuditState", auditState.toString());
		}
		return;
	}

																#line 1350 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via initAuditStorage()
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitAuditStorageActor>
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActorState {
															#line 1356 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActorState(Reference<DataDistributor> const& self) 
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 1363 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	~InitAuditStorageActorState() 
	{
		fdb_probe_actor_destroy("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 576 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->auditStorageInitStarted = true;
															#line 577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			MoveKeyLockInfo lockInfo;
															#line 578 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.myOwner = self->lock.myOwner;
															#line 579 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevOwner = self->lock.prevOwner;
															#line 580 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevWrite = self->lock.prevWrite;
															#line 581 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<std::vector<AuditStorageState>> __when_expr_0 = initAuditMetadata(self->txnProcessor->context(), lockInfo, self->context->isDDEnabled(), self->ddId, SERVER_KNOBS->PERSIST_FINISH_AUDIT_COUNT);
															#line 581 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1390 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 1;
															#line 581 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*>(static_cast<InitAuditStorageActor*>(this)));
															#line 1395 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
															#line 587 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 588 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 589 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1422 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
															#line 587 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 588 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 589 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1438 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(auditStatesToResume, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(auditStatesToResume), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state > 0) static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 0;
		static_cast<InitAuditStorageActor*>(this)->ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >::remove();

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> const& value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> && value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,Error err) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1523 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via initAuditStorage()
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActor final : public Actor<Void>, public ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >, public FastAllocated<InitAuditStorageActor>, public InitAuditStorageActorState<InitAuditStorageActor> {
															#line 1528 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitAuditStorageActor>::operator new;
	using FastAllocated<InitAuditStorageActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3516744452796790016UL, 11753483326962227712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >;
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActor(Reference<DataDistributor> const& self) 
															#line 1545 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitAuditStorageActorState<InitAuditStorageActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8855224104647952640UL, 5194562846350463232UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("initAuditStorage");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> initAuditStorage( Reference<DataDistributor> const& self ) {
															#line 575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitAuditStorageActor(self));
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 591 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 1582 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class WaitUntilDataDistributorExitSecurityModeActor>
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActorState {
															#line 1588 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActorState(Reference<DataDistributor> const& self) 
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 593 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(self->txnProcessor->context())
															#line 1597 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	~WaitUntilDataDistributorExitSecurityModeActorState() 
	{
		fdb_probe_actor_destroy("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 594 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 1612 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitUntilDataDistributorExitSecurityModeActorState();
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 595 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->DD_ENABLED_CHECK_DELAY, TaskPriority::DataDistribution);
															#line 595 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1644 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 1;
															#line 595 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1649 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 596 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 597 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 598 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1662 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1673 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 596 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 597 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 598 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1692 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1698 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1703 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 613 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 613 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1802 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 4;
															#line 613 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1807 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> const& mode,int loopDepth) 
	{
															#line 601 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1822 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 602 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1826 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 604 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 605 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 606 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 607 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1840 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 608 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1844 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 610 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 610 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1854 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 610 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1859 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> && mode,int loopDepth) 
	{
															#line 601 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1868 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 602 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1872 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 604 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 605 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 606 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 607 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1886 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 608 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1890 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 610 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 610 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1900 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 610 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1905 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> const& mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(mode, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> && mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(mode), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 611 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1989 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 611 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1998 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 593 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 2182 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActor final : public Actor<Void>, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >, public FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>, public WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor> {
															#line 2187 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator new;
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5155640281385761024UL, 6211096133817367296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >;
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActor(Reference<DataDistributor> const& self) 
															#line 2207 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4834519635010961920UL, 15563594834600371456UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitUntilDataDistributorExitSecurityMode");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> waitUntilDataDistributorExitSecurityMode( Reference<DataDistributor> const& self ) {
															#line 592 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new WaitUntilDataDistributorExitSecurityModeActor(self));
															#line 2242 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 617 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Initialize the required internal states of DataDistributor from system metadata. It's necessary before
	// DataDistributor start working. Doesn't include initialization of optional components, like TenantCache, DDQueue,
	// Tracker, TeamCollection. The components should call its own ::init methods.
																#line 2250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via init()
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitActor>
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActorState {
															#line 2256 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActorState(Reference<DataDistributor> const& self) 
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 2263 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("init", reinterpret_cast<unsigned long>(this));

	}
	~InitActorState() 
	{
		fdb_probe_actor_destroy("init", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 622 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 2278 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitActorState();
		static_cast<InitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 734 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitActorState(); static_cast<InitActor*>(this)->destroy(); return 0; }
															#line 2301 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitActorState();
		static_cast<InitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 623 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->waitDataDistributorEnabled();
															#line 623 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2322 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 1;
															#line 623 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitActor, 0, Void >*>(static_cast<InitActor*>(this)));
															#line 2327 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 624 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 626 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 627 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 627 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2355 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 627 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2360 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 624 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 626 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 627 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 627 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2375 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 627 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2380 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 628 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 632 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2466 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 634 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2470 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 636 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2474 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 643 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2479 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2485 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2490 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 628 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 632 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2506 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 634 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2510 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 636 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2514 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 643 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2519 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2525 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2530 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< InitActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 653 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributorExitSecurityMode").log();
															#line 655 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->loadDatabaseConfiguration();
															#line 655 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2623 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 4;
															#line 655 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< InitActor, 3, Void >*>(static_cast<InitActor*>(this)));
															#line 2628 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< InitActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 657 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 661 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2728 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 663 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 665 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2734 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2740 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2745 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 657 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 661 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2758 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 663 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 665 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2764 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2775 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< InitActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
															#line 669 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 671 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 671 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2863 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 671 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2868 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
															#line 669 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 671 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 671 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2881 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 671 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2886 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< InitActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 673 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 2970 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 674 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2974 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 681 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2980 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 689 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 2984 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 691 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 2988 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 695 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 699 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 721 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 727 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 2999 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 673 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 3008 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 674 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 3012 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 681 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 3018 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 689 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 3022 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 691 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 3026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 695 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 699 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 721 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 727 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 3037 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< InitActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 3119 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via init()
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActor final : public Actor<Void>, public ActorCallback< InitActor, 0, Void >, public ActorCallback< InitActor, 1, Void >, public ActorCallback< InitActor, 2, Void >, public ActorCallback< InitActor, 3, Void >, public ActorCallback< InitActor, 4, Void >, public ActorCallback< InitActor, 5, Void >, public FastAllocated<InitActor>, public InitActorState<InitActor> {
															#line 3124 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitActor>::operator new;
	using FastAllocated<InitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1397076796425805824UL, 5349528246170351360UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitActor, 0, Void >;
friend struct ActorCallback< InitActor, 1, Void >;
friend struct ActorCallback< InitActor, 2, Void >;
friend struct ActorCallback< InitActor, 3, Void >;
friend struct ActorCallback< InitActor, 4, Void >;
friend struct ActorCallback< InitActor, 5, Void >;
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActor(Reference<DataDistributor> const& self) 
															#line 3146 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitActorState<InitActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1770802396446596864UL, 4776746127134105600UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("init");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< InitActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< InitActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< InitActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< InitActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< InitActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> init( Reference<DataDistributor> const& self ) {
															#line 621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitActor(self));
															#line 3183 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 736 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3188 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoveDataMoveTombstoneBackgroundActor>
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActorState {
															#line 3194 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActorState(Reference<DataDistributor> const& self) 
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 738 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentID()
															#line 3203 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	~RemoveDataMoveTombstoneBackgroundActorState() 
	{
		fdb_probe_actor_destroy("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 740 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				cx = openDBOnServer(self->dbInfo, TaskPriority::DefaultEndpoint, LockAware::True);
															#line 741 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr = Transaction(cx);
															#line 742 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				;
															#line 3223 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 767 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoveDataMoveTombstoneBackgroundActorState(); static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->destroy(); return 0; }
															#line 3252 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 758 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 3265 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 759 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3269 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 761 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "RemoveDataMoveTombstoneError", self->ddId) .errorUnsuppressed(e) .detail("CurrentDataMoveID", currentID);
															#line 3273 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 744 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 745 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 746 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			for( UID& dataMoveID : self->initData->toCleanDataMoveTombstone ) {
															#line 747 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				currentID = dataMoveID;
															#line 748 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr.clear(dataMoveKeyFor(currentID));
															#line 749 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "RemoveDataMoveTombstone", self->ddId).detail("DataMoveID", currentID);
															#line 3312 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 751 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr.commit();
															#line 751 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3318 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 1;
															#line 751 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3323 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 754 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 754 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3360 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 2;
															#line 754 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3365 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 738 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID currentID;
															#line 740 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 741 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 3571 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActor final : public Actor<Void>, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >, public FastAllocated<RemoveDataMoveTombstoneBackgroundActor>, public RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor> {
															#line 3576 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator new;
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14084596469594320640UL, 17240230242477472512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >;
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >;
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActor(Reference<DataDistributor> const& self) 
															#line 3594 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929663207261195520UL, 2170269584154920960UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("removeDataMoveTombstoneBackground");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> removeDataMoveTombstoneBackground( Reference<DataDistributor> const& self ) {
															#line 737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoveDataMoveTombstoneBackgroundActor(self));
															#line 3627 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 769 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3632 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromShards()
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromShardsActor>
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActorState {
															#line 3638 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActorState(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   traceShard(traceShard)
															#line 3647 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromShardsActorState() 
	{
		fdb_probe_actor_destroy("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 772 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 3662 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 773 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for(int i = 0;i < self->initData->shards.size() - 1;i++) {
															#line 774 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					const DDShardInfo& iShard = self->initData->shards[i];
															#line 775 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					KeyRangeRef keys = KeyRangeRef(iShard.key, self->initData->shards[i + 1].key);
															#line 776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					teams.emplace_back(iShard.primarySrc, true);
															#line 778 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (self->configuration.usableRegions > 1)
															#line 3676 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 779 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.emplace_back(iShard.remoteSrc, false);
															#line 3680 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					}
															#line 781 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->physicalShardCollection->initPhysicalShardCollection(keys, teams, iShard.srcId.first(), 0);
															#line 3684 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 785 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundaries = std::vector<Key>();
															#line 786 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadIsEnabled(self->initData->bulkLoadMode))
															#line 3691 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 788 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevInfo, "DDInitCustomRangeConfigDisabledByBulkLoadMode", self->ddId);
															#line 3695 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 790 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for( auto it : self->initData->userRangeConfig->ranges() ) {
															#line 791 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					auto range = it->range();
															#line 792 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					customBoundaries.push_back(range.begin);
															#line 793 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDInitCustomRangeConfig", self->ddId) .detail("Range", KeyRangeRef(range.begin, range.end)) .detail("Config", it->value());
															#line 3707 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 799 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			shard = 0;
															#line 800 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary = 0;
															#line 801 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			overreplicatedCount = 0;
															#line 802 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 3718 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 876 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromShardsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromShardsActorState(); static_cast<ResumeFromShardsActor*>(this)->destroy(); return 0; }
															#line 3741 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromShardsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 802 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(shard < self->initData->shards.size() - 1))
															#line 3760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 803 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DDShardInfo& iShard = self->initData->shards[shard];
															#line 804 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<KeyRangeRef> ranges;
															#line 806 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key beginKey = iShard.key;
															#line 807 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key endKey = self->initData->shards[shard + 1].key;
															#line 808 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] <= beginKey;) {
															#line 809 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 811 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] < endKey;) {
															#line 812 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ranges.push_back(KeyRangeRef(beginKey, customBoundaries[customBoundary]));
															#line 813 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = customBoundaries[customBoundary];
															#line 814 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3786 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 816 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ranges.push_back(KeyRangeRef(beginKey, endKey));
															#line 818 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 819 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.primarySrc, true));
															#line 820 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.usableRegions > 1)
															#line 3796 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 821 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.remoteSrc, false));
															#line 3800 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 824 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int r = 0;r < ranges.size();r++) {
															#line 825 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto& keys = ranges[r];
															#line 826 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->defineShard(keys);
															#line 828 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->initData->userRangeConfig->rangeContaining(keys.begin);
															#line 829 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			int customReplicas = std::max(self->configuration.storageTeamSize, it->value().replicationFactor.orDefault(0));
															#line 831 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT_WE_THINK(KeyRangeRef(it->range().begin, it->range().end).contains(keys));
															#line 833 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bool unhealthy = iShard.primarySrc.size() != customReplicas;
															#line 834 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && self->configuration.usableRegions > 1)
															#line 3818 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 835 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				unhealthy = iShard.remoteSrc.size() != customReplicas;
															#line 3822 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 837 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && iShard.primarySrc.size() > self->configuration.storageTeamSize)
															#line 3826 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 838 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (++overreplicatedCount > SERVER_KNOBS->DD_MAX_SHARDS_ON_LARGE_TEAMS)
															#line 3830 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 839 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					unhealthy = true;
															#line 3834 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 843 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (traceShard)
															#line 3839 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 844 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDInitShard", self->ddId) .detail("Keys", keys) .detail("PrimarySrc", describe(iShard.primarySrc)) .detail("RemoteSrc", describe(iShard.remoteSrc)) .detail("PrimaryDest", describe(iShard.primaryDest)) .detail("RemoteDest", describe(iShard.remoteDest)) .detail("SrcID", iShard.srcId) .detail("DestID", iShard.destId) .detail("CustomReplicas", customReplicas) .detail("StorageTeamSize", self->configuration.storageTeamSize) .detail("Unhealthy", unhealthy) .detail("Overreplicated", overreplicatedCount);
															#line 3843 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 858 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->moveShard(keys, teams);
															#line 859 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if ((ddLargeTeamEnabled() && (unhealthy || r > 0)) || (iShard.hasDest && iShard.destId == anonymousShardId))
															#line 3849 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				DataMovementReason reason = DataMovementReason::RECOVER_MOVE;
															#line 865 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (unhealthy)
															#line 3855 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 866 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					reason = DataMovementReason::TEAM_UNHEALTHY;
															#line 3859 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 867 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (r > 0)
															#line 3865 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 868 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						reason = DataMovementReason::SPLIT_SHARD;
															#line 3869 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					}
				}
															#line 870 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(RelocateShard(keys, reason, RelocateReason::OTHER));
															#line 3874 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 874 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = yield(TaskPriority::DataDistribution);
															#line 874 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3881 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 1;
															#line 874 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromShardsActor, 0, Void >*>(static_cast<ResumeFromShardsActor*>(this)));
															#line 3886 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 802 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3908 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 802 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3917 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromShardsActor*>(this)->ActorCallback< ResumeFromShardsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromShardsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool traceShard;
															#line 785 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Key> customBoundaries;
															#line 799 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int shard;
															#line 800 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int customBoundary;
															#line 801 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int overreplicatedCount;
															#line 4009 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromShards()
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActor final : public Actor<Void>, public ActorCallback< ResumeFromShardsActor, 0, Void >, public FastAllocated<ResumeFromShardsActor>, public ResumeFromShardsActorState<ResumeFromShardsActor> {
															#line 4014 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromShardsActor>::operator new;
	using FastAllocated<ResumeFromShardsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14293783825270388736UL, 17005530747489984512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromShardsActor, 0, Void >;
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActor(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 4031 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromShardsActorState<ResumeFromShardsActor>(self, traceShard),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17921411212936075776UL, 15009917662088599808UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromShards");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromShardsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromShards( Reference<DataDistributor> const& self, bool const& traceShard ) {
															#line 770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromShardsActor(self, traceShard));
															#line 4063 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 878 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// TODO: unit test needed
																#line 4069 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromDataMoves()
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromDataMovesActor>
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActorState {
															#line 4075 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart),
															#line 881 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   it(self->initData->dataMoveMap.ranges().begin())
															#line 4086 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromDataMovesActorState() 
	{
		fdb_probe_actor_destroy("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 883 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 883 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4103 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 1;
															#line 883 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 0, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4108 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 885 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4131 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 885 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4140 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 947 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 4224 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 948 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->addActor.send(self->removeDataMoveTombstoneBackground(self));
															#line 4228 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 950 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromDataMovesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromDataMovesActorState(); static_cast<ResumeFromDataMovesActor*>(this)->destroy(); return 0; }
															#line 4232 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromDataMovesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 885 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(it != self->initData->dataMoveMap.ranges().end()))
															#line 4251 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 886 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DataMoveMetaData& meta = it.value()->meta;
															#line 887 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		DataMoveType dataMoveType = getDataMoveTypeFromDataMoveId(meta.id);
															#line 888 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.ranges.empty())
															#line 4261 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 889 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "EmptyDataMoveRange", self->ddId).detail("DataMoveMetaData", meta.toString());
															#line 4265 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1cont1continue1(loopDepth); // continue
		}
															#line 892 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.bulkLoadTaskState.present())
															#line 4270 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 893 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 894 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.dataMoveId = meta.id;
															#line 895 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.cancelled = true;
															#line 896 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->relocationProducer.send(rs);
															#line 899 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMove", self->ddId) .detail("Reason", "DDInit") .detail("DataMove", meta.toString());
															#line 4282 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		else
		{
															#line 902 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (dataMoveType == DataMoveType::LOGICAL_BULKLOAD || dataMoveType == DataMoveType::PHYSICAL_BULKLOAD)
															#line 4289 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 905 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 906 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.dataMoveId = meta.id;
															#line 907 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.cancelled = true;
															#line 908 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(rs);
															#line 909 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMoveForWrongType", self->ddId) .detail("Reason", "WrongTypeWhenDDInit") .detail("DataMoveType", dataMoveType) .detail("DataMove", meta.toString());
															#line 4301 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1cont1loopBody1cont5(loopDepth);
			}
			else
			{
															#line 913 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (it.value()->isCancelled() || (it.value()->valid && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA))
															#line 4308 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 915 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 916 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.dataMoveId = meta.id;
															#line 917 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.cancelled = true;
															#line 918 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->relocationProducer.send(rs);
															#line 919 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent("DDInitScheduledCancelDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 4320 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = a_body1cont1loopBody1cont7(loopDepth);
				}
				else
				{
															#line 920 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (it.value()->valid)
															#line 4327 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 921 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						TraceEvent(SevDebug, "DDInitFoundDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 922 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						ASSERT(meta.ranges.front() == it.range());
															#line 924 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 925 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMoveId = meta.id;
															#line 926 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMove = it.value();
															#line 927 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 928 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->primaryDest, true));
															#line 929 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (!rs.dataMove->remoteDest.empty())
															#line 4345 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 930 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
							teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->remoteDest, false));
															#line 4349 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						}
															#line 935 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->restartShardTracker.send(rs.keys);
															#line 936 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->defineShard(rs.keys);
															#line 940 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->moveShard(rs.keys, teams);
															#line 941 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->relocationProducer.send(rs);
															#line 942 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						StrictFuture<Void> __when_expr_1 = yield(TaskPriority::DataDistribution);
															#line 942 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4363 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
						static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 2;
															#line 942 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 1, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4368 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1cont1loopBody1cont9(loopDepth);
					}
				}
			}
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1continue1(int loopDepth) 
	{
															#line 885 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4398 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 885 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4407 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 881 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRangeMap<std::shared_ptr<DataMove>>::iterator it;
															#line 4523 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromDataMoves()
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActor final : public Actor<Void>, public ActorCallback< ResumeFromDataMovesActor, 0, Void >, public ActorCallback< ResumeFromDataMovesActor, 1, Void >, public FastAllocated<ResumeFromDataMovesActor>, public ResumeFromDataMovesActorState<ResumeFromDataMovesActor> {
															#line 4528 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromDataMovesActor>::operator new;
	using FastAllocated<ResumeFromDataMovesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13827890446419531776UL, 7577173214841465088UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromDataMovesActor, 0, Void >;
friend struct ActorCallback< ResumeFromDataMovesActor, 1, Void >;
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 4546 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromDataMovesActorState<ResumeFromDataMovesActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929780632836526848UL, 14642014520803479552UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromDataMoves");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromDataMoves( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromDataMovesActor(self, readyToStart));
															#line 4579 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 952 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Resume inflight relocations from the previous DD
	// TODO: The initialDataDistribution is unused once resumeRelocations,
	// DataDistributionTracker::trackInitialShards, and DDTeamCollection::init are done. In the future, we can release
	// the object to save memory usage if it turns out to be a problem.
	Future<Void> resumeRelocations() {
		ASSERT(shardsAffectedByTeamFailure); // has to be allocated
		Future<Void> shardsReady = resumeFromShards(Reference<DataDistributor>::addRef(this), g_network->isSimulated());
		return resumeFromDataMoves(Reference<DataDistributor>::addRef(this), shardsReady);
	}

	Future<Void> pollMoveKeysLock() const {
		return txnProcessor->pollMoveKeysLock(lock, context->ddEnabledState.get());
	}

	Future<bool> isDataDistributionEnabled() const {
		return txnProcessor->isDataDistributionEnabled(context->ddEnabledState.get());
	}

	Future<Void> removeKeysFromFailedServer(const UID& serverID, const std::vector<UID>& teamForDroppedRange) const {
		return txnProcessor->removeKeysFromFailedServer(
		    serverID, teamForDroppedRange, lock, context->ddEnabledState.get());
	}

	Future<Void> removeStorageServer(const UID& serverID, const Optional<UID>& tssPairID = Optional<UID>()) const {
		return txnProcessor->removeStorageServer(serverID, tssPairID, lock, context->ddEnabledState.get());
	}

	Future<Void> initDDConfigWatch();

	Future<Void> initTenantCache();
};

Future<Void> DataDistributor::initDDConfigWatch() {
	if (txnProcessor->isMocked()) {
		onConfigChange = Never();
		return Void();
	}
	onConfigChange = map(DDConfiguration().trigger.onChange(
	                         SystemDBWriteLockedNow(txnProcessor->context().getReference()), {}, configChangeWatching),
	                     [](Version v) {
		                     CODE_PROBE(true, "DataDistribution change detected");
		                     TraceEvent("DataDistributionConfigChanged").detail("ChangeVersion", v);
		                     throw dd_config_changed();
		                     return Void();
	                     });

	return success(configChangeWatching.getFuture());
}

Future<Void> DataDistributor::initTenantCache() {
	// SOMEDAY: support tenant cache in MockDD
	ASSERT(!txnProcessor->isMocked());
	ddTenantCache = makeReference<TenantCache>(txnProcessor->context(), ddId);
	return ddTenantCache.get()->build();
}

inline void addAuditToAuditMap(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit) {
	AuditType auditType = audit->coreState.getType();
	UID auditID = audit->coreState.id;
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "addAuditToAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(!self->audits[auditType].contains(auditID));
	self->audits[auditType][auditID] = audit;
	return;
}

inline std::shared_ptr<DDAudit> getAuditFromAuditMap(Reference<DataDistributor> self,
                                                     AuditType auditType,
                                                     UID auditID) {
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "getAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	return self->audits[auditType][auditID];
}

inline void removeAuditFromAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	std::shared_ptr<DDAudit> audit = self->audits[auditType][auditID];
	audit->cancel();
	self->audits[auditType].erase(auditID);
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "removeAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	return;
}

inline bool auditExistInAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	return self->audits.contains(auditType) && self->audits[auditType].contains(auditID);
}

inline bool existAuditInAuditMapForType(Reference<DataDistributor> self, AuditType auditType) {
	return self->audits.contains(auditType) && !self->audits[auditType].empty();
}

inline std::unordered_map<UID, std::shared_ptr<DDAudit>> getAuditsForType(Reference<DataDistributor> self,
                                                                          AuditType auditType) {
	ASSERT(self->audits.contains(auditType));
	return self->audits[auditType];
}

// Periodically check and log the physicalShard status; clean up empty physicalShard;
															#line 4690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MonitorPhysicalShardStatusActor>
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActorState {
															#line 4697 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActorState(Reference<PhysicalShardCollection> const& self) 
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 4704 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	~MonitorPhysicalShardStatusActorState() 
	{
		fdb_probe_actor_destroy("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1060 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA);
															#line 1061 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD);
															#line 1062 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 4723 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorPhysicalShardStatusActorState();
		static_cast<MonitorPhysicalShardStatusActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1063 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->cleanUpPhysicalShardCollection();
															#line 1064 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->logPhysicalShardCollection();
															#line 1065 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->PHYSICAL_SHARD_METRICS_DELAY);
															#line 1065 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4759 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 1;
															#line 1065 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*>(static_cast<MonitorPhysicalShardStatusActor*>(this)));
															#line 4764 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state > 0) static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorPhysicalShardStatusActor*>(this)->ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<PhysicalShardCollection> self;
															#line 4858 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActor final : public Actor<Void>, public ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >, public FastAllocated<MonitorPhysicalShardStatusActor>, public MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor> {
															#line 4863 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator new;
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14369291547690294016UL, 5970906808210286080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >;
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActor(Reference<PhysicalShardCollection> const& self) 
															#line 4880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8017156993063803648UL, 8326163650824365056UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorPhysicalShardStatus");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> monitorPhysicalShardStatus( Reference<PhysicalShardCollection> const& self ) {
															#line 1059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MonitorPhysicalShardStatusActor(self));
															#line 4913 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1068 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Trigger a task on range based on the current bulk load task metadata
															#line 4919 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class TriggerBulkLoadTaskActor>
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActorState {
															#line 4926 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId)
															#line 4937 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~TriggerBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1073 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 4952 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1074 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Database cx = self->txnProcessor->context();
															#line 1075 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr = Transaction(cx);
															#line 1076 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState = BulkLoadTaskState();
															#line 4986 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 1078 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1079 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1080 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1080 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4996 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1080 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5001 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1106 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 5023 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1107 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistTriggerStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 5027 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1112 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1112 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5033 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1112 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5038 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1081 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1082 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1082 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5057 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1082 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5062 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1081 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1082 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1082 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5075 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1082 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5080 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1088 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1089 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1090 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1091 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1092 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1092 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5174 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1092 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5179 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1088 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1089 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1090 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1091 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1092 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1092 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5198 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1092 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5203 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1096 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1096 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5289 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1096 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5294 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1096 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1096 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5305 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1096 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5310 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1097 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1102 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1103 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 5400 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1097 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1102 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1103 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 5418 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1075 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1076 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState newBulkLoadTaskState;
															#line 5598 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActor final : public Actor<std::pair<BulkLoadTaskState, Version>>, public ActorCallback< TriggerBulkLoadTaskActor, 0, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 1, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 2, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 3, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 4, Void >, public FastAllocated<TriggerBulkLoadTaskActor>, public TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor> {
															#line 5603 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<TriggerBulkLoadTaskActor>::operator new;
	using FastAllocated<TriggerBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18149212826064514560UL, 3787090545169753600UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::pair<BulkLoadTaskState, Version>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TriggerBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 4, Void >;
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 5624 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<std::pair<BulkLoadTaskState, Version>>(),
		   TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9998265325154424320UL, 14265371869313678848UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("triggerBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<std::pair<BulkLoadTaskState, Version>> triggerBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1070 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<std::pair<BulkLoadTaskState, Version>>(new TriggerBulkLoadTaskActor(self, taskRange, taskId));
															#line 5661 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1116 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// TODO(BulkLoad): add reason to persist
															#line 5667 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via failBulkLoadTask()
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FailBulkLoadTaskActor>
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActorState {
															#line 5674 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cancelledDataMovePriority(cancelledDataMovePriority),
															#line 1122 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1123 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1124 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState()
															#line 5693 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~FailBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1125 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 5708 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1155 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FailBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FailBulkLoadTaskActorState(); static_cast<FailBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 5731 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FailBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1127 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1128 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1129 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1129 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5757 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1129 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 0, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5762 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1146 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 5797 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1147 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistErrorStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 5801 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1152 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1152 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5807 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1152 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 4, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5812 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1130 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1130 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5829 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1130 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5834 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1130 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1130 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5845 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1130 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5850 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1132 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1133 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1134 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1135 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1136 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1136 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5944 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1136 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5949 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1132 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1133 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1134 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1135 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1136 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1136 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1136 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5973 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1138 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1138 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6059 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1138 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6064 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1138 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1138 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6075 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1138 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6080 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1139 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1140 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6166 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1139 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1140 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6177 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int cancelledDataMovePriority;
															#line 1122 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1123 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1124 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 6358 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via failBulkLoadTask()
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< FailBulkLoadTaskActor, 0, Void >, public ActorCallback< FailBulkLoadTaskActor, 1, Void >, public ActorCallback< FailBulkLoadTaskActor, 2, Void >, public ActorCallback< FailBulkLoadTaskActor, 3, Void >, public ActorCallback< FailBulkLoadTaskActor, 4, Void >, public FastAllocated<FailBulkLoadTaskActor>, public FailBulkLoadTaskActorState<FailBulkLoadTaskActor> {
															#line 6363 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FailBulkLoadTaskActor>::operator new;
	using FastAllocated<FailBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5223403940077872128UL, 5383740727703489280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FailBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 4, Void >;
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 6384 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FailBulkLoadTaskActorState<FailBulkLoadTaskActor>(self, taskRange, taskId, cancelledDataMovePriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5276853651913060352UL, 273265509463260672UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("failBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> failBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId, int const& cancelledDataMovePriority ) {
															#line 1118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FailBulkLoadTaskActor(self, taskRange, taskId, cancelledDataMovePriority));
															#line 6421 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1157 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// A bulk load task is guaranteed to be either complete or overwritten by another task
															#line 6427 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via doBulkLoadTask()
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DoBulkLoadTaskActor>
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActorState {
															#line 6434 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1160 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   completeAck(),
															#line 1161 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   triggeredBulkLoadTask(),
															#line 1162 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   commitVersion(invalidVersion),
															#line 1163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 6453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~DoBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1164 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadEngineParallelismLimitor.canStart());
															#line 1165 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadEngineParallelismLimitor.incrementTaskCounter();
															#line 6470 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1168 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<std::pair<BulkLoadTaskState, Version>> __when_expr_0 = triggerBulkLoadTask(self, range, taskId);
															#line 1168 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6476 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1168 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 6481 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1246 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadEngineParallelismLimitor.decrementTaskCounter();
															#line 1247 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DoBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBulkLoadTaskActorState(); static_cast<DoBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6512 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DoBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1232 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 6525 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1233 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6529 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1235 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(e) .detail("Phase", "Error") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1241 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 6535 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1242 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6539 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1169 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1170 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1171 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1178 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1184 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1189 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6569 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 6574 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1169 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1170 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1171 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1178 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1184 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1189 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6597 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 6602 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(triggeredBulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(triggeredBulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1191 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ack.unretryableError)
															#line 6686 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1192 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "See unretryable error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 6690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1200 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_2 = failBulkLoadTask(self, range, taskId, ack.dataMovePriority);
															#line 1200 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 6696 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1200 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 2, Void >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 6701 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont3Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 1225 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Task complete") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 6714 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck const& __ack,int loopDepth) 
	{
															#line 1190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ack = __ack;
															#line 6724 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck && __ack,int loopDepth) 
	{
		ack = std::move(__ack);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& failTaskError,int loopDepth=0) 
	{
		try {
															#line 1208 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_actor_cancelled)
															#line 6816 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1209 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 6820 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1211 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(failTaskError) .detail("Phase", "Failed to mark task error. Error should be bulkload_task_outdated") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1218 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_movekeys_conflict)
															#line 6826 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1219 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 6830 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1221 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(failTaskError.code() == error_code_bulkload_task_outdated);
															#line 6834 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 1201 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 6849 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 1201 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 6858 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1160 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<BulkLoadAck> completeAck;
															#line 1161 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState triggeredBulkLoadTask;
															#line 1162 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Version commitVersion;
															#line 1163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadAck ack;
															#line 6980 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via doBulkLoadTask()
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >, public ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >, public ActorCallback< DoBulkLoadTaskActor, 2, Void >, public FastAllocated<DoBulkLoadTaskActor>, public DoBulkLoadTaskActorState<DoBulkLoadTaskActor> {
															#line 6985 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DoBulkLoadTaskActor>::operator new;
	using FastAllocated<DoBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(153084964318861568UL, 4062044265424883200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >;
friend struct ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >;
friend struct ActorCallback< DoBulkLoadTaskActor, 2, Void >;
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 7004 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DoBulkLoadTaskActorState<DoBulkLoadTaskActor>(self, range, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9618644142710713856UL, 6256719351634756608UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> doBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& taskId ) {
															#line 1159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DoBulkLoadTaskActor(self, range, taskId));
															#line 7039 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1249 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 7044 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class EraseBulkLoadTaskActor>
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActorState {
															#line 7051 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1251 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1252 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1253 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask()
															#line 7068 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~EraseBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1254 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 7083 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1294 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<EraseBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EraseBulkLoadTaskActorState(); static_cast<EraseBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 7106 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<EraseBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1256 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1257 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1258 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(bulkLoadTask, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Acknowledged }));
															#line 1258 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7132 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1258 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 0, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7137 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1281 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 7172 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1282 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskEraseStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7176 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1287 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_bulkload_task_outdated)
															#line 7180 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1291 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1291 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7188 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1291 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 3, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7193 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1259 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1259 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7210 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1259 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7215 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1259 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1259 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7226 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1259 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7231 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1261 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1261 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1261 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7322 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1261 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1261 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7333 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1261 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7338 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1262 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1263 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1267 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1268 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1269 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 7430 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1273 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1274 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7436 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1262 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1263 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1267 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1268 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1269 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 7454 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1273 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1274 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7460 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1251 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1252 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1253 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 7640 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< EraseBulkLoadTaskActor, 0, Void >, public ActorCallback< EraseBulkLoadTaskActor, 1, Void >, public ActorCallback< EraseBulkLoadTaskActor, 2, Void >, public ActorCallback< EraseBulkLoadTaskActor, 3, Void >, public FastAllocated<EraseBulkLoadTaskActor>, public EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor> {
															#line 7645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<EraseBulkLoadTaskActor>::operator new;
	using FastAllocated<EraseBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6115671476778857472UL, 17647158324803002624UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< EraseBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 3, Void >;
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 7665 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(445300323391651072UL, 14768580555042305792UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("eraseBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> eraseBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1250 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new EraseBulkLoadTaskActor(self, taskRange, taskId));
															#line 7701 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1296 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 7706 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadTasksActor>
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActorState {
															#line 7713 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActorState(Reference<DataDistributor> const& self) 
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1298 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginKey(normalKeys.begin),
															#line 1299 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   endKey(normalKeys.end),
															#line 1300 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeToRead(),
															#line 1301 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1302 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1303 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   i(0),
															#line 1304 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1305 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   result(),
															#line 1306 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadActors(),
															#line 1307 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range()
															#line 7740 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadTasksActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1308 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 7755 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1369 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = waitForAll(bulkLoadActors);
															#line 1369 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7780 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 7;
															#line 1369 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 7785 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1308 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 7801 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1310 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1311 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			result.clear();
															#line 1312 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store( result, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead, SERVER_KNOBS->DD_BULKLOAD_TASK_METADATA_READ_SIZE));
															#line 1312 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7814 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 1;
															#line 1312 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 7819 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1363 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 7854 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1364 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 7858 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1366 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = tr.onError(e);
															#line 1366 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_5.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 6;
															#line 1366 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 7869 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1315 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1316 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 7886 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1315 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1316 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 7897 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1361 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = result.back().key;
															#line 7981 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1316 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < result.size() - 1))
															#line 7997 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1317 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (result[i].value.empty())
															#line 8003 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3continue1(loopDepth); // continue
		}
															#line 1320 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		range = Standalone(KeyRangeRef(result[i].key, result[i + 1].key));
															#line 1321 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[i].value);
															#line 1322 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 8013 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1323 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 1323 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8019 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 2;
															#line 1323 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8024 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3continue1(int loopDepth) 
	{
															#line 1316 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 8051 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1326 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (range != bulkLoadTaskState.getRange())
															#line 8060 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(0.1);
															#line 1328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8066 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 3;
															#line 1328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8071 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont6(int loopDepth) 
	{
															#line 1331 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.phase == BulkLoadPhase::Submitted || bulkLoadTaskState.phase == BulkLoadPhase::Triggered || bulkLoadTaskState.phase == BulkLoadPhase::Running)
															#line 8172 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1335 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 8176 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont3loopBody1cont6loopHead1(loopDepth);
		}
		else
		{
															#line 1347 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadTaskState.phase == BulkLoadPhase::Acknowledged)
															#line 8183 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1348 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskClearMetadata", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 1352 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_4 = eraseBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId());
															#line 1352 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8191 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
				static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 5;
															#line 1352 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8196 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1353 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (bulkLoadTaskState.phase == BulkLoadPhase::Error)
															#line 8203 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1354 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevWarnAlways, "DDBulkLoadTaskUnretriableError", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 8207 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 1358 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(bulkLoadTaskState.phase == BulkLoadPhase::Complete);
															#line 8213 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont3loopBody1cont9(int loopDepth) 
	{
															#line 1316 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 8312 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont10(int loopDepth) 
	{
															#line 1341 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSchedule", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("Phase", bulkLoadTaskState.phase);
															#line 1345 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadActors.push_back( doBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId()));
															#line 8323 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1(int loopDepth) 
	{
															#line 1336 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadEngineParallelismLimitor.canStart())
															#line 8339 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3loopBody1cont6break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1339 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->bulkLoadEngineParallelismLimitor.waitUntilCounterChanged();
															#line 1339 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8347 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont6loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 4;
															#line 1339 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8352 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont3loopBody1cont11(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1370 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 8654 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1370 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 8666 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1298 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1299 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1300 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1301 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1302 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1303 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1304 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1305 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult result;
															#line 1306 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> bulkLoadActors;
															#line 1307 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 8771 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >, public FastAllocated<ScheduleBulkLoadTasksActor>, public ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor> {
															#line 8776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator new;
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9318979424182662912UL, 11639249398714609152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >;
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActor(Reference<DataDistributor> const& self) 
															#line 8799 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14856297450692458752UL, 7460955458194361088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadTasks");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self ) {
															#line 1297 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadTasksActor(self));
															#line 8838 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1372 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 8843 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadTaskCoreActor>
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActorState {
															#line 8850 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart)
															#line 8859 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadTaskCoreActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1374 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 1374 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8876 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 1;
															#line 1374 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 0, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 8881 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadTaskCoreActorState();
		static_cast<BulkLoadTaskCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1375 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1376 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1376 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8908 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1376 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 8913 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1375 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1376 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1376 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8926 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1376 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 8931 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1377 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1378 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9017 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1377 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1378 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9028 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
		try {
															#line 1380 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = scheduleBulkLoadTasks(self);
															#line 1380 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9122 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 3;
															#line 1380 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 2, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9127 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1390 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(SERVER_KNOBS->DD_BULKLOAD_SCHEDULE_MIN_INTERVAL_SEC);
															#line 1390 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9144 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 4;
															#line 1390 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 3, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9149 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1382 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 9159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1383 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1385 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadTaskCoreError", self->ddId).errorUnsuppressed(e);
															#line 1386 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 9169 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1387 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9173 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 1375 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 9378 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActor final : public Actor<Void>, public ActorCallback< BulkLoadTaskCoreActor, 0, Void >, public ActorCallback< BulkLoadTaskCoreActor, 1, Void >, public ActorCallback< BulkLoadTaskCoreActor, 2, Void >, public ActorCallback< BulkLoadTaskCoreActor, 3, Void >, public FastAllocated<BulkLoadTaskCoreActor>, public BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor> {
															#line 9383 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadTaskCoreActor>::operator new;
	using FastAllocated<BulkLoadTaskCoreActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4238962634128586240UL, 14180994814442334720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadTaskCoreActor, 0, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 1, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 2, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 3, Void >;
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 9403 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12210363843676168704UL, 8182787864321598208UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadTaskCore");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadTaskCore( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 1373 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadTaskCoreActor(self, readyToStart));
															#line 9439 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1393 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

void explainBulkLoadJobGetRangeResult(const RangeResult& rangeResult) {
	TraceEvent(SevInfo, "DDBulkLoadJobExplainRangeResult").detail("Size", rangeResult.size());
	for (int i = 0; i < rangeResult.size(); i++) {
		TraceEvent e(SevInfo, "DDBulkLoadJobExplainRangeResultKV");
		e.detail("Key", rangeResult[i].key);
		if (rangeResult[i].value.empty()) {
			e.detail("Val", "empty");
		} else {
			BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[i].value);
			e.detail("Val", currentJobState.toString());
		}
	}
}

// Return the current bulk load job state for the given jobId and jobRange
// If the job is not found or outdated, throw bulkload_task_outdated error
															#line 9460 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via getBulkLoadJob()
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class GetBulkLoadJobActor>
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActorState {
															#line 9467 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActorState(Transaction* const& tr,UID const& jobId,KeyRange const& jobRange) 
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : tr(tr),
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRange(jobRange),
															#line 1411 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeResult(),
															#line 1412 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorMessage()
															#line 9482 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~GetBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1414 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(rangeResult, krmGetRanges(tr, bulkLoadJobPrefix, jobRange));
															#line 1414 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9500 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1414 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetBulkLoadJobActor, 0, Void >*>(static_cast<GetBulkLoadJobActor*>(this)));
															#line 9505 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1441 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 9535 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1442 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 9539 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1444 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (g_network->isSimulated() && e.code() == error_code_bulkload_task_outdated)
															#line 9543 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1445 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				explainBulkLoadJobGetRangeResult(rangeResult);
															#line 9547 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1447 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobGetError", jobId).errorUnsuppressed(e).detail("Message", errorMessage);
															#line 1448 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 9553 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1415 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult.size() != 2 && rangeResult.size() != 3)
															#line 9567 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1417 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("Expected 2 or 3 results for bulk load job {} in range {}, but got {} results", jobId.toString(), jobRange.toString(), std::to_string(rangeResult.size()));
															#line 1421 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9573 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1423 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 9577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1424 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("No bulk load job found in range {}", jobRange.toString());
															#line 1425 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9583 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1427 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1428 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 9589 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1429 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("No valid bulk load job in range {}", jobRange.toString());
															#line 1430 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9595 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1432 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId || currentJobState.getJobRange() != jobRange)
															#line 9599 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1433 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("Bulk load job {} in range {} is outdated, expected jobId {}", currentJobState.getJobId().toString(), jobRange.toString(), jobId.toString());
															#line 1437 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9605 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1439 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 9609 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1415 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult.size() != 2 && rangeResult.size() != 3)
															#line 9621 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1417 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("Expected 2 or 3 results for bulk load job {} in range {}, but got {} results", jobId.toString(), jobRange.toString(), std::to_string(rangeResult.size()));
															#line 1421 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9627 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1423 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 9631 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1424 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("No bulk load job found in range {}", jobRange.toString());
															#line 1425 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9637 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1427 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1428 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 9643 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1429 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("No valid bulk load job in range {}", jobRange.toString());
															#line 1430 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9649 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1432 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId || currentJobState.getJobRange() != jobRange)
															#line 9653 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1433 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("Bulk load job {} in range {} is outdated, expected jobId {}", currentJobState.getJobId().toString(), jobRange.toString(), jobId.toString());
															#line 1437 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9659 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1439 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 9663 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<GetBulkLoadJobActor*>(this)->ActorCallback< GetBulkLoadJobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction* tr;
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1411 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult rangeResult;
															#line 1412 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string errorMessage;
															#line 9756 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via getBulkLoadJob()
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActor final : public Actor<BulkLoadJobState>, public ActorCallback< GetBulkLoadJobActor, 0, Void >, public FastAllocated<GetBulkLoadJobActor>, public GetBulkLoadJobActorState<GetBulkLoadJobActor> {
															#line 9761 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<GetBulkLoadJobActor>::operator new;
	using FastAllocated<GetBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14858018564646913280UL, 12692252306853982976UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadJobState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetBulkLoadJobActor, 0, Void >;
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActor(Transaction* const& tr,UID const& jobId,KeyRange const& jobRange) 
															#line 9778 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadJobState>(),
		   GetBulkLoadJobActorState<GetBulkLoadJobActor>(tr, jobId, jobRange),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17307579102949951488UL, 17075247084345564672UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetBulkLoadJobActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadJobState> getBulkLoadJob( Transaction* const& tr, UID const& jobId, KeyRange const& jobRange ) {
															#line 1410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadJobState>(new GetBulkLoadJobActor(tr, jobId, jobRange));
															#line 9811 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1451 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Find task metadata for a bulk load job with jobId and input range
															#line 9817 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobFindTaskActor>
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActorState {
															#line 9824 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,KeyRange const& jobRange,UID const& logId) 
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRange(jobRange),
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   logId(logId),
															#line 1458 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1459 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1460 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 9845 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobFindTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1461 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 9860 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1491 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(bulkLoadTaskState); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 9883 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(std::move(bulkLoadTaskState)); // state_var_RVO
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1463 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1464 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobId, jobRange);
															#line 1464 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9907 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 1;
															#line 1464 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 9912 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1488 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1488 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9949 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 3;
															#line 1488 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 9954 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(BulkLoadJobState const& checkJobState,int loopDepth) 
	{
															#line 1466 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1467 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1467 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9973 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1467 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 9978 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(BulkLoadJobState && checkJobState,int loopDepth) 
	{
															#line 1466 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1467 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1467 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9991 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1467 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 9996 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& checkJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(checkJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && checkJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(checkJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(RangeResult const& result,int loopDepth) 
	{
															#line 1469 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1470 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1471 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 10084 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1472 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 10088 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1474 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1475 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1476 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 10100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1477 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1484 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 10106 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3(RangeResult && result,int loopDepth) 
	{
															#line 1469 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1470 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1471 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 10120 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1472 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 10124 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1474 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1475 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1476 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 10136 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1477 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1484 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 10142 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(result, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID logId;
															#line 1458 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1459 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1460 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 10326 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActor final : public Actor<Optional<BulkLoadTaskState>>, public ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >, public ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >, public ActorCallback< BulkLoadJobFindTaskActor, 2, Void >, public FastAllocated<BulkLoadJobFindTaskActor>, public BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor> {
															#line 10331 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobFindTaskActor>::operator new;
	using FastAllocated<BulkLoadJobFindTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12961909473727699968UL, 343810888892950528UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<BulkLoadTaskState>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 2, Void >;
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,KeyRange const& jobRange,UID const& logId) 
															#line 10350 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Optional<BulkLoadTaskState>>(),
		   BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor>(self, range, jobId, jobRange, logId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3995601487305067776UL, 1783749224615820032UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobFindTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Optional<BulkLoadTaskState>> bulkLoadJobFindTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& jobId, KeyRange const& jobRange, UID const& logId ) {
															#line 1453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Optional<BulkLoadTaskState>>(new BulkLoadJobFindTaskActor(self, range, jobId, jobRange, logId));
															#line 10385 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1493 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Submit a bulkload task for the given jobId
															#line 10391 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobSubmitTaskActor>
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActorState {
															#line 10398 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests,KeyRange const& taskRange) 
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(manifests),
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1499 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1500 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1502 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(jobId, manifests, taskRange)
															#line 10417 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobSubmitTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1503 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 10432 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1523 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV<BulkLoadTaskState>::futures) { (void)(bulkLoadTask); this->~BulkLoadJobSubmitTaskActorState(); static_cast<BulkLoadJobSubmitTaskActor*>(this)->destroy(); return 0; }
															#line 10455 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV< BulkLoadTaskState >::value()) BulkLoadTaskState(std::move(bulkLoadTask)); // state_var_RVO
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1506 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1506 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10477 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 1;
															#line 1506 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10482 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1520 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1520 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10519 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 4;
															#line 1520 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10524 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1507 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1507 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10541 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1507 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10546 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1507 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1507 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10557 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1507 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10562 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1509 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1509 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10648 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1509 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10653 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1509 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1509 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10664 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1509 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10669 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1510 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1511 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 10755 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1510 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1511 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 10766 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1499 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1500 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1502 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 10947 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActor final : public Actor<BulkLoadTaskState>, public ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >, public FastAllocated<BulkLoadJobSubmitTaskActor>, public BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor> {
															#line 10952 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator new;
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4729090124343168000UL, 7828571502575976448UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadTaskState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >;
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests,KeyRange const& taskRange) 
															#line 10972 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadTaskState>(),
		   BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor>(self, jobId, manifests, taskRange),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9698732467664213248UL, 5402256740808584704UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobSubmitTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadTaskState> bulkLoadJobSubmitTask( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadManifestSet const& manifests, KeyRange const& taskRange ) {
															#line 1495 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadTaskState>(new BulkLoadJobSubmitTaskActor(self, jobId, manifests, taskRange));
															#line 11008 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1525 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 11013 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobWaitUntilTaskCompleteOrErrorActor>
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActorState {
															#line 11020 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(bulkLoadTask)
															#line 11031 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1529 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(bulkLoadTask.isValid());
															#line 1530 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1531 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1532 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentTask = BulkLoadTaskState();
															#line 1533 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 11054 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1535 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentTask, getBulkLoadTask(&tr, bulkLoadTask.getRange(), bulkLoadTask.getTaskId(), { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running, BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }));
															#line 1535 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11087 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1535 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 11092 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1562 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_JOB_MONITOR_PERIOD_SEC);
															#line 1562 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11109 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 1562 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 11114 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1560 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1560 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11126 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1560 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 11131 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1545 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 11146 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1546 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 11150 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1548 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 11154 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1549 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretryableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1554 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 11160 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1556 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 11168 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1557 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 11172 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1545 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 11186 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1546 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 11190 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1548 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 11194 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1549 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretryableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1554 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 11200 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1556 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 11208 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1557 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 11212 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1530 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1531 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1532 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState currentTask;
															#line 11496 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActor final : public Actor<Void>, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>, public BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor> {
															#line 11501 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(970719442108548864UL, 15716972367471270912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >;
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 11520 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>(self, jobId, bulkLoadTask),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16093638368250768128UL, 12615896230081500416UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobWaitUntilTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobWaitUntilTaskCompleteOrError( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadTaskState const& bulkLoadTask ) {
															#line 1526 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobWaitUntilTaskCompleteOrErrorActor(self, jobId, bulkLoadTask));
															#line 11555 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1565 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// A bulkload job can contain multiple tasks. Each task can contain multiple manifests.
// Given a job range, the bulkload task range is defined as the range between the min begin key and the max end key of
// all manifests, overlapping with the maxRange (i.e. the job range).
KeyRange generateBulkLoadTaskRange(const BulkLoadManifestSet& manifests, const KeyRange& maxRange) {
	KeyRange manifestsRange = Standalone(KeyRangeRef(manifests.getMinBeginKey(), manifests.getMaxEndKey()));
	return manifestsRange & maxRange; // ensure the task range is within the maxRange
}

// Given a list of manifestEntry, create a bulkload task.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 11571 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobNewTaskActor>
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActorState {
															#line 11578 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,KeyRange const& jobRange,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRoot(jobRoot),
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRange(jobRange),
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestLocalTempFolder(manifestLocalTempFolder),
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntries(manifestEntries),
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1585 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1586 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1587 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(),
															#line 1588 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now()),
															#line 1589 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange()
															#line 11609 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobNewTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1590 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!manifestEntries.empty());
															#line 11624 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1593 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(manifests, getBulkLoadManifestMetadataFromEntry( manifestEntries, manifestLocalTempFolder, jobTransportMethod, jobRoot, self->ddId));
															#line 1593 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11630 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 1;
															#line 1593 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11635 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1650 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 11664 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1634 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 11677 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1635 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 11681 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1637 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("JobRange", jobRange) .detail("Duration", now() - beginTime);
															#line 1644 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 11687 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1645 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 11691 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1596 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		taskRange = generateBulkLoadTaskRange(manifests, jobRange);
															#line 1600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask(self, taskRange, jobId, jobRange, self->ddId);
															#line 1600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11711 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1596 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		taskRange = generateBulkLoadTaskRange(manifests, jobRange);
															#line 1600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask(self, taskRange, jobId, jobRange, self->ddId);
															#line 1600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11729 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1600 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11734 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1602 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 11818 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1604 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 11822 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1611 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1618 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests, taskRange));
															#line 1618 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11834 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1618 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11839 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1602 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 11848 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1604 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 11852 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1611 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1618 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests, taskRange));
															#line 1618 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1618 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11869 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1620 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("JobRange", jobRange) .detail("Duration", now() - beginTime);
															#line 1628 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 11955 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1630 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1631 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 11961 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1620 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("JobRange", jobRange) .detail("Duration", now() - beginTime);
															#line 1628 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 11973 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1630 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1631 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 11979 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string jobRoot;
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string manifestLocalTempFolder;
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1585 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1586 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1587 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1588 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1589 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 12099 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobNewTaskActor, 0, Void >, public ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobNewTaskActor, 2, Void >, public FastAllocated<BulkLoadJobNewTaskActor>, public BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor> {
															#line 12104 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobNewTaskActor>::operator new;
	using FastAllocated<BulkLoadJobNewTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5951069460245211904UL, 15197391359679875584UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobNewTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 2, Void >;
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActor(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,KeyRange const& jobRange,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 12123 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor>(self, jobId, jobRoot, jobRange, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8655289167937893888UL, 2026549218517418240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobNewTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobNewTask( Reference<DataDistributor> const& self, UID const& jobId, std::string const& jobRoot, KeyRange const& jobRange, BulkLoadTransportMethod const& jobTransportMethod, std::string const& manifestLocalTempFolder, std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries, Promise<Void> const& errorOut ) {
															#line 1577 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobNewTaskActor(self, jobId, jobRoot, jobRange, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut));
															#line 12158 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1652 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Given a bulkload task range, find the task and wait until the task is complete or error.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 12166 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobMonitorTaskActor>
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActorState {
															#line 12173 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& jobRange,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRange(jobRange),
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1661 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1662 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1663 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 12194 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobMonitorTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1664 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadParallelismLimitor.canStart());
															#line 1665 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.incrementTaskCounter();
															#line 12211 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Optional<BulkLoadTaskState>> __when_expr_0 = bulkLoadJobFindTask(self, taskRange, jobId, jobRange, self->ddId);
															#line 1668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12217 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 1;
															#line 1668 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 12222 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1713 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 12251 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1698 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 12264 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1699 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 12268 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1701 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskMonitorError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1706 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1707 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 12276 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1708 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 12280 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1670 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 12296 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1672 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1673 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 12302 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1675 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1676 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1683 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12314 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1685 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1686 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12320 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12326 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 12331 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1670 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 12340 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1672 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1673 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 12346 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1675 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1676 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1683 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12358 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1685 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1686 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12364 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12370 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 12375 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1691 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1696 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 12461 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1691 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1696 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 12472 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1661 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1662 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1663 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 12581 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >, public FastAllocated<BulkLoadJobMonitorTaskActor>, public BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor> {
															#line 12586 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator new;
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1362678644441450240UL, 8035647473974000896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >;
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActor(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& jobRange,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 12604 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor>(self, jobId, jobRange, taskRange, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3763012194873441280UL, 18076434267272298240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobMonitorTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobMonitorTask( Reference<DataDistributor> const& self, UID const& jobId, KeyRange const& jobRange, KeyRange const& taskRange, Promise<Void> const& errorOut ) {
															#line 1656 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobMonitorTaskActor(self, jobId, jobRange, taskRange, errorOut));
															#line 12638 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1715 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 12643 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class PersistBulkLoadJobTaskCountActor>
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActorState {
															#line 12650 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActorState(Reference<DataDistributor> const& self) 
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1717 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1718 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 12661 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	~PersistBulkLoadJobTaskCountActorState() 
	{
		fdb_probe_actor_destroy("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1719 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1720 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1721 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = jobState.getJobId();
															#line 1722 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = jobState.getJobRange();
															#line 1723 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(jobState.getTaskCount().present());
															#line 1724 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			taskCount = jobState.getTaskCount().get();
															#line 1725 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentJobState = BulkLoadJobState();
															#line 1726 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 12690 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1755 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 12713 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1728 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentJobState, getBulkLoadJob(&tr, jobId, jobRange));
															#line 1728 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12735 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 1;
															#line 1728 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12740 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1752 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1752 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 4;
															#line 1752 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12782 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1729 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 12797 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1730 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 12801 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1731 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 12807 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1739 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 12811 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1741 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1742 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1742 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12823 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1742 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12828 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1729 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 12837 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1730 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 12841 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1731 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1737 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 12847 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1739 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 12851 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1741 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1742 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1742 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12863 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1742 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12868 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1743 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1743 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12954 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1743 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12959 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1743 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1743 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12970 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1743 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12975 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1744 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1745 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 13061 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1744 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1745 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 13072 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1717 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1718 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1720 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1721 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1722 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1724 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	uint64_t taskCount;
															#line 1725 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 13255 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActor final : public Actor<Void>, public ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >, public FastAllocated<PersistBulkLoadJobTaskCountActor>, public PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor> {
															#line 13260 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator new;
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15986110780134494464UL, 5569169756173766144UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >;
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActor(Reference<DataDistributor> const& self) 
															#line 13280 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10651365252717723648UL, 4800485421425282816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistBulkLoadJobTaskCount");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> persistBulkLoadJobTaskCount( Reference<DataDistributor> const& self ) {
															#line 1716 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new PersistBulkLoadJobTaskCountActor(self));
															#line 13316 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1757 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Remove the bulkload job metadata from the range map metadata.
// Add the job metadata to the history metadata with the error message.
															#line 13323 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MoveErrorBulkLoadJobToHistoryActor>
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActorState {
															#line 13330 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActorState(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorMessage(errorMessage),
															#line 1761 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1762 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1763 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentJobState()
															#line 13345 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	~MoveErrorBulkLoadJobToHistoryActorState() 
	{
		fdb_probe_actor_destroy("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1764 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1765 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = self->bulkLoadJobManager.get().jobState.getJobId();
															#line 1766 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = self->bulkLoadJobManager.get().jobState.getJobRange();
															#line 1767 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 13366 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1783 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveErrorBulkLoadJobToHistoryActorState(); static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->destroy(); return 0; }
															#line 13389 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1769 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1769 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13411 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 1;
															#line 1769 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13416 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1780 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_6 = tr.onError(e);
															#line 1780 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13453 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_6.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 7;
															#line 1780 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13458 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId, jobRange));
															#line 1770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13475 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13480 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId, jobRange));
															#line 1770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13491 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1770 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13496 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1771 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1771 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13582 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1771 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13587 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1771 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1771 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13598 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1771 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13603 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1773 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1774 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1775 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1775 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13693 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1775 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13698 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1773 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1774 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1775 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1775 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13713 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1775 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13718 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = releaseExclusiveReadLockOnRange(&tr, jobRange, rangeLockNameForBulkLoad);
															#line 1776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13804 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13809 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = releaseExclusiveReadLockOnRange(&tr, jobRange, rangeLockNameForBulkLoad);
															#line 1776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13820 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1776 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13825 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = tr.commit();
															#line 1777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13911 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_5.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 6;
															#line 1777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13916 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = tr.commit();
															#line 1777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13927 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_5.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 6;
															#line 1777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13932 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string errorMessage;
															#line 1761 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1762 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1763 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 1765 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1766 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 14200 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActor final : public Actor<Void>, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >, public FastAllocated<MoveErrorBulkLoadJobToHistoryActor>, public MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor> {
															#line 14205 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator new;
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13709090341972565760UL, 10260222719689235456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >;
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActor(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 14228 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor>(self, errorMessage),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10076677940119608320UL, 17212560479686694656UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("moveErrorBulkLoadJobToHistory");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> moveErrorBulkLoadJobToHistory( Reference<DataDistributor> const& self, std::string const& errorMessage ) {
															#line 1760 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MoveErrorBulkLoadJobToHistoryActor(self, errorMessage));
															#line 14267 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1785 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Download the job manifest file from the remoteJobManifestFilePath to the localJobManifestFilePath.
// Build the bulkload manifest range map based on the localJobManifestFilePath file content.
// For any failure, we mark the job metadata as error and move the metadata to the history.
															#line 14275 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FetchBulkLoadTaskManifestEntryMapActor>
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActorState {
															#line 14282 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActorState(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   localJobManifestFilePath(localJobManifestFilePath),
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   remoteJobManifestFilePath(remoteJobManifestFilePath)
															#line 14295 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	~FetchBulkLoadTaskManifestEntryMapActorState() 
	{
		fdb_probe_actor_destroy("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1793 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid() && self->bulkLoadJobManager.get().manifestEntryMap->empty());
															#line 1795 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginTime = now();
															#line 1796 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = self->bulkLoadJobManager.get().jobState.getJobRange();
															#line 1797 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestMapRange = KeyRange();
															#line 14316 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1799 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!fileExists(abspath(localJobManifestFilePath)))
															#line 14320 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1800 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDBulkLoadJobManagerDownloadJobManifest", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath);
															#line 1804 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					StrictFuture<Void> __when_expr_0 = downloadBulkLoadJobManifestFile( jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath, self->ddId);
															#line 1804 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 14328 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 1;
															#line 1804 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 14333 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1861 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchBulkLoadTaskManifestEntryMapActorState(); static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->destroy(); return 0; }
															#line 14367 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1837 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 14380 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1838 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 14384 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1840 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			err = e;
															#line 1841 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindUnretryableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1847 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string errorMessage = "Failed to build job-manifest map with error code " + std::to_string(err.code()) + ". The remote file path is " + remoteJobManifestFilePath + ". The local file path is " + localJobManifestFilePath + ". The transport method is " + convertBulkLoadTransportMethodToString(jobTransportMethod) + ".";
															#line 1852 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = moveErrorBulkLoadJobToHistory(self, errorMessage);
															#line 1852 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14396 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_2.get(), loopDepth); };
			static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 3;
															#line 1852 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 14401 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1815 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(manifestMapRange, getBulkLoadJobFileManifestEntryFromJobManifestFile( localJobManifestFilePath, jobRange, self->ddId, self->bulkLoadJobManager.get().manifestEntryMap));
															#line 1815 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 14418 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 2;
															#line 1815 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 14423 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1806 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 14432 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1806 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 14441 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1823 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadJobManager.get().jobState.getJobRange() != manifestMapRange)
															#line 14525 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1824 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerManifestMapRangeMismatch", self->ddId) .detail("JobRange", jobRange) .detail("ManifestMapRange", manifestMapRange);
															#line 1827 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_dataset_not_cover_required_range(), loopDepth);
															#line 14531 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1829 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1830 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 14537 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1823 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadJobManager.get().jobState.getJobRange() != manifestMapRange)
															#line 14546 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1824 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerManifestMapRangeMismatch", self->ddId) .detail("JobRange", jobRange) .detail("ManifestMapRange", manifestMapRange);
															#line 1827 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_dataset_not_cover_required_range(), loopDepth);
															#line 14552 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1829 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1830 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 14558 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch2cont1(Void const& _,int loopDepth) 
	{
															#line 1853 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretryableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1859 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 14657 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2cont1(Void && _,int loopDepth) 
	{
															#line 1853 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretryableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1859 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 14667 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string localJobManifestFilePath;
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string remoteJobManifestFilePath;
															#line 1795 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1796 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1797 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange manifestMapRange;
															#line 1840 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Error err;
															#line 14762 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActor final : public Actor<Void>, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >, public FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>, public FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor> {
															#line 14767 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator new;
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3843995777285612800UL, 14455391712667517440UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >;
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActor(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 14786 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor>(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7119443845485172480UL, 13876208275314761984UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchBulkLoadTaskManifestEntryMap");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> fetchBulkLoadTaskManifestEntryMap( Reference<DataDistributor> const& self, BulkLoadTransportMethod const& jobTransportMethod, std::string const& localJobManifestFilePath, std::string const& remoteJobManifestFilePath ) {
															#line 1789 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FetchBulkLoadTaskManifestEntryMapActor(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath));
															#line 14821 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1863 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 14826 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadJobActor>
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActorState {
															#line 14833 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActorState(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1865 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntry()
															#line 14844 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1866 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1867 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1868 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1869 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors = std::vector<Future<Void>>();
															#line 1870 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1871 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1875 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 14871 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1964 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(actors);
															#line 1964 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14896 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 1964 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 14901 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1877 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, jobState.getJobRange().end));
															#line 1877 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14920 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1877 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 14925 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1961 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1961 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14962 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 1961 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 14967 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1879 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 14984 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 1877 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		res = __res;
															#line 14993 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1955 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (beginKey >= jobState.getJobRange().end)
															#line 15072 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1957 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager.get().allTaskSubmitted = true;
															#line 15076 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < res.size() - 1))
															#line 15094 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1884 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!res[i].value.empty())
															#line 15100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1885 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			task = decodeBulkLoadTaskState(res[i].value);
															#line 1886 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (task.isValid())
															#line 15106 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1887 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.getJobId() != self->bulkLoadJobManager.get().jobState.getJobId())
															#line 15110 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1888 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					return a_body1loopBody1Catch1(bulkload_task_outdated(), std::max(0, loopDepth - 1));
															#line 15114 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
															#line 1894 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(self->bulkLoadJobManager.get().manifestEntryMap->find(task.getRange().begin) != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1896 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.onAnyPhase( { BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }))
															#line 15120 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1898 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(task.getRange().end == res[i + 1].key);
															#line 1899 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					beginKey = task.getRange().end;
															#line 15126 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					return a_body1loopBody1cont2continue1(loopDepth); // continue
				}
				else
				{
															#line 1902 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (task.onAnyPhase({ BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }))
															#line 15133 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 1907 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (self->bulkLoadJobManager.get().allTaskSubmitted)
															#line 15137 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 1909 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
							;
															#line 15141 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
							loopDepth = a_body1loopBody1cont2loopBody1loopHead1(loopDepth);
						}
						else
						{
							loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);
						}
					}
					else
					{
															#line 1923 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						UNREACHABLE();
															#line 15153 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = a_body1loopBody1cont2loopBody1cont7(loopDepth);
					}
				}
			}
			else
			{
				loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2continue1(int loopDepth) 
	{
															#line 1880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 15187 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1931 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i].key);
															#line 1932 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 15198 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont8(int loopDepth) 
	{
															#line 1919 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(task.getRange().end == res[i + 1].key);
															#line 1920 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = task.getRange().end;
															#line 15227 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont2continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont9(int loopDepth) 
	{
															#line 1916 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back(bulkLoadJobMonitorTask( self, task.getJobId(), jobState.getJobRange(), task.getRange(), errorOut));
															#line 15236 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1(int loopDepth) 
	{
															#line 1910 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadParallelismLimitor.canStart())
															#line 15252 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1913 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->bulkLoadParallelismLimitor.waitUntilCounterChanged();
															#line 1913 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 15260 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 1913 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15265 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2loopBody1cont13(int loopDepth) 
	{
															#line 1953 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i + 1].key || beginKey > jobState.getJobRange().end);
															#line 1880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 15376 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 1932 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < res[i + 1].key))
															#line 15392 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1933 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1934 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;manifestEntries.size() < SERVER_KNOBS->MANIFEST_COUNT_MAX_PER_BULKLOAD_TASK && beginKey < res[i + 1].key;) {
															#line 1936 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->bulkLoadJobManager.get().manifestEntryMap->find(beginKey);
															#line 1937 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(it != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1938 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntry = it->second;
															#line 1939 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntries.push_back(manifestEntry);
															#line 1940 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = manifestEntry.getEndKey();
															#line 15410 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1942 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!manifestEntries.empty());
															#line 1943 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back(bulkLoadJobNewTask(self, jobState.getJobId(), jobState.getJobRoot(), jobState.getJobRange(), jobState.getTransportMethod(), self->bulkLoadJobManager.get().manifestLocalTempFolder, manifestEntries, errorOut));
															#line 1951 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_TASK_SUBMISSION_INTERVAL_SEC);
															#line 1951 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 15420 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 1951 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15425 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont13(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1965 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 15634 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1965 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 15646 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1865 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFileManifestEntry manifestEntry;
															#line 1867 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1868 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1869 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> actors;
															#line 1870 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1871 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1877 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult res;
															#line 1879 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1885 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState task;
															#line 15751 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >, public ActorCallback< ScheduleBulkLoadJobActor, 1, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 2, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 3, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 4, Void >, public FastAllocated<ScheduleBulkLoadJobActor>, public ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor> {
															#line 15756 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadJobActor>::operator new;
	using FastAllocated<ScheduleBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5083990794612706816UL, 18056034711455202304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 4, Void >;
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActor(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 15777 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor>(self, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4684338048225246976UL, 14035353769110992896UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadJob( Reference<DataDistributor> const& self, Promise<Void> const& errorOut ) {
															#line 1864 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadJobActor(self, errorOut));
															#line 15814 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1967 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 15819 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class CheckBulkLoadTaskCompleteOrErrorActor>
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActorState {
															#line 15826 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self) 
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1969 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1970 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 15837 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~CheckBulkLoadTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1971 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1972 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1973 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1974 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 1975 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 1976 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = KeyRange();
															#line 1977 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 1978 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 15866 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2023 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(true); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 15889 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(true);
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1978 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 15908 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1980 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 1981 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1982 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobState.getJobId(), jobState.getJobRange());
															#line 1982 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15921 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1982 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 15926 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2020 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2020 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15963 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 2020 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 15968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState const& checkJobState,int loopDepth) 
	{
															#line 1983 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1983 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15985 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1983 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 15990 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState && checkJobState,int loopDepth) 
	{
															#line 1983 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1983 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16001 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1983 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16006 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& checkJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(checkJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && checkJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(checkJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1984 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1985 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1986 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1987 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16096 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1989 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1993 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16108 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1994 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1999 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16114 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2001 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2002 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16122 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 2009 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16127 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2010 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 2015 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16133 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2018 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 16142 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1984 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1985 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1986 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1987 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16157 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1989 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16161 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1993 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16169 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1994 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1999 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16175 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2001 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16179 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2002 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16183 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 2009 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16188 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2010 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 2015 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16194 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2018 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 16203 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1969 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1970 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1972 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1973 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1974 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1975 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState existTask;
															#line 1976 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1977 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult bulkLoadTaskResult;
															#line 16401 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActor final : public Actor<bool>, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>, public CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor> {
															#line 16406 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3488768710841563904UL, 3984617401565417216UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >;
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActor(Reference<DataDistributor> const& self) 
															#line 16425 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<bool>(),
		   CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13835388531052272896UL, 2426800946393579264UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBulkLoadTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<bool> checkBulkLoadTaskCompleteOrError( Reference<DataDistributor> const& self ) {
															#line 1968 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<bool>(new CheckBulkLoadTaskCompleteOrErrorActor(self));
															#line 16460 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 2025 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 16465 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via finalizeBulkLoadJob()
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FinalizeBulkLoadJobActor>
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FinalizeBulkLoadJobActorState {
															#line 16472 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FinalizeBulkLoadJobActorState(Reference<DataDistributor> const& self) 
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2027 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 2028 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 16483 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~FinalizeBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2029 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 2030 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 2031 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 2032 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 2033 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = Optional<Key>();
															#line 2034 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 2035 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 2036 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobCompleteRange = KeyRange();
															#line 2037 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = false;
															#line 2038 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			i = 0;
															#line 2039 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			allFinish = false;
															#line 2040 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 16520 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2123 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FinalizeBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FinalizeBulkLoadJobActorState(); static_cast<FinalizeBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 16543 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FinalizeBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2040 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 16562 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 2042 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.reset();
															#line 2043 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 2044 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobState.getJobId(), jobState.getJobRange());
															#line 2044 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16575 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 2044 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16580 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2119 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(e.code() != error_code_range_unlock_reject);
															#line 2120 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_9 = tr.onError(e);
															#line 2120 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16619 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_9.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 10;
															#line 2120 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16624 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
															#line 2045 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2046 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2046 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16643 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2046 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16648 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState && currentJobState,int loopDepth) 
	{
															#line 2045 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2046 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2046 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16661 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2046 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16666 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(currentJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(currentJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 2047 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2048 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 16752 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 2047 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2048 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 16763 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 2086 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (hasError)
															#line 16847 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2087 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setErrorPhase("BulkLoadTask Error.");
															#line 16851 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 2089 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setCompletePhase();
															#line 16857 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 2091 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(lastKey.present());
															#line 2092 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobCompleteRange = KeyRangeRef(jobState.getJobRange().begin, lastKey.get());
															#line 2093 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 2093 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16867 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 2093 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16872 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1(int loopDepth) 
	{
															#line 2048 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < bulkLoadTaskResult.size() - 1))
															#line 16888 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2049 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 2050 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 2051 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!existTask.isValid())
															#line 16898 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2052 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = bulkLoadTaskResult[i + 1].key;
															#line 16902 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1loopBody1cont4continue1(loopDepth); // continue
		}
															#line 2055 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getRange() == KeyRangeRef(bulkLoadTaskResult[i].key, bulkLoadTaskResult[i + 1].key));
															#line 2060 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getJobId() == jobState.getJobId());
															#line 2061 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (existTask.phase == BulkLoadPhase::Error)
															#line 16911 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2062 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerStopClearMetadata", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2069 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = true;
															#line 16917 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);
		}
		else
		{
															#line 2071 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete && existTask.phase != BulkLoadPhase::Acknowledged)
															#line 16924 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2072 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerWrongTaskPhase", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2078 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16930 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2081 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = setBulkLoadFinalizeTransaction(&tr, existTask.getRange(), existTask.getTaskId());
															#line 2081 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16936 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 2081 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16941 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4continue1(int loopDepth) 
	{
															#line 2048 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 16964 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont1(int loopDepth) 
	{
															#line 2083 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		lastKey = bulkLoadTaskResult[i + 1].key;
															#line 16973 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2094 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2095 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17071 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2097 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17079 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17084 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2102 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_7 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2102 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17093 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_7.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 8;
															#line 2102 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2094 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2095 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17110 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2097 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17118 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2098 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17123 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2102 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_7 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2102 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17132 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_7.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 8;
															#line 2102 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17137 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 2104 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_8 = tr.commit();
															#line 2104 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17224 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_8.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 9;
															#line 2104 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17229 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 2099 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2099 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17240 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2099 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17245 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 2099 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2099 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17256 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2099 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17261 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10cont1(Void const& _,int loopDepth) 
	{
															#line 2100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = releaseExclusiveReadLockOnRange(&tr, jobState.getJobRange(), rangeLockNameForBulkLoad);
															#line 2100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17347 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont10cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17352 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1(Void && _,int loopDepth) 
	{
															#line 2100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = releaseExclusiveReadLockOnRange(&tr, jobState.getJobRange(), rangeLockNameForBulkLoad);
															#line 2100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17363 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont10cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2100 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17368 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont10cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
															#line 2105 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 2106 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerFinalizeRange", self->ddId) .detail("JobCompleteRange", jobCompleteRange) .detail("CommitVersion", commitVersion) .detail("AllFinish", allFinish) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2115 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = lastKey.get();
															#line 17630 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont12cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
															#line 2105 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 2106 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerFinalizeRange", self->ddId) .detail("JobCompleteRange", jobCompleteRange) .detail("CommitVersion", commitVersion) .detail("AllFinish", allFinish) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2115 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = lastKey.get();
															#line 17643 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont12cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1loopBody1cont12cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);

	}
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 2027 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 2028 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 2030 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 2031 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 2032 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 2033 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Optional<Key> lastKey;
															#line 2034 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState existTask;
															#line 2035 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult bulkLoadTaskResult;
															#line 2036 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobCompleteRange;
															#line 2037 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool hasError;
															#line 2038 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 2039 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool allFinish;
															#line 17849 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via finalizeBulkLoadJob()
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FinalizeBulkLoadJobActor final : public Actor<Void>, public ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >, public ActorCallback< FinalizeBulkLoadJobActor, 1, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 2, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 3, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 4, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 5, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 6, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 7, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 8, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 9, Void >, public FastAllocated<FinalizeBulkLoadJobActor>, public FinalizeBulkLoadJobActorState<FinalizeBulkLoadJobActor> {
															#line 17854 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FinalizeBulkLoadJobActor>::operator new;
	using FastAllocated<FinalizeBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9393649724427001856UL, 5522754037006193408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 1, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 2, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 3, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 4, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 5, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 6, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 7, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 8, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 9, Void >;
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FinalizeBulkLoadJobActor(Reference<DataDistributor> const& self) 
															#line 17880 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FinalizeBulkLoadJobActorState<FinalizeBulkLoadJobActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13125497442541706752UL, 11431860539656438528UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("finalizeBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> finalizeBulkLoadJob( Reference<DataDistributor> const& self ) {
															#line 2026 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FinalizeBulkLoadJobActor(self));
															#line 17922 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 2125 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 17927 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobManager()
															#line 2126 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobManagerActor>
															#line 2126 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobManagerActorState {
															#line 17934 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 2126 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobManagerActorState(Reference<DataDistributor> const& self) 
															#line 2126 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 2126 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2128 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context())
															#line 17943 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobManager", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobManagerActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobManager", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2129 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<BulkLoadJobState>> __when_expr_0 = getRunningBulkLoadJob(cx);
															#line 2129 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17960 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 1;
															#line 2129 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 17965 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobManagerActorState();
		static_cast<BulkLoadJobManagerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2130 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!job.present())
															#line 17988 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2131 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobManagerNoJobExist", self->ddId);
															#line 2132 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager.reset();
															#line 2133 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadTaskCollection->removeBulkLoadJobRange();
															#line 2134 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobManagerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobManagerActorState(); static_cast<BulkLoadJobManagerActor*>(this)->destroy(); return 0; }
															#line 17998 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobManagerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobManagerActorState();
			static_cast<BulkLoadJobManagerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2136 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobId = job.get().getJobId();
															#line 2137 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobRange = job.get().getJobRange();
															#line 2138 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobRoot = job.get().getJobRoot();
															#line 2139 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobTransportMethod = job.get().getTransportMethod();
															#line 2140 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->setBulkLoadJobRange(jobRange);
															#line 2143 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->bulkLoadJobManager.present() || self->bulkLoadJobManager.get().jobState.getJobId() != jobId)
															#line 18016 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2144 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadJobManagerBuild", self->ddId) .detail("OldJobID", self->bulkLoadJobManager.present() ? self->bulkLoadJobManager.get().jobState.getJobId().toString() : "No old job") .detail("NewJobId", jobId) .detail("NewJobRange", jobRange) .detail("NewJobRoot", jobRoot) .detail("NewJobTransportMethod", jobTransportMethod);
															#line 2153 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string localFolder = getBulkLoadJobRoot(self->bulkLoadFolder, jobId);
															#line 2154 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string manifestLocalTempFolder = abspath(joinPath(localFolder, "manifest-temp"));
															#line 2155 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			resetFileFolder(manifestLocalTempFolder);
															#line 2156 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string remoteFolder = getBulkLoadJobRoot(jobRoot, jobId);
															#line 2157 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string jobManifestFileName = getBulkLoadJobManifestFileName();
															#line 2158 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string localJobManifestFilePath = joinPath(localFolder, jobManifestFileName);
															#line 2159 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string remoteJobManifestFilePath = appendToPath(remoteFolder, jobManifestFileName);
															#line 2160 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager = DDBulkLoadJobManager(job.get(), manifestLocalTempFolder);
															#line 2161 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = fetchBulkLoadTaskManifestEntryMap( self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath);
															#line 2161 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18040 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 2;
															#line 2161 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 1, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18045 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2165 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadJobManagerExist", self->ddId) .detail("JobID", self->bulkLoadJobManager.get().jobState.getJobId()) .detail("JobRange", self->bulkLoadJobManager.get().jobState.getJobRange());
															#line 18052 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadJobState> const& __job,int loopDepth) 
	{
															#line 2129 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		job = __job;
															#line 18062 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadJobState> && __job,int loopDepth) 
	{
		job = std::move(__job);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Optional<BulkLoadJobState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Optional<BulkLoadJobState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2171 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 2177 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		errorOut = Promise<Void>();
															#line 2178 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 18145 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = persistBulkLoadJobTaskCount(self);
															#line 2163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18156 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 3;
															#line 2163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 2, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18161 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = persistBulkLoadJobTaskCount(self);
															#line 2163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18172 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 3;
															#line 2163 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 2, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18177 "/codebuild/output/src2118757142/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(th