#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
/*
 * FileDecoder.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <algorithm>
#include <cstdlib>
#include <iostream>
#include <limits>
#include <memory>
#include <string>
#include <vector>
#include <fcntl.h>

#ifdef _WIN32
#include <io.h>
#endif

#include "fdbclient/BackupTLSConfig.h"
#include "fdbclient/BuildFlags.h"
#include "fdbbackup/FileConverter.h"
#include "fdbbackup/Decode.h"
#include "fdbclient/BackupAgent.actor.h"
#include "fdbclient/BackupContainer.h"
#include "fdbclient/BackupContainerFileSystem.h"
#include "fdbclient/BuildFlags.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/IKnobCollection.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/MutationList.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/versions.h"
#include "flow/ArgParseUtil.h"
#include "flow/FastRef.h"
#include "flow/IRandom.h"
#include "flow/Platform.h"
#include "flow/Trace.h"
#include "flow/flow.h"
#include "flow/serialize.h"

#include "flow/actorcompiler.h" // has to be last include

#define SevDecodeInfo SevVerbose

extern bool g_crashOnError;
extern const char* getSourceVersion();

namespace file_converter {

void printDecodeUsage() {
	std::cout << "Decoder for FoundationDB backup mutation logs.\n"
	             "Usage: fdbdecode  [OPTIONS]\n"
	             "  -r, --container URL\n"
	             "                 Backup container URL, e.g., file:///some/path/.\n"
	             "  -i, --input    FILE\n"
	             "                 Log file filter, only matched files are decoded.\n"
	             "  --log          Enables trace file logging for the CLI session.\n"
	             "  --logdir PATH  Specifies the output directory for trace files. If\n"
	             "                 unspecified, defaults to the current directory. Has\n"
	             "                 no effect unless --log is specified.\n"
	             "  --loggroup     LOG_GROUP\n"
	             "                 Sets the LogGroup field with the specified value for all\n"
	             "                 events in the trace output (defaults to `default').\n"
	             "  --trace-format FORMAT\n"
	             "                 Select the format of the trace files, xml (the default) or json.\n"
	             "                 Has no effect unless --log is specified.\n"
	             "  --crash        Crash on serious error.\n"
	             "  --blob-credentials FILE\n"
	             "                 File containing blob credentials in JSON format.\n"
	             "                 The same credential format/file fdbbackup uses.\n" TLS_HELP
	             "  -t, --file-type [log|range|both]\n"
	             "                 Specifies the backup file type to decode.\n"
	             "  --build-flags  Print build information and exit.\n"
	             "  --list-only    Print file list and exit.\n"
	             "  --validate-filters Validate the default RangeMap filtering logic with a slower one.\n"
	             "  -k KEY_PREFIX  Use a single prefix for filtering mutations.\n"
	             "  --filters PREFIX_FILTER_FILE\n"
	             "                 A file containing a list of prefix filters in HEX format separated by \";\",\n"
	             "                 e.g., \"\\x05\\x01;\\x15\\x2b\"\n"
	             "  --hex-prefix   HEX_PREFIX\n"
	             "                 The prefix specified in HEX format, e.g., --hex-prefix \"\\\\x05\\\\x01\".\n"
	             "  --begin-version-filter BEGIN_VERSION\n"
	             "                 The version range's begin version (inclusive) for filtering.\n"
	             "  --end-version-filter END_VERSION\n"
	             "                 The version range's end version (exclusive) for filtering.\n"
	             "  --knob-KNOBNAME KNOBVALUE\n"
	             "                 Changes a knob value. KNOBNAME should be lowercase.\n"
	             "  -s, --save     Save a copy of downloaded files (default: not saving).\n"
	             "\n";
	return;
}

void printBuildInformation() {
	std::cout << jsonBuildInformation() << "\n";
}

struct DecodeParams : public ReferenceCounted<DecodeParams> {
	std::string container_url;
	Optional<std::string> proxy;
	std::string fileFilter; // only files match the filter will be decoded
	bool log_enabled = true;
	std::string log_dir, trace_format, trace_log_group;
	BackupTLSConfig tlsConfig;
	bool list_only = false;
	bool decode_logs = true;
	bool decode_range = true;
	bool save_file_locally = false;
	bool validate_filters = false;
	std::vector<std::string> prefixes; // Key prefixes for filtering
	// more efficient data structure for intersection queries than "prefixes"
	fileBackup::RangeMapFilters filters;
	Version beginVersionFilter = 0;
	Version endVersionFilter = std::numeric_limits<Version>::max();

	std::vector<std::pair<std::string, std::string>> knobs;

	// Returns if [begin, end) overlap with the filter range
	bool overlap(Version begin, Version end) const {
		// Filter [100, 200),  [50,75) [200, 300)
		return !(begin >= endVersionFilter || end <= beginVersionFilter);
	}

	bool overlap(Version version) const { return version >= beginVersionFilter && version < endVersionFilter; }

	void updateRangeMap() { filters.updateFilters(prefixes); }

	bool matchFilters(const MutationRef& m) const {
		bool match = filters.match(m);
		if (!validate_filters) {
			return match;
		}

		// If we choose to validate the filters, go through filters one by one
		for (const auto& prefix : prefixes) {
			if (isSingleKeyMutation((MutationRef::Type)m.type)) {
				if (m.param1.startsWith(StringRef(prefix))) {
					ASSERT(match);
					return true;
				}
			} else if (m.type == MutationRef::ClearRange) {
				KeyRange range(KeyRangeRef(m.param1, m.param2));
				KeyRange range2 = prefixRange(StringRef(prefix));
				if (range.intersects(range2)) {
					ASSERT(match);
					return true;
				}
			} else {
				ASSERT(false);
			}
		}
		ASSERT(!match);
		return false;
	}

	bool matchFilters(const KeyRange& range) const {
		bool match = filters.match(range);
		if (!validate_filters) {
			return match;
		}

		for (const auto& prefix : prefixes) {
			if (range.intersects(prefixRange(StringRef(prefix)))) {
				ASSERT(match);
				return true;
			}
		}
		return false;
	}

	bool matchFilters(KeyValueRef kv) const {
		bool match = filters.match(kv);

		if (!validate_filters) {
			return match;
		}

		for (const auto& prefix : prefixes) {
			if (kv.key.startsWith(StringRef(prefix))) {
				ASSERT(match);
				return true;
			}
		}

		return match;
	}

	std::string toString() {
		std::string s;
		s.append("ContainerURL: ");
		s.append(container_url);
		if (proxy.present()) {
			s.append(", Proxy: ");
			s.append(proxy.get());
		}
		s.append(", FileFilter: ");
		s.append(fileFilter);
		if (log_enabled) {
			if (!log_dir.empty()) {
				s.append(" LogDir:").append(log_dir);
			}
			if (!trace_format.empty()) {
				s.append(" Format:").append(trace_format);
			}
			if (!trace_log_group.empty()) {
				s.append(" LogGroup:").append(trace_log_group);
			}
		}
		s.append(", list_only: ").append(list_only ? "true" : "false");
		s.append(", validate_filters: ").append(validate_filters ? "true" : "false");
		if (beginVersionFilter != 0) {
			s.append(", beginVersionFilter: ").append(std::to_string(beginVersionFilter));
		}
		if (endVersionFilter < std::numeric_limits<Version>::max()) {
			s.append(", endVersionFilter: ").append(std::to_string(endVersionFilter));
		}
		if (!prefixes.empty()) {
			s.append(", KeyPrefixes: ").append(printable(describe(prefixes)));
		}
		for (const auto& [knob, value] : knobs) {
			s.append(", KNOB-").append(knob).append(" = ").append(value);
		}
		s.append(", SaveFile: ").append(save_file_locally ? "true" : "false");
		return s;
	}

	void updateKnobs() {
		IKnobCollection::setupKnobs(knobs);

		// Reinitialize knobs in order to update knobs that are dependent on explicitly set knobs
		IKnobCollection::getMutableGlobalKnobCollection().initialize(Randomize::False, IsSimulated::False);
	}
};

// Parses and returns a ";" separated HEX encoded strings. So the ";" in
// the string should be escaped as "\;".
// Sets "err" to true if there is any parsing error.
std::vector<std::string> parsePrefixesLine(const std::string& line, bool& err) {
	std::vector<std::string> results;
	err = false;

	int p = 0;
	while (p < line.size()) {
		int end = line.find_first_of(';', p);
		if (end == line.npos) {
			end = line.size();
		}
		auto prefix = decode_hex_string(line.substr(p, end - p), err);
		if (err) {
			return results;
		}
		results.push_back(prefix);
		p = end + 1;
	}
	return results;
}

std::vector<std::string> parsePrefixFile(const std::string& filename, bool& err) {
	std::string line = readFileBytes(filename, 64 * 1024 * 1024);
	return parsePrefixesLine(line, err);
}

int parseDecodeCommandLine(Reference<DecodeParams> param, CSimpleOpt* args) {
	bool err = false;

	while (args->Next()) {
		auto lastError = args->LastError();
		switch (lastError) {
		case SO_SUCCESS:
			break;

		default:
			std::cerr << "ERROR: argument given for option: " << args->OptionText() << "\n";
			return FDB_EXIT_ERROR;
			break;
		}
		int optId = args->OptionId();
		switch (optId) {
		case OPT_HELP:
			return FDB_EXIT_ERROR;

		case OPT_CONTAINER:
			param->container_url = args->OptionArg();
			break;

		case OPT_FILE_TYPE: {
			auto ftype = std::string(args->OptionArg());
			if (ftype == "log") {
				param->decode_range = false;
			} else if (ftype == "range") {
				param->decode_logs = false;
			} else if (ftype != "both" && ftype != "") {
				err = true;
				std::cerr << "ERROR: Unrecognized backup file type option: " << args->OptionArg() << "\n";
				return FDB_EXIT_ERROR;
			}
			break;
		}

		case OPT_LIST_ONLY:
			param->list_only = true;
			break;

		case OPT_VALIDATE_FILTERS:
			param->validate_filters = true;
			break;

		case OPT_KEY_PREFIX:
			param->prefixes.push_back(args->OptionArg());
			break;

		case OPT_FILTERS:
			param->prefixes = parsePrefixFile(args->OptionArg(), err);
			if (err) {
				throw std::runtime_error("ERROR:" + std::string(args->OptionArg()) + "contains invalid prefix(es)");
			}
			break;

		case OPT_HEX_KEY_PREFIX:
			param->prefixes.push_back(decode_hex_string(args->OptionArg(), err));
			break;

		case OPT_PROXY:
			param->proxy = args->OptionArg();
			break;

		case OPT_BEGIN_VERSION_FILTER:
			param->beginVersionFilter = std::atoll(args->OptionArg());
			break;

		case OPT_END_VERSION_FILTER:
			param->endVersionFilter = std::atoll(args->OptionArg());
			break;

		case OPT_CRASHONERROR:
			g_crashOnError = true;
			break;

		case OPT_INPUT_FILE:
			param->fileFilter = args->OptionArg();
			break;

		case OPT_TRACE:
			param->log_enabled = true;
			break;

		case OPT_TRACE_DIR:
			param->log_dir = args->OptionArg();
			break;

		case OPT_TRACE_FORMAT:
			if (!selectTraceFormatter(args->OptionArg())) {
				std::cerr << "ERROR: Unrecognized trace format " << args->OptionArg() << "\n";
				return FDB_EXIT_ERROR;
			}
			param->trace_format = args->OptionArg();
			break;

		case OPT_TRACE_LOG_GROUP:
			param->trace_log_group = args->OptionArg();
			break;

		case OPT_BLOB_CREDENTIALS:
			param->tlsConfig.blobCredentials.push_back(args->OptionArg());
			break;

		case OPT_KNOB: {
			Optional<std::string> knobName = extractPrefixedArgument("--knob", args->OptionSyntax());
			if (!knobName.present()) {
				std::cerr << "ERROR: unable to parse knob option '" << args->OptionSyntax() << "'\n";
				return FDB_EXIT_ERROR;
			}
			param->knobs.emplace_back(knobName.get(), args->OptionArg());
			break;
		}

		case OPT_SAVE_FILE:
			param->save_file_locally = true;
			break;

		case TLSConfig::OPT_TLS_PLUGIN:
			args->OptionArg();
			break;

		case TLSConfig::OPT_TLS_CERTIFICATES:
			param->tlsConfig.tlsCertPath = args->OptionArg();
			break;

		case TLSConfig::OPT_TLS_PASSWORD:
			param->tlsConfig.tlsPassword = args->OptionArg();
			break;

		case TLSConfig::OPT_TLS_CA_FILE:
			param->tlsConfig.tlsCAPath = args->OptionArg();
			break;

		case TLSConfig::OPT_TLS_KEY:
			param->tlsConfig.tlsKeyPath = args->OptionArg();
			break;

		case TLSConfig::OPT_TLS_VERIFY_PEERS:
			param->tlsConfig.tlsVerifyPeers = args->OptionArg();
			break;

		case OPT_BUILD_FLAGS:
			printBuildInformation();
			return FDB_EXIT_ERROR;
			break;
		}
	}
	return FDB_EXIT_SUCCESS;
}

template <class BackupFile>
void printLogFiles(std::string msg, const std::vector<BackupFile>& files) {
	std::cout << msg << " " << files.size() << " total\n";
	for (const auto& file : files) {
		std::cout << file.toString() << "\n";
	}
	std::cout << std::endl;
}

std::vector<LogFile> getRelevantLogFiles(const std::vector<LogFile>& files, const Reference<DecodeParams> params) {
	std::vector<LogFile> filtered;
	for (const auto& file : files) {
		if (file.fileName.find(params->fileFilter) != std::string::npos &&
		    params->overlap(file.beginVersion, file.endVersion + 1)) {
			filtered.push_back(file);
		}
	}
	return filtered;
}

std::vector<RangeFile> getRelevantRangeFiles(const std::vector<RangeFile>& files,
                                             const Reference<DecodeParams> params) {
	std::vector<RangeFile> filtered;
	for (const auto& file : files) {
		if (file.fileName.find(params->fileFilter) != std::string::npos && params->overlap(file.version)) {
			filtered.push_back(file);
		}
	}
	return filtered;
}

struct VersionedMutations {
	Version version;
	std::vector<MutationRef> mutations;
	std::string serializedMutations; // buffer that contains mutations
};

/*
 * Model a decoding progress for a mutation file. Usage is:
 *
 *    DecodeProgress progress(logfile);
 *    wait(progress->openFile(container));
 *    while (1) {
 *        Optional<VersionedMutations> batch = wait(progress->getNextBatch());
 *        if (!batch.present()) break;
 *        ... // process the batch mutations
 *    }
 *
 * Internally, the decoding process is done block by block -- each block is
 * decoded into a list of key/value pairs, which are then decoded into batches
 * of mutations. Because a version's mutations can be split into many key/value
 * pairs, the decoding of mutation needs to look ahead to find all batches that
 * belong to the same version.
 */
class DecodeProgress {
	std::vector<Standalone<VectorRef<KeyValueRef>>> blocks;
	std::unordered_map<Version, fileBackup::AccumulatedMutations> mutationBlocksByVersion;

public:
	DecodeProgress() = default;
	DecodeProgress(const LogFile& file, bool save) : file(file), save(save) {}

	~DecodeProgress() {
		if (lfd != -1) {
			close(lfd);
		}
	}

	// Open and loads file into memory
	Future<Void> openFile(Reference<IBackupContainer> container) { return openFileImpl(this, container); }

	// The following are private APIs:

	// Returns the next batch of mutations along with the arena backing it.
	// Note the returned batch can be empty when the file has unfinished
	// version batch data that are in the next file.
	Optional<VersionedMutations> getNextBatch() {
		for (auto& [version, m] : mutationBlocksByVersion) {
			if (m.isComplete()) {
				VersionedMutations vms;
				vms.version = version;
				vms.serializedMutations = m.serializedMutations;
				vms.mutations = fileBackup::decodeMutationLogValue(vms.serializedMutations);
				TraceEvent("Decode").detail("Version", vms.version).detail("N", vms.mutations.size());
				mutationBlocksByVersion.erase(version);
				return vms;
			}
		}

		// No complete versions
		if (!mutationBlocksByVersion.empty()) {
			TraceEvent(SevWarn, "UnfishedBlocks").detail("NumberOfVersions", mutationBlocksByVersion.size());
		}
		return Optional<VersionedMutations>();
	}

																#line 528 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
// This generated class is to be used only via openFileImpl()
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
template <class OpenFileImplActor>
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class OpenFileImplActorState {
															#line 534 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	OpenFileImplActorState(DecodeProgress* const& self,Reference<IBackupContainer> const& container) 
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		 : self(self),
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   container(container)
															#line 543 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("openFileImpl", reinterpret_cast<unsigned long>(this));

	}
	~OpenFileImplActorState() 
	{
		fdb_probe_actor_destroy("openFileImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 527 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<Reference<IAsyncFile>> __when_expr_0 = container->readFile(self->file.fileName);
															#line 527 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<OpenFileImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 560 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<OpenFileImplActor*>(this)->actor_wait_state = 1;
															#line 527 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< OpenFileImplActor, 0, Reference<IAsyncFile> >*>(static_cast<OpenFileImplActor*>(this)));
															#line 565 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~OpenFileImplActorState();
		static_cast<OpenFileImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Reference<IAsyncFile> const& fd,int loopDepth) 
	{
															#line 528 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		self->fd = fd;
															#line 529 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		buf = makeString(self->file.fileSize);
															#line 530 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		StrictFuture<int> __when_expr_1 = self->fd->read(mutateString(buf), self->file.fileSize, 0);
															#line 530 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (static_cast<OpenFileImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 594 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<OpenFileImplActor*>(this)->actor_wait_state = 2;
															#line 530 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< OpenFileImplActor, 1, int >*>(static_cast<OpenFileImplActor*>(this)));
															#line 599 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Reference<IAsyncFile> && fd,int loopDepth) 
	{
															#line 528 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		self->fd = fd;
															#line 529 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		buf = makeString(self->file.fileSize);
															#line 530 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		StrictFuture<int> __when_expr_1 = self->fd->read(mutateString(buf), self->file.fileSize, 0);
															#line 530 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (static_cast<OpenFileImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 614 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<OpenFileImplActor*>(this)->actor_wait_state = 2;
															#line 530 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< OpenFileImplActor, 1, int >*>(static_cast<OpenFileImplActor*>(this)));
															#line 619 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& fd,int loopDepth) 
	{
		loopDepth = a_body1cont1(fd, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && fd,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(fd), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<OpenFileImplActor*>(this)->actor_wait_state > 0) static_cast<OpenFileImplActor*>(this)->actor_wait_state = 0;
		static_cast<OpenFileImplActor*>(this)->ActorCallback< OpenFileImplActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< OpenFileImplActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< OpenFileImplActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< OpenFileImplActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int const& rLen,int loopDepth) 
	{
															#line 531 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (rLen != self->file.fileSize)
															#line 703 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 532 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			return a_body1Catch1(restore_bad_read(), loopDepth);
															#line 707 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 535 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (self->save)
															#line 711 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 536 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			std::string dir = self->file.fileName;
															#line 537 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			std::size_t found = self->file.fileName.find_last_of('/');
															#line 538 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (found != std::string::npos)
															#line 719 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 539 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				std::string path = self->file.fileName.substr(0, found);
															#line 540 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!directoryExists(path))
															#line 725 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 541 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					platform::createDirectory(path);
															#line 729 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
			}
															#line 544 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			self->lfd = open(self->file.fileName.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0600);
															#line 545 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (self->lfd == -1)
															#line 736 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 546 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent(SevError, "OpenLocalFileFailed").detail("File", self->file.fileName);
															#line 547 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(platform_error(), loopDepth);
															#line 742 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
															#line 549 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			int wlen = write(self->lfd, buf.begin(), self->file.fileSize);
															#line 550 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (wlen != self->file.fileSize)
															#line 748 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 551 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent(SevError, "WriteLocalFileFailed") .detail("File", self->file.fileName) .detail("Len", self->file.fileSize);
															#line 554 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(platform_error(), loopDepth);
															#line 754 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
															#line 556 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			TraceEvent("WriteLocalFile").detail("Name", self->file.fileName).detail("Len", self->file.fileSize);
															#line 758 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 559 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		self->decodeFile(buf);
															#line 560 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<OpenFileImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OpenFileImplActorState(); static_cast<OpenFileImplActor*>(this)->destroy(); return 0; }
															#line 764 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<OpenFileImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~OpenFileImplActorState();
		static_cast<OpenFileImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(int && rLen,int loopDepth) 
	{
															#line 531 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (rLen != self->file.fileSize)
															#line 776 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 532 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			return a_body1Catch1(restore_bad_read(), loopDepth);
															#line 780 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 535 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (self->save)
															#line 784 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 536 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			std::string dir = self->file.fileName;
															#line 537 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			std::size_t found = self->file.fileName.find_last_of('/');
															#line 538 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (found != std::string::npos)
															#line 792 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 539 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				std::string path = self->file.fileName.substr(0, found);
															#line 540 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!directoryExists(path))
															#line 798 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 541 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					platform::createDirectory(path);
															#line 802 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
			}
															#line 544 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			self->lfd = open(self->file.fileName.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0600);
															#line 545 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (self->lfd == -1)
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 546 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent(SevError, "OpenLocalFileFailed").detail("File", self->file.fileName);
															#line 547 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(platform_error(), loopDepth);
															#line 815 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
															#line 549 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			int wlen = write(self->lfd, buf.begin(), self->file.fileSize);
															#line 550 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (wlen != self->file.fileSize)
															#line 821 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 551 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent(SevError, "WriteLocalFileFailed") .detail("File", self->file.fileName) .detail("Len", self->file.fileSize);
															#line 554 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(platform_error(), loopDepth);
															#line 827 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
															#line 556 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			TraceEvent("WriteLocalFile").detail("Name", self->file.fileName).detail("Len", self->file.fileSize);
															#line 831 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 559 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		self->decodeFile(buf);
															#line 560 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<OpenFileImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OpenFileImplActorState(); static_cast<OpenFileImplActor*>(this)->destroy(); return 0; }
															#line 837 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<OpenFileImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~OpenFileImplActorState();
		static_cast<OpenFileImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(int const& rLen,int loopDepth) 
	{
		loopDepth = a_body1cont2(rLen, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(int && rLen,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(rLen), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<OpenFileImplActor*>(this)->actor_wait_state > 0) static_cast<OpenFileImplActor*>(this)->actor_wait_state = 0;
		static_cast<OpenFileImplActor*>(this)->ActorCallback< OpenFileImplActor, 1, int >::remove();

	}
	void a_callback_fire(ActorCallback< OpenFileImplActor, 1, int >*,int const& value) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< OpenFileImplActor, 1, int >*,int && value) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< OpenFileImplActor, 1, int >*,Error err) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	DecodeProgress* self;
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<IBackupContainer> container;
															#line 529 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Standalone<StringRef> buf;
															#line 926 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
};
// This generated class is to be used only via openFileImpl()
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class OpenFileImplActor final : public Actor<Void>, public ActorCallback< OpenFileImplActor, 0, Reference<IAsyncFile> >, public ActorCallback< OpenFileImplActor, 1, int >, public FastAllocated<OpenFileImplActor>, public OpenFileImplActorState<OpenFileImplActor> {
															#line 931 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
	using FastAllocated<OpenFileImplActor>::operator new;
	using FastAllocated<OpenFileImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14988936481147074304UL, 9551464234944654080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< OpenFileImplActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< OpenFileImplActor, 1, int >;
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	OpenFileImplActor(DecodeProgress* const& self,Reference<IBackupContainer> const& container) 
															#line 949 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   OpenFileImplActorState<OpenFileImplActor>(self, container),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5478526420490648320UL, 17246077299049572864UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("openFileImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< OpenFileImplActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< OpenFileImplActor, 1, int >*)0, actor_cancelled()); break;
		}

	}
};
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
[[nodiscard]] static Future<Void> openFileImpl( DecodeProgress* const& self, Reference<IBackupContainer> const& container ) {
															#line 526 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	return Future<Void>(new OpenFileImplActor(self, container));
															#line 982 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
}

#line 562 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"

	// Add chunks to mutationBlocksByVersion
	void addBlockKVPairs(VectorRef<KeyValueRef> chunks) {
		for (auto& kv : chunks) {
			auto versionAndChunkNumber = fileBackup::decodeMutationLogKey(kv.key);
			mutationBlocksByVersion[versionAndChunkNumber.first].addChunk(versionAndChunkNumber.second, kv);
		}
	}

	// Reads a file a file content in the buffer, decodes it into key/value pairs, and stores these pairs.
	void decodeFile(const Standalone<StringRef>& buf) {
		try {
			loop {
				int64_t len = std::min<int64_t>(file.blockSize, file.fileSize - offset);
				if (len == 0) {
					return;
				}

				// Decode a file block into log_key and log_value chunks
				Standalone<VectorRef<KeyValueRef>> chunks =
				    fileBackup::decodeMutationLogFileBlock(buf.substr(offset, len));
				blocks.push_back(chunks);
				addBlockKVPairs(chunks);
				offset += len;
			}
		} catch (Error& e) {
			TraceEvent(SevWarn, "CorruptLogFileBlock")
			    .error(e)
			    .detail("Filename", file.fileName)
			    .detail("BlockOffset", offset)
			    .detail("BlockLen", file.blockSize);
			throw;
		}
	}

	LogFile file;
	Reference<IAsyncFile> fd;
	int64_t offset = 0;
	bool eof = false;
	bool save = false;
	int lfd = -1; // local file descriptor
};

class DecodeRangeProgress {
public:
	std::vector<Standalone<VectorRef<KeyValueRef>>> blocks;

	DecodeRangeProgress() = default;
	DecodeRangeProgress(const RangeFile& file, bool save) : file(file), save(save) {}
	~DecodeRangeProgress() {
		if (lfd != -1) {
			close(lfd);
		}
	}

	// Open and loads file into memory
	Future<Void> openFile(Reference<IBackupContainer> container) { return openFileImpl(this, container); }

																#line 1044 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
// This generated class is to be used only via openFileImpl()
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
template <class OpenFileImplActor1>
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class OpenFileImplActor1State {
															#line 1050 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	OpenFileImplActor1State(DecodeRangeProgress* const& self,Reference<IBackupContainer> const& container) 
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		 : self(self),
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   container(container)
															#line 1059 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("openFileImpl", reinterpret_cast<unsigned long>(this));

	}
	~OpenFileImplActor1State() 
	{
		fdb_probe_actor_destroy("openFileImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 621 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			TraceEvent("ReadFile").detail("Name", self->file.fileName).detail("Len", self->file.fileSize);
															#line 623 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<Reference<IAsyncFile>> __when_expr_0 = container->readFile(self->file.fileName);
															#line 623 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<OpenFileImplActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1078 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<OpenFileImplActor1*>(this)->actor_wait_state = 1;
															#line 623 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< OpenFileImplActor1, 0, Reference<IAsyncFile> >*>(static_cast<OpenFileImplActor1*>(this)));
															#line 1083 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~OpenFileImplActor1State();
		static_cast<OpenFileImplActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Reference<IAsyncFile> const& fd,int loopDepth) 
	{
															#line 624 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		self->fd = fd;
															#line 625 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		buf = makeString(self->file.fileSize);
															#line 626 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		StrictFuture<int> __when_expr_1 = self->fd->read(mutateString(buf), self->file.fileSize, 0);
															#line 626 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (static_cast<OpenFileImplActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1112 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<OpenFileImplActor1*>(this)->actor_wait_state = 2;
															#line 626 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< OpenFileImplActor1, 1, int >*>(static_cast<OpenFileImplActor1*>(this)));
															#line 1117 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Reference<IAsyncFile> && fd,int loopDepth) 
	{
															#line 624 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		self->fd = fd;
															#line 625 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		buf = makeString(self->file.fileSize);
															#line 626 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		StrictFuture<int> __when_expr_1 = self->fd->read(mutateString(buf), self->file.fileSize, 0);
															#line 626 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (static_cast<OpenFileImplActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1132 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<OpenFileImplActor1*>(this)->actor_wait_state = 2;
															#line 626 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< OpenFileImplActor1, 1, int >*>(static_cast<OpenFileImplActor1*>(this)));
															#line 1137 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& fd,int loopDepth) 
	{
		loopDepth = a_body1cont1(fd, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && fd,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(fd), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<OpenFileImplActor1*>(this)->actor_wait_state > 0) static_cast<OpenFileImplActor1*>(this)->actor_wait_state = 0;
		static_cast<OpenFileImplActor1*>(this)->ActorCallback< OpenFileImplActor1, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< OpenFileImplActor1, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< OpenFileImplActor1, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< OpenFileImplActor1, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int const& rLen,int loopDepth) 
	{
															#line 627 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (rLen != self->file.fileSize)
															#line 1221 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 628 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			return a_body1Catch1(restore_bad_read(), loopDepth);
															#line 1225 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 631 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (self->save)
															#line 1229 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 632 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			std::string dir = self->file.fileName;
															#line 633 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			std::size_t found = self->file.fileName.find_last_of('/');
															#line 634 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (found != std::string::npos)
															#line 1237 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 635 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				std::string path = self->file.fileName.substr(0, found);
															#line 636 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!directoryExists(path))
															#line 1243 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 637 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					platform::createDirectory(path);
															#line 1247 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
			}
															#line 641 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			self->lfd = open(self->file.fileName.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0600);
															#line 642 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (self->lfd == -1)
															#line 1254 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 643 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent(SevError, "OpenLocalFileFailed").detail("File", self->file.fileName);
															#line 644 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(platform_error(), loopDepth);
															#line 1260 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
															#line 646 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			int wlen = write(self->lfd, buf.begin(), self->file.fileSize);
															#line 647 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (wlen != self->file.fileSize)
															#line 1266 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 648 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent(SevError, "WriteLocalFileFailed") .detail("File", self->file.fileName) .detail("Len", self->file.fileSize);
															#line 651 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(platform_error(), loopDepth);
															#line 1272 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
															#line 653 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			TraceEvent("WriteLocalFile").detail("Name", self->file.fileName).detail("Len", self->file.fileSize);
															#line 1276 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 656 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		self->decodeFile(buf);
															#line 657 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<OpenFileImplActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OpenFileImplActor1State(); static_cast<OpenFileImplActor1*>(this)->destroy(); return 0; }
															#line 1282 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<OpenFileImplActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~OpenFileImplActor1State();
		static_cast<OpenFileImplActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(int && rLen,int loopDepth) 
	{
															#line 627 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (rLen != self->file.fileSize)
															#line 1294 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 628 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			return a_body1Catch1(restore_bad_read(), loopDepth);
															#line 1298 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 631 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (self->save)
															#line 1302 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 632 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			std::string dir = self->file.fileName;
															#line 633 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			std::size_t found = self->file.fileName.find_last_of('/');
															#line 634 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (found != std::string::npos)
															#line 1310 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 635 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				std::string path = self->file.fileName.substr(0, found);
															#line 636 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!directoryExists(path))
															#line 1316 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 637 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					platform::createDirectory(path);
															#line 1320 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
			}
															#line 641 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			self->lfd = open(self->file.fileName.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0600);
															#line 642 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (self->lfd == -1)
															#line 1327 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 643 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent(SevError, "OpenLocalFileFailed").detail("File", self->file.fileName);
															#line 644 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(platform_error(), loopDepth);
															#line 1333 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
															#line 646 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			int wlen = write(self->lfd, buf.begin(), self->file.fileSize);
															#line 647 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (wlen != self->file.fileSize)
															#line 1339 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 648 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent(SevError, "WriteLocalFileFailed") .detail("File", self->file.fileName) .detail("Len", self->file.fileSize);
															#line 651 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(platform_error(), loopDepth);
															#line 1345 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
															#line 653 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			TraceEvent("WriteLocalFile").detail("Name", self->file.fileName).detail("Len", self->file.fileSize);
															#line 1349 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 656 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		self->decodeFile(buf);
															#line 657 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<OpenFileImplActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~OpenFileImplActor1State(); static_cast<OpenFileImplActor1*>(this)->destroy(); return 0; }
															#line 1355 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<OpenFileImplActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~OpenFileImplActor1State();
		static_cast<OpenFileImplActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(int const& rLen,int loopDepth) 
	{
		loopDepth = a_body1cont2(rLen, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(int && rLen,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(rLen), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<OpenFileImplActor1*>(this)->actor_wait_state > 0) static_cast<OpenFileImplActor1*>(this)->actor_wait_state = 0;
		static_cast<OpenFileImplActor1*>(this)->ActorCallback< OpenFileImplActor1, 1, int >::remove();

	}
	void a_callback_fire(ActorCallback< OpenFileImplActor1, 1, int >*,int const& value) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< OpenFileImplActor1, 1, int >*,int && value) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< OpenFileImplActor1, 1, int >*,Error err) 
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	DecodeRangeProgress* self;
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<IBackupContainer> container;
															#line 625 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Standalone<StringRef> buf;
															#line 1444 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
};
// This generated class is to be used only via openFileImpl()
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class OpenFileImplActor1 final : public Actor<Void>, public ActorCallback< OpenFileImplActor1, 0, Reference<IAsyncFile> >, public ActorCallback< OpenFileImplActor1, 1, int >, public FastAllocated<OpenFileImplActor1>, public OpenFileImplActor1State<OpenFileImplActor1> {
															#line 1449 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
	using FastAllocated<OpenFileImplActor1>::operator new;
	using FastAllocated<OpenFileImplActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14988936481147074304UL, 9551464234944654080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< OpenFileImplActor1, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< OpenFileImplActor1, 1, int >;
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	OpenFileImplActor1(DecodeRangeProgress* const& self,Reference<IBackupContainer> const& container) 
															#line 1467 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   OpenFileImplActor1State<OpenFileImplActor1>(self, container),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("openFileImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3444886599382641408UL, 7355907575748708608UL);
		ActorExecutionContextHelper __helper(static_cast<OpenFileImplActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("openFileImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("openFileImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< OpenFileImplActor1, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< OpenFileImplActor1, 1, int >*)0, actor_cancelled()); break;
		}

	}
};
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
[[nodiscard]] static Future<Void> openFileImpl( DecodeRangeProgress* const& self, Reference<IBackupContainer> const& container ) {
															#line 620 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	return Future<Void>(new OpenFileImplActor1(self, container));
															#line 1500 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
}

#line 659 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"

	// Reads a file content in the buffer, decodes it into key/value pairs, and stores these pairs.
	void decodeFile(const Standalone<StringRef>& buf) {
		try {
			loop {
				// process one block at a time
				int64_t len = std::min<int64_t>(file.blockSize, file.fileSize - offset);
				if (len == 0) {
					return;
				}

				Standalone<VectorRef<KeyValueRef>> chunks = fileBackup::decodeRangeFileBlock(buf.substr(offset, len));
				blocks.push_back(chunks);
				offset += len;
			}
		} catch (Error& e) {
			TraceEvent(SevWarn, "CorruptRangeFileBlock")
			    .error(e)
			    .detail("Filename", file.fileName)
			    .detail("BlockOffset", offset)
			    .detail("BlockLen", file.blockSize);
			throw;
		}
	}

	RangeFile file;
	Reference<IAsyncFile> fd;
	int64_t offset = 0;
	bool save = false;
	int lfd = -1; // local file descriptor
};

// convert a StringRef to Hex string
std::string hexStringRef(const StringRef& s) {
	std::string result;
	result.reserve(s.size() * 2);
	for (int i = 0; i < s.size(); i++) {
		result.append(format("%02x", s[i]));
	}
	return result;
}

															#line 1546 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
// This generated class is to be used only via process_range_file()
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
template <class Process_range_fileActor>
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class Process_range_fileActorState {
															#line 1552 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Process_range_fileActorState(Reference<IBackupContainer> const& container,RangeFile const& file,UID const& uid,Reference<DecodeParams> const& params) 
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		 : container(container),
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   file(file),
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   uid(uid),
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   params(params)
															#line 1565 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("process_range_file", reinterpret_cast<unsigned long>(this));

	}
	~Process_range_fileActorState() 
	{
		fdb_probe_actor_destroy("process_range_file", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 706 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (file.fileSize == 0)
															#line 1580 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 707 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent("SkipEmptyFile", uid).detail("Name", file.fileName);
															#line 708 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!static_cast<Process_range_fileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Process_range_fileActorState(); static_cast<Process_range_fileActor*>(this)->destroy(); return 0; }
															#line 1586 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				new (&static_cast<Process_range_fileActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~Process_range_fileActorState();
				static_cast<Process_range_fileActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 711 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			progress = DecodeRangeProgress(file, params->save_file_locally);
															#line 712 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<Void> __when_expr_0 = progress.openFile(container);
															#line 712 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<Process_range_fileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1598 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Process_range_fileActor*>(this)->actor_wait_state = 1;
															#line 712 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Process_range_fileActor, 0, Void >*>(static_cast<Process_range_fileActor*>(this)));
															#line 1603 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Process_range_fileActorState();
		static_cast<Process_range_fileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 714 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		for( auto& block : progress.blocks ) {
															#line 715 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			for( const auto& kv : block ) {
															#line 716 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				bool print = params->prefixes.empty();
															#line 718 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!print)
															#line 1632 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 719 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					print = params->matchFilters(kv);
															#line 1636 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
															#line 722 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (print)
															#line 1640 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 723 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					TraceEvent(format("KVPair_%llu", file.version).c_str(), uid) .detail("Version", file.version) .setMaxFieldLength(1000) .detail("KV", kv);
															#line 727 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					std::cout << file.version << " key: " << hexStringRef(kv.key) << "  value: " << hexStringRef(kv.value) << std::endl;
															#line 1646 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
			}
		}
															#line 732 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("ProcessRangeFileDone", uid).detail("File", file.fileName);
															#line 734 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<Process_range_fileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Process_range_fileActorState(); static_cast<Process_range_fileActor*>(this)->destroy(); return 0; }
															#line 1654 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<Process_range_fileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Process_range_fileActorState();
		static_cast<Process_range_fileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 714 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		for( auto& block : progress.blocks ) {
															#line 715 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			for( const auto& kv : block ) {
															#line 716 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				bool print = params->prefixes.empty();
															#line 718 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!print)
															#line 1672 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 719 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					print = params->matchFilters(kv);
															#line 1676 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
															#line 722 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (print)
															#line 1680 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 723 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					TraceEvent(format("KVPair_%llu", file.version).c_str(), uid) .detail("Version", file.version) .setMaxFieldLength(1000) .detail("KV", kv);
															#line 727 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					std::cout << file.version << " key: " << hexStringRef(kv.key) << "  value: " << hexStringRef(kv.value) << std::endl;
															#line 1686 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
			}
		}
															#line 732 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("ProcessRangeFileDone", uid).detail("File", file.fileName);
															#line 734 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<Process_range_fileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Process_range_fileActorState(); static_cast<Process_range_fileActor*>(this)->destroy(); return 0; }
															#line 1694 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<Process_range_fileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Process_range_fileActorState();
		static_cast<Process_range_fileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Process_range_fileActor*>(this)->actor_wait_state > 0) static_cast<Process_range_fileActor*>(this)->actor_wait_state = 0;
		static_cast<Process_range_fileActor*>(this)->ActorCallback< Process_range_fileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Process_range_fileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("process_range_file", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Process_range_fileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("process_range_file", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Process_range_fileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("process_range_file", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Process_range_fileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("process_range_file", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Process_range_fileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("process_range_file", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Process_range_fileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("process_range_file", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<IBackupContainer> container;
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	RangeFile file;
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	UID uid;
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<DecodeParams> params;
															#line 711 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	DecodeRangeProgress progress;
															#line 1787 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
};
// This generated class is to be used only via process_range_file()
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class Process_range_fileActor final : public Actor<Void>, public ActorCallback< Process_range_fileActor, 0, Void >, public FastAllocated<Process_range_fileActor>, public Process_range_fileActorState<Process_range_fileActor> {
															#line 1792 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
	using FastAllocated<Process_range_fileActor>::operator new;
	using FastAllocated<Process_range_fileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7470376618748136192UL, 18314679639838892544UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Process_range_fileActor, 0, Void >;
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Process_range_fileActor(Reference<IBackupContainer> const& container,RangeFile const& file,UID const& uid,Reference<DecodeParams> const& params) 
															#line 1809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   Process_range_fileActorState<Process_range_fileActor>(container, file, uid, params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("process_range_file", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5099434470983954432UL, 6526348685823392768UL);
		ActorExecutionContextHelper __helper(static_cast<Process_range_fileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("process_range_file");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("process_range_file", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Process_range_fileActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
[[nodiscard]] Future<Void> process_range_file( Reference<IBackupContainer> const& container, RangeFile const& file, UID const& uid, Reference<DecodeParams> const& params ) {
															#line 701 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	return Future<Void>(new Process_range_fileActor(container, file, uid, params));
															#line 1841 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
}

#line 736 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"

															#line 1846 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
// This generated class is to be used only via process_file()
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
template <class Process_fileActor>
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class Process_fileActorState {
															#line 1852 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Process_fileActorState(Reference<IBackupContainer> const& container,LogFile const& file,UID const& uid,Reference<DecodeParams> const& params) 
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		 : container(container),
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   file(file),
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   uid(uid),
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   params(params)
															#line 1865 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("process_file", reinterpret_cast<unsigned long>(this));

	}
	~Process_fileActorState() 
	{
		fdb_probe_actor_destroy("process_file", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 741 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (file.fileSize == 0)
															#line 1880 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 742 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent("SkipEmptyFile", uid).detail("Name", file.fileName);
															#line 743 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!static_cast<Process_fileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Process_fileActorState(); static_cast<Process_fileActor*>(this)->destroy(); return 0; }
															#line 1886 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				new (&static_cast<Process_fileActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~Process_fileActorState();
				static_cast<Process_fileActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 746 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			progress = DecodeProgress(file, params->save_file_locally);
															#line 747 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<Void> __when_expr_0 = progress.openFile(container);
															#line 747 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<Process_fileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1898 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Process_fileActor*>(this)->actor_wait_state = 1;
															#line 747 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Process_fileActor, 0, Void >*>(static_cast<Process_fileActor*>(this)));
															#line 1903 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Process_fileActorState();
		static_cast<Process_fileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 748 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		for(;true;) {
															#line 749 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			auto batch = progress.getNextBatch();
															#line 750 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (!batch.present())
															#line 1930 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
				break;
			}
															#line 753 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			const VersionedMutations& vms = batch.get();
															#line 754 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (vms.version < params->beginVersionFilter || vms.version >= params->endVersionFilter)
															#line 1938 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 755 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent("SkipVersion").detail("Version", vms.version);
															#line 1942 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				continue;
			}
															#line 759 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			int sub = 0;
															#line 760 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			for( const auto& m : vms.mutations ) {
															#line 761 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				sub++;
															#line 762 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				bool print = params->prefixes.empty();
															#line 764 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!print)
															#line 1955 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 765 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					print = params->matchFilters(m);
															#line 1959 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
															#line 767 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (print)
															#line 1963 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 768 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					TraceEvent(format("Mutation_%llu_%d", vms.version, sub).c_str(), uid) .detail("Version", vms.version) .setMaxFieldLength(1000) .detail("M", m.toString());
															#line 772 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					std::cout << vms.version << "." << sub << " " << typeString[(int)m.type] << " param1: " << hexStringRef(m.param1) << " param2: " << hexStringRef(m.param2) << "\n";
															#line 1969 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
			}
		}
															#line 777 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("ProcessFileDone", uid).detail("File", file.fileName);
															#line 778 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<Process_fileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Process_fileActorState(); static_cast<Process_fileActor*>(this)->destroy(); return 0; }
															#line 1977 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<Process_fileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Process_fileActorState();
		static_cast<Process_fileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 748 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		for(;true;) {
															#line 749 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			auto batch = progress.getNextBatch();
															#line 750 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (!batch.present())
															#line 1993 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
				break;
			}
															#line 753 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			const VersionedMutations& vms = batch.get();
															#line 754 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (vms.version < params->beginVersionFilter || vms.version >= params->endVersionFilter)
															#line 2001 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 755 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				TraceEvent("SkipVersion").detail("Version", vms.version);
															#line 2005 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				continue;
			}
															#line 759 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			int sub = 0;
															#line 760 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			for( const auto& m : vms.mutations ) {
															#line 761 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				sub++;
															#line 762 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				bool print = params->prefixes.empty();
															#line 764 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (!print)
															#line 2018 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 765 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					print = params->matchFilters(m);
															#line 2022 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
															#line 767 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				if (print)
															#line 2026 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				{
															#line 768 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					TraceEvent(format("Mutation_%llu_%d", vms.version, sub).c_str(), uid) .detail("Version", vms.version) .setMaxFieldLength(1000) .detail("M", m.toString());
															#line 772 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
					std::cout << vms.version << "." << sub << " " << typeString[(int)m.type] << " param1: " << hexStringRef(m.param1) << " param2: " << hexStringRef(m.param2) << "\n";
															#line 2032 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
				}
			}
		}
															#line 777 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("ProcessFileDone", uid).detail("File", file.fileName);
															#line 778 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<Process_fileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Process_fileActorState(); static_cast<Process_fileActor*>(this)->destroy(); return 0; }
															#line 2040 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<Process_fileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Process_fileActorState();
		static_cast<Process_fileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Process_fileActor*>(this)->actor_wait_state > 0) static_cast<Process_fileActor*>(this)->actor_wait_state = 0;
		static_cast<Process_fileActor*>(this)->ActorCallback< Process_fileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Process_fileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("process_file", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Process_fileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("process_file", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Process_fileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("process_file", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Process_fileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("process_file", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Process_fileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("process_file", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Process_fileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("process_file", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<IBackupContainer> container;
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	LogFile file;
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	UID uid;
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<DecodeParams> params;
															#line 746 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	DecodeProgress progress;
															#line 2133 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
};
// This generated class is to be used only via process_file()
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class Process_fileActor final : public Actor<Void>, public ActorCallback< Process_fileActor, 0, Void >, public FastAllocated<Process_fileActor>, public Process_fileActorState<Process_fileActor> {
															#line 2138 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
	using FastAllocated<Process_fileActor>::operator new;
	using FastAllocated<Process_fileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18298539725318361600UL, 6298896217623337728UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Process_fileActor, 0, Void >;
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Process_fileActor(Reference<IBackupContainer> const& container,LogFile const& file,UID const& uid,Reference<DecodeParams> const& params) 
															#line 2155 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   Process_fileActorState<Process_fileActor>(container, file, uid, params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("process_file", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11319499860388264192UL, 2422351918744320512UL);
		ActorExecutionContextHelper __helper(static_cast<Process_fileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("process_file");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("process_file", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Process_fileActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
[[nodiscard]] Future<Void> process_file( Reference<IBackupContainer> const& container, LogFile const& file, UID const& uid, Reference<DecodeParams> const& params ) {
															#line 737 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	return Future<Void>(new Process_fileActor(container, file, uid, params));
															#line 2187 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
}

#line 780 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"

// Use the snapshot metadata to quickly identify relevant range files and
// then filter by versions.
															#line 2194 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
// This generated class is to be used only via getRangeFiles()
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
template <class GetRangeFilesActor>
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class GetRangeFilesActorState {
															#line 2200 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	GetRangeFilesActorState(Reference<IBackupContainer> const& bc,Reference<DecodeParams> const& params) 
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		 : bc(bc),
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   params(params)
															#line 2209 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getRangeFiles", reinterpret_cast<unsigned long>(this));

	}
	~GetRangeFilesActorState() 
	{
		fdb_probe_actor_destroy("getRangeFiles", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 784 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<std::vector<KeyspaceSnapshotFile>> __when_expr_0 = (dynamic_cast<BackupContainerFileSystem*>(bc.getPtr()))->listKeyspaceSnapshots();
															#line 784 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<GetRangeFilesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2226 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetRangeFilesActor*>(this)->actor_wait_state = 1;
															#line 784 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetRangeFilesActor, 0, std::vector<KeyspaceSnapshotFile> >*>(static_cast<GetRangeFilesActor*>(this)));
															#line 2231 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetRangeFilesActorState();
		static_cast<GetRangeFilesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 786 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		files = std::vector<RangeFile>();
															#line 788 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		i = 0;
															#line 789 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		;
															#line 2258 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<KeyspaceSnapshotFile> const& __snapshots,int loopDepth) 
	{
															#line 784 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		snapshots = __snapshots;
															#line 2267 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<KeyspaceSnapshotFile> && __snapshots,int loopDepth) 
	{
		snapshots = std::move(__snapshots);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetRangeFilesActor*>(this)->actor_wait_state > 0) static_cast<GetRangeFilesActor*>(this)->actor_wait_state = 0;
		static_cast<GetRangeFilesActor*>(this)->ActorCallback< GetRangeFilesActor, 0, std::vector<KeyspaceSnapshotFile> >::remove();

	}
	void a_callback_fire(ActorCallback< GetRangeFilesActor, 0, std::vector<KeyspaceSnapshotFile> >*,std::vector<KeyspaceSnapshotFile> const& value) 
	{
		fdb_probe_actor_enter("getRangeFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRangeFilesActor, 0, std::vector<KeyspaceSnapshotFile> >*,std::vector<KeyspaceSnapshotFile> && value) 
	{
		fdb_probe_actor_enter("getRangeFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetRangeFilesActor, 0, std::vector<KeyspaceSnapshotFile> >*,Error err) 
	{
		fdb_probe_actor_enter("getRangeFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 806 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<GetRangeFilesActor*>(this)->SAV<std::vector<RangeFile>>::futures) { (void)(getRelevantRangeFiles(files, params)); this->~GetRangeFilesActorState(); static_cast<GetRangeFilesActor*>(this)->destroy(); return 0; }
															#line 2346 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<GetRangeFilesActor*>(this)->SAV< std::vector<RangeFile> >::value()) std::vector<RangeFile>(getRelevantRangeFiles(files, params));
		this->~GetRangeFilesActorState();
		static_cast<GetRangeFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 789 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!(i < snapshots.size()))
															#line 2365 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 791 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>>> __when_expr_1 = (dynamic_cast<BackupContainerFileSystem*>(bc.getPtr()))->readKeyspaceSnapshot(snapshots[i]);
															#line 791 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<GetRangeFilesActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2374 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetRangeFilesActor*>(this)->actor_wait_state = 2;
															#line 791 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetRangeFilesActor, 1, std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> >*>(static_cast<GetRangeFilesActor*>(this)));
															#line 2379 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 789 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		i++;
															#line 2407 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 800 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			TraceEvent("ReadKeyspaceSnapshotError").error(e).detail("I", i);
															#line 801 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (e.code() != error_code_restore_missing_data)
															#line 2419 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 802 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 2423 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> const& results,int loopDepth) 
	{
															#line 793 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		for( const auto& rangeFile : results.first ) {
															#line 794 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			const auto& keyRange = results.second.at(rangeFile.fileName);
															#line 795 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (params->matchFilters(keyRange))
															#line 2443 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 796 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				files.push_back(rangeFile);
															#line 2447 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> && results,int loopDepth) 
	{
															#line 793 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		for( const auto& rangeFile : results.first ) {
															#line 794 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			const auto& keyRange = results.second.at(rangeFile.fileName);
															#line 795 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (params->matchFilters(keyRange))
															#line 2462 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			{
															#line 796 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
				files.push_back(rangeFile);
															#line 2466 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> const& results,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(results, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> && results,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(results), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetRangeFilesActor*>(this)->actor_wait_state > 0) static_cast<GetRangeFilesActor*>(this)->actor_wait_state = 0;
		static_cast<GetRangeFilesActor*>(this)->ActorCallback< GetRangeFilesActor, 1, std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> >::remove();

	}
	void a_callback_fire(ActorCallback< GetRangeFilesActor, 1, std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> >*,std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> const& value) 
	{
		fdb_probe_actor_enter("getRangeFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeFiles", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetRangeFilesActor, 1, std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> >*,std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> && value) 
	{
		fdb_probe_actor_enter("getRangeFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeFiles", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetRangeFilesActor, 1, std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> >*,Error err) 
	{
		fdb_probe_actor_enter("getRangeFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRangeFiles", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<IBackupContainer> bc;
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<DecodeParams> params;
															#line 784 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	std::vector<KeyspaceSnapshotFile> snapshots;
															#line 786 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	std::vector<RangeFile> files;
															#line 788 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	int i;
															#line 2571 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getRangeFiles()
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class GetRangeFilesActor final : public Actor<std::vector<RangeFile>>, public ActorCallback< GetRangeFilesActor, 0, std::vector<KeyspaceSnapshotFile> >, public ActorCallback< GetRangeFilesActor, 1, std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> >, public FastAllocated<GetRangeFilesActor>, public GetRangeFilesActorState<GetRangeFilesActor> {
															#line 2576 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetRangeFilesActor>::operator new;
	using FastAllocated<GetRangeFilesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13453654841254808576UL, 456730207392406784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<RangeFile>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetRangeFilesActor, 0, std::vector<KeyspaceSnapshotFile> >;
friend struct ActorCallback< GetRangeFilesActor, 1, std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> >;
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	GetRangeFilesActor(Reference<IBackupContainer> const& bc,Reference<DecodeParams> const& params) 
															#line 2594 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		 : Actor<std::vector<RangeFile>>(),
		   GetRangeFilesActorState<GetRangeFilesActor>(bc, params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getRangeFiles", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5259650601384680448UL, 9179222199584152832UL);
		ActorExecutionContextHelper __helper(static_cast<GetRangeFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getRangeFiles");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getRangeFiles", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetRangeFilesActor, 0, std::vector<KeyspaceSnapshotFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetRangeFilesActor, 1, std::pair<std::vector<RangeFile>, std::map<std::string, KeyRange>> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
[[nodiscard]] Future<std::vector<RangeFile>> getRangeFiles( Reference<IBackupContainer> const& bc, Reference<DecodeParams> const& params ) {
															#line 783 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	return Future<std::vector<RangeFile>>(new GetRangeFilesActor(bc, params));
															#line 2627 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
}

#line 808 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"

															#line 2632 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
// This generated class is to be used only via decode_logs()
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
template <class Decode_logsActor>
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class Decode_logsActorState {
															#line 2638 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Decode_logsActorState(Reference<DecodeParams> const& params) 
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		 : params(params),
															#line 810 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   container(IBackupContainer::openContainer(params->container_url, params->proxy, {})),
															#line 812 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		   uid(deterministicRandom()->randomUniqueID())
															#line 2649 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("decode_logs", reinterpret_cast<unsigned long>(this));

	}
	~Decode_logsActorState() 
	{
		fdb_probe_actor_destroy("decode_logs", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 813 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<BackupFileList> __when_expr_0 = container->dumpFileList();
															#line 813 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<Decode_logsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2666 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Decode_logsActor*>(this)->actor_wait_state = 1;
															#line 813 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Decode_logsActor, 0, BackupFileList >*>(static_cast<Decode_logsActor*>(this)));
															#line 2671 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Decode_logsActorState();
		static_cast<Decode_logsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 815 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		listing.logs.erase(std::remove_if(listing.logs.begin(), listing.logs.end(), [](const LogFile& file) { std::string prefix("plogs/"); return file.fileName.substr(0, prefix.size()) == prefix; }), listing.logs.end());
															#line 822 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		std::sort(listing.logs.begin(), listing.logs.end());
															#line 823 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("Container", uid).detail("URL", params->container_url).detail("Logs", listing.logs.size());
															#line 824 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("DecodeParam", uid).setMaxFieldLength(100000).detail("Value", params->toString());
															#line 826 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		StrictFuture<BackupDescription> __when_expr_1 = container->describeBackup();
															#line 826 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (static_cast<Decode_logsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2704 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Decode_logsActor*>(this)->actor_wait_state = 2;
															#line 826 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Decode_logsActor, 1, BackupDescription >*>(static_cast<Decode_logsActor*>(this)));
															#line 2709 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(BackupFileList const& __listing,int loopDepth) 
	{
															#line 813 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		listing = __listing;
															#line 2718 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(BackupFileList && __listing,int loopDepth) 
	{
		listing = std::move(__listing);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Decode_logsActor*>(this)->actor_wait_state > 0) static_cast<Decode_logsActor*>(this)->actor_wait_state = 0;
		static_cast<Decode_logsActor*>(this)->ActorCallback< Decode_logsActor, 0, BackupFileList >::remove();

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 0, BackupFileList >*,BackupFileList const& value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 0, BackupFileList >*,BackupFileList && value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Decode_logsActor, 0, BackupFileList >*,Error err) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(BackupDescription const& desc,int loopDepth) 
	{
															#line 827 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		std::cout << "\n" << desc.toString() << "\n";
															#line 829 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		logFiles = std::vector<LogFile>();
															#line 830 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		rangeFiles = std::vector<RangeFile>();
															#line 832 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (params->decode_logs)
															#line 2803 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 833 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			logFiles = getRelevantLogFiles(listing.logs, params);
															#line 834 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			printLogFiles("Relevant log files are: ", logFiles);
															#line 2809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 837 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (params->decode_range)
															#line 2813 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 839 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<std::vector<RangeFile>> __when_expr_2 = getRangeFiles(container, params);
															#line 839 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<Decode_logsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2819 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<Decode_logsActor*>(this)->actor_wait_state = 3;
															#line 839 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >*>(static_cast<Decode_logsActor*>(this)));
															#line 2824 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(BackupDescription && desc,int loopDepth) 
	{
															#line 827 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		std::cout << "\n" << desc.toString() << "\n";
															#line 829 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		logFiles = std::vector<LogFile>();
															#line 830 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		rangeFiles = std::vector<RangeFile>();
															#line 832 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (params->decode_logs)
															#line 2844 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 833 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			logFiles = getRelevantLogFiles(listing.logs, params);
															#line 834 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			printLogFiles("Relevant log files are: ", logFiles);
															#line 2850 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		}
															#line 837 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (params->decode_range)
															#line 2854 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 839 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			StrictFuture<std::vector<RangeFile>> __when_expr_2 = getRangeFiles(container, params);
															#line 839 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (static_cast<Decode_logsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2860 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<Decode_logsActor*>(this)->actor_wait_state = 3;
															#line 839 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >*>(static_cast<Decode_logsActor*>(this)));
															#line 2865 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(BackupDescription const& desc,int loopDepth) 
	{
		loopDepth = a_body1cont2(desc, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(BackupDescription && desc,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(desc), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Decode_logsActor*>(this)->actor_wait_state > 0) static_cast<Decode_logsActor*>(this)->actor_wait_state = 0;
		static_cast<Decode_logsActor*>(this)->ActorCallback< Decode_logsActor, 1, BackupDescription >::remove();

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 1, BackupDescription >*,BackupDescription const& value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 1, BackupDescription >*,BackupDescription && value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Decode_logsActor, 1, BackupDescription >*,Error err) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 844 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("TotalFiles", uid).detail("LogFiles", logFiles.size()).detail("RangeFiles", rangeFiles.size());
															#line 846 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (params->list_only)
															#line 2956 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 847 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			if (!static_cast<Decode_logsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Decode_logsActorState(); static_cast<Decode_logsActor*>(this)->destroy(); return 0; }
															#line 2960 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			new (&static_cast<Decode_logsActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~Decode_logsActorState();
			static_cast<Decode_logsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 850 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		idx = 0;
															#line 851 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (params->decode_logs)
															#line 2970 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 852 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			;
															#line 2974 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = a_body1cont3loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(std::vector<RangeFile> const& files,int loopDepth) 
	{
															#line 840 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		rangeFiles = files;
															#line 841 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		printLogFiles("Relevant range files are: ", rangeFiles);
															#line 2990 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(std::vector<RangeFile> && files,int loopDepth) 
	{
															#line 840 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		rangeFiles = files;
															#line 841 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		printLogFiles("Relevant range files are: ", rangeFiles);
															#line 3001 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(std::vector<RangeFile> const& files,int loopDepth) 
	{
		loopDepth = a_body1cont5(files, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(std::vector<RangeFile> && files,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(files), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<Decode_logsActor*>(this)->actor_wait_state > 0) static_cast<Decode_logsActor*>(this)->actor_wait_state = 0;
		static_cast<Decode_logsActor*>(this)->ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >::remove();

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >*,std::vector<RangeFile> const& value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >*,std::vector<RangeFile> && value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >*,Error err) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 861 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (params->decode_range)
															#line 3085 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
															#line 862 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			idx = 0;
															#line 863 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
			;
															#line 3091 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
			loopDepth = a_body1cont7loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(int loopDepth) 
	{
															#line 857 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("DecodeLogsDone", uid).log();
															#line 3105 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 852 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!(idx < logFiles.size()))
															#line 3121 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
			return a_body1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 853 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("ProcessFile").detail("Name", logFiles[idx].fileName).detail("I", idx);
															#line 854 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		StrictFuture<Void> __when_expr_3 = process_file(container, logFiles[idx], uid, params);
															#line 854 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (static_cast<Decode_logsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3131 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<Decode_logsActor*>(this)->actor_wait_state = 4;
															#line 854 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< Decode_logsActor, 3, Void >*>(static_cast<Decode_logsActor*>(this)));
															#line 3136 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3break1(int loopDepth) 
	{
		try {
			return a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 855 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		idx++;
															#line 3158 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 855 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		idx++;
															#line 3167 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<Decode_logsActor*>(this)->actor_wait_state > 0) static_cast<Decode_logsActor*>(this)->actor_wait_state = 0;
		static_cast<Decode_logsActor*>(this)->ActorCallback< Decode_logsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< Decode_logsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont11(int loopDepth) 
	{
															#line 871 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!static_cast<Decode_logsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~Decode_logsActorState(); static_cast<Decode_logsActor*>(this)->destroy(); return 0; }
															#line 3251 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		new (&static_cast<Decode_logsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~Decode_logsActorState();
		static_cast<Decode_logsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont12(int loopDepth) 
	{
															#line 868 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("DecodeRangeFileDone", uid).log();
															#line 3263 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont7loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1(int loopDepth) 
	{
															#line 863 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (!(idx < rangeFiles.size()))
															#line 3279 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		{
			return a_body1cont7break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 864 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		TraceEvent("ProcessFile").detail("Name", rangeFiles[idx].fileName).detail("I", idx);
															#line 865 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		StrictFuture<Void> __when_expr_4 = process_range_file(container, rangeFiles[idx], uid, params);
															#line 865 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		if (static_cast<Decode_logsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3289 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1when1(__when_expr_4.get(), loopDepth); };
		static_cast<Decode_logsActor*>(this)->actor_wait_state = 5;
															#line 865 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< Decode_logsActor, 4, Void >*>(static_cast<Decode_logsActor*>(this)));
															#line 3294 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7break1(int loopDepth) 
	{
		try {
			return a_body1cont12(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 866 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		idx++;
															#line 3316 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont7loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 866 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
		idx++;
															#line 3325 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont7loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<Decode_logsActor*>(this)->actor_wait_state > 0) static_cast<Decode_logsActor*>(this)->actor_wait_state = 0;
		static_cast<Decode_logsActor*>(this)->ActorCallback< Decode_logsActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< Decode_logsActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< Decode_logsActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<DecodeParams> params;
															#line 810 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Reference<IBackupContainer> container;
															#line 812 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	UID uid;
															#line 813 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	BackupFileList listing;
															#line 829 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	std::vector<LogFile> logFiles;
															#line 830 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	std::vector<RangeFile> rangeFiles;
															#line 850 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	int idx;
															#line 3419 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
};
// This generated class is to be used only via decode_logs()
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
class Decode_logsActor final : public Actor<Void>, public ActorCallback< Decode_logsActor, 0, BackupFileList >, public ActorCallback< Decode_logsActor, 1, BackupDescription >, public ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >, public ActorCallback< Decode_logsActor, 3, Void >, public ActorCallback< Decode_logsActor, 4, Void >, public FastAllocated<Decode_logsActor>, public Decode_logsActorState<Decode_logsActor> {
															#line 3424 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
public:
	using FastAllocated<Decode_logsActor>::operator new;
	using FastAllocated<Decode_logsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1731355596332898560UL, 17562445197479026432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Decode_logsActor, 0, BackupFileList >;
friend struct ActorCallback< Decode_logsActor, 1, BackupDescription >;
friend struct ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >;
friend struct ActorCallback< Decode_logsActor, 3, Void >;
friend struct ActorCallback< Decode_logsActor, 4, Void >;
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	Decode_logsActor(Reference<DecodeParams> const& params) 
															#line 3445 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   Decode_logsActorState<Decode_logsActor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("decode_logs", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3616119142404391680UL, 14480948263783511808UL);
		ActorExecutionContextHelper __helper(static_cast<Decode_logsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("decode_logs");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("decode_logs", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Decode_logsActor, 0, BackupFileList >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Decode_logsActor, 1, BackupDescription >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< Decode_logsActor, 2, std::vector<RangeFile> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< Decode_logsActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< Decode_logsActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
[[nodiscard]] Future<Void> decode_logs( Reference<DecodeParams> const& params ) {
															#line 809 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"
	return Future<Void>(new Decode_logsActor(params));
															#line 3481 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/build_output/fdbbackup/FileDecoder.actor.g.cpp.py_gen"
}

#line 873 "/codebuild/output/src2038086582/src/github.com/apple/foundationdb/fdbbackup/FileDecoder.actor.cpp"

} // namespace file_converter

int main(int argc, char** argv) {
	std::string commandLine;
	for (int a = 0; a < argc; a++) {
		if (a)
			commandLine += ' ';
		commandLine += argv[a];
	}

	try {
		std::unique_ptr<CSimpleOpt> args(
		    new CSimpleOpt(argc, argv, file_converter::gConverterOptions, SO_O_EXACT | SO_O_HYPHEN_TO_UNDERSCORE));
		auto param = makeReference<file_converter::DecodeParams>();
		int status = file_converter::parseDecodeCommandLine(param, args.get());
		std::cout << "Params: " << param->toString() << "\n";
		param->updateRangeMap();
		if (status != FDB_EXIT_SUCCESS) {
			file_converter::printDecodeUsage();
			return status;
		}

		if (param->log_enabled) {
			if (param->log_dir.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_ENABLE);
			} else {
				setNetworkOption(FDBNetworkOptions::TRACE_ENABLE, StringRef(param->log_dir));
			}
			if (!param->trace_format.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_FORMAT, StringRef(param->trace_format));
			} else {
				setNetworkOption(FDBNetworkOptions::TRACE_FORMAT, "json"_sr);
			}
			if (!param->trace_log_group.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_LOG_GROUP, StringRef(param->trace_log_group));
			}
		}

		if (!param->tlsConfig.setupTLS()) {
			TraceEvent(SevError, "TLSError").log();
			throw tls_error();
		}

		platformInit();
		Error::init();

		StringRef url(param->container_url);
		setupNetwork(0, UseMetrics::True);

		// Must be called after setupNetwork() to be effective
		param->updateKnobs();

		TraceEvent("ProgramStart")
		    .setMaxEventLength(12000)
		    .detail("SourceVersion", getSourceVersion())
		    .detail("Version", FDB_VT_VERSION)
		    .detail("PackageName", FDB_VT_PACKAGE_NAME)
		    .detailf("ActualTime", "%lld", DEBUG_DETERMINISM ? 0 : time(NULL))
		    .setMaxFieldLength(10000)
		    .detail("CommandLine", commandLine)
		    .setMaxFieldLength(0)
		    .trackLatest("ProgramStart");

		TraceEvent::setNetworkThread();
		openTraceFile({}, 10 << 20, 500 << 20, param->log_dir, "decode", param->trace_log_group);
		param->tlsConfig.setupBlobCredentials();

		auto f = stopAfter(decode_logs(param));

		runNetwork();

		flushTraceFileVoid();
		fflush(stdout);
		closeTraceFile();

		return status;
	} catch (Error& e) {
		std::cerr << "ERROR: " << e.what() << "\n";
		return FDB_EXIT_ERROR;
	} catch (std::exception& e) {
		TraceEvent(SevError, "MainError").error(unknown_error()).detail("RootException", e.what());
		return FDB_EXIT_MAIN_EXCEPTION;
	}
}
