#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
/*
 * RESTSimKmsVault.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/BlobMetadataUtils.h"
#include "fdbserver/RESTSimKmsVault.h"
#include "fdbclient/SimKmsVault.h"
#include "fdbrpc/HTTP.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/RESTKmsConnectorUtils.h"
#include "flow/Arena.h"
#include "flow/EncryptUtils.h"

#include <cstring>
#include <rapidjson/document.h>
#include <rapidjson/rapidjson.h>
#include <rapidjson/stringbuffer.h>
#include <rapidjson/writer.h>

#include "flow/FastRef.h"
#include "flow/IRandom.h"
#include "flow/Knobs.h"
#include "flow/Trace.h"
#include "flow/actorcompiler.h" // This must be the last #include.

using DomIdVec = std::vector<EncryptCipherDomainId>;
using BaseCipherDomIdVec = std::vector<std::pair<EncryptCipherBaseKeyId, Optional<EncryptCipherDomainId>>>;

using namespace RESTKmsConnectorUtils;

namespace {
const std::string missingVersionMsg = "Missing version";
const std::string missingVersionCode = "1234";
const std::string invalidVersionMsg = "Invalid version";
const std::string invalidVersionCode = "5678";
const std::string missingTokensMsg = "Missing validation tokens";
const std::string missingTokenCode = "0123";

const std::string bgUrl = "file://simfdb/fdbblob/";

struct VaultResponse {
	bool failed;
	std::string buff;

	VaultResponse() : failed(false), buff("") {}
};

} // namespace

int64_t getRefreshInterval(const int64_t now, const int64_t defaultTtl) {
	if (BUGGIFY) {
		return now;
	}
	return (now + defaultTtl);
}

int64_t getExpireInterval(const int64_t refTS, const int64_t defaultTtl) {

	if (BUGGIFY) {
		return -1;
	}
	return (refTS + defaultTtl);
}

void validateHeaders(const HTTP::Headers& toCompare) {
	if (toCompare != RESTKmsConnectorUtils::getHTTPHeaders()) {
		throw rest_malformed_response();
	}
}

void addErrorToDoc(rapidjson::Document& doc, const ErrorDetail& details) {
	rapidjson::Value errorDetail(rapidjson::kObjectType);
	if (!details.errorMsg.empty()) {
		// Add "errorMsg"
		rapidjson::Value key(ERROR_MSG_TAG, doc.GetAllocator());
		rapidjson::Value errMsg;
		errMsg.SetString(details.errorMsg.data(), details.errorMsg.size(), doc.GetAllocator());
		errorDetail.AddMember(key, errMsg, doc.GetAllocator());
	}
	if (!details.errorCode.empty()) {
		// Add "value" - token value
		rapidjson::Value key(ERROR_CODE_TAG, doc.GetAllocator());
		rapidjson::Value errCode;
		errCode.SetString(details.errorCode.data(), details.errorCode.size(), doc.GetAllocator());
		errorDetail.AddMember(key, errCode, doc.GetAllocator());
	}

	// Append "error"
	rapidjson::Value key(ERROR_TAG, doc.GetAllocator());
	doc.AddMember(key, errorDetail, doc.GetAllocator());
}

void prepareErrorResponse(VaultResponse* response,
                          const ErrorDetail& errorDetail,
                          Optional<int> version = Optional<int>()) {
	rapidjson::Document doc;
	doc.SetObject();

	if (version.present()) {
		addVersionToDoc(doc, version.get());
	}

	addErrorToDoc(doc, errorDetail);

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	ASSERT(!response->failed);
	response->failed = true;
	response->buff = std::string(sb.GetString(), sb.GetSize());
}

// Helper routine to extract 'version' from the input json document. If 'version' is missing or 'invalid', the routine
// is responsible to populate required error details to the 'response'
bool extractVersion(const rapidjson::Document& doc, VaultResponse* response, int* version) {
	// check version tag sanityrest_malformed_response
	if (!doc.HasMember(REQUEST_VERSION_TAG) || !doc[REQUEST_VERSION_TAG].IsInt()) {
		prepareErrorResponse(response, ErrorDetail(missingVersionCode, missingVersionMsg));
		CODE_PROBE(true, "RESTSimKmsVault missing version");
		return false;
	}

	*version = doc[REQUEST_VERSION_TAG].GetInt();
	if (*version < 0 || *version > SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION) {
		prepareErrorResponse(response, ErrorDetail(invalidVersionCode, invalidVersionMsg));
		CODE_PROBE(true, "RESTSimKmsVault invalid version");
		return false;
	}

	return true;
}

// Helper routine to validate 'validation-token(s)' from the input json document. If tokens are missing the routine is
// responsible to populate appropriate error to the 'response'
bool checkValidationTokens(const rapidjson::Document& doc, const int version, VaultResponse* response) {
	ASSERT(!response->failed);
	if (!doc.HasMember(VALIDATION_TOKENS_TAG) || !doc[VALIDATION_TOKENS_TAG].IsArray()) {
		prepareErrorResponse(response, ErrorDetail(missingTokenCode, missingTokensMsg), version);
		CODE_PROBE(true, "RESTSimKmsVault missing validation tokens");
		return false;
	}
	ASSERT(!response->failed);
	return true;
}

void addCipherDetailToRespDoc(rapidjson::Document& doc,
                              rapidjson::Value& cipherDetails,
                              const Reference<SimKmsVaultKeyCtx>& keyCtx,
                              const Optional<EncryptCipherDomainId> domId) {
	rapidjson::Value cipherDetail(rapidjson::kObjectType);

	// Add 'base_cipher_id'
	rapidjson::Value key(BASE_CIPHER_ID_TAG, doc.GetAllocator());
	rapidjson::Value baseKeyId;
	baseKeyId.SetUint64(keyCtx->id);
	cipherDetail.AddMember(key, baseKeyId, doc.GetAllocator());

	// Add 'encrypt_domain_id'
	if (domId.present()) {
		key.SetString(ENCRYPT_DOMAIN_ID_TAG, doc.GetAllocator());
		rapidjson::Value domainId;
		domainId.SetInt64(domId.get());
		cipherDetail.AddMember(key, domainId, doc.GetAllocator());
	}

	// Add 'cipher'
	key.SetString(BASE_CIPHER_TAG, doc.GetAllocator());
	rapidjson::Value cipher;
	ASSERT_EQ(keyCtx->key.size(), keyCtx->keyLen);
	cipher.SetString(reinterpret_cast<const char*>(keyCtx->key.begin()), keyCtx->keyLen, doc.GetAllocator());
	cipherDetail.AddMember(key, cipher, doc.GetAllocator());

	// Add 'refreshAt'
	key.SetString(REFRESH_AFTER_SEC, doc.GetAllocator());
	const int64_t refreshAt = getRefreshInterval(now(), FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value refreshInterval;
	refreshInterval.SetInt64(refreshAt);
	cipherDetail.AddMember(key, refreshInterval, doc.GetAllocator());

	// Add 'expireAt
	key.SetString(EXPIRE_AFTER_SEC, doc.GetAllocator());
	const int64_t expireAt = getExpireInterval(refreshAt, FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value expireInterval;
	expireInterval.SetInt64(expireAt);
	cipherDetail.AddMember(key, expireInterval, doc.GetAllocator());

	// push above object to the array
	cipherDetails.PushBack(cipherDetail, doc.GetAllocator());
}

void addBlobMetadaToResDoc(rapidjson::Document& doc, rapidjson::Value& blobDetails, const EncryptCipherDomainId domId) {
	Standalone<BlobMetadataDetailsRef> detailsRef = SimKmsVault::getBlobMetadata(domId, bgUrl);
	rapidjson::Value blobDetail(rapidjson::kObjectType);

	rapidjson::Value key(BLOB_METADATA_DOMAIN_ID_TAG, doc.GetAllocator());
	rapidjson::Value domainId;
	domainId.SetInt64(domId);
	blobDetail.AddMember(key, domainId, doc.GetAllocator());

	rapidjson::Value locations(rapidjson::kArrayType);
	for (const auto& loc : detailsRef.locations) {
		rapidjson::Value location(rapidjson::kObjectType);

		// set location-id
		key.SetString(BLOB_METADATA_LOCATION_ID_TAG, doc.GetAllocator());
		rapidjson::Value id;
		id.SetInt64(loc.locationId);
		location.AddMember(key, id, doc.GetAllocator());

		// set location-path
		key.SetString(BLOB_METADATA_LOCATION_PATH_TAG, doc.GetAllocator());
		rapidjson::Value path;
		path.SetString(reinterpret_cast<const char*>(loc.path.begin()), loc.path.size(), doc.GetAllocator());
		location.AddMember(key, path, doc.GetAllocator());

		locations.PushBack(location, doc.GetAllocator());
	}
	key.SetString(BLOB_METADATA_LOCATIONS_TAG, doc.GetAllocator());
	blobDetail.AddMember(key, locations, doc.GetAllocator());

	// Add 'refreshAt'
	rapidjson::Value refreshKey(REFRESH_AFTER_SEC, doc.GetAllocator());
	const int64_t refreshAt = getRefreshInterval(now(), FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value refreshInterval;
	refreshInterval.SetInt64(refreshAt);
	blobDetail.AddMember(refreshKey, refreshInterval, doc.GetAllocator());

	// Add 'expireAt'
	rapidjson::Value expireKey(EXPIRE_AFTER_SEC, doc.GetAllocator());
	const int64_t expireAt = getExpireInterval(refreshAt, FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value expireInterval;
	expireInterval.SetInt64(expireAt);
	blobDetail.AddMember(expireKey, expireInterval, doc.GetAllocator());

	blobDetails.PushBack(blobDetail, doc.GetAllocator());
}

void addKmsUrlsToDoc(rapidjson::Document& doc) {
	rapidjson::Value kmsUrls(rapidjson::kArrayType);
	// FIXME: fetch latest KMS URLs && append to the doc
	rapidjson::Value memberKey(KMS_URLS_TAG, doc.GetAllocator());
	doc.AddMember(memberKey, kmsUrls, doc.GetAllocator());
}

// TODO: inject faults
VaultResponse handleFetchKeysByDomainIds(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;
	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'cipher_key_details' as json array
	rapidjson::Value cipherDetails(rapidjson::kArrayType);
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		// Check if ENCRYPT_DOMAIN_ID_TAG exists before accessing it
		if (!cipherDetail.HasMember(ENCRYPT_DOMAIN_ID_TAG) || !cipherDetail[ENCRYPT_DOMAIN_ID_TAG].IsInt64()) {
			continue; // Skip invalid entries
		}
		EncryptCipherDomainId domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domainId);
		ASSERT(keyCtx.isValid());
		addCipherDetailToRespDoc(result, cipherDetails, keyCtx, domainId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, cipherDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchByDomainIdsResponseStr").detail("Str", response->buff);
	return response;
}

VaultResponse handleFetchKeysByKeyIds(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;

	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'cipher_key_details' as json array
	rapidjson::Value cipherDetails(rapidjson::kArrayType);
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		Optional<EncryptCipherDomainId> domainId;
		if (cipherDetail.HasMember(ENCRYPT_DOMAIN_ID_TAG) && cipherDetail[ENCRYPT_DOMAIN_ID_TAG].IsInt64()) {
			domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		}
		EncryptCipherBaseKeyId baseCipherId = cipherDetail[BASE_CIPHER_ID_TAG].GetUint64();
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByBaseCipherId(baseCipherId);
		ASSERT(keyCtx.isValid());
		addCipherDetailToRespDoc(result, cipherDetails, keyCtx, domainId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, cipherDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchByKeyIdsResponseStr").detail("Str", response.buff);
	return response;
}

VaultResponse handleFetchBlobMetada(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;

	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'blob_metadata_details' as json array
	rapidjson::Value blobDetails(rapidjson::kArrayType);
	for (const auto& blobDetail : doc[BLOB_METADATA_DETAILS_TAG].GetArray()) {
		// Check if BLOB_METADATA_DOMAIN_ID_TAG exists before accessing it
		if (!blobDetail.HasMember(BLOB_METADATA_DOMAIN_ID_TAG) || !blobDetail[BLOB_METADATA_DOMAIN_ID_TAG].IsInt64()) {
			continue; // Skip invalid entries
		}
		EncryptCipherDomainId domainId = blobDetail[BLOB_METADATA_DOMAIN_ID_TAG].GetInt64();
		addBlobMetadaToResDoc(doc, blobDetails, domainId);
	}
	rapidjson::Value memberKey(BLOB_METADATA_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, blobDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchBlobMetadataResponeStr").detail("Str", response.buff);
	return response;
}

															#line 437 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via simKmsVaultRequestHandler()
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class SimKmsVaultRequestHandlerActor>
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class SimKmsVaultRequestHandlerActorState {
															#line 444 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	SimKmsVaultRequestHandlerActorState(Reference<HTTP::IncomingRequest> const& request,Reference<HTTP::OutgoingResponse> const& response) 
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : request(request),
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(response)
															#line 453 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this));

	}
	~SimKmsVaultRequestHandlerActorState() 
	{
		fdb_probe_actor_destroy("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 437 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0);
															#line 437 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 470 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state = 1;
															#line 437 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*>(static_cast<SimKmsVaultRequestHandlerActor*>(this)));
															#line 475 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 438 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT_EQ(request->verb, HTTP::HTTP_VERB_POST);
															#line 440 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		validateHeaders(request->data.headers);
															#line 442 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		vaultResponse = VaultResponse();
															#line 443 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_KEY_IDS_RESOURCE) == 0)
															#line 504 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		{
															#line 444 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			vaultResponse = handleFetchKeysByKeyIds(request->data.content);
															#line 508 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		}
		else
		{
															#line 445 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_DOMAIN_IDS_RESOURCE) == 0)
															#line 514 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			{
															#line 446 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				vaultResponse = handleFetchKeysByDomainIds(request->data.content);
															#line 518 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			}
			else
			{
															#line 447 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (request->resource.compare(REST_SIM_KMS_VAULT_GET_BLOB_METADATA_RESOURCE) == 0)
															#line 524 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				{
															#line 448 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					vaultResponse = handleFetchBlobMetada(request->data.content);
															#line 528 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				}
				else
				{
															#line 450 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					TraceEvent("UnexpectedResource").detail("Resource", request->resource);
															#line 451 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					return a_body1Catch1(http_bad_response(), loopDepth);
															#line 536 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				}
			}
		}
															#line 454 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->code = HTTP::HTTP_STATUS_CODE_OK;
															#line 455 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.headers = request->data.headers;
															#line 456 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		PacketWriter pw(response->data.content->getWriteBuffer(vaultResponse.buff.size()), nullptr, Unversioned());
															#line 457 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		pw.serializeBytes(vaultResponse.buff.data(), vaultResponse.buff.size());
															#line 458 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.contentLen = vaultResponse.buff.size();
															#line 460 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SimKmsVaultRequestHandlerActorState(); static_cast<SimKmsVaultRequestHandlerActor*>(this)->destroy(); return 0; }
															#line 552 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		new (&static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 438 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT_EQ(request->verb, HTTP::HTTP_VERB_POST);
															#line 440 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		validateHeaders(request->data.headers);
															#line 442 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		vaultResponse = VaultResponse();
															#line 443 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_KEY_IDS_RESOURCE) == 0)
															#line 570 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		{
															#line 444 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			vaultResponse = handleFetchKeysByKeyIds(request->data.content);
															#line 574 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		}
		else
		{
															#line 445 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_DOMAIN_IDS_RESOURCE) == 0)
															#line 580 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			{
															#line 446 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				vaultResponse = handleFetchKeysByDomainIds(request->data.content);
															#line 584 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			}
			else
			{
															#line 447 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (request->resource.compare(REST_SIM_KMS_VAULT_GET_BLOB_METADATA_RESOURCE) == 0)
															#line 590 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				{
															#line 448 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					vaultResponse = handleFetchBlobMetada(request->data.content);
															#line 594 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				}
				else
				{
															#line 450 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					TraceEvent("UnexpectedResource").detail("Resource", request->resource);
															#line 451 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					return a_body1Catch1(http_bad_response(), loopDepth);
															#line 602 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				}
			}
		}
															#line 454 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->code = HTTP::HTTP_STATUS_CODE_OK;
															#line 455 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.headers = request->data.headers;
															#line 456 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		PacketWriter pw(response->data.content->getWriteBuffer(vaultResponse.buff.size()), nullptr, Unversioned());
															#line 457 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		pw.serializeBytes(vaultResponse.buff.data(), vaultResponse.buff.size());
															#line 458 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.contentLen = vaultResponse.buff.size();
															#line 460 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SimKmsVaultRequestHandlerActorState(); static_cast<SimKmsVaultRequestHandlerActor*>(this)->destroy(); return 0; }
															#line 618 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		new (&static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state > 0) static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state = 0;
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 442 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	VaultResponse vaultResponse;
															#line 707 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via simKmsVaultRequestHandler()
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class SimKmsVaultRequestHandlerActor final : public Actor<Void>, public ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >, public FastAllocated<SimKmsVaultRequestHandlerActor>, public SimKmsVaultRequestHandlerActorState<SimKmsVaultRequestHandlerActor> {
															#line 712 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<SimKmsVaultRequestHandlerActor>::operator new;
	using FastAllocated<SimKmsVaultRequestHandlerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17841741362034161408UL, 12350771949214984704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >;
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	SimKmsVaultRequestHandlerActor(Reference<HTTP::IncomingRequest> const& request,Reference<HTTP::OutgoingResponse> const& response) 
															#line 729 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   SimKmsVaultRequestHandlerActorState<SimKmsVaultRequestHandlerActor>(request, response),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3523274803376955648UL, 4651189297213340672UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("simKmsVaultRequestHandler");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
[[nodiscard]] Future<Void> simKmsVaultRequestHandler( Reference<HTTP::IncomingRequest> const& request, Reference<HTTP::OutgoingResponse> const& response ) {
															#line 435 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new SimKmsVaultRequestHandlerActor(request, response));
															#line 762 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}

#line 462 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

Future<Void> RESTSimKmsVaultRequestHandler::handleRequest(Reference<HTTP::IncomingRequest> request,
                                                          Reference<HTTP::OutgoingResponse> response) {
	return simKmsVaultRequestHandler(request, response);
}

// Only used to link unit tests
void forceLinkRESTSimKmsVaultTest() {}

namespace {

enum class FaultType { NONE = 1, MISSING_VERSION = 2, INVALID_VERSION = 3, MISSING_VALIDATION_TOKEN = 4 };

void addFakeValidationTokens(rapidjson::Document& doc) {
	ValidationTokenMap tokenMap;
	tokenMap.emplace("foo", ValidationTokenCtx("bar", ValidationTokenSource::VALIDATION_TOKEN_SOURCE_FILE));

	addValidationTokensSectionToJsonDoc(doc, tokenMap);
}

void constructDomainIds(EncryptCipherDomainIdVec& domIds) {
	domIds.push_back(SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID);
	domIds.push_back(FDB_DEFAULT_ENCRYPT_DOMAIN_ID);
	domIds.push_back(ENCRYPT_HEADER_DOMAIN_ID);

	int idx = deterministicRandom()->randomInt(512, 786);
	int count = deterministicRandom()->randomInt(5, 100);
	while (count--) {
		domIds.push_back(idx++);
	}
}

std::string getFakeDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds,
                                           const char* rootTag,
                                           const char* elementTag,
                                           FaultType fault = FaultType::NONE) {
	rapidjson::Document doc;
	doc.SetObject();

	if (fault == FaultType::INVALID_VERSION) {
		addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION + 1);
	} else if (fault == FaultType::MISSING_VERSION) {
		// Skip adding the version
	} else {
		addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION);
	}

	if (fault != FaultType::MISSING_VALIDATION_TOKEN) {
		addFakeValidationTokens(doc);
	}

	constructDomainIds(domIds);
	addLatestDomainDetailsToDoc(doc, rootTag, elementTag, domIds);

	addRefreshKmsUrlsSectionToJsonDoc(doc, deterministicRandom()->coinflip());

	if (deterministicRandom()->coinflip()) {
		addDebugUidSectionToJsonDoc(doc, deterministicRandom()->randomUniqueID());
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	std::string resp(sb.GetString(), sb.GetSize());
	/*TraceEvent(SevDebug, "FakeDomainIdsRequest")
	    .detail("Str", resp)
	    .detail("RootTag", rootTag)
	    .detail("ElementTag", elementTag);*/
	return resp;
}

std::string getFakeEncryptDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	return getFakeDomainIdsRequestContent(domIds, CIPHER_KEY_DETAILS_TAG, ENCRYPT_DOMAIN_ID_TAG, fault);
}

std::string getFakeBlobDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	return getFakeDomainIdsRequestContent(domIds, BLOB_METADATA_DETAILS_TAG, BLOB_METADATA_DOMAIN_ID_TAG, fault);
}

std::string getFakeBaseCipherIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	rapidjson::Document doc;
	doc.SetObject();

	if (fault != FaultType::MISSING_VERSION) {
		if (fault == FaultType::INVALID_VERSION) {
			addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION + 1);
		} else {
			addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION);
		}
	}

	if (fault != FaultType::MISSING_VALIDATION_TOKEN) {
		addFakeValidationTokens(doc);
	}

	constructDomainIds(domIds);
	rapidjson::Value keyIdDetails(rapidjson::kArrayType);
	for (auto domId : domIds) {
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domId);
		ASSERT(keyCtx.isValid());
		addBaseCipherIdDomIdToDoc(doc, keyIdDetails, keyCtx->id, domId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, doc.GetAllocator());
	doc.AddMember(memberKey, keyIdDetails, doc.GetAllocator());

	addRefreshKmsUrlsSectionToJsonDoc(doc, deterministicRandom()->coinflip());

	if (deterministicRandom()->coinflip()) {
		addDebugUidSectionToJsonDoc(doc, deterministicRandom()->randomUniqueID());
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	std::string resp(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FakeKeyIdsRequest").detail("Str", resp);
	return resp;
}

Optional<ErrorDetail> getErrorDetail(const std::string& buff) {
	rapidjson::Document doc;
	doc.Parse(buff.data());
	return RESTKmsConnectorUtils::getError(doc);
}

void validateEncryptLookup(const VaultResponse& response, const EncryptCipherDomainIdVec& domIds) {
	ASSERT(!response.failed);

	//TraceEvent(SevDebug, "VaultEncryptResponse").detail("Str", response.buff);

	rapidjson::Document doc;
	doc.Parse(response.buff.data());

	ASSERT(doc.HasMember(CIPHER_KEY_DETAILS_TAG) && doc[CIPHER_KEY_DETAILS_TAG].IsArray());

	std::unordered_set<EncryptCipherDomainId> domIdSet(domIds.begin(), domIds.end());
	int count = 0;
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		EncryptCipherDomainId domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		EncryptCipherBaseKeyId baseCipherId = cipherDetail[BASE_CIPHER_ID_TAG].GetUint64();
		const int cipherKeyLen = cipherDetail[BASE_CIPHER_TAG].GetStringLength();
		Standalone<StringRef> cipherKeyRef = makeString(cipherKeyLen);
		memcpy(mutateString(cipherKeyRef), cipherDetail[BASE_CIPHER_TAG].GetString(), cipherKeyLen);

		ASSERT(domIdSet.find(domainId) != domIdSet.end());

		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domainId);
		ASSERT_EQ(keyCtx->id, baseCipherId);
		ASSERT_EQ(keyCtx->key.compare(cipherKeyRef), 0);
		const int64_t refreshAfterSec = cipherDetail[REFRESH_AFTER_SEC].GetInt64();
		const int64_t expireAfterSec = cipherDetail[EXPIRE_AFTER_SEC].GetInt64();
		ASSERT(refreshAfterSec <= expireAfterSec || expireAfterSec == -1);
		count++;
	}
	ASSERT_EQ(count, domIds.size());
}

void validateBlobLookup(const VaultResponse& response, const EncryptCipherDomainIdVec& domIds) {
	ASSERT(!response.failed);

	//TraceEvent(SevDebug, "VaultBlobResponse").detail("Str", response.buff);

	rapidjson::Document doc;
	doc.Parse(response.buff.data());

	ASSERT(doc.HasMember(BLOB_METADATA_DETAILS_TAG) && doc[BLOB_METADATA_DETAILS_TAG].IsArray());

	std::unordered_set<EncryptCipherDomainId> domIdSet(domIds.begin(), domIds.end());
	int count = 0;
	for (const auto& blobDetail : doc[BLOB_METADATA_DETAILS_TAG].GetArray()) {
		EncryptCipherDomainId domainId = blobDetail[BLOB_METADATA_DOMAIN_ID_TAG].GetInt64();
		Standalone<BlobMetadataDetailsRef> details = SimKmsVault::getBlobMetadata(domainId, bgUrl);

		std::unordered_map<BlobMetadataLocationId, Standalone<StringRef>> locMap;
		for (const auto& loc : details.locations) {
			locMap[loc.locationId] = loc.path;
		}
		for (const auto& location : blobDetail[BLOB_METADATA_LOCATIONS_TAG].GetArray()) {
			BlobMetadataLocationId locationId = location[BLOB_METADATA_LOCATION_ID_TAG].GetInt64();
			Standalone<StringRef> path = makeString(location[BLOB_METADATA_LOCATION_PATH_TAG].GetStringLength());
			memcpy(mutateString(path),
			       location[BLOB_METADATA_LOCATION_PATH_TAG].GetString(),
			       location[BLOB_METADATA_LOCATION_PATH_TAG].GetStringLength());
			auto it = locMap.find(locationId);
			ASSERT(it != locMap.end());
			ASSERT_EQ(it->second.compare(path), 0);
		}
		const int64_t refreshAfterSec = blobDetail[REFRESH_AFTER_SEC].GetInt64();
		const int64_t expireAfterSec = blobDetail[EXPIRE_AFTER_SEC].GetInt64();
		ASSERT(refreshAfterSec <= expireAfterSec || expireAfterSec == -1);
		count++;
	}
	ASSERT_EQ(count, domIds.size());
}

} // namespace

															#line 967 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase663()
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase663Actor>
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase663ActorState {
															#line 974 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase663ActorState(UnitTestParameters const& params) 
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params),
															#line 664 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   request(makeReference<HTTP::IncomingRequest>()),
															#line 665 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(makeReference<HTTP::OutgoingResponse>())
															#line 985 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase663", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase663ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase663", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 667 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->verb = HTTP::HTTP_VERB_POST;
															#line 668 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->resource = "/whatever";
															#line 669 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers = RESTKmsConnectorUtils::getHTTPHeaders();
															#line 1004 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			try {
															#line 671 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				StrictFuture<Void> __when_expr_0 = simKmsVaultRequestHandler(request, response);
															#line 671 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (static_cast<FlowTestCase663Actor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1010 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FlowTestCase663Actor*>(this)->actor_wait_state = 1;
															#line 671 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase663Actor, 0, Void >*>(static_cast<FlowTestCase663Actor*>(this)));
															#line 1015 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase663ActorState();
		static_cast<FlowTestCase663Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 676 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<FlowTestCase663Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase663ActorState(); static_cast<FlowTestCase663Actor*>(this)->destroy(); return 0; }
															#line 1044 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		new (&static_cast<FlowTestCase663Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase663ActorState();
		static_cast<FlowTestCase663Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 674 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT_EQ(e.code(), error_code_http_bad_response);
															#line 1057 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 672 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1072 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 672 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1081 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase663Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase663Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase663Actor*>(this)->ActorCallback< FlowTestCase663Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase663Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase663", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase663Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase663", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase663Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase663", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase663Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase663", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase663Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase663", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase663Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase663", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 664 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 665 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1180 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase663()
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase663Actor final : public Actor<Void>, public ActorCallback< FlowTestCase663Actor, 0, Void >, public FastAllocated<FlowTestCase663Actor>, public FlowTestCase663ActorState<FlowTestCase663Actor> {
															#line 1185 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase663Actor>::operator new;
	using FastAllocated<FlowTestCase663Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17575798683687473920UL, 13767457608483909632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase663Actor, 0, Void >;
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase663Actor(UnitTestParameters const& params) 
															#line 1202 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase663ActorState<FlowTestCase663Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase663", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14033886408453573632UL, 3992666851526065152UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase663Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase663");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase663", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase663Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase663( UnitTestParameters const& params ) {
															#line 663 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase663Actor(params));
															#line 1235 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase663, "/restSimKmsVault/invalidResource")

#line 678 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1241 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase679()
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase679Actor>
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase679ActorState {
															#line 1248 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase679ActorState(UnitTestParameters const& params) 
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params),
															#line 680 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   request(makeReference<HTTP::IncomingRequest>()),
															#line 681 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(makeReference<HTTP::OutgoingResponse>())
															#line 1259 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase679", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase679ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase679", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 683 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->verb = HTTP::HTTP_VERB_POST;
															#line 684 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->resource = "/whatever";
															#line 685 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers = RESTKmsConnectorUtils::getHTTPHeaders();
															#line 686 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers["Foo"] = "Bar";
															#line 1280 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			try {
															#line 688 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				StrictFuture<Void> __when_expr_0 = simKmsVaultRequestHandler(request, response);
															#line 688 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (static_cast<FlowTestCase679Actor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1286 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FlowTestCase679Actor*>(this)->actor_wait_state = 1;
															#line 688 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase679Actor, 0, Void >*>(static_cast<FlowTestCase679Actor*>(this)));
															#line 1291 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase679ActorState();
		static_cast<FlowTestCase679Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 693 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<FlowTestCase679Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase679ActorState(); static_cast<FlowTestCase679Actor*>(this)->destroy(); return 0; }
															#line 1320 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		new (&static_cast<FlowTestCase679Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase679ActorState();
		static_cast<FlowTestCase679Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 691 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT_EQ(e.code(), error_code_rest_malformed_response);
															#line 1333 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 689 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1348 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 689 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1357 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase679Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase679Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase679Actor*>(this)->ActorCallback< FlowTestCase679Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase679Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase679", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase679Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase679", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase679Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase679", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase679Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase679", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase679Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase679", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase679Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase679", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 680 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 681 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1456 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase679()
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase679Actor final : public Actor<Void>, public ActorCallback< FlowTestCase679Actor, 0, Void >, public FastAllocated<FlowTestCase679Actor>, public FlowTestCase679ActorState<FlowTestCase679Actor> {
															#line 1461 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase679Actor>::operator new;
	using FastAllocated<FlowTestCase679Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17600733871583919104UL, 11282820223930337536UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase679Actor, 0, Void >;
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase679Actor(UnitTestParameters const& params) 
															#line 1478 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase679ActorState<FlowTestCase679Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase679", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8555188639991951360UL, 13387254527781047552UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase679Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase679");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase679", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase679Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase679( UnitTestParameters const& params ) {
															#line 679 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase679Actor(params));
															#line 1511 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase679, "/restSimKmsVault/invalidHeader")

#line 695 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1517 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase696()
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase696Actor>
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase696ActorState {
															#line 1524 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase696ActorState(UnitTestParameters const& params) 
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1531 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase696", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase696ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase696", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 697 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 698 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 699 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 700 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 701 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 702 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 703 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 705 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase696Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase696ActorState(); static_cast<FlowTestCase696Actor*>(this)->destroy(); return 0; }
															#line 1560 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase696Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase696ActorState();
			static_cast<FlowTestCase696Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase696ActorState();
		static_cast<FlowTestCase696Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1584 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase696()
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase696Actor final : public Actor<Void>, public FastAllocated<FlowTestCase696Actor>, public FlowTestCase696ActorState<FlowTestCase696Actor> {
															#line 1589 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase696Actor>::operator new;
	using FastAllocated<FlowTestCase696Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11286274031933408256UL, 3613334082459226112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase696Actor(UnitTestParameters const& params) 
															#line 1605 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase696ActorState<FlowTestCase696Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase696", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13704272775011032064UL, 11541543819778809856UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase696Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase696");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase696", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase696( UnitTestParameters const& params ) {
															#line 696 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase696Actor(params));
															#line 1637 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase696, "/restSimKmsVault/GetByDomainIds/missingVersion")

#line 707 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1643 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase708()
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase708Actor>
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase708ActorState {
															#line 1650 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase708ActorState(UnitTestParameters const& params) 
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1657 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase708", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase708ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase708", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 709 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 710 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 711 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 712 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 713 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 714 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 715 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 717 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase708Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase708ActorState(); static_cast<FlowTestCase708Actor*>(this)->destroy(); return 0; }
															#line 1686 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase708Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase708ActorState();
			static_cast<FlowTestCase708Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase708ActorState();
		static_cast<FlowTestCase708Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1710 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase708()
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase708Actor final : public Actor<Void>, public FastAllocated<FlowTestCase708Actor>, public FlowTestCase708ActorState<FlowTestCase708Actor> {
															#line 1715 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase708Actor>::operator new;
	using FastAllocated<FlowTestCase708Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17483397936932786688UL, 11775313864980461568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase708Actor(UnitTestParameters const& params) 
															#line 1731 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase708ActorState<FlowTestCase708Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase708", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10126996590177419520UL, 12103493544181703168UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase708Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase708");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase708", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase708( UnitTestParameters const& params ) {
															#line 708 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase708Actor(params));
															#line 1763 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase708, "/restSimKmsVault/GetByDomainIds/invalidVersion")

#line 719 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1769 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase720()
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase720Actor>
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase720ActorState {
															#line 1776 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase720ActorState(UnitTestParameters const& params) 
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1783 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase720", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase720ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase720", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 721 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 722 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 724 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 725 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 726 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 727 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 728 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 730 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase720Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase720ActorState(); static_cast<FlowTestCase720Actor*>(this)->destroy(); return 0; }
															#line 1812 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase720Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase720ActorState();
			static_cast<FlowTestCase720Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase720ActorState();
		static_cast<FlowTestCase720Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1836 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase720()
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase720Actor final : public Actor<Void>, public FastAllocated<FlowTestCase720Actor>, public FlowTestCase720ActorState<FlowTestCase720Actor> {
															#line 1841 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase720Actor>::operator new;
	using FastAllocated<FlowTestCase720Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13761276624946795264UL, 15044639428854138368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase720Actor(UnitTestParameters const& params) 
															#line 1857 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase720ActorState<FlowTestCase720Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase720", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4532594586167917056UL, 11044513145831470592UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase720Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase720");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase720", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase720( UnitTestParameters const& params ) {
															#line 720 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase720Actor(params));
															#line 1889 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase720, "/restSimKmsVault/GetByDomainIds/missingValidationTokens")

#line 732 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1895 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase733()
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase733Actor>
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase733ActorState {
															#line 1902 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase733ActorState(UnitTestParameters const& params) 
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1909 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase733", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase733ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase733", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 734 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 735 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds);
															#line 737 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 738 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateEncryptLookup(response, domIds);
															#line 739 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase733Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase733ActorState(); static_cast<FlowTestCase733Actor*>(this)->destroy(); return 0; }
															#line 1932 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase733Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase733ActorState();
			static_cast<FlowTestCase733Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase733ActorState();
		static_cast<FlowTestCase733Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1956 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase733()
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase733Actor final : public Actor<Void>, public FastAllocated<FlowTestCase733Actor>, public FlowTestCase733ActorState<FlowTestCase733Actor> {
															#line 1961 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase733Actor>::operator new;
	using FastAllocated<FlowTestCase733Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13850542843639965184UL, 6688334853315751936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase733Actor(UnitTestParameters const& params) 
															#line 1977 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase733ActorState<FlowTestCase733Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase733", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4080047510121817344UL, 11574546535354078720UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase733Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase733");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase733", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase733( UnitTestParameters const& params ) {
															#line 733 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase733Actor(params));
															#line 2009 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase733, "/restSimKmsVault/GetByDomainIds")

#line 741 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2015 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase742()
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase742Actor>
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase742ActorState {
															#line 2022 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase742ActorState(UnitTestParameters const& params) 
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2029 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase742", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase742ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase742", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 743 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 744 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 746 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 747 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 748 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 749 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 750 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 752 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase742Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase742ActorState(); static_cast<FlowTestCase742Actor*>(this)->destroy(); return 0; }
															#line 2058 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase742Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase742ActorState();
			static_cast<FlowTestCase742Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase742ActorState();
		static_cast<FlowTestCase742Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2082 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase742()
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase742Actor final : public Actor<Void>, public FastAllocated<FlowTestCase742Actor>, public FlowTestCase742ActorState<FlowTestCase742Actor> {
															#line 2087 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase742Actor>::operator new;
	using FastAllocated<FlowTestCase742Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13732450143860902144UL, 13510191130601230336UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase742Actor(UnitTestParameters const& params) 
															#line 2103 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase742ActorState<FlowTestCase742Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase742", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14468103045155587072UL, 332753245024101632UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase742Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase742");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase742", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase742( UnitTestParameters const& params ) {
															#line 742 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase742Actor(params));
															#line 2135 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase742, "/restSimKmsVault/GetByKeyIds/missingVersion")

#line 754 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2141 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase755()
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase755Actor>
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase755ActorState {
															#line 2148 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase755ActorState(UnitTestParameters const& params) 
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2155 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase755", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase755ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase755", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 756 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 757 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 759 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 760 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 761 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 762 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 763 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 765 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase755Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase755ActorState(); static_cast<FlowTestCase755Actor*>(this)->destroy(); return 0; }
															#line 2184 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase755Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase755ActorState();
			static_cast<FlowTestCase755Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase755ActorState();
		static_cast<FlowTestCase755Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2208 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase755()
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase755Actor final : public Actor<Void>, public FastAllocated<FlowTestCase755Actor>, public FlowTestCase755ActorState<FlowTestCase755Actor> {
															#line 2213 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase755Actor>::operator new;
	using FastAllocated<FlowTestCase755Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13643780447980653568UL, 16230568933400054784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase755Actor(UnitTestParameters const& params) 
															#line 2229 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase755ActorState<FlowTestCase755Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase755", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13466744430676719104UL, 15113891646312456448UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase755Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase755");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase755", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase755( UnitTestParameters const& params ) {
															#line 755 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase755Actor(params));
															#line 2261 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase755, "/restSimKmsVault/GetByKeyIds/invalidVersion")

#line 767 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2267 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase768()
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase768Actor>
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase768ActorState {
															#line 2274 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase768ActorState(UnitTestParameters const& params) 
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2281 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase768", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase768ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase768", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 769 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 770 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 772 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 773 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 774 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 775 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 776 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 778 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase768Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase768ActorState(); static_cast<FlowTestCase768Actor*>(this)->destroy(); return 0; }
															#line 2310 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase768Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase768ActorState();
			static_cast<FlowTestCase768Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase768ActorState();
		static_cast<FlowTestCase768Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2334 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase768()
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase768Actor final : public Actor<Void>, public FastAllocated<FlowTestCase768Actor>, public FlowTestCase768ActorState<FlowTestCase768Actor> {
															#line 2339 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase768Actor>::operator new;
	using FastAllocated<FlowTestCase768Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7941473791827802368UL, 8909618321482111488UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase768Actor(UnitTestParameters const& params) 
															#line 2355 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase768ActorState<FlowTestCase768Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase768", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8182258776000434944UL, 9833904026492196608UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase768Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase768");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase768", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase768( UnitTestParameters const& params ) {
															#line 768 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase768Actor(params));
															#line 2387 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase768, "/restSimKmsVault/GetByKeyIds/missingValidationTokens")

#line 780 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2393 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase781()
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase781Actor>
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase781ActorState {
															#line 2400 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase781ActorState(UnitTestParameters const& params) 
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2407 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase781", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase781ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase781", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 782 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 783 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 785 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 786 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 787 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 788 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 789 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 791 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase781Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase781ActorState(); static_cast<FlowTestCase781Actor*>(this)->destroy(); return 0; }
															#line 2436 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase781Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase781ActorState();
			static_cast<FlowTestCase781Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase781ActorState();
		static_cast<FlowTestCase781Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2460 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase781()
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase781Actor final : public Actor<Void>, public FastAllocated<FlowTestCase781Actor>, public FlowTestCase781ActorState<FlowTestCase781Actor> {
															#line 2465 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase781Actor>::operator new;
	using FastAllocated<FlowTestCase781Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2117547270585542656UL, 7883065912352130048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase781Actor(UnitTestParameters const& params) 
															#line 2481 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase781ActorState<FlowTestCase781Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase781", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8834200973892582400UL, 504856189617749504UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase781Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase781");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase781", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase781( UnitTestParameters const& params ) {
															#line 781 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase781Actor(params));
															#line 2513 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase781, "/restSimKmsVault/GetByKeyIds/missingValidationTokensBlob")

#line 793 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2519 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase794()
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase794Actor>
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase794ActorState {
															#line 2526 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase794ActorState(UnitTestParameters const& params) 
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2533 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase794", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase794ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase794", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 795 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 796 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds);
															#line 798 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 799 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateEncryptLookup(response, domIds);
															#line 800 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase794Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase794ActorState(); static_cast<FlowTestCase794Actor*>(this)->destroy(); return 0; }
															#line 2556 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase794Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase794ActorState();
			static_cast<FlowTestCase794Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase794ActorState();
		static_cast<FlowTestCase794Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2580 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase794()
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase794Actor final : public Actor<Void>, public FastAllocated<FlowTestCase794Actor>, public FlowTestCase794ActorState<FlowTestCase794Actor> {
															#line 2585 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase794Actor>::operator new;
	using FastAllocated<FlowTestCase794Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3738967284284965376UL, 12496794308474524928UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase794Actor(UnitTestParameters const& params) 
															#line 2601 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase794ActorState<FlowTestCase794Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase794", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5560007075699582976UL, 11557308979492832000UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase794Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase794");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase794", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase794( UnitTestParameters const& params ) {
															#line 794 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase794Actor(params));
															#line 2633 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase794, "/restSimKmsVault/GetByKeyIds")

#line 802 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2639 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase803()
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase803Actor>
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase803ActorState {
															#line 2646 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase803ActorState(UnitTestParameters const& params) 
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2653 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase803", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase803ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase803", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 804 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 805 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 807 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 808 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 809 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 810 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 811 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 813 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase803Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase803ActorState(); static_cast<FlowTestCase803Actor*>(this)->destroy(); return 0; }
															#line 2682 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase803Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase803ActorState();
			static_cast<FlowTestCase803Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase803ActorState();
		static_cast<FlowTestCase803Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2706 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase803()
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase803Actor final : public Actor<Void>, public FastAllocated<FlowTestCase803Actor>, public FlowTestCase803ActorState<FlowTestCase803Actor> {
															#line 2711 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase803Actor>::operator new;
	using FastAllocated<FlowTestCase803Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5638247264289550080UL, 16184809166648077312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase803Actor(UnitTestParameters const& params) 
															#line 2727 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase803ActorState<FlowTestCase803Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase803", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12980222041527406848UL, 1205056615461489408UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase803Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase803");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase803", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase803( UnitTestParameters const& params ) {
															#line 803 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase803Actor(params));
															#line 2759 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase803, "/restSimKmsVault/GetBlobMetadata/missingVersion")

#line 815 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2765 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase816()
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase816Actor>
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase816ActorState {
															#line 2772 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase816ActorState(UnitTestParameters const& params) 
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2779 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase816", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase816ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase816", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 817 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 818 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 820 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 821 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 822 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 823 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 824 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 826 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase816Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase816ActorState(); static_cast<FlowTestCase816Actor*>(this)->destroy(); return 0; }
															#line 2808 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase816Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase816ActorState();
			static_cast<FlowTestCase816Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase816ActorState();
		static_cast<FlowTestCase816Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2832 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase816()
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase816Actor final : public Actor<Void>, public FastAllocated<FlowTestCase816Actor>, public FlowTestCase816ActorState<FlowTestCase816Actor> {
															#line 2837 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase816Actor>::operator new;
	using FastAllocated<FlowTestCase816Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12171895019461292288UL, 3336761999257191680UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase816Actor(UnitTestParameters const& params) 
															#line 2853 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase816ActorState<FlowTestCase816Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase816", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9142938843537595136UL, 9074228770382408960UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase816Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase816");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase816", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase816( UnitTestParameters const& params ) {
															#line 816 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase816Actor(params));
															#line 2885 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase816, "/restSimKmsVault/GetBlobMetadata/invalidVersion")

#line 828 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2891 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase829()
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase829Actor>
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase829ActorState {
															#line 2898 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase829ActorState(UnitTestParameters const& params) 
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2905 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase829", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase829ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase829", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 830 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 831 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds);
															#line 833 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 834 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateBlobLookup(response, domIds);
															#line 835 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase829Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase829ActorState(); static_cast<FlowTestCase829Actor*>(this)->destroy(); return 0; }
															#line 2928 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase829Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase829ActorState();
			static_cast<FlowTestCase829Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase829ActorState();
		static_cast<FlowTestCase829Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2952 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase829()
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase829Actor final : public Actor<Void>, public FastAllocated<FlowTestCase829Actor>, public FlowTestCase829ActorState<FlowTestCase829Actor> {
															#line 2957 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase829Actor>::operator new;
	using FastAllocated<FlowTestCase829Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10784014243802296064UL, 2905419140018493696UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase829Actor(UnitTestParameters const& params) 
															#line 2973 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase829ActorState<FlowTestCase829Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase829", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6821780616658211840UL, 3970267137474583040UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase829Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase829");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase829", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase829( UnitTestParameters const& params ) {
															#line 829 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase829Actor(params));
															#line 3005 "/codebuild/output/src4166514139/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase829, "/restSimKmsVault/GetBlobMetadata/foo")

