"""Base generator class for language bindings."""

from abc import ABC, abstractmethod
from pathlib import Path
from typing import List, Dict, Any
import jinja2
from ..models import Option, Scope, ParamType
from .. import utils


class BaseGenerator(ABC):
    """Abstract base class for language-specific generators."""

    def __init__(self, options: List[Option]):
        """Initialize the generator with parsed options.

        Args:
            options: List of Option objects from the parsed XML
        """
        self.options = options
        self.env = self._create_jinja_env()

    def _create_jinja_env(self) -> jinja2.Environment:
        """Create and configure a Jinja2 environment with custom filters."""
        # Get the templates directory
        templates_dir = Path(__file__).parent.parent / "templates"

        env = jinja2.Environment(
            loader=jinja2.FileSystemLoader(str(templates_dir)),
            trim_blocks=True,
            lstrip_blocks=False,  # Keep indentation in templates
            keep_trailing_newline=True,
        )

        # Add custom filters
        env.filters["snake_to_camel"] = utils.snake_to_camel
        env.filters["snake_to_lower_camel"] = utils.snake_to_lower_camel
        env.filters["snake_to_upper_snake"] = utils.snake_to_upper_snake
        env.filters["format_c_comment"] = utils.format_c_comment
        env.filters["format_javadoc_comment"] = utils.format_javadoc_comment
        env.filters["format_python_comment"] = utils.format_python_comment
        env.filters["escape_string"] = lambda text, lang: utils.escape_string(
            text, lang
        )
        env.filters["replace_backticks"] = utils.replace_backticks_for_java

        return env

    def group_by_scope(self) -> Dict[Scope, List[Option]]:
        """Group options by their scope.

        Returns:
            Dictionary mapping Scope to list of Options
        """
        groups = {}
        for option in self.options:
            if option.scope not in groups:
                groups[option.scope] = []
            groups[option.scope].append(option)
        return groups

    def get_options_for_scope(self, scope: Scope) -> List[Option]:
        """Get all options for a specific scope.

        Args:
            scope: The scope to filter by

        Returns:
            List of options for that scope
        """
        return [opt for opt in self.options if opt.scope == scope]

    def is_deprecated(self, option: Option) -> bool:
        """Check if an option is deprecated.

        Args:
            option: The option to check

        Returns:
            True if the option's comment starts with "Deprecated"
        """
        return option.comment.startswith("Deprecated")

    def get_parameter_comment(self, option: Option) -> str:
        """Get a formatted parameter comment for an option.

        Args:
            option: The option to get comment for

        Returns:
            Formatted string describing the parameter
        """
        if option.param_type == ParamType.NONE:
            return "Option takes no parameter"
        return f"({option.param_type.value}) {option.param_desc or ''}"

    @abstractmethod
    def write_files(self, output_path: str) -> None:
        """Generate and write output files.

        Args:
            output_path: Path where output file(s) should be written
        """
        pass

    def render_template(self, template_name: str, context: Dict[str, Any]) -> str:
        """Render a Jinja2 template with the given context.

        Args:
            template_name: Name of the template file
            context: Dictionary of variables to pass to the template

        Returns:
            Rendered template as string
        """
        template = self.env.get_template(template_name)
        return template.render(**context)

    def write_file(self, path: str, content: str) -> None:
        """Write content to a file.

        Args:
            path: File path to write to
            content: Content to write
        """
        output_path = Path(path)
        output_path.parent.mkdir(parents=True, exist_ok=True)

        with open(output_path, "w", encoding="utf-8") as f:
            f.write(content)

    @staticmethod
    def get_license_header() -> str:
        """Get the Apache 2.0 license header for generated files."""
        return """/*
 * This file is automatically generated by vexillographer.
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */"""

    @staticmethod
    def get_python_license_header() -> str:
        """Get the Apache 2.0 license header for Python files."""
        return """#
# This file is automatically generated by vexillographer.
#
# This source file is part of the FoundationDB open source project
#
# Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#"""
