#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
/*
 * MockS3Server.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2025 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbserver/MockS3Server.h"

#include "fdbrpc/HTTP.h"
#include "fdbrpc/simulator.h"
#include "flow/Trace.h"
#include "flow/IRandom.h"
#include "flow/serialize.h"

#include <string>
#include <map>
#include <sstream>
#include <regex>
#include <utility>
#include <iostream>
#include <algorithm>

#include "flow/IAsyncFile.h" // For IAsyncFileSystem
#include "flow/Platform.h" // For platform::createDirectory

#include "rapidjson/document.h"
#include "rapidjson/writer.h"
#include "rapidjson/stringbuffer.h"
#include "rapidjson/prettywriter.h"

#include "flow/actorcompiler.h" // This must be the last #include.

/*
 * ACTOR STATE VARIABLE INITIALIZATION REQUIREMENT
 *
 * ACTORs with early returns (before any wait()) crash with canBeSet() assertion if no state
 * variable is declared before the return. The actor compiler generates a member initialization
 * list (": member(value)") in the state class constructor only when it sees state variables.
 * This initialization list ensures the Actor<T> base class and its internal Promise are fully
 * initialized before any code runs. Without it, early returns try to use an uninitialized Promise.
 *
 * FIX: Declare at least one state variable BEFORE any early return. Declaration alone is enough.
 *
 * CORRECT:
 *   ACTOR Future<Void> someActor(...) {
 *       state std::string data;                // Triggers member init list (requires default ctor)
 *       if (earlyExitCondition) return Void(); // Safe - Promise is initialized
 *       data = computeValue();                 // Can initialize later
 *   }
 *
 *   // Or if no default constructor: state MyType x(params); and initialize at declaration
 *
 * WRONG (canBeSet() crash):
 *   ACTOR Future<Void> someActor(...) {
 *       if (earlyExitCondition) return Void(); // CRASH - no member init list generated yet
 *       state std::string data;                // Too late - compiler didn't see it early enough
 *   }
 */

// MockS3 persistence file extensions
namespace {
constexpr const char* OBJECT_DATA_SUFFIX = ".data";
constexpr const char* OBJECT_META_SUFFIX = ".meta.json";
constexpr const char* MULTIPART_STATE_SUFFIX = ".state.json";
constexpr size_t OBJECT_META_SUFFIX_LEN = 10; // strlen(".meta.json")
constexpr size_t MULTIPART_STATE_SUFFIX_LEN = 11; // strlen(".state.json")
} // namespace

// Global storage for MockS3 (shared across all simulated processes)
struct MockS3GlobalStorage {
	struct ObjectData {
		std::string content;
		HTTP::Headers headers;
		std::map<std::string, std::string> tags;
		std::string etag;
		double lastModified;

		ObjectData() : lastModified(now()) {}
		ObjectData(const std::string& data) : content(data), lastModified(now()) { etag = generateETag(data); }

		static std::string generateETag(const std::string& content) {
			return "\"" + HTTP::computeMD5Sum(content) + "\"";
		}
	};

	struct MultipartUpload {
		std::string uploadId;
		std::string bucket;
		std::string object;
		std::map<int, std::pair<std::string, std::string>> parts; // partNum -> {etag, content}
		HTTP::Headers metadata;
		double initiated;

		MultipartUpload() = default;
		MultipartUpload(const std::string& b, const std::string& o) : bucket(b), object(o), initiated(now()) {
			uploadId = deterministicRandom()->randomUniqueID().toString();
		}
	};

	std::map<std::string, std::map<std::string, ObjectData>> buckets;
	std::map<std::string, MultipartUpload> multipartUploads;

	// Persistence configuration
	std::string persistenceDir;
	bool persistenceEnabled = false;
	bool persistenceLoaded = false;

	// Note: In FDB simulation, function-local statics are SHARED across all simulated processes
	// because they all run on the same OS thread. This is exactly what we want for MockS3 storage.
	MockS3GlobalStorage() { TraceEvent("MockS3GlobalStorageCreated").detail("Address", format("%p", this)); }

	// Clear all stored data - called at the start of each simulation test to prevent
	// data accumulation across multiple tests
	void clearStorage() {
		buckets.clear();
		multipartUploads.clear();
		TraceEvent("MockS3GlobalStorageCleared").detail("Address", format("%p", this));
	}

	// Enable persistence to specified directory
	// Note: When using simulation filesystem, directories are created automatically by file open()
	// and each simulated machine has its own isolated directory structure
	void enablePersistence(const std::string& dir) {
		persistenceDir = dir;
		persistenceEnabled = true;
		persistenceLoaded = false;

		TraceEvent("MockS3PersistenceEnabled")
		    .detail("Directory", dir)
		    .detail("Address", format("%p", this))
		    .detail("UsingSimulationFS", g_network->isSimulated());
	}

	// Get paths for persistence files
	// Note: Object names with slashes map directly to filesystem directory structure
	std::string getObjectDataPath(const std::string& bucket, const std::string& object) const {
		return persistenceDir + "/objects/" + bucket + "/" + object + OBJECT_DATA_SUFFIX;
	}

	std::string getObjectMetaPath(const std::string& bucket, const std::string& object) const {
		return persistenceDir + "/objects/" + bucket + "/" + object + OBJECT_META_SUFFIX;
	}

	std::string getMultipartStatePath(const std::string& uploadId) const {
		return persistenceDir + "/multipart/" + uploadId + MULTIPART_STATE_SUFFIX;
	}

	std::string getMultipartPartPath(const std::string& uploadId, int partNum) const {
		return persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
	}

	std::string getMultipartPartMetaPath(const std::string& uploadId, int partNum) const {
		return persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum) + OBJECT_META_SUFFIX;
	}
};

// Accessor function - uses function-local static for lazy initialization
// In simulation, this static is shared across all simulated processes (same OS thread)
static MockS3GlobalStorage& getGlobalStorage() {
	static MockS3GlobalStorage storage;
	return storage;
}

// Helper: Create all parent directories for a file path
// Uses platform::createDirectory which handles recursive creation and EEXIST errors
static void createParentDirectories(const std::string& filePath) {
	size_t lastSlash = filePath.find_last_of('/');
	if (lastSlash != std::string::npos && lastSlash > 0) {
		std::string parentDir = filePath.substr(0, lastSlash);
		platform::createDirectory(parentDir); // Handles recursive creation and EEXIST
	}
}

// ACTOR: Atomic file write using simulation filesystem without chaos injection
// Chaos-free because AsyncFileChaos only affects files with "storage-" in the name
// (see AsyncFileChaos.h:40). OPEN_NO_AIO controls AsyncFileNonDurable behavior.
															#line 193 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via atomicWriteFile()
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class AtomicWriteFileActor>
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class AtomicWriteFileActorState {
															#line 200 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	AtomicWriteFileActorState(std::string const& path,std::string const& content) 
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path),
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   content(content)
															#line 209 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("atomicWriteFile", reinterpret_cast<unsigned long>(this));

	}
	~AtomicWriteFileActorState() 
	{
		fdb_probe_actor_destroy("atomicWriteFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 194 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				createParentDirectories(path);
															#line 198 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_ATOMIC_WRITE_AND_CREATE | IAsyncFile::OPEN_CREATE | IAsyncFile::OPEN_READWRITE | IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_NO_AIO, 0644);
															#line 198 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 229 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 1;
															#line 198 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 234 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AtomicWriteFileActorState();
		static_cast<AtomicWriteFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 212 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<AtomicWriteFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AtomicWriteFileActorState(); static_cast<AtomicWriteFileActor*>(this)->destroy(); return 0; }
															#line 263 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<AtomicWriteFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AtomicWriteFileActorState();
		static_cast<AtomicWriteFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 210 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceWriteException").error(e).detail("Path", path);
															#line 276 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 204 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = file->write(content.data(), content.size(), 0);
															#line 204 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 293 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 2;
															#line 204 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 1, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 298 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 198 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = __file;
															#line 307 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 205 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = file->sync();
															#line 205 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 388 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 3;
															#line 205 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 2, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 393 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 205 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = file->sync();
															#line 205 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 404 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 3;
															#line 205 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 2, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 409 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 206 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 208 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceWriteSuccess").detail("Path", path).detail("Size", content.size());
															#line 495 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 206 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 208 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceWriteSuccess").detail("Path", path).detail("Size", content.size());
															#line 506 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 198 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<IAsyncFile> file;
															#line 605 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via atomicWriteFile()
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class AtomicWriteFileActor final : public Actor<Void>, public ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >, public ActorCallback< AtomicWriteFileActor, 1, Void >, public ActorCallback< AtomicWriteFileActor, 2, Void >, public FastAllocated<AtomicWriteFileActor>, public AtomicWriteFileActorState<AtomicWriteFileActor> {
															#line 610 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<AtomicWriteFileActor>::operator new;
	using FastAllocated<AtomicWriteFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8728863702490288384UL, 9556352406639863808UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< AtomicWriteFileActor, 1, Void >;
friend struct ActorCallback< AtomicWriteFileActor, 2, Void >;
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	AtomicWriteFileActor(std::string const& path,std::string const& content) 
															#line 629 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   AtomicWriteFileActorState<AtomicWriteFileActor>(path, content),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1551474338982279168UL, 5682969234323159552UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("atomicWriteFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> atomicWriteFile( std::string const& path, std::string const& content ) {
															#line 191 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new AtomicWriteFileActor(path, content));
															#line 664 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 214 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Read file content using simulation filesystem without chaos
// Chaos-free because AsyncFileChaos only affects files with "storage-" in the name
															#line 671 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via readFileContent()
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class ReadFileContentActor>
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class ReadFileContentActorState {
															#line 678 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	ReadFileContentActorState(std::string const& path) 
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path),
															#line 218 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   exists(fileExists(path))
															#line 687 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("readFileContent", reinterpret_cast<unsigned long>(this));

	}
	~ReadFileContentActorState() 
	{
		fdb_probe_actor_destroy("readFileContent", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 221 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!exists)
															#line 703 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 222 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(std::string()); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 707 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::string());
					this->~ReadFileContentActorState();
					static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 225 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_NO_AIO, 0644);
															#line 225 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 717 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ReadFileContentActor*>(this)->actor_wait_state = 1;
															#line 225 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*>(static_cast<ReadFileContentActor*>(this)));
															#line 722 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 243 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadException").error(e).detail("Path", path);
															#line 244 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(std::string()); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 754 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::string());
			this->~ReadFileContentActorState();
			static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 227 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = file->size();
															#line 227 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 774 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadFileContentActor*>(this)->actor_wait_state = 2;
															#line 227 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 1, int64_t >*>(static_cast<ReadFileContentActor*>(this)));
															#line 779 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 225 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = __file;
															#line 788 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 229 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content = std::string();
															#line 230 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content.resize(fileSize);
															#line 231 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<int> __when_expr_2 = file->read((uint8_t*)content.data(), fileSize, 0);
															#line 231 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 873 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadFileContentActor*>(this)->actor_wait_state = 3;
															#line 231 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 2, int >*>(static_cast<ReadFileContentActor*>(this)));
															#line 878 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(int64_t const& __fileSize,int loopDepth) 
	{
															#line 227 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileSize = __fileSize;
															#line 887 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(int64_t && __fileSize,int loopDepth) 
	{
		fileSize = std::move(__fileSize);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int const& bytesRead,int loopDepth) 
	{
															#line 232 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 234 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bytesRead != fileSize)
															#line 968 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 235 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadShort") .detail("Path", path) .detail("Expected", fileSize) .detail("Actual", bytesRead);
															#line 972 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
															#line 241 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(content); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 976 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::move(content)); // state_var_RVO
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(int && bytesRead,int loopDepth) 
	{
															#line 232 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 234 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bytesRead != fileSize)
															#line 990 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 235 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadShort") .detail("Path", path) .detail("Expected", fileSize) .detail("Actual", bytesRead);
															#line 994 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
															#line 241 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(content); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::move(content)); // state_var_RVO
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(int const& bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont5(bytesRead, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(int && bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(bytesRead), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 218 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	bool exists;
															#line 225 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<IAsyncFile> file;
															#line 227 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int64_t fileSize;
															#line 229 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 1091 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via readFileContent()
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class ReadFileContentActor final : public Actor<std::string>, public ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >, public ActorCallback< ReadFileContentActor, 1, int64_t >, public ActorCallback< ReadFileContentActor, 2, int >, public FastAllocated<ReadFileContentActor>, public ReadFileContentActorState<ReadFileContentActor> {
															#line 1096 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<ReadFileContentActor>::operator new;
	using FastAllocated<ReadFileContentActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9238952750619444224UL, 5342367047575436800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< ReadFileContentActor, 1, int64_t >;
friend struct ActorCallback< ReadFileContentActor, 2, int >;
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	ReadFileContentActor(std::string const& path) 
															#line 1115 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<std::string>(),
		   ReadFileContentActorState<ReadFileContentActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7623102229524567296UL, 7126613595367428096UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readFileContent");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadFileContentActor, 1, int64_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadFileContentActor, 2, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<std::string> readFileContent( std::string const& path ) {
															#line 217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<std::string>(new ReadFileContentActor(path));
															#line 1150 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 247 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete file using simulation filesystem
// Wraps deleteFile with trace events and error handling for MockS3 persistence cleanup
															#line 1157 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedFile()
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedFileActor>
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedFileActorState {
															#line 1164 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedFileActorState(std::string const& path) 
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path)
															#line 1171 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedFile", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedFileActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 252 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = IAsyncFileSystem::filesystem()->deleteFile(path, true);
															#line 252 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1189 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedFileActor*>(this)->actor_wait_state = 1;
															#line 252 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedFileActor, 0, Void >*>(static_cast<DeletePersistedFileActor*>(this)));
															#line 1194 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedFileActorState();
		static_cast<DeletePersistedFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 257 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedFileActorState(); static_cast<DeletePersistedFileActor*>(this)->destroy(); return 0; }
															#line 1223 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedFileActorState();
		static_cast<DeletePersistedFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 255 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceDeleteException").error(e).detail("Path", path);
															#line 1236 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 253 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceDelete").detail("Path", path);
															#line 1251 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 253 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceDelete").detail("Path", path);
															#line 1260 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedFileActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedFileActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedFileActor*>(this)->ActorCallback< DeletePersistedFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 1355 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedFile()
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedFileActor final : public Actor<Void>, public ActorCallback< DeletePersistedFileActor, 0, Void >, public FastAllocated<DeletePersistedFileActor>, public DeletePersistedFileActorState<DeletePersistedFileActor> {
															#line 1360 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedFileActor>::operator new;
	using FastAllocated<DeletePersistedFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15100301275766143232UL, 7557021202268476928UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedFileActor, 0, Void >;
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedFileActor(std::string const& path) 
															#line 1377 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedFileActorState<DeletePersistedFileActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8878611606913210624UL, 16427746229923233536UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedFileActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedFile( std::string const& path ) {
															#line 250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedFileActor(path));
															#line 1410 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 259 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// JSON Serialization using rapidjson
static std::string serializeObjectMeta(const MockS3GlobalStorage::ObjectData& obj) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();

	doc.AddMember("etag", Value(obj.etag.c_str(), allocator), allocator);
	doc.AddMember("lastModified", obj.lastModified, allocator);

	Value tagsObj(kObjectType);
	for (const auto& tag : obj.tags) {
		tagsObj.AddMember(Value(tag.first.c_str(), allocator), Value(tag.second.c_str(), allocator), allocator);
	}
	doc.AddMember("tags", tagsObj, allocator);

	StringBuffer buffer;
	PrettyWriter<StringBuffer> writer(buffer);
	doc.Accept(writer);
	return buffer.GetString();
}

// JSON Deserialization using rapidjson
static void deserializeObjectMeta(const std::string& jsonStr, MockS3GlobalStorage::ObjectData& obj) {
	using namespace rapidjson;
	Document doc;
	doc.Parse(jsonStr.c_str());

	if (doc.HasMember("etag") && doc["etag"].IsString())
		obj.etag = doc["etag"].GetString();
	if (doc.HasMember("lastModified") && doc["lastModified"].IsNumber())
		obj.lastModified = doc["lastModified"].GetDouble();
	if (doc.HasMember("tags") && doc["tags"].IsObject()) {
		for (auto& m : doc["tags"].GetObject()) {
			if (m.value.IsString())
				obj.tags[m.name.GetString()] = m.value.GetString();
		}
	}
}

static std::string serializeMultipartState(const MockS3GlobalStorage::MultipartUpload& upload) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();

	doc.AddMember("uploadId", Value(upload.uploadId.c_str(), allocator), allocator);
	doc.AddMember("bucket", Value(upload.bucket.c_str(), allocator), allocator);
	doc.AddMember("object", Value(upload.object.c_str(), allocator), allocator);
	doc.AddMember("initiated", upload.initiated, allocator);

	Value partsArray(kArrayType);
	for (const auto& part : upload.parts) {
		Value partObj(kObjectType);
		partObj.AddMember("partNum", part.first, allocator);
		partObj.AddMember("etag", Value(part.second.first.c_str(), allocator), allocator);
		partsArray.PushBack(partObj, allocator);
	}
	doc.AddMember("parts", partsArray, allocator);

	StringBuffer buffer;
	PrettyWriter<StringBuffer> writer(buffer);
	doc.Accept(writer);
	return buffer.GetString();
}

static void deserializeMultipartState(const std::string& jsonStr, MockS3GlobalStorage::MultipartUpload& upload) {
	using namespace rapidjson;
	Document doc;
	doc.Parse(jsonStr.c_str());

	if (doc.HasMember("uploadId") && doc["uploadId"].IsString())
		upload.uploadId = doc["uploadId"].GetString();
	if (doc.HasMember("bucket") && doc["bucket"].IsString())
		upload.bucket = doc["bucket"].GetString();
	if (doc.HasMember("object") && doc["object"].IsString())
		upload.object = doc["object"].GetString();
	if (doc.HasMember("initiated") && doc["initiated"].IsNumber())
		upload.initiated = doc["initiated"].GetDouble();
	if (doc.HasMember("parts") && doc["parts"].IsArray()) {
		for (auto& partVal : doc["parts"].GetArray()) {
			if (partVal.HasMember("partNum") && partVal["partNum"].IsInt() && partVal.HasMember("etag") &&
			    partVal["etag"].IsString()) {
				int partNum = partVal["partNum"].GetInt();
				std::string etag = partVal["etag"].GetString();
				upload.parts[partNum] = { etag, "" }; // Content loaded separately from .part.N files
			}
		}
	}
}

// Forward declarations for state loading functions
															#line 1507 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadPersistedObjects( std::string const& persistenceDir );

#line 353 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1511 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadPersistedMultipartUploads( std::string const& persistenceDir );

#line 354 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1515 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadMockS3PersistedStateImpl(  );

#line 355 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
Future<Void> loadMockS3PersistedStateFuture();

static std::string serializePartMeta(const std::string& etag) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();
	doc.AddMember("etag", Value(etag.c_str(), allocator), allocator);

	StringBuffer buffer;
	Writer<StringBuffer> writer(buffer); // Use Writer instead of PrettyWriter for compact output
	doc.Accept(writer);
	return buffer.GetString();
}

// ACTOR: Persist object data and metadata
															#line 1535 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via persistObject()
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class PersistObjectActor>
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistObjectActorState {
															#line 1542 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistObjectActorState(std::string const& bucket,std::string const& object) 
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : bucket(bucket),
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("persistObject", reinterpret_cast<unsigned long>(this));

	}
	~PersistObjectActorState() 
	{
		fdb_probe_actor_destroy("persistObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 372 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 373 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 375 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto bucketIter = storage.buckets.find(bucket);
															#line 376 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (bucketIter == storage.buckets.end())
															#line 1572 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 377 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1576 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 380 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto objectIter = bucketIter->second.find(object);
															#line 381 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (objectIter == bucketIter->second.end())
															#line 1586 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 382 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1590 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 386 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			content = objectIter->second.content;
															#line 387 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaJson = serializeObjectMeta(objectIter->second);
															#line 1600 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 392 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				dataPath = storage.getObjectDataPath(bucket, object);
															#line 393 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				metaPath = storage.getObjectMetaPath(bucket, object);
															#line 397 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (fileExists(dataPath) || fileExists(dataPath + ".part"))
															#line 1608 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 398 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3ObjectAlreadyPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("DataPath", dataPath);
															#line 402 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1614 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~PersistObjectActorState();
					static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 405 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3PersistingObject") .detail("Bucket", bucket) .detail("Object", object) .detail("DataPath", dataPath) .detail("Size", content.size());
															#line 412 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = atomicWriteFile(dataPath, content);
															#line 412 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1626 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PersistObjectActor*>(this)->actor_wait_state = 1;
															#line 412 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 0, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1631 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistObjectActorState();
		static_cast<PersistObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 425 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1660 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistObjectActorState();
		static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 422 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3PersistObjectFailed").error(e).detail("Bucket", bucket).detail("Object", object);
															#line 1673 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 415 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(metaPath, metaJson);
															#line 415 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1690 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont4when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistObjectActor*>(this)->actor_wait_state = 2;
															#line 415 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 1, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1695 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 415 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(metaPath, metaJson);
															#line 415 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1706 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont4when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistObjectActor*>(this)->actor_wait_state = 2;
															#line 415 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 1, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1711 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state > 0) static_cast<PersistObjectActor*>(this)->actor_wait_state = 0;
		static_cast<PersistObjectActor*>(this)->ActorCallback< PersistObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 417 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("Size", content.size());
															#line 1795 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 417 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("Size", content.size());
															#line 1804 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state > 0) static_cast<PersistObjectActor*>(this)->actor_wait_state = 0;
		static_cast<PersistObjectActor*>(this)->ActorCallback< PersistObjectActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistObjectActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 386 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 387 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaJson;
															#line 392 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 393 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 1909 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via persistObject()
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistObjectActor final : public Actor<Void>, public ActorCallback< PersistObjectActor, 0, Void >, public ActorCallback< PersistObjectActor, 1, Void >, public FastAllocated<PersistObjectActor>, public PersistObjectActorState<PersistObjectActor> {
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<PersistObjectActor>::operator new;
	using FastAllocated<PersistObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6942402780644172800UL, 16823859749055905536UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistObjectActor, 0, Void >;
friend struct ActorCallback< PersistObjectActor, 1, Void >;
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistObjectActor(std::string const& bucket,std::string const& object) 
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   PersistObjectActorState<PersistObjectActor>(bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1247695763033469440UL, 1096140585632004864UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistObjectActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistObjectActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> persistObject( std::string const& bucket, std::string const& object ) {
															#line 371 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new PersistObjectActor(bucket, object));
															#line 1966 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 427 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Persist multipart upload state
															#line 1972 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via persistMultipartState()
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class PersistMultipartStateActor>
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistMultipartStateActorState {
															#line 1979 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistMultipartStateActorState(std::string const& uploadId) 
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : uploadId(uploadId),
															#line 430 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(),
															#line 431 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   parts()
															#line 1990 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("persistMultipartState", reinterpret_cast<unsigned long>(this));

	}
	~PersistMultipartStateActorState() 
	{
		fdb_probe_actor_destroy("persistMultipartState", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 433 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 434 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 436 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = storage.multipartUploads.find(uploadId);
															#line 437 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == storage.multipartUploads.end())
															#line 2011 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 438 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2015 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistMultipartStateActorState();
				static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 441 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			const auto& upload = uploadIter->second;
															#line 442 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			persistenceDir = storage.persistenceDir;
															#line 443 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			parts = upload.parts;
															#line 2027 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 448 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string statePath = persistenceDir + "/multipart/" + uploadId + ".state.json";
															#line 452 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (fileExists(statePath) || fileExists(statePath + ".part"))
															#line 2033 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 453 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3MultipartAlreadyPersisted").detail("UploadId", uploadId).detail("StatePath", statePath);
															#line 454 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2039 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~PersistMultipartStateActorState();
					static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 457 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string stateJson = serializeMultipartState(upload);
															#line 458 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = atomicWriteFile(statePath, stateJson);
															#line 458 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2051 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 1;
															#line 458 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 0, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2056 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistMultipartStateActorState();
		static_cast<PersistMultipartStateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 482 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2085 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistMultipartStateActorState();
		static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 479 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistMultipartFailed").error(e).detail("UploadId", uploadId);
															#line 2098 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 461 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter = parts.begin();
															#line 462 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 2115 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 461 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter = parts.begin();
															#line 462 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 2126 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 477 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartPersisted").detail("UploadId", uploadId).detail("PartsCount", parts.size());
															#line 2210 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 462 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partIter != parts.end()))
															#line 2226 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 463 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = partIter->first;
															#line 464 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		etag = partIter->second.first;
															#line 465 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partData = partIter->second.second;
															#line 467 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
															#line 468 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(partPath, partData);
															#line 468 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2242 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 2;
															#line 468 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 1, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2247 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 470 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 471 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = serializePartMeta(etag);
															#line 472 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = atomicWriteFile(partMetaPath, partMetaJson);
															#line 472 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2275 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 3;
															#line 472 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 2, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2280 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 470 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 471 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = serializePartMeta(etag);
															#line 472 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = atomicWriteFile(partMetaPath, partMetaJson);
															#line 472 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2295 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 3;
															#line 472 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 2, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2300 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 474 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter++;
															#line 2384 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 474 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter++;
															#line 2393 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 430 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 431 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<int, std::pair<std::string, std::string>> parts;
															#line 461 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<int, std::pair<std::string, std::string>>::iterator partIter;
															#line 463 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 464 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 465 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partData;
															#line 467 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 470 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 471 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaJson;
															#line 2506 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via persistMultipartState()
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistMultipartStateActor final : public Actor<Void>, public ActorCallback< PersistMultipartStateActor, 0, Void >, public ActorCallback< PersistMultipartStateActor, 1, Void >, public ActorCallback< PersistMultipartStateActor, 2, Void >, public FastAllocated<PersistMultipartStateActor>, public PersistMultipartStateActorState<PersistMultipartStateActor> {
															#line 2511 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<PersistMultipartStateActor>::operator new;
	using FastAllocated<PersistMultipartStateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1105101071747711232UL, 4272063935684789504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistMultipartStateActor, 0, Void >;
friend struct ActorCallback< PersistMultipartStateActor, 1, Void >;
friend struct ActorCallback< PersistMultipartStateActor, 2, Void >;
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistMultipartStateActor(std::string const& uploadId) 
															#line 2530 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   PersistMultipartStateActorState<PersistMultipartStateActor>(uploadId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9066204211401481728UL, 7216261254142226944UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistMultipartState");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> persistMultipartState( std::string const& uploadId ) {
															#line 429 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new PersistMultipartStateActor(uploadId));
															#line 2565 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 484 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete persisted object
															#line 2571 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedObject()
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedObjectActor>
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedObjectActorState {
															#line 2578 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedObjectActorState(std::string const& bucket,std::string const& object) 
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : bucket(bucket),
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 487 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   dataPath(),
															#line 488 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   metaPath()
															#line 2591 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedObject", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedObjectActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 490 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 491 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			dataPath = storage.getObjectDataPath(bucket, object);
															#line 494 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaPath = storage.getObjectMetaPath(bucket, object);
															#line 2612 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 497 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedFile(dataPath);
															#line 497 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2618 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 1;
															#line 497 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 0, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2623 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedObjectActorState();
		static_cast<DeletePersistedObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 508 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedObjectActorState(); static_cast<DeletePersistedObjectActor*>(this)->destroy(); return 0; }
															#line 2652 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedObjectActorState();
		static_cast<DeletePersistedObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 502 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3DeletePersistedObjectFailed") .error(e) .detail("Bucket", bucket) .detail("Object", object);
															#line 2665 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 498 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(metaPath);
															#line 498 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2682 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 2;
															#line 498 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 1, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2687 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 498 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(metaPath);
															#line 498 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2698 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 2;
															#line 498 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 1, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2703 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedObjectActor*>(this)->ActorCallback< DeletePersistedObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 500 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 2787 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 500 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 2796 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedObjectActor*>(this)->ActorCallback< DeletePersistedObjectActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 487 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 488 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 2897 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedObject()
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedObjectActor final : public Actor<Void>, public ActorCallback< DeletePersistedObjectActor, 0, Void >, public ActorCallback< DeletePersistedObjectActor, 1, Void >, public FastAllocated<DeletePersistedObjectActor>, public DeletePersistedObjectActorState<DeletePersistedObjectActor> {
															#line 2902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedObjectActor>::operator new;
	using FastAllocated<DeletePersistedObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(971388197619982848UL, 10743470974752692224UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedObjectActor, 0, Void >;
friend struct ActorCallback< DeletePersistedObjectActor, 1, Void >;
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedObjectActor(std::string const& bucket,std::string const& object) 
															#line 2920 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedObjectActorState<DeletePersistedObjectActor>(bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5402173951988437248UL, 4607022702540543488UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedObjectActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeletePersistedObjectActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedObject( std::string const& bucket, std::string const& object ) {
															#line 486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedObjectActor(bucket, object));
															#line 2954 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 510 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete persisted multipart upload
															#line 2960 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedMultipart()
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedMultipartActor>
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedMultipartActorState {
															#line 2967 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedMultipartActorState(std::string const& uploadId) 
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : uploadId(uploadId),
															#line 513 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   maxPart(),
															#line 514 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(),
															#line 515 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partNum(),
															#line 516 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partPath(),
															#line 517 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partMetaPath()
															#line 2984 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedMultipart", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedMultipartActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedMultipart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 519 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 520 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 3001 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 524 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				auto uploadIter = storage.multipartUploads.find(uploadId);
															#line 525 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				maxPart = 100;
															#line 526 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (uploadIter != storage.multipartUploads.end())
															#line 3009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 527 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					for( const auto& part : uploadIter->second.parts ) {
															#line 528 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						maxPart = std::max(maxPart, part.first);
															#line 3015 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					}
				}
															#line 533 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				persistenceDir = storage.persistenceDir;
															#line 536 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string statePath = persistenceDir + "/multipart/" + uploadId + ".state.json";
															#line 537 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedFile(statePath);
															#line 537 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3026 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 1;
															#line 537 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 0, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedMultipartActorState();
		static_cast<DeletePersistedMultipartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 561 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedMultipartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedMultipartActorState(); static_cast<DeletePersistedMultipartActor*>(this)->destroy(); return 0; }
															#line 3060 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedMultipartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedMultipartActorState();
		static_cast<DeletePersistedMultipartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 558 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3DeletePersistedMultipartFailed").error(e).detail("UploadId", uploadId);
															#line 3073 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 541 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 542 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 3090 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 541 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 542 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 3101 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 556 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartDeleted").detail("UploadId", uploadId);
															#line 3185 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 542 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partNum <= maxPart + 10))
															#line 3201 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 543 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 545 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(partPath);
															#line 545 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3213 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 2;
															#line 545 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 1, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3218 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 546 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deletePersistedFile(partMetaPath);
															#line 546 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3242 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 3;
															#line 546 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 2, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3247 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 546 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deletePersistedFile(partMetaPath);
															#line 546 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3258 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 3;
															#line 546 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 2, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3263 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 547 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (partNum % 2 == 0)
															#line 3349 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0.02);
															#line 552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3355 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 4;
															#line 552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 3, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3360 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 547 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (partNum % 2 == 0)
															#line 3376 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0.02);
															#line 552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3382 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 4;
															#line 552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 3, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3387 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont4(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 513 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int maxPart;
															#line 514 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 515 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 516 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 517 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 3590 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedMultipart()
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedMultipartActor final : public Actor<Void>, public ActorCallback< DeletePersistedMultipartActor, 0, Void >, public ActorCallback< DeletePersistedMultipartActor, 1, Void >, public ActorCallback< DeletePersistedMultipartActor, 2, Void >, public ActorCallback< DeletePersistedMultipartActor, 3, Void >, public FastAllocated<DeletePersistedMultipartActor>, public DeletePersistedMultipartActorState<DeletePersistedMultipartActor> {
															#line 3595 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedMultipartActor>::operator new;
	using FastAllocated<DeletePersistedMultipartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17640238135541691392UL, 4703693954702609152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedMultipartActor, 0, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 1, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 2, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 3, Void >;
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedMultipartActor(std::string const& uploadId) 
															#line 3615 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedMultipartActorState<DeletePersistedMultipartActor>(uploadId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15695025802244908800UL, 1209722574274508288UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedMultipart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedMultipart( std::string const& uploadId ) {
															#line 512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedMultipartActor(uploadId));
															#line 3651 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 563 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Mock S3 Server Implementation for deterministic testing
class MockS3ServerImpl {
public:
	using ObjectData = MockS3GlobalStorage::ObjectData;
	using MultipartUpload = MockS3GlobalStorage::MultipartUpload;

	MockS3ServerImpl() { TraceEvent("MockS3ServerImpl_Constructor").detail("Address", format("%p", this)); }

	~MockS3ServerImpl() { TraceEvent("MockS3ServerImpl_Destructor").detail("Address", format("%p", this)); }

	// S3 Operation Handlers
																#line 3667 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleRequest()
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleRequestActor>
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleRequestActorState {
															#line 3673 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleRequestActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response) 
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response)
															#line 3684 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleRequest", reinterpret_cast<unsigned long>(this));

	}
	~HandleRequestActorState() 
	{
		fdb_probe_actor_destroy("handleRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 579 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3Request") .detail("Method", req->verb) .detail("Resource", req->resource) .detail("ContentLength", req->data.contentLen) .detail("Headers", req->data.headers.size()) .detail("UserAgent", req->data.headers.find("User-Agent") != req->data.headers.end() ? req->data.headers.at("User-Agent") : "N/A") .detail("Host", req->data.headers.find("Host") != req->data.headers.end() ? req->data.headers.at("Host") : "N/A");
															#line 3699 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 592 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string bucket, object;
															#line 593 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::map<std::string, std::string> queryParams;
															#line 594 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->parseS3Request(req->resource, bucket, object, queryParams);
															#line 596 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ParsedRequest") .detail("Bucket", bucket) .detail("Object", object) .detail("QueryParamCount", queryParams.size());
															#line 602 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (queryParams.count("uploads"))
															#line 3711 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 603 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					StrictFuture<Void> __when_expr_0 = self->handleMultipartStart(self, req, response, bucket, object);
															#line 603 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3717 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<HandleRequestActor*>(this)->actor_wait_state = 1;
															#line 603 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 0, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3722 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 604 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (queryParams.count("uploadId"))
															#line 3729 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					{
															#line 605 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						if (queryParams.count("partNumber"))
															#line 3733 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
						{
															#line 606 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							StrictFuture<Void> __when_expr_1 = self->handleUploadPart(self, req, response, bucket, object, queryParams);
															#line 606 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3739 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
							static_cast<HandleRequestActor*>(this)->actor_wait_state = 2;
															#line 606 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 1, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3744 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							loopDepth = 0;
						}
						else
						{
															#line 607 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (req->verb == "POST")
															#line 3751 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 608 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_2 = self->handleMultipartComplete(self, req, response, bucket, object, queryParams);
															#line 608 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3757 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1when3(__when_expr_2.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 3;
															#line 608 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 2, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3762 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 609 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (req->verb == "DELETE")
															#line 3769 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 610 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									StrictFuture<Void> __when_expr_3 = self->handleMultipartAbort(self, req, response, bucket, object, queryParams);
															#line 610 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3775 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1when4(__when_expr_3.get(), loopDepth); };
									static_cast<HandleRequestActor*>(this)->actor_wait_state = 4;
															#line 610 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 3, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3780 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = 0;
								}
								else
								{
															#line 612 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									self->sendError( response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "InvalidRequest", "Unknown multipart operation");
															#line 3787 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = a_body1cont9(loopDepth);
								}
							}
						}
					}
					else
					{
															#line 615 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						if (queryParams.count("tagging"))
															#line 3797 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
						{
															#line 616 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (req->verb == "PUT")
															#line 3801 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_4 = self->handlePutObjectTags(self, req, response, bucket, object);
															#line 617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3807 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), loopDepth); else return a_body1when5(__when_expr_4.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 5;
															#line 617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 4, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3812 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 618 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (req->verb == "GET")
															#line 3819 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 619 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									StrictFuture<Void> __when_expr_5 = self->handleGetObjectTags(self, req, response, bucket, object);
															#line 619 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3825 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), loopDepth); else return a_body1when6(__when_expr_5.get(), loopDepth); };
									static_cast<HandleRequestActor*>(this)->actor_wait_state = 6;
															#line 619 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 5, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3830 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = 0;
								}
								else
								{
															#line 621 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									self->sendError(response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Method not allowed for tagging");
															#line 3837 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = a_body1cont15(loopDepth);
								}
							}
						}
						else
						{
															#line 626 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (queryParams.count("list-type") || (req->verb == "GET" && object.empty()))
															#line 3846 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 628 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_6 = self->handleListObjects(self, req, response, bucket, queryParams);
															#line 628 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3852 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch2(__when_expr_6.getError(), loopDepth); else return a_body1when7(__when_expr_6.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 7;
															#line 628 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 6, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3857 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 629 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (object.empty())
															#line 3864 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 631 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (req->verb == "HEAD")
															#line 3868 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									{
															#line 632 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										StrictFuture<Void> __when_expr_7 = self->handleHeadBucket(self, req, response, bucket);
															#line 632 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3874 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch2(__when_expr_7.getError(), loopDepth); else return a_body1when8(__when_expr_7.get(), loopDepth); };
										static_cast<HandleRequestActor*>(this)->actor_wait_state = 8;
															#line 632 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 7, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3879 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										loopDepth = 0;
									}
									else
									{
															#line 633 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (req->verb == "PUT")
															#line 3886 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										{
															#line 634 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											StrictFuture<Void> __when_expr_8 = self->handlePutBucket(self, req, response, bucket);
															#line 634 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3892 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch2(__when_expr_8.getError(), loopDepth); else return a_body1when9(__when_expr_8.get(), loopDepth); };
											static_cast<HandleRequestActor*>(this)->actor_wait_state = 9;
															#line 634 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 8, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3897 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = 0;
										}
										else
										{
															#line 636 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											self->sendError(response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Bucket operation not supported");
															#line 3904 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = a_body1cont23(loopDepth);
										}
									}
								}
								else
								{
															#line 643 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (req->verb == "PUT")
															#line 3913 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									{
															#line 644 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										StrictFuture<Void> __when_expr_9 = self->handlePutObject(self, req, response, bucket, object);
															#line 644 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3919 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch2(__when_expr_9.getError(), loopDepth); else return a_body1when10(__when_expr_9.get(), loopDepth); };
										static_cast<HandleRequestActor*>(this)->actor_wait_state = 10;
															#line 644 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 9, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3924 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										loopDepth = 0;
									}
									else
									{
															#line 645 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (req->verb == "GET")
															#line 3931 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										{
															#line 646 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											StrictFuture<Void> __when_expr_10 = self->handleGetObject(self, req, response, bucket, object);
															#line 646 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3937 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch2(__when_expr_10.getError(), loopDepth); else return a_body1when11(__when_expr_10.get(), loopDepth); };
											static_cast<HandleRequestActor*>(this)->actor_wait_state = 11;
															#line 646 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 10, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3942 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = 0;
										}
										else
										{
															#line 647 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (req->verb == "DELETE")
															#line 3949 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											{
															#line 648 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												StrictFuture<Void> __when_expr_11 = self->handleDeleteObject(self, req, response, bucket, object);
															#line 648 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3955 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch2(__when_expr_11.getError(), loopDepth); else return a_body1when12(__when_expr_11.get(), loopDepth); };
												static_cast<HandleRequestActor*>(this)->actor_wait_state = 12;
															#line 648 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 11, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3960 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												loopDepth = 0;
											}
											else
											{
															#line 649 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												if (req->verb == "HEAD")
															#line 3967 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												{
															#line 650 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													StrictFuture<Void> __when_expr_12 = self->handleHeadObject(self, req, response, bucket, object);
															#line 650 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3973 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1Catch2(__when_expr_12.getError(), loopDepth); else return a_body1when13(__when_expr_12.get(), loopDepth); };
													static_cast<HandleRequestActor*>(this)->actor_wait_state = 13;
															#line 650 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 12, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3978 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													loopDepth = 0;
												}
												else
												{
															#line 652 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													self->sendError( response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Method not supported");
															#line 3985 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													loopDepth = a_body1cont32(loopDepth);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleRequestActorState();
		static_cast<HandleRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleRequestActorState(); static_cast<HandleRequestActor*>(this)->destroy(); return 0; }
															#line 4023 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleRequestActorState();
		static_cast<HandleRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 658 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3RequestError").error(e).detail("Resource", req->resource);
															#line 659 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			self->sendError(response, 500, "InternalError", "Internal server error");
															#line 4038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont35(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(_, loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont15(int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont18(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont19(_, loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont19(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont20(int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont21(int loopDepth) 
	{
		loopDepth = a_body1cont20(loopDepth);

		return loopDepth;
	}
	int a_body1cont22(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1cont22(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1when8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont22(_, loopDepth);

		return loopDepth;
	}
	int a_body1when8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont22(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1when8(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont23(int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1cont24(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont23(loopDepth);

		return loopDepth;
	}
	int a_body1cont24(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont23(loopDepth);

		return loopDepth;
	}
	int a_body1when9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont24(_, loopDepth);

		return loopDepth;
	}
	int a_body1when9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont24(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1when9(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1when9(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont26(int loopDepth) 
	{
		loopDepth = a_body1cont20(loopDepth);

		return loopDepth;
	}
	int a_body1cont27(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont27(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1when10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont27(_, loopDepth);

		return loopDepth;
	}
	int a_body1when10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont27(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1when10(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1when10(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont28(int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont29(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1cont29(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1when11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1when11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1when11(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1when11(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont30(int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1cont31(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont31(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1when12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont31(_, loopDepth);

		return loopDepth;
	}
	int a_body1when12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont31(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1when12(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1when12(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont32(int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont33(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1cont33(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1when13(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont33(_, loopDepth);

		return loopDepth;
	}
	int a_body1when13(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont33(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1when13(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1when13(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont35(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 5295 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleRequest()
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleRequestActor final : public Actor<Void>, public ActorCallback< HandleRequestActor, 0, Void >, public ActorCallback< HandleRequestActor, 1, Void >, public ActorCallback< HandleRequestActor, 2, Void >, public ActorCallback< HandleRequestActor, 3, Void >, public ActorCallback< HandleRequestActor, 4, Void >, public ActorCallback< HandleRequestActor, 5, Void >, public ActorCallback< HandleRequestActor, 6, Void >, public ActorCallback< HandleRequestActor, 7, Void >, public ActorCallback< HandleRequestActor, 8, Void >, public ActorCallback< HandleRequestActor, 9, Void >, public ActorCallback< HandleRequestActor, 10, Void >, public ActorCallback< HandleRequestActor, 11, Void >, public ActorCallback< HandleRequestActor, 12, Void >, public FastAllocated<HandleRequestActor>, public HandleRequestActorState<HandleRequestActor> {
															#line 5300 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleRequestActor>::operator new;
	using FastAllocated<HandleRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8902812581340183808UL, 4040273312443320064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleRequestActor, 0, Void >;
friend struct ActorCallback< HandleRequestActor, 1, Void >;
friend struct ActorCallback< HandleRequestActor, 2, Void >;
friend struct ActorCallback< HandleRequestActor, 3, Void >;
friend struct ActorCallback< HandleRequestActor, 4, Void >;
friend struct ActorCallback< HandleRequestActor, 5, Void >;
friend struct ActorCallback< HandleRequestActor, 6, Void >;
friend struct ActorCallback< HandleRequestActor, 7, Void >;
friend struct ActorCallback< HandleRequestActor, 8, Void >;
friend struct ActorCallback< HandleRequestActor, 9, Void >;
friend struct ActorCallback< HandleRequestActor, 10, Void >;
friend struct ActorCallback< HandleRequestActor, 11, Void >;
friend struct ActorCallback< HandleRequestActor, 12, Void >;
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleRequestActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response) 
															#line 5329 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleRequestActorState<HandleRequestActor>(self, req, response),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3085149730912622080UL, 127431907239638784UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleRequestActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleRequestActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< HandleRequestActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< HandleRequestActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< HandleRequestActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< HandleRequestActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< HandleRequestActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< HandleRequestActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< HandleRequestActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< HandleRequestActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< HandleRequestActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< HandleRequestActor, 11, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< HandleRequestActor, 12, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleRequest( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response ) {
															#line 575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleRequestActor(self, req, response));
															#line 5373 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 664 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	void parseS3Request(const std::string& resource,
	                    std::string& bucket,
	                    std::string& object,
	                    std::map<std::string, std::string>& queryParams) {

		// Split resource into path and query string
		size_t queryPos = resource.find('?');
		std::string path = (queryPos != std::string::npos) ? resource.substr(0, queryPos) : resource;
		std::string query = (queryPos != std::string::npos) ? resource.substr(queryPos + 1) : "";

		// Parse path: /bucket/object (like real S3)
		if (path.size() > 1) {
			path = path.substr(1); // Remove leading /
			size_t slashPos = path.find('/');
			if (slashPos != std::string::npos) {
				bucket = path.substr(0, slashPos);
				object = path.substr(slashPos + 1);
			} else {
				bucket = path;
				object = "";
			}
		}

		// Parse query parameters
		if (!query.empty()) {
			std::regex paramRegex("([^&=]+)=?([^&]*)");
			std::sregex_iterator iter(query.begin(), query.end(), paramRegex);
			std::sregex_iterator end;

			for (; iter != end; ++iter) {
				std::string key = iter->str(1);
				std::string value = iter->str(2);
				// URL decode the parameter value
				queryParams[key] = HTTP::urlDecode(value);
			}
		}

		// MockS3Server handles S3 HTTP requests where bucket is always the first path component
		// For bucket operations: HEAD /bucket_name
		// For object operations: HEAD /bucket_name/object_path
		if (bucket.empty()) {
			TraceEvent(SevWarn, "MockS3MissingBucketInPath").detail("Resource", resource).detail("QueryString", query);
			throw backup_invalid_url();
		}

		TraceEvent("MockS3ParsedPath")
		    .detail("OriginalResource", resource)
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("QueryString", query);
	}

	// Parse HTTP Range header: "bytes=start-end"
	// Returns true if parsing succeeded, false otherwise
	// Sets rangeStart and rangeEnd to the parsed values
	static bool parseRangeHeader(const std::string& rangeHeader, int64_t& rangeStart, int64_t& rangeEnd) {
		if (rangeHeader.empty()) {
			return false;
		}

		// Check for "bytes=" prefix
		if (rangeHeader.substr(0, 6) != "bytes=") {
			return false;
		}

		std::string range = rangeHeader.substr(6);
		size_t dashPos = range.find('-');
		if (dashPos == std::string::npos) {
			return false;
		}

		try {
			rangeStart = std::stoll(range.substr(0, dashPos));
			std::string endStr = range.substr(dashPos + 1);
			if (endStr.empty()) {
				// Open-ended range (e.g., "bytes=100-")
				rangeEnd = -1; // Indicates open-ended
			} else {
				rangeEnd = std::stoll(endStr);
			}
			return true;
		} catch (...) {
			return false;
		}
	}

	// Multipart Upload Operations
																#line 5465 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartStart()
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartStartActor>
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartStartActorState {
															#line 5471 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartStartActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 5486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartStart", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartStartActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartStart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 758 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartStart").detail("Bucket", bucket).detail("Object", object);
															#line 761 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MultipartUpload upload(bucket, object);
															#line 762 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadId = upload.uploadId;
															#line 763 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().multipartUploads[uploadId] = std::move(upload);
															#line 766 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = persistMultipartState(uploadId);
															#line 766 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleMultipartStartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5511 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleMultipartStartActor*>(this)->actor_wait_state = 1;
															#line 766 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartStartActor, 0, Void >*>(static_cast<HandleMultipartStartActor*>(this)));
															#line 5516 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 769 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<InitiateMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <UploadId>%s</UploadId>\n" "</InitiateMultipartUploadResult>", bucket.c_str(), object.c_str(), uploadId.c_str());
															#line 779 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 781 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartStarted").detail("UploadId", uploadId);
															#line 783 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartStartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartStartActorState(); static_cast<HandleMultipartStartActor*>(this)->destroy(); return 0; }
															#line 5545 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartStartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 769 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<InitiateMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <UploadId>%s</UploadId>\n" "</InitiateMultipartUploadResult>", bucket.c_str(), object.c_str(), uploadId.c_str());
															#line 779 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 781 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartStarted").detail("UploadId", uploadId);
															#line 783 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartStartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartStartActorState(); static_cast<HandleMultipartStartActor*>(this)->destroy(); return 0; }
															#line 5563 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartStartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartStartActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartStartActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartStartActor*>(this)->ActorCallback< HandleMultipartStartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartStartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartStartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartStartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 762 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 5658 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartStart()
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartStartActor final : public Actor<Void>, public ActorCallback< HandleMultipartStartActor, 0, Void >, public FastAllocated<HandleMultipartStartActor>, public HandleMultipartStartActorState<HandleMultipartStartActor> {
															#line 5663 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartStartActor>::operator new;
	using FastAllocated<HandleMultipartStartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(835319250341852160UL, 5406377397076386816UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartStartActor, 0, Void >;
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartStartActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 5680 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartStartActorState<HandleMultipartStartActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10763588338010117376UL, 16830117502434504448UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartStart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartStartActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartStart( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartStartActor(self, req, response, bucket, object));
															#line 5712 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 785 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 5717 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleUploadPart()
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleUploadPartActor>
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleUploadPartActorState {
															#line 5723 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleUploadPartActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 793 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId")),
															#line 794 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partNumber(std::stoi(queryParams.at("partNumber")))
															#line 5744 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleUploadPart", reinterpret_cast<unsigned long>(this));

	}
	~HandleUploadPartActorState() 
	{
		fdb_probe_actor_destroy("handleUploadPart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 796 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3UploadPart") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ContentLength", req->data.contentLen) .detail("ActualContentSize", req->data.content.size()) .detail("ContentPreview", req->data.content.size() > 0 ? req->data.content.substr(0, std::min((size_t)20, req->data.content.size())) : "EMPTY");
															#line 806 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 807 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 5763 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 808 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 809 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5769 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleUploadPartActorState();
				static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 813 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			etag = ObjectData::generateETag(req->data.content);
															#line 814 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadIter->second.parts[partNumber] = { etag, req->data.content };
															#line 817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = persistMultipartState(uploadId);
															#line 817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleUploadPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5783 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleUploadPartActor*>(this)->actor_wait_state = 1;
															#line 817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleUploadPartActor, 0, Void >*>(static_cast<HandleUploadPartActor*>(this)));
															#line 5788 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 820 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 821 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 822 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 823 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 825 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PartUploaded") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ETag", etag);
															#line 830 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5821 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 820 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 821 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 822 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 823 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 825 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PartUploaded") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ETag", etag);
															#line 830 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5843 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleUploadPartActor*>(this)->actor_wait_state > 0) static_cast<HandleUploadPartActor*>(this)->actor_wait_state = 0;
		static_cast<HandleUploadPartActor*>(this)->ActorCallback< HandleUploadPartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleUploadPartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleUploadPartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleUploadPartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 793 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 794 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNumber;
															#line 813 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 5944 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleUploadPart()
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleUploadPartActor final : public Actor<Void>, public ActorCallback< HandleUploadPartActor, 0, Void >, public FastAllocated<HandleUploadPartActor>, public HandleUploadPartActorState<HandleUploadPartActor> {
															#line 5949 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleUploadPartActor>::operator new;
	using FastAllocated<HandleUploadPartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(517276852873744640UL, 12437731594760678144UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleUploadPartActor, 0, Void >;
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleUploadPartActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 5966 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleUploadPartActorState<HandleUploadPartActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2867735766649073408UL, 15512396953593662720UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleUploadPart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleUploadPartActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleUploadPart( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleUploadPartActor(self, req, response, bucket, object, queryParams));
															#line 5998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 832 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 6003 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartComplete()
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartCompleteActor>
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartCompleteActorState {
															#line 6009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartCompleteActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 840 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId"))
															#line 6028 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartComplete", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartCompleteActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartComplete", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 842 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartComplete").detail("UploadId", uploadId);
															#line 844 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 845 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 6047 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 846 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6053 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleMultipartCompleteActorState();
				static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 851 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			combinedContent = std::string();
															#line 852 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			for( auto& part : uploadIter->second.parts ) {
															#line 853 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				combinedContent += part.second.second;
															#line 6065 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 856 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartDebug") .detail("UploadId", uploadId) .detail("PartsCount", uploadIter->second.parts.size()) .detail("CombinedSize", combinedContent.size()) .detail("CombinedPreview", combinedContent.size() > 0 ? combinedContent.substr(0, std::min((size_t)20, combinedContent.size())) : "EMPTY");
															#line 865 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ObjectData obj(combinedContent);
															#line 866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().buckets[bucket][object] = std::move(obj);
															#line 868 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartFinalObject") .detail("UploadId", uploadId) .detail("StoredSize", getGlobalStorage().buckets[bucket][object].content.size()) .detail("StoredPreview", getGlobalStorage().buckets[bucket][object].content.size() > 0 ? getGlobalStorage().buckets[bucket][object].content.substr( 0, std::min((size_t)20, getGlobalStorage().buckets[bucket][object].content.size())) : "EMPTY");
															#line 878 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = persistObject(bucket, object);
															#line 878 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6079 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 1;
															#line 878 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 0, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6084 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 881 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads.erase(uploadId);
															#line 882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedMultipart(uploadId);
															#line 882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6111 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 2;
															#line 882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 1, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6116 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 881 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads.erase(uploadId);
															#line 882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedMultipart(uploadId);
															#line 882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6129 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 2;
															#line 882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 1, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6134 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartCompleteActor*>(this)->ActorCallback< HandleMultipartCompleteActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 885 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<CompleteMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <ETag>%s</ETag>\n" "</CompleteMultipartUploadResult>", bucket.c_str(), object.c_str(), getGlobalStorage().buckets[bucket][object].etag.c_str());
															#line 895 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 897 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartCompleted").detail("UploadId", uploadId).detail("FinalSize", combinedContent.size());
															#line 899 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6224 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 885 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<CompleteMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <ETag>%s</ETag>\n" "</CompleteMultipartUploadResult>", bucket.c_str(), object.c_str(), getGlobalStorage().buckets[bucket][object].etag.c_str());
															#line 895 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 897 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartCompleted").detail("UploadId", uploadId).detail("FinalSize", combinedContent.size());
															#line 899 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6242 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartCompleteActor*>(this)->ActorCallback< HandleMultipartCompleteActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 840 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 851 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string combinedContent;
															#line 6341 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartComplete()
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartCompleteActor final : public Actor<Void>, public ActorCallback< HandleMultipartCompleteActor, 0, Void >, public ActorCallback< HandleMultipartCompleteActor, 1, Void >, public FastAllocated<HandleMultipartCompleteActor>, public HandleMultipartCompleteActorState<HandleMultipartCompleteActor> {
															#line 6346 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartCompleteActor>::operator new;
	using FastAllocated<HandleMultipartCompleteActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18094636290313807104UL, 8712445954361780992UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartCompleteActor, 0, Void >;
friend struct ActorCallback< HandleMultipartCompleteActor, 1, Void >;
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartCompleteActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6364 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartCompleteActorState<HandleMultipartCompleteActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7470605948036193280UL, 16274542347142593024UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartComplete");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartCompleteActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleMultipartCompleteActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartComplete( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartCompleteActor(self, req, response, bucket, object, queryParams));
															#line 6397 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 901 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 6402 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartAbort()
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartAbortActor>
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartAbortActorState {
															#line 6408 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartAbortActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 909 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId"))
															#line 6427 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartAbort", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartAbortActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartAbort", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 911 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartAbort").detail("UploadId", uploadId);
															#line 913 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 6446 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 915 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 916 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6452 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleMultipartAbortActorState();
				static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 920 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().multipartUploads.erase(uploadId);
															#line 921 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = deletePersistedMultipart(uploadId);
															#line 921 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6464 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state = 1;
															#line 921 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartAbortActor, 0, Void >*>(static_cast<HandleMultipartAbortActor*>(this)));
															#line 6469 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 923 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 924 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 925 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 927 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartAborted").detail("UploadId", uploadId);
															#line 929 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6500 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 923 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 924 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 925 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 927 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartAborted").detail("UploadId", uploadId);
															#line 929 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6520 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartAbortActor*>(this)->ActorCallback< HandleMultipartAbortActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 909 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 6617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartAbort()
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartAbortActor final : public Actor<Void>, public ActorCallback< HandleMultipartAbortActor, 0, Void >, public FastAllocated<HandleMultipartAbortActor>, public HandleMultipartAbortActorState<HandleMultipartAbortActor> {
															#line 6622 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartAbortActor>::operator new;
	using FastAllocated<HandleMultipartAbortActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11028151519016293120UL, 5796803521859974144UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartAbortActor, 0, Void >;
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartAbortActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6639 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartAbortActorState<HandleMultipartAbortActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3855017746569260032UL, 6136799802664629248UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartAbort");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartAbortActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartAbort( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartAbortActor(self, req, response, bucket, object, queryParams));
															#line 6671 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 931 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	// Object Tagging Operations
	static Future<Void> handlePutObjectTags(MockS3ServerImpl* self,
	                                        Reference<HTTP::IncomingRequest> req,
	                                        Reference<HTTP::OutgoingResponse> response,
	                                        std::string bucket,
	                                        std::string object) {

		TraceEvent("MockS3PutObjectTags")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagsXML", req->data.content);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		// Parse tags XML (simplified parser)
		std::map<std::string, std::string> tags = self->parseTagsXML(req->data.content);
		objectIter->second.tags = tags;

		response->code = 200;
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3ObjectTagsSet")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagCount", tags.size());

		return Void();
	}

	static Future<Void> handleGetObjectTags(MockS3ServerImpl* self,
	                                        Reference<HTTP::IncomingRequest> req,
	                                        Reference<HTTP::OutgoingResponse> response,
	                                        std::string bucket,
	                                        std::string object) {

		TraceEvent("MockS3GetObjectTags").detail("Bucket", bucket).detail("Object", object);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		// Generate tags XML response
		std::string xml = self->generateTagsXML(objectIter->second.tags);
		self->sendXMLResponse(response, 200, xml);

		TraceEvent("MockS3ObjectTagsRetrieved")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagCount", objectIter->second.tags.size());

		return Void();
	}

	// Basic Object Operations
																#line 6749 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handlePutObject()
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandlePutObjectActor>
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandlePutObjectActorState {
															#line 6755 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandlePutObjectActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 6770 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handlePutObject", reinterpret_cast<unsigned long>(this));

	}
	~HandlePutObjectActorState() 
	{
		fdb_probe_actor_destroy("handlePutObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1011 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3PutObject_Debug") .detail("Bucket", bucket) .detail("Object", object) .detail("ContentLength", req->data.contentLen) .detail("ContentSize", req->data.content.size()) .detail("ContentPreview", req->data.content.substr(0, std::min(100, (int)req->data.content.size())));
															#line 1018 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ObjectData obj(req->data.content);
															#line 1019 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			etag = obj.etag;
															#line 1020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().buckets[bucket][object] = std::move(obj);
															#line 1022 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3PutObject_Stored") .detail("Bucket", bucket) .detail("Object", object) .detail("ETag", etag) .detail("StoredSize", getGlobalStorage().buckets[bucket][object].content.size());
															#line 1029 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = persistObject(bucket, object);
															#line 1029 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandlePutObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6797 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandlePutObjectActor*>(this)->actor_wait_state = 1;
															#line 1029 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandlePutObjectActor, 0, Void >*>(static_cast<HandlePutObjectActor*>(this)));
															#line 6802 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 1032 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 1033 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1036 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PutObject_Response") .detail("Bucket", bucket) .detail("Object", object) .detail("ResponseCode", response->code) .detail("ContentLen", response->data.contentLen) .detail("HasContent", response->data.content != nullptr);
															#line 1043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandlePutObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandlePutObjectActorState(); static_cast<HandlePutObjectActor*>(this)->destroy(); return 0; }
															#line 6833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandlePutObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 1032 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 1033 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1036 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PutObject_Response") .detail("Bucket", bucket) .detail("Object", object) .detail("ResponseCode", response->code) .detail("ContentLen", response->data.contentLen) .detail("HasContent", response->data.content != nullptr);
															#line 1043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandlePutObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandlePutObjectActorState(); static_cast<HandlePutObjectActor*>(this)->destroy(); return 0; }
															#line 6853 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandlePutObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandlePutObjectActor*>(this)->actor_wait_state > 0) static_cast<HandlePutObjectActor*>(this)->actor_wait_state = 0;
		static_cast<HandlePutObjectActor*>(this)->ActorCallback< HandlePutObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandlePutObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandlePutObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandlePutObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 1019 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 6948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handlePutObject()
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandlePutObjectActor final : public Actor<Void>, public ActorCallback< HandlePutObjectActor, 0, Void >, public FastAllocated<HandlePutObjectActor>, public HandlePutObjectActorState<HandlePutObjectActor> {
															#line 6953 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandlePutObjectActor>::operator new;
	using FastAllocated<HandlePutObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11410395138152932608UL, 5385727599133671936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandlePutObjectActor, 0, Void >;
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandlePutObjectActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 6970 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandlePutObjectActorState<HandlePutObjectActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4560658774791891968UL, 15471708591759102464UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handlePutObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandlePutObjectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handlePutObject( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 1005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandlePutObjectActor(self, req, response, bucket, object));
															#line 7002 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1045 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	static Future<Void> handleGetObject(MockS3ServerImpl* self,
	                                    Reference<HTTP::IncomingRequest> req,
	                                    Reference<HTTP::OutgoingResponse> response,
	                                    std::string bucket,
	                                    std::string object) {

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		std::string content = objectIter->second.content;
		std::string etag = objectIter->second.etag;
		std::string contentMD5 = HTTP::computeMD5Sum(content);

		// Handle HTTP Range header for partial content requests
		// This is essential for AsyncFileEncrypted to read encrypted blocks correctly
		int64_t rangeStart = 0;
		int64_t rangeEnd = static_cast<int64_t>(content.size() - 1);
		bool isRangeRequest = false;

		auto rangeHeader = req->data.headers.find("Range");
		if (rangeHeader != req->data.headers.end()) {
			int64_t parsedStart, parsedEnd;
			if (parseRangeHeader(rangeHeader->second, parsedStart, parsedEnd)) {
				rangeStart = parsedStart;
				if (parsedEnd == -1) {
					// Open-ended range (e.g., "bytes=100-")
					rangeEnd = static_cast<int64_t>(content.size() - 1);
				} else {
					rangeEnd = parsedEnd;
				}
				// Clamp range to actual content size
				int64_t contentSize = static_cast<int64_t>(content.size() - 1);
				rangeEnd = std::min(rangeEnd, contentSize);
				rangeStart = std::min(rangeStart, contentSize);
				if (rangeStart <= rangeEnd) {
					isRangeRequest = true;
				}
			}
		}

		// Extract the requested range
		std::string responseContent;
		if (isRangeRequest && rangeStart <= rangeEnd) {
			responseContent =
			    content.substr(static_cast<size_t>(rangeStart), static_cast<size_t>(rangeEnd - rangeStart + 1));
			response->code = 206; // Partial Content
			response->data.headers["Content-Range"] =
			    format("bytes %lld-%lld/%zu", rangeStart, rangeEnd, content.size());
			// For range requests, calculate MD5 of the partial content, not full content
			contentMD5 = HTTP::computeMD5Sum(responseContent);
		} else {
			responseContent = content;
			response->code = 200;
		}

		response->data.headers["ETag"] = etag;
		response->data.headers["Content-Type"] = "binary/octet-stream";
		response->data.headers["Content-MD5"] = contentMD5;

		// Write content to response
		response->data.contentLen = responseContent.size();
		response->data.headers["Content-Length"] = std::to_string(responseContent.size());
		response->data.content->discardAll(); // Clear existing content

		if (!responseContent.empty()) {
			// Use the correct approach: getWriteBuffer from the UnsentPacketQueue
			PacketBuffer* buffer = response->data.content->getWriteBuffer(responseContent.size());
			PacketWriter pw(buffer, nullptr, Unversioned());
			pw.serializeBytes(responseContent);
			pw.finish();
		}

		return Void();
	}

																#line 7091 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleDeleteObject()
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleDeleteObjectActor>
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleDeleteObjectActorState {
															#line 7097 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleDeleteObjectActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 7112 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleDeleteObject", reinterpret_cast<unsigned long>(this));

	}
	~HandleDeleteObjectActorState() 
	{
		fdb_probe_actor_destroy("handleDeleteObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1136 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3DeleteObject").detail("Bucket", bucket).detail("Object", object);
															#line 1138 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto bucketIter = getGlobalStorage().buckets.find(bucket);
															#line 1139 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (bucketIter != getGlobalStorage().buckets.end())
															#line 7131 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1140 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIter->second.erase(object);
															#line 7135 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 1144 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = deletePersistedObject(bucket, object);
															#line 1144 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7141 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state = 1;
															#line 1144 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleDeleteObjectActor, 0, Void >*>(static_cast<HandleDeleteObjectActor*>(this)));
															#line 7146 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1146 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 1147 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1148 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 1150 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 1152 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleDeleteObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleDeleteObjectActorState(); static_cast<HandleDeleteObjectActor*>(this)->destroy(); return 0; }
															#line 7177 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleDeleteObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1146 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 1147 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1148 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 1150 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 1152 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleDeleteObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleDeleteObjectActorState(); static_cast<HandleDeleteObjectActor*>(this)->destroy(); return 0; }
															#line 7197 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleDeleteObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state > 0) static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state = 0;
		static_cast<HandleDeleteObjectActor*>(this)->ActorCallback< HandleDeleteObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 7290 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleDeleteObject()
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleDeleteObjectActor final : public Actor<Void>, public ActorCallback< HandleDeleteObjectActor, 0, Void >, public FastAllocated<HandleDeleteObjectActor>, public HandleDeleteObjectActorState<HandleDeleteObjectActor> {
															#line 7295 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleDeleteObjectActor>::operator new;
	using FastAllocated<HandleDeleteObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5271123296247428096UL, 9694226392590225664UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleDeleteObjectActor, 0, Void >;
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleDeleteObjectActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 7312 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleDeleteObjectActorState<HandleDeleteObjectActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7788094060450111232UL, 5901935141690097664UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleDeleteObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleDeleteObjectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleDeleteObject( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 1130 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleDeleteObjectActor(self, req, response, bucket, object));
															#line 7344 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1154 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	static Future<Void> handleHeadObject(MockS3ServerImpl* self,
	                                     Reference<HTTP::IncomingRequest> req,
	                                     Reference<HTTP::OutgoingResponse> response,
	                                     std::string bucket,
	                                     std::string object) {

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			TraceEvent("MockS3HeadObjectNoBucket")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("AvailableBuckets", getGlobalStorage().buckets.size());
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			TraceEvent("MockS3HeadObjectNoObject")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("ObjectsInBucket", bucketIter->second.size());
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		const ObjectData& obj = objectIter->second;
		std::string etag = obj.etag;
		size_t contentSize = obj.content.size();
		std::string preview = contentSize > 0 ? obj.content.substr(0, std::min((size_t)20, contentSize)) : "EMPTY";

		TraceEvent("MockS3HeadObjectFound")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("Size", contentSize)
		    .detail("Preview", preview);

		response->code = 200;
		response->data.headers["ETag"] = etag;
		response->data.headers["Content-Length"] = std::to_string(contentSize);
		response->data.headers["Content-Type"] = "binary/octet-stream";
		// HEAD requests need contentLen set to actual size for headers
		response->data.contentLen = contentSize; // This controls ResponseContentSize in HTTP logs

		return Void();
	}

	// S3 ListObjects Operation
	static Future<Void> handleListObjects(MockS3ServerImpl* self,
	                                      Reference<HTTP::IncomingRequest> req,
	                                      Reference<HTTP::OutgoingResponse> response,
	                                      std::string bucket,
	                                      std::map<std::string, std::string> queryParams) {

		TraceEvent("MockS3ListObjects").detail("Bucket", bucket).detail("QueryParamCount", queryParams.size());

		// Get query parameters for listing
		std::string prefix = queryParams.count("prefix") ? queryParams.at("prefix") : "";
		std::string delimiter = queryParams.count("delimiter") ? queryParams.at("delimiter") : "";
		std::string marker = queryParams.count("marker") ? queryParams.at("marker") : "";
		std::string continuationToken =
		    queryParams.count("continuation-token") ? queryParams.at("continuation-token") : "";
		int maxKeys = queryParams.count("max-keys") ? std::stoi(queryParams.at("max-keys")) : 1000;

		TraceEvent("MockS3ListObjectsDebug")
		    .detail("Bucket", bucket)
		    .detail("Prefix", prefix)
		    .detail("Delimiter", delimiter)
		    .detail("Marker", marker)
		    .detail("ContinuationToken", continuationToken)
		    .detail("MaxKeys", maxKeys);

		// Find bucket
		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		// Collect all matching objects first
		std::vector<std::pair<std::string, const ObjectData*>> matchingObjects;
		for (const auto& objectPair : bucketIter->second) {
			const std::string& objectName = objectPair.first;
			const ObjectData& objectData = objectPair.second;

			// Apply prefix filter
			if (!prefix.empty() && objectName.find(prefix) != 0) {
				continue;
			}

			matchingObjects.push_back({ objectName, &objectData });
		}

		// Sort objects by name for consistent pagination
		std::sort(matchingObjects.begin(), matchingObjects.end());

		// Find starting point for pagination
		size_t startIndex = 0;
		if (!marker.empty()) {
			for (size_t i = 0; i < matchingObjects.size(); i++) {
				if (matchingObjects[i].first > marker) {
					startIndex = i;
					break;
				}
			}
		} else if (!continuationToken.empty()) {
			// Simple continuation token implementation (just use the last object name)
			for (size_t i = 0; i < matchingObjects.size(); i++) {
				if (matchingObjects[i].first > continuationToken) {
					startIndex = i;
					break;
				}
			}
		}

		// Build list of objects for this page
		std::string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<ListBucketResult>\n";
		xml += "<Name>" + bucket + "</Name>\n";
		xml += "<Prefix>" + prefix + "</Prefix>\n";
		xml += "<MaxKeys>" + std::to_string(maxKeys) + "</MaxKeys>\n";

		if (!marker.empty()) {
			xml += "<Marker>" + marker + "</Marker>\n";
		}

		int count = 0;
		std::string lastKey;
		size_t totalMatching = matchingObjects.size();

		for (size_t i = startIndex; i < matchingObjects.size() && count < maxKeys; i++) {
			const std::string& objectName = matchingObjects[i].first;
			const ObjectData* objectData = matchingObjects[i].second;

			xml += "<Contents>\n";
			xml += "<Key>" + objectName + "</Key>\n";
			xml += "<LastModified>" + std::to_string((int64_t)objectData->lastModified) + "</LastModified>\n";
			xml += "<ETag>" + objectData->etag + "</ETag>\n";
			xml += "<Size>" + std::to_string(objectData->content.size()) + "</Size>\n";
			xml += "<StorageClass>STANDARD</StorageClass>\n";
			xml += "</Contents>\n";

			lastKey = objectName;
			count++;
		}

		// Determine if there are more results
		bool isTruncated = (startIndex + count) < totalMatching;
		xml += "<IsTruncated>" + std::string(isTruncated ? "true" : "false") + "</IsTruncated>\n";

		if (isTruncated && !lastKey.empty()) {
			xml += "<NextMarker>" + lastKey + "</NextMarker>\n";
		}

		xml += "</ListBucketResult>";

		self->sendXMLResponse(response, 200, xml);

		TraceEvent("MockS3ListObjectsCompleted")
		    .detail("Bucket", bucket)
		    .detail("Prefix", prefix)
		    .detail("ObjectCount", count)
		    .detail("StartIndex", startIndex)
		    .detail("TotalMatching", totalMatching)
		    .detail("IsTruncated", isTruncated)
		    .detail("NextMarker", isTruncated ? lastKey : "");

		return Void();
	}

	// S3 Bucket Operations
	static Future<Void> handleHeadBucket(MockS3ServerImpl* self,
	                                     Reference<HTTP::IncomingRequest> req,
	                                     Reference<HTTP::OutgoingResponse> response,
	                                     std::string bucket) {

		TraceEvent("MockS3HeadBucket").detail("Bucket", bucket);

		// Ensure bucket exists in our storage (implicit creation like real S3)
		if (getGlobalStorage().buckets.find(bucket) == getGlobalStorage().buckets.end()) {
			getGlobalStorage().buckets[bucket] = std::map<std::string, ObjectData>();
		}

		response->code = 200;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3BucketHead").detail("Bucket", bucket);

		return Void();
	}

	static Future<Void> handlePutBucket(MockS3ServerImpl* self,
	                                    Reference<HTTP::IncomingRequest> req,
	                                    Reference<HTTP::OutgoingResponse> response,
	                                    std::string bucket) {

		TraceEvent("MockS3PutBucket").detail("Bucket", bucket);

		// Ensure bucket exists in our storage (implicit creation)
		if (getGlobalStorage().buckets.find(bucket) == getGlobalStorage().buckets.end()) {
			getGlobalStorage().buckets[bucket] = std::map<std::string, ObjectData>();
		}

		response->code = 200;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3BucketCreated").detail("Bucket", bucket);

		return Void();
	}

	// Utility Methods

	void sendError(Reference<HTTP::OutgoingResponse> response,
	               int code,
	               const std::string& errorCode,
	               const std::string& message) {

		TraceEvent("MockS3Error").detail("Code", code).detail("ErrorCode", errorCode).detail("Message", message);

		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		                         "<Error>\n"
		                         "  <Code>%s</Code>\n"
		                         "  <Message>%s</Message>\n"
		                         "</Error>",
		                         errorCode.c_str(),
		                         message.c_str());

		sendXMLResponse(response, code, xml);
	}

	void sendXMLResponse(Reference<HTTP::OutgoingResponse> response, int code, const std::string& xml) {
		TraceEvent("MockS3SendXMLResponse_Start")
		    .detail("Code", code)
		    .detail("XMLSize", xml.size())
		    .detail("XMLPreview", xml.size() > 0 ? xml.substr(0, std::min((size_t)50, xml.size())) : "EMPTY");

		response->code = code;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.headers["Content-Length"] = std::to_string(xml.size());
		response->data.headers["Content-MD5"] = HTTP::computeMD5Sum(xml);

		// Actually put the XML content into the response
		if (xml.empty()) {
			response->data.contentLen = 0;
			TraceEvent("MockS3SendXMLResponse_Empty").detail("ResponseCode", response->code);
		} else {
			// Use the existing content queue instead of creating a new one
			// This prevents memory management issues and potential canBeSet() failures
			size_t contentSize = xml.size();
			response->data.contentLen = contentSize;

			// Clear any existing content and write the XML
			response->data.content->discardAll();
			PacketBuffer* buffer = response->data.content->getWriteBuffer(contentSize);
			PacketWriter pw(buffer, nullptr, Unversioned());
			pw.serializeBytes(xml);
			pw.finish();
		}

		TraceEvent("MockS3SendXMLResponse_Complete")
		    .detail("FinalCode", response->code)
		    .detail("FinalContentLen", response->data.contentLen)
		    .detail("XMLSize", xml.size());
	}

	std::map<std::string, std::string> parseTagsXML(const std::string& xml) {
		std::map<std::string, std::string> tags;

		// Simplified XML parsing for tags - this would need a proper XML parser in production
		std::regex tagRegex("<Tag><Key>([^<]+)</Key><Value>([^<]*)</Value></Tag>");
		std::sregex_iterator iter(xml.begin(), xml.end(), tagRegex);
		std::sregex_iterator end;

		for (; iter != end; ++iter) {
			std::string key = iter->str(1);
			std::string value = iter->str(2);
			tags[key] = value;

			TraceEvent("MockS3ParsedTag").detail("Key", key).detail("Value", value);
		}

		return tags;
	}

	std::string generateTagsXML(const std::map<std::string, std::string>& tags) {
		std::string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<Tagging><TagSet>";

		for (const auto& tag : tags) {
			xml += "<Tag><Key>" + tag.first + "</Key><Value>" + tag.second + "</Value></Tag>";
		}

		xml += "</TagSet></Tagging>";
		return xml;
	}
};

// Global registry to track registered servers and avoid conflicts
static std::map<std::string, bool> registeredServers;

// Clear global storage state for clean test runs
static void clearSingletonState() {
	getGlobalStorage().buckets.clear();
	getGlobalStorage().multipartUploads.clear();
	TraceEvent("MockS3ServerImpl_StateCleared");
}

// Request Handler Implementation - Each handler instance works with global storage
Future<Void> MockS3RequestHandler::handleRequest(Reference<HTTP::IncomingRequest> req,
                                                 Reference<HTTP::OutgoingResponse> response) {
	// Guard against calling virtual functions during destruction
	if (destructing.load()) {
		TraceEvent(SevWarn, "MockS3RequestHandler_DestructingGuard")
		    .detail("Method", "handleRequest")
		    .detail("Resource", req->resource);
		return Void();
	}

	// Create a temporary instance just to use its static handleRequest method
	// All actual storage is in g_mockS3Storage which is truly global
	static MockS3ServerImpl serverInstance;
	return MockS3ServerImpl::handleRequest(&serverInstance, req, response);
}

Reference<HTTP::IRequestHandler> MockS3RequestHandler::clone() {
	// Guard against calling virtual functions during destruction
	if (destructing.load()) {
		TraceEvent(SevWarn, "MockS3RequestHandler_DestructingGuard").detail("Method", "clone");
		// Return nullptr - caller must handle this gracefully
		return Reference<HTTP::IRequestHandler>();
	}
	return makeReference<MockS3RequestHandler>();
}

// Safe server registration that prevents conflicts (internal implementation)
															#line 7687 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via registerMockS3Server_impl()
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class RegisterMockS3Server_implActor>
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class RegisterMockS3Server_implActorState {
															#line 7694 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	RegisterMockS3Server_implActorState(std::string const& ip,std::string const& port) 
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : ip(ip),
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   port(port),
															#line 1494 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   serverKey(ip + ":" + port)
															#line 7705 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this));

	}
	~RegisterMockS3Server_implActorState() 
	{
		fdb_probe_actor_destroy("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1497 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Start") .detail("IP", ip) .detail("Port", port) .detail("ServerKey", serverKey) .detail("IsSimulated", g_network->isSimulated()) .detail("AlreadyRegistered", registeredServers.count(serverKey) > 0);
															#line 1506 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (registeredServers.count(serverKey))
															#line 7722 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1507 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent(SevWarn, "MockS3ServerAlreadyRegistered").detail("Address", serverKey);
															#line 1508 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<RegisterMockS3Server_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterMockS3Server_implActorState(); static_cast<RegisterMockS3Server_implActor*>(this)->destroy(); return 0; }
															#line 7728 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<RegisterMockS3Server_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~RegisterMockS3Server_implActorState();
				static_cast<RegisterMockS3Server_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1512 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Calling registerSimHTTPServer") .detail("Address", serverKey);
															#line 1516 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = g_simulator->registerSimHTTPServer(ip, port, makeReference<MockS3RequestHandler>());
															#line 1516 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7741 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 1;
															#line 1516 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 0, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7746 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RegisterMockS3Server_implActorState();
		static_cast<RegisterMockS3Server_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1547 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<RegisterMockS3Server_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterMockS3Server_implActorState(); static_cast<RegisterMockS3Server_implActor*>(this)->destroy(); return 0; }
															#line 7775 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<RegisterMockS3Server_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterMockS3Server_implActorState();
		static_cast<RegisterMockS3Server_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1539 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3ServerRegistrationFailed") .error(e) .detail("Address", serverKey) .detail("ErrorCode", e.code()) .detail("ErrorName", e.name());
															#line 1544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 7790 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1517 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		registeredServers[serverKey] = true;
															#line 1521 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!getGlobalStorage().persistenceEnabled)
															#line 7806 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1522 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string persistenceDir = "simfdb/mocks3";
															#line 1523 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			enableMockS3Persistence(persistenceDir);
															#line 1524 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerPersistenceEnabled") .detail("Address", serverKey) .detail("PersistenceDir", persistenceDir);
															#line 1529 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_1 = loadMockS3PersistedStateFuture();
															#line 1529 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7818 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
			static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 2;
															#line 1529 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 1, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7823 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1517 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		registeredServers[serverKey] = true;
															#line 1521 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!getGlobalStorage().persistenceEnabled)
															#line 7839 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1522 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string persistenceDir = "simfdb/mocks3";
															#line 1523 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			enableMockS3Persistence(persistenceDir);
															#line 1524 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerPersistenceEnabled") .detail("Address", serverKey) .detail("PersistenceDir", persistenceDir);
															#line 1529 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_1 = loadMockS3PersistedStateFuture();
															#line 1529 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7851 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
			static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 2;
															#line 1529 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 1, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7856 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state > 0) static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterMockS3Server_implActor*>(this)->ActorCallback< RegisterMockS3Server_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 1532 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRegistered").detail("Address", serverKey).detail("Success", true);
															#line 1534 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Complete") .detail("Address", serverKey) .detail("TotalRegistered", registeredServers.size());
															#line 7947 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state > 0) static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterMockS3Server_implActor*>(this)->ActorCallback< RegisterMockS3Server_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string ip;
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string port;
															#line 1494 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string serverKey;
															#line 8058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via registerMockS3Server_impl()
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class RegisterMockS3Server_implActor final : public Actor<Void>, public ActorCallback< RegisterMockS3Server_implActor, 0, Void >, public ActorCallback< RegisterMockS3Server_implActor, 1, Void >, public FastAllocated<RegisterMockS3Server_implActor>, public RegisterMockS3Server_implActorState<RegisterMockS3Server_implActor> {
															#line 8063 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<RegisterMockS3Server_implActor>::operator new;
	using FastAllocated<RegisterMockS3Server_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7871210858336287232UL, 6799410786342460416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RegisterMockS3Server_implActor, 0, Void >;
friend struct ActorCallback< RegisterMockS3Server_implActor, 1, Void >;
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	RegisterMockS3Server_implActor(std::string const& ip,std::string const& port) 
															#line 8081 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   RegisterMockS3Server_implActorState<RegisterMockS3Server_implActor>(ip, port),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13948469852865528320UL, 9795983047084281344UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("registerMockS3Server_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RegisterMockS3Server_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RegisterMockS3Server_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> registerMockS3Server_impl( std::string const& ip, std::string const& port ) {
															#line 1493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new RegisterMockS3Server_implActor(ip, port));
															#line 8115 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1549 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Public Interface Implementation
															#line 8121 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via startMockS3Server()
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class StartMockS3ServerActor>
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerActorState {
															#line 8128 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerActorState(NetworkAddress const& listenAddress) 
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : listenAddress(listenAddress)
															#line 8135 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("startMockS3Server", reinterpret_cast<unsigned long>(this));

	}
	~StartMockS3ServerActorState() 
	{
		fdb_probe_actor_destroy("startMockS3Server", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerStarting").detail("ListenAddress", listenAddress.toString());
															#line 8150 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1555 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerRegistering") .detail("IP", listenAddress.ip.toString()) .detail("Port", std::to_string(listenAddress.port)) .detail("IsSimulated", g_network->isSimulated());
															#line 1561 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = registerMockS3Server_impl(listenAddress.ip.toString(), std::to_string(listenAddress.port));
															#line 1561 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<StartMockS3ServerActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 8158 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StartMockS3ServerActor*>(this)->actor_wait_state = 1;
															#line 1561 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerActor, 0, Void >*>(static_cast<StartMockS3ServerActor*>(this)));
															#line 8163 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartMockS3ServerActorState();
		static_cast<StartMockS3ServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1572 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerActorState(); static_cast<StartMockS3ServerActor*>(this)->destroy(); return 0; }
															#line 8192 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerActorState();
		static_cast<StartMockS3ServerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1568 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3ServerStartError").error(e).detail("ListenAddress", listenAddress.toString());
															#line 1569 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 8207 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1563 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerStarted") .detail("ListenAddress", listenAddress.toString()) .detail("HandlerCreated", true);
															#line 8221 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1563 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerStarted") .detail("ListenAddress", listenAddress.toString()) .detail("HandlerCreated", true);
															#line 8230 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartMockS3ServerActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerActor*>(this)->ActorCallback< StartMockS3ServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	NetworkAddress listenAddress;
															#line 8325 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via startMockS3Server()
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerActor final : public Actor<Void>, public ActorCallback< StartMockS3ServerActor, 0, Void >, public FastAllocated<StartMockS3ServerActor>, public StartMockS3ServerActorState<StartMockS3ServerActor> {
															#line 8330 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<StartMockS3ServerActor>::operator new;
	using FastAllocated<StartMockS3ServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5931505061029545728UL, 11551306937193052928UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartMockS3ServerActor, 0, Void >;
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerActor(NetworkAddress const& listenAddress) 
															#line 8347 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   StartMockS3ServerActorState<StartMockS3ServerActor>(listenAddress),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16442790091880884736UL, 12597676879247326208UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startMockS3Server");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartMockS3ServerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> startMockS3Server( NetworkAddress const& listenAddress ) {
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new StartMockS3ServerActor(listenAddress));
															#line 8380 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1574 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Clear all MockS3 global storage - called at the start of each simulation test
void clearMockS3Storage() {
	getGlobalStorage().clearStorage();
}

// Enable persistence for MockS3 storage
void enableMockS3Persistence(const std::string& persistenceDir) {
	getGlobalStorage().enablePersistence(persistenceDir);
	TraceEvent("MockS3PersistenceConfigured").detail("Directory", persistenceDir);
}

// ACTOR: Load persisted objects from disk
															#line 8397 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadPersistedObjects()
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadPersistedObjectsActor>
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedObjectsActorState {
															#line 8404 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedObjectsActorState(std::string const& persistenceDir) 
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir(persistenceDir),
															#line 1588 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   objectsDir(persistenceDir + "/objects")
															#line 8413 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPersistedObjects", reinterpret_cast<unsigned long>(this));

	}
	~LoadPersistedObjectsActorState() 
	{
		fdb_probe_actor_destroy("loadPersistedObjects", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1590 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(objectsDir))
															#line 8428 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1591 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3LoadObjects").detail("Status", "NoObjectsDir");
															#line 1592 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadPersistedObjectsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedObjectsActorState(); static_cast<LoadPersistedObjectsActor*>(this)->destroy(); return 0; }
															#line 8434 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadPersistedObjectsActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadPersistedObjectsActorState();
				static_cast<LoadPersistedObjectsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1597 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				buckets = platform::listFiles(objectsDir, "");
															#line 1599 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::sort(buckets.begin(), buckets.end());
															#line 1600 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIdx = 0;
															#line 1602 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIdx = 0;
															#line 8449 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPersistedObjectsActorState();
		static_cast<LoadPersistedObjectsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1657 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadPersistedObjectsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedObjectsActorState(); static_cast<LoadPersistedObjectsActor*>(this)->destroy(); return 0; }
															#line 8478 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadPersistedObjectsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPersistedObjectsActorState();
		static_cast<LoadPersistedObjectsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1654 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadObjectsFailed").error(e);
															#line 8491 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1652 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectsLoaded").detail("BucketsCount", getGlobalStorage().buckets.size());
															#line 8506 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1602 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(bucketIdx < buckets.size()))
															#line 8522 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1603 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucket = buckets[bucketIdx];
															#line 1604 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bucket == "." || bucket == "..")
															#line 8530 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1continue1(loopDepth); // continue
		}
															#line 1607 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketDir = objectsDir + "/" + bucket;
															#line 1608 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!directoryExists(bucketDir))
															#line 8538 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1continue1(loopDepth); // continue
		}
															#line 1612 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		files = platform::listFiles(bucketDir, "");
															#line 1613 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::sort(files.begin(), files.end());
															#line 1614 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx = 0;
															#line 1616 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx = 0;
															#line 8550 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1continue1(int loopDepth) 
	{
															#line 1602 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketIdx++;
															#line 8572 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1602 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketIdx++;
															#line 8581 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 1616 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(fileIdx < files.size()))
															#line 8597 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileName = files[fileIdx];
															#line 1620 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (fileName.size() > OBJECT_META_SUFFIX_LEN && fileName.substr(fileName.size() - OBJECT_META_SUFFIX_LEN) == OBJECT_META_SUFFIX)
															#line 8605 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1623 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			objectName = fileName.substr(0, fileName.size() - OBJECT_META_SUFFIX_LEN);
															#line 1624 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			dataPath = bucketDir + "/" + objectName + OBJECT_DATA_SUFFIX;
															#line 1625 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaPath = bucketDir + "/" + fileName;
															#line 1627 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(dataPath))
															#line 8615 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1628 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent(SevWarn, "MockS3LoadObjectSkipped") .detail("Bucket", bucket) .detail("Object", objectName) .detail("Reason", "NoDataFile");
															#line 8619 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				return a_body1loopBody1continue1(loopDepth); // continue
			}
															#line 1636 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<std::string> __when_expr_0 = readFileContent(dataPath);
															#line 1636 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8626 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 1;
															#line 1636 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedObjectsActor, 0, std::string >*>(static_cast<LoadPersistedObjectsActor*>(this)));
															#line 8631 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1continue1(int loopDepth) 
	{
															#line 1616 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 8658 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1616 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 8667 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
															#line 1637 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = readFileContent(metaPath);
															#line 1637 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8678 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 2;
															#line 1637 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedObjectsActor, 1, std::string >*>(static_cast<LoadPersistedObjectsActor*>(this)));
															#line 8683 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(std::string const& __content,int loopDepth) 
	{
															#line 1636 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content = __content;
															#line 8692 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(std::string && __content,int loopDepth) 
	{
		content = std::move(__content);
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedObjectsActor*>(this)->ActorCallback< LoadPersistedObjectsActor, 0, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont5(int loopDepth) 
	{
															#line 1640 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		MockS3GlobalStorage::ObjectData obj(content);
															#line 1641 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		deserializeObjectMeta(metaJson, obj);
															#line 1642 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().buckets[bucket][objectName] = std::move(obj);
															#line 1644 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectRestored") .detail("Bucket", bucket) .detail("Object", objectName) .detail("Size", content.size());
															#line 8777 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(std::string const& __metaJson,int loopDepth) 
	{
															#line 1637 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		metaJson = __metaJson;
															#line 8786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(std::string && __metaJson,int loopDepth) 
	{
		metaJson = std::move(__metaJson);
		loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedObjectsActor*>(this)->ActorCallback< LoadPersistedObjectsActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 1588 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string objectsDir;
															#line 1597 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> buckets;
															#line 1600 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int bucketIdx;
															#line 1603 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1607 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucketDir;
															#line 1612 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> files;
															#line 1614 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int fileIdx;
															#line 1617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string fileName;
															#line 1623 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string objectName;
															#line 1624 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 1625 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 1636 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 1637 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaJson;
															#line 8902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadPersistedObjects()
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedObjectsActor final : public Actor<Void>, public ActorCallback< LoadPersistedObjectsActor, 0, std::string >, public ActorCallback< LoadPersistedObjectsActor, 1, std::string >, public FastAllocated<LoadPersistedObjectsActor>, public LoadPersistedObjectsActorState<LoadPersistedObjectsActor> {
															#line 8907 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadPersistedObjectsActor>::operator new;
	using FastAllocated<LoadPersistedObjectsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11150201489787593728UL, 10520520283075096576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPersistedObjectsActor, 0, std::string >;
friend struct ActorCallback< LoadPersistedObjectsActor, 1, std::string >;
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedObjectsActor(std::string const& persistenceDir) 
															#line 8925 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadPersistedObjectsActorState<LoadPersistedObjectsActor>(persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15447654064422368256UL, 7597335790883620352UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPersistedObjects");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPersistedObjectsActor, 0, std::string >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPersistedObjectsActor, 1, std::string >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadPersistedObjects( std::string const& persistenceDir ) {
															#line 1587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadPersistedObjectsActor(persistenceDir));
															#line 8959 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1659 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Load persisted multipart uploads from disk
															#line 8965 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadPersistedMultipartUploads()
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadPersistedMultipartUploadsActor>
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedMultipartUploadsActorState {
															#line 8972 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedMultipartUploadsActorState(std::string const& persistenceDir) 
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir(persistenceDir),
															#line 1662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   multipartDir(persistenceDir + "/multipart")
															#line 8981 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this));

	}
	~LoadPersistedMultipartUploadsActorState() 
	{
		fdb_probe_actor_destroy("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1664 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(multipartDir))
															#line 8996 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1665 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3LoadMultipart").detail("Status", "NoMultipartDir");
															#line 1666 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedMultipartUploadsActorState(); static_cast<LoadPersistedMultipartUploadsActor*>(this)->destroy(); return 0; }
															#line 9002 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadPersistedMultipartUploadsActorState();
				static_cast<LoadPersistedMultipartUploadsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1671 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				files = platform::listFiles(multipartDir, "");
															#line 1672 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::sort(files.begin(), files.end());
															#line 1673 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				fileIdx = 0;
															#line 1675 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				fileIdx = 0;
															#line 9017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPersistedMultipartUploadsActorState();
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1741 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedMultipartUploadsActorState(); static_cast<LoadPersistedMultipartUploadsActor*>(this)->destroy(); return 0; }
															#line 9046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPersistedMultipartUploadsActorState();
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1738 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadMultipartFailed").error(e);
															#line 9059 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1736 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartUploadsLoaded").detail("UploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9074 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1675 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(fileIdx < files.size()))
															#line 9090 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1676 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileName = files[fileIdx];
															#line 1679 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (fileName.size() > MULTIPART_STATE_SUFFIX_LEN && fileName.substr(fileName.size() - MULTIPART_STATE_SUFFIX_LEN) == MULTIPART_STATE_SUFFIX)
															#line 9098 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1681 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadId = fileName.substr(0, fileName.size() - MULTIPART_STATE_SUFFIX_LEN);
															#line 1682 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			statePath = multipartDir + "/" + fileName;
															#line 1685 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<std::string> __when_expr_0 = readFileContent(statePath);
															#line 1685 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9108 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 1;
															#line 1685 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9113 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1continue1(int loopDepth) 
	{
															#line 1675 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 9140 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1675 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 9149 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1686 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (stateJson.empty())
															#line 9158 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1687 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadMultipartSkipped") .detail("UploadId", uploadId) .detail("Reason", "EmptyStateFile");
															#line 9162 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			return a_body1continue1(loopDepth); // continue
		}
															#line 1694 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload = MockS3GlobalStorage::MultipartUpload("", "");
															#line 1695 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload.uploadId = uploadId;
															#line 1696 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		deserializeMultipartState(stateJson, upload);
															#line 1699 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 1700 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		maxAttempts = 10000;
															#line 1701 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 9177 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::string const& __stateJson,int loopDepth) 
	{
															#line 1685 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		stateJson = __stateJson;
															#line 9186 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::string && __stateJson,int loopDepth) 
	{
		stateJson = std::move(__stateJson);
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1726 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartUploadRestored") .detail("UploadId", uploadId) .detail("Bucket", upload.bucket) .detail("Object", upload.object) .detail("PartsCount", upload.parts.size());
															#line 1732 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads[uploadId] = std::move(upload);
															#line 9267 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1701 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partNum <= maxAttempts))
															#line 9283 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1702 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = multipartDir + "/" + uploadId + ".part." + std::to_string(partNum);
															#line 1703 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 1705 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!fileExists(partPath) || !fileExists(partMetaPath))
															#line 9293 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1709 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = readFileContent(partPath);
															#line 1709 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9301 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 2;
															#line 1709 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9306 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1710 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_2 = readFileContent(partMetaPath);
															#line 1710 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9330 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 3;
															#line 1710 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9335 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::string const& __partData,int loopDepth) 
	{
															#line 1709 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partData = __partData;
															#line 9344 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::string && __partData,int loopDepth) 
	{
		partData = std::move(__partData);
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont4(int loopDepth) 
	{
															#line 1713 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		using namespace rapidjson;
															#line 1714 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		Document doc;
															#line 1715 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		doc.Parse(partMetaJson.c_str());
															#line 1716 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string etag = doc.HasMember("etag") && doc["etag"].IsString() ? doc["etag"].GetString() : "";
															#line 1717 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload.parts[partNum] = { etag, partData };
															#line 1719 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartPartRestored") .detail("UploadId", uploadId) .detail("PartNumber", partNum) .detail("Size", partData.size());
															#line 1701 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 9435 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(std::string const& __partMetaJson,int loopDepth) 
	{
															#line 1710 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = __partMetaJson;
															#line 9444 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(std::string && __partMetaJson,int loopDepth) 
	{
		partMetaJson = std::move(__partMetaJson);
		loopDepth = a_body1loopBody1cont3loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 1662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string multipartDir;
															#line 1671 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> files;
															#line 1673 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int fileIdx;
															#line 1676 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string fileName;
															#line 1681 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 1682 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string statePath;
															#line 1685 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string stateJson;
															#line 1694 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3GlobalStorage::MultipartUpload upload;
															#line 1699 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 1700 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int maxAttempts;
															#line 1702 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 1703 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 1709 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partData;
															#line 1710 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaJson;
															#line 9562 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadPersistedMultipartUploads()
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedMultipartUploadsActor final : public Actor<Void>, public ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >, public ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >, public ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >, public FastAllocated<LoadPersistedMultipartUploadsActor>, public LoadPersistedMultipartUploadsActorState<LoadPersistedMultipartUploadsActor> {
															#line 9567 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadPersistedMultipartUploadsActor>::operator new;
	using FastAllocated<LoadPersistedMultipartUploadsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5352484692135970816UL, 233000732787558656UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >;
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >;
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >;
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedMultipartUploadsActor(std::string const& persistenceDir) 
															#line 9586 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadPersistedMultipartUploadsActorState<LoadPersistedMultipartUploadsActor>(persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17523876871774881024UL, 15657059887496463360UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPersistedMultipartUploads");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadPersistedMultipartUploads( std::string const& persistenceDir ) {
															#line 1661 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadPersistedMultipartUploadsActor(persistenceDir));
															#line 9621 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1743 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Load all persisted state from disk
															#line 9627 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadMockS3PersistedStateImpl()
															#line 1745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadMockS3PersistedStateImplActor>
															#line 1745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadMockS3PersistedStateImplActorState {
															#line 9634 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadMockS3PersistedStateImplActorState() 
															#line 1745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1746 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir()
															#line 9641 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this));

	}
	~LoadMockS3PersistedStateImplActorState() 
	{
		fdb_probe_actor_destroy("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1748 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled || getGlobalStorage().persistenceLoaded)
															#line 9656 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1749 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadMockS3PersistedStateImplActorState(); static_cast<LoadMockS3PersistedStateImplActor*>(this)->destroy(); return 0; }
															#line 9660 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadMockS3PersistedStateImplActorState();
				static_cast<LoadMockS3PersistedStateImplActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			persistenceDir = getGlobalStorage().persistenceDir;
															#line 1753 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3LoadPersistedStateStart").detail("PersistenceDir", persistenceDir);
															#line 9670 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1757 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadPersistedObjects(persistenceDir);
															#line 1757 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9676 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 1;
															#line 1757 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9681 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadMockS3PersistedStateImplActorState();
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1772 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadMockS3PersistedStateImplActorState(); static_cast<LoadMockS3PersistedStateImplActor*>(this)->destroy(); return 0; }
															#line 9710 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadMockS3PersistedStateImplActorState();
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1768 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3LoadPersistedStateFailed").error(e);
															#line 1769 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 9725 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1760 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = loadPersistedMultipartUploads(persistenceDir);
															#line 1760 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9741 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 2;
															#line 1760 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9746 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1760 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = loadPersistedMultipartUploads(persistenceDir);
															#line 1760 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9757 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 2;
															#line 1760 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9762 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state > 0) static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 0;
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1762 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().persistenceLoaded = true;
															#line 1764 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3LoadPersistedStateComplete") .detail("ObjectsCount", getGlobalStorage().buckets.size()) .detail("MultipartUploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9848 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1762 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().persistenceLoaded = true;
															#line 1764 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3LoadPersistedStateComplete") .detail("ObjectsCount", getGlobalStorage().buckets.size()) .detail("MultipartUploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9859 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state > 0) static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 0;
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1746 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 9954 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadMockS3PersistedStateImpl()
															#line 1745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadMockS3PersistedStateImplActor final : public Actor<Void>, public ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >, public ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >, public FastAllocated<LoadMockS3PersistedStateImplActor>, public LoadMockS3PersistedStateImplActorState<LoadMockS3PersistedStateImplActor> {
															#line 9959 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadMockS3PersistedStateImplActor>::operator new;
	using FastAllocated<LoadMockS3PersistedStateImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17648339566407769600UL, 17092097485286911488UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >;
friend struct ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >;
															#line 1745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadMockS3PersistedStateImplActor() 
															#line 9977 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadMockS3PersistedStateImplActorState<LoadMockS3PersistedStateImplActor>(),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10516119037031120128UL, 15944437332465256704UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadMockS3PersistedStateImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadMockS3PersistedStateImpl(  ) {
															#line 1745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadMockS3PersistedStateImplActor());
															#line 10011 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1774 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Load persisted state from disk (called at server startup) - returns Future for use in ACTOR context
Future<Void> loadMockS3PersistedStateFuture() {
	if (getGlobalStorage().persistenceEnabled && !getGlobalStorage().persistenceLoaded) {
		return loadMockS3PersistedStateImpl();
	}
	return Void();
}

// Unit Tests for MockS3Server
															#line 10025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1784()
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1784Actor>
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1784ActorState {
															#line 10032 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1784ActorState(UnitTestParameters const& params) 
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10039 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1784", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1784ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1784", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1787 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1788 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1789 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1791 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1793 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1794 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1795 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1797 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1784Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1784ActorState(); static_cast<FlowTestCase1784Actor*>(this)->destroy(); return 0; }
															#line 10070 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1784Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1784ActorState();
			static_cast<FlowTestCase1784Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1784ActorState();
		static_cast<FlowTestCase1784Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10094 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1784()
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1784Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1784Actor>, public FlowTestCase1784ActorState<FlowTestCase1784Actor> {
															#line 10099 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1784Actor>::operator new;
	using FastAllocated<FlowTestCase1784Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16948787920329529344UL, 2454776816441209344UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1784Actor(UnitTestParameters const& params) 
															#line 10115 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1784ActorState<FlowTestCase1784Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1784", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11607728916305283328UL, 9842114729776368640UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1784Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1784");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1784", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1784( UnitTestParameters const& params ) {
															#line 1784 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1784Actor(params));
															#line 10147 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1784, "/MockS3Server/parseS3Request/ValidBucketParameter")

#line 1799 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10153 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1800()
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1800Actor>
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1800ActorState {
															#line 10160 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1800ActorState(UnitTestParameters const& params) 
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10167 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1800", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1800ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1800", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1802 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1803 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/?region=us-east-1";
															#line 1804 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1805 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 10188 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1808 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				server.parseS3Request(resource, bucket, object, queryParams);
															#line 1809 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				ASSERT(false);
															#line 10194 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1800ActorState();
		static_cast<FlowTestCase1800Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1814 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<FlowTestCase1800Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1800ActorState(); static_cast<FlowTestCase1800Actor*>(this)->destroy(); return 0; }
															#line 10223 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<FlowTestCase1800Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1800ActorState();
		static_cast<FlowTestCase1800Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1811 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(e.code() == error_code_backup_invalid_url);
															#line 10236 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10262 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1800()
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1800Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1800Actor>, public FlowTestCase1800ActorState<FlowTestCase1800Actor> {
															#line 10267 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1800Actor>::operator new;
	using FastAllocated<FlowTestCase1800Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4500284784983267840UL, 12389897531492087040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1800Actor(UnitTestParameters const& params) 
															#line 10283 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1800ActorState<FlowTestCase1800Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1800", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16485787300354454528UL, 14510205305306813952UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1800Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1800");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1800", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1800( UnitTestParameters const& params ) {
															#line 1800 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1800Actor(params));
															#line 10315 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1800, "/MockS3Server/parseS3Request/MissingBucketParameter")

#line 1816 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10321 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1817()
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1817Actor>
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1817ActorState {
															#line 10328 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1817ActorState(UnitTestParameters const& params) 
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10335 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1817", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1817ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1817", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1819 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1820 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/";
															#line 1821 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1822 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 10356 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1825 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				server.parseS3Request(resource, bucket, object, queryParams);
															#line 1826 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				ASSERT(false);
															#line 10362 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1817ActorState();
		static_cast<FlowTestCase1817Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1831 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<FlowTestCase1817Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1817ActorState(); static_cast<FlowTestCase1817Actor*>(this)->destroy(); return 0; }
															#line 10391 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<FlowTestCase1817Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1817ActorState();
		static_cast<FlowTestCase1817Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1828 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(e.code() == error_code_backup_invalid_url);
															#line 10404 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10430 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1817()
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1817Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1817Actor>, public FlowTestCase1817ActorState<FlowTestCase1817Actor> {
															#line 10435 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1817Actor>::operator new;
	using FastAllocated<FlowTestCase1817Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1817502507915809792UL, 3447025138014552320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1817Actor(UnitTestParameters const& params) 
															#line 10451 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1817ActorState<FlowTestCase1817Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1817", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3434592911225446912UL, 13671043530582969856UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1817Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1817");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1817", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1817( UnitTestParameters const& params ) {
															#line 1817 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1817Actor(params));
															#line 10483 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1817, "/MockS3Server/parseS3Request/EmptyQueryString")

#line 1833 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10489 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1834()
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1834Actor>
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1834ActorState {
															#line 10496 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1834ActorState(UnitTestParameters const& params) 
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10503 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1834", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1834ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1834", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1836 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1837 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket/testobject?region=us-east-1";
															#line 1838 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1839 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1841 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1843 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1844 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "testobject");
															#line 1845 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1834Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1834ActorState(); static_cast<FlowTestCase1834Actor*>(this)->destroy(); return 0; }
															#line 10534 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1834Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1834ActorState();
			static_cast<FlowTestCase1834Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1834ActorState();
		static_cast<FlowTestCase1834Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10558 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1834()
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1834Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1834Actor>, public FlowTestCase1834ActorState<FlowTestCase1834Actor> {
															#line 10563 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1834Actor>::operator new;
	using FastAllocated<FlowTestCase1834Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9666354170249957376UL, 3892055186670973184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1834Actor(UnitTestParameters const& params) 
															#line 10579 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1834ActorState<FlowTestCase1834Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1834", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5986881767939646464UL, 13438081518914690048UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1834Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1834");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1834", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1834( UnitTestParameters const& params ) {
															#line 1834 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1834Actor(params));
															#line 10611 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1834, "/MockS3Server/parseS3Request/BucketParameterOverride")

#line 1849 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1850()
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1850Actor>
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1850ActorState {
															#line 10624 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1850ActorState(UnitTestParameters const& params) 
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10631 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1850", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1850ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1850", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1852 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1853 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket/folder/subfolder/file.txt?region=us-east-1";
															#line 1854 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1855 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1857 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1859 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1860 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "folder/subfolder/file.txt");
															#line 1861 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1863 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1850Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1850ActorState(); static_cast<FlowTestCase1850Actor*>(this)->destroy(); return 0; }
															#line 10662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1850Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1850ActorState();
			static_cast<FlowTestCase1850Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1850ActorState();
		static_cast<FlowTestCase1850Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10686 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1850()
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1850Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1850Actor>, public FlowTestCase1850ActorState<FlowTestCase1850Actor> {
															#line 10691 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1850Actor>::operator new;
	using FastAllocated<FlowTestCase1850Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13481357895651890176UL, 13709750181894544896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1850Actor(UnitTestParameters const& params) 
															#line 10707 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1850ActorState<FlowTestCase1850Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1850", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11168038530737468160UL, 869792128700489472UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1850Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1850");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1850", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1850( UnitTestParameters const& params ) {
															#line 1850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1850Actor(params));
															#line 10739 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1850, "/MockS3Server/parseS3Request/ComplexPath")

#line 1865 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1866()
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1866Actor>
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1866ActorState {
															#line 10752 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1866ActorState(UnitTestParameters const& params) 
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10759 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1866", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1866ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1866", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1868 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1869 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1&param=value%3Dtest";
															#line 1870 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1871 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1873 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1875 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1876 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1877 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["param"] == "value=test");
															#line 1879 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1866Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1866ActorState(); static_cast<FlowTestCase1866Actor*>(this)->destroy(); return 0; }
															#line 10790 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1866Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1866ActorState();
			static_cast<FlowTestCase1866Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1866ActorState();
		static_cast<FlowTestCase1866Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10814 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1866()
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1866Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1866Actor>, public FlowTestCase1866ActorState<FlowTestCase1866Actor> {
															#line 10819 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1866Actor>::operator new;
	using FastAllocated<FlowTestCase1866Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16171511120181492736UL, 14607837861413978368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1866Actor(UnitTestParameters const& params) 
															#line 10835 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1866ActorState<FlowTestCase1866Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1866", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(946916491978723584UL, 9681847544094974464UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1866Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1866");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1866", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1866( UnitTestParameters const& params ) {
															#line 1866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1866Actor(params));
															#line 10867 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1866, "/MockS3Server/parseS3Request/URLEncodedParameters")

#line 1881 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10873 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1882()
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1882Actor>
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1882ActorState {
															#line 10880 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1882ActorState(UnitTestParameters const& params) 
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10887 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1882", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1882ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1882", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1884 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1885 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1886 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1887 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1889 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1891 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1892 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1893 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1895 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1882Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1882ActorState(); static_cast<FlowTestCase1882Actor*>(this)->destroy(); return 0; }
															#line 10918 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1882Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1882ActorState();
			static_cast<FlowTestCase1882Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1882ActorState();
		static_cast<FlowTestCase1882Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10942 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1882()
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1882Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1882Actor>, public FlowTestCase1882ActorState<FlowTestCase1882Actor> {
															#line 10947 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1882Actor>::operator new;
	using FastAllocated<FlowTestCase1882Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3859835689207676416UL, 7113178143331569920UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1882Actor(UnitTestParameters const& params) 
															#line 10963 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1882ActorState<FlowTestCase1882Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1882", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(289423043390919936UL, 15431653937774174976UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1882Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1882");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1882", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1882( UnitTestParameters const& params ) {
															#line 1882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1882Actor(params));
															#line 10995 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1882, "/MockS3Server/parseS3Request/EmptyPath")

#line 1897 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11001 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1898()
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1898Actor>
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1898ActorState {
															#line 11008 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1898ActorState(UnitTestParameters const& params) 
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11015 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1898", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1898ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1898", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1900 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1901 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1902 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1903 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1905 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1907 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1908 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1909 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1911 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1898Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1898ActorState(); static_cast<FlowTestCase1898Actor*>(this)->destroy(); return 0; }
															#line 11046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1898Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1898ActorState();
			static_cast<FlowTestCase1898Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1898ActorState();
		static_cast<FlowTestCase1898Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11070 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1898()
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1898Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1898Actor>, public FlowTestCase1898ActorState<FlowTestCase1898Actor> {
															#line 11075 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1898Actor>::operator new;
	using FastAllocated<FlowTestCase1898Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17327543475289893632UL, 14241212039495398144UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1898Actor(UnitTestParameters const& params) 
															#line 11091 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1898ActorState<FlowTestCase1898Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1898", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9448047514357098752UL, 15438263196725003264UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1898Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1898");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1898", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1898( UnitTestParameters const& params ) {
															#line 1898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1898Actor(params));
															#line 11123 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1898, "/MockS3Server/parseS3Request/OnlyBucketInPath")

#line 1913 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11129 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1914()
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1914Actor>
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1914ActorState {
															#line 11136 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1914ActorState(UnitTestParameters const& params) 
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11143 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1914", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1914ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1914", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1916 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1917 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1&version=1&encoding=utf8";
															#line 1918 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1919 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1921 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1923 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1924 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1925 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["version"] == "1");
															#line 1926 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["encoding"] == "utf8");
															#line 1927 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams.size() == 3);
															#line 1929 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1914Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1914ActorState(); static_cast<FlowTestCase1914Actor*>(this)->destroy(); return 0; }
															#line 11178 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1914Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1914ActorState();
			static_cast<FlowTestCase1914Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1914ActorState();
		static_cast<FlowTestCase1914Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11202 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1914()
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1914Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1914Actor>, public FlowTestCase1914ActorState<FlowTestCase1914Actor> {
															#line 11207 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1914Actor>::operator new;
	using FastAllocated<FlowTestCase1914Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16321451886057106432UL, 18258430631502980864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1914Actor(UnitTestParameters const& params) 
															#line 11223 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1914ActorState<FlowTestCase1914Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1914", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5105896747875338240UL, 15143610386852061184UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1914Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1914");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1914", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1914( UnitTestParameters const& params ) {
															#line 1914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1914Actor(params));
															#line 11255 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1914, "/MockS3Server/parseS3Request/MultipleParameters")

#line 1931 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11261 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1932()
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1932Actor>
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1932ActorState {
															#line 11268 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1932ActorState(UnitTestParameters const& params) 
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11275 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1932", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1932ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1932", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1934 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1935 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?flag&region=us-east-1";
															#line 1936 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1937 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1939 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1941 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1942 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["flag"] == "");
															#line 1943 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1945 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1932Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1932ActorState(); static_cast<FlowTestCase1932Actor*>(this)->destroy(); return 0; }
															#line 11306 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1932Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1932ActorState();
			static_cast<FlowTestCase1932Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1932ActorState();
		static_cast<FlowTestCase1932Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11330 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1932()
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1932Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1932Actor>, public FlowTestCase1932ActorState<FlowTestCase1932Actor> {
															#line 11335 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1932Actor>::operator new;
	using FastAllocated<FlowTestCase1932Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9288087460049433088UL, 4407493080307887360UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1932Actor(UnitTestParameters const& params) 
															#line 11351 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1932ActorState<FlowTestCase1932Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1932", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8050393363254702592UL, 12380887009022492928UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1932Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1932");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1932", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1932( UnitTestParameters const& params ) {
															#line 1932 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1932Actor(params));
															#line 11383 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1932, "/MockS3Server/parseS3Request/ParametersWithoutValues")

#line 1947 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11389 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1948()
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1948Actor>
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1948ActorState {
															#line 11396 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1948ActorState(UnitTestParameters const& params) 
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11403 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1948", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1948ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1948", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1949 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=0-99";
															#line 1950 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 1952 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 1954 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 1955 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 0);
															#line 1956 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 99);
															#line 1958 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1948Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1948ActorState(); static_cast<FlowTestCase1948Actor*>(this)->destroy(); return 0; }
															#line 11430 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1948Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1948ActorState();
			static_cast<FlowTestCase1948Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1948ActorState();
		static_cast<FlowTestCase1948Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11454 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1948()
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1948Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1948Actor>, public FlowTestCase1948ActorState<FlowTestCase1948Actor> {
															#line 11459 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1948Actor>::operator new;
	using FastAllocated<FlowTestCase1948Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4014042282458235136UL, 9783010135690385408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1948Actor(UnitTestParameters const& params) 
															#line 11475 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1948ActorState<FlowTestCase1948Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1948", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16930020380649221120UL, 17401853989703125760UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1948Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1948");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1948", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1948( UnitTestParameters const& params ) {
															#line 1948 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1948Actor(params));
															#line 11507 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1948, "/MockS3Server/RangeHeader/SimpleByteRange")

#line 1960 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11513 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1961()
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1961Actor>
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1961ActorState {
															#line 11520 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1961ActorState(UnitTestParameters const& params) 
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11527 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1961", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1961ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1961", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1962 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=100-199";
															#line 1963 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 1965 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 1967 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 1968 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 100);
															#line 1969 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 199);
															#line 1971 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1961Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1961ActorState(); static_cast<FlowTestCase1961Actor*>(this)->destroy(); return 0; }
															#line 11554 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1961Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1961ActorState();
			static_cast<FlowTestCase1961Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1961ActorState();
		static_cast<FlowTestCase1961Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11578 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1961()
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1961Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1961Actor>, public FlowTestCase1961ActorState<FlowTestCase1961Actor> {
															#line 11583 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1961Actor>::operator new;
	using FastAllocated<FlowTestCase1961Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(793975194682047488UL, 8938071902769137408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1961Actor(UnitTestParameters const& params) 
															#line 11599 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1961ActorState<FlowTestCase1961Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1961", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12216455348170519296UL, 2396457093511273216UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1961Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1961");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1961", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1961( UnitTestParameters const& params ) {
															#line 1961 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1961Actor(params));
															#line 11631 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1961, "/MockS3Server/RangeHeader/MiddleRange")

#line 1973 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11637 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1974()
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1974Actor>
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1974ActorState {
															#line 11644 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1974ActorState(UnitTestParameters const& params) 
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11651 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1974", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1974ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1974", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1975 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=1000000-1999999";
															#line 1976 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 1978 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 1980 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 1981 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 1000000);
															#line 1982 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 1999999);
															#line 1984 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1974Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1974ActorState(); static_cast<FlowTestCase1974Actor*>(this)->destroy(); return 0; }
															#line 11678 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1974Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1974ActorState();
			static_cast<FlowTestCase1974Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1974ActorState();
		static_cast<FlowTestCase1974Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11702 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1974()
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1974Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1974Actor>, public FlowTestCase1974ActorState<FlowTestCase1974Actor> {
															#line 11707 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1974Actor>::operator new;
	using FastAllocated<FlowTestCase1974Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6973925449035563008UL, 16433379421658942208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1974Actor(UnitTestParameters const& params) 
															#line 11723 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1974ActorState<FlowTestCase1974Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18100637140583322368UL, 7823974797634655744UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1974");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1974( UnitTestParameters const& params ) {
															#line 1974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1974Actor(params));
															#line 11755 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1974, "/MockS3Server/RangeHeader/LargeOffsets")

#line 1986 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11761 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1987()
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1987Actor>
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1987ActorState {
															#line 11768 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1987ActorState(UnitTestParameters const& params) 
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11775 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1987", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1987ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1987", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1988 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "invalid-range";
															#line 1989 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 1991 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 1993 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 1995 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1987Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1987ActorState(); static_cast<FlowTestCase1987Actor*>(this)->destroy(); return 0; }
															#line 11798 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1987Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1987ActorState();
			static_cast<FlowTestCase1987Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1987ActorState();
		static_cast<FlowTestCase1987Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11822 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1987()
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1987Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1987Actor>, public FlowTestCase1987ActorState<FlowTestCase1987Actor> {
															#line 11827 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1987Actor>::operator new;
	using FastAllocated<FlowTestCase1987Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13686795967867269376UL, 3559276309531765504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1987Actor(UnitTestParameters const& params) 
															#line 11843 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1987ActorState<FlowTestCase1987Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1987", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15748006011262807808UL, 5841495732011407872UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1987Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1987");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1987", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1987( UnitTestParameters const& params ) {
															#line 1987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1987Actor(params));
															#line 11875 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1987, "/MockS3Server/RangeHeader/InvalidFormat")

#line 1997 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11881 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1998()
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1998Actor>
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1998ActorState {
															#line 11888 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1998ActorState(UnitTestParameters const& params) 
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11895 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1998", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1998ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1998", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1999 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "0-99";
															#line 2000 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2002 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2006 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1998Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1998ActorState(); static_cast<FlowTestCase1998Actor*>(this)->destroy(); return 0; }
															#line 11918 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1998Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1998ActorState();
			static_cast<FlowTestCase1998Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1998ActorState();
		static_cast<FlowTestCase1998Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11942 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1998()
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1998Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1998Actor>, public FlowTestCase1998ActorState<FlowTestCase1998Actor> {
															#line 11947 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1998Actor>::operator new;
	using FastAllocated<FlowTestCase1998Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10329938919211382784UL, 18086566910545403392UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1998Actor(UnitTestParameters const& params) 
															#line 11963 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1998ActorState<FlowTestCase1998Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1998", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11811051189418804480UL, 2669518114347836928UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1998Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1998");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1998", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1998( UnitTestParameters const& params ) {
															#line 1998 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1998Actor(params));
															#line 11995 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1998, "/MockS3Server/RangeHeader/MissingBytesPrefix")

#line 2008 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12001 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2009()
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2009Actor>
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2009ActorState {
															#line 12008 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2009ActorState(UnitTestParameters const& params) 
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12015 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2009", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2009ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2009", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=0";
															#line 2011 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2013 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2015 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2009Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2009ActorState(); static_cast<FlowTestCase2009Actor*>(this)->destroy(); return 0; }
															#line 12038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2009Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2009ActorState();
			static_cast<FlowTestCase2009Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2009ActorState();
		static_cast<FlowTestCase2009Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12062 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2009()
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2009Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2009Actor>, public FlowTestCase2009ActorState<FlowTestCase2009Actor> {
															#line 12067 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2009Actor>::operator new;
	using FastAllocated<FlowTestCase2009Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2071256877798207488UL, 5120142801278257664UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2009Actor(UnitTestParameters const& params) 
															#line 12083 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2009ActorState<FlowTestCase2009Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2009", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13576115846491637248UL, 1211027230964504576UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2009Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2009");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2009", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2009( UnitTestParameters const& params ) {
															#line 2009 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2009Actor(params));
															#line 12115 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2009, "/MockS3Server/RangeHeader/MissingDash")

#line 2019 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12121 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2020()
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2020Actor>
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2020ActorState {
															#line 12128 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2020ActorState(UnitTestParameters const& params) 
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12135 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2020", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2020ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2020", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2021 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "";
															#line 2022 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2024 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2026 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2028 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2020Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2020ActorState(); static_cast<FlowTestCase2020Actor*>(this)->destroy(); return 0; }
															#line 12158 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2020Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2020ActorState();
			static_cast<FlowTestCase2020Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2020ActorState();
		static_cast<FlowTestCase2020Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12182 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2020()
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2020Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2020Actor>, public FlowTestCase2020ActorState<FlowTestCase2020Actor> {
															#line 12187 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2020Actor>::operator new;
	using FastAllocated<FlowTestCase2020Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3024627740721431808UL, 17563837830889683200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2020Actor(UnitTestParameters const& params) 
															#line 12203 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2020ActorState<FlowTestCase2020Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2020", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17337883412881757440UL, 16813064531799584000UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2020Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2020");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2020", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2020( UnitTestParameters const& params ) {
															#line 2020 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2020Actor(params));
															#line 12235 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2020, "/MockS3Server/RangeHeader/EmptyString")

#line 2030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12241 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2031()
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2031Actor>
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2031ActorState {
															#line 12248 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2031ActorState(UnitTestParameters const& params) 
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12255 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2031", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2031ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2031", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2032 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=-100-200";
															#line 2033 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2035 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2040 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2031Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2031ActorState(); static_cast<FlowTestCase2031Actor*>(this)->destroy(); return 0; }
															#line 12278 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2031Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2031ActorState();
			static_cast<FlowTestCase2031Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2031ActorState();
		static_cast<FlowTestCase2031Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12302 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2031()
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2031Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2031Actor>, public FlowTestCase2031ActorState<FlowTestCase2031Actor> {
															#line 12307 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2031Actor>::operator new;
	using FastAllocated<FlowTestCase2031Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2745356680241092864UL, 9978123522935076864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2031Actor(UnitTestParameters const& params) 
															#line 12323 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2031ActorState<FlowTestCase2031Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2031", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9156885137961901056UL, 3636434032296420608UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2031Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2031");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2031", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2031( UnitTestParameters const& params ) {
															#line 2031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2031Actor(params));
															#line 12355 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2031, "/MockS3Server/RangeHeader/NegativeStart")

#line 2042 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12361 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2043()
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2043Actor>
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2043ActorState {
															#line 12368 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2043ActorState(UnitTestParameters const& params) 
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12375 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2043", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2043ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2043", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2044 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=200-100";
															#line 2045 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2047 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2050 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2051 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 200);
															#line 2052 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 100);
															#line 2054 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2043Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2043ActorState(); static_cast<FlowTestCase2043Actor*>(this)->destroy(); return 0; }
															#line 12402 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2043Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2043ActorState();
			static_cast<FlowTestCase2043Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2043ActorState();
		static_cast<FlowTestCase2043Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12426 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2043()
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2043Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2043Actor>, public FlowTestCase2043ActorState<FlowTestCase2043Actor> {
															#line 12431 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2043Actor>::operator new;
	using FastAllocated<FlowTestCase2043Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4343434244677155840UL, 2170638355787690752UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2043Actor(UnitTestParameters const& params) 
															#line 12447 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2043ActorState<FlowTestCase2043Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2043", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(875482208392099072UL, 4979150467743823104UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2043Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2043");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2043", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2043( UnitTestParameters const& params ) {
															#line 2043 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2043Actor(params));
															#line 12479 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2043, "/MockS3Server/RangeHeader/StartGreaterThanEnd")

#line 2056 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Real HTTP Server Implementation for ctests
															#line 12486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via startMockS3ServerReal_impl()
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class StartMockS3ServerReal_implActor>
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerReal_implActorState {
															#line 12493 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerReal_implActorState(NetworkAddress const& listenAddress,std::string const& persistenceDir) 
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : listenAddress(listenAddress),
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(persistenceDir)
															#line 12502 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this));

	}
	~StartMockS3ServerReal_implActorState() 
	{
		fdb_probe_actor_destroy("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2059 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerRealStarting").detail("ListenAddress", listenAddress.toString());
															#line 2062 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled)
															#line 12519 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 2064 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (persistenceDir.empty())
															#line 12523 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 2065 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					persistenceDir = "simfdb/mocks3";
															#line 12527 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				}
															#line 2067 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				enableMockS3Persistence(persistenceDir);
															#line 2068 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerRealPersistenceEnabled") .detail("ListenAddress", listenAddress.toString()) .detail("PersistenceDir", persistenceDir);
															#line 2073 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadMockS3PersistedStateFuture();
															#line 2073 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12537 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 1;
															#line 2073 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*>(static_cast<StartMockS3ServerReal_implActor*>(this)));
															#line 12542 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2076 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		server = makeReference<HTTP::SimServerContext>();
															#line 2077 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		server->registerNewServer(listenAddress, makeReference<MockS3RequestHandler>());
															#line 2079 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRealStarted") .detail("ListenAddress", listenAddress.toString()) .detail("ServerPtr", format("%p", server.getPtr()));
															#line 2084 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = Never();
															#line 2084 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12578 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 2;
															#line 2084 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*>(static_cast<StartMockS3ServerReal_implActor*>(this)));
															#line 12583 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerReal_implActor*>(this)->ActorCallback< StartMockS3ServerReal_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2085 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerReal_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerReal_implActorState(); static_cast<StartMockS3ServerReal_implActor*>(this)->destroy(); return 0; }
															#line 12679 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerReal_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2085 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerReal_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerReal_implActorState(); static_cast<StartMockS3ServerReal_implActor*>(this)->destroy(); return 0; }
															#line 12691 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerReal_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerReal_implActor*>(this)->ActorCallback< StartMockS3ServerReal_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	NetworkAddress listenAddress;
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 2076 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::SimServerContext> server;
															#line 12780 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via startMockS3ServerReal_impl()
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerReal_implActor final : public Actor<Void>, public ActorCallback< StartMockS3ServerReal_implActor, 0, Void >, public ActorCallback< StartMockS3ServerReal_implActor, 1, Void >, public FastAllocated<StartMockS3ServerReal_implActor>, public StartMockS3ServerReal_implActorState<StartMockS3ServerReal_implActor> {
															#line 12785 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<StartMockS3ServerReal_implActor>::operator new;
	using FastAllocated<StartMockS3ServerReal_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11075511836881354240UL, 5254431837679208704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartMockS3ServerReal_implActor, 0, Void >;
friend struct ActorCallback< StartMockS3ServerReal_implActor, 1, Void >;
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerReal_implActor(NetworkAddress const& listenAddress,std::string const& persistenceDir) 
															#line 12803 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   StartMockS3ServerReal_implActorState<StartMockS3ServerReal_implActor>(listenAddress, persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15792334911365886208UL, 3052106341508935936UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startMockS3ServerReal_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> startMockS3ServerReal_impl( NetworkAddress const& listenAddress, std::string const& persistenceDir ) {
															#line 2058 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new StartMockS3ServerReal_implActor(listenAddress, persistenceDir));
															#line 12837 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 2087 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

Future<Void> startMockS3ServerReal(const NetworkAddress& listenAddress, const std::string& persistenceDir) {
	return startMockS3ServerReal_impl(listenAddress, persistenceDir);
}

// Wrapper for registerMockS3Server (calls the ACTOR implementation)
Future<Void> registerMockS3Server(std::string ip, std::string port) {
	return registerMockS3Server_impl(ip, port);
}
