#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
/*
 * KeyValueStoreMemory.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/BlobCipher.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/Notified.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/ServerDBInfo.actor.h"
#include "fdbserver/DeltaTree.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbserver/IDiskQueue.h"
#include "fdbserver/IKeyValueContainer.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/RadixTree.h"
#include "fdbserver/TransactionStoreMutationTracking.h"
#include "flow/ActorCollection.h"
#include "flow/EncryptUtils.h"
#include "flow/Knobs.h"
#include "flow/actorcompiler.h" // This must be the last #include.

#define OP_DISK_OVERHEAD (sizeof(OpHeader) + 1)
#define ENCRYPTION_ENABLED_BIT 31
static_assert(sizeof(uint32_t) == 4);

template <typename Container>
class KeyValueStoreMemory final : public IKeyValueStore, NonCopyable {
public:
	KeyValueStoreMemory(IDiskQueue* log,
	                    Reference<AsyncVar<ServerDBInfo> const> db,
	                    UID id,
	                    int64_t memoryLimit,
	                    KeyValueStoreType storeType,
	                    bool disableSnapshot,
	                    bool replaceContent,
	                    bool exactRecovery,
	                    bool enableEncryption);

	bool getReplaceContent() const override { return replaceContent; }
	// IClosable
	Future<Void> getError() const override { return log->getError(); }
	Future<Void> onClosed() const override { return log->onClosed(); }
	void dispose() override {
		recovering.cancel();
		log->dispose();
		if (reserved_buffer != nullptr) {
			delete[] reserved_buffer;
			reserved_buffer = nullptr;
		}
		delete this;
	}
	void close() override {
		recovering.cancel();
		log->close();
		if (reserved_buffer != nullptr) {
			delete[] reserved_buffer;
			reserved_buffer = nullptr;
		}
		delete this;
	}

	// IKeyValueStore
	KeyValueStoreType getType() const override { return type; }

	std::tuple<size_t, size_t, size_t> getSize() const override { return data.size(); }

	int64_t getAvailableSize() const {
		int64_t residentSize = data.sumTo(data.end()) + queue.totalSize() + // doesn't account for overhead in queue
		                       transactionSize;

		return memoryLimit - residentSize;
	}

	StorageBytes getStorageBytes() const override {
		StorageBytes diskQueueBytes = log->getStorageBytes();

		// Try to bound how many in-memory bytes we might need to write to disk if we commit() now
		int64_t uncommittedBytes = queue.totalSize() + transactionSize;

		// Check that we have enough space in memory and on disk
		int64_t freeSize = std::min(getAvailableSize(), diskQueueBytes.free / 4 - uncommittedBytes);
		int64_t availableSize = std::min(getAvailableSize(), diskQueueBytes.available / 4 - uncommittedBytes);
		int64_t totalSize = std::min(memoryLimit, diskQueueBytes.total / 4 - uncommittedBytes);

		return StorageBytes(std::max((int64_t)0, freeSize),
		                    std::max((int64_t)0, totalSize),
		                    diskQueueBytes.used,
		                    std::max((int64_t)0, availableSize));
	}

	void semiCommit() {
		transactionSize += queue.totalSize();
		if (transactionSize > 0.5 * committedDataSize) {
			transactionIsLarge = true;
			TraceEvent("KVSMemSwitchingToLargeTransactionMode", id)
			    .detail("TransactionSize", transactionSize)
			    .detail("DataSize", committedDataSize);
			CODE_PROBE(true, "KeyValueStoreMemory switching to large transaction mode");
			CODE_PROBE(committedDataSize > 1e3,
			           "KeyValueStoreMemory switching to large transaction mode with committed data");
		}

		int64_t bytesWritten = commit_queue(queue, true);
		committedWriteBytes += bytesWritten;
	}

	void set(KeyValueRef keyValue, const Arena* arena) override {
		// A commit that occurs with no available space returns Never, so we can throw out all modifications
		if (getAvailableSize() <= 0)
			return;

		if (transactionIsLarge) {
			data.insert(keyValue.key, keyValue.value);
		} else {
			queue.set(keyValue, arena);
			if (recovering.isReady() && !disableSnapshot) {
				semiCommit();
			}
		}
	}

	void clear(KeyRangeRef range, const Arena* arena) override {
		// A commit that occurs with no available space returns Never, so we can throw out all modifications
		if (getAvailableSize() <= 0)
			return;

		if (transactionIsLarge) {
			data.erase(data.lower_bound(range.begin), data.lower_bound(range.end));
		} else {
			queue.clear(range, arena);
			if (recovering.isReady() && !disableSnapshot) {
				semiCommit();
			}
		}
	}

	Future<Void> commit(bool sequential) override {
		if (getAvailableSize() <= 0) {
			TraceEvent(SevError, "KeyValueStoreMemory_OutOfSpace", id).log();
			return Never();
		}

		if (recovering.isError())
			throw recovering.getError();
		if (!recovering.isReady())
			return waitAndCommit(this, sequential);

		if (!disableSnapshot && replaceContent && !firstCommitWithSnapshot) {
			transactionSize += SERVER_KNOBS->REPLACE_CONTENTS_BYTES;
			committedWriteBytes += SERVER_KNOBS->REPLACE_CONTENTS_BYTES;
			semiCommit();
		}

		if (transactionIsLarge) {
			fullSnapshot(data);
			resetSnapshot = true;
			committedWriteBytes = notifiedCommittedWriteBytes.get();
			overheadWriteBytes = 0;

			if (disableSnapshot) {
				return Void();
			}
			log_op(OpCommit, StringRef(), StringRef());
		} else {
			int64_t bytesWritten = commit_queue(queue, !disableSnapshot, sequential);

			if (disableSnapshot) {
				return Void();
			}

			if (bytesWritten > 0 || committedWriteBytes > notifiedCommittedWriteBytes.get()) {
				committedWriteBytes += bytesWritten + overheadWriteBytes +
				                       OP_DISK_OVERHEAD; // OP_DISK_OVERHEAD is for the following log_op(OpCommit)
				notifiedCommittedWriteBytes.set(committedWriteBytes); // This set will cause snapshot items to be
				                                                      // written, so it must happen before the OpCommit
				log_op(OpCommit, StringRef(), StringRef());
				overheadWriteBytes = log->getCommitOverhead();
			}
		}

		auto c = log->commit();

		committedDataSize = data.sumTo(data.end());
		transactionSize = 0;
		transactionIsLarge = false;
		firstCommitWithSnapshot = false;

		addActor.send(commitAndUpdateVersions(this, c, previousSnapshotEnd));
		return c;
	}

	Future<Optional<Value>> readValue(KeyRef key, Optional<ReadOptions> options) override {
		if (recovering.isError())
			throw recovering.getError();
		if (!recovering.isReady())
			return waitAndReadValue(this, key, options);

		auto it = data.find(key);
		if (it == data.end())
			return Optional<Value>();
		return Optional<Value>(it.getValue());
	}

	Future<Optional<Value>> readValuePrefix(KeyRef key, int maxLength, Optional<ReadOptions> options) override {
		if (recovering.isError())
			throw recovering.getError();
		if (!recovering.isReady())
			return waitAndReadValuePrefix(this, key, maxLength, options);

		auto it = data.find(key);
		if (it == data.end())
			return Optional<Value>();
		auto val = it.getValue();
		if (maxLength < val.size()) {
			return Optional<Value>(val.substr(0, maxLength));
		} else {
			return Optional<Value>(val);
		}
	}

	// If rowLimit>=0, reads first rows sorted ascending, otherwise reads last rows sorted descending
	// The total size of the returned value (less the last entry) will be less than byteLimit
	Future<RangeResult> readRange(KeyRangeRef keys,
	                              int rowLimit,
	                              int byteLimit,
	                              Optional<ReadOptions> options) override {
		if (recovering.isError())
			throw recovering.getError();
		if (!recovering.isReady())
			return waitAndReadRange(this, keys, rowLimit, byteLimit, options);

		RangeResult result;
		if (rowLimit == 0) {
			return result;
		}

		if (rowLimit > 0) {
			auto it = data.lower_bound(keys.begin);
			while (it != data.end() && rowLimit && byteLimit > 0) {
				StringRef tempKey = it.getKey(reserved_buffer);
				if (tempKey >= keys.end)
					break;

				byteLimit -= sizeof(KeyValueRef) + tempKey.size() + it.getValue().size();
				result.push_back_deep(result.arena(), KeyValueRef(tempKey, it.getValue()));
				++it;
				--rowLimit;
			}
		} else {
			rowLimit = -rowLimit;
			auto it = data.previous(data.lower_bound(keys.end));
			while (it != data.end() && rowLimit && byteLimit > 0) {
				StringRef tempKey = it.getKey(reserved_buffer);
				if (tempKey < keys.begin)
					break;

				byteLimit -= sizeof(KeyValueRef) + tempKey.size() + it.getValue().size();
				result.push_back_deep(result.arena(), KeyValueRef(tempKey, it.getValue()));
				it = data.previous(it);
				--rowLimit;
			}
		}

		result.more = rowLimit == 0 || byteLimit <= 0;
		return result;
	}

	void resyncLog() override {
		ASSERT(recovering.isReady());
		resetSnapshot = true;
		log_op(OpSnapshotAbort, StringRef(), StringRef());
	}

	void enableSnapshot() override { disableSnapshot = false; }

	int uncommittedBytes() { return queue.totalSize(); }

	// KeyValueStoreMemory does not support encryption-at-rest in general, despite it supports encryption
	// when being used as TxnStateStore backend.
	Future<EncryptionAtRestMode> encryptionMode() override {
		return EncryptionAtRestMode(EncryptionAtRestMode::DISABLED);
	}

private:
	enum OpType {
		OpSet,
		OpClear,
		OpClearToEnd,
		OpSnapshotItem,
		OpSnapshotEnd,
		OpSnapshotAbort, // terminate an in progress snapshot in order to start a full snapshot
		OpCommit, // only in log, not in queue
		OpRollback, // only in log, not in queue
		OpSnapshotItemDelta,
		OpEncrypted_Deprecated // deprecated since we now store the encryption status in the first bit of the opType
	};

	struct OpRef {
		OpType op;
		StringRef p1, p2;
		OpRef() {}
		OpRef(Arena& a, OpRef const& o) : op(o.op), p1(a, o.p1), p2(a, o.p2) {}
		size_t expectedSize() const { return p1.expectedSize() + p2.expectedSize(); }
	};
	struct OpHeader {
		uint32_t op;
		int len1, len2;
	};

	struct OpQueue {
		OpQueue() : numBytes(0) {}

		int totalSize() const { return numBytes; }

		void clear() {
			numBytes = 0;
			operations = Standalone<VectorRef<OpRef>>();
			arenas.clear();
		}

		void rollback() { clear(); }

		void set(KeyValueRef keyValue, const Arena* arena = nullptr) {
			queue_op(OpSet, keyValue.key, keyValue.value, arena);
		}

		void clear(KeyRangeRef range, const Arena* arena = nullptr) {
			queue_op(OpClear, range.begin, range.end, arena);
		}

		void clear_to_end(StringRef fromKey, const Arena* arena = nullptr) {
			queue_op(OpClearToEnd, fromKey, StringRef(), arena);
		}

		void queue_op(OpType op, StringRef p1, StringRef p2, const Arena* arena) {
			numBytes += p1.size() + p2.size() + sizeof(OpHeader) + sizeof(OpRef);

			OpRef r;
			r.op = op;
			r.p1 = p1;
			r.p2 = p2;
			if (arena == nullptr) {
				operations.push_back_deep(operations.arena(), r);
			} else {
				operations.push_back(operations.arena(), r);
				arenas.push_back(*arena);
			}
		}

		const OpRef* begin() { return operations.begin(); }

		const OpRef* end() { return operations.end(); }

	private:
		Standalone<VectorRef<OpRef>> operations;
		uint64_t numBytes;
		std::vector<Arena> arenas;
	};
	KeyValueStoreType type;
	UID id;

	Container data;
	// reserved buffer for snapshot/fullsnapshot
	uint8_t* reserved_buffer;

	OpQueue queue; // mutations not yet commit()ted
	IDiskQueue* log;
	Reference<AsyncVar<ServerDBInfo> const> db;
	Future<Void> recovering, snapshotting;
	int64_t committedWriteBytes;
	int64_t overheadWriteBytes;
	NotifiedVersion notifiedCommittedWriteBytes;
	Key recoveredSnapshotKey; // After recovery, the next key in the currently uncompleted snapshot
	IDiskQueue::location
	    currentSnapshotEnd; // The end of the most recently completed snapshot (this snapshot cannot be discarded)
	IDiskQueue::location previousSnapshotEnd; // The end of the second most recently completed snapshot (on commit, this
	                                          // snapshot can be discarded)
	PromiseStream<Future<Void>> addActor;
	Future<Void> commitActors;

	int64_t committedDataSize;
	int64_t transactionSize;
	bool transactionIsLarge;

	bool resetSnapshot; // Set to true after a fullSnapshot is performed.  This causes the regular snapshot mechanism to
	                    // restart
	bool disableSnapshot;
	bool replaceContent;
	bool firstCommitWithSnapshot;
	int snapshotCount;

	int64_t memoryLimit; // The upper limit on the memory used by the store (excluding, possibly, some clear operations)
	std::vector<std::pair<KeyValueMapPair, uint64_t>> dataSets;

	bool enableEncryption;
	TextAndHeaderCipherKeys cipherKeys;
	Future<Void> refreshCipherKeysActor;

	int64_t commit_queue(OpQueue& ops, bool log, bool sequential = false) {
		int64_t total = 0, count = 0;
		IDiskQueue::location log_location = 0;

		for (auto o = ops.begin(); o != ops.end(); ++o) {
			++count;
			total += o->p1.size() + o->p2.size() + OP_DISK_OVERHEAD;
			if (o->op == OpSet) {
				if (sequential) {
					KeyValueMapPair pair(o->p1, o->p2);
					dataSets.emplace_back(pair, pair.arena.getSize() + data.getElementBytes());
				} else {
					data.insert(o->p1, o->p2);
				}
			} else if (o->op == OpClear) {
				if (sequential) {
					data.insert(dataSets);
					dataSets.clear();
				}
				data.erase(data.lower_bound(o->p1), data.lower_bound(o->p2));
			} else if (o->op == OpClearToEnd) {
				if (sequential) {
					data.insert(dataSets);
					dataSets.clear();
				}
				data.erase(data.lower_bound(o->p1), data.end());
			} else
				ASSERT(false);
			if (log)
				log_location = log_op(o->op, o->p1, o->p2);
		}
		if (sequential) {
			data.insert(dataSets);
			dataSets.clear();
		}

		bool ok = count < 1e6;
		if (!ok) {
			TraceEvent(/*ok ? SevInfo : */ SevWarnAlways, "KVSMemCommitQueue", id)
			    .detail("Bytes", total)
			    .detail("Log", log)
			    .detail("Ops", count)
			    .detail("LastLoggedLocation", log_location)
			    .detail("Details", count);
		}

		ops.clear();
		return total;
	}

	static bool isOpEncrypted(OpHeader* header) { return header->op >> ENCRYPTION_ENABLED_BIT == 1; }

	static void setEncryptFlag(OpHeader* header, bool set) {
		if (set) {
			header->op |= (1UL << ENCRYPTION_ENABLED_BIT);
		} else {
			header->op &= ~(1UL << ENCRYPTION_ENABLED_BIT);
		}
	}

	// NOTE: The first bit of opType indicates whether the entry is encrypted or not. This is fine for backwards
	// compatibility since the first bit was never used previously
	//
	// Unencrypted data format:
	// +-------------+-------------+-------------+--------+--------+-----------+
	// | opType      | len1        | len2        | param2 | param2 |   \x01    |
	// | sizeof(int) | sizeof(int) | sizeof(int) | len1   | len2   |  1 byte   |
	// +-------------+-------------+-------------+--------+--------+-----------+
	//
	// Encrypted data format:
	// +-------------+-------+---------+------------+-----------------------------+--------+--------+------------+
	// |   opType    |len1   | len2    | headerSize | BlobCipherEncryptHeader     | param1 | param2 |    \x01    |
	// |  s(uint32)  | s(int)| s(int)  | s(uint16)  |  s(BlobCipherEncryptHeader) | len1   | len2   |   1 byte   |
	// +-------------+-------+---------+------------+-----------------------------+--------+--------+------------+
	// |                                plaintext                                 |    encrypted    |            |
	// +--------------------------------------------------------------------------+-----------------+------------+
	//
	IDiskQueue::location log_op(OpType op, StringRef v1, StringRef v2) {
		// Metadata op types to be excluded from encryption.
		static std::unordered_set<OpType> metaOps = { OpSnapshotEnd, OpSnapshotAbort, OpCommit, OpRollback };
		uint32_t opType = (uint32_t)op;
		// Make sure the first bit of the optype is empty
		ASSERT(opType >> ENCRYPTION_ENABLED_BIT == 0);
		if (!enableEncryption || metaOps.contains(op)) {
			OpHeader h = { opType, v1.size(), v2.size() };
			log->push(StringRef((const uint8_t*)&h, sizeof(h)));
			log->push(v1);
			log->push(v2);
		} else {
			OpHeader h = { opType, v1.size(), v2.size() };
			// Set the first bit of the header to 1 to indicate that the log entry is encrypted
			setEncryptFlag(&h, true);
			log->push(StringRef((const uint8_t*)&h, sizeof(h)));

			uint8_t* plaintext = new uint8_t[v1.size() + v2.size()];
			if (v1.size()) {
				memcpy(plaintext, v1.begin(), v1.size());
			}
			if (v2.size()) {
				memcpy(plaintext + v1.size(), v2.begin(), v2.size());
			}

			ASSERT(cipherKeys.cipherTextKey.isValid());
			ASSERT(cipherKeys.cipherHeaderKey.isValid());
			EncryptBlobCipherAes265Ctr cipher(
			    cipherKeys.cipherTextKey,
			    cipherKeys.cipherHeaderKey,
			    getEncryptAuthTokenMode(EncryptAuthTokenMode::ENCRYPT_HEADER_AUTH_TOKEN_MODE_SINGLE),
			    BlobCipherMetrics::KV_MEMORY);
			uint16_t encryptHeaderSize;
			// TODO: If possible we want to avoid memcpy to the disk log by using the same arena used by IDiskQueue
			Arena arena;
			BlobCipherEncryptHeaderRef headerRef;
			StringRef cipherText = cipher.encrypt(plaintext, v1.size() + v2.size(), &headerRef, arena);
			Standalone<StringRef> headerRefStr = BlobCipherEncryptHeaderRef::toStringRef(headerRef);
			encryptHeaderSize = headerRefStr.size();
			ASSERT(encryptHeaderSize > 0);
			log->push(StringRef((const uint8_t*)&encryptHeaderSize, sizeof(encryptHeaderSize)));
			log->push(headerRefStr);
			log->push(cipherText);
		}
		IDiskQueue::location loc = log->push("\x01"_sr); // Changes here should be reflected in OP_DISK_OVERHEAD
		DEBUG_TRANSACTION_STATE_STORE("LogOp", v1, id, loc);
		return loc;
	}

	// In case the op data is not encrypted, simply read the operands and the zero fill flag.
	// Otherwise, decrypt the op type and data.
																#line 546 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via readOpData()
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class ReadOpDataActor>
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class ReadOpDataActorState {
															#line 552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	ReadOpDataActorState(KeyValueStoreMemory* const& self,OpHeader const& h,bool* const& isZeroFilled,int* const& zeroFillSize,bool const& encryptedOp) 
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   h(h),
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   isZeroFilled(isZeroFilled),
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   zeroFillSize(zeroFillSize),
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   encryptedOp(encryptedOp)
															#line 567 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("readOpData", reinterpret_cast<unsigned long>(this));

	}
	~ReadOpDataActorState() 
	{
		fdb_probe_actor_destroy("readOpData", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 549 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			ASSERT(!isOpEncrypted(&h));
															#line 551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			static std::unordered_set<OpType> metaOps = { OpSnapshotEnd, OpSnapshotAbort, OpCommit, OpRollback };
															#line 552 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (!metaOps.contains((OpType)h.op))
															#line 586 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 554 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				ASSERT_EQ(encryptedOp, self->enableEncryption);
															#line 590 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			}
															#line 557 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			encryptHeaderSize = 0;
															#line 558 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (encryptedOp)
															#line 596 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 559 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				StrictFuture<Standalone<StringRef>> __when_expr_0 = self->log->readNext(sizeof(encryptHeaderSize));
															#line 559 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (static_cast<ReadOpDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 602 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ReadOpDataActor*>(this)->actor_wait_state = 1;
															#line 559 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*>(static_cast<ReadOpDataActor*>(this)));
															#line 607 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadOpDataActorState();
		static_cast<ReadOpDataActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 573 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		remainingBytes = h.len1 + h.len2 + 1;
															#line 574 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (encryptedOp)
															#line 637 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 576 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			remainingBytes += encryptHeaderSize;
															#line 641 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 578 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Standalone<StringRef>> __when_expr_1 = self->log->readNext(remainingBytes);
															#line 578 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<ReadOpDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 647 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadOpDataActor*>(this)->actor_wait_state = 2;
															#line 578 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadOpDataActor, 1, Standalone<StringRef> >*>(static_cast<ReadOpDataActor*>(this)));
															#line 652 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 560 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		ASSERT(headerSizeStr.size() <= sizeof(encryptHeaderSize));
															#line 562 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		memset(&encryptHeaderSize, 0, sizeof(encryptHeaderSize));
															#line 563 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		memcpy(&encryptHeaderSize, headerSizeStr.begin(), headerSizeStr.size());
															#line 564 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (headerSizeStr.size() < sizeof(encryptHeaderSize))
															#line 667 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 565 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			CODE_PROBE(true, "zero fill partial encryption header size", probe::decoration::rare);
															#line 566 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			*zeroFillSize = (sizeof(encryptHeaderSize) - headerSizeStr.size()) + encryptHeaderSize + h.len1 + h.len2 + 1;
															#line 673 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 569 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (*zeroFillSize > 0)
															#line 677 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 570 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (!static_cast<ReadOpDataActor*>(this)->SAV<Standalone<StringRef>>::futures) { (void)(headerSizeStr); this->~ReadOpDataActorState(); static_cast<ReadOpDataActor*>(this)->destroy(); return 0; }
															#line 681 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			new (&static_cast<ReadOpDataActor*>(this)->SAV< Standalone<StringRef> >::value()) Standalone<StringRef>(std::move(headerSizeStr)); // state_var_RVO
			this->~ReadOpDataActorState();
			static_cast<ReadOpDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Standalone<StringRef> const& __headerSizeStr,int loopDepth) 
	{
															#line 559 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		headerSizeStr = __headerSizeStr;
															#line 695 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Standalone<StringRef> && __headerSizeStr,int loopDepth) 
	{
		headerSizeStr = std::move(__headerSizeStr);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadOpDataActor*>(this)->actor_wait_state > 0) static_cast<ReadOpDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadOpDataActor*>(this)->ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 579 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		ASSERT(data.size() <= remainingBytes);
															#line 580 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		*zeroFillSize = remainingBytes - data.size();
															#line 581 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (*zeroFillSize == 0)
															#line 778 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 582 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			*isZeroFilled = (data[data.size() - 1] == 0);
															#line 782 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 584 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!encryptedOp || *zeroFillSize > 0 || *isZeroFilled)
															#line 786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 585 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (!static_cast<ReadOpDataActor*>(this)->SAV<Standalone<StringRef>>::futures) { (void)(data); this->~ReadOpDataActorState(); static_cast<ReadOpDataActor*>(this)->destroy(); return 0; }
															#line 790 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			new (&static_cast<ReadOpDataActor*>(this)->SAV< Standalone<StringRef> >::value()) Standalone<StringRef>(std::move(data)); // state_var_RVO
			this->~ReadOpDataActorState();
			static_cast<ReadOpDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		arena = Arena();
															#line 588 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		plaintext = StringRef();
															#line 589 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		cipherHeaderRef = BlobCipherEncryptHeaderRef::fromStringRef(StringRef(data.begin(), encryptHeaderSize));
															#line 591 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<TextAndHeaderCipherKeys> __when_expr_2 = GetEncryptCipherKeys<ServerDBInfo>::getEncryptCipherKeys( self->db, cipherHeaderRef, BlobCipherMetrics::KV_MEMORY);
															#line 591 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<ReadOpDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 806 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont7when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadOpDataActor*>(this)->actor_wait_state = 3;
															#line 591 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadOpDataActor, 2, TextAndHeaderCipherKeys >*>(static_cast<ReadOpDataActor*>(this)));
															#line 811 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Standalone<StringRef> const& __data,int loopDepth) 
	{
															#line 578 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		data = __data;
															#line 820 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Standalone<StringRef> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadOpDataActor*>(this)->actor_wait_state > 0) static_cast<ReadOpDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadOpDataActor*>(this)->ActorCallback< ReadOpDataActor, 1, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadOpDataActor, 1, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadOpDataActor, 1, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadOpDataActor, 1, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont9(TextAndHeaderCipherKeys const& cipherKeys,int loopDepth) 
	{
															#line 593 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		DecryptBlobCipherAes256Ctr cipher(cipherKeys.cipherTextKey, cipherKeys.cipherHeaderKey, cipherHeaderRef.getIV(), BlobCipherMetrics::KV_MEMORY);
															#line 597 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		plaintext = cipher.decrypt(data.begin() + encryptHeaderSize, h.len1 + h.len2, cipherHeaderRef, arena);
															#line 598 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<ReadOpDataActor*>(this)->SAV<Standalone<StringRef>>::futures) { (void)(Standalone<StringRef>(plaintext, arena)); this->~ReadOpDataActorState(); static_cast<ReadOpDataActor*>(this)->destroy(); return 0; }
															#line 903 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<ReadOpDataActor*>(this)->SAV< Standalone<StringRef> >::value()) Standalone<StringRef>(Standalone<StringRef>(plaintext, arena));
		this->~ReadOpDataActorState();
		static_cast<ReadOpDataActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(TextAndHeaderCipherKeys && cipherKeys,int loopDepth) 
	{
															#line 593 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		DecryptBlobCipherAes256Ctr cipher(cipherKeys.cipherTextKey, cipherKeys.cipherHeaderKey, cipherHeaderRef.getIV(), BlobCipherMetrics::KV_MEMORY);
															#line 597 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		plaintext = cipher.decrypt(data.begin() + encryptHeaderSize, h.len1 + h.len2, cipherHeaderRef, arena);
															#line 598 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<ReadOpDataActor*>(this)->SAV<Standalone<StringRef>>::futures) { (void)(Standalone<StringRef>(plaintext, arena)); this->~ReadOpDataActorState(); static_cast<ReadOpDataActor*>(this)->destroy(); return 0; }
															#line 919 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<ReadOpDataActor*>(this)->SAV< Standalone<StringRef> >::value()) Standalone<StringRef>(Standalone<StringRef>(plaintext, arena));
		this->~ReadOpDataActorState();
		static_cast<ReadOpDataActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7when1(TextAndHeaderCipherKeys const& cipherKeys,int loopDepth) 
	{
		loopDepth = a_body1cont9(cipherKeys, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(TextAndHeaderCipherKeys && cipherKeys,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(cipherKeys), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadOpDataActor*>(this)->actor_wait_state > 0) static_cast<ReadOpDataActor*>(this)->actor_wait_state = 0;
		static_cast<ReadOpDataActor*>(this)->ActorCallback< ReadOpDataActor, 2, TextAndHeaderCipherKeys >::remove();

	}
	void a_callback_fire(ActorCallback< ReadOpDataActor, 2, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys const& value) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadOpDataActor, 2, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys && value) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadOpDataActor, 2, TextAndHeaderCipherKeys >*,Error err) 
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	OpHeader h;
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool* isZeroFilled;
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int* zeroFillSize;
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool encryptedOp;
															#line 557 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	uint16_t encryptHeaderSize;
															#line 559 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Standalone<StringRef> headerSizeStr;
															#line 573 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int remainingBytes;
															#line 578 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Standalone<StringRef> data;
															#line 587 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Arena arena;
															#line 588 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	StringRef plaintext;
															#line 589 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	BlobCipherEncryptHeaderRef cipherHeaderRef;
															#line 1026 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via readOpData()
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class ReadOpDataActor final : public Actor<Standalone<StringRef>>, public ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >, public ActorCallback< ReadOpDataActor, 1, Standalone<StringRef> >, public ActorCallback< ReadOpDataActor, 2, TextAndHeaderCipherKeys >, public FastAllocated<ReadOpDataActor>, public ReadOpDataActorState<ReadOpDataActor> {
															#line 1031 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<ReadOpDataActor>::operator new;
	using FastAllocated<ReadOpDataActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15217714447076573952UL, 10603855098540297216UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Standalone<StringRef>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >;
friend struct ActorCallback< ReadOpDataActor, 1, Standalone<StringRef> >;
friend struct ActorCallback< ReadOpDataActor, 2, TextAndHeaderCipherKeys >;
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	ReadOpDataActor(KeyValueStoreMemory* const& self,OpHeader const& h,bool* const& isZeroFilled,int* const& zeroFillSize,bool const& encryptedOp) 
															#line 1050 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Standalone<StringRef>>(),
		   ReadOpDataActorState<ReadOpDataActor>(self, h, isZeroFilled, zeroFillSize, encryptedOp),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readOpData", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16657327943355146496UL, 2700099178469709056UL);
		ActorExecutionContextHelper __helper(static_cast<ReadOpDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readOpData");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readOpData", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadOpDataActor, 0, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadOpDataActor, 1, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadOpDataActor, 2, TextAndHeaderCipherKeys >*)0, actor_cancelled()); break;
		}

	}
};
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Standalone<StringRef>> readOpData( KeyValueStoreMemory* const& self, OpHeader const& h, bool* const& isZeroFilled, int* const& zeroFillSize, bool const& encryptedOp ) {
															#line 544 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Standalone<StringRef>>(new ReadOpDataActor(self, h, isZeroFilled, zeroFillSize, encryptedOp));
															#line 1084 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 600 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"

																#line 1089 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via recover()
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class RecoverActor>
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class RecoverActorState {
															#line 1095 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	RecoverActorState(KeyValueStoreMemory* const& self,bool const& exactRecovery) 
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   exactRecovery(exactRecovery)
															#line 1104 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("recover", reinterpret_cast<unsigned long>(this));

	}
	~RecoverActorState() 
	{
		fdb_probe_actor_destroy("recover", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 602 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			;
															#line 1119 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RecoverActorState();
		static_cast<RecoverActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 607 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		uncommittedNextKey = self->recoveredSnapshotKey;
															#line 608 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		uncommittedPrevSnapshotEnd = self->previousSnapshotEnd = self->log->getNextReadLocation();
															#line 610 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		uncommittedSnapshotEnd = self->currentSnapshotEnd = uncommittedPrevSnapshotEnd;
															#line 612 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		zeroFillSize = 0;
															#line 613 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgSnapshotItemCount = 0;
															#line 614 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgSnapshotEndCount = 0;
															#line 615 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgMutationCount = 0;
															#line 616 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgCommitCount = 0;
															#line 617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		startt = now();
															#line 618 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		dbgid = self->id;
															#line 620 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		loggingDelay = delay(1.0);
															#line 622 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		recoveryQueue = OpQueue();
															#line 623 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		h = OpHeader();
															#line 624 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKey = Standalone<StringRef>();
															#line 625 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		isZeroFilled = bool();
															#line 627 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		TraceEvent("KVSMemRecoveryStarted", self->id).detail("SnapshotEndLocation", uncommittedSnapshotEnd);
															#line 1179 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		try {
															#line 630 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			;
															#line 1183 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = a_body1loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 809 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			bool ok = e.code() == error_code_operation_cancelled || e.code() == error_code_file_not_found || e.code() == error_code_disk_adapter_reset;
															#line 811 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent(ok ? SevInfo : SevError, "ErrorDuringRecovery", dbgid).errorUnsuppressed(e);
															#line 812 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (e.code() != error_code_disk_adapter_reset)
															#line 1209 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 813 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 1213 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			}
															#line 815 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			self->data.clear();
															#line 816 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			self->dataSets.clear();
															#line 1219 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 771 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (zeroFillSize)
															#line 1234 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 772 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (exactRecovery)
															#line 1238 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 773 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				TraceEvent(SevError, "KVSMemExpectedExact", self->id).log();
															#line 774 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				ASSERT(false);
															#line 1244 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			}
															#line 777 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			CODE_PROBE(true, "Fixing a partial commit at the end of the KeyValueStoreMemory log");
															#line 778 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			for(int i = 0;i < zeroFillSize;i++) {
															#line 779 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->log->push(StringRef((const uint8_t*)"", 1));
															#line 1252 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			}
		}
															#line 785 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->log_op(OpRollback, StringRef(), StringRef());
															#line 787 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->committedDataSize = self->data.sumTo(self->data.end());
															#line 789 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		TraceEvent("KVSMemRecovered", self->id) .detail("SnapshotItems", dbgSnapshotItemCount) .detail("SnapshotEnd", dbgSnapshotEndCount) .detail("Mutations", dbgMutationCount) .detail("Commits", dbgCommitCount) .detail("TimeTaken", now() - startt);
															#line 798 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (self->enableEncryption)
															#line 1263 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 799 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_3 = updateCipherKeys(self);
															#line 799 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1269 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_3.get(), loopDepth); };
			static_cast<RecoverActor*>(this)->actor_wait_state = 4;
															#line 799 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 3, Void >*>(static_cast<RecoverActor*>(this)));
															#line 1274 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 631 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		encryptedOp = false;
															#line 1295 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 633 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Standalone<StringRef>> __when_expr_0 = self->log->readNext(sizeof(OpHeader));
															#line 633 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1301 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RecoverActor*>(this)->actor_wait_state = 1;
															#line 633 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 0, Standalone<StringRef> >*>(static_cast<RecoverActor*>(this)));
															#line 1306 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Standalone<StringRef>> __when_expr_1 = readOpData(self, h, &isZeroFilled, &zeroFillSize, encryptedOp);
															#line 662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1331 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 2;
															#line 662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 1, Standalone<StringRef> >*>(static_cast<RecoverActor*>(this)));
															#line 1336 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Standalone<StringRef> const& data,int loopDepth) 
	{
															#line 634 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (data.size() != sizeof(OpHeader))
															#line 1345 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 635 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (data.size())
															#line 1349 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 636 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				CODE_PROBE( true, "zero fill partial header in KeyValueStoreMemory", probe::decoration::rare);
															#line 638 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				memset(&h, 0, sizeof(OpHeader));
															#line 639 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				memcpy(&h, data.begin(), data.size());
															#line 640 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				zeroFillSize = sizeof(OpHeader) - data.size() + h.len1 + h.len2 + 1;
															#line 641 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (isOpEncrypted(&h))
															#line 1361 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 645 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					zeroFillSize += 0 + sizeof(uint16_t);
															#line 1365 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
			}
															#line 648 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoveryComplete", self->id) .detail("Reason", "Non-header sized data read") .detail("DataSize", data.size()) .detail("ZeroFillSize", zeroFillSize) .detail("SnapshotEndLocation", uncommittedSnapshotEnd) .detail("NextReadLoc", self->log->getNextReadLocation());
															#line 1370 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 656 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		h = *(OpHeader*)data.begin();
															#line 657 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		encryptedOp = isOpEncrypted(&h);
															#line 659 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		setEncryptFlag(&h, false);
															#line 660 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		ASSERT(h.op != OpEncrypted_Deprecated);
															#line 1381 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Standalone<StringRef> && data,int loopDepth) 
	{
															#line 634 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (data.size() != sizeof(OpHeader))
															#line 1390 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 635 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (data.size())
															#line 1394 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 636 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				CODE_PROBE( true, "zero fill partial header in KeyValueStoreMemory", probe::decoration::rare);
															#line 638 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				memset(&h, 0, sizeof(OpHeader));
															#line 639 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				memcpy(&h, data.begin(), data.size());
															#line 640 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				zeroFillSize = sizeof(OpHeader) - data.size() + h.len1 + h.len2 + 1;
															#line 641 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (isOpEncrypted(&h))
															#line 1406 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 645 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					zeroFillSize += 0 + sizeof(uint16_t);
															#line 1410 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
			}
															#line 648 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoveryComplete", self->id) .detail("Reason", "Non-header sized data read") .detail("DataSize", data.size()) .detail("ZeroFillSize", zeroFillSize) .detail("SnapshotEndLocation", uncommittedSnapshotEnd) .detail("NextReadLoc", self->log->getNextReadLocation());
															#line 1415 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 656 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		h = *(OpHeader*)data.begin();
															#line 657 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		encryptedOp = isOpEncrypted(&h);
															#line 659 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		setEncryptFlag(&h, false);
															#line 660 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		ASSERT(h.op != OpEncrypted_Deprecated);
															#line 1426 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Standalone<StringRef> const& data,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(data, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Standalone<StringRef> && data,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(std::move(data), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 0, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 0, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 0, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RecoverActor, 0, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont7(int loopDepth) 
	{
															#line 664 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (zeroFillSize > 0)
															#line 1510 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 665 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoveryComplete", self->id) .detail("Reason", "data specified by header does not exist") .detail("DataSize", data.size()) .detail("ZeroFillSize", zeroFillSize) .detail("SnapshotEndLocation", uncommittedSnapshotEnd) .detail("OpCode", h.op) .detail("NextReadLoc", self->log->getNextReadLocation());
															#line 1514 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 675 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!isZeroFilled)
															#line 1519 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 676 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StringRef p1 = data.substr(0, h.len1);
															#line 677 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StringRef p2 = data.substr(h.len1, h.len2);
															#line 679 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			DEBUG_TRANSACTION_STATE_STORE("Recover", p1, self->id);
															#line 681 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (h.op == OpSnapshotItem || h.op == OpSnapshotItemDelta)
															#line 1529 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 689 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (h.op == OpSnapshotItemDelta)
															#line 1533 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 690 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					ASSERT(p1.size() > 1);
															#line 692 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					int borrowed = *(uint8_t*)p1.begin();
															#line 693 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					ASSERT(borrowed <= lastSnapshotKey.size());
															#line 695 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					StringRef suffix = p1.substr(1);
															#line 697 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					Arena& dataArena = *(Arena*)&data.arena();
															#line 698 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					p1 = makeString(borrowed + suffix.size(), dataArena);
															#line 700 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					memcpy(mutateString(p1), lastSnapshotKey.begin(), borrowed);
															#line 702 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					memcpy(mutateString(p1) + borrowed, suffix.begin(), suffix.size());
															#line 1551 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 704 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (p1 >= uncommittedNextKey)
															#line 1555 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 705 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					recoveryQueue.clear( KeyRangeRef(uncommittedNextKey, p1), &uncommittedNextKey .arena());
															#line 1559 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 709 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				recoveryQueue.set(KeyValueRef(p1, p2), &data.arena());
															#line 710 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				uncommittedNextKey = keyAfter(p1);
															#line 711 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				++dbgSnapshotItemCount;
															#line 712 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				lastSnapshotKey = Key(p1, data.arena());
															#line 1569 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			}
			else
			{
															#line 713 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (h.op == OpSnapshotEnd || h.op == OpSnapshotAbort)
															#line 1575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 714 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					TraceEvent("RecSnapshotEnd", self->id) .detail("NextKey", uncommittedNextKey) .detail("Nextlocation", self->log->getNextReadLocation()) .detail("IsSnapshotEnd", h.op == OpSnapshotEnd);
															#line 719 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					if (h.op == OpSnapshotEnd)
															#line 1581 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
					{
															#line 720 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						uncommittedPrevSnapshotEnd = uncommittedSnapshotEnd;
															#line 721 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						uncommittedSnapshotEnd = self->log->getNextReadLocation();
															#line 722 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						recoveryQueue.clear_to_end(uncommittedNextKey, &uncommittedNextKey.arena());
															#line 1589 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
					}
															#line 725 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uncommittedNextKey = Key();
															#line 726 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					lastSnapshotKey = Key();
															#line 727 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					++dbgSnapshotEndCount;
															#line 1597 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 728 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					if (h.op == OpSet)
															#line 1603 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
					{
															#line 729 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						recoveryQueue.set(KeyValueRef(p1, p2), &data.arena());
															#line 730 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						++dbgMutationCount;
															#line 1609 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
					}
					else
					{
															#line 731 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
						if (h.op == OpClear)
															#line 1615 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
						{
															#line 732 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
							recoveryQueue.clear(KeyRangeRef(p1, p2), &data.arena());
															#line 733 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
							++dbgMutationCount;
															#line 1621 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
						}
						else
						{
															#line 734 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
							if (h.op == OpClearToEnd)
															#line 1627 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
							{
															#line 735 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
								recoveryQueue.clear_to_end(p1, &data.arena());
															#line 1631 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
							}
							else
							{
															#line 736 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
								if (h.op == OpCommit)
															#line 1637 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
								{
															#line 737 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									self->commit_queue(recoveryQueue, false);
															#line 738 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									++dbgCommitCount;
															#line 739 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									self->recoveredSnapshotKey = uncommittedNextKey;
															#line 740 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									self->previousSnapshotEnd = uncommittedPrevSnapshotEnd;
															#line 741 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									self->currentSnapshotEnd = uncommittedSnapshotEnd;
															#line 1649 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
								}
								else
								{
															#line 742 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
									if (h.op == OpRollback)
															#line 1655 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
									{
															#line 743 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										recoveryQueue.rollback();
															#line 744 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										TraceEvent("KVSMemRecSnapshotRollback", self->id).detail("NextKey", uncommittedNextKey);
															#line 745 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										uncommittedNextKey = self->recoveredSnapshotKey;
															#line 746 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										uncommittedPrevSnapshotEnd = self->previousSnapshotEnd;
															#line 747 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										uncommittedSnapshotEnd = self->currentSnapshotEnd;
															#line 1667 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
									}
									else
									{
															#line 749 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
										ASSERT(false);
															#line 1673 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
									}
								}
							}
						}
					}
				}
			}
		}
		else
		{
															#line 751 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoverySkippedZeroFill", self->id) .detail("PayloadSize", data.size()) .detail("ExpectedSize", h.len1 + h.len2 + 1) .detail("OpCode", h.op) .detail("EndsAt", self->log->getNextReadLocation());
															#line 1686 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 758 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (loggingDelay.isReady())
															#line 1690 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 759 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent("KVSMemRecoveryLogSnap", self->id) .detail("SnapshotItems", dbgSnapshotItemCount) .detail("SnapshotEnd", dbgSnapshotEndCount) .detail("Mutations", dbgMutationCount) .detail("Commits", dbgCommitCount) .detail("EndsAt", self->log->getNextReadLocation());
															#line 765 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			loggingDelay = delay(1.0);
															#line 1696 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 768 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield();
															#line 768 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<RecoverActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1702 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont7when1(__when_expr_2.get(), loopDepth); };
		static_cast<RecoverActor*>(this)->actor_wait_state = 3;
															#line 768 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RecoverActor, 2, Void >*>(static_cast<RecoverActor*>(this)));
															#line 1707 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Standalone<StringRef> const& __data,int loopDepth) 
	{
															#line 662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		data = __data;
															#line 1716 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1when1(Standalone<StringRef> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 1, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 1, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 1, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RecoverActor, 1, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont8(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont8(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RecoverActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 802 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		CODE_PROBE(self->enableEncryption && self->uncommittedBytes() > 0, "KeyValueStoreMemory recovered partial transaction while encryption-at-rest is enabled", probe::decoration::rare);
															#line 805 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->semiCommit();
															#line 807 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<RecoverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RecoverActorState(); static_cast<RecoverActor*>(this)->destroy(); return 0; }
															#line 1886 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<RecoverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RecoverActorState();
		static_cast<RecoverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<RecoverActor*>(this)->actor_wait_state > 0) static_cast<RecoverActor*>(this)->actor_wait_state = 0;
		static_cast<RecoverActor*>(this)->ActorCallback< RecoverActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RecoverActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< RecoverActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< RecoverActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool exactRecovery;
															#line 607 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key uncommittedNextKey;
															#line 608 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	IDiskQueue::location uncommittedPrevSnapshotEnd;
															#line 610 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	IDiskQueue::location uncommittedSnapshotEnd;
															#line 612 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int zeroFillSize;
															#line 613 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int dbgSnapshotItemCount;
															#line 614 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int dbgSnapshotEndCount;
															#line 615 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int dbgMutationCount;
															#line 616 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int dbgCommitCount;
															#line 617 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	double startt;
															#line 618 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	UID dbgid;
															#line 620 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Future<Void> loggingDelay;
															#line 622 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	OpQueue recoveryQueue;
															#line 623 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	OpHeader h;
															#line 624 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Standalone<StringRef> lastSnapshotKey;
															#line 625 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool isZeroFilled;
															#line 631 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool encryptedOp;
															#line 662 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Standalone<StringRef> data;
															#line 2019 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via recover()
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class RecoverActor final : public Actor<Void>, public ActorCallback< RecoverActor, 0, Standalone<StringRef> >, public ActorCallback< RecoverActor, 1, Standalone<StringRef> >, public ActorCallback< RecoverActor, 2, Void >, public ActorCallback< RecoverActor, 3, Void >, public FastAllocated<RecoverActor>, public RecoverActorState<RecoverActor> {
															#line 2024 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<RecoverActor>::operator new;
	using FastAllocated<RecoverActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11097144617715412224UL, 6289188855623280640UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RecoverActor, 0, Standalone<StringRef> >;
friend struct ActorCallback< RecoverActor, 1, Standalone<StringRef> >;
friend struct ActorCallback< RecoverActor, 2, Void >;
friend struct ActorCallback< RecoverActor, 3, Void >;
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	RecoverActor(KeyValueStoreMemory* const& self,bool const& exactRecovery) 
															#line 2044 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Void>(),
		   RecoverActorState<RecoverActor>(self, exactRecovery),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("recover", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7334007120712012544UL, 4865582193762289920UL);
		ActorExecutionContextHelper __helper(static_cast<RecoverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("recover");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("recover", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RecoverActor, 0, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RecoverActor, 1, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RecoverActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< RecoverActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> recover( KeyValueStoreMemory* const& self, bool const& exactRecovery ) {
															#line 601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new RecoverActor(self, exactRecovery));
															#line 2079 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 820 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"

	// Snapshots an entire data set
	void fullSnapshot(Container& snapshotData) {
		previousSnapshotEnd = log_op(OpSnapshotAbort, StringRef(), StringRef());
		replaceContent = false;

		// Clear everything since we are about to write the whole database
		log_op(OpClearToEnd, allKeys.begin, StringRef());

		int count = 0;
		int64_t snapshotSize = 0;
		for (auto kv = snapshotData.begin(); kv != snapshotData.end(); ++kv) {
			StringRef tempKey = kv.getKey(reserved_buffer);
			DEBUG_TRANSACTION_STATE_STORE("FullSnapshot", tempKey, id);
			log_op(OpSnapshotItem, tempKey, kv.getValue());
			snapshotSize += tempKey.size() + kv.getValue().size() + OP_DISK_OVERHEAD;
			++count;
		}

		TraceEvent("FullSnapshotEnd", id)
		    .detail("PreviousSnapshotEndLoc", previousSnapshotEnd)
		    .detail("SnapshotSize", snapshotSize)
		    .detail("SnapshotElements", count);

		currentSnapshotEnd = log_op(OpSnapshotEnd, StringRef(), StringRef());
	}

																#line 2110 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via snapshot()
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class SnapshotActor>
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class SnapshotActorState {
															#line 2116 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	SnapshotActorState(KeyValueStoreMemory* const& self) 
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self)
															#line 2123 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("snapshot", reinterpret_cast<unsigned long>(this));

	}
	~SnapshotActorState() 
	{
		fdb_probe_actor_destroy("snapshot", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 848 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 848 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<SnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2140 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SnapshotActor*>(this)->actor_wait_state = 1;
															#line 848 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SnapshotActor, 0, Void >*>(static_cast<SnapshotActor*>(this)));
															#line 2145 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SnapshotActorState();
		static_cast<SnapshotActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		nextKey = self->recoveredSnapshotKey;
															#line 851 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		nextKeyAfter = false;
															#line 852 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapshotTotalWrittenBytes = 0;
															#line 853 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastDiff = 0;
															#line 854 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapItems = 0;
															#line 855 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapshotBytes = 0;
															#line 860 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyA = makeString(CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 861 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyB = makeString(CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 862 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyUsingA = true;
															#line 864 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		TraceEvent("KVSMemStartingSnapshot", self->id).detail("StartKey", nextKey);
															#line 866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		;
															#line 2188 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		nextKey = self->recoveredSnapshotKey;
															#line 851 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		nextKeyAfter = false;
															#line 852 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapshotTotalWrittenBytes = 0;
															#line 853 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastDiff = 0;
															#line 854 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapItems = 0;
															#line 855 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		snapshotBytes = 0;
															#line 860 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyA = makeString(CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 861 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyB = makeString(CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 862 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastSnapshotKeyUsingA = true;
															#line 864 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		TraceEvent("KVSMemStartingSnapshot", self->id).detail("StartKey", nextKey);
															#line 866 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		;
															#line 2217 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SnapshotActor*>(this)->actor_wait_state > 0) static_cast<SnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<SnapshotActor*>(this)->ActorCallback< SnapshotActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SnapshotActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SnapshotActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SnapshotActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 867 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->notifiedCommittedWriteBytes.whenAtLeast(snapshotTotalWrittenBytes + 1);
															#line 867 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<SnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2310 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SnapshotActor*>(this)->actor_wait_state = 2;
															#line 867 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SnapshotActor, 1, Void >*>(static_cast<SnapshotActor*>(this)));
															#line 2315 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 869 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (self->resetSnapshot)
															#line 2324 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 870 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			nextKey = Key();
															#line 871 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			nextKeyAfter = false;
															#line 872 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			snapItems = 0;
															#line 873 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			snapshotBytes = 0;
															#line 874 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			self->resetSnapshot = false;
															#line 2336 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 877 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		auto next = nextKeyAfter ? self->data.upper_bound(nextKey) : self->data.lower_bound(nextKey);
															#line 878 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		int diff = self->notifiedCommittedWriteBytes.get() - snapshotTotalWrittenBytes;
															#line 879 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (diff > lastDiff && diff > 5e7)
															#line 2344 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 880 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent(SevWarnAlways, "ManyWritesAtOnce", self->id) .detail("CommittedWrites", self->notifiedCommittedWriteBytes.get()) .detail("SnapshotWrites", snapshotTotalWrittenBytes) .detail("Diff", diff) .detail("LastOperationWasASnapshot", nextKey == Key() && !nextKeyAfter);
															#line 2348 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 885 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastDiff = diff;
															#line 890 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		bool useDelta = false;
															#line 893 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		for(;;) {
															#line 895 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (next == self->data.end())
															#line 2358 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				useDelta = false;
															#line 900 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				auto thisSnapshotEnd = self->log_op(OpSnapshotEnd, StringRef(), StringRef());
															#line 901 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				DisabledTraceEvent("SnapshotEnd", self->id) .detail("CurrentSnapshotEndLoc", self->currentSnapshotEnd) .detail("PreviousSnapshotEndLoc", self->previousSnapshotEnd) .detail("ThisSnapshotEnd", thisSnapshotEnd) .detail("Items", snapItems) .detail("CommittedWrites", self->notifiedCommittedWriteBytes.get()) .detail("SnapshotSize", snapshotBytes);
															#line 909 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				ASSERT(thisSnapshotEnd >= self->currentSnapshotEnd);
															#line 910 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->previousSnapshotEnd = self->currentSnapshotEnd;
															#line 911 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->currentSnapshotEnd = thisSnapshotEnd;
															#line 913 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (++self->snapshotCount == 2)
															#line 2374 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->replaceContent = false;
															#line 2378 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 917 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapItems = 0;
															#line 918 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotBytes = 0;
															#line 919 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotTotalWrittenBytes += OP_DISK_OVERHEAD;
															#line 922 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (snapshotTotalWrittenBytes < self->notifiedCommittedWriteBytes.get())
															#line 2388 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 923 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					next = self->data.begin();
															#line 2392 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 926 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKey = Key();
															#line 927 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKeyAfter = false;
															#line 2400 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
					break;
				}
			}
			else
			{
															#line 933 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				Key& destKey = lastSnapshotKeyUsingA ? lastSnapshotKeyA : lastSnapshotKeyB;
															#line 936 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				KeyRef tempKey = next.getKey(mutateString(destKey));
															#line 937 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				int opKeySize = tempKey.size();
															#line 942 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (tempKey.begin() != destKey.begin())
															#line 2414 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 943 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					memcpy(mutateString(destKey), tempKey.begin(), tempKey.size());
															#line 2418 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 949 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				destKey.contents() = KeyRef(destKey.begin(), tempKey.size());
															#line 951 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				DEBUG_TRANSACTION_STATE_STORE("SnapshotItem", destKey.toString(), self->id);
															#line 954 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				int commonPrefix;
															#line 955 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (useDelta && SERVER_KNOBS->PREFIX_COMPRESS_KVS_MEM_SNAPSHOTS)
															#line 2428 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 956 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = commonPrefixLength(lastSnapshotKeyA, lastSnapshotKeyB);
															#line 2432 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 958 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = 0;
															#line 959 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					useDelta = true;
															#line 2440 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 964 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (commonPrefix > 1)
															#line 2444 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 966 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = std::min<int>(commonPrefix, std::numeric_limits<uint8_t>::max());
															#line 970 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uint8_t& prefixLength = mutateString(destKey)[commonPrefix - 1];
															#line 971 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uint8_t backupByte = prefixLength;
															#line 972 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					prefixLength = commonPrefix;
															#line 974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					opKeySize = opKeySize - commonPrefix + 1;
															#line 975 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					KeyRef opKey(&prefixLength, opKeySize);
															#line 976 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->log_op(OpSnapshotItemDelta, opKey, next.getValue());
															#line 979 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					prefixLength = backupByte;
															#line 2462 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 981 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->log_op(OpSnapshotItem, tempKey, next.getValue());
															#line 2468 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 984 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapItems++;
															#line 985 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				uint64_t opBytes = opKeySize + next.getValue().size() + OP_DISK_OVERHEAD;
															#line 986 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotBytes += opBytes;
															#line 987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotTotalWrittenBytes += opBytes;
															#line 988 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				lastSnapshotKeyUsingA = !lastSnapshotKeyUsingA;
															#line 991 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (snapshotTotalWrittenBytes < self->notifiedCommittedWriteBytes.get())
															#line 2482 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 992 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					++next;
															#line 2486 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 995 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKey = destKey;
															#line 996 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKeyAfter = true;
															#line 2494 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
					break;
				}
			}
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 869 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (self->resetSnapshot)
															#line 2507 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 870 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			nextKey = Key();
															#line 871 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			nextKeyAfter = false;
															#line 872 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			snapItems = 0;
															#line 873 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			snapshotBytes = 0;
															#line 874 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			self->resetSnapshot = false;
															#line 2519 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 877 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		auto next = nextKeyAfter ? self->data.upper_bound(nextKey) : self->data.lower_bound(nextKey);
															#line 878 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		int diff = self->notifiedCommittedWriteBytes.get() - snapshotTotalWrittenBytes;
															#line 879 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (diff > lastDiff && diff > 5e7)
															#line 2527 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		{
															#line 880 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			TraceEvent(SevWarnAlways, "ManyWritesAtOnce", self->id) .detail("CommittedWrites", self->notifiedCommittedWriteBytes.get()) .detail("SnapshotWrites", snapshotTotalWrittenBytes) .detail("Diff", diff) .detail("LastOperationWasASnapshot", nextKey == Key() && !nextKeyAfter);
															#line 2531 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		}
															#line 885 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		lastDiff = diff;
															#line 890 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		bool useDelta = false;
															#line 893 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		for(;;) {
															#line 895 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (next == self->data.end())
															#line 2541 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			{
															#line 898 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				useDelta = false;
															#line 900 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				auto thisSnapshotEnd = self->log_op(OpSnapshotEnd, StringRef(), StringRef());
															#line 901 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				DisabledTraceEvent("SnapshotEnd", self->id) .detail("CurrentSnapshotEndLoc", self->currentSnapshotEnd) .detail("PreviousSnapshotEndLoc", self->previousSnapshotEnd) .detail("ThisSnapshotEnd", thisSnapshotEnd) .detail("Items", snapItems) .detail("CommittedWrites", self->notifiedCommittedWriteBytes.get()) .detail("SnapshotSize", snapshotBytes);
															#line 909 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				ASSERT(thisSnapshotEnd >= self->currentSnapshotEnd);
															#line 910 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->previousSnapshotEnd = self->currentSnapshotEnd;
															#line 911 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				self->currentSnapshotEnd = thisSnapshotEnd;
															#line 913 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (++self->snapshotCount == 2)
															#line 2557 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 914 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->replaceContent = false;
															#line 2561 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 917 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapItems = 0;
															#line 918 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotBytes = 0;
															#line 919 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotTotalWrittenBytes += OP_DISK_OVERHEAD;
															#line 922 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (snapshotTotalWrittenBytes < self->notifiedCommittedWriteBytes.get())
															#line 2571 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 923 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					next = self->data.begin();
															#line 2575 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 926 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKey = Key();
															#line 927 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKeyAfter = false;
															#line 2583 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
					break;
				}
			}
			else
			{
															#line 933 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				Key& destKey = lastSnapshotKeyUsingA ? lastSnapshotKeyA : lastSnapshotKeyB;
															#line 936 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				KeyRef tempKey = next.getKey(mutateString(destKey));
															#line 937 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				int opKeySize = tempKey.size();
															#line 942 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (tempKey.begin() != destKey.begin())
															#line 2597 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 943 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					memcpy(mutateString(destKey), tempKey.begin(), tempKey.size());
															#line 2601 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 949 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				destKey.contents() = KeyRef(destKey.begin(), tempKey.size());
															#line 951 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				DEBUG_TRANSACTION_STATE_STORE("SnapshotItem", destKey.toString(), self->id);
															#line 954 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				int commonPrefix;
															#line 955 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (useDelta && SERVER_KNOBS->PREFIX_COMPRESS_KVS_MEM_SNAPSHOTS)
															#line 2611 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 956 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = commonPrefixLength(lastSnapshotKeyA, lastSnapshotKeyB);
															#line 2615 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 958 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = 0;
															#line 959 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					useDelta = true;
															#line 2623 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 964 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (commonPrefix > 1)
															#line 2627 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 966 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					commonPrefix = std::min<int>(commonPrefix, std::numeric_limits<uint8_t>::max());
															#line 970 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uint8_t& prefixLength = mutateString(destKey)[commonPrefix - 1];
															#line 971 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					uint8_t backupByte = prefixLength;
															#line 972 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					prefixLength = commonPrefix;
															#line 974 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					opKeySize = opKeySize - commonPrefix + 1;
															#line 975 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					KeyRef opKey(&prefixLength, opKeySize);
															#line 976 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->log_op(OpSnapshotItemDelta, opKey, next.getValue());
															#line 979 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					prefixLength = backupByte;
															#line 2645 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 981 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					self->log_op(OpSnapshotItem, tempKey, next.getValue());
															#line 2651 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
															#line 984 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapItems++;
															#line 985 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				uint64_t opBytes = opKeySize + next.getValue().size() + OP_DISK_OVERHEAD;
															#line 986 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotBytes += opBytes;
															#line 987 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				snapshotTotalWrittenBytes += opBytes;
															#line 988 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				lastSnapshotKeyUsingA = !lastSnapshotKeyUsingA;
															#line 991 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
				if (snapshotTotalWrittenBytes < self->notifiedCommittedWriteBytes.get())
															#line 2665 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				{
															#line 992 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					++next;
															#line 2669 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
				}
				else
				{
															#line 995 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKey = destKey;
															#line 996 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
					nextKeyAfter = true;
															#line 2677 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
					break;
				}
			}
		}
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SnapshotActor*>(this)->actor_wait_state > 0) static_cast<SnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<SnapshotActor*>(this)->ActorCallback< SnapshotActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SnapshotActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SnapshotActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SnapshotActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 850 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key nextKey;
															#line 851 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool nextKeyAfter;
															#line 852 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	uint64_t snapshotTotalWrittenBytes;
															#line 853 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int lastDiff;
															#line 854 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int snapItems;
															#line 855 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	uint64_t snapshotBytes;
															#line 860 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key lastSnapshotKeyA;
															#line 861 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key lastSnapshotKeyB;
															#line 862 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool lastSnapshotKeyUsingA;
															#line 2781 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via snapshot()
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class SnapshotActor final : public Actor<Void>, public ActorCallback< SnapshotActor, 0, Void >, public ActorCallback< SnapshotActor, 1, Void >, public FastAllocated<SnapshotActor>, public SnapshotActorState<SnapshotActor> {
															#line 2786 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<SnapshotActor>::operator new;
	using FastAllocated<SnapshotActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17794427142074985728UL, 8165848171199806720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SnapshotActor, 0, Void >;
friend struct ActorCallback< SnapshotActor, 1, Void >;
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	SnapshotActor(KeyValueStoreMemory* const& self) 
															#line 2804 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Void>(),
		   SnapshotActorState<SnapshotActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("snapshot", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7446917862669081600UL, 3777547187618145280UL);
		ActorExecutionContextHelper __helper(static_cast<SnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("snapshot");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("snapshot", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SnapshotActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SnapshotActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> snapshot( KeyValueStoreMemory* const& self ) {
															#line 847 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new SnapshotActor(self));
															#line 2837 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 1003 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"

																#line 2842 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via waitAndReadValue()
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class WaitAndReadValueActor>
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadValueActorState {
															#line 2848 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadValueActorState(KeyValueStoreMemory* const& self,Key const& key,Optional<ReadOptions> const& options) 
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   key(key),
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   options(options)
															#line 2859 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("waitAndReadValue", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndReadValueActorState() 
	{
		fdb_probe_actor_destroy("waitAndReadValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1007 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 1007 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<WaitAndReadValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2876 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitAndReadValueActor*>(this)->actor_wait_state = 1;
															#line 1007 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndReadValueActor, 0, Void >*>(static_cast<WaitAndReadValueActor*>(this)));
															#line 2881 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndReadValueActorState();
		static_cast<WaitAndReadValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1008 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadValueActor*>(this)->SAV<Optional<Value>>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readValue(key, options).get()); this->~WaitAndReadValueActorState(); static_cast<WaitAndReadValueActor*>(this)->destroy(); return 0; }
															#line 2904 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<WaitAndReadValueActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(static_cast<IKeyValueStore*>(self)->readValue(key, options).get());
		this->~WaitAndReadValueActorState();
		static_cast<WaitAndReadValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1008 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadValueActor*>(this)->SAV<Optional<Value>>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readValue(key, options).get()); this->~WaitAndReadValueActorState(); static_cast<WaitAndReadValueActor*>(this)->destroy(); return 0; }
															#line 2916 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<WaitAndReadValueActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(static_cast<IKeyValueStore*>(self)->readValue(key, options).get());
		this->~WaitAndReadValueActorState();
		static_cast<WaitAndReadValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndReadValueActor*>(this)->actor_wait_state > 0) static_cast<WaitAndReadValueActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndReadValueActor*>(this)->ActorCallback< WaitAndReadValueActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndReadValueActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndReadValueActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndReadValueActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValue", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key key;
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Optional<ReadOptions> options;
															#line 3005 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via waitAndReadValue()
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadValueActor final : public Actor<Optional<Value>>, public ActorCallback< WaitAndReadValueActor, 0, Void >, public FastAllocated<WaitAndReadValueActor>, public WaitAndReadValueActorState<WaitAndReadValueActor> {
															#line 3010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<WaitAndReadValueActor>::operator new;
	using FastAllocated<WaitAndReadValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4869330000137521664UL, 9824998808168647936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Value>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndReadValueActor, 0, Void >;
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadValueActor(KeyValueStoreMemory* const& self,Key const& key,Optional<ReadOptions> const& options) 
															#line 3027 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Optional<Value>>(),
		   WaitAndReadValueActorState<WaitAndReadValueActor>(self, key, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndReadValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(866617973594577152UL, 12678521189896779008UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndReadValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndReadValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndReadValueActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Optional<Value>> waitAndReadValue( KeyValueStoreMemory* const& self, Key const& key, Optional<ReadOptions> const& options ) {
															#line 1004 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Optional<Value>>(new WaitAndReadValueActor(self, key, options));
															#line 3059 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
																#line 3063 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via waitAndReadValuePrefix()
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class WaitAndReadValuePrefixActor>
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadValuePrefixActorState {
															#line 3069 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadValuePrefixActorState(KeyValueStoreMemory* const& self,Key const& key,int const& maxLength,Optional<ReadOptions> const& options) 
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   key(key),
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   maxLength(maxLength),
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   options(options)
															#line 3082 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndReadValuePrefixActorState() 
	{
		fdb_probe_actor_destroy("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1014 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 1014 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<WaitAndReadValuePrefixActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3099 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitAndReadValuePrefixActor*>(this)->actor_wait_state = 1;
															#line 1014 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*>(static_cast<WaitAndReadValuePrefixActor*>(this)));
															#line 3104 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndReadValuePrefixActorState();
		static_cast<WaitAndReadValuePrefixActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1015 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadValuePrefixActor*>(this)->SAV<Optional<Value>>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readValuePrefix(key, maxLength, options).get()); this->~WaitAndReadValuePrefixActorState(); static_cast<WaitAndReadValuePrefixActor*>(this)->destroy(); return 0; }
															#line 3127 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<WaitAndReadValuePrefixActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(static_cast<IKeyValueStore*>(self)->readValuePrefix(key, maxLength, options).get());
		this->~WaitAndReadValuePrefixActorState();
		static_cast<WaitAndReadValuePrefixActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1015 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadValuePrefixActor*>(this)->SAV<Optional<Value>>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readValuePrefix(key, maxLength, options).get()); this->~WaitAndReadValuePrefixActorState(); static_cast<WaitAndReadValuePrefixActor*>(this)->destroy(); return 0; }
															#line 3139 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<WaitAndReadValuePrefixActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(static_cast<IKeyValueStore*>(self)->readValuePrefix(key, maxLength, options).get());
		this->~WaitAndReadValuePrefixActorState();
		static_cast<WaitAndReadValuePrefixActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndReadValuePrefixActor*>(this)->actor_wait_state > 0) static_cast<WaitAndReadValuePrefixActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndReadValuePrefixActor*>(this)->ActorCallback< WaitAndReadValuePrefixActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValuePrefixActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValuePrefixActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValuePrefixActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Key key;
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int maxLength;
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Optional<ReadOptions> options;
															#line 3230 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via waitAndReadValuePrefix()
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadValuePrefixActor final : public Actor<Optional<Value>>, public ActorCallback< WaitAndReadValuePrefixActor, 0, Void >, public FastAllocated<WaitAndReadValuePrefixActor>, public WaitAndReadValuePrefixActorState<WaitAndReadValuePrefixActor> {
															#line 3235 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<WaitAndReadValuePrefixActor>::operator new;
	using FastAllocated<WaitAndReadValuePrefixActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4011101930503615744UL, 12520281052128563456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Value>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndReadValuePrefixActor, 0, Void >;
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadValuePrefixActor(KeyValueStoreMemory* const& self,Key const& key,int const& maxLength,Optional<ReadOptions> const& options) 
															#line 3252 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Optional<Value>>(),
		   WaitAndReadValuePrefixActorState<WaitAndReadValuePrefixActor>(self, key, maxLength, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1850494756222781696UL, 8275989234370189312UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadValuePrefixActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndReadValuePrefix");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndReadValuePrefix", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndReadValuePrefixActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Optional<Value>> waitAndReadValuePrefix( KeyValueStoreMemory* const& self, Key const& key, int const& maxLength, Optional<ReadOptions> const& options ) {
															#line 1010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Optional<Value>>(new WaitAndReadValuePrefixActor(self, key, maxLength, options));
															#line 3284 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
																#line 3288 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via waitAndReadRange()
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class WaitAndReadRangeActor>
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadRangeActorState {
															#line 3294 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadRangeActorState(KeyValueStoreMemory* const& self,KeyRange const& keys,int const& rowLimit,int const& byteLimit,Optional<ReadOptions> const& options) 
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   keys(keys),
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   rowLimit(rowLimit),
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   byteLimit(byteLimit),
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   options(options)
															#line 3309 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("waitAndReadRange", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndReadRangeActorState() 
	{
		fdb_probe_actor_destroy("waitAndReadRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1022 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 1022 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<WaitAndReadRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3326 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitAndReadRangeActor*>(this)->actor_wait_state = 1;
															#line 1022 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndReadRangeActor, 0, Void >*>(static_cast<WaitAndReadRangeActor*>(this)));
															#line 3331 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndReadRangeActorState();
		static_cast<WaitAndReadRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1023 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadRangeActor*>(this)->SAV<RangeResult>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readRange(keys, rowLimit, byteLimit, options).get()); this->~WaitAndReadRangeActorState(); static_cast<WaitAndReadRangeActor*>(this)->destroy(); return 0; }
															#line 3354 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<WaitAndReadRangeActor*>(this)->SAV< RangeResult >::value()) RangeResult(static_cast<IKeyValueStore*>(self)->readRange(keys, rowLimit, byteLimit, options).get());
		this->~WaitAndReadRangeActorState();
		static_cast<WaitAndReadRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1023 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndReadRangeActor*>(this)->SAV<RangeResult>::futures) { (void)(static_cast<IKeyValueStore*>(self)->readRange(keys, rowLimit, byteLimit, options).get()); this->~WaitAndReadRangeActorState(); static_cast<WaitAndReadRangeActor*>(this)->destroy(); return 0; }
															#line 3366 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<WaitAndReadRangeActor*>(this)->SAV< RangeResult >::value()) RangeResult(static_cast<IKeyValueStore*>(self)->readRange(keys, rowLimit, byteLimit, options).get());
		this->~WaitAndReadRangeActorState();
		static_cast<WaitAndReadRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndReadRangeActor*>(this)->actor_wait_state > 0) static_cast<WaitAndReadRangeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndReadRangeActor*>(this)->ActorCallback< WaitAndReadRangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndReadRangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndReadRangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndReadRangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndReadRange", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyRange keys;
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int rowLimit;
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	int byteLimit;
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Optional<ReadOptions> options;
															#line 3459 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via waitAndReadRange()
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndReadRangeActor final : public Actor<RangeResult>, public ActorCallback< WaitAndReadRangeActor, 0, Void >, public FastAllocated<WaitAndReadRangeActor>, public WaitAndReadRangeActorState<WaitAndReadRangeActor> {
															#line 3464 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<WaitAndReadRangeActor>::operator new;
	using FastAllocated<WaitAndReadRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10958461083723798272UL, 14323893006120782080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<RangeResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndReadRangeActor, 0, Void >;
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndReadRangeActor(KeyValueStoreMemory* const& self,KeyRange const& keys,int const& rowLimit,int const& byteLimit,Optional<ReadOptions> const& options) 
															#line 3481 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<RangeResult>(),
		   WaitAndReadRangeActorState<WaitAndReadRangeActor>(self, keys, rowLimit, byteLimit, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndReadRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12404310014652120320UL, 13044338866170580992UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndReadRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndReadRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndReadRangeActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<RangeResult> waitAndReadRange( KeyValueStoreMemory* const& self, KeyRange const& keys, int const& rowLimit, int const& byteLimit, Optional<ReadOptions> const& options ) {
															#line 1017 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<RangeResult>(new WaitAndReadRangeActor(self, keys, rowLimit, byteLimit, options));
															#line 3513 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
																#line 3517 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via waitAndCommit()
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class WaitAndCommitActor>
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndCommitActorState {
															#line 3523 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndCommitActorState(KeyValueStoreMemory* const& self,bool const& sequential) 
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   sequential(sequential)
															#line 3532 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("waitAndCommit", reinterpret_cast<unsigned long>(this));

	}
	~WaitAndCommitActorState() 
	{
		fdb_probe_actor_destroy("waitAndCommit", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1026 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->recovering;
															#line 1026 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3549 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 1;
															#line 1026 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitAndCommitActor, 0, Void >*>(static_cast<WaitAndCommitActor*>(this)));
															#line 3554 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitAndCommitActorState();
		static_cast<WaitAndCommitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1027 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->commit(sequential);
															#line 1027 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3579 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 2;
															#line 1027 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitAndCommitActor, 1, Void >*>(static_cast<WaitAndCommitActor*>(this)));
															#line 3584 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1027 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->commit(sequential);
															#line 1027 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3595 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 2;
															#line 1027 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitAndCommitActor, 1, Void >*>(static_cast<WaitAndCommitActor*>(this)));
															#line 3600 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state > 0) static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndCommitActor*>(this)->ActorCallback< WaitAndCommitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndCommitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitAndCommitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitAndCommitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1028 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndCommitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitAndCommitActorState(); static_cast<WaitAndCommitActor*>(this)->destroy(); return 0; }
															#line 3684 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<WaitAndCommitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitAndCommitActorState();
		static_cast<WaitAndCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1028 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<WaitAndCommitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitAndCommitActorState(); static_cast<WaitAndCommitActor*>(this)->destroy(); return 0; }
															#line 3696 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<WaitAndCommitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitAndCommitActorState();
		static_cast<WaitAndCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitAndCommitActor*>(this)->actor_wait_state > 0) static_cast<WaitAndCommitActor*>(this)->actor_wait_state = 0;
		static_cast<WaitAndCommitActor*>(this)->ActorCallback< WaitAndCommitActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitAndCommitActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitAndCommitActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitAndCommitActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	bool sequential;
															#line 3783 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via waitAndCommit()
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class WaitAndCommitActor final : public Actor<Void>, public ActorCallback< WaitAndCommitActor, 0, Void >, public ActorCallback< WaitAndCommitActor, 1, Void >, public FastAllocated<WaitAndCommitActor>, public WaitAndCommitActorState<WaitAndCommitActor> {
															#line 3788 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<WaitAndCommitActor>::operator new;
	using FastAllocated<WaitAndCommitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3949170071466462464UL, 12551930479255635200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitAndCommitActor, 0, Void >;
friend struct ActorCallback< WaitAndCommitActor, 1, Void >;
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	WaitAndCommitActor(KeyValueStoreMemory* const& self,bool const& sequential) 
															#line 3806 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Void>(),
		   WaitAndCommitActorState<WaitAndCommitActor>(self, sequential),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitAndCommit", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2545403049674954496UL, 15925460025590017024UL);
		ActorExecutionContextHelper __helper(static_cast<WaitAndCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitAndCommit");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitAndCommit", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitAndCommitActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitAndCommitActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> waitAndCommit( KeyValueStoreMemory* const& self, bool const& sequential ) {
															#line 1025 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new WaitAndCommitActor(self, sequential));
															#line 3839 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
																#line 3843 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via commitAndUpdateVersions()
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class CommitAndUpdateVersionsActor>
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class CommitAndUpdateVersionsActorState {
															#line 3849 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	CommitAndUpdateVersionsActorState(KeyValueStoreMemory* const& self,Future<Void> const& commit,IDiskQueue::location const& location) 
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self),
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   commit(commit),
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		   location(location)
															#line 3860 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this));

	}
	~CommitAndUpdateVersionsActorState() 
	{
		fdb_probe_actor_destroy("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1033 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<Void> __when_expr_0 = commit;
															#line 1033 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<CommitAndUpdateVersionsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3877 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CommitAndUpdateVersionsActor*>(this)->actor_wait_state = 1;
															#line 1033 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*>(static_cast<CommitAndUpdateVersionsActor*>(this)));
															#line 3882 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CommitAndUpdateVersionsActorState();
		static_cast<CommitAndUpdateVersionsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1034 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->log->pop(location);
															#line 1035 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<CommitAndUpdateVersionsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitAndUpdateVersionsActorState(); static_cast<CommitAndUpdateVersionsActor*>(this)->destroy(); return 0; }
															#line 3907 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<CommitAndUpdateVersionsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CommitAndUpdateVersionsActorState();
		static_cast<CommitAndUpdateVersionsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1034 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->log->pop(location);
															#line 1035 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<CommitAndUpdateVersionsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitAndUpdateVersionsActorState(); static_cast<CommitAndUpdateVersionsActor*>(this)->destroy(); return 0; }
															#line 3921 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<CommitAndUpdateVersionsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CommitAndUpdateVersionsActorState();
		static_cast<CommitAndUpdateVersionsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CommitAndUpdateVersionsActor*>(this)->actor_wait_state > 0) static_cast<CommitAndUpdateVersionsActor*>(this)->actor_wait_state = 0;
		static_cast<CommitAndUpdateVersionsActor*>(this)->ActorCallback< CommitAndUpdateVersionsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CommitAndUpdateVersionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CommitAndUpdateVersionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CommitAndUpdateVersionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	Future<Void> commit;
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	IDiskQueue::location location;
															#line 4010 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via commitAndUpdateVersions()
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class CommitAndUpdateVersionsActor final : public Actor<Void>, public ActorCallback< CommitAndUpdateVersionsActor, 0, Void >, public FastAllocated<CommitAndUpdateVersionsActor>, public CommitAndUpdateVersionsActorState<CommitAndUpdateVersionsActor> {
															#line 4015 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<CommitAndUpdateVersionsActor>::operator new;
	using FastAllocated<CommitAndUpdateVersionsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8244092255721425920UL, 3385717809184010496UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CommitAndUpdateVersionsActor, 0, Void >;
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	CommitAndUpdateVersionsActor(KeyValueStoreMemory* const& self,Future<Void> const& commit,IDiskQueue::location const& location) 
															#line 4032 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Void>(),
		   CommitAndUpdateVersionsActorState<CommitAndUpdateVersionsActor>(self, commit, location),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5732578105651983360UL, 2263904282379971584UL);
		ActorExecutionContextHelper __helper(static_cast<CommitAndUpdateVersionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("commitAndUpdateVersions");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("commitAndUpdateVersions", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CommitAndUpdateVersionsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> commitAndUpdateVersions( KeyValueStoreMemory* const& self, Future<Void> const& commit, IDiskQueue::location const& location ) {
															#line 1030 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new CommitAndUpdateVersionsActor(self, commit, location));
															#line 4064 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 1037 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"

																#line 4069 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via updateCipherKeys()
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class UpdateCipherKeysActor>
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class UpdateCipherKeysActorState {
															#line 4075 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	UpdateCipherKeysActorState(KeyValueStoreMemory* const& self) 
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self)
															#line 4082 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("updateCipherKeys", reinterpret_cast<unsigned long>(this));

	}
	~UpdateCipherKeysActorState() 
	{
		fdb_probe_actor_destroy("updateCipherKeys", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1039 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			StrictFuture<TextAndHeaderCipherKeys> __when_expr_0 = GetEncryptCipherKeys<ServerDBInfo>::getLatestSystemEncryptCipherKeys( self->db, BlobCipherMetrics::KV_MEMORY);
															#line 1039 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			if (static_cast<UpdateCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4099 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateCipherKeysActor*>(this)->actor_wait_state = 1;
															#line 1039 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateCipherKeysActor, 0, TextAndHeaderCipherKeys >*>(static_cast<UpdateCipherKeysActor*>(this)));
															#line 4104 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateCipherKeysActorState();
		static_cast<UpdateCipherKeysActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(TextAndHeaderCipherKeys const& cipherKeys,int loopDepth) 
	{
															#line 1041 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->cipherKeys = cipherKeys;
															#line 1042 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<UpdateCipherKeysActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateCipherKeysActorState(); static_cast<UpdateCipherKeysActor*>(this)->destroy(); return 0; }
															#line 4129 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<UpdateCipherKeysActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateCipherKeysActorState();
		static_cast<UpdateCipherKeysActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(TextAndHeaderCipherKeys && cipherKeys,int loopDepth) 
	{
															#line 1041 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		self->cipherKeys = cipherKeys;
															#line 1042 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (!static_cast<UpdateCipherKeysActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateCipherKeysActorState(); static_cast<UpdateCipherKeysActor*>(this)->destroy(); return 0; }
															#line 4143 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		new (&static_cast<UpdateCipherKeysActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateCipherKeysActorState();
		static_cast<UpdateCipherKeysActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(TextAndHeaderCipherKeys const& cipherKeys,int loopDepth) 
	{
		loopDepth = a_body1cont1(cipherKeys, loopDepth);

		return loopDepth;
	}
	int a_body1when1(TextAndHeaderCipherKeys && cipherKeys,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(cipherKeys), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateCipherKeysActor*>(this)->actor_wait_state > 0) static_cast<UpdateCipherKeysActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateCipherKeysActor*>(this)->ActorCallback< UpdateCipherKeysActor, 0, TextAndHeaderCipherKeys >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateCipherKeysActor, 0, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys const& value) 
	{
		fdb_probe_actor_enter("updateCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateCipherKeysActor, 0, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys && value) 
	{
		fdb_probe_actor_enter("updateCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateCipherKeysActor, 0, TextAndHeaderCipherKeys >*,Error err) 
	{
		fdb_probe_actor_enter("updateCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 4228 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via updateCipherKeys()
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class UpdateCipherKeysActor final : public Actor<Void>, public ActorCallback< UpdateCipherKeysActor, 0, TextAndHeaderCipherKeys >, public FastAllocated<UpdateCipherKeysActor>, public UpdateCipherKeysActorState<UpdateCipherKeysActor> {
															#line 4233 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<UpdateCipherKeysActor>::operator new;
	using FastAllocated<UpdateCipherKeysActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1477356905995963392UL, 15957312430088878848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateCipherKeysActor, 0, TextAndHeaderCipherKeys >;
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	UpdateCipherKeysActor(KeyValueStoreMemory* const& self) 
															#line 4250 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateCipherKeysActorState<UpdateCipherKeysActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateCipherKeys", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15712068675808657664UL, 3854966057158990080UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateCipherKeys");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateCipherKeys", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateCipherKeysActor, 0, TextAndHeaderCipherKeys >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> updateCipherKeys( KeyValueStoreMemory* const& self ) {
															#line 1038 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new UpdateCipherKeysActor(self));
															#line 4282 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 1044 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"

	// TODO(yiwu): Implement background refresh mechanism for BlobCipher and use that mechanism to refresh cipher key.
																#line 4288 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
// This generated class is to be used only via refreshCipherKeys()
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
template <class RefreshCipherKeysActor>
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class RefreshCipherKeysActorState {
															#line 4294 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	RefreshCipherKeysActorState(KeyValueStoreMemory* const& self) 
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		 : self(self)
															#line 4301 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
	{
		fdb_probe_actor_create("refreshCipherKeys", reinterpret_cast<unsigned long>(this));

	}
	~RefreshCipherKeysActorState() 
	{
		fdb_probe_actor_destroy("refreshCipherKeys", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1047 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
			;
															#line 4316 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RefreshCipherKeysActorState();
		static_cast<RefreshCipherKeysActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1048 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_0 = updateCipherKeys(self);
															#line 1048 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4348 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state = 1;
															#line 1048 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RefreshCipherKeysActor, 0, Void >*>(static_cast<RefreshCipherKeysActor*>(this)));
															#line 4353 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1049 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
															#line 1049 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4364 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state = 2;
															#line 1049 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RefreshCipherKeysActor, 1, Void >*>(static_cast<RefreshCipherKeysActor*>(this)));
															#line 4369 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1049 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
															#line 1049 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		if (static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4380 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state = 2;
															#line 1049 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RefreshCipherKeysActor, 1, Void >*>(static_cast<RefreshCipherKeysActor*>(this)));
															#line 4385 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state > 0) static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshCipherKeysActor*>(this)->ActorCallback< RefreshCipherKeysActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshCipherKeysActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RefreshCipherKeysActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RefreshCipherKeysActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state > 0) static_cast<RefreshCipherKeysActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshCipherKeysActor*>(this)->ActorCallback< RefreshCipherKeysActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshCipherKeysActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RefreshCipherKeysActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RefreshCipherKeysActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshCipherKeys", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	KeyValueStoreMemory* self;
															#line 4554 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
};
// This generated class is to be used only via refreshCipherKeys()
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
class RefreshCipherKeysActor final : public Actor<Void>, public ActorCallback< RefreshCipherKeysActor, 0, Void >, public ActorCallback< RefreshCipherKeysActor, 1, Void >, public FastAllocated<RefreshCipherKeysActor>, public RefreshCipherKeysActorState<RefreshCipherKeysActor> {
															#line 4559 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
public:
	using FastAllocated<RefreshCipherKeysActor>::operator new;
	using FastAllocated<RefreshCipherKeysActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17268956155072303104UL, 8543242843905270528UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RefreshCipherKeysActor, 0, Void >;
friend struct ActorCallback< RefreshCipherKeysActor, 1, Void >;
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	RefreshCipherKeysActor(KeyValueStoreMemory* const& self) 
															#line 4577 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
		 : Actor<Void>(),
		   RefreshCipherKeysActorState<RefreshCipherKeysActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("refreshCipherKeys", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14668996033429024256UL, 9771597919636766720UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("refreshCipherKeys");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("refreshCipherKeys", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RefreshCipherKeysActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RefreshCipherKeysActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
[[nodiscard]] static Future<Void> refreshCipherKeys( KeyValueStoreMemory* const& self ) {
															#line 1046 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
	return Future<Void>(new RefreshCipherKeysActor(self));
															#line 4610 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreMemory.actor.g.cpp"
}

#line 1052 "/codebuild/output/src1143352350/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreMemory.actor.cpp"
};

template <typename Container>
KeyValueStoreMemory<Container>::KeyValueStoreMemory(IDiskQueue* log,
                                                    Reference<AsyncVar<ServerDBInfo> const> db,
                                                    UID id,
                                                    int64_t memoryLimit,
                                                    KeyValueStoreType storeType,
                                                    bool disableSnapshot,
                                                    bool replaceContent,
                                                    bool exactRecovery,
                                                    bool enableEncryption)
  : type(storeType), id(id), log(log), db(db), committedWriteBytes(0), overheadWriteBytes(0), currentSnapshotEnd(-1),
    previousSnapshotEnd(-1), committedDataSize(0), transactionSize(0), transactionIsLarge(false), resetSnapshot(false),
    disableSnapshot(disableSnapshot), replaceContent(replaceContent), firstCommitWithSnapshot(true), snapshotCount(0),
    memoryLimit(memoryLimit), enableEncryption(enableEncryption) {
	// create reserved buffer for radixtree store type
	this->reserved_buffer =
	    (storeType == KeyValueStoreType::MEMORY) ? nullptr : new uint8_t[CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT];
	if (this->reserved_buffer != nullptr)
		memset(this->reserved_buffer, 0, CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);

	recovering = recover(this, exactRecovery);
	snapshotting = snapshot(this);
	commitActors = actorCollection(addActor.getFuture());
	if (enableEncryption) {
		refreshCipherKeysActor = refreshCipherKeys(this);
	}
}

IKeyValueStore* keyValueStoreMemory(std::string const& basename,
                                    UID logID,
                                    int64_t memoryLimit,
                                    std::string ext,
                                    KeyValueStoreType storeType) {
	TraceEvent("KVSMemOpening", logID)
	    .detail("Basename", basename)
	    .detail("MemoryLimit", memoryLimit)
	    .detail("StoreType", storeType);

	// Use DiskQueueVersion::V2 with xxhash3 checksum
	IDiskQueue* log = openDiskQueue(basename, ext, logID, DiskQueueVersion::V2);
	if (storeType == KeyValueStoreType::MEMORY_RADIXTREE) {
		return new KeyValueStoreMemory<radix_tree>(
		    log, Reference<AsyncVar<ServerDBInfo> const>(), logID, memoryLimit, storeType, false, false, false, false);
	} else {
		return new KeyValueStoreMemory<IKeyValueContainer>(
		    log, Reference<AsyncVar<ServerDBInfo> const>(), logID, memoryLimit, storeType, false, false, false, false);
	}
}

IKeyValueStore* keyValueStoreLogSystem(class IDiskQueue* queue,
                                       Reference<AsyncVar<ServerDBInfo> const> db,
                                       UID logID,
                                       int64_t memoryLimit,
                                       bool disableSnapshot,
                                       bool replaceContent,
                                       bool exactRecovery,
                                       bool enableEncryption) {
	// ServerDBInfo is required if encryption is to be enabled, or the KV store instance have been encrypted.
	ASSERT(!enableEncryption || db.isValid());
	return new KeyValueStoreMemory<IKeyValueContainer>(queue,
	                                                   db,
	                                                   logID,
	                                                   memoryLimit,
	                                                   KeyValueStoreType::MEMORY,
	                                                   disableSnapshot,
	                                                   replaceContent,
	                                                   exactRecovery,
	                                                   enableEncryption);
}
