#
# generate_options.py
#
# This source file is part of the FoundationDB open source project
#
# Copyright 2016-2025 Apple Inc. and the FoundationDB project authors
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

'''
This script generates the Options file for the Swift FoundationDB binding.
'''

from vexillographer import Vexillographer

class SwiftEmitter(object):
    '''
    This class provides an emitter for generating the Swift options file.
    '''

    def __init__(self, output_file):
        '''
        This initializer creates the Swift emitter.

        :param output_file: The file that we are printing into.
        '''

        self.output_file = output_file

    def print_header_warning(self):
        '''
        This method prints the warning at the top of the file that tells people
        not to edit it by hand.
        '''

        self.output_file.write(
            """/*
 * Options.swift
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2016-2025 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// WARNING: This file is automatically generated, and must not be edited by hand.

extension FDB {
"""
        )

    def print_footer(self):
        '''
        This method prints the warning at the end of the file.
        '''
        self.output_file.write("}")

    def get_comment(self, enum_name):
        '''
        This method gets the documentation comment for an enum type.

        :param enum_name: The name of the enum
        :returns: The text for the comment
        '''

        if enum_name == 'NetworkOption':
            return "A set of options that can be set globally for the FoundationDB API."
        elif enum_name == 'DatabaseOption':
            return "A set of options that can be set on a database."
        elif enum_name == 'TransactionOption':
            return "A set of options that can be set on a transaction."
        elif enum_name == 'StreamingMode':
            return "Options that control the way the binding performs range reads."
        elif enum_name == 'MutationType':
            return "A set of operations that can be performed atomically on a database."
        elif enum_name == 'ConflictRangeType':
            return "Conflict range types used internally by the C API."
        elif enum_name == 'ErrorPredicate':
            return "Error code predicates for binding writers and non-standard layer implementers."
        else:
            return ""

    def write_scope_start(self, name, signed):
        '''
        This method writes the beginning of the type containing a kind of
        option.

        :param name: The name of the scope
        :param signed: Whether these options can have negative values.
        '''

        self.output_file.write("/** %s */\n" % self.get_comment(name))
        enum_type = 'Int32' if signed else 'UInt32'
        self.output_file.write("public enum %s: %s {\n" % (name, enum_type))

    def write_scope_end(self):
        '''
        This method writes the end of the type containing a kind of option.
        '''

        self.output_file.write("}\n\n")

    def write_option(self, name, code, description, scope_name, deprecated):
        '''
        This method writes a single option.

        :param scope_name: The name of the type that this option is part of.
        :param name: The name of the option.
        :param code: The numeric value for the option.
        :param description: A comment describing the option.
        :param deprecated: Whether the option has been deprecated.
        '''

        components = name.split('_')
        name = components[0].lower() + "".join([
            component.title() for component in components[1:]
        ])
        if deprecated:
            self.output_file.write("\t@available(*, deprecated)\n")
            self.output_file.write(
                "\tpublic static let %s = %s(rawValue: %d)\n\n" % (name, scope_name, code)
            )
        else:
            if len(description) > 0:
                self.output_file.write("\t/** %s */\n" % description)
            self.output_file.write("\tcase %s = %d\n\n" % (name, code))

if __name__ == '__main__':
    Vexillographer.run(SwiftEmitter)
