#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
/*
 * TransactionTagCounter.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/NativeAPI.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/TransactionTagCounter.h"
#include "flow/Trace.h"
#include "flow/actorcompiler.h"

class TransactionTagCounterImpl {
	UID thisServerID;
	TransactionTagMap<double> intervalCosts;
	double intervalTotalCost = 0;
	double intervalStart = 0;
	int maxTagsTracked;
	double minRateTracked;

	std::vector<BusyTagInfo> previousBusiestTags;
	Reference<EventCacheHolder> busiestReadTagEventHolder;

	std::vector<BusyTagInfo> getBusiestTagsFromLastInterval(double elapsed) const {
		std::priority_queue<BusyTagInfo, std::vector<BusyTagInfo>, std::greater<BusyTagInfo>> topKTags;
		for (auto const& [tag, cost] : intervalCosts) {
			auto const rate = cost / elapsed;
			auto const fractionalBusyness = std::min(1.0, cost / intervalTotalCost);
			if (rate < minRateTracked) {
				continue;
			} else if (topKTags.size() < maxTagsTracked) {
				topKTags.emplace(tag, rate, fractionalBusyness);
			} else if (topKTags.top().rate < rate) {
				topKTags.pop();
				topKTags.emplace(tag, rate, fractionalBusyness);
			}
		}
		std::vector<BusyTagInfo> result;
		while (!topKTags.empty()) {
			result.push_back(std::move(topKTags.top()));
			topKTags.pop();
		}
		return result;
	}

public:
	TransactionTagCounterImpl(UID thisServerID, int maxTagsTracked, double minRateTracked)
	  : thisServerID(thisServerID), maxTagsTracked(maxTagsTracked), minRateTracked(minRateTracked),
	    busiestReadTagEventHolder(makeReference<EventCacheHolder>(thisServerID.toString() + "/BusiestReadTag")) {}

	void addRequest(Optional<TagSet> const& tags, int64_t bytes) {
		auto const cost = getReadOperationCost(bytes);
		intervalTotalCost += cost;
		if (tags.present()) {
			for (auto const& tag : tags.get()) {
				CODE_PROBE(true, "Tracking transaction tag in TransactionTagCounter");
				intervalCosts[TransactionTag(tag, tags.get().getArena())] += cost / CLIENT_KNOBS->READ_TAG_SAMPLE_RATE;
			}
		}
	}

	void startNewInterval() {
		double elapsed = now() - intervalStart;
		previousBusiestTags.clear();
		if (intervalStart > 0 && CLIENT_KNOBS->READ_TAG_SAMPLE_RATE > 0 && elapsed > 0) {
			previousBusiestTags = getBusiestTagsFromLastInterval(elapsed);

			// For status, report the busiest tag:
			if (previousBusiestTags.empty()) {
				TraceEvent("BusiestReadTag", thisServerID).detail("TagCost", 0.0);
			} else {
				auto busiestTagInfo = previousBusiestTags[0];
				for (int i = 1; i < previousBusiestTags.size(); ++i) {
					auto const& tagInfo = previousBusiestTags[i];
					if (tagInfo.rate > busiestTagInfo.rate) {
						busiestTagInfo = tagInfo;
					}
				}
				TraceEvent("BusiestReadTag", thisServerID)
				    .detail("Tag", printable(busiestTagInfo.tag))
				    .detail("TagCost", busiestTagInfo.rate)
				    .detail("FractionalBusyness", busiestTagInfo.fractionalBusyness);
			}

			for (const auto& tagInfo : previousBusiestTags) {
				TraceEvent("BusyReadTag", thisServerID)
				    .detail("Tag", printable(tagInfo.tag))
				    .detail("TagCost", tagInfo.rate)
				    .detail("FractionalBusyness", tagInfo.fractionalBusyness);
			}
		}

		intervalCosts.clear();
		intervalTotalCost = 0;
		intervalStart = now();
	}

	std::vector<BusyTagInfo> const& getBusiestTags() const { return previousBusiestTags; }
};

TransactionTagCounter::TransactionTagCounter(UID thisServerID, int maxTagsTracked, double minRateTracked)
  : impl(PImpl<TransactionTagCounterImpl>::create(thisServerID, maxTagsTracked, minRateTracked)) {}

TransactionTagCounter::~TransactionTagCounter() = default;

void TransactionTagCounter::addRequest(Optional<TagSet> const& tags, int64_t bytes) {
	return impl->addRequest(tags, bytes);
}

void TransactionTagCounter::startNewInterval() {
	return impl->startNewInterval();
}

std::vector<BusyTagInfo> const& TransactionTagCounter::getBusiestTags() const {
	return impl->getBusiestTags();
}

namespace {

bool containsTag(std::vector<BusyTagInfo> const& busyTags, TransactionTagRef tag) {
	return std::count_if(busyTags.begin(), busyTags.end(), [tag](auto const& tagInfo) { return tagInfo.tag == tag; }) ==
	       1;
}

TagSet getTagSet(TransactionTagRef tag) {
	TagSet result;
	result.addTag(tag);
	return result;
}

} // namespace

															#line 149 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase147()
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
template <class FlowTestCase147Actor>
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
class FlowTestCase147ActorState {
															#line 156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
public:
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	FlowTestCase147ActorState(UnitTestParameters const& params) 
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		 : params(params),
															#line 148 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		   counter(UID(), 2, 10.0 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE / CLIENT_KNOBS->READ_TAG_SAMPLE_RATE)
															#line 165 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase147", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase147ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase147", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 152 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
			counter.startNewInterval();
															#line 153 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
			ASSERT_EQ(counter.getBusiestTags().size(), 0);
															#line 182 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
			{
															#line 155 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
				StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 155 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
				if (static_cast<FlowTestCase147Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 188 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FlowTestCase147Actor*>(this)->actor_wait_state = 1;
															#line 155 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase147Actor, 0, Void >*>(static_cast<FlowTestCase147Actor*>(this)));
															#line 193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase147ActorState();
		static_cast<FlowTestCase147Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 167 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		if (!static_cast<FlowTestCase147Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase147ActorState(); static_cast<FlowTestCase147Actor*>(this)->destroy(); return 0; }
															#line 217 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
		new (&static_cast<FlowTestCase147Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase147ActorState();
		static_cast<FlowTestCase147Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagA"_sr), 10 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 157 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagA"_sr), 10 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagB"_sr), 15 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagC"_sr), 20 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.startNewInterval();
															#line 161 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		auto const busiestTags = counter.getBusiestTags();
															#line 162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT_EQ(busiestTags.size(), 2);
															#line 163 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT(containsTag(busiestTags, "tagA"_sr));
															#line 164 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT(!containsTag(busiestTags, "tagB"_sr));
															#line 165 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT(containsTag(busiestTags, "tagC"_sr));
															#line 247 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagA"_sr), 10 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 157 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagA"_sr), 10 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagB"_sr), 15 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagC"_sr), 20 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.startNewInterval();
															#line 161 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		auto const busiestTags = counter.getBusiestTags();
															#line 162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT_EQ(busiestTags.size(), 2);
															#line 163 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT(containsTag(busiestTags, "tagA"_sr));
															#line 164 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT(!containsTag(busiestTags, "tagB"_sr));
															#line 165 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT(containsTag(busiestTags, "tagC"_sr));
															#line 274 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase147Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase147Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase147Actor*>(this)->ActorCallback< FlowTestCase147Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase147Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase147", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase147Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase147", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase147Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase147", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase147Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase147", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase147Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase147", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase147Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase147", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	UnitTestParameters params;
															#line 148 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	TransactionTagCounter counter;
															#line 358 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase147()
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
class FlowTestCase147Actor final : public Actor<Void>, public ActorCallback< FlowTestCase147Actor, 0, Void >, public FastAllocated<FlowTestCase147Actor>, public FlowTestCase147ActorState<FlowTestCase147Actor> {
															#line 363 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase147Actor>::operator new;
	using FastAllocated<FlowTestCase147Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6126632016679082752UL, 2291431803158066944UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase147Actor, 0, Void >;
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	FlowTestCase147Actor(UnitTestParameters const& params) 
															#line 380 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase147ActorState<FlowTestCase147Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase147", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12081458885367482368UL, 14967863624694478336UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase147Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase147");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase147", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase147Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
static Future<Void> flowTestCase147( UnitTestParameters const& params ) {
															#line 147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	return Future<Void>(new FlowTestCase147Actor(params));
															#line 413 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase147, "/fdbserver/TransactionTagCounter/IgnoreBeyondMaxTags")

#line 169 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"

															#line 419 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase170()
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
template <class FlowTestCase170Actor>
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
class FlowTestCase170ActorState {
															#line 426 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
public:
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	FlowTestCase170ActorState(UnitTestParameters const& params) 
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		 : params(params),
															#line 171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		   counter(UID(), 2, 10.0 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE / CLIENT_KNOBS->READ_TAG_SAMPLE_RATE)
															#line 435 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase170", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase170ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase170", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 175 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
			counter.startNewInterval();
															#line 176 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
			ASSERT_EQ(counter.getBusiestTags().size(), 0);
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
			{
															#line 178 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
				StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 178 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
				if (static_cast<FlowTestCase170Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 458 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FlowTestCase170Actor*>(this)->actor_wait_state = 1;
															#line 178 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase170Actor, 0, Void >*>(static_cast<FlowTestCase170Actor*>(this)));
															#line 463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase170ActorState();
		static_cast<FlowTestCase170Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		if (!static_cast<FlowTestCase170Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase170ActorState(); static_cast<FlowTestCase170Actor*>(this)->destroy(); return 0; }
															#line 487 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
		new (&static_cast<FlowTestCase170Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase170ActorState();
		static_cast<FlowTestCase170Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 179 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagA"_sr), 5 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 180 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.startNewInterval();
															#line 181 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		auto const busiestTags = counter.getBusiestTags();
															#line 182 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT_EQ(busiestTags.size(), 0);
															#line 505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 179 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.addRequest(getTagSet("tagA"_sr), 5 * CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE);
															#line 180 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		counter.startNewInterval();
															#line 181 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		auto const busiestTags = counter.getBusiestTags();
															#line 182 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
		ASSERT_EQ(busiestTags.size(), 0);
															#line 520 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase170Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase170Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase170Actor*>(this)->ActorCallback< FlowTestCase170Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase170Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase170", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase170Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase170", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase170Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase170", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase170Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase170", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase170Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase170", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase170Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase170", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	UnitTestParameters params;
															#line 171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	TransactionTagCounter counter;
															#line 604 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase170()
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
class FlowTestCase170Actor final : public Actor<Void>, public ActorCallback< FlowTestCase170Actor, 0, Void >, public FastAllocated<FlowTestCase170Actor>, public FlowTestCase170ActorState<FlowTestCase170Actor> {
															#line 609 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase170Actor>::operator new;
	using FastAllocated<FlowTestCase170Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16360701599388247296UL, 7120199409076237824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase170Actor, 0, Void >;
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	FlowTestCase170Actor(UnitTestParameters const& params) 
															#line 626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase170ActorState<FlowTestCase170Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase170", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16263197351376690432UL, 4393530446601720320UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase170Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase170");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase170", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase170Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
static Future<Void> flowTestCase170( UnitTestParameters const& params ) {
															#line 170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
	return Future<Void>(new FlowTestCase170Actor(params));
															#line 659 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/TransactionTagCounter.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase170, "/fdbserver/TransactionTagCounter/IgnoreBelowMinRate")

#line 186 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/TransactionTagCounter.actor.cpp"
