#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
/*
 * StorageCache.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/BlobCipher.h"
#include "fdbclient/FDBTypes.h"
#include "fdbserver/OTELSpanContextMessage.h"
#include "flow/Arena.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/ServerDBInfo.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/VersionedMap.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/Atomic.h"
#include "fdbclient/Notified.h"
#include "fdbserver/LogProtocolMessage.h"
#include "fdbserver/LogSystem.h"
#include "fdbserver/MutationTracking.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/NativeAPI.actor.h"
#include "flow/Trace.h"
#include "flow/actorcompiler.h" // This must be the last #include.

// TODO storageCache server shares quite a bit of storageServer functionality, although simplified
// Need to look into refactoring common code out for better code readability and to avoid duplication

namespace {
// TODO rename wrong_shard_server error to wrong_cache_server
bool canReplyWith(Error e) {
	switch (e.code()) {
	case error_code_transaction_too_old:
	case error_code_future_version:
	case error_code_wrong_shard_server:
	case error_code_cold_cache_server:
	case error_code_process_behind:
		// case error_code_all_alternatives_failed:
		return true;
	default:
		return false;
	};
}
} // namespace

class StorageCacheUpdater;

struct AddingCacheRange : NonCopyable {
	KeyRange keys;
	Future<Void> fetchClient; // holds FetchKeys() actor
	Promise<Void> fetchComplete;
	Promise<Void> readWrite;

	std::deque<Standalone<VerUpdateRef>>
	    updates; // during the Fetching phase, mutations with key in keys and version>=(fetchClient's) fetchVersion;

	struct StorageCacheData* server;
	Version transferredVersion;

	enum Phase { WaitPrevious, Fetching, Waiting };

	Phase phase;

	AddingCacheRange(StorageCacheData* server, KeyRangeRef const& keys);

	~AddingCacheRange() {
		if (!fetchComplete.isSet())
			fetchComplete.send(Void());
		if (!readWrite.isSet())
			readWrite.send(Void());
	}

	void addMutation(Version version, MutationRef const& mutation);

	bool isTransferred() const { return phase == Waiting; }
};

class CacheRangeInfo : public ReferenceCounted<CacheRangeInfo>, NonCopyable {
	CacheRangeInfo(KeyRange keys, std::unique_ptr<AddingCacheRange>&& adding, StorageCacheData* readWrite)
	  : adding(std::move(adding)), readWrite(readWrite), keys(keys) {}

public:
	std::unique_ptr<AddingCacheRange> adding;
	struct StorageCacheData* readWrite;
	KeyRange keys;
	uint64_t changeCounter;

	static CacheRangeInfo* newNotAssigned(KeyRange keys) { return new CacheRangeInfo(keys, nullptr, nullptr); }
	static CacheRangeInfo* newReadWrite(KeyRange keys, StorageCacheData* data) {
		return new CacheRangeInfo(keys, nullptr, data);
	}
	static CacheRangeInfo* newAdding(StorageCacheData* data, KeyRange keys) {
		return new CacheRangeInfo(keys, std::make_unique<AddingCacheRange>(data, keys), nullptr);
	}

	bool isReadable() const { return readWrite != nullptr; }
	bool isAdding() const { return adding != nullptr; }
	bool notAssigned() const { return !readWrite && !adding; }
	bool assigned() const { return readWrite || adding; }
	bool isInVersionedData() const { return readWrite || (adding && adding->isTransferred()); }
	void addMutation(Version version, MutationRef const& mutation);
	bool isFetched() const { return readWrite || (adding && adding->fetchComplete.isSet()); }

	const char* debugDescribeState() const {
		if (notAssigned())
			return "NotAssigned";
		else if (adding && !adding->isTransferred())
			return "AddingFetching";
		else if (adding)
			return "AddingTransferred";
		else
			return "ReadWrite";
	}
};

const int VERSION_OVERHEAD =
    64 + sizeof(Version) + sizeof(Standalone<VerUpdateRef>) + // mutationLog, 64b overhead for map
    2 * (64 + sizeof(Version) +
         sizeof(
             Reference<VersionedMap<KeyRef,
                                    ValueOrClearToRef>::PTreeT>)); // versioned map [ x2 for createNewVersion(version+1)
                                                                   // ], 64b overhead for map
static int mvccStorageBytes(MutationRef const& m) {
	return VersionedMap<KeyRef, ValueOrClearToRef>::overheadPerItem * 2 +
	       (MutationRef::OVERHEAD_BYTES + m.param1.size() + m.param2.size()) * 2;
}

struct FetchInjectionInfo {
	Arena arena;
	std::vector<VerUpdateRef> changes;
};

struct StorageCacheData {
	typedef VersionedMap<KeyRef, ValueOrClearToRef> VersionedData;
	// typedef VersionedMap<KeyRef, ValueOrClearToRef, FastAllocPTree<KeyRef>> VersionedData;
private:
	// in-memory versioned struct (PTree as of now. Subject to change)
	VersionedData versionedData;
	// in-memory mutationLog that the versionedData contains references to
	// TODO change it to a deque, already contains mutations in version order
	std::map<Version, Standalone<VerUpdateRef>> mutationLog; // versions (durableVersion, version]

public:
	UID thisServerID; // unique id
	uint16_t index; // server index
	ProtocolVersion logProtocol;
	Reference<ILogSystem> logSystem;
	Key ck; // cacheKey
	Reference<AsyncVar<ServerDBInfo> const> db;
	Database cx;
	StorageCacheUpdater* updater;

	// KeyRangeMap <bool> cachedRangeMap; // map of cached key-ranges
	KeyRangeMap<Reference<CacheRangeInfo>> cachedRangeMap; // map of cached key-ranges
	uint64_t cacheRangeChangeCounter; // Max( CacheRangeInfo->changecounter )

	// TODO Add cache metrics, such as available memory/in-use memory etc to help dat adistributor assign cached ranges
	// StorageCacheMetrics metrics;

	// newestAvailableVersion[k]
	//   == invalidVersion -> k is unavailable at all versions
	//   <= compactVersion -> k is unavailable at all versions
	//   == v              -> k is readable (from versionedData) @ (oldestVersion,v], and not being updated when version
	//   increases
	//   == latestVersion  -> k is readable (from versionedData) @ (oldestVersion,version.get()], and thus stays
	//   available when version increases
	CoalescedKeyRangeMap<Version> newestAvailableVersion;

	CoalescedKeyRangeMap<Version> newestDirtyVersion; // Similar to newestAvailableVersion, but includes (only) keys
	                                                  // that were only partly available (due to cancelled fetchKeys)

	// The following are in rough order from newest to oldest
	// TODO double check which ones we need for storageCache servers
	Version lastTLogVersion, lastVersionWithData;
	Version peekVersion; // version to peek the log at
	NotifiedVersion version; // current version i.e. the max version that can be read from the cache
	NotifiedVersion desiredOldestVersion; // oldestVersion can be increased to this after compaction
	NotifiedVersion oldestVersion; // Min version that might be read from the cache

	// TODO not really in use as of now. may need in some failure cases. Revisit and remove if no plausible use
	Future<Void> compactionInProgress;

	FlowLock updateVersionLock;
	FlowLock fetchKeysParallelismLock;
	std::vector<Promise<FetchInjectionInfo*>> readyFetchKeys;

	// TODO do we need otherError here?
	Promise<Void> otherError;
	Promise<Void> coreStarted;

	bool debug_inApplyUpdate;
	double debug_lastValidateTime;

	int64_t versionLag; // An estimate for how many versions it takes for the data to move from the logs to this cache
	                    // server
	bool behind;

	// TODO double check which ones we need for storageCache servers
	struct Counters {
		CounterCollection cc;
		Counter allQueries, getKeyQueries, getValueQueries, getRangeQueries, finishedQueries, rowsQueried, bytesQueried;
		Counter bytesInput, bytesFetched, mutationBytes; // Like bytesInput but without MVCC accounting
		Counter mutations, setMutations, clearRangeMutations, atomicMutations;
		Counter updateBatches, updateVersions;
		Counter loops;
		Counter readsRejected;

		// LatencyBands readLatencyBands;

		Counters(StorageCacheData* self)
		  : cc("StorageCacheServer", self->thisServerID.toString()), allQueries("QueryQueue", cc),
		    getKeyQueries("GetKeyQueries", cc), getValueQueries("GetValueQueries", cc),
		    getRangeQueries("GetRangeQueries", cc), finishedQueries("FinishedQueries", cc),
		    rowsQueried("RowsQueried", cc), bytesQueried("BytesQueried", cc), bytesInput("BytesInput", cc),
		    bytesFetched("BytesFetched", cc), mutationBytes("MutationBytes", cc), mutations("Mutations", cc),
		    setMutations("SetMutations", cc), clearRangeMutations("ClearRangeMutations", cc),
		    atomicMutations("AtomicMutations", cc), updateBatches("UpdateBatches", cc),
		    updateVersions("UpdateVersions", cc), loops("Loops", cc), readsRejected("ReadsRejected", cc) {
			specialCounter(cc, "LastTLogVersion", [self]() { return self->lastTLogVersion; });
			specialCounter(cc, "Version", [self]() { return self->version.get(); });
			specialCounter(cc, "VersionLag", [self]() { return self->versionLag; });
		}
	} counters;

	explicit StorageCacheData(UID thisServerID, uint16_t index, Reference<AsyncVar<ServerDBInfo> const> const& db)
	  : /*versionedData(FastAllocPTree<KeyRef>{std::make_shared<int>(0)}), */
	    thisServerID(thisServerID), index(index), logProtocol(0), db(db), cacheRangeChangeCounter(0),
	    lastTLogVersion(0), lastVersionWithData(0), peekVersion(0), compactionInProgress(Void()),
	    fetchKeysParallelismLock(SERVER_KNOBS->FETCH_KEYS_PARALLELISM_BYTES), debug_inApplyUpdate(false),
	    debug_lastValidateTime(0), versionLag(0), behind(false), counters(this) {
		version.initMetric("StorageCacheData.Version"_sr, counters.cc.getId());
		desiredOldestVersion.initMetric("StorageCacheData.DesriedOldestVersion"_sr, counters.cc.getId());
		oldestVersion.initMetric("StorageCacheData.OldestVersion"_sr, counters.cc.getId());

		newestAvailableVersion.insert(allKeys, invalidVersion);
		newestDirtyVersion.insert(allKeys, invalidVersion);
		addCacheRange(CacheRangeInfo::newNotAssigned(allKeys));
		cx = openDBOnServer(db, TaskPriority::DefaultEndpoint, LockAware::True);
	}

	// Puts the given cacheRange into cachedRangeMap.  The caller is responsible for adding cacheRanges
	//   for all ranges in cachedRangeMap.getAffectedRangesAfterInsertion(newCacheRange->keys)), because these
	//   cacheRanges are invalidated by the call.
	void addCacheRange(CacheRangeInfo* newCacheRange) {
		ASSERT(!newCacheRange->keys.empty());
		newCacheRange->changeCounter = ++cacheRangeChangeCounter;
		//TraceEvent(SevDebug, "AddCacheRange", this->thisServerID).detail("KeyBegin", newCacheRange->keys.begin).detail("KeyEnd", newCacheRange->keys.end).
		// detail("State", newCacheRange->isReadable() ? "Readable" : newCacheRange->notAssigned() ? "NotAssigned" :
		// "Adding").detail("Version", this->version.get());
		cachedRangeMap.insert(newCacheRange->keys, Reference<CacheRangeInfo>(newCacheRange));
	}
	void addMutation(KeyRangeRef const& cachedKeyRange, Version version, MutationRef const& mutation);
	void applyMutation(MutationRef const& m, Arena& arena, VersionedData& data);

	bool isReadable(KeyRangeRef const& keys) {
		auto cr = cachedRangeMap.intersectingRanges(keys);
		for (auto i = cr.begin(); i != cr.end(); ++i)
			if (!i->value()->isReadable())
				return false;
		return true;
	}

	void checkChangeCounter(uint64_t oldCacheRangeChangeCounter, KeyRef const& key) {
		if (oldCacheRangeChangeCounter != cacheRangeChangeCounter &&
		    cachedRangeMap[key]->changeCounter > oldCacheRangeChangeCounter) {
			// CODE_PROBE(true, "CacheRange change during getValueQ");
			// TODO: should we throw the cold_cache_server() error here instead?
			throw wrong_shard_server();
		}
	}

	void checkChangeCounter(uint64_t oldCacheRangeChangeCounter, KeyRangeRef const& keys) {
		if (oldCacheRangeChangeCounter != cacheRangeChangeCounter) {
			auto sh = cachedRangeMap.intersectingRanges(keys);
			for (auto i = sh.begin(); i != sh.end(); ++i)
				if (i->value()->changeCounter > oldCacheRangeChangeCounter) {
					// CODE_PROBE(true, "CacheRange change during range operation");
					//  TODO: should we throw the cold_cache_server() error here instead?
					throw wrong_shard_server();
				}
		}
	}

	Arena lastArena;
	std::map<Version, Standalone<VerUpdateRef>> const& getMutationLog() const { return mutationLog; }
	std::map<Version, Standalone<VerUpdateRef>>& getMutableMutationLog() { return mutationLog; }
	VersionedData const& data() const { return versionedData; }
	VersionedData& mutableData() { return versionedData; }

	Standalone<VerUpdateRef>& addVersionToMutationLog(Version v) {
		// return existing version...
		auto m = mutationLog.find(v);
		if (m != mutationLog.end())
			return m->second;

		// ...or create a new one
		auto& u = mutationLog[v];
		u.version = v;
		if (lastArena.getSize() >= 65536)
			lastArena = Arena(4096);
		u.arena() = lastArena;
		counters.bytesInput += VERSION_OVERHEAD;
		return u;
	}

	MutationRef addMutationToMutationLog(Standalone<VerUpdateRef>& mLV, MutationRef const& m) {
		// TODO find out more
		// byteSampleApplyMutation(m, mLV.version);
		counters.bytesInput += mvccStorageBytes(m);
		return mLV.push_back_deep(mLV.arena(), m);
	}
};
void applyMutation(StorageCacheUpdater* updater, StorageCacheData* data, MutationRef const& mutation, Version version);

/////////////////////////////////// Validation ///////////////////////////////////////
#pragma region Validation
bool validateCacheRange(StorageCacheData::VersionedData::ViewAtVersion const& view,
                        KeyRangeRef range,
                        Version version,
                        UID id,
                        Version minInsertVersion) {
	// * Nonoverlapping: No clear overlaps a set or another clear, or adjoins another clear.
	// * Old mutations are erased: All items in versionedData.atLatest() have insertVersion() > oldestVersion()

	//TraceEvent(SevDebug, "ValidateRange", id).detail("KeyBegin", range.begin).detail("KeyEnd", range.end).detail("Version", version);
	KeyRef k;
	bool ok = true;
	bool kIsClear = false;
	auto i = view.lower_bound(range.begin);
	if (i != view.begin())
		--i;
	for (; i != view.end() && i.key() < range.end; ++i) {
		// TODO revisit this check. there could be nodes in PTree that were inserted, but never updated. their
		// insertVersion thus maybe lower than the current oldest version of the versioned map
		// if (i.insertVersion() <= minInsertVersion)
		//	TraceEvent(SevError,"SCValidateCacheRange",id).detail("IKey", i.key()).detail("Version",
		// version).detail("InsertVersion", i.insertVersion()).detail("MinInsertVersion", minInsertVersion); ASSERT(
		// i.insertVersion() > minInsertVersion );
		if (kIsClear && i->isClearTo() ? i.key() <= k : i.key() < k) {
			TraceEvent(SevError, "SCInvalidRange", id)
			    .detail("Key1", k)
			    .detail("Key2", i.key())
			    .detail("Version", version);
			ok = false;
		}
		// ASSERT( i.key() >= k );
		kIsClear = i->isClearTo();
		k = kIsClear ? i->getEndKey() : i.key();
	}
	return ok;
}

void validate(StorageCacheData* data, bool force = false) {
	try {
		if (force || (EXPENSIVE_VALIDATION)) {
			data->newestAvailableVersion.validateCoalesced();
			data->newestDirtyVersion.validateCoalesced();

			for (auto range = data->cachedRangeMap.ranges().begin(); range != data->cachedRangeMap.ranges().end();
			     ++range) {
				ASSERT(range->value()->keys == range->range());
				ASSERT(!range->value()->keys.empty());
			}

			for (auto range = data->cachedRangeMap.ranges().begin(); range != data->cachedRangeMap.ranges().end();
			     ++range)
				if (range->value()->isReadable()) {
					auto ar = data->newestAvailableVersion.intersectingRanges(range->range());
					for (auto a = ar.begin(); a != ar.end(); ++a)
						ASSERT(a->value() == latestVersion);
				}

			// * versionedData contains versions [oldestVersion.get(), version.get()].  It might also contain later
			// versions if applyUpdate is on the stack.
			ASSERT(data->data().getOldestVersion() == data->oldestVersion.get());
			ASSERT(data->data().getLatestVersion() == data->version.get() ||
			       data->data().getLatestVersion() == data->version.get() + 1 ||
			       (data->debug_inApplyUpdate && data->data().getLatestVersion() > data->version.get()));

			auto latest = data->data().atLatest();

			latest.validate();
			validateCacheRange(latest, allKeys, data->version.get(), data->thisServerID, data->oldestVersion.get());

			data->debug_lastValidateTime = now();
			//TraceEvent(SevDebug, "SCValidationDone", data->thisServerID).detail("LastValidTime", data->debug_lastValidateTime);
		}
	} catch (...) {
		TraceEvent(SevError, "SCValidationFailure", data->thisServerID)
		    .detail("LastValidTime", data->debug_lastValidateTime);
		throw;
	}
}
#pragma endregion

///////////////////////////////////// Queries /////////////////////////////////
#pragma region Queries
															#line 420 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForVersion()
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class WaitForVersionActor>
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class WaitForVersionActorState {
															#line 427 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	WaitForVersionActorState(StorageCacheData* const& data,Version const& version) 
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : data(data),
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   version(version)
															#line 436 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForVersion", reinterpret_cast<unsigned long>(this));

	}
	~WaitForVersionActorState() 
	{
		fdb_probe_actor_destroy("waitForVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 420 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (version == latestVersion)
															#line 451 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 421 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				version = std::max(Version(1), data->version.get());
															#line 455 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 422 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (version < data->oldestVersion.get() || version <= 0)
															#line 459 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 423 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 424 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (version <= data->version.get())
															#line 469 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 425 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					if (!static_cast<WaitForVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorState(); static_cast<WaitForVersionActor*>(this)->destroy(); return 0; }
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
					new (&static_cast<WaitForVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
					this->~WaitForVersionActorState();
					static_cast<WaitForVersionActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 427 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (data->behind && version > data->version.get())
															#line 482 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 428 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch1(process_behind(), loopDepth);
															#line 486 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 431 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (deterministicRandom()->random01() < 0.001)
															#line 490 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 432 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				TraceEvent("WaitForVersion1000x").log();
															#line 494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 434 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 433 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<WaitForVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 500 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 441 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 504 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForVersionActor*>(this)->actor_wait_state = 1;
															#line 434 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionActor, 0, Void >*>(static_cast<WaitForVersionActor*>(this)));
															#line 441 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionActor, 1, Void >*>(static_cast<WaitForVersionActor*>(this)));
															#line 511 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForVersionActorState();
		static_cast<WaitForVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 437 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (version < data->oldestVersion.get())
															#line 534 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 438 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 538 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 439 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<WaitForVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorState(); static_cast<WaitForVersionActor*>(this)->destroy(); return 0; }
															#line 542 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<WaitForVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionActorState();
		static_cast<WaitForVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 437 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (version < data->oldestVersion.get())
															#line 554 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 438 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 558 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 439 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<WaitForVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorState(); static_cast<WaitForVersionActor*>(this)->destroy(); return 0; }
															#line 562 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<WaitForVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionActorState();
		static_cast<WaitForVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 442 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 443 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevWarn, "CacheServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 578 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 447 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 582 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 442 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 590 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 443 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevWarn, "CacheServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 594 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 447 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 598 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitForVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForVersionActor*>(this)->ActorCallback< WaitForVersionActor, 0, Void >::remove();
		static_cast<WaitForVersionActor*>(this)->ActorCallback< WaitForVersionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForVersionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersion", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* data;
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version version;
															#line 727 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via waitForVersion()
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class WaitForVersionActor final : public Actor<Version>, public ActorCallback< WaitForVersionActor, 0, Void >, public ActorCallback< WaitForVersionActor, 1, Void >, public FastAllocated<WaitForVersionActor>, public WaitForVersionActorState<WaitForVersionActor> {
															#line 732 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<WaitForVersionActor>::operator new;
	using FastAllocated<WaitForVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14325360290389040640UL, 12515744419785195008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForVersionActor, 0, Void >;
friend struct ActorCallback< WaitForVersionActor, 1, Void >;
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	WaitForVersionActor(StorageCacheData* const& data,Version const& version) 
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForVersionActorState<WaitForVersionActor>(data, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12860986568707665920UL, 7455632835290028288UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForVersionActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Version> waitForVersion( StorageCacheData* const& data, Version const& version ) {
															#line 418 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Version>(new WaitForVersionActor(data, version));
															#line 783 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 451 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

															#line 788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForVersionNoTooOld()
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class WaitForVersionNoTooOldActor>
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class WaitForVersionNoTooOldActorState {
															#line 795 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	WaitForVersionNoTooOldActorState(StorageCacheData* const& data,Version const& version) 
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : data(data),
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   version(version)
															#line 804 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this));

	}
	~WaitForVersionNoTooOldActorState() 
	{
		fdb_probe_actor_destroy("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 454 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (version == latestVersion)
															#line 819 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 455 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				version = std::max(Version(1), data->version.get());
															#line 823 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 456 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (version <= data->version.get())
															#line 827 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 457 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 831 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
				this->~WaitForVersionNoTooOldActorState();
				static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 459 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 458 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 841 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 462 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state = 1;
															#line 459 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*>(static_cast<WaitForVersionNoTooOldActor*>(this)));
															#line 462 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*>(static_cast<WaitForVersionNoTooOldActor*>(this)));
															#line 852 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 460 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 875 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 460 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 887 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 899 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 464 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevWarn, "CacheServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 903 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 468 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 907 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 915 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 464 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevWarn, "CacheServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 919 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 468 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 923 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state > 0) static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForVersionNoTooOldActor*>(this)->ActorCallback< WaitForVersionNoTooOldActor, 0, Void >::remove();
		static_cast<WaitForVersionNoTooOldActor*>(this)->ActorCallback< WaitForVersionNoTooOldActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* data;
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version version;
															#line 1052 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via waitForVersionNoTooOld()
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class WaitForVersionNoTooOldActor final : public Actor<Version>, public ActorCallback< WaitForVersionNoTooOldActor, 0, Void >, public ActorCallback< WaitForVersionNoTooOldActor, 1, Void >, public FastAllocated<WaitForVersionNoTooOldActor>, public WaitForVersionNoTooOldActorState<WaitForVersionNoTooOldActor> {
															#line 1057 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<WaitForVersionNoTooOldActor>::operator new;
	using FastAllocated<WaitForVersionNoTooOldActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12489731901460959744UL, 13530565364871332608UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForVersionNoTooOldActor, 0, Void >;
friend struct ActorCallback< WaitForVersionNoTooOldActor, 1, Void >;
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	WaitForVersionNoTooOldActor(StorageCacheData* const& data,Version const& version) 
															#line 1075 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForVersionNoTooOldActorState<WaitForVersionNoTooOldActor>(data, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14315589494631704320UL, 1805358750176616448UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForVersionNoTooOld");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Version> waitForVersionNoTooOld( StorageCacheData* const& data, Version const& version ) {
															#line 452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Version>(new WaitForVersionNoTooOldActor(data, version));
															#line 1108 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 472 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

															#line 1113 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via getValueQ()
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class GetValueQActor>
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class GetValueQActorState {
															#line 1120 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetValueQActorState(StorageCacheData* const& data,GetValueRequest const& req) 
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : data(data),
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   req(req),
															#line 474 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   resultSize(0)
															#line 1131 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("getValueQ", reinterpret_cast<unsigned long>(this));

	}
	~GetValueQActorState() 
	{
		fdb_probe_actor_destroy("getValueQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 476 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				++data->counters.getValueQueries;
															#line 477 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				++data->counters.allQueries;
															#line 485 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				StrictFuture<Void> __when_expr_0 = delay(0, TaskPriority::DefaultEndpoint);
															#line 485 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1153 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetValueQActor*>(this)->actor_wait_state = 1;
															#line 485 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 0, Void >*>(static_cast<GetValueQActor*>(this)));
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetValueQActorState();
		static_cast<GetValueQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 548 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		++data->counters.finishedQueries;
															#line 555 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<GetValueQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetValueQActorState(); static_cast<GetValueQActor*>(this)->destroy(); return 0; }
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<GetValueQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetValueQActorState();
		static_cast<GetValueQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 543 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (!canReplyWith(e))
															#line 1202 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 544 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 1206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 545 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			req.reply.sendError(e);
															#line 1210 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 487 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 1225 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 488 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.DoRead");
															#line 1229 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		v = Optional<Value>();
															#line 495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Version> __when_expr_1 = waitForVersion(data, req.version);
															#line 495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1237 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 2;
															#line 495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 1, Version >*>(static_cast<GetValueQActor*>(this)));
															#line 1242 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 487 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 1251 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 488 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.DoRead");
															#line 1255 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		v = Optional<Value>();
															#line 495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Version> __when_expr_1 = waitForVersion(data, req.version);
															#line 495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1263 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 2;
															#line 495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 1, Version >*>(static_cast<GetValueQActor*>(this)));
															#line 1268 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 496 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 1352 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 497 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.AfterVersion");
															#line 1356 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 501 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		changeCounter = data->cacheRangeChangeCounter;
															#line 503 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (data->cachedRangeMap[req.key]->notAssigned())
															#line 1362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1Catch2(wrong_shard_server(), loopDepth);
															#line 1366 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		else
		{
															#line 506 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (!data->cachedRangeMap[req.key]->isReadable())
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 509 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch2(future_version(), loopDepth);
															#line 1376 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
															#line 512 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		path = 0;
															#line 513 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto i = data->data().at(version).lastLessOrEqual(req.key);
															#line 514 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (i && i->isValue() && i.key() == req.key)
															#line 1385 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 515 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			v = (Value)i->getValue();
															#line 516 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			path = 1;
															#line 518 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->checkChangeCounter(changeCounter, req.key);
															#line 1393 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 526 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (v.present())
															#line 1397 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 527 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			++data->counters.rowsQueried;
															#line 528 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			resultSize = v.get().size();
															#line 529 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->counters.bytesQueried += resultSize;
															#line 1405 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 533 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 534 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.AfterRead");
															#line 1413 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 538 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		GetValueReply reply(v, true);
															#line 539 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		req.reply.send(reply);
															#line 1419 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Version const& __version,int loopDepth) 
	{
															#line 495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		version = __version;
															#line 1428 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 1, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 1, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 1, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 1, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont13(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* data;
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetValueRequest req;
															#line 474 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int64_t resultSize;
															#line 494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Optional<Value> v;
															#line 495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version version;
															#line 501 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	uint64_t changeCounter;
															#line 512 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int path;
															#line 1530 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via getValueQ()
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class GetValueQActor final : public Actor<Void>, public ActorCallback< GetValueQActor, 0, Void >, public ActorCallback< GetValueQActor, 1, Version >, public FastAllocated<GetValueQActor>, public GetValueQActorState<GetValueQActor> {
															#line 1535 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<GetValueQActor>::operator new;
	using FastAllocated<GetValueQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18102186071519693312UL, 12959499486293763072UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetValueQActor, 0, Void >;
friend struct ActorCallback< GetValueQActor, 1, Version >;
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetValueQActor(StorageCacheData* const& data,GetValueRequest const& req) 
															#line 1553 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   GetValueQActorState<GetValueQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17920039446934625280UL, 7083594275336435200UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getValueQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetValueQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetValueQActor, 1, Version >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> getValueQ( StorageCacheData* const& data, GetValueRequest const& req ) {
															#line 473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new GetValueQActor(data, req));
															#line 1587 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 556 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"


GetKeyValuesReply readRange(StorageCacheData* data, Version version, KeyRangeRef range, int limit, int* pLimitBytes) {
	GetKeyValuesReply result;
	StorageCacheData::VersionedData::ViewAtVersion view = data->data().at(version);
	StorageCacheData::VersionedData::iterator vCurrent = view.end();
	KeyRef readBegin;
	KeyRef readEnd;
	KeyRef rangeBegin = range.begin;
	KeyRef rangeEnd = range.end;
	int accumulatedBytes = 0;
	// printf("\nSCReadRange\n");

	// if (limit >= 0) we are reading forward, else backward
	if (limit >= 0) {
		// We might care about a clear beginning before start that runs into range
		vCurrent = view.lastLessOrEqual(rangeBegin);
		if (vCurrent && vCurrent->isClearTo() && vCurrent->getEndKey() > rangeBegin)
			readBegin = vCurrent->getEndKey();
		else
			readBegin = rangeBegin;

		vCurrent = view.lower_bound(readBegin);
		ASSERT(!vCurrent || vCurrent.key() >= readBegin);
		if (vCurrent) {
			auto b = vCurrent;
			--b;
			ASSERT(!b || b.key() < readBegin);
		}
		accumulatedBytes = 0;
		while (vCurrent && vCurrent.key() < rangeEnd && limit > 0 && accumulatedBytes < *pLimitBytes) {
			if (!vCurrent->isClearTo()) {
				result.data.push_back_deep(result.arena, KeyValueRef(vCurrent.key(), vCurrent->getValue()));
				accumulatedBytes += sizeof(KeyValueRef) + result.data.end()[-1].expectedSize();
				--limit;
			}
			++vCurrent;
		}
	} else { // reverse readRange
		vCurrent = view.lastLess(rangeEnd);

		// A clear might extend all the way to range.end
		if (vCurrent && vCurrent->isClearTo() && vCurrent->getEndKey() >= rangeEnd) {
			readEnd = vCurrent.key();
			--vCurrent;
		} else {
			readEnd = rangeEnd;
		}
		ASSERT(!vCurrent || vCurrent.key() < readEnd);
		if (vCurrent) {
			auto b = vCurrent;
			--b;
			ASSERT(!b || b.key() >= readEnd);
		}
		accumulatedBytes = 0;
		while (vCurrent && vCurrent.key() >= rangeEnd && limit > 0 && accumulatedBytes < *pLimitBytes) {
			if (!vCurrent->isClearTo()) {
				result.data.push_back_deep(result.arena, KeyValueRef(vCurrent.key(), vCurrent->getValue()));
				accumulatedBytes += sizeof(KeyValueRef) + result.data.end()[-1].expectedSize();
				--limit;
			}
			--vCurrent;
		}
	}

	*pLimitBytes -= accumulatedBytes;
	ASSERT(result.data.size() == 0 || *pLimitBytes + result.data.end()[-1].expectedSize() + sizeof(KeyValueRef) > 0);
	result.more = limit == 0 || *pLimitBytes <= 0; // FIXME: Does this have to be exact?
	result.version = version;
	result.cached = true;
	return result;
}

Key findKey(StorageCacheData* data, KeySelectorRef sel, Version version, KeyRange range, int* pOffset)
// Attempts to find the key indicated by sel in the data at version, within range.
// Precondition: selectorInRange(sel, range)
// If it is found, offset is set to 0 and a key is returned which falls inside range.
// If the search would depend on any key outside range OR if the key selector offset is too large (range read returns
// too many bytes), it returns either
//   a negative offset and a key in [range.begin, sel.getKey()], indicating the key is (the first key <= returned key) +
//   offset, or a positive offset and a key in (sel.getKey(), range.end], indicating the key is (the first key >=
//   returned key) + offset-1
// The range passed in to this function should specify a cacheRange.  If range.begin is repeatedly not the beginning of
// a cacheRange, then it is possible to get stuck looping here
{
	ASSERT(version != latestVersion);
	ASSERT(selectorInRange(sel, range) && version >= data->oldestVersion.get());

	// Count forward or backward distance items, skipping the first one if it == key and skipEqualKey
	bool forward = sel.offset > 0; // If forward, result >= sel.getKey(); else result <= sel.getKey()
	int sign = forward ? +1 : -1;
	bool skipEqualKey = sel.orEqual == forward;
	int distance = forward ? sel.offset : 1 - sel.offset;

	// Don't limit the number of bytes if this is a trivial key selector (there will be at most two items returned from
	// the read range in this case)
	int maxBytes;
	if (sel.offset <= 1 && sel.offset >= 0)
		maxBytes = std::numeric_limits<int>::max();
	else
		maxBytes = BUGGIFY ? SERVER_KNOBS->BUGGIFY_LIMIT_BYTES : SERVER_KNOBS->STORAGE_LIMIT_BYTES;

	GetKeyValuesReply rep =
	    readRange(data,
	              version,
	              forward ? KeyRangeRef(sel.getKey(), range.end) : KeyRangeRef(range.begin, keyAfter(sel.getKey())),
	              (distance + skipEqualKey) * sign,
	              &maxBytes);
	bool more = rep.more && rep.data.size() != distance + skipEqualKey;

	// If we get only one result in the reverse direction as a result of the data being too large, we could get stuck in
	// a loop
	if (more && !forward && rep.data.size() == 1) {
		// CODE_PROBE(true, "Reverse key selector returned only one result in range read");
		maxBytes = std::numeric_limits<int>::max();
		GetKeyValuesReply rep2 =
		    readRange(data, version, KeyRangeRef(range.begin, keyAfter(sel.getKey())), -2, &maxBytes);
		rep = rep2;
		more = rep.more && rep.data.size() != distance + skipEqualKey;
		ASSERT(rep.data.size() == 2 || !more);
	}

	int index = distance - 1;
	if (skipEqualKey && rep.data.size() && rep.data[0].key == sel.getKey())
		++index;

	if (index < rep.data.size()) {
		*pOffset = 0;
		return rep.data[index].key;
	} else {
		// FIXME: If range.begin=="" && !forward, return success?
		*pOffset = index - rep.data.size() + 1;
		if (!forward)
			*pOffset = -*pOffset;

		if (more) {
			// CODE_PROBE(true, "Key selector read range had more results");

			ASSERT(rep.data.size());
			Key returnKey = forward ? keyAfter(rep.data.back().key) : rep.data.back().key;

			// This is possible if key/value pairs are very large and only one result is returned on a last less than
			// query SOMEDAY: graceful handling of exceptionally sized values
			ASSERT(returnKey != sel.getKey());

			return returnKey;
		} else
			return forward ? range.end : range.begin;
	}
}

KeyRange getCachedKeyRange(StorageCacheData* data, const KeySelectorRef& sel)
// Returns largest range that is cached on this server and selectorInRange(sel, range) or wrong_shard_server if no such
// range exists
{
	auto i = sel.isBackward() ? data->cachedRangeMap.rangeContainingKeyBefore(sel.getKey())
	                          : data->cachedRangeMap.rangeContaining(sel.getKey());

	if (i->value()->notAssigned())
		throw wrong_shard_server();
	else if (!i->value()->isReadable())
		throw future_version();

	ASSERT(selectorInRange(sel, i->range()));
	return i->range();
}

															#line 1758 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via getKeyValues()
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class GetKeyValuesActor>
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class GetKeyValuesActorState {
															#line 1765 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetKeyValuesActorState(StorageCacheData* const& data,GetKeyValuesRequest const& req) 
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : data(data),
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   req(req),
															#line 727 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   resultSize(0)
															#line 1776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("getKeyValues", reinterpret_cast<unsigned long>(this));

	}
	~GetKeyValuesActorState() 
	{
		fdb_probe_actor_destroy("getKeyValues", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 729 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			++data->counters.getRangeQueries;
															#line 730 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			++data->counters.allQueries;
															#line 737 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TaskPriority taskType = TaskPriority::DefaultEndpoint;
															#line 738 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (SERVER_KNOBS->FETCH_KEYS_LOWER_PRIORITY && req.options.present() && req.options.get().type == ReadType::FETCH)
															#line 1797 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 739 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				taskType = TaskPriority::FetchKeys;
															#line 1801 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 744 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0, taskType);
															#line 744 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<GetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1807 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetKeyValuesActor*>(this)->actor_wait_state = 1;
															#line 744 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesActor, 0, Void >*>(static_cast<GetKeyValuesActor*>(this)));
															#line 1812 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetKeyValuesActorState();
		static_cast<GetKeyValuesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 747 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 1836 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 748 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storagecache.getKeyValues.Before");
															#line 1840 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Version> __when_expr_1 = waitForVersion(data, req.version);
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<GetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1846 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetKeyValuesActor*>(this)->actor_wait_state = 2;
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesActor, 1, Version >*>(static_cast<GetKeyValuesActor*>(this)));
															#line 1851 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 747 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 1867 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 748 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storagecache.getKeyValues.Before");
															#line 1871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Version> __when_expr_1 = waitForVersion(data, req.version);
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<GetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1877 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetKeyValuesActor*>(this)->actor_wait_state = 2;
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesActor, 1, Version >*>(static_cast<GetKeyValuesActor*>(this)));
															#line 1882 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetKeyValuesActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesActor*>(this)->ActorCallback< GetKeyValuesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetKeyValuesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 854 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		++data->counters.finishedQueries;
															#line 856 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<GetKeyValuesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetKeyValuesActorState(); static_cast<GetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 1974 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<GetKeyValuesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetKeyValuesActorState();
		static_cast<GetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 843 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevWarn, "SCGetKeyValuesError", data->thisServerID) .detail("Code", e.code()) .detail("ReqBegin", req.begin.getKey()) .detail("ReqEnd", req.end.getKey()) .detail("ReqVersion", req.version) .detail("DataVersion", data->version.get());
															#line 849 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (!canReplyWith(e))
															#line 1989 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 850 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 1993 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 851 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			req.reply.sendError(e);
															#line 1997 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
															#line 752 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		changeCounter = data->cacheRangeChangeCounter;
															#line 754 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cachedKeyRange = getCachedKeyRange(data, req.begin);
															#line 756 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 2016 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 757 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storagecache.getKeyValues.AfterVersion");
															#line 2020 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 761 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!selectorInRange(req.end, cachedKeyRange) && !(req.end.isFirstGreaterOrEqual() && req.end.getKey() == cachedKeyRange.end))
															#line 2024 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 766 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1cont1Catch1(wrong_shard_server(), loopDepth);
															#line 2028 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		offset1 = int();
															#line 770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		offset2 = int();
															#line 771 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		begin = req.begin.isFirstGreaterOrEqual() ? req.begin.getKey() : findKey(data, req.begin, version, cachedKeyRange, &offset1);
															#line 774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		end = req.end.isFirstGreaterOrEqual() ? req.end.getKey() : findKey(data, req.end, version, cachedKeyRange, &offset2);
															#line 776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 2040 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 777 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storagecache.getKeyValues.AfterKeys");
															#line 2044 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 785 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if ((offset1 && offset1 != 1) || (offset2 && offset2 != 1))
															#line 2048 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 795 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1cont1Catch1(wrong_shard_server(), loopDepth);
															#line 2052 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 802 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (begin >= end)
															#line 2056 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 803 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 2060 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 804 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storagecache.getKeyValues.Send");
															#line 2064 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 808 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			GetKeyValuesReply none;
															#line 809 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			none.version = version;
															#line 810 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			none.more = false;
															#line 811 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->checkChangeCounter(changeCounter, KeyRangeRef(std::min<KeyRef>(req.begin.getKey(), req.end.getKey()), std::max<KeyRef>(req.begin.getKey(), req.end.getKey())));
															#line 814 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			req.reply.send(none);
															#line 2076 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		else
		{
															#line 816 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			remainingLimitBytes = req.limitBytes;
															#line 818 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			GetKeyValuesReply _r = readRange(data, version, KeyRangeRef(begin, end), req.limit, &remainingLimitBytes);
															#line 819 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			GetKeyValuesReply r = _r;
															#line 821 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 2088 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 822 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", req.options.get().debugID.get().first(), "storagecache.getKeyValues.AfterReadRange");
															#line 2092 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 825 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->checkChangeCounter( changeCounter, KeyRangeRef(std::min<KeyRef>(begin, std::min<KeyRef>(req.begin.getKey(), req.end.getKey())), std::max<KeyRef>(end, std::max<KeyRef>(req.begin.getKey(), req.end.getKey()))));
															#line 830 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (EXPENSIVE_VALIDATION)
															#line 2098 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 831 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				for(int i = 0;i < r.data.size();i++) {
															#line 832 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					ASSERT(r.data[i].key >= begin && r.data[i].key < end);
															#line 2104 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
															#line 833 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				ASSERT(r.data.size() <= std::abs(req.limit));
															#line 2108 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 836 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			req.reply.send(r);
															#line 838 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			resultSize = req.limitBytes - remainingLimitBytes;
															#line 839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->counters.bytesQueried += resultSize;
															#line 840 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->counters.rowsQueried += r.data.size();
															#line 2118 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		loopDepth = a_body1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Version const& __version,int loopDepth) 
	{
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		version = __version;
															#line 2128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetKeyValuesActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesActor*>(this)->ActorCallback< GetKeyValuesActor, 1, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesActor, 1, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesActor, 1, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetKeyValuesActor, 1, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont17(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* data;
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetKeyValuesRequest req;
															#line 727 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int64_t resultSize;
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version version;
															#line 752 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	uint64_t changeCounter;
															#line 754 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	KeyRange cachedKeyRange;
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int offset1;
															#line 770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int offset2;
															#line 771 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Key begin;
															#line 774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Key end;
															#line 816 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int remainingLimitBytes;
															#line 2238 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via getKeyValues()
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class GetKeyValuesActor final : public Actor<Void>, public ActorCallback< GetKeyValuesActor, 0, Void >, public ActorCallback< GetKeyValuesActor, 1, Version >, public FastAllocated<GetKeyValuesActor>, public GetKeyValuesActorState<GetKeyValuesActor> {
															#line 2243 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<GetKeyValuesActor>::operator new;
	using FastAllocated<GetKeyValuesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7767961594618237696UL, 15238951859707809792UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetKeyValuesActor, 0, Void >;
friend struct ActorCallback< GetKeyValuesActor, 1, Version >;
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetKeyValuesActor(StorageCacheData* const& data,GetKeyValuesRequest const& req) 
															#line 2261 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   GetKeyValuesActorState<GetKeyValuesActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getKeyValues", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6773873305336683264UL, 9370560631201917440UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getKeyValues");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getKeyValues", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetKeyValuesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetKeyValuesActor, 1, Version >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> getKeyValues( StorageCacheData* const& data, GetKeyValuesRequest const& req ) {
															#line 723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new GetKeyValuesActor(data, req));
															#line 2295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

															#line 2300 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via getKey()
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class GetKeyActor>
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class GetKeyActorState {
															#line 2307 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetKeyActorState(StorageCacheData* const& data,GetKeyRequest const& req) 
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : data(data),
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   req(req),
															#line 860 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   resultSize(0)
															#line 2318 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("getKey", reinterpret_cast<unsigned long>(this));

	}
	~GetKeyActorState() 
	{
		fdb_probe_actor_destroy("getKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 862 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			++data->counters.getKeyQueries;
															#line 863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			++data->counters.allQueries;
															#line 868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0, TaskPriority::DefaultEndpoint);
															#line 868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<GetKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2339 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetKeyActor*>(this)->actor_wait_state = 1;
															#line 868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetKeyActor, 0, Void >*>(static_cast<GetKeyActor*>(this)));
															#line 2344 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetKeyActorState();
		static_cast<GetKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Version> __when_expr_1 = waitForVersion(data, req.version);
															#line 871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<GetKeyActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 2370 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetKeyActor*>(this)->actor_wait_state = 2;
															#line 871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyActor, 1, Version >*>(static_cast<GetKeyActor*>(this)));
															#line 2375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Version> __when_expr_1 = waitForVersion(data, req.version);
															#line 871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<GetKeyActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 2393 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetKeyActor*>(this)->actor_wait_state = 2;
															#line 871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyActor, 1, Version >*>(static_cast<GetKeyActor*>(this)));
															#line 2398 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetKeyActor*>(this)->actor_wait_state > 0) static_cast<GetKeyActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyActor*>(this)->ActorCallback< GetKeyActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetKeyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 907 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		++data->counters.finishedQueries;
															#line 909 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<GetKeyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetKeyActorState(); static_cast<GetKeyActor*>(this)->destroy(); return 0; }
															#line 2490 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<GetKeyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetKeyActorState();
		static_cast<GetKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 902 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (!canReplyWith(e))
															#line 2503 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 903 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2507 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 904 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			req.reply.sendError(e);
															#line 2511 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 872 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		changeCounter = data->cacheRangeChangeCounter;
															#line 873 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cachedKeyRange = getCachedKeyRange(data, req.sel);
															#line 875 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		offset = int();
															#line 876 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		Key k = findKey(data, req.sel, version, cachedKeyRange, &offset);
															#line 878 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->checkChangeCounter( changeCounter, KeyRangeRef(std::min<KeyRef>(req.sel.getKey(), k), std::max<KeyRef>(req.sel.getKey(), k)));
															#line 881 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		KeySelector updated;
															#line 882 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (offset < 0)
															#line 2538 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 883 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			updated = firstGreaterOrEqual(k) + offset;
															#line 2542 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		else
		{
															#line 886 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (offset > 0)
															#line 2548 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 887 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				updated = firstGreaterOrEqual(k) + offset - 1;
															#line 2552 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 891 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				updated = KeySelectorRef(k, true, 0);
															#line 2558 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
															#line 893 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		resultSize = k.size();
															#line 894 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->counters.bytesQueried += resultSize;
															#line 895 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		++data->counters.rowsQueried;
															#line 897 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		GetKeyReply reply(updated, true);
															#line 898 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		req.reply.send(reply);
															#line 2571 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Version const& __version,int loopDepth) 
	{
															#line 871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		version = __version;
															#line 2580 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetKeyActor*>(this)->actor_wait_state > 0) static_cast<GetKeyActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyActor*>(this)->ActorCallback< GetKeyActor, 1, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 1, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 1, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetKeyActor, 1, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* data;
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetKeyRequest req;
															#line 860 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int64_t resultSize;
															#line 871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version version;
															#line 872 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	uint64_t changeCounter;
															#line 873 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	KeyRange cachedKeyRange;
															#line 875 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int offset;
															#line 2682 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via getKey()
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class GetKeyActor final : public Actor<Void>, public ActorCallback< GetKeyActor, 0, Void >, public ActorCallback< GetKeyActor, 1, Version >, public FastAllocated<GetKeyActor>, public GetKeyActorState<GetKeyActor> {
															#line 2687 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<GetKeyActor>::operator new;
	using FastAllocated<GetKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15279425201425226496UL, 10623874055647603712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetKeyActor, 0, Void >;
friend struct ActorCallback< GetKeyActor, 1, Version >;
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetKeyActor(StorageCacheData* const& data,GetKeyRequest const& req) 
															#line 2705 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   GetKeyActorState<GetKeyActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3161838746568296704UL, 16159820804706226432UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetKeyActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetKeyActor, 1, Version >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> getKey( StorageCacheData* const& data, GetKeyRequest const& req ) {
															#line 859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new GetKeyActor(data, req));
															#line 2739 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 911 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

#pragma endregion

bool expandMutation(MutationRef& m, StorageCacheData::VersionedData const& data, KeyRef eagerTrustedEnd, Arena& ar) {
	// After this function call, m should be copied into an arena immediately (before modifying data, cacheRanges, or
	// eager)
	if (m.type == MutationRef::ClearRange) {
		// Expand the clear
		const auto& d = data.atLatest();

		// If another clear overlaps the beginning of this one, engulf it
		auto i = d.lastLess(m.param1);
		if (i && i->isClearTo() && i->getEndKey() >= m.param1)
			m.param1 = i.key();

		// If another clear overlaps the end of this one, engulf it; otherwise expand
		i = d.lastLessOrEqual(m.param2);
		if (i && i->isClearTo() && i->getEndKey() >= m.param2) {
			m.param2 = i->getEndKey();
		} else {
			// Expand to the next set or clear (from storage or latestVersion), and if it
			// is a clear, engulf it as well
			i = d.lower_bound(m.param2);
			// TODO check if the following is correct
			KeyRef endKey = eagerTrustedEnd;
			if (!i || endKey < i.key())
				m.param2 = endKey;
			else if (i->isClearTo())
				m.param2 = i->getEndKey();
			else
				m.param2 = i.key();
		}
	} else if (m.type != MutationRef::SetValue && (m.type)) {

		Optional<StringRef> oldVal;
		auto it = data.atLatest().lastLessOrEqual(m.param1);
		if (it != data.atLatest().end() && it->isValue() && it.key() == m.param1)
			oldVal = it->getValue();
		else if (it != data.atLatest().end() && it->isClearTo() && it->getEndKey() > m.param1) {
			// CODE_PROBE(true, "Atomic op right after a clear.");
		}

		switch (m.type) {
		case MutationRef::AddValue:
			m.param2 = doLittleEndianAdd(oldVal, m.param2, ar);
			break;
		case MutationRef::And:
			m.param2 = doAnd(oldVal, m.param2, ar);
			break;
		case MutationRef::Or:
			m.param2 = doOr(oldVal, m.param2, ar);
			break;
		case MutationRef::Xor:
			m.param2 = doXor(oldVal, m.param2, ar);
			break;
		case MutationRef::AppendIfFits:
			m.param2 = doAppendIfFits(oldVal, m.param2, ar);
			break;
		case MutationRef::Max:
			m.param2 = doMax(oldVal, m.param2, ar);
			break;
		case MutationRef::Min:
			m.param2 = doMin(oldVal, m.param2, ar);
			break;
		case MutationRef::ByteMin:
			m.param2 = doByteMin(oldVal, m.param2, ar);
			break;
		case MutationRef::ByteMax:
			m.param2 = doByteMax(oldVal, m.param2, ar);
			break;
		case MutationRef::MinV2:
			m.param2 = doMinV2(oldVal, m.param2, ar);
			break;
		case MutationRef::AndV2:
			m.param2 = doAndV2(oldVal, m.param2, ar);
			break;
		case MutationRef::CompareAndClear:
			if (oldVal.present() && m.param2 == oldVal.get()) {
				m.type = MutationRef::ClearRange;
				m.param2 = keyAfter(m.param1, ar);
				return expandMutation(m, data, eagerTrustedEnd, ar);
			}
			return false;
		}
		m.type = MutationRef::SetValue;
	}

	return true;
}

// Applies a write mutation (SetValue or ClearRange) to the in-memory versioned data structure
void StorageCacheData::applyMutation(MutationRef const& m, Arena& arena, StorageCacheData::VersionedData& data) {
	// m is expected to be in arena already
	// Clear split keys are added to arena

	if (m.type == MutationRef::SetValue) {
		auto prev = data.atLatest().lastLessOrEqual(m.param1);
		if (prev && prev->isClearTo() && prev->getEndKey() > m.param1) {
			ASSERT(prev.key() <= m.param1);
			KeyRef end = prev->getEndKey();
			// TODO double check if the insert version of the previous clear needs to be preserved for the "left half",
			// insert() invalidates prev, so prev.key() is not safe to pass to it by reference
			data.insert(KeyRef(prev.key()),
			            ValueOrClearToRef::clearTo(m.param1),
			            prev.insertVersion()); // overwritten by below insert if empty
			//TraceEvent(SevDebug, "ApplyMutationClearTo")
			//.detail("Key1", prev.key())
			//.detail("Key2",m.param1)
			//.detail("Version1", prev.insertVersion());
			KeyRef nextKey = keyAfter(m.param1, arena);
			if (end != nextKey) {
				ASSERT(end > nextKey);
				// TODO double check if it's okay to let go of the the insert version of the "right half"
				// FIXME: This copy is technically an asymptotic problem, definitely a waste of memory (copy of keyAfter
				// is a waste, but not asymptotic)
				data.insert(nextKey, ValueOrClearToRef::clearTo(KeyRef(arena, end)));
				//TraceEvent(SevDebug, "ApplyMutationClearTo2")
				//.detail("K1", nextKey)
				//.detail("K2", end)
				//.detail("V", data.latestVersion);
			}
		}
		data.insert(m.param1, ValueOrClearToRef::value(m.param2));
		//TraceEvent(SevDebug, "ApplyMutation")
		//	.detail("Key", m.param1)
		//	.detail("Value",m.param2)
		//	.detail("Version", data.latestVersion);
	} else if (m.type == MutationRef::ClearRange) {
		data.erase(m.param1, m.param2);
		ASSERT(m.param2 > m.param1);
		ASSERT(!data.isClearContaining(data.atLatest(), m.param1));
		data.insert(m.param1, ValueOrClearToRef::clearTo(m.param2));
		//TraceEvent(SevDebug, "ApplyMutationClearTo3")
		//	.detail("Key21", m.param1)
		//	.detail("Key22", m.param2)
		//	.detail("V2", data.latestVersion);
	}
}

template <class T>
void addMutation(T& target, Version version, MutationRef const& mutation) {
	target.addMutation(version, mutation);
}

template <class T>
void addMutation(Reference<T>& target, Version version, MutationRef const& mutation) {
	addMutation(*target, version, mutation);
}

template <class T>
void splitMutation(StorageCacheData* data, KeyRangeMap<T>& map, MutationRef const& m, Version ver) {
	if (isSingleKeyMutation((MutationRef::Type)m.type)) {
		auto i = map.rangeContaining(m.param1);
		if (i->value()) // If this key lies in the cached key-range on this server
			data->addMutation(i->range(), ver, m);
	} else if (m.type == MutationRef::ClearRange) {
		KeyRangeRef mKeys(m.param1, m.param2);
		auto r = map.intersectingRanges(mKeys);
		for (auto i = r.begin(); i != r.end(); ++i) {
			if (i->value()) { // if this sub-range exists on this cache server
				KeyRangeRef k = mKeys & i->range();
				data->addMutation(i->range(), ver, MutationRef((MutationRef::Type)m.type, k.begin, k.end));
			}
		}
	} else
		ASSERT(false); // Unknown mutation type in splitMutations
}

void rollback(StorageCacheData* data, Version rollbackVersion, Version nextVersion) {
	// CODE_PROBE(true, "call to cacheRange rollback");
	// FIXME: enable when debugKeyRange is active
	// debugKeyRange("Rollback", rollbackVersion, allKeys);

	// FIXME: It's not straightforward to rollback certain versions from the VersionedMap.
	// It's doable. But for now, we choose to just throw away this cache role

	throw please_reboot();
}

void StorageCacheData::addMutation(KeyRangeRef const& cachedKeyRange, Version version, MutationRef const& mutation) {
	MutationRef expanded = mutation;
	auto& mLog = addVersionToMutationLog(version);

	if (!expandMutation(expanded, data(), cachedKeyRange.end, mLog.arena())) {
		return;
	}
	expanded = addMutationToMutationLog(mLog, expanded);

	DEBUG_MUTATION("expandedMutation", version, expanded)
	    .detail("Begin", cachedKeyRange.begin)
	    .detail("End", cachedKeyRange.end);
	applyMutation(expanded, mLog.arena(), mutableData());
	// printf("\nSCUpdate: Printing versioned tree after applying mutation\n");
	// mutableData().printTree(version);
}

void removeDataRange(StorageCacheData* sc,
                     Standalone<VerUpdateRef>& mLV,
                     KeyRangeMap<Reference<CacheRangeInfo>>& cacheRanges,
                     KeyRangeRef range) {
	// modify the latest version of data to remove all sets and trim all clears to exclude range.
	// Add a clear to mLV (mutationLog[data.getLatestVersion()]) that ensures all keys in range are removed from the
	// disk when this latest version becomes durable mLV is also modified if necessary to ensure that split clears can
	// be forgotten

	MutationRef clearRange(MutationRef::ClearRange, range.begin, range.end);
	clearRange = sc->addMutationToMutationLog(mLV, clearRange);

	auto& data = sc->mutableData();

	// Expand the range to the right to include other cacheRanges not in versionedData
	for (auto r = cacheRanges.rangeContaining(range.end);
	     r != cacheRanges.ranges().end() && !r->value()->isInVersionedData();
	     ++r)
		range = KeyRangeRef(range.begin, r->end());

	auto endClear = data.atLatest().lastLess(range.end);
	if (endClear && endClear->isClearTo() && endClear->getEndKey() > range.end) {
		// This clear has been bumped up to insertVersion==data.getLatestVersion and needs a corresponding mutation log
		// entry to forget
		MutationRef m(MutationRef::ClearRange, range.end, endClear->getEndKey());
		m = sc->addMutationToMutationLog(mLV, m);
		data.insert(m.param1, ValueOrClearToRef::clearTo(m.param2));
	}

	auto beginClear = data.atLatest().lastLess(range.begin);
	if (beginClear && beginClear->isClearTo() && beginClear->getEndKey() > range.begin) {
		// We don't need any special mutationLog entry - because the begin key and insert version are unchanged the
		// original clear
		//   mutation works to forget this one - but we need range.begin in the right arena
		KeyRef rb(mLV.arena(), range.begin);
		// insert() invalidates beginClear, so beginClear.key() is not safe to pass to it by reference
		data.insert(KeyRef(beginClear.key()), ValueOrClearToRef::clearTo(rb), beginClear.insertVersion());
	}

	data.erase(range.begin, range.end);
}

// void setAvailableStatus( StorageServer* self, KeyRangeRef keys, bool available );
// void setAssignedStatus( StorageServer* self, KeyRangeRef keys, bool nowAssigned );

void coalesceCacheRanges(StorageCacheData* data, KeyRangeRef keys) {
	auto cacheRanges = data->cachedRangeMap.intersectingRanges(keys);
	auto fullRange = data->cachedRangeMap.ranges();

	auto iter = cacheRanges.begin();
	if (iter != fullRange.begin())
		--iter;
	auto iterEnd = cacheRanges.end();
	if (iterEnd != fullRange.end())
		++iterEnd;

	bool lastReadable = false;
	bool lastNotAssigned = false;
	KeyRangeMap<Reference<CacheRangeInfo>>::iterator lastRange;

	for (; iter != iterEnd; ++iter) {
		if (lastReadable && iter->value()->isReadable()) {
			KeyRange range = KeyRangeRef(lastRange->begin(), iter->end());
			data->addCacheRange(CacheRangeInfo::newReadWrite(range, data));
			iter = data->cachedRangeMap.rangeContaining(range.begin);
		} else if (lastNotAssigned && iter->value()->notAssigned()) {
			KeyRange range = KeyRangeRef(lastRange->begin(), iter->end());
			data->addCacheRange(CacheRangeInfo::newNotAssigned(range));
			iter = data->cachedRangeMap.rangeContaining(range.begin);
		}

		lastReadable = iter->value()->isReadable();
		lastNotAssigned = iter->value()->notAssigned();
		lastRange = iter;
	}
}

															#line 3016 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via tryFetchRange()
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class TryFetchRangeActor>
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class TryFetchRangeActorState {
															#line 3023 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	TryFetchRangeActorState(Database const& cx,Version const& version,KeyRangeRef const& keys,GetRangeLimits const& limits,bool* const& isTooOld) 
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : cx(cx),
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   version(version),
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   keys(keys),
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   limits(limits),
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   isTooOld(isTooOld),
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   tr(cx),
															#line 1190 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   output(),
															#line 1191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   begin(firstGreaterOrEqual(keys.begin)),
															#line 1192 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   end(firstGreaterOrEqual(keys.end)),
															#line 1193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   options(ReadOptions(ReadType::FETCH, CacheResult::False))
															#line 3048 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("tryFetchRange", reinterpret_cast<unsigned long>(this));

	}
	~TryFetchRangeActorState() 
	{
		fdb_probe_actor_destroy("tryFetchRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1195 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (*isTooOld)
															#line 3063 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1196 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 3067 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1198 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			ASSERT(!cx->switchable);
															#line 1199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			tr.setVersion(version);
															#line 1200 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			tr.trState->taskID = TaskPriority::FetchKeys;
															#line 1201 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			tr.trState->readOptions = options;
															#line 1202 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			limits.minRows = 0;
															#line 3079 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			try {
															#line 1205 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				;
															#line 3083 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TryFetchRangeActorState();
		static_cast<TryFetchRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1238 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (begin.getKey() != keys.begin && (e.code() == error_code_transaction_too_old || e.code() == error_code_future_version || e.code() == error_code_process_behind))
															#line 3113 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1241 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (e.code() == error_code_transaction_too_old)
															#line 3117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1242 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					*isTooOld = true;
															#line 3121 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
															#line 1243 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.more = true;
															#line 1244 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (begin.isFirstGreaterOrEqual())
															#line 3127 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1245 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					output.setReadThrough(begin.getKey());
															#line 3131 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
															#line 1247 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (!static_cast<TryFetchRangeActor*>(this)->SAV<RangeResult>::futures) { (void)(output); this->~TryFetchRangeActorState(); static_cast<TryFetchRangeActor*>(this)->destroy(); return 0; }
															#line 3135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				new (&static_cast<TryFetchRangeActor*>(this)->SAV< RangeResult >::value()) RangeResult(std::move(output)); // state_var_RVO
				this->~TryFetchRangeActorState();
				static_cast<TryFetchRangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 3143 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<RangeResult> __when_expr_0 = tr.getRange(begin, end, limits, Snapshot::True);
															#line 1206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<TryFetchRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3166 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<TryFetchRangeActor*>(this)->actor_wait_state = 1;
															#line 1206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TryFetchRangeActor, 0, RangeResult >*>(static_cast<TryFetchRangeActor*>(this)));
															#line 3171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(RangeResult const& rep,int loopDepth) 
	{
															#line 1207 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		limits.decrement(rep);
															#line 1209 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (limits.isReached() || !rep.more)
															#line 3182 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1210 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (output.size())
															#line 3186 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1211 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.arena().dependsOn(rep.arena());
															#line 1212 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.append(output.arena(), rep.begin(), rep.size());
															#line 1213 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (limits.isReached() && rep.readThrough.present())
															#line 3194 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1214 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					output.readThrough = rep.readThrough.get();
															#line 3198 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
			}
			else
			{
															#line 1216 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output = rep;
															#line 3205 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1219 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			output.more = limits.isReached();
															#line 1221 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (!static_cast<TryFetchRangeActor*>(this)->SAV<RangeResult>::futures) { (void)(output); this->~TryFetchRangeActorState(); static_cast<TryFetchRangeActor*>(this)->destroy(); return 0; }
															#line 3211 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			new (&static_cast<TryFetchRangeActor*>(this)->SAV< RangeResult >::value()) RangeResult(std::move(output)); // state_var_RVO
			this->~TryFetchRangeActorState();
			static_cast<TryFetchRangeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 1222 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (rep.readThrough.present())
															#line 3221 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1223 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.arena().dependsOn(rep.arena());
															#line 1224 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (rep.size())
															#line 3227 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1225 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					output.append(output.arena(), rep.begin(), rep.size());
															#line 1226 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					ASSERT(rep.readThrough.get() > rep.end()[-1].key);
															#line 3233 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
				else
				{
															#line 1228 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					ASSERT(rep.readThrough.get() > keys.begin);
															#line 3239 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
															#line 1230 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				begin = firstGreaterOrEqual(rep.readThrough.get());
															#line 3243 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 1232 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.arena().dependsOn(rep.arena());
															#line 1233 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.append(output.arena(), rep.begin(), rep.size());
															#line 1234 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				begin = firstGreaterThan(output.end()[-1].key);
															#line 3253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(RangeResult && rep,int loopDepth) 
	{
															#line 1207 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		limits.decrement(rep);
															#line 1209 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (limits.isReached() || !rep.more)
															#line 3266 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1210 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (output.size())
															#line 3270 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1211 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.arena().dependsOn(rep.arena());
															#line 1212 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.append(output.arena(), rep.begin(), rep.size());
															#line 1213 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (limits.isReached() && rep.readThrough.present())
															#line 3278 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1214 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					output.readThrough = rep.readThrough.get();
															#line 3282 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
			}
			else
			{
															#line 1216 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output = rep;
															#line 3289 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1219 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			output.more = limits.isReached();
															#line 1221 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (!static_cast<TryFetchRangeActor*>(this)->SAV<RangeResult>::futures) { (void)(output); this->~TryFetchRangeActorState(); static_cast<TryFetchRangeActor*>(this)->destroy(); return 0; }
															#line 3295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			new (&static_cast<TryFetchRangeActor*>(this)->SAV< RangeResult >::value()) RangeResult(std::move(output)); // state_var_RVO
			this->~TryFetchRangeActorState();
			static_cast<TryFetchRangeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 1222 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (rep.readThrough.present())
															#line 3305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1223 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.arena().dependsOn(rep.arena());
															#line 1224 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (rep.size())
															#line 3311 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1225 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					output.append(output.arena(), rep.begin(), rep.size());
															#line 1226 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					ASSERT(rep.readThrough.get() > rep.end()[-1].key);
															#line 3317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
				else
				{
															#line 1228 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					ASSERT(rep.readThrough.get() > keys.begin);
															#line 3323 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
															#line 1230 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				begin = firstGreaterOrEqual(rep.readThrough.get());
															#line 3327 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 1232 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.arena().dependsOn(rep.arena());
															#line 1233 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				output.append(output.arena(), rep.begin(), rep.size());
															#line 1234 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				begin = firstGreaterThan(output.end()[-1].key);
															#line 3337 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& rep,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(rep, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && rep,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TryFetchRangeActor*>(this)->actor_wait_state > 0) static_cast<TryFetchRangeActor*>(this)->actor_wait_state = 0;
		static_cast<TryFetchRangeActor*>(this)->ActorCallback< TryFetchRangeActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< TryFetchRangeActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("tryFetchRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryFetchRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryFetchRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TryFetchRangeActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("tryFetchRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryFetchRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryFetchRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TryFetchRangeActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("tryFetchRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryFetchRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryFetchRange", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Database cx;
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version version;
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	KeyRangeRef keys;
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	GetRangeLimits limits;
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	bool* isTooOld;
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Transaction tr;
															#line 1190 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	RangeResult output;
															#line 1191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	KeySelectorRef begin;
															#line 1192 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	KeySelectorRef end;
															#line 1193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	ReadOptions options;
															#line 3439 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via tryFetchRange()
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class TryFetchRangeActor final : public Actor<RangeResult>, public ActorCallback< TryFetchRangeActor, 0, RangeResult >, public FastAllocated<TryFetchRangeActor>, public TryFetchRangeActorState<TryFetchRangeActor> {
															#line 3444 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<TryFetchRangeActor>::operator new;
	using FastAllocated<TryFetchRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7554894882940294400UL, 15577267165319745792UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<RangeResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TryFetchRangeActor, 0, RangeResult >;
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	TryFetchRangeActor(Database const& cx,Version const& version,KeyRangeRef const& keys,GetRangeLimits const& limits,bool* const& isTooOld) 
															#line 3461 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<RangeResult>(),
		   TryFetchRangeActorState<TryFetchRangeActor>(cx, version, keys, limits, isTooOld),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tryFetchRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3437381331639141888UL, 3896193855494158592UL);
		ActorExecutionContextHelper __helper(static_cast<TryFetchRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tryFetchRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tryFetchRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TryFetchRangeActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<RangeResult> tryFetchRange( Database const& cx, Version const& version, KeyRangeRef const& keys, GetRangeLimits const& limits, bool* const& isTooOld ) {
															#line 1184 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<RangeResult>(new TryFetchRangeActor(cx, version, keys, limits, isTooOld));
															#line 3494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 1252 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

															#line 3499 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchKeys()
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class FetchKeysActor>
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class FetchKeysActorState {
															#line 3506 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	FetchKeysActorState(StorageCacheData* const& data,AddingCacheRange* const& cacheRange) 
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : data(data),
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   cacheRange(cacheRange),
															#line 1254 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   interval("SCFetchKeys"),
															#line 1255 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   keys(cacheRange->keys),
															#line 1257 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   startt(now()),
															#line 1259 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   fetchBlockBytes(BUGGIFY ? SERVER_KNOBS->BUGGIFY_BLOCK_BYTES : SERVER_KNOBS->FETCH_BLOCK_BYTES)
															#line 3523 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchKeys", reinterpret_cast<unsigned long>(this));

	}
	~FetchKeysActorState() 
	{
		fdb_probe_actor_destroy("fetchKeys", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1263 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->coreStarted.getFuture() && delay(0);
															#line 1263 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3540 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FetchKeysActor*>(this)->actor_wait_state = 1;
															#line 1263 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 0, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 3545 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchKeysActorState();
		static_cast<FetchKeysActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 1273 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			validate(data);
															#line 1278 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			auto navr = data->newestAvailableVersion.intersectingRanges(keys);
															#line 1279 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			Version lastAvailable = invalidVersion;
															#line 1280 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			for(auto r = navr.begin();r != navr.end();++r) {
															#line 1281 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				ASSERT(r->value() != latestVersion);
															#line 1282 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				lastAvailable = std::max(lastAvailable, r->value());
															#line 3579 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1284 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			auto ndvr = data->newestDirtyVersion.intersectingRanges(keys);
															#line 1285 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			for(auto r = ndvr.begin();r != ndvr.end();++r) {
															#line 1286 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				lastAvailable = std::max(lastAvailable, r->value());
															#line 3587 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1288 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (lastAvailable != invalidVersion && lastAvailable >= data->oldestVersion.get())
															#line 3591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				StrictFuture<Void> __when_expr_1 = data->oldestVersion.whenAtLeast(lastAvailable + 1);
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3597 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<FetchKeysActor*>(this)->actor_wait_state = 2;
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 1, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 3602 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont3(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 1273 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			validate(data);
															#line 1278 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			auto navr = data->newestAvailableVersion.intersectingRanges(keys);
															#line 1279 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			Version lastAvailable = invalidVersion;
															#line 1280 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			for(auto r = navr.begin();r != navr.end();++r) {
															#line 1281 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				ASSERT(r->value() != latestVersion);
															#line 1282 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				lastAvailable = std::max(lastAvailable, r->value());
															#line 3633 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1284 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			auto ndvr = data->newestDirtyVersion.intersectingRanges(keys);
															#line 1285 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			for(auto r = ndvr.begin();r != ndvr.end();++r) {
															#line 1286 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				lastAvailable = std::max(lastAvailable, r->value());
															#line 3641 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1288 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (lastAvailable != invalidVersion && lastAvailable >= data->oldestVersion.get())
															#line 3645 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				StrictFuture<Void> __when_expr_1 = data->oldestVersion.whenAtLeast(lastAvailable + 1);
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3651 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<FetchKeysActor*>(this)->actor_wait_state = 2;
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 1, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 3656 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont3(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1553 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<FetchKeysActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchKeysActorState(); static_cast<FetchKeysActor*>(this)->destroy(); return 0; }
															#line 3751 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<FetchKeysActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchKeysActorState();
		static_cast<FetchKeysActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (e.code() == error_code_actor_cancelled && cacheRange->phase >= AddingCacheRange::Fetching)
															#line 3764 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1527 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (cacheRange->phase < AddingCacheRange::Waiting)
															#line 3768 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1529 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					removeDataRange( data, data->addVersionToMutationLog(data->data().getLatestVersion()), data->cachedRangeMap, keys);
															#line 3772 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
				else
				{
															#line 1533 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					ASSERT(data->data().getLatestVersion() > data->version.get());
															#line 1534 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					removeDataRange( data, data->addVersionToMutationLog(data->data().getLatestVersion()), data->cachedRangeMap, keys);
															#line 1539 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					data->newestDirtyVersion.insert(keys, data->data().getLatestVersion());
															#line 3782 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
			}
															#line 1543 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevError, "SCFetchKeysError", data->thisServerID) .error(e) .detail("Elapsed", now() - startt) .detail("KeyBegin", keys.begin) .detail("KeyEnd", keys.end);
															#line 1548 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 3789 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1549 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				data->otherError.sendError(e);
															#line 3793 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1550 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 3797 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1293 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent(SevDebug, "SCFetchKeysVersionSatisfied", data->thisServerID).detail("FKID", interval.pairID);
															#line 1295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_2 = data->fetchKeysParallelismLock.take(TaskPriority::DefaultYield, fetchBlockBytes);
															#line 1295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3815 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 3;
															#line 1295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 2, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 3820 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		holdingFKPL = FlowLock::Releaser(data->fetchKeysParallelismLock, fetchBlockBytes);
															#line 1305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_3 = data->updateVersionLock.take();
															#line 1305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3920 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont7when1(__when_expr_3.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 4;
															#line 1305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 3, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 3925 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		holdingFKPL = FlowLock::Releaser(data->fetchKeysParallelismLock, fetchBlockBytes);
															#line 1305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_3 = data->updateVersionLock.take();
															#line 1305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3938 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont7when1(__when_expr_3.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 4;
															#line 1305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 3, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 3943 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
															#line 1307 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->phase = AddingCacheRange::Fetching;
															#line 1308 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		fetchVersion = data->version.get();
															#line 1310 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->updateVersionLock.release();
															#line 1312 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(0);
															#line 1312 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 4035 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont8when1(__when_expr_4.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 5;
															#line 1312 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 4, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 4040 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
															#line 1307 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->phase = AddingCacheRange::Fetching;
															#line 1308 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		fetchVersion = data->version.get();
															#line 1310 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->updateVersionLock.release();
															#line 1312 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(0);
															#line 1312 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 4057 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont8when1(__when_expr_4.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 5;
															#line 1312 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 4, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 4062 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
															#line 1314 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent(SevDebug, "SCFetchKeysUnblocked", data->thisServerID) .detail("FKID", interval.pairID) .detail("Version", fetchVersion);
															#line 1319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		debug_getRangeRetries = 0;
															#line 1320 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		debug_nextRetryToLog = 1;
															#line 1321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		isTooOld = false;
															#line 1326 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 4154 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont9loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
															#line 1314 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent(SevDebug, "SCFetchKeysUnblocked", data->thisServerID) .detail("FKID", interval.pairID) .detail("Version", fetchVersion);
															#line 1319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		debug_getRangeRetries = 0;
															#line 1320 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		debug_nextRetryToLog = 1;
															#line 1321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		isTooOld = false;
															#line 1326 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 4171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont9loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont10(int loopDepth) 
	{
															#line 1432 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->fetchComplete.send(Void());
															#line 1443 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		holdingFKPL.release();
															#line 1448 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		Promise<FetchInjectionInfo*> p;
															#line 1449 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->readyFetchKeys.push_back(p);
															#line 1451 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<FetchInjectionInfo*> __when_expr_9 = p.getFuture();
															#line 1451 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 4265 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont10when1(__when_expr_9.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 10;
															#line 1451 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 9, FetchInjectionInfo* >*>(static_cast<FetchKeysActor*>(this)));
															#line 4270 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont9loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont9loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1(int loopDepth) 
	{
		try {
															#line 1330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<RangeResult> __when_expr_5 = tryFetchRange(data->cx, fetchVersion, keys, GetRangeLimits(GetRangeLimits::ROW_LIMIT_UNLIMITED, fetchBlockBytes), &isTooOld);
															#line 1330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont9loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4289 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont9loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont9loopBody1when1(__when_expr_5.get(), loopDepth); };
			static_cast<FetchKeysActor*>(this)->actor_wait_state = 6;
															#line 1330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 5, RangeResult >*>(static_cast<FetchKeysActor*>(this)));
															#line 4294 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont9loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont9loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9break1(int loopDepth) 
	{
		try {
			return a_body1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont9loopHead1(0);

		return loopDepth;
	}
	int a_body1cont9loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1387 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent("SCFKBlockFail", data->thisServerID) .errorUnsuppressed(e) .suppressFor(1.0) .detail("FKID", interval.pairID);
															#line 1391 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (e.code() == error_code_transaction_too_old)
															#line 4331 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1393 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				Version lastFV = fetchVersion;
															#line 1394 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				fetchVersion = data->version.get();
															#line 1395 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				isTooOld = false;
															#line 1399 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				for(;!cacheRange->updates.empty() && cacheRange->updates[0].version <= fetchVersion;) {
															#line 1400 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					cacheRange->updates.pop_front();
															#line 4343 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
															#line 1404 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (debug_getRangeRetries >= 100)
															#line 4347 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1405 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					ASSERT(false);
															#line 1406 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					data->cx->enableLocalityLoadBalance = EnableLocalityLoadBalance::False;
															#line 4353 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				debug_getRangeRetries++;
															#line 1410 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (debug_nextRetryToLog == debug_getRangeRetries)
															#line 4359 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1411 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					debug_nextRetryToLog += std::min(debug_nextRetryToLog, 1024);
															#line 1412 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					TraceEvent(SevWarn, "SCFetchPast", data->thisServerID) .detail("TotalAttempts", debug_getRangeRetries) .detail("FKID", interval.pairID) .detail("V", lastFV) .detail("N", fetchVersion) .detail("E", data->version.get());
															#line 4365 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
			}
			else
			{
															#line 1419 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (e.code() == error_code_future_version || e.code() == error_code_process_behind)
															#line 4372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
				}
				else
				{
															#line 1424 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					return a_body1cont1Catch1(e, std::max(0, loopDepth - 1));
															#line 4379 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
			}
															#line 1426 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_8 = delayJittered(FLOW_KNOBS->PREVENT_FAST_SPIN_DELAY);
															#line 1426 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4386 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont9loopBody1Catch1when1(__when_expr_8.get(), loopDepth); };
			static_cast<FetchKeysActor*>(this)->actor_wait_state = 9;
															#line 1426 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 8, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 4391 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont9loopBody1cont2(int loopDepth) 
	{
															#line 1337 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		expectedSize = (int)this_block.expectedSize() + (8 - (int)sizeof(KeyValueRef)) * this_block.size();
															#line 1340 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent(SevDebug, "SCFetchKeysBlock", data->thisServerID) .detail("FKID", interval.pairID) .detail("BlockRows", this_block.size()) .detail("BlockBytes", expectedSize) .detail("KeyBegin", keys.begin) .detail("KeyEnd", keys.end) .detail("Last", this_block.size() ? this_block.end()[-1].key : std::string()) .detail("Version", fetchVersion) .detail("More", this_block.more);
															#line 1356 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->counters.bytesFetched += expectedSize;
															#line 1357 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (fetchBlockBytes > expectedSize)
															#line 4412 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1358 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			holdingFKPL.release(fetchBlockBytes - expectedSize);
															#line 4416 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		kvItr = this_block.begin();
															#line 1363 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 4422 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont9loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1when1(RangeResult const& __this_block,int loopDepth) 
	{
															#line 1330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		this_block = __this_block;
															#line 4431 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont9loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1when1(RangeResult && __this_block,int loopDepth) 
	{
		this_block = std::move(__this_block);
		loopDepth = a_body1cont9loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 5, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 5, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont9loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 5, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont9loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 5, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont9loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont9loopBody1cont3(int loopDepth) 
	{
															#line 1375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (this_block.more)
															#line 4510 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1376 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevDebug, "CacheWarmupMoreDataThanLimit", data->thisServerID).log();
															#line 1377 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1cont9loopBody1Catch1(please_reboot(), loopDepth);
															#line 4516 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1380 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		this_block = RangeResult();
															#line 1382 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (BUGGIFY)
															#line 4522 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1383 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_7 = delay(1);
															#line 1383 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont9loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4528 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont9loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont9loopBody1cont3when1(__when_expr_7.get(), loopDepth); };
			static_cast<FetchKeysActor*>(this)->actor_wait_state = 8;
															#line 1383 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 7, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 4533 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont9loopBody1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont9loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1363 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!(kvItr != this_block.end()))
															#line 4554 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
			return a_body1cont9loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1364 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		applyMutation(data->updater, data, MutationRef(MutationRef::SetValue, kvItr->key, kvItr->value), fetchVersion);
															#line 1368 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->counters.bytesFetched += expectedSize;
															#line 1369 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_6 = yield();
															#line 1369 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont9loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4566 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont9loopBody1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont9loopBody1cont2loopBody1when1(__when_expr_6.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 7;
															#line 1369 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 6, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 4571 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont9loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont9loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont9loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont9loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9loopBody1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1363 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		++kvItr;
															#line 4593 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont9loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont9loopBody1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1363 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		++kvItr;
															#line 4602 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont9loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont9loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont9loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont9loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont9loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont9loopBody1cont5(int loopDepth) 
	{
		return a_body1cont9break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont9loopBody1cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont9loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont9loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont9loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont9loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont9loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont9loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont9loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont9loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont9loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont10cont1(FetchInjectionInfo* const& batch,int loopDepth) 
	{
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent(SevDebug, "SCFKUpdateBatch", data->thisServerID).detail("FKID", interval.pairID);
															#line 1454 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->phase = AddingCacheRange::Waiting;
															#line 1461 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->transferredVersion = data->version.get() + 1;
															#line 1462 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->mutableData().createNewVersion(cacheRange->transferredVersion);
															#line 1463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(cacheRange->transferredVersion > data->oldestVersion.get());
															#line 1464 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(cacheRange->transferredVersion == data->data().getLatestVersion());
															#line 1466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent(SevDebug, "SCFetchKeysHaveData", data->thisServerID) .detail("FKID", interval.pairID) .detail("Version", cacheRange->transferredVersion) .detail("OldestVersion", data->oldestVersion.get());
															#line 1470 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		validate(data);
															#line 1476 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		for(auto i = cacheRange->updates.begin();i != cacheRange->updates.end();++i) {
															#line 1477 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			i->version = cacheRange->transferredVersion;
															#line 1478 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			batch->arena.dependsOn(i->arena());
															#line 4886 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1481 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		int startSize = batch->changes.size();
															#line 1483 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		batch->changes.resize(batch->changes.size() + cacheRange->updates.size());
															#line 1486 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		std::copy(cacheRange->updates.begin(), cacheRange->updates.end(), batch->changes.begin() + startSize);
															#line 1487 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		Version checkv = cacheRange->transferredVersion;
															#line 1489 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		for(auto b = batch->changes.begin() + startSize;b != batch->changes.end();++b) {
															#line 1490 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			ASSERT(b->version >= checkv);
															#line 1491 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			checkv = b->version;
															#line 4902 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1497 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->updates.clear();
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_10 = data->oldestVersion.whenAtLeast(cacheRange->transferredVersion);
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 4910 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont10cont1when1(__when_expr_10.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 11;
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 10, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 4915 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10cont1(FetchInjectionInfo* && batch,int loopDepth) 
	{
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent(SevDebug, "SCFKUpdateBatch", data->thisServerID).detail("FKID", interval.pairID);
															#line 1454 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->phase = AddingCacheRange::Waiting;
															#line 1461 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->transferredVersion = data->version.get() + 1;
															#line 1462 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->mutableData().createNewVersion(cacheRange->transferredVersion);
															#line 1463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(cacheRange->transferredVersion > data->oldestVersion.get());
															#line 1464 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(cacheRange->transferredVersion == data->data().getLatestVersion());
															#line 1466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent(SevDebug, "SCFetchKeysHaveData", data->thisServerID) .detail("FKID", interval.pairID) .detail("Version", cacheRange->transferredVersion) .detail("OldestVersion", data->oldestVersion.get());
															#line 1470 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		validate(data);
															#line 1476 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		for(auto i = cacheRange->updates.begin();i != cacheRange->updates.end();++i) {
															#line 1477 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			i->version = cacheRange->transferredVersion;
															#line 1478 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			batch->arena.dependsOn(i->arena());
															#line 4944 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1481 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		int startSize = batch->changes.size();
															#line 1483 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		batch->changes.resize(batch->changes.size() + cacheRange->updates.size());
															#line 1486 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		std::copy(cacheRange->updates.begin(), cacheRange->updates.end(), batch->changes.begin() + startSize);
															#line 1487 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		Version checkv = cacheRange->transferredVersion;
															#line 1489 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		for(auto b = batch->changes.begin() + startSize;b != batch->changes.end();++b) {
															#line 1490 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			ASSERT(b->version >= checkv);
															#line 1491 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			checkv = b->version;
															#line 4960 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1497 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->updates.clear();
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_10 = data->oldestVersion.whenAtLeast(cacheRange->transferredVersion);
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 4968 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont10cont1when1(__when_expr_10.get(), loopDepth); };
		static_cast<FetchKeysActor*>(this)->actor_wait_state = 11;
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< FetchKeysActor, 10, Void >*>(static_cast<FetchKeysActor*>(this)));
															#line 4973 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10when1(FetchInjectionInfo* const& batch,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(batch, loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(FetchInjectionInfo* && batch,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(std::move(batch), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 9, FetchInjectionInfo* >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 9, FetchInjectionInfo* >*,FetchInjectionInfo* const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 9, FetchInjectionInfo* >*,FetchInjectionInfo* && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 9, FetchInjectionInfo* >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont10cont2(Void const& _,int loopDepth) 
	{
															#line 1507 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(data->cachedRangeMap[cacheRange->keys.begin]->assigned() && data->cachedRangeMap[cacheRange->keys.begin]->keys == cacheRange->keys);
															#line 1510 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->newestAvailableVersion.insert(cacheRange->keys, latestVersion);
															#line 1511 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->readWrite.send(Void());
															#line 1512 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->addCacheRange(CacheRangeInfo::newReadWrite(cacheRange->keys, data));
															#line 1513 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		coalesceCacheRanges(data, keys);
															#line 1515 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		validate(data);
															#line 5067 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont10cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont2(Void && _,int loopDepth) 
	{
															#line 1507 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(data->cachedRangeMap[cacheRange->keys.begin]->assigned() && data->cachedRangeMap[cacheRange->keys.begin]->keys == cacheRange->keys);
															#line 1510 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->newestAvailableVersion.insert(cacheRange->keys, latestVersion);
															#line 1511 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cacheRange->readWrite.send(Void());
															#line 1512 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->addCacheRange(CacheRangeInfo::newReadWrite(cacheRange->keys, data));
															#line 1513 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		coalesceCacheRanges(data, keys);
															#line 1515 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		validate(data);
															#line 5086 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont10cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<FetchKeysActor*>(this)->actor_wait_state > 0) static_cast<FetchKeysActor*>(this)->actor_wait_state = 0;
		static_cast<FetchKeysActor*>(this)->ActorCallback< FetchKeysActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< FetchKeysActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< FetchKeysActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont10cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* data;
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	AddingCacheRange* cacheRange;
															#line 1254 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	TraceInterval interval;
															#line 1255 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	KeyRange keys;
															#line 1257 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	double startt;
															#line 1259 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int fetchBlockBytes;
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	FlowLock::Releaser holdingFKPL;
															#line 1308 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version fetchVersion;
															#line 1319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int debug_getRangeRetries;
															#line 1320 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int debug_nextRetryToLog;
															#line 1321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	bool isTooOld;
															#line 1330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	RangeResult this_block;
															#line 1337 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int expectedSize;
															#line 1362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	KeyValueRef* kvItr;
															#line 5207 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via fetchKeys()
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class FetchKeysActor final : public Actor<Void>, public ActorCallback< FetchKeysActor, 0, Void >, public ActorCallback< FetchKeysActor, 1, Void >, public ActorCallback< FetchKeysActor, 2, Void >, public ActorCallback< FetchKeysActor, 3, Void >, public ActorCallback< FetchKeysActor, 4, Void >, public ActorCallback< FetchKeysActor, 5, RangeResult >, public ActorCallback< FetchKeysActor, 6, Void >, public ActorCallback< FetchKeysActor, 7, Void >, public ActorCallback< FetchKeysActor, 8, Void >, public ActorCallback< FetchKeysActor, 9, FetchInjectionInfo* >, public ActorCallback< FetchKeysActor, 10, Void >, public FastAllocated<FetchKeysActor>, public FetchKeysActorState<FetchKeysActor> {
															#line 5212 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<FetchKeysActor>::operator new;
	using FastAllocated<FetchKeysActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5850718379138268672UL, 18008283544871592960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchKeysActor, 0, Void >;
friend struct ActorCallback< FetchKeysActor, 1, Void >;
friend struct ActorCallback< FetchKeysActor, 2, Void >;
friend struct ActorCallback< FetchKeysActor, 3, Void >;
friend struct ActorCallback< FetchKeysActor, 4, Void >;
friend struct ActorCallback< FetchKeysActor, 5, RangeResult >;
friend struct ActorCallback< FetchKeysActor, 6, Void >;
friend struct ActorCallback< FetchKeysActor, 7, Void >;
friend struct ActorCallback< FetchKeysActor, 8, Void >;
friend struct ActorCallback< FetchKeysActor, 9, FetchInjectionInfo* >;
friend struct ActorCallback< FetchKeysActor, 10, Void >;
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	FetchKeysActor(StorageCacheData* const& data,AddingCacheRange* const& cacheRange) 
															#line 5239 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   FetchKeysActorState<FetchKeysActor>(data, cacheRange),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchKeys", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12649395534889573888UL, 8182797599733655808UL);
		ActorExecutionContextHelper __helper(static_cast<FetchKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchKeys");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchKeys", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchKeysActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchKeysActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchKeysActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FetchKeysActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FetchKeysActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FetchKeysActor, 5, RangeResult >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< FetchKeysActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< FetchKeysActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< FetchKeysActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< FetchKeysActor, 9, FetchInjectionInfo* >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< FetchKeysActor, 10, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> fetchKeys( StorageCacheData* const& data, AddingCacheRange* const& cacheRange ) {
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new FetchKeysActor(data, cacheRange));
															#line 5282 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 1554 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"


AddingCacheRange::AddingCacheRange(StorageCacheData* server, KeyRangeRef const& keys)
  : keys(keys), server(server), transferredVersion(invalidVersion), phase(WaitPrevious) {
	fetchClient = fetchKeys(server, this);
}

void AddingCacheRange::addMutation(Version version, MutationRef const& mutation) {
	if (mutation.type == mutation.ClearRange) {
		ASSERT(keys.begin <= mutation.param1 && mutation.param2 <= keys.end);
	} else if (isSingleKeyMutation((MutationRef::Type)mutation.type)) {
		ASSERT(keys.contains(mutation.param1));
	}

	if (phase == WaitPrevious) {
		// Updates can be discarded
	} else if (phase == Fetching) {
		if (!updates.size() || version > updates.end()[-1].version) {
			VerUpdateRef v;
			v.version = version;
			v.isPrivateData = false;
			updates.push_back(v);
		} else {
			ASSERT(version == updates.end()[-1].version);
		}
		updates.back().mutations.push_back_deep(updates.back().arena(), mutation);
	} else if (phase == Waiting) {
		server->addMutation(keys, version, mutation);
	} else
		ASSERT(false);
}

void CacheRangeInfo::addMutation(Version version, MutationRef const& mutation) {
	ASSERT((void*)this);
	ASSERT(keys.contains(mutation.param1));
	if (adding)
		adding->addMutation(version, mutation);
	else if (readWrite)
		readWrite->addMutation(this->keys, version, mutation);
	else if (mutation.type != MutationRef::ClearRange) { // TODO NEELAM: ClearRange mutations are ignored (why do we
		                                                 // even allow them on un-assigned range?)
		TraceEvent(SevError, "DeliveredToNotAssigned").detail("Version", version).detail("Mutation", mutation);
		ASSERT(false); // Mutation delivered to notAssigned cacheRange!
	}
}

void cacheWarmup(StorageCacheData* data, const KeyRangeRef& keys, bool nowAssigned, Version version) {

	ASSERT(!keys.empty());

	validate(data);

	// FIXME: enable when debugKeyRange is active
	// debugKeyRange( nowAssigned ? "KeysAssigned" : "KeysUnassigned", version, keys );

	bool isDifferent = false;
	auto existingCacheRanges = data->cachedRangeMap.intersectingRanges(keys);
	for (auto it = existingCacheRanges.begin(); it != existingCacheRanges.end(); ++it) {
		if (nowAssigned != it->value()->assigned()) {
			isDifferent = true;
			TraceEvent("SCWRangeDifferent", data->thisServerID)
			    .detail("KeyBegin", it->range().begin)
			    .detail("KeyEnd", it->range().end);
			break;
		}
	}
	if (!isDifferent) {
		TraceEvent("SCWShortCircuit", data->thisServerID).detail("KeyBegin", keys.begin).detail("KeyEnd", keys.end);
		return;
	}

	// Save a backup of the CacheRangeInfo references before we start messing with cacheRanges, in order to defer
	// fetchKeys cancellation (and its potential call to removeDataRange()) until cacheRanges is again valid
	std::vector<Reference<CacheRangeInfo>> oldCacheRanges;
	auto ocr = data->cachedRangeMap.intersectingRanges(keys);
	for (auto r = ocr.begin(); r != ocr.end(); ++r)
		oldCacheRanges.push_back(r->value());

	// As addCacheRange (called below)'s documentation requires, reinitialize any overlapping range(s)
	auto ranges = data->cachedRangeMap.getAffectedRangesAfterInsertion(
	    keys, Reference<CacheRangeInfo>()); // null reference indicates the range being changed
	for (int i = 0; i < ranges.size(); i++) {
		if (!ranges[i].value) {
			ASSERT((KeyRangeRef&)ranges[i] == keys); // there shouldn't be any nulls except for the range being inserted
		} else if (ranges[i].value->notAssigned())
			data->addCacheRange(CacheRangeInfo::newNotAssigned(ranges[i]));
		else if (ranges[i].value->isReadable())
			data->addCacheRange(CacheRangeInfo::newReadWrite(ranges[i], data));
		else {
			ASSERT(ranges[i].value->adding);
			data->addCacheRange(CacheRangeInfo::newAdding(data, ranges[i]));
			// CODE_PROBE(true, "cacheWarmup reFetchKeys");
		}
	}

	// CacheRange state depends on nowAssigned and whether the data is available (actually assigned in memory or on the
	// disk) up to the given version.  The latter depends on data->newestAvailableVersion, so loop over the ranges of
	// that. SOMEDAY: Could this just use cacheRanges?  Then we could explicitly do the removeDataRange here when an
	// adding/transferred cacheRange is cancelled
	auto vr = data->newestAvailableVersion.intersectingRanges(keys);
	std::vector<std::pair<KeyRange, Version>> changeNewestAvailable;
	std::vector<KeyRange> removeRanges;
	for (auto r = vr.begin(); r != vr.end(); ++r) {
		KeyRangeRef range = keys & r->range();
		bool dataAvailable = r->value() == latestVersion || r->value() >= version;
		/*TraceEvent("CSKRange", data->thisServerID)
		    .detail("KeyBegin", range.begin)
		    .detail("KeyEnd", range.end)
		    .detail("Available", dataAvailable)
		    .detail("NowAssigned", nowAssigned)
		    .detail("NewestAvailable", r->value())
		    .detail("CacheRangeState0", data->cachedRangeMap[range.begin]->debugDescribeState());*/
		if (!nowAssigned) {
			if (dataAvailable) {
				ASSERT(r->value() ==
				       latestVersion); // Not that we care, but this used to be checked instead of dataAvailable
				ASSERT(data->mutableData().getLatestVersion() > version);
				changeNewestAvailable.emplace_back(range, version);
				removeRanges.push_back(range);
			}
			data->addCacheRange(CacheRangeInfo::newNotAssigned(range));
		} else if (!dataAvailable) {
			// SOMEDAY: Avoid restarting adding/transferred cacheRanges
			if (version == 0) { // bypass fetchkeys; cacheRange is known empty at version 0
				changeNewestAvailable.emplace_back(range, latestVersion);
				data->addCacheRange(CacheRangeInfo::newReadWrite(range, data));
				// setAvailableStatus(data, range, true);
			} else {
				auto& cacheRange = data->cachedRangeMap[range.begin];
				if (!cacheRange->assigned() || cacheRange->keys != range)
					data->addCacheRange(CacheRangeInfo::newAdding(data, range));
			}
		} else {
			changeNewestAvailable.emplace_back(range, latestVersion);
			data->addCacheRange(CacheRangeInfo::newReadWrite(range, data));
		}
	}
	// Update newestAvailableVersion when a cacheRange becomes (un)available (in a separate loop to avoid invalidating
	// vr above)
	for (auto r = changeNewestAvailable.begin(); r != changeNewestAvailable.end(); ++r)
		data->newestAvailableVersion.insert(r->first, r->second);

	// TODO
	// if (!nowAssigned)
	//	data->metrics.notifyNotReadable( keys );

	coalesceCacheRanges(data, KeyRangeRef(ranges[0].begin, ranges[ranges.size() - 1].end));

	// Now it is OK to do removeDataRanges, directly and through fetchKeys cancellation (and we have to do so before
	// validate())
	oldCacheRanges.clear();
	ranges.clear();
	for (auto r = removeRanges.begin(); r != removeRanges.end(); ++r) {
		removeDataRange(data, data->addVersionToMutationLog(data->data().getLatestVersion()), data->cachedRangeMap, *r);
		// setAvailableStatus(data, *r, false);
	}
	validate(data);
}

// Helper class for updating the storage cache (i.e. applying mutations)
class StorageCacheUpdater {
public:
	StorageCacheUpdater()
	  : currentVersion(invalidVersion), fromVersion(invalidVersion), processedCacheStartKey(false) {}
	StorageCacheUpdater(Version currentVersion)
	  : currentVersion(invalidVersion), fromVersion(currentVersion), processedCacheStartKey(false) {}

	void applyMutation(StorageCacheData* data, MutationRef const& m, Version ver) {
		//TraceEvent("SCNewVersion", data->thisServerID).detail("VerWas", data->mutableData().latestVersion).detail("ChVer", ver);

		if (currentVersion != ver) {
			fromVersion = currentVersion;
			currentVersion = ver;
			data->mutableData().createNewVersion(ver);
		}

		DEBUG_MUTATION("SCUpdateMutation", ver, m);
		if (m.param1.startsWith(systemKeys.end)) {
			//TraceEvent("SCPrivateData", data->thisServerID).detail("Mutation", m).detail("Version", ver);
			applyPrivateCacheData(data, m);
		} else {
			splitMutation(data, data->cachedRangeMap, m, ver);
		}

		// TODO
		if (data->otherError.getFuture().isReady())
			data->otherError.getFuture().get();
	}

	Version currentVersion;

private:
	Version fromVersion;
	KeyRef cacheStartKey;
	bool nowAssigned;
	bool processedCacheStartKey;

	// Applies private mutations, as the name suggests. It basically establishes the key-ranges
	// that this cache server is responsible for
	// TODO Revisit during failure handling. Might we loose some private mutations?
	void applyPrivateCacheData(StorageCacheData* data, MutationRef const& m) {

		//TraceEvent(SevDebug, "SCPrivateCacheMutation", data->thisServerID).detail("Mutation", m);

		if (processedCacheStartKey) {
			// we expect changes in pairs, [begin,end). This mutation is for end key of the range
			ASSERT(m.type == MutationRef::SetValue && m.param1.startsWith(data->ck));
			KeyRangeRef keys(cacheStartKey.removePrefix(data->ck), m.param1.removePrefix(data->ck));
			// setAssignedStatus( data, keys, nowAssigned );
			// data->cachedRangeMap.insert(keys, true);
			// fprintf(stderr, "SCPrivateCacheMutation: begin: %s, end: %s\n", printable(keys.begin).c_str(),
			// printable(keys.end).c_str());

			// Warmup the cache for the newly added key-range
			cacheWarmup(data, /*this,*/ keys, nowAssigned, currentVersion - 1);
			processedCacheStartKey = false;
		} else if (m.type == MutationRef::SetValue && m.param1.startsWith(data->ck)) {
			// We expect changes in pairs, [begin,end), This mutation is for start key of the range
			cacheStartKey = m.param1;
			nowAssigned = m.param2 != serverKeysFalse;
			processedCacheStartKey = true;
		} else if (m.type == MutationRef::SetValue && m.param1 == lastEpochEndPrivateKey) {
			// lastEpochEnd transactions are guaranteed by the master to be alone in their own batch (version)
			// That means we don't have to worry about the impact on changeServerKeys

			Version rollbackVersion;
			BinaryReader br(m.param2, Unversioned());
			br >> rollbackVersion;

			if (rollbackVersion < fromVersion && rollbackVersion > data->oldestVersion.get()) {
				// CODE_PROBE(true, "CacheRangeApplyPrivateData cacheRange rollback");
				TraceEvent(SevWarn, "Rollback", data->thisServerID)
				    .detail("FromVersion", fromVersion)
				    .detail("ToVersion", rollbackVersion)
				    .detail("AtVersion", currentVersion)
				    .detail("OldestVersion", data->oldestVersion.get());
				rollback(data, rollbackVersion, currentVersion);
			}
		} else {
			TraceEvent(SevWarn, "SCPrivateCacheMutation: Unknown private mutation").log();
			// ASSERT(false);  // Unknown private mutation
		}
	}
};

void applyMutation(StorageCacheUpdater* updater, StorageCacheData* data, MutationRef const& mutation, Version version) {
	updater->applyMutation(data, mutation, version);
}

// Compacts the in-memory VersionedMap, i.e. removes versions below the desiredOldestVersion
// TODO revisit if we change the data structure of the VersionedMap
															#line 5537 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via compactCache()
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class CompactCacheActor>
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class CompactCacheActorState {
															#line 5544 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	CompactCacheActorState(StorageCacheData* const& data) 
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : data(data)
															#line 5551 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("compactCache", reinterpret_cast<unsigned long>(this));

	}
	~CompactCacheActorState() 
	{
		fdb_probe_actor_destroy("compactCache", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1806 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			;
															#line 5566 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CompactCacheActorState();
		static_cast<CompactCacheActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1816 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_0 = data->desiredOldestVersion.whenAtLeast(data->oldestVersion.get() + 1);
															#line 1816 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5598 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 1;
															#line 1816 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 0, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5603 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::CompactCache);
															#line 1817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5614 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 2;
															#line 1817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 1, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5619 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::CompactCache);
															#line 1817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5630 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 2;
															#line 1817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 1, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5635 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state > 0) static_cast<CompactCacheActor*>(this)->actor_wait_state = 0;
		static_cast<CompactCacheActor*>(this)->ActorCallback< CompactCacheActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CompactCacheActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1820 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		compactionInProgress = Promise<Void>();
															#line 1821 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->compactionInProgress = compactionInProgress.getFuture();
															#line 1823 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		desiredVersion = data->desiredOldestVersion.get();
															#line 1827 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->mutableData().compact(desiredVersion);
															#line 1828 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		Future<Void> finishedForgetting = data->mutableData().forgetVersionsBeforeAsync(desiredVersion, TaskPriority::CompactCache);
															#line 1830 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->oldestVersion.set(desiredVersion);
															#line 1831 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_2 = finishedForgetting;
															#line 1831 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5733 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 3;
															#line 1831 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 2, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5738 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1820 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		compactionInProgress = Promise<Void>();
															#line 1821 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->compactionInProgress = compactionInProgress.getFuture();
															#line 1823 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		desiredVersion = data->desiredOldestVersion.get();
															#line 1827 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->mutableData().compact(desiredVersion);
															#line 1828 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		Future<Void> finishedForgetting = data->mutableData().forgetVersionsBeforeAsync(desiredVersion, TaskPriority::CompactCache);
															#line 1830 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->oldestVersion.set(desiredVersion);
															#line 1831 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_2 = finishedForgetting;
															#line 1831 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5761 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 3;
															#line 1831 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 2, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5766 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state > 0) static_cast<CompactCacheActor*>(this)->actor_wait_state = 0;
		static_cast<CompactCacheActor*>(this)->ActorCallback< CompactCacheActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CompactCacheActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1835 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_3 = yield(TaskPriority::CompactCache);
															#line 1835 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5852 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 4;
															#line 1835 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 3, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5857 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1835 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_3 = yield(TaskPriority::CompactCache);
															#line 1835 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 4;
															#line 1835 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 3, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5873 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state > 0) static_cast<CompactCacheActor*>(this)->actor_wait_state = 0;
		static_cast<CompactCacheActor*>(this)->ActorCallback< CompactCacheActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CompactCacheActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1838 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		compactionInProgress.send(Void());
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(2.0);
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5961 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 5;
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 4, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5966 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1838 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		compactionInProgress.send(Void());
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(2.0);
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5979 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<CompactCacheActor*>(this)->actor_wait_state = 5;
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CompactCacheActor, 4, Void >*>(static_cast<CompactCacheActor*>(this)));
															#line 5984 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state > 0) static_cast<CompactCacheActor*>(this)->actor_wait_state = 0;
		static_cast<CompactCacheActor*>(this)->ActorCallback< CompactCacheActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CompactCacheActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<CompactCacheActor*>(this)->actor_wait_state > 0) static_cast<CompactCacheActor*>(this)->actor_wait_state = 0;
		static_cast<CompactCacheActor*>(this)->ActorCallback< CompactCacheActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CompactCacheActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< CompactCacheActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* data;
															#line 1820 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Promise<Void> compactionInProgress;
															#line 1823 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version desiredVersion;
															#line 6157 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via compactCache()
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class CompactCacheActor final : public Actor<Void>, public ActorCallback< CompactCacheActor, 0, Void >, public ActorCallback< CompactCacheActor, 1, Void >, public ActorCallback< CompactCacheActor, 2, Void >, public ActorCallback< CompactCacheActor, 3, Void >, public ActorCallback< CompactCacheActor, 4, Void >, public FastAllocated<CompactCacheActor>, public CompactCacheActorState<CompactCacheActor> {
															#line 6162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<CompactCacheActor>::operator new;
	using FastAllocated<CompactCacheActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15674058025648427776UL, 17450202420943773184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CompactCacheActor, 0, Void >;
friend struct ActorCallback< CompactCacheActor, 1, Void >;
friend struct ActorCallback< CompactCacheActor, 2, Void >;
friend struct ActorCallback< CompactCacheActor, 3, Void >;
friend struct ActorCallback< CompactCacheActor, 4, Void >;
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	CompactCacheActor(StorageCacheData* const& data) 
															#line 6183 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   CompactCacheActorState<CompactCacheActor>(data),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("compactCache", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3102351636106060032UL, 848003727438058496UL);
		ActorExecutionContextHelper __helper(static_cast<CompactCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("compactCache");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("compactCache", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CompactCacheActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CompactCacheActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CompactCacheActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CompactCacheActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< CompactCacheActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> compactCache( StorageCacheData* const& data ) {
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new CompactCacheActor(data));
															#line 6220 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 1844 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

															#line 6225 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via pullAsyncData()
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class PullAsyncDataActor>
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class PullAsyncDataActorState {
															#line 6232 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	PullAsyncDataActorState(StorageCacheData* const& data) 
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : data(data),
															#line 1846 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   dbInfoChange(Void()),
															#line 1847 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   cursor(),
															#line 1848 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   tagAt(0),
															#line 1849 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   start(now()),
															#line 1850 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   ver(invalidVersion)
															#line 6249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("pullAsyncData", reinterpret_cast<unsigned long>(this));

	}
	~PullAsyncDataActorState() 
	{
		fdb_probe_actor_destroy("pullAsyncData", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1851 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			++data->counters.updateBatches;
															#line 1853 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			;
															#line 6266 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PullAsyncDataActorState();
		static_cast<PullAsyncDataActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1854 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 6296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		try {
															#line 1873 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (cursor->version().version <= cursor->popped())
															#line 6306 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1874 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1loopBody1cont1Catch1(please_reboot(), loopDepth);
															#line 6310 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1876 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->lastTLogVersion = cursor->getMaxKnownVersion();
															#line 1877 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->versionLag = std::max<int64_t>(0, data->lastTLogVersion - data->version.get());
															#line 1879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			start = now();
															#line 1880 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_2 = data->updateVersionLock.take(TaskPriority::TLogPeekReply, 1);
															#line 1880 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<PullAsyncDataActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 6322 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<PullAsyncDataActor*>(this)->actor_wait_state = 2;
															#line 1880 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PullAsyncDataActor, 2, Void >*>(static_cast<PullAsyncDataActor*>(this)));
															#line 6327 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 1855 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_0 = cursor ? cursor->getMore(TaskPriority::TLogCommit) : Never();
															#line 1854 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<PullAsyncDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 6351 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 1858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_1 = dbInfoChange;
															#line 6355 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<PullAsyncDataActor*>(this)->actor_wait_state = 1;
															#line 1855 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PullAsyncDataActor, 0, Void >*>(static_cast<PullAsyncDataActor*>(this)));
															#line 1858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PullAsyncDataActor, 1, Void >*>(static_cast<PullAsyncDataActor*>(this)));
															#line 6362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 1859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		dbInfoChange = data->db->onChange();
															#line 1860 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (data->db->get().recoveryState >= RecoveryState::ACCEPTING_COMMITS)
															#line 6404 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1861 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->logSystem = ILogSystem::fromServerDBInfo(data->thisServerID, data->db->get());
															#line 1862 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (data->logSystem)
															#line 6410 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				cursor = data->logSystem->peekSingle( data->thisServerID, data->peekVersion, cacheTag, std::vector<std::pair<Version, Tag>>());
															#line 6414 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 1866 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				cursor = Reference<ILogSystem::IPeekCursor>();
															#line 6420 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 1859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		dbInfoChange = data->db->onChange();
															#line 1860 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (data->db->get().recoveryState >= RecoveryState::ACCEPTING_COMMITS)
															#line 6433 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1861 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->logSystem = ILogSystem::fromServerDBInfo(data->thisServerID, data->db->get());
															#line 1862 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (data->logSystem)
															#line 6439 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				cursor = data->logSystem->peekSingle( data->thisServerID, data->peekVersion, cacheTag, std::vector<std::pair<Version, Tag>>());
															#line 6443 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 1866 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				cursor = Reference<ILogSystem::IPeekCursor>();
															#line 6449 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PullAsyncDataActor*>(this)->actor_wait_state > 0) static_cast<PullAsyncDataActor*>(this)->actor_wait_state = 0;
		static_cast<PullAsyncDataActor*>(this)->ActorCallback< PullAsyncDataActor, 0, Void >::remove();
		static_cast<PullAsyncDataActor*>(this)->ActorCallback< PullAsyncDataActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PullAsyncDataActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PullAsyncDataActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 2134 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		tagAt = std::max(tagAt, cursor->version().version);
															#line 6581 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& err,int loopDepth=0) 
	{
		try {
															#line 2125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			e = err;
															#line 2126 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevDebug, "SCUpdateError", data->thisServerID).error(e).backtrace();
															#line 2127 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (e.code() == error_code_worker_removed)
															#line 6595 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 2128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch1(please_reboot(), std::max(0, loopDepth - 1));
															#line 6599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 2130 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 6605 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1881 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		holdingDVL = FlowLock::Releaser(data->updateVersionLock);
															#line 1882 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (now() - start > 0.1)
															#line 6622 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1883 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent("SCSlowTakeLock1", data->thisServerID) .detailf("From", "%016llx", debug_lastLoadBalanceResultEndpointToken) .detail("Duration", now() - start) .detail("Version", data->version.get());
															#line 6626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1888 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		fii = FetchInjectionInfo();
															#line 1889 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cloneCursor2 = Reference<ILogSystem::IPeekCursor>();
															#line 1890 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cipherKeys = Optional<std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>>>();
															#line 1891 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		collectingCipherKeys = false;
															#line 1893 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 6638 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1881 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		holdingDVL = FlowLock::Releaser(data->updateVersionLock);
															#line 1882 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (now() - start > 0.1)
															#line 6649 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1883 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent("SCSlowTakeLock1", data->thisServerID) .detailf("From", "%016llx", debug_lastLoadBalanceResultEndpointToken) .detail("Duration", now() - start) .detail("Version", data->version.get());
															#line 6653 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1888 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		fii = FetchInjectionInfo();
															#line 1889 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cloneCursor2 = Reference<ILogSystem::IPeekCursor>();
															#line 1890 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cipherKeys = Optional<std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>>>();
															#line 1891 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		collectingCipherKeys = false;
															#line 1893 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 6665 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PullAsyncDataActor*>(this)->actor_wait_state > 0) static_cast<PullAsyncDataActor*>(this)->actor_wait_state = 0;
		static_cast<PullAsyncDataActor*>(this)->ActorCallback< PullAsyncDataActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PullAsyncDataActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 1976 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->debug_inApplyUpdate = true;
															#line 1977 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (EXPENSIVE_VALIDATION)
															#line 6751 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1978 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->data().atLatest().validate();
															#line 6755 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 1979 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		validate(data);
															#line 1981 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		injectedChanges = false;
															#line 1982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		changeNum = 0;
															#line 1983 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		mutationBytes = 0;
															#line 1984 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 6767 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1loopBody1cont5loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1894 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		changeCounter = data->cacheRangeChangeCounter;
															#line 1895 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		bool epochEnd = false;
															#line 1896 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		bool hasPrivateData = false;
															#line 1897 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		bool firstMutation = true;
															#line 1898 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		bool dbgLastMessageWasProtocol = false;
															#line 1900 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		std::unordered_set<BlobCipherDetails> cipherDetails;
															#line 1902 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		Reference<ILogSystem::IPeekCursor> cloneCursor1 = cursor->cloneNoMore();
															#line 1903 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cloneCursor2 = cursor->cloneNoMore();
															#line 1907 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cloneCursor1->setProtocolVersion(g_network->protocolVersion());
															#line 1909 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		for(;cloneCursor1->hasMessage();cloneCursor1->nextMessage()) {
															#line 1910 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			ArenaReader& cloneReader = *cloneCursor1->reader();
															#line 1912 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (LogProtocolMessage::isNextIn(cloneReader))
															#line 6805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 1913 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				LogProtocolMessage lpm;
															#line 1914 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				cloneReader >> lpm;
															#line 1915 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				dbgLastMessageWasProtocol = true;
															#line 1916 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				cloneCursor1->setProtocolVersion(cloneReader.protocolVersion());
															#line 6815 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 1917 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (cloneReader.protocolVersion().hasSpanContext() && SpanContextMessage::isNextIn(cloneReader))
															#line 6821 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 1919 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					SpanContextMessage scm;
															#line 1920 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					cloneReader >> scm;
															#line 6827 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
				else
				{
															#line 1921 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					if (cloneReader.protocolVersion().hasOTELSpanContext() && OTELSpanContextMessage::isNextIn(cloneReader))
															#line 6833 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
					{
															#line 1923 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						OTELSpanContextMessage scm;
															#line 1924 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						cloneReader >> scm;
															#line 6839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
					}
					else
					{
															#line 1926 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						MutationRef msg;
															#line 1927 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						cloneReader >> msg;
															#line 1928 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						if (msg.isEncrypted())
															#line 6849 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
						{
															#line 1929 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							if (!cipherKeys.present())
															#line 6853 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
							{
															#line 1930 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
								msg.updateEncryptCipherDetails(cipherDetails);
															#line 1931 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
								collectingCipherKeys = true;
															#line 6859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
							}
							else
							{
															#line 1933 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
								msg = msg.decrypt(cipherKeys.get(), cloneReader.arena(), BlobCipherMetrics::TLOG);
															#line 6865 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
							}
						}
															#line 1936 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						if (!collectingCipherKeys)
															#line 6870 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
						{
															#line 1937 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							if (firstMutation && msg.param1.startsWith(systemKeys.end))
															#line 6874 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
							{
															#line 1938 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
								hasPrivateData = true;
															#line 6878 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
							}
															#line 1939 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							firstMutation = false;
															#line 1941 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							if (msg.param1 == lastEpochEndPrivateKey)
															#line 6884 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
							{
															#line 1942 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
								epochEnd = true;
															#line 1944 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
								ASSERT(dbgLastMessageWasProtocol);
															#line 6890 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
							}
															#line 1947 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							dbgLastMessageWasProtocol = false;
															#line 6894 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
						}
					}
				}
			}
		}
															#line 1952 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (collectingCipherKeys)
															#line 6902 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1953 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>>> __when_expr_3 = GetEncryptCipherKeys<ServerDBInfo>::getEncryptCipherKeys( data->db, cipherDetails, BlobCipherMetrics::TLOG);
															#line 1953 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<PullAsyncDataActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6908 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_3.get(), loopDepth); };
			static_cast<PullAsyncDataActor*>(this)->actor_wait_state = 3;
															#line 1953 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PullAsyncDataActor, 3, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*>(static_cast<PullAsyncDataActor*>(this)));
															#line 6913 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1962 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			for(;!hasPrivateData && !epochEnd && !data->readyFetchKeys.empty();) {
															#line 1963 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				auto fk = data->readyFetchKeys.back();
															#line 1964 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				data->readyFetchKeys.pop_back();
															#line 1965 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				fk.send(&fii);
															#line 6926 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (data->cacheRangeChangeCounter == changeCounter)
															#line 6930 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
				return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
			}
			loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont15(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> const& result,int loopDepth) 
	{
															#line 1956 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cipherKeys = result;
															#line 1957 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		collectingCipherKeys = false;
															#line 6964 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont15(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> && result,int loopDepth) 
	{
															#line 1956 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cipherKeys = result;
															#line 1957 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		collectingCipherKeys = false;
															#line 6975 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> const& result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont15(result, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> && result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont15(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PullAsyncDataActor*>(this)->actor_wait_state > 0) static_cast<PullAsyncDataActor*>(this)->actor_wait_state = 0;
		static_cast<PullAsyncDataActor*>(this)->ActorCallback< PullAsyncDataActor, 3, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >::remove();

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 3, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> const& value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 3, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> && value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PullAsyncDataActor, 3, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,Error err) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
															#line 2001 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cloneCursor2->setProtocolVersion(g_network->protocolVersion());
															#line 2002 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ver = invalidVersion;
															#line 2005 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		for(;cloneCursor2->hasMessage();cloneCursor2->nextMessage()) {
															#line 2006 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			ArenaReader& reader = *cloneCursor2->reader();
															#line 2008 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (cloneCursor2->version().version > ver && cloneCursor2->version().version > data->version.get())
															#line 7067 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 2009 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				++data->counters.updateVersions;
															#line 2010 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				ver = cloneCursor2->version().version;
															#line 7073 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 2012 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (LogProtocolMessage::isNextIn(reader))
															#line 7077 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 2013 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				LogProtocolMessage lpm;
															#line 2014 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				reader >> lpm;
															#line 2017 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				data->logProtocol = reader.protocolVersion();
															#line 2018 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				cloneCursor2->setProtocolVersion(data->logProtocol);
															#line 7087 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 2019 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				if (reader.protocolVersion().hasSpanContext() && SpanContextMessage::isNextIn(reader))
															#line 7093 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				{
															#line 2020 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					SpanContextMessage scm;
															#line 2021 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					reader >> scm;
															#line 7099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				}
				else
				{
															#line 2022 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
					if (reader.protocolVersion().hasOTELSpanContext() && OTELSpanContextMessage::isNextIn(reader))
															#line 7105 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
					{
															#line 2024 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						OTELSpanContextMessage oscm;
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						reader >> oscm;
															#line 7111 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
					}
					else
					{
															#line 2027 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						MutationRef msg;
															#line 2028 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						reader >> msg;
															#line 2029 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						if (msg.isEncrypted())
															#line 7121 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
						{
															#line 2030 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							msg = msg.decrypt(cipherKeys.get(), reader.arena(), BlobCipherMetrics::TLOG);
															#line 7125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
						}
															#line 2033 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						if (ver != invalidVersion)
															#line 7129 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
						{
															#line 2035 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							applyMutation(data->updater, data, msg, ver);
															#line 2036 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							data->counters.mutationBytes += msg.totalSize();
															#line 2037 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							++data->counters.mutations;
															#line 2039 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							switch (msg.type) { case MutationRef::SetValue: ++data->counters.setMutations; break; case MutationRef::ClearRange: ++data->counters.clearRangeMutations; break; case MutationRef::AddValue: case MutationRef::And: case MutationRef::AndV2: case MutationRef::AppendIfFits: case MutationRef::ByteMax: case MutationRef::ByteMin: case MutationRef::Max: case MutationRef::Min: case MutationRef::MinV2: case MutationRef::Or: case MutationRef::Xor: case MutationRef::CompareAndClear: ++data->counters.atomicMutations; break; };
															#line 7139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
						}
						else
						{
															#line 2062 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
							TraceEvent(SevError, "DiscardingPeekedData", data->thisServerID) .detail("Mutation", msg) .detail("CursorVersion", cloneCursor2->version().version) .detail("DataVersion", data->version.get());
															#line 7145 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
						}
															#line 2068 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
						tagAt = cursor->version().version + 1;
															#line 7149 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
					}
				}
			}
		}
															#line 2072 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (ver != invalidVersion)
															#line 7156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 2073 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->lastVersionWithData = ver;
															#line 7160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		else
		{
															#line 2075 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			ver = cloneCursor2->version().version - 1;
															#line 7166 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 2077 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (injectedChanges)
															#line 7170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 2078 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->lastVersionWithData = ver;
															#line 7174 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 2080 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->debug_inApplyUpdate = false;
															#line 2082 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (ver != invalidVersion && ver > data->version.get())
															#line 7180 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 2083 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			DEBUG_KEY_RANGE("SCUpdate", ver, allKeys);
															#line 2085 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->mutableData().createNewVersion(ver);
															#line 2088 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (data->otherError.getFuture().isReady())
															#line 7188 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 2089 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				data->otherError.getFuture().get();
															#line 7192 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 2094 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->version.set(ver);
															#line 2095 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->peekVersion = ver + 1;
															#line 2100 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (data->otherError.getFuture().isReady())
															#line 7200 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 2101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				data->otherError.getFuture().get();
															#line 7204 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 2105 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			Version maxVersionsInMemory = SERVER_KNOBS->MAX_READ_TRANSACTION_LIFE_VERSIONS;
															#line 2106 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			Version proposedOldestVersion = data->version.get() - maxVersionsInMemory;
															#line 2107 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			proposedOldestVersion = std::max(proposedOldestVersion, data->oldestVersion.get());
															#line 2108 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->desiredOldestVersion.set(proposedOldestVersion);
															#line 7214 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
															#line 2111 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		validate(data);
															#line 2113 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->lastTLogVersion = cloneCursor2->getMaxKnownVersion();
															#line 2114 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		cursor->advanceTo(cloneCursor2->version());
															#line 2115 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		data->versionLag = std::max<int64_t>(0, data->lastTLogVersion - data->version.get());
															#line 2116 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (cursor->version().version >= data->lastTLogVersion)
															#line 7226 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 2117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (data->behind)
															#line 7230 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 2118 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				TraceEvent("StorageCacheNoLongerBehind", data->thisServerID) .detail("CursorVersion", cursor->version().version) .detail("TLogVersion", data->lastTLogVersion);
															#line 7234 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
															#line 2122 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			data->behind = false;
															#line 7238 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont29(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont5loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1(int loopDepth) 
	{
															#line 1984 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!(changeNum < fii.changes.size()))
															#line 7255 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
			return a_body1loopBody1cont5break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1985 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		mutationNum = 0;
															#line 1986 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		pUpdate = &fii.changes[changeNum];
															#line 1987 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 7265 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1loopBody1cont5loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1cont1(int loopDepth) 
	{
															#line 1984 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		changeNum++;
															#line 7287 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont5loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1loopBody1(int loopDepth) 
	{
															#line 1987 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!(mutationNum < pUpdate->mutations.size()))
															#line 7303 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
			return a_body1loopBody1cont5loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1988 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		TraceEvent("SCInjectedChanges", data->thisServerID).detail("Version", pUpdate->version);
															#line 1989 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		applyMutation(data->updater, data, pUpdate->mutations[mutationNum], pUpdate->version);
															#line 1990 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		mutationBytes += pUpdate->mutations[mutationNum].totalSize();
															#line 1991 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		injectedChanges = true;
															#line 1992 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (false && mutationBytes > SERVER_KNOBS->DESIRED_UPDATE_BYTES)
															#line 7317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 1993 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			mutationBytes = 0;
															#line 1994 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(SERVER_KNOBS->UPDATE_DELAY);
															#line 1994 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<PullAsyncDataActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 7325 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont5loopBody1loopBody1when1(__when_expr_4.get(), loopDepth); };
			static_cast<PullAsyncDataActor*>(this)->actor_wait_state = 4;
															#line 1994 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< PullAsyncDataActor, 4, Void >*>(static_cast<PullAsyncDataActor*>(this)));
															#line 7330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont5loopBody1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont5loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1987 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		mutationNum++;
															#line 7357 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont5loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PullAsyncDataActor*>(this)->actor_wait_state > 0) static_cast<PullAsyncDataActor*>(this)->actor_wait_state = 0;
		static_cast<PullAsyncDataActor*>(this)->ActorCallback< PullAsyncDataActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< PullAsyncDataActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< PullAsyncDataActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont29(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* data;
															#line 1846 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Future<Void> dbInfoChange;
															#line 1847 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Reference<ILogSystem::IPeekCursor> cursor;
															#line 1848 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version tagAt;
															#line 1849 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	double start;
															#line 1850 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version ver;
															#line 1881 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	FlowLock::Releaser holdingDVL;
															#line 1888 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	FetchInjectionInfo fii;
															#line 1889 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Reference<ILogSystem::IPeekCursor> cloneCursor2;
															#line 1890 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Optional<std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>>> cipherKeys;
															#line 1891 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	bool collectingCipherKeys;
															#line 1894 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	uint64_t changeCounter;
															#line 1981 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	bool injectedChanges;
															#line 1982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int changeNum;
															#line 1983 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int mutationBytes;
															#line 1985 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	int mutationNum;
															#line 1986 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	VerUpdateRef* pUpdate;
															#line 2125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Error e;
															#line 7498 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via pullAsyncData()
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class PullAsyncDataActor final : public Actor<Void>, public ActorCallback< PullAsyncDataActor, 0, Void >, public ActorCallback< PullAsyncDataActor, 1, Void >, public ActorCallback< PullAsyncDataActor, 2, Void >, public ActorCallback< PullAsyncDataActor, 3, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >, public ActorCallback< PullAsyncDataActor, 4, Void >, public FastAllocated<PullAsyncDataActor>, public PullAsyncDataActorState<PullAsyncDataActor> {
															#line 7503 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<PullAsyncDataActor>::operator new;
	using FastAllocated<PullAsyncDataActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4423111402720948736UL, 10326625094548387328UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PullAsyncDataActor, 0, Void >;
friend struct ActorCallback< PullAsyncDataActor, 1, Void >;
friend struct ActorCallback< PullAsyncDataActor, 2, Void >;
friend struct ActorCallback< PullAsyncDataActor, 3, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >;
friend struct ActorCallback< PullAsyncDataActor, 4, Void >;
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	PullAsyncDataActor(StorageCacheData* const& data) 
															#line 7524 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   PullAsyncDataActorState<PullAsyncDataActor>(data),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("pullAsyncData", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8097422354448463360UL, 16625194706777794048UL);
		ActorExecutionContextHelper __helper(static_cast<PullAsyncDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("pullAsyncData");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("pullAsyncData", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PullAsyncDataActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PullAsyncDataActor, 2, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PullAsyncDataActor, 3, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PullAsyncDataActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> pullAsyncData( StorageCacheData* const& data ) {
															#line 1845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new PullAsyncDataActor(data));
															#line 7560 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 2137 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

// Fetch metadata mutation from the database to establish cache ranges and apply them
															#line 7566 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via storageCacheStartUpWarmup()
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class StorageCacheStartUpWarmupActor>
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class StorageCacheStartUpWarmupActorState {
															#line 7573 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheStartUpWarmupActorState(StorageCacheData* const& self) 
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : self(self),
															#line 2140 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   tr(self->cx),
															#line 2141 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   trueValue(storageCacheValue(std::vector<uint16_t>{ 0 })),
															#line 2142 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   falseValue(storageCacheValue(std::vector<uint16_t>{})),
															#line 2143 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   privatized()
															#line 7588 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this));

	}
	~StorageCacheStartUpWarmupActorState() 
	{
		fdb_probe_actor_destroy("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2144 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			privatized.type = MutationRef::SetValue;
															#line 2145 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			readVersion = Version();
															#line 7605 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			try {
															#line 2147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				;
															#line 7609 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StorageCacheStartUpWarmupActorState();
		static_cast<StorageCacheStartUpWarmupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2188 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!static_cast<StorageCacheStartUpWarmupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StorageCacheStartUpWarmupActorState(); static_cast<StorageCacheStartUpWarmupActor*>(this)->destroy(); return 0; }
															#line 7638 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		new (&static_cast<StorageCacheStartUpWarmupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StorageCacheStartUpWarmupActorState();
		static_cast<StorageCacheStartUpWarmupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2185 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			TraceEvent(SevError, "SCStartUpFailed").error(e);
															#line 2186 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 7653 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2148 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 2149 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 7682 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		try {
															#line 2151 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(storageCacheKeys, CLIENT_KNOBS->TOO_MANY);
															#line 2151 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<StorageCacheStartUpWarmupActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7688 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<StorageCacheStartUpWarmupActor*>(this)->actor_wait_state = 1;
															#line 2151 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StorageCacheStartUpWarmupActor, 0, RangeResult >*>(static_cast<StorageCacheStartUpWarmupActor*>(this)));
															#line 7693 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2181 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2181 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<StorageCacheStartUpWarmupActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7730 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<StorageCacheStartUpWarmupActor*>(this)->actor_wait_state = 2;
															#line 2181 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StorageCacheStartUpWarmupActor, 1, Void >*>(static_cast<StorageCacheStartUpWarmupActor*>(this)));
															#line 7735 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(RangeResult const& range,int loopDepth) 
	{
															#line 2152 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(!range.more);
															#line 2153 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		readVersion = tr.getReadVersion().get();
															#line 2154 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		bool currCached = false;
															#line 2155 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		KeyRef begin, end;
															#line 2156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		for( const auto& kv : range ) {
															#line 2158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			ASSERT(currCached == (kv.value == falseValue));
															#line 2159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (kv.value == trueValue)
															#line 7762 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 2160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				begin = kv.key;
															#line 2161 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				privatized.param1 = begin.withPrefix(systemKeys.begin, privatized.arena());
															#line 2162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				privatized.param2 = serverKeysTrue;
															#line 2166 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				applyMutation(self->updater, self, privatized, readVersion);
															#line 2167 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				currCached = true;
															#line 7774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 2169 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				currCached = false;
															#line 2170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				end = kv.key;
															#line 2171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				privatized.param1 = begin.withPrefix(systemKeys.begin, privatized.arena());
															#line 2172 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				privatized.param2 = serverKeysFalse;
															#line 2175 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				applyMutation(self->updater, self, privatized, readVersion);
															#line 7788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
															#line 2178 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		self->peekVersion = readVersion + 1;
															#line 7793 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(RangeResult && range,int loopDepth) 
	{
															#line 2152 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(!range.more);
															#line 2153 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		readVersion = tr.getReadVersion().get();
															#line 2154 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		bool currCached = false;
															#line 2155 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		KeyRef begin, end;
															#line 2156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		for( const auto& kv : range ) {
															#line 2158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			ASSERT(currCached == (kv.value == falseValue));
															#line 2159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (kv.value == trueValue)
															#line 7814 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			{
															#line 2160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				begin = kv.key;
															#line 2161 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				privatized.param1 = begin.withPrefix(systemKeys.begin, privatized.arena());
															#line 2162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				privatized.param2 = serverKeysTrue;
															#line 2166 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				applyMutation(self->updater, self, privatized, readVersion);
															#line 2167 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				currCached = true;
															#line 7826 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
			else
			{
															#line 2169 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				currCached = false;
															#line 2170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				end = kv.key;
															#line 2171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				privatized.param1 = begin.withPrefix(systemKeys.begin, privatized.arena());
															#line 2172 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				privatized.param2 = serverKeysFalse;
															#line 2175 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
				applyMutation(self->updater, self, privatized, readVersion);
															#line 7840 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			}
		}
															#line 2178 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		self->peekVersion = readVersion + 1;
															#line 7845 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& range,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(range, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && range,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(range), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StorageCacheStartUpWarmupActor*>(this)->actor_wait_state > 0) static_cast<StorageCacheStartUpWarmupActor*>(this)->actor_wait_state = 0;
		static_cast<StorageCacheStartUpWarmupActor*>(this)->ActorCallback< StorageCacheStartUpWarmupActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< StorageCacheStartUpWarmupActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheStartUpWarmupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StorageCacheStartUpWarmupActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheStartUpWarmupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StorageCacheStartUpWarmupActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheStartUpWarmupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<StorageCacheStartUpWarmupActor*>(this)->actor_wait_state > 0) static_cast<StorageCacheStartUpWarmupActor*>(this)->actor_wait_state = 0;
		static_cast<StorageCacheStartUpWarmupActor*>(this)->ActorCallback< StorageCacheStartUpWarmupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StorageCacheStartUpWarmupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheStartUpWarmupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< StorageCacheStartUpWarmupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheStartUpWarmupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< StorageCacheStartUpWarmupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheStartUpWarmupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* self;
															#line 2140 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Transaction tr;
															#line 2141 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Value trueValue;
															#line 2142 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Value falseValue;
															#line 2143 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Standalone<MutationRef> privatized;
															#line 2145 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Version readVersion;
															#line 8037 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via storageCacheStartUpWarmup()
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class StorageCacheStartUpWarmupActor final : public Actor<Void>, public ActorCallback< StorageCacheStartUpWarmupActor, 0, RangeResult >, public ActorCallback< StorageCacheStartUpWarmupActor, 1, Void >, public FastAllocated<StorageCacheStartUpWarmupActor>, public StorageCacheStartUpWarmupActorState<StorageCacheStartUpWarmupActor> {
															#line 8042 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<StorageCacheStartUpWarmupActor>::operator new;
	using FastAllocated<StorageCacheStartUpWarmupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3367769231298781184UL, 328980788963017984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StorageCacheStartUpWarmupActor, 0, RangeResult >;
friend struct ActorCallback< StorageCacheStartUpWarmupActor, 1, Void >;
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheStartUpWarmupActor(StorageCacheData* const& self) 
															#line 8060 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   StorageCacheStartUpWarmupActorState<StorageCacheStartUpWarmupActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14642452718017494784UL, 16198399489316498432UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheStartUpWarmupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("storageCacheStartUpWarmup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("storageCacheStartUpWarmup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StorageCacheStartUpWarmupActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< StorageCacheStartUpWarmupActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> storageCacheStartUpWarmup( StorageCacheData* const& self ) {
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new StorageCacheStartUpWarmupActor(self));
															#line 8094 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 2190 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

															#line 8099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via watchInterface()
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class WatchInterfaceActor>
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class WatchInterfaceActorState {
															#line 8106 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	WatchInterfaceActorState(StorageCacheData* const& self,StorageServerInterface const& ssi) 
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : self(self),
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   ssi(ssi),
															#line 2192 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   tr(self->cx),
															#line 2193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   storageKey(storageCacheServerKey(ssi.id()))
															#line 8119 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("watchInterface", reinterpret_cast<unsigned long>(this));

	}
	~WatchInterfaceActorState() 
	{
		fdb_probe_actor_destroy("watchInterface", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2194 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			;
															#line 8134 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchInterfaceActorState();
		static_cast<WatchInterfaceActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2195 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 8164 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 2214 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(5.0);
															#line 2214 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8175 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 4;
															#line 2214 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WatchInterfaceActor, 3, Void >*>(static_cast<WatchInterfaceActor*>(this)));
															#line 8180 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 2196 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2197 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 8198 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		try {
															#line 2199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr.get(storageKey);
															#line 2199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8204 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 1;
															#line 2199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchInterfaceActor, 0, Optional<Value> >*>(static_cast<WatchInterfaceActor*>(this)));
															#line 8209 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2211 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2211 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8246 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 3;
															#line 2211 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchInterfaceActor, 2, Void >*>(static_cast<WatchInterfaceActor*>(this)));
															#line 8251 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Optional<Value> const& val,int loopDepth) 
	{
															#line 2204 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!val.present())
															#line 8266 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 2205 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			tr.set(storageKey, storageCacheServerValue(ssi));
															#line 2206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8274 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 2;
															#line 2206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchInterfaceActor, 1, Void >*>(static_cast<WatchInterfaceActor*>(this)));
															#line 8279 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Optional<Value> && val,int loopDepth) 
	{
															#line 2204 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (!val.present())
															#line 8293 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		{
															#line 2205 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			tr.set(storageKey, storageCacheServerValue(ssi));
															#line 2206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.commit();
															#line 2206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8301 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 2;
															#line 2206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchInterfaceActor, 1, Void >*>(static_cast<WatchInterfaceActor*>(this)));
															#line 8306 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(val, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state > 0) static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 0;
		static_cast<WatchInterfaceActor*>(this)->ActorCallback< WatchInterfaceActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< WatchInterfaceActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchInterfaceActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchInterfaceActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
															#line 2208 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		tr.reset();
															#line 8395 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state > 0) static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 0;
		static_cast<WatchInterfaceActor*>(this)->ActorCallback< WatchInterfaceActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchInterfaceActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchInterfaceActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WatchInterfaceActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state > 0) static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 0;
		static_cast<WatchInterfaceActor*>(this)->ActorCallback< WatchInterfaceActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchInterfaceActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WatchInterfaceActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WatchInterfaceActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WatchInterfaceActor*>(this)->actor_wait_state > 0) static_cast<WatchInterfaceActor*>(this)->actor_wait_state = 0;
		static_cast<WatchInterfaceActor*>(this)->ActorCallback< WatchInterfaceActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchInterfaceActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WatchInterfaceActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WatchInterfaceActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData* self;
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageServerInterface ssi;
															#line 2192 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Transaction tr;
															#line 2193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Key storageKey;
															#line 8669 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via watchInterface()
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class WatchInterfaceActor final : public Actor<Void>, public ActorCallback< WatchInterfaceActor, 0, Optional<Value> >, public ActorCallback< WatchInterfaceActor, 1, Void >, public ActorCallback< WatchInterfaceActor, 2, Void >, public ActorCallback< WatchInterfaceActor, 3, Void >, public FastAllocated<WatchInterfaceActor>, public WatchInterfaceActorState<WatchInterfaceActor> {
															#line 8674 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<WatchInterfaceActor>::operator new;
	using FastAllocated<WatchInterfaceActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3979602234916261120UL, 3667126638773906432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WatchInterfaceActor, 0, Optional<Value> >;
friend struct ActorCallback< WatchInterfaceActor, 1, Void >;
friend struct ActorCallback< WatchInterfaceActor, 2, Void >;
friend struct ActorCallback< WatchInterfaceActor, 3, Void >;
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	WatchInterfaceActor(StorageCacheData* const& self,StorageServerInterface const& ssi) 
															#line 8694 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   WatchInterfaceActorState<WatchInterfaceActor>(self, ssi),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("watchInterface", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(752533449356400640UL, 9629102451919627776UL);
		ActorExecutionContextHelper __helper(static_cast<WatchInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("watchInterface");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("watchInterface", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WatchInterfaceActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WatchInterfaceActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WatchInterfaceActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WatchInterfaceActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> watchInterface( StorageCacheData* const& self, StorageServerInterface const& ssi ) {
															#line 2191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new WatchInterfaceActor(self, ssi));
															#line 8730 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 2217 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"

															#line 8735 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
namespace {
// This generated class is to be used only via storageCacheServer()
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
template <class StorageCacheServerActor>
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class StorageCacheServerActorState {
															#line 8742 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheServerActorState(StorageServerInterface const& ssi,uint16_t const& id,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		 : ssi(ssi),
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   id(id),
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   db(db),
															#line 2221 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   self(ssi.id(), id, db),
															#line 2222 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   actors(false),
															#line 2223 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		   updater(self.lastVersionWithData)
															#line 8759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
	{
		fdb_probe_actor_create("storageCacheServer", reinterpret_cast<unsigned long>(this));

	}
	~StorageCacheServerActorState() 
	{
		fdb_probe_actor_destroy("storageCacheServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2224 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			self.updater = &updater;
															#line 2229 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			self.ck = cacheKeysPrefixFor(id).withPrefix(systemKeys.begin);
															#line 2231 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			actors.add(waitFailureServer(ssi.waitFailure.getFuture()));
															#line 2232 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			actors.add(self.counters.cc.traceCounters("CacheMetrics", self.thisServerID, SERVER_KNOBS->STORAGE_LOGGING_DELAY, self.thisServerID.toString() + "/CacheMetrics"));
															#line 2238 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			StrictFuture<Void> __when_expr_0 = storageCacheStartUpWarmup(&self);
															#line 2238 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			if (static_cast<StorageCacheServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8784 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<StorageCacheServerActor*>(this)->actor_wait_state = 1;
															#line 2238 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StorageCacheServerActor, 0, Void >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 8789 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StorageCacheServerActorState();
		static_cast<StorageCacheServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2241 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(compactCache(&self));
															#line 2244 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(pullAsyncData(&self));
															#line 2245 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(watchInterface(&self, ssi));
															#line 2247 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(traceRole(Role::STORAGE_CACHE, ssi.id()));
															#line 2248 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		self.coreStarted.send(Void());
															#line 2250 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 8822 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2241 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(compactCache(&self));
															#line 2244 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(pullAsyncData(&self));
															#line 2245 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(watchInterface(&self, ssi));
															#line 2247 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(traceRole(Role::STORAGE_CACHE, ssi.id()));
															#line 2248 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		self.coreStarted.send(Void());
															#line 2250 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		;
															#line 8841 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StorageCacheServerActor*>(this)->actor_wait_state > 0) static_cast<StorageCacheServerActor*>(this)->actor_wait_state = 0;
		static_cast<StorageCacheServerActor*>(this)->ActorCallback< StorageCacheServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StorageCacheServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StorageCacheServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StorageCacheServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 2251 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		++self.counters.loops;
															#line 2253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_1 = ssi.getValue.getFuture();
															#line 8934 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<GetValueRequest>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<GetValueRequest>>::value, "invalid type");
															#line 2252 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		if (static_cast<StorageCacheServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8938 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.pop(), loopDepth); };
															#line 2258 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_2 = ssi.watchValue.getFuture();
															#line 8942 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_2), FutureStream<WatchValueRequest>>::value || std::is_same<decltype(__when_expr_2), ThreadFutureStream<WatchValueRequest>>::value, "invalid type");
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when2(__when_expr_2.pop(), loopDepth); };
															#line 2261 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_3 = ssi.getKey.getFuture();
															#line 8947 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_3), FutureStream<GetKeyRequest>>::value || std::is_same<decltype(__when_expr_3), ThreadFutureStream<GetKeyRequest>>::value, "invalid type");
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when3(__when_expr_3.pop(), loopDepth); };
															#line 2264 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_4 = ssi.getKeyValues.getFuture();
															#line 8952 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_4), FutureStream<GetKeyValuesRequest>>::value || std::is_same<decltype(__when_expr_4), ThreadFutureStream<GetKeyValuesRequest>>::value, "invalid type");
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when4(__when_expr_4.pop(), loopDepth); };
															#line 2267 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_5 = ssi.getShardState.getFuture();
															#line 8957 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_5), FutureStream<GetShardStateRequest>>::value || std::is_same<decltype(__when_expr_5), ThreadFutureStream<GetShardStateRequest>>::value, "invalid type");
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when5(__when_expr_5.pop(), loopDepth); };
															#line 2270 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_6 = ssi.getQueuingMetrics.getFuture();
															#line 8962 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_6), FutureStream<StorageQueuingMetricsRequest>>::value || std::is_same<decltype(__when_expr_6), ThreadFutureStream<StorageQueuingMetricsRequest>>::value, "invalid type");
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when6(__when_expr_6.pop(), loopDepth); };
															#line 2276 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_7 = ssi.getKeyValueStoreType.getFuture();
															#line 8967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_7), FutureStream<ReplyPromise<KeyValueStoreType>>>::value || std::is_same<decltype(__when_expr_7), ThreadFutureStream<ReplyPromise<KeyValueStoreType>>>::value, "invalid type");
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when7(__when_expr_7.pop(), loopDepth); };
															#line 2280 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_8 = ssi.getMappedKeyValues.getFuture();
															#line 8972 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_8), FutureStream<GetMappedKeyValuesRequest>>::value || std::is_same<decltype(__when_expr_8), ThreadFutureStream<GetMappedKeyValuesRequest>>::value, "invalid type");
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when8(__when_expr_8.pop(), loopDepth); };
															#line 2283 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_9 = ssi.waitMetrics.getFuture();
															#line 8977 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_9), FutureStream<WaitMetricsRequest>>::value || std::is_same<decltype(__when_expr_9), ThreadFutureStream<WaitMetricsRequest>>::value, "invalid type");
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when9(__when_expr_9.pop(), loopDepth); };
															#line 2286 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_10 = ssi.splitMetrics.getFuture();
															#line 8982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_10), FutureStream<SplitMetricsRequest>>::value || std::is_same<decltype(__when_expr_10), ThreadFutureStream<SplitMetricsRequest>>::value, "invalid type");
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when10(__when_expr_10.pop(), loopDepth); };
															#line 2289 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_11 = ssi.getStorageMetrics.getFuture();
															#line 8987 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_11), FutureStream<GetStorageMetricsRequest>>::value || std::is_same<decltype(__when_expr_11), ThreadFutureStream<GetStorageMetricsRequest>>::value, "invalid type");
		if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch1(__when_expr_11.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when11(__when_expr_11.pop(), loopDepth); };
															#line 2292 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_12 = ssi.getReadHotRanges.getFuture();
															#line 8992 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_12), FutureStream<ReadHotSubRangeRequest>>::value || std::is_same<decltype(__when_expr_12), ThreadFutureStream<ReadHotSubRangeRequest>>::value, "invalid type");
		if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1Catch1(__when_expr_12.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when12(__when_expr_12.pop(), loopDepth); };
															#line 2295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_13 = ssi.getRangeSplitPoints.getFuture();
															#line 8997 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_13), FutureStream<SplitRangeRequest>>::value || std::is_same<decltype(__when_expr_13), ThreadFutureStream<SplitRangeRequest>>::value, "invalid type");
		if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1Catch1(__when_expr_13.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when13(__when_expr_13.pop(), loopDepth); };
															#line 2298 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		auto __when_expr_14 = ssi.getKeyValuesStream.getFuture();
															#line 9002 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_14), FutureStream<GetKeyValuesStreamRequest>>::value || std::is_same<decltype(__when_expr_14), ThreadFutureStream<GetKeyValuesStreamRequest>>::value, "invalid type");
		if (__when_expr_14.isReady()) { if (__when_expr_14.isError()) return a_body1Catch1(__when_expr_14.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when14(__when_expr_14.pop(), loopDepth); };
															#line 2301 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		StrictFuture<Void> __when_expr_15 = actors.getResult();
															#line 9007 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		if (__when_expr_15.isReady()) { if (__when_expr_15.isError()) return a_body1Catch1(__when_expr_15.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when15(__when_expr_15.get(), loopDepth); };
		static_cast<StorageCacheServerActor*>(this)->actor_wait_state = 2;
															#line 2253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 1, GetValueRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2258 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 2, WatchValueRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2261 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 3, GetKeyRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2264 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 4, GetKeyValuesRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2267 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 5, GetShardStateRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2270 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 6, StorageQueuingMetricsRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2276 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 7, ReplyPromise<KeyValueStoreType> >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2280 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 8, GetMappedKeyValuesRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2283 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 9, WaitMetricsRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2286 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 10, SplitMetricsRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2289 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_11.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 11, GetStorageMetricsRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2292 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_12.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 12, ReadHotSubRangeRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_13.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 13, SplitRangeRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2298 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_14.addCallbackAndClear(static_cast<ActorSingleCallback< StorageCacheServerActor, 14, GetKeyValuesStreamRequest >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 2301 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		__when_expr_15.addCallbackAndClear(static_cast<ActorCallback< StorageCacheServerActor, 15, Void >*>(static_cast<StorageCacheServerActor*>(this)));
															#line 9040 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GetValueRequest const& req,int loopDepth) 
	{
															#line 2256 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(getValueQ(&self, req));
															#line 9055 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GetValueRequest && req,int loopDepth) 
	{
															#line 2256 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(getValueQ(&self, req));
															#line 9064 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when2(WatchValueRequest const& req,int loopDepth) 
	{
															#line 2259 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9073 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when2(WatchValueRequest && req,int loopDepth) 
	{
															#line 2259 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9082 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when3(GetKeyRequest const& req,int loopDepth) 
	{
															#line 2262 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(getKey(&self, req));
															#line 9091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when3(GetKeyRequest && req,int loopDepth) 
	{
															#line 2262 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(getKey(&self, req));
															#line 9100 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when4(GetKeyValuesRequest const& req,int loopDepth) 
	{
															#line 2265 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(getKeyValues(&self, req));
															#line 9109 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when4(GetKeyValuesRequest && req,int loopDepth) 
	{
															#line 2265 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		actors.add(getKeyValues(&self, req));
															#line 9118 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when5(GetShardStateRequest const& req,int loopDepth) 
	{
															#line 2268 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9127 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when5(GetShardStateRequest && req,int loopDepth) 
	{
															#line 2268 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9136 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when6(StorageQueuingMetricsRequest const& req,int loopDepth) 
	{
															#line 2271 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9145 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when6(StorageQueuingMetricsRequest && req,int loopDepth) 
	{
															#line 2271 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9154 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when7(ReplyPromise<KeyValueStoreType> const& reply,int loopDepth) 
	{
															#line 2277 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9163 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when7(ReplyPromise<KeyValueStoreType> && reply,int loopDepth) 
	{
															#line 2277 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9172 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when8(GetMappedKeyValuesRequest const& req,int loopDepth) 
	{
															#line 2281 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9181 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when8(GetMappedKeyValuesRequest && req,int loopDepth) 
	{
															#line 2281 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9190 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when9(WaitMetricsRequest const& req,int loopDepth) 
	{
															#line 2284 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when9(WaitMetricsRequest && req,int loopDepth) 
	{
															#line 2284 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9208 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when10(SplitMetricsRequest const& req,int loopDepth) 
	{
															#line 2287 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9217 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when10(SplitMetricsRequest && req,int loopDepth) 
	{
															#line 2287 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9226 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when11(GetStorageMetricsRequest const& req,int loopDepth) 
	{
															#line 2290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9235 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when11(GetStorageMetricsRequest && req,int loopDepth) 
	{
															#line 2290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9244 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when12(ReadHotSubRangeRequest const& req,int loopDepth) 
	{
															#line 2293 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when12(ReadHotSubRangeRequest && req,int loopDepth) 
	{
															#line 2293 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9262 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when13(SplitRangeRequest const& req,int loopDepth) 
	{
															#line 2296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9271 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when13(SplitRangeRequest && req,int loopDepth) 
	{
															#line 2296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9280 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when14(GetKeyValuesStreamRequest const& req,int loopDepth) 
	{
															#line 2299 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9289 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when14(GetKeyValuesStreamRequest && req,int loopDepth) 
	{
															#line 2299 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
		ASSERT(false);
															#line 9298 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when15(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when15(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<StorageCacheServerActor*>(this)->actor_wait_state > 0) static_cast<StorageCacheServerActor*>(this)->actor_wait_state = 0;
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 1, GetValueRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 2, WatchValueRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 3, GetKeyRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 4, GetKeyValuesRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 5, GetShardStateRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 6, StorageQueuingMetricsRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 7, ReplyPromise<KeyValueStoreType> >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 8, GetMappedKeyValuesRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 9, WaitMetricsRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 10, SplitMetricsRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 11, GetStorageMetricsRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 12, ReadHotSubRangeRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 13, SplitRangeRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorSingleCallback< StorageCacheServerActor, 14, GetKeyValuesStreamRequest >::remove();
		static_cast<StorageCacheServerActor*>(this)->ActorCallback< StorageCacheServerActor, 15, Void >::remove();

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 1, GetValueRequest >*,GetValueRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 1, GetValueRequest >*,GetValueRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 1, GetValueRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 2, WatchValueRequest >*,WatchValueRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 2, WatchValueRequest >*,WatchValueRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 2, WatchValueRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 3, GetKeyRequest >*,GetKeyRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 3, GetKeyRequest >*,GetKeyRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 3, GetKeyRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 4, GetKeyValuesRequest >*,GetKeyValuesRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 4, GetKeyValuesRequest >*,GetKeyValuesRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 4, GetKeyValuesRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 5, GetShardStateRequest >*,GetShardStateRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 5, GetShardStateRequest >*,GetShardStateRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 5, GetShardStateRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 6, StorageQueuingMetricsRequest >*,StorageQueuingMetricsRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 6, StorageQueuingMetricsRequest >*,StorageQueuingMetricsRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 6, StorageQueuingMetricsRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 7, ReplyPromise<KeyValueStoreType> >*,ReplyPromise<KeyValueStoreType> const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 7, ReplyPromise<KeyValueStoreType> >*,ReplyPromise<KeyValueStoreType> && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 7, ReplyPromise<KeyValueStoreType> >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 8, GetMappedKeyValuesRequest >*,GetMappedKeyValuesRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when8(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 8, GetMappedKeyValuesRequest >*,GetMappedKeyValuesRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 8, GetMappedKeyValuesRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 9, WaitMetricsRequest >*,WaitMetricsRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when9(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 9, WaitMetricsRequest >*,WaitMetricsRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when9(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 9, WaitMetricsRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 10, SplitMetricsRequest >*,SplitMetricsRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when10(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 10, SplitMetricsRequest >*,SplitMetricsRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when10(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 10, SplitMetricsRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 11, GetStorageMetricsRequest >*,GetStorageMetricsRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when11(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 11, GetStorageMetricsRequest >*,GetStorageMetricsRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when11(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 11, GetStorageMetricsRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 12, ReadHotSubRangeRequest >*,ReadHotSubRangeRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when12(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 12, ReadHotSubRangeRequest >*,ReadHotSubRangeRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when12(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 12, ReadHotSubRangeRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 13, SplitRangeRequest >*,SplitRangeRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when13(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 13, SplitRangeRequest >*,SplitRangeRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when13(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 13, SplitRangeRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 14, GetKeyValuesStreamRequest >*,GetKeyValuesStreamRequest const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when14(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_fire(ActorSingleCallback< StorageCacheServerActor, 14, GetKeyValuesStreamRequest >*,GetKeyValuesStreamRequest && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when14(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_error(ActorSingleCallback< StorageCacheServerActor, 14, GetKeyValuesStreamRequest >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_fire(ActorCallback< StorageCacheServerActor, 15, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when15(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 15);

	}
	void a_callback_fire(ActorCallback< StorageCacheServerActor, 15, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when15(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 15);

	}
	void a_callback_error(ActorCallback< StorageCacheServerActor, 15, Void >*,Error err) 
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), 15);

	}
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageServerInterface ssi;
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	uint16_t id;
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 2221 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheData self;
															#line 2222 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	ActorCollection actors;
															#line 2223 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheUpdater updater;
															#line 10202 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
};
// This generated class is to be used only via storageCacheServer()
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
class StorageCacheServerActor final : public Actor<Void>, public ActorCallback< StorageCacheServerActor, 0, Void >, public ActorSingleCallback< StorageCacheServerActor, 1, GetValueRequest >, public ActorSingleCallback< StorageCacheServerActor, 2, WatchValueRequest >, public ActorSingleCallback< StorageCacheServerActor, 3, GetKeyRequest >, public ActorSingleCallback< StorageCacheServerActor, 4, GetKeyValuesRequest >, public ActorSingleCallback< StorageCacheServerActor, 5, GetShardStateRequest >, public ActorSingleCallback< StorageCacheServerActor, 6, StorageQueuingMetricsRequest >, public ActorSingleCallback< StorageCacheServerActor, 7, ReplyPromise<KeyValueStoreType> >, public ActorSingleCallback< StorageCacheServerActor, 8, GetMappedKeyValuesRequest >, public ActorSingleCallback< StorageCacheServerActor, 9, WaitMetricsRequest >, public ActorSingleCallback< StorageCacheServerActor, 10, SplitMetricsRequest >, public ActorSingleCallback< StorageCacheServerActor, 11, GetStorageMetricsRequest >, public ActorSingleCallback< StorageCacheServerActor, 12, ReadHotSubRangeRequest >, public ActorSingleCallback< StorageCacheServerActor, 13, SplitRangeRequest >, public ActorSingleCallback< StorageCacheServerActor, 14, GetKeyValuesStreamRequest >, public ActorCallback< StorageCacheServerActor, 15, Void >, public FastAllocated<StorageCacheServerActor>, public StorageCacheServerActorState<StorageCacheServerActor> {
															#line 10207 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
public:
	using FastAllocated<StorageCacheServerActor>::operator new;
	using FastAllocated<StorageCacheServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1355939343985200896UL, 10023636653269466112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StorageCacheServerActor, 0, Void >;
friend struct ActorSingleCallback< StorageCacheServerActor, 1, GetValueRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 2, WatchValueRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 3, GetKeyRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 4, GetKeyValuesRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 5, GetShardStateRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 6, StorageQueuingMetricsRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 7, ReplyPromise<KeyValueStoreType> >;
friend struct ActorSingleCallback< StorageCacheServerActor, 8, GetMappedKeyValuesRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 9, WaitMetricsRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 10, SplitMetricsRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 11, GetStorageMetricsRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 12, ReadHotSubRangeRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 13, SplitRangeRequest >;
friend struct ActorSingleCallback< StorageCacheServerActor, 14, GetKeyValuesStreamRequest >;
friend struct ActorCallback< StorageCacheServerActor, 15, Void >;
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	StorageCacheServerActor(StorageServerInterface const& ssi,uint16_t const& id,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 10239 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
		 : Actor<Void>(),
		   StorageCacheServerActorState<StorageCacheServerActor>(ssi, id, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("storageCacheServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7742917301014674176UL, 1973907054760801280UL);
		ActorExecutionContextHelper __helper(static_cast<StorageCacheServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("storageCacheServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("storageCacheServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StorageCacheServerActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< StorageCacheServerActor, 1, GetValueRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
[[nodiscard]] Future<Void> storageCacheServer( StorageServerInterface const& ssi, uint16_t const& id, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 2218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
	return Future<Void>(new StorageCacheServerActor(ssi, id, db));
															#line 10273 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/StorageCache.actor.g.cpp"
}

#line 2305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/StorageCache.actor.cpp"
