#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
/*
 * DataDistribution.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/Audit.h"
#include "fdbclient/AuditUtils.actor.h"
#include "fdbclient/BulkDumping.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/RunRYWTransaction.actor.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbserver/BulkDumpUtil.actor.h"
#include "fdbserver/BulkLoadUtil.actor.h"
#include "fdbserver/DDSharedContext.h"
#include "fdbserver/DDTeamCollection.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/FDBExecHelper.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/QuietDatabase.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/TenantCache.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbserver/MockDataDistributor.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Buggify.h"
#include "flow/Error.h"
#include "flow/Platform.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/serialize.h"
#include "flow/actorcompiler.h" // This must be the last #include.

static const std::string ddServerBulkDumpFolder = "ddBulkDumpFiles";
static const std::string ddServerBulkLoadFolder = "ddBulkLoadFiles";

DataMoveType getDataMoveTypeFromDataMoveId(const UID& dataMoveId) {
	bool assigned, emptyRange;
	DataMoveType dataMoveType;
	DataMovementReason dataMoveReason;
	decodeDataMoveId(dataMoveId, assigned, emptyRange, dataMoveType, dataMoveReason);
	return dataMoveType;
}

void RelocateShard::setParentRange(KeyRange const& parent) {
	ASSERT(reason == RelocateReason::WRITE_SPLIT || reason == RelocateReason::SIZE_SPLIT);
	parent_range = parent;
}

Optional<KeyRange> RelocateShard::getParentRange() const {
	return parent_range;
}

ShardSizeBounds ShardSizeBounds::shardSizeBoundsBeforeTrack() {
	return ShardSizeBounds{ .max = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                           .iosPerKSecond = StorageMetrics::infinity,
		                                           .bytesReadPerKSecond = StorageMetrics::infinity,
		                                           .opsReadPerKSecond = StorageMetrics::infinity },
		                    .min = StorageMetrics{ .bytes = -1,
		                                           .bytesWrittenPerKSecond = 0,
		                                           .iosPerKSecond = 0,
		                                           .bytesReadPerKSecond = 0,
		                                           .opsReadPerKSecond = 0 },
		                    .permittedError = StorageMetrics{ .bytes = -1,
		                                                      .bytesWrittenPerKSecond = StorageMetrics::infinity,
		                                                      .iosPerKSecond = StorageMetrics::infinity,
		                                                      .bytesReadPerKSecond = StorageMetrics::infinity,
		                                                      .opsReadPerKSecond = StorageMetrics::infinity } };
}

namespace {

std::set<int> const& normalDDQueueErrors() {
	static std::set<int> s{ error_code_movekeys_conflict,
		                    error_code_broken_promise,
		                    error_code_data_move_cancelled,
		                    error_code_data_move_dest_team_not_found };
	return s;
}

} // anonymous namespace

enum class DDAuditContext : uint8_t {
	INVALID = 0,
	RESUME = 1,
	LAUNCH = 2,
	RETRY = 3,
};

struct DDAudit {
	DDAudit(AuditStorageState coreState)
	  : coreState(coreState), actors(true), foundError(false), auditStorageAnyChildFailed(false), retryCount(0),
	    cancelled(false), overallCompleteDoAuditCount(0), overallIssuedDoAuditCount(0), overallSkippedDoAuditCount(0),
	    remainingBudgetForAuditTasks(SERVER_KNOBS->CONCURRENT_AUDIT_TASK_COUNT_MAX), context(DDAuditContext::INVALID) {}

	AuditStorageState coreState;
	ActorCollection actors;
	Future<Void> auditActor;
	bool foundError;
	int retryCount;
	bool auditStorageAnyChildFailed;
	bool cancelled; // use to cancel any actor beyond auditActor
	int64_t overallIssuedDoAuditCount;
	int64_t overallCompleteDoAuditCount;
	int64_t overallSkippedDoAuditCount;
	AsyncVar<int> remainingBudgetForAuditTasks;
	DDAuditContext context;
	std::unordered_set<UID> serversFinishedSSShardAudit; // dedicated to ssshard

	inline void setAuditRunActor(Future<Void> actor) { auditActor = actor; }
	inline Future<Void> getAuditRunActor() { return auditActor; }

	inline void setDDAuditContext(DDAuditContext context_) { this->context = context_; }
	inline DDAuditContext getDDAuditContext() const { return context; }

	// auditActor and actors are guaranteed to deliver a cancel signal
	void cancel() {
		auditActor.cancel();
		actors.clear(true);
		cancelled = true;
	}

	bool isCancelled() const { return cancelled; }
};

void DataMove::validateShard(const DDShardInfo& shard, KeyRangeRef range, int priority) {
	if (!valid) {
		if (shard.hasDest && shard.destId != anonymousShardId) {
			TraceEvent(SevError, "DataMoveValidationError")
			    .detail("Range", range)
			    .detail("Reason", "DataMoveMissing")
			    .detail("DestID", shard.destId)
			    .detail("ShardPrimaryDest", describe(shard.primaryDest))
			    .detail("ShardRemoteDest", describe(shard.remoteDest));
		}
		return;
	}

	if (this->meta.ranges.empty()) {
		TraceEvent(SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveMetatdataRangeEmpty")
		    .detail("DestID", shard.destId)
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		ASSERT(false);
	}

	if (!this->meta.ranges.front().contains(range)) {
		TraceEvent(SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveMetatdataRangeMismatch")
		    .detail("DestID", shard.destId)
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		ASSERT(false);
	}

	if (!shard.hasDest) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "ShardMissingDest")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest));
		cancelled = true;
		return;
	}

	if (shard.destId != this->meta.id) {
		TraceEvent(SevWarnAlways, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveIDMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("ShardMoveID", shard.destId);
		cancelled = true;
		return;
	}

	if (!std::equal(
	        this->primaryDest.begin(), this->primaryDest.end(), shard.primaryDest.begin(), shard.primaryDest.end()) ||
	    !std::equal(
	        this->remoteDest.begin(), this->remoteDest.end(), shard.remoteDest.begin(), shard.remoteDest.end())) {
		TraceEvent(g_network->isSimulated() ? SevWarn : SevError, "DataMoveValidationError")
		    .detail("Range", range)
		    .detail("Reason", "DataMoveDestMissMatch")
		    .detail("DataMoveMetaData", this->meta.toString())
		    .detail("DataMovePrimaryDest", describe(this->primaryDest))
		    .detail("DataMoveRemoteDest", describe(this->remoteDest))
		    .detail("ShardPrimaryDest", describe(shard.primaryDest))
		    .detail("ShardRemoteDest", describe(shard.remoteDest));
		cancelled = true;
	}
}

Future<Void> StorageWiggler::onCheck() const {
	return delay(MIN_ON_CHECK_DELAY_SEC);
}

// add server to wiggling queue
void StorageWiggler::addServer(const UID& serverId, const StorageMetadataType& metadata) {
	// std::cout << "size: " << pq_handles.size() << " add " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	ASSERT(!pq_handles.contains(serverId));
	pq_handles[serverId] = wiggle_pq.emplace(metadata, serverId);
}

void StorageWiggler::removeServer(const UID& serverId) {
	// std::cout << "size: " << pq_handles.size() << " remove " << serverId.toString() << " DC: "
	//           << teamCollection->isPrimary() << std::endl;
	if (contains(serverId)) { // server haven't been popped
		auto handle = pq_handles.at(serverId);
		pq_handles.erase(serverId);
		wiggle_pq.erase(handle);
	}
}

void StorageWiggler::updateMetadata(const UID& serverId, const StorageMetadataType& metadata) {
	//	std::cout << "size: " << pq_handles.size() << " update " << serverId.toString()
	//	          << " DC: " << teamCollection->isPrimary() << std::endl;
	auto handle = pq_handles.at(serverId);
	if ((*handle).first == metadata) {
		return;
	}
	wiggle_pq.update(handle, std::make_pair(metadata, serverId));
}

bool StorageWiggler::necessary(const UID& serverId, const StorageMetadataType& metadata) const {
	return metadata.wrongConfiguredForWiggle ||
	       (now() - metadata.createdTime > SERVER_KNOBS->DD_STORAGE_WIGGLE_MIN_SS_AGE_SEC);
}

Optional<UID> StorageWiggler::getNextServerId(bool necessaryOnly) {
	if (!wiggle_pq.empty()) {
		auto [metadata, id] = wiggle_pq.top();
		if (necessaryOnly && !necessary(id, metadata)) {
			return {};
		}
		wiggle_pq.pop();
		pq_handles.erase(id);
		return Optional<UID>(id);
	}
	return Optional<UID>();
}

Future<Void> StorageWiggler::resetStats() {
	metrics.reset();
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.resetStorageWiggleMetrics(tr, PrimaryRegion(teamCollection->isPrimary()), metrics);
	    });
}

Future<Void> StorageWiggler::restoreStats() {
	auto readFuture = wiggleData.storageWiggleMetrics(PrimaryRegion(teamCollection->isPrimary()))
	                      .getD(teamCollection->dbContext().getReference(), Snapshot::False, metrics);
	return store(metrics, readFuture);
}

Future<Void> StorageWiggler::startWiggle() {
	metrics.last_wiggle_start = StorageMetadataType::currentTime();
	if (shouldStartNewRound()) {
		metrics.last_round_start = metrics.last_wiggle_start;
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

Future<Void> StorageWiggler::finishWiggle() {
	metrics.last_wiggle_finish = StorageMetadataType::currentTime();
	metrics.finished_wiggle += 1;
	auto duration = metrics.last_wiggle_finish - metrics.last_wiggle_start;
	metrics.smoothed_wiggle_duration.setTotal((double)duration);

	if (shouldFinishRound()) {
		metrics.last_round_finish = metrics.last_wiggle_finish;
		metrics.finished_round += 1;
		duration = metrics.last_round_finish - metrics.last_round_start;
		metrics.smoothed_round_duration.setTotal((double)duration);
	}
	return runRYWTransaction(
	    teamCollection->dbContext(), [this](Reference<ReadYourWritesTransaction> tr) -> Future<Void> {
		    return wiggleData.updateStorageWiggleMetrics(tr, metrics, PrimaryRegion(teamCollection->isPrimary()));
	    });
}

															#line 319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via remoteRecovered()
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoteRecoveredActor>
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActorState {
															#line 326 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActorState(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : db(db)
															#line 333 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	~RemoteRecoveredActorState() 
	{
		fdb_probe_actor_destroy("remoteRecovered", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 318 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDTrackerStarting").log();
															#line 319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 350 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 323 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoteRecoveredActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoteRecoveredActorState(); static_cast<RemoteRecoveredActor*>(this)->destroy(); return 0; }
															#line 373 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoteRecoveredActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoteRecoveredActorState();
		static_cast<RemoteRecoveredActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(db->get().recoveryState < RecoveryState::ALL_LOGS_RECRUITED))
															#line 392 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 320 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDTrackerStarting").detail("RecoveryState", (int)db->get().recoveryState);
															#line 321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 402 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 1;
															#line 321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoteRecoveredActor, 0, Void >*>(static_cast<RemoteRecoveredActor*>(this)));
															#line 407 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoteRecoveredActor*>(this)->actor_wait_state > 0) static_cast<RemoteRecoveredActor*>(this)->actor_wait_state = 0;
		static_cast<RemoteRecoveredActor*>(this)->ActorCallback< RemoteRecoveredActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoteRecoveredActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoteRecoveredActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 514 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via remoteRecovered()
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoteRecoveredActor final : public Actor<Void>, public ActorCallback< RemoteRecoveredActor, 0, Void >, public FastAllocated<RemoteRecoveredActor>, public RemoteRecoveredActorState<RemoteRecoveredActor> {
															#line 519 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoteRecoveredActor>::operator new;
	using FastAllocated<RemoteRecoveredActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5231580983973467392UL, 15648275617585340160UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoteRecoveredActor, 0, Void >;
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoteRecoveredActor(Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 536 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoteRecoveredActorState<RemoteRecoveredActor>(db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17462346244829468672UL, 10124612030083111680UL);
		ActorExecutionContextHelper __helper(static_cast<RemoteRecoveredActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("remoteRecovered");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("remoteRecovered", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoteRecoveredActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> remoteRecovered( Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoteRecoveredActor(db));
															#line 569 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 325 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Ensures that the serverKeys key space is properly coalesced
// This method is only used for testing and is not implemented in a manner that is safe for large databases
															#line 576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via debugCheckCoalescing()
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DebugCheckCoalescingActor>
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActorState {
															#line 583 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActorState(Database const& cx) 
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : cx(cx),
															#line 329 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 592 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	~DebugCheckCoalescingActorState() 
	{
		fdb_probe_actor_destroy("debugCheckCoalescing", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 607 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 332 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(serverListKeys, CLIENT_KNOBS->TOO_MANY);
															#line 332 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 640 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 1;
															#line 332 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 645 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 352 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 352 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 669 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 3;
															#line 352 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 2, Void >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 674 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 333 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!serverList.more && serverList.size() < CLIENT_KNOBS->TOO_MANY);
															#line 335 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = int();
															#line 336 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 693 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __serverList,int loopDepth) 
	{
															#line 332 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		serverList = __serverList;
															#line 702 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __serverList,int loopDepth) 
	{
		serverList = std::move(__serverList);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 349 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DoneCheckingCoalescing").log();
															#line 350 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DebugCheckCoalescingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DebugCheckCoalescingActorState(); static_cast<DebugCheckCoalescingActor*>(this)->destroy(); return 0; }
															#line 783 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DebugCheckCoalescingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DebugCheckCoalescingActorState();
		static_cast<DebugCheckCoalescingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 336 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < serverList.size()))
															#line 802 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 337 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		id = decodeServerListValue(serverList[i].value).id();
															#line 338 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, serverKeysPrefixFor(id), allKeys);
															#line 338 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 812 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 2;
															#line 338 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*>(static_cast<DebugCheckCoalescingActor*>(this)));
															#line 817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult const& ranges,int loopDepth) 
	{
															#line 339 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 341 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 342 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 843 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 343 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 847 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 336 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 852 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(RangeResult && ranges,int loopDepth) 
	{
															#line 339 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(ranges.end()[-1].key == allKeys.end);
															#line 341 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int j = 0;j < ranges.size() - 2;j++) {
															#line 342 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (ranges[j].value == ranges[j + 1].value)
															#line 865 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 343 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "UncoalescedValues", id) .detail("Key1", ranges[j].key) .detail("Key2", ranges[j + 1].key) .detail("Value", ranges[j].value);
															#line 869 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 336 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 874 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult const& ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(ranges, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(RangeResult && ranges,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(std::move(ranges), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state > 0) static_cast<DebugCheckCoalescingActor*>(this)->actor_wait_state = 0;
		static_cast<DebugCheckCoalescingActor*>(this)->ActorCallback< DebugCheckCoalescingActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DebugCheckCoalescingActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 329 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 332 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult serverList;
															#line 335 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 337 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID id;
															#line 1051 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via debugCheckCoalescing()
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DebugCheckCoalescingActor final : public Actor<Void>, public ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >, public ActorCallback< DebugCheckCoalescingActor, 2, Void >, public FastAllocated<DebugCheckCoalescingActor>, public DebugCheckCoalescingActorState<DebugCheckCoalescingActor> {
															#line 1056 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DebugCheckCoalescingActor>::operator new;
	using FastAllocated<DebugCheckCoalescingActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(670325166405734912UL, 3439559408532819456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >;
friend struct ActorCallback< DebugCheckCoalescingActor, 2, Void >;
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DebugCheckCoalescingActor(Database const& cx) 
															#line 1075 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DebugCheckCoalescingActorState<DebugCheckCoalescingActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13012747114799893760UL, 10690014687488268032UL);
		ActorExecutionContextHelper __helper(static_cast<DebugCheckCoalescingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("debugCheckCoalescing");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("debugCheckCoalescing", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DebugCheckCoalescingActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> debugCheckCoalescing( Database const& cx ) {
															#line 328 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DebugCheckCoalescingActor(cx));
															#line 1110 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 356 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DataDistributor;
void runAuditStorage(
    Reference<DataDistributor> self,
    AuditStorageState auditStates,
    int retryCount,
    DDAuditContext context,
    Optional<std::unordered_set<UID>> serversFinishedSSShardAudit = Optional<std::unordered_set<UID>>());
															#line 1122 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorageCore( Reference<DataDistributor> const& self, UID const& auditID, AuditType const& auditType, int const& currentRetryCount );

#line 368 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1126 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<UID> launchAudit( Reference<DataDistributor> const& self, KeyRange const& auditRange, AuditType const& auditType, KeyValueStoreType const& auditStorageEngineType );

#line 372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1130 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> auditStorage( Reference<DataDistributor> const& self, TriggerAuditRequest const& req );

#line 373 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
void loadAndDispatchAudit(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit);
															#line 1135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorageServerShard( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditStorageShardOnServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi );

#line 378 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1143 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditStorage( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit );

#line 379 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1147 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> dispatchAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 382 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1151 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditLocationMetadata( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& auditRange );

#line 385 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1155 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& range );

#line 388 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> doAuditOnStorageServer( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, StorageServerInterface const& ssi, AuditStorageRequest const& req );

#line 392 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1163 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> skipAuditOnRange( Reference<DataDistributor> const& self, std::shared_ptr<DDAudit> const& audit, KeyRange const& rangeToSkip );

#line 395 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

void runBulkLoadTaskAsync(Reference<DataDistributor> self, KeyRange range, UID taskId, bool restart);
															#line 1169 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self );

#line 398 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

struct DDBulkLoadJobManager {
	BulkLoadJobState jobState;
	BulkLoadTransportMethod jobTransportMethod;
	// manifestEntryMap is a map from the begin key of the manifest to the manifest entry.
	// The end key of the current manifest is the begin key of the next manifest.
	// When the task range is aligned with the manifest range, every key is the begin key of the corresponding manifest.
	// When the task range is not aligned with the manifest range, the first key is the task begin key which can be
	// larger than the corresponding manifest begin key.
	std::shared_ptr<BulkLoadManifestFileMap> manifestEntryMap;
	std::string manifestLocalTempFolder;
	bool allTaskSubmitted = false;

	DDBulkLoadJobManager() = default;
	DDBulkLoadJobManager(const BulkLoadJobState& jobState, const std::string& manifestLocalTempFolder)
	  : jobState(jobState), manifestLocalTempFolder(manifestLocalTempFolder), allTaskSubmitted(false) {
		manifestEntryMap = std::make_shared<BulkLoadManifestFileMap>();
	}

	bool isValid() const { return jobState.isValid(); }
};

struct DDBulkDumpJobManager {
	BulkDumpState jobState;
	std::map<Key, BulkLoadManifest> taskManifestMap;

	DDBulkDumpJobManager() = default;
	DDBulkDumpJobManager(const BulkDumpState& jobState) : jobState(jobState) {}

	bool isValid() const { return jobState.isValid(); }
};

struct DataDistributor : NonCopyable, ReferenceCounted<DataDistributor> {
public:
	Reference<AsyncVar<ServerDBInfo> const> dbInfo;
	Reference<DDSharedContext> context;
	UID ddId;
	PromiseStream<Future<Void>> addActor;

	// State initialized when bootstrap
	Reference<IDDTxnProcessor> txnProcessor;
	MoveKeysLock& lock; // reference to context->lock
	DatabaseConfiguration& configuration; // reference to context->configuration
	std::vector<Optional<Key>> primaryDcId;
	std::vector<Optional<Key>> remoteDcIds;
	Reference<InitialDataDistribution> initData;

	Reference<EventCacheHolder> initialDDEventHolder;
	Reference<EventCacheHolder> movingDataEventHolder;
	Reference<EventCacheHolder> totalDataInFlightEventHolder;
	Reference<EventCacheHolder> totalDataInFlightRemoteEventHolder;

	// Optional components that can be set after ::init(). They're optional when test, but required for DD being
	// fully-functional.
	DDTeamCollection* teamCollection;
	Reference<ShardsAffectedByTeamFailure> shardsAffectedByTeamFailure;
	// consumer is a yield stream from producer. The RelocateShard is pushed into relocationProducer and popped from
	// relocationConsumer (by DDQueue)
	PromiseStream<RelocateShard> relocationProducer, relocationConsumer;
	PromiseStream<BulkLoadShardRequest> triggerShardBulkLoading;
	Reference<PhysicalShardCollection> physicalShardCollection;
	Reference<BulkLoadTaskCollection> bulkLoadTaskCollection;

	Promise<Void> initialized;

	std::unordered_map<AuditType, std::unordered_map<UID, std::shared_ptr<DDAudit>>> audits;
	FlowLock auditStorageHaLaunchingLock;
	FlowLock auditStorageReplicaLaunchingLock;
	FlowLock auditStorageLocationMetadataLaunchingLock;
	FlowLock auditStorageSsShardLaunchingLock;
	Promise<Void> auditStorageInitialized;
	bool auditStorageInitStarted;

	Optional<Reference<TenantCache>> ddTenantCache;

	// monitor DD configuration change
	Promise<Version> configChangeWatching;
	Future<Void> onConfigChange;

	ActorCollection bulkLoadActors;
	bool bulkLoadEnabled = false;
	ParallelismLimitor bulkLoadParallelismLimitor;
	ParallelismLimitor bulkLoadEngineParallelismLimitor;
	std::string bulkLoadFolder;

	Optional<DDBulkLoadJobManager> bulkLoadJobManager;

	bool bulkDumpEnabled = false;
	ParallelismLimitor bulkDumpParallelismLimitor;
	std::string folder;
	std::string bulkDumpFolder;

	DDBulkDumpJobManager bulkDumpJobManager;

	DataDistributor(Reference<AsyncVar<ServerDBInfo> const> const& db,
	                UID id,
	                Reference<DDSharedContext> context,
	                std::string folder)
	  : dbInfo(db), context(context), ddId(id), txnProcessor(nullptr), lock(context->lock),
	    configuration(context->configuration), initialDDEventHolder(makeReference<EventCacheHolder>("InitialDD")),
	    movingDataEventHolder(makeReference<EventCacheHolder>("MovingData")),
	    totalDataInFlightEventHolder(makeReference<EventCacheHolder>("TotalDataInFlight")),
	    totalDataInFlightRemoteEventHolder(makeReference<EventCacheHolder>("TotalDataInFlightRemote")),
	    teamCollection(nullptr), bulkLoadTaskCollection(nullptr), auditStorageHaLaunchingLock(1),
	    auditStorageReplicaLaunchingLock(1), auditStorageLocationMetadataLaunchingLock(1),
	    auditStorageSsShardLaunchingLock(1), auditStorageInitStarted(false), bulkLoadActors(false),
	    bulkLoadEnabled(false), bulkLoadParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM),
	    bulkLoadEngineParallelismLimitor(SERVER_KNOBS->DD_BULKLOAD_PARALLELISM), bulkDumpEnabled(false),
	    bulkDumpParallelismLimitor(SERVER_KNOBS->DD_BULKDUMP_PARALLELISM), folder(folder) {
		if (!folder.empty()) {
			bulkDumpFolder = abspath(joinPath(folder, ddServerBulkDumpFolder));
			// TODO(BulkDump): clear this folder in the presence of crash
			bulkLoadFolder = abspath(joinPath(folder, ddServerBulkLoadFolder));
			// TODO(BulkLoad): clear this folder in the presence of crash
		}
	}

	// bootstrap steps

	Future<Void> takeMoveKeysLock() { return store(lock, txnProcessor->takeMoveKeysLock(ddId)); }

	Future<Void> loadDatabaseConfiguration() { return store(configuration, txnProcessor->getDatabaseConfiguration()); }

	Future<Void> updateReplicaKeys() {
		return txnProcessor->updateReplicaKeys(primaryDcId, remoteDcIds, configuration);
	}

	Future<Void> loadInitialDataDistribution() {
		return store(initData,
		             txnProcessor->getInitialDataDistribution(
		                 ddId,
		                 lock,
		                 configuration.usableRegions > 1 ? remoteDcIds : std::vector<Optional<Key>>(),
		                 context->ddEnabledState.get(),
		                 SkipDDModeCheck::False));
	}

	void initDcInfo() {
		primaryDcId.clear();
		remoteDcIds.clear();
		const std::vector<RegionInfo>& regions = configuration.regions;
		if (configuration.regions.size() > 0) {
			primaryDcId.push_back(regions[0].dcId);
		}
		if (configuration.regions.size() > 1) {
			remoteDcIds.push_back(regions[1].dcId);
		}
	}

	Future<Void> waitDataDistributorEnabled() const {
		return txnProcessor->waitForDataDistributionEnabled(context->ddEnabledState.get());
	}

	// Resume in-memory audit instances and issue background audit metadata cleanup
	void resumeAuditStorage(Reference<DataDistributor> self, std::vector<AuditStorageState> auditStates) {
		for (const auto& auditState : auditStates) {
			if (auditState.getPhase() != AuditPhase::Running) {
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "WrongAuditStateToResume")
				    .detail("AuditState", auditState.toString());
				return;
			}
			if (self->audits.contains(auditState.getType()) &&
			    self->audits[auditState.getType()].contains(auditState.id)) {
				// Ignore any RUNNING auditState with an alive audit
				// instance in DD audits map
				continue;
			}
			runAuditStorage(self, auditState, 0, DDAuditContext::RESUME);
			TraceEvent(SevInfo, "AuditStorageResumed", self->ddId)
			    .detail("AuditID", auditState.id)
			    .detail("AuditType", auditState.getType())
			    .detail("AuditState", auditState.toString());
		}
		return;
	}

																#line 1349 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via initAuditStorage()
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitAuditStorageActor>
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActorState {
															#line 1355 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActorState(Reference<DataDistributor> const& self) 
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 1362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	~InitAuditStorageActorState() 
	{
		fdb_probe_actor_destroy("initAuditStorage", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 575 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->auditStorageInitStarted = true;
															#line 576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			MoveKeyLockInfo lockInfo;
															#line 577 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.myOwner = self->lock.myOwner;
															#line 578 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevOwner = self->lock.prevOwner;
															#line 579 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lockInfo.prevWrite = self->lock.prevWrite;
															#line 580 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<std::vector<AuditStorageState>> __when_expr_0 = initAuditMetadata(self->txnProcessor->context(), lockInfo, self->context->isDDEnabled(), self->ddId, SERVER_KNOBS->PERSIST_FINISH_AUDIT_COUNT);
															#line 580 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1389 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 1;
															#line 580 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*>(static_cast<InitAuditStorageActor*>(this)));
															#line 1394 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
															#line 586 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 587 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 588 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1421 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
															#line 586 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->resumeAuditStorage(self, auditStatesToResume);
															#line 587 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->auditStorageInitialized.send(Void());
															#line 588 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitAuditStorageActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitAuditStorageActorState(); static_cast<InitAuditStorageActor*>(this)->destroy(); return 0; }
															#line 1437 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitAuditStorageActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitAuditStorageActorState();
		static_cast<InitAuditStorageActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> const& auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(auditStatesToResume, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<AuditStorageState> && auditStatesToResume,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(auditStatesToResume), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitAuditStorageActor*>(this)->actor_wait_state > 0) static_cast<InitAuditStorageActor*>(this)->actor_wait_state = 0;
		static_cast<InitAuditStorageActor*>(this)->ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >::remove();

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> const& value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,std::vector<AuditStorageState> && value) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*,Error err) 
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1522 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via initAuditStorage()
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitAuditStorageActor final : public Actor<Void>, public ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >, public FastAllocated<InitAuditStorageActor>, public InitAuditStorageActorState<InitAuditStorageActor> {
															#line 1527 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitAuditStorageActor>::operator new;
	using FastAllocated<InitAuditStorageActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6511514937350378496UL, 13499228371663661824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >;
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitAuditStorageActor(Reference<DataDistributor> const& self) 
															#line 1544 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitAuditStorageActorState<InitAuditStorageActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8855224104647952640UL, 5194562846350463232UL);
		ActorExecutionContextHelper __helper(static_cast<InitAuditStorageActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("initAuditStorage");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("initAuditStorage", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitAuditStorageActor, 0, std::vector<AuditStorageState> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> initAuditStorage( Reference<DataDistributor> const& self ) {
															#line 574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitAuditStorageActor(self));
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 590 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 1581 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class WaitUntilDataDistributorExitSecurityModeActor>
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActorState {
															#line 1587 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActorState(Reference<DataDistributor> const& self) 
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 592 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(self->txnProcessor->context())
															#line 1596 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	~WaitUntilDataDistributorExitSecurityModeActorState() 
	{
		fdb_probe_actor_destroy("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 593 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 1611 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitUntilDataDistributorExitSecurityModeActorState();
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 594 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->DD_ENABLED_CHECK_DELAY, TaskPriority::DataDistribution);
															#line 594 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1643 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 1;
															#line 594 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1648 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 595 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 596 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 597 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1661 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1672 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 595 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 596 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 597 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1691 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(dataDistributionModeKey);
															#line 599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1697 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 2;
															#line 599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1702 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 612 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 612 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1801 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 4;
															#line 612 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1806 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> const& mode,int loopDepth) 
	{
															#line 600 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1821 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 601 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1825 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 603 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 604 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 605 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 606 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 607 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1843 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 609 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 609 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1853 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 609 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> && mode,int loopDepth) 
	{
															#line 600 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!mode.present())
															#line 1867 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 601 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 603 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BinaryReader rd(mode.get(), Unversioned());
															#line 604 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		int ddMode = 1;
															#line 605 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		rd >> ddMode;
															#line 606 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ddMode != 2)
															#line 1885 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 607 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitUntilDataDistributorExitSecurityModeActorState(); static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->destroy(); return 0; }
															#line 1889 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WaitUntilDataDistributorExitSecurityModeActorState();
			static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 609 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = checkMoveKeysLockReadOnly(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 609 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1899 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 3;
															#line 609 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*>(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)));
															#line 1904 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> const& mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(mode, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Optional<Value> && mode,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(mode), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 610 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1988 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 610 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr.reset();
															#line 1997 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state > 0) static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->actor_wait_state = 0;
		static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 592 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 2181 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via waitUntilDataDistributorExitSecurityMode()
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class WaitUntilDataDistributorExitSecurityModeActor final : public Actor<Void>, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >, public ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >, public FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>, public WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor> {
															#line 2186 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator new;
	using FastAllocated<WaitUntilDataDistributorExitSecurityModeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4448034742154865664UL, 695348556396565504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >;
friend struct ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >;
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	WaitUntilDataDistributorExitSecurityModeActor(Reference<DataDistributor> const& self) 
															#line 2206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   WaitUntilDataDistributorExitSecurityModeActorState<WaitUntilDataDistributorExitSecurityModeActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4834519635010961920UL, 15563594834600371456UL);
		ActorExecutionContextHelper __helper(static_cast<WaitUntilDataDistributorExitSecurityModeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitUntilDataDistributorExitSecurityMode");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitUntilDataDistributorExitSecurityMode", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WaitUntilDataDistributorExitSecurityModeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> waitUntilDataDistributorExitSecurityMode( Reference<DataDistributor> const& self ) {
															#line 591 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new WaitUntilDataDistributorExitSecurityModeActor(self));
															#line 2241 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 616 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Initialize the required internal states of DataDistributor from system metadata. It's necessary before
	// DataDistributor start working. Doesn't include initialization of optional components, like TenantCache, DDQueue,
	// Tracker, TeamCollection. The components should call its own ::init methods.
																#line 2249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via init()
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class InitActor>
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActorState {
															#line 2255 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActorState(Reference<DataDistributor> const& self) 
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 2262 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("init", reinterpret_cast<unsigned long>(this));

	}
	~InitActorState() 
	{
		fdb_probe_actor_destroy("init", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 621 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 2277 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitActorState();
		static_cast<InitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 733 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<InitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitActorState(); static_cast<InitActor*>(this)->destroy(); return 0; }
															#line 2300 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<InitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitActorState();
		static_cast<InitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 622 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->waitDataDistributorEnabled();
															#line 622 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 1;
															#line 622 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitActor, 0, Void >*>(static_cast<InitActor*>(this)));
															#line 2326 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 623 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 625 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2354 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2359 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 623 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionEnabled").log();
															#line 625 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTakingMoveKeysLock", self->ddId).log();
															#line 626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->takeMoveKeysLock();
															#line 626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2374 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 2;
															#line 626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< InitActor, 1, Void >*>(static_cast<InitActor*>(this)));
															#line 2379 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 627 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 631 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2465 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 633 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2469 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 635 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 642 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2478 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 644 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 644 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2484 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 644 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2489 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 627 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitTookMoveKeysLock", self->ddId).log();
															#line 631 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->auditStorageInitStarted)
															#line 2505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 633 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!self->txnProcessor->isMocked())
															#line 2509 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 635 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->addActor.send(self->initAuditStorage(self));
															#line 2513 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 642 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 2518 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 644 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = waitUntilDataDistributorExitSecurityMode(self);
															#line 644 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2524 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<InitActor*>(this)->actor_wait_state = 3;
															#line 644 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< InitActor, 2, Void >*>(static_cast<InitActor*>(this)));
															#line 2529 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< InitActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< InitActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 652 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributorExitSecurityMode").log();
															#line 654 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->loadDatabaseConfiguration();
															#line 654 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2622 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 4;
															#line 654 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< InitActor, 3, Void >*>(static_cast<InitActor*>(this)));
															#line 2627 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< InitActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< InitActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 656 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 660 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2727 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 662 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 664 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2733 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2739 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2744 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->initDcInfo();
															#line 656 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitGotConfiguration", self->ddId) .setMaxFieldLength(-1) .detail("Conf", self->configuration.toString());
															#line 660 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.storageServerStoreType == KeyValueStoreType::SSD_SHARDED_ROCKSDB && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 2757 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 662 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "PhysicalShardNotEnabledForShardedRocks", self->ddId) .detail("EnableServerKnob", "SHARD_ENCODE_LOCATION_METADATA");
															#line 664 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(internal_error(), std::max(0, loopDepth - 1));
															#line 2763 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = self->updateReplicaKeys();
															#line 667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 5;
															#line 667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< InitActor, 4, Void >*>(static_cast<InitActor*>(this)));
															#line 2774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< InitActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< InitActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
															#line 668 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 670 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 670 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2862 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 670 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2867 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
															#line 668 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DDInitUpdatedReplicaKeys", self->ddId).log();
															#line 670 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = self->loadInitialDataDistribution();
															#line 670 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<InitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2880 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<InitActor*>(this)->actor_wait_state = 6;
															#line 670 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< InitActor, 5, Void >*>(static_cast<InitActor*>(this)));
															#line 2885 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< InitActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< InitActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 672 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 2969 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 673 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2973 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 680 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 2979 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 688 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 2983 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 690 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 2987 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 694 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 698 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 720 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 726 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 2998 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 672 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->shards.size() > 1)
															#line 3007 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 673 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", self->initData->shards.end()[-2].key) .detail("E", self->initData->shards.end()[-1].key) .detail("Src", describe(self->initData->shards.end()[-2].primarySrc)) .detail("Dest", describe(self->initData->shards.end()[-2].primaryDest)) .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 3011 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 680 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DDInitGotInitialDD", self->ddId) .detail("B", "") .detail("E", "") .detail("Src", "[no items]") .detail("Dest", "[no items]") .trackLatest(self->initialDDEventHolder->trackingKey);
															#line 3017 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 688 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->initData->mode == 1 && self->context->isDDEnabled())
															#line 3021 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 690 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent("DataDistributionInitComplete", self->ddId).log();
															#line 3025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 694 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("DataDistributionDisabled", self->ddId) .detail("Mode", self->initData->mode) .detail("Enabled", self->context->isDDEnabled());
															#line 698 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("MovingData", self->ddId) .detail("InFlight", 0) .detail("InQueue", 0) .detail("AverageShardSize", -1) .detail("UnhealthyRelocations", 0) .detail("HighestPriority", 0) .detail("BytesWritten", 0) .detail("BytesWrittenAverageRate", 0) .detail("PriorityRecoverMove", 0) .detail("PriorityRebalanceUnderutilizedTeam", 0) .detail("PriorityRebalannceOverutilizedTeam", 0) .detail("PriorityTeamHealthy", 0) .detail("PriorityTeamContainsUndesiredServer", 0) .detail("PriorityTeamRedundant", 0) .detail("PriorityMergeShard", 0) .detail("PriorityTeamUnhealthy", 0) .detail("PriorityTeam2Left", 0) .detail("PriorityTeam1Left", 0) .detail("PriorityTeam0Left", 0) .detail("PrioritySplitShard", 0) .trackLatest(self->movingDataEventHolder->trackingKey);
															#line 720 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", true) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", 0) .trackLatest(self->totalDataInFlightEventHolder->trackingKey);
															#line 726 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent("TotalDataInFlight", self->ddId) .detail("Primary", false) .detail("TotalBytes", 0) .detail("UnhealthyServers", 0) .detail("HighestPriority", self->configuration.usableRegions > 1 ? 0 : -1) .trackLatest(self->totalDataInFlightRemoteEventHolder->trackingKey);
															#line 3036 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<InitActor*>(this)->actor_wait_state > 0) static_cast<InitActor*>(this)->actor_wait_state = 0;
		static_cast<InitActor*>(this)->ActorCallback< InitActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< InitActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< InitActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 3118 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via init()
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class InitActor final : public Actor<Void>, public ActorCallback< InitActor, 0, Void >, public ActorCallback< InitActor, 1, Void >, public ActorCallback< InitActor, 2, Void >, public ActorCallback< InitActor, 3, Void >, public ActorCallback< InitActor, 4, Void >, public ActorCallback< InitActor, 5, Void >, public FastAllocated<InitActor>, public InitActorState<InitActor> {
															#line 3123 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<InitActor>::operator new;
	using FastAllocated<InitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11340017199883505920UL, 3596579065592238336UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitActor, 0, Void >;
friend struct ActorCallback< InitActor, 1, Void >;
friend struct ActorCallback< InitActor, 2, Void >;
friend struct ActorCallback< InitActor, 3, Void >;
friend struct ActorCallback< InitActor, 4, Void >;
friend struct ActorCallback< InitActor, 5, Void >;
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	InitActor(Reference<DataDistributor> const& self) 
															#line 3145 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   InitActorState<InitActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("init", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1770802396446596864UL, 4776746127134105600UL);
		ActorExecutionContextHelper __helper(static_cast<InitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("init");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("init", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< InitActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< InitActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< InitActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< InitActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< InitActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> init( Reference<DataDistributor> const& self ) {
															#line 620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new InitActor(self));
															#line 3182 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 735 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3187 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class RemoveDataMoveTombstoneBackgroundActor>
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActorState {
															#line 3193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActorState(Reference<DataDistributor> const& self) 
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 737 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentID()
															#line 3202 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	~RemoveDataMoveTombstoneBackgroundActorState() 
	{
		fdb_probe_actor_destroy("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 739 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				cx = openDBOnServer(self->dbInfo, TaskPriority::DefaultEndpoint, LockAware::True);
															#line 740 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr = Transaction(cx);
															#line 741 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				;
															#line 3222 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 766 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoveDataMoveTombstoneBackgroundActorState(); static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->destroy(); return 0; }
															#line 3251 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoveDataMoveTombstoneBackgroundActorState();
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 757 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 3264 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 758 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3268 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 760 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "RemoveDataMoveTombstoneError", self->ddId) .errorUnsuppressed(e) .detail("CurrentDataMoveID", currentID);
															#line 3272 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 743 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 744 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 745 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			for( UID& dataMoveID : self->initData->toCleanDataMoveTombstone ) {
															#line 746 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				currentID = dataMoveID;
															#line 747 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				tr.clear(dataMoveKeyFor(currentID));
															#line 748 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "RemoveDataMoveTombstone", self->ddId).detail("DataMoveID", currentID);
															#line 3311 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr.commit();
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3317 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 1;
															#line 750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3322 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 753 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 753 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3359 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 2;
															#line 753 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*>(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)));
															#line 3364 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state > 0) static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 737 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID currentID;
															#line 739 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 740 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 3570 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via removeDataMoveTombstoneBackground()
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class RemoveDataMoveTombstoneBackgroundActor final : public Actor<Void>, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >, public ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >, public FastAllocated<RemoveDataMoveTombstoneBackgroundActor>, public RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor> {
															#line 3575 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator new;
	using FastAllocated<RemoveDataMoveTombstoneBackgroundActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17496662904649400832UL, 6758512950865835264UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >;
friend struct ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >;
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RemoveDataMoveTombstoneBackgroundActor(Reference<DataDistributor> const& self) 
															#line 3593 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   RemoveDataMoveTombstoneBackgroundActorState<RemoveDataMoveTombstoneBackgroundActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929663207261195520UL, 2170269584154920960UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveDataMoveTombstoneBackgroundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("removeDataMoveTombstoneBackground");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("removeDataMoveTombstoneBackground", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RemoveDataMoveTombstoneBackgroundActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> removeDataMoveTombstoneBackground( Reference<DataDistributor> const& self ) {
															#line 736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new RemoveDataMoveTombstoneBackgroundActor(self));
															#line 3626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 768 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

																#line 3631 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromShards()
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromShardsActor>
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActorState {
															#line 3637 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActorState(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   traceShard(traceShard)
															#line 3646 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromShardsActorState() 
	{
		fdb_probe_actor_destroy("resumeFromShards", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 771 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD)
															#line 3661 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 772 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for(int i = 0;i < self->initData->shards.size() - 1;i++) {
															#line 773 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					const DDShardInfo& iShard = self->initData->shards[i];
															#line 774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					KeyRangeRef keys = KeyRangeRef(iShard.key, self->initData->shards[i + 1].key);
															#line 775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					teams.emplace_back(iShard.primarySrc, true);
															#line 777 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (self->configuration.usableRegions > 1)
															#line 3675 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 778 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.emplace_back(iShard.remoteSrc, false);
															#line 3679 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					}
															#line 780 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->physicalShardCollection->initPhysicalShardCollection(keys, teams, iShard.srcId.first(), 0);
															#line 3683 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 784 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundaries = std::vector<Key>();
															#line 785 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadIsEnabled(self->initData->bulkLoadMode))
															#line 3690 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 787 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevInfo, "DDInitCustomRangeConfigDisabledByBulkLoadMode", self->ddId);
															#line 3694 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			else
			{
															#line 789 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				for( auto it : self->initData->userRangeConfig->ranges() ) {
															#line 790 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					auto range = it->range();
															#line 791 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					customBoundaries.push_back(range.begin);
															#line 792 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDInitCustomRangeConfig", self->ddId) .detail("Range", KeyRangeRef(range.begin, range.end)) .detail("Config", it->value());
															#line 3706 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 798 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			shard = 0;
															#line 799 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary = 0;
															#line 800 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			overreplicatedCount = 0;
															#line 801 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 3717 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 875 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromShardsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromShardsActorState(); static_cast<ResumeFromShardsActor*>(this)->destroy(); return 0; }
															#line 3740 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromShardsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromShardsActorState();
		static_cast<ResumeFromShardsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 801 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(shard < self->initData->shards.size() - 1))
															#line 3759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 802 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DDShardInfo& iShard = self->initData->shards[shard];
															#line 803 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<KeyRangeRef> ranges;
															#line 805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key beginKey = iShard.key;
															#line 806 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Key endKey = self->initData->shards[shard + 1].key;
															#line 807 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] <= beginKey;) {
															#line 808 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 810 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;customBoundary < customBoundaries.size() && customBoundaries[customBoundary] < endKey;) {
															#line 811 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ranges.push_back(KeyRangeRef(beginKey, customBoundaries[customBoundary]));
															#line 812 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = customBoundaries[customBoundary];
															#line 813 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			customBoundary++;
															#line 3785 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 815 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ranges.push_back(KeyRangeRef(beginKey, endKey));
															#line 817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 818 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.primarySrc, true));
															#line 819 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->configuration.usableRegions > 1)
															#line 3795 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 820 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			teams.push_back(ShardsAffectedByTeamFailure::Team(iShard.remoteSrc, false));
															#line 3799 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 823 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int r = 0;r < ranges.size();r++) {
															#line 824 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto& keys = ranges[r];
															#line 825 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->defineShard(keys);
															#line 827 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->initData->userRangeConfig->rangeContaining(keys.begin);
															#line 828 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			int customReplicas = std::max(self->configuration.storageTeamSize, it->value().replicationFactor.orDefault(0));
															#line 830 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT_WE_THINK(KeyRangeRef(it->range().begin, it->range().end).contains(keys));
															#line 832 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bool unhealthy = iShard.primarySrc.size() != customReplicas;
															#line 833 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && self->configuration.usableRegions > 1)
															#line 3817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 834 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				unhealthy = iShard.remoteSrc.size() != customReplicas;
															#line 3821 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 836 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!unhealthy && iShard.primarySrc.size() > self->configuration.storageTeamSize)
															#line 3825 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 837 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (++overreplicatedCount > SERVER_KNOBS->DD_MAX_SHARDS_ON_LARGE_TEAMS)
															#line 3829 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 838 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					unhealthy = true;
															#line 3833 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
			}
															#line 842 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (traceShard)
															#line 3838 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 843 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDInitShard", self->ddId) .detail("Keys", keys) .detail("PrimarySrc", describe(iShard.primarySrc)) .detail("RemoteSrc", describe(iShard.remoteSrc)) .detail("PrimaryDest", describe(iShard.primaryDest)) .detail("RemoteDest", describe(iShard.remoteDest)) .detail("SrcID", iShard.srcId) .detail("DestID", iShard.destId) .detail("CustomReplicas", customReplicas) .detail("StorageTeamSize", self->configuration.storageTeamSize) .detail("Unhealthy", unhealthy) .detail("Overreplicated", overreplicatedCount);
															#line 3842 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 857 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->shardsAffectedByTeamFailure->moveShard(keys, teams);
															#line 858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if ((ddLargeTeamEnabled() && (unhealthy || r > 0)) || (iShard.hasDest && iShard.destId == anonymousShardId))
															#line 3848 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				DataMovementReason reason = DataMovementReason::RECOVER_MOVE;
															#line 864 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (unhealthy)
															#line 3854 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 865 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					reason = DataMovementReason::TEAM_UNHEALTHY;
															#line 3858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 866 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (r > 0)
															#line 3864 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 867 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						reason = DataMovementReason::SPLIT_SHARD;
															#line 3868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					}
				}
															#line 869 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(RelocateShard(keys, reason, RelocateReason::OTHER));
															#line 3873 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
		}
															#line 873 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = yield(TaskPriority::DataDistribution);
															#line 873 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3880 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 1;
															#line 873 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromShardsActor, 0, Void >*>(static_cast<ResumeFromShardsActor*>(this)));
															#line 3885 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 801 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3907 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 801 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		shard++;
															#line 3916 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromShardsActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromShardsActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromShardsActor*>(this)->ActorCallback< ResumeFromShardsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromShardsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromShardsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool traceShard;
															#line 784 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Key> customBoundaries;
															#line 798 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int shard;
															#line 799 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int customBoundary;
															#line 800 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int overreplicatedCount;
															#line 4008 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromShards()
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromShardsActor final : public Actor<Void>, public ActorCallback< ResumeFromShardsActor, 0, Void >, public FastAllocated<ResumeFromShardsActor>, public ResumeFromShardsActorState<ResumeFromShardsActor> {
															#line 4013 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromShardsActor>::operator new;
	using FastAllocated<ResumeFromShardsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3908311487560045824UL, 8643588808709793792UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromShardsActor, 0, Void >;
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromShardsActor(Reference<DataDistributor> const& self,bool const& traceShard) 
															#line 4030 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromShardsActorState<ResumeFromShardsActor>(self, traceShard),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17921411212936075776UL, 15009917662088599808UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromShardsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromShards");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromShards", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromShardsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromShards( Reference<DataDistributor> const& self, bool const& traceShard ) {
															#line 769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromShardsActor(self, traceShard));
															#line 4062 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 877 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// TODO: unit test needed
																#line 4068 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
// This generated class is to be used only via resumeFromDataMoves()
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ResumeFromDataMovesActor>
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActorState {
															#line 4074 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart),
															#line 880 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   it(self->initData->dataMoveMap.ranges().begin())
															#line 4085 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	~ResumeFromDataMovesActorState() 
	{
		fdb_probe_actor_destroy("resumeFromDataMoves", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 882 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 882 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4102 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 1;
															#line 882 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 0, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4107 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 884 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4130 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 884 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 4139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 946 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->txnProcessor->isMocked())
															#line 4223 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 947 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->addActor.send(self->removeDataMoveTombstoneBackground(self));
															#line 4227 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 949 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ResumeFromDataMovesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResumeFromDataMovesActorState(); static_cast<ResumeFromDataMovesActor*>(this)->destroy(); return 0; }
															#line 4231 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ResumeFromDataMovesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResumeFromDataMovesActorState();
		static_cast<ResumeFromDataMovesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 884 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(it != self->initData->dataMoveMap.ranges().end()))
															#line 4250 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 885 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		const DataMoveMetaData& meta = it.value()->meta;
															#line 886 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		DataMoveType dataMoveType = getDataMoveTypeFromDataMoveId(meta.id);
															#line 887 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.ranges.empty())
															#line 4260 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 888 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "EmptyDataMoveRange", self->ddId).detail("DataMoveMetaData", meta.toString());
															#line 4264 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1cont1continue1(loopDepth); // continue
		}
															#line 891 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (meta.bulkLoadTaskState.present())
															#line 4269 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 892 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 893 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.dataMoveId = meta.id;
															#line 894 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rs.cancelled = true;
															#line 895 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->relocationProducer.send(rs);
															#line 898 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMove", self->ddId) .detail("Reason", "DDInit") .detail("DataMove", meta.toString());
															#line 4281 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		else
		{
															#line 901 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (dataMoveType == DataMoveType::LOGICAL_BULKLOAD || dataMoveType == DataMoveType::PHYSICAL_BULKLOAD)
															#line 4288 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 904 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 905 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.dataMoveId = meta.id;
															#line 906 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				rs.cancelled = true;
															#line 907 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				self->relocationProducer.send(rs);
															#line 908 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadTaskCancelDataMoveForWrongType", self->ddId) .detail("Reason", "WrongTypeWhenDDInit") .detail("DataMoveType", dataMoveType) .detail("DataMove", meta.toString());
															#line 4300 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = a_body1cont1loopBody1cont5(loopDepth);
			}
			else
			{
															#line 912 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (it.value()->isCancelled() || (it.value()->valid && !SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA))
															#line 4307 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 914 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 915 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.dataMoveId = meta.id;
															#line 916 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					rs.cancelled = true;
															#line 917 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					self->relocationProducer.send(rs);
															#line 918 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent("DDInitScheduledCancelDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 4319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = a_body1cont1loopBody1cont7(loopDepth);
				}
				else
				{
															#line 919 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (it.value()->valid)
															#line 4326 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 920 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						TraceEvent(SevDebug, "DDInitFoundDataMove", self->ddId).detail("DataMove", meta.toString());
															#line 921 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						ASSERT(meta.ranges.front() == it.range());
															#line 923 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						RelocateShard rs(meta.ranges.front(), DataMovementReason::RECOVER_MOVE, RelocateReason::OTHER);
															#line 924 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMoveId = meta.id;
															#line 925 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						rs.dataMove = it.value();
															#line 926 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						std::vector<ShardsAffectedByTeamFailure::Team> teams;
															#line 927 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->primaryDest, true));
															#line 928 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (!rs.dataMove->remoteDest.empty())
															#line 4344 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 929 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
							teams.push_back(ShardsAffectedByTeamFailure::Team(rs.dataMove->remoteDest, false));
															#line 4348 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						}
															#line 934 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->restartShardTracker.send(rs.keys);
															#line 935 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->defineShard(rs.keys);
															#line 939 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->shardsAffectedByTeamFailure->moveShard(rs.keys, teams);
															#line 940 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						self->relocationProducer.send(rs);
															#line 941 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						StrictFuture<Void> __when_expr_1 = yield(TaskPriority::DataDistribution);
															#line 941 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
						static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 2;
															#line 941 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResumeFromDataMovesActor, 1, Void >*>(static_cast<ResumeFromDataMovesActor*>(this)));
															#line 4367 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1cont1loopBody1cont9(loopDepth);
					}
				}
			}
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1continue1(int loopDepth) 
	{
															#line 884 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4397 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 884 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		++it;
															#line 4406 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state > 0) static_cast<ResumeFromDataMovesActor*>(this)->actor_wait_state = 0;
		static_cast<ResumeFromDataMovesActor*>(this)->ActorCallback< ResumeFromDataMovesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResumeFromDataMovesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 880 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRangeMap<std::shared_ptr<DataMove>>::iterator it;
															#line 4522 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via resumeFromDataMoves()
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ResumeFromDataMovesActor final : public Actor<Void>, public ActorCallback< ResumeFromDataMovesActor, 0, Void >, public ActorCallback< ResumeFromDataMovesActor, 1, Void >, public FastAllocated<ResumeFromDataMovesActor>, public ResumeFromDataMovesActorState<ResumeFromDataMovesActor> {
															#line 4527 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ResumeFromDataMovesActor>::operator new;
	using FastAllocated<ResumeFromDataMovesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12026926373210566656UL, 15239320423948385280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResumeFromDataMovesActor, 0, Void >;
friend struct ActorCallback< ResumeFromDataMovesActor, 1, Void >;
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ResumeFromDataMovesActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 4545 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ResumeFromDataMovesActorState<ResumeFromDataMovesActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11929780632836526848UL, 14642014520803479552UL);
		ActorExecutionContextHelper __helper(static_cast<ResumeFromDataMovesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resumeFromDataMoves");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resumeFromDataMoves", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResumeFromDataMovesActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] static Future<Void> resumeFromDataMoves( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ResumeFromDataMovesActor(self, readyToStart));
															#line 4578 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 951 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

	// Resume inflight relocations from the previous DD
	// TODO: The initialDataDistribution is unused once resumeRelocations,
	// DataDistributionTracker::trackInitialShards, and DDTeamCollection::init are done. In the future, we can release
	// the object to save memory usage if it turns out to be a problem.
	Future<Void> resumeRelocations() {
		ASSERT(shardsAffectedByTeamFailure); // has to be allocated
		Future<Void> shardsReady = resumeFromShards(Reference<DataDistributor>::addRef(this), g_network->isSimulated());
		return resumeFromDataMoves(Reference<DataDistributor>::addRef(this), shardsReady);
	}

	Future<Void> pollMoveKeysLock() const {
		return txnProcessor->pollMoveKeysLock(lock, context->ddEnabledState.get());
	}

	Future<bool> isDataDistributionEnabled() const {
		return txnProcessor->isDataDistributionEnabled(context->ddEnabledState.get());
	}

	Future<Void> removeKeysFromFailedServer(const UID& serverID, const std::vector<UID>& teamForDroppedRange) const {
		return txnProcessor->removeKeysFromFailedServer(
		    serverID, teamForDroppedRange, lock, context->ddEnabledState.get());
	}

	Future<Void> removeStorageServer(const UID& serverID, const Optional<UID>& tssPairID = Optional<UID>()) const {
		return txnProcessor->removeStorageServer(serverID, tssPairID, lock, context->ddEnabledState.get());
	}

	Future<Void> initDDConfigWatch();

	Future<Void> initTenantCache();
};

Future<Void> DataDistributor::initDDConfigWatch() {
	if (txnProcessor->isMocked()) {
		onConfigChange = Never();
		return Void();
	}
	onConfigChange = map(DDConfiguration().trigger.onChange(
	                         SystemDBWriteLockedNow(txnProcessor->context().getReference()), {}, configChangeWatching),
	                     [](Version v) {
		                     CODE_PROBE(true, "DataDistribution change detected");
		                     TraceEvent("DataDistributionConfigChanged").detail("ChangeVersion", v);
		                     throw dd_config_changed();
		                     return Void();
	                     });

	return success(configChangeWatching.getFuture());
}

Future<Void> DataDistributor::initTenantCache() {
	// SOMEDAY: support tenant cache in MockDD
	ASSERT(!txnProcessor->isMocked());
	ddTenantCache = makeReference<TenantCache>(txnProcessor->context(), ddId);
	return ddTenantCache.get()->build();
}

inline void addAuditToAuditMap(Reference<DataDistributor> self, std::shared_ptr<DDAudit> audit) {
	AuditType auditType = audit->coreState.getType();
	UID auditID = audit->coreState.id;
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "addAuditToAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(!self->audits[auditType].contains(auditID));
	self->audits[auditType][auditID] = audit;
	return;
}

inline std::shared_ptr<DDAudit> getAuditFromAuditMap(Reference<DataDistributor> self,
                                                     AuditType auditType,
                                                     UID auditID) {
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "getAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	return self->audits[auditType][auditID];
}

inline void removeAuditFromAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	ASSERT(self->audits.contains(auditType) && self->audits[auditType].contains(auditID));
	std::shared_ptr<DDAudit> audit = self->audits[auditType][auditID];
	audit->cancel();
	self->audits[auditType].erase(auditID);
	TraceEvent(SevDebug, "AuditMapOps", self->ddId)
	    .detail("Ops", "removeAuditFromAuditMap")
	    .detail("AuditType", auditType)
	    .detail("AuditID", auditID);
	return;
}

inline bool auditExistInAuditMap(Reference<DataDistributor> self, AuditType auditType, UID auditID) {
	return self->audits.contains(auditType) && self->audits[auditType].contains(auditID);
}

inline bool existAuditInAuditMapForType(Reference<DataDistributor> self, AuditType auditType) {
	return self->audits.contains(auditType) && !self->audits[auditType].empty();
}

inline std::unordered_map<UID, std::shared_ptr<DDAudit>> getAuditsForType(Reference<DataDistributor> self,
                                                                          AuditType auditType) {
	ASSERT(self->audits.contains(auditType));
	return self->audits[auditType];
}

// Periodically check and log the physicalShard status; clean up empty physicalShard;
															#line 4689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MonitorPhysicalShardStatusActor>
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActorState {
															#line 4696 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActorState(Reference<PhysicalShardCollection> const& self) 
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self)
															#line 4703 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	~MonitorPhysicalShardStatusActorState() 
	{
		fdb_probe_actor_destroy("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1059 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA);
															#line 1060 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(SERVER_KNOBS->ENABLE_DD_PHYSICAL_SHARD);
															#line 1061 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 4722 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorPhysicalShardStatusActorState();
		static_cast<MonitorPhysicalShardStatusActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1062 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->cleanUpPhysicalShardCollection();
															#line 1063 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->logPhysicalShardCollection();
															#line 1064 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->PHYSICAL_SHARD_METRICS_DELAY);
															#line 1064 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4758 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 1;
															#line 1064 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*>(static_cast<MonitorPhysicalShardStatusActor*>(this)));
															#line 4763 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state > 0) static_cast<MonitorPhysicalShardStatusActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorPhysicalShardStatusActor*>(this)->ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<PhysicalShardCollection> self;
															#line 4857 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via monitorPhysicalShardStatus()
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MonitorPhysicalShardStatusActor final : public Actor<Void>, public ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >, public FastAllocated<MonitorPhysicalShardStatusActor>, public MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor> {
															#line 4862 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator new;
	using FastAllocated<MonitorPhysicalShardStatusActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16006330598435399936UL, 8957219675499161088UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >;
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MonitorPhysicalShardStatusActor(Reference<PhysicalShardCollection> const& self) 
															#line 4879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorPhysicalShardStatusActorState<MonitorPhysicalShardStatusActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8017156993063803648UL, 8326163650824365056UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorPhysicalShardStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorPhysicalShardStatus");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorPhysicalShardStatus", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorPhysicalShardStatusActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> monitorPhysicalShardStatus( Reference<PhysicalShardCollection> const& self ) {
															#line 1058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MonitorPhysicalShardStatusActor(self));
															#line 4912 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1067 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Trigger a task on range based on the current bulk load task metadata
															#line 4918 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class TriggerBulkLoadTaskActor>
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActorState {
															#line 4925 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId)
															#line 4936 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~TriggerBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1072 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 4951 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1073 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Database cx = self->txnProcessor->context();
															#line 1074 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		tr = Transaction(cx);
															#line 1075 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState = BulkLoadTaskState();
															#line 4985 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		try {
															#line 1077 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1078 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1079 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1079 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4995 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1079 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5000 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1105 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 5022 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1106 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistTriggerStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 5026 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1111 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1111 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5032 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1111 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5037 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1080 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1081 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1081 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5056 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1081 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5061 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1080 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadPhase> phase;
															#line 1081 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(newBulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1081 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5074 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1081 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5079 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1087 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1088 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1089 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1090 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5173 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5178 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1087 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.phase = BulkLoadPhase::Triggered;
															#line 1088 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.clearDataMoveId();
															#line 1089 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.restartCount = newBulkLoadTaskState.restartCount + 1;
															#line 1090 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		newBulkLoadTaskState.triggerTime = now();
															#line 1091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange(&tr, bulkLoadTaskPrefix, newBulkLoadTaskState.getRange(), bulkLoadTaskStateValue(newBulkLoadTaskState));
															#line 1091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5197 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5202 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1095 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1095 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5288 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1095 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5293 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1095 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1095 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5304 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1095 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*>(static_cast<TriggerBulkLoadTaskActor*>(this)));
															#line 5309 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1096 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1102 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 5399 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1096 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistTriggerState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", newBulkLoadTaskState.getTaskId()) .detail("JobID", newBulkLoadTaskState.getJobId());
															#line 1101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(commitVersion != invalidVersion);
															#line 1102 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<TriggerBulkLoadTaskActor*>(this)->SAV<std::pair<BulkLoadTaskState, Version>>::futures) { (void)(std::make_pair(newBulkLoadTaskState, commitVersion)); this->~TriggerBulkLoadTaskActorState(); static_cast<TriggerBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 5417 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<TriggerBulkLoadTaskActor*>(this)->SAV< std::pair<BulkLoadTaskState, Version> >::value()) std::pair<BulkLoadTaskState, Version>(std::make_pair(newBulkLoadTaskState, commitVersion));
		this->~TriggerBulkLoadTaskActorState();
		static_cast<TriggerBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<TriggerBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<TriggerBulkLoadTaskActor*>(this)->ActorCallback< TriggerBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1074 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1075 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState newBulkLoadTaskState;
															#line 5597 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via triggerBulkLoadTask()
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class TriggerBulkLoadTaskActor final : public Actor<std::pair<BulkLoadTaskState, Version>>, public ActorCallback< TriggerBulkLoadTaskActor, 0, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 1, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 2, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 3, Void >, public ActorCallback< TriggerBulkLoadTaskActor, 4, Void >, public FastAllocated<TriggerBulkLoadTaskActor>, public TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor> {
															#line 5602 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<TriggerBulkLoadTaskActor>::operator new;
	using FastAllocated<TriggerBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12930693789163536896UL, 15765819414618321664UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::pair<BulkLoadTaskState, Version>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TriggerBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< TriggerBulkLoadTaskActor, 4, Void >;
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	TriggerBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 5623 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<std::pair<BulkLoadTaskState, Version>>(),
		   TriggerBulkLoadTaskActorState<TriggerBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9998265325154424320UL, 14265371869313678848UL);
		ActorExecutionContextHelper __helper(static_cast<TriggerBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("triggerBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("triggerBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< TriggerBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<std::pair<BulkLoadTaskState, Version>> triggerBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1069 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<std::pair<BulkLoadTaskState, Version>>(new TriggerBulkLoadTaskActor(self, taskRange, taskId));
															#line 5660 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1115 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// TODO(BulkLoad): add reason to persist
															#line 5666 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via failBulkLoadTask()
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FailBulkLoadTaskActor>
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActorState {
															#line 5673 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cancelledDataMovePriority(cancelledDataMovePriority),
															#line 1121 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1122 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1123 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState()
															#line 5692 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~FailBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("failBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1124 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 5707 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1154 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FailBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FailBulkLoadTaskActorState(); static_cast<FailBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 5730 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FailBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FailBulkLoadTaskActorState();
		static_cast<FailBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1126 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1127 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5756 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 0, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5761 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1145 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 5796 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1146 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskPersistErrorStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 5800 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1151 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1151 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5806 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 5;
															#line 1151 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 4, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5811 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1129 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1129 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5828 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1129 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5833 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1129 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskState, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Triggered, BulkLoadPhase::Running }));
															#line 1129 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5844 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1129 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 1, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5849 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1131 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1132 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1133 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1134 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5943 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5948 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1131 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.phase = BulkLoadPhase::Error;
															#line 1132 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState.setCancelledDataMovePriority(cancelledDataMovePriority);
															#line 1133 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(taskRange == bulkLoadTaskState.getRange() && taskId == bulkLoadTaskState.getTaskId());
															#line 1134 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(normalKeys.contains(taskRange));
															#line 1135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadTaskPrefix, bulkLoadTaskState.getRange(), bulkLoadTaskStateValue(bulkLoadTaskState));
															#line 1135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 2, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 5972 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1137 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1137 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6058 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1137 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6063 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1137 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr.commit();
															#line 1137 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6074 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1137 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FailBulkLoadTaskActor, 3, Void >*>(static_cast<FailBulkLoadTaskActor*>(this)));
															#line 6079 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1138 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6165 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1138 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskPersistErrorState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 6176 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<FailBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<FailBulkLoadTaskActor*>(this)->ActorCallback< FailBulkLoadTaskActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FailBulkLoadTaskActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int cancelledDataMovePriority;
															#line 1121 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1122 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1123 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 6357 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via failBulkLoadTask()
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FailBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< FailBulkLoadTaskActor, 0, Void >, public ActorCallback< FailBulkLoadTaskActor, 1, Void >, public ActorCallback< FailBulkLoadTaskActor, 2, Void >, public ActorCallback< FailBulkLoadTaskActor, 3, Void >, public ActorCallback< FailBulkLoadTaskActor, 4, Void >, public FastAllocated<FailBulkLoadTaskActor>, public FailBulkLoadTaskActorState<FailBulkLoadTaskActor> {
															#line 6362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FailBulkLoadTaskActor>::operator new;
	using FastAllocated<FailBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8788720969351100672UL, 2149448230096599040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FailBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 3, Void >;
friend struct ActorCallback< FailBulkLoadTaskActor, 4, Void >;
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FailBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId,int const& cancelledDataMovePriority) 
															#line 6383 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FailBulkLoadTaskActorState<FailBulkLoadTaskActor>(self, taskRange, taskId, cancelledDataMovePriority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5276853651913060352UL, 273265509463260672UL);
		ActorExecutionContextHelper __helper(static_cast<FailBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("failBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("failBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FailBulkLoadTaskActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> failBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId, int const& cancelledDataMovePriority ) {
															#line 1117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FailBulkLoadTaskActor(self, taskRange, taskId, cancelledDataMovePriority));
															#line 6420 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// A bulk load task is guaranteed to be either complete or overwritten by another task
															#line 6426 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via doBulkLoadTask()
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class DoBulkLoadTaskActor>
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActorState {
															#line 6433 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   completeAck(),
															#line 1160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   triggeredBulkLoadTask(),
															#line 1161 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   commitVersion(invalidVersion),
															#line 1162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 6452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~DoBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("doBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1163 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadEngineParallelismLimitor.canStart());
															#line 1164 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadEngineParallelismLimitor.incrementTaskCounter();
															#line 6469 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1167 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<std::pair<BulkLoadTaskState, Version>> __when_expr_0 = triggerBulkLoadTask(self, range, taskId);
															#line 1167 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6475 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1167 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 6480 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1245 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadEngineParallelismLimitor.decrementTaskCounter();
															#line 1246 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<DoBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBulkLoadTaskActorState(); static_cast<DoBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 6511 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<DoBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoBulkLoadTaskActorState();
		static_cast<DoBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1231 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 6524 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1232 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6528 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1234 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(e) .detail("Phase", "Error") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1240 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 6534 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1241 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6538 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1168 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1169 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1177 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1183 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1188 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6568 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 6573 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
															#line 1168 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		triggeredBulkLoadTask = triggeredBulkLoadTask_.first;
															#line 1169 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		commitVersion = triggeredBulkLoadTask_.second;
															#line 1170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Triggered") .detail("TaskID", triggeredBulkLoadTask.getTaskId()) .detail("TaskRange", triggeredBulkLoadTask.getRange()) .detail("JobID", triggeredBulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion) .detail("Duration", now() - beginTime);
															#line 1177 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(triggeredBulkLoadTask.getRange() == range);
															#line 1183 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->publishTask(triggeredBulkLoadTask, commitVersion, completeAck);
															#line 1188 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->triggerShardBulkLoading.send(BulkLoadShardRequest(triggeredBulkLoadTask));
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<BulkLoadAck> __when_expr_1 = completeAck.getFuture();
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6596 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 6601 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> const& triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(triggeredBulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::pair<BulkLoadTaskState, Version> && triggeredBulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(triggeredBulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,std::pair<BulkLoadTaskState, Version> && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1190 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (ack.unretryableError)
															#line 6685 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1191 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "See unretryable error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 6689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_2 = failBulkLoadTask(self, range, taskId, ack.dataMovePriority);
															#line 1199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 6695 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
				static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoBulkLoadTaskActor, 2, Void >*>(static_cast<DoBulkLoadTaskActor*>(this)));
															#line 6700 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont3Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 1224 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Task complete") .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 6713 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck const& __ack,int loopDepth) 
	{
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ack = __ack;
															#line 6723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(BulkLoadAck && __ack,int loopDepth) 
	{
		ack = std::move(__ack);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,BulkLoadAck && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& failTaskError,int loopDepth=0) 
	{
		try {
															#line 1207 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_actor_cancelled)
															#line 6815 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1208 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 6819 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1210 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .errorUnsuppressed(failTaskError) .detail("Phase", "Failed to mark task error. Error should be bulkload_task_outdated") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 1217 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (failTaskError.code() == error_code_movekeys_conflict)
															#line 6825 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1218 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch2(failTaskError, loopDepth);
															#line 6829 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1220 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(failTaskError.code() == error_code_bulkload_task_outdated);
															#line 6833 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 1200 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 6848 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 1200 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadTaskDoTask", self->ddId) .detail("Phase", "Marked as error") .detail("CancelledDataMovePriority", ack.dataMovePriority) .detail("Range", range) .detail("TaskID", taskId) .detail("Duration", now() - beginTime);
															#line 6857 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<DoBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<DoBulkLoadTaskActor*>(this)->ActorCallback< DoBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<BulkLoadAck> completeAck;
															#line 1160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState triggeredBulkLoadTask;
															#line 1161 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Version commitVersion;
															#line 1162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadAck ack;
															#line 6979 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via doBulkLoadTask()
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class DoBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >, public ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >, public ActorCallback< DoBulkLoadTaskActor, 2, Void >, public FastAllocated<DoBulkLoadTaskActor>, public DoBulkLoadTaskActorState<DoBulkLoadTaskActor> {
															#line 6984 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<DoBulkLoadTaskActor>::operator new;
	using FastAllocated<DoBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4265611007971684096UL, 13270315467166168832UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >;
friend struct ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >;
friend struct ActorCallback< DoBulkLoadTaskActor, 2, Void >;
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	DoBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& taskId) 
															#line 7003 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   DoBulkLoadTaskActorState<DoBulkLoadTaskActor>(self, range, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9618644142710713856UL, 6256719351634756608UL);
		ActorExecutionContextHelper __helper(static_cast<DoBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 0, std::pair<BulkLoadTaskState, Version> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 1, BulkLoadAck >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> doBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& taskId ) {
															#line 1158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new DoBulkLoadTaskActor(self, range, taskId));
															#line 7038 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1248 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 7043 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class EraseBulkLoadTaskActor>
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActorState {
															#line 7050 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActorState(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskId(taskId),
															#line 1250 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1251 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1252 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask()
															#line 7067 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	~EraseBulkLoadTaskActorState() 
	{
		fdb_probe_actor_destroy("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1253 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 7082 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1293 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<EraseBulkLoadTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EraseBulkLoadTaskActorState(); static_cast<EraseBulkLoadTaskActor*>(this)->destroy(); return 0; }
															#line 7105 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<EraseBulkLoadTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EraseBulkLoadTaskActorState();
		static_cast<EraseBulkLoadTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1255 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1256 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1257 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(bulkLoadTask, getBulkLoadTask(&tr, taskRange, taskId, { BulkLoadPhase::Acknowledged }));
															#line 1257 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7131 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 1;
															#line 1257 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 0, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7136 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1280 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 7171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1281 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarn, "DDBulkLoadTaskEraseStateError", self->ddId) .errorUnsuppressed(e) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7175 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1286 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_bulkload_task_outdated)
															#line 7179 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7187 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 4;
															#line 1290 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 3, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7192 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1258 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1258 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7209 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1258 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7214 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1258 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRangeCoalescing( &tr, bulkLoadTaskPrefix, taskRange, normalKeys, bulkLoadTaskStateValue(BulkLoadTaskState()));
															#line 1258 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7225 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 2;
															#line 1258 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 1, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7230 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1260 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1260 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7316 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1260 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1260 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1260 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7332 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 3;
															#line 1260 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EraseBulkLoadTaskActor, 2, Void >*>(static_cast<EraseBulkLoadTaskActor*>(this)));
															#line 7337 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1261 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1262 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1266 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1267 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1268 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 7429 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1272 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1273 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7435 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1261 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1262 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskEraseState", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 1266 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->eraseTask(bulkLoadTask);
															#line 1267 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Optional<int> cancelledDataMovePriority = bulkLoadTask.getCancelledDataMovePriority();
															#line 1268 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (cancelledDataMovePriority.present() && cancelledDataMovePriority.get() != SERVER_KNOBS->PRIORITY_TEAM_HEALTHY)
															#line 7453 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1272 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->triggerShardBulkLoading.send(BulkLoadShardRequest(bulkLoadTask, cancelledDataMovePriority.get()));
															#line 1273 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskTriggerShardDatamove", self->ddId) .detail("CommitVersion", commitVersion) .detail("TaskRange", taskRange) .detail("TaskID", taskId);
															#line 7459 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state > 0) static_cast<EraseBulkLoadTaskActor*>(this)->actor_wait_state = 0;
		static_cast<EraseBulkLoadTaskActor*>(this)->ActorCallback< EraseBulkLoadTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< EraseBulkLoadTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID taskId;
															#line 1250 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1251 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1252 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 7639 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via eraseBulkLoadTask()
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class EraseBulkLoadTaskActor final : public Actor<Void>, public ActorCallback< EraseBulkLoadTaskActor, 0, Void >, public ActorCallback< EraseBulkLoadTaskActor, 1, Void >, public ActorCallback< EraseBulkLoadTaskActor, 2, Void >, public ActorCallback< EraseBulkLoadTaskActor, 3, Void >, public FastAllocated<EraseBulkLoadTaskActor>, public EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor> {
															#line 7644 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<EraseBulkLoadTaskActor>::operator new;
	using FastAllocated<EraseBulkLoadTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6814465310027435264UL, 796243771611466752UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< EraseBulkLoadTaskActor, 0, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 1, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 2, Void >;
friend struct ActorCallback< EraseBulkLoadTaskActor, 3, Void >;
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	EraseBulkLoadTaskActor(Reference<DataDistributor> const& self,KeyRange const& taskRange,UID const& taskId) 
															#line 7664 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   EraseBulkLoadTaskActorState<EraseBulkLoadTaskActor>(self, taskRange, taskId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(445300323391651072UL, 14768580555042305792UL);
		ActorExecutionContextHelper __helper(static_cast<EraseBulkLoadTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("eraseBulkLoadTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("eraseBulkLoadTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< EraseBulkLoadTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> eraseBulkLoadTask( Reference<DataDistributor> const& self, KeyRange const& taskRange, UID const& taskId ) {
															#line 1249 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new EraseBulkLoadTaskActor(self, taskRange, taskId));
															#line 7700 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 7705 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadTasksActor>
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActorState {
															#line 7712 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActorState(Reference<DataDistributor> const& self) 
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1297 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginKey(normalKeys.begin),
															#line 1298 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   endKey(normalKeys.end),
															#line 1299 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeToRead(),
															#line 1300 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1301 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1302 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   i(0),
															#line 1303 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1304 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   result(),
															#line 1305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadActors(),
															#line 1306 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range()
															#line 7739 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadTasksActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1307 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 7754 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1368 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = waitForAll(bulkLoadActors);
															#line 1368 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7779 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 7;
															#line 1368 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 7784 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1307 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 7800 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1309 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1310 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			result.clear();
															#line 1311 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store( result, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead, SERVER_KNOBS->DD_BULKLOAD_TASK_METADATA_READ_SIZE));
															#line 1311 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7813 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 1;
															#line 1311 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 7818 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 7853 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1363 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 7857 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1365 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_5 = tr.onError(e);
															#line 1365 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_5.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 6;
															#line 1365 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 7868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1314 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1315 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 7885 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1314 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1315 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 7896 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1360 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = result.back().key;
															#line 7980 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1315 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < result.size() - 1))
															#line 7996 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1316 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (result[i].value.empty())
															#line 8002 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3continue1(loopDepth); // continue
		}
															#line 1319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		range = Standalone(KeyRangeRef(result[i].key, result[i + 1].key));
															#line 1320 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[i].value);
															#line 1321 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 8012 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1322 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 1322 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8018 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 2;
															#line 1322 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8023 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3continue1(int loopDepth) 
	{
															#line 1315 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 8050 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1325 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (range != bulkLoadTaskState.getRange())
															#line 8059 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1327 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(0.1);
															#line 1327 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8065 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 3;
															#line 1327 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8070 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont6(int loopDepth) 
	{
															#line 1330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.phase == BulkLoadPhase::Submitted || bulkLoadTaskState.phase == BulkLoadPhase::Triggered || bulkLoadTaskState.phase == BulkLoadPhase::Running)
															#line 8171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1334 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 8175 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont3loopBody1cont6loopHead1(loopDepth);
		}
		else
		{
															#line 1346 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (bulkLoadTaskState.phase == BulkLoadPhase::Acknowledged)
															#line 8182 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1347 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskClearMetadata", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 1351 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_4 = eraseBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId());
															#line 1351 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8190 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
				static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 5;
															#line 1351 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8195 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1352 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (bulkLoadTaskState.phase == BulkLoadPhase::Error)
															#line 8202 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1353 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevWarnAlways, "DDBulkLoadTaskUnretriableError", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId());
															#line 8206 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				else
				{
															#line 1357 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(bulkLoadTaskState.phase == BulkLoadPhase::Complete);
															#line 8212 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
				loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont7(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont3loopBody1cont9(int loopDepth) 
	{
															#line 1315 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 8311 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont10(int loopDepth) 
	{
															#line 1340 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadTaskSchedule", self->ddId) .detail("Range", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("Phase", bulkLoadTaskState.phase);
															#line 1344 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadActors.push_back( doBulkLoadTask(self, bulkLoadTaskState.getRange(), bulkLoadTaskState.getTaskId()));
															#line 8322 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1(int loopDepth) 
	{
															#line 1335 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadEngineParallelismLimitor.canStart())
															#line 8338 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont3loopBody1cont6break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1338 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = self->bulkLoadEngineParallelismLimitor.waitUntilCounterChanged();
															#line 1338 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8346 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont6loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 4;
															#line 1338 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*>(static_cast<ScheduleBulkLoadTasksActor*>(this)));
															#line 8351 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont6loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3loopBody1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont3loopBody1cont11(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont3loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1369 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 8653 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1369 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadTasksActorState(); static_cast<ScheduleBulkLoadTasksActor*>(this)->destroy(); return 0; }
															#line 8665 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadTasksActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadTasksActorState();
		static_cast<ScheduleBulkLoadTasksActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadTasksActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadTasksActor*>(this)->ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1297 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1298 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1299 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1300 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1301 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1302 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1303 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1304 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult result;
															#line 1305 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> bulkLoadActors;
															#line 1306 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 8770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadTasks()
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadTasksActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >, public ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >, public FastAllocated<ScheduleBulkLoadTasksActor>, public ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor> {
															#line 8775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator new;
	using FastAllocated<ScheduleBulkLoadTasksActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13439003834593533440UL, 15551853557381680896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >;
friend struct ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >;
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadTasksActor(Reference<DataDistributor> const& self) 
															#line 8798 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadTasksActorState<ScheduleBulkLoadTasksActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14856297450692458752UL, 7460955458194361088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadTasksActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadTasks");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadTasks", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ScheduleBulkLoadTasksActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadTasks( Reference<DataDistributor> const& self ) {
															#line 1296 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadTasksActor(self));
															#line 8837 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1371 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 8842 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadTaskCoreActor>
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActorState {
															#line 8849 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActorState(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   readyToStart(readyToStart)
															#line 8858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadTaskCoreActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1373 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readyToStart;
															#line 1373 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8875 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 1;
															#line 1373 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 0, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 8880 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadTaskCoreActorState();
		static_cast<BulkLoadTaskCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1374 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8907 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 8912 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1374 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		cx = self->txnProcessor->context();
															#line 1375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = registerRangeLockOwner(cx, rangeLockNameForBulkLoad, rangeLockNameForBulkLoad);
															#line 1375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8925 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 2;
															#line 1375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 1, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 8930 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1376 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1377 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9016 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1376 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadTaskCoreStart", self->ddId);
															#line 1377 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 9027 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
		try {
															#line 1379 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = scheduleBulkLoadTasks(self);
															#line 1379 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9121 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 3;
															#line 1379 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 2, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9126 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1389 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(SERVER_KNOBS->DD_BULKLOAD_SCHEDULE_MIN_INTERVAL_SEC);
															#line 1389 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9143 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 4;
															#line 1389 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadTaskCoreActor, 3, Void >*>(static_cast<BulkLoadTaskCoreActor*>(this)));
															#line 9148 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1381 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 9158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1382 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1384 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadTaskCoreError", self->ddId).errorUnsuppressed(e);
															#line 1385 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_movekeys_conflict)
															#line 9168 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1386 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9172 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadTaskCoreActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadTaskCoreActor*>(this)->ActorCallback< BulkLoadTaskCoreActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadTaskCoreActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Future<Void> readyToStart;
															#line 1374 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 9377 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadTaskCore()
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadTaskCoreActor final : public Actor<Void>, public ActorCallback< BulkLoadTaskCoreActor, 0, Void >, public ActorCallback< BulkLoadTaskCoreActor, 1, Void >, public ActorCallback< BulkLoadTaskCoreActor, 2, Void >, public ActorCallback< BulkLoadTaskCoreActor, 3, Void >, public FastAllocated<BulkLoadTaskCoreActor>, public BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor> {
															#line 9382 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadTaskCoreActor>::operator new;
	using FastAllocated<BulkLoadTaskCoreActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3343432385513159936UL, 7317564238242518272UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadTaskCoreActor, 0, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 1, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 2, Void >;
friend struct ActorCallback< BulkLoadTaskCoreActor, 3, Void >;
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskCoreActor(Reference<DataDistributor> const& self,Future<Void> const& readyToStart) 
															#line 9402 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadTaskCoreActorState<BulkLoadTaskCoreActor>(self, readyToStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12210363843676168704UL, 8182787864321598208UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadTaskCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadTaskCore");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadTaskCore", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadTaskCoreActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadTaskCore( Reference<DataDistributor> const& self, Future<Void> const& readyToStart ) {
															#line 1372 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadTaskCoreActor(self, readyToStart));
															#line 9438 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1392 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

void explainBulkLoadJobGetRangeResult(const RangeResult& rangeResult) {
	TraceEvent(SevInfo, "DDBulkLoadJobExplainRangeResult").detail("Size", rangeResult.size());
	for (int i = 0; i < rangeResult.size(); i++) {
		TraceEvent e(SevInfo, "DDBulkLoadJobExplainRangeResultKV");
		e.detail("Key", rangeResult[i].key);
		if (rangeResult[i].value.empty()) {
			e.detail("Val", "empty");
		} else {
			BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[i].value);
			e.detail("Val", currentJobState.toString());
		}
	}
}

// Return the current bulk load job state for the given jobId and jobRange
// If the job is not found or outdated, throw bulkload_task_outdated error
															#line 9459 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via getBulkLoadJob()
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class GetBulkLoadJobActor>
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActorState {
															#line 9466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActorState(Transaction* const& tr,UID const& jobId,KeyRange const& jobRange) 
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : tr(tr),
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRange(jobRange),
															#line 1410 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   rangeResult(),
															#line 1411 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorMessage()
															#line 9481 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~GetBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("getBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1413 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(rangeResult, krmGetRanges(tr, bulkLoadJobPrefix, jobRange));
															#line 1413 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9499 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1413 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetBulkLoadJobActor, 0, Void >*>(static_cast<GetBulkLoadJobActor*>(this)));
															#line 9504 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1440 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 9534 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1441 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 9538 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1443 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (g_network->isSimulated() && e.code() == error_code_bulkload_task_outdated)
															#line 9542 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1444 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				explainBulkLoadJobGetRangeResult(rangeResult);
															#line 9546 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1446 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobGetError", jobId).errorUnsuppressed(e).detail("Message", errorMessage);
															#line 1447 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 9552 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1414 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult.size() != 2 && rangeResult.size() != 3)
															#line 9566 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1416 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("Expected 2 or 3 results for bulk load job {} in range {}, but got {} results", jobId.toString(), jobRange.toString(), std::to_string(rangeResult.size()));
															#line 1420 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9572 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1422 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 9576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1423 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("No bulk load job found in range {}", jobRange.toString());
															#line 1424 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9582 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1426 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1427 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 9588 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1428 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("No valid bulk load job in range {}", jobRange.toString());
															#line 1429 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9594 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1431 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId || currentJobState.getJobRange() != jobRange)
															#line 9598 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1432 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("Bulk load job {} in range {} is outdated, expected jobId {}", currentJobState.getJobId().toString(), jobRange.toString(), jobId.toString());
															#line 1436 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9604 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1438 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 9608 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1414 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult.size() != 2 && rangeResult.size() != 3)
															#line 9620 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1416 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("Expected 2 or 3 results for bulk load job {} in range {}, but got {} results", jobId.toString(), jobRange.toString(), std::to_string(rangeResult.size()));
															#line 1420 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9626 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1422 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (rangeResult[0].value.empty())
															#line 9630 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1423 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("No bulk load job found in range {}", jobRange.toString());
															#line 1424 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9636 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1426 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		BulkLoadJobState currentJobState = decodeBulkLoadJobState(rangeResult[0].value);
															#line 1427 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!currentJobState.isValid())
															#line 9642 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1428 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("No valid bulk load job in range {}", jobRange.toString());
															#line 1429 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9648 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1431 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getJobId() != jobId || currentJobState.getJobRange() != jobRange)
															#line 9652 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1432 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			errorMessage = fmt::format("Bulk load job {} in range {} is outdated, expected jobId {}", currentJobState.getJobId().toString(), jobRange.toString(), jobId.toString());
															#line 1436 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_task_outdated(), loopDepth);
															#line 9658 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1438 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<GetBulkLoadJobActor*>(this)->SAV<BulkLoadJobState>::futures) { (void)(currentJobState); this->~GetBulkLoadJobActorState(); static_cast<GetBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 9662 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<GetBulkLoadJobActor*>(this)->SAV< BulkLoadJobState >::value()) BulkLoadJobState(currentJobState);
		this->~GetBulkLoadJobActorState();
		static_cast<GetBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<GetBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<GetBulkLoadJobActor*>(this)->ActorCallback< GetBulkLoadJobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetBulkLoadJobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction* tr;
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1410 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult rangeResult;
															#line 1411 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string errorMessage;
															#line 9755 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via getBulkLoadJob()
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class GetBulkLoadJobActor final : public Actor<BulkLoadJobState>, public ActorCallback< GetBulkLoadJobActor, 0, Void >, public FastAllocated<GetBulkLoadJobActor>, public GetBulkLoadJobActorState<GetBulkLoadJobActor> {
															#line 9760 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<GetBulkLoadJobActor>::operator new;
	using FastAllocated<GetBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6140539518944699392UL, 13112899644562208768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadJobState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetBulkLoadJobActor, 0, Void >;
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	GetBulkLoadJobActor(Transaction* const& tr,UID const& jobId,KeyRange const& jobRange) 
															#line 9777 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadJobState>(),
		   GetBulkLoadJobActorState<GetBulkLoadJobActor>(tr, jobId, jobRange),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17307579102949951488UL, 17075247084345564672UL);
		ActorExecutionContextHelper __helper(static_cast<GetBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetBulkLoadJobActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadJobState> getBulkLoadJob( Transaction* const& tr, UID const& jobId, KeyRange const& jobRange ) {
															#line 1409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadJobState>(new GetBulkLoadJobActor(tr, jobId, jobRange));
															#line 9810 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1450 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Find task metadata for a bulk load job with jobId and input range
															#line 9816 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobFindTaskActor>
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActorState {
															#line 9823 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActorState(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,KeyRange const& jobRange,UID const& logId) 
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   range(range),
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRange(jobRange),
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   logId(logId),
															#line 1457 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTaskState(),
															#line 1458 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1459 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 9844 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobFindTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1460 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 9859 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1490 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(bulkLoadTaskState); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 9882 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(std::move(bulkLoadTaskState)); // state_var_RVO
		this->~BulkLoadJobFindTaskActorState();
		static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1462 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 1463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobId, jobRange);
															#line 1463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9906 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 1;
															#line 1463 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 9911 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1487 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1487 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9948 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 3;
															#line 1487 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 9953 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(BulkLoadJobState const& checkJobState,int loopDepth) 
	{
															#line 1465 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9972 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 9977 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(BulkLoadJobState && checkJobState,int loopDepth) 
	{
															#line 1465 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!range.empty());
															#line 1466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(&tr, bulkLoadTaskPrefix, range);
															#line 1466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9990 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 2;
															#line 1466 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*>(static_cast<BulkLoadJobFindTaskActor*>(this)));
															#line 9995 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& checkJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(checkJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && checkJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(checkJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(RangeResult const& result,int loopDepth) 
	{
															#line 1468 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1469 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1470 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 10083 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1471 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 10087 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1474 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1475 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 10099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1476 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1483 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 10105 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3(RangeResult && result,int loopDepth) 
	{
															#line 1468 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!result[0].value.empty() && result.size() == 2);
															#line 1469 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTaskState = decodeBulkLoadTaskState(result[0].value);
															#line 1470 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTaskState.isValid())
															#line 10119 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1471 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobFindTaskActor*>(this)->SAV<Optional<BulkLoadTaskState>>::futures) { (void)(Optional<BulkLoadTaskState>()); this->~BulkLoadJobFindTaskActorState(); static_cast<BulkLoadJobFindTaskActor*>(this)->destroy(); return 0; }
															#line 10123 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobFindTaskActor*>(this)->SAV< Optional<BulkLoadTaskState> >::value()) Optional<BulkLoadTaskState>(Optional<BulkLoadTaskState>());
			this->~BulkLoadJobFindTaskActorState();
			static_cast<BulkLoadJobFindTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1473 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		KeyRange currentRange = Standalone(KeyRangeRef(result[0].key, result[1].key));
															#line 1474 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(result[0].key != result[1].key);
															#line 1475 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTaskState.getRange() != currentRange)
															#line 10135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1476 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevError, "DDBulkLoadJobExecutorFindRangeMismatch", logId) .detail("InputRange", range) .detail("InputJobID", jobId) .detail("CurrentRange", currentRange) .detail("TaskRange", bulkLoadTaskState.getRange()) .detail("TaskID", bulkLoadTaskState.getTaskId()) .detail("JobID", bulkLoadTaskState.getJobId());
															#line 1483 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(false);
															#line 10141 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(result, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && result,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobFindTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobFindTaskActor*>(this)->ActorCallback< BulkLoadJobFindTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange range;
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID logId;
															#line 1457 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTaskState;
															#line 1458 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1459 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 10325 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobFindTask()
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobFindTaskActor final : public Actor<Optional<BulkLoadTaskState>>, public ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >, public ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >, public ActorCallback< BulkLoadJobFindTaskActor, 2, Void >, public FastAllocated<BulkLoadJobFindTaskActor>, public BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor> {
															#line 10330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobFindTaskActor>::operator new;
	using FastAllocated<BulkLoadJobFindTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2369128981543804160UL, 3552869588895698944UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<BulkLoadTaskState>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >;
friend struct ActorCallback< BulkLoadJobFindTaskActor, 2, Void >;
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFindTaskActor(Reference<DataDistributor> const& self,KeyRange const& range,UID const& jobId,KeyRange const& jobRange,UID const& logId) 
															#line 10349 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Optional<BulkLoadTaskState>>(),
		   BulkLoadJobFindTaskActorState<BulkLoadJobFindTaskActor>(self, range, jobId, jobRange, logId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3995601487305067776UL, 1783749224615820032UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobFindTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobFindTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobFindTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 0, BulkLoadJobState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobFindTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Optional<BulkLoadTaskState>> bulkLoadJobFindTask( Reference<DataDistributor> const& self, KeyRange const& range, UID const& jobId, KeyRange const& jobRange, UID const& logId ) {
															#line 1452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Optional<BulkLoadTaskState>>(new BulkLoadJobFindTaskActor(self, range, jobId, jobRange, logId));
															#line 10384 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1492 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Submit a bulkload task for the given jobId
															#line 10390 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobSubmitTaskActor>
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActorState {
															#line 10397 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests,KeyRange const& taskRange) 
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(manifests),
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1498 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1499 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1501 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(jobId, manifests, taskRange)
															#line 10416 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobSubmitTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1502 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 10431 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1522 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV<BulkLoadTaskState>::futures) { (void)(bulkLoadTask); this->~BulkLoadJobSubmitTaskActorState(); static_cast<BulkLoadJobSubmitTaskActor*>(this)->destroy(); return 0; }
															#line 10454 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobSubmitTaskActor*>(this)->SAV< BulkLoadTaskState >::value()) BulkLoadTaskState(std::move(bulkLoadTask)); // state_var_RVO
		this->~BulkLoadJobSubmitTaskActorState();
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10476 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 1;
															#line 1505 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10481 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1519 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1519 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10518 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 4;
															#line 1519 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10523 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1506 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1506 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10540 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1506 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10545 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1506 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = setBulkLoadSubmissionTransaction(&tr, bulkLoadTask);
															#line 1506 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10556 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 2;
															#line 1506 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10561 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1508 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1508 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10647 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1508 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10652 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1508 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1508 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10663 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 3;
															#line 1508 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*>(static_cast<BulkLoadJobSubmitTaskActor*>(this)));
															#line 10668 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1509 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1510 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 10754 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1509 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1510 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorSubmitTask", self->ddId) .detail("InputJobID", jobId) .detail("ManifestCount", manifests.size()) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("TaskJobID", bulkLoadTask.getJobId()) .detail("CommitVersion", commitVersion);
															#line 10765 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobSubmitTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobSubmitTaskActor*>(this)->ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1498 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1499 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1501 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 10946 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobSubmitTask()
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobSubmitTaskActor final : public Actor<BulkLoadTaskState>, public ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >, public ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >, public FastAllocated<BulkLoadJobSubmitTaskActor>, public BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor> {
															#line 10951 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator new;
	using FastAllocated<BulkLoadJobSubmitTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12905596175179700736UL, 4375547748691078912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BulkLoadTaskState>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >;
friend struct ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >;
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobSubmitTaskActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadManifestSet const& manifests,KeyRange const& taskRange) 
															#line 10971 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<BulkLoadTaskState>(),
		   BulkLoadJobSubmitTaskActorState<BulkLoadJobSubmitTaskActor>(self, jobId, manifests, taskRange),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9698732467664213248UL, 5402256740808584704UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobSubmitTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobSubmitTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobSubmitTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BulkLoadJobSubmitTaskActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<BulkLoadTaskState> bulkLoadJobSubmitTask( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadManifestSet const& manifests, KeyRange const& taskRange ) {
															#line 1494 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<BulkLoadTaskState>(new BulkLoadJobSubmitTaskActor(self, jobId, manifests, taskRange));
															#line 11007 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1524 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 11012 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobWaitUntilTaskCompleteOrErrorActor>
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActorState {
															#line 11019 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(bulkLoadTask)
															#line 11030 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1528 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(bulkLoadTask.isValid());
															#line 1529 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1530 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1531 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentTask = BulkLoadTaskState();
															#line 1532 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 11053 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1534 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentTask, getBulkLoadTask(&tr, bulkLoadTask.getRange(), bulkLoadTask.getTaskId(), { BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running, BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }));
															#line 1534 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11086 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1534 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 11091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1561 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_JOB_MONITOR_PERIOD_SEC);
															#line 1561 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11108 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 1561 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 11113 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1559 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 1559 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1559 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*>(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)));
															#line 11130 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1544 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 11145 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1545 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 11149 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1547 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 11153 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1548 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretryableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1553 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 11159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1555 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 11167 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1556 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 11171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1544 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.getJobId() != bulkLoadTask.getJobId())
															#line 11185 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1545 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1loopBody1Catch1(bulkload_task_outdated(), loopDepth);
															#line 11189 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1547 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Error)
															#line 11193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1548 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorFindUnretryableError", self->ddId) .detail("InputJobID", jobId) .detail("TaskJobID", currentTask.getJobId()) .detail("TaskRange", currentTask.getRange()) .detail("TaskID", currentTask.getTaskId());
															#line 1553 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 11199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1555 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentTask.phase == BulkLoadPhase::Complete || currentTask.phase == BulkLoadPhase::Acknowledged)
															#line 11207 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1556 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState(); static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 11211 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobWaitUntilTaskCompleteOrErrorActorState();
			static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1529 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1530 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1531 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState currentTask;
															#line 11495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobWaitUntilTaskCompleteOrError()
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobWaitUntilTaskCompleteOrErrorActor final : public Actor<Void>, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>, public BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor> {
															#line 11500 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16686839104351345664UL, 9870948140854504192UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >;
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobWaitUntilTaskCompleteOrErrorActor(Reference<DataDistributor> const& self,UID const& jobId,BulkLoadTaskState const& bulkLoadTask) 
															#line 11519 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobWaitUntilTaskCompleteOrErrorActorState<BulkLoadJobWaitUntilTaskCompleteOrErrorActor>(self, jobId, bulkLoadTask),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16093638368250768128UL, 12615896230081500416UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobWaitUntilTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobWaitUntilTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobWaitUntilTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobWaitUntilTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobWaitUntilTaskCompleteOrError( Reference<DataDistributor> const& self, UID const& jobId, BulkLoadTaskState const& bulkLoadTask ) {
															#line 1525 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobWaitUntilTaskCompleteOrErrorActor(self, jobId, bulkLoadTask));
															#line 11554 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1564 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// A bulkload job can contain multiple tasks. Each task can contain multiple manifests.
// Given a job range, the bulkload task range is defined as the range between the min begin key and the max end key of
// all manifests, overlapping with the maxRange (i.e. the job range).
KeyRange generateBulkLoadTaskRange(const BulkLoadManifestSet& manifests, const KeyRange& maxRange) {
	KeyRange manifestsRange = Standalone(KeyRangeRef(manifests.getMinBeginKey(), manifests.getMaxEndKey()));
	return manifestsRange & maxRange; // ensure the task range is within the maxRange
}

// Given a list of manifestEntry, create a bulkload task.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 11570 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobNewTaskActor>
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActorState {
															#line 11577 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,KeyRange const& jobRange,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRoot(jobRoot),
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRange(jobRange),
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestLocalTempFolder(manifestLocalTempFolder),
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntries(manifestEntries),
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1584 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1585 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1586 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifests(),
															#line 1587 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now()),
															#line 1588 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange()
															#line 11608 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobNewTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1589 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!manifestEntries.empty());
															#line 11623 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1592 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(manifests, getBulkLoadManifestMetadataFromEntry( manifestEntries, manifestLocalTempFolder, jobTransportMethod, jobRoot, self->ddId));
															#line 1592 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11629 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 1;
															#line 1592 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11634 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1649 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 11663 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobNewTaskActorState();
		static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1633 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 11676 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1634 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 11680 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1636 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("JobRange", jobRange) .detail("Duration", now() - beginTime);
															#line 1643 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 11686 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1644 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 11690 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1595 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		taskRange = generateBulkLoadTaskRange(manifests, jobRange);
															#line 1599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask(self, taskRange, jobId, jobRange, self->ddId);
															#line 1599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11710 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1595 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		taskRange = generateBulkLoadTaskRange(manifests, jobRange);
															#line 1599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Optional<BulkLoadTaskState>> __when_expr_1 = bulkLoadJobFindTask(self, taskRange, jobId, jobRange, self->ddId);
															#line 1599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11728 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 2;
															#line 1599 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11733 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1601 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 11817 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1603 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 11821 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1610 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1617 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests, taskRange));
															#line 1617 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11833 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1617 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11838 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1601 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (bulkLoadTask_.present())
															#line 11847 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1603 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobNewTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobNewTaskActorState(); static_cast<BulkLoadJobNewTaskActor*>(this)->destroy(); return 0; }
															#line 11851 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobNewTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobNewTaskActorState();
			static_cast<BulkLoadJobNewTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1610 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		manifests.setRootPath(jobRoot);
															#line 1617 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(bulkLoadTask, bulkLoadJobSubmitTask(self, jobId, manifests, taskRange));
															#line 1617 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 11863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 3;
															#line 1617 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*>(static_cast<BulkLoadJobNewTaskActor*>(this)));
															#line 11868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1619 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("JobRange", jobRange) .detail("Duration", now() - beginTime);
															#line 1627 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 11954 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1629 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1630 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 11960 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1619 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task submitted") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("JobRange", jobRange) .detail("Duration", now() - beginTime);
															#line 1627 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 11972 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1629 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "New");
															#line 1630 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 11978 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobNewTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobNewTaskActor*>(this)->ActorCallback< BulkLoadJobNewTaskActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string jobRoot;
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string manifestLocalTempFolder;
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1584 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1585 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1586 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadManifestSet manifests;
															#line 1587 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1588 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 12098 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobNewTask()
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobNewTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobNewTaskActor, 0, Void >, public ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobNewTaskActor, 2, Void >, public FastAllocated<BulkLoadJobNewTaskActor>, public BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor> {
															#line 12103 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobNewTaskActor>::operator new;
	using FastAllocated<BulkLoadJobNewTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4595848115385945088UL, 15338312177362530304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobNewTaskActor, 0, Void >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobNewTaskActor, 2, Void >;
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobNewTaskActor(Reference<DataDistributor> const& self,UID const& jobId,std::string const& jobRoot,KeyRange const& jobRange,BulkLoadTransportMethod const& jobTransportMethod,std::string const& manifestLocalTempFolder,std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries,Promise<Void> const& errorOut) 
															#line 12122 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobNewTaskActorState<BulkLoadJobNewTaskActor>(self, jobId, jobRoot, jobRange, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8655289167937893888UL, 2026549218517418240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobNewTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobNewTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobNewTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 1, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BulkLoadJobNewTaskActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobNewTask( Reference<DataDistributor> const& self, UID const& jobId, std::string const& jobRoot, KeyRange const& jobRange, BulkLoadTransportMethod const& jobTransportMethod, std::string const& manifestLocalTempFolder, std::vector<BulkLoadJobFileManifestEntry> const& manifestEntries, Promise<Void> const& errorOut ) {
															#line 1576 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobNewTaskActor(self, jobId, jobRoot, jobRange, jobTransportMethod, manifestLocalTempFolder, manifestEntries, errorOut));
															#line 12157 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1651 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Given a bulkload task range, find the task and wait until the task is complete or error.
// There is an invariant by bulkload engine: if a task metadata is persisted, the task
// is guaranteed to be eventually marked as complete or error.
															#line 12165 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobMonitorTaskActor>
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActorState {
															#line 12172 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActorState(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& jobRange,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobId(jobId),
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobRange(jobRange),
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   taskRange(taskRange),
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1660 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1661 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   bulkLoadTask(),
															#line 1662 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   beginTime(now())
															#line 12193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobMonitorTaskActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1663 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadParallelismLimitor.canStart());
															#line 1664 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.incrementTaskCounter();
															#line 12210 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				StrictFuture<Optional<BulkLoadTaskState>> __when_expr_0 = bulkLoadJobFindTask(self, taskRange, jobId, jobRange, self->ddId);
															#line 1667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12216 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 1;
															#line 1667 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 12221 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1712 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 12250 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~BulkLoadJobMonitorTaskActorState();
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1697 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 12263 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1698 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 12267 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1700 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarn, "DDBulkLoadJobExecutorTaskMonitorError", self->ddId) .errorUnsuppressed(e) .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1705 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1706 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (errorOut.canBeSet())
															#line 12275 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1707 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				errorOut.sendError(e);
															#line 12279 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
															#line 1669 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 12295 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1671 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1672 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 12301 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1674 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1675 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1682 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12313 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1684 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1685 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12325 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 12330 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
															#line 1669 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!bulkLoadTask_.present())
															#line 12339 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1671 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 1672 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobMonitorTaskActorState(); static_cast<BulkLoadJobMonitorTaskActor*>(this)->destroy(); return 0; }
															#line 12345 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobMonitorTaskActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobMonitorTaskActorState();
			static_cast<BulkLoadJobMonitorTaskActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1674 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		bulkLoadTask = bulkLoadTask_.get();
															#line 1675 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Task found") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("TaskRange", bulkLoadTask.getRange()) .detail("Duration", now() - beginTime);
															#line 1682 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (g_network->isSimulated() && SERVER_KNOBS->BULKLOAD_SIM_FAILURE_INJECTION && deterministicRandom()->random01() < 0.1)
															#line 12357 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1684 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobExecutorInjectDDRestart", self->ddId).detail("Context", "Monitor");
															#line 1685 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(movekeys_conflict(), loopDepth);
															#line 12363 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bulkLoadJobWaitUntilTaskCompleteOrError(self, jobId, bulkLoadTask);
															#line 1689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12369 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 2;
															#line 1689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*>(static_cast<BulkLoadJobMonitorTaskActor*>(this)));
															#line 12374 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> const& bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(bulkLoadTask_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadTaskState> && bulkLoadTask_,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(bulkLoadTask_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Optional<BulkLoadTaskState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1690 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1695 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 12460 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1690 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(bulkLoadPerfEventSev(), "DDBulkLoadJobExecutorTask", self->ddId) .detail("Phase", "Found task complete") .detail("JobID", jobId) .detail("TaskID", bulkLoadTask.getTaskId()) .detail("Duration", now() - beginTime);
															#line 1695 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadParallelismLimitor.decrementTaskCounter();
															#line 12471 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobMonitorTaskActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobMonitorTaskActor*>(this)->ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange taskRange;
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1660 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1661 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState bulkLoadTask;
															#line 1662 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 12580 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via bulkLoadJobMonitorTask()
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobMonitorTaskActor final : public Actor<Void>, public ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >, public ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >, public FastAllocated<BulkLoadJobMonitorTaskActor>, public BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor> {
															#line 12585 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator new;
	using FastAllocated<BulkLoadJobMonitorTaskActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11207423198098799616UL, 4490477742821538560UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >;
friend struct ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >;
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobMonitorTaskActor(Reference<DataDistributor> const& self,UID const& jobId,KeyRange const& jobRange,KeyRange const& taskRange,Promise<Void> const& errorOut) 
															#line 12603 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   BulkLoadJobMonitorTaskActorState<BulkLoadJobMonitorTaskActor>(self, jobId, jobRange, taskRange, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3763012194873441280UL, 18076434267272298240UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobMonitorTaskActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bulkLoadJobMonitorTask");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bulkLoadJobMonitorTask", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 0, Optional<BulkLoadTaskState> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BulkLoadJobMonitorTaskActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> bulkLoadJobMonitorTask( Reference<DataDistributor> const& self, UID const& jobId, KeyRange const& jobRange, KeyRange const& taskRange, Promise<Void> const& errorOut ) {
															#line 1655 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new BulkLoadJobMonitorTaskActor(self, jobId, jobRange, taskRange, errorOut));
															#line 12637 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1714 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 12642 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class PersistBulkLoadJobTaskCountActor>
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActorState {
															#line 12649 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActorState(Reference<DataDistributor> const& self) 
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1716 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1717 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 12660 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	~PersistBulkLoadJobTaskCountActorState() 
	{
		fdb_probe_actor_destroy("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1718 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1719 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1720 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = jobState.getJobId();
															#line 1721 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = jobState.getJobRange();
															#line 1722 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(jobState.getTaskCount().present());
															#line 1723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			taskCount = jobState.getTaskCount().get();
															#line 1724 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			currentJobState = BulkLoadJobState();
															#line 1725 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 12689 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1754 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 12712 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistBulkLoadJobTaskCountActorState();
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1727 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(currentJobState, getBulkLoadJob(&tr, jobId, jobRange));
															#line 1727 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12734 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 1;
															#line 1727 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12739 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1751 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1751 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 4;
															#line 1751 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12781 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1728 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 12796 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1729 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 12800 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1730 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 12806 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1738 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 12810 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1740 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1741 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1741 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12822 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1741 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12827 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1728 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (currentJobState.getTaskCount().present())
															#line 12836 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1729 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (currentJobState.getTaskCount().get() != taskCount)
															#line 12840 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1730 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindTaskCountMismatch", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("InputTaskCount", taskCount) .detail("CurrentJobID", currentJobState.getJobId()) .detail("CurrentJobRange", currentJobState.getJobRange());
															#line 1736 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 12846 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1738 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistBulkLoadJobTaskCountActorState(); static_cast<PersistBulkLoadJobTaskCountActor*>(this)->destroy(); return 0; }
															#line 12850 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<PersistBulkLoadJobTaskCountActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PersistBulkLoadJobTaskCountActorState();
			static_cast<PersistBulkLoadJobTaskCountActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1740 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setTaskCount(taskCount);
															#line 1741 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(&tr, bulkLoadJobPrefix, jobRange, bulkLoadJobValue(currentJobState));
															#line 1741 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12862 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 2;
															#line 1741 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12867 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1742 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1742 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12953 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1742 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12958 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1742 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 1742 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12969 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 3;
															#line 1742 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*>(static_cast<PersistBulkLoadJobTaskCountActor*>(this)));
															#line 12974 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1743 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1744 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 13060 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1743 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 1744 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerPersistTaskCountToJobMetadata", self->ddId) .detail("JobID", jobId) .detail("JobRange", jobRange) .detail("CommitVersion", commitVersion) .detail("TaskCount", taskCount);
															#line 13071 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state > 0) static_cast<PersistBulkLoadJobTaskCountActor*>(this)->actor_wait_state = 0;
		static_cast<PersistBulkLoadJobTaskCountActor*>(this)->ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1716 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1717 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1719 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1720 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1721 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1723 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	uint64_t taskCount;
															#line 1724 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 13254 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via persistBulkLoadJobTaskCount()
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class PersistBulkLoadJobTaskCountActor final : public Actor<Void>, public ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >, public ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >, public FastAllocated<PersistBulkLoadJobTaskCountActor>, public PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor> {
															#line 13259 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator new;
	using FastAllocated<PersistBulkLoadJobTaskCountActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12058533146494468096UL, 1883443237883798784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >;
friend struct ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >;
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	PersistBulkLoadJobTaskCountActor(Reference<DataDistributor> const& self) 
															#line 13279 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   PersistBulkLoadJobTaskCountActorState<PersistBulkLoadJobTaskCountActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10651365252717723648UL, 4800485421425282816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistBulkLoadJobTaskCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistBulkLoadJobTaskCount");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistBulkLoadJobTaskCount", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PersistBulkLoadJobTaskCountActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> persistBulkLoadJobTaskCount( Reference<DataDistributor> const& self ) {
															#line 1715 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new PersistBulkLoadJobTaskCountActor(self));
															#line 13315 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1756 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Remove the bulkload job metadata from the range map metadata.
// Add the job metadata to the history metadata with the error message.
															#line 13322 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class MoveErrorBulkLoadJobToHistoryActor>
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActorState {
															#line 13329 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActorState(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorMessage(errorMessage),
															#line 1760 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1761 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx),
															#line 1762 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   currentJobState()
															#line 13344 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	~MoveErrorBulkLoadJobToHistoryActorState() 
	{
		fdb_probe_actor_destroy("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1763 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1764 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobId = self->bulkLoadJobManager.get().jobState.getJobId();
															#line 1765 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = self->bulkLoadJobManager.get().jobState.getJobRange();
															#line 1766 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 13365 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1782 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveErrorBulkLoadJobToHistoryActorState(); static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->destroy(); return 0; }
															#line 13388 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveErrorBulkLoadJobToHistoryActorState();
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1768 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 1768 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13410 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 1;
															#line 1768 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13415 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1779 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_6 = tr.onError(e);
															#line 1779 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13452 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_6.get(), loopDepth); };
			static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 7;
															#line 1779 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13457 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId, jobRange));
															#line 1769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13474 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13479 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(currentJobState, getBulkLoadJob(&tr, jobId, jobRange));
															#line 1769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13490 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 2;
															#line 1769 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13495 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13581 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13586 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = krmSetRange( &tr, bulkLoadJobPrefix, currentJobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 1770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13597 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 3;
															#line 1770 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13602 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1772 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1773 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13692 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13697 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1772 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setErrorPhase(errorMessage);
															#line 1773 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		currentJobState.setEndTime(now());
															#line 1774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = addBulkLoadJobToHistory(&tr, currentJobState);
															#line 1774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13712 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 4;
															#line 1774 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13717 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = releaseExclusiveReadLockOnRange(&tr, jobRange, rangeLockNameForBulkLoad);
															#line 1775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13803 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13808 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = releaseExclusiveReadLockOnRange(&tr, jobRange, rangeLockNameForBulkLoad);
															#line 1775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13819 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 5;
															#line 1775 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13824 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = tr.commit();
															#line 1776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13910 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_5.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 6;
															#line 1776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13915 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = tr.commit();
															#line 1776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13926 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_5.get(), loopDepth); };
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 6;
															#line 1776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*>(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)));
															#line 13931 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state > 0) static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->actor_wait_state = 0;
		static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string errorMessage;
															#line 1760 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1761 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1762 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState currentJobState;
															#line 1764 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	UID jobId;
															#line 1765 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 14199 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via moveErrorBulkLoadJobToHistory()
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class MoveErrorBulkLoadJobToHistoryActor final : public Actor<Void>, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >, public ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >, public FastAllocated<MoveErrorBulkLoadJobToHistoryActor>, public MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor> {
															#line 14204 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator new;
	using FastAllocated<MoveErrorBulkLoadJobToHistoryActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7234497663512538368UL, 4642326606929645312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >;
friend struct ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >;
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	MoveErrorBulkLoadJobToHistoryActor(Reference<DataDistributor> const& self,std::string const& errorMessage) 
															#line 14227 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   MoveErrorBulkLoadJobToHistoryActorState<MoveErrorBulkLoadJobToHistoryActor>(self, errorMessage),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10076677940119608320UL, 17212560479686694656UL);
		ActorExecutionContextHelper __helper(static_cast<MoveErrorBulkLoadJobToHistoryActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("moveErrorBulkLoadJobToHistory");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("moveErrorBulkLoadJobToHistory", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< MoveErrorBulkLoadJobToHistoryActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> moveErrorBulkLoadJobToHistory( Reference<DataDistributor> const& self, std::string const& errorMessage ) {
															#line 1759 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new MoveErrorBulkLoadJobToHistoryActor(self, errorMessage));
															#line 14266 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1784 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

// Download the job manifest file from the remoteJobManifestFilePath to the localJobManifestFilePath.
// Build the bulkload manifest range map based on the localJobManifestFilePath file content.
// For any failure, we mark the job metadata as error and move the metadata to the history.
															#line 14274 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FetchBulkLoadTaskManifestEntryMapActor>
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActorState {
															#line 14281 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActorState(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   jobTransportMethod(jobTransportMethod),
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   localJobManifestFilePath(localJobManifestFilePath),
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   remoteJobManifestFilePath(remoteJobManifestFilePath)
															#line 14294 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	~FetchBulkLoadTaskManifestEntryMapActorState() 
	{
		fdb_probe_actor_destroy("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1792 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid() && self->bulkLoadJobManager.get().manifestEntryMap->empty());
															#line 1794 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginTime = now();
															#line 1795 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobRange = self->bulkLoadJobManager.get().jobState.getJobRange();
															#line 1796 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestMapRange = KeyRange();
															#line 14315 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			try {
															#line 1798 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!fileExists(abspath(localJobManifestFilePath)))
															#line 14319 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1799 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					TraceEvent(SevDebug, "DDBulkLoadJobManagerDownloadJobManifest", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath);
															#line 1803 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					StrictFuture<Void> __when_expr_0 = downloadBulkLoadJobManifestFile( jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath, self->ddId);
															#line 1803 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 14327 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 1;
															#line 1803 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 14332 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont2(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1860 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchBulkLoadTaskManifestEntryMapActorState(); static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->destroy(); return 0; }
															#line 14366 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchBulkLoadTaskManifestEntryMapActorState();
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1836 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 14379 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1837 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 14383 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			err = e;
															#line 1840 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindUnretryableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1846 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string errorMessage = "Failed to build job-manifest map with error code " + std::to_string(err.code()) + ". The remote file path is " + remoteJobManifestFilePath + ". The local file path is " + localJobManifestFilePath + ". The transport method is " + convertBulkLoadTransportMethodToString(jobTransportMethod) + ".";
															#line 1851 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = moveErrorBulkLoadJobToHistory(self, errorMessage);
															#line 1851 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14395 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_2.get(), loopDepth); };
			static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 3;
															#line 1851 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 14400 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1814 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(manifestMapRange, getBulkLoadJobFileManifestEntryFromJobManifestFile( localJobManifestFilePath, jobRange, self->ddId, self->bulkLoadJobManager.get().manifestEntryMap));
															#line 1814 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 14417 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 2;
															#line 1814 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*>(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)));
															#line 14422 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 14431 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1805 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestDownloaded", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 14440 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1822 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadJobManager.get().jobState.getJobRange() != manifestMapRange)
															#line 14524 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1823 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerManifestMapRangeMismatch", self->ddId) .detail("JobRange", jobRange) .detail("ManifestMapRange", manifestMapRange);
															#line 1826 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_dataset_not_cover_required_range(), loopDepth);
															#line 14530 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1828 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1829 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 14536 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1822 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadJobManager.get().jobState.getJobRange() != manifestMapRange)
															#line 14545 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1823 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerManifestMapRangeMismatch", self->ddId) .detail("JobRange", jobRange) .detail("ManifestMapRange", manifestMapRange);
															#line 1826 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			return a_body1Catch2(bulkload_dataset_not_cover_required_range(), loopDepth);
															#line 14551 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1828 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadJobManager.get().jobState.setTaskCount(self->bulkLoadJobManager.get().manifestEntryMap->size());
															#line 1829 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerManifestMapBuilt", self->ddId) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("TaskCount", self->bulkLoadJobManager.get().manifestEntryMap->size()) .detail("Duration", now() - beginTime);
															#line 14557 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch2cont1(Void const& _,int loopDepth) 
	{
															#line 1852 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretryableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 14656 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2cont1(Void && _,int loopDepth) 
	{
															#line 1852 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerPersistUnretryableError", self->ddId) .errorUnsuppressed(err) .detail("JobTransportMethod", jobTransportMethod) .detail("LocalJobManifestFilePath", localJobManifestFilePath) .detail("RemoteJobManifestFilePath", remoteJobManifestFilePath) .detail("Duration", now() - beginTime);
															#line 1858 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 14666 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state > 0) static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTransportMethod jobTransportMethod;
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string localJobManifestFilePath;
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::string remoteJobManifestFilePath;
															#line 1794 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	double beginTime;
															#line 1795 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobRange;
															#line 1796 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange manifestMapRange;
															#line 1839 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Error err;
															#line 14761 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via fetchBulkLoadTaskManifestEntryMap()
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FetchBulkLoadTaskManifestEntryMapActor final : public Actor<Void>, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >, public ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >, public FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>, public FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor> {
															#line 14766 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator new;
	using FastAllocated<FetchBulkLoadTaskManifestEntryMapActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4482655466642462208UL, 2694729033411194368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >;
friend struct ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >;
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FetchBulkLoadTaskManifestEntryMapActor(Reference<DataDistributor> const& self,BulkLoadTransportMethod const& jobTransportMethod,std::string const& localJobManifestFilePath,std::string const& remoteJobManifestFilePath) 
															#line 14785 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FetchBulkLoadTaskManifestEntryMapActorState<FetchBulkLoadTaskManifestEntryMapActor>(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7119443845485172480UL, 13876208275314761984UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBulkLoadTaskManifestEntryMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchBulkLoadTaskManifestEntryMap");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchBulkLoadTaskManifestEntryMap", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchBulkLoadTaskManifestEntryMapActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> fetchBulkLoadTaskManifestEntryMap( Reference<DataDistributor> const& self, BulkLoadTransportMethod const& jobTransportMethod, std::string const& localJobManifestFilePath, std::string const& remoteJobManifestFilePath ) {
															#line 1788 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FetchBulkLoadTaskManifestEntryMapActor(self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath));
															#line 14820 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1862 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 14825 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class ScheduleBulkLoadJobActor>
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActorState {
															#line 14832 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActorState(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   errorOut(errorOut),
															#line 1864 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   manifestEntry()
															#line 14843 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~ScheduleBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1865 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1866 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1867 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			actors = std::vector<Future<Void>>();
															#line 1869 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			cx = self->txnProcessor->context();
															#line 1870 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr = Transaction(cx);
															#line 1874 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 14870 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1963 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(actors);
															#line 1963 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14895 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 1963 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 14900 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1876 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, jobState.getJobRange().end));
															#line 1876 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14919 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 1876 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 14924 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1960 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 1960 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14961 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 1960 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 14966 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1878 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 1879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 14983 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 1876 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		res = __res;
															#line 14992 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1954 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (beginKey >= jobState.getJobRange().end)
															#line 15071 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1956 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager.get().allTaskSubmitted = true;
															#line 15075 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < res.size() - 1))
															#line 15093 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1883 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!res[i].value.empty())
															#line 15099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 1884 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			task = decodeBulkLoadTaskState(res[i].value);
															#line 1885 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (task.isValid())
															#line 15105 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1886 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.getJobId() != self->bulkLoadJobManager.get().jobState.getJobId())
															#line 15109 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1887 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					return a_body1loopBody1Catch1(bulkload_task_outdated(), std::max(0, loopDepth - 1));
															#line 15113 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				}
															#line 1893 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(self->bulkLoadJobManager.get().manifestEntryMap->find(task.getRange().begin) != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1895 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (task.onAnyPhase( { BulkLoadPhase::Complete, BulkLoadPhase::Acknowledged, BulkLoadPhase::Error }))
															#line 15119 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				{
															#line 1897 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					ASSERT(task.getRange().end == res[i + 1].key);
															#line 1898 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					beginKey = task.getRange().end;
															#line 15125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					return a_body1loopBody1cont2continue1(loopDepth); // continue
				}
				else
				{
															#line 1901 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
					if (task.onAnyPhase({ BulkLoadPhase::Submitted, BulkLoadPhase::Triggered, BulkLoadPhase::Running }))
															#line 15132 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
					{
															#line 1906 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						if (self->bulkLoadJobManager.get().allTaskSubmitted)
															#line 15136 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						{
															#line 1908 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
							;
															#line 15140 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
							loopDepth = a_body1loopBody1cont2loopBody1loopHead1(loopDepth);
						}
						else
						{
							loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);
						}
					}
					else
					{
															#line 1922 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
						UNREACHABLE();
															#line 15152 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
						loopDepth = a_body1loopBody1cont2loopBody1cont7(loopDepth);
					}
				}
			}
			else
			{
				loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2continue1(int loopDepth) 
	{
															#line 1879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 15186 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1930 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i].key);
															#line 1931 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 15197 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont7(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont8(int loopDepth) 
	{
															#line 1918 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(task.getRange().end == res[i + 1].key);
															#line 1919 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = task.getRange().end;
															#line 15226 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont2continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont9(int loopDepth) 
	{
															#line 1915 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back(bulkLoadJobMonitorTask( self, task.getJobId(), jobState.getJobRange(), task.getRange(), errorOut));
															#line 15235 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1(int loopDepth) 
	{
															#line 1909 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (self->bulkLoadParallelismLimitor.canStart())
															#line 15251 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1912 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->bulkLoadParallelismLimitor.waitUntilCounterChanged();
															#line 1912 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 15259 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 1912 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15264 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2loopBody1cont13(int loopDepth) 
	{
															#line 1952 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(beginKey == res[i + 1].key || beginKey > jobState.getJobRange().end);
															#line 1879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 15375 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 1931 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < res[i + 1].key))
															#line 15391 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont2loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1932 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		std::vector<BulkLoadJobFileManifestEntry> manifestEntries;
															#line 1933 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(;manifestEntries.size() < SERVER_KNOBS->MANIFEST_COUNT_MAX_PER_BULKLOAD_TASK && beginKey < res[i + 1].key;) {
															#line 1935 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			auto it = self->bulkLoadJobManager.get().manifestEntryMap->find(beginKey);
															#line 1936 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(it != self->bulkLoadJobManager.get().manifestEntryMap->end());
															#line 1937 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntry = it->second;
															#line 1938 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			manifestEntries.push_back(manifestEntry);
															#line 1939 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = manifestEntry.getEndKey();
															#line 15409 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 1941 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!manifestEntries.empty());
															#line 1942 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		actors.push_back(bulkLoadJobNewTask(self, jobState.getJobId(), jobState.getJobRoot(), jobState.getJobRange(), jobState.getTransportMethod(), self->bulkLoadJobManager.get().manifestLocalTempFolder, manifestEntries, errorOut));
															#line 1950 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(SERVER_KNOBS->DD_BULKLOAD_TASK_SUBMISSION_INTERVAL_SEC);
															#line 1950 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 15419 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 1950 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*>(static_cast<ScheduleBulkLoadJobActor*>(this)));
															#line 15424 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont13(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1964 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 15633 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1964 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<ScheduleBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ScheduleBulkLoadJobActorState(); static_cast<ScheduleBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 15645 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<ScheduleBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ScheduleBulkLoadJobActorState();
		static_cast<ScheduleBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<ScheduleBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<ScheduleBulkLoadJobActor*>(this)->ActorCallback< ScheduleBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Promise<Void> errorOut;
															#line 1864 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobFileManifestEntry manifestEntry;
															#line 1866 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1867 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1868 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	std::vector<Future<Void>> actors;
															#line 1869 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1870 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1876 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult res;
															#line 1878 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 1884 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState task;
															#line 15750 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via scheduleBulkLoadJob()
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class ScheduleBulkLoadJobActor final : public Actor<Void>, public ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >, public ActorCallback< ScheduleBulkLoadJobActor, 1, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 2, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 3, Void >, public ActorCallback< ScheduleBulkLoadJobActor, 4, Void >, public FastAllocated<ScheduleBulkLoadJobActor>, public ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor> {
															#line 15755 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<ScheduleBulkLoadJobActor>::operator new;
	using FastAllocated<ScheduleBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9491747527610224128UL, 8514777735603801856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 1, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 2, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 3, Void >;
friend struct ActorCallback< ScheduleBulkLoadJobActor, 4, Void >;
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	ScheduleBulkLoadJobActor(Reference<DataDistributor> const& self,Promise<Void> const& errorOut) 
															#line 15776 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   ScheduleBulkLoadJobActorState<ScheduleBulkLoadJobActor>(self, errorOut),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4684338048225246976UL, 14035353769110992896UL);
		ActorExecutionContextHelper __helper(static_cast<ScheduleBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("scheduleBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("scheduleBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ScheduleBulkLoadJobActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> scheduleBulkLoadJob( Reference<DataDistributor> const& self, Promise<Void> const& errorOut ) {
															#line 1863 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new ScheduleBulkLoadJobActor(self, errorOut));
															#line 15813 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 1966 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 15818 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class CheckBulkLoadTaskCompleteOrErrorActor>
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActorState {
															#line 15825 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActorState(Reference<DataDistributor> const& self) 
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 1968 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 1969 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 15836 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	~CheckBulkLoadTaskCompleteOrErrorActorState() 
	{
		fdb_probe_actor_destroy("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1970 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 1971 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 1972 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 1973 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 1974 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 1975 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = KeyRange();
															#line 1976 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 1977 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 15865 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2022 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(true); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 15888 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(true);
		this->~CheckBulkLoadTaskCompleteOrErrorActorState();
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1977 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 15907 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 1979 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 1980 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			rangeToRead = Standalone(KeyRangeRef(beginKey, endKey));
															#line 1981 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobState.getJobId(), jobState.getJobRange());
															#line 1981 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15920 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 1;
															#line 1981 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 15925 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2019 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 2019 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15962 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 3;
															#line 2019 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 15967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState const& checkJobState,int loopDepth) 
	{
															#line 1982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15984 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 15989 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState && checkJobState,int loopDepth) 
	{
															#line 1982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, rangeToRead));
															#line 1982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16000 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 2;
															#line 1982 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*>(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)));
															#line 16005 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& checkJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(checkJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && checkJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(checkJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1983 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1984 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1985 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1986 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16095 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1988 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1992 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16107 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1993 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1998 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16113 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2000 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2001 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16121 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 2008 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16126 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2009 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 2014 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16132 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2017 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 16141 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1983 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		for(int i = 0;i < bulkLoadTaskResult.size() - 1;i++) {
															#line 1984 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 1985 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 1986 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!existTask.isValid())
															#line 16156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1988 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1992 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.getJobId() != jobState.getJobId())
															#line 16168 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 1993 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerFindIdMisMatch", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 1998 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16174 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2000 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase == BulkLoadPhase::Error)
															#line 16178 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2001 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerFindErrorTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 16182 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				continue;
			}
															#line 2008 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete)
															#line 16187 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2009 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevDebug, "DDBulkLoadJobManageFindRunningTask", self->ddId) .detail("TaskJobID", existTask.getJobId()) .detail("TaskID", existTask.getTaskId()) .detail("TaskRange", existTask.getRange()) .detail("InputJobID", jobState.getJobId());
															#line 2014 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				if (!static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CheckBulkLoadTaskCompleteOrErrorActorState(); static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->destroy(); return 0; }
															#line 16193 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
				new (&static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->SAV< bool >::value()) bool(false);
				this->~CheckBulkLoadTaskCompleteOrErrorActorState();
				static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 2017 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = bulkLoadTaskResult.back().key;
															#line 16202 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state > 0) static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 1968 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 1969 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 1971 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 1972 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 1973 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 1974 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState existTask;
															#line 1975 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange rangeToRead;
															#line 1976 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult bulkLoadTaskResult;
															#line 16400 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via checkBulkLoadTaskCompleteOrError()
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class CheckBulkLoadTaskCompleteOrErrorActor final : public Actor<bool>, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >, public ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >, public FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>, public CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor> {
															#line 16405 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator new;
	using FastAllocated<CheckBulkLoadTaskCompleteOrErrorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6191242435325497600UL, 8331583742875600896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >;
friend struct ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >;
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	CheckBulkLoadTaskCompleteOrErrorActor(Reference<DataDistributor> const& self) 
															#line 16424 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<bool>(),
		   CheckBulkLoadTaskCompleteOrErrorActorState<CheckBulkLoadTaskCompleteOrErrorActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13835388531052272896UL, 2426800946393579264UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBulkLoadTaskCompleteOrErrorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBulkLoadTaskCompleteOrError");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBulkLoadTaskCompleteOrError", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 0, BulkLoadJobState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckBulkLoadTaskCompleteOrErrorActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<bool> checkBulkLoadTaskCompleteOrError( Reference<DataDistributor> const& self ) {
															#line 1967 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<bool>(new CheckBulkLoadTaskCompleteOrErrorActor(self));
															#line 16459 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 2024 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 16464 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via finalizeBulkLoadJob()
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class FinalizeBulkLoadJobActor>
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FinalizeBulkLoadJobActorState {
															#line 16471 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FinalizeBulkLoadJobActorState(Reference<DataDistributor> const& self) 
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2026 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context()),
															#line 2027 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   tr(cx)
															#line 16482 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	~FinalizeBulkLoadJobActorState() 
	{
		fdb_probe_actor_destroy("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2028 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 2029 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState = self->bulkLoadJobManager.get().jobState;
															#line 2030 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			beginKey = jobState.getJobRange().begin;
															#line 2031 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			endKey = jobState.getJobRange().end;
															#line 2032 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = Optional<Key>();
															#line 2033 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			existTask = BulkLoadTaskState();
															#line 2034 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult = RangeResult();
															#line 2035 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobCompleteRange = KeyRange();
															#line 2036 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = false;
															#line 2037 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			i = 0;
															#line 2038 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			allFinish = false;
															#line 2039 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			;
															#line 16519 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2122 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!static_cast<FinalizeBulkLoadJobActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FinalizeBulkLoadJobActorState(); static_cast<FinalizeBulkLoadJobActor*>(this)->destroy(); return 0; }
															#line 16542 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		new (&static_cast<FinalizeBulkLoadJobActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FinalizeBulkLoadJobActorState();
		static_cast<FinalizeBulkLoadJobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2039 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(beginKey < endKey))
															#line 16561 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 2041 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			tr.reset();
															#line 2042 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			bulkLoadTaskResult.clear();
															#line 2043 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<BulkLoadJobState> __when_expr_0 = getBulkLoadJob(&tr, jobState.getJobId(), jobState.getJobRange());
															#line 2043 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16574 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 1;
															#line 2043 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16579 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2118 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			ASSERT(e.code() != error_code_range_unlock_reject);
															#line 2119 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_9 = tr.onError(e);
															#line 2119 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16618 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_9.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 10;
															#line 2119 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16623 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
															#line 2044 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2045 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2045 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16642 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2045 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16647 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(BulkLoadJobState && currentJobState,int loopDepth) 
	{
															#line 2044 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		hasError = hasError && (currentJobState.getPhase() == BulkLoadJobPhase::Error);
															#line 2045 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_1 = store(bulkLoadTaskResult, krmGetRanges(&tr, bulkLoadTaskPrefix, KeyRangeRef(beginKey, endKey)));
															#line 2045 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16660 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 2;
															#line 2045 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16665 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState const& currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(currentJobState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(BulkLoadJobState && currentJobState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(currentJobState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,BulkLoadJobState && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 2046 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2047 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 16751 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 2046 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i = 0;
															#line 2047 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 16762 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 2085 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (hasError)
															#line 16846 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2086 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setErrorPhase("BulkLoadTask Error.");
															#line 16850 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
		else
		{
															#line 2088 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setCompletePhase();
															#line 16856 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		}
															#line 2090 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(lastKey.present());
															#line 2091 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobCompleteRange = KeyRangeRef(jobState.getJobRange().begin, lastKey.get());
															#line 2092 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_3 = checkMoveKeysLock(&tr, self->context->lock, self->context->ddEnabledState.get());
															#line 2092 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16866 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 4;
															#line 2092 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16871 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1(int loopDepth) 
	{
															#line 2047 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!(i < bulkLoadTaskResult.size() - 1))
															#line 16887 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
			return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2048 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(!bulkLoadTaskResult[i].value.empty());
															#line 2049 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		existTask = decodeBulkLoadTaskState(bulkLoadTaskResult[i].value);
															#line 2050 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!existTask.isValid())
															#line 16897 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2051 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			lastKey = bulkLoadTaskResult[i + 1].key;
															#line 16901 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			return a_body1loopBody1cont4continue1(loopDepth); // continue
		}
															#line 2054 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getRange() == KeyRangeRef(bulkLoadTaskResult[i].key, bulkLoadTaskResult[i + 1].key));
															#line 2059 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(existTask.getJobId() == jobState.getJobId());
															#line 2060 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (existTask.phase == BulkLoadPhase::Error)
															#line 16910 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2061 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevWarnAlways, "DDBulkLoadJobManagerStopClearMetadata", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2068 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			hasError = true;
															#line 16916 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);
		}
		else
		{
															#line 2070 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (existTask.phase != BulkLoadPhase::Complete && existTask.phase != BulkLoadPhase::Acknowledged)
															#line 16923 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			{
															#line 2071 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				TraceEvent(SevError, "DDBulkLoadJobManagerWrongTaskPhase", self->ddId) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2077 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
				ASSERT(false);
															#line 16929 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			}
															#line 2080 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_2 = setBulkLoadFinalizeTransaction(&tr, existTask.getRange(), existTask.getTaskId());
															#line 2080 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16935 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 3;
															#line 2080 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 16940 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4continue1(int loopDepth) 
	{
															#line 2047 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		i++;
															#line 16963 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont1(int loopDepth) 
	{
															#line 2082 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		lastKey = bulkLoadTaskResult[i + 1].key;
															#line 16972 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		return a_body1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2093 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2094 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17070 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2096 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17078 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17083 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_7 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17092 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_7.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 8;
															#line 2101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2093 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		allFinish = jobCompleteRange == jobState.getJobRange();
															#line 2094 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (allFinish)
															#line 17109 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2096 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			jobState.setEndTime(now());
															#line 2097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_4 = krmSetRange(&tr, bulkLoadJobPrefix, jobState.getJobRange(), bulkLoadJobValue(BulkLoadJobState()));
															#line 2097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17117 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 5;
															#line 2097 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17122 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_7 = krmSetRange(&tr, bulkLoadJobPrefix, jobCompleteRange, bulkLoadJobValue(jobState));
															#line 2101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17131 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1loopBody1cont6when2(__when_expr_7.get(), loopDepth); };
			static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 8;
															#line 2101 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17136 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 2103 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_8 = tr.commit();
															#line 2103 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17223 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_8.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 9;
															#line 2103 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17228 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 2098 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2098 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17239 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2098 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17244 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 2098 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_5 = addBulkLoadJobToHistory(&tr, jobState);
															#line 2098 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17255 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_5.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 6;
															#line 2098 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17260 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10cont1(Void const& _,int loopDepth) 
	{
															#line 2099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = releaseExclusiveReadLockOnRange(&tr, jobState.getJobRange(), rangeLockNameForBulkLoad);
															#line 2099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17346 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont10cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17351 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1(Void && _,int loopDepth) 
	{
															#line 2099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_6 = releaseExclusiveReadLockOnRange(&tr, jobState.getJobRange(), rangeLockNameForBulkLoad);
															#line 2099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17362 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont10cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 7;
															#line 2099 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*>(static_cast<FinalizeBulkLoadJobActor*>(this)));
															#line 17367 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont10cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
															#line 2104 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 2105 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerFinalizeRange", self->ddId) .detail("JobCompleteRange", jobCompleteRange) .detail("CommitVersion", commitVersion) .detail("AllFinish", allFinish) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2114 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = lastKey.get();
															#line 17629 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont12cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
															#line 2104 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		Version commitVersion = tr.getCommittedVersion();
															#line 2105 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		TraceEvent(SevInfo, "DDBulkLoadJobManagerFinalizeRange", self->ddId) .detail("JobCompleteRange", jobCompleteRange) .detail("CommitVersion", commitVersion) .detail("AllFinish", allFinish) .detail("JobID", jobState.getJobId()) .detail("JobRange", jobState.getJobRange()) .detail("ExistTaskJobID", existTask.getJobId()) .detail("ExistTaskID", existTask.getTaskId()) .detail("ExistTaskRange", existTask.getRange());
															#line 2114 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		beginKey = lastKey.get();
															#line 17642 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1loopBody1cont12cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1loopBody1cont12cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state > 0) static_cast<FinalizeBulkLoadJobActor*>(this)->actor_wait_state = 0;
		static_cast<FinalizeBulkLoadJobActor*>(this)->ActorCallback< FinalizeBulkLoadJobActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), 9);

	}
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Reference<DataDistributor> self;
															#line 2026 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Database cx;
															#line 2027 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Transaction tr;
															#line 2029 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobState jobState;
															#line 2030 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key beginKey;
															#line 2031 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Key endKey;
															#line 2032 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	Optional<Key> lastKey;
															#line 2033 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadTaskState existTask;
															#line 2034 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	RangeResult bulkLoadTaskResult;
															#line 2035 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	KeyRange jobCompleteRange;
															#line 2036 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool hasError;
															#line 2037 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	int i;
															#line 2038 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	bool allFinish;
															#line 17848 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
};
// This generated class is to be used only via finalizeBulkLoadJob()
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class FinalizeBulkLoadJobActor final : public Actor<Void>, public ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >, public ActorCallback< FinalizeBulkLoadJobActor, 1, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 2, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 3, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 4, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 5, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 6, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 7, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 8, Void >, public ActorCallback< FinalizeBulkLoadJobActor, 9, Void >, public FastAllocated<FinalizeBulkLoadJobActor>, public FinalizeBulkLoadJobActorState<FinalizeBulkLoadJobActor> {
															#line 17853 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
	using FastAllocated<FinalizeBulkLoadJobActor>::operator new;
	using FastAllocated<FinalizeBulkLoadJobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14459677264439105536UL, 14460170800430941696UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 1, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 2, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 3, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 4, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 5, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 6, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 7, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 8, Void >;
friend struct ActorCallback< FinalizeBulkLoadJobActor, 9, Void >;
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	FinalizeBulkLoadJobActor(Reference<DataDistributor> const& self) 
															#line 17879 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		 : Actor<Void>(),
		   FinalizeBulkLoadJobActorState<FinalizeBulkLoadJobActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13125497442541706752UL, 11431860539656438528UL);
		ActorExecutionContextHelper __helper(static_cast<FinalizeBulkLoadJobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("finalizeBulkLoadJob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("finalizeBulkLoadJob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 0, BulkLoadJobState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< FinalizeBulkLoadJobActor, 9, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
[[nodiscard]] Future<Void> finalizeBulkLoadJob( Reference<DataDistributor> const& self ) {
															#line 2025 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	return Future<Void>(new FinalizeBulkLoadJobActor(self));
															#line 17921 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
}

#line 2124 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"

															#line 17926 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
namespace {
// This generated class is to be used only via bulkLoadJobManager()
															#line 2125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
template <class BulkLoadJobManagerActor>
															#line 2125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
class BulkLoadJobManagerActorState {
															#line 17933 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
public:
															#line 2125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
	BulkLoadJobManagerActorState(Reference<DataDistributor> const& self) 
															#line 2125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
															#line 2125 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		 : self(self),
															#line 2127 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		   cx(self->txnProcessor->context())
															#line 17942 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
	{
		fdb_probe_actor_create("bulkLoadJobManager", reinterpret_cast<unsigned long>(this));

	}
	~BulkLoadJobManagerActorState() 
	{
		fdb_probe_actor_destroy("bulkLoadJobManager", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Optional<BulkLoadJobState>> __when_expr_0 = getRunningBulkLoadJob(cx);
															#line 2128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17959 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 1;
															#line 2128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 17964 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BulkLoadJobManagerActorState();
		static_cast<BulkLoadJobManagerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2129 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!job.present())
															#line 17987 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2130 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(bulkLoadVerboseEventSev(), "DDBulkLoadJobManagerNoJobExist", self->ddId);
															#line 2131 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager.reset();
															#line 2132 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadTaskCollection->removeBulkLoadJobRange();
															#line 2133 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (!static_cast<BulkLoadJobManagerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BulkLoadJobManagerActorState(); static_cast<BulkLoadJobManagerActor*>(this)->destroy(); return 0; }
															#line 17997 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			new (&static_cast<BulkLoadJobManagerActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BulkLoadJobManagerActorState();
			static_cast<BulkLoadJobManagerActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2135 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobId = job.get().getJobId();
															#line 2136 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobRange = job.get().getJobRange();
															#line 2137 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobRoot = job.get().getJobRoot();
															#line 2138 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		jobTransportMethod = job.get().getTransportMethod();
															#line 2139 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		self->bulkLoadTaskCollection->setBulkLoadJobRange(jobRange);
															#line 2142 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (!self->bulkLoadJobManager.present() || self->bulkLoadJobManager.get().jobState.getJobId() != jobId)
															#line 18015 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		{
															#line 2143 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadJobManagerBuild", self->ddId) .detail("OldJobID", self->bulkLoadJobManager.present() ? self->bulkLoadJobManager.get().jobState.getJobId().toString() : "No old job") .detail("NewJobId", jobId) .detail("NewJobRange", jobRange) .detail("NewJobRoot", jobRoot) .detail("NewJobTransportMethod", jobTransportMethod);
															#line 2152 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string localFolder = getBulkLoadJobRoot(self->bulkLoadFolder, jobId);
															#line 2153 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string manifestLocalTempFolder = abspath(joinPath(localFolder, "manifest-temp"));
															#line 2154 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			resetFileFolder(manifestLocalTempFolder);
															#line 2155 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string remoteFolder = getBulkLoadJobRoot(jobRoot, jobId);
															#line 2156 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string jobManifestFileName = getBulkLoadJobManifestFileName();
															#line 2157 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string localJobManifestFilePath = joinPath(localFolder, jobManifestFileName);
															#line 2158 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			std::string remoteJobManifestFilePath = appendToPath(remoteFolder, jobManifestFileName);
															#line 2159 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			self->bulkLoadJobManager = DDBulkLoadJobManager(job.get(), manifestLocalTempFolder);
															#line 2160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			StrictFuture<Void> __when_expr_1 = fetchBulkLoadTaskManifestEntryMap( self, jobTransportMethod, localJobManifestFilePath, remoteJobManifestFilePath);
															#line 2160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18039 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 2;
															#line 2160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 1, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18044 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 2164 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
			TraceEvent(SevInfo, "DDBulkLoadJobManagerExist", self->ddId) .detail("JobID", self->bulkLoadJobManager.get().jobState.getJobId()) .detail("JobRange", self->bulkLoadJobManager.get().jobState.getJobRange());
															#line 18051 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadJobState> const& __job,int loopDepth) 
	{
															#line 2128 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		job = __job;
															#line 18061 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<BulkLoadJobState> && __job,int loopDepth) 
	{
		job = std::move(__job);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Optional<BulkLoadJobState> const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Optional<BulkLoadJobState> && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 0, Optional<BulkLoadJobState> >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2170 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		ASSERT(self->bulkLoadJobManager.present() && self->bulkLoadJobManager.get().isValid());
															#line 2176 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		errorOut = Promise<Void>();
															#line 2177 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		;
															#line 18144 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = persistBulkLoadJobTaskCount(self);
															#line 2162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18155 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 3;
															#line 2162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 2, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18160 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		StrictFuture<Void> __when_expr_2 = persistBulkLoadJobTaskCount(self);
															#line 2162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18171 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 3;
															#line 2162 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/fdbserver/DataDistribution.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BulkLoadJobManagerActor, 2, Void >*>(static_cast<BulkLoadJobManagerActor*>(this)));
															#line 18176 "/codebuild/output/src17799134/src/github.com/apple/foundationdb/build_output/fdbserver/DataDistribution.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state > 0) static_cast<BulkLoadJobManagerActor*>(this)->actor_wait_state = 0;
		static_cast<BulkLoadJobManagerActor*>(this)->ActorCallback< BulkLoadJobManagerActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BulkLoadJobManagerActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BulkLoadJobManagerActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bulkLoadJobManager", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BulkLoadJobManagerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_act