#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
/*
 * AuthzSecurity.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cstring>
#include <unordered_set>

#include "fdbclient/BlobWorkerInterface.h"
#include "flow/Arena.h"
#include "flow/IRandom.h"
#include "flow/Trace.h"
#include "flow/WipedString.h"
#include "flow/serialize.h"
#include "fdbrpc/simulator.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbserver/LogSystemConfig.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbserver/TesterInterface.actor.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/workloads/workloads.actor.h"

#include "flow/actorcompiler.h" // This must be the last #include.

FDB_BOOLEAN_PARAM(PositiveTestcase);

bool checkGranuleLocations(ErrorOr<GetBlobGranuleLocationsReply> rep, TenantInfo tenant) {
	if (rep.isError()) {
		if (rep.getError().code() == error_code_permission_denied) {
			TraceEvent(SevError, "AuthzSecurityError")
			    .detail("Case", "CrossTenantGranuleLocationCheckDisallowed")
			    .log();
		}
		return false;
	} else {
		ASSERT(!rep.get().results.empty());
		for (auto const& [range, bwIface] : rep.get().results) {
			if (!range.begin.startsWith(tenant.prefix.get())) {
				TraceEvent(SevError, "AuthzSecurityBlobGranuleRangeLeak")
				    .detail("TenantId", tenant.tenantId)
				    .detail("LeakingRangeBegin", range.begin.printable());
			}
			if (!range.end.startsWith(tenant.prefix.get())) {
				TraceEvent(SevError, "AuthzSecurityBlobGranuleRangeLeak")
				    .detail("TenantId", tenant.tenantId)
				    .detail("LeakingRangeEnd", range.end.printable());
			}
		}
		return true;
	}
}

struct AuthzSecurityWorkload : TestWorkload {
	static constexpr auto NAME = "AuthzSecurity";
	int actorCount;
	double testDuration, transactionsPerSecond;

	std::vector<Future<Void>> clients;
	Arena arena;
	Reference<Tenant> tenant;
	Reference<Tenant> anotherTenant;
	TenantName tenantName;
	TenantName anotherTenantName;
	WipedString signedToken;
	WipedString signedTokenAnotherTenant;
	Standalone<StringRef> tLogConfigKey;
	PerfIntCounter crossTenantGetPositive, crossTenantGetNegative, crossTenantCommitPositive, crossTenantCommitNegative,
	    publicNonTenantRequestPositive, tLogReadNegative, keyLocationLeakNegative, bgLocationLeakNegative,
	    crossTenantBGLocPositive, crossTenantBGLocNegative, crossTenantBGReqPositive, crossTenantBGReqNegative,
	    crossTenantBGReadPositive, crossTenantBGReadNegative, crossTenantGetGranulesPositive,
	    crossTenantGetGranulesNegative, blobbifyNegative, unblobbifyNegative, listBlobNegative, verifyBlobNegative,
	    flushBlobNegative, purgeBlobNegative;
	std::vector<std::function<Future<Void>(Database cx)>> testFunctions;
	bool checkBlobGranules, checkBlobManagement;

	AuthzSecurityWorkload(WorkloadContext const& wcx)
	  : TestWorkload(wcx), crossTenantGetPositive("CrossTenantGetPositive"),
	    crossTenantGetNegative("CrossTenantGetNegative"), crossTenantCommitPositive("CrossTenantCommitPositive"),
	    crossTenantCommitNegative("CrossTenantCommitNegative"),
	    publicNonTenantRequestPositive("PublicNonTenantRequestPositive"), tLogReadNegative("TLogReadNegative"),
	    keyLocationLeakNegative("KeyLocationLeakNegative"), bgLocationLeakNegative("BGLocationLeakNegative"),
	    crossTenantBGLocPositive("CrossTenantBGLocPositive"), crossTenantBGLocNegative("CrossTenantBGLocNegative"),
	    crossTenantBGReqPositive("CrossTenantBGReqPositive"), crossTenantBGReqNegative("CrossTenantBGReqNegative"),
	    crossTenantBGReadPositive("CrossTenantBGReadPositive"), crossTenantBGReadNegative("CrossTenantBGReadNegative"),
	    crossTenantGetGranulesPositive("CrossTenantGetGranulesPositive"),
	    crossTenantGetGranulesNegative("CrossTenantGetGranulesNegative"), blobbifyNegative("BlobbifyNegative"),
	    unblobbifyNegative("UnblobbifyNegative"), listBlobNegative("ListBlobNegative"),
	    verifyBlobNegative("VerifyBlobNegative"), flushBlobNegative("FlushBlobNegative"),
	    purgeBlobNegative("PurgeBlobNegative") {
		testDuration = getOption(options, "testDuration"_sr, 10.0);
		transactionsPerSecond = getOption(options, "transactionsPerSecond"_sr, 500.0) / clientCount;
		actorCount = getOption(options, "actorsPerClient"_sr, transactionsPerSecond / 5);
		tenantName = getOption(options, "tenantA"_sr, "authzSecurityTestTenant"_sr);
		anotherTenantName = getOption(options, "tenantB"_sr, "authzSecurityTestTenant"_sr);
		tLogConfigKey = getOption(options, "tLogConfigKey"_sr, "TLogInterface"_sr);
		checkBlobGranules = getOption(options, "checkBlobGranules"_sr, false);
		checkBlobManagement =
		    checkBlobGranules && getOption(options, "checkBlobManagement"_sr, sharedRandomNumber % 2 == 0);
		sharedRandomNumber /= 2;

		ASSERT(g_network->isSimulated());
		// make it comfortably longer than the timeout of the workload
		testFunctions.push_back(
		    [this](Database cx) { return testCrossTenantGetDisallowed(this, cx, PositiveTestcase::True); });
		testFunctions.push_back(
		    [this](Database cx) { return testCrossTenantGetDisallowed(this, cx, PositiveTestcase::False); });
		testFunctions.push_back(
		    [this](Database cx) { return testCrossTenantCommitDisallowed(this, cx, PositiveTestcase::True); });
		testFunctions.push_back(
		    [this](Database cx) { return testCrossTenantCommitDisallowed(this, cx, PositiveTestcase::False); });
		testFunctions.push_back(
		    [this](Database cx) { return testPublicNonTenantRequestsAllowedWithoutTokens(this, cx); });
		testFunctions.push_back([this](Database cx) { return testTLogReadDisallowed(this, cx); });
		testFunctions.push_back([this](Database cx) { return testKeyLocationLeakDisallowed(this, cx); });

		if (checkBlobGranules) {
			testFunctions.push_back([this](Database cx) { return testBlobGranuleLocationLeakDisallowed(this, cx); });
			testFunctions.push_back(
			    [this](Database cx) { return testCrossTenantBGLocDisallowed(this, cx, PositiveTestcase::True); });
			testFunctions.push_back(
			    [this](Database cx) { return testCrossTenantBGLocDisallowed(this, cx, PositiveTestcase::False); });
			testFunctions.push_back(
			    [this](Database cx) { return testCrossTenantBGRequestDisallowed(this, cx, PositiveTestcase::True); });
			testFunctions.push_back(
			    [this](Database cx) { return testCrossTenantBGRequestDisallowed(this, cx, PositiveTestcase::False); });
			testFunctions.push_back(
			    [this](Database cx) { return testCrossTenantBGReadDisallowed(this, cx, PositiveTestcase::True); });
			testFunctions.push_back(
			    [this](Database cx) { return testCrossTenantBGReadDisallowed(this, cx, PositiveTestcase::False); });
			testFunctions.push_back(
			    [this](Database cx) { return testCrossTenantGetGranulesDisallowed(this, cx, PositiveTestcase::True); });
			testFunctions.push_back([this](Database cx) {
				return testCrossTenantGetGranulesDisallowed(this, cx, PositiveTestcase::False);
			});
		}
		if (checkBlobManagement) {
			testFunctions.push_back([this](Database cx) { return testBlobbifyDisallowed(this, cx); });
			testFunctions.push_back([this](Database cx) { return testUnblobbifyDisallowed(this, cx); });
			testFunctions.push_back([this](Database cx) { return testListBlobDisallowed(this, cx); });
			testFunctions.push_back([this](Database cx) { return testVerifyBlobDisallowed(this, cx); });
			testFunctions.push_back([this](Database cx) { return testFlushBlobDisallowed(this, cx); });
			testFunctions.push_back([this](Database cx) { return testPurgeBlobDisallowed(this, cx); });
		}
	}

	Future<Void> setup(Database const& cx) override {
		tenant = makeReference<Tenant>(cx, tenantName);
		anotherTenant = makeReference<Tenant>(cx, anotherTenantName);
		return tenant->ready() && anotherTenant->ready();
	}

	Future<Void> start(Database const& cx) override {
		signedToken = g_simulator->makeToken(
		    tenant->id(), uint64_t(std::lround(getCheckTimeout())) + uint64_t(std::lround(testDuration)) + 100);
		signedTokenAnotherTenant = g_simulator->makeToken(
		    anotherTenant->id(), uint64_t(std::lround(getCheckTimeout())) + uint64_t(std::lround(testDuration)) + 100);
		for (int c = 0; c < actorCount; c++)
			clients.push_back(timeout(runTestClient(this, cx->clone()), testDuration, Void()));
		return waitForAll(clients);
	}

	Future<bool> check(Database const& cx) override {
		int errors = 0;
		for (int c = 0; c < clients.size(); c++)
			errors += clients[c].isError();
		if (errors)
			TraceEvent(SevError, "TestFailure").detail("Reason", "There were client errors.");
		clients.clear();
		bool success = errors == 0 && crossTenantGetPositive.getValue() > 0 && crossTenantGetNegative.getValue() > 0 &&
		               crossTenantCommitPositive.getValue() > 0 && crossTenantCommitNegative.getValue() > 0 &&
		               publicNonTenantRequestPositive.getValue() > 0 && tLogReadNegative.getValue() > 0 &&
		               keyLocationLeakNegative.getValue() > 0;
		if (checkBlobGranules) {
			success &= bgLocationLeakNegative.getValue() > 0 && crossTenantBGLocPositive.getValue() > 0 &&
			           crossTenantBGLocNegative.getValue() > 0 && crossTenantBGReqPositive.getValue() > 0 &&
			           crossTenantBGReqNegative.getValue() > 0 && crossTenantBGReadPositive.getValue() > 0 &&
			           crossTenantBGReadNegative.getValue() > 0 && crossTenantGetGranulesPositive.getValue() > 0 &&
			           crossTenantGetGranulesNegative.getValue() > 0;
		}
		if (checkBlobManagement) {
			success &= blobbifyNegative.getValue() > 0 && unblobbifyNegative.getValue() > 0 &&
			           listBlobNegative.getValue() > 0 && verifyBlobNegative.getValue() > 0 &&
			           flushBlobNegative.getValue() > 0 && purgeBlobNegative.getValue() > 0;
		}
		return success;
	}

	void getMetrics(std::vector<PerfMetric>& m) override {
		m.push_back(crossTenantGetPositive.getMetric());
		m.push_back(crossTenantGetNegative.getMetric());
		m.push_back(crossTenantCommitPositive.getMetric());
		m.push_back(crossTenantCommitNegative.getMetric());
		m.push_back(publicNonTenantRequestPositive.getMetric());
		m.push_back(tLogReadNegative.getMetric());
		m.push_back(keyLocationLeakNegative.getMetric());
		if (checkBlobGranules) {
			m.push_back(bgLocationLeakNegative.getMetric());
			m.push_back(crossTenantBGLocPositive.getMetric());
			m.push_back(crossTenantBGLocNegative.getMetric());
			m.push_back(crossTenantBGReqPositive.getMetric());
			m.push_back(crossTenantBGReqNegative.getMetric());
			m.push_back(crossTenantBGReadPositive.getMetric());
			m.push_back(crossTenantBGReadNegative.getMetric());
			m.push_back(crossTenantGetGranulesPositive.getMetric());
			m.push_back(crossTenantGetGranulesNegative.getMetric());
		}
		if (checkBlobManagement) {
			m.push_back(blobbifyNegative.getMetric());
			m.push_back(unblobbifyNegative.getMetric());
			m.push_back(listBlobNegative.getMetric());
			m.push_back(verifyBlobNegative.getMetric());
			m.push_back(flushBlobNegative.getMetric());
			m.push_back(purgeBlobNegative.getMetric());
		}
	}

	void setAuthToken(Transaction& tr, StringRef token) {
		tr.setOption(FDBTransactionOptions::AUTHORIZATION_TOKEN, token);
	}

																#line 239 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via setAndCommitKeyValueAndGetVersion()
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class SetAndCommitKeyValueAndGetVersionActor>
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class SetAndCommitKeyValueAndGetVersionActorState {
															#line 245 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	SetAndCommitKeyValueAndGetVersionActorState(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,WipedString const& token,StringRef const& key,StringRef const& value) 
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tenant(tenant),
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   token(token),
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(key),
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(value),
															#line 243 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tr(cx, tenant)
															#line 264 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this));

	}
	~SetAndCommitKeyValueAndGetVersionActorState() 
	{
		fdb_probe_actor_destroy("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 244 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			;
															#line 279 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SetAndCommitKeyValueAndGetVersionActorState();
		static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 246 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			self->setAuthToken(tr, token);
															#line 247 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			tr.set(key, value);
															#line 248 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr.commit();
															#line 248 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 316 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->actor_wait_state = 1;
															#line 248 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 0, Void >*>(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)));
															#line 321 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 251 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 251 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 345 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->actor_wait_state = 2;
															#line 251 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 1, Void >*>(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)));
															#line 350 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 249 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->SAV<Version>::futures) { (void)(tr.getCommittedVersion()); this->~SetAndCommitKeyValueAndGetVersionActorState(); static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->destroy(); return 0; }
															#line 365 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->SAV< Version >::value()) Version(tr.getCommittedVersion());
		this->~SetAndCommitKeyValueAndGetVersionActorState();
		static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 249 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->SAV<Version>::futures) { (void)(tr.getCommittedVersion()); this->~SetAndCommitKeyValueAndGetVersionActorState(); static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->destroy(); return 0; }
															#line 377 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->SAV< Version >::value()) Version(tr.getCommittedVersion());
		this->~SetAndCommitKeyValueAndGetVersionActorState();
		static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->actor_wait_state > 0) static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->actor_wait_state = 0;
		static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->actor_wait_state > 0) static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->actor_wait_state = 0;
		static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Reference<Tenant> tenant;
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	WipedString token;
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	StringRef key;
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	StringRef value;
															#line 243 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Transaction tr;
															#line 561 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via setAndCommitKeyValueAndGetVersion()
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class SetAndCommitKeyValueAndGetVersionActor final : public Actor<Version>, public ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 0, Void >, public ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 1, Void >, public FastAllocated<SetAndCommitKeyValueAndGetVersionActor>, public SetAndCommitKeyValueAndGetVersionActorState<SetAndCommitKeyValueAndGetVersionActor> {
															#line 566 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<SetAndCommitKeyValueAndGetVersionActor>::operator new;
	using FastAllocated<SetAndCommitKeyValueAndGetVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6730501009889597440UL, 7708152145847342848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 0, Void >;
friend struct ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 1, Void >;
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	SetAndCommitKeyValueAndGetVersionActor(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,WipedString const& token,StringRef const& key,StringRef const& value) 
															#line 584 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Version>(),
		   SetAndCommitKeyValueAndGetVersionActorState<SetAndCommitKeyValueAndGetVersionActor>(self, cx, tenant, token, key, value),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12538110609011233792UL, 13155861937347160832UL);
		ActorExecutionContextHelper __helper(static_cast<SetAndCommitKeyValueAndGetVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("setAndCommitKeyValueAndGetVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("setAndCommitKeyValueAndGetVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SetAndCommitKeyValueAndGetVersionActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Version> setAndCommitKeyValueAndGetVersion( AuthzSecurityWorkload* const& self, Database const& cx, Reference<Tenant> const& tenant, WipedString const& token, StringRef const& key, StringRef const& value ) {
															#line 237 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Version>(new SetAndCommitKeyValueAndGetVersionActor(self, cx, tenant, token, key, value));
															#line 617 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 255 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 622 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via refreshAndGetCachedLocation()
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class RefreshAndGetCachedLocationActor>
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class RefreshAndGetCachedLocationActorState {
															#line 628 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	RefreshAndGetCachedLocationActorState(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,WipedString const& token,StringRef const& key) 
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tenant(tenant),
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   token(token),
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(key),
															#line 261 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tr(cx, tenant)
															#line 645 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this));

	}
	~RefreshAndGetCachedLocationActorState() 
	{
		fdb_probe_actor_destroy("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 262 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			;
															#line 660 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RefreshAndGetCachedLocationActorState();
		static_cast<RefreshAndGetCachedLocationActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 265 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			self->setAuthToken(tr, token);
															#line 266 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(tr.get(key));
															#line 266 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 695 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state = 1;
															#line 266 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RefreshAndGetCachedLocationActor, 0, Void >*>(static_cast<RefreshAndGetCachedLocationActor*>(this)));
															#line 700 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 274 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 274 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 724 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state = 3;
															#line 274 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RefreshAndGetCachedLocationActor, 2, Void >*>(static_cast<RefreshAndGetCachedLocationActor*>(this)));
															#line 729 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 267 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		auto loc = cx->getCachedLocation(tr.trState->getTenantInfo(), key);
															#line 268 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (loc.present())
															#line 746 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 269 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!static_cast<RefreshAndGetCachedLocationActor*>(this)->SAV<KeyRangeLocationInfo>::futures) { (void)(loc.get()); this->~RefreshAndGetCachedLocationActorState(); static_cast<RefreshAndGetCachedLocationActor*>(this)->destroy(); return 0; }
															#line 750 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			new (&static_cast<RefreshAndGetCachedLocationActor*>(this)->SAV< KeyRangeLocationInfo >::value()) KeyRangeLocationInfo(loc.get());
			this->~RefreshAndGetCachedLocationActorState();
			static_cast<RefreshAndGetCachedLocationActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 271 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 271 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 762 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state = 2;
															#line 271 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >*>(static_cast<RefreshAndGetCachedLocationActor*>(this)));
															#line 767 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 267 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		auto loc = cx->getCachedLocation(tr.trState->getTenantInfo(), key);
															#line 268 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (loc.present())
															#line 779 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 269 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!static_cast<RefreshAndGetCachedLocationActor*>(this)->SAV<KeyRangeLocationInfo>::futures) { (void)(loc.get()); this->~RefreshAndGetCachedLocationActorState(); static_cast<RefreshAndGetCachedLocationActor*>(this)->destroy(); return 0; }
															#line 783 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			new (&static_cast<RefreshAndGetCachedLocationActor*>(this)->SAV< KeyRangeLocationInfo >::value()) KeyRangeLocationInfo(loc.get());
			this->~RefreshAndGetCachedLocationActorState();
			static_cast<RefreshAndGetCachedLocationActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 271 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 271 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 795 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state = 2;
															#line 271 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >*>(static_cast<RefreshAndGetCachedLocationActor*>(this)));
															#line 800 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state > 0) static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshAndGetCachedLocationActor*>(this)->ActorCallback< RefreshAndGetCachedLocationActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshAndGetCachedLocationActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RefreshAndGetCachedLocationActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RefreshAndGetCachedLocationActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state > 0) static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshAndGetCachedLocationActor*>(this)->ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state > 0) static_cast<RefreshAndGetCachedLocationActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshAndGetCachedLocationActor*>(this)->ActorCallback< RefreshAndGetCachedLocationActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshAndGetCachedLocationActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RefreshAndGetCachedLocationActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RefreshAndGetCachedLocationActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Reference<Tenant> tenant;
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	WipedString token;
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	StringRef key;
															#line 261 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Transaction tr;
															#line 1086 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via refreshAndGetCachedLocation()
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class RefreshAndGetCachedLocationActor final : public Actor<KeyRangeLocationInfo>, public ActorCallback< RefreshAndGetCachedLocationActor, 0, Void >, public ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >, public ActorCallback< RefreshAndGetCachedLocationActor, 2, Void >, public FastAllocated<RefreshAndGetCachedLocationActor>, public RefreshAndGetCachedLocationActorState<RefreshAndGetCachedLocationActor> {
															#line 1091 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<RefreshAndGetCachedLocationActor>::operator new;
	using FastAllocated<RefreshAndGetCachedLocationActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15569709376710214144UL, 2267545653368561920UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<KeyRangeLocationInfo>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RefreshAndGetCachedLocationActor, 0, Void >;
friend struct ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >;
friend struct ActorCallback< RefreshAndGetCachedLocationActor, 2, Void >;
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	RefreshAndGetCachedLocationActor(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,WipedString const& token,StringRef const& key) 
															#line 1110 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<KeyRangeLocationInfo>(),
		   RefreshAndGetCachedLocationActorState<RefreshAndGetCachedLocationActor>(self, cx, tenant, token, key),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9503796058120611840UL, 8903243772475254528UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshAndGetCachedLocationActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("refreshAndGetCachedLocation");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("refreshAndGetCachedLocation", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RefreshAndGetCachedLocationActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RefreshAndGetCachedLocationActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RefreshAndGetCachedLocationActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<KeyRangeLocationInfo> refreshAndGetCachedLocation( AuthzSecurityWorkload* const& self, Database const& cx, Reference<Tenant> const& tenant, WipedString const& token, StringRef const& key ) {
															#line 256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<KeyRangeLocationInfo>(new RefreshAndGetCachedLocationActor(self, cx, tenant, token, key));
															#line 1144 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 278 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

	Standalone<StringRef> randomString() {
		auto const minLength = tLogConfigKey.size() + 1;
		return StringRef(
		    deterministicRandom()->randomAlphaNumeric(deterministicRandom()->randomInt(minLength, minLength + 100)));
	}

																#line 1155 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via tryGetValue()
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TryGetValueActor>
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryGetValueActorState {
															#line 1161 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryGetValueActorState(AuthzSecurityWorkload* const& self,Reference<Tenant> const& tenant,Version const& committedVersion,Standalone<StringRef> const& key,Optional<Standalone<StringRef>> const& expectedValue,WipedString const& token,Database const& cx,KeyRangeLocationInfo const& loc) 
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tenant(tenant),
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   committedVersion(committedVersion),
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(key),
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   expectedValue(expectedValue),
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   token(token),
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   loc(loc)
															#line 1182 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("tryGetValue", reinterpret_cast<unsigned long>(this));

	}
	~TryGetValueActorState() 
	{
		fdb_probe_actor_destroy("tryGetValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 293 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			;
															#line 1197 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TryGetValueActorState();
		static_cast<TryGetValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 319 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryGetValueActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryGetValueActorState(); static_cast<TryGetValueActor*>(this)->destroy(); return 0; }
															#line 1220 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryGetValueActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryGetValueActorState();
		static_cast<TryGetValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 294 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		GetValueRequest req;
															#line 295 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.key = key;
															#line 296 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.version = committedVersion;
															#line 297 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.tenantInfo.tenantId = tenant->id();
															#line 298 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.tenantInfo.token = token;
															#line 1247 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		try {
															#line 300 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<GetValueReply> __when_expr_0 = loadBalance(loc.locations->locations(), &StorageServerInterface::getValue, req, TaskPriority::DefaultPromiseEndpoint, AtMostOnce::False, nullptr);
															#line 300 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TryGetValueActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1253 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TryGetValueActor*>(this)->actor_wait_state = 1;
															#line 300 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TryGetValueActor, 0, GetValueReply >*>(static_cast<TryGetValueActor*>(this)));
															#line 1258 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 315 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			CODE_PROBE(e.code() == error_code_permission_denied, "Cross tenant get meets permission_denied");
															#line 316 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!static_cast<TryGetValueActor*>(this)->SAV<Optional<Error>>::futures) { (void)(e); this->~TryGetValueActorState(); static_cast<TryGetValueActor*>(this)->destroy(); return 0; }
															#line 1289 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			new (&static_cast<TryGetValueActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(e);
			this->~TryGetValueActorState();
			static_cast<TryGetValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(GetValueReply const& reply,int loopDepth) 
	{
															#line 307 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (reply.value != expectedValue)
															#line 1307 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 308 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			TraceEvent(SevError, "AuthzSecurityUnmatchedValue") .detail("Expected", expectedValue) .detail("Actual", reply.value) .log();
															#line 1311 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2(GetValueReply && reply,int loopDepth) 
	{
															#line 307 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (reply.value != expectedValue)
															#line 1321 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 308 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			TraceEvent(SevError, "AuthzSecurityUnmatchedValue") .detail("Expected", expectedValue) .detail("Actual", reply.value) .log();
															#line 1325 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(GetValueReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(reply, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(GetValueReply && reply,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TryGetValueActor*>(this)->actor_wait_state > 0) static_cast<TryGetValueActor*>(this)->actor_wait_state = 0;
		static_cast<TryGetValueActor*>(this)->ActorCallback< TryGetValueActor, 0, GetValueReply >::remove();

	}
	void a_callback_fire(ActorCallback< TryGetValueActor, 0, GetValueReply >*,GetValueReply const& value) 
	{
		fdb_probe_actor_enter("tryGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TryGetValueActor, 0, GetValueReply >*,GetValueReply && value) 
	{
		fdb_probe_actor_enter("tryGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TryGetValueActor, 0, GetValueReply >*,Error err) 
	{
		fdb_probe_actor_enter("tryGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Reference<Tenant> tenant;
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Standalone<StringRef> key;
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Optional<Standalone<StringRef>> expectedValue;
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	WipedString token;
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	KeyRangeLocationInfo loc;
															#line 1422 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via tryGetValue()
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryGetValueActor final : public Actor<Optional<Error>>, public ActorCallback< TryGetValueActor, 0, GetValueReply >, public FastAllocated<TryGetValueActor>, public TryGetValueActorState<TryGetValueActor> {
															#line 1427 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TryGetValueActor>::operator new;
	using FastAllocated<TryGetValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17303771001712039168UL, 8838172268332070144UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Error>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TryGetValueActor, 0, GetValueReply >;
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryGetValueActor(AuthzSecurityWorkload* const& self,Reference<Tenant> const& tenant,Version const& committedVersion,Standalone<StringRef> const& key,Optional<Standalone<StringRef>> const& expectedValue,WipedString const& token,Database const& cx,KeyRangeLocationInfo const& loc) 
															#line 1444 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Optional<Error>>(),
		   TryGetValueActorState<TryGetValueActor>(self, tenant, committedVersion, key, expectedValue, token, cx, loc),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tryGetValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13211099537176528384UL, 11894003926691458304UL);
		ActorExecutionContextHelper __helper(static_cast<TryGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tryGetValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tryGetValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TryGetValueActor, 0, GetValueReply >*)0, actor_cancelled()); break;
		}

	}
};
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Optional<Error>> tryGetValue( AuthzSecurityWorkload* const& self, Reference<Tenant> const& tenant, Version const& committedVersion, Standalone<StringRef> const& key, Optional<Standalone<StringRef>> const& expectedValue, WipedString const& token, Database const& cx, KeyRangeLocationInfo const& loc ) {
															#line 285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Optional<Error>>(new TryGetValueActor(self, tenant, committedVersion, key, expectedValue, token, cx, loc));
															#line 1476 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 321 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 1481 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testCrossTenantGetDisallowed()
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestCrossTenantGetDisallowedActor>
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantGetDisallowedActorState {
															#line 1487 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantGetDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   positive(positive),
															#line 325 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 326 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString())
															#line 1502 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestCrossTenantGetDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 327 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Version> __when_expr_0 = setAndCommitKeyValueAndGetVersion(self, cx, self->tenant, self->signedToken, key, value);
															#line 327 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1519 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 1;
															#line 327 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantGetDisallowedActor, 0, Version >*>(static_cast<TestCrossTenantGetDisallowedActor*>(this)));
															#line 1524 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestCrossTenantGetDisallowedActorState();
		static_cast<TestCrossTenantGetDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 330 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<KeyRangeLocationInfo> __when_expr_1 = refreshAndGetCachedLocation(self, cx, self->tenant, self->signedToken, key);
															#line 330 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1549 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 2;
															#line 330 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantGetDisallowedActor, 1, KeyRangeLocationInfo >*>(static_cast<TestCrossTenantGetDisallowedActor*>(this)));
															#line 1554 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Version const& __committedVersion,int loopDepth) 
	{
															#line 327 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		committedVersion = __committedVersion;
															#line 1563 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Version && __committedVersion,int loopDepth) 
	{
		committedVersion = std::move(__committedVersion);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantGetDisallowedActor*>(this)->ActorCallback< TestCrossTenantGetDisallowedActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetDisallowedActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetDisallowedActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestCrossTenantGetDisallowedActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(KeyRangeLocationInfo const& loc,int loopDepth) 
	{
															#line 331 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (positive)
															#line 1642 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 334 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Optional<Error>> __when_expr_2 = tryGetValue(self, self->tenant, committedVersion, key, value, self->signedToken , cx, loc);
															#line 334 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1648 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 3;
															#line 334 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >*>(static_cast<TestCrossTenantGetDisallowedActor*>(this)));
															#line 1653 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 351 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Optional<Error>> __when_expr_3 = tryGetValue(self, self->tenant, committedVersion, key, value, self->signedTokenAnotherTenant , cx, loc);
															#line 351 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1662 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when2(__when_expr_3.get(), loopDepth); };
			static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 4;
															#line 351 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >*>(static_cast<TestCrossTenantGetDisallowedActor*>(this)));
															#line 1667 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont2(KeyRangeLocationInfo && loc,int loopDepth) 
	{
															#line 331 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (positive)
															#line 1677 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 334 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Optional<Error>> __when_expr_2 = tryGetValue(self, self->tenant, committedVersion, key, value, self->signedToken , cx, loc);
															#line 334 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1683 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 3;
															#line 334 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >*>(static_cast<TestCrossTenantGetDisallowedActor*>(this)));
															#line 1688 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 351 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Optional<Error>> __when_expr_3 = tryGetValue(self, self->tenant, committedVersion, key, value, self->signedTokenAnotherTenant , cx, loc);
															#line 351 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1697 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when2(__when_expr_3.get(), loopDepth); };
			static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 4;
															#line 351 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >*>(static_cast<TestCrossTenantGetDisallowedActor*>(this)));
															#line 1702 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1when1(KeyRangeLocationInfo const& loc,int loopDepth) 
	{
		loopDepth = a_body1cont2(loc, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(KeyRangeLocationInfo && loc,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(loc), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantGetDisallowedActor*>(this)->ActorCallback< TestCrossTenantGetDisallowedActor, 1, KeyRangeLocationInfo >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetDisallowedActor, 1, KeyRangeLocationInfo >*,KeyRangeLocationInfo const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetDisallowedActor, 1, KeyRangeLocationInfo >*,KeyRangeLocationInfo && value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestCrossTenantGetDisallowedActor, 1, KeyRangeLocationInfo >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 371 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantGetDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantGetDisallowedActorState(); static_cast<TestCrossTenantGetDisallowedActor*>(this)->destroy(); return 0; }
															#line 1787 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantGetDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantGetDisallowedActorState();
		static_cast<TestCrossTenantGetDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Optional<Error> const& outcome,int loopDepth) 
	{
															#line 342 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!outcome.present())
															#line 1799 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 343 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			++self->crossTenantGetPositive;
															#line 1803 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		else
		{
															#line 344 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (outcome.get().code() == error_code_permission_denied)
															#line 1809 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 345 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "CrossTenantGetDisallowed") .detail("Subcase", "Positive") .log();
															#line 1813 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Optional<Error> && outcome,int loopDepth) 
	{
															#line 342 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!outcome.present())
															#line 1824 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 343 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			++self->crossTenantGetPositive;
															#line 1828 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		else
		{
															#line 344 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (outcome.get().code() == error_code_permission_denied)
															#line 1834 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 345 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "CrossTenantGetDisallowed") .detail("Subcase", "Positive") .log();
															#line 1838 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Error> const& outcome,int loopDepth) 
	{
		loopDepth = a_body1cont4(outcome, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Error> && outcome,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(outcome), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantGetDisallowedActor*>(this)->ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >*,Optional<Error> const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >*,Optional<Error> && value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont9(Optional<Error> const& outcome,int loopDepth) 
	{
															#line 362 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!outcome.present())
															#line 1924 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 363 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "CrossTenantGetDisallowed") .detail("Subcase", "Negative") .log();
															#line 1928 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		else
		{
															#line 367 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (outcome.get().code() == error_code_permission_denied)
															#line 1934 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 368 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				++self->crossTenantGetNegative;
															#line 1938 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Optional<Error> && outcome,int loopDepth) 
	{
															#line 362 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!outcome.present())
															#line 1949 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 363 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "CrossTenantGetDisallowed") .detail("Subcase", "Negative") .log();
															#line 1953 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		else
		{
															#line 367 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (outcome.get().code() == error_code_permission_denied)
															#line 1959 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 368 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				++self->crossTenantGetNegative;
															#line 1963 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Optional<Error> const& outcome,int loopDepth) 
	{
		loopDepth = a_body1cont9(outcome, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Optional<Error> && outcome,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(outcome), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantGetDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantGetDisallowedActor*>(this)->ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >*,Optional<Error> const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >*,Optional<Error> && value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	PositiveTestcase positive;
															#line 325 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 326 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 327 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 2057 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testCrossTenantGetDisallowed()
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantGetDisallowedActor final : public Actor<Void>, public ActorCallback< TestCrossTenantGetDisallowedActor, 0, Version >, public ActorCallback< TestCrossTenantGetDisallowedActor, 1, KeyRangeLocationInfo >, public ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >, public ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >, public FastAllocated<TestCrossTenantGetDisallowedActor>, public TestCrossTenantGetDisallowedActorState<TestCrossTenantGetDisallowedActor> {
															#line 2062 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestCrossTenantGetDisallowedActor>::operator new;
	using FastAllocated<TestCrossTenantGetDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15850877558777029632UL, 2077826687381901568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestCrossTenantGetDisallowedActor, 0, Version >;
friend struct ActorCallback< TestCrossTenantGetDisallowedActor, 1, KeyRangeLocationInfo >;
friend struct ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >;
friend struct ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >;
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantGetDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 2082 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestCrossTenantGetDisallowedActorState<TestCrossTenantGetDisallowedActor>(self, cx, positive),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8888239307471471104UL, 360649738962053632UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testCrossTenantGetDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testCrossTenantGetDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestCrossTenantGetDisallowedActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestCrossTenantGetDisallowedActor, 1, KeyRangeLocationInfo >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TestCrossTenantGetDisallowedActor, 2, Optional<Error> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TestCrossTenantGetDisallowedActor, 3, Optional<Error> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testCrossTenantGetDisallowed( AuthzSecurityWorkload* const& self, Database const& cx, PositiveTestcase const& positive ) {
															#line 322 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestCrossTenantGetDisallowedActor(self, cx, positive));
															#line 2117 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 373 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 2122 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via tryCommit()
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TryCommitActor>
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryCommitActorState {
															#line 2128 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryCommitActorState(AuthzSecurityWorkload* const& self,Reference<Tenant> const& tenant,WipedString const& token,Key const& key,Value const& newValue,Version const& readVersion,Database const& cx) 
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tenant(tenant),
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   token(token),
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(key),
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   newValue(newValue),
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   readVersion(readVersion),
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx)
															#line 2147 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("tryCommit", reinterpret_cast<unsigned long>(this));

	}
	~TryCommitActorState() 
	{
		fdb_probe_actor_destroy("tryCommit", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 381 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			;
															#line 2162 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TryCommitActorState();
		static_cast<TryCommitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 382 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		prefixedKey = key.withPrefix(tenant->prefix());
															#line 383 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		CommitTransactionRequest req;
															#line 384 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.transaction.mutations.push_back(req.arena, MutationRef(MutationRef::SetValue, prefixedKey, newValue));
															#line 385 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.transaction.read_snapshot = readVersion;
															#line 386 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.tenantInfo.token = token;
															#line 387 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.tenantInfo.tenantId = tenant->id();
															#line 2202 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		try {
															#line 389 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<CommitID> __when_expr_0 = basicLoadBalance(cx->getCommitProxies(UseProvisionalProxies::False), &CommitProxyInterface::commit, req, TaskPriority::DefaultPromiseEndpoint, AtMostOnce::False);
															#line 389 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TryCommitActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2208 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TryCommitActor*>(this)->actor_wait_state = 1;
															#line 389 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TryCommitActor, 0, CommitID >*>(static_cast<TryCommitActor*>(this)));
															#line 2213 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 396 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			CODE_PROBE(e.code() == error_code_permission_denied, "Cross tenant commit meets permission_denied");
															#line 397 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!static_cast<TryCommitActor*>(this)->SAV<Optional<Error>>::futures) { (void)(e); this->~TryCommitActorState(); static_cast<TryCommitActor*>(this)->destroy(); return 0; }
															#line 2231 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			new (&static_cast<TryCommitActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(e);
			this->~TryCommitActorState();
			static_cast<TryCommitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(CommitID const& reply,int loopDepth) 
	{
															#line 394 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryCommitActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryCommitActorState(); static_cast<TryCommitActor*>(this)->destroy(); return 0; }
															#line 2249 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryCommitActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryCommitActorState();
		static_cast<TryCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(CommitID && reply,int loopDepth) 
	{
															#line 394 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryCommitActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryCommitActorState(); static_cast<TryCommitActor*>(this)->destroy(); return 0; }
															#line 2261 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryCommitActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryCommitActorState();
		static_cast<TryCommitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(CommitID const& reply,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(reply, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(CommitID && reply,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TryCommitActor*>(this)->actor_wait_state > 0) static_cast<TryCommitActor*>(this)->actor_wait_state = 0;
		static_cast<TryCommitActor*>(this)->ActorCallback< TryCommitActor, 0, CommitID >::remove();

	}
	void a_callback_fire(ActorCallback< TryCommitActor, 0, CommitID >*,CommitID const& value) 
	{
		fdb_probe_actor_enter("tryCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TryCommitActor, 0, CommitID >*,CommitID && value) 
	{
		fdb_probe_actor_enter("tryCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryCommit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TryCommitActor, 0, CommitID >*,Error err) 
	{
		fdb_probe_actor_enter("tryCommit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryCommit", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Reference<Tenant> tenant;
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	WipedString token;
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value newValue;
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version readVersion;
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 382 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key prefixedKey;
															#line 2360 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via tryCommit()
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryCommitActor final : public Actor<Optional<Error>>, public ActorCallback< TryCommitActor, 0, CommitID >, public FastAllocated<TryCommitActor>, public TryCommitActorState<TryCommitActor> {
															#line 2365 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TryCommitActor>::operator new;
	using FastAllocated<TryCommitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2481671307236499712UL, 4552933470011460864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Error>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TryCommitActor, 0, CommitID >;
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryCommitActor(AuthzSecurityWorkload* const& self,Reference<Tenant> const& tenant,WipedString const& token,Key const& key,Value const& newValue,Version const& readVersion,Database const& cx) 
															#line 2382 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Optional<Error>>(),
		   TryCommitActorState<TryCommitActor>(self, tenant, token, key, newValue, readVersion, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tryCommit", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(927202980068370176UL, 11563951767143569664UL);
		ActorExecutionContextHelper __helper(static_cast<TryCommitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tryCommit");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tryCommit", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TryCommitActor, 0, CommitID >*)0, actor_cancelled()); break;
		}

	}
};
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Optional<Error>> tryCommit( AuthzSecurityWorkload* const& self, Reference<Tenant> const& tenant, WipedString const& token, Key const& key, Value const& newValue, Version const& readVersion, Database const& cx ) {
															#line 374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Optional<Error>>(new TryCommitActor(self, tenant, token, key, newValue, readVersion, cx));
															#line 2414 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 401 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 2419 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testCrossTenantCommitDisallowed()
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestCrossTenantCommitDisallowedActor>
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantCommitDisallowedActorState {
															#line 2425 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantCommitDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   positive(positive),
															#line 405 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 406 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString()),
															#line 407 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   newValue(self->randomString())
															#line 2442 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestCrossTenantCommitDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 408 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Version> __when_expr_0 = setAndCommitKeyValueAndGetVersion(self, cx, self->tenant, self->signedToken, key, value);
															#line 408 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2459 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state = 1;
															#line 408 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantCommitDisallowedActor, 0, Version >*>(static_cast<TestCrossTenantCommitDisallowedActor*>(this)));
															#line 2464 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestCrossTenantCommitDisallowedActorState();
		static_cast<TestCrossTenantCommitDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 410 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (positive)
															#line 2487 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 412 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Optional<Error>> __when_expr_1 = tryCommit(self, self->tenant, self->signedToken, key, newValue, committedVersion, cx);
															#line 412 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2493 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state = 2;
															#line 412 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantCommitDisallowedActor, 1, Optional<Error> >*>(static_cast<TestCrossTenantCommitDisallowedActor*>(this)));
															#line 2498 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 423 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Optional<Error>> __when_expr_2 = tryCommit(self, self->tenant, self->signedTokenAnotherTenant, key, newValue, committedVersion, cx);
															#line 423 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2507 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
			static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state = 3;
															#line 423 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantCommitDisallowedActor, 2, Optional<Error> >*>(static_cast<TestCrossTenantCommitDisallowedActor*>(this)));
															#line 2512 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1when1(Version const& __committedVersion,int loopDepth) 
	{
															#line 408 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		committedVersion = __committedVersion;
															#line 2522 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Version && __committedVersion,int loopDepth) 
	{
		committedVersion = std::move(__committedVersion);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantCommitDisallowedActor*>(this)->ActorCallback< TestCrossTenantCommitDisallowedActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantCommitDisallowedActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantCommitDisallowedActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestCrossTenantCommitDisallowedActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 434 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantCommitDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantCommitDisallowedActorState(); static_cast<TestCrossTenantCommitDisallowedActor*>(this)->destroy(); return 0; }
															#line 2601 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantCommitDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantCommitDisallowedActorState();
		static_cast<TestCrossTenantCommitDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<Error> const& outcome,int loopDepth) 
	{
															#line 414 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!outcome.present())
															#line 2613 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 415 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			++self->crossTenantCommitPositive;
															#line 2617 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		else
		{
															#line 416 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (outcome.get().code() == error_code_permission_denied)
															#line 2623 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 417 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "CrossTenantCommitDisallowed") .detail("Subcase", "Positive") .log();
															#line 2627 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Optional<Error> && outcome,int loopDepth) 
	{
															#line 414 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!outcome.present())
															#line 2638 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 415 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			++self->crossTenantCommitPositive;
															#line 2642 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		else
		{
															#line 416 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (outcome.get().code() == error_code_permission_denied)
															#line 2648 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 417 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "CrossTenantCommitDisallowed") .detail("Subcase", "Positive") .log();
															#line 2652 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Error> const& outcome,int loopDepth) 
	{
		loopDepth = a_body1cont3(outcome, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Error> && outcome,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(outcome), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantCommitDisallowedActor*>(this)->ActorCallback< TestCrossTenantCommitDisallowedActor, 1, Optional<Error> >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantCommitDisallowedActor, 1, Optional<Error> >*,Optional<Error> const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantCommitDisallowedActor, 1, Optional<Error> >*,Optional<Error> && value) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestCrossTenantCommitDisallowedActor, 1, Optional<Error> >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont8(Optional<Error> const& outcome,int loopDepth) 
	{
															#line 425 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!outcome.present())
															#line 2738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 426 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "CrossTenantCommitDisallowed") .detail("Subcase", "Negative") .log();
															#line 2742 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		else
		{
															#line 430 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (outcome.get().code() == error_code_permission_denied)
															#line 2748 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 431 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				++self->crossTenantCommitNegative;
															#line 2752 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Optional<Error> && outcome,int loopDepth) 
	{
															#line 425 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!outcome.present())
															#line 2763 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 426 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "CrossTenantCommitDisallowed") .detail("Subcase", "Negative") .log();
															#line 2767 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		else
		{
															#line 430 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (outcome.get().code() == error_code_permission_denied)
															#line 2773 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 431 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				++self->crossTenantCommitNegative;
															#line 2777 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Optional<Error> const& outcome,int loopDepth) 
	{
		loopDepth = a_body1cont8(outcome, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Optional<Error> && outcome,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(outcome), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantCommitDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantCommitDisallowedActor*>(this)->ActorCallback< TestCrossTenantCommitDisallowedActor, 2, Optional<Error> >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantCommitDisallowedActor, 2, Optional<Error> >*,Optional<Error> const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantCommitDisallowedActor, 2, Optional<Error> >*,Optional<Error> && value) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TestCrossTenantCommitDisallowedActor, 2, Optional<Error> >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	PositiveTestcase positive;
															#line 405 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 406 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 407 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value newValue;
															#line 408 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 2873 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testCrossTenantCommitDisallowed()
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantCommitDisallowedActor final : public Actor<Void>, public ActorCallback< TestCrossTenantCommitDisallowedActor, 0, Version >, public ActorCallback< TestCrossTenantCommitDisallowedActor, 1, Optional<Error> >, public ActorCallback< TestCrossTenantCommitDisallowedActor, 2, Optional<Error> >, public FastAllocated<TestCrossTenantCommitDisallowedActor>, public TestCrossTenantCommitDisallowedActorState<TestCrossTenantCommitDisallowedActor> {
															#line 2878 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestCrossTenantCommitDisallowedActor>::operator new;
	using FastAllocated<TestCrossTenantCommitDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4376770939920392192UL, 1736054308880912896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestCrossTenantCommitDisallowedActor, 0, Version >;
friend struct ActorCallback< TestCrossTenantCommitDisallowedActor, 1, Optional<Error> >;
friend struct ActorCallback< TestCrossTenantCommitDisallowedActor, 2, Optional<Error> >;
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantCommitDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 2897 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestCrossTenantCommitDisallowedActorState<TestCrossTenantCommitDisallowedActor>(self, cx, positive),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5412969517208966656UL, 10653583560272852992UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantCommitDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testCrossTenantCommitDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testCrossTenantCommitDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestCrossTenantCommitDisallowedActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestCrossTenantCommitDisallowedActor, 1, Optional<Error> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TestCrossTenantCommitDisallowedActor, 2, Optional<Error> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testCrossTenantCommitDisallowed( AuthzSecurityWorkload* const& self, Database const& cx, PositiveTestcase const& positive ) {
															#line 402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestCrossTenantCommitDisallowedActor(self, cx, positive));
															#line 2931 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 436 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 2936 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testPublicNonTenantRequestsAllowedWithoutTokens()
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestPublicNonTenantRequestsAllowedWithoutTokensActor>
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestPublicNonTenantRequestsAllowedWithoutTokensActorState {
															#line 2942 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestPublicNonTenantRequestsAllowedWithoutTokensActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 439 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tr(cx, self->tenant)
															#line 2953 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this));

	}
	~TestPublicNonTenantRequestsAllowedWithoutTokensActorState() 
	{
		fdb_probe_actor_destroy("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 440 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			;
															#line 2968 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestPublicNonTenantRequestsAllowedWithoutTokensActorState();
		static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 442 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Version> __when_expr_0 = tr.getReadVersion();
															#line 442 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3001 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->actor_wait_state = 1;
															#line 442 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 0, Version >*>(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)));
															#line 3006 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 447 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 447 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3030 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->actor_wait_state = 2;
															#line 447 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 1, Void >*>(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)));
															#line 3035 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Version const& version,int loopDepth) 
	{
															#line 443 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		(void)version;
															#line 444 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		++self->publicNonTenantRequestPositive;
															#line 445 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestPublicNonTenantRequestsAllowedWithoutTokensActorState(); static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->destroy(); return 0; }
															#line 3054 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestPublicNonTenantRequestsAllowedWithoutTokensActorState();
		static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Version && version,int loopDepth) 
	{
															#line 443 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		(void)version;
															#line 444 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		++self->publicNonTenantRequestPositive;
															#line 445 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestPublicNonTenantRequestsAllowedWithoutTokensActorState(); static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->destroy(); return 0; }
															#line 3070 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestPublicNonTenantRequestsAllowedWithoutTokensActorState();
		static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Version const& version,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(version, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version && version,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(version), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->actor_wait_state > 0) static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->actor_wait_state = 0;
		static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->actor_wait_state > 0) static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->actor_wait_state = 0;
		static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 439 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Transaction tr;
															#line 3246 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testPublicNonTenantRequestsAllowedWithoutTokens()
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestPublicNonTenantRequestsAllowedWithoutTokensActor final : public Actor<Void>, public ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 0, Version >, public ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 1, Void >, public FastAllocated<TestPublicNonTenantRequestsAllowedWithoutTokensActor>, public TestPublicNonTenantRequestsAllowedWithoutTokensActorState<TestPublicNonTenantRequestsAllowedWithoutTokensActor> {
															#line 3251 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestPublicNonTenantRequestsAllowedWithoutTokensActor>::operator new;
	using FastAllocated<TestPublicNonTenantRequestsAllowedWithoutTokensActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12581652276829963008UL, 8639271722952096000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 0, Version >;
friend struct ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 1, Void >;
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestPublicNonTenantRequestsAllowedWithoutTokensActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 3269 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestPublicNonTenantRequestsAllowedWithoutTokensActorState<TestPublicNonTenantRequestsAllowedWithoutTokensActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7275377086168935936UL, 280063435299632128UL);
		ActorExecutionContextHelper __helper(static_cast<TestPublicNonTenantRequestsAllowedWithoutTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testPublicNonTenantRequestsAllowedWithoutTokens");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testPublicNonTenantRequestsAllowedWithoutTokens", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestPublicNonTenantRequestsAllowedWithoutTokensActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testPublicNonTenantRequestsAllowedWithoutTokens( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 437 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestPublicNonTenantRequestsAllowedWithoutTokensActor(self, cx));
															#line 3302 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 451 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 3307 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testTLogReadDisallowed()
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestTLogReadDisallowedActor>
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestTLogReadDisallowedActorState {
															#line 3313 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestTLogReadDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 453 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 454 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString())
															#line 3326 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestTLogReadDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 455 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Version> __when_expr_0 = setAndCommitKeyValueAndGetVersion(self, cx, self->tenant, self->signedToken, key, value);
															#line 455 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3343 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state = 1;
															#line 455 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestTLogReadDisallowedActor, 0, Version >*>(static_cast<TestTLogReadDisallowedActor*>(this)));
															#line 3348 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestTLogReadDisallowedActorState();
		static_cast<TestTLogReadDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 457 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		tr = Transaction(cx, self->tenant);
															#line 458 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		tLogConfigString = Optional<Value>();
															#line 459 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		;
															#line 3375 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Version const& __committedVersion,int loopDepth) 
	{
															#line 455 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		committedVersion = __committedVersion;
															#line 3384 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Version && __committedVersion,int loopDepth) 
	{
		committedVersion = std::move(__committedVersion);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestTLogReadDisallowedActor*>(this)->ActorCallback< TestTLogReadDisallowedActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestTLogReadDisallowedActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestTLogReadDisallowedActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestTLogReadDisallowedActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 470 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(tLogConfigString.present());
															#line 471 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		logSystemConfig = ObjectReader::fromStringRef<LogSystemConfig>(tLogConfigString.get(), IncludeVersion());
															#line 473 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		logs = logSystemConfig.allPresentLogs();
															#line 474 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		replies = std::vector<Future<ErrorOr<TLogPeekReply>>>();
															#line 475 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for( const auto& log : logs ) {
															#line 476 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			replies.push_back(log.peekMessages.tryGetReply( TLogPeekRequest(committedVersion, Tag(0, committedVersion), false, false)));
															#line 3473 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
															#line 479 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Void> __when_expr_3 = waitForAllReady(replies);
															#line 479 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3479 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state = 4;
															#line 479 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TestTLogReadDisallowedActor, 3, Void >*>(static_cast<TestTLogReadDisallowedActor*>(this)));
															#line 3484 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 461 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			self->setAuthToken(tr, self->signedToken);
															#line 462 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = tr.get(self->tLogConfigKey);
															#line 462 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3505 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state = 2;
															#line 462 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestTLogReadDisallowedActor, 1, Optional<Value> >*>(static_cast<TestTLogReadDisallowedActor*>(this)));
															#line 3510 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 467 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 467 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3547 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state = 3;
															#line 467 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TestTLogReadDisallowedActor, 2, Void >*>(static_cast<TestTLogReadDisallowedActor*>(this)));
															#line 3552 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Optional<Value> const& value,int loopDepth) 
	{
															#line 463 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(value.present());
															#line 464 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		tLogConfigString = value;
															#line 3569 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Optional<Value> && value,int loopDepth) 
	{
															#line 463 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(value.present());
															#line 464 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		tLogConfigString = value;
															#line 3580 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Optional<Value> const& value,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(value, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Optional<Value> && value,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(value), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestTLogReadDisallowedActor*>(this)->ActorCallback< TestTLogReadDisallowedActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< TestTLogReadDisallowedActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestTLogReadDisallowedActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestTLogReadDisallowedActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestTLogReadDisallowedActor*>(this)->ActorCallback< TestTLogReadDisallowedActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestTLogReadDisallowedActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TestTLogReadDisallowedActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TestTLogReadDisallowedActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 480 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for(auto i = 0u;i < logs.size();i++) {
															#line 481 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			const auto& reply = replies[i];
															#line 482 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			ASSERT(reply.isValid());
															#line 483 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (reply.canGet())
															#line 3757 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 484 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				ErrorOr<TLogPeekReply> r = reply.getValue();
															#line 485 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				if (!r.isError())
															#line 3763 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				{
															#line 486 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
					const TLogPeekReply& rpcReply = r.get();
															#line 487 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
					TraceEvent(SevError, "AuthzExpectedErrorNotFound") .detail("TLogIndex", i) .detail("Messages", rpcReply.messages.toString()) .detail("End", rpcReply.end) .detail("Popped", rpcReply.popped) .detail("MaxKnownVersion", rpcReply.maxKnownVersion) .detail("MinKnownCommitVersion", rpcReply.minKnownCommittedVersion) .detail("Begin", rpcReply.begin) .detail("OnlySpilled", rpcReply.onlySpilled) .log();
															#line 3769 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				}
				else
				{
															#line 498 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
					Error e = r.getError();
															#line 499 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
					if (e.code() == error_code_unauthorized_attempt)
															#line 3777 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
					{
															#line 500 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
						++self->tLogReadNegative;
															#line 3781 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
					}
					else
					{
															#line 501 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
						if (e.code() != error_code_actor_cancelled && e.code() != error_code_request_maybe_delivered)
															#line 3787 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
						{
															#line 503 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
							TraceEvent(SevError, "AuthzSecurityUnexpectedError").detail("Error", e.name()).log();
															#line 3791 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
						}
					}
				}
			}
			else
			{
															#line 507 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityUnexpectedError").detail("Error", reply.getError().name()).log();
															#line 3800 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
															#line 510 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestTLogReadDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestTLogReadDisallowedActorState(); static_cast<TestTLogReadDisallowedActor*>(this)->destroy(); return 0; }
															#line 3805 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestTLogReadDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestTLogReadDisallowedActorState();
		static_cast<TestTLogReadDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 480 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for(auto i = 0u;i < logs.size();i++) {
															#line 481 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			const auto& reply = replies[i];
															#line 482 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			ASSERT(reply.isValid());
															#line 483 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (reply.canGet())
															#line 3823 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 484 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				ErrorOr<TLogPeekReply> r = reply.getValue();
															#line 485 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				if (!r.isError())
															#line 3829 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				{
															#line 486 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
					const TLogPeekReply& rpcReply = r.get();
															#line 487 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
					TraceEvent(SevError, "AuthzExpectedErrorNotFound") .detail("TLogIndex", i) .detail("Messages", rpcReply.messages.toString()) .detail("End", rpcReply.end) .detail("Popped", rpcReply.popped) .detail("MaxKnownVersion", rpcReply.maxKnownVersion) .detail("MinKnownCommitVersion", rpcReply.minKnownCommittedVersion) .detail("Begin", rpcReply.begin) .detail("OnlySpilled", rpcReply.onlySpilled) .log();
															#line 3835 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				}
				else
				{
															#line 498 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
					Error e = r.getError();
															#line 499 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
					if (e.code() == error_code_unauthorized_attempt)
															#line 3843 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
					{
															#line 500 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
						++self->tLogReadNegative;
															#line 3847 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
					}
					else
					{
															#line 501 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
						if (e.code() != error_code_actor_cancelled && e.code() != error_code_request_maybe_delivered)
															#line 3853 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
						{
															#line 503 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
							TraceEvent(SevError, "AuthzSecurityUnexpectedError").detail("Error", e.name()).log();
															#line 3857 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
						}
					}
				}
			}
			else
			{
															#line 507 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityUnexpectedError").detail("Error", reply.getError().name()).log();
															#line 3866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
															#line 510 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestTLogReadDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestTLogReadDisallowedActorState(); static_cast<TestTLogReadDisallowedActor*>(this)->destroy(); return 0; }
															#line 3871 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestTLogReadDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestTLogReadDisallowedActorState();
		static_cast<TestTLogReadDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestTLogReadDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestTLogReadDisallowedActor*>(this)->ActorCallback< TestTLogReadDisallowedActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestTLogReadDisallowedActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TestTLogReadDisallowedActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TestTLogReadDisallowedActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 453 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 454 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 455 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 457 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Transaction tr;
															#line 458 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Optional<Value> tLogConfigString;
															#line 471 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	LogSystemConfig logSystemConfig;
															#line 473 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	std::vector<TLogInterface> logs;
															#line 474 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	std::vector<Future<ErrorOr<TLogPeekReply>>> replies;
															#line 3974 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testTLogReadDisallowed()
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestTLogReadDisallowedActor final : public Actor<Void>, public ActorCallback< TestTLogReadDisallowedActor, 0, Version >, public ActorCallback< TestTLogReadDisallowedActor, 1, Optional<Value> >, public ActorCallback< TestTLogReadDisallowedActor, 2, Void >, public ActorCallback< TestTLogReadDisallowedActor, 3, Void >, public FastAllocated<TestTLogReadDisallowedActor>, public TestTLogReadDisallowedActorState<TestTLogReadDisallowedActor> {
															#line 3979 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestTLogReadDisallowedActor>::operator new;
	using FastAllocated<TestTLogReadDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17634031010262895360UL, 2879131107878006784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestTLogReadDisallowedActor, 0, Version >;
friend struct ActorCallback< TestTLogReadDisallowedActor, 1, Optional<Value> >;
friend struct ActorCallback< TestTLogReadDisallowedActor, 2, Void >;
friend struct ActorCallback< TestTLogReadDisallowedActor, 3, Void >;
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestTLogReadDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 3999 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestTLogReadDisallowedActorState<TestTLogReadDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4039462977600321024UL, 17875274534978066688UL);
		ActorExecutionContextHelper __helper(static_cast<TestTLogReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testTLogReadDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testTLogReadDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestTLogReadDisallowedActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestTLogReadDisallowedActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TestTLogReadDisallowedActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TestTLogReadDisallowedActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testTLogReadDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestTLogReadDisallowedActor(self, cx));
															#line 4034 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 512 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 4039 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testKeyLocationLeakDisallowed()
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestKeyLocationLeakDisallowedActor>
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestKeyLocationLeakDisallowedActorState {
															#line 4045 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestKeyLocationLeakDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 514 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 515 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString())
															#line 4058 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestKeyLocationLeakDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 517 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(setAndCommitKeyValueAndGetVersion(self, cx, self->tenant, self->signedToken, key, value));
															#line 517 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4075 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 1;
															#line 517 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestKeyLocationLeakDisallowedActor, 0, Void >*>(static_cast<TestKeyLocationLeakDisallowedActor*>(this)));
															#line 4080 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestKeyLocationLeakDisallowedActorState();
		static_cast<TestKeyLocationLeakDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 518 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Version> __when_expr_1 = setAndCommitKeyValueAndGetVersion( self, cx, self->anotherTenant, self->signedTokenAnotherTenant, key, value);
															#line 518 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4105 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 2;
															#line 518 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >*>(static_cast<TestKeyLocationLeakDisallowedActor*>(this)));
															#line 4110 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 518 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Version> __when_expr_1 = setAndCommitKeyValueAndGetVersion( self, cx, self->anotherTenant, self->signedTokenAnotherTenant, key, value);
															#line 518 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4121 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 2;
															#line 518 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >*>(static_cast<TestKeyLocationLeakDisallowedActor*>(this)));
															#line 4126 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestKeyLocationLeakDisallowedActor*>(this)->ActorCallback< TestKeyLocationLeakDisallowedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestKeyLocationLeakDisallowedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestKeyLocationLeakDisallowedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestKeyLocationLeakDisallowedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
		try {
															#line 522 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<GetKeyServerLocationsReply> __when_expr_2 = basicLoadBalance(cx->getCommitProxies(UseProvisionalProxies::False), &CommitProxyInterface::getKeyServersLocations, GetKeyServerLocationsRequest(SpanContext(), TenantInfo(self->tenant->id(), self->signedToken), key, Optional<KeyRef>(), 100, false, v2, Arena()));
															#line 522 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1cont2Catch1(actor_cancelled(), loopDepth);
															#line 4213 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 3;
															#line 522 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TestKeyLocationLeakDisallowedActor, 2, GetKeyServerLocationsReply >*>(static_cast<TestKeyLocationLeakDisallowedActor*>(this)));
															#line 4218 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Version const& __v2,int loopDepth) 
	{
															#line 518 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		v2 = __v2;
															#line 4233 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Version && __v2,int loopDepth) 
	{
		v2 = std::move(__v2);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestKeyLocationLeakDisallowedActor*>(this)->ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
															#line 552 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<GetKeyServerLocationsReply> __when_expr_3 = basicLoadBalance( cx->getCommitProxies(UseProvisionalProxies::False), &CommitProxyInterface::getKeyServersLocations, GetKeyServerLocationsRequest(SpanContext(), TenantInfo(self->anotherTenant->id(), self->signedTokenAnotherTenant), key, Optional<KeyRef>(), 100, false, v2, Arena()));
															#line 552 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 4315 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont3Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 4;
															#line 552 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TestKeyLocationLeakDisallowedActor, 3, GetKeyServerLocationsReply >*>(static_cast<TestKeyLocationLeakDisallowedActor*>(this)));
															#line 4320 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 546 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 4336 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 547 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4340 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 549 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			ASSERT(e.code() == error_code_commit_proxy_memory_limit_exceeded);
															#line 4344 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(GetKeyServerLocationsReply const& rep,int loopDepth) 
	{
															#line 533 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for( auto const& [range, ssIfaces] : rep.results ) {
															#line 534 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!range.begin.startsWith(self->tenant->prefix()))
															#line 4361 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 535 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityKeyRangeLeak") .detail("TenantId", self->tenant->id()) .detail("LeakingRangeBegin", range.begin.printable());
															#line 4365 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 539 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!range.end.startsWith(self->tenant->prefix()))
															#line 4369 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 540 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityKeyRangeLeak") .detail("TenantId", self->tenant->id()) .detail("LeakingRangeEnd", range.end.printable());
															#line 4373 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(GetKeyServerLocationsReply && rep,int loopDepth) 
	{
															#line 533 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for( auto const& [range, ssIfaces] : rep.results ) {
															#line 534 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!range.begin.startsWith(self->tenant->prefix()))
															#line 4386 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 535 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityKeyRangeLeak") .detail("TenantId", self->tenant->id()) .detail("LeakingRangeBegin", range.begin.printable());
															#line 4390 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 539 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!range.end.startsWith(self->tenant->prefix()))
															#line 4394 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 540 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityKeyRangeLeak") .detail("TenantId", self->tenant->id()) .detail("LeakingRangeEnd", range.end.printable());
															#line 4398 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(GetKeyServerLocationsReply const& rep,int loopDepth) 
	{
		loopDepth = a_body1cont4(rep, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(GetKeyServerLocationsReply && rep,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestKeyLocationLeakDisallowedActor*>(this)->ActorCallback< TestKeyLocationLeakDisallowedActor, 2, GetKeyServerLocationsReply >::remove();

	}
	void a_callback_fire(ActorCallback< TestKeyLocationLeakDisallowedActor, 2, GetKeyServerLocationsReply >*,GetKeyServerLocationsReply const& value) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TestKeyLocationLeakDisallowedActor, 2, GetKeyServerLocationsReply >*,GetKeyServerLocationsReply && value) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TestKeyLocationLeakDisallowedActor, 2, GetKeyServerLocationsReply >*,Error err) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
															#line 581 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		++self->keyLocationLeakNegative;
															#line 583 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestKeyLocationLeakDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestKeyLocationLeakDisallowedActorState(); static_cast<TestKeyLocationLeakDisallowedActor*>(this)->destroy(); return 0; }
															#line 4499 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestKeyLocationLeakDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestKeyLocationLeakDisallowedActorState();
		static_cast<TestKeyLocationLeakDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 576 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 4512 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 577 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4516 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 579 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			ASSERT(e.code() == error_code_commit_proxy_memory_limit_exceeded);
															#line 4520 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = a_body1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11(GetKeyServerLocationsReply const& rep,int loopDepth) 
	{
															#line 563 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for( auto const& [range, ssIfaces] : rep.results ) {
															#line 564 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!range.begin.startsWith(self->anotherTenant->prefix()))
															#line 4537 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 565 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityKeyRangeLeak") .detail("TenantId", self->anotherTenant->id()) .detail("LeakingRangeBegin", range.begin.printable());
															#line 4541 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 569 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!range.end.startsWith(self->anotherTenant->prefix()))
															#line 4545 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 570 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityKeyRangeLeak") .detail("TenantId", self->anotherTenant->id()) .detail("LeakingRangeEnd", range.end.printable());
															#line 4549 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont11cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(GetKeyServerLocationsReply && rep,int loopDepth) 
	{
															#line 563 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for( auto const& [range, ssIfaces] : rep.results ) {
															#line 564 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!range.begin.startsWith(self->anotherTenant->prefix()))
															#line 4562 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 565 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityKeyRangeLeak") .detail("TenantId", self->anotherTenant->id()) .detail("LeakingRangeBegin", range.begin.printable());
															#line 4566 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 569 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!range.end.startsWith(self->anotherTenant->prefix()))
															#line 4570 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 570 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityKeyRangeLeak") .detail("TenantId", self->anotherTenant->id()) .detail("LeakingRangeEnd", range.end.printable());
															#line 4574 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
		}
		loopDepth = a_body1cont11cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(GetKeyServerLocationsReply const& rep,int loopDepth) 
	{
		loopDepth = a_body1cont11(rep, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(GetKeyServerLocationsReply && rep,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestKeyLocationLeakDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestKeyLocationLeakDisallowedActor*>(this)->ActorCallback< TestKeyLocationLeakDisallowedActor, 3, GetKeyServerLocationsReply >::remove();

	}
	void a_callback_fire(ActorCallback< TestKeyLocationLeakDisallowedActor, 3, GetKeyServerLocationsReply >*,GetKeyServerLocationsReply const& value) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TestKeyLocationLeakDisallowedActor, 3, GetKeyServerLocationsReply >*,GetKeyServerLocationsReply && value) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TestKeyLocationLeakDisallowedActor, 3, GetKeyServerLocationsReply >*,Error err) 
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont11cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 514 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 515 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 518 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version v2;
															#line 4679 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testKeyLocationLeakDisallowed()
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestKeyLocationLeakDisallowedActor final : public Actor<Void>, public ActorCallback< TestKeyLocationLeakDisallowedActor, 0, Void >, public ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >, public ActorCallback< TestKeyLocationLeakDisallowedActor, 2, GetKeyServerLocationsReply >, public ActorCallback< TestKeyLocationLeakDisallowedActor, 3, GetKeyServerLocationsReply >, public FastAllocated<TestKeyLocationLeakDisallowedActor>, public TestKeyLocationLeakDisallowedActorState<TestKeyLocationLeakDisallowedActor> {
															#line 4684 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestKeyLocationLeakDisallowedActor>::operator new;
	using FastAllocated<TestKeyLocationLeakDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3542850813660815872UL, 7329437095430169856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestKeyLocationLeakDisallowedActor, 0, Void >;
friend struct ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >;
friend struct ActorCallback< TestKeyLocationLeakDisallowedActor, 2, GetKeyServerLocationsReply >;
friend struct ActorCallback< TestKeyLocationLeakDisallowedActor, 3, GetKeyServerLocationsReply >;
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestKeyLocationLeakDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 4704 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestKeyLocationLeakDisallowedActorState<TestKeyLocationLeakDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10833391251506936576UL, 13576044464119215104UL);
		ActorExecutionContextHelper __helper(static_cast<TestKeyLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testKeyLocationLeakDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testKeyLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestKeyLocationLeakDisallowedActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestKeyLocationLeakDisallowedActor, 1, Version >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TestKeyLocationLeakDisallowedActor, 2, GetKeyServerLocationsReply >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TestKeyLocationLeakDisallowedActor, 3, GetKeyServerLocationsReply >*)0, actor_cancelled()); break;
		}

	}
};
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testKeyLocationLeakDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 513 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestKeyLocationLeakDisallowedActor(self, cx));
															#line 4739 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 585 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 4744 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via getGranuleLocations()
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class GetGranuleLocationsActor>
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class GetGranuleLocationsActorState {
															#line 4750 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	GetGranuleLocationsActorState(AuthzSecurityWorkload* const& self,Database const& cx,TenantInfo const& tenant,Version const& v) 
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tenant(tenant),
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   v(v)
															#line 4763 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("getGranuleLocations", reinterpret_cast<unsigned long>(this));

	}
	~GetGranuleLocationsActorState() 
	{
		fdb_probe_actor_destroy("getGranuleLocations", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 591 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				StrictFuture<GetBlobGranuleLocationsReply> __when_expr_0 = basicLoadBalance(cx->getCommitProxies(UseProvisionalProxies::False), &CommitProxyInterface::getBlobGranuleLocations, GetBlobGranuleLocationsRequest( SpanContext(), tenant, ""_sr, Optional<KeyRef>(), 100, false, false, v, Arena()));
															#line 591 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				if (static_cast<GetGranuleLocationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4781 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetGranuleLocationsActor*>(this)->actor_wait_state = 1;
															#line 591 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetGranuleLocationsActor, 0, GetBlobGranuleLocationsReply >*>(static_cast<GetGranuleLocationsActor*>(this)));
															#line 4786 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetGranuleLocationsActorState();
		static_cast<GetGranuleLocationsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 598 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 4816 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 599 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4820 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 601 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			CODE_PROBE(e.code() == error_code_permission_denied, "Cross tenant blob granule locations meets permission_denied");
															#line 603 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!static_cast<GetGranuleLocationsActor*>(this)->SAV<ErrorOr<GetBlobGranuleLocationsReply>>::futures) { (void)(e); this->~GetGranuleLocationsActorState(); static_cast<GetGranuleLocationsActor*>(this)->destroy(); return 0; }
															#line 4826 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			new (&static_cast<GetGranuleLocationsActor*>(this)->SAV< ErrorOr<GetBlobGranuleLocationsReply> >::value()) ErrorOr<GetBlobGranuleLocationsReply>(e);
			this->~GetGranuleLocationsActorState();
			static_cast<GetGranuleLocationsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(GetBlobGranuleLocationsReply const& reply,int loopDepth) 
	{
															#line 596 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<GetGranuleLocationsActor*>(this)->SAV<ErrorOr<GetBlobGranuleLocationsReply>>::futures) { (void)(reply); this->~GetGranuleLocationsActorState(); static_cast<GetGranuleLocationsActor*>(this)->destroy(); return 0; }
															#line 4844 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<GetGranuleLocationsActor*>(this)->SAV< ErrorOr<GetBlobGranuleLocationsReply> >::value()) ErrorOr<GetBlobGranuleLocationsReply>(reply);
		this->~GetGranuleLocationsActorState();
		static_cast<GetGranuleLocationsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(GetBlobGranuleLocationsReply && reply,int loopDepth) 
	{
															#line 596 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<GetGranuleLocationsActor*>(this)->SAV<ErrorOr<GetBlobGranuleLocationsReply>>::futures) { (void)(reply); this->~GetGranuleLocationsActorState(); static_cast<GetGranuleLocationsActor*>(this)->destroy(); return 0; }
															#line 4856 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<GetGranuleLocationsActor*>(this)->SAV< ErrorOr<GetBlobGranuleLocationsReply> >::value()) ErrorOr<GetBlobGranuleLocationsReply>(reply);
		this->~GetGranuleLocationsActorState();
		static_cast<GetGranuleLocationsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(GetBlobGranuleLocationsReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont2(reply, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetBlobGranuleLocationsReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetGranuleLocationsActor*>(this)->actor_wait_state > 0) static_cast<GetGranuleLocationsActor*>(this)->actor_wait_state = 0;
		static_cast<GetGranuleLocationsActor*>(this)->ActorCallback< GetGranuleLocationsActor, 0, GetBlobGranuleLocationsReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetGranuleLocationsActor, 0, GetBlobGranuleLocationsReply >*,GetBlobGranuleLocationsReply const& value) 
	{
		fdb_probe_actor_enter("getGranuleLocations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleLocationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleLocations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetGranuleLocationsActor, 0, GetBlobGranuleLocationsReply >*,GetBlobGranuleLocationsReply && value) 
	{
		fdb_probe_actor_enter("getGranuleLocations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleLocationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleLocations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetGranuleLocationsActor, 0, GetBlobGranuleLocationsReply >*,Error err) 
	{
		fdb_probe_actor_enter("getGranuleLocations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleLocationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleLocations", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TenantInfo tenant;
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version v;
															#line 4947 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via getGranuleLocations()
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class GetGranuleLocationsActor final : public Actor<ErrorOr<GetBlobGranuleLocationsReply>>, public ActorCallback< GetGranuleLocationsActor, 0, GetBlobGranuleLocationsReply >, public FastAllocated<GetGranuleLocationsActor>, public GetGranuleLocationsActorState<GetGranuleLocationsActor> {
															#line 4952 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<GetGranuleLocationsActor>::operator new;
	using FastAllocated<GetGranuleLocationsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7744631235489150720UL, 16399604180891624704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<ErrorOr<GetBlobGranuleLocationsReply>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetGranuleLocationsActor, 0, GetBlobGranuleLocationsReply >;
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	GetGranuleLocationsActor(AuthzSecurityWorkload* const& self,Database const& cx,TenantInfo const& tenant,Version const& v) 
															#line 4969 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<ErrorOr<GetBlobGranuleLocationsReply>>(),
		   GetGranuleLocationsActorState<GetGranuleLocationsActor>(self, cx, tenant, v),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getGranuleLocations", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14506698547047788544UL, 3621267239456157440UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleLocationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getGranuleLocations");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getGranuleLocations", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetGranuleLocationsActor, 0, GetBlobGranuleLocationsReply >*)0, actor_cancelled()); break;
		}

	}
};
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<ErrorOr<GetBlobGranuleLocationsReply>> getGranuleLocations( AuthzSecurityWorkload* const& self, Database const& cx, TenantInfo const& tenant, Version const& v ) {
															#line 586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<ErrorOr<GetBlobGranuleLocationsReply>>(new GetGranuleLocationsActor(self, cx, tenant, v));
															#line 5001 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 606 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 5006 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testBlobGranuleLocationLeakDisallowed()
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestBlobGranuleLocationLeakDisallowedActor>
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestBlobGranuleLocationLeakDisallowedActorState {
															#line 5012 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestBlobGranuleLocationLeakDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 608 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 609 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString())
															#line 5025 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestBlobGranuleLocationLeakDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 610 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Version> __when_expr_0 = setAndCommitKeyValueAndGetVersion(self, cx, self->tenant, self->signedToken, key, value);
															#line 610 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5042 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state = 1;
															#line 610 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 0, Version >*>(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)));
															#line 5047 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestBlobGranuleLocationLeakDisallowedActorState();
		static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 612 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Version> __when_expr_1 = setAndCommitKeyValueAndGetVersion( self, cx, self->anotherTenant, self->signedTokenAnotherTenant, key, value);
															#line 612 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5072 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state = 2;
															#line 612 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 1, Version >*>(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)));
															#line 5077 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Version const& __v1,int loopDepth) 
	{
															#line 610 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		v1 = __v1;
															#line 5086 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Version && __v1,int loopDepth) 
	{
		v1 = std::move(__v1);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 615 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		success = true;
															#line 616 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		tenantInfo = TenantInfo();
															#line 5167 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 619 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			tenantInfo = TenantInfo(self->tenant->id(), self->signedToken);
															#line 620 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<ErrorOr<GetBlobGranuleLocationsReply>> __when_expr_2 = getGranuleLocations(self, cx, tenantInfo, v2);
															#line 620 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5175 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state = 3;
															#line 620 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 2, ErrorOr<GetBlobGranuleLocationsReply> >*>(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)));
															#line 5180 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1when1(Version const& __v2,int loopDepth) 
	{
															#line 612 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		v2 = __v2;
															#line 5190 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Version && __v2,int loopDepth) 
	{
		v2 = std::move(__v2);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 1, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 1, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 1, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 1, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
		{
															#line 625 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			tenantInfo = TenantInfo(self->anotherTenant->id(), self->signedTokenAnotherTenant);
															#line 626 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<ErrorOr<GetBlobGranuleLocationsReply>> __when_expr_3 = getGranuleLocations(self, cx, tenantInfo, v2);
															#line 626 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5274 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state = 4;
															#line 626 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 3, ErrorOr<GetBlobGranuleLocationsReply> >*>(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)));
															#line 5279 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont4(ErrorOr<GetBlobGranuleLocationsReply> const& rep,int loopDepth) 
	{
															#line 621 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		bool checkSuccess = checkGranuleLocations(rep, tenantInfo);
															#line 622 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		success &= checkSuccess;
															#line 5291 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(ErrorOr<GetBlobGranuleLocationsReply> && rep,int loopDepth) 
	{
															#line 621 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		bool checkSuccess = checkGranuleLocations(rep, tenantInfo);
															#line 622 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		success &= checkSuccess;
															#line 5302 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(ErrorOr<GetBlobGranuleLocationsReply> const& rep,int loopDepth) 
	{
		loopDepth = a_body1cont4(rep, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(ErrorOr<GetBlobGranuleLocationsReply> && rep,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 2, ErrorOr<GetBlobGranuleLocationsReply> >::remove();

	}
	void a_callback_fire(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 2, ErrorOr<GetBlobGranuleLocationsReply> >*,ErrorOr<GetBlobGranuleLocationsReply> const& value) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 2, ErrorOr<GetBlobGranuleLocationsReply> >*,ErrorOr<GetBlobGranuleLocationsReply> && value) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 2, ErrorOr<GetBlobGranuleLocationsReply> >*,Error err) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 630 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (success)
															#line 5386 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 631 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			++self->bgLocationLeakNegative;
															#line 5390 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
															#line 634 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestBlobGranuleLocationLeakDisallowedActorState(); static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->destroy(); return 0; }
															#line 5394 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestBlobGranuleLocationLeakDisallowedActorState();
		static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(ErrorOr<GetBlobGranuleLocationsReply> const& rep,int loopDepth) 
	{
															#line 627 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		bool checkSuccess = checkGranuleLocations(rep, tenantInfo);
															#line 628 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		success &= checkSuccess;
															#line 5408 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(ErrorOr<GetBlobGranuleLocationsReply> && rep,int loopDepth) 
	{
															#line 627 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		bool checkSuccess = checkGranuleLocations(rep, tenantInfo);
															#line 628 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		success &= checkSuccess;
															#line 5419 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(ErrorOr<GetBlobGranuleLocationsReply> const& rep,int loopDepth) 
	{
		loopDepth = a_body1cont7(rep, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(ErrorOr<GetBlobGranuleLocationsReply> && rep,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 3, ErrorOr<GetBlobGranuleLocationsReply> >::remove();

	}
	void a_callback_fire(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 3, ErrorOr<GetBlobGranuleLocationsReply> >*,ErrorOr<GetBlobGranuleLocationsReply> const& value) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 3, ErrorOr<GetBlobGranuleLocationsReply> >*,ErrorOr<GetBlobGranuleLocationsReply> && value) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 3, ErrorOr<GetBlobGranuleLocationsReply> >*,Error err) 
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 608 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 609 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 610 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version v1;
															#line 612 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version v2;
															#line 615 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	bool success;
															#line 616 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TenantInfo tenantInfo;
															#line 5515 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testBlobGranuleLocationLeakDisallowed()
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestBlobGranuleLocationLeakDisallowedActor final : public Actor<Void>, public ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 0, Version >, public ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 1, Version >, public ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 2, ErrorOr<GetBlobGranuleLocationsReply> >, public ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 3, ErrorOr<GetBlobGranuleLocationsReply> >, public FastAllocated<TestBlobGranuleLocationLeakDisallowedActor>, public TestBlobGranuleLocationLeakDisallowedActorState<TestBlobGranuleLocationLeakDisallowedActor> {
															#line 5520 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestBlobGranuleLocationLeakDisallowedActor>::operator new;
	using FastAllocated<TestBlobGranuleLocationLeakDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16078807626618803968UL, 17664185133771264768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 0, Version >;
friend struct ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 1, Version >;
friend struct ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 2, ErrorOr<GetBlobGranuleLocationsReply> >;
friend struct ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 3, ErrorOr<GetBlobGranuleLocationsReply> >;
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestBlobGranuleLocationLeakDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 5540 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestBlobGranuleLocationLeakDisallowedActorState<TestBlobGranuleLocationLeakDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3537623704447409408UL, 9864912767611079936UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobGranuleLocationLeakDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testBlobGranuleLocationLeakDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testBlobGranuleLocationLeakDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 1, Version >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 2, ErrorOr<GetBlobGranuleLocationsReply> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< TestBlobGranuleLocationLeakDisallowedActor, 3, ErrorOr<GetBlobGranuleLocationsReply> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testBlobGranuleLocationLeakDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestBlobGranuleLocationLeakDisallowedActor(self, cx));
															#line 5575 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 636 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 5580 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via tryBlobGranuleRequest()
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TryBlobGranuleRequestActor>
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryBlobGranuleRequestActorState {
															#line 5586 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryBlobGranuleRequestActorState(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,WipedString const& locToken,WipedString const& reqToken,Version const& committedVersion) 
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tenant(tenant),
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   locToken(locToken),
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   reqToken(reqToken),
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   committedVersion(committedVersion)
															#line 5603 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this));

	}
	~TryBlobGranuleRequestActorState() 
	{
		fdb_probe_actor_destroy("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 644 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				StrictFuture<ErrorOr<GetBlobGranuleLocationsReply>> __when_expr_0 = getGranuleLocations(self, cx, TenantInfo(tenant->id(), locToken), committedVersion);
															#line 644 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				if (static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5621 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state = 1;
															#line 644 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TryBlobGranuleRequestActor, 0, ErrorOr<GetBlobGranuleLocationsReply> >*>(static_cast<TryBlobGranuleRequestActor*>(this)));
															#line 5626 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TryBlobGranuleRequestActorState();
		static_cast<TryBlobGranuleRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 683 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			CODE_PROBE(e.code() == error_code_permission_denied, "Cross tenant blob granule read meets permission_denied");
															#line 685 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!static_cast<TryBlobGranuleRequestActor*>(this)->SAV<Optional<Error>>::futures) { (void)(e); this->~TryBlobGranuleRequestActorState(); static_cast<TryBlobGranuleRequestActor*>(this)->destroy(); return 0; }
															#line 5658 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			new (&static_cast<TryBlobGranuleRequestActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(e);
			this->~TryBlobGranuleRequestActorState();
			static_cast<TryBlobGranuleRequestActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(ErrorOr<GetBlobGranuleLocationsReply> const& rep,int loopDepth) 
	{
															#line 646 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (rep.isError())
															#line 5676 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 647 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (rep.getError().code() == error_code_permission_denied)
															#line 5680 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 648 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "GranuleLocBeforeRequestDisallowed") .log();
															#line 5684 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 652 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			return a_body1Catch2(rep.getError(), loopDepth);
															#line 5688 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
															#line 655 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		int locIdx = deterministicRandom()->randomInt(0, rep.get().results.size());
															#line 657 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(!rep.get().results.empty());
															#line 658 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(!rep.get().bwInterfs.empty());
															#line 659 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		BlobGranuleFileRequest req;
															#line 660 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.arena.dependsOn(rep.get().arena);
															#line 661 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.keyRange = rep.get().results[locIdx].first;
															#line 662 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.tenantInfo = TenantInfo(tenant->id(), reqToken);
															#line 663 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.readVersion = committedVersion;
															#line 665 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		UID bwId = rep.get().results[locIdx].second;
															#line 666 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(bwId != UID());
															#line 667 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		int bwInterfIdx;
															#line 668 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for(bwInterfIdx = 0;bwInterfIdx < rep.get().bwInterfs.size();bwInterfIdx++) {
															#line 669 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (rep.get().bwInterfs[bwInterfIdx].id() == bwId)
															#line 5716 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
				break;
			}
		}
															#line 673 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(bwInterfIdx < rep.get().bwInterfs.size());
															#line 674 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		auto& bwInterf = rep.get().bwInterfs[bwInterfIdx];
															#line 675 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<ErrorOr<BlobGranuleFileReply>> __when_expr_1 = bwInterf.blobGranuleFileRequest.tryGetReply(req);
															#line 675 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5729 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state = 2;
															#line 675 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >*>(static_cast<TryBlobGranuleRequestActor*>(this)));
															#line 5734 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(ErrorOr<GetBlobGranuleLocationsReply> && rep,int loopDepth) 
	{
															#line 646 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (rep.isError())
															#line 5743 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 647 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (rep.getError().code() == error_code_permission_denied)
															#line 5747 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 648 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityError") .detail("Case", "GranuleLocBeforeRequestDisallowed") .log();
															#line 5751 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 652 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			return a_body1Catch2(rep.getError(), loopDepth);
															#line 5755 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
															#line 655 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		int locIdx = deterministicRandom()->randomInt(0, rep.get().results.size());
															#line 657 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(!rep.get().results.empty());
															#line 658 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(!rep.get().bwInterfs.empty());
															#line 659 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		BlobGranuleFileRequest req;
															#line 660 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.arena.dependsOn(rep.get().arena);
															#line 661 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.keyRange = rep.get().results[locIdx].first;
															#line 662 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.tenantInfo = TenantInfo(tenant->id(), reqToken);
															#line 663 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		req.readVersion = committedVersion;
															#line 665 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		UID bwId = rep.get().results[locIdx].second;
															#line 666 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(bwId != UID());
															#line 667 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		int bwInterfIdx;
															#line 668 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		for(bwInterfIdx = 0;bwInterfIdx < rep.get().bwInterfs.size();bwInterfIdx++) {
															#line 669 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (rep.get().bwInterfs[bwInterfIdx].id() == bwId)
															#line 5783 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
				break;
			}
		}
															#line 673 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(bwInterfIdx < rep.get().bwInterfs.size());
															#line 674 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		auto& bwInterf = rep.get().bwInterfs[bwInterfIdx];
															#line 675 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<ErrorOr<BlobGranuleFileReply>> __when_expr_1 = bwInterf.blobGranuleFileRequest.tryGetReply(req);
															#line 675 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5796 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state = 2;
															#line 675 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >*>(static_cast<TryBlobGranuleRequestActor*>(this)));
															#line 5801 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(ErrorOr<GetBlobGranuleLocationsReply> const& rep,int loopDepth) 
	{
		loopDepth = a_body1cont2(rep, loopDepth);

		return loopDepth;
	}
	int a_body1when1(ErrorOr<GetBlobGranuleLocationsReply> && rep,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state > 0) static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<TryBlobGranuleRequestActor*>(this)->ActorCallback< TryBlobGranuleRequestActor, 0, ErrorOr<GetBlobGranuleLocationsReply> >::remove();

	}
	void a_callback_fire(ActorCallback< TryBlobGranuleRequestActor, 0, ErrorOr<GetBlobGranuleLocationsReply> >*,ErrorOr<GetBlobGranuleLocationsReply> const& value) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TryBlobGranuleRequestActor, 0, ErrorOr<GetBlobGranuleLocationsReply> >*,ErrorOr<GetBlobGranuleLocationsReply> && value) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TryBlobGranuleRequestActor, 0, ErrorOr<GetBlobGranuleLocationsReply> >*,Error err) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(ErrorOr<BlobGranuleFileReply> const& fileRep,int loopDepth) 
	{
															#line 676 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (fileRep.isError())
															#line 5885 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 677 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			return a_body1Catch2(fileRep.getError(), loopDepth);
															#line 5889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
															#line 679 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(!fileRep.get().chunks.empty());
															#line 681 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryBlobGranuleRequestActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryBlobGranuleRequestActorState(); static_cast<TryBlobGranuleRequestActor*>(this)->destroy(); return 0; }
															#line 5895 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryBlobGranuleRequestActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryBlobGranuleRequestActorState();
		static_cast<TryBlobGranuleRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(ErrorOr<BlobGranuleFileReply> && fileRep,int loopDepth) 
	{
															#line 676 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (fileRep.isError())
															#line 5907 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 677 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			return a_body1Catch2(fileRep.getError(), loopDepth);
															#line 5911 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
															#line 679 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(!fileRep.get().chunks.empty());
															#line 681 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryBlobGranuleRequestActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryBlobGranuleRequestActorState(); static_cast<TryBlobGranuleRequestActor*>(this)->destroy(); return 0; }
															#line 5917 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryBlobGranuleRequestActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryBlobGranuleRequestActorState();
		static_cast<TryBlobGranuleRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(ErrorOr<BlobGranuleFileReply> const& fileRep,int loopDepth) 
	{
		loopDepth = a_body1cont3(fileRep, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(ErrorOr<BlobGranuleFileReply> && fileRep,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(fileRep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state > 0) static_cast<TryBlobGranuleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<TryBlobGranuleRequestActor*>(this)->ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >::remove();

	}
	void a_callback_fire(ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >*,ErrorOr<BlobGranuleFileReply> const& value) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >*,ErrorOr<BlobGranuleFileReply> && value) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >*,Error err) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Reference<Tenant> tenant;
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	WipedString locToken;
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	WipedString reqToken;
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 6012 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via tryBlobGranuleRequest()
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryBlobGranuleRequestActor final : public Actor<Optional<Error>>, public ActorCallback< TryBlobGranuleRequestActor, 0, ErrorOr<GetBlobGranuleLocationsReply> >, public ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >, public FastAllocated<TryBlobGranuleRequestActor>, public TryBlobGranuleRequestActorState<TryBlobGranuleRequestActor> {
															#line 6017 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TryBlobGranuleRequestActor>::operator new;
	using FastAllocated<TryBlobGranuleRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9588914846262183168UL, 8295909306021442304UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Error>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TryBlobGranuleRequestActor, 0, ErrorOr<GetBlobGranuleLocationsReply> >;
friend struct ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >;
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryBlobGranuleRequestActor(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,WipedString const& locToken,WipedString const& reqToken,Version const& committedVersion) 
															#line 6035 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Optional<Error>>(),
		   TryBlobGranuleRequestActorState<TryBlobGranuleRequestActor>(self, cx, tenant, locToken, reqToken, committedVersion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6189181962894187264UL, 10417441849823724032UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tryBlobGranuleRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tryBlobGranuleRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TryBlobGranuleRequestActor, 0, ErrorOr<GetBlobGranuleLocationsReply> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TryBlobGranuleRequestActor, 1, ErrorOr<BlobGranuleFileReply> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Optional<Error>> tryBlobGranuleRequest( AuthzSecurityWorkload* const& self, Database const& cx, Reference<Tenant> const& tenant, WipedString const& locToken, WipedString const& reqToken, Version const& committedVersion ) {
															#line 637 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Optional<Error>>(new TryBlobGranuleRequestActor(self, cx, tenant, locToken, reqToken, committedVersion));
															#line 6068 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 688 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 6073 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via tryBlobGranuleRead()
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TryBlobGranuleReadActor>
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryBlobGranuleReadActorState {
															#line 6079 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryBlobGranuleReadActorState(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,Key const& key,WipedString const& token,Version const& committedVersion) 
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tenant(tenant),
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(key),
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   token(token),
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   committedVersion(committedVersion),
															#line 695 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tr(cx, tenant)
															#line 6098 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this));

	}
	~TryBlobGranuleReadActorState() 
	{
		fdb_probe_actor_destroy("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 696 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			self->setAuthToken(tr, token);
															#line 697 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			KeyRange range(KeyRangeRef(key, keyAfter(key)));
															#line 6115 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			try {
															#line 699 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				StrictFuture<Void> __when_expr_0 = success(tr.readBlobGranules(range, 0, committedVersion));
															#line 699 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				if (static_cast<TryBlobGranuleReadActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6121 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TryBlobGranuleReadActor*>(this)->actor_wait_state = 1;
															#line 699 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TryBlobGranuleReadActor, 0, Void >*>(static_cast<TryBlobGranuleReadActor*>(this)));
															#line 6126 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TryBlobGranuleReadActorState();
		static_cast<TryBlobGranuleReadActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 702 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			CODE_PROBE(e.code() == error_code_permission_denied, "Cross tenant blob granule read meets permission_denied");
															#line 704 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!static_cast<TryBlobGranuleReadActor*>(this)->SAV<Optional<Error>>::futures) { (void)(e); this->~TryBlobGranuleReadActorState(); static_cast<TryBlobGranuleReadActor*>(this)->destroy(); return 0; }
															#line 6158 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			new (&static_cast<TryBlobGranuleReadActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(e);
			this->~TryBlobGranuleReadActorState();
			static_cast<TryBlobGranuleReadActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 700 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryBlobGranuleReadActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryBlobGranuleReadActorState(); static_cast<TryBlobGranuleReadActor*>(this)->destroy(); return 0; }
															#line 6176 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryBlobGranuleReadActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryBlobGranuleReadActorState();
		static_cast<TryBlobGranuleReadActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 700 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryBlobGranuleReadActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryBlobGranuleReadActorState(); static_cast<TryBlobGranuleReadActor*>(this)->destroy(); return 0; }
															#line 6188 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryBlobGranuleReadActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryBlobGranuleReadActorState();
		static_cast<TryBlobGranuleReadActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TryBlobGranuleReadActor*>(this)->actor_wait_state > 0) static_cast<TryBlobGranuleReadActor*>(this)->actor_wait_state = 0;
		static_cast<TryBlobGranuleReadActor*>(this)->ActorCallback< TryBlobGranuleReadActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TryBlobGranuleReadActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TryBlobGranuleReadActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TryBlobGranuleReadActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Reference<Tenant> tenant;
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	WipedString token;
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 695 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Transaction tr;
															#line 6285 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via tryBlobGranuleRead()
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryBlobGranuleReadActor final : public Actor<Optional<Error>>, public ActorCallback< TryBlobGranuleReadActor, 0, Void >, public FastAllocated<TryBlobGranuleReadActor>, public TryBlobGranuleReadActorState<TryBlobGranuleReadActor> {
															#line 6290 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TryBlobGranuleReadActor>::operator new;
	using FastAllocated<TryBlobGranuleReadActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2228673044320490496UL, 15830880124850703872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Error>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TryBlobGranuleReadActor, 0, Void >;
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryBlobGranuleReadActor(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,Key const& key,WipedString const& token,Version const& committedVersion) 
															#line 6307 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Optional<Error>>(),
		   TryBlobGranuleReadActorState<TryBlobGranuleReadActor>(self, cx, tenant, key, token, committedVersion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8721091505943554816UL, 6875589684655197440UL);
		ActorExecutionContextHelper __helper(static_cast<TryBlobGranuleReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tryBlobGranuleRead");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tryBlobGranuleRead", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TryBlobGranuleReadActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Optional<Error>> tryBlobGranuleRead( AuthzSecurityWorkload* const& self, Database const& cx, Reference<Tenant> const& tenant, Key const& key, WipedString const& token, Version const& committedVersion ) {
															#line 689 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Optional<Error>>(new TryBlobGranuleReadActor(self, cx, tenant, key, token, committedVersion));
															#line 6339 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 707 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

	static void checkCrossTenantOutcome(std::string testcase,
	                                    PerfIntCounter& positiveCounter,
	                                    PerfIntCounter& negativeCounter,
	                                    Optional<Error> outcome,
	                                    PositiveTestcase positive) {
		if (positive) {
			// Supposed to succeed. Expected to occasionally fail because of buggify, faultInjection, or data
			// distribution, but should not return permission_denied
			if (!outcome.present()) {
				++positiveCounter;
			} else if (outcome.get().code() == error_code_permission_denied) {
				TraceEvent(SevError, "AuthzSecurityError")
				    .detail("Case", "CrossTenant" + testcase + "Disallowed")
				    .detail("Subcase", "Positive")
				    .log();
			}
		} else {
			// Should always fail. Expected to return permission_denied, but expected to occasionally fail with
			// different errors
			if (!outcome.present()) {
				TraceEvent(SevError, "AuthzSecurityError")
				    .detail("Case", "CrossTenant" + testcase + "Disallowed")
				    .detail("Subcase", "Negative")
				    .log();
			} else if (outcome.get().code() == error_code_permission_denied) {
				++negativeCounter;
			}
		}
	}

																#line 6374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testCrossTenantBGLocDisallowed()
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestCrossTenantBGLocDisallowedActor>
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantBGLocDisallowedActorState {
															#line 6380 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantBGLocDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   positive(positive),
															#line 741 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 742 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString())
															#line 6395 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestCrossTenantBGLocDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 743 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Version> __when_expr_0 = setAndCommitKeyValueAndGetVersion(self, cx, self->tenant, self->signedToken, key, value);
															#line 743 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6412 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->actor_wait_state = 1;
															#line 743 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantBGLocDisallowedActor, 0, Version >*>(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)));
															#line 6417 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestCrossTenantBGLocDisallowedActorState();
		static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 745 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		TenantInfo tenantInfo(self->tenant->id(), positive ? self->signedToken : self->signedTokenAnotherTenant);
															#line 746 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<ErrorOr<GetBlobGranuleLocationsReply>> __when_expr_1 = getGranuleLocations(self, cx, tenantInfo, committedVersion);
															#line 746 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6444 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->actor_wait_state = 2;
															#line 746 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantBGLocDisallowedActor, 1, ErrorOr<GetBlobGranuleLocationsReply> >*>(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)));
															#line 6449 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Version const& __committedVersion,int loopDepth) 
	{
															#line 743 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		committedVersion = __committedVersion;
															#line 6458 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Version && __committedVersion,int loopDepth) 
	{
		committedVersion = std::move(__committedVersion);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->ActorCallback< TestCrossTenantBGLocDisallowedActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGLocDisallowedActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGLocDisallowedActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestCrossTenantBGLocDisallowedActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(ErrorOr<GetBlobGranuleLocationsReply> const& rep,int loopDepth) 
	{
															#line 747 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		Optional<Error> outcome = rep.isError() ? rep.getError() : Optional<Error>();
															#line 748 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		checkCrossTenantOutcome( "BGLoc", self->crossTenantBGLocPositive, self->crossTenantBGLocNegative, outcome, positive);
															#line 750 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantBGLocDisallowedActorState(); static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->destroy(); return 0; }
															#line 6541 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantBGLocDisallowedActorState();
		static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(ErrorOr<GetBlobGranuleLocationsReply> && rep,int loopDepth) 
	{
															#line 747 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		Optional<Error> outcome = rep.isError() ? rep.getError() : Optional<Error>();
															#line 748 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		checkCrossTenantOutcome( "BGLoc", self->crossTenantBGLocPositive, self->crossTenantBGLocNegative, outcome, positive);
															#line 750 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantBGLocDisallowedActorState(); static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->destroy(); return 0; }
															#line 6557 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantBGLocDisallowedActorState();
		static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(ErrorOr<GetBlobGranuleLocationsReply> const& rep,int loopDepth) 
	{
		loopDepth = a_body1cont2(rep, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(ErrorOr<GetBlobGranuleLocationsReply> && rep,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->ActorCallback< TestCrossTenantBGLocDisallowedActor, 1, ErrorOr<GetBlobGranuleLocationsReply> >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGLocDisallowedActor, 1, ErrorOr<GetBlobGranuleLocationsReply> >*,ErrorOr<GetBlobGranuleLocationsReply> const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGLocDisallowedActor, 1, ErrorOr<GetBlobGranuleLocationsReply> >*,ErrorOr<GetBlobGranuleLocationsReply> && value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestCrossTenantBGLocDisallowedActor, 1, ErrorOr<GetBlobGranuleLocationsReply> >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	PositiveTestcase positive;
															#line 741 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 742 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 743 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 6652 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testCrossTenantBGLocDisallowed()
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantBGLocDisallowedActor final : public Actor<Void>, public ActorCallback< TestCrossTenantBGLocDisallowedActor, 0, Version >, public ActorCallback< TestCrossTenantBGLocDisallowedActor, 1, ErrorOr<GetBlobGranuleLocationsReply> >, public FastAllocated<TestCrossTenantBGLocDisallowedActor>, public TestCrossTenantBGLocDisallowedActorState<TestCrossTenantBGLocDisallowedActor> {
															#line 6657 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestCrossTenantBGLocDisallowedActor>::operator new;
	using FastAllocated<TestCrossTenantBGLocDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18072098383265936896UL, 13405428852143800064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestCrossTenantBGLocDisallowedActor, 0, Version >;
friend struct ActorCallback< TestCrossTenantBGLocDisallowedActor, 1, ErrorOr<GetBlobGranuleLocationsReply> >;
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantBGLocDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 6675 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestCrossTenantBGLocDisallowedActorState<TestCrossTenantBGLocDisallowedActor>(self, cx, positive),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18046329005639638784UL, 8735862984854389504UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGLocDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testCrossTenantBGLocDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testCrossTenantBGLocDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestCrossTenantBGLocDisallowedActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestCrossTenantBGLocDisallowedActor, 1, ErrorOr<GetBlobGranuleLocationsReply> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testCrossTenantBGLocDisallowed( AuthzSecurityWorkload* const& self, Database const& cx, PositiveTestcase const& positive ) {
															#line 738 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestCrossTenantBGLocDisallowedActor(self, cx, positive));
															#line 6708 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 752 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 6713 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testCrossTenantBGRequestDisallowed()
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestCrossTenantBGRequestDisallowedActor>
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantBGRequestDisallowedActorState {
															#line 6719 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantBGRequestDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   positive(positive),
															#line 756 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 757 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString())
															#line 6734 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestCrossTenantBGRequestDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 758 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Version> __when_expr_0 = setAndCommitKeyValueAndGetVersion(self, cx, self->tenant, self->signedToken, key, value);
															#line 758 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6751 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->actor_wait_state = 1;
															#line 758 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantBGRequestDisallowedActor, 0, Version >*>(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)));
															#line 6756 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestCrossTenantBGRequestDisallowedActorState();
		static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 760 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Optional<Error>> __when_expr_1 = tryBlobGranuleRequest(self, cx, self->tenant, self->signedToken, positive ? self->signedToken : self->signedTokenAnotherTenant, committedVersion);
															#line 760 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6781 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->actor_wait_state = 2;
															#line 760 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantBGRequestDisallowedActor, 1, Optional<Error> >*>(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)));
															#line 6786 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Version const& __committedVersion,int loopDepth) 
	{
															#line 758 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		committedVersion = __committedVersion;
															#line 6795 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Version && __committedVersion,int loopDepth) 
	{
		committedVersion = std::move(__committedVersion);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->ActorCallback< TestCrossTenantBGRequestDisallowedActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGRequestDisallowedActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGRequestDisallowedActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestCrossTenantBGRequestDisallowedActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Optional<Error> const& outcome,int loopDepth) 
	{
															#line 767 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		checkCrossTenantOutcome( "BGRequest", self->crossTenantBGReqPositive, self->crossTenantBGReqNegative, outcome, positive);
															#line 769 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantBGRequestDisallowedActorState(); static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->destroy(); return 0; }
															#line 6876 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantBGRequestDisallowedActorState();
		static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<Error> && outcome,int loopDepth) 
	{
															#line 767 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		checkCrossTenantOutcome( "BGRequest", self->crossTenantBGReqPositive, self->crossTenantBGReqNegative, outcome, positive);
															#line 769 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantBGRequestDisallowedActorState(); static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->destroy(); return 0; }
															#line 6890 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantBGRequestDisallowedActorState();
		static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Error> const& outcome,int loopDepth) 
	{
		loopDepth = a_body1cont2(outcome, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Error> && outcome,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(outcome), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->ActorCallback< TestCrossTenantBGRequestDisallowedActor, 1, Optional<Error> >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGRequestDisallowedActor, 1, Optional<Error> >*,Optional<Error> const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGRequestDisallowedActor, 1, Optional<Error> >*,Optional<Error> && value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestCrossTenantBGRequestDisallowedActor, 1, Optional<Error> >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	PositiveTestcase positive;
															#line 756 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 757 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 758 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 6985 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testCrossTenantBGRequestDisallowed()
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantBGRequestDisallowedActor final : public Actor<Void>, public ActorCallback< TestCrossTenantBGRequestDisallowedActor, 0, Version >, public ActorCallback< TestCrossTenantBGRequestDisallowedActor, 1, Optional<Error> >, public FastAllocated<TestCrossTenantBGRequestDisallowedActor>, public TestCrossTenantBGRequestDisallowedActorState<TestCrossTenantBGRequestDisallowedActor> {
															#line 6990 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestCrossTenantBGRequestDisallowedActor>::operator new;
	using FastAllocated<TestCrossTenantBGRequestDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7695445243524879360UL, 16715655586179681792UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestCrossTenantBGRequestDisallowedActor, 0, Version >;
friend struct ActorCallback< TestCrossTenantBGRequestDisallowedActor, 1, Optional<Error> >;
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantBGRequestDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 7008 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestCrossTenantBGRequestDisallowedActorState<TestCrossTenantBGRequestDisallowedActor>(self, cx, positive),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15705087106959557888UL, 5519479006261502976UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGRequestDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testCrossTenantBGRequestDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testCrossTenantBGRequestDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestCrossTenantBGRequestDisallowedActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestCrossTenantBGRequestDisallowedActor, 1, Optional<Error> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testCrossTenantBGRequestDisallowed( AuthzSecurityWorkload* const& self, Database const& cx, PositiveTestcase const& positive ) {
															#line 753 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestCrossTenantBGRequestDisallowedActor(self, cx, positive));
															#line 7041 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 771 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 7046 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testCrossTenantBGReadDisallowed()
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestCrossTenantBGReadDisallowedActor>
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantBGReadDisallowedActorState {
															#line 7052 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantBGReadDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   positive(positive),
															#line 775 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 776 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString())
															#line 7067 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestCrossTenantBGReadDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 777 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Version> __when_expr_0 = setAndCommitKeyValueAndGetVersion(self, cx, self->tenant, self->signedToken, key, value);
															#line 777 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7084 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->actor_wait_state = 1;
															#line 777 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantBGReadDisallowedActor, 0, Version >*>(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)));
															#line 7089 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestCrossTenantBGReadDisallowedActorState();
		static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 779 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Optional<Error>> __when_expr_1 = tryBlobGranuleRead(self, cx, self->tenant, key, positive ? self->signedToken : self->signedTokenAnotherTenant, committedVersion);
															#line 779 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7114 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->actor_wait_state = 2;
															#line 779 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantBGReadDisallowedActor, 1, Optional<Error> >*>(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)));
															#line 7119 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Version const& __committedVersion,int loopDepth) 
	{
															#line 777 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		committedVersion = __committedVersion;
															#line 7128 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Version && __committedVersion,int loopDepth) 
	{
		committedVersion = std::move(__committedVersion);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->ActorCallback< TestCrossTenantBGReadDisallowedActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGReadDisallowedActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGReadDisallowedActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestCrossTenantBGReadDisallowedActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Optional<Error> const& outcome,int loopDepth) 
	{
															#line 785 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		checkCrossTenantOutcome( "BGRead", self->crossTenantBGReadPositive, self->crossTenantBGReadNegative, outcome, positive);
															#line 787 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantBGReadDisallowedActorState(); static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->destroy(); return 0; }
															#line 7209 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantBGReadDisallowedActorState();
		static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<Error> && outcome,int loopDepth) 
	{
															#line 785 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		checkCrossTenantOutcome( "BGRead", self->crossTenantBGReadPositive, self->crossTenantBGReadNegative, outcome, positive);
															#line 787 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantBGReadDisallowedActorState(); static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->destroy(); return 0; }
															#line 7223 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantBGReadDisallowedActorState();
		static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Error> const& outcome,int loopDepth) 
	{
		loopDepth = a_body1cont2(outcome, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Error> && outcome,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(outcome), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->ActorCallback< TestCrossTenantBGReadDisallowedActor, 1, Optional<Error> >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGReadDisallowedActor, 1, Optional<Error> >*,Optional<Error> const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantBGReadDisallowedActor, 1, Optional<Error> >*,Optional<Error> && value) 
	{
		fdb_probe_actor_enter("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestCrossTenantBGReadDisallowedActor, 1, Optional<Error> >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	PositiveTestcase positive;
															#line 775 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 776 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 777 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Version committedVersion;
															#line 7318 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testCrossTenantBGReadDisallowed()
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantBGReadDisallowedActor final : public Actor<Void>, public ActorCallback< TestCrossTenantBGReadDisallowedActor, 0, Version >, public ActorCallback< TestCrossTenantBGReadDisallowedActor, 1, Optional<Error> >, public FastAllocated<TestCrossTenantBGReadDisallowedActor>, public TestCrossTenantBGReadDisallowedActorState<TestCrossTenantBGReadDisallowedActor> {
															#line 7323 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestCrossTenantBGReadDisallowedActor>::operator new;
	using FastAllocated<TestCrossTenantBGReadDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10130962909670040576UL, 266998849259613184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestCrossTenantBGReadDisallowedActor, 0, Version >;
friend struct ActorCallback< TestCrossTenantBGReadDisallowedActor, 1, Optional<Error> >;
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantBGReadDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 7341 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestCrossTenantBGReadDisallowedActorState<TestCrossTenantBGReadDisallowedActor>(self, cx, positive),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17759322379277882368UL, 10151089882934257408UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantBGReadDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testCrossTenantBGReadDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testCrossTenantBGReadDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestCrossTenantBGReadDisallowedActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestCrossTenantBGReadDisallowedActor, 1, Optional<Error> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testCrossTenantBGReadDisallowed( AuthzSecurityWorkload* const& self, Database const& cx, PositiveTestcase const& positive ) {
															#line 772 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestCrossTenantBGReadDisallowedActor(self, cx, positive));
															#line 7374 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 789 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 7379 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via tryGetGranules()
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TryGetGranulesActor>
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryGetGranulesActorState {
															#line 7385 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryGetGranulesActorState(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,Key const& key,WipedString const& token) 
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tenant(tenant),
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(key),
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   token(token),
															#line 795 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   tr(cx, tenant)
															#line 7402 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("tryGetGranules", reinterpret_cast<unsigned long>(this));

	}
	~TryGetGranulesActorState() 
	{
		fdb_probe_actor_destroy("tryGetGranules", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 796 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			self->setAuthToken(tr, token);
															#line 797 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			KeyRange range(KeyRangeRef(key, keyAfter(key)));
															#line 7419 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			try {
															#line 799 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				StrictFuture<Standalone<VectorRef<KeyRangeRef>>> __when_expr_0 = tr.getBlobGranuleRanges(range, 1000);
															#line 799 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				if (static_cast<TryGetGranulesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7425 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TryGetGranulesActor*>(this)->actor_wait_state = 1;
															#line 799 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TryGetGranulesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*>(static_cast<TryGetGranulesActor*>(this)));
															#line 7430 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TryGetGranulesActorState();
		static_cast<TryGetGranulesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 807 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			CODE_PROBE(e.code() == error_code_permission_denied, "Cross tenant get granules meets permission_denied");
															#line 808 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (!static_cast<TryGetGranulesActor*>(this)->SAV<Optional<Error>>::futures) { (void)(e); this->~TryGetGranulesActorState(); static_cast<TryGetGranulesActor*>(this)->destroy(); return 0; }
															#line 7462 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			new (&static_cast<TryGetGranulesActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(e);
			this->~TryGetGranulesActorState();
			static_cast<TryGetGranulesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Standalone<VectorRef<KeyRangeRef>> const& result,int loopDepth) 
	{
															#line 801 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(result.size() <= 1);
															#line 802 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!result.empty())
															#line 7482 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 803 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			ASSERT(result[0].contains(key));
															#line 7486 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
															#line 805 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryGetGranulesActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryGetGranulesActorState(); static_cast<TryGetGranulesActor*>(this)->destroy(); return 0; }
															#line 7490 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryGetGranulesActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryGetGranulesActorState();
		static_cast<TryGetGranulesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Standalone<VectorRef<KeyRangeRef>> && result,int loopDepth) 
	{
															#line 801 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(result.size() <= 1);
															#line 802 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!result.empty())
															#line 7504 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		{
															#line 803 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			ASSERT(result[0].contains(key));
															#line 7508 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
															#line 805 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TryGetGranulesActor*>(this)->SAV<Optional<Error>>::futures) { (void)(Optional<Error>()); this->~TryGetGranulesActorState(); static_cast<TryGetGranulesActor*>(this)->destroy(); return 0; }
															#line 7512 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TryGetGranulesActor*>(this)->SAV< Optional<Error> >::value()) Optional<Error>(Optional<Error>());
		this->~TryGetGranulesActorState();
		static_cast<TryGetGranulesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Standalone<VectorRef<KeyRangeRef>> const& result,int loopDepth) 
	{
		loopDepth = a_body1cont2(result, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Standalone<VectorRef<KeyRangeRef>> && result,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TryGetGranulesActor*>(this)->actor_wait_state > 0) static_cast<TryGetGranulesActor*>(this)->actor_wait_state = 0;
		static_cast<TryGetGranulesActor*>(this)->ActorCallback< TryGetGranulesActor, 0, Standalone<VectorRef<KeyRangeRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< TryGetGranulesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*,Standalone<VectorRef<KeyRangeRef>> const& value) 
	{
		fdb_probe_actor_enter("tryGetGranules", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryGetGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryGetGranules", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TryGetGranulesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*,Standalone<VectorRef<KeyRangeRef>> && value) 
	{
		fdb_probe_actor_enter("tryGetGranules", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryGetGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryGetGranules", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TryGetGranulesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("tryGetGranules", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryGetGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryGetGranules", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Reference<Tenant> tenant;
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	WipedString token;
															#line 795 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Transaction tr;
															#line 7607 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via tryGetGranules()
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TryGetGranulesActor final : public Actor<Optional<Error>>, public ActorCallback< TryGetGranulesActor, 0, Standalone<VectorRef<KeyRangeRef>> >, public FastAllocated<TryGetGranulesActor>, public TryGetGranulesActorState<TryGetGranulesActor> {
															#line 7612 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TryGetGranulesActor>::operator new;
	using FastAllocated<TryGetGranulesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12832120108903371776UL, 4217179510448864000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Error>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TryGetGranulesActor, 0, Standalone<VectorRef<KeyRangeRef>> >;
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TryGetGranulesActor(AuthzSecurityWorkload* const& self,Database const& cx,Reference<Tenant> const& tenant,Key const& key,WipedString const& token) 
															#line 7629 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Optional<Error>>(),
		   TryGetGranulesActorState<TryGetGranulesActor>(self, cx, tenant, key, token),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tryGetGranules", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4534066620853387264UL, 6857905859765023744UL);
		ActorExecutionContextHelper __helper(static_cast<TryGetGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tryGetGranules");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tryGetGranules", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TryGetGranulesActor, 0, Standalone<VectorRef<KeyRangeRef>> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Optional<Error>> tryGetGranules( AuthzSecurityWorkload* const& self, Database const& cx, Reference<Tenant> const& tenant, Key const& key, WipedString const& token ) {
															#line 790 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Optional<Error>>(new TryGetGranulesActor(self, cx, tenant, key, token));
															#line 7661 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 811 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 7666 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testCrossTenantGetGranulesDisallowed()
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestCrossTenantGetGranulesDisallowedActor>
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantGetGranulesDisallowedActorState {
															#line 7672 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantGetGranulesDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   positive(positive),
															#line 815 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   key(self->randomString()),
															#line 816 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   value(self->randomString())
															#line 7687 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestCrossTenantGetGranulesDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 817 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Optional<Error>> __when_expr_0 = tryGetGranules(self, cx, self->tenant, key, positive ? self->signedToken : self->signedTokenAnotherTenant);
															#line 817 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7704 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->actor_wait_state = 1;
															#line 817 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestCrossTenantGetGranulesDisallowedActor, 0, Optional<Error> >*>(static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)));
															#line 7709 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestCrossTenantGetGranulesDisallowedActorState();
		static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Error> const& outcome,int loopDepth) 
	{
															#line 819 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		checkCrossTenantOutcome("GetGranules", self->crossTenantGetGranulesPositive, self->crossTenantGetGranulesNegative, outcome, positive);
															#line 824 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantGetGranulesDisallowedActorState(); static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->destroy(); return 0; }
															#line 7734 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantGetGranulesDisallowedActorState();
		static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Error> && outcome,int loopDepth) 
	{
															#line 819 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		checkCrossTenantOutcome("GetGranules", self->crossTenantGetGranulesPositive, self->crossTenantGetGranulesNegative, outcome, positive);
															#line 824 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestCrossTenantGetGranulesDisallowedActorState(); static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->destroy(); return 0; }
															#line 7748 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestCrossTenantGetGranulesDisallowedActorState();
		static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Error> const& outcome,int loopDepth) 
	{
		loopDepth = a_body1cont1(outcome, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Error> && outcome,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(outcome), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->ActorCallback< TestCrossTenantGetGranulesDisallowedActor, 0, Optional<Error> >::remove();

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetGranulesDisallowedActor, 0, Optional<Error> >*,Optional<Error> const& value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestCrossTenantGetGranulesDisallowedActor, 0, Optional<Error> >*,Optional<Error> && value) 
	{
		fdb_probe_actor_enter("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestCrossTenantGetGranulesDisallowedActor, 0, Optional<Error> >*,Error err) 
	{
		fdb_probe_actor_enter("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	PositiveTestcase positive;
															#line 815 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Key key;
															#line 816 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Value value;
															#line 7841 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testCrossTenantGetGranulesDisallowed()
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestCrossTenantGetGranulesDisallowedActor final : public Actor<Void>, public ActorCallback< TestCrossTenantGetGranulesDisallowedActor, 0, Optional<Error> >, public FastAllocated<TestCrossTenantGetGranulesDisallowedActor>, public TestCrossTenantGetGranulesDisallowedActorState<TestCrossTenantGetGranulesDisallowedActor> {
															#line 7846 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestCrossTenantGetGranulesDisallowedActor>::operator new;
	using FastAllocated<TestCrossTenantGetGranulesDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16355929443948372480UL, 10699555915425812224UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestCrossTenantGetGranulesDisallowedActor, 0, Optional<Error> >;
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestCrossTenantGetGranulesDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx,PositiveTestcase const& positive) 
															#line 7863 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestCrossTenantGetGranulesDisallowedActorState<TestCrossTenantGetGranulesDisallowedActor>(self, cx, positive),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3504735454169136896UL, 16347380847100728320UL);
		ActorExecutionContextHelper __helper(static_cast<TestCrossTenantGetGranulesDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testCrossTenantGetGranulesDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testCrossTenantGetGranulesDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestCrossTenantGetGranulesDisallowedActor, 0, Optional<Error> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testCrossTenantGetGranulesDisallowed( AuthzSecurityWorkload* const& self, Database const& cx, PositiveTestcase const& positive ) {
															#line 812 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestCrossTenantGetGranulesDisallowedActor(self, cx, positive));
															#line 7895 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 826 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 7900 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via checkBlobManagementNegative()
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class CheckBlobManagementNegativeActor>
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class CheckBlobManagementNegativeActorState {
															#line 7906 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	CheckBlobManagementNegativeActorState(AuthzSecurityWorkload* const& self,std::string const& opName,Future<Void> const& op,PerfIntCounter* const& counter) 
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   opName(opName),
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   op(op),
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   counter(counter)
															#line 7919 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this));

	}
	~CheckBlobManagementNegativeActorState() 
	{
		fdb_probe_actor_destroy("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 832 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				StrictFuture<Void> __when_expr_0 = op;
															#line 832 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				if (static_cast<CheckBlobManagementNegativeActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7937 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<CheckBlobManagementNegativeActor*>(this)->actor_wait_state = 1;
															#line 832 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckBlobManagementNegativeActor, 0, Void >*>(static_cast<CheckBlobManagementNegativeActor*>(this)));
															#line 7942 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckBlobManagementNegativeActorState();
		static_cast<CheckBlobManagementNegativeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 846 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<CheckBlobManagementNegativeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckBlobManagementNegativeActorState(); static_cast<CheckBlobManagementNegativeActor*>(this)->destroy(); return 0; }
															#line 7971 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<CheckBlobManagementNegativeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckBlobManagementNegativeActorState();
		static_cast<CheckBlobManagementNegativeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 835 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 7984 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 836 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 7988 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 838 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (e.code() == error_code_permission_denied)
															#line 7992 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 839 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				++(*counter);
															#line 7996 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
			else
			{
															#line 841 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				TraceEvent(SevError, "AuthzSecurityBlobManagementAllowed") .detail("OpType", opName) .detail("TenantId", self->tenant->id());
															#line 8002 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 833 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(false);
															#line 8018 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 833 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		ASSERT(false);
															#line 8027 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckBlobManagementNegativeActor*>(this)->actor_wait_state > 0) static_cast<CheckBlobManagementNegativeActor*>(this)->actor_wait_state = 0;
		static_cast<CheckBlobManagementNegativeActor*>(this)->ActorCallback< CheckBlobManagementNegativeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckBlobManagementNegativeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBlobManagementNegativeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckBlobManagementNegativeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBlobManagementNegativeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckBlobManagementNegativeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBlobManagementNegativeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	std::string opName;
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Future<Void> op;
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	PerfIntCounter* counter;
															#line 8128 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via checkBlobManagementNegative()
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class CheckBlobManagementNegativeActor final : public Actor<Void>, public ActorCallback< CheckBlobManagementNegativeActor, 0, Void >, public FastAllocated<CheckBlobManagementNegativeActor>, public CheckBlobManagementNegativeActorState<CheckBlobManagementNegativeActor> {
															#line 8133 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<CheckBlobManagementNegativeActor>::operator new;
	using FastAllocated<CheckBlobManagementNegativeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(395001561854029056UL, 1439452588984143104UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckBlobManagementNegativeActor, 0, Void >;
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	CheckBlobManagementNegativeActor(AuthzSecurityWorkload* const& self,std::string const& opName,Future<Void> const& op,PerfIntCounter* const& counter) 
															#line 8150 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   CheckBlobManagementNegativeActorState<CheckBlobManagementNegativeActor>(self, opName, op, counter),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12583280040637808896UL, 15576375202023826688UL);
		ActorExecutionContextHelper __helper(static_cast<CheckBlobManagementNegativeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkBlobManagementNegative");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkBlobManagementNegative", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckBlobManagementNegativeActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> checkBlobManagementNegative( AuthzSecurityWorkload* const& self, std::string const& opName, Future<Void> const& op, PerfIntCounter* const& counter ) {
															#line 827 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new CheckBlobManagementNegativeActor(self, opName, op, counter));
															#line 8182 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 848 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 8187 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testBlobbifyDisallowed()
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestBlobbifyDisallowedActor>
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestBlobbifyDisallowedActorState {
															#line 8193 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestBlobbifyDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx)
															#line 8202 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestBlobbifyDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 850 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			Future<Void> op;
															#line 851 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (deterministicRandom()->coinflip())
															#line 8219 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			{
															#line 852 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				op = success(cx->blobbifyRange(normalKeys, self->tenant));
															#line 8223 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
			else
			{
															#line 854 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				op = success(cx->blobbifyRangeBlocking(normalKeys, self->tenant));
															#line 8229 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			}
															#line 856 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBlobManagementNegative(self, "Blobbify", op, &self->blobbifyNegative);
															#line 856 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestBlobbifyDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8235 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestBlobbifyDisallowedActor*>(this)->actor_wait_state = 1;
															#line 856 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestBlobbifyDisallowedActor, 0, Void >*>(static_cast<TestBlobbifyDisallowedActor*>(this)));
															#line 8240 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestBlobbifyDisallowedActorState();
		static_cast<TestBlobbifyDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 857 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestBlobbifyDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestBlobbifyDisallowedActorState(); static_cast<TestBlobbifyDisallowedActor*>(this)->destroy(); return 0; }
															#line 8263 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestBlobbifyDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestBlobbifyDisallowedActorState();
		static_cast<TestBlobbifyDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 857 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestBlobbifyDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestBlobbifyDisallowedActorState(); static_cast<TestBlobbifyDisallowedActor*>(this)->destroy(); return 0; }
															#line 8275 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestBlobbifyDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestBlobbifyDisallowedActorState();
		static_cast<TestBlobbifyDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestBlobbifyDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestBlobbifyDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestBlobbifyDisallowedActor*>(this)->ActorCallback< TestBlobbifyDisallowedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestBlobbifyDisallowedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobbifyDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestBlobbifyDisallowedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobbifyDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestBlobbifyDisallowedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobbifyDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 8362 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testBlobbifyDisallowed()
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestBlobbifyDisallowedActor final : public Actor<Void>, public ActorCallback< TestBlobbifyDisallowedActor, 0, Void >, public FastAllocated<TestBlobbifyDisallowedActor>, public TestBlobbifyDisallowedActorState<TestBlobbifyDisallowedActor> {
															#line 8367 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestBlobbifyDisallowedActor>::operator new;
	using FastAllocated<TestBlobbifyDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4441226037112875264UL, 3944988200366820608UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestBlobbifyDisallowedActor, 0, Void >;
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestBlobbifyDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 8384 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestBlobbifyDisallowedActorState<TestBlobbifyDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5551264151975452416UL, 11603279331529178368UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobbifyDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testBlobbifyDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testBlobbifyDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestBlobbifyDisallowedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testBlobbifyDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 849 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestBlobbifyDisallowedActor(self, cx));
															#line 8416 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 859 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 8421 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testUnblobbifyDisallowed()
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestUnblobbifyDisallowedActor>
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestUnblobbifyDisallowedActorState {
															#line 8427 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestUnblobbifyDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx)
															#line 8436 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestUnblobbifyDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 861 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBlobManagementNegative( self, "Unblobbify", success(cx->unblobbifyRange(normalKeys, self->tenant)), &self->unblobbifyNegative);
															#line 861 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestUnblobbifyDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8453 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestUnblobbifyDisallowedActor*>(this)->actor_wait_state = 1;
															#line 861 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestUnblobbifyDisallowedActor, 0, Void >*>(static_cast<TestUnblobbifyDisallowedActor*>(this)));
															#line 8458 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestUnblobbifyDisallowedActorState();
		static_cast<TestUnblobbifyDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 863 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestUnblobbifyDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestUnblobbifyDisallowedActorState(); static_cast<TestUnblobbifyDisallowedActor*>(this)->destroy(); return 0; }
															#line 8481 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestUnblobbifyDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestUnblobbifyDisallowedActorState();
		static_cast<TestUnblobbifyDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 863 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestUnblobbifyDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestUnblobbifyDisallowedActorState(); static_cast<TestUnblobbifyDisallowedActor*>(this)->destroy(); return 0; }
															#line 8493 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestUnblobbifyDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestUnblobbifyDisallowedActorState();
		static_cast<TestUnblobbifyDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestUnblobbifyDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestUnblobbifyDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestUnblobbifyDisallowedActor*>(this)->ActorCallback< TestUnblobbifyDisallowedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestUnblobbifyDisallowedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestUnblobbifyDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestUnblobbifyDisallowedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestUnblobbifyDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestUnblobbifyDisallowedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestUnblobbifyDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 8580 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testUnblobbifyDisallowed()
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestUnblobbifyDisallowedActor final : public Actor<Void>, public ActorCallback< TestUnblobbifyDisallowedActor, 0, Void >, public FastAllocated<TestUnblobbifyDisallowedActor>, public TestUnblobbifyDisallowedActorState<TestUnblobbifyDisallowedActor> {
															#line 8585 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestUnblobbifyDisallowedActor>::operator new;
	using FastAllocated<TestUnblobbifyDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17447246993537973248UL, 3866845444609275392UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestUnblobbifyDisallowedActor, 0, Void >;
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestUnblobbifyDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 8602 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestUnblobbifyDisallowedActorState<TestUnblobbifyDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2844293577886192896UL, 2147400212900971008UL);
		ActorExecutionContextHelper __helper(static_cast<TestUnblobbifyDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testUnblobbifyDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testUnblobbifyDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestUnblobbifyDisallowedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testUnblobbifyDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 860 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestUnblobbifyDisallowedActor(self, cx));
															#line 8634 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 865 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 8639 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testListBlobDisallowed()
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestListBlobDisallowedActor>
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestListBlobDisallowedActorState {
															#line 8645 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestListBlobDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx)
															#line 8654 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testListBlobDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestListBlobDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testListBlobDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 867 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBlobManagementNegative(self, "ListBlob", success(cx->listBlobbifiedRanges(normalKeys, 1000, self->tenant)), &self->listBlobNegative);
															#line 867 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestListBlobDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8671 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestListBlobDisallowedActor*>(this)->actor_wait_state = 1;
															#line 867 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestListBlobDisallowedActor, 0, Void >*>(static_cast<TestListBlobDisallowedActor*>(this)));
															#line 8676 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestListBlobDisallowedActorState();
		static_cast<TestListBlobDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 871 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestListBlobDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestListBlobDisallowedActorState(); static_cast<TestListBlobDisallowedActor*>(this)->destroy(); return 0; }
															#line 8699 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestListBlobDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestListBlobDisallowedActorState();
		static_cast<TestListBlobDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 871 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestListBlobDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestListBlobDisallowedActorState(); static_cast<TestListBlobDisallowedActor*>(this)->destroy(); return 0; }
															#line 8711 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestListBlobDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestListBlobDisallowedActorState();
		static_cast<TestListBlobDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestListBlobDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestListBlobDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestListBlobDisallowedActor*>(this)->ActorCallback< TestListBlobDisallowedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestListBlobDisallowedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testListBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestListBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testListBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestListBlobDisallowedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testListBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestListBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testListBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestListBlobDisallowedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testListBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestListBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testListBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 8798 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testListBlobDisallowed()
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestListBlobDisallowedActor final : public Actor<Void>, public ActorCallback< TestListBlobDisallowedActor, 0, Void >, public FastAllocated<TestListBlobDisallowedActor>, public TestListBlobDisallowedActorState<TestListBlobDisallowedActor> {
															#line 8803 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestListBlobDisallowedActor>::operator new;
	using FastAllocated<TestListBlobDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5409137454864262656UL, 2607432886329045248UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestListBlobDisallowedActor, 0, Void >;
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestListBlobDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 8820 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestListBlobDisallowedActorState<TestListBlobDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testListBlobDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16295516239261100288UL, 3529313529806670848UL);
		ActorExecutionContextHelper __helper(static_cast<TestListBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testListBlobDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testListBlobDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestListBlobDisallowedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testListBlobDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 866 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestListBlobDisallowedActor(self, cx));
															#line 8852 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 873 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 8857 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testVerifyBlobDisallowed()
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestVerifyBlobDisallowedActor>
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestVerifyBlobDisallowedActorState {
															#line 8863 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestVerifyBlobDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx)
															#line 8872 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestVerifyBlobDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 875 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBlobManagementNegative( self, "VerifyBlob", success(cx->verifyBlobRange(normalKeys, {}, self->tenant)), &self->verifyBlobNegative);
															#line 875 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestVerifyBlobDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestVerifyBlobDisallowedActor*>(this)->actor_wait_state = 1;
															#line 875 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestVerifyBlobDisallowedActor, 0, Void >*>(static_cast<TestVerifyBlobDisallowedActor*>(this)));
															#line 8894 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestVerifyBlobDisallowedActorState();
		static_cast<TestVerifyBlobDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 877 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestVerifyBlobDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestVerifyBlobDisallowedActorState(); static_cast<TestVerifyBlobDisallowedActor*>(this)->destroy(); return 0; }
															#line 8917 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestVerifyBlobDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestVerifyBlobDisallowedActorState();
		static_cast<TestVerifyBlobDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 877 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestVerifyBlobDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestVerifyBlobDisallowedActorState(); static_cast<TestVerifyBlobDisallowedActor*>(this)->destroy(); return 0; }
															#line 8929 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestVerifyBlobDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestVerifyBlobDisallowedActorState();
		static_cast<TestVerifyBlobDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestVerifyBlobDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestVerifyBlobDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestVerifyBlobDisallowedActor*>(this)->ActorCallback< TestVerifyBlobDisallowedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestVerifyBlobDisallowedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestVerifyBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestVerifyBlobDisallowedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestVerifyBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestVerifyBlobDisallowedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestVerifyBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 9016 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testVerifyBlobDisallowed()
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestVerifyBlobDisallowedActor final : public Actor<Void>, public ActorCallback< TestVerifyBlobDisallowedActor, 0, Void >, public FastAllocated<TestVerifyBlobDisallowedActor>, public TestVerifyBlobDisallowedActorState<TestVerifyBlobDisallowedActor> {
															#line 9021 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestVerifyBlobDisallowedActor>::operator new;
	using FastAllocated<TestVerifyBlobDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9214741120394285312UL, 11971280494280109312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestVerifyBlobDisallowedActor, 0, Void >;
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestVerifyBlobDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 9038 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestVerifyBlobDisallowedActorState<TestVerifyBlobDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(621197671170830592UL, 9511802508287332352UL);
		ActorExecutionContextHelper __helper(static_cast<TestVerifyBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testVerifyBlobDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testVerifyBlobDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestVerifyBlobDisallowedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testVerifyBlobDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 874 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestVerifyBlobDisallowedActor(self, cx));
															#line 9070 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 879 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 9075 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testFlushBlobDisallowed()
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestFlushBlobDisallowedActor>
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestFlushBlobDisallowedActorState {
															#line 9081 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestFlushBlobDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx)
															#line 9090 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestFlushBlobDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 881 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBlobManagementNegative( self, "FlushBlob", success(cx->flushBlobRange(normalKeys, {}, deterministicRandom()->coinflip(), self->tenant)), &self->flushBlobNegative);
															#line 881 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestFlushBlobDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9107 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestFlushBlobDisallowedActor*>(this)->actor_wait_state = 1;
															#line 881 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestFlushBlobDisallowedActor, 0, Void >*>(static_cast<TestFlushBlobDisallowedActor*>(this)));
															#line 9112 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestFlushBlobDisallowedActorState();
		static_cast<TestFlushBlobDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 886 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestFlushBlobDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestFlushBlobDisallowedActorState(); static_cast<TestFlushBlobDisallowedActor*>(this)->destroy(); return 0; }
															#line 9135 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestFlushBlobDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestFlushBlobDisallowedActorState();
		static_cast<TestFlushBlobDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 886 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestFlushBlobDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestFlushBlobDisallowedActorState(); static_cast<TestFlushBlobDisallowedActor*>(this)->destroy(); return 0; }
															#line 9147 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestFlushBlobDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestFlushBlobDisallowedActorState();
		static_cast<TestFlushBlobDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestFlushBlobDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestFlushBlobDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestFlushBlobDisallowedActor*>(this)->ActorCallback< TestFlushBlobDisallowedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestFlushBlobDisallowedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestFlushBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestFlushBlobDisallowedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestFlushBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestFlushBlobDisallowedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestFlushBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 9234 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testFlushBlobDisallowed()
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestFlushBlobDisallowedActor final : public Actor<Void>, public ActorCallback< TestFlushBlobDisallowedActor, 0, Void >, public FastAllocated<TestFlushBlobDisallowedActor>, public TestFlushBlobDisallowedActorState<TestFlushBlobDisallowedActor> {
															#line 9239 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestFlushBlobDisallowedActor>::operator new;
	using FastAllocated<TestFlushBlobDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10816303406716688896UL, 295143773112144640UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestFlushBlobDisallowedActor, 0, Void >;
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestFlushBlobDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 9256 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestFlushBlobDisallowedActorState<TestFlushBlobDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15995609244438088704UL, 17238092445537812992UL);
		ActorExecutionContextHelper __helper(static_cast<TestFlushBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testFlushBlobDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testFlushBlobDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestFlushBlobDisallowedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testFlushBlobDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 880 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestFlushBlobDisallowedActor(self, cx));
															#line 9288 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 888 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 9293 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via testPurgeBlobDisallowed()
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class TestPurgeBlobDisallowedActor>
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestPurgeBlobDisallowedActorState {
															#line 9299 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestPurgeBlobDisallowedActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx)
															#line 9308 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this));

	}
	~TestPurgeBlobDisallowedActorState() 
	{
		fdb_probe_actor_destroy("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 890 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkBlobManagementNegative( self, "PurgeBlob", success(cx->purgeBlobGranules(normalKeys, 1, self->tenant, deterministicRandom()->coinflip())), &self->purgeBlobNegative);
															#line 890 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			if (static_cast<TestPurgeBlobDisallowedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9325 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestPurgeBlobDisallowedActor*>(this)->actor_wait_state = 1;
															#line 890 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestPurgeBlobDisallowedActor, 0, Void >*>(static_cast<TestPurgeBlobDisallowedActor*>(this)));
															#line 9330 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestPurgeBlobDisallowedActorState();
		static_cast<TestPurgeBlobDisallowedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 895 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestPurgeBlobDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestPurgeBlobDisallowedActorState(); static_cast<TestPurgeBlobDisallowedActor*>(this)->destroy(); return 0; }
															#line 9353 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestPurgeBlobDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestPurgeBlobDisallowedActorState();
		static_cast<TestPurgeBlobDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 895 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (!static_cast<TestPurgeBlobDisallowedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestPurgeBlobDisallowedActorState(); static_cast<TestPurgeBlobDisallowedActor*>(this)->destroy(); return 0; }
															#line 9365 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		new (&static_cast<TestPurgeBlobDisallowedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestPurgeBlobDisallowedActorState();
		static_cast<TestPurgeBlobDisallowedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestPurgeBlobDisallowedActor*>(this)->actor_wait_state > 0) static_cast<TestPurgeBlobDisallowedActor*>(this)->actor_wait_state = 0;
		static_cast<TestPurgeBlobDisallowedActor*>(this)->ActorCallback< TestPurgeBlobDisallowedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestPurgeBlobDisallowedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestPurgeBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestPurgeBlobDisallowedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestPurgeBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestPurgeBlobDisallowedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestPurgeBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 9452 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via testPurgeBlobDisallowed()
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class TestPurgeBlobDisallowedActor final : public Actor<Void>, public ActorCallback< TestPurgeBlobDisallowedActor, 0, Void >, public FastAllocated<TestPurgeBlobDisallowedActor>, public TestPurgeBlobDisallowedActorState<TestPurgeBlobDisallowedActor> {
															#line 9457 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<TestPurgeBlobDisallowedActor>::operator new;
	using FastAllocated<TestPurgeBlobDisallowedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3596542480497768192UL, 6095311943636489472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestPurgeBlobDisallowedActor, 0, Void >;
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	TestPurgeBlobDisallowedActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 9474 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   TestPurgeBlobDisallowedActorState<TestPurgeBlobDisallowedActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4579883313560370944UL, 14442594427480965888UL);
		ActorExecutionContextHelper __helper(static_cast<TestPurgeBlobDisallowedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testPurgeBlobDisallowed");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testPurgeBlobDisallowed", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestPurgeBlobDisallowedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> testPurgeBlobDisallowed( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 889 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new TestPurgeBlobDisallowedActor(self, cx));
															#line 9506 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 897 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"

																#line 9511 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
// This generated class is to be used only via runTestClient()
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
template <class RunTestClientActor>
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class RunTestClientActorState {
															#line 9517 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	RunTestClientActorState(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		 : self(self),
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   cx(cx),
															#line 899 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   lastTime(now()),
															#line 900 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		   delay(self->actorCount / self->transactionsPerSecond)
															#line 9530 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
	{
		fdb_probe_actor_create("runTestClient", reinterpret_cast<unsigned long>(this));

	}
	~RunTestClientActorState() 
	{
		fdb_probe_actor_destroy("runTestClient", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 902 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
				;
															#line 9546 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunTestClientActorState();
		static_cast<RunTestClientActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 907 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			TraceEvent(SevError, "AuthzSecurityClient").error(e);
															#line 908 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 9578 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 903 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Void> __when_expr_0 = poisson(&lastTime, delay);
															#line 903 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<RunTestClientActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9601 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RunTestClientActor*>(this)->actor_wait_state = 1;
															#line 903 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunTestClientActor, 0, Void >*>(static_cast<RunTestClientActor*>(this)));
															#line 9606 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 904 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deterministicRandom()->randomChoice(self->testFunctions)(cx);
															#line 904 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<RunTestClientActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9617 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RunTestClientActor*>(this)->actor_wait_state = 2;
															#line 904 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunTestClientActor, 1, Void >*>(static_cast<RunTestClientActor*>(this)));
															#line 9622 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 904 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deterministicRandom()->randomChoice(self->testFunctions)(cx);
															#line 904 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		if (static_cast<RunTestClientActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9633 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RunTestClientActor*>(this)->actor_wait_state = 2;
															#line 904 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunTestClientActor, 1, Void >*>(static_cast<RunTestClientActor*>(this)));
															#line 9638 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunTestClientActor*>(this)->actor_wait_state > 0) static_cast<RunTestClientActor*>(this)->actor_wait_state = 0;
		static_cast<RunTestClientActor*>(this)->ActorCallback< RunTestClientActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunTestClientActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runTestClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTestClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunTestClientActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runTestClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTestClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunTestClientActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runTestClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTestClient", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RunTestClientActor*>(this)->actor_wait_state > 0) static_cast<RunTestClientActor*>(this)->actor_wait_state = 0;
		static_cast<RunTestClientActor*>(this)->ActorCallback< RunTestClientActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunTestClientActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runTestClient", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTestClient", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RunTestClientActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runTestClient", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTestClient", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RunTestClientActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runTestClient", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runTestClient", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	AuthzSecurityWorkload* self;
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	Database cx;
															#line 899 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	double lastTime;
															#line 900 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	double delay;
															#line 9813 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
};
// This generated class is to be used only via runTestClient()
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
class RunTestClientActor final : public Actor<Void>, public ActorCallback< RunTestClientActor, 0, Void >, public ActorCallback< RunTestClientActor, 1, Void >, public FastAllocated<RunTestClientActor>, public RunTestClientActorState<RunTestClientActor> {
															#line 9818 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
public:
	using FastAllocated<RunTestClientActor>::operator new;
	using FastAllocated<RunTestClientActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4728881515056000768UL, 13107351287866254080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunTestClientActor, 0, Void >;
friend struct ActorCallback< RunTestClientActor, 1, Void >;
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	RunTestClientActor(AuthzSecurityWorkload* const& self,Database const& cx) 
															#line 9836 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
		 : Actor<Void>(),
		   RunTestClientActorState<RunTestClientActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runTestClient", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10875659283561385728UL, 8396279508576182272UL);
		ActorExecutionContextHelper __helper(static_cast<RunTestClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runTestClient");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runTestClient", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunTestClientActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RunTestClientActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
[[nodiscard]] static Future<Void> runTestClient( AuthzSecurityWorkload* const& self, Database const& cx ) {
															#line 898 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
	return Future<Void>(new RunTestClientActor(self, cx));
															#line 9869 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AuthzSecurity.actor.g.cpp"
}

#line 911 "/codebuild/output/src530198565/src/github.com/apple/foundationdb/fdbserver/workloads/AuthzSecurity.actor.cpp"
};

WorkloadFactory<AuthzSecurityWorkload> AuthzSecurityWorkloadFactory(UntrustedMode::True);
