#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
/*
 * RESTSimKmsVault.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/BlobMetadataUtils.h"
#include "fdbserver/RESTSimKmsVault.h"
#include "fdbclient/SimKmsVault.h"
#include "fdbrpc/HTTP.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/RESTKmsConnectorUtils.h"
#include "flow/Arena.h"
#include "flow/EncryptUtils.h"

#include <cstring>
#include <rapidjson/document.h>
#include <rapidjson/rapidjson.h>
#include <rapidjson/stringbuffer.h>
#include <rapidjson/writer.h>

#include "flow/FastRef.h"
#include "flow/IRandom.h"
#include "flow/Knobs.h"
#include "flow/Trace.h"
#include "flow/actorcompiler.h" // This must be the last #include.

using DomIdVec = std::vector<EncryptCipherDomainId>;
using BaseCipherDomIdVec = std::vector<std::pair<EncryptCipherBaseKeyId, Optional<EncryptCipherDomainId>>>;

using namespace RESTKmsConnectorUtils;

namespace {
const std::string missingVersionMsg = "Missing version";
const std::string missingVersionCode = "1234";
const std::string invalidVersionMsg = "Invalid version";
const std::string invalidVersionCode = "5678";
const std::string missingTokensMsg = "Missing validation tokens";
const std::string missingTokenCode = "0123";

const std::string bgUrl = "file://simfdb/fdbblob/";

struct VaultResponse {
	bool failed;
	std::string buff;

	VaultResponse() : failed(false), buff("") {}
};

} // namespace

int64_t getRefreshInterval(const int64_t now, const int64_t defaultTtl) {
	if (BUGGIFY) {
		return now;
	}
	return (now + defaultTtl);
}

int64_t getExpireInterval(const int64_t refTS, const int64_t defaultTtl) {

	if (BUGGIFY) {
		return -1;
	}
	return (refTS + defaultTtl);
}

void validateHeaders(const HTTP::Headers& toCompare) {
	if (toCompare != RESTKmsConnectorUtils::getHTTPHeaders()) {
		throw rest_malformed_response();
	}
}

void addErrorToDoc(rapidjson::Document& doc, const ErrorDetail& details) {
	rapidjson::Value errorDetail(rapidjson::kObjectType);
	if (!details.errorMsg.empty()) {
		// Add "errorMsg"
		rapidjson::Value key(ERROR_MSG_TAG, doc.GetAllocator());
		rapidjson::Value errMsg;
		errMsg.SetString(details.errorMsg.data(), details.errorMsg.size(), doc.GetAllocator());
		errorDetail.AddMember(key, errMsg, doc.GetAllocator());
	}
	if (!details.errorCode.empty()) {
		// Add "value" - token value
		rapidjson::Value key(ERROR_CODE_TAG, doc.GetAllocator());
		rapidjson::Value errCode;
		errCode.SetString(details.errorCode.data(), details.errorCode.size(), doc.GetAllocator());
		errorDetail.AddMember(key, errCode, doc.GetAllocator());
	}

	// Append "error"
	rapidjson::Value key(ERROR_TAG, doc.GetAllocator());
	doc.AddMember(key, errorDetail, doc.GetAllocator());
}

void prepareErrorResponse(VaultResponse* response,
                          const ErrorDetail& errorDetail,
                          Optional<int> version = Optional<int>()) {
	rapidjson::Document doc;
	doc.SetObject();

	if (version.present()) {
		addVersionToDoc(doc, version.get());
	}

	addErrorToDoc(doc, errorDetail);

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	ASSERT(!response->failed);
	response->failed = true;
	response->buff = std::string(sb.GetString(), sb.GetSize());
}

// Helper routine to extract 'version' from the input json document. If 'version' is missing or 'invalid', the routine
// is responsible to populate required error details to the 'response'
bool extractVersion(const rapidjson::Document& doc, VaultResponse* response, int* version) {
	// check version tag sanityrest_malformed_response
	if (!doc.HasMember(REQUEST_VERSION_TAG) || !doc[REQUEST_VERSION_TAG].IsInt()) {
		prepareErrorResponse(response, ErrorDetail(missingVersionCode, missingVersionMsg));
		CODE_PROBE(true, "RESTSimKmsVault missing version");
		return false;
	}

	*version = doc[REQUEST_VERSION_TAG].GetInt();
	if (*version < 0 || *version > SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION) {
		prepareErrorResponse(response, ErrorDetail(invalidVersionCode, invalidVersionMsg));
		CODE_PROBE(true, "RESTSimKmsVault invalid version");
		return false;
	}

	return true;
}

// Helper routine to validate 'validation-token(s)' from the input json document. If tokens are missing the routine is
// responsible to populate appropriate error to the 'response'
bool checkValidationTokens(const rapidjson::Document& doc, const int version, VaultResponse* response) {
	ASSERT(!response->failed);
	if (!doc.HasMember(VALIDATION_TOKENS_TAG) || !doc[VALIDATION_TOKENS_TAG].IsArray()) {
		prepareErrorResponse(response, ErrorDetail(missingTokenCode, missingTokensMsg), version);
		CODE_PROBE(true, "RESTSimKmsVault missing validation tokens");
		return false;
	}
	ASSERT(!response->failed);
	return true;
}

void addCipherDetailToRespDoc(rapidjson::Document& doc,
                              rapidjson::Value& cipherDetails,
                              const Reference<SimKmsVaultKeyCtx>& keyCtx,
                              const Optional<EncryptCipherDomainId> domId) {
	rapidjson::Value cipherDetail(rapidjson::kObjectType);

	// Add 'base_cipher_id'
	rapidjson::Value key(BASE_CIPHER_ID_TAG, doc.GetAllocator());
	rapidjson::Value baseKeyId;
	baseKeyId.SetUint64(keyCtx->id);
	cipherDetail.AddMember(key, baseKeyId, doc.GetAllocator());

	// Add 'encrypt_domain_id'
	if (domId.present()) {
		key.SetString(ENCRYPT_DOMAIN_ID_TAG, doc.GetAllocator());
		rapidjson::Value domainId;
		domainId.SetInt64(domId.get());
		cipherDetail.AddMember(key, domainId, doc.GetAllocator());
	}

	// Add 'cipher'
	key.SetString(BASE_CIPHER_TAG, doc.GetAllocator());
	rapidjson::Value cipher;
	ASSERT_EQ(keyCtx->key.size(), keyCtx->keyLen);
	cipher.SetString(reinterpret_cast<const char*>(keyCtx->key.begin()), keyCtx->keyLen, doc.GetAllocator());
	cipherDetail.AddMember(key, cipher, doc.GetAllocator());

	// Add 'refreshAt'
	key.SetString(REFRESH_AFTER_SEC, doc.GetAllocator());
	const int64_t refreshAt = getRefreshInterval(now(), FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value refreshInterval;
	refreshInterval.SetInt64(refreshAt);
	cipherDetail.AddMember(key, refreshInterval, doc.GetAllocator());

	// Add 'expireAt
	key.SetString(EXPIRE_AFTER_SEC, doc.GetAllocator());
	const int64_t expireAt = getExpireInterval(refreshAt, FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL);
	rapidjson::Value expireInterval;
	expireInterval.SetInt64(expireAt);
	cipherDetail.AddMember(key, expireInterval, doc.GetAllocator());

	// push above object to the array
	cipherDetails.PushBack(cipherDetail, doc.GetAllocator());
}

void addBlobMetadaToResDoc(rapidjson::Document& doc, rapidjson::Value& blobDetails, const EncryptCipherDomainId domId) {
	Standalone<BlobMetadataDetailsRef> detailsRef = SimKmsVault::getBlobMetadata(domId, bgUrl);
	rapidjson::Value blobDetail(rapidjson::kObjectType);

	rapidjson::Value key(BLOB_METADATA_DOMAIN_ID_TAG, doc.GetAllocator());
	rapidjson::Value domainId;
	domainId.SetInt64(domId);
	blobDetail.AddMember(key, domainId, doc.GetAllocator());

	rapidjson::Value locations(rapidjson::kArrayType);
	for (const auto& loc : detailsRef.locations) {
		rapidjson::Value location(rapidjson::kObjectType);

		// set location-id
		key.SetString(BLOB_METADATA_LOCATION_ID_TAG, doc.GetAllocator());
		rapidjson::Value id;
		id.SetInt64(loc.locationId);
		location.AddMember(key, id, doc.GetAllocator());

		// set location-path
		key.SetString(BLOB_METADATA_LOCATION_PATH_TAG, doc.GetAllocator());
		rapidjson::Value path;
		path.SetString(reinterpret_cast<const char*>(loc.path.begin()), loc.path.size(), doc.GetAllocator());
		location.AddMember(key, path, doc.GetAllocator());

		locations.PushBack(location, doc.GetAllocator());
	}
	key.SetString(BLOB_METADATA_LOCATIONS_TAG, doc.GetAllocator());
	blobDetail.AddMember(key, locations, doc.GetAllocator());

	blobDetails.PushBack(blobDetail, doc.GetAllocator());
}

void addKmsUrlsToDoc(rapidjson::Document& doc) {
	rapidjson::Value kmsUrls(rapidjson::kArrayType);
	// FIXME: fetch latest KMS URLs && append to the doc
	rapidjson::Value memberKey(KMS_URLS_TAG, doc.GetAllocator());
	doc.AddMember(memberKey, kmsUrls, doc.GetAllocator());
}

// TODO: inject faults
VaultResponse handleFetchKeysByDomainIds(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;
	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'cipher_key_details' as json array
	rapidjson::Value cipherDetails(rapidjson::kArrayType);
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		EncryptCipherDomainId domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domainId);
		ASSERT(keyCtx.isValid());
		addCipherDetailToRespDoc(result, cipherDetails, keyCtx, domainId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, cipherDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchByDomainIdsResponseStr").detail("Str", response->buff);
	return response;
}

VaultResponse handleFetchKeysByKeyIds(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;

	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'cipher_key_details' as json array
	rapidjson::Value cipherDetails(rapidjson::kArrayType);
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		Optional<EncryptCipherDomainId> domainId;
		if (cipherDetail.HasMember(ENCRYPT_DOMAIN_ID_TAG) && cipherDetail[ENCRYPT_DOMAIN_ID_TAG].IsInt64()) {
			domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		}
		EncryptCipherBaseKeyId baseCipherId = cipherDetail[BASE_CIPHER_ID_TAG].GetUint64();
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByBaseCipherId(baseCipherId);
		ASSERT(keyCtx.isValid());
		addCipherDetailToRespDoc(result, cipherDetails, keyCtx, domainId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, cipherDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchByKeyIdsResponseStr").detail("Str", response.buff);
	return response;
}

VaultResponse handleFetchBlobMetada(const std::string& content) {
	VaultResponse response;
	rapidjson::Document doc;

	doc.Parse(content.data());

	int version;

	if (!extractVersion(doc, &response, &version)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	if (!checkValidationTokens(doc, version, &response)) {
		// Return HTTP::HTTP_STATUS_CODE_OK with appropriate 'error' details
		ASSERT(response.failed);
		return response;
	}
	ASSERT(!response.failed);

	rapidjson::Document result;
	result.SetObject();

	// Append 'request version'
	addVersionToDoc(result, version);

	// Append 'blob_metadata_details' as json array
	rapidjson::Value blobDetails(rapidjson::kArrayType);
	for (const auto& blobDetail : doc[BLOB_METADATA_DETAILS_TAG].GetArray()) {
		EncryptCipherDomainId domainId = blobDetail[BLOB_METADATA_DOMAIN_ID_TAG].GetInt64();
		addBlobMetadaToResDoc(doc, blobDetails, domainId);
	}
	rapidjson::Value memberKey(BLOB_METADATA_DETAILS_TAG, result.GetAllocator());
	result.AddMember(memberKey, blobDetails, result.GetAllocator());

	if (doc.HasMember(KMS_URLS_TAG) && doc[KMS_URLS_TAG].GetBool()) {
		addKmsUrlsToDoc(result);
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	result.Accept(writer);

	ASSERT(!response.failed);
	response.buff = std::string(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FetchBlobMetadataResponeStr").detail("Str", response.buff);
	return response;
}

															#line 415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via simKmsVaultRequestHandler()
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class SimKmsVaultRequestHandlerActor>
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class SimKmsVaultRequestHandlerActorState {
															#line 422 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	SimKmsVaultRequestHandlerActorState(Reference<HTTP::IncomingRequest> const& request,Reference<HTTP::OutgoingResponse> const& response) 
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : request(request),
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(response)
															#line 431 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this));

	}
	~SimKmsVaultRequestHandlerActorState() 
	{
		fdb_probe_actor_destroy("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0);
															#line 415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 448 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state = 1;
															#line 415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*>(static_cast<SimKmsVaultRequestHandlerActor*>(this)));
															#line 453 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 416 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT_EQ(request->verb, HTTP::HTTP_VERB_POST);
															#line 418 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		validateHeaders(request->data.headers);
															#line 420 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		vaultResponse = VaultResponse();
															#line 421 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_KEY_IDS_RESOURCE) == 0)
															#line 482 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		{
															#line 422 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			vaultResponse = handleFetchKeysByKeyIds(request->data.content);
															#line 486 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		}
		else
		{
															#line 423 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_DOMAIN_IDS_RESOURCE) == 0)
															#line 492 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			{
															#line 424 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				vaultResponse = handleFetchKeysByDomainIds(request->data.content);
															#line 496 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			}
			else
			{
															#line 425 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (request->resource.compare(REST_SIM_KMS_VAULT_GET_BLOB_METADATA_RESOURCE) == 0)
															#line 502 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				{
															#line 426 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					vaultResponse = handleFetchBlobMetada(request->data.content);
															#line 506 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				}
				else
				{
															#line 428 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					TraceEvent("UnexpectedResource").detail("Resource", request->resource);
															#line 429 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					return a_body1Catch1(http_bad_response(), loopDepth);
															#line 514 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				}
			}
		}
															#line 432 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->code = HTTP::HTTP_STATUS_CODE_OK;
															#line 433 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.headers = request->data.headers;
															#line 434 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		PacketWriter pw(response->data.content->getWriteBuffer(vaultResponse.buff.size()), nullptr, Unversioned());
															#line 435 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		pw.serializeBytes(vaultResponse.buff.data(), vaultResponse.buff.size());
															#line 436 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.contentLen = vaultResponse.buff.size();
															#line 438 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SimKmsVaultRequestHandlerActorState(); static_cast<SimKmsVaultRequestHandlerActor*>(this)->destroy(); return 0; }
															#line 530 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		new (&static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 416 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT_EQ(request->verb, HTTP::HTTP_VERB_POST);
															#line 418 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		validateHeaders(request->data.headers);
															#line 420 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		vaultResponse = VaultResponse();
															#line 421 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_KEY_IDS_RESOURCE) == 0)
															#line 548 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		{
															#line 422 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			vaultResponse = handleFetchKeysByKeyIds(request->data.content);
															#line 552 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		}
		else
		{
															#line 423 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (request->resource.compare(REST_SIM_KMS_VAULT_GET_ENCRYPTION_KEYS_BY_DOMAIN_IDS_RESOURCE) == 0)
															#line 558 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			{
															#line 424 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				vaultResponse = handleFetchKeysByDomainIds(request->data.content);
															#line 562 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			}
			else
			{
															#line 425 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (request->resource.compare(REST_SIM_KMS_VAULT_GET_BLOB_METADATA_RESOURCE) == 0)
															#line 568 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				{
															#line 426 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					vaultResponse = handleFetchBlobMetada(request->data.content);
															#line 572 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				}
				else
				{
															#line 428 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					TraceEvent("UnexpectedResource").detail("Resource", request->resource);
															#line 429 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
					return a_body1Catch1(http_bad_response(), loopDepth);
															#line 580 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				}
			}
		}
															#line 432 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->code = HTTP::HTTP_STATUS_CODE_OK;
															#line 433 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.headers = request->data.headers;
															#line 434 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		PacketWriter pw(response->data.content->getWriteBuffer(vaultResponse.buff.size()), nullptr, Unversioned());
															#line 435 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		pw.serializeBytes(vaultResponse.buff.data(), vaultResponse.buff.size());
															#line 436 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		response->data.contentLen = vaultResponse.buff.size();
															#line 438 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SimKmsVaultRequestHandlerActorState(); static_cast<SimKmsVaultRequestHandlerActor*>(this)->destroy(); return 0; }
															#line 596 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		new (&static_cast<SimKmsVaultRequestHandlerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SimKmsVaultRequestHandlerActorState();
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state > 0) static_cast<SimKmsVaultRequestHandlerActor*>(this)->actor_wait_state = 0;
		static_cast<SimKmsVaultRequestHandlerActor*>(this)->ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 420 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	VaultResponse vaultResponse;
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via simKmsVaultRequestHandler()
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class SimKmsVaultRequestHandlerActor final : public Actor<Void>, public ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >, public FastAllocated<SimKmsVaultRequestHandlerActor>, public SimKmsVaultRequestHandlerActorState<SimKmsVaultRequestHandlerActor> {
															#line 690 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<SimKmsVaultRequestHandlerActor>::operator new;
	using FastAllocated<SimKmsVaultRequestHandlerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3574980721292837888UL, 3159392666326331136UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >;
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	SimKmsVaultRequestHandlerActor(Reference<HTTP::IncomingRequest> const& request,Reference<HTTP::OutgoingResponse> const& response) 
															#line 707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   SimKmsVaultRequestHandlerActorState<SimKmsVaultRequestHandlerActor>(request, response),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3523274803376955648UL, 4651189297213340672UL);
		ActorExecutionContextHelper __helper(static_cast<SimKmsVaultRequestHandlerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("simKmsVaultRequestHandler");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("simKmsVaultRequestHandler", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SimKmsVaultRequestHandlerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
[[nodiscard]] Future<Void> simKmsVaultRequestHandler( Reference<HTTP::IncomingRequest> const& request, Reference<HTTP::OutgoingResponse> const& response ) {
															#line 413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new SimKmsVaultRequestHandlerActor(request, response));
															#line 740 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}

#line 440 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

Future<Void> RESTSimKmsVaultRequestHandler::handleRequest(Reference<HTTP::IncomingRequest> request,
                                                          Reference<HTTP::OutgoingResponse> response) {
	return simKmsVaultRequestHandler(request, response);
}

// Only used to link unit tests
void forceLinkRESTSimKmsVaultTest() {}

namespace {

enum class FaultType { NONE = 1, MISSING_VERSION = 2, INVALID_VERSION = 3, MISSING_VALIDATION_TOKEN = 4 };

void addFakeValidationTokens(rapidjson::Document& doc) {
	ValidationTokenMap tokenMap;
	tokenMap.emplace("foo", ValidationTokenCtx("bar", ValidationTokenSource::VALIDATION_TOKEN_SOURCE_FILE));

	addValidationTokensSectionToJsonDoc(doc, tokenMap);
}

void constructDomainIds(EncryptCipherDomainIdVec& domIds) {
	domIds.push_back(SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID);
	domIds.push_back(FDB_DEFAULT_ENCRYPT_DOMAIN_ID);
	domIds.push_back(ENCRYPT_HEADER_DOMAIN_ID);

	int idx = deterministicRandom()->randomInt(512, 786);
	int count = deterministicRandom()->randomInt(5, 100);
	while (count--) {
		domIds.push_back(idx++);
	}
}

std::string getFakeDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds,
                                           const char* rootTag,
                                           const char* elementTag,
                                           FaultType fault = FaultType::NONE) {
	rapidjson::Document doc;
	doc.SetObject();

	if (fault == FaultType::INVALID_VERSION) {
		addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION + 1);
	} else if (fault == FaultType::MISSING_VERSION) {
		// Skip adding the version
	} else {
		addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION);
	}

	if (fault != FaultType::MISSING_VALIDATION_TOKEN) {
		addFakeValidationTokens(doc);
	}

	constructDomainIds(domIds);
	addLatestDomainDetailsToDoc(doc, rootTag, elementTag, domIds);

	addRefreshKmsUrlsSectionToJsonDoc(doc, deterministicRandom()->coinflip());

	if (deterministicRandom()->coinflip()) {
		addDebugUidSectionToJsonDoc(doc, deterministicRandom()->randomUniqueID());
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	std::string resp(sb.GetString(), sb.GetSize());
	/*TraceEvent(SevDebug, "FakeDomainIdsRequest")
	    .detail("Str", resp)
	    .detail("RootTag", rootTag)
	    .detail("ElementTag", elementTag);*/
	return resp;
}

std::string getFakeEncryptDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	return getFakeDomainIdsRequestContent(domIds, CIPHER_KEY_DETAILS_TAG, ENCRYPT_DOMAIN_ID_TAG, fault);
}

std::string getFakeBlobDomainIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	return getFakeDomainIdsRequestContent(domIds, BLOB_METADATA_DETAILS_TAG, BLOB_METADATA_DOMAIN_ID_TAG, fault);
}

std::string getFakeBaseCipherIdsRequestContent(EncryptCipherDomainIdVec& domIds, FaultType fault = FaultType::NONE) {
	rapidjson::Document doc;
	doc.SetObject();

	if (fault != FaultType::MISSING_VERSION) {
		if (fault == FaultType::INVALID_VERSION) {
			addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION + 1);
		}
		addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION);
	}

	if (fault != FaultType::MISSING_VALIDATION_TOKEN) {
		addFakeValidationTokens(doc);
	}

	constructDomainIds(domIds);
	rapidjson::Value keyIdDetails(rapidjson::kArrayType);
	for (auto domId : domIds) {
		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domId);
		ASSERT(keyCtx.isValid());
		addBaseCipherIdDomIdToDoc(doc, keyIdDetails, keyCtx->id, domId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, doc.GetAllocator());
	doc.AddMember(memberKey, keyIdDetails, doc.GetAllocator());

	addRefreshKmsUrlsSectionToJsonDoc(doc, deterministicRandom()->coinflip());

	if (deterministicRandom()->coinflip()) {
		addDebugUidSectionToJsonDoc(doc, deterministicRandom()->randomUniqueID());
	}

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	std::string resp(sb.GetString(), sb.GetSize());
	//TraceEvent(SevDebug, "FakeKeyIdsRequest").detail("Str", resp);
	return resp;
}

Optional<ErrorDetail> getErrorDetail(const std::string& buff) {
	rapidjson::Document doc;
	doc.Parse(buff.data());
	return RESTKmsConnectorUtils::getError(doc);
}

void validateEncryptLookup(const VaultResponse& response, const EncryptCipherDomainIdVec& domIds) {
	ASSERT(!response.failed);

	//TraceEvent(SevDebug, "VaultEncryptResponse").detail("Str", response.buff);

	rapidjson::Document doc;
	doc.Parse(response.buff.data());

	ASSERT(doc.HasMember(CIPHER_KEY_DETAILS_TAG) && doc[CIPHER_KEY_DETAILS_TAG].IsArray());

	std::unordered_set<EncryptCipherDomainId> domIdSet(domIds.begin(), domIds.end());
	int count = 0;
	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		EncryptCipherDomainId domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		EncryptCipherBaseKeyId baseCipherId = cipherDetail[BASE_CIPHER_ID_TAG].GetUint64();
		const int cipherKeyLen = cipherDetail[BASE_CIPHER_TAG].GetStringLength();
		Standalone<StringRef> cipherKeyRef = makeString(cipherKeyLen);
		memcpy(mutateString(cipherKeyRef), cipherDetail[BASE_CIPHER_TAG].GetString(), cipherKeyLen);

		ASSERT(domIdSet.find(domainId) != domIdSet.end());

		Reference<SimKmsVaultKeyCtx> keyCtx = SimKmsVault::getByDomainId(domainId);
		ASSERT_EQ(keyCtx->id, baseCipherId);
		ASSERT_EQ(keyCtx->key.compare(cipherKeyRef), 0);
		const int64_t refreshAfterSec = cipherDetail[REFRESH_AFTER_SEC].GetInt64();
		const int64_t expireAfterSec = cipherDetail[EXPIRE_AFTER_SEC].GetInt64();
		ASSERT(refreshAfterSec <= expireAfterSec || expireAfterSec == -1);
		count++;
	}
	ASSERT_EQ(count, domIds.size());
}

void validateBlobLookup(const VaultResponse& response, const EncryptCipherDomainIdVec& domIds) {
	ASSERT(!response.failed);

	//TraceEvent(SevDebug, "VaultBlobResponse").detail("Str", response.buff);

	rapidjson::Document doc;
	doc.Parse(response.buff.data());

	ASSERT(doc.HasMember(BLOB_METADATA_DETAILS_TAG) && doc[BLOB_METADATA_DETAILS_TAG].IsArray());

	std::unordered_set<EncryptCipherDomainId> domIdSet(domIds.begin(), domIds.end());
	int count = 0;
	for (const auto& blobDetail : doc[BLOB_METADATA_DETAILS_TAG].GetArray()) {
		EncryptCipherDomainId domainId = blobDetail[BLOB_METADATA_DOMAIN_ID_TAG].GetInt64();
		Standalone<BlobMetadataDetailsRef> details = SimKmsVault::getBlobMetadata(domainId, bgUrl);

		std::unordered_map<BlobMetadataLocationId, Standalone<StringRef>> locMap;
		for (const auto& loc : details.locations) {
			locMap[loc.locationId] = loc.path;
		}
		for (const auto& location : blobDetail[BLOB_METADATA_LOCATIONS_TAG].GetArray()) {
			BlobMetadataLocationId locationId = location[BLOB_METADATA_LOCATION_ID_TAG].GetInt64();
			Standalone<StringRef> path = makeString(location[BLOB_METADATA_LOCATION_PATH_TAG].GetStringLength());
			memcpy(mutateString(path),
			       location[BLOB_METADATA_LOCATION_PATH_TAG].GetString(),
			       location[BLOB_METADATA_LOCATION_PATH_TAG].GetStringLength());
			auto it = locMap.find(locationId);
			ASSERT(it != locMap.end());
			ASSERT_EQ(it->second.compare(path), 0);
		}
		const int64_t refreshAfterSec = blobDetail[REFRESH_AFTER_SEC].GetInt64();
		const int64_t expireAfterSec = blobDetail[EXPIRE_AFTER_SEC].GetInt64();
		ASSERT(refreshAfterSec <= expireAfterSec || expireAfterSec == -1);
		count++;
	}
	ASSERT_EQ(count, domIds.size());
}

} // namespace

															#line 944 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase640()
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase640Actor>
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase640ActorState {
															#line 951 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase640ActorState(UnitTestParameters const& params) 
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params),
															#line 641 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   request(makeReference<HTTP::IncomingRequest>()),
															#line 642 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(makeReference<HTTP::OutgoingResponse>())
															#line 962 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase640", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase640ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase640", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 644 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->verb = HTTP::HTTP_VERB_POST;
															#line 645 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->resource = "/whatever";
															#line 646 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers = RESTKmsConnectorUtils::getHTTPHeaders();
															#line 981 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			try {
															#line 648 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				StrictFuture<Void> __when_expr_0 = simKmsVaultRequestHandler(request, response);
															#line 648 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (static_cast<FlowTestCase640Actor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 987 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FlowTestCase640Actor*>(this)->actor_wait_state = 1;
															#line 648 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase640Actor, 0, Void >*>(static_cast<FlowTestCase640Actor*>(this)));
															#line 992 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase640ActorState();
		static_cast<FlowTestCase640Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 653 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<FlowTestCase640Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase640ActorState(); static_cast<FlowTestCase640Actor*>(this)->destroy(); return 0; }
															#line 1021 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		new (&static_cast<FlowTestCase640Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase640ActorState();
		static_cast<FlowTestCase640Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 651 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT_EQ(e.code(), error_code_http_bad_response);
															#line 1034 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 649 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1049 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 649 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1058 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase640Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase640Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase640Actor*>(this)->ActorCallback< FlowTestCase640Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase640Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase640", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase640Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase640", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase640Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase640", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase640Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase640", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase640Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase640", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase640Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase640", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 641 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 642 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1157 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase640()
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase640Actor final : public Actor<Void>, public ActorCallback< FlowTestCase640Actor, 0, Void >, public FastAllocated<FlowTestCase640Actor>, public FlowTestCase640ActorState<FlowTestCase640Actor> {
															#line 1162 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase640Actor>::operator new;
	using FastAllocated<FlowTestCase640Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5599781024070457344UL, 10730451647888408064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase640Actor, 0, Void >;
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase640Actor(UnitTestParameters const& params) 
															#line 1179 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase640ActorState<FlowTestCase640Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase640", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14993956912757602560UL, 8378303830399465472UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase640Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase640");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase640", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase640Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase640( UnitTestParameters const& params ) {
															#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase640Actor(params));
															#line 1212 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase640, "/restSimKmsVault/invalidResource")

#line 655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1218 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase656()
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase656Actor>
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase656ActorState {
															#line 1225 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase656ActorState(UnitTestParameters const& params) 
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params),
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   request(makeReference<HTTP::IncomingRequest>()),
															#line 658 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		   response(makeReference<HTTP::OutgoingResponse>())
															#line 1236 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase656", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase656ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase656", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 660 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->verb = HTTP::HTTP_VERB_POST;
															#line 661 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->resource = "/whatever";
															#line 662 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers = RESTKmsConnectorUtils::getHTTPHeaders();
															#line 663 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			request->data.headers["Foo"] = "Bar";
															#line 1257 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			try {
															#line 665 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				StrictFuture<Void> __when_expr_0 = simKmsVaultRequestHandler(request, response);
															#line 665 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				if (static_cast<FlowTestCase656Actor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1263 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FlowTestCase656Actor*>(this)->actor_wait_state = 1;
															#line 665 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase656Actor, 0, Void >*>(static_cast<FlowTestCase656Actor*>(this)));
															#line 1268 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase656ActorState();
		static_cast<FlowTestCase656Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 670 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		if (!static_cast<FlowTestCase656Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase656ActorState(); static_cast<FlowTestCase656Actor*>(this)->destroy(); return 0; }
															#line 1297 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		new (&static_cast<FlowTestCase656Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase656ActorState();
		static_cast<FlowTestCase656Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 668 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT_EQ(e.code(), error_code_rest_malformed_response);
															#line 1310 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 666 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1325 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 666 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		ASSERT(false);
															#line 1334 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase656Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase656Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase656Actor*>(this)->ActorCallback< FlowTestCase656Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase656Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase656", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase656Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase656", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase656Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase656", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase656Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase656", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase656Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase656", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase656Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase656", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::IncomingRequest> request;
															#line 658 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1433 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase656()
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase656Actor final : public Actor<Void>, public ActorCallback< FlowTestCase656Actor, 0, Void >, public FastAllocated<FlowTestCase656Actor>, public FlowTestCase656ActorState<FlowTestCase656Actor> {
															#line 1438 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase656Actor>::operator new;
	using FastAllocated<FlowTestCase656Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16539613886444633088UL, 17158257385955440128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase656Actor, 0, Void >;
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase656Actor(UnitTestParameters const& params) 
															#line 1455 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase656ActorState<FlowTestCase656Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase656", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15191057593148120320UL, 277194864508418560UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase656Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase656");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase656", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase656Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase656( UnitTestParameters const& params ) {
															#line 656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase656Actor(params));
															#line 1488 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase656, "/restSimKmsVault/invalidHeader")

#line 672 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1494 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase673()
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase673Actor>
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase673ActorState {
															#line 1501 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase673ActorState(UnitTestParameters const& params) 
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1508 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase673", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase673ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase673", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 674 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 675 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 676 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 677 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 678 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 679 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 680 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase673Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase673ActorState(); static_cast<FlowTestCase673Actor*>(this)->destroy(); return 0; }
															#line 1537 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase673Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase673ActorState();
			static_cast<FlowTestCase673Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase673ActorState();
		static_cast<FlowTestCase673Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1561 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase673()
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase673Actor final : public Actor<Void>, public FastAllocated<FlowTestCase673Actor>, public FlowTestCase673ActorState<FlowTestCase673Actor> {
															#line 1566 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase673Actor>::operator new;
	using FastAllocated<FlowTestCase673Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4283528354313475072UL, 12253159219852449280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase673Actor(UnitTestParameters const& params) 
															#line 1582 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase673ActorState<FlowTestCase673Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase673", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6451569151394209792UL, 15567082101615168000UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase673Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase673");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase673", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase673( UnitTestParameters const& params ) {
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase673Actor(params));
															#line 1614 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase673, "/restSimKmsVault/GetByDomainIds/missingVersion")

#line 684 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1620 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase685()
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase685Actor>
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase685ActorState {
															#line 1627 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase685ActorState(UnitTestParameters const& params) 
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1634 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase685", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase685ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase685", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 686 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 687 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 688 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 689 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 690 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 691 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 692 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 694 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase685Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase685ActorState(); static_cast<FlowTestCase685Actor*>(this)->destroy(); return 0; }
															#line 1663 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase685Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase685ActorState();
			static_cast<FlowTestCase685Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase685ActorState();
		static_cast<FlowTestCase685Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1687 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase685()
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase685Actor final : public Actor<Void>, public FastAllocated<FlowTestCase685Actor>, public FlowTestCase685ActorState<FlowTestCase685Actor> {
															#line 1692 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase685Actor>::operator new;
	using FastAllocated<FlowTestCase685Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9828393954718337536UL, 15345948469611887360UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase685Actor(UnitTestParameters const& params) 
															#line 1708 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase685ActorState<FlowTestCase685Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase685", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11251837935623794432UL, 14876149652932320000UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase685Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase685");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase685", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase685( UnitTestParameters const& params ) {
															#line 685 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase685Actor(params));
															#line 1740 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase685, "/restSimKmsVault/GetByDomainIds/invalidVersion")

#line 696 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1746 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase697()
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase697Actor>
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase697ActorState {
															#line 1753 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase697ActorState(UnitTestParameters const& params) 
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1760 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase697", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase697ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase697", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 698 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 699 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 701 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 704 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 705 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase697Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase697ActorState(); static_cast<FlowTestCase697Actor*>(this)->destroy(); return 0; }
															#line 1789 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase697Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase697ActorState();
			static_cast<FlowTestCase697Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase697ActorState();
		static_cast<FlowTestCase697Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1813 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase697()
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase697Actor final : public Actor<Void>, public FastAllocated<FlowTestCase697Actor>, public FlowTestCase697ActorState<FlowTestCase697Actor> {
															#line 1818 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase697Actor>::operator new;
	using FastAllocated<FlowTestCase697Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11855073372293458176UL, 16114482374926401024UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase697Actor(UnitTestParameters const& params) 
															#line 1834 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase697ActorState<FlowTestCase697Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase697", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5629460012882567424UL, 6444844894057364224UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase697Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase697");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase697", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase697( UnitTestParameters const& params ) {
															#line 697 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase697Actor(params));
															#line 1866 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase697, "/restSimKmsVault/GetByDomainIds/missingValidationTokens")

#line 709 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1872 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase710()
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase710Actor>
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase710ActorState {
															#line 1879 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase710ActorState(UnitTestParameters const& params) 
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 1886 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase710", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase710ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase710", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 711 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 712 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeEncryptDomainIdsRequestContent(domIds);
															#line 714 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByDomainIds(requestContent);
															#line 715 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateEncryptLookup(response, domIds);
															#line 716 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase710Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase710ActorState(); static_cast<FlowTestCase710Actor*>(this)->destroy(); return 0; }
															#line 1909 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase710Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase710ActorState();
			static_cast<FlowTestCase710Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase710ActorState();
		static_cast<FlowTestCase710Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 1933 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase710()
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase710Actor final : public Actor<Void>, public FastAllocated<FlowTestCase710Actor>, public FlowTestCase710ActorState<FlowTestCase710Actor> {
															#line 1938 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase710Actor>::operator new;
	using FastAllocated<FlowTestCase710Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2910465282144164096UL, 10868324639114874880UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase710Actor(UnitTestParameters const& params) 
															#line 1954 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase710ActorState<FlowTestCase710Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase710", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15754981172551286784UL, 8976077587038495488UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase710Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase710");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase710", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase710( UnitTestParameters const& params ) {
															#line 710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase710Actor(params));
															#line 1986 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase710, "/restSimKmsVault/GetByDomainIds")

#line 718 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 1992 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase719()
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase719Actor>
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase719ActorState {
															#line 1999 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase719ActorState(UnitTestParameters const& params) 
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2006 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase719", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase719ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase719", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 720 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 721 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 723 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 724 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 725 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 726 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 727 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 729 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase719Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase719ActorState(); static_cast<FlowTestCase719Actor*>(this)->destroy(); return 0; }
															#line 2035 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase719Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase719ActorState();
			static_cast<FlowTestCase719Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase719ActorState();
		static_cast<FlowTestCase719Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2059 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase719()
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase719Actor final : public Actor<Void>, public FastAllocated<FlowTestCase719Actor>, public FlowTestCase719ActorState<FlowTestCase719Actor> {
															#line 2064 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase719Actor>::operator new;
	using FastAllocated<FlowTestCase719Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13332964172335601408UL, 11789565742938419968UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase719Actor(UnitTestParameters const& params) 
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase719ActorState<FlowTestCase719Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase719", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12646733125748677120UL, 5412006093491094528UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase719Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase719");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase719", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase719( UnitTestParameters const& params ) {
															#line 719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase719Actor(params));
															#line 2112 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase719, "/restSimKmsVault/GetByKeyIds/missingVersion")

#line 731 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2118 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase732()
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase732Actor>
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase732ActorState {
															#line 2125 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase732ActorState(UnitTestParameters const& params) 
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2132 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase732", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase732ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase732", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 733 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 734 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 736 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 737 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 738 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 739 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 740 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 742 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase732Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase732ActorState(); static_cast<FlowTestCase732Actor*>(this)->destroy(); return 0; }
															#line 2161 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase732Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase732ActorState();
			static_cast<FlowTestCase732Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase732ActorState();
		static_cast<FlowTestCase732Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2185 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase732()
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase732Actor final : public Actor<Void>, public FastAllocated<FlowTestCase732Actor>, public FlowTestCase732ActorState<FlowTestCase732Actor> {
															#line 2190 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase732Actor>::operator new;
	using FastAllocated<FlowTestCase732Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13705306467120434688UL, 16067943563733685504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase732Actor(UnitTestParameters const& params) 
															#line 2206 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase732ActorState<FlowTestCase732Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase732", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8498616368470848768UL, 4168663777638849792UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase732Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase732");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase732", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase732( UnitTestParameters const& params ) {
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase732Actor(params));
															#line 2238 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase732, "/restSimKmsVault/GetByKeyIds/invalidVersion")

#line 744 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2244 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase745()
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase745Actor>
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase745ActorState {
															#line 2251 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase745ActorState(UnitTestParameters const& params) 
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2258 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase745", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase745ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase745", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 746 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 747 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 749 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 750 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 751 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 752 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 753 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 755 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase745Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase745ActorState(); static_cast<FlowTestCase745Actor*>(this)->destroy(); return 0; }
															#line 2287 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase745Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase745ActorState();
			static_cast<FlowTestCase745Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase745ActorState();
		static_cast<FlowTestCase745Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2311 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase745()
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase745Actor final : public Actor<Void>, public FastAllocated<FlowTestCase745Actor>, public FlowTestCase745ActorState<FlowTestCase745Actor> {
															#line 2316 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase745Actor>::operator new;
	using FastAllocated<FlowTestCase745Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3438191716655232256UL, 14236418294881300736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase745Actor(UnitTestParameters const& params) 
															#line 2332 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase745ActorState<FlowTestCase745Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase745", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5601893455724070144UL, 680185364163385344UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase745Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase745");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase745", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase745( UnitTestParameters const& params ) {
															#line 745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase745Actor(params));
															#line 2364 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase745, "/restSimKmsVault/GetByKeyIds/missingValidationTokens")

#line 757 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2370 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase758()
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase758Actor>
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase758ActorState {
															#line 2377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase758ActorState(UnitTestParameters const& params) 
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2384 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase758", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase758ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase758", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 759 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 760 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::MISSING_VALIDATION_TOKEN);
															#line 762 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 763 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 764 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 765 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 766 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingTokenCode, missingTokensMsg)));
															#line 768 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase758Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase758ActorState(); static_cast<FlowTestCase758Actor*>(this)->destroy(); return 0; }
															#line 2413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase758Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase758ActorState();
			static_cast<FlowTestCase758Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase758ActorState();
		static_cast<FlowTestCase758Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2437 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase758()
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase758Actor final : public Actor<Void>, public FastAllocated<FlowTestCase758Actor>, public FlowTestCase758ActorState<FlowTestCase758Actor> {
															#line 2442 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase758Actor>::operator new;
	using FastAllocated<FlowTestCase758Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8215789800684477952UL, 17374070592432506880UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase758Actor(UnitTestParameters const& params) 
															#line 2458 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase758ActorState<FlowTestCase758Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase758", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5558394056721664UL, 3471552018802204672UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase758Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase758");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase758", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase758( UnitTestParameters const& params ) {
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase758Actor(params));
															#line 2490 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase758, "/restSimKmsVault/GetByKeyIds/missingValidationTokensBlob")

#line 770 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2496 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase771()
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase771Actor>
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase771ActorState {
															#line 2503 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase771ActorState(UnitTestParameters const& params) 
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2510 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase771", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase771ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase771", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 772 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 773 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBaseCipherIdsRequestContent(domIds);
															#line 775 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchKeysByKeyIds(requestContent);
															#line 776 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateEncryptLookup(response, domIds);
															#line 777 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase771Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase771ActorState(); static_cast<FlowTestCase771Actor*>(this)->destroy(); return 0; }
															#line 2533 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase771Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase771ActorState();
			static_cast<FlowTestCase771Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase771ActorState();
		static_cast<FlowTestCase771Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2557 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase771()
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase771Actor final : public Actor<Void>, public FastAllocated<FlowTestCase771Actor>, public FlowTestCase771ActorState<FlowTestCase771Actor> {
															#line 2562 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase771Actor>::operator new;
	using FastAllocated<FlowTestCase771Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14185148519809311232UL, 5967373557166436608UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase771Actor(UnitTestParameters const& params) 
															#line 2578 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase771ActorState<FlowTestCase771Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase771", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15424100140347215616UL, 5145059038419179008UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase771Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase771");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase771", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase771( UnitTestParameters const& params ) {
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase771Actor(params));
															#line 2610 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase771, "/restSimKmsVault/GetByKeyIds")

#line 779 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2616 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase780()
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase780Actor>
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase780ActorState {
															#line 2623 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase780ActorState(UnitTestParameters const& params) 
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2630 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase780", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase780ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase780", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 781 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 782 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::MISSING_VERSION);
															#line 784 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 785 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 786 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 787 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 788 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(missingVersionCode, missingVersionMsg)));
															#line 790 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase780Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase780ActorState(); static_cast<FlowTestCase780Actor*>(this)->destroy(); return 0; }
															#line 2659 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase780Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase780ActorState();
			static_cast<FlowTestCase780Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase780ActorState();
		static_cast<FlowTestCase780Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2683 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase780()
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase780Actor final : public Actor<Void>, public FastAllocated<FlowTestCase780Actor>, public FlowTestCase780ActorState<FlowTestCase780Actor> {
															#line 2688 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase780Actor>::operator new;
	using FastAllocated<FlowTestCase780Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14383158173650749440UL, 11230982466096653568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase780Actor(UnitTestParameters const& params) 
															#line 2704 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase780ActorState<FlowTestCase780Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase780", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7008406062230896896UL, 8003078899513820416UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase780Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase780");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase780", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase780( UnitTestParameters const& params ) {
															#line 780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase780Actor(params));
															#line 2736 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase780, "/restSimKmsVault/GetBlobMetadata/missingVersion")

#line 792 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2742 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase793()
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase793Actor>
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase793ActorState {
															#line 2749 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase793ActorState(UnitTestParameters const& params) 
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2756 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase793", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase793ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase793", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 794 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 795 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds, FaultType::INVALID_VERSION);
															#line 797 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 798 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(response.failed);
															#line 799 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			Optional<ErrorDetail> detail = getErrorDetail(response.buff);
															#line 800 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail.present());
															#line 801 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			ASSERT(detail->isEqual(ErrorDetail(invalidVersionCode, invalidVersionMsg)));
															#line 803 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase793Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase793ActorState(); static_cast<FlowTestCase793Actor*>(this)->destroy(); return 0; }
															#line 2785 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase793Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase793ActorState();
			static_cast<FlowTestCase793Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase793ActorState();
		static_cast<FlowTestCase793Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2809 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase793()
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase793Actor final : public Actor<Void>, public FastAllocated<FlowTestCase793Actor>, public FlowTestCase793ActorState<FlowTestCase793Actor> {
															#line 2814 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase793Actor>::operator new;
	using FastAllocated<FlowTestCase793Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6012685983061911552UL, 2068858817259130112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase793Actor(UnitTestParameters const& params) 
															#line 2830 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase793ActorState<FlowTestCase793Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase793", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4931143392259050496UL, 9536034588804408064UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase793Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase793");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase793", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase793( UnitTestParameters const& params ) {
															#line 793 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase793Actor(params));
															#line 2862 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase793, "/restSimKmsVault/GetBlobMetadata/invalidVersion")

#line 805 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"

															#line 2868 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase806()
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
template <class FlowTestCase806Actor>
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase806ActorState {
															#line 2875 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase806ActorState(UnitTestParameters const& params) 
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
		 : params(params)
															#line 2882 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase806", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase806ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase806", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 807 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			EncryptCipherDomainIdVec domIds;
															#line 808 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			std::string requestContent = getFakeBlobDomainIdsRequestContent(domIds);
															#line 810 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			VaultResponse response = handleFetchBlobMetada(requestContent);
															#line 811 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			validateBlobLookup(response, domIds);
															#line 812 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
			if (!static_cast<FlowTestCase806Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase806ActorState(); static_cast<FlowTestCase806Actor*>(this)->destroy(); return 0; }
															#line 2905 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
			new (&static_cast<FlowTestCase806Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase806ActorState();
			static_cast<FlowTestCase806Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase806ActorState();
		static_cast<FlowTestCase806Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	UnitTestParameters params;
															#line 2929 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase806()
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
class FlowTestCase806Actor final : public Actor<Void>, public FastAllocated<FlowTestCase806Actor>, public FlowTestCase806ActorState<FlowTestCase806Actor> {
															#line 2934 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase806Actor>::operator new;
	using FastAllocated<FlowTestCase806Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10276186503729910272UL, 8249449618457664256UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	FlowTestCase806Actor(UnitTestParameters const& params) 
															#line 2950 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase806ActorState<FlowTestCase806Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase806", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13119145203896912896UL, 7539303562061006336UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase806Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase806");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase806", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
static Future<Void> flowTestCase806( UnitTestParameters const& params ) {
															#line 806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/RESTSimKmsVault.actor.cpp"
	return Future<Void>(new FlowTestCase806Actor(params));
															#line 2982 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/RESTSimKmsVault.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase806, "/restSimKmsVault/GetBlobMetadata/foo")

